# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import namedtuple
from Ark import (
   switchTimeToUtc
)
import CliCommand
import LazyMount
import Smash
import SmashLazyMount
import SharedMem
import Tac
import Tracing
from Arnet import IpGenAddr
from BasicCli import (
   addShowCommandClass,
)
from CliToken.Flow import (
   flowMatcherForShow,
)
from IpLibConsts import ALL_VRF_NAME
from FlowTrackingCliLib import (
   exporterKw,
   exporterNameMatcher,
   FtConsts,
   getIfIndexMap,
   getFlowGroups,
   getFlowGroupsAndIds,
   getTrackerNamesAndIds,
   isSftAgentRunning,
   sampledShowKw,
   trackerKw,
   trackingShowKw,
   trackerNameMatcher,
   IP_GROUP,
   IP6_GROUP,
   flowTableKw,
   groupKw,
   allGroupNameMatcher,
   getFtrTypeFromArgs,
)
from TypeFuture import TacLazyType
from SftLib import (
   ifindexToIntf,
   vniOrUnknown,
)
from FlowTrackerCliUtil import (
   ftrTypeSampled,
   ipStr,
   protocolStr,
   tcpFlagStr,
   ftrTypeInbandTelemetry,
)
from SftCliLib import (
      getFlowTable,
      intfRangeRule,
      sampledHardwareKw,
      tokenDstIp,
      tokenDstPort,
      tokenInterface,
      tokenIPv4,
      tokenIPv6,
      tokenPort,
      tokenProtocol,
      tokenProtocolValue,
      tokenSrcIp,
      tokenSrcPort,
      tokenVlan,
      tokenVlanValue,
      tokenVrf,
      tokenVrfValue,
      tokenState,
      tokenStateValue,
)
from SftCliUtil import (
   HoFlowStateEnum,
   HoFlowConfig,
   HoFlowCounter,
   HoFlowStatus,
   HosStateCounter,
   HosStateCounterKey,
)
import SftModel
import ShowCommand
import CliMatcher
from CliPlugin.FlowTrackingCounterCliLib import (
   FlowGroupCounterKey,
   FlowGroupCounterEntry,
   TemplateIdType,
   CollectorStatisticsKey,
   CollectorInfo,
   addExporters,
   FlowCounterKey,
   FlowCounterEntry,
)
from Toggles.FlowTrackerToggleLib import toggleHwOffloadIpv4Enabled

FtConstants = TacLazyType( 'FlowTracking::Constants' )
IpfixConst = TacLazyType( 'Arnet::IpfixConst' )
SmashCollectorInfo = TacLazyType( 'Smash::Ipfix::CollectorInfo' )
TemplateIdEnum = TacLazyType( 'FlowTracking::StaticTemplateId' )
TemplateIdType = TacLazyType( 'FlowTracking::TemplateId' )

traceHandle = Tracing.Handle( 'SftCliShow' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

activeAgentDir = None
arpVrfIdMap = None
entityManager = None
ethIntfStatusDir = None
ipfixStats = None
sftCounters = None
sftHoCounters = None
hoStatus = None
hosStateCounter = None
sftConfig = {}
sftConfig[ ftrTypeSampled ] = None
sftConfig[ ftrTypeInbandTelemetry ] = None
shmemEntityManager = None

FgCount = namedtuple( 'FgCount', [ 'flows', 'expiredFlows', 'packets' ] )

# SHOW COMMANDS
#------------------------------------------------------------
# show flow tracking sampled flow-table [detail]
# [ tracker <tracker-name> ] [ group <group-name> ]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ interface <interface-range> ]
#------------------------------------------------------------
class ShowFlowTable( object ):

   def showFlowDetail( self, flowDetailModel, ipGroup, entry, bgp, ifIndexMap ):
      flowDetailModel.srcEthAddr = entry.srcEthAddr
      flowDetailModel.dstEthAddr = entry.dstEthAddr
      flowDetailModel.tcpFlags = tcpFlagStr( entry.tcpFlags )
      flowDetailModel.lastPktTime = entry.lastPktTime
      flowDetailModel.tos = entry.tos
      flowDetailModel.ingressIntf = ifindexToIntf( ifIndexMap, entry.inIfIndex )
      flowDetailModel.egressVlanId = entry.egressVlanId
      flowDetailModel.egressIntf = ifIndexMap.get( entry.outIfIndex, 'unknown' )
      flowDetailModel.vni = vniOrUnknown( entry.vni )
      flowDetailModel.tunnelState = entry.tunnelState
      if bgp:
         flowDetailModel.srcAs = bgp.srcAs
         flowDetailModel.dstAs = bgp.dstAs
         flowDetailModel.nextHopIp = IpGenAddr( ipStr( bgp.nextHopIp ) )
         flowDetailModel.bgpNextHopIp = IpGenAddr( ipStr( bgp.bgpNextHopIp ) )
         flowDetailModel.srcPrefixLen = bgp.srcPrefixLen
         flowDetailModel.dstPrefixLen = bgp.dstPrefixLen
      else:
         flowDetailModel.srcAs = 0
         flowDetailModel.dstAs = 0
         flowDetailModel.nextHopIp = IpGenAddr( "::" )
         flowDetailModel.bgpNextHopIp = IpGenAddr( "::" )
         flowDetailModel.srcPrefixLen = 0
         flowDetailModel.dstPrefixLen = 0

      if toggleHwOffloadIpv4Enabled():
         if ipGroup:
            flowDetailModel.sampledPktsReceived = entry.pkts
            flowDetailModel.sampledBytesReceived = entry.bytes
            flowDetailModel.hwPktsReceived = entry.hwPkts
            flowDetailModel.hwBytesReceived = entry.hwBytes
      if not ipGroup:
         flowDetailModel.flowLabel = entry.flowLabel

   def isEntryMatchingFilter( self, key, entry, ifIndexMap=None, srcIpAddr=None,
         dstIpAddr=None, srcPort=None, dstPort=None, vrfName=None, vlanId=None,
         protocol=None, intfs=None ):

      if srcIpAddr is not None and srcIpAddr != key.srcAddr:
         return False

      if dstIpAddr is not None and dstIpAddr != key.dstAddr:
         return False

      if srcPort is not None and srcPort != key.srcPort:
         return False

      if dstPort is not None and dstPort != key.dstPort:
         return False

      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      if vrfName is not None and vrfName != vrfNameStr and vrfName != ALL_VRF_NAME:
         return False

      if ( protocol is not None and
            ( protocol.lower() != protocolStr( key.ipProtocol,
               key.ipProtocolNumber ).lower() ) and
            protocol != str( key.ipProtocolNumber ) ):
         return False

      if vlanId is not None and vlanId != key.vlanId:
         return False

      if intfs is not None and \
            ifindexToIntf( ifIndexMap, entry.inIfIndex ).stringValue \
            not in list( intfs.intfNames() ):
         return False

      return True

   def populateFlowModel( self, ipGroup, key, entry ):
      flowModel = SftModel.FlowModel()
      flowModel.bytesReceived = entry.bytes
      flowModel.pktsReceived = entry.pkts
      if toggleHwOffloadIpv4Enabled():
         if ipGroup:
            flowModel.pktsReceived += entry.hwPkts
            flowModel.bytesReceived += entry.hwBytes
      flowModel.startTime = entry.startTime

      flowKeyModel = SftModel.FlowKeyModel()
      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      flowKeyModel.vrfName = vrfNameStr
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowModel.key = flowKeyModel

      return flowModel

   def showFlowEntry( self, flowTable, ipGroup, key, entry, bgp=None,
         ifIndexMap=None, displayType=None ):

      flowModel = self.populateFlowModel( ipGroup, key, entry )

      if displayType == "detail":
         flowDetailModel = SftModel.FlowDetailModel()
         self.showFlowDetail( flowDetailModel, ipGroup, entry, bgp, ifIndexMap )
         flowModel.flowDetail = flowDetailModel

      return flowModel

   def showFlowTableEntries( self, grName, groupModel, flowTable, ifIndexMap,
         displayType, srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId,
         protocol, intfs ):
      flowEntry = {}
      bgpEntry = {}
      ipGroup = True
      if grName == IP_GROUP:
         flowEntry = flowTable.ipFlowEntry
         bgpEntry = flowTable.bgpFlowEntry
      else:
         ipGroup = False
         flowEntry = flowTable.ip6FlowEntry
         bgpEntry = flowTable.bgp6FlowEntry
      bgp = None
      for key, entry in flowEntry.items():
         if not self.isEntryMatchingFilter( key, entry, ifIndexMap, srcIpAddr,
               dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol, intfs ):
            continue
         if displayType == "detail":
            bgp = bgpEntry.get( key )
         flowModel = self.showFlowEntry( flowTable, ipGroup, key, entry, bgp,
               ifIndexMap, displayType=displayType )
         groupModel.flows.append( flowModel )

   def fetchTrackingModel( self ):
      trackingModel = SftModel.TrackingModel()
      trackingModel.running = isSftAgentRunning( entityManager, activeAgentDir )
      trackingModel.ftrType = ftrTypeSampled
      return trackingModel

   def getFlowTable( self, sMount, trackerName ):
      return getFlowTable( sMount, trackerName )

   def showFlowTable( self, mode, args ):
      ftrType = getFtrTypeFromArgs( args )
      srcIpAddr = None
      dstIpAddr = None
      intfs = None
      groupName = args.get( 'GROUP_NAME' )
      trackerName = args.get( 'TRACKER_NAME' )
      displayType = args.get( 'detail' )
      if 'SRC_IPV4' in args:
         srcIpAddr = args.get( 'SRC_IPV4' )
      else:
         srcIpAddr = args.get( 'SRC_IPV6' )
      if 'DST_IPV4' in args:
         dstIpAddr = args.get( 'DST_IPV4' )
      else:
         dstIpAddr = args.get( 'DST_IPV6' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      intfs = args.get( 'INTERFACE_RANGE' )

      ifIndexMap = getIfIndexMap( ethIntfStatusDir )

      trackingModel = self.fetchTrackingModel()
      if not trackingModel.running:
         return trackingModel

      for trName in sftConfig[ ftrType ].hwFtConfig:
         if trackerName and trName != trackerName:
            continue

         flowTable = self.getFlowTable( shmemEntityManager, trName )
         if not flowTable:
            continue

         trackerModel = SftModel.TrackerModel()
         trackingModel.trackers[ trName ] = trackerModel
         trackerModel.numFlows = 0

         for grName in getFlowGroups( ftrType, trName ):
            if groupName and grName != groupName:
               continue
            groupModel = SftModel.GroupModel()
            trackerModel.groups[ grName ] = groupModel
            self.showFlowTableEntries( grName, groupModel, flowTable, ifIndexMap,
                  displayType, srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName,
                  vlanId, protocol, intfs )
            trackerModel.numFlows += len( groupModel.flows )
            if groupName:
               break
         if trackerName:
            break
      return trackingModel

def showFlowTable( mode, args ):
   t = ShowFlowTable()
   return t.showFlowTable( mode, args )

tokenDetail = CliCommand.Node(
                        CliMatcher.KeywordMatcher( 'detail',
                                                       "Detailed flow information" ),
                         storeSharedResult=True, maxMatches=1 )

def generateShowTrackingFilter():
   _expression = '[ { ( ( tracker TRACKER_NAME ) | ( group GROUP_NAME ) | ' + \
         '( src-ip ( SRC_IPV4 | SRC_IPV6 ) ) | ( vlan VLAN ) | ' + \
         '( dst-ip ( DST_IPV4 | DST_IPV6 ) ) | ( vrf VRF ) | ' + \
         '( src-port SRC_PORT ) | ( dst-port DST_PORT ) | ( detail ) | ' + \
         '( protocol PROTOCOL ) | ( interface INTERFACE_RANGE ) ) } ] '
   _data = {
      'detail' : tokenDetail,
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'group' : CliCommand.Node( groupKw, maxMatches=1 ),
      'GROUP_NAME' : allGroupNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'SRC_IPV6' : tokenIPv6,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'DST_IPV6' : tokenIPv6,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'interface' : tokenInterface,
      'INTERFACE_RANGE' : intfRangeRule
   }

   class ShowTrackingFilter( CliCommand.CliExpression ):
      expression = _expression
      data = _data

   return ShowTrackingFilter

class ShowFlowTrackingFlowTable( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled flow-table [ FILTER ] '
   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'flow-table' : flowTableKw,
      'FILTER' : generateShowTrackingFilter()
   }

   handler = showFlowTable
   cliModel = SftModel.TrackingModel

addShowCommandClass( ShowFlowTrackingFlowTable )

#------------------------------------------------------------
# show flow tracking sampled flow-table hardware [detail]
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ tracker <tracker-name> ] [ interface <interface-range> ]
#------------------------------------------------------------
class HoFlowTable( object ):

   def isEntryMatchingFilter( self, key, entry, srcIpAddr=None, dstIpAddr=None,
         srcPort=None, dstPort=None, vrfName=None, vlanId=None,
         protocol=None, state=None, intfs=None ):

      if srcIpAddr is not None and srcIpAddr != key.srcAddr:
         return False

      if dstIpAddr is not None and dstIpAddr != key.dstAddr:
         return False

      if srcPort is not None and srcPort != key.srcPort:
         return False

      if dstPort is not None and dstPort != key.dstPort:
         return False

      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      if vrfName is not None and vrfName != vrfNameStr and vrfName != ALL_VRF_NAME:
         return False

      if ( protocol is not None and
            ( protocol.lower() != protocolStr( key.ipProtocol,
               key.ipProtocolNumber ).lower() ) and
            protocol != str( key.ipProtocolNumber ) ):
         return False

      if vlanId is not None and vlanId != key.vlanId:
         return False

      if state is not None and state != entry.hoFlowState:
         return False

      if intfs is not None and key.intfId not in list( intfs.intfNames() ):
         return False

      return True

   def hoFlowEntry( self, key, entry ):
      flowEntryModel = SftModel.HoFlowEntryModel()
      flowEntryModel.ingressIntf = key.intfId
      flowEntryModel.state = entry.hoFlowState

      flowKeyModel = SftModel.FlowKeyModel()
      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      flowKeyModel.vrfName = vrfNameStr
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowEntryModel.key = flowKeyModel
      flowEntryModel.hwCreateTime = entry.createTime

      return flowEntryModel

   def getHoFlowConfig( self, shmemEm ):
      smashHoFlowConfig = shmemEm.doMount( HoFlowConfig.mountPath( ftrTypeSampled ),
                                           'FlowTracking::HoFlowConfig',
                                           Smash.mountInfo( 'reader' ) )
      return smashHoFlowConfig

   def getHoFlowStatus( self, shmemEm ):
      smashHoFlowStatus = shmemEm.doMount( HoFlowStatus.mountPath( ftrTypeSampled ),
                                           'FlowTracking::HoFlowStatus',
                                           Smash.mountInfo( 'reader' ) )
      return smashHoFlowStatus

   def hoFlowDetail( self, key, config, status ):
      hoFlowDetailModel = SftModel.HoFlowDetailModel()
      hoFlowDetailModel.hwUpdateTime = status.updateTime
      if config:
         hoFlowDetailModel.swCreateTime = config.createTime
         hoFlowDetailModel.srcEthAddr = config.srcEthAddr
         hoFlowDetailModel.dstEthAddr = config.dstEthAddr

      return hoFlowDetailModel

   def hoFlowEntries( self, hoFlowStatus, hoFlowTableModel, detail,
         srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol, state,
         intfs, trackerName ):
      hoFlowConfig = None
      if detail:
         hoFlowConfig = self.getHoFlowConfig( shmemEntityManager )

      for key, entry in hoFlowStatus.hoIpFlowStatus.items():
         if not self.isEntryMatchingFilter( key, entry, srcIpAddr, dstIpAddr,
               srcPort, dstPort, vrfName, vlanId, protocol, state, intfs ):
            continue
         tracker = 'Unknown'
         intfId = key.intfId
         if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfId ):
            intfId = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intfId )
         if intfId in sftConfig[ ftrTypeSampled ].hwIntfFtConfig:
            intfFtConfig = sftConfig[ ftrTypeSampled ].hwIntfFtConfig[ intfId ]
            tracker = intfFtConfig.hwFtConfig.name
         if trackerName is not None and tracker != trackerName:
            continue

         trackerModel = None
         groupModel = None
         groupName = 'Unknown'
         if tracker != 'Unknown':
            ipv4FgId = \
                  FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateIpv4 )
            for fgName, fgId in getFlowGroupsAndIds( ftrTypeSampled, tracker ):
               if fgId == ipv4FgId:
                  groupName = fgName
                  break
         if tracker not in hoFlowTableModel.trackers:
            trackerModel = SftModel.HoTrackerModel()
            hoFlowTableModel.trackers[ tracker ] = trackerModel
            trackerModel.numFlows = 0
            groupModel = SftModel.HoGroupModel()
            trackerModel.groups[ groupName ] = groupModel
         else:
            trackerModel = hoFlowTableModel.trackers[ tracker ]
            groupModel = trackerModel.groups[ groupName ]

         flowModel = self.hoFlowEntry( key, entry )
         if detail:
            hoFlowDetailModel = SftModel.HoFlowDetailModel()
            config = None
            if key in hoFlowConfig.hoIpFlowConfig:
               config = hoFlowConfig.hoIpFlowConfig[ key ]
            hoFlowDetailModel = self.hoFlowDetail( key, config, entry )
            flowModel.flowDetail = hoFlowDetailModel
         groupModel.flows.append( flowModel )
         trackerModel.numFlows += 1

   def hoFlowTable( self, mode, args ):
      srcIpAddr = None
      dstIpAddr = None
      intfs = None
      detail = 'detail' in args
      srcIpAddr = args.get( 'SRC_IPV4' )
      dstIpAddr = args.get( 'DST_IPV4' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      trackerName = args.get( 'TRACKER_NAME' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      state = args.get( 'STATE' )
      intfs = args.get( 'INTERFACE_RANGE' )

      hoFlowTableModel = SftModel.HoFlowTableModel()
      hoFlowTableModel.running = isSftAgentRunning( entityManager, activeAgentDir )
      hoFlowTableModel.ftrType = ftrTypeSampled
      if not hoFlowTableModel.running:
         return hoFlowTableModel

      hoFlowStatus = self.getHoFlowStatus( shmemEntityManager )
      if not hoFlowStatus:
         return hoFlowTableModel

      self.hoFlowEntries( hoFlowStatus, hoFlowTableModel, detail,
            srcIpAddr, dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol,
            state, intfs, trackerName )
      return hoFlowTableModel

def doShowHoFlowTable( mode, args ):
   t = HoFlowTable()
   return t.hoFlowTable( mode, args )

class HoTableFilter( CliCommand.CliExpression ):
   expression = ( '''[ { ( tracker TRACKER_NAME )
                       | ( vlan VLAN )
                       | ( vrf VRF )
                       | ( src-ip SRC_IPV4 )
                       | ( dst-ip DST_IPV4 )
                       | ( src-port SRC_PORT )
                       | ( dst-port DST_PORT )
                       | ( protocol PROTOCOL )
                       | ( state STATE )
                       | ( interface INTERFACE_RANGE )
                       | ( detail ) } ]''' )
   data = {
      'detail' : tokenDetail,
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'state' : tokenState,
      'STATE' : tokenStateValue,
      'interface' : tokenInterface,
      'INTERFACE_RANGE' : intfRangeRule
   }

if toggleHwOffloadIpv4Enabled():
   class ShowFlowTrackingHoFlowTable( ShowCommand.ShowCliCommandClass ):
      syntax = 'show flow tracking sampled flow-table hardware [ FILTER ] '
      data = {
         'flow' : flowMatcherForShow,
         'tracking' : trackingShowKw,
         'sampled' : sampledShowKw,
         'flow-table' : flowTableKw,
         'hardware' : sampledHardwareKw,
         'FILTER' : HoTableFilter
      }

      handler = doShowHoFlowTable
      cliModel = SftModel.HoFlowTableModel

   addShowCommandClass( ShowFlowTrackingHoFlowTable )

#------------------------------------------------------------
# show flow tracking sampled flow-table hardware counters
# [ src-ip <ip> ] [ dst-ip <ip> ] [ src-port <port> ] [ dst-port <port> ]
# [ protocol <protocol> ] [ vrf <vrf> ] [ vlan <vlan> ]
# [ tracker <tracker-name> ] [ interface <interface-range> ]
#------------------------------------------------------------
class HoFlowCounters( object ):

   def isEntryMatchingFilter( self, key, entry, srcIpAddr=None, dstIpAddr=None,
                              srcPort=None, dstPort=None, vrfName=None, vlanId=None,
                              protocol=None, intfs=None ):

      if srcIpAddr is not None and srcIpAddr != key.srcAddr:
         return False

      if dstIpAddr is not None and dstIpAddr != key.dstAddr:
         return False

      if srcPort is not None and srcPort != key.srcPort:
         return False

      if dstPort is not None and dstPort != key.dstPort:
         return False

      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      if vrfName is not None and vrfName != vrfNameStr and vrfName != ALL_VRF_NAME:
         return False

      if ( protocol is not None and
            ( protocol.lower() != protocolStr( key.ipProtocol,
               key.ipProtocolNumber ).lower() ) and
            protocol != str( key.ipProtocolNumber ) ):
         return False

      if vlanId is not None and vlanId != key.vlanId:
         return False

      if intfs is not None and entry.intfId not in list( intfs.intfNames() ):
         return False

      return True

   def hoFlowCounterEntry( self, key, entry ):
      flowCounterEntryModel = SftModel.HoFlowCounterEntryModel()
      flowCounterEntryModel.ingressIntf = entry.intfId

      flowKeyModel = SftModel.FlowKeyModel()
      vrfNameStr = "unknown"
      if key.vrfId in arpVrfIdMap.vrfIdToName:
         vrfNameStr = arpVrfIdMap.vrfIdToName[ key.vrfId ].vrfName
      flowKeyModel.vrfName = vrfNameStr
      flowKeyModel.vlanId = key.vlanId
      flowKeyModel.srcAddr = IpGenAddr( ipStr( key.srcAddr ) )
      flowKeyModel.dstAddr = IpGenAddr( ipStr( key.dstAddr ) )
      flowKeyModel.ipProtocolNumber = key.ipProtocolNumber
      try:
         flowKeyModel.ipProtocol = key.ipProtocol
      except ValueError:
         # unassigned protocol number will not have ipProtocol Enum
         pass
      flowKeyModel.srcPort = key.srcPort
      flowKeyModel.dstPort = key.dstPort
      flowCounterEntryModel.key = flowKeyModel
      flowCounterEntryModel.pktsReceived = entry.packets
      flowCounterEntryModel.bytesReceived = entry.bytes
      flowCounterEntryModel.createTime = entry.createTime
      flowCounterEntryModel.updateTime = entry.updateTime

      return flowCounterEntryModel

   def getHoFlowCounters( self, shmemEm ):
      smashHoFlowCounter = \
            shmemEm.doMount( HoFlowCounter.mountPath( ftrTypeSampled, "hfc" ),
                                            'FlowTracking::HoFlowCounter',
                                            Smash.mountInfo( 'reader' ) )
      return smashHoFlowCounter

   def hoFlowEntries( self, hoFlowCounters, hoFlowCountersModel, srcIpAddr,
                      dstIpAddr, srcPort, dstPort, vrfName, vlanId, protocol,
                      intfs, trackerName ):
      for key, entry in hoFlowCounters.hoIpFlowCounter.items():
         if not self.isEntryMatchingFilter( key, entry, srcIpAddr, dstIpAddr,
                           srcPort, dstPort, vrfName, vlanId, protocol, intfs ):
            continue
         tracker = 'Unknown'
         intfId = entry.intfId
         if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfId ):
            intfId = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intfId )
         if intfId in sftConfig[ ftrTypeSampled ].hwIntfFtConfig:
            intfFtConfig = sftConfig[ ftrTypeSampled ].hwIntfFtConfig[ intfId ]
            tracker = intfFtConfig.hwFtConfig.name
         if trackerName is not None and tracker != trackerName:
            continue

         trackerModel = None
         groupModel = None
         groupName = 'Unknown'
         if tracker != 'Unknown':
            ipv4FgId = \
                  FtConstants.reservedTemplateId( TemplateIdEnum.dataTemplateIpv4 )
            for fgName, fgId in getFlowGroupsAndIds( ftrTypeSampled, tracker ):
               if fgId == ipv4FgId:
                  groupName = fgName
                  break
         if tracker not in hoFlowCountersModel.trackers:
            trackerModel = SftModel.HoTrackerCounterModel()
            hoFlowCountersModel.trackers[ tracker ] = trackerModel
            trackerModel.numFlows = 0
            groupModel = SftModel.HoGroupCounterModel()
            trackerModel.groups[ groupName ] = groupModel
         else:
            trackerModel = hoFlowCountersModel.trackers[ tracker ]
            groupModel = trackerModel.groups[ groupName ]

         flowCounterModel = self.hoFlowCounterEntry( key, entry )
         groupModel.flows.append( flowCounterModel )
         trackerModel.numFlows += 1

   def hoFlowCounters( self, mode, args ):
      srcIpAddr = None
      dstIpAddr = None
      intfs = None
      srcIpAddr = args.get( 'SRC_IPV4' )
      dstIpAddr = args.get( 'DST_IPV4' )
      srcPort = args.get( 'SRC_PORT' )
      dstPort = args.get( 'DST_PORT' )
      protocol = args.get( 'PROTOCOL' )
      trackerName = args.get( 'TRACKER_NAME' )
      vrfName = args.get( 'VRF' )
      vlanId = args.get( 'VLAN' )
      intfs = args.get( 'INTERFACE_RANGE' )

      hoFlowCountersModel = SftModel.HoFlowCountersModel()
      hoFlowCountersModel.running = isSftAgentRunning( entityManager,
                                                       activeAgentDir )
      hoFlowCountersModel.ftrType = ftrTypeSampled
      if not hoFlowCountersModel.running:
         return hoFlowCountersModel

      hoFlowCounters = self.getHoFlowCounters( shmemEntityManager )
      if not hoFlowCounters:
         return hoFlowCountersModel

      self.hoFlowEntries( hoFlowCounters, hoFlowCountersModel, srcIpAddr, dstIpAddr,
                          srcPort, dstPort, vrfName, vlanId, protocol, intfs,
                          trackerName )
      return hoFlowCountersModel

def doShowHoFlowCounters( mode, args ):
   t = HoFlowCounters()
   return t.hoFlowCounters( mode, args )

class HoCountersFilter( CliCommand.CliExpression ):
   expression = ( '''[ { ( tracker TRACKER_NAME )
                       | ( vlan VLAN )
                       | ( vrf VRF )
                       | ( src-ip SRC_IPV4 )
                       | ( dst-ip DST_IPV4 )
                       | ( src-port SRC_PORT )
                       | ( dst-port DST_PORT )
                       | ( protocol PROTOCOL )
                       | ( interface INTERFACE_RANGE ) } ]''' )
   data = {
      'tracker' : CliCommand.Node( trackerKw, maxMatches=1 ),
      'TRACKER_NAME' : trackerNameMatcher,
      'src-ip' : tokenSrcIp,
      'SRC_IPV4' : tokenIPv4,
      'dst-ip' : tokenDstIp,
      'DST_IPV4' : tokenIPv4,
      'src-port' : tokenSrcPort,
      'SRC_PORT' : tokenPort,
      'dst-port' : tokenDstPort,
      'DST_PORT' : tokenPort,
      'protocol' : tokenProtocol,
      'PROTOCOL' : tokenProtocolValue,
      'vrf' : tokenVrf,
      'VRF' : tokenVrfValue,
      'vlan' : tokenVlan,
      'VLAN' : tokenVlanValue,
      'interface' : tokenInterface,
      'INTERFACE_RANGE' : intfRangeRule
   }

if toggleHwOffloadIpv4Enabled():
   class ShowFlowTrackingHoFlowCounters( ShowCommand.ShowCliCommandClass ):
      syntax = 'show flow tracking sampled flow-table hardware counters [ FILTER ] '
      data = {
         'flow' : flowMatcherForShow,
         'tracking' : trackingShowKw,
         'sampled' : sampledShowKw,
         'flow-table' : flowTableKw,
         'hardware' : sampledHardwareKw,
         'counters' : 'Show flow tracking sampled hardware counters',
         'FILTER' : HoCountersFilter
      }

      handler = doShowHoFlowCounters
      cliModel = SftModel.HoFlowCountersModel

   addShowCommandClass( ShowFlowTrackingHoFlowCounters )

#--------------------------
class ShowCounters( object ):

   def addExporters( self, ftrType, trModel, trName, expFilter ):
      ftConfig = sftConfig[ ftrType ]
      addExporters( trModel, trName, expFilter, ftConfig, ipfixStats,
                    ftConfig.swExport, counterModel=SftModel )

   if toggleHwOffloadIpv4Enabled():
      def getHoModel( self, model ):
         if not model.hardwareOffload:
            hoModel = SftModel.HardwareOffloadCounters()
            hoModel.flowsCreated = 0
            hoModel.flowsDeleted = 0
            hoModel.pktsReceived = 0
            hoModel.pktsHardwareFailed = 0
            hoModel.pktsHardwareMiss = 0
            hoModel.pktsDiscarded = 0
            model.hardwareOffload = hoModel
         return model.hardwareOffload

      def populateHoCounters( self, hoModel, hoCounts ):
         hoModel.flowsCreated += hoCounts.hoFlowsCreated
         hoModel.flowsDeleted += hoCounts.hoFlowsDeleted
         hoModel.pktsReceived += hoCounts.hoPackets
         hoModel.pktsDiscarded += hoCounts.hoFlowTrackerDisabled + \
                                    hoCounts.hoInvalidIfIndex + \
                                    hoCounts.hoFlowConfigMiss
         hoModel.pktsHardwareMiss += hoCounts.hoFlowOffloadMiss
         hoModel.pktsHardwareFailed += hoCounts.hoFlowConfigLimit

      def addFgHoCounters( self, fgModel, ftId ):
         if sftHoCounters:
            ipv4FgId = FtConsts.reservedTemplateId( 'dataTemplateIpv4' )
            counterKey = FlowGroupCounterKey( ftId, ipv4FgId )
            hoCounts = sftHoCounters.hoCounter.get( counterKey )
            if hoCounts:
               hoModel = self.getHoModel( fgModel )
               self.populateHoCounters( hoModel, hoCounts )
               t1( 'addFgHoCounters', hoCounts )

      def addHoCounters( self, ftrType, model ):
         if sftHoCounters:
            for _, ftId in getTrackerNamesAndIds( model.ftrType ):
               self.addFgHoCounters( model, ftId )
            self.addFgHoCounters( model, 0 )

         if hosStateCounter:
            hoModel = self.getHoModel( model )
            flowsActive = 0
            flowsPending = 0
            inHardware = HosStateCounterKey( HoFlowStateEnum.inHardware )
            for key, flow in hosStateCounter.hosStateIpCounter.items():
               if key == inHardware:
                  flowsActive += flow.hoFlows
               else:
                  flowsPending += flow.hoFlows
               t1( 'hosStateIpCounter', key, flow )
            hoModel.flowsActive = flowsActive
            hoModel.flowsPending = flowsPending

   def addFlowGroups( self, ftrType, ftrCounters, trModel, trName,
                      ftId, v4Flows, v6Flows ):
      anyGroups = False
      flows = 0
      expiredFlows = 0
      packets = 0
      for fgName, fgId in getFlowGroupsAndIds( ftrType, trName ):
         anyGroups = True
         t1( 'process flow group', fgName )
         fgModel = SftModel.FlowGroupCounters()
         if fgName == IP_GROUP:
            fgModel.activeFlows = v4Flows
            if toggleHwOffloadIpv4Enabled():
               self.addFgHoCounters( fgModel, ftId )
         else:
            assert fgName == IP6_GROUP
            fgModel.activeFlows = v6Flows

         counterKey = FlowGroupCounterKey( ftId, fgId )
         fgCounts = ftrCounters.flowGroupCounter.get( counterKey )
         if not fgCounts:
            t0( 'No counters for', counterKey.smashString() )
            fgCounts = FlowGroupCounterEntry()
         flows += fgCounts.flowEntry.flows
         fgModel.flows = fgCounts.flowEntry.flows
         expiredFlows += fgCounts.flowEntry.expiredFlows
         fgModel.expiredFlows = fgCounts.flowEntry.expiredFlows
         packets += fgCounts.flowEntry.packets
         fgModel.packets = fgCounts.flowEntry.packets

         trModel.flowGroups[ fgName ] = fgModel

      clearTimeKey = FlowGroupCounterKey( ftId, TemplateIdType.maxTemplateId )
      clearTime = ftrCounters.flowGroupCounter.get( clearTimeKey )
      if clearTime and clearTime.key == clearTimeKey:
         # clearTime should be most recent lastClearedTime from either
         # sftCounters or ipfixStats
         lastClearedTime = switchTimeToUtc( clearTime.flowEntry.lastClearedTime )
         if lastClearedTime > trModel.clearTime:
            trModel.clearTime = lastClearedTime

      if not anyGroups:
         t0( 'WARNING: no flow groups for tracker', trName )

      return FgCount( flows=flows, expiredFlows=expiredFlows, packets=packets )

   def addTrackers( self, model, ftrCounters, trFilter, expFilter ):
      allTrackerFlows = 0
      allActiveFlows = 0
      for trName, ftId in getTrackerNamesAndIds( model.ftrType ):
         if trFilter and trFilter != trName:
            t1( 'tracker', trName, 'did not match filter' )
            continue
         t1( 'process tracker', trName )
         flowTable = self.getFlowTable( shmemEntityManager, trName )
         if not flowTable:
            continue
         trModel = SftModel.TrackerCounters()
         v4Flows = len( flowTable.ipFlowEntry )
         v6Flows = len( flowTable.ip6FlowEntry )
         trModel.activeFlows = v4Flows + v6Flows
         clearTimeKey = CollectorStatisticsKey( trName, "", CollectorInfo() )
         clearTime = ipfixStats.stats.get( clearTimeKey )
         if clearTime and clearTime.key == clearTimeKey:
            trModel.clearTime = switchTimeToUtc(
               clearTime.lastClearedTime )
         counts = self.addFlowGroups( model.ftrType, ftrCounters, trModel,
                                      trName, ftId, v4Flows, v6Flows )
         trModel.flows = counts.flows
         trModel.expiredFlows = counts.expiredFlows
         allTrackerFlows += counts.flows
         allActiveFlows += trModel.activeFlows
         trModel.packets = counts.packets
         self.addExporters( model.ftrType, trModel, trName, expFilter )
         model.trackers[ trName ] = trModel
      ftrKey = FlowCounterKey( 0 )
      ftrCounts = ftrCounters.flowsCounter.get( ftrKey )
      if not ftrCounts:
         ftrCounts = FlowCounterEntry()
      if ftrCounts.flowEntry.lastClearedTime != 0:
         lastClearedTime = switchTimeToUtc(
            ftrCounts.flowEntry.lastClearedTime )
         model.clearTime = lastClearedTime
      model.activeFlows = allActiveFlows
      model.flows = ftrCounts.flowEntry.flows
      model.expiredFlows = ftrCounts.flowEntry.expiredFlows
      model.packets = ftrCounts.flowEntry.packets

   def getFlowTable( self, sMount, trackerName ):
      return getFlowTable( sMount, trackerName )

   def showCounters( self, mode, args ):
      model = SftModel.FtrCounters()
      model.ftrType = ftrTypeSampled
      model.running = isSftAgentRunning( entityManager, activeAgentDir )
      if model.running:
         trFilter = args.get( 'TRACKER_NAME' )
         expFilter = args.get( 'EXPORTER_NAME' )
         self.addTrackers( model, sftCounters, trFilter, expFilter )
         if toggleHwOffloadIpv4Enabled():
            self.addHoCounters( model.ftrType, model )
      return model

def showSftFlowCounters( mode, args ):
   t = ShowCounters()
   return t.showCounters( mode, args )

class ShowSampledCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show flow tracking sampled counters ' + \
            '[ tracker TRACKER_NAME [ exporter EXPORTER_NAME ] ]'

   data = {
      'flow' : flowMatcherForShow,
      'tracking' : trackingShowKw,
      'sampled' : sampledShowKw,
      'counters' : 'Show flow tracking sampled counters',
      'tracker' : trackerKw,
      'TRACKER_NAME' : trackerNameMatcher,
      'exporter' : exporterKw,
      'EXPORTER_NAME' : exporterNameMatcher,
   }

   handler = showSftFlowCounters
   cliModel = SftModel.FtrCounters

addShowCommandClass( ShowSampledCounters )

#--------------------------
def Plugin( em ):
   global ipfixStats
   global sftConfig
   global sftCounters
   global activeAgentDir
   global arpVrfIdMap
   global ethIntfStatusDir
   global entityManager
   global shmemEntityManager

   entityManager = em
   shmemEntityManager = SharedMem.entityManager( sysdbEm=em )

   ipfixStats = SmashLazyMount.mount( em, 'flowtracking/sampled/ipfix/statistics',
                                      'Smash::Ipfix::CollectorStatistics',
                                      SmashLazyMount.mountInfo( 'reader' ) )
   sftConfig[ ftrTypeSampled ] = LazyMount.mount( em,
                                'hardware/flowtracking/config/sampled',
                                'HwFlowTracking::Config', 'r' )
   sftConfig[ ftrTypeInbandTelemetry ] = LazyMount.mount( em,
                                'hardware/flowtracking/config/inbandTelemetry',
                                'HwFlowTracking::Config', 'r' )
   sftCounters = SmashLazyMount.mount( em,
                                       'flowtracking/sampled/counters',
                                       'Smash::FlowTracker::FtCounters',
                                       SmashLazyMount.mountInfo( 'reader' ) )
   if toggleHwOffloadIpv4Enabled():
      global sftHoCounters
      global hoStatus
      global hosStateCounter
      hoStatus = SmashLazyMount.mount( em,
                                       HoFlowStatus.mountPath( ftrTypeSampled ),
                                       'FlowTracking::HoFlowStatus',
                                       SmashLazyMount.mountInfo( 'reader' ) )
      hosStateCounter = SmashLazyMount.mount( em,
                                       HosStateCounter.mountPath( ftrTypeSampled ),
                                       'FlowTracking::HosStateCounter',
                                       SmashLazyMount.mountInfo( 'reader' ) )
      sftHoCounters = SmashLazyMount.mount( em,
                                       'flowtracking/sampled/hoCounters',
                                       'Smash::SampledFlowTracker::HoCounters',
                                       SmashLazyMount.mountInfo( 'reader' ) )
   activeAgentDir = LazyMount.mount( em, 'flowtracking/activeAgent',
                                     'Tac::Dir', 'ri' )
   arpVrfIdMap = SmashLazyMount.mount( em, "vrf/vrfIdMapStatus",
                                       "Vrf::VrfIdMap::Status",
                                       SmashLazyMount.mountInfo( 'reader' ) )
   ethIntfStatusDir = LazyMount.mount( em, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )
