#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from IntfModel import Interface
import SflowConst
import Tac
from Toggles.SflowLibToggleLib import toggleEgressSflowEnabled

SampleTruncateSizeType = Tac.Type( "Sflow::SampleTruncateSizeType" )

_sendingReasonMap = {
   'noError' : '',
   'notRunning' : ' (Sflow not running)',
   'invalidSource' : ' (Invalid source IP address)',
   'invalidDestination' : ' (Invalid destination IP address)',
   'invalidSourceDestination' : ' (Invalid combination of IPv4 '
   'and IPv6 addresses)',
   'invalidVrf' : ' (Invalid VRF)',
   'bindingInProgress' : ' (Binding to source IP in progress)',
}

def _convertToYesNo( booleanToConvert ):
   return 'Yes' if booleanToConvert else 'No'

class SflowStatus( Model ):
   class Details( Model ):
      hardwareSamplingEnabled = Bool( help='Hardware sampling enabled' )
      samplesDiscarded = Int( help='Number of discarded samples' )
      sampleOutputInterface = Bool( help='Include output interface in '
                                    'packet sample' )
      sampleMplsExtension = Bool( help='Include MPLS extension header in '
                                  'packet sample', optional=True )
      sampleSwitchExtension = Bool( help='Include switch extension header in '
                                    'packet sample', optional=True )
      sampleRouterExtension = Bool( help='Include router extension header in '
                                    'packet sample', optional=True )
      sampleTunnelIpv4EgrExtension = Bool( help='Include tunnel IPv4 egress '
                                           'extension header in packet sample',
                                           optional=True )
      sampleIngressSubintf = Bool( help='Include input subinterface information '
                                       'in packet sample; sample sent '
                                       'in expanded format', optional=True )
      sampleEgressSubintf = Bool( help='Include output subinterface information '
                                       'in packet sample; sample sent '
                                       'in expanded format', optional=True )
      portChannelOutputIfIndex = Enum( help='Output interface index for '
                                       'port channels',
                                       values=( 'member', 'portchannel' ),
                                       optional=True )
      accelUnsupportedExtensions = List( help='Extensions that are not supported for'
                                         ' hardware acceleration',
                                         valueType=str, optional=True )
      sampleEncodingFormat = Enum( help='Encoding format for samples',
                                   values=( 'compact', 'expanded' ),
                                   optional=True )

   class Destination( Model ):
      port = Int( help='Port of sFlow collector' )
      hostname = Str( help='Hostname of sFlow collector' )

   class Ipv4Destination( Destination ):
      ipv4Address = Ip4Address( help='IPv4 address of sFlow collector' )
      vrfName = Str( help='Associated VRF for sFlow collector' )

      def render( self ):
         if self.hostname == str( self.ipv4Address ):
            print '  %s:%s (VRF: %s)' % ( self.ipv4Address, self.port,
                                          self.vrfName )
         else:
            print '  %s (%s):%s (VRF: %s)' % ( self.hostname, self.ipv4Address,
                                               self.port, self.vrfName )

   class Ipv6Destination( Destination ):
      ipv6Address = Ip6Address( help='IPv6 address of sFlow collector' )
      vrfName = Str( help='Associated VRF for sFlow collector' )

      def render( self ):
         if self.hostname == str( self.ipv6Address ):
            print '  [%s]:%s (VRF: %s)' % ( self.ipv6Address, self.port,
                                            self.vrfName )
         else:
            print '  %s (%s):%s (VRF: %s)' % ( self.hostname, self.ipv6Address,
                                               self.port, self.vrfName )

   class Ipv4Source( Model ):
      ipv4Address = Ip4Address( help='Source IPv4 address' )
      sourceInterface = Interface( help='Source Interface', optional=True )
      vrfName = Str( help='Associated VRF for sFlow source' )

      def render( self ):
         finalIpAddr = SflowConst.valueWithDefaultInd( str( self.ipv4Address ),
                                                       SflowConst.defaultIp )
         if self.sourceInterface:
            interfaceName = str( self.sourceInterface ).replace( '\'', '' )
            print '  %s from %s (VRF: %s)' % ( finalIpAddr, interfaceName,
                                               self.vrfName )
         else:
            print '  %s (VRF: %s)' % ( finalIpAddr, self.vrfName )

   class Ipv6Source( Model ):
      ipv6Address = Ip6Address( help='Source IPv6 address' )
      sourceInterface = Interface( help='Source Interface', optional=True )
      vrfName = Str( help='Associated VRF for sFlow source' )

      def render( self ):
         finalIpAddr = SflowConst.valueWithDefaultInd( str( self.ipv6Address ),
                                                       SflowConst.defaultIp6 )
         if self.sourceInterface:
            interfaceName = str( self.sourceInterface ).replace( '\'', '' )
            print '  %s from %s (VRF: %s)' % ( finalIpAddr, interfaceName,
                                               self.vrfName )
         else:
            print '  %s (VRF: %s)' % ( finalIpAddr, self.vrfName )

   class SendingDatagram( Model ):
      sending = Bool( help='sFlow samples being sent' )
      reason = Enum( values=_sendingReasonMap.keys(),
                     help='Reason for sendingDatagrams state' )
      vrfName = Str( help='VRF name' )

      def _getSendingReason( self, sendingDatagram, sendingDatagramReason ):
         if sendingDatagram:
            return ''
         else:
            return _sendingReasonMap[ sendingDatagramReason ]

      def render( self ):
         print '  %s%s (VRF: %s)' % ( _convertToYesNo( self.sending ),
                                        self._getSendingReason( self.sending,
                                                                self.reason ),
                                        self.vrfName )

   class BgpExport( Model ):
      export = Bool( help='Export BGP information' )
      vrfName = Str( help='VRF name' )

      def render( self ):
         print '  %s (VRF: %s)' % ( _convertToYesNo( self.export ), self.vrfName )

   # Configuration
   ipv4Destinations = List( help='List of IPv4 destinations',
                            valueType=Ipv4Destination )
   ipv6Destinations = List( help='List of IPv6 destinations',
                            valueType=Ipv6Destination )
   ipv4Sources = List( help='List of Source IPv4 addresses', valueType=Ipv4Source )
   ipv6Sources = List( help='List of Source IPv6 addresses', valueType=Ipv6Source )
   sampleRate = Int( help='Sample rate, 1 sample per N packets' )
   sampleTruncateSize = Int( help='Sample truncate size limit in bytes' )
   hwSampleTruncateSize = Int( help='HW Sample truncate size limit in bytes',
                               optional=True )
   accelSampleRate = Int( help='Sample rate for hardware acceleration, 1 sample per'
                          ' N packets', optional=True )
   pollingInterval = Float( help='Polling interval in seconds' )
   rewriteDscp = Bool( help='Rewrite DSCP value in sFlow samples',
                              optional=True )
   dscpValue = Int( help='DSCP value in sFlow datagram' )

   # Status
   enabled = Bool( help='sFlow globally enabled' )
   accelEnabled = Bool( help='sFlow hardware acceleration running', optional=True )
   polling = Bool( help='sFlow polling' )
   samplingEnabled = Bool( help='sFlow sampling enabled' )
   bgpExports = List( help='List of BGP Exports',
                      valueType=BgpExport, optional=True )
   sendingDatagrams = List( help='List of Sending datagrams',
                            valueType=SendingDatagram )
   hardwareSampleRate = Int( help='Hardware sample rate for software Sflow, 1 sample'
                             ' per N packets' )
   hardwareAccelSampleRate = Int( help='Hardware sample rate for hardware'
                                  ' accelerated Sflow, 1 sample per N packets',
                                  optional=True )

   accelSupported = Bool( help='Is hardware acceleration supported', optional=True,
                          default=False )

   # Statistics
   # The following counters contain SflowAccel counters too.
   totalPackets = Int( help='Total packets' )
   softwareSamples = Int( help='Number of samples' )
   samplePool = Int( help='Sample pool' )
   hardwareSamples = Int( help='Number of hardware samples' )
   datagrams = Int( help='Number of datagrams' )

   # Details
   details = Submodel( valueType=Details, help='Detailed sFlow information',
                       optional=True )

   def render( self ):
      print 'sFlow Configuration'
      print '-------------------'
      self._renderDestinations()
      self._renderSrcIpAddrs()
      print 'Hardware Sample Rate for SW sFlow: %s' % self._getSampleRate()
      if self.accelSupported:
         print 'Hardware Sample Rate for HW sFlow: {}'.format(
            self._getAccelSampleRate() )
      print 'Polling Interval (sec): %s' % self._getPollingInterval()
      print 'Rewrite DSCP value: %s' % \
            _convertToYesNo( self._isRewriteDscpOn() )
      print 'Datagram DSCP value: %s' % self._dscpValue()

      print ''
      print 'Status'
      print '------'
      print 'Running: %s' % _convertToYesNo( self.enabled )
      if self.accelSupported:
         print 'Hardware acceleration running: {}'.format(
            _convertToYesNo( self.accelEnabled ) )
      print 'Polling On: %s' % ( 'Yes (default)' if self.polling else 'No' )
      print 'Sampling On: %s' % ( 'Yes (default)' if self._isSamplingOn()
                                  else 'No' )
      print 'Sample Truncation Size for SW sFlow: %s' % \
                                                ( self._getSampleTruncateSize() )
      if self.accelSupported and self.accelEnabled:
         print 'Sample Truncation Size for HW sFlow: %s' % \
                                                ( self._getHwSampleTruncateSize() )
      self._renderSendingDatagrams()
      self._renderBgpExports()
      print 'Hardware Sample Rate for SW sFlow: %s' % self._getHardwareSampleRate()
      if self.accelSupported and self.accelEnabled:
         print 'Hardware Sample Rate for HW sFlow: {}'.format(
            self._getHardwareAccelSampleRate() )
      if self.details:
         print 'Hardware Sampling On: %s' % _convertToYesNo(
                                             self.details.hardwareSamplingEnabled )
         sampleOutputText = _convertToYesNo(
            self.details.sampleOutputInterface )
         print "Sample Output Interface: %s" % sampleOutputText
         sampleMplsExtensionText = _convertToYesNo(
            self.details.sampleMplsExtension )
         print "Sample MPLS Extension: %s" % sampleMplsExtensionText
         sampleSwitchExtensionText = _convertToYesNo(
            self.details.sampleSwitchExtension )
         print "Sample Switch Extension: %s" % sampleSwitchExtensionText
         sampleRouterExtensionText = _convertToYesNo(
            self.details.sampleRouterExtension )
         print "Sample Router Extension: %s" % sampleRouterExtensionText
         sampleTunnelIpv4EgrExtensionText = _convertToYesNo(
            self.details.sampleTunnelIpv4EgrExtension )
         print "Sample Tunnel IPv4 Egress Extension: %s" % \
               sampleTunnelIpv4EgrExtensionText
         sampleIngressSubintfText = _convertToYesNo(
            self.details.sampleIngressSubintf )
         print "Sample Input Subinterface: %s" % \
               sampleIngressSubintfText
         sampleEgressSubintfText = _convertToYesNo(
            self.details.sampleEgressSubintf )
         print "Sample Output Subinterface: %s" % \
               sampleEgressSubintfText
         portchOutputIfIndexText = self.details.portChannelOutputIfIndex
         print "Port Channel Output Interface Index: %s" % portchOutputIfIndexText
         if self.accelSupported and self.accelEnabled:
            if self.details.accelUnsupportedExtensions:
               print 'Hardware acceleration unsupported features:'
            for unsupportedFeature in self.details.accelUnsupportedExtensions:
               print '  {}'.format( unsupportedFeature )
         sampleEncodingFormatText = self.details.sampleEncodingFormat
         print "Sample Encoding Format: %s" % sampleEncodingFormatText
      print ''
      print 'Statistics'
      print '----------'
      print 'Total Packets: %d' % self.totalPackets
      print 'Number of Samples: %d' % self.softwareSamples
      print 'Sample Pool: %d' % self.samplePool
      print 'Hardware Trigger: %d' % self.hardwareSamples
      print 'Number of Datagrams: %d' % self.datagrams
      if self.details:
         print 'Number of Samples Discarded: %d' % self.details.samplesDiscarded

   def _getPollingInterval( self ):
      return SflowConst.valueWithDefaultInd( self.pollingInterval,
                                             SflowConst.defaultPollingInterval )

   def _isSamplingOn( self ):
      return self.samplingEnabled and self.enabled

   def _isRewriteDscpOn( self ):
      return self.rewriteDscp

   def _dscpValue( self ):
      return self.dscpValue

   def _getSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.sampleRate,
                                                SflowConst.defaultSampleRate )

   def _getSampleTruncateSize( self ):
      return SflowConst.valueWithDefaultInd( self.sampleTruncateSize,
                                             SampleTruncateSizeType.min )

   def _getHwSampleTruncateSize( self ):
      return self.hwSampleTruncateSize

   def _getAccelSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.accelSampleRate,
                                                SflowConst.defaultSampleRate )

   def _getHardwareSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.hardwareSampleRate,
                                                SflowConst.defaultSampleRate )

   def _getHardwareAccelSampleRate( self ):
      if not self.samplingEnabled:
         return 'None'
      else:
         return SflowConst.valueWithDefaultInd( self.hardwareAccelSampleRate,
                                                SflowConst.defaultSampleRate )

   def _renderSrcIpAddrs( self ):
      if not self.ipv4Sources and not self.ipv6Sources:
         print SflowConst.sflowSourceDefaultMsg
      else:
         print 'Source(s):'

         allIpv4Sources = {}
         for source in self.ipv4Sources:
            allIpv4Sources[ source.vrfName ] = source

         allIpv6Sources = {}
         for source in self.ipv6Sources:
            allIpv6Sources[ source.vrfName ] = source

         for vrfName in sorted( set( allIpv4Sources.keys() +
                                     allIpv6Sources.keys() ) ):
            if vrfName in allIpv4Sources:
               allIpv4Sources[ vrfName ].render()
            if vrfName in allIpv6Sources:
               allIpv6Sources[ vrfName ].render()

   def _renderDestinations( self ):
      if not self.ipv4Destinations and not self.ipv6Destinations:
         print SflowConst.sflowDestDefaultMsg
      else:
         print 'Destination(s):'

         allIpv4Destinations = {}
         for dest in self.ipv4Destinations:
            allIpv4Destinations.setdefault( dest.vrfName, [] ).append( dest )

         allIpv6Destinations = {}
         for dest in self.ipv6Destinations:
            allIpv6Destinations.setdefault( dest.vrfName, [] ).append( dest )

         for vrfName in sorted( set( allIpv4Destinations.keys() +
                                     allIpv6Destinations.keys() ) ):
            if vrfName in allIpv4Destinations:
               for dest in allIpv4Destinations[ vrfName ]:
                  dest.render()
            if vrfName in allIpv6Destinations:
               for dest in allIpv6Destinations[ vrfName ]:
                  dest.render()

   def _renderSendingDatagrams( self ):
      print 'Send Datagrams:'
      for send in self.sendingDatagrams:
         send.render()

   def _renderBgpExports( self ):
      print 'BGP Export:'
      for export in self.bgpExports:
         export.render()

class SflowInterfaces( Model ):
   __revision__ = 2
   enabled = Bool( help='sFlow globally enabled' )
   ingressInterfacesEnabledDefault = Bool(
                           help='Global ingress sFlow configuration for interfaces' )
   egressInterfacesEnabledDefault = Bool(
                           help='Global egress sFlow configuration for interfaces' )
   interfaces = Dict( keyType=Interface, valueType=str,
                      help='Mapping between an interface and its status' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'interfacesEnabledDefault' ] = \
               dictRepr[ 'ingressInterfacesEnabledDefault' ]
         del dictRepr[ 'ingressInterfacesEnabledDefault' ]
         del dictRepr[ 'egressInterfacesEnabledDefault' ]
      return dictRepr

   def render( self ):
      ingressIntfConfigStr = "Default ingress sFlow configuration for an interface: "
      if self.ingressInterfacesEnabledDefault:
         print ingressIntfConfigStr + 'Enabled'
      else:
         print ingressIntfConfigStr + 'Disabled'

      egressIntfConfigStr = "Default egress sFlow configuration for an interface: "
      if toggleEgressSflowEnabled():
         if self.egressInterfacesEnabledDefault:
            print egressIntfConfigStr + 'Enabled'
         else:
            print egressIntfConfigStr + 'Disabled'
      print "sFlow Interface (s):"
      print "--------------------"
      if not self.enabled:
         print "sFlow is not running"
      elif len( self.interfaces ) == 0:
         print "No interfaces have been configured for sFlow"
      else:
         for interface in Arnet.sortIntf( self.interfaces ):
            print "%s - %s" % ( interface, self.interfaces[ interface ] )
