#!/usr/bin/env python
# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements sFlow configuration.  In particular, it provides:
# -  the "[no|default] sflow enable" command
# -  the "[no|default] sflow egress enable" command
# -  the "[no|default] sflow egress unmodified enable" command
# -  the "[no|default] sflow polling-interval <secs>" command
# -  the "no sflow sample" command
# -  the "sflow sample [dangerous] <rate>" command
# -  the "[no|default] sflow destination <ip-addr> [<dest-udp-port>]" command
# -  the "[no|default] sflow source <ip-addr>" command
# -  the "[no|default] sflow sample rewrite dscp" command
# -  the "[no|default] sflow sample output interface" command
# -  the "[no|default] sflow sample output subinterface" command
# -  the "[no|default] sflow extension bgp" command
# -  the "[no|default] sflow extension mpls" command
# -  the "[no|default] sflow extension router" command
# -  the "[no|default] sflow extension switch" command
# -  the "[no|default] sflow extension tunnel ipv4 egress" command
# -  the "[no|default] sflow sample output portchannel ifindex" command
# -  the "[no|default] sflow run" command
# -  the "[no|default] sflow qos dscp <0-63>" command
# -  the "[no|default] sflow interface disable default" command
# -  the "[no|default] sflow interface egress enable default" command
# -  the "[no|default] sflow interface egress unmodified enable default" command
# -  the "[no|default] sflow sample include drop reason acl" command
# -  the "clear sflow counters" command
# -  the "show sflow" command
# -  the "show sflow detail" command
# -  the "show sflow interfaces" command
#-------------------------------------------------------------------------------

import os
import re

import Arnet
import Tac
import CliParser
import LazyMount
import ConfigMount
import SflowConst
from SflowModel import SflowStatus, SflowInterfaces
from IpLibConsts import DEFAULT_VRF
import SflowUtil
import DscpCliLib
import SharedMem
import Smash
import CliExtensions
from Toggles.SflowLibToggleLib import toggleEgressSflowEnabled

dscpConfig = None
sflowHwStatusDir = None
veosStatus = None
sflowAccelCapabilities = None
sflowAccelRunnability = None
sflowAccelConfig = None
sflowAccelHwConfig = None
sflowAccelStatusDir = None
smashEntityManager = None
sflowAccelCountersTable = None
sflowConfig = None
sflowCounterConfig = None
sflowStatus = None
sflowHwConfig = None

AccelId = Tac.Type( "Hardware::SflowAccel::AccelId" )
MessageHelper = Tac.Type( "Hardware::SflowAccel::FeatureMessageHelper" )
FpgaCounterKey = Tac.Type( 'Hardware::SflowAccel::FpgaCounterKey' )
SflowAccelPPCountersTableType = \
      Tac.Type( "Hardware::SflowAccel::SflowAccelCounterTable" )

def sflowHwStatus():
   return SflowUtil.sflowHwStatus( sflowHwStatusDir )

def sflowHwSampleRate():
   return SflowUtil.sflowHwSampleRate( sflowHwStatusDir )

def sflowHwSamples():
   totalHwSamples = max( [ s.hwSamples for s in sflowHwStatusDir.values() ] )
   return max( veosStatus.hwSamples, totalHwSamples )

def sflowAccelSupported():
   return sflowAccelCapabilities.sflowAccelSupported or \
      sflowAccelCapabilities.sflowAccelPPSupported

def sflowSupportedGuard( mode, token ):
   if sflowHwStatus().sflowSupported:
      return None
   return CliParser.guardNotThisPlatform

def sflowAccelGuard( mode, token ):
   if sflowAccelSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowSampleIncludeDropReasonAclSupportedGuard( mode, token ):
   if sflowHwStatus().sflowSampleIncludeDropReasonAclSupported:
      return sflowSupportedGuard( mode, token )
   else:
      return CliParser.guardNotThisPlatform

def sflowTunnelExtensionGuard( mode, token ):
   # Add additional tunnel extensions here as they are added
   if sflowHwStatus().tunnelIpv4EgrExtensionSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowEgressSubintfGuard( mode, token ):
   if sflowHwStatus().egressSubintfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# On some switches ( currently Petra ) only few sample rates are supported
# These cli guards are used to choose the sample rate parser rule
# based on the HwStatus::sampleRangeSupported attribute published by the
# platform
def sflowSampleRangeGuard( mode, token ):
   if sflowHwStatus().sampleRangeSupported:
      return None
   return CliParser.guardNotThisPlatform

def sflowSampleSetGuard( mode, token ):
   if not sflowHwStatus().sampleRangeSupported:
      return None
   return CliParser.guardNotThisPlatform

# Takes the counters object for an accelerator and returned a python dic equivalent,
# potentially zeroing out the counters if the object timestamp is older than the last
# clear counters request.
def rectifiedSflowAccelCounters( accelCounters ):
   rectifiedCounters = {}
   fields = [
      'inPktCnt',
      'outSflowDgramCnt',
      'outFlowSampleCnt',
      'inProcPktCnt',
      'rcvPktDropCnt',
      'rcvPktTrunCnt',
      'inPktErrCnt',
      'outProcSflowDgramCnt',
      'samplePool',
   ]
   # If the last clear in smash is before the last clear request made by the Cli, it
   # means the writer agent hasn't processed the request yet, so displaying all 0s.
   needClear = accelCounters.lastClear < sflowCounterConfig.countersCleared
   for field in fields:
      value = 0 if needClear else getattr( accelCounters, field )
      rectifiedCounters[ field ] = value

   return rectifiedCounters

# Returns a python dic containing SflowAccel counters:
# {
#    sflowAccelName : {
#       counterName : counterValue,
#       ..,
#    },
#    ...
# }
def sflowAccelCounters( forAggregatedCounters=False ):
   if sflowAccelRunnability.sflowAccelPPRunnable:
      if forAggregatedCounters:
         aggregatedCountersAccelKey = FpgaCounterKey( AccelId.totalAccelId() )
         accelKeyFilter = \
               lambda accelKey: ( accelKey == aggregatedCountersAccelKey )
      else:
         accelKeyFilter = lambda accelKey: False
   else:
      accelKeyFilter = lambda accelKey: True
   accelNameFunc = lambda accelId: \
         ( 'Total' if ( accelId == AccelId.totalAccelId() )
           else AccelId( accelId ).accelName() )

   result = {}
   for accelCountersKey, accelCounters in \
         sflowAccelCountersTable.fpgaCounter.iteritems():
      if not accelKeyFilter( accelCountersKey ):
         continue
      accelName = accelNameFunc( accelCountersKey.accelId )
      result[ accelName ] = rectifiedSflowAccelCounters( accelCounters )

   return result

# Sum of sflow accel counters for all accelerators.
def aggregatedSflowAccelCounters():
   accelCountersSum = {}
   resultCounters = sflowAccelCounters( forAggregatedCounters=True )
   for accelCounters in resultCounters.itervalues():
      for counterName, counterValue in accelCounters.iteritems():
         accelCountersSum[ counterName ] = ( accelCountersSum.get( counterName, 0 ) +
                                             counterValue )

   return accelCountersSum

def sflowPortChannelOutputIfindexConfigGuard( mode, token ):
   if sflowHwStatus().portChannelOutputIfindexConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def sflowEgressSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressModifiedSflowIntfSupported or \
      sflowHwStatus().egressUnmodifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowEgressModifiedSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressModifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowEgressUnmodifiedSflowIntfGuard( mode, token ):
   if sflowHwStatus().egressUnmodifiedSflowIntfSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# the "[no] sflow run" command in "config" mode
# full syntax: {no|default} sflow run
#-------------------------------------------------------------------------------
def runSflow( mode, args ):
   sflowConfig.enabled = True

def noRunSflow( mode, args ):
   sflowConfig.enabled = False

#-------------------------------------------------------------------------------
# the "[no] sflow polling-interval <secs>" command in "config" mode
# full syntax: {no|default} sflow polling-interval <secs>
#-------------------------------------------------------------------------------
def setPollingInterval( mode, args ):
   sflowConfig.pollingInterval = args[ 'POLLING_INTERVAL' ]

def defaultPollingInterval( mode, args ):
   sflowConfig.pollingInterval = sflowConfig.pollingIntervalDefault

def noPollingInterval( mode, args ):
   # setting pollingInterval to 0 implies polling is disabled.
   sflowConfig.pollingInterval = 0

#-------------------------------------------------------------------------------
# the "[no] sflow sample <rate>" command in "config" mode
# full syntax: {default} sflow sample <rate>
#              no sflow sample
#-------------------------------------------------------------------------------
def rateRangeFn( mode ):
   # This function returns the range supported for the Sflow sample rate
   return ( sflowHwStatus().minSampleRate, sflowHwStatus().maxSampleRate )

def dangRateRangeFn( mode ):
   # This function returns the entire/dangerous Sflow sample rate range
   return ( 1, sflowHwStatus().maxSampleRate )

def setSampleRate( mode, args ):
   rate = args[ 'RATE' ] if 'RATE' in args else args[ 'DANGEROUS_RATE' ]
   sflowConfig.sampleRate = int( rate )
   sflowConfig.samplingEnabled = True
   sflowConfig.sampleDangerous = 'dangerous' in args

def noSampleRate( mode, args ):
   sflowConfig.samplingEnabled = False

def defaultSampleRate( mode, args ):
   sflowConfig.sampleRate = sflowConfig.sampleRateDefault
   sflowConfig.samplingEnabled = True
   sflowConfig.sampleDangerous = False

#-------------------------------------------------------------------------------
# the "[no] sflow destination <ipv4/6-addr-or-hostname> [<dest-udp-port>]" command
# in "config" mode
# full syntax:
#   {default|no} sflow destination <ip4/6-addr-or-hostname> [<dst-udp-port>]
#-------------------------------------------------------------------------------
def setDestination( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   port = args.get( 'PORT', SflowConst.defaultPort )
   addr = str( args[ 'DESTINATION' ] )
   # BUG23291 - Reenable this when resolution is asynchronous
   # HostnameCli.resolveHostname( hostname, printWarnMsg=True )
   # BUG23291-Use hostname instead of ipAddr for an argument after the fix

   SflowUtil.addConfigCollectorInfo( sflowConfig, addr, port, vrfName )
   updateDscpRules()

def noDestination( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   port = args.get( 'PORT' )
   addr = str( args[ 'DESTINATION' ] )
   SflowUtil.removeConfigCollectorInfo( sflowConfig, addr, port, vrfName )
   updateDscpRules()

# tokenHostname = HostnameCli.IpAddrOrHostnameRule(
#   name='hostname',
#   helpname='WORD',
#   helpdesc='Hostname or IP address of Collector',
#   ipv6=False )

# BUG23291 - Reenable this when resolution is asynchronous
# BasicCli.GlobalConfigMode.addCommand(
#    ( tokenSflow, tokenDestination, tokenHostname, [ '>>port', port ],
#      setDestination ) )

# BUG23291 - Reenable this when resolution is asynchronous
# BasicCli.GlobalConfigMode.addCommand(
#    ( BasicCli.noOrDefault, tokenSflow, tokenDestination,
#       tokenHostname, [ '>>port', port ], noDestination ) )

#-------------------------------------------------------------------------------
# -  the "[no] sflow source <ip-addr>" command
# full syntax: {no|default} sflow source <ip-addr>
#-------------------------------------------------------------------------------
def setSource( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   ipAddr = args[ 'SOURCE' ]

   if ( vrfName in sflowConfig.sflowVrfConfig and
        sflowConfig.sflowVrfConfig[ vrfName ].srcIntfName ):
      mode.addError( "Source-interface is already configured for VRF " + vrfName )
   else:
      SflowUtil.addConfigSourceIp( sflowConfig, ipAddr, vrfName )

def noSource( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   SflowUtil.removeConfigSourceIp( sflowConfig, vrfName )

#-------------------------------------------------------------------------------
# -  the "[no] sflow source-interface <intf-name>" command
# full syntax: {no|default} sflow source-interface <intf-name>
#-------------------------------------------------------------------------------
def setSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   intf = args[ 'IPINTF' ]

   if ( vrfName in sflowConfig.sflowVrfConfig and
        sflowConfig.sflowVrfConfig[ vrfName ].switchIpAddr != SflowConst.defaultIp ):
      mode.addError( "Source IP address is already configured for VRF " + vrfName )
   else:
      SflowUtil.addConfigSourceIntf( sflowConfig, intf.name, vrfName )

def noSourceIntf( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   SflowUtil.removeConfigSourceIntf( sflowConfig, vrfName )

#-------------------------------------------------------------------------------
# the "[no] sflow sample rewrite dscp" command in "config" mode
# full syntax: {no|default} sflow sample rewrite dscp
#-------------------------------------------------------------------------------
def rewriteDscp( mode, args ):
   sflowConfig.rewriteDscp = True

def noRewriteDscp( mode, args ):
   sflowConfig.rewriteDscp = False

#-------------------------------------------------------------------------------
# the "[no] sflow sample output interface" command in "config" mode
# full syntax: {no|default} sflow sample output interface
#
# the "[no] sflow sample output subinterface" command in "config" mode
# full syntax: {no|default} sflow sample output subinterface
#-------------------------------------------------------------------------------
def sflowSampleOutputHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = True
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = True

def sflowSampleOutputNoHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = False
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = False

def sflowSampleOutputDefaultHandler( mode, args ):
   if 'interface' in args:
      sflowConfig.outputInterface = sflowConfig.outputInterfaceDefault
   elif 'subinterface' in args:
      sflowConfig.egressSubintf = sflowConfig.egressSubintfDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample input subinterface" command in "config" mode
# full syntax: {no|default} sflow sample input subinterface
#-------------------------------------------------------------------------------
def sflowSampleInputHandler( mode, args ):
   sflowConfig.ingressSubintf = True

def sflowSampleInputNoHandler( mode, args ):
   sflowConfig.ingressSubintf = False

def sflowSampleInputDefaultHandler( mode, args ):
   sflowConfig.ingressSubintf = sflowConfig.ingressSubintfDefault

#-------------------------------------------------------------------------------
# the "[no] sflow sample truncate size SIZE" command in "config" mode
# full syntax: {no|default} sflow sample truncate size ...
#-------------------------------------------------------------------------------
def sflowSampleTruncateSizeHandler( mode, args ):
   size = args.get( 'SIZE', sflowConfig.sampleTruncateSizeDefault )
   sflowConfig.sampleTruncateSize = size

#-------------------------------------------------------------------------------
# the "[no|default] sflow extension bgp" command in global "config" mode
# the "[no|default] sflow extension mpls" command in global "config" mode
# the "[no|default] sflow extension router" command in global "config" mode
# the "[no|default] sflow extension switch" command in global "config" mode
# the "[no|default] sflow extension tunnel ipv4 egress" command in global "config"
# mode
#-------------------------------------------------------------------------------
def setExtension( attrName, value ):
   # None means to use the default value of this extension.
   effectiveValue = getattr( sflowConfig, attrName + 'Default' ) \
                    if value is None else value
   setattr( sflowConfig, attrName, effectiveValue )

#-------------------------------------------------------------------------------
# the "sflow sample output portchannel ifindex" command in "config" mode
# full syntax:
# {default} sflow sample output portchannel ifindex { member | portchannel }
#-------------------------------------------------------------------------------
def setPortChannelIfindexMode( mode, args ):
   sflowConfig.portChannelIfindex = args[ 'IFINDEX_MODE' ]

def defaultPortChannelIfindex( mode, args ):
   sflowConfig.portChannelIfindex = 'platformDefault'

#-------------------------------------------------------------------------------
# the "[no] sflow enable" command in "config-if" mode
# full syntax: [no] sflow enable
#-------------------------------------------------------------------------------
def intfIngressSflowEnableHandler( mode, args ):
   if mode.intf.name in sflowConfig.disabledIngressIntf:
      del sflowConfig.disabledIngressIntf[ mode.intf.name ]
   sflowConfig.enabledIngressIntf[ mode.intf.name ] = True

def intfIngressSflowEnableNoHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledIngressIntf:
      del sflowConfig.enabledIngressIntf[ mode.intf.name ]
   sflowConfig.disabledIngressIntf[ mode.intf.name ] = True

def intfIngressSflowEnableDefaultHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledIngressIntf:
      del sflowConfig.enabledIngressIntf[ mode.intf.name ]
   if mode.intf.name in sflowConfig.disabledIngressIntf:
      del sflowConfig.disabledIngressIntf[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow egress enable" command in "config-if" mode
# full syntax: [no] sflow egress enable
#-------------------------------------------------------------------------------
def intfEgressSflowEnableHandler( mode, args ):
   if mode.intf.name in sflowConfig.disabledEgressIntf:
      del sflowConfig.disabledEgressIntf[ mode.intf.name ]
   sflowConfig.enabledEgressIntf[ mode.intf.name ] = True

def intfEgressSflowEnableNoHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledEgressIntf:
      del sflowConfig.enabledEgressIntf[ mode.intf.name ]
   sflowConfig.disabledEgressIntf[ mode.intf.name ] = True

def intfEgressSflowEnableDefaultHandler( mode, args ):
   if mode.intf.name in sflowConfig.enabledEgressIntf:
      del sflowConfig.enabledEgressIntf[ mode.intf.name ]
   if mode.intf.name in sflowConfig.disabledEgressIntf:
      del sflowConfig.disabledEgressIntf[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow id ID" command in "config-if" mode
# full syntax: [no|default] sflow id ...
#-------------------------------------------------------------------------------
def intfIdHandler( mode, args ):
   configuredId = args.get( 'ID' )
   sflowConfig.intfConfig.newMember( mode.intf.name ).id = configuredId

def intfIdNoDefaultHandler( mode, args ):
   del sflowConfig.intfConfig[ mode.intf.name ]

#-------------------------------------------------------------------------------
# the "[no] sflow interface disable default" command in "config" mode
# full syntax: {no|default} sflow interface disable default
#-------------------------------------------------------------------------------
def disableIngressInterfacesByDefault( mode, args ):
   sflowConfig.ingressIntfsEnabledDefault = False

def enableIngressInterfacesByDefault( mode, args ):
   sflowConfig.ingressIntfsEnabledDefault = True

#-------------------------------------------------------------------------------
# the "[no] sflow interface egress enable default" command in "config" mode
# full syntax: {no|default} sflow interface egress enable default
#-------------------------------------------------------------------------------
def disableEgressInterfacesByDefault( mode, args ):
   sflowConfig.egressIntfsEnabledDefault = False

def enableEgressInterfacesByDefault( mode, args ):
   sflowConfig.egressIntfsEnabledDefault = True

#-------------------------------------------------------------------------------
# the "[no] sflow sample include drop reason acl" command in "config" mode
# full syntax: {no|default} sflow sample include drop reason acl
#-------------------------------------------------------------------------------
def includeDropReasonAcl( mode, args ):
   sflowConfig.sampleIncludeDropReasonAcl = True

def noIncludeDropReasonAcl( mode, args ):
   sflowConfig.sampleIncludeDropReasonAcl = False

#-------------------------------------------------------------------------------
# the "clear sflow counters" command in "config" mode
# full syntax: clear sflow counters
#-------------------------------------------------------------------------------
def setCountersCleared( mode, args ):
   sflowCounterConfig.countersCleared = sflowCounterConfig.countersCleared + 1

#-------------------------------------------------------------------------------
# the "[no|default] sflow qos dscp <0-63>" command in global "config" mode
#-------------------------------------------------------------------------------
def updateDscpRules():
   dscpValue = sflowConfig.dscpValue

   if not dscpValue:
      del dscpConfig.protoConfig[ 'sflow' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'sflow' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for vrfConfig in sflowConfig.sflowVrfConfig.itervalues():
      for hostAndPort in vrfConfig.collectorHostAndPort.itervalues():
         DscpCliLib.addDscpRule( ruleColl, hostAndPort.hostname, hostAndPort.port,
                                 False, vrfConfig.vrfName, 'udp', dscpValue )
         DscpCliLib.addDscpRule( ruleColl, hostAndPort.hostname, hostAndPort.port,
                                 False, vrfConfig.vrfName, 'udp', dscpValue,
                                 v6=True )

def setDscp( mode, args ):
   sflowConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   sflowConfig.dscpValue = sflowConfig.dscpValueDefault
   updateDscpRules()

#-------------------------------------------------------------------------------
# the "show sflow" and "show sflow detail" commands
#-------------------------------------------------------------------------------

def updateCounters( mode ):
   if os.environ.get( 'SIMULATION_VMID' ):
      # Bypass this in btests
      return

   sflowCounterConfig.counterUpdateRequestTime = Tac.now() # monotonic time

   def countersUpdated():
      return ( sflowStatus.counterUpdateTime >=
               sflowCounterConfig.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def getDestIpAddresses( ):
   ipv4Destinations = []
   ipv6Destinations = []
   for vrfName in sorted( sflowConfig.sflowVrfConfig ):
      collectorHostAndPort = \
         sflowConfig.sflowVrfConfig[ vrfName ].collectorHostAndPort
      for hostname in sorted( collectorHostAndPort ):
         ipv4Match = re.search( SflowUtil.ipAddrPattern, hostname )
         ipv6Match = re.search( Arnet.Ip6AddrRe, hostname )
         if ipv4Match:
            ipv4Destination = SflowStatus.Ipv4Destination()
            try:
               _ = Arnet.IpAddress( hostname )
               ipv4Destination.hostname = hostname
               ipv4Destination.ipv4Address = hostname
            except ValueError:
               ipv4Destination.hostname = hostname
               sflowVrfStatus = sflowStatus.sflowVrfStatus[ vrfName ]
               if hostname in sflowVrfStatus.collectorIpAndPort:
                  ipv4Destination.ipv4Address = \
                     sflowVrfStatus.collectorIpAndPort[ hostname ].ip
            ipv4Destination.port = collectorHostAndPort[ hostname ].port
            ipv4Destination.vrfName = vrfName
            ipv4Destinations.append( ipv4Destination )
         elif ipv6Match:
            ipv6Destination = SflowStatus.Ipv6Destination()
            try:
               _ = Arnet.Ip6Addr( str( hostname ) )
               ipv6Destination.hostname = hostname
               ipv6Destination.ipv6Address = hostname
            except ValueError:
               ipv6Destination.hostname = hostname
               if hostname in ( sflowStatus.sflowVrfStatus[ vrfName ].
                                collectorIpAndPort ):
                  ipv6Destination.ipv6Address = \
                     ( sflowStatus.sflowVrfStatus[ vrfName ].
                       collectorIpAndPort[ hostname ].ip )
            ipv6Destination.port = collectorHostAndPort[ hostname ].port
            ipv6Destination.vrfName = vrfName
            ipv6Destinations.append( ipv6Destination )

   return ipv4Destinations, ipv6Destinations

def getSourceIpAddresses():
   ipv4Sources = []
   ipv6Sources = []
   for vrfName in sorted( sflowConfig.sflowVrfConfig ):
      ipv4Source = SflowStatus.Ipv4Source()
      ipv6Source = SflowStatus.Ipv6Source()

      sflowVrfConfig = sflowConfig.sflowVrfConfig[ vrfName ]
      if sflowVrfConfig.srcIntfName:
         sflowVrfStatus = sflowStatus.sflowVrfStatus.get( vrfName )
         ipv4Source.ipv4Address = \
            sflowVrfStatus.srcIpAddr if sflowVrfStatus else SflowConst.defaultIp
         ipv4Source.sourceInterface = sflowVrfConfig.srcIntfName
         ipv6Source.ipv6Address = \
            sflowVrfStatus.srcIp6Addr if sflowVrfStatus else SflowConst.defaultIp6
         ipv6Source.sourceInterface = sflowVrfConfig.srcIntfName
      else:
         ipv4Source.ipv4Address = sflowVrfConfig.switchIpAddr
         ipv4Source.sourceInterface = None
         ipv6Source.ipv6Address = SflowConst.defaultIp6
         ipv6Source.sourceInterface = None

      ipv4Source.vrfName = vrfName
      ipv6Source.vrfName = vrfName
      ipv4Sources.append( ipv4Source )
      ipv6Sources.append( ipv6Source )

   if not ipv4Sources:
      ipv4Source = SflowStatus.Ipv4Source()
      ipv4Source.ipv4Address = SflowConst.defaultIp
      ipv4Source.sourceInterface = None
      ipv4Source.vrfName = DEFAULT_VRF
      ipv4Sources.append( ipv4Source )

   if not ipv6Sources:
      ipv6Source = SflowStatus.Ipv6Source()
      ipv6Source.ipv6Address = SflowConst.defaultIp6
      ipv6Source.sourceInterface = None
      ipv6Source.vrfName = DEFAULT_VRF
      ipv6Sources.append( ipv6Source )

   return ipv4Sources, ipv6Sources

def getSendingDatagrams():
   sendingDatagrams = []
   for vrfName in sorted( sflowConfig.sflowVrfConfig ):
      if vrfName in sflowStatus.sflowVrfStatus:
         sendingDatagram = SflowStatus.SendingDatagram()
         sflowVrfStatus = sflowStatus.sflowVrfStatus[ vrfName ]
         sendingDatagram.sending = sflowVrfStatus.sendingSflowDatagrams
         sendingDatagram.reason = sflowVrfStatus.sendingSflowDatagramsReason
         sendingDatagram.vrfName = vrfName
         sendingDatagrams.append( sendingDatagram )

   if not sendingDatagrams:
      sendingDatagram = SflowStatus.SendingDatagram()
      sendingDatagram.sending = sflowStatus.sendingSflowDatagrams
      sendingDatagram.reason = sflowStatus.sendingSflowDatagramsReason
      sendingDatagram.vrfName = DEFAULT_VRF
      sendingDatagrams.append( sendingDatagram )

   return sendingDatagrams

def getBgpExports():
   bgpExports = []
   for vrfName in sorted( sflowConfig.sflowVrfConfig ):
      bgpExport = SflowStatus.BgpExport()
      # Bgp extension is currently not per vrf
      bgpExport.export = sflowConfig.bgpExtension
      bgpExport.vrfName = vrfName
      bgpExports.append( bgpExport )

   if not bgpExports:
      bgpExport = SflowStatus.BgpExport()
      bgpExport.export = sflowConfig.bgpExtension
      bgpExport.vrfName = DEFAULT_VRF
      bgpExports.append( bgpExport )

   return bgpExports

def sflowEncodingFormat():
   if sflowConfig.ingressSubintf or sflowConfig.egressSubintf:
      return "expanded"
   else:
      return "compact"

def getSflowDetails():
   details = SflowStatus.Details()
   details.hardwareSamplingEnabled = sflowHwConfig.enabled
   details.samplesDiscarded = sflowStatus.samplesDiscarded
   details.sampleOutputInterface = sflowConfig.outputInterface
   details.sampleMplsExtension = sflowConfig.mplsExtension
   details.sampleSwitchExtension = sflowConfig.switchExtension
   details.sampleRouterExtension = sflowConfig.routerExtension
   details.sampleTunnelIpv4EgrExtension = sflowConfig.tunnelIpv4EgrExtension
   details.portChannelOutputIfIndex = sflowHwStatus().portChannelOutputIfindex
   details.sampleIngressSubintf = sflowConfig.ingressSubintf
   details.sampleEgressSubintf = sflowConfig.egressSubintf
   details.sampleEncodingFormat = sflowEncodingFormat()
   LazyMount.force( sflowAccelStatusDir )
   statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                   sflowAccelStatusDir )
   sflowAccelStatus = statusHelper.sflowAccelStatus()
   # No point displaying SflowAccel status if not enabled
   if sflowAccelSupported() and sflowAccelStatus.running and \
      sflowAccelStatus.inactiveFeature:
      for feature in sflowAccelStatus.inactiveFeature.values():
         descriptionStr = MessageHelper.featureDescription( feature )
         details.accelUnsupportedExtensions.append( descriptionStr )

   return details

def doPrepShowSflow( mode, args ):
   if sflowConfig.enabled:
      updateCounters( mode )
   else:
      mode.addWarning( "Displaying counters that may be stale" )

def doShowSflow( mode, args ):
   detail = 'detail' in args
   ret = SflowStatus()
   ret.enabled = sflowConfig.enabled
   ret.samplingEnabled = sflowConfig.samplingEnabled
   ret.sendingDatagrams = getSendingDatagrams()
   ret.bgpExports = getBgpExports()
   ret.polling = sflowStatus.pollingEnabled
   ret.pollingInterval = sflowConfig.pollingInterval
   ret.sampleRate = sflowConfig.sampleRate
   ret.sampleTruncateSize = sflowConfig.sampleTruncateSize
   ret.rewriteDscp = sflowConfig.rewriteDscp
   ret.dscpValue = sflowConfig.dscpValue
   ret.hardwareSampleRate = sflowHwSampleRate()
   ret.ipv4Destinations, ret.ipv6Destinations = getDestIpAddresses()
   ret.ipv4Sources, ret.ipv6Sources = getSourceIpAddresses()
   ret.totalPackets = sflowStatus.totalPackets
   ret.softwareSamples = sflowStatus.swSamples
   ret.samplePool = sflowStatus.samplePool
   ret.hardwareSamples = sflowHwSamples()
   ret.datagrams = sflowStatus.datagrams
   if detail:
      ret.details = getSflowDetails()

   if sflowAccelSupported():
      LazyMount.force( sflowAccelStatusDir )
      statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                      sflowAccelStatusDir )
      sflowAccelStatus = statusHelper.sflowAccelStatus()
      ret.accelSupported = True
      ret.accelEnabled = sflowAccelStatus.running
      if sflowAccelConfig.sampleRate == 0:
         # 0 means use the SW sflow rate
         ret.accelSampleRate = sflowConfig.sampleRate
      else:
         ret.accelSampleRate = sflowAccelConfig.sampleRate
      # No point displaying SflowAccel status if not enabled
      if sflowAccelStatus.running:
         ret.hardwareAccelSampleRate = sflowAccelHwConfig.sampleRate
         ret.hwSampleTruncateSize = sflowHwStatus().hwSampleTruncateSize

      accelCtrs = aggregatedSflowAccelCounters()
      # Don't need to update ret.totalPackets, as it's the sum of all packets on all
      # interfaces, it is only updated by PollerSms when running counters polling.

      # softwareSamples is actually showed as "Number of Samples" in the Cli output,
      # for SW Sflow it actually represents the packets processed by the Sflow
      # agent. The closest equivalent for SflowAccel is inProcPktCnt which represents
      # the packets that made it to the accelerator and were successfuly processed.
      ret.softwareSamples += accelCtrs.get( 'inProcPktCnt', 0 )

      # Sample pool as given by Sflow accelerators.
      ret.samplePool += accelCtrs.get( 'samplePool', 0 )

      # hardwareSamples is actually showed as "Hardware Trigger" in the Cli output,
      # for SW Sflow it is updated using sand-dma counter. The closest equivalent for
      # SflowAccel is inPktCnt which represents all the packets that made it to the
      # accelerator, they may then be processed or dropped.
      ret.hardwareSamples += accelCtrs.get( 'inPktCnt', 0 )

      # Number of datagrams generated by the Aclow accelerators.
      ret.datagrams += accelCtrs.get( 'outSflowDgramCnt', 0 )

      if detail:
         # Use the sum of packets that arrived with errors to the accelerator + the
         # packets that had to be dropped due to bandwidth limitations of the
         # accelerator.
         ret.details.samplesDiscarded += ( accelCtrs.get( 'rcvPktDropCnt', 0 ) +
                                           accelCtrs.get( 'inPktErrCnt', 0 ) )

   return ret

#-------------------------------------------------------------------------------
# the "show sflow interfaces" command
#-------------------------------------------------------------------------------
# the hook should return the egress runIntfs disabled by platform, currently provided
# by SandFap only
egressIntfDisabledHook = CliExtensions.CliHook()

def doShowSflowIntfs( mode, args ):
   ret = SflowInterfaces()
   ret.enabled = sflowConfig.enabled
   ret.ingressInterfacesEnabledDefault = sflowConfig.ingressIntfsEnabledDefault
   ret.egressInterfacesEnabledDefault = sflowConfig.egressIntfsEnabledDefault
   egressSamplingIntfs = set()
   disabledEgressIntfs = []

   # This is a temporary workaround for BUG173418. RunIntfs should be cleared by
   # ConfigSm by reacting to pollingEnabled and samplingEnabled
   if sflowConfig.samplingEnabled:
      samplingIntfs = set( sflowHwConfig.runIntf )
      egressSamplingIntfs = set( sflowHwConfig.runEgressIntf )
   else:
      samplingIntfs = set()

   if sflowStatus.pollingEnabled:
      pollingIntfs = set( sflowStatus.runIntfForPolling )
   else:
      pollingIntfs = set()

   for hook in egressIntfDisabledHook.extensions():
      disabledEgressIntfs += hook( mode )

   for intfName in pollingIntfs | samplingIntfs | egressSamplingIntfs:
      if ( intfName in sflowHwStatus().intfOperStatus and
           not sflowHwStatus().intfOperStatus[ intfName ].running ):
         reason = sflowHwStatus().intfOperStatus[ intfName ].reason
      else:
         sflowTypes = []
         inactiveTypes = []
         if intfName in samplingIntfs:
            if toggleEgressSflowEnabled():
               sflowTypes.append( 'Ingress' )
            else:
               sflowTypes.append( 'Flow' )
         if intfName in egressSamplingIntfs:
            if toggleEgressSflowEnabled():
               if intfName in disabledEgressIntfs:
                  inactiveTypes.append( 'Egress' )
               else:
                  sflowTypes.append( 'Egress' )
         if intfName in pollingIntfs:
            sflowTypes.append( 'Counter' )

         sFlowTypeStr = ','.join( sflowTypes )
         reason = 'running({})'.format( sFlowTypeStr )
         if inactiveTypes:
            inactiveTypeStr = ','.join( inactiveTypes )
            inactiveStr = 'inactive({})'.format( inactiveTypeStr )
            if sflowTypes:
               reason = reason + ', ' + inactiveStr
            else:
               reason = inactiveStr
      ret.interfaces[ intfName ] = reason
   return ret

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global sflowConfig, sflowCounterConfig, sflowStatus, sflowHwConfig
   global sflowHwStatusDir
   global dscpConfig, veosStatus
   global sflowAccelCapabilities
   global sflowAccelRunnability
   global sflowAccelConfig
   global sflowAccelHwConfig
   global sflowAccelStatusDir
   global smashEntityManager
   global sflowAccelCountersTable

   sflowConfig = ConfigMount.mount(
      entityManager, 'sflow/config', 'Sflow::Config', 'w' )
   sflowCounterConfig = LazyMount.mount(
      entityManager, 'sflow/counterConfig', 'Sflow::CounterConfig', 'w' )
   sflowStatus = LazyMount.mount(
      entityManager, 'sflow/status', 'Sflow::Status', 'rO' )
   sflowHwConfig = LazyMount.mount(
      entityManager, 'sflow/hwconfigdir/sflow', 'Sflow::HwConfig', 'rO' )
   sflowHwStatusDir = LazyMount.mount(
      entityManager, 'sflow/hwstatus', 'Tac::Dir', 'ri' )
   veosStatus = LazyMount.mount(
      entityManager, 'sflow/veosstatus', 'Sflow::HwStatus', 'rO' )
   dscpConfig = ConfigMount.mount( entityManager, 'mgmt/dscp/config',
                                   'Mgmt::Dscp::Config', 'w' )

   # SflowAccel
   sflowAccelCapabilities = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/capabilities',
      'Hardware::SflowAccel::Capabilities', 'r' )
   sflowAccelRunnability = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/runnability',
      'Hardware::SflowAccel::Runnability', 'r' )
   sflowAccelConfig = ConfigMount.mount(
      entityManager, 'hardware/sflowAccel/config',
      'Hardware::SflowAccel::Config', 'w' )
   sflowAccelHwConfig = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/hwConfig',
      'Hardware::SflowAccel::HwConfig', 'r' )
   sflowAccelStatusDir = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/status',
      'Tac::Dir', 'ri' )

   smashEntityManager = SharedMem.entityManager( sysdbEm=entityManager )
   counterMountInfo = Smash.mountInfo( 'reader' )
   sflowAccelCountersTable = smashEntityManager.doMount(
      "hardware/sflowAccel/counters",
      "Hardware::SflowAccel::Counters", counterMountInfo )
