#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This module loads hardware/modularSystem/* from Sysdb, so add a dependency on
# whatever package provides this path.
# pkgdeps: rpmwith %{_libdir}/preinit/ModularSystem

#-------------------------------------------------------------------------------
# This module implements HW sFlow acceleration configuration.
# In particular, it provides:
# - the '[no|default] sflow hardware acceleration' command
# - the '[no|default] sflow hardware acceleration module linecard <num>' command
# - the '[no|default] sflow hardware acceleration sample <rate>' command
# - the '[no|default] sflow hardware acceleration profile sflowaccel-v2' command
# - the 'show sflow hardware status' command
# - the 'show sflow hardware accelerators' command
# - the 'show sflow hardware mapping' command
# - the 'show sflow hardware counters' command
#-------------------------------------------------------------------------------

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.SflowCli as SflowCli
import CliPlugin.SflowAccelCliModel as SflowAccelCliModel
import CliPlugin.SflowLibShowCommands as SflowLibShowCommands
import CliPlugin.FruCli as FruCli
import ConfigMount
import LazyMount
import Tac
import MultiRangeRule
import ShowCommand
from TypeFuture import TacLazyType

AccelId = TacLazyType( 'Hardware::SflowAccel::AccelId' )

sflowConfig = None
sflowAccelCapabilities = None
sflowAccelConfig = None
sflowAccelHwConfig = None
sflowAccelStatusDir = None
sflowAccelFpgaMapping = None
sflowAccelRunnability = None
hwSflowAccelFpgaConfigDir = None
hwSflowAccelFpgaStatusDir = None
pciDeviceStatusDir = None
slotConfig = None
slotStatus = None

TacSflowAccelFpgaType = TacLazyType(
   'Inventory::SflowAccelFpga::SflowAccelFpgaType' )
MessageHelper = TacLazyType( 'Hardware::SflowAccel::MessageHelper' )
FeatureMessageHelper = TacLazyType( 'Hardware::SflowAccel::FeatureMessageHelper' )
TransmissionThresholdByteRange = TacLazyType(
   'Hardware::SflowAccel::TransmissionThresholdByteRange' )
TransmissionThresholdSampleRange = TacLazyType(
   'Hardware::SflowAccel::TransmissionThresholdSampleRange' )
ThresholdUnit = TacLazyType( 'Hardware::SflowAccel::ThresholdUnit' )
TransmissionThreshold = TacLazyType( 'Hardware::SflowAccel::TransmissionThreshold' )
SflowFeature = TacLazyType( 'Hardware::SflowAccel::SflowFeature' )

def sflowAccelFpgaTypeStr( fpgaType ):
   if fpgaType == TacSflowAccelFpgaType.unknown:
      return 'unknown'
   elif fpgaType == TacSflowAccelFpgaType.halo:
      return 'halo'
   elif fpgaType == TacSflowAccelFpgaType.scd:
      return 'NiagaraScd'
   elif fpgaType == TacSflowAccelFpgaType.oyster:
      return 'Oyster'
   else:
      return 'error'

def cardInserted( slotName ):
   slotFruStatus = slotStatus.slotStatus.get( slotName, None )
   return slotFruStatus and slotFruStatus.state == 'owned'

def sflowAccel2Guard( mode, token ):
   if 'sflowaccelV2' in sflowAccelCapabilities.advancedCapabilities:
      return None
   else:
      return CliParser.guardNotThisPlatform

def sflowAccelDatagramGuard( mode, token ):
   if ( SflowFeature.datagramTransmissionThreshold in
        sflowAccelCapabilities.advancedCapabilities ):
      return None
   else:
      return CliParser.guardNotThisPlatform

matcherAcceleration = CliMatcher.KeywordMatcher( 'acceleration',
      helpdesc='Enable hardware sFlow acceleration' )
nodeSflow = CliCommand.guardedKeyword( 'sflow', helpdesc='sFlow configuration',
      guard=SflowCli.sflowSupportedGuard )
nodeHardware = CliCommand.guardedKeyword( 'hardware',
      helpdesc='Configure hardware acceleration for sFlow',
      guard=SflowCli.sflowAccelGuard )
nodeHardwareForShow = CliCommand.guardedKeyword( 'hardware',
      helpdesc='Show hardware state', guard=SflowCli.sflowAccelGuard )
matcherMapping = CliMatcher.KeywordMatcher( 'mapping',
      helpdesc='Display the mapping between chip and FPGA' )
nodeDatagram = CliCommand.guardedKeyword( 'datagram',
      helpdesc='Set datagram characteristics for sFlow',
      guard=sflowAccelDatagramGuard, hidden=True )
matcherThreshold = CliMatcher.KeywordMatcher( 'threshold',
      helpdesc='Set sFlow datagram threshold values' )
matcherTransmission = CliMatcher.KeywordMatcher( 'transmission',
      helpdesc='Set transmission threshold value' )

#--------------------------------------------------------------------------------
# [ no | default ] sflow hardware acceleration
#--------------------------------------------------------------------------------
class SflowHardwareAccelerationCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow hardware acceleration'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : nodeSflow,
      'hardware' : nodeHardware,
      'acceleration' : 'Enable hardware sFlow acceleration',
   }

   @staticmethod
   def handler( mode, args ):
      sflowAccelConfig.sflowAccelEnabled = 'enabledByCli'

   @staticmethod
   def noHandler( mode, args ):
      sflowAccelConfig.sflowAccelEnabled = 'disabledByCli'

   @staticmethod
   def defaultHandler( mode, args ):
      sflowAccelConfig.sflowAccelEnabled = 'platformDefault'

BasicCli.GlobalConfigMode.addCommandClass( SflowHardwareAccelerationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sflow hardware acceleration module CARD_LIST
#--------------------------------------------------------------------------------
class SflowLcSflowAccelCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow hardware acceleration module CARD_LIST'
   noOrDefaultSyntax = syntax
   data = {
      'sflow' : nodeSflow,
      'hardware' : nodeHardware,
      'acceleration' : matcherAcceleration,
      'module' : CliCommand.guardedKeyword( 'module',
         helpdesc='Module specific configurations for hardware sFlow acceleration',
         guard=FruCli.modularSystemGuard ),
      'CARD_LIST' : MultiRangeRule.MultiRangeMatcher( 
         rangeFn=lambda: FruCli.rangeFn( 'Linecard' ),
         noSingletons=False,
         helpdesc='Linecard number',
         tagLong='Linecard' ),
   }

   @staticmethod
   def handler( mode, args ):
      for card in args[ 'CARD_LIST' ].values():
         if card in sflowAccelConfig.disableHwAccelInSlot:
            del sflowAccelConfig.disableHwAccelInSlot[ card ]

   @staticmethod
   def noHandler( mode, args ):
      for card in args[ 'CARD_LIST' ].values():
         sflowAccelConfig.disableHwAccelInSlot[ card ] = True

   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( SflowLcSflowAccelCmd )

#--------------------------------------------------------------------------------
# sflow hardware acceleration sample RATE
#--------------------------------------------------------------------------------
class SflowSampleRateCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow hardware acceleration sample RATE'
   noOrDefaultSyntax = 'sflow hardware acceleration sample ...'
   data = {
      'sflow' : nodeSflow,
      'hardware' : nodeHardware,
      'acceleration' : matcherAcceleration,
      'sample' : 'Set sample characteristics for sFlow',
      'RATE' : CliMatcher.DynamicIntegerMatcher( SflowCli.dangRateRangeFn,
         helpdesc='Rate' ),
   }

   @staticmethod
   def handler( mode, args ):
      sflowAccelConfig.sampleRate = int( args[ 'RATE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sflowAccelConfig.sampleRate = sflowAccelConfig.sampleRateDefault

BasicCli.GlobalConfigMode.addCommandClass( SflowSampleRateCmd )

#--------------------------------------------------------------------------------
# sflow hardware acceleration profile sflowaccel-v2
#--------------------------------------------------------------------------------
# Since we support only one profile for now, also guard the token 'profile' with the
# SflowAccel2 guard. Remove the guard on tokenProfile when we support more profiles.

class SflowProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow hardware acceleration profile sflowaccel-v2'
   noOrDefaultSyntax = 'sflow hardware acceleration profile ...'
   data = {
      'sflow' : nodeSflow,
      'hardware' : nodeHardware,
      'acceleration' : matcherAcceleration,
      'profile' : CliCommand.guardedKeyword( 'profile',
         helpdesc='Profile for hardware sFlow acceleration',
         guard=sflowAccel2Guard ),
      'sflowaccel-v2' : 'SflowAccel v2 profile'
   }

   @staticmethod
   def handler( mode, args ):
      sflowAccelConfig.profile = 'profileSflowAccelV2'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sflowAccelConfig.profile = sflowAccelConfig.profileDefault

BasicCli.GlobalConfigMode.addCommandClass( SflowProfileCmd )

#--------------------------------------------------------------------------------
# sflow hardware acceleration datagram transmission threshold <b> bytes
#--------------------------------------------------------------------------------

class SflowTransmissionThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'sflow hardware acceleration datagram '\
            'threshold transmission ( BYTES bytes ) | ( SAMPLES samples )'
   noOrDefaultSyntax = 'sflow hardware acceleration datagram ...'
   data = {
      'sflow' : nodeSflow,
      'hardware' : nodeHardware,
      'acceleration' : matcherAcceleration,
      'datagram' : nodeDatagram,
      'threshold' : matcherThreshold,
      'transmission' : matcherTransmission,
      'BYTES' : CliMatcher.IntegerMatcher(
         TransmissionThresholdByteRange.lo,
         TransmissionThresholdByteRange.hi,
         helpdesc='Transmission threshold value (bytes)' ),
      'bytes' : 'Threshold value in bytes',
      'SAMPLES' : CliMatcher.IntegerMatcher(
         TransmissionThresholdSampleRange.lo,
         TransmissionThresholdSampleRange.hi,
         helpdesc='Transmission threshold value (samples)' ),
      'samples' : 'Threshold value in number of samples',
   }

   @staticmethod
   def handler( mode, args ):
      if 'bytes' in args:
         unit = ThresholdUnit.bytes
         val = args[ 'BYTES' ]
      else:
         unit = ThresholdUnit.samples
         val = args[ 'SAMPLES' ]
      sflowAccelConfig.datagramTransmissionThreshold = \
         TransmissionThreshold( val, unit )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sflowAccelConfig.datagramTransmissionThreshold = \
         sflowAccelConfig.defaultDatagramTransmissionThreshold

BasicCli.GlobalConfigMode.addCommandClass( SflowTransmissionThresholdCmd )

#--------------------------------------------------------------------------------
# show sflow hardware status
#--------------------------------------------------------------------------------
def createSflowAccelModuleStatus( slotName, slotId, sflowAccelStatus ):
   moduleStatus = SflowAccelCliModel.SflowAccelModuleStatus()
   moduleStatus.sflowMode = SflowAccelCliModel.ModuleSflowMode()

   moduleStatus.sflowMode.configured = 'hardware-accelerated'
   if sflowAccelConfig.disableHwAccelInSlot.get( slotId, False ):
      moduleStatus.sflowMode.configured = 'software'

   if not sflowConfig.enabled:
      moduleStatus.sflowMode.active = 'disabled'
   elif not sflowAccelStatus.running:
      moduleStatus.sflowMode.active = 'software'
   else:
      moduleStatus.sflowMode.active = moduleStatus.sflowMode.configured

   fpgaSliceConfigDir = hwSflowAccelFpgaConfigDir.get( slotName, None )

   hasAccelerators = ( sflowAccelCapabilities.sflowAccelSupported and
                       bool( fpgaSliceConfigDir and
                             fpgaSliceConfigDir.config ) )
   hasAccelerators = hasAccelerators or sflowAccelCapabilities.sflowAccelPPSupported
   moduleStatus.hasAccelerators = hasAccelerators

   return moduleStatus

class SflowHardwareStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sflow hardware status'
   data = {
      'sflow' : SflowLibShowCommands.nodeSflow,
      'hardware' : nodeHardwareForShow,
      'status' : 'Show sFlow hardware acceleration status',
   }
   cliModel = SflowAccelCliModel.SflowAccelStatus

   @staticmethod
   def handler( mode, args ):
      statusModel = SflowAccelCliModel.SflowAccelStatus()
      LazyMount.force( sflowAccelStatusDir )
      statusHelper = Tac.newInstance( 'Hardware::SflowAccel::SflowAccelStatusHelper',
                                      sflowAccelStatusDir )
      sflowAccelStatus = statusHelper.sflowAccelStatus()
      statusModel.running = sflowAccelStatus.running
      for condition in sflowAccelRunnability.unmetCondition.values():
         descriptionStr = MessageHelper.unmetConditionDescription( condition )
         statusModel.reasonsNotRunning.append( descriptionStr )

      statusModel.sampleRate = sflowAccelHwConfig.sampleRate

      for feature in sflowAccelStatus.inactiveFeature.values():
         descriptionStr = FeatureMessageHelper.featureDescription( feature )
         statusModel.inactiveFeatures.append( descriptionStr )

      if slotConfig:
         for slotName, fruConfig in slotConfig.slotConfig.iteritems():
            if slotName.startswith( 'Linecard' ) and cardInserted( slotName ):
               moduleStatus = createSflowAccelModuleStatus( slotName,
                                                            fruConfig.slotId,
                                                            sflowAccelStatus )
               statusModel.modules[ slotName ] = moduleStatus

      return statusModel

BasicCli.addShowCommandClass( SflowHardwareStatusCmd )

#--------------------------------------------------------------------------------
# show sflow hardware accelerators
#--------------------------------------------------------------------------------
def accelPciAddress( accelHam ):
   deviceStatus = pciDeviceStatusDir.pciDeviceStatus.get( accelHam.deviceName, None )
   return deviceStatus.addr.stringValue() if deviceStatus else 'Unknown'

def chipsConnectedToAccel( accelId ):
   def chipPhysicallyConnectedToFpga( accelMapping ):
      return accelMapping.accelId == accelId and accelMapping.directLink()

   return [ chipName
            for chipName, accelMapping in
            sflowAccelFpgaMapping.chipMapping.iteritems()
            if chipPhysicallyConnectedToFpga( accelMapping ) ]

def createSflowAcceleratorInfo( sliceName, fpgaConfig ):
   accelInfo = SflowAccelCliModel.SflowAcceleratorInfo()
   accelInfo.sliceName = sliceName
   accelInfo.sflowAccelFpgaType = \
         sflowAccelFpgaTypeStr( fpgaConfig.sflowAccelFpgaType )
   accelInfo.pciAddress = accelPciAddress( fpgaConfig.ham )
   accelInfo.chips = sorted( chipsConnectedToAccel( fpgaConfig.accelId ) )
   accelInfo.accelRevision = 0
   statusDir = hwSflowAccelFpgaStatusDir.get( sliceName )
   if statusDir is not None:
      fpgaStatus = statusDir.status.get( fpgaConfig.name )
      if fpgaStatus is not None:
         accelInfo.accelRevision = fpgaStatus.fpgaRevision
   return accelInfo

class SflowHardwareAcceleratorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sflow hardware accelerators'
   data = {
      'sflow' : SflowLibShowCommands.nodeSflow,
      'hardware' : nodeHardwareForShow,
      'accelerators' : 'Show sFlow hardware accelerators',
   }
   cliModel = SflowAccelCliModel.SflowAccelerators

   @staticmethod
   def handler( mode, args ):
      sflowAcceleratorsModel = SflowAccelCliModel.SflowAccelerators()

      for sliceName, fpgaSliceConfigDir in hwSflowAccelFpgaConfigDir.iteritems():
         for fpgaName, fpgaConfig in fpgaSliceConfigDir.config.iteritems():
            accelInfo = createSflowAcceleratorInfo( sliceName, fpgaConfig )
            sflowAcceleratorsModel.accelerators[ fpgaName ] = accelInfo

      return sflowAcceleratorsModel

BasicCli.addShowCommandClass( SflowHardwareAcceleratorsCmd )

#--------------------------------------------------------------------------------
# show sflow hardware mapping
#--------------------------------------------------------------------------------
def createHwSflowMappingModel( chipToAccelMap ):
   hwSflowMappingModel = SflowAccelCliModel.SflowAcceleratorMappingInfo()
   hwSflowMappingModel.sflowAccelerator = AccelId(
      chipToAccelMap.accelId ).accelName()
   hwSflowMappingModel.directConnection = chipToAccelMap.directLink()
   return hwSflowMappingModel

class SflowHardwareMappingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sflow hardware mapping'
   data = {
      'sflow' : SflowLibShowCommands.nodeSflow,
      'hardware' : nodeHardwareForShow,
      'mapping' : matcherMapping,
   }
   cliModel = SflowAccelCliModel.HardwareSflowMapping

   @staticmethod
   def handler( mode, args ):
      showSflowMapping = SflowAccelCliModel.HardwareSflowMapping()

      for chip, chipToAccelMap in sflowAccelFpgaMapping.chipMapping.iteritems():
         showSflowMapping.chips[ chip ] = \
               createHwSflowMappingModel( chipToAccelMap )

      return showSflowMapping

BasicCli.addShowCommandClass( SflowHardwareMappingCmd )

#--------------------------------------------------------------------------------
# show sflow hardware mapping lag
#--------------------------------------------------------------------------------
def createHwSflowMappingLagModel( lagToAccelMap ):
   hwSflowMappingModel = SflowAccelCliModel.SflowAcceleratorMappingLagInfo()
   hwSflowMappingModel.sflowAccelerator = AccelId(
      lagToAccelMap.accelId ).accelName()
   return hwSflowMappingModel

class SflowHardwareMappingLagCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sflow hardware mapping lag'
   data = {
      'sflow' : SflowLibShowCommands.nodeSflow,
      'hardware' : nodeHardwareForShow,
      'mapping' : matcherMapping,
      'lag' : 'Display the mapping between LAG and FPGA',
   }

   cliModel = SflowAccelCliModel.HardwareSflowMappingLag

   @staticmethod
   def handler( mode, args ):
      showSflowMappingLag = SflowAccelCliModel.HardwareSflowMappingLag()

      for lag, lagToAccelMap in sflowAccelFpgaMapping.lagMapping.iteritems():
         showSflowMappingLag.lags[ lag ] = \
               createHwSflowMappingLagModel( lagToAccelMap )

      return showSflowMappingLag

BasicCli.addShowCommandClass( SflowHardwareMappingLagCmd )

#--------------------------------------------------------------------------------
# show sflow hardware counters
#--------------------------------------------------------------------------------
def createHardwareSflowCountersInfo( accelCounters ):
   hwSflowCountersInfo = SflowAccelCliModel.HardwareSflowCountersInfo()
   for fieldName, value in accelCounters.iteritems():
      setattr( hwSflowCountersInfo, fieldName, value )

   return hwSflowCountersInfo

class SflowHardwareCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show sflow hardware counters'
   data = {
      'sflow' : SflowLibShowCommands.nodeSflow,
      'hardware' : nodeHardwareForShow,
      'counters' : 'Display the Sflow counters',
   }

   cliModel = SflowAccelCliModel.HardwareSflowCounters

   @staticmethod
   def handler( mode, args ):
      showSflowCounter = SflowAccelCliModel.HardwareSflowCounters()

      fpgaCounters = SflowCli.sflowAccelCounters()
      for accelName, accelCounters in fpgaCounters.iteritems():
         countersInfoModel = createHardwareSflowCountersInfo( accelCounters )
         showSflowCounter.counters[ accelName ] = countersInfoModel

      return showSflowCounter

BasicCli.addShowCommandClass( SflowHardwareCountersCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed states from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global sflowConfig
   global sflowAccelCapabilities
   global sflowAccelConfig
   global sflowAccelHwConfig
   global sflowAccelStatusDir
   global sflowAccelFpgaMapping
   global sflowAccelRunnability
   global hwSflowAccelFpgaConfigDir
   global hwSflowAccelFpgaStatusDir
   global pciDeviceStatusDir
   global slotConfig
   global slotStatus

   sflowConfig = LazyMount.mount(
      entityManager, 'sflow/config',
      'Sflow::Config', 'r' )
   sflowAccelCapabilities = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/capabilities',
      'Hardware::SflowAccel::Capabilities', 'r' )
   sflowAccelConfig = ConfigMount.mount(
      entityManager, 'hardware/sflowAccel/config',
      'Hardware::SflowAccel::Config', 'w' )
   sflowAccelHwConfig = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/hwConfig',
      'Hardware::SflowAccel::HwConfig', 'r' )
   sflowAccelStatusDir = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/status', 'Tac::Dir', 'ri' )
   sflowAccelFpgaMapping = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/fpgaBalancingStatus',
      'Hardware::SflowAccel::AccelMappingStatus', 'r' )
   sflowAccelRunnability = LazyMount.mount(
      entityManager, 'hardware/sflowAccel/runnability',
      'Hardware::SflowAccel::Runnability', 'r' )
   hwSflowAccelFpgaConfigDir = LazyMount.mount(
      entityManager, 'hardware/sflowAccelFpga/config/slice',
      'Tac::Dir', 'ri' )
   hwSflowAccelFpgaStatusDir = LazyMount.mount(
      entityManager, 'hardware/sflowAccelFpga/status/slice',
      'Tac::Dir', 'ri' )
   pciDeviceStatusDir = LazyMount.mount(
      entityManager, Cell.path( 'hardware/pciDeviceStatusDir' ),
      'Hardware::PciDeviceStatusDir', 'r' )

   if Cell.cellType() == 'supervisor':
      slotConfig = LazyMount.mount( entityManager,
                                    'hardware/modularSystem/config/slot',
                                    'Hardware::ModularSystem::Config', 'r' )
      slotStatus = LazyMount.mount( entityManager,
                                    'hardware/modularSystem/status',
                                    'Hardware::ModularSystem::Status', 'r' )
