#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Model
from CliModel import Str
from IntfModels import Interface

class SfeMirrorSessions( Model ):
   class MirrorSession( Model ):
      """ Represents a MirroringSession """
      class SourceIntf( Model ):
         direction = Enum( values=( 'Rx', 'Tx', 'Both' ),
                           help="direction of traffic to be mirrored" )

      class TargetIntf( Model ):
         encapType = Str( help="Encapsulation type" )

      class SfeModule( Model ):
         class SfeSourceIntf( Model ):
            direction = Enum( values=( 'Rx', 'Tx', 'Both' ),
                              help="direction of traffic to be mirrored" )
            gate = Int( help="Output gate number" )

         class SfeTargetIntf( Model ):
            gate = Int( help="Input gate number" )

         class SfeSuccessFailCounter( Model ):
            success = Int( help="Success counter" )
            fail = Int( help="Failure counter" )

         sfeSourceIntfs = Dict( keyType=Interface, valueType=SfeSourceIntf,
                               help="Map from source interface to output gate" )
         sfeTargetIntfs = Dict( keyType=Interface, valueType=SfeTargetIntf,
                               help="Map from target interface to input gate" )

         sfeCounters = Dict( keyType=str, valueType=SfeSuccessFailCounter,
                             help="Counter for success and failure" )

      mirrorGroup = Int( help="Mirror group" )
      globalId = Int( help="Global Id" )
      srcIntfs = Dict( keyType=Interface, valueType=SourceIntf,
                      help="Map from source interface to direction" )
      tgtIntfs = Dict( keyType=Interface, valueType=TargetIntf,
                      help="Map from target interface to encapsulation type" )
      sfeModules = Dict( keyType=str, valueType=SfeModule,
                        help="Map from module name to module details" )

   sessions = Dict( keyType=str, valueType=MirrorSession,
                    help="Map from session name to session details" )

   def render( self ):
      for sessionName in self.sessions:
         session = self.sessions[ sessionName ]
         print "Session: %s, group: %d, global Id: %d" % \
            ( sessionName, session.mirrorGroup, session.globalId )

         for srcIntf in session.srcIntfs:
            direction = session.srcIntfs[ srcIntf ].direction
            print " Source interface: %s, direction: %s" % ( srcIntf, direction )

         for tgtIntf in session.tgtIntfs:
            encapType = session.tgtIntfs[ tgtIntf ].encapType
            print " Target interface: %s, encapsulation: %s" % \
               ( tgtIntf, encapType )

         for sfeModuleName in session.sfeModules:
            print " Sfe module: %s" % sfeModuleName
            sfeModule = session.sfeModules[ sfeModuleName ]
            for sfeSrcIntf in sfeModule.sfeSourceIntfs:
               direction = sfeModule.sfeSourceIntfs[ sfeSrcIntf ].direction
               gate = sfeModule.sfeSourceIntfs[ sfeSrcIntf ].gate
               print "  Source interface: %s, direction: %s, gate: %d" % \
                  ( sfeSrcIntf, direction, gate )

            for sfeTgtIntf in sfeModule.sfeTargetIntfs:
               gate = sfeModule.sfeTargetIntfs[ sfeTgtIntf ].gate
               print "  Target interface: %s, gate: %s" % ( sfeTgtIntf, gate )

            for c in sorted( sfeModule.sfeCounters.keys() ):
               print "  %s success: %u, fail: %u" % \
                  ( c, sfeModule.sfeCounters[ c ].success,
                    sfeModule.sfeCounters[ c ].fail )
