#!/usr/bin/env python
# Copyright (c) 2016-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool, Float, Int, Model, Enum, Dict, List, Submodel
from IntfModels import Interface
from ArnetModel import MacAddress
from socket import inet_aton

from TableOutput import Format, createTable

from CliModel import Str

import TableOutput
import Tac

# NOTE: This list currently duplicates the values in the SfeL3Unicast::NextHopPktCmd
# enum. Loading the enum type was causing too much slowdown in loading this plugin.
# BUG395932 tracks deduplicating this information in a way that does not cause as
# much slowdown in the CLI.
NextHopPktCmdTypes = [
      'pktCmdNone',
      'pktCmdForward',
      'pktCmdArpTrap',
      'pktCmdReceive',
      'pktCmdSlowReceive',
      'pktCmdTrapVxlanULFlood',
      'pktCmdDrop',
      'pktCmdForwardAndTrap' ]

class L3AgentSummaryUnicast( Model ):

   # Routing state
   ucastEnabledV4 = Bool( help="Status of IPV4 unicast routing" )
   ucastEnabledV6 = Bool( help="Status of IPV6 unicast routing" )

   mcastEnabledV4 = Bool( help="Status of IPV4 multicast routing" )
   mplsEnabled = Bool( help="Status of MPLS routing" )

   # Forwarding state
   forwardingMode = Enum( values=[ "defaultMode", "cutThrough",
                          "storeAndForward" ], help="Forwarding mode" )
   platformAdjSharing = Bool( help="Status of Platform Adjacency sharing" )
   routeDeletionDelay = Float( help="Route deletion delay" )

   # Unicast routes
   lpmRoutesV4 = Int( help="Number of IPV4 LPM routes" )
   lpmRoutes = Int( help="Number of IPV4 and IPV6 LPM routes" )
   lpmCreateV4 = Int( help="Number of IPV4 LPM creates" )
   lpmDeleteV4 = Int( help="Number of IPV4 LPM deletes" )
   lpmUpdateV4 = Int( help="Number of IPV4 LPM updates" )

   def render( self ):
      # Routing state
      print( "IPV4 unicast routing: %s" %
         ( "Enabled" if self.ucastEnabledV4 else "Disabled" ) )
      print( "IPV6 unicast routing: %s" %
         ( "Enabled" if self.ucastEnabledV6 else "Disabled" ) )

      print( "IPV4 multicast routing: %s\n" %
         ( "Enabled" if self.mcastEnabledV4 else "Disabled" ) )
      print( "MPLS routing: %s" % ( "Enabled" if self.mplsEnabled else "Disabled" ) )

      # Forwarding State
      print( "Forwarding mode: %s\n" % self.forwardingMode )
      print( "FIB ADJ sharing: enabled" )
      print( "Platform ADJ sharing: %s" %
            ( "enabled" if self.platformAdjSharing else "disabled" ) )
      print( "Route deletion delay: %0.0f seconds\n" % ( self.routeDeletionDelay ) )

      # LPM table
      print( "LPM IPV4 routes: %u" % ( self.lpmRoutesV4 ) )
      print( "LPM IPV4 creates: %u" % ( self.lpmCreateV4 ) )
      print( "LPM IPV4 updates: %u" % ( self.lpmUpdateV4 ) )
      print( "LPM IPV4 deletes: %u" % ( self.lpmDeleteV4 ) )

      # Unicast routes
      print( "LPM routes (IPV4 + IPV6): %u" % ( self.lpmRoutes ) )

      print()

class Counter( Model ):
   name = Str( help="Name of the counter", optional=True )
   ownerId = Int( help="ID used to look up the module owner of the counter "
                  "in the owners dictionary", optional=True )
   counterType = Enum( values=( "gatehook", "module" ), help="Counter type" )
   unit = Enum( values=( "bytes", "packets" ), help="Basic unit of the counter" )
   count = Int( help="Number of occurrences" )

class CounterModel( Model ):
   counters = Dict( keyType=int, valueType=Counter,
                    help="A list of counters used by Bessd keyed by the counterId" )
   owners = Dict( keyType=int, valueType=str,
                  help="A list of counter owners keyed by the ownerId" )

   def render( self ):
      header = [ "Name", "Owner", "Counter Type", "Unit", "Count" ]
      table = TableOutput.createTable( header )

      for ( counterId, entry ) in self.counters.items():
         name = entry.name if entry.name else "ID:%d" % counterId
         owner = self.owners.get( entry.ownerId, "-" )
         table.newRow( name, owner, entry.counterType, entry.unit, entry.count )

      print table.output()

#
# Start of Next Hop code
#

class NextHop( Model ):

   pktCmd = Enum( optional=True, help="Next Hop Type",
                  values=NextHopPktCmdTypes )
   intf = Interface( help="Egress interface" )
   ethAddr = MacAddress( help="Destination MAC address" )
   nhLocalIndex = Int( help="Next Hop Local Index" )

class FecInfo( Model ):

   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )

   fecInfos = List( valueType=NextHop,
                    help='Lists of NextHop' )

   def setAttrsFromDict( self, data ):
      self.numberOfUniqueNextHops = data[ 0 ] # 1st element in tuple
      # fecInfo is local dict, self.fecInfos is CliModel List obj.
      fecInfo = data[ 1 ]
      for _, value in fecInfo.iteritems():
         nextHop = NextHop()
         self.fecInfos.append( nextHop )
         nextHop.setAttrsFromDict( value )

class Gates( Model ):

   outputGate = Int( help="Output gate" )

   def setOutputGate( self, data ):
      self.outputGate = data

class pModNextHop( Model ):

   gate = Int( help="pModule Gate" )
   nhLocalIndex = Int( help="Next Hop Local Index" )

class FibFecInfo( Model ):

   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )

   fecInfos = List( valueType=pModNextHop,
                    help='Lists of pModNextHop' )

   def setAttrsFromDict( self, data ):
      self.numberOfUniqueNextHops = data[ 0 ] # 1st element in tuple
      # fecInfo is local dict, self.fecInfos is CliModel List obj.
      fecInfo = data[ 1 ]
      for _, value in fecInfo.iteritems():
         nextHop = pModNextHop()
         self.fecInfos.append( nextHop )
         nextHop.setAttrsFromDict( value )

def compareL2Adjs( adj1, adj2 ):
   """Compares two adjacencies. """
   return int( adj1 ) - int( adj2 )

def compareFecKeys( fecKey1, fecKey2 ):
   """Compares two fecKeys. """
   return fecKey1 - fecKey2

def compareIpPrefixAndLen( a, b ):
   """Compares two Ip addresses in this form 1.2.3.4/32 """
   slashPosA = a.prefix.find( '/' )
   addrA = a.prefix[ 0 : slashPosA ]
   slashPosB = b.prefix.find( '/' )
   addrB = b.prefix[ 0 : slashPosB ]
   numA = inet_aton( addrA )
   numB = inet_aton( addrB )
   if numA == numB:
      prefixLenA = int( a.prefix[ slashPosA + 1 : ] )
      prefixLenB = int( b.prefix[ slashPosB + 1 : ] )
      return prefixLenA - prefixLenB
   elif numA > numB:
      return 1
   else:
      return -1

def compareIpAndLen( a, b ):
   """Compares two Ip addresses in this form vrf/1.2.3.4/32 """
   slashPosA = a.find( '/' )
   slashPosEndA = a.find( '/', slashPosA + 1 )
   addrA = a[ slashPosA + 1 : slashPosEndA ]
   slashPosB = b.find( '/' )
   slashPosEndB = b.find( '/', slashPosB + 1 )
   addrB = b[ slashPosB + 1 : slashPosEndB ]
   numA = inet_aton( addrA )
   numB = inet_aton( addrB )
   if numA == numB:
      prefixLenA = int( a[ slashPosEndA + 1 : ] )
      prefixLenB = int( b[ slashPosEndB + 1 : ] )
      return prefixLenA - prefixLenB
   elif numA > numB:
      return 1
   else:
      return -1

def compareVrf( a, b ):
   slashPosA = a.find( '/' )
   vrfA = a[ 0 : slashPosA ]
   slashPosB = b.find( '/' )
   vrfB = b[ 0 : slashPosB ]
   if vrfA == vrfB:
      return 0
   elif vrfA > vrfB:
      return 1
   else:
      return -1

class Fec( Model ):

   fecKey = Int( help="FecKey associated with L2Adj" )

   def setAttr( self, data ):
      self.fecKey = data

class Fib( Model ):

   vrfName = Str( help="VrfName" )
   prefix = Str( help="Prefix" )
   fecKey = Int( help="FecKey associated with L2Adj" )
   nextHops = Str( help="Comma separated list of Next Hop(s)" )

class ModuleTable( Model ):

   nhInfos = List( valueType=NextHop,
                   help='Lists of NextHop' )

   fibs = List( valueType=Fib,
                help='Lists of Fib' )

   fecIndexes = Dict( keyType=int, valueType=FibFecInfo,
                      help='Maps a FecKey index to FibFecInfo' )

   def setNHAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = intfId.id(), v[1] = nhId, v[2] = pktCmd v[3] = macaddr
      # Sample data :('Ethernet1', '2', 0, '2e:10:ef:f6:f8:17')

      nextHop = NextHop()
      valueDict = { 'intf' : data[ 0 ], 'nhLocalIndex' : int( data[ 1 ] ),
                    'pktCmd' : data[ 2 ], 'ethAddr' : data[ 3 ] }
      nextHop.setAttrsFromDict( valueDict )
      self.nhInfos.append( nextHop )

   def setFibAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = prefix v[1] = fecKey, v[2] = vrfName, v[3] = Next Hop
      # Sample data :('5.6.7.8', 1, 'default', '10,11')

      fib = Fib()
      valueDict = { 'prefix' : data[ 0 ], 'fecKey' : data[ 1 ],
                    'vrfName' : data[ 2 ], 'nextHops' : data[ 3 ] }
      fib.setAttrsFromDict( valueDict )
      self.fibs.append( fib )

   def setFecAttrsFromTuple( self, data ):
      """ Receive a tuple and convert to dict. """

      # v[0] = fec, v[1] = tuple of (nh + gates) / could be ECMP

      self.fecIndexes[ int( data[ 0 ] ) ] = FibFecInfo()
      self.fecIndexes[ int( data[ 0 ] ) ].setAttrsFromDict( data[ 1 ].values()[ 0 ] )

   def setFecAttrsFromDiff( self, data ):
      """ Receive a tuple and convert to dict. """

      self.fecIndexes[ int( data[ 0 ] ) ] = FibFecInfo()
      shimTuple = ( 1, { '0' : { 'nhLocalIndex' : int( data[ 1 ] ), 'gate' : 0 } } )
      self.fecIndexes[ int( data[ 0 ] ) ].setAttrsFromDict( shimTuple )

class PModStatusTable( Model ):

   activeStatus = Bool( help='pModule status' )

   def setAttrsFromDict( self, data ):
      setattr( self, "activeStatus", data )

class AsyncNextHop( Model ):
   l2Adj = Int( help="L2Adj from fecToL2Adjs" )
   nhFecIndex = Int( help="Index of this nh in the NextHopKey hashmap" )
   nhLocalIndex = Int( help="Next Hop Local Index" )
   packetCmd = Enum( optional=True, help="Next Hop Type",
                  values=NextHopPktCmdTypes )
   nhIntf = Interface( help="Egress interface" )
   nhMac = MacAddress( help="Destination MAC address" )

class AsyncFecInfo( Model ):
   numberOfUniqueNextHops = Int( help="Number of Unique Next Hops" )
   nhInfos = List( valueType=AsyncNextHop,
                    help='Lists of NextHops in this fec' )

class L3AgentAsyncNextHops( Model ):
   """Replacement class for async NextHops, needed for testing"""
   fecs = Dict( keyType=int, valueType=AsyncFecInfo,
         help="Maps a FecKey index to FecInfo" )

class L3AgentNextHops( Model ):
   """Major class for NextHops."""

   # key defaults to a string (l2Adj index)
   l2Adjs = Dict( valueType=Fec,
                  help='Maps a L2Adj index to FecKey' )

   # map fec to l2Adj string
   fecToL2Adjs = Dict( keyType=int, valueType=str,
                       help='Maps a FecKey to L2Adj' )

   fecs = Dict( keyType=int, valueType=FecInfo,
                help='Maps a FecKey index to FecInfo' )

   diffNH = Bool( help='Diff Next Hop packet Module with control Module' )

   pModIntfs = Dict( keyType=Interface, valueType=PModStatusTable,
                     help='pModule interfaces' )

   pModule = Submodel( valueType=ModuleTable,
                       help='packet Module Next Hop Table' )

   cModule = Submodel( valueType=ModuleTable,
                       help='control Module Next Hop Table' )

   diffModule = Submodel( valueType=ModuleTable,
                          help='Next Hop Table Diff' )

   def convertFromPktCmd( self, pktCmd ):
      return {
         'pktCmdArpTrap' : 0,
         'pktCmdForward' : 1,
         'pktCmdDrop' : 2,
         }[ pktCmd ]

   def convertToPktCmd( self, pktCmd ):
      return {
         0 : 'pktCmdArpTrap',
         1 : 'pktCmdForward',
         2 : 'pktCmdDrop',
         # 3 : 'pktCmdFwdMirror',
         }[ pktCmd ]

   def render( self ):

      if not self.diffNH:
         table = createTable( ( "FEC", "L2ADJ", "Next Hop 1 Of N", "Next Hop",
                                "Packet CMD", "Interface", "Ethernet MAC Address" ),
                              tableWidth=9999 )
         fmt = Format( justify="left" )
         table.formatColumns( fmt, fmt, fmt, fmt, fmt, fmt, fmt )

         for key in sorted( self.fecs, cmp=compareFecKeys ):
            for index, nh in enumerate( sorted( self.fecs[ key ].fecInfos ), 1 ):
               table.newRow( str( key ),
                             self.fecToL2Adjs[ key ],
                             str( index ),
                             str( nh.nhLocalIndex ),
                             nh.pktCmd,
                             str( nh.intf ).strip( '\'' ) if nh.intf else '-',
                             nh.ethAddr )
         output = table.output()
         print "                                             SFE Control Module"
         print "                                             =================="
         print output
      else:
         #
         # Three tables
         # cModule, pModule and diffModule

         # cModule
         if self.cModule:
            cModTable = createTable( ( "Next Hop", "Packet CMD", "Interface",
                                       "Ethernet MAC Address" ),
                                     tableWidth=9999 )
            fmt = Format( justify="left" )
            cModTable.formatColumns( fmt, fmt, fmt, fmt )

            for nh in sorted( self.cModule.nhInfos ):
               cModTable.newRow( str( nh.nhLocalIndex ),
                                 nh.pktCmd,
                                 str( nh.intf ).strip( '\'' ),
                                 nh.ethAddr )
            output = cModTable.output()
            print "                            SFE Control Module"
            print "                            =================="
            print output
            print

         # pModule
         if self.pModule:
            pModTable = createTable( ( "Next Hop", "Packet CMD", "Interface",
                                       "Ethernet MAC Address" ),
                                     tableWidth=9999 )
            fmt = Format( justify="left" )
            pModTable.formatColumns( fmt, fmt, fmt, fmt )

            for nh in sorted( self.pModule.nhInfos ):
               pModTable.newRow( str( nh.nhLocalIndex ),
                                 nh.pktCmd,
                                 str( nh.intf ).strip( '\'' ),
                                 nh.ethAddr )
            output = pModTable.output()
            print "                            SFE Packet Module"
            print "                            =================="
            print output
            print

         # diffModule
         if self.diffModule:
            diffModTable = createTable(
               ( "Next Hop", "Packet CMD", "Interface", "Ethernet MAC Address" ),
               tableWidth=9999 )
            fmt = Format( justify="left" )
            diffModTable.formatColumns( fmt, fmt, fmt, fmt )

            for nh in self.diffModule.nhInfos:
               diffModTable.newRow( str( nh.nhLocalIndex ),
                                    nh.pktCmd,
                                    str( nh.intf ).strip( '\'' ),
                                    nh.ethAddr )
            output = diffModTable.output()
            print "                   Difference between pModule and cModule"
            print "                   ======================================"
            print output
            print

         #
         # Dump the pModule Interface list
         #
         if self.pModIntfs:
            pModIntfTable = createTable( ( "Interface", "Status" ), tableWidth=9999 )
            fmt = Format( justify="left" )
            pModIntfTable.formatColumns( fmt, fmt )

            for k, v in sorted( self.pModIntfs.iteritems() ):
               statusString = "Active" if v.activeStatus else "Missing"
               pModIntfTable.newRow( k, statusString )

            output = pModIntfTable.output()
            print "  pModule Interface(s)"
            print "  ===================="
            print output
            print

   def initTables( self ):
      self.cModule = ModuleTable()
      self.pModule = ModuleTable()
      self.diffModule = ModuleTable()

   def diffTables( self ):
      tableOneSet = set() # cModule
      for key in sorted( self.fecs, cmp=compareFecKeys ):
         for nh in self.fecs[ key ].fecInfos:
            if not nh.intf.stringValue:
               continue
            # Filter out Management ports
            if "Management" in nh.intf.stringValue:
               continue
            tupleEntry = ( nh.intf.stringValue,
                           str( nh.nhLocalIndex ),
                           nh.pktCmd,
                           nh.ethAddr.stringValue )
            tableOneSet.add( tupleEntry )

      # Populate cModule with all data
      if tableOneSet:
         self.cModule = ModuleTable()
         for value in tableOneSet:
            self.cModule.setNHAttrsFromTuple( value )

      # We must do a symmetric diff on the two tables

      if self.pModule:
         tableTwoSet = set()
         for nh in self.pModule.nhInfos:
            tupleEntry = ( nh.intf.stringValue,
                           str( nh.nhLocalIndex ),
                           nh.pktCmd,
                           nh.ethAddr.stringValue )
            tableTwoSet.add( tupleEntry ) # Add tuple to set
         symmetric_diff = tableOneSet ^ tableTwoSet

         # Convert back into the model List
         if symmetric_diff:
            self.diffModule = ModuleTable()
            for value in symmetric_diff:
               self.diffModule.setNHAttrsFromTuple( value ) # Add to diff table

   def setAttrsFromDict( self, data ):
      tableTwoSet = set() # pModule
      for key in data:
         if key == 'l2Adj':
            l2AdjInfo = data[ key ]
            for l2AdjIndex, value in l2AdjInfo.iteritems():
               self.l2Adjs[ l2AdjIndex ] = Fec()
               self.l2Adjs[ l2AdjIndex ].setAttr( value )
               self.fecToL2Adjs[ value ] = l2AdjIndex
         elif key == 'fec':
            fecInfo = data[ key ]
            for fecKey, value in fecInfo.iteritems():
               self.fecs[ int( fecKey ) ] = FecInfo()
               self.fecs[ int( fecKey ) ].setAttrsFromDict( value )
         elif key == 'pModIntf':
            pModIntf = data[ key ]
            for intf, value in pModIntf.iteritems():
               # Filter out Management ports
               if "Management" in intf:
                  continue
               self.pModIntfs[ intf ] = PModStatusTable()
               self.pModIntfs[ intf ].setAttrsFromDict( value )
         elif key.startswith( 'pModule' ):
            pModDict = data[ key ]
            for _, v in pModDict.iteritems():
               IntfId = Tac.Type( 'Arnet::IntfId' )
               nhIntfId = IntfId()
               nhIntfId.intfId = v[ 0 ]
               nhId = str( v[ 1 ] )
               nhPktCmd = self.convertToPktCmd( v[ 2 ] )
               updatedVal = ( nhIntfId.stringValue, nhId, nhPktCmd, v[ 3 ] )
               tableTwoSet.add( updatedVal )
         else:
            setattr( self, key, data[ key ] )

      # Populate pModule with all data
      if tableTwoSet:
         self.pModule = ModuleTable()
         for value in tableTwoSet:
            self.pModule.setNHAttrsFromTuple( value ) # send a tuple

class L3AgentSummarySoftware( Model ):

   # Route Helper
   ipVersion = Enum( values=( "IPV4", "IPV6" ), help="Version of IP" )

   # Route stats
   routeInserts = Int( help="Number of routes inserted" )
   routeDeletes = Int( help="Number of routes deleted" )
   routeUpdates = Int( help="Number of routes updated" )
   routeUpdatesIgnored = Int( help="Number of route updates ignored" )
   routeUpdatesFecChanged = Int( help="Number of route updates where Fec Changed" )
   routeUpdatesFecUpdated = Int( help="Number of route updates where Fec Updated" )

   # LPM stats
   lpmCreates = Int( help="Number of LPM creates" )
   lpmDeletes = Int( help="Number of LPM deletes" )
   lpmUpdates = Int( help="Number of LPM updates" )
   lpmUpdatesFecChanged = Int( help="Number of LPM updates where Fec Changed" )
   lpmUpdatesFecUpdated = Int( help="Number of LPM updates where Fec Updated" )

   # Adjacencies
   adjCreates = Int( help="Number of adjacency creates" )
   adjUpdates = Int( help="Number of adjacency updates" )
   adjDeletes = Int( help="Number of adjacency deletes" )

   def render( self ):
      print
      print "%s Summary" % self.ipVersion
      print " Route inserts: %u, deletes: %u, updates: %u, ignores: %u" % (
         self.routeInserts, self.routeDeletes, self.routeUpdates,
         self.routeUpdatesIgnored )

      print " Route updates fec changed: %u, updated: %u" % (
         self.routeUpdatesFecChanged, self.routeUpdatesFecUpdated )

      print " LPM creates: %u, deletes: %u, updates: %u" % (
         self.lpmCreates, self.lpmDeletes, self.lpmUpdates )
      print " LPM updates fec changed: %u, updated: %u" % (
         self.lpmUpdatesFecChanged, self.lpmUpdatesFecUpdated )

      print
      print "Adjacency Summary"
      print " ADJ creates: %u, updates: %u, deletes: %u" % (
         self.adjCreates, self.adjUpdates, self.adjDeletes )

# FibV4
class FibNextHop( Model ):

   # key defaults to a string (next hop index)
   nextHops = List( valueType=int,
                    help="nexthop local index" )

   def setAttrsFromDictForNextHop( self, data ):
      self.nextHops.append( data )

class AsyncNhLocalIndex( Model ):
   """Used only for testing"""
   nhLocalIndex = Int( help="Next Hop Local Index" )

class AsyncGate( Model ):
   vrfName = Str( help="VRF associated with this interface" )
   outputGate = Int( help="Output gate" )

class AsyncFec( Model ):
   """Used only for testing"""
   fecKey = Int( help="FecKey associated with L2Adj" )
   # List cannot have valueType=int, due to a limitation in the cpp renderer
   nhLocalIndexes = List( valueType=AsyncNhLocalIndex, help="nexthop local indexes" )

class AsyncPrefix( Model ):
   routes = Dict( keyType=str, valueType=AsyncFec, help="Maps prefix to FEC info" )

class L3AgentAsyncIpRoutes( Model ):
   """Replacement class for async IpRoutes, needed for testing"""
   vrfs = Dict( keyType=str, valueType=AsyncPrefix, help="Maps vrf to prefix" )
   # Key is vrfInterface string
   gates = Dict( keyType=str, valueType=AsyncGate,
                 help='Output Gates keyed by interface string' )

class L3AgentIpRoutes( Model ):
   """Major class for Ipv4 Routes."""

   routes = Dict( keyType=str, valueType=Fec,
                  help="Maps vrf/prefix to its FecKey" )

   fecs = Dict( keyType=int, valueType=FibFecInfo,
                help='Maps a FecKey index to FecInfo' )

   # Key is vrfInterface string
   gates = Dict( keyType=str, valueType=Gates,
                 help='Output Gates keyed by interface string' )

   diffFIB = Bool( help='Diff FIBV4 packet Module with control Module' )

   pModVrfs = Dict( keyType=str, valueType=PModStatusTable,
                    help='pModule VRFS' )

   pModuleFib = Submodel( valueType=ModuleTable,
                          help='packet Module FIBV4 Table Dump' )

   pModuleFec = Submodel( valueType=ModuleTable,
                          help='packet Module FEC Table Diff' )

   cModule = Submodel( valueType=ModuleTable,
                       help='control Module FIBV4 Table' )

   diffModuleFec = Submodel( valueType=ModuleTable,
                             help='FIBV4 (FEC) Table Diff' )

   def render( self ):

      if not self.diffFIB:
         table = createTable( ( "VRF", "Prefix", "FEC", "Next Hops" ),
                              tableWidth=9999 )

         fmt = Format( justify="left" )
         table.formatColumns( fmt, fmt, fmt, fmt )

         sortedList = sorted( self.routes.iterkeys(), cmp=compareIpAndLen )
         sortedList = sorted( sortedList, cmp=compareVrf )
         for key in sortedList:
            slashPos = key.find( '/' )
            vrfName = key[ 0 : slashPos ]
            prefix = key[ slashPos + 1 : ]
            fecKey = self.routes[ key ].fecKey

            nhs = []
            # Walk fecInfo
            for nh in sorted( self.fecs[ fecKey ].fecInfos ):
               nhs.append( str( nh.nhLocalIndex ) )
            nhsStr = ",".join( sorted( nhs ) )
            table.newRow( vrfName,
                          prefix,
                          fecKey,
                          nhsStr )
         output = table.output()
         print "                  SFE Control Module"
         print "                  =================="
         print output
         print
         print

         table = createTable( ( "Interface", "VRF", "Output Gate" ),
                              tableWidth=9999 )
         fmt = Format( justify="left" )
         table.formatColumns( fmt, fmt, fmt )

         for key in sorted( self.gates.iterkeys() ):
            slashPos = key.find( '/' )
            vrfName = key[ 0 : slashPos ]
            intf = key[ slashPos + 1 : ]
            gate = self.gates[ key ].outputGate
            table.newRow( intf,
                          vrfName,
                          gate )
         output = table.output()
         print "                  Gate Info"
         print "                  ========="
         print output
         print
         print
      else:
         #
         # Four tables
         # cModule, pModuleFib, pModuleFec and diffModuleFec

         # cModule
         if self.cModule:
            cModTable = createTable( ( "VRF", "Prefix", "FEC", "Next Hops" ),
                                     tableWidth=9999 )

            fmt = Format( justify="left" )
            cModTable.formatColumns( fmt, fmt, fmt, fmt )

            sortedList = sorted( self.cModule.fibs, cmp=compareIpPrefixAndLen )
            sortedList = sorted( sortedList, key=lambda fib: fib.vrfName )
            for fib in sortedList:
               cModTable.newRow( fib.vrfName,
                                 fib.prefix,
                                 fib.fecKey,
                                 fib.nextHops )

            output = cModTable.output()
            print "                  SFE Control Module"
            print "                  =================="
            print output
            print

         # pModuleFib
         if self.pModuleFib:
            pModFibTable = createTable(
               ( "VRF", "Prefix/Depth", "FEC" ), tableWidth=9999 )

            fmt = Format( justify="left" )
            pModFibTable.formatColumns( fmt, fmt, fmt )

            sortedList = sorted( self.pModuleFib.fibs, cmp=compareIpPrefixAndLen )
            sortedList = sorted( sortedList, key=lambda fib: fib.vrfName )
            for fib in sortedList:
               pModFibTable.newRow( fib.vrfName,
                                    fib.prefix,
                                    fib.fecKey )
            output = pModFibTable.output()
            print "      SFE Packet Module (FIB)"
            print "      ======================="
            print output
            print

         # pModuleFec
         if self.pModuleFec:
            pModFecTable = createTable( ( "FEC", "Next Hop", "Gate" ),
                                        tableWidth=9999 )
            fmt = Format( justify="left" )
            pModFecTable.formatColumns( fmt, fmt, fmt )

            for key in sorted( self.pModuleFec.fecIndexes.iterkeys() ):
               for nh in sorted( self.pModuleFec.fecIndexes[ key ].fecInfos ):
                  pModFecTable.newRow( str( key ),
                                       str( nh.nhLocalIndex ),
                                       str( nh.gate ) )
            output = pModFecTable.output()
            print "    SFE Packet Module (FEC)"
            print "    ======================="
            print output
            print

         # diffModuleFec
         if self.diffModuleFec:
            diffModFecTable = createTable( ( "FEC", "Next Hop" ),
                                           tableWidth=9999 )

            fmt = Format( justify="left" )
            diffModFecTable.formatColumns( fmt, fmt )

            for key in sorted( self.diffModuleFec.fecIndexes.iterkeys() ):
               for nh in sorted( self.diffModuleFec.fecIndexes[ key ].fecInfos ):
                  diffModFecTable.newRow( str( key ),
                                          str( nh.nhLocalIndex ) )
            output = diffModFecTable.output()

            print "Difference between pModule and cModule (FEC)"
            print "============================================"
            print output
            print

         #
         # Dump the pModule Vrf list
         #
         if self.pModVrfs:
            pModVrfTable = createTable( ( "Vrf", "Status" ), tableWidth=9999 )
            fmt = Format( justify="left" )
            pModVrfTable.formatColumns( fmt, fmt )

            for k, v in sorted( self.pModVrfs.iteritems() ):
               statusString = "Active" if v.activeStatus else "Missing"
               pModVrfTable.newRow( k, statusString )

            output = pModVrfTable.output()
            print "  pModule VRF(s)"
            print "  =============="
            print output
            print

   def initTables( self ):
      self.cModule = ModuleTable()
      self.pModuleFib = ModuleTable()
      self.pModuleFec = ModuleTable()
      self.diffModuleFec = ModuleTable()

   def diffTables( self ):

      tableOneSet = set() # cModule
      tableOneTrimSet = set() # cModule

      sortedList = sorted( self.routes.iterkeys(), cmp=compareIpAndLen )
      sortedList = sorted( sortedList, cmp=compareVrf )
      for key in sortedList:

         slashPos = key.find( '/' )
         vrfName = key[ 0 : slashPos ]
         prefix = key[ slashPos + 1 : ]
         fecKey = self.routes[ key ].fecKey

         nhs = []
         # Walk fecInfo
         for nh in sorted( self.fecs[ fecKey ].fecInfos ):
            nhs.append( str( nh.nhLocalIndex ) )
         nhsStr = ",".join( sorted( nhs ) )
         tupleEntry = ( prefix, fecKey, vrfName, nhsStr )
         tableOneSet.add( tupleEntry )

         # TRIM
         for nh in sorted( self.fecs[ fecKey ].fecInfos ):
            trimTupleEntry = ( str( fecKey ),
                               str( nh.nhLocalIndex ) )
            tableOneTrimSet.add( trimTupleEntry )

      # Populate cModule with all data
      if tableOneSet:
         self.cModule = ModuleTable()
         for value in tableOneSet:
            self.cModule.setFibAttrsFromTuple( value )

      # We must do a symmetric diff on the two tables

      if self.pModuleFec:
         tableTwoFecSet = set()

         for key in sorted( self.pModuleFec.fecIndexes.iterkeys() ):
            for nh in sorted( self.pModuleFec.fecIndexes[ key ].fecInfos ):
               tupleEntry = ( str( key ),
                              str( nh.nhLocalIndex ) )
               tableTwoFecSet.add( tupleEntry ) # Add tuple to set

         symmetric_diff = tableOneTrimSet ^ tableTwoFecSet

         # Convert back into the model List
         if symmetric_diff:
            self.diffModuleFec = ModuleTable()

            for value in symmetric_diff:
               self.diffModuleFec.setFecAttrsFromDiff( value ) # Add to diff table

   def setAttrsFromDict( self, data ):
      tableTwoFec = [] # pModule
      tableTwoFib = set() # pModule
      for key in data:
         if key == 'route':
            routeInfo = data[ key ]
            for aleVrfKey, value in routeInfo.iteritems():
               self.routes[ aleVrfKey ] = Fec()
               self.routes[ aleVrfKey ].setAttr( value )
         elif key == 'fec':
            fecInfo = data[ key ]
            for fecKey, value in fecInfo.iteritems():
               self.fecs[ int( fecKey ) ] = FibFecInfo()
               self.fecs[ int( fecKey ) ].setAttrsFromDict( value )
         elif key == 'gate':
            gateInfo = data[ key ]
            for vrfIntfKey, value in gateInfo.iteritems():
               self.gates[ vrfIntfKey ] = Gates()
               self.gates[ vrfIntfKey ].setOutputGate( value )
         elif key == 'pModVrf':
            pModVrf = data[ key ]
            for vrf, value in pModVrf.iteritems():
               self.pModVrfs[ vrf ] = PModStatusTable()
               self.pModVrfs[ vrf ].setAttrsFromDict( value )
         elif key.startswith( 'pModuleFIB' ):
            pModTupl = data[ key ]
            prefixes = []
            vrfs = []
            fecs = []

            for prefix in pModTupl[ : : 2 ]:
               prefixes.append( prefix )
            for value in pModTupl[ 1 : : 2 ]:
               vrfs.append( value[ 0 ] )
               fecs.append( value[ 1 ] )

            #
            # TableTwoFib:
            # Will be a set of FIB info used for a dump
            # of pModule FIB state and not a diff.
            # Diff is hard to do because the FIB
            # info cannot be rebuilt because of prefix len is
            # not easily reconstructed.
            #
            for index, prefix in enumerate( prefixes ):
               tableTwoFib.add( ( prefix, fecs[ index ], vrfs[ index ],
                                     "" ) )

         elif key.startswith( 'pModuleFEC' ):
            pModDict = data[ key ]

            # Test data, standard and ecmp
            # {'11': ( 10, 0 ), '1': ( 0, 1 ), '13': ( 11, 0 )}
            # ECMP
            # {'12': ( 1, 2 ), '14': ( 12, 0, 13, 1, 14, 2, 15, 3 ) }
            #pModDict = {'12': ( 1, 2 ), '14': ( 12, 0, 13, 1, 14, 2, 15, 3 ) }

            iterCount = 0
            for fec, v in pModDict.iteritems():
               iterCount = iterCount + 1
               nhCount = len( v ) / 2
               updatedVal = "(" + str( fec ) + \
                            ", {'" + str( fec ) + "':( " + str( nhCount ) + ", "
               count = len( v )
               for i in range( 0, count, 2 ):
                  updatedVal += "{ '" + str( iterCount ) + \
                                "': { 'nhLocalIndex':" + \
                                str( v[ i ] ) + \
                                ", 'gate':" + str( v[ i + 1 ] ) + "} }, "
               updatedVal += "),})"
               flyTupl = eval( updatedVal )
               tableTwoFec.append( flyTupl )
         else:
            setattr( self, key, data[ key ] )

      # Populate pModule with all data
      if tableTwoFib:
         self.pModuleFib = ModuleTable()
         for value in tableTwoFib:
            self.pModuleFib.setFibAttrsFromTuple( value ) # send a tuple

      if tableTwoFec:
         self.pModuleFec = ModuleTable()
         for value in tableTwoFec:
            self.pModuleFec.setFecAttrsFromTuple( value ) # send a tuple
