#!/usr/bin/env python
# Copyright (c) 2017-2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliMatcher import KeywordMatcher
from CliMatcher import IntegerMatcher
import AgentCommandRequest
import cStringIO
import ast
import SfeAgent
from SfeL3UnicastShowCliModel import L3AgentSummaryUnicast
from SfeL3UnicastShowCliModel import L3AgentSummarySoftware
from SfeL3UnicastShowCliModel import L3AgentNextHops
from SfeL3UnicastShowCliModel import L3AgentAsyncNextHops
from SfeL3UnicastShowCliModel import L3AgentIpRoutes
from SfeL3UnicastShowCliModel import L3AgentAsyncIpRoutes

import ShowCommand
import CliCommand
import CliModel
from SfeCliLib import nodeSfe, notHwPlatform
from CliToken.Platform import platformMatcherForShow
from IraIpRouteCliLib import prefixMatcher
import CliPlugin.VrfCli as VrfCli

matcherDiff = CliCommand.guardedKeyword( 'diff', helpdesc='Display cModule diff',
                                         guard=notHwPlatform )
matcherL3 = KeywordMatcher( 'l3', 'Display routing information' )
matcherSummary = KeywordMatcher( 'summary', 'Display L3 software summary' )
matcherFecId = IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF, helpdesc='FEC identifier' )

#---------------------------------------------------------------------------------
# show platform sfe ip route [ vrf <vrfName> [ <prefix> ] ] [ diff ]
#---------------------------------------------------------------------------------

def doShowIpRouteVrf( mode, args ):
   buff = cStringIO.StringIO()
   cmd = "SIPRD"
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", cmd, stringBuff=buff, timeout=50,
                                         keepalive=True )
   try:
      siprDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return

   sipr = L3AgentIpRoutes()
   #
   # Need these to prevent CliModel warning
   #
   sipr.initTables()
   sipr.setAttrsFromDict( siprDict )
   sipr.diffTables()

   return sipr

class PlatformSfeIpRouteDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe ip route diff'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : 'IPV4 related commands',
      'route' : 'Show hardware routing table',
      'diff' : matcherDiff,
   }

   handler = doShowIpRouteVrf
   cliModel = L3AgentIpRoutes
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeIpRouteDiffCmd )

class PlatformSfeIpRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe ip route [ VRF [ PREFIX ] ]'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'ip' : 'IPV4 related commands',
      'route' : 'Show hardware routing table',
      'VRF' : VrfCli.VrfExprFactory( helpdesc='Show VRF routing table' ),
      'PREFIX' : prefixMatcher,
   }

   cliModel = L3AgentAsyncIpRoutes
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      cmd = "SIPRN+"

      if 'VRF' in args:
         cmd += "VRF+" + args[ 'VRF' ] + "+"

      if 'PREFIX' in args:
         cmd += "PREFIX+" + args[ 'PREFIX' ] + "+"
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                                outputFormat=outputFormat,
                                                throwException=True,
                                                errors=[],
                                                asyncCommand=True )
         return CliModel.cliPrinted( L3AgentAsyncIpRoutes )
      except AgentCommandRequest.RunSocketCommandException, e:
         mode.addError( str( e ) )

BasicCli.addShowCommandClass( PlatformSfeIpRouteCmd )




#---------------------------------------------------------------------------------
# show platform sfe l3 unicast summary
#---------------------------------------------------------------------------------
class PlatformSfeL3UnicastSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 unicast summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'unicast' : 'Display unicast routing state',
      'summary' : matcherSummary,
   }

   cliModel = L3AgentSummaryUnicast
   privileged = True

   @staticmethod
   def handler( mode, args ):
      outputFormat = mode.session_.outputFormat_
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               SfeAgent.name(),
                                               "SfeCliWithFormat",
                                               "SUMM",
                                               stringBuff=None,
                                               outputFormat=outputFormat,
                                               throwException=True,
                                               errors=[] )
         return CliModel.cliPrinted( L3AgentSummaryUnicast )
      except AgentCommandRequest.RunSocketCommandException, e:
         mode.addError( str( e ) )

BasicCli.addShowCommandClass( PlatformSfeL3UnicastSummaryCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 software summary
#---------------------------------------------------------------------------------
def doShowL3SummarySoftware( mode, args ):
   buff = cStringIO.StringIO()
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "SSus", stringBuff=buff )
   try:
      summDict = ast.literal_eval( buff.getvalue() )
   except SyntaxError:
      mode.addError( buff.getvalue() )
      return
   summ = L3AgentSummarySoftware()
   summ.setAttrsFromDict( summDict )
   return summ

class PlatformSfeL3SoftwareSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 software summary'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'software' : 'Display L3 software state',
      'summary' : matcherSummary,
   }

   handler = doShowL3SummarySoftware
   cliModel = L3AgentSummarySoftware
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeL3SoftwareSummaryCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 next-hops [ diff ]
#---------------------------------------------------------------------------------
class PlatformSfeL3NextHopsDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 next-hops diff'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'next-hops' : 'Show next-hop entries',
      'diff' : matcherDiff,
   }

   cliModel = L3AgentNextHops
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cmd = "L3NHD"
      keep = True
      buff = cStringIO.StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", cmd, stringBuff=buff, timeout=50,
                                         keepalive=keep )
      try:
         nhDict = ast.literal_eval( buff.getvalue() )
      except SyntaxError:
         mode.addError( buff.getvalue() )
         return

      l3Nh = L3AgentNextHops()
      #
      # Need these to prevent CliModel warning
      #
      l3Nh.initTables()

      l3Nh.setAttrsFromDict( nhDict )
      l3Nh.diffTables()

      return l3Nh

BasicCli.addShowCommandClass( PlatformSfeL3NextHopsDiffCmd )

#---------------------------------------------------------------------------------
# show platform sfe l3 next-hops [ FECID ]
#---------------------------------------------------------------------------------

class PlatformSfeL3NextHopsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe l3 next-hops [ FECID ]'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'l3' : matcherL3,
      'next-hops' : 'Show next-hop entries',
      'FECID' : matcherFecId,
   }

   cliModel = L3AgentAsyncNextHops
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cmd = "L3NH"
      fecId = args.get( 'FECID' )
      if fecId:
         cmd += '#F' + str( fecId )
      outputFormat = mode.session_.outputFormat_
      try:
         AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                               "SfeCliWithFormat", cmd,
                                                outputFormat=outputFormat,
                                                throwException=True, errors=[],
                                                asyncCommand=True )
         return CliModel.cliPrinted( L3AgentAsyncNextHops )
      except AgentCommandRequest.RunSocketCommandException, e:
         mode.addError( str( e ) )

BasicCli.addShowCommandClass( PlatformSfeL3NextHopsCmd )
