#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=protected-access

from __future__ import absolute_import, division, print_function
from ArnetModel import (
   IpGenericAddress,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from FlowTrackerCliUtil import (
   ftrTypeHardware,
   ftrTypeShowStr,
   addressStr,
   protocolStr,
   timeStr,
)
from TypeFuture import TacLazyType
import Tac
import TacSigint
import TableOutput

IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )
FtConsts = TacLazyType( 'FlowTracking::Constants' )

def pindent( indent, *args ):
   print( ' ' * ( 2 * indent - 1 ), *args )

def templateIdNum( staticTemplateIdEnum ):
   return FtConsts.reservedTemplateId( staticTemplateIdEnum )

class FlowKeyModel( Model ):
   vrfName = Str( help="VRF name" )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes,
                      optional=True )
   ipProtocolNumber = Int( help="IP protocol number" )
   srcPort = Int( help="Source port" )
   dstPort = Int( help="Destination port" )
   _combinedKey = Int( help="Combined flow key" )
   _direction = Enum( values=( 'fcRight', 'fcLeft' ),
                     help="Combined flow key direction" )

class FlowDetailModel( Model ):
   tcpFlags = Str( help="Flow TCP flags" )
   lastPktTime = Float( help="Last packet received time" )

class FlowModel( Model ):
   key = Submodel( valueType=FlowKeyModel, help="Flow key" )
   bytesReceived = Int( help="Number of bytes received" )
   pktsReceived = Int( help="Number of packets received" )
   startTime = Float( help="Flow start time" )
   flowDetail = Submodel( valueType=FlowDetailModel, optional=True,
                          help="Flow detailed information" )
   _exportedTime = Float( help="Time last exported to collector" )
   _exportedBytes = Int( help="Number of bytes exported to collector" )
   _exportedPkts = Int( help="Number of packet exported to collector" )

class GroupModel( Model ):
   flows = List( valueType=FlowModel, help="List of flows" )

class TrackerModel( Model ):
   groups = Dict( keyType=str, valueType=GroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

class TrackingModel( Model ):
   _detail = Bool( 'Print info in details' )
   trackers = Dict( keyType=str, valueType=TrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Flow tracking is active" )
   softwareFlowTable = Bool( help="Software flow table is available" )

   def renderDetail( self ):
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( "Tracker: {}, Flows: {}".format( trackerName, tracker.numFlows ) )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue
            print( "  Group: {}, Flows: {}".format( groupName,
                                                    len( group.flows ) ) )
            for flow in group.flows:
               key = flow.key
               detail = flow.flowDetail
               tab = " " * 4
               print( "{}Flow: {} {} - {}, VRF: {}".format(
                  tab,
                  protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                  addressStr( key.srcAddr, key.srcPort ),
                  addressStr( key.dstAddr, key.dstPort ), key.vrfName ) )
               tab = " " * 6
               print( "{}Combined Key: {} Direction: {}".format(
                  tab, str( key._combinedKey ), str( key._direction ) ) )
               print( "{}Start time: {}, Last packet time: {}".format(
                  tab,
                  timeStr( flow.startTime ), timeStr( detail.lastPktTime ) ) )
               print( "{}Packets: {}, Bytes: {}, "
                      "TCP Flags: {}".format( tab, flow.pktsReceived,
                                              flow.bytesReceived,
                                              detail.tcpFlags ) )
               print( "{}Last exported time: {}, Packets: {}, Bytes: {}".
                      format( tab, timeStr( flow._exportedTime ),
                              flow._exportedPkts,
                              flow._exportedBytes ) )
               TacSigint.check()

   def renderSummary( self ):
      headings = ( "VRF", "Source", "Destination", "Protocol",
                   "Start Time", "Pkts", "Bytes" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )
      for trackerName, tracker in sorted( self.trackers.items() ):
         print( "Tracker: {}, Flows: {}".format( trackerName, tracker.numFlows ) )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue
            table = TableOutput.createTable( headings )
            table.formatColumns( formatLeft, formatLeft, formatLeft,
                                 formatCenter, formatRight, formatRight,
                                 formatRight )
            print( "Group: {}, Flows: {}".format( groupName,
                                                  len( group.flows ) ) )
            for flow in group.flows:
               key = flow.key
               table.newRow( key.vrfName,
                             addressStr( key.srcAddr, key.srcPort ),
                             addressStr( key.dstAddr, key.dstPort ),
                             protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                             timeStr( flow.startTime ),
                             flow.pktsReceived,
                             flow.bytesReceived )
               TacSigint.check()
            print( table.output() )

   def render( self ):
      ftrStr = ftrTypeShowStr[ ftrTypeHardware ]
      if not self.running:
         print( ftrStr, 'flow tracking is not active' )
         return

      if self._detail:
         self.renderDetail()
      else:
         self.renderSummary()

class PlatformCountersModel( Model ):
   """ Represents sfeFtShowCounters """
   grpcErr = Int( help="Number of errors in GRPC processing",
                  default=0 )
   flowAdd = Int( help="Number flow added", default=0 )
   flowDel = Int( help="Number flow deleted", default=0 )

   exportInfoReq = Int( help="Number of export GRPC "
                        "request message sent", default=0 )
   exportInfoRsp = Int( help="Number of export GRPC response "
                        "message received", default=0 )
   exportInfoInRspAdd = Int( help="Number of add export message received",
                             default=0 )
   exportInfoInRspRemove = Int( help="Number of remove export message received",
                                default=0 )
   exportInfoInRspRemoveNoFlow = Int( help="Number of remove export message "
                                      "received for invalid flow", default=0 )
   exportInfoRspInvalidFt = Int( help="Number of export received "
                                 "for invalid flow traker", default=0 )
   exportInfoRspInvalidIntf = Int( help="Number of export received "
                                   "for invalid flow tracker interface", default=0 )
   exportInfoRspNoModule = Int( help="Number of export received "
                                "for invalid flow tracker module", default=0 )

   flowUdReq = Int( help="Number of user data GRPC request sent", default=0 )
   flowUdRsp = Int( help="Number of user data GRPC response received",
                    default=0 )
   flowKeyUdInReq = Int( help="Number of user data requested", default=0 )
   flowKeyUdInRsp = Int( help="Number of user data response received",
                         default=0 )
   flowKeyUdInRspNotFound = Int( help="Number of user data response not found",
                                 default=0 )
   flowKeyUdInRspNoFlow = Int( help="Number of user data response for "
                               "invalid flow", default=0 )
   flowKeyUdInRspNoPktSeen = Int( help="Number of user data response "
                                  "with no update", default=0 )
   flowKeyUdInRspInvalidFt = Int( help="Number of user data response "
                                 "for invalid flow tracker", default=0 )
   flowKeyUdInRspInvalidIntf = Int( help="Number of user data response "
                                    "for invalid flow tracker interface", default=0 )
   flowKeyUdInRspNoModule = Int( help="Number of user data response "
                                "for invalid flow tracker module", default=0 )

   exportQueueWriteUnavailable = Int(
      help="Number of times export queue write unavailable",
      default=0 )
   exportQueuePushAttempt = Int( help="Number of export record enqueue attempted",
                                 default=0 )
   exportQueuePushFail = Int( help="Number of export record enqueue failed",
                              default=0 )

   getflowUd = Int( help="Number of BESS get user data requests", default=0 )
   getExportInfo = Int( help="Number of BESS get export info requests", default=0 )

   fcudPoolSize = Int( help="Size of BESS flow-cache user data pool", default=0 )
   fcudPoolGet = Int( help="Number of BESS flow-cache user data pool allocation",
                      default=0 )
   fcudPoolPut = Int( help="Number of BESS flow-cache user data pool "
                      "deallocation", default=0 )
   fcudPoolFull = Int( help="Number of BESS flow-cache user data pool allocation "
                       "failure", default=0 )
   fcudPoolAvailable = Int( help="Number of BESS flow-cache user data available",
                            default=0 )
   fcudPoolInuse = Int( help="Number of BESS flow-cache user data in use",
                        default=0 )

   exportRingSize = Int( help="Size of BESS export ring", default=0 )
   exportRingInuse = Int( help="Number of export ring data in use", default=0 )
   exportRingAddEnqFail = Int( help="Number of BESS export ring add enqueue "
                               "failure", default=0 )
   exportRingDelEnqFail = Int( help="Number of BESS export ring delete enqueue "
                               "failure", default=0 )

   exportfd = Int( help="Export ring BESS file descriptor", default=0 )
   exportfdReq = Int( help="Number of export ring read indication", default=0 )
   exportfdReqFail = Int( help="Number of export ring read indication failure",
                          default=0 )

   def render( self ):
      print ( "Number of flows added: %u" % self.flowAdd )
      print ( "Number of flows deleted: %u" % self.flowDel )

      print ( "Number of errors in GRPC processing: %u" % self.grpcErr )
      print ( "Number of export GRPC request message sent: %u" %
              self.exportInfoReq )
      print ( "Number of export GRPC response message received: %u" %
              self.exportInfoRsp )
      print( "Number of add export message received: %u" %
             self.exportInfoInRspAdd )
      print( "Number of remove export message received: %u" %
             self.exportInfoInRspRemove )
      print( "Number of remove export message received "
             "for non existing flow: %u" % self.exportInfoInRspRemoveNoFlow )
      print( "Number of export received for invalid flow tracker: %u" %
         self.exportInfoRspInvalidFt )
      print( "Number of export received for "
              "invalid flow tracker interface : %u" % self.exportInfoRspInvalidIntf )
      print( "Number of export received for invalid flow tracker "
         "module: %u" % self.exportInfoRspNoModule )

      print( "Number of user data GRPC request sent: %u" % self.flowUdReq )
      print( "Number of user data GRPC response received: %u" % self.flowUdRsp )
      print( "Number of user data requested: %u" % self.flowKeyUdInReq )
      print( "Number of user data response received: %u" % self.flowKeyUdInRsp )
      print( "Number of user data response not found: %u" %
             self.flowKeyUdInRspNotFound )
      print( "Number of user data response for invalid flow: %u" %
             self.flowKeyUdInRspNoFlow )
      print( "Number of user data response with no update: %u" %
             self.flowKeyUdInRspNoPktSeen )
      print( "Number of user data response for invalid flow tracker: %u" %
             self.flowKeyUdInRspInvalidFt )
      print( "Number of user data response for invalid "
             "flow tracker interface: %u" % self.flowKeyUdInRspInvalidIntf )
      print( "Number of user data response for invalid "
             "flow tracker module: %u" % self.flowKeyUdInRspNoModule )

      print( "Number of times export queue write unavailable: %u" %
             self.exportQueueWriteUnavailable )
      print( "Number of export record enqueue attempts: %u" %
             self.exportQueuePushAttempt )
      print( "Number of export record enqueue failed: %u" %
             self.exportQueuePushFail )

      print( "Number of BESS get user data requests: %u" % self.getflowUd )
      print( "Number of BESS get export info requests: %u" % self.getExportInfo )

      print( "Size of BESS flow-cache user data pool: %u" % self.fcudPoolSize )
      print( "Number of BESS flow-cache user data pool allocation: %u" %
             self.fcudPoolGet )
      print( "Number of BESS flow-cache user data pool deallocation: %u" %
             self.fcudPoolPut )
      print( "Number of BESS flow-cache user data pool allocation failure: %u" %
             self.fcudPoolFull )
      print( "Number of BESS flow-cache user data available: %u" %
             self.fcudPoolAvailable )
      print( "Number of BESS flow-cache user data in use: %u" %
             self.fcudPoolInuse )

      print( "Size of BESS export ring: %u" % self.exportRingSize )
      print( "Number of BESS export ring data in use: %u" % self.exportRingInuse )
      print( "Number of BESS export ring add enqueue failure: %u" %
             self.exportRingAddEnqFail )
      print( "Number of BESS export ring delete enqueue failure: %u" %
             self.exportRingDelEnqFail )

      print( "Export ring BESS file descriptor: %u" % self.exportfd )
      print( "Number of BESS export ring read indication: %u" %
             self.exportfdReq )
      print( "Number of BESS export ring read indication failure: %u" %
             self.exportfdReqFail )
