#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMatcher import KeywordMatcher
from CliToken.Platform import (
      platformMatcherForClear,
      platformMatcherForConfig,
      platformMatcherForShow,
)
from CliToken.Clear import clearKwNode
from CliToken.Trace import matcherTrace
from CliToken.Ip import ipMatcherForConfig
import ConfigMount
import SmashLazyMount
import LazyMount
import Tac
import SfeAgent
from QosTypes import tacRateUnit
from SfeL3UnicastShowCliModel import Counter
from SfeL3UnicastShowCliModel import CounterModel
from SfeCliLib import nodeSfe, nodeSfeEos, sfe
import ShowCommand

bessCliConfig = None
bessCounterDb = None
bessCounterInfo = None
bessCounterSnapshot = None
bessdStatus = None

#------------------------------------------------------------------------------
# The "platform sfe debug trace" hidden command, in "config" mode.
#-------------------------------------------------------------------------------
nodeDebug = CliCommand.Node(
            matcher=KeywordMatcher( 'debug',
                    helpdesc='Bess debugging setting commands' ),
            hidden=True )

def unsetTrace( mode, args ):
   bessCliConfig.traceConfig = ''

def setTrace( mode, args ):
   traceConfig = args[ 'TRACECONFIG' ]
   bessCliConfig.traceConfig = traceConfig

class PlatformSfeDebugTraceCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe debug trace TRACECONFIG'
   noOrDefaultSyntax = 'platform sfe debug trace ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'debug' : nodeDebug,
      'trace' : matcherTrace,
      'TRACECONFIG' : CliMatcher.PatternMatcher( pattern='.+',
                      helpdesc='Trace config in the format of FACILITY/LEVELS,...',
                      helpname='EXPR' ),
   }

   handler = setTrace
   noOrDefaultHandler = unsetTrace

BasicCliModes.GlobalConfigMode.addCommandClass( PlatformSfeDebugTraceCmd )

#------------------------------------------------------------------------------
# The "ip load-sharing sfe hash funcId" command, in "config" mode.
#-------------------------------------------------------------------------------
def setTableIndex( mode, args ):
   bessCliConfig.hashFuncId = args.get( 'FUNC', 0 )

matcherLoadSharing = CliMatcher.KeywordMatcher( 'load-sharing',
        helpdesc='Configure ECMP route selection parameters' )

matcherHash = CliMatcher.KeywordMatcher( 'hash',
        helpdesc='Configure hash function used for load balancing' )

class SfeLoadSharingConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ip load-sharing sfe hash FUNC'
   noOrDefaultSyntax = 'ip load-sharing sfe hash ...'
   data = {
      'ip' : ipMatcherForConfig,
      'load-sharing' : matcherLoadSharing,
      'sfe' : nodeSfe,
      'hash' : matcherHash,
      'FUNC' : CliMatcher.IntegerMatcher( 0, 3, helpdesc='hash function ID' ),
   }

   handler = setTableIndex

   # noOrDefault syntax wil ignore the FUNC and anything after that.
   # hence hanlder will return 0
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( SfeLoadSharingConfigCmd )

#------------------------------------------------------------------------------
# The "platform sfe debug vlog" hidden command, in "config" mode.
#-------------------------------------------------------------------------------
def unsetVlog( mode, args ):
   bessCliConfig.vlogConfig = ''

def setVlog( mode, args ):
   vlogConfig = args[ 'VLOGCONFIG' ]
   bessCliConfig.vlogConfig = vlogConfig

class PlatformSfeDebugVlogCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe debug vlog VLOGCONFIG'
   noOrDefaultSyntax = 'platform sfe debug vlog ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'debug' : nodeDebug,
      'vlog' : 'Enable Bess VLog facility',
      'VLOGCONFIG' : CliMatcher.PatternMatcher( pattern='.+',
                     helpdesc='Vlog config in the format of FACILITY=LEVEL,...',
                     helpname='EXPR' ),
   }

   handler = setVlog
   noOrDefaultHandler = unsetVlog

BasicCliModes.GlobalConfigMode.addCommandClass( PlatformSfeDebugVlogCmd )

#---------------------------------------------------------------------------------
# show platform sfe qos
#---------------------------------------------------------------------------------
def doShowSfeQos( mode, args ):
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "QOS" )

class PlatformSfeQosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe qos'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'qos' : 'Show qos info',
   }

   handler = doShowSfeQos
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeQosCmd )

#--------------------------------------------------------------------------------
# show platform sfe counters [ ( module | module-class | type ) [ RULE ] ]
#--------------------------------------------------------------------------------
matcherCounters = KeywordMatcher( 'counters', 'Bess debug counters' )
matcherCounterMatchRule = CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
                          helpdesc='Counter matching rule', helpname='WORD' )
matcherCounterMatchType = CliMatcher.EnumMatcher( {
                          'module' : 'Name of the module holding the counter',
                          'module-class' : 'Module class of the counter owner',
                          'type' : 'Counter type' } )

def showBessCounter( mode, args ):
   matchingType = args.get( 'MATCHTYPE' )
   if matchingType == 'module-class':
      matchingType = 'moduleClass'
   matchingRule = args.get( 'RULE' )

   counterDict = {}
   ownerDict = {}
   counterOwnerInfoInvMap = \
      { v : k for k, v in bessCounterInfo.counterOwnerInfo.iteritems() }

   for ( counterId, counterInfo ) in bessCounterInfo.counterInfo.items():
      if counterInfo.hide: # hidden counters are skipped in show
         continue
      ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
      if not matchingRule or \
         ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
         ( matchingRule == getattr( ownerInfo, matchingType, None ) ):

         smashCount = bessCounterDb.smashCount.get( counterId )
         if smashCount is None:
            continue

         snapshotCount = 0
         snapshotCounter = bessCounterSnapshot.counterCopy.get( counterId )
         if snapshotCounter and snapshotCounter.timestamp > counterInfo.timestamp:
            snapshotCount = snapshotCounter.count
         counter = Counter( counterType=counterInfo.type,
                            unit=counterInfo.unit,
                            count=smashCount.count - snapshotCount )
         if counterInfo.name:
            counter.name = counterInfo.name

         if ownerInfo:
            counter.ownerId = counterInfo.ownerId
            ownerDict.setdefault( counterInfo.ownerId, ownerInfo.module )

         counterDict[ counterId ] = counter

   return CounterModel( counters=counterDict, owners=ownerDict )

class PlatformSfeCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe counters [ MATCHTYPE [ RULE ] ]'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'counters' : matcherCounters,
      'MATCHTYPE' : matcherCounterMatchType,
      'RULE' : matcherCounterMatchRule,
   }
   hidden = not sfe()

   handler = showBessCounter
   cliModel = CounterModel

BasicCli.addShowCommandClass( PlatformSfeCountersCmd )

#---------------------------------------------------------------------------------
# clear platform sfe counters [ ( module | module-class | type ) [ RULE ] ]
#---------------------------------------------------------------------------------
def clearBessCounter( mode, args ):
   matchingType = args.get( 'MATCHTYPE' )
   if matchingType == 'module-class':
      matchingType = 'moduleClass'
   matchingRule = args.get( 'RULE' )

   counterOwnerInfoInvMap = \
      { v : k for k, v in bessCounterInfo.counterOwnerInfo.iteritems() }

   # walk through current counter to do snapshot
   for ( counterId, counterInfo ) in bessCounterInfo.counterInfo.items():
      ownerInfo = counterOwnerInfoInvMap.get( counterInfo.ownerId )
      if not matchingRule or \
         ( matchingRule == getattr( counterInfo, matchingType, None ) ) or \
         ( matchingRule == getattr( ownerInfo, matchingType, None ) ):

         smashCount = bessCounterDb.smashCount.get( counterId )
         if smashCount is None or smashCount.count == 0:
            continue

         bessCounterSnapshot.counterCopy[ counterId ] = \
            Tac.Value( "Sfe::BessCounterCopy", count=smashCount.count,
                       timestamp=Tac.now() )

   # walk through snapshot to cleanup stale counterIds
   for ( counterId, counterCopy ) in bessCounterSnapshot.counterCopy.items():
      counterInfo = bessCounterInfo.counterInfo.get( counterId )
      if counterInfo is None or counterInfo.timestamp >= counterCopy.timestamp:
         del bessCounterSnapshot.counterCopy[ counterId ]

class ClearPlatformSfeCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear platform sfe counters [ MATCHTYPE [ RULE ] ]'
   data = {
      'clear' : clearKwNode,
      'platform' : platformMatcherForClear,
      'sfe' : nodeSfeEos,
      'counters' : matcherCounters,
      'MATCHTYPE' : matcherCounterMatchType,
      'RULE' : matcherCounterMatchRule,
   }
   hidden = not sfe()

   handler = clearBessCounter

BasicCliModes.EnableMode.addCommandClass( ClearPlatformSfeCountersCmd )

#---------------------------------------------------------------------------------
# clear platform sfe counters session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters module [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters module-class [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# clear platform sfe counters type [ < WORD > ] session
#---------------------------------------------------------------------------------
# TODO

#---------------------------------------------------------------------------------
# show platform sfe licensing
#---------------------------------------------------------------------------------
def doShowSfeLicensing( mode, args ):
   AgentCommandRequest.runSocketCommand( mode.entityManager, SfeAgent.name(),
                                         "sfe", "LICENSE" )

class PlatformSfeLicensingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe licensing'
   data = {
      'platform' : platformMatcherForShow,
      'sfe' : nodeSfe,
      'licensing' : 'Show licensing info',
   }

   handler = doShowSfeLicensing
   privileged = True

BasicCli.addShowCommandClass( PlatformSfeLicensingCmd )

def Plugin( entityManager ):
   global bessCliConfig, bessCounterDb, bessCounterInfo, bessCounterSnapshot
   global bessdStatus

   bessCliConfig = ConfigMount.mount(
      entityManager, "bess/cli/config", "Sfe::BessdCliConfig", "w" )
   bessCounterDbMountInfo = SmashLazyMount.mountInfo( 'reader' )
   bessCounterDb = SmashLazyMount.mount( entityManager, "bess/counter",
                                         "SfeModules::SmashCounterDbStatus",
                                         bessCounterDbMountInfo )
   bessCounterInfo = LazyMount.mount( entityManager, "bess/counter/info",
                                      "Sfe::BessCounterInfo", "r" )
   bessCounterSnapshot = LazyMount.mount( entityManager, "bess/cli/counter/snapshot",
                                          "Sfe::BessCounterSnapshot", "wr" )
   bessdStatus = LazyMount.mount( entityManager, "bess/bessd/status",
                                    "Sfe::BessdStatus", "r" )

#------------------------------------------------------------------------------
# [ no | default ] paltform sfe control-plane receive-limit <value> <kbps|mbps>
#-------------------------------------------------------------------------------
def cpuReceivelimit( mode, args ):
   receiveLimitValue = args[ 'LIMIT' ]
   if 'kbps' in args:
      receiveLimitUnit = tacRateUnit.rateUnitKbps
   else:
      receiveLimitUnit = tacRateUnit.rateUnitMbps
   bessCliConfig.cpuReceiveLimit = Tac.Value(
         "Sfe::CpuReceiveLimitConfig", receiveLimitValue, receiveLimitUnit )

def noCpuReceivelimit( mode, args ):
   bessCliConfig.cpuReceiveLimit = Tac.Value( "Sfe::CpuReceiveLimitConfig" )

class PlatformSfeControlPlaneReceiveLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'platform sfe control-plane receive-limit LIMIT ( kbps | mbps )'
   noOrDefaultSyntax = 'platform sfe control-plane receive-limit ...'
   data = {
      'platform' : platformMatcherForConfig,
      'sfe' : nodeSfe,
      'control-plane' : 'Set control plane configuration',
      'receive-limit' : 'Set the rate limit for control plane traffic',
      'LIMIT' : CliMatcher.IntegerMatcher( 1, 10000000,
                                           helpdesc='Rate limit' ),
      'kbps' : 'Rate limit unit kbps',
      'mbps' : 'Rate limit unit mbps',
   }

   handler = cpuReceivelimit
   noOrDefaultHandler = noCpuReceivelimit

BasicCliModes.GlobalConfigMode.addCommandClass(
      PlatformSfeControlPlaneReceiveLimitCmd )
