#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import threading

import Ark
import BasicCli
import BasicCliSession
import ConfigMount
import ConfigSessionCommon
import CliCommand
import CliMatcher
import CliParser
import CliSession
import CliToken.Configure
import CliToken.Service
import CliToken.Terminal
import FileCliUtil
import LazyMount
import Tac
import Tracing
import Url
import DateTimeRule

t0 = Tracing.trace0

cliSessionStatus = None
cliConfig = None
checkpointFunc = None
sessionLock = threading.RLock()

# Minimal config session commands, needed by Cli. The rest of the Clis are
# implemented in ConfigSession/CliPlugin.
# configure session [<name> [description <description>]]
# commit [ timer <hours:minutes:seconds> ]
# abort [ name ]
# rollback clean-config | [ name ]
#  name is name of a previously committed session,
# rollback with name also does a clean-config first, in order
#  to add all the roots to this session (so it replaces the config, not
#  simply adds to it)
# future: update: if there are no conflicts, removes preempted by copying
#         current running-config to ancestor.

class ConfigSessionMode( BasicCli.GlobalConfigMode ):
   modeParseTreeShared = True

   def __init__( self, parent, session ):
      BasicCli.GlobalConfigMode.__init__( self, parent=parent, session=session )
      self.cs = CliSession.currentSession( self.session_.entityManager_ )
      self.userSession = False      # True if started by "configure session" CLI

   def enter( self ):
      pass

   def onExit( self ):
      if self.userSession:
         CliSession.exitSession( self.session_.entityManager_ )
         ConfigSessionCommon.doLog( ConfigSessionCommon.SYS_CONFIG_SESSION_EXITED,
                                    self.cs )
      BasicCli.GlobalConfigMode.onExit( self )

def setCheckpointFunc( callback ):
   # solve the dependency problem with ConfigSession package,
   # only start to create checkpoint files when the ConfigSession package
   # has been loaded
   global checkpointFunc
   checkpointFunc = callback

def commitSession( mode, timerValue=None, sessionName=None ):
   timerValue1 = timerValue
   sessionName1 = sessionName
   if timerValue:
      hr, minutes, sec = timerValue
      timerValue = hr * 3600 + minutes * 60 + sec
      if timerValue == 0:
         mode.addError( "Time value cannot be zero." )
         return "Error"
   t0( 'Committing session ', sessionName,
       ' in ', mode, 'with timerValue ', timerValue )

   em = mode.session_.entityManager_
   commitTimerInProgress = CliSession.isCommitTimerInProgress()
   if sessionName is None:
      sessionName = CliSession.currentSession( em )
      if sessionName is None:
         mode.addError( "Cannot commit session due to missing name." )
         return "Error"

   t0( 'Is commit timer in progress ? ', commitTimerInProgress )
   response = CliSession.canCommitSession( em, sessionName1 )
   if not response:
      # Checkpoint only the commit timer(not commit) for
      # the first time.
      if timerValue and not commitTimerInProgress:
         savePreCommitConfig( mode )
   else:
      mode.addError( response )
      return "Error"

   if not BasicCliSession.CONFIG_LOCK.canRunConfigureCmds():
      raise ConfigMount.ConfigMountProhibitedError(
         ConfigMount.ConfigMountProhibitedError.CONFIG_LOCKED )

   # make checkpoint file only when ConfigSession package is loaded and
   # CheckpointUrl is registered
   if checkpointFunc:
      checkpointUrl = checkpointFunc( mode, cliConfig.sysdbObj().maxCheckpoints )
   else:
      checkpointUrl = None

   # Ask the session manager to commit the session. If None is returned,
   # the session was concurrently deleted. Otherwise a non-empty string
   # means the session could not be committed.
   response = CliSession.commitSession( em, None, timerValue, sessionName )
   if response:
      mode.addError( response )
      ConfigSessionCommon.doLog(
            ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_FAILURE, sessionName )
      return "Error"
   if commitTimerInProgress:
      t0( 'Session ', sessionName, 'already pending commit timer. ',
          'Changing timer only ' )
   if response is not None:
      # Invoke commit handlers and rollback to checkpointUrl if error
      onCommitStatus = ConfigSessionCommon.invokeOnCommitHandlersOrRevert( mode,
                          sessionName, checkpointUrl )
      if timerValue:
         if commitTimerInProgress:
            logId = ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED
         else:
            logId = ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED
         ConfigSessionCommon.doLog( logId, sessionName, time=timerValue1 )
      elif onCommitStatus is None:
         ConfigSessionCommon.doLog(
               ConfigSessionCommon.SYS_CONFIG_SESSION_COMMIT_SUCCESS,
               sessionName )
         mode.session_.addToHistoryEventTable( "commandLine", "session",
                                               "running", sessionName, "",
                                               runningConfigChanged=True )
      else:
         mode.addError( onCommitStatus )
   return None

def abortSession( mode, sessionName ):
   t0( 'Aborting Session', sessionName, 'in mode', mode )
   em = mode.session_.entityManager_
   response = CliSession.abortSession( em, sessionName )
   if response:
      mode.addError( response )
   return response

# Some default settings are configurable by the customer, that is, not set in tacc.
# So when rolling back to a clean config, those defaults have to be honored, so the
# owner of the attribute with configurable default can register an save function
# to record the state as well as a restore function to restore it. The save function
# is called before rolling back, th restore function when the rollback is done. The
# restore function is passed in what the save function returned.
customizableDefaultsCbs = []

def registerCustomizableDefaults( save, restore ):
   cbinfo = { "save": save, "context": None, "restore": restore }
   customizableDefaultsCbs.append( cbinfo )

class ReInstallCustomerConfigurableDefaults( object ):
   """Context manager that on entry remembers the state of defaults that are
   customizable, so that they can be set again on exit. To be used as part of
   a config session rollback clean-config operation.
   """
   def __enter__( self ):
      for cbinfo in customizableDefaultsCbs:
         cbinfo[ "context" ] = cbinfo[ "save" ]()
      return self

   def __exit__( self, excType, value, excTraceback ):
      for cbinfo in customizableDefaultsCbs:
         cbinfo[ "restore" ]( cbinfo[ "context" ] )

def setMaxPendingSessionNum( mode, args ):
   cliConfig.maxOpenSessions = args.get( '<maxPendingSessions>', 5 )

def savePreCommitConfig( mode ):
   """Save running-config into a checkpoint file."""
   t0( 'savePreCommitConfig in mode', mode )
   ctx = Url.Context( mode.session_.entityManager_, disableAaa=True,
                      cliSession=None )
   surl = Url.parseUrl( 'system:/running-config', ctx )
   # this could be invoked under different user IDs; so make sure we can write to it
   preCommitConfig = cliSessionStatus.preCommitConfig
   Tac.run( [ 'rm', '-f', preCommitConfig.lstrip( 'file:' ) ], asRoot=True )
   durl = Url.parseUrl( preCommitConfig, ctx )
   FileCliUtil.copyFile( None, mode, surl, durl, commandSource="configure replace" )

#-----------------------------------------------------------------------------------
# [ no | default ] configure session [ <sessionName> ]
#-----------------------------------------------------------------------------------
sessionKwForConfig = CliMatcher.KeywordMatcher( 'session',
                                          helpdesc=( 'Enter configuration session; '
                                                'commands applied only on commit' ) )
sessionNameMatcher = CliMatcher.PatternMatcher( r'\S+', helpname='WORD',
                                                helpdesc='Name for session' )

@Ark.synchronized( sessionLock )
def enterConfigSession( mode, args ):
   sessionName = args.get( '<sessionName>' )
   noCommit = 'no-commit' in args
   openSessionNum = 0
   for sname, pcs in cliSessionStatus.pendingChangeStatusDir.status.iteritems():
      if sname == sessionName:
         openSessionNum = -1
         break
      if not pcs.completed:
         openSessionNum = openSessionNum + 1
   if openSessionNum >= cliConfig.maxOpenSessions:
      mode.addError( "Maximum number of pending sessions has been reached. "
                     "Please commit or abort previous sessions "
                     "before creating a new one." )
      return
   em = mode.session_.entityManager_
   if not sessionName:
      sessionName = CliSession.uniqueSessionName( em, "sess" )

   response = CliSession.canEnterSession( sessionName, em, noCommit )
   if response:
      mode.addError( response )
      return

   # We know we'll be entering config-s mode from exec mode, so
   # in case we are typed in a config mode already, we need to
   # re-enable ConfigMounts and call the onExit() handlers for
   # config modes we've entered.
   with ConfigMount.ConfigMountDisabler( disable=False ):
      mode.session_.commitMode()
   response = CliSession.enterSession( sessionName, em, noCommit )
   if response:
      mode.addError( response )
      return

   # add a description to userString if Cli provides one. TODO
   newMode = ConfigSessionMode( mode, mode.session_ )
   newMode.userSession = True
   mode.session_.gotoChildMode( newMode )
   ConfigSessionCommon.doLog( ConfigSessionCommon.SYS_CONFIG_SESSION_ENTERED,
                              CliSession.currentSession( em ) )

class EnterConfigSession( CliCommand.CliCommandClass ):
   syntax = '''configure session [ <sessionName> ]'''
   noOrDefaultSyntax = '''configure session <sessionName>'''
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher
          }
   handler = enterConfigSession

   @staticmethod
   @Ark.synchronized( sessionLock )
   def noOrDefaultHandler( mode, args ):
      sessionName = args[ '<sessionName>' ]
      em = mode.session_.entityManager_
      response = CliSession.deleteSession( em, sessionName )
      if response:
         mode.addError( response )
      else:
         ConfigSessionCommon.doLog( ConfigSessionCommon.SYS_CONFIG_SESSION_DELETED,
                                    sessionName )

BasicCli.EnableMode.addCommandClass( EnterConfigSession )

#-----------------------------------------------------------------------------------
# configure session <sessionName> no-commit
#-----------------------------------------------------------------------------------
class EnterConfigNoCommitSession( CliCommand.CliCommandClass ):
   syntax = '''configure session <sessionName> no-commit'''
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher,
            'no-commit': CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'no-commit',
                                       helpdesc='Per user configure session that '
                                                'disallows commit' ),
                              hidden=True )
          }
   handler = enterConfigSession

BasicCli.EnableMode.addCommandClass( EnterConfigNoCommitSession )

#-----------------------------------------------------------------------------------
# configure session <sessionName> commit [ timer <timerValue> ]
#-----------------------------------------------------------------------------------
timerKwMatcher = CliMatcher.KeywordMatcher(
   'timer',
   helpdesc='commit session with a timeout. '
            'If not committed within this time, config will be reverted.' )
timerValueMatcher = DateTimeRule.TimeMatcher( helpdesc='timeout' )

commitKwMatcher = CliMatcher.KeywordMatcher(
   'commit',
   helpdesc='Commit pending session' )

class ConfigSessionCommit( CliCommand.CliCommandClass ):
   syntax = '''configure session <sessionName> commit [ timer <timerValue> ]'''
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher,
            'commit': commitKwMatcher,
            'timer': timerKwMatcher,
            '<timerValue>': timerValueMatcher
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      commitSession( mode, args.get( '<timerValue>' ), args[ '<sessionName>' ] )

BasicCli.EnableMode.addCommandClass( ConfigSessionCommit )

#-----------------------------------------------------------------------------------
# configure session <sessionName> abort
#-----------------------------------------------------------------------------------
class ConfigSessionAbort( CliCommand.CliCommandClass ):
   syntax = '''configure session <sessionName> abort'''
   data = {
            'configure': CliToken.Configure.configureParseNode,
            'session': sessionKwForConfig,
            '<sessionName>': sessionNameMatcher,
            'abort': 'Abort configuration session'
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      sessionName = args[ '<sessionName>' ]
      if not CliSession.isSessionPresent( sessionName ):
         mode.addError( "Cannot abort non-existent session %s." % sessionName )
         return
      if abortSession( mode, sessionName ):
         return
      isOkToLog = not CliSession.isSessionPendingCommitTimer( sessionName )
      if isOkToLog:
         ConfigSessionCommon.doLog( ConfigSessionCommon.SYS_CONFIG_SESSION_ABORTED,
                                    sessionName )

BasicCli.EnableMode.addCommandClass( ConfigSessionAbort )

#-----------------------------------------------------------------------------------
# [ no | default ] terminal auto-session
#-----------------------------------------------------------------------------------
class AutoSession( CliCommand.CliCommandClass ):
   syntax = '''terminal auto-session'''
   noOrDefaultSyntax = syntax
   data = {
             'terminal': CliToken.Terminal.terminalKwForExec,
             'auto-session': CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'auto-session',
                                       helpdesc='Run each command in an individual '
                                                'config session in this Cli' ),
                              hidden=True )
          }

   @staticmethod
   def handler( mode, args ):
      mode.session_.autoConfigSessionIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session_.autoConfigSessionIs( False )

BasicCli.UnprivMode.addCommandClass( AutoSession )
BasicCli.EnableMode.addCommandClass( AutoSession )

#-----------------------------------------------------------------------------------
# commit [ timer <timerValue> ]
#-----------------------------------------------------------------------------------
class CommitInConfigureSession( CliCommand.CliCommandClass ):
   syntax = '''commit [ timer <timerValue> ]'''
   data = {
            'commit': CliCommand.Node( matcher=commitKwMatcher,
                                       guard=ConfigSessionCommon.sessionGuard ),
            'timer': timerKwMatcher,
            '<timerValue>': timerValueMatcher
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      response = commitSession( mode, args.get( '<timerValue>' ) )
      if response is None:
         mode.session_.gotoParentMode()

ConfigSessionMode.addCommandClass( CommitInConfigureSession )

#-----------------------------------------------------------------------------------
# abort
#-----------------------------------------------------------------------------------
class AbortInConfigureSession( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort': CliCommand.Node(
                           matcher=CliMatcher.KeywordMatcher( 'abort',
                                            helpdesc='Abort configuration session' ),
                           guard=ConfigSessionCommon.sessionGuard )
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      sessionName = CliSession.currentSession( mode.session_.entityManager_ )
      response = abortSession( mode, sessionName )
      mode.session_.gotoParentMode()
      if not response:
         ConfigSessionCommon.doLog( ConfigSessionCommon.SYS_CONFIG_SESSION_ABORTED,
                                    sessionName )

ConfigSessionMode.addCommandClass( AbortInConfigureSession )

#-----------------------------------------------------------------------------------
# rollback clean-config [ { <configGroups> } ]
#-----------------------------------------------------------------------------------
emptyCompletion = [ CliParser.Completion( 'WORD', 'Config Group(s)',
   literal=False ) ]

class RollbackSession( CliCommand.CliCommandClass ):
   syntax = 'rollback clean-config [ { <configGroups> } ]'
   data = {
             'rollback': CliCommand.Node(
                                    matcher=CliMatcher.KeywordMatcher( 'rollback',
                                              helpdesc='Clear configuration state' ),
                                    guard=ConfigSessionCommon.sessionGuard ),
             'clean-config': 'Copy config state from clean, default, config',
             '<configGroups>': CliMatcher.DynamicKeywordMatcher(
                                 lambda mode: CliSession.sessionStatus.configGroup,
                                 emptyTokenCompletion=emptyCompletion )
          }

   @staticmethod
   @Ark.synchronized( sessionLock )
   def handler( mode, args ):
      em = mode.session_.entityManager_
      configGroups = args.get( '<configGroups>' )
      response = CliSession.rollbackSession( em, configGroups=configGroups )
      if response:
         mode.addError( response )
      # Replay the commands in base-config as it contains default settings that
      # should survive a rollback clean-config (or write-erase if in kickstart-conf)
      if os.path.exists( "/mnt/flash/base-config" ):
         with open( "/mnt/flash/base-config" ) as f:
            for line in f:
               line = line.strip( "\n" )
               if line:
                  mode.session_.runCmd( line, aaa=False )

ConfigSessionMode.addCommandClass( RollbackSession )

#-----------------------------------------------------------------------------------
# service configuration session max completed <maxSavedSessions>
#-----------------------------------------------------------------------------------
sessionKwMatcher = CliMatcher.KeywordMatcher( 'session',
                                              helpdesc='configure session settings' )
maxKwMatcher = CliMatcher.KeywordMatcher( 'max', helpdesc='set a max limit' )

class ConfigSessionMaxCompleted( CliCommand.CliCommandClass ):
   syntax = 'service configuration session max completed <maxSavedSessions>'
   noOrDefaultSyntax = ( 'service configuration session max completed '
                         '...' )
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'max': maxKwMatcher,
            'completed': 'maximum number of completed sessions kept in memory',
            '<maxSavedSessions>': CliMatcher.IntegerMatcher( 0, 20,
                                       helpdesc='Maximum number of saved sessions' )
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.maxSavedSessions = args[ '<maxSavedSessions>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.maxSavedSessions = 1

BasicCli.GlobalConfigMode.addCommandClass( ConfigSessionMaxCompleted )

#-----------------------------------------------------------------------------------
# service configuration session max pending <maxPendingSessions>
#-----------------------------------------------------------------------------------
class ConfigSessionMaxPending( CliCommand.CliCommandClass ):
   syntax = 'service configuration session max pending <maxPendingSessions>'
   noOrDefaultSyntax = 'service configuration session max pending ...'
   data = {
            'service': CliToken.Service.serviceKw,
            'configuration': CliToken.Service.configKwAfterService,
            'session': sessionKwMatcher,
            'max': maxKwMatcher,
            'pending': 'maximum number of pending sessions',
            '<maxPendingSessions>': CliMatcher.IntegerMatcher( 1, 20,
                                      helpdesc='Maximum number of pending sessions' )
          }

   @staticmethod
   def handler( mode, args ):
      cliConfig.maxOpenSessions = args[ '<maxPendingSessions>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.maxOpenSessions = 5

BasicCli.GlobalConfigMode.addCommandClass( ConfigSessionMaxPending )

#-----------------------------------------------------------------------------------
# Plugin Func
#-----------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig, cliSessionStatus

   cliConfig = ConfigMount.mount( entityManager, "cli/session/input/config",
                                  "Cli::Session::CliConfig", "w" )

   cliSessionStatus = LazyMount.mount( entityManager, "cli/session/status",
                                       "Cli::Session::Status", "r" )
