# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Implements CLI command for generating a log message with the following format:
         %SYS-2-LOGMSG: Message from <username> on <terminal> (<ip-address>): <text>
Supported in privileged EXEC mode, with the following syntax:
         send log [level <level>] message <text>
where <text> is an arbitrary non-empty string.
"""

import BasicCli
import CliCommand
import CliMatcher
import Logging
import LoggingLib
import UtmpDump

#------------------------------------------------------------------------------------
# The "send log" command, in enable mode.
#------------------------------------------------------------------------------------
class SendCmd( CliCommand.CliCommandClass ):
   syntax = "send log [ level LEVEL ] message MESSAGE"
   data = {
      'send' : 'Generate log message',
      'log' : 'Generate log message',
      'level' : 'Specify the severity level (default is \'informational\')',
      'LEVEL' : LoggingLib.SeverityExpression,
      'message' : 'Specify the log message',
      'MESSAGE' : CliMatcher.StringMatcher( helpname='LOG',
                                            helpdesc='Log message' )
      }

   @staticmethod
   def handler( mode, args ):
      info = UtmpDump.getUserInfo()
      # Set 'informational' as the default value of log severity level
      severity = args.get( 'SEVERITY', 6 )
      Logging.log( LoggingLib.logSevValueToSendLogIdMap[ severity ],
                   info[ 'user' ], info[ 'tty' ], info[ 'ipAddr' ],
                   args[ 'MESSAGE' ] )


# Add the command to the CLI system
BasicCli.EnableMode.addCommandClass( SendCmd )

