# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import ConfigMount
import LazyMount
import CliExtensions
import CliParser
import CliMatcher
import CliCommand
import IntfCli
import Tac
import Tracing
from CliMode.SegmentRoutingMode import SrModeBase
from RouterGeneralCli import ( RouterGeneralMode, routerGeneralCleanupHook )
import BasicCli
from Toggles import SegmentRoutingCliToggleLib

traceHandle = Tracing.Handle( 'SegmentRoutingCli' )
t1 = traceHandle.trace1 # Info

# pkgdeps: library MplsSysdbTypes

srConfig = None
mplsHwCapability = None
mplsConfig = None
ip6Config = None
ipConfig = None
srSidInvalid = Tac.Type( "Routing::SegmentRoutingCli::Constants" ).srSidInvalid

#--------------------------------------------------------
# Segment-routing currently only support mpls dataplane
#--------------------------------------------------------
def mplsSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

# The node-segment ipv[4|6] command is common to both ISIS and OSPF,
# and implemented here. The below callback is used to check for
# conflicts with other prefix/proxy-node configs in ISIS/OSPF.
# If a configuration is acceptable to both OSPF and ISIS only then
# is it accepted. This should typically not be a problem as the protocol
# callbacks will check that the interface is of interest to them.
# NOTE : Bugs 373190 & 147320 track removing the conflict resolution
# logic from the Cli, once fixed we could removed the below hooks.
validateNodeSegmentHook = CliExtensions.CliHook()

def checkForConflicts( mode, index, ipv6=False ):
   for hook in validateNodeSegmentHook.extensions():
      ( accept, hookMsg ) = hook( mode.intf.name, index, ipv6=ipv6 )
      if hookMsg:
         if accept:
            mode.addWarning( hookMsg )
         else:
            mode.addError( hookMsg )

      if not accept:
         t1( '%s reported conflict for %s' % ( hook.__str__, mode.intf.name ) )
         return True

   return False

def _deleteSrIntfConfigIfAllDefaults( intfName ):
   srIntfConfig = srConfig.intfConfig.get( intfName, None )
   if srIntfConfig:
      if srIntfConfig.srV6NodeSegmentIndex == srSidInvalid and \
         srIntfConfig.srNodeSegmentIndex == srSidInvalid:

         del srConfig.intfConfig[ intfName ]

def updateSrNodeSid( intfName, index, ipv6=False ):
   # All registered hooks find this configuration acceptable
   # so update the config
   srIntfConfig = srConfig.intfConfig.get( intfName, None )
   if not srIntfConfig:
      srIntfConfig = srConfig.intfConfig.newMember( intfName )

   t1( 'Updating %s node-segment for %s to %s' %
       ( 'ipv6' if ipv6 else 'ipv4', intfName, index ) )
   if ipv6:
      srIntfConfig.srV6NodeSegmentIndex = index
   else:
      srIntfConfig.srNodeSegmentIndex = index

def deleteSrNodeSid( intfName, ipv6=False ):
   srIntfConfig = srConfig.intfConfig.get( intfName, None )

   if not srIntfConfig:
      return

   t1( 'Deleting %s node-segment for %s' % ( 'ipv6' if ipv6 else 'ipv4', intfName ) )
   if ipv6:
      srIntfConfig.srV6NodeSegmentIndex = srSidInvalid
   else:
      srIntfConfig.srNodeSegmentIndex = srSidInvalid
   _deleteSrIntfConfigIfAllDefaults( intfName )

#---------------------------------------------------------------------------------
# Create a new mode SrMode to add segment routing related command agnostic to IGP
#---------------------------------------------------------------------------------
class SrMode( SrModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      SrModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# Create a new modelet so that we can add node-segment command only to relavant
# interfaces
#-------------------------------------------------------------------------------
class SrIntfConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.mode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.routingSupported() and
               not mode.intf.name.startswith( "Management" ) )

# Add SrIntfConfigModelet to IntfConfigMode
IntfCli.IntfConfigMode.addModelet( SrIntfConfigModelet )
srIntfModelet = SrIntfConfigModelet

class SegmentRoutingIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      deleteSrNodeSid( self.intf_.name )
      deleteSrNodeSid( self.intf_.name, ipv6=True )

#-------------------------------------------------------------------------------
# [no|default] node-segment ipv4|ipv6 index <index>
# also adds node-segment index <index> which is hidden since it is deprecated
#-------------------------------------------------------------------------------
def setNodeSegmentV4( mode, index ):
   intfName = mode.intf.name
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )

   if ( ipIntfConfig is None ) or \
          ( ipIntfConfig is not None and
              ipIntfConfig.addrWithMask.address != '0.0.0.0'
              and ipIntfConfig.addrWithMask.len != 32 ):
      mode.addWarning( "/32 IPv4 address is not configured on the interface" )

   if not checkForConflicts( mode, index ):
      updateSrNodeSid( intfName, index )

def setNodeSegmentV6( mode, index ):
   intfName = mode.intf.name
   ip6IntfConfig = ip6Config.intf.get( intfName )
   v6HostAddrPresent = False
   if ip6IntfConfig:
      for prefix in ip6IntfConfig.addr:
         prefix = Arnet.IpGenPrefix( str( prefix ) )
         if prefix.isHost:
            v6HostAddrPresent = True

   if not v6HostAddrPresent:
      mode.addWarning( "/128 IPv6 address is not configured on the interface" )

   if not checkForConflicts( mode, index, ipv6=True ):
      updateSrNodeSid( intfName, index, ipv6=True )

def setNodeSegment( mode, af, index ):
   if af == 'ipv4':
      setNodeSegmentV4( mode, index )
   elif af == 'ipv6':
      setNodeSegmentV6( mode, index )

def noNodeSegment( mode, af ):
   deleteSrNodeSid( mode.intf.name, ipv6=( af == 'ipv6' ) )

# The node index can either be from the ISIS or OSPF sr ranges,
# use the maximum of the two here. In the field we don't see any
# practical use case of having OSPF and ISIS running together so
# no point complicating the logic determining which protocol is
# applicable to the interface.
def nodeIndexRangeFn( mode ):
   isisSrgb = mplsConfig.labelRange[ 'isis-sr' ].size
   # TODO Bug370639 : Once the toggle is removed this check can be relaxed
   ospfSrgb = mplsConfig.labelRange.get( 'ospf-sr',
         Tac.Value( 'Mpls::LabelRange', 1, 0 ) ).size

   maxSize = ospfSrgb if ospfSrgb > isisSrgb else isisSrgb
   return ( 0, maxSize - 1 )

srIndexMatcher = CliMatcher.DynamicIntegerMatcher( rangeFn=nodeIndexRangeFn,
                              helpdesc='Index to be mapped with IP prefix' )
#-------------------------------------------------------------------------------
# [no|default] node-segment ipv4|ipv6 index <index>
#-------------------------------------------------------------------------------

class NodeSegmentCommand( CliCommand.CliCommandClass ):
   syntax = 'node-segment ( ipv4 | ipv6 ) index INDEX'
   noOrDefaultSyntax = 'node-segment (ipv4 | ipv6) ...'

   data = {
         'node-segment': 'Configure a node segment',
         'ipv4': 'IPv4 node config',
         'ipv6': 'IPv6 node config',
         'index': 'Node segment identifier',
         'INDEX': srIndexMatcher
   }

   @staticmethod
   def handler( mode, args ):
      af = 'ipv4' if 'ipv4' in args else 'ipv6'
      index = args[ 'INDEX' ]
      setNodeSegment( mode, af, index )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      af = 'ipv4' if 'ipv4' in args else 'ipv6'
      noNodeSegment( mode, af )

class HiddenNodeSegmentCommand( CliCommand.CliCommandClass ):
   syntax = 'node-segment index INDEX'
   noOrDefaultSyntax = 'node-segment index ...'

   data = {
         'node-segment': 'Configure a node segment',
         'index': 'Node segment identifier',
         'INDEX': srIndexMatcher
   }

   @staticmethod
   def handler( mode, args ):
      setNodeSegment( mode, 'ipv4', args[ 'INDEX' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noNodeSegment( mode, 'ipv4' )

   hidden = True

srIntfModelet.addCommandClass( NodeSegmentCommand )
srIntfModelet.addCommandClass( HiddenNodeSegmentCommand )

def delSegmentRouting():
   mplsConfig.tunnelIgpFecSharing = True
   srConfig.enabled = False

class CfgSegmentRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing': CliCommand.guardedKeyword( 'segment-routing',
                                                    'Segment Routing configuration',
                                                    guard=mplsSupportedGuard )
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SrMode )
      mode.session_.gotoChildMode( childMode )
      srConfig.enabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delSegmentRouting()

if SegmentRoutingCliToggleLib.toggleTunnelIgpFecSharingEnabled():
   RouterGeneralMode.addCommandClass( CfgSegmentRoutingCmd )

#-------------------------------------------------------------------------------#
# rtr1(config-sr)#[no|default] fec sharing igp tunnel disabled
# No need to add guard because segment routing mode is already protected by guard
#-------------------------------------------------------------------------------#
class TunnelIgpFecSharingCmd( CliCommand.CliCommandClass ):
   syntax = 'fec sharing igp tunnel disabled'
   noOrDefaultSyntax = syntax

   data = {
      'fec': 'FEC configuration',
      'sharing': 'Share the FEC',
      'igp': 'Running IGP protocol',
      'tunnel': 'Tunnel for the IGP',
      'disabled': 'Disable FEC sharing',
   }

   @staticmethod
   def handler( mode, args ):
      mplsConfig.tunnelIgpFecSharing = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mplsConfig.tunnelIgpFecSharing = True

if SegmentRoutingCliToggleLib.toggleTunnelIgpFecSharingEnabled():
   SrMode.addCommandClass( TunnelIgpFecSharingCmd )
#---------------------------------------------------------------
# Remove all segment-routing configs when the parent is removed
# i.e., "no router general" in config mode.
#---------------------------------------------------------------

def noOrDefaultRouterGeneralMode( mode=None ):
   delSegmentRouting()

routerGeneralCleanupHook.addExtension( noOrDefaultRouterGeneralMode )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entMan ):
   global srConfig
   global mplsConfig
   global ipConfig
   global ip6Config
   global mplsHwCapability

   entityManager = entMan

   # pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/ConfigAgent-SegmentRoutingCli
   srConfig = ConfigMount.mount( entityManager, "routing/sr/config",
                                 "Routing::SegmentRoutingCli::Config", 'w' )

   mplsConfig = ConfigMount.mount( entityManager, "routing/mpls/config",
                                   "Mpls::Config", "w" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )

   IntfCli.Intf.registerDependentClass( SegmentRoutingIntf, priority=20 )
