# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import array
import fcntl
import os
import re
import termios

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliModel
import CliParser
from CliPlugin import ConfigMgmtMode
import ConfigMount
import LazyMount
import SecretCli
import ShowCommand
from MgmtSecurityLib import mgmtSecurityConfigType
import Tac

# ------------------------------
# Register callback for the forwarding chip information

forwardingInfoCallbacks = []

def registerForwardingChipInfoCallback( callbackFunc ):
   forwardingInfoCallbacks.append( callbackFunc )

# ------------------------------
# Security config commands
# 
# From config mode
#    management security - enter security config mode
#

cpuModelRegex = re.compile( r'model name\s+: (.*)' )
securityConfig = None
securityStatus = None
sslProfileConfig = None

securityShowMatcher = CliMatcher.KeywordMatcher(
   'security',
   helpdesc='Show security status' )

securityKwMatcher = CliMatcher.KeywordMatcher(
   'security',
   helpdesc='Configure security related options' )

class SecurityConfigMode( ConfigMgmtMode.ConfigMgmtMode ):

   name = "Security configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "security" )
      self.config_ = securityConfig

   def setEntropySource( self, args ):
      if "hardware" in args:
         self.config_.entropySourceHardware = True
      if "haveged" in args:
         self.config_.entropySourceHaveged = True

   def noEntropySource( self, args=None ):
      if not args or ( "hardware" not in args and "haveged" not in args ):
         self.config_.entropySourceHardware = False
         self.config_.entropySourceHaveged = False
      else:
         if "hardware" in args:
            self.config_.entropySourceHardware = False
         if "haveged" in args:
            self.config_.entropySourceHaveged = False

   def defaultEntropySource( self, args ):
      if "hardware" not in args and "haveged" not in args:
         self.config_.entropySourceHardware = \
               self.config_.defaultEntropySourceHardware
         self.config_.entropySourceHaveged = \
               self.config_.defaultEntropySourceHaveged
      else:
         if "hardware" in args:
            self.config_.entropySourceHardware = \
                  self.config_.defaultEntropySourceHardware
         if "haveged" in args:
            self.config_.entropySourceHaveged = \
                  self.config_.defaultEntropySourceHaveged

   def setEntropyServer( self, args ):
      self.config_.useEntropyServer = True

   def defaultEntropyServer( self, args ):
      self.config_.useEntropyServer = self.config_.defaultUseEntropyServer

   def setPasswordMinimumLength( self, args ):
      self.config_.minPasswordLength = args[ 'LENGTH' ]

   def noPasswordMinimumLength( self, args ):
      self.config_.minPasswordLength = 0

   def setPasswordEncryptionKeyCommon( self, args ):
      self.config_.commonKeyEnabled = True

   def noPasswordEncryptionKeyCommon( self, args ):
      self.config_.commonKeyEnabled = False

   def defaultPasswordEncryptionKeyCommon( self, args ):
      self.config_.commonKeyEnabled = self.config_.commonKeyEnabledDefault

   def setSignatureVerification( self, args ):
      sslProfile = args.get( 'PROFILE', self.config_.defaultSslProfile )
      self.config_.enforceSignature = True
      self.config_.sslProfile = sslProfile

   def noSignatureVerification( self, args ):
      self.config_.enforceSignature = False
      self.config_.sslProfile = self.config_.defaultSslProfile

   def defaultSignatureVerification( self, args ):
      self.config_.enforceSignature = self.config_.defaultEnforceSignature
      self.config_.sslProfile = self.config_.defaultSslProfile

   def unblockNetworkProtocol( self, args ):
      protocol = args[ 'PROTO' ]
      del self.config_.blockedNetworkProtocols[ protocol ]

   def blockNetworkProtocol( self, args ):
      protocol = args[ 'PROTO' ]
      self.config_.blockedNetworkProtocols[ protocol ] = True

   def clearAllBlockNetworkProtocols( self, args=None ):
      self.config_.blockedNetworkProtocols.clear()

def entropyHardwareSupported( mode, token ):
   if securityStatus.entropySourceHardwareSupported:
      return None
   return CliParser.guardNotThisPlatform

def gotoSecurityConfigMode( mode, args ):
   childMode = mode.childMode( SecurityConfigMode )
   mode.session_.gotoChildMode( childMode )

def noSecurityConfig( mode, args ):
   childMode = mode.childMode( SecurityConfigMode )
   childMode.noEntropySource()
   childMode.defaultSignatureVerification( args )
   childMode.clearAllBlockNetworkProtocols()
   childMode.removeComment()
   import CliPlugin.Ssl
   import CliPlugin.SharedSecretProfileCli
   CliPlugin.Ssl.noSecurityConfig( mode )
   CliPlugin.SharedSecretProfileCli.noSecurityConfig( mode )

class EnterSecurityMode( CliCommand.CliCommandClass ):
   syntax = 'management security'
   noOrDefaultSyntax = syntax
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'security': securityKwMatcher
          }
   handler = gotoSecurityConfigMode
   noOrDefaultHandler = noSecurityConfig

BasicCli.GlobalConfigMode.addCommandClass( EnterSecurityMode )

#------------------------------------------------------------------------------------
# The "entropy source" commands, in config mode.
#
# [no|default] entropy source hardware [exclusive]
#------------------------------------------------------------------------------------
entryKwMatcher = CliMatcher.KeywordMatcher( 'entropy',
      helpdesc='Entropy configuration' )
sourceKwMatcher = CliMatcher.KeywordMatcher( 'source', helpdesc='Source of entropy' )
hwMatcherMatcher = CliMatcher.KeywordMatcher( 'hardware',
      helpdesc='Set entropy source to hardware' )
hwNode = CliCommand.Node( matcher=hwMatcherMatcher, guard=entropyHardwareSupported )
havegedMatcher = CliMatcher.KeywordMatcher( 'haveged',
      helpdesc='Set entropy source to haveged' )

class EntropySourceHardware( CliCommand.CliCommandClass ):
   syntax = 'entropy source { hardware | haveged }'
   noOrDefaultSyntax = 'entropy source [ { hardware | haveged } ]'
   data = {
            'entropy': entryKwMatcher,
            'source': sourceKwMatcher,
            'hardware': hwNode,
            'haveged': havegedMatcher,
          }
   handler = SecurityConfigMode.setEntropySource
   noHandler = SecurityConfigMode.noEntropySource
   defaultHandler = SecurityConfigMode.defaultEntropySource

SecurityConfigMode.addCommandClass( EntropySourceHardware )

# NOTE: This is a separate command since there are plans to make the set of entropy
# sources a set in the future
class EntropySourceHardwareExclusive( CliCommand.CliCommandClass ):
   syntax = 'entropy source hardware exclusive'
   noOrDefaultSyntax = 'entropy source hardware exclusive'
   data = {
            'entropy': entryKwMatcher,
            'source': sourceKwMatcher,
            'hardware': hwNode,
            'exclusive': 'Only use entropy from the hardware source'
          }
   handler = SecurityConfigMode.setEntropyServer
   noOrDefaultHandler = SecurityConfigMode.defaultEntropyServer

SecurityConfigMode.addCommandClass( EntropySourceHardwareExclusive )

#------------------------------------------------------------------------------------
# The "show management security" command
#
# show management security
#------------------------------------------------------------------------------------
def opt( var, default ):
   return var if var else default

class SecurityStatus( CliModel.Model ):
   cpuModel = CliModel.Str( help='CPU Model name and manufacturer', 
                              optional=True )
   switchChip = CliModel.Str( help='Model of the switch chip',
                                        optional=True )
   securityChipVersion = CliModel.Str( help='Security chip version. Not present on '
                                       'older platforms.', optional=True )
   hardwareEntropyEnabled = CliModel.Bool( help='True if the source of entropy is '
                                                'hardware' )
   havegedEntropyEnabled = CliModel.Bool( help='True if the source of entropy is '
                                               'haveged' )
   entropyServerEnabled = CliModel.Bool( help='True if entropy server is serving '
                                       'entropy in the entropy queue' )
   hardwareEntropyQueueSize = CliModel.Int( help='Number of bytes of hardware '
                                                 'generated entropy in the queue',
                                            optional=True )
   blockedNetworkProtocols = CliModel.List( valueType=str,
                                            help='Blocked unencrypted network '
                                                 'protocols', optional=True )

   def render( self ):
      hardwareEntropyMsg = 'enabled' if self.hardwareEntropyEnabled else 'disabled'
      havegedEntropyMsg = 'enabled' if self.havegedEntropyEnabled else 'disabled'
      if self.entropyServerEnabled:
         hardwareEntropyMsg += "\nEntropy Queue Size (bytes): %d" % \
                               self.hardwareEntropyQueueSize
      print 'CPU Model:', opt( self.cpuModel, 'Not available' )
      print 'Security Chip:', opt( self.securityChipVersion, 'Not available' )

      asicLine = 'Forwarding ASIC:'
      switchChipMsg = ''
      if self.switchChip:
         lines = self.switchChip.splitlines()
         switchChipMsg = lines[ 0 ]
         header = ' ' * ( len( asicLine ) + 1 )
         for remainingInfo in lines[ 1: ]:
            switchChipMsg += '\n%s%s' % ( header, remainingInfo )
      else:
         switchChipMsg = 'Not available'
      print asicLine, switchChipMsg
      blockedProtocolsMsg = None
      if self.blockedNetworkProtocols:
         blockedProtocolsMsg = ', '.join( sorted( self.blockedNetworkProtocols ) )
      print 'Blocked client protocols:', blockedProtocolsMsg
      print 'Hardware entropy generation is %s' % hardwareEntropyMsg
      print 'Haveged entropy generation is %s' % havegedEntropyMsg
      print

def getBytesInFifoPipe( mgmtSecConfig ):
   try:
      fifoFd = os.open( mgmtSecConfig.entropyFifoPath,
                        os.O_RDONLY | os.O_NONBLOCK )
      bytesInPipe = array.array( 'i', [ 0 ] )
      assert fcntl.ioctl( fifoFd, termios.FIONREAD, bytesInPipe, 1 ) == 0
      os.close( fifoFd )
      return bytesInPipe[ 0 ]
   except OSError:
      return 0

def showSecurityStatus( mode, args ):
   cpuInfo = open( '/proc/cpuinfo', 'r' ).read()
   cpuModelSearch = re.search( cpuModelRegex, cpuInfo )
   if cpuModelSearch:
      cpuModel = cpuModelSearch.group( 1 )
   else:
      cpuModel = None
   hardwareInfo = None
   for callback in forwardingInfoCallbacks:
      chipInfo = callback( mode )
      if chipInfo:
         hardwareInfo = chipInfo
         break

   res = SecurityStatus()
   res.cpuModel = opt( cpuModel, None )
   res.switchChip = opt( hardwareInfo, None )
   res.securityChipVersion = opt( securityStatus.securityChipVersion, None )
   res.hardwareEntropyEnabled = securityStatus.entropySourceHardwareEnabled
   res.havegedEntropyEnabled = securityStatus.entropySourceHavegedEnabled
   res.entropyServerEnabled = securityStatus.entropyServerEnabled
   totalBytes = securityStatus.entropyQueueSizeBytes + \
                getBytesInFifoPipe( securityConfig )
   res.hardwareEntropyQueueSize = totalBytes
   res.blockedNetworkProtocols = securityConfig.blockedNetworkProtocols.keys()
   
   return res

class ShowSecurityStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': securityShowMatcher
          }
   cliModel = SecurityStatus
   handler = showSecurityStatus

BasicCli.addShowCommandClass( ShowSecurityStatus )

#------------------------------------------------------------------------------------
# The "password minimum length" commands, in config mode.
#
# [no|default] password minimum length <n>
#------------------------------------------------------------------------------------
class PasswordMinLengthCmd( CliCommand.CliCommandClass ):
   syntax = 'password minimum length LENGTH'
   noOrDefaultSyntax = 'password minimum length [ LENGTH ]'
   data = {
            'password': 'Password configuration',
            'minimum': 'Minimum setting',
            'length': 'Number of characters',
            'LENGTH': CliMatcher.IntegerMatcher( 1, 32,
               helpdesc='Value between 1 and 32' )
          }
   handler = SecurityConfigMode.setPasswordMinimumLength
   noOrDefaultHandler = SecurityConfigMode.noPasswordMinimumLength

SecurityConfigMode.addCommandClass( PasswordMinLengthCmd )

#------------------------------------------------------------------------------------
# The "password encryption-key common" commands, in config mode.
#
# [no|default] password encryption-key common <n>
#------------------------------------------------------------------------------------
class PasswordEncryptionKeyCommonCmd( CliCommand.CliCommandClass ):
   syntax = 'password encryption-key common'
   noOrDefaultSyntax = syntax
   data = {
            'password': 'Password configuration',
            'encryption-key': 'internal storage encryption-key',
            'common': 'use a common internal key'
          }
   handler = SecurityConfigMode.setPasswordEncryptionKeyCommon
   noHandler = SecurityConfigMode.noPasswordEncryptionKeyCommon
   defaultHandler = SecurityConfigMode.defaultPasswordEncryptionKeyCommon

SecurityConfigMode.addCommandClass( PasswordEncryptionKeyCommonCmd )

#-----------------------------------------------------------------------------------
# The "signature-verification extension" commands, in config mode.
#
# [no|default] signature-verification extension [ssl profile <profile>]
#-----------------------------------------------------------------------------------
class SignatureVerificationCmd( CliCommand.CliCommandClass ):
   syntax = "signature-verification extension [ ssl profile PROFILE ]"
   noOrDefaultSyntax = "signature-verification extension ..."
   data = {
      "signature-verification": "Configure whether to verify signatures",
      "extension": "Verify SWIX signatures",
      "ssl": "SSL profile",
      "profile": "SSL profile",
      "PROFILE": CliMatcher.DynamicNameMatcher(
                        lambda mode: sslProfileConfig.profileConfig,
                        "SSL profile name" )
   }
   handler = SecurityConfigMode.setSignatureVerification
   noHandler = SecurityConfigMode.noSignatureVerification
   defaultHandler = SecurityConfigMode.defaultSignatureVerification

SecurityConfigMode.addCommandClass( SignatureVerificationCmd )

#-----------------------------------------------------------------------------------
# The "network client protocol PROTO disabled" commands, in config mode.
#
# [no|default] network client protocol PROTO disabled
#-----------------------------------------------------------------------------------
networkProtocols = {}
for proto in [ "tftp", "ftp", "http" ]:
   networkProtocols[ proto ] = "%s protocol configuration" % proto.upper()

class BlockNetworkProtocolCmd( CliCommand.CliCommandClass ):
   syntax = "network client protocol PROTO disabled"
   noOrDefaultSyntax = syntax
   data = {
      "network": "Configure network protocols",
      "client": "Client traffic",
      "protocol": "Unencrypted network protocol",
      "PROTO": CliMatcher.EnumMatcher( networkProtocols ),
      "disabled": "Disable network protocol"
   }
   handler = SecurityConfigMode.blockNetworkProtocol
   noOrDefaultHandler = SecurityConfigMode.unblockNetworkProtocol

SecurityConfigMode.addCommandClass( BlockNetworkProtocolCmd )

def Plugin( entityManager ):
   global securityConfig, securityStatus, sslProfileConfig
   securityConfig = ConfigMount.mount( entityManager, "mgmt/security/config",
                                       mgmtSecurityConfigType, "w" )
   securityStatus = LazyMount.mount( entityManager,
                                     Cell.path( "mgmt/security/status" ),
                                     "Mgmt::Security::Status", "r" )
   sslProfileConfig = LazyMount.mount( entityManager, "mgmt/security/ssl/config",
                                       "Mgmt::Security::Ssl::Config", "r" )
   SecretCli.setMinPasswordLengthFunc( lambda: securityConfig.minPasswordLength )
