# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import AgentDirectory
import Ark
import Cell
from CliModel import Dict
from CliModel import Int
from CliModel import Model
from CliModel import Str
import CliParser
import CliPlugin
import LazyMount

satScdConfig = None
satScdStatus = None
sysname = None

def satelliteScdNameFn( mode ):
   names = []
   if not AgentDirectory.agent( sysname, 'Scd' ):
      return names

   for scdName in satScdConfig.scdConfig:
      if scdName not in satScdStatus.scdStatus:
         continue
      satStatusDir = satScdStatus.scdStatus[ scdName ].satelliteScd
      names.extend( [ '%s-%s' % ( scdName, x )
                      for x in satScdConfig.scdConfig[ scdName
                                  ].satelliteScd.keys() if x in satStatusDir ] )
   return sorted( names )

def satelliteScdGuard( mode, token ):
   # Satellite Scds are present even if no linecards are present, so availability
   # of the satellite config is a good test for the command's guard.
   if satelliteScdNameFn( None ):
      return None
   return CliParser.guardNotThisPlatform

class DebugRegister( Model ):
   name = Str( help="Register name" )
   offset = Int( help="Register offset" )
   value = Int( help="Register value" )

class SatelliteScd( Model ):
   name = Str( help="Satellite Scd Name" )
   collectionTime = Str( help="Collection time" )
   registers = Dict( keyType=int, valueType=DebugRegister,
                     help="Debug registers" )

   def render( self ):
      print( '%s register values:' % self.name )

      if not self.registers:
         print( 'No values collected' )
         return

      print( 'Last collection time:', self.collectionTime )

      for offset in sorted( self.registers ):
         debugReg = self.registers[ offset ]
         print( '%s %s: %s' % ( hex( offset ), debugReg.name,
                                hex( debugReg.value ) ) )

class SatelliteDebug( Model ):
   scds = Dict( keyType=str, valueType=SatelliteScd,
                help="Satellite Scds" )

   def render( self ):
      for name in sorted( self.scds ):
         self.scds[ name ].render()
         print()

def _doShowSatScdDebug( scds, name, debugBlock, debugVals ):
   collectionTime = Ark.timestampToStr( debugVals.timestamp, relative=False )
   registers = {}

   if debugVals:
      for regName in debugBlock:
         assert regName in debugVals.val
         offset = debugBlock[ regName ].offset
         registers[ offset ] = DebugRegister( name=regName,
                                               offset=offset,
                                               value=debugVals.val[ regName ] )

   scds[ name ] = SatelliteScd( name=name, collectionTime=collectionTime,
                                registers=registers )

def doShowSatelliteDebug( mode, args ):
   scds = {}

   cliScdName = args[ 'SCD-NAME' ] if 'SCD-NAME' in args else None
   for scdName in satScdConfig.scdConfig:
      if scdName not in satScdStatus.scdStatus:
         continue
      satConfigDir = satScdConfig.scdConfig[ scdName ].satelliteScd
      satStatusDir = satScdStatus.scdStatus[ scdName ].satelliteScd

      for sliceId in satConfigDir:
         scdDesc = '%s-%s' % ( scdName, sliceId )
         if cliScdName and scdDesc != cliScdName:
            continue
         if sliceId not in satStatusDir:
            continue
         _doShowSatScdDebug(
            scds, scdDesc, satConfigDir[ sliceId ].debugBlock,
            satStatusDir[ sliceId ].debugVals )

   return SatelliteDebug( scds=scds )

#---------------------------------------------------------------------------------
# Register to show tech-support
#---------------------------------------------------------------------------------
def _showTechCmds():
   cmds = [ 'show platform scd satellite debug', ]
   if satelliteScdNameFn( None ):
      return cmds
   return []

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2019-05-22 09:08:26', _showTechCmds )
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2019-05-22 09:08:26', _showTechCmds, extended='platform' )

# -------------------------------------------------------------------------------
# Plugin
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global satScdConfig, satScdStatus, sysname
   sysname = entityManager.sysname()
   satScdConfig = LazyMount.mount( entityManager,
      'hardware/cell/%d/scd/config' % Cell.cellId(),
      'Hardware::Scd::Config', 'r' )
   satScdStatus = LazyMount.mount( entityManager,
      'hardware/archer/scd/status/cell/%d' % Cell.cellId(),
      'Hardware::Scd::Status', 'r' )
