# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libRsvp.so*

import sys

import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.MplsCli as MplsCli
import CliPlugin.RsvpModel as RsvpModel
import CliPlugin.TechSupportCli
import CliToken.Rsvp
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import SmashLazyMount
import Tac

# Type Alias
BandwidthClientStatus = Tac.Type( 'TrafficEngineering::BandwidthClientStatus' )
BandwidthClientUsage = Tac.Type( 'TrafficEngineering::BandwidthClientUsage' )
FrrMode = Tac.Type( 'Rsvp::Cli::FrrMode' )

# global
routingVrfInfoDir = None
routingHardwareRouteStatus = None
mplsRoutingConfig = None
rsvpCliConfig = None
rsvpConfig = None
rsvpMessageCounters = None
rsvpStatus = None
sysname = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
rsvpBwClientUsage = None
rsvpBwClientStatus = None
mplsHwCapability = None
rsvpSessionHistory = None
rsvpSpHistory = None

# Helpers
def rsvpAgentRunning():
   return AgentDirectory.agentIsRunning( sysname, 'Rsvp' )

def mplsAgentRunning():
   return AgentDirectory.agentIsRunning( sysname, 'Mpls' )

# Decorators
def showWarnings( showCmd ):
   def warningShowCmd( mode, *args, **kwargs ):
      routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
      if not rsvpCliConfig.enabled:
         mode.addWarning( "RSVP is not enabled" )
      if not mplsRoutingConfig.mplsRouting:
         mode.addWarning( "MPLS routing is not enabled" )
      if not ( routingInfo and routingInfo.routing ):
         mode.addWarning( "IP routing is not enabled" )
      if not rsvpAgentRunning():
         mode.addWarning( "Agent RSVP is not running" )
      if not mplsAgentRunning():
         mode.addWarning( "Agent MPLS is not running" )
      return showCmd( mode, *args, **kwargs )
   return warningShowCmd

def showHFECDisabledWarning( showCmd ):
   def warningShowCmd( mode, *args, **kwargs ):
      if rsvpCliConfig.frrMode != FrrMode.frrNone and \
         not routingHardwareRouteStatus.hierarchicalFecsEnabled:
         mode.addWarning( "Fast Re-Route support is not available without "
                          "Hierarchical FECs" )
      return showCmd( mode, *args, **kwargs )
   return warningShowCmd


# Tokens
matcherBandwidth = CliMatcher.KeywordMatcher( 'bandwidth',
      helpdesc='Show RSVP bandwidth information' )
matcherNeighbor = CliMatcher.KeywordMatcher( 'neighbor',
      helpdesc='Show RSVP neighbors' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='Show summarized information' )
matcherHistory = CliMatcher.KeywordMatcher( 'history',
      helpdesc='Show last 10 history events' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='More comprehensive output' )

# Constants
RsvpCliId = Tac.Type( "Rsvp::RsvpCliId" )
MIN_CLI_ID = RsvpCliId.minValue
MAX_CLI_ID = RsvpCliId.maxValue

def sessionNameList( mode ):
   return [ spState.getSessionName() for spState in
            rsvpStatus.spStateColl.spState.values() ]

#------------------------------------------------------------------------------------
# show mpls rsvp neighbor[<neighborAddress>]
#------------------------------------------------------------------------------------
@showWarnings
def showMplsRsvpNeighbor( mode, args ):
   neighborAddr = args.get( 'NEIGHBOR_ADDR' )
   summary = 'summary' in args
   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   if neighborAddr is None:
      neighborAddr = Tac.Value( "Arnet::IpGenAddr" )

   LazyMount.force( rsvpStatus )
   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpNeighbor", rsvpStatus )
   renderer.render( socket, fmt, neighborAddr, summary )

   if summary:
      return RsvpModel.RsvpNeighborSummary
   else:
      return RsvpModel.RsvpNeighbors

class MplsRsvpNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp neighbor [ NEIGHBOR_ADDR ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'neighbor': matcherNeighbor,
      'NEIGHBOR_ADDR': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP (v4 or v6) address of neighbor' ),
   }
   handler = showMplsRsvpNeighbor
   cliModel = RsvpModel.RsvpNeighbors

BasicCli.addShowCommandClass( MplsRsvpNeighborCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp neighbor summary
#------------------------------------------------------------------------------------
class MplsRsvpNeighborSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp neighbor summary'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'neighbor': matcherNeighbor,
      'summary': matcherSummary,
   }
   handler = showMplsRsvpNeighbor
   cliModel = RsvpModel.RsvpNeighborSummary

BasicCli.addShowCommandClass( MplsRsvpNeighborSummaryCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp
#------------------------------------------------------------------------------------
@showWarnings
@showHFECDisabledWarning
def showMplsRsvp( mode, args ):
   ReversionMode = Tac.Type( 'Rsvp::Cli::ReversionMode' )
   model = RsvpModel.RsvpState()
   model.adminState = 'enabled' if rsvpCliConfig.enabled else 'disabled'
   if rsvpConfig.runningState == 'rsvpRunning':
      model.operationalState = "up"
   else:
      model.operationalState = "down"
   model.refreshInterval = rsvpConfig.refreshPeriod
   model.refreshReduction = rsvpConfig.rorEnabled

   model.helloEnabled = rsvpConfig.helloEnabled
   if model.helloEnabled:
      model.helloInterval = rsvpConfig.helloInterval
      model.helloMultiplier = rsvpConfig.helloTimeoutFactor

   model.fastReroute = rsvpConfig.frrEnabled
   if rsvpConfig.frrEnabled:
      if rsvpConfig.frrMode == FrrMode.frrNodeProtection:
         model.fastRerouteMode = 'nodeProtection'
      elif rsvpConfig.frrMode == FrrMode.frrLinkProtection:
         model.fastRerouteMode = 'linkProtection'
   else:
      model.fastRerouteMode = 'none'
   model.hierarchicalFecsEnabled = routingHardwareRouteStatus.hierarchicalFecsEnabled
   if rsvpConfig.frrEnabled:
      if rsvpConfig.frrLocalReversion:
         model.reversionMode = 'local'
      else:
         model.reversionMode = 'global'
      if rsvpCliConfig.reversionMode == ReversionMode.reversionLocal:
         model.reversionModeConfig = 'local'
      else:
         model.reversionModeConfig = 'global'

   if rsvpConfig.authType == 'rsvpAuthMd5':
      model.authentication = 'MD5'
   else:
      model.authentication = 'disabled'

   model.srlgMode = rsvpConfig.srlgMode

   model.preemptionPeriod = rsvpConfig.preemptionPeriod
   model.mtuSignalingEnabled = rsvpCliConfig.mtuSignalingEnabled

   model.labelLocalTerminationMode = rsvpConfig.labelLocalTerminationMode

   model.ingressCount = 0
   model.transitCount = 0
   model.egressCount = 0
   if rsvpStatus.sessionStateColl:
      model.sessionCount = len( rsvpStatus.sessionStateColl.sessionState )
      for sessionState in rsvpStatus.sessionStateColl.sessionState.itervalues():
         if sessionState.sessionRole == 'ingressSessionRole':
            model.ingressCount += 1
         elif sessionState.sessionRole == 'transitSessionRole':
            model.transitCount += 1
         elif sessionState.sessionRole == 'egressSessionRole':
            model.egressCount += 1
   else:
      model.sessionCount = 0
   model.lspCount = RsvpModel.LspCount()
   model.lspCount.ingress = 0
   model.lspCount.transit = 0
   model.lspCount.egress = 0
   if rsvpStatus.spStateColl:
      model.lspCount.total = len( rsvpStatus.spStateColl.spState )
      for spState in rsvpStatus.spStateColl.spState.itervalues():
         if spState.sessionRole == 'ingressSessionRole':
            model.lspCount.ingress += 1
         elif spState.sessionRole == 'transitSessionRole':
            model.lspCount.transit += 1
         elif spState.sessionRole == 'egressSessionRole':
            model.lspCount.egress += 1
   else:
      model.lspCount.total = 0

   if rsvpStatus.spStateColl:
      model.lspCount.operational = rsvpStatus.spStateColl.operationalCount
      model.bypassTunnelCount = rsvpStatus.spStateColl.bypassCount
      model.lspCount.usingBypassTunnels = rsvpStatus.spStateColl.bypassInUseCount
   else:
      model.lspCount.operational = 0
      model.bypassTunnelCount = 0
      model.lspCount.usingBypassTunnels = 0

   if rsvpStatus.neighborStateColl:
      model.neighborCount = len( rsvpStatus.neighborStateColl.neighborState )
   else:
      model.neighborCount = 0

   if rsvpStatus.neighborIntfMap:
      model.interfaceCount = rsvpStatus.neighborIntfMap.intfCount
   else:
      model.interfaceCount = 0

   return model

class MplsRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
   }
   handler = showMplsRsvp
   cliModel = RsvpModel.RsvpState

BasicCli.addShowCommandClass( MplsRsvpCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp session [filters]
# Session Filters:
#  - destination <destination>
#  - id <cliId>
#  - role (transit|ingress|egress)
# Sp Filters:
#  - name <sessionName>
#  - lsp <lsp>
#  - state <up, down, path-only>
#  - bypass
#------------------------------------------------------------------------------------
@showWarnings
def showMplsRsvpSession( mode, args ):
   filterArg = args.get( 'filterArg' )
   summary = 'summary' in args
   history = 'history' in args
   detail = 'detail' in args

   tacFilter = Tac.Value( "Rsvp::Cli::ShowRsvpSessionFilter" )
   tacSpFilter = Tac.Value( "Rsvp::Cli::ShowRsvpSessionSpFilter" )

   if filterArg:
      # filterArg [('spFilter', [('sessionName', 'foo')]),
      #            ('spOperationalFilter', [('spOperational', 'up')]),
      #            ('sessionFilter', [('cliId', 2)]),
      #            ('sessionRole', 'ingress')]
      for filterType, arg in filterArg:
         if filterType == 'sessionFilter':
            # Cli Filter rule names are same as RsvpSessionFilter attributes
            for filterAttr, filterValue in arg:
               if filterAttr in [ 'cliId', 'dstIp' ]:
                  setattr( tacFilter, filterAttr, filterValue )

         if filterType == 'sessionRole':
            setattr( tacFilter, filterType, arg )

         if filterType == 'spFilter':
            # Cli Filter rule names are same as RsvpSessionSpFilter attributes
            for filterAttr, filterValue in arg:
               if filterAttr in [ 'sessionName', 'lsp', 'bypass' ]:
                  setattr( tacSpFilter, filterAttr, filterValue )

         if filterType == 'spOperationalFilter':
            # Cli Filter rule name is the same as one of the RsvpSessionSpFilter
            # attributes
            for filterAttr, filterValue in arg:
               if filterAttr == 'spOperational':
                  setattr( tacSpFilter, filterAttr, filterValue )

   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   LazyMount.force( rsvpStatus )
   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpSession", rsvpStatus,
                               rsvpSessionHistory, rsvpSpHistory )
   renderer.render( socket, fmt, summary, history, detail, tacFilter, tacSpFilter )

   return RsvpModel.RsvpSessions

sessionFilterSharedObj = object()
spFilterSharedObj = object()

class MplsRsvpSessionFilterArg( CliCommand.CliExpression ):
   expression = ( '{ '
                  '( ( destination DEST_ADDR ) | ( id ID ) ) | '
                  '( ( name NAME ) | ( lsp LSP ) | bypass ) | '
                  '( role ROLE ) | '
                  '( state STATE ) '
                  '}')
   data = {
      'destination': CliCommand.singleKeyword( 'destination',
         helpdesc='Filter by destination address',
         sharedMatchObj=sessionFilterSharedObj ),
      'DEST_ADDR': IpGenAddrMatcher.IpGenAddrMatcher(
         helpdesc='IP (v4 or v6) address of destination' ),
      'id': CliCommand.singleKeyword( 'id', helpdesc='Filter by Session ID',
         sharedMatchObj=sessionFilterSharedObj ),
      'ID': CliMatcher.IntegerMatcher( MIN_CLI_ID, MAX_CLI_ID,
         helpdesc='Session number' ),
      'name': CliCommand.singleKeyword( 'name', helpdesc='Filter by session name',
         sharedMatchObj=spFilterSharedObj ),
      'NAME': CliMatcher.DynamicNameMatcher( sessionNameList, 'Session name',
         pattern=CliParser.namePattern ),
      'lsp': CliCommand.singleKeyword( 'lsp', helpdesc='Filter by LSP#',
         sharedMatchObj=spFilterSharedObj ),
      'LSP': CliMatcher.IntegerMatcher( MIN_CLI_ID, MAX_CLI_ID, helpdesc='LSP#' ),
      'bypass': CliCommand.singleKeyword( 'bypass',
         helpdesc='Filter for bypass LSPs', sharedMatchObj=spFilterSharedObj ),
      'role': CliCommand.singleKeyword( 'role', helpdesc='Filter by router role' ),
      'ROLE': CliMatcher.EnumMatcher( {
         'transit': 'Show transit LSPs',
         'ingress': 'Show locally originated LSPs',
         'egress': 'Show locally terminating LSPs',
      } ),
      'state': CliCommand.singleKeyword( 'state', helpdesc='Filter by LSP state' ),
      'STATE': CliMatcher.EnumMatcher( {
         'up': 'LSP state up',
         'down': 'LSP state down',
         'path-only': 'LSP state path-only',
      } )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      filterArg = []
      if 'destination' in args:
         filterArg.append( ( 'sessionFilter', [ ( 'dstIp',
                                                args[ 'DEST_ADDR' ][ 0 ] ) ] ) )
      if 'id' in args:
         filterArg.append( ( 'sessionFilter', [ ( 'cliId', args[ 'ID' ][ 0 ] ) ] ) )
      if 'name' in args:
         filterArg.append( ( 'spFilter',  [ ( 'sessionName',
                                            args[ 'NAME' ][ 0 ] ) ] ) )
      if 'lsp' in args:
         filterArg.append( ( 'spFilter',  [ ( 'lsp', args[ 'LSP' ][ 0 ] ) ] ) )
      if 'bypass' in args:
         filterArg.append( ( 'spFilter', [ ( 'bypass', True ) ] ) )
      if 'role' in args:
         filterArg.append( ( 'sessionRole', args[ 'ROLE' ][ 0 ] ) )
      if 'state' in args:
         filterArg.append( ( 'spOperationalFilter', [ ( 'spOperational',
                                                        args[ 'STATE' ][ 0 ] ) ] ) )

      if filterArg:
         args[ 'filterArg' ] = filterArg

class MplsRsvpSessionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp session [ FILTERS ] [ summary | history | detail ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'session': 'Show RSVP session information',
      'FILTERS': MplsRsvpSessionFilterArg,
      'summary': matcherSummary,
      'history' : matcherHistory,
      'detail' : matcherDetail,
   }
   handler = showMplsRsvpSession
   cliModel = RsvpModel.RsvpSessions

BasicCli.addShowCommandClass( MplsRsvpSessionCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp ip access-list [ <acl> ] [ summary ]
# show mpls rsvp ip6 access-list [ <acl> ] [ summary ]
#------------------------------------------------------------------------------------
class ShowMplsRsvpAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mpls rsvp'
              '('
              ' ( ip access-list [ <ipAclName> ] ) | '
              ' ( ipv6 access-list [ <ipv6AclName> ] ) '
              ')' )
   data = {
            'mpls': MplsCli.mplsNodeForShow,
            'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            '<ipAclName>': AclCli.ipAclNameExpression,
            '<ipv6AclName>': AclCli.ip6AclNameExpression
          }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      aclType = 'ip' if 'ip' in args else 'ipv6'
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    aclType,
                                    args[ '<aclNameExpr>' ],
                                    serviceName='rsvp' )

BasicCli.addShowCommandClass( ShowMplsRsvpAclCmd )

#------------------------------------------------------------------------------------
# show mpls rsvp counters [ipv4|ipv6][interface <interfaceRange>]
#------------------------------------------------------------------------------------
def messageTypeFromTacc( tac ):
   if tac == 'rsvpPathMsg':
      return "Path"
   if tac == 'rsvpResvMsg':
      return "Resv"
   if tac == 'rsvpPathErrMsg':
      return "PathErr"
   if tac == 'rsvpResvErrMsg':
      return "ResvErr"
   if tac == 'rsvpPathTearMsg':
      return "PathTear"
   if tac == 'rsvpResvTearMsg':
      return "ResvTear"
   if tac == 'rsvpSrefreshMsg':
      return "Srefresh"
   if tac == 'rsvpUnknownMsg':
      return "Errors"
   else:
      return "Other"

@showWarnings
def showMplsRsvpCounters( mode, args ):
   intfs = args.get( 'INTF' )
   af = None
   if 'ipv4' in args:
      af = 'ipv4'
   elif 'ipv6' in args:
      af = 'ipv6'

   model = RsvpModel.RsvpMessageCounters()
   rxColls = []
   txColls = []

   if af != 'ipv6':
      rxColls.append( rsvpMessageCounters.rxIpv4Count )
      txColls.append( rsvpMessageCounters.txIpv4Count )
   if af != 'ipv4':
      rxColls.append( rsvpMessageCounters.rxIpv6Count )
      txColls.append( rsvpMessageCounters.txIpv6Count )

   if intfs:
      intfNames = [ intf for intf in intfs.intfNames() ]

   def accept( counterKey ):
      if intfs:
         return counterKey.intfId in intfNames
      else:
         return True

   for rxColl in rxColls:
      for counterKey, counter in rxColl.iteritems():
         if accept( counterKey ):
            intfModel = model.intfModel( counterKey.intfId )
            mType = messageTypeFromTacc( counterKey.mType )
            intfModel.rx.countInc( mType, counter.count )

   for txColl in txColls:
      for counterKey, counter in txColl.iteritems():
         if accept( counterKey ):
            intfModel = model.intfModel( counterKey.intfId )
            mType = messageTypeFromTacc( counterKey.mType )
            intfModel.tx.countInc( mType, counter.count )

   return model

class MplsRsvpCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp counters [ ipv4 | ipv6 ] [ interface INTF ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'counters': 'Show RSVP message counters',
      'ipv4': 'Filter by ipv4',
      'ipv6': 'Filter by ipv6',
      'interface': 'Filter by interface',
      'INTF': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpCounters
   cliModel = RsvpModel.RsvpMessageCounters

BasicCli.addShowCommandClass( MplsRsvpCountersCmd )

# -----------------------------------------------------------------------------------
# show mpls rsvp bandwidth summary [interface <interfaceRange>]
# -----------------------------------------------------------------------------------
@showWarnings
def showMplsRsvpBandwidthSummary( mode, args ):
   intfs = args.get( 'INTF' )
   if intfs:
      intfNames = set( intfs.intfNames() )
   else:
      intfNames = set( rsvpBwClientUsage.reservedBandwidth )
      intfNames.update( rsvpBwClientStatus.maxReservableBandwidth )


   model = RsvpModel.RsvpBandwidthSummaryModel()
   model.interfaces = {}

   for intfId in intfNames:
      usage = rsvpBwClientUsage.reservedBandwidth.get( intfId )
      maxReservableBandwidth = \
            rsvpBwClientStatus.maxReservableBandwidth.get( intfId )

      if usage:
         bw = sum( usage.bandwidth.itervalues(), 0.0 )
      elif maxReservableBandwidth:
         bw = 0.0
      else:
         continue

      intfModel = RsvpModel.RsvpBandwidthIntfSummaryModel()
      # convert bytes-per-sec to bits-per-sec
      intfModel.reservedBandwidth = bw * 8
      if maxReservableBandwidth:
         intfModel.totalBandwidth = maxReservableBandwidth.bandwidth * 8
      else:
         intfModel.totalBandwidth = 0.0

      model.interfaces[ intfId ] = intfModel
   return model

class MplsRsvpBandwidthSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp bandwidth summary [ INTF ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'bandwidth': matcherBandwidth,
      'summary': matcherSummary,
      'INTF': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpBandwidthSummary
   cliModel = RsvpModel.RsvpBandwidthSummaryModel

BasicCli.addShowCommandClass( MplsRsvpBandwidthSummaryCmd )

# -----------------------------------------------------------------------------------
# show mpls rsvp bandwidth [interface <interfaceRange>]
# -----------------------------------------------------------------------------------
@showWarnings
def showMplsRsvpBandwidth( mode, args ):
   intfs = args.get( 'INTF' )
   if intfs:
      intfNames = set( intfs.intfNames() )
   else:
      intfNames = set( rsvpBwClientUsage.reservedBandwidth.keys() )
      intfNames.update( rsvpBwClientStatus.maxReservableBandwidth.keys() )

   model = RsvpModel.RsvpBandwidthModel()
   model.interfaces = {}

   for intfId in intfNames:
      usage = rsvpBwClientUsage.reservedBandwidth.get( intfId )
      maxReservableBandwidth = \
            rsvpBwClientStatus.maxReservableBandwidth.get( intfId )
      if not usage and not maxReservableBandwidth:
         continue

      intfModel = RsvpModel.RsvpBandwidthIntfModel()
      intfModel.reservedBandwidth = {}
      bandwidth = usage.bandwidth if usage else {}
      # convert bytes-per-sec to bits-per-sec
      for priority in range( 0, 8 ):
         intfModel.reservedBandwidth[ priority ] = \
               float( bandwidth.get( priority, 0 ) * 8 )
      if maxReservableBandwidth:
         intfModel.totalBandwidth = maxReservableBandwidth.bandwidth * 8
      else:
         intfModel.totalBandwidth = 0.0

      model.interfaces[ intfId ] = intfModel
   return model

class MplsRsvpBandwidthCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls rsvp bandwidth [ INTF ]'
   data = {
      'mpls': MplsCli.mplsNodeForShow,
      'rsvp': CliToken.Rsvp.rsvpMatcherForShow,
      'bandwidth': matcherBandwidth,
      'INTF': IntfCli.Intf.rangeMatcher,
   }
   handler = showMplsRsvpBandwidth
   cliModel = RsvpModel.RsvpBandwidthModel

BasicCli.addShowCommandClass( MplsRsvpBandwidthCmd )

#---
# Support for show tech-support
def _showTechCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   cmds = [
            'show mpls rsvp',
            'show mpls rsvp ip access-list',
            'show mpls rsvp ipv6 access-list',
            'show mpls rsvp counter',
            'show mpls rsvp neighbor',
            'show mpls rsvp session detail',
            'show mpls rsvp bandwidth',
            'show mpls rsvp session history',
          ]
   return cmds

def _showTechSummaryCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   return [ 'show mpls rsvp neighbor',
            'show mpls rsvp session summary',
          ]

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( '2018-07-24 16:30:00',
      _showTechCmds, summaryCmdCallback=_showTechSummaryCmds )

def Plugin( entityManager ):
   global routingVrfInfoDir
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                        "routing/vrf/routingInfo/status",
                                        "Tac::Dir", "ri" )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                                 "routing/hardware/route/status",
                                                 "Routing::Hardware::RouteStatus",
                                                 'r' )

   global mplsRoutingConfig
   mplsRoutingConfig = LazyMount.mount( entityManager,
                                        "routing/mpls/config",
                                        "Mpls::Config", 'r' )
   global rsvpCliConfig
   RsvpCliConfig = Tac.Type( "Rsvp::RsvpCliConfig" )
   rsvpCliConfig = LazyMount.mount( entityManager,
                                    RsvpCliConfig.mountPath,
                                    "Rsvp::RsvpCliConfig", 'r' )
   global rsvpConfig
   RsvpConfig = Tac.Type( "Rsvp::RsvpConfig" )
   rsvpConfig = LazyMount.mount( entityManager, RsvpConfig.mountPath,
                                 "Rsvp::RsvpConfig", 'r' )

   global rsvpStatus
   RsvpStatus = Tac.Type( "Rsvp::RsvpStatus" )
   rsvpStatus = LazyMount.mount( entityManager,
                                 RsvpStatus.mountPath,
                                 "Rsvp::RsvpStatus", 'rS' )

   global rsvpMessageCounters
   RsvpMessageCounter = Tac.Type( "Rsvp::Smash::MessageCounter" )
   rsvpMessageCounters = SmashLazyMount.mount(
                           entityManager,
                           RsvpMessageCounter.mountPath,
                           "Rsvp::Smash::MessageCounter",
                           SmashLazyMount.mountInfo( 'reader' ) )

   global sysname
   sysname = entityManager.sysname()

   global aclCpConfig
   aclCpConfig = LazyMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "r" )

   global aclStatus
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )

   global aclCheckpoint
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )

   BW_CLIENT_NAME = 'rsvpFrr'
   global rsvpBwClientUsage
   rsvpBwClientUsage = SmashLazyMount.mount(
         entityManager,
         BandwidthClientUsage.mountPath( DEFAULT_VRF, BW_CLIENT_NAME ),
         "TrafficEngineering::BandwidthClientUsage",
         SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpBwClientStatus
   rsvpBwClientStatus = SmashLazyMount.mount(
         entityManager,
         BandwidthClientStatus.mountPath( DEFAULT_VRF, BW_CLIENT_NAME ),
         "TrafficEngineering::BandwidthClientStatus",
         SmashLazyMount.mountInfo( 'reader' ) )
   global mplsHwCapability
   mplsHwCapability = LazyMount.mount( entityManager,
                                 "routing/hardware/mpls/capability",
                                 "Mpls::Hardware::Capability", 'r' )

   global rsvpSessionHistory
   RsvpSessionHistory = Tac.Type( "Rsvp::RsvpSessionHistory" )
   rsvpSessionHistory = SmashLazyMount.mount( entityManager,
                                              RsvpSessionHistory.mountPath,
                                              "Rsvp::RsvpSessionHistory",
                                              SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpSpHistory
   RsvpSpHistory = Tac.Type( "Rsvp::RsvpSpHistory" )
   rsvpSpHistory = SmashLazyMount.mount( entityManager, RsvpSpHistory.mountPath,
                                         "Rsvp::RsvpSpHistory",
                                         SmashLazyMount.mountInfo( 'reader' ) )
