# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys

from AgentDirectory import agentIsRunning
import BasicCli
import CliCommand
from CliMatcher import (
      DynamicNameMatcher,
      KeywordMatcher,
   )
import CliParser
import CliPlugin.RsvpLerModel as RsvpLerModel
import CliPlugin.TechSupportCli
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ShowCommand
import SmashLazyMount
import Tac
from TypeFuture import TacLazyType

# Type Alias
FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
FeatureState = TacLazyType( 'Ale::FlexCounter::ConfigState' )

RsvpLerLspGroupState = TacLazyType( "Rsvp::RsvpLerLspGroupState" )
RsvpLerLspRequestConfigId = TacLazyType( "Rsvp::RsvpLerLspRequestConfigId" )
RsvpLerPathSpecId = TacLazyType( "Rsvp::RsvpLerPathSpecId" )
RsvpLerTunnelState = TacLazyType( "Rsvp::RsvpLerTunnelState" )

# Tokens
teMatcherForShow = KeywordMatcher( "traffic-engineering",
      helpdesc="Traffic Engineering related information" )
rsvpMatcherForShow = KeywordMatcher( "rsvp", helpdesc="Show RSVP information" )
tunnelMatcherForShow = KeywordMatcher( "tunnel",
      helpdesc="RSVP tunnel information" )
lspDetailsMatcherForShow = KeywordMatcher( "lsp",
      helpdesc="Show LSP details" )
summaryMatcherForShow = KeywordMatcher( "summary",
      helpdesc="Show summarized information" )
detailMatcherForShow = KeywordMatcher( "detail",
      helpdesc="More comprehensive output" )
historyMatcherForShow = KeywordMatcher( "history",
      helpdesc="Show history information" )

# Globals
fcFeatureConfigDir = None
mplsRoutingConfig = None
routingVrfInfoDir = None
rsvpCliConfig = None
rsvpLerCliConfig = None
rsvpLerConfig = None
rsvpLerStatus = None
rsvpLerTunnelHistory = None
rsvpLerLspHistory = None
sysname = None
mplsHwCapability = None
routingHardwareRouteStatus = None

# RsvpLer is supported only on mpls supported platforms that have hfec enabled
def rsvpLerSupportedGuard( mode, token ):
   if ( mplsHwCapability.mplsSupported and
        routingHardwareRouteStatus.hierarchicalFecsEnabled ):
      return None
   else:
      return CliParser.guardNotThisPlatform

# Decorators
def showWarnings( showCmd ):
   def warningShowCmd( mode, *args, **kwargs ):
      routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
      if not rsvpCliConfig.enabled:
         mode.addWarning( "RSVP is not enabled" )
      if not mplsRoutingConfig.mplsRouting:
         mode.addWarning( "MPLS routing is not enabled" )
      if not ( routingInfo and routingInfo.routing ):
         mode.addWarning( "IP routing is not enabled" )
      if not rsvpAgentRunning():
         mode.addWarning( "Agent RSVP is not running" )
      if not mplsAgentRunning():
         mode.addWarning( "Agent MPLS is not running" )
      return showCmd( mode, *args, **kwargs )
   return warningShowCmd

# Helpers
def rsvpAgentRunning():
   return agentIsRunning( sysname, 'Rsvp' )

def mplsAgentRunning():
   return agentIsRunning( sysname, 'Mpls' )

def tunnelNameList( mode ):
   return [ tunnelSpecId.tunnelName for tunnelSpecId in
            rsvpLerStatus.tunnelSpecColl.tunnelSpec ]

def tunnelCountersEnabled():
   mplsFeatureConfigEnabled = \
      fcFeatureConfigDir.feature.get( FeatureId.MplsTunnel ) == FeatureState.enabled
   nhFeatureConfigDisabled = \
      fcFeatureConfigDir.feature.get( FeatureId.Nexthop ) != FeatureState.enabled
   # Return true if mpls tunnel counter feature is eanbled and nexthop group counter
   # feature is disabled. The reason for this is that currently nexthop-group
   # counters and mpls tunnel counters cannot be enabled simultaneously as the
   # mpls tunnel counter feature is reusing the nexthop-group's featureId.
   return mplsFeatureConfigEnabled and nhFeatureConfigDisabled

@showWarnings
def showTeRsvpTunnel( mode, args ):
   filters = args.get( 'FILTERS' )
   summary = "summary" in args
   lspDetails = "lsp" in args
   detail = "detail" in args
   history = "history" in args

   tacTunnelFilter = Tac.Value( "Rsvp::Cli::ShowRsvpLerTunnelFilter" )

   if filters:
      # filters [('tunnelFilter', ('tunnelName', 'myTunnel')),
      #         ]
      for filterType, arg in filters:
         if filterType == 'tunnelFilter':
            filterAttr, filterValue = arg
            if filterAttr in [ 'tunnelName' ]:
               setattr( tacTunnelFilter, filterAttr, filterValue )

   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   LazyMount.force( rsvpLerStatus )
   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpLerTunnel", rsvpCliConfig,
                               rsvpLerStatus, rsvpLerTunnelHistory,
                               rsvpLerLspHistory )
   renderer.render( socket, fmt, tacTunnelFilter, summary, lspDetails, detail,
                    history )
   return RsvpLerModel.RsvpLerTunnels

tunnelFilterSharedObj = object()

class RsvpLerTunnelFilterArg( CliCommand.CliExpression ):
   expression = ( '{ '
                  '( name NAME ) '
                  '}' )
   data = {
      'name': CliCommand.singleKeyword( 'name', helpdesc='Filter by tunnel name',
         sharedMatchObj=tunnelFilterSharedObj ),
      'NAME': DynamicNameMatcher( tunnelNameList, 'Tunnel name',
         pattern=CliParser.namePattern ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      filters = []
      if 'name' in args and 'NAME' in args:
         filters.append( ( 'tunnelFilter', ( 'tunnelName',
                                             args[ 'NAME' ][ 0 ] ) ) )

      if filters:
         args[ 'FILTERS' ] = filters

class TeRsvpTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show traffic-engineering rsvp tunnel [ FILTERS ] "
              "[ summary | lsp | detail | history ]" )
   data = {
         "traffic-engineering": teMatcherForShow,
         "rsvp": rsvpMatcherForShow,
         "tunnel": tunnelMatcherForShow,
         "FILTERS": RsvpLerTunnelFilterArg,
         "lsp": lspDetailsMatcherForShow,
         "summary": summaryMatcherForShow,
         "detail": detailMatcherForShow,
         "history": historyMatcherForShow,
         }
   handler = showTeRsvpTunnel
   cliModel = RsvpLerModel.RsvpLerTunnels

BasicCli.addShowCommandClass( TeRsvpTunnelCmd )

#------------------------------------------------------------------------------------
# show traffic-engineering rsvp
#------------------------------------------------------------------------------------
@showWarnings
def showTeRsvp( mode, args ):
   tunnelCount = 0
   tunnelUpCount = 0
   tunnelSecondaryEnabledCount = 0
   tunnelSecondaryInUseCount = 0
   lspCount = 0
   lspUpCount = 0

   if rsvpLerStatus.tunnelGroupConfigColl:
      tunnelGroupConfig = rsvpLerStatus.tunnelGroupConfigColl.tunnelGroupConfig
      tunnelCount = len( tunnelGroupConfig )
      tunnelSecondaryEnabledCount = len( [ t for t in tunnelGroupConfig.values()
                                           if t.secondaryPathSpec is not None ] )
   if rsvpLerStatus.tunnelGroupStatusColl:
      tunnelGroupStatus = rsvpLerStatus.tunnelGroupStatusColl.tunnelGroupStatus
      tunnelUpCount = len( [ t for t in tunnelGroupStatus.values() if
                             t.tunnelState is RsvpLerTunnelState.tunnelStateUp ] )
   if rsvpLerStatus.lspGroupStatusCollTable and rsvpLerStatus.tunnelStatusCollTable:
      lspGroupStatusColl = rsvpLerStatus.lspGroupStatusCollTable.lspGroupStatusColl
      tunnelStatusColl = rsvpLerStatus.tunnelStatusCollTable.tunnelStatusColl
      for tunnelConfigId, tunnel in lspGroupStatusColl.items():
         activeLspGroupId = None
         tunStatusColl = tunnelStatusColl.get( tunnelConfigId.tunnelSpecId )
         if tunStatusColl and tunnelConfigId in tunStatusColl.tunnelStatus:
            tunnelStatus = tunStatusColl.tunnelStatus[ tunnelConfigId ]
            activeLspGroupId = tunnelStatus.activeLspGroup
         for lsp in tunnel.lspGroupStatus.values():
            isLspGroupUp = \
               ( lsp.lspGroupState == RsvpLerLspGroupState.lspGroupStateUp or
                 lsp.lspGroupState == RsvpLerLspGroupState.lspGroupStateUpError )
            isSecondary = ( lsp.lspGroupId.groupId != 0 )
            isLspGroupInUse = ( activeLspGroupId is not None and
                                lsp.lspGroupId == activeLspGroupId )

            lspCount += ( lsp.activeLsp != RsvpLerLspRequestConfigId() )
            lspCount += ( lsp.pendingLsp != RsvpLerLspRequestConfigId() )
            lspUpCount += isLspGroupUp
            tunnelSecondaryInUseCount += ( isLspGroupUp and isLspGroupInUse and
                                           isSecondary )

   model = RsvpLerModel.RsvpLerState()
   model.localIntf = rsvpLerCliConfig.localIntf
   model.localIntfIp = rsvpLerConfig.localIntfIp
   model.autoBwSupported = tunnelCountersEnabled()

   model.tunnelCount = tunnelCount
   model.tunnelUpCount = tunnelUpCount
   model.tunnelSecondaryEnabledCount = tunnelSecondaryEnabledCount
   model.tunnelSecondaryInUseCount = tunnelSecondaryInUseCount
   model.lspCount = lspCount
   model.lspUpCount = lspUpCount
   model.optimizationInterval = rsvpLerConfig.optimizationInterval

   return model

class TeRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show traffic-engineering rsvp" )
   data = {
         "traffic-engineering": teMatcherForShow,
         "rsvp": rsvpMatcherForShow,
         }
   handler = showTeRsvp
   cliModel = RsvpLerModel.RsvpLerState

BasicCli.addShowCommandClass( TeRsvpCmd )

# Support for show tech-support
def _showTechCmds():
   if not mplsHwCapability.mplsSupported:
      return []
   cmds = [
            'show traffic-engineering rsvp',
            'show traffic-engineering rsvp tunnel detail',
            'show traffic-engineering rsvp tunnel lsp',
            'show traffic-engineering rsvp tunnel history',
          ]
   return cmds

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2020-04-02 15:00:00', _showTechCmds )

def Plugin( entityManager ):
   global sysname
   sysname = entityManager.sysname()

   global routingVrfInfoDir
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                        "routing/vrf/routingInfo/status",
                                        "Tac::Dir", "ri" )
   global rsvpLerStatus
   RsvpLerStatus = Tac.Type( "Rsvp::RsvpLerStatus" )
   rsvpLerStatus = LazyMount.mount( entityManager, RsvpLerStatus.mountPath,
                                    "Rsvp::RsvpLerStatus", "rS" )

   global mplsRoutingConfig
   mplsRoutingConfig = LazyMount.mount( entityManager,
                                        "routing/mpls/config",
                                        "Mpls::Config", "r" )

   global rsvpCliConfig
   RsvpCliConfig = Tac.Type( "Rsvp::RsvpCliConfig" )
   rsvpCliConfig = LazyMount.mount( entityManager,
                                    RsvpCliConfig.mountPath,
                                    "Rsvp::RsvpCliConfig", 'r' )

   global mplsHwCapability
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability", 'r' )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                                 "routing/hardware/route/status",
                                                 "Routing::Hardware::RouteStatus",
                                                 'r' )

   global rsvpLerCliConfig
   RsvpLerCliConfig = Tac.Type( "Rsvp::RsvpLerCliConfig" )
   rsvpLerCliConfig = LazyMount.mount( entityManager,
                                       RsvpLerCliConfig.mountPath,
                                       "Rsvp::RsvpLerCliConfig", 'r' )

   global rsvpLerTunnelHistory
   RsvpLerTunnelHistory = Tac.Type( "Rsvp::RsvpLerTunnelHistory" )
   rsvpLerTunnelHistory = SmashLazyMount.mount(
                            entityManager,
                            RsvpLerTunnelHistory.mountPath,
                            "Rsvp::RsvpLerTunnelHistory",
                            SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpLerLspHistory
   RsvpLerLspHistory = Tac.Type( "Rsvp::RsvpLerLspHistory" )
   rsvpLerLspHistory = SmashLazyMount.mount( entityManager,
                                             RsvpLerLspHistory.mountPath,
                                             "Rsvp::RsvpLerLspHistory",
                                             SmashLazyMount.mountInfo( 'reader' ) )

   global rsvpLerConfig
   RsvpLerConfig = Tac.Type( "Rsvp::RsvpLerConfig" )
   rsvpLerConfig = LazyMount.mount( entityManager,
                                    RsvpLerConfig.mountPath,
                                    "Rsvp::RsvpLerConfig", 'r' )

   global fcFeatureConfigDir
   fcFeatureConfigDir = LazyMount.mount( entityManager,
                                           'flexCounter/featureConfigDir/cliAgent',
                                           'Ale::FlexCounter::FeatureConfigDir',
                                           'r' )
