# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import IpGenericAddress
from IntfModels import Interface
from CliModel import ( Bool, DeferredModel, Dict, Enum, Float, Model, Submodel,
                       Int, Str, List )
from TypeFuture import TacLazyType

RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )

class RsvpLerError( Model ):
   code = Int( help="RSVP error code" )
   value = Int( help="RSVP error value" )
   node = IpGenericAddress( help="Router on which the error occurred" )

class RsvpLerTunnelPath( Model ):
   specName = Str( help="Path specification name" )
   specType = Enum( help="Path specification type",
                    values=( 'explicit', 'dynamic', 'unknown' ) )
   state = Enum( help="Path state",
                 values=( 'up', 'down', 'establishing', 'coldStandby', 'unknown' ) )
   cspfPathKeyId = Int( help="CSPF Path ID", optional=True )
   mbb = Bool( help="Path is being re-established with make-before-break" )
   inUse = Bool( help="Path is in use" )
   preSignaled = Bool( help="Path is pre-signaled", optional=True )
   hops = List( help="Path hops", valueType=IpGenericAddress )
   pathErrors = List( help="Errors on this path", valueType=RsvpLerError )
   # MTU signaling attributes (only shown in "detail", therefore optional)
   # Only shown when MTU signaling is enabled
   intfMtu = Int( help="Interface MTU", optional=True )
   pathMtu = Int( help="Path MTU", optional=True )

class RsvpLerAutoBandwidthParams( Model ):
   adjustmentPeriod = Float(
         help="Bandwidth adjustment period for auto bandwidth" )
   minBandwidth = Int(
         help="Minimum bandwidth for auto bandwidth in bps" )
   maxBandwidth = Int(
         help="Maximum bandwidth for auto bandwidth in bps" )
   lastSampledBandwidth = Int(
         help="Last sampled bandwidth for auto bandwidth in bps" )

class RsvpLerLsp( Model ):
   lspId = Int( help="LSP ID" )
   cspfPathKeyId = Int( help="CSPF Path ID", optional=True )
   pathType = Enum( help="Path type",
                    values=( 'primary', 'secondary', 'unknown' ) )
   pathSpecName = Str( help="Path specification name" )
   pathSpecType = Enum( help="Path specification type",
                        values=( 'explicit', 'dynamic', 'unknown' ) )
   bandwidth = Int( help="LSP requested bandwidth in bps" )
   state = Enum( help="LSP state",
                 values=( 'up', 'down', 'establishing', 'unknown' ) )
   inUse = Bool( help="LSP is in use" )
   lspErrors = List( help="Errors on this LSP", valueType=RsvpLerError )
   hops = List( help="Path hops", valueType=IpGenericAddress )

class RsvpLerHistoryEntry( Model ):
   timestamp = Float( help='Timestamp of the event' )
   event = Str( help="History event" )

class RsvpLerHistory( Model ):
   history = List( help="Recent history events", valueType=RsvpLerHistoryEntry )

class RsvpLerTunnel( Model ):
   source = IpGenericAddress( help="Destination address" )
   destination = IpGenericAddress( help="Destination address" )
   state = Enum( help="Tunnel state", values=( 'up', 'down', 'unknown' ) )
   mbb = Bool( help="Tunnel is getting re-established with make-before-break" )

   # Bandwidth-related attributes
   currentBandwidth = Int( help="Currently reserved bandwidth in bps" )
   requestedBandwidth = Int( help="Requested bandwidth in bps" )
   autoBandwidth = Bool( help="Bandwidth is automatically determined" )
   # Optional since only shown in "detail" and if autoBandwidth==True
   autoBandwidthParams = \
         Submodel( valueType=RsvpLerAutoBandwidthParams,
                   help="Parameters for tuning automagic bandwidth determination",
                   optional=True )
   # Optional since only shown in "detail"
   setupPriority = Int( help="Setup priority", optional=True )
   holdPriority = Int( help="Hold priority", optional=True )

   # MTU signaling attributes (only shown in "detail", therefore optional)
   mtuSignalingEnabled = Bool( help="MTU Signaling is enabled", optional=True )

   sessionId = Int( help="Session ID associated with this tunnel", optional=True )

   tunnelIndex = Int( help="Index of this tunnel in the tunnel FIB", optional=True )

   lspCount = Int( help="Number of LSPs in this tunnel" )

   # LDP Tunneling and IGP Shortcut eligibility flags. Optional since only in detail.
   ldpTunneling = Bool( help="Tunnel eligible to be used for LDP tunneling",
                        optional=True )
   igpShortcut = Bool( help="Tunnel eligible to be used for IGP shortcuts",
                       optional=True )

   activePathType = Enum( help="Active path type",
                          values=( 'primary', 'secondary', 'unknown' ) )

   # Path info (not included in "summary")
   primaryPath = Submodel( valueType=RsvpLerTunnelPath,
                           help="Primary path information",
                           optional=True )
   secondaryPaths = List( help="Secondary paths information",
                          valueType=RsvpLerTunnelPath,
                          optional=True )

   # Optional since only used in "history" variant
   tunnelHistory = Submodel( valueType=RsvpLerHistory, help='Tunnel history events',
                             optional=True )
   lspHistory = Dict( help="LSP history events keyed by LSP ID",
                      keyType=int, valueType=RsvpLerHistory, optional=True )

   # Optional since only used in "lsp" variant
   lsps = List( help="LSPs part of this tunnel", valueType=RsvpLerLsp,
                optional=True )

   # Tunnel optimization interval (only shown in "detail", therefore optional)
   optimizationInterval = Int( help="Optimization interval value in seconds",
                               optional=True )

class RsvpLerTunnels( DeferredModel ):
   '''Model for "show traffic-engineering rsvp tunnel"'''
   tunnels = Dict( help="Table of tunnel information keyed by tunnel name",
                   keyType=str,
                   valueType=RsvpLerTunnel )

class RsvpLerState( Model ):
   '''Model for "show traffic-engineering rsvp" '''
   localIntf = Interface( help="Local interface" )
   localIntfIp = IpGenericAddress( help="Local interface IP address" )
   tunnelCount = Int( help="Number of tunnels" )
   tunnelUpCount = Int( help="Number of tunnels that are up" )
   tunnelSecondaryEnabledCount = Int(
      help="Number of tunnels with secondary path configured" )
   tunnelSecondaryInUseCount = Int(
      help="Number of tunnels with secondary path in use" )
   lspCount = Int( help="Number of LSPs" )
   lspUpCount = Int( help="Number of LSPs that are up" )
   optimizationInterval = Int( help="Optimization interval value in seconds" )
   autoBwSupported = Bool( help="Is auto bandwidth supported" )

   def render( self ):
      print ( "Local interface: %s" % self.localIntf )
      print ( "IP address: %s" % self.localIntfIp )
      autoBwSupportStr = "supported" if self.autoBwSupported else "unsupported"
      print ( "Auto bandwidth: %s" % autoBwSupportStr )
      print ( "" )

      print ( "Number of tunnels: %d" % self.tunnelCount )
      print ( "Operational: %d" % self.tunnelUpCount )
      print ( "Secondary path enabled: %d" % self.tunnelSecondaryEnabledCount )
      print ( "Secondary path in use: %d" % self.tunnelSecondaryInUseCount )
      if self.optimizationInterval == RsvpLerConstants.optimizationIntervalDisabled:
         print ( "Periodic optimization: disabled" )
      else:
         print ( "Periodic optimization: enabled, interval: %d seconds" %
                 self.optimizationInterval )

      print ( "" )
      print ( "Number of LSPs: %d" % self.lspCount )
      print ( "Operational: %d" % self.lspUpCount )
