# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import difflib

import BasicCli
import CliCommand
from CliCommand import Node
from CliMode.RsvpLer import (
      RsvpTeMode,
      RsvpPathSpecMode,
      RsvpTunnelSpecMode,
      )
import CliMatcher
import CliParser
from CliPlugin.IpAddrMatcher import IpAddrMatcher as IpAddrMatcher
import CliPlugin.IntfCli as IntfCli
import CliPlugin.TeCli as TeCli
from CliPlugin.TeCli import RouterGlobalTeMode
import CliToken.RsvpLer
import ConfigMount
import LazyMount
from MultiRangeRule import MultiRangeMatcher
from RsvpLib import (
      dictToTaccForCli,
      bandwidthBitsToBytes,
      )
import ShowCommand
import Tac
from TypeFuture import TacLazyType
import Tracing
import Toggles.RsvpToggleLib
import Toggles.gatedToggleLib

def adminGroupRange():
   return ( 0, 31 )

t0 = Tracing.trace0

def IpGenAddr( *args, **kwargs ):
   # Need constness for dict keys
   return Tac.ValueConst( 'Arnet::IpGenAddr', *args, **kwargs )

CspfConstraintIdx = TacLazyType( "Cspf::ConstraintAttrIndex" )

FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
FeatureState = TacLazyType( 'Ale::FlexCounter::ConfigState' )

RsvpLerAutoBwParam = TacLazyType( 'Rsvp::RsvpLerAutoBwParam' )
RsvpLerCliConfig = TacLazyType( 'Rsvp::RsvpLerCliConfig' )
RsvpLerConstants = TacLazyType( 'Rsvp::RsvpLerConstants' )
RsvpLerPathSpec = TacLazyType( 'Rsvp::RsvpLerPathSpec' )
RsvpLerPathSpecHop = TacLazyType( 'Rsvp::RsvpLerPathSpecHop' )
RsvpLerPathSpecId = TacLazyType( 'Rsvp::RsvpLerPathSpecId' )
RsvpLerPathSpecType = TacLazyType( 'Rsvp::RsvpLerPathSpecType' )
RsvpLerTunnelSpecId = TacLazyType( 'Rsvp::RsvpLerTunnelSpecId' )

pathSpecDynamicType = RsvpLerPathSpecType.pathSpecDynamicType
pathSpecExplicitType = RsvpLerPathSpecType.pathSpecExplicitType
tunnelSourceCli = TacLazyType( 'Rsvp::RsvpLerTunnelSource' ).tunnelSourceCli

config = None
fcFeatureConfigDir = None

class RsvpTeConfigMode( RsvpTeMode, BasicCli.ConfigModeBase ):
   name = "RSVP LER Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RsvpTeMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#---------------------------------
# [ no | default ] rsvp
#---------------------------------
class CfgRsvpCmd( CliCommand.CliCommandClass ):
   syntax = 'rsvp'
   noOrDefaultSyntax = syntax
   data = {
      'rsvp': CliToken.RsvpLer.rsvpNodeForLerConfig,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RsvpTeConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.reset()

RouterGlobalTeMode.addCommandClass( CfgRsvpCmd )

#---------------------------------------------------------------
# Remove all rsvp configs when the parent is removed
# i.e., "no router traffic-engineering" in config mode.
#---------------------------------------------------------------
class TeSubMode( TeCli.TeDependentBase ):
   def setDefault( self ):
      config.reset()

#--------------------------------------------------------------------------------
# [ no | default ] local-interface INTF
#--------------------------------------------------------------------------------
class LocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': 'Local interface',
      'INTF': IntfCli.Intf.matcher,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      intfId = intf.name
      config.localIntf = intfId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.localIntf = ''

RsvpTeConfigMode.addCommandClass( LocalInterfaceCmd )

MAX_OPTIMIZATION_INTERVAL = 2**32 - 1
matcherOptimizationKeyword = CliMatcher.KeywordMatcher( 'optimization',
   helpdesc='Configure periodic tunnel optimization' )
matcherIntervalKeyword = CliMatcher.KeywordMatcher( 'interval',
   helpdesc='Time between tunnel optimizations' )
matcherInterval = CliMatcher.IntegerMatcher( 1, MAX_OPTIMIZATION_INTERVAL,
   helpdesc='Interval value' )
matcherSecondsKeyword = CliMatcher.KeywordMatcher( 'seconds',
   helpdesc='Optimization interval to be specified as seconds' )

class RsvpOptimizationCmd( CliCommand.CliCommandClass ):
   syntax = 'optimization interval INTERVAL seconds'
   noOrDefaultSyntax = 'optimization ...'
   data = {
      'optimization': matcherOptimizationKeyword,
      'interval': matcherIntervalKeyword,
      'INTERVAL': matcherInterval,
      'seconds': matcherSecondsKeyword,
   }

   @staticmethod
   def handler( mode, args ):
      config.optimizationInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.optimizationInterval = RsvpLerConstants.optimizationIntervalDisabled

if Toggles.RsvpToggleLib.toggleRsvpLerReoptimizationEnabled():
   RsvpTeConfigMode.addCommandClass( RsvpOptimizationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] path <name> ( explicit | dynamic )
#--------------------------------------------------------------------------------

class RsvpPathSpecConfigMode( RsvpPathSpecMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, pathSpecName, pathSpecType ):
      param = ( pathSpecName, pathSpecType )
      RsvpPathSpecMode.__init__( self, param=param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.pathSpecId = RsvpLerPathSpecId( self.pathSpecName, tunnelSourceCli )
      self.pathSpec = None
      self.pendingConfig = self.buildPendingConfigFromPathSpec()

   def getActivePathSpec( self, createIfNotExist=False ):
      pathSpec = None
      if createIfNotExist:
         pathSpec = config.pathSpecCollTable.pathSpecColl.\
               newMember( tunnelSourceCli ).pathSpec.\
               newMember( self.pathSpecId )
      else:
         pathSpecColl = config.pathSpecCollTable.pathSpecColl.get( tunnelSourceCli )
         pathSpec = pathSpecColl.pathSpec.get( self.pathSpecId ) \
                    if pathSpecColl else None
      return pathSpec

   def getPendingPathSpec( self ):
      hops = self.pendingConfig[ 'hops' ]
      excludeHops = self.pendingConfig[ 'excludeHops' ]
      includeAllAdminGroup = self.pendingConfig[ 'includeAllAdminGroup' ]
      includeAnyAdminGroup = self.pendingConfig[ 'includeAnyAdminGroup' ]
      excludeAdminGroup = self.pendingConfig[ 'excludeAdminGroup' ]
      pathSpec = self.buildPathSpec( hops, excludeHops=excludeHops,
                                     includeAllAdminGroup=includeAllAdminGroup,
                                     includeAnyAdminGroup=includeAnyAdminGroup,
                                     excludeAdminGroup=excludeAdminGroup )
      return pathSpec

   def buildPathSpec( self, hops, excludeHops=None, includeAllAdminGroup=0,
                      includeAnyAdminGroup=0, excludeAdminGroup=0, pathSpec=None ):
      '''
      If pathSpec is not provided, a new PathSpec (orphan) will be created.
      The pathSpec is then populated using the arguments passed to this function.
      Being able to create a temporary pathSpec is useful for `show pending` and
      `show diff`, when we need to check the difference between two objects,
      including one that is only 'pending' from now, i.e only exists through a
      pending configuration (self.pendingConfig).
      '''
      # Set default values
      excludeHops = excludeHops or set()
      # Create actual pathSpec if needed
      if pathSpec is None:
         pathId = RsvpLerPathSpecId( self.pathSpecName, tunnelSourceCli )
         pathSpec = RsvpLerPathSpec( pathId )

      # Build a config dict to use dictToTaccForCli
      includeHopList = []
      for i, ( hopIp, loose ) in enumerate( hops ):
         pathHop = RsvpLerPathSpecHop( hopIp, loose )
         includeHopList.append( ( i, pathHop ) )
      excludeHopList = []
      for excludeHop in excludeHops:
         excludeHopList.append( ( excludeHop, True ) )

      configDict = {
            'pathSpecType': self.pathSpecType,
            'includeHop': includeHopList,
            'excludeHop': excludeHopList,
            'includeAllAdminGroup': includeAllAdminGroup,
            'includeAnyAdminGroup': includeAnyAdminGroup,
            'excludeAdminGroup': excludeAdminGroup,
      }
      changed = dictToTaccForCli( configDict, pathSpec )
      if changed:
         pathSpec.version += 1
         pathSpec.changeCount = Tac.now()

      return pathSpec

   def buildPendingConfigFromPathSpec( self ):
      pendingConfig = { 'hops': [], 'excludeHops': set(),
                        'includeAllAdminGroup': 0,
                        'includeAnyAdminGroup': 0,
                        'excludeAdminGroup': 0 }
      pathSpec = self.getActivePathSpec( createIfNotExist=False )
      if pathSpec is None:
         return pendingConfig

      for pathSpecHop in pathSpec.includeHop.values():
         pendingConfig[ 'hops' ].append( ( pathSpecHop.hopIp, pathSpecHop.loose ) )
      for excludeHop in pathSpec.excludeHop.keys():
         pendingConfig[ 'excludeHops' ].add( excludeHop )
      pendingConfig[ 'includeAllAdminGroup' ] = pathSpec.includeAllAdminGroup
      pendingConfig[ 'includeAnyAdminGroup' ] = pathSpec.includeAnyAdminGroup
      pendingConfig[ 'excludeAdminGroup' ] = pathSpec.excludeAdminGroup
      return pendingConfig

   def commitContext( self ):
      if self.pendingConfig is None:
         # Abort
         return
      hops = self.pendingConfig[ 'hops' ]
      excludeHops = self.pendingConfig[ 'excludeHops' ]
      includeAllAdminGroup = self.pendingConfig[ 'includeAllAdminGroup' ]
      includeAnyAdminGroup = self.pendingConfig[ 'includeAnyAdminGroup' ]
      excludeAdminGroup = self.pendingConfig[ 'excludeAdminGroup' ]
      self.pathSpec = self.getActivePathSpec( createIfNotExist=True )
      self.buildPathSpec( hops, excludeHops=excludeHops,
                          includeAllAdminGroup=includeAllAdminGroup,
                          includeAnyAdminGroup=includeAnyAdminGroup,
                          excludeAdminGroup=excludeAdminGroup,
                          pathSpec=self.pathSpec )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      pathSpec = self.getPendingPathSpec()
      cliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )
      print( self.enterCmd() )
      for cliCmd in cliCmds:
         print( '   ' + cliCmd )

   def showDiff( self, args ):
      pathSpec = self.getPendingPathSpec()
      pendingCliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )
      # Current configuration
      pathSpec = self.getActivePathSpec( createIfNotExist=False )
      activeCliCmds = self.pathSpecToCliCmds( pathSpec, saveAll=False )

      # Generate diff
      diff = difflib.unified_diff( activeCliCmds, pendingCliCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

   def findIndex_( self, hopList, hopToFind ):
      return next( i for i, ( hop, _ ) in enumerate( hopList ) if hop == hopToFind )

class RsvpPathSpecExplicitConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER Explicit Path Specification"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecExplicitType )

class RsvpPathSpecDynamicConfigMode( RsvpPathSpecConfigMode ):
   name = "RSVP LER Dynamic Path Specification"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, pathSpecName ):
      RsvpPathSpecConfigMode.__init__( self, parent, session, pathSpecName,
                                       pathSpecType=pathSpecDynamicType )

def pathSpecNameList( mode ):
   cliPathSpecColl = config.pathSpecCollTable.pathSpecColl.get( tunnelSourceCli )
   if cliPathSpecColl is None:
      return []
   return [ pathSpec.pathSpecName for pathSpec in cliPathSpecColl.pathSpec.values() ]

class RsvpPathSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'path PATH_NAME PATH_MODE'
   noOrDefaultSyntax = 'path PATH_NAME ...'
   data = {
      'path': 'Enter the configuration mode for a new path specification',
      'PATH_NAME': CliMatcher.DynamicNameMatcher( pathSpecNameList,
                helpdesc='Path Specification name', pattern=r'.+' ),
      'PATH_MODE': CliMatcher.EnumMatcher( {
         'explicit': 'Explicit route specification',
         'dynamic': 'Dynamic route specification',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      pathSpecType = args[ 'PATH_MODE' ]
      pathSpecName = args[ 'PATH_NAME' ]
      if pathSpecType == 'explicit':
         childMode = mode.childMode( RsvpPathSpecExplicitConfigMode,
                                     pathSpecName=pathSpecName )
      else:
         childMode = mode.childMode( RsvpPathSpecDynamicConfigMode,
                                     pathSpecName=pathSpecName )
      # Check already existing entries with different pathSpecType
      activePathSpec = childMode.getActivePathSpec()
      if activePathSpec and activePathSpec.pathSpecType != childMode.pathSpecType:
         mode.addError( 'Changing the type of existing path specification '
                        'is not allowed' )
         return

      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pathSpecName = args[ 'PATH_NAME' ]
      pathSpecId = RsvpLerPathSpecId( pathSpecName, tunnelSourceCli )
      cliPathSpecColl = config.pathSpecCollTable.pathSpecColl.get( tunnelSourceCli )
      if cliPathSpecColl is None:
         return

      if pathSpecId in cliPathSpecColl.pathSpec:
         del cliPathSpecColl.pathSpec[ pathSpecId ]

RsvpTeConfigMode.addCommandClass( RsvpPathSpecCmd )

matcherHopBefore = CliMatcher.KeywordMatcher( 'before',
   helpdesc='Specify the hop which this hop should be placed directly before' )
matcherHopAfter = CliMatcher.KeywordMatcher( 'after',
    helpdesc='Specify the hop which this hop should be placed directly after' )
matcherHopExclude = CliMatcher.KeywordMatcher( 'exclude',
      helpdesc='Exclude this hop from the dynamic path computation' )
matcherHopLoose = CliMatcher.KeywordMatcher( 'loose',
      helpdesc='Define this hop as a loose hop' )
matcherHop = IpAddrMatcher( helpdesc='IPv4 Address' )
matcherHopKeyword = CliMatcher.KeywordMatcher( 'hop',
                    helpdesc='Hop along the path to include or exclude' )

#--------------------------------------------------------------------------------
# Common Path (Explicit/Dynamic) commands
#--------------------------------------------------------------------------------

class RsvpPathSpecAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit path configuration without committing changes'
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

RsvpPathSpecExplicitConfigMode.addCommandClass( RsvpPathSpecAbortCmd )
RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecAbortCmd )

class RsvpPathSpecShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show the pending path specification'
   }
   handler = RsvpPathSpecConfigMode.showPending

RsvpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )
RsvpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowPendingCmd )

class RsvpPathSpecShowDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show a diff between active and pending path specification'
   }
   handler = RsvpPathSpecConfigMode.showDiff

RsvpPathSpecExplicitConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )
RsvpPathSpecDynamicConfigMode.addShowCommandClass( RsvpPathSpecShowDiffCmd )

class RsvpPathSpecGenericHopCmd( CliCommand.CliCommandClass ):

   @staticmethod
   def handler( mode, args ):
      pendingConfigHops = mode.pendingConfig[ 'hops' ]
      hop = IpGenAddr( args[ 'HOP_ADDR' ] )
      maxHop = CspfConstraintIdx.max - CspfConstraintIdx.min + 1
      exclude = 'exclude' in args
      if exclude:
         # Marking a hop as excluded
         pendingConfigExcludeHops = mode.pendingConfig[ 'excludeHops' ]
         if len( pendingConfigExcludeHops ) >= maxHop:
            mode.addWarning(
               'Can not configure more than %d exclude hops, ignoring...' % maxHop )
            return
         hop = IpGenAddr( args[ 'HOP_ADDR' ] )
         pendingConfigExcludeHops.add( hop )
      else:
         # Marking a hop as included
         if len( pendingConfigHops ) >= maxHop:
            mode.addWarning(
               'Can not configure more than %d hops, ignoring...' % maxHop )
            return
         loose = 'loose' in args
         after = 'after' in args
         before = 'before' in args
         extHopAddr = IpGenAddr( args.get( 'EXT_HOP_ADDR' ) or '' )
         existingHops = [ pendingHop for pendingHop, _ in pendingConfigHops ]
         if extHopAddr and extHopAddr not in existingHops:
            mode.addWarning( '{} has not been configured for this path, '
                             'ignoring...'.format( extHopAddr ) )
            return
         if after:
            prevIndex = mode.findIndex_( pendingConfigHops, extHopAddr )
            pendingConfigHops.insert( prevIndex + 1, ( hop, loose ) )
         elif before:
            prevIndex = mode.findIndex_( pendingConfigHops, extHopAddr )
            pendingConfigHops.insert( prevIndex, ( hop, loose ) )
         else:
            pendingConfigHops.append( ( hop, loose ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'HOP_ADDR' not in args:
         mode.pendingConfig[ 'hops' ] = []
         mode.pendingConfig[ 'excludeHops' ].clear()
      else:
         pendingConfigHops = mode.pendingConfig[ 'hops' ]
         hop = IpGenAddr( args[ 'HOP_ADDR' ] )
         mode.pendingConfig[ 'hops' ] = [ ( pendingHop, loose ) for pendingHop, loose
                                          in pendingConfigHops if pendingHop != hop ]
         if hop in mode.pendingConfig[ 'excludeHops' ]:
            mode.pendingConfig[ 'excludeHops' ].remove( hop )

#--------------------------------------------------------------------------------
# Explicit Path Mode commands
#--------------------------------------------------------------------------------

class RsvpPathSpecExplicitHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = 'hop HOP_ADDR [ ( before | after ) EXT_HOP_ADDR ]'
   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopKeyword,
      'HOP_ADDR': matcherHop,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }

RsvpPathSpecExplicitConfigMode.addCommandClass( RsvpPathSpecExplicitHopCmd )

#--------------------------------------------------------------------------------
# Dynamic Path Mode commands
#--------------------------------------------------------------------------------

class RsvpPathSpecDynamicHopCmd( RsvpPathSpecGenericHopCmd ):
   syntax = (
      '''hop HOP_ADDR
         [ exclude | ( [ loose ] [ ( before | after ) EXT_HOP_ADDR ] ) ]'''
      if Toggles.RsvpToggleLib.toggleRsvpLerIncludeHopEnabled() else
      'hop HOP_ADDR exclude' )

   noOrDefaultSyntax = 'hop [ HOP_ADDR ]'
   data = {
      'hop': matcherHopKeyword,
      'HOP_ADDR': matcherHop,
      'exclude': matcherHopExclude,
      'loose': matcherHopLoose,
      'before': matcherHopBefore,
      'after': matcherHopAfter,
      'EXT_HOP_ADDR': matcherHop,
   }

RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecDynamicHopCmd )

matcherAdminGroupInc = CliMatcher.KeywordMatcher( 'include',
            helpdesc='Must include the following admin groups' )
matcherAdminGroupExc = CliMatcher.KeywordMatcher( 'exclude',
            helpdesc='Must exclude the following admin groups' )
matcherAdminGrpIds = MultiRangeMatcher( adminGroupRange, False,
                                        'Admininistrative Group Ids' )

def rangeToValue( rangeList ):
   bitmask = 0
   if rangeList:
      for i in rangeList:
         bitmask += 1 << i
   return bitmask

# Admin group Cli syntax:
# administrative-group include all <num> include any <num> exclude <num>
#
# where 'include all', 'include any', and 'exclude' can appear at most
# only once. Since the 'include' keyword is repeated twice, and it can not
# appear twice in the 'syntax' below, it is referred to as 'include_all'
# and 'include_any'.
class RsvpPathSpecDynamicAdminGrpCmd( CliCommand.CliCommandClass ):
   syntax = '''administrative-group
                { ( include_all all ID_RANGE_INCL_ALL ) |
                  ( include_any any ID_RANGE_INCL_ANY ) |
                  ( exclude ID_RANGE_EXCL ) }'''
   noOrDefaultSyntax = 'administrative-group'
   data = {
      'administrative-group': 'Administrative groups for that path',
      'include_all': Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'all': 'Include all of the following admin groups',
      'ID_RANGE_INCL_ALL': matcherAdminGrpIds,
      'include_any': Node( matcher=matcherAdminGroupInc, maxMatches=1 ),
      'any': 'Include any of the following admin groups',
      'ID_RANGE_INCL_ANY': matcherAdminGrpIds,
      'exclude': Node( matcher=matcherAdminGroupExc, maxMatches=1 ),
      'ID_RANGE_EXCL': matcherAdminGrpIds,
   }

   @staticmethod
   def handler( mode, args ):
      incAllRange = args.get( 'ID_RANGE_INCL_ALL' )
      incAnyRange = args.get( 'ID_RANGE_INCL_ANY' )
      excRange = args.get( 'ID_RANGE_EXCL' )
      # Convert the include values to bitmask
      incAllMask = rangeToValue( incAllRange[ 0 ].values() if incAllRange else [] )
      incAnyMask = rangeToValue( incAnyRange[ 0 ].values() if incAnyRange else [] )
      # Convert the exclude values to bitmask
      excMask = rangeToValue( excRange[ 0 ].values() if excRange else [] )

      mode.pendingConfig[ 'includeAllAdminGroup' ] = incAllMask
      mode.pendingConfig[ 'includeAnyAdminGroup' ] = incAnyMask
      mode.pendingConfig[ 'excludeAdminGroup' ] = excMask

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'includeAllAdminGroup' ] = 0
      mode.pendingConfig[ 'includeAnyAdminGroup' ] = 0
      mode.pendingConfig[ 'excludeAdminGroup' ] = 0


RsvpPathSpecDynamicConfigMode.addCommandClass( RsvpPathSpecDynamicAdminGrpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel <name>
#--------------------------------------------------------------------------------

class RsvpTunnelSpecConfigMode( RsvpTunnelSpecMode, BasicCli.ConfigModeBase ):
   name = "RSVP LER Tunnel Specification"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunnelSpecName ):
      param = tunnelSpecName
      RsvpTunnelSpecMode.__init__( self, param=param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.tunnelSpecId = RsvpLerTunnelSpecId( self.tunnelSpecName, tunnelSourceCli )
      self.tunnelSpec = None
      self.pendingConfig = self.buildPendingConfigFromTunnelSpec()

   def getActiveTunnelSpec( self, create=False ):
      tunnelSpec = None
      if create:
         tunnelSpec = config.tunnelSpecCollTable.tunnelSpecColl.\
               newMember( tunnelSourceCli ).tunnelSpec.\
               newMember( self.tunnelSpecId )
      else:
         tunnelSpecColl = config.tunnelSpecCollTable.tunnelSpecColl.get(
                             tunnelSourceCli )
         tunnelSpec = tunnelSpecColl.tunnelSpec.get( self.tunnelSpecId ) \
                    if tunnelSpecColl else None
      return tunnelSpec

   def getPendingTunnelSpec( self ):
      tunnelSpec = self.buildTunnelSpec( **self.pendingConfig )
      return tunnelSpec

   def buildTunnelSpec(
         self,
         enabled=False,
         dstIp=IpGenAddr(),
         primaryPath=RsvpLerPathSpecId(),
         secondaryPath=RsvpLerPathSpecId(),
         secondaryPathPreSignaled=False,
         explicitBandwidth=0,
         autoBw=False,
         autoBwParam=RsvpLerAutoBwParam(),
         setupPriority=RsvpLerConstants.defaultSetupPriority,
         holdPriority=RsvpLerConstants.defaultHoldPriority,
         eligibleForLdpTunneling=False,
         eligibleForIgpShortcut=False,
         optimizationInterval=RsvpLerConstants.optimizationIntervalDisabled,
         optimizationExplicitlyDisabled=False,
         tunnelSpec=None ):

      '''
      If tunnelSpec is not provided, a new TunnelSpec (orphan) will be created.
      The tunnelSpec is then populated using the arguments passed to this function.
      Being able to create a temporary tunnelSpec is useful for `show pending` and
      `show diff`, when we need to check the difference between two objects,
      including one that is only 'pending' from now, i.e only exists through a
      pending configuration (self.pendingConfig).
      '''
      # Create actual tunnelSpec if needed
      if tunnelSpec is None:
         tunnelId = RsvpLerTunnelSpecId( self.tunnelSpecName, tunnelSourceCli )
         tunnelSpec = Tac.newInstance( 'Rsvp::RsvpLerTunnelSpec', tunnelId )

      configDict = {
            'enabled': enabled,
            'dstIp': dstIp,
            'primaryPath': primaryPath,
            'secondaryPath': secondaryPath,
            'secondaryPathPreSignaled': secondaryPathPreSignaled,
            'explicitBandwidth': explicitBandwidth,
            'autoBw': autoBw,
            'autoBwParam': autoBwParam,
            'setupPriority': setupPriority,
            'holdPriority': holdPriority,
            'eligibleForLdpTunneling': eligibleForLdpTunneling,
            'eligibleForIgpShortcut': eligibleForIgpShortcut,
            'optimizationInterval': optimizationInterval,
            'optimizationExplicitlyDisabled': optimizationExplicitlyDisabled
      }
      changed = dictToTaccForCli( configDict, tunnelSpec )
      if changed:
         tunnelSpec.changeCount = Tac.now()

      return tunnelSpec

   def buildPendingConfigFromTunnelSpec( self ):
      pendingConfig = {
         'enabled': False,
         'dstIp': IpGenAddr(),
         'primaryPath': RsvpLerPathSpecId(),
         'secondaryPath': RsvpLerPathSpecId(),
         'secondaryPathPreSignaled': False,
         'explicitBandwidth': 0,
         'autoBw': False,
         'autoBwParam': RsvpLerAutoBwParam(),
         'setupPriority': RsvpLerConstants.defaultSetupPriority,
         'holdPriority': RsvpLerConstants.defaultHoldPriority,
         'eligibleForLdpTunneling': False,
         'eligibleForIgpShortcut': False,
         'optimizationInterval': RsvpLerConstants.optimizationIntervalDisabled,
         'optimizationExplicitlyDisabled': False
      }
      tunnelSpec = self.getActiveTunnelSpec( create=False )
      if tunnelSpec is None:
         return pendingConfig

      pendingConfig[ 'enabled' ] = tunnelSpec.enabled
      pendingConfig[ 'dstIp' ] = tunnelSpec.dstIp
      pendingConfig[ 'primaryPath' ] = tunnelSpec.primaryPath
      pendingConfig[ 'secondaryPath' ] = tunnelSpec.secondaryPath
      pendingConfig[ 'secondaryPathPreSignaled' ] = \
         tunnelSpec.secondaryPathPreSignaled
      pendingConfig[ 'explicitBandwidth' ] = tunnelSpec.explicitBandwidth
      pendingConfig[ 'autoBw' ] = tunnelSpec.autoBw
      pendingConfig[ 'autoBwParam' ] = tunnelSpec.autoBwParam
      pendingConfig[ 'setupPriority' ] = tunnelSpec.setupPriority
      pendingConfig[ 'holdPriority' ] = tunnelSpec.holdPriority
      pendingConfig[ 'eligibleForLdpTunneling' ] = tunnelSpec.eligibleForLdpTunneling
      pendingConfig[ 'eligibleForIgpShortcut' ] = tunnelSpec.eligibleForIgpShortcut
      pendingConfig[ 'optimizationInterval' ] = tunnelSpec.optimizationInterval
      pendingConfig[ 'optimizationExplicitlyDisabled' ] = \
         tunnelSpec.optimizationExplicitlyDisabled

      return pendingConfig

   def commitContext( self ):
      if self.pendingConfig is None:
         # Abort
         return

      self.tunnelSpec = self.getActiveTunnelSpec( create=True )
      self.buildTunnelSpec( tunnelSpec=self.tunnelSpec, **self.pendingConfig )

   def abort( self ):
      self.pendingConfig = None
      self.session_.gotoParentMode()

   def showPending( self, args ):
      tunnelSpec = self.getPendingTunnelSpec()
      cliCmds = self.tunnelSpecToCliCmds( tunnelSpec, saveAll=False )
      print( self.enterCmd() )
      for cliCmd in cliCmds:
         print( '   ' + cliCmd )

   def showDiff( self, args ):
      tunnelSpec = self.getPendingTunnelSpec()
      pendingCliCmds = self.tunnelSpecToCliCmds( tunnelSpec, saveAll=False )
      # Current configuration
      tunnelSpec = self.getActiveTunnelSpec( create=False )
      activeCliCmds = self.tunnelSpecToCliCmds( tunnelSpec, saveAll=False )

      # Generate diff
      diff = difflib.unified_diff( activeCliCmds, pendingCliCmds, lineterm='' )
      diff = list( diff )
      if diff:
         # Remove control lines (---,+++, '@@')
         diff = diff[ 3 : ]
         print( '\n'.join( diff ) )

def tunSpecNameList( mode ):
   cliTunSpecColl = config.tunnelSpecCollTable.tunnelSpecColl.get( tunnelSourceCli )
   if cliTunSpecColl is None:
      return []
   return [ tunSpec.tunnelName for tunSpec in cliTunSpecColl.tunnelSpec.values() ]

class RsvpTunnelSpecCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNEL_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': 'Enter the configuration mode for a new tunnel specification',
      'TUNNEL_NAME': CliMatcher.DynamicNameMatcher( tunSpecNameList,
                     helpdesc='Tunnel name', pattern=r'.+' ),
   }

   @staticmethod
   def handler( mode, args ):
      tunnelSpecName = args[ 'TUNNEL_NAME' ]
      childMode = mode.childMode( RsvpTunnelSpecConfigMode,
                                  tunnelSpecName=tunnelSpecName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tunnelSpecName = args[ 'TUNNEL_NAME' ]
      tunnelSpecId = RsvpLerTunnelSpecId( tunnelSpecName, tunnelSourceCli )
      cliTunnelSpecColl = config.tunnelSpecCollTable.tunnelSpecColl.get(
                             tunnelSourceCli )
      if cliTunnelSpecColl is None:
         return

      del cliTunnelSpecColl.tunnelSpec[ tunnelSpecId ]

RsvpTeConfigMode.addCommandClass( RsvpTunnelSpecCmd )

class RsvpTunnelSpecAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Exit tunnel configuration without committing changes'
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecAbortCmd )

class RsvpTunnelSpecShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pending'
   data = {
      'pending': 'Show the pending tunnel specification'
   }
   handler = RsvpTunnelSpecConfigMode.showPending

RsvpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowPendingCmd )

class RsvpTunnelSpecShowDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show a diff between active and pending tunnel specification'
   }
   handler = RsvpTunnelSpecConfigMode.showDiff

RsvpTunnelSpecConfigMode.addShowCommandClass( RsvpTunnelSpecShowDiffCmd )

#--------------------------------------------------------------------------------
# Tunnel Mode commands
#--------------------------------------------------------------------------------

class RsvpTunnelSpecShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable this tunnel',
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'enabled' ] = False

   @staticmethod
   def noHandler( mode, args ):
      mode.pendingConfig[ 'enabled' ] = True

   # default is the tunnel to be shutdown
   defaultHandler = handler

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecShutdownCmd )

class RsvpTunnelSpecDstIpCmd( CliCommand.CliCommandClass ):
   syntax = 'destination ip IP_ADDR'
   noOrDefaultSyntax = 'destination ip ...'
   data = {
      'destination': 'Tunnel destination',
      'ip': 'IP address of the destination',
      'IP_ADDR': IpAddrMatcher( helpdesc='IPv4 Address' ),
   }

   @staticmethod
   def handler( mode, args ):
      dstIp = IpGenAddr( args[ 'IP_ADDR' ] )
      mode.pendingConfig[ 'dstIp' ] = dstIp

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'dstIp' ] = IpGenAddr()

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecDstIpCmd )

class RsvpTunnelSpecPathCmd( CliCommand.CliCommandClass ):
   syntax = 'path PATH_NAME [ secondary [ pre-signaled ] ]'
   noOrDefaultSyntax = 'path [ PATH_NAME ] [ secondary ] ...'
   data = {
      'path': 'Primary path specification name',
      'PATH_NAME': CliMatcher.DynamicNameMatcher( pathSpecNameList,
                   helpdesc='Path Specification name', pattern=r'.+' ),
      'secondary': 'Secondary path specification name',
      'pre-signaled': 'Make this secondary path pre-signaled',
   }

   @staticmethod
   def handler( mode, args ):
      # Create pathSepcId from passed path name
      path = args[ 'PATH_NAME' ]
      pathSpecId = RsvpLerPathSpecId( path, tunnelSourceCli )
      if 'secondary' in args:
         mode.pendingConfig[ 'secondaryPath' ] = pathSpecId
         mode.pendingConfig[ 'secondaryPathPreSignaled' ] = 'pre-signaled' in args
      else:
         mode.pendingConfig[ 'primaryPath' ] = pathSpecId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      path = args.get( 'PATH_NAME' )
      if path is not None:
         # Create pathSepcId from passed path name and remove primary if
         # if secondary keyword is not passed and path spec matches
         # the existing primary spec. Likewise if secondary keyword is
         # passed remove secondary if path spec matches that of
         # existing secondary.
         pathSpecId = RsvpLerPathSpecId( path, tunnelSourceCli )
         if not 'secondary' in args and pathSpecId == \
               mode.pendingConfig[ 'primaryPath' ]:
            mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()
         if 'secondary' in args and pathSpecId == \
               mode.pendingConfig[ 'secondaryPath' ]:
            mode.pendingConfig[ 'secondaryPath' ] = RsvpLerPathSpecId()
            mode.pendingConfig[ 'secondaryPathPreSignaled' ] = False
      else:
         # No path name specified, remove all primary and secondary paths.
         mode.pendingConfig[ 'primaryPath' ] = RsvpLerPathSpecId()
         mode.pendingConfig[ 'secondaryPath' ] = RsvpLerPathSpecId()
         mode.pendingConfig[ 'secondaryPathPreSignaled' ] = False

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecPathCmd )

MAX_ADJUSTMENT_PERIOD = 2**32 - 1
matcherBwValueKeyword = CliMatcher.FloatMatcher( 0, 1000,
                        helpdesc='Bandwidth value', precisionString='%.2f' )
matcherBwUnitKeyword = CliMatcher.EnumMatcher( {
                       'bps': 'bandwidth in bits per second',
                       'kbps': 'bandwidth in kilobits per second',
                       'mbps': 'bandwidth in megabits per second',
                       'gbps': 'bandwidth in gigabits per second',
                       } )

class RsvpTunnelSpecBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = '''bandwidth (
                  ( FIXED_BW FIXED_BW_UNIT ) |
                  ( auto min MIN_BW MIN_BW_UNIT max MAX_BW MAX_BW_UNIT
                       [ adjustment-period ADJUSTMENT_PERIOD ] ) )'''
   noOrDefaultSyntax = 'bandwidth'
   data = {
      'bandwidth': 'Bandwidth requirement of the tunnel',
      'FIXED_BW': matcherBwValueKeyword,
      'FIXED_BW_UNIT': matcherBwUnitKeyword,
      'auto': 'Auto bandwidth',
      'min': 'Minimum bandwidth requirement of the tunnel',
      'MIN_BW': matcherBwValueKeyword,
      'MIN_BW_UNIT': matcherBwUnitKeyword,
      'max': 'Maximum bandwidth requirement of the tunnel',
      'MAX_BW': matcherBwValueKeyword,
      'MAX_BW_UNIT': matcherBwUnitKeyword,
      'adjustment-period': 'Minimal time between bw adjustments',
      'ADJUSTMENT_PERIOD': CliMatcher.IntegerMatcher( 1, MAX_ADJUSTMENT_PERIOD,
         helpdesc='Interval in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      # Convert Bandwidth to Bps
      bwMultiplier = { 'bps': 1, 'kbps': 1e3, 'mbps': 1e6, 'gbps': 1e9 }
      if 'auto' in args:
         if FeatureId.MplsTunnel not in fcFeatureConfigDir.feature or \
            fcFeatureConfigDir.feature[ FeatureId.MplsTunnel ] == \
               FeatureState.disabled:
            # Warn the user if mpls tunnel counter feature is not enabled
            mode.addWarning( "Autobandwidth support is not "
              "available when the mpls tunnel counter feature is disabled" )
         elif FeatureId.Nexthop in fcFeatureConfigDir.feature and \
            fcFeatureConfigDir.feature[ FeatureId.Nexthop ] != \
               FeatureState.disabled:
            # Warn the user if nexthop group counter feature is enabled. The reason
            # for this is that currently nexthop-group counters and mpls tunnel
            # counters cannot be enabled simultaneously as the mpls tunnel counter
            # feature is reusing the nexthop-group's featureId.
            mode.addWarning( "Autobandwidth support is not "
              "available when the nexthop group counter feature is enabled" )

         # First reset explicit bw config related param.
         mode.pendingConfig[ 'explicitBandwidth' ] = 0

         # Set autoBw to True and set other autoBwParam
         mode.pendingConfig[ 'autoBw' ] = True
         adjustmentPeriod = args.get( 'ADJUSTMENT_PERIOD', 0 )
         minBw = long( args[ 'MIN_BW' ] * bwMultiplier[ args[ 'MIN_BW_UNIT' ] ] )
         if minBw % 8 > 0:
            mode.addWarning( "Minimal bandwidth will be rounded to whole bytes" )
         minBw = bandwidthBitsToBytes( minBw ) # convert to bytes-per-sec
         maxBw = long( args[ 'MAX_BW' ] * bwMultiplier[ args[ 'MAX_BW_UNIT' ] ] )
         if maxBw % 8 > 0:
            mode.addWarning( "Maximal bandwidth will be rounded to whole bytes" )
         maxBw = bandwidthBitsToBytes( maxBw ) # convert to bytes-per-sec
         if minBw > maxBw:
            mode.addWarning(
            "Minimal bandwidth configured to be greater than maximal bandwidth." +
            " Will be set to %s %s." % ( args[ 'MAX_BW' ], args[ 'MAX_BW_UNIT' ] ) )
         minBw = min( minBw, maxBw )
         autoBwParam = RsvpLerAutoBwParam( adjustmentPeriod, minBw, maxBw )
         mode.pendingConfig[ 'autoBwParam' ] = autoBwParam
      else:
         # First reset auto bw config related param.
         mode.pendingConfig[ 'autoBw' ] = False
         mode.pendingConfig[ 'autoBwParam' ] = RsvpLerAutoBwParam()

         explicitBw = \
            long( args[ 'FIXED_BW' ] * bwMultiplier[ args[ 'FIXED_BW_UNIT' ] ] )
         if explicitBw % 8 > 0:
            mode.addWarning( "Bandwidth will be rounded to whole bytes" )
         explicitBw = bandwidthBitsToBytes( explicitBw ) # convert to bytes-per-sec
         mode.pendingConfig[ 'explicitBandwidth' ] = explicitBw

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'autoBw' ] = False
      mode.pendingConfig[ 'autoBwParam' ] = RsvpLerAutoBwParam()
      mode.pendingConfig[ 'explicitBandwidth' ] = 0

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecBandwidthCmd )

MIN_PRI = 0
MAX_PRI = 7
matcherPriority = CliMatcher.IntegerMatcher( MIN_PRI, MAX_PRI, helpdesc='Priority' )

class RsvpTunnelSpecPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'priority setup SETUP_PRIORITY hold HOLD_PRIORITY'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': 'Tunnel priority',
      'setup': 'Setup priority',
      'SETUP_PRIORITY': matcherPriority,
      'hold': 'Hold priority',
      'HOLD_PRIORITY': matcherPriority,
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'setupPriority' ] = args[ 'SETUP_PRIORITY' ]
      mode.pendingConfig[ 'holdPriority' ] = args[ 'HOLD_PRIORITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'setupPriority' ] = RsvpLerConstants.defaultSetupPriority
      mode.pendingConfig[ 'holdPriority' ] = RsvpLerConstants.defaultHoldPriority

RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecPriorityCmd )

class RsvpTunnelSpecLdpTunnelingCmd( CliCommand.CliCommandClass ):
   syntax = 'tunneling ldp'
   noOrDefaultSyntax = 'tunneling ...'
   data = {
      'tunneling': 'Tunneling configuration',
      'ldp': 'Allow RSVP tunnel to be used for LDP tunneling',
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'eligibleForLdpTunneling' ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'eligibleForLdpTunneling' ] = False

if Toggles.RsvpToggleLib.toggleLdpOverRsvpEnabled():
   RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecLdpTunnelingCmd )

class RsvpTunnelSpecIgpShortcutCmd( CliCommand.CliCommandClass ):
   syntax = 'igp shortcut'
   noOrDefaultSyntax = 'igp ...'
   data = {
      'igp': 'IGP configuration',
      'shortcut': 'Allow RSVP tunnel to be used for IGP shortcuts',
   }

   @staticmethod
   def handler( mode, args ):
      mode.pendingConfig[ 'eligibleForIgpShortcut' ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pendingConfig[ 'eligibleForIgpShortcut' ] = False

if Toggles.gatedToggleLib.toggleIsisIgpShortcutEnabled():
   RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecIgpShortcutCmd )

class RsvpTunnelSpecOptimizationCmd( CliCommand.CliCommandClass ):
   syntax = 'optimization ( ( interval INTERVAL seconds ) | disabled )'
   noOrDefaultSyntax = 'optimization ...'
   data = {
      'optimization': matcherOptimizationKeyword,
      'interval': matcherIntervalKeyword,
      'INTERVAL': matcherInterval,
      'seconds': matcherSecondsKeyword,
      'disabled': 'Tunnel optimization disabled',
   }

   @staticmethod
   def handler( mode, args ):
      interval = args.get( 'INTERVAL',
                           RsvpLerConstants.optimizationIntervalDisabled )
      mode.pendingConfig[ 'optimizationInterval' ] = interval
      mode.pendingConfig[ 'optimizationExplicitlyDisabled' ] = 'disabled' in args

   noOrDefaultHandler = handler

if Toggles.RsvpToggleLib.toggleRsvpLerReoptimizationEnabled():
   RsvpTunnelSpecConfigMode.addCommandClass( RsvpTunnelSpecOptimizationCmd )

#--------------------------------------------------------------------------------
# Plugin
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   config = ConfigMount.mount( entityManager, RsvpLerCliConfig.mountPath,
                               'Rsvp::RsvpLerCliConfig', 'w' )
   global fcFeatureConfigDir
   fcFeatureConfigDir = LazyMount.mount( entityManager,
                                           'flexCounter/featureConfigDir/cliAgent',
                                           'Ale::FlexCounter::FeatureConfigDir',
                                           'r' )

   TeCli.TeModeDependents.registerDependentClass( TeSubMode, priority=20 )
