# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpmwith %{_libdir}/libRsvp.so*

from __future__ import absolute_import, division, print_function

import AclCliLib
import AclLib
import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
from CliMode.Rsvp import RsvpMode
import CliPlugin.AclCli as AclCli
import CliPlugin.MplsCli as MplsCli
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliToken.Rsvp
import CliParser
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ReversibleSecretCli
from RsvpLib import bandwidthBitsToBytes
import Tac
import Tracing
import Toggles.RsvpToggleLib

t0 = Tracing.trace0

RsvpCliConfig = Tac.Type( 'Rsvp::RsvpCliConfig' )
config = None
routingHardwareRouteStatus = None
aclConfig = None
aclCpConfig = None

EthType = Tac.Type( 'Arnet::EthType' )

RsvpLspRequestId = Tac.Type( 'Rsvp::RsvpLspRequestId' )
FrrMode = Tac.Type( 'Rsvp::Cli::FrrMode' )
SrlgMode = Tac.Type( 'Rsvp::Cli::SrlgMode' )
LabelLocalTerminationMode = Tac.Type( 'Rsvp::Cli::LabelLocalTerminationMode' )

class RsvpConfigMode( RsvpMode, BasicCli.ConfigModeBase ):
   name = 'Rsvp Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName
      RsvpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------------------------------
# Cli Tokens
#------------------------------------------------------------------------------------
authKwMatcher = CliMatcher.KeywordMatcher( 'authentication',
                        helpdesc='Configure cryptographic authentication' )
matcherFastReroute = CliMatcher.KeywordMatcher( 'fast-reroute',
                        helpdesc='Configure fast reroute' )

# Constants
MAX_INDEX = 2**32 - 1
MAX_INTERVAL = 2**16 - 1
MAX_MUL = 2**8 - 1
MAX_WINDOW = 2**8 - 1
MAX_PREEMPTION_TIMER = 65535

#--------------------------------------------------------------------------------
# [ no | default ] mpls rsvp
#--------------------------------------------------------------------------------
class MplsRsvpCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls rsvp'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MplsCli.mplsNodeForConfig,
      'rsvp': CliToken.Rsvp.rsvpMatcherForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      rsvpMode = mode.childMode( RsvpConfigMode )
      mode.session_.gotoChildMode( rsvpMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for aclType in [ 'ip', 'ipv6' ]:
         _noServiceAcl( mode, aclType )
      config.reset()

BasicCliModes.GlobalConfigMode.addCommandClass( MplsRsvpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable RSVP',
   }

   @staticmethod
   def handler( mode, args ):
      config.enabled = False

   @staticmethod
   def noHandler( mode, args ):
      config.enabled = True

   # default is the to be shutdown
   defaultHandler = handler

RsvpConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mtu signaling
#--------------------------------------------------------------------------------
class MtuSignalingCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu signaling'
   noOrDefaultSyntax = syntax
   data = {
      'mtu' : 'MTU parameters',
      'signaling' : 'Enable MTU signaling',
   }

   @staticmethod
   def handler( mode, args ):
      config.mtuSignalingEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.mtuSignalingEnabled = False

RsvpConfigMode.addCommandClass( MtuSignalingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] refresh interval INTERVAL
#--------------------------------------------------------------------------------
class RefreshIntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh interval INTERVAL'
   noOrDefaultSyntax = 'refresh interval ...'
   data = {
      'refresh' : 'Configure neighbor state refresh',
      'interval' : 'Time between refreshes',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, MAX_INTERVAL,
         helpdesc='Interval in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      config.refreshInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.refreshInterval = config.refreshIntervalDefault

RsvpConfigMode.addCommandClass( RefreshIntervalIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] refresh method METHOD
#--------------------------------------------------------------------------------
class RefreshMethodCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh method METHOD'
   noOrDefaultSyntax = 'refresh method ...'
   data = {
      'refresh': 'Configure neighbor state refresh',
      'method': 'Neighbor refresh mechanism',
      'METHOD': CliMatcher.EnumMatcher( {
         'explicit': 'Send each message individually',
         'bundled': 'Refresh states using message identifier lists',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'METHOD' ] == 'explicit':
         config.refreshMethod = 'refreshExplicit'
      else:
         config.refreshMethod = 'refreshBundled'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.refreshMethod = config.refreshMethodDefault

RsvpConfigMode.addCommandClass( RefreshMethodCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hello interval INTERVAL multiplier MULTIPLIER
#--------------------------------------------------------------------------------
class HelloIntervalMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'hello interval INTERVAL multiplier MULTIPLIER'
   noOrDefaultSyntax = 'hello interval ...'
   data = {
      'hello': 'Configure hello messages',
      'interval': 'Time between hello messages',
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Interval in units of seconds' ),
      'multiplier': 'Timeout multiplier',
      'MULTIPLIER': CliMatcher.IntegerMatcher( 1, 255,
         helpdesc='Number of missed hellos after which the neighbor is expired' ),
   }
   @staticmethod
   def handler( mode, args ):
      config.helloInterval = args[ 'INTERVAL' ]
      config.helloMultiplier = args[ 'MULTIPLIER' ]

   @staticmethod
   def noHandler( mode, args ):
      config.helloInterval = config.helloIntervalDisabled
      config.helloMultiplier = config.helloMultiplierDefault

   @staticmethod
   def defaultHandler( mode, args ):
      config.helloInterval = config.helloIntervalDefault
      config.helloMultiplier = config.helloMultiplierDefault

RsvpConfigMode.addCommandClass( HelloIntervalMultiplierCmd )

#--------------------------------------------------------------------------------
# [ no | default ] authentication type AUTH_TYPE
#--------------------------------------------------------------------------------
class AuthenticationTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication type AUTH_TYPE'
   noOrDefaultSyntax = 'authentication type ...'
   data = {
      'authentication': authKwMatcher,
      'type': 'Authentication mechanism',
      'AUTH_TYPE': CliMatcher.EnumMatcher( {
         'none': 'No Authentication mechanism',
         'md5': 'MD5 Hash',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      config.authType = args[ 'AUTH_TYPE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.authType = config.authTypeDefault

RsvpConfigMode.addCommandClass( AuthenticationTypeCmd )

#--------------------------------------------------------------------------------
#  [ no | default ] authentication sequence-number window WINDOW
#--------------------------------------------------------------------------------
class AuthenticationSequenceNumberWindowWindowCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication sequence-number window WINDOW'
   noOrDefaultSyntax = 'authentication sequence-number window ...'
   data = {
      'authentication': authKwMatcher,
      'sequence-number': 'Index in the sequence',
      'window': 'Reorder window size',
      'WINDOW': CliMatcher.IntegerMatcher( 1, MAX_WINDOW,
         helpdesc='Size of reorder window' ),
   }
   
   @staticmethod
   def handler( mode, args ):
      config.authSeqWindowSize = args[ 'WINDOW' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.authSeqWindowSize = config.authSeqWindowSizeDefault

RsvpConfigMode.addCommandClass( AuthenticationSequenceNumberWindowWindowCmd )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] authentication index INDEX password PASSWORD
#------------------------------------------------------------------------------------
authIndexKwMatcher = CliMatcher.KeywordMatcher( 'index',
                                                helpdesc='Password index' )
authIndexIntegerMatcher = CliMatcher.IntegerMatcher( 1, MAX_INDEX,
                                                     helpdesc='Password index' )

class AuthCommand( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX password PASSWORD'
   noOrDefaultSyntax = 'authentication index INDEX password ...'
   data = {
      'authentication': authKwMatcher,
      'index': authIndexKwMatcher,
      'INDEX': authIndexIntegerMatcher,
      'password': 'Password',
      'PASSWORD':
      ReversibleSecretCli.reversibleSecretCliExpression( 'PASSWORD' )
     }

   @staticmethod
   def handler( mode, args ):
      config.encodedPassword[ args[ 'INDEX' ] ] = args[ 'PASSWORD' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del config.encodedPassword[ args[ 'INDEX' ] ]

RsvpConfigMode.addCommandClass( AuthCommand )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] authentication index INDEX active
#------------------------------------------------------------------------------------
class ActiveAuthCommand( CliCommand.CliCommandClass ):
   syntax = 'authentication index INDEX active'
   noOrDefaultSyntax = syntax
   data = {
      'authentication': authKwMatcher,
      'index': authIndexKwMatcher,
      'INDEX': authIndexIntegerMatcher,
      'active': 'Use index as active password'
     }

   @staticmethod
   def handler( mode, args ):
      index = args[ 'INDEX' ]
      if index not in config.encodedPassword:
         mode.addWarning( 'Password with index %s has not been configured' % index )
      config.activePasswordIndex = index

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.activePasswordIndex = 0

RsvpConfigMode.addCommandClass( ActiveAuthCommand )

#--------------------------------------------------------------------------------
# [ no | default ] ip access-group ACL_NAME [ in ]
# [ no | default ] ipv6 access-group ACL_NAME [ in ]
#--------------------------------------------------------------------------------
def _setServiceAcl( mode, aclName, aclType='ip' ):
   AclCliLib.setServiceAcl( mode, 'rsvp', AclLib.IPPROTO_RSVP,
                            aclConfig, aclCpConfig, aclName, aclType )

def _noServiceAcl( mode, aclType='ip' ):
   AclCliLib.noServiceAcl( mode, 'rsvp', aclConfig, aclCpConfig, None, aclType )

class IpAccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ in ]'
   data = {
      'ip': AclCli.ipKwForServiceAclMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL_NAME': AclCli.ipAclNameMatcher,
      'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      _setServiceAcl( mode, args[ 'ACL_NAME' ], aclType='ip' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _noServiceAcl( mode, aclType='ip' )

RsvpConfigMode.addCommandClass( IpAccessGroupAclnameCmd )

class Ipv6AccessGroupAclnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ in ]'
   data = {
      'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL_NAME': AclCli.ip6AclNameMatcher,
      'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      _setServiceAcl( mode, args[ 'ACL_NAME' ], aclType='ipv6' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _noServiceAcl( mode, aclType='ipv6' )

RsvpConfigMode.addCommandClass( Ipv6AccessGroupAclnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] fast-reroute mode FRR_MODE
#--------------------------------------------------------------------------------
class FastRerouteModeCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute mode FRR_MODE'
   noOrDefaultSyntax = 'fast-reroute mode ...'
   data = {
      'fast-reroute': matcherFastReroute,
      'mode': 'Fast reroute mode',
      'FRR_MODE': CliMatcher.EnumMatcher( {
         'link-protection': 'Protect against failure of the next link',
         'none': 'Disable fast reroute',
         'node-protection': 'Protect against failure of the next node',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      frrStrToMode = {
         'node-protection' : FrrMode.frrNodeProtection,
         'link-protection' : FrrMode.frrLinkProtection,
         'none' : FrrMode.frrNone,
      }
      frrMode = args[ 'FRR_MODE' ]
      frrModeValue = frrStrToMode.get( frrMode )
      if frrModeValue != FrrMode.frrNone and \
         not routingHardwareRouteStatus.hierarchicalFecsEnabled:
         mode.addWarning( 'Fast Re-Route support is not '
                          'available without Hierarchical FECs' )
      config.frrMode = frrModeValue

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.frrMode = config.frrModeDefault

RsvpConfigMode.addCommandClass( FastRerouteModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] srlg [ strict ]
#--------------------------------------------------------------------------------
class SrlgModeCmd( CliCommand.CliCommandClass ):
   syntax = 'srlg [ strict ]'
   noOrDefaultSyntax = 'srlg ...'
   data = {
      'srlg': 'Select SRLG behavior',
      'strict': 'Apply strict SRLG constraint'
   }

   @staticmethod
   def handler( mode, args ):
      srlgMode = SrlgMode.srlgStrict if 'strict' in args else SrlgMode.srlgLoose
      config.srlgMode = srlgMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.srlgMode = config.srlgModeDefault

RsvpConfigMode.addCommandClass( SrlgModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] fast-reroute reversion REVERSION_MODE
#--------------------------------------------------------------------------------
class FastRerouteReversionCmd( CliCommand.CliCommandClass ):
   syntax = 'fast-reroute reversion REVERSION_MODE'
   noOrDefaultSyntax = 'fast-reroute reversion ...'
   data = {
      'fast-reroute': matcherFastReroute,
      'reversion': 'Select reversion behavior',
      'REVERSION_MODE': CliMatcher.EnumMatcher( {
         'local': 'Local revertive repair',
         'global': 'Global revertive repair',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'REVERSION_MODE' ] == 'local':
         config.reversionMode = 'reversionLocal'
      else:
         config.reversionMode = 'reversionGlobal'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.reversionMode = config.reversionModeDefault

RsvpConfigMode.addCommandClass( FastRerouteReversionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] preemption method METHOD [ timer TIMER ]
#--------------------------------------------------------------------------------
class PreemptionTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'preemption method METHOD [ timer TIMER ]'
   noOrDefaultSyntax = 'preemption method ...'
   data = {
      'preemption' : 'Configure preemption',
      'method' : 'Select preemption method',
      'METHOD' : CliMatcher.EnumMatcher( {
         'soft' : 'Soft preemption',
         'hard' : 'Hard preemption',
      } ),
      'timer' : 'Time limit for LSP teardown',
      'TIMER' : CliMatcher.IntegerMatcher( 1, MAX_PREEMPTION_TIMER,
         helpdesc='Timer value in units of seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'METHOD' ] == 'soft':
         config.preemptionTimer = config.preemptionTimerDefault
         if 'timer' in args:
            config.preemptionTimer = args[ 'TIMER' ]
      else:
         config.preemptionTimer = config.preemptionTimerDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.preemptionTimer = config.preemptionTimerDefault

RsvpConfigMode.addCommandClass( PreemptionTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] label local-termination { implicit-null | explicit-null }
#--------------------------------------------------------------------------------
class LabelLocalTerminationCmd( CliCommand.CliCommandClass ):
   syntax = 'label local-termination ( implicit-null | explicit-null )'
   noOrDefaultSyntax = 'label local-termination ...'
   data = {
      'label' : 'Label to be advertised',
      'local-termination' : 'Local termination label to be advertised',
      'implicit-null' : 'Advertise implicit null',
      'explicit-null' : 'Advertise explicit null'
   }

   @staticmethod
   def handler( mode, args ):
      termMode = LabelLocalTerminationMode.explicitNull if 'explicit-null' in args \
                    else LabelLocalTerminationMode.implicitNull
      config.labelLocalTerminationMode = termMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.labelLocalTerminationMode = config.labelLocalTerminationModeDefault

if Toggles.RsvpToggleLib.toggleRsvpExplicitNullEnabled():
   RsvpConfigMode.addCommandClass( LabelLocalTerminationCmd )

#------------------------------------------------------------------------------------
# (config-mpls-rsvp) # [no|default] debug tunnel id <id>
#                                                sub-id <subId>
#                                                src <srcIp
#                                                dst <dstIp>
#                                                ether-type <ethType>
#                                                lpd <node|link|off>
#                                                session-name <name>
#                                                [bandwidth <0-1000>
#                                                (bps|kbps|mbps|gbps)]
#                                                [preemption <soft|hard>]
#                                                [setup-pri <0-7>]
#                                                [hold-pri <0-7>]
#                                                hops <ip1> [<ip2 ...]
#------------------------------------------------------------------------------------
MIN_TUNNEL_ID = 0
MAX_TUNNEL_ID = 2**16 - 1
MIN_TUNNEL_SUBID = 0
MAX_TUNNEL_SUBID = 2**16 - 1
MIN_PRI = 0
MAX_PRI = 7

matcherPriority = CliMatcher.IntegerMatcher( MIN_PRI, MAX_PRI, helpdesc='Priority' )

class DebugTunnelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'debug tunnel id REQ_ID sub-id REQ_SUB_ID src SRC_IP dst DST_IP '
              ' ether-type ETH_TYPE lpd LPD session-name SESSION_NAME '
              ' [ bandwidth BANDWIDTH UNIT ] [ preemption PREEMPTION ] '
              ' [ setup-prio SETUP_PRIO ] [ hold-prio HOLD_PRIO ] '
              ' [ mtu-signaling ] hops { HOPS }' )
   noOrDefaultSyntax = 'debug tunnel [ id REQ_ID ] [ sub-id REQ_SUB_ID ] ...'
   data = {
      'debug': 'Debug configuration',
      'tunnel': 'Tunnel headend setup',
      'id': 'Tunnel identifier',
      'REQ_ID': CliMatcher.IntegerMatcher( MIN_TUNNEL_ID, MAX_TUNNEL_ID,
         helpdesc='Tunnel ID integer' ),
      'sub-id': 'Tunnel sub-ID',
      'REQ_SUB_ID': CliMatcher.IntegerMatcher( MIN_TUNNEL_SUBID, MAX_TUNNEL_SUBID,
         helpdesc='Tunnel sub-ID integer' ),
      'src': 'Tunnel source IP address',
      'SRC_IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' ),
      'dst': 'Tunnel desination IP address',
      'DST_IP': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' ),
      'ether-type': 'Tunnel EtherType',
      'ETH_TYPE': CliMatcher.EnumMatcher( {
         'ip': 'IP EtherType (0x800)',
         'ip6': 'IPv6 EtherType (0x86DD)',
         'mpls': 'MPLS EtherType (0x8847)',
      } ),
      'lpd': 'Tunnel local protection desired',
      'LPD': CliMatcher.EnumMatcher( {
         'node': 'Local protection on with node protection',
         'link': 'Local protection on with link protection',
         'off': 'Local protection off',
      } ),
      'session-name': 'Tunnel Session Name',
      'SESSION_NAME': CliMatcher.QuotedStringMatcher( helpdesc='Tunnel SessionName',
         pattern=CliParser.namePattern ),
      'bandwidth': 'Bandwidth characteristics of the flow',
      'BANDWIDTH': CliMatcher.FloatMatcher( 0, 1000,
         helpdesc='Bandwidth requested', precisionString='%.2f' ),
      'UNIT': CliMatcher.EnumMatcher( {
         'bps' : 'Maximum reservable bandwidth in bits per second',
         'kbps' : 'Maximum reservable bandwidth in kilobits per second',
         'mbps' : 'Maximum reservable bandwidth in megabits per second',
         'gbps' : 'Maximum reservable bandwidth in gigabits per second',
      } ),
      'preemption' : 'Desired LSP preemption mode',
      'PREEMPTION' : CliMatcher.EnumMatcher( {
         'soft' : 'Soft preemption',
         'hard' : 'Hard preemption',
      } ),
      'setup-prio' : 'Setup priority',
      'SETUP_PRIO' : matcherPriority,
      'hold-prio' : 'Hold priority',
      'HOLD_PRIO' : matcherPriority,
      'mtu-signaling' : 'Enable MTU Signaling',
      'hops': 'Tunnel Hops',
      'HOPS': IpGenAddrMatcher.IpGenAddrMatcher( helpdesc='IPv4 or IPv6 Address' )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      reqId = args[ 'REQ_ID' ]
      reqSubId = args[ 'REQ_SUB_ID' ]
      srcIp = args[ 'SRC_IP' ]
      dstIp = args[ 'DST_IP' ]
      ethType = args[ 'ETH_TYPE' ]
      lpd = args[ 'LPD' ]
      sessionName = args[ 'SESSION_NAME' ]
      bandwidth = None
      if 'bandwidth' in args:
         bandwidth = ( args[ 'BANDWIDTH' ], args[ 'UNIT' ] )
      spd = args.get( 'PREEMPTION' )
      setupPriority = args.get( 'SETUP_PRIO', 7 )
      holdPriority = args.get( 'HOLD_PRIO', 0 )
      mtuSignalingEnabled = 'mtu-signaling' in args
      hops = args[ 'HOPS' ]
      # pylint: disable-msg=W0102, dangerous-default-value
      lspReqId = RsvpLspRequestId( reqId, reqSubId )

      # Convert ethType to Enum value
      ethTypes = { 'ip' : EthType.ethTypeIp,
                   'ip6' : EthType.ethTypeIp6,
                   'mpls' : EthType.ethTypeMpls }
      ethType = ethTypes[ ethType ]

      # Identify if nodeProtectionDesired should be set
      npd = lpd == 'node'

      # Convert lpd to Bool
      lpd = ( lpd == 'node' or lpd == 'link' )

      # Convert Bandwidth to Bps
      bwMultiplier = { 'bps': 1, 'kbps': 1e3, 'mbps': 1e6, 'gbps': 1e9 }

      if bandwidth is None:
         bandwidthBps = 0
      else:
         unit = bandwidth[ 1 ]
         bw = long( bandwidth[ 0 ] * bwMultiplier[ unit ] )
         if bw % 8 > 0:
            mode.addWarning( "Bandwidth will be rounded to whole bytes" )
         bandwidthBps = bandwidthBitsToBytes( bw )

      # Convert spd to Bool
      spd = ( spd == 'soft' )

      oldReq = config.debugLspRequestColl.debugLspReq.get( lspReqId )
      # If this ID already has a request, and it has not changed, don't do anything
      if oldReq is not None:
         if ( oldReq.srcIp == srcIp
              and oldReq.dstIp == dstIp
              and oldReq.ethType == ethType
              and oldReq.setupPriority == setupPriority
              and oldReq.holdPriority == holdPriority
              and oldReq.localProtectionDesired == lpd
              and oldReq.nodeProtectionDesired == npd
              and oldReq.softPreemptionDesired == spd
              and oldReq.mtuSignalingEnabled == mtuSignalingEnabled
              and oldReq.sessionName == sessionName
              and oldReq.bandwidth == bandwidthBps
              and oldReq.hop.items() == list( enumerate( hops ) ) ):
            t0( 'Debug tunnel request has not changed' )
            return
         t0( 'Debug tunnel request has changed' )
      else:
         t0( 'New debug tunnel request' )

      # Request is new or has changed
      req = config.debugLspRequestColl.debugLspReq.newMember( lspReqId )
      req.srcIp = srcIp
      req.dstIp = dstIp
      req.ethType = ethType
      req.setupPriority = setupPriority
      req.holdPriority = holdPriority
      req.localProtectionDesired = lpd
      req.nodeProtectionDesired = npd
      req.softPreemptionDesired = spd
      req.mtuSignalingEnabled = mtuSignalingEnabled
      req.sessionName = sessionName
      req.bandwidth = bandwidthBps
      req.hop.clear()
      for i, hop in enumerate( hops ):
         req.hop[ i ] = hop
      req.version += 1

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      reqId = args.get( 'REQ_ID' )
      reqSubId = args.get( 'REQ_SUB_ID' )

      if reqId is not None and reqSubId is not None:
         lspReqId = RsvpLspRequestId( reqId, reqSubId )
         del config.debugLspRequestColl.debugLspReq[ lspReqId ]
      else:
         # Wipe all requests
         config.debugLspRequestColl.debugLspReq.clear()

RsvpConfigMode.addCommandClass( DebugTunnelCmd )

def Plugin( entityManager ):
   global config
   config = ConfigMount.mount( entityManager, RsvpCliConfig.mountPath,
                               'Rsvp::RsvpCliConfig', 'w' )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                           'routing/hardware/route/status',
                           'Routing::Hardware::RouteStatus',
                           'r' )

   global aclConfig, aclCpConfig
   aclConfig = ConfigMount.mount( entityManager, 'acl/config/cli',
                                  'Acl::Input::Config', 'w' )

   aclCpConfig = ConfigMount.mount( entityManager, 'acl/cpconfig/cli',
                                    'Acl::Input::CpConfig', 'w' )
