#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys
import BasicCli
import ShowCommand
from CliPlugin.RpkiCliModels import ( RpkiShowRoaModel,
                                      RpkiShowCacheModel,
                                      CacheModel,
                                      CacheDetailModel,
                                      RpkiShowCacheCounterModel,
                                      RpkiShowCacheErrorCounterModel,
                                      CacheCounterModel,
                                      CacheRxErrorCounterModel,
                                      CacheTxErrorCounterModel,
                                      RpkiShowRoaSummaryModel,
                                      CacheRoaSummaryModel,
                                      TcpInformationModel )
from CliPlugin.RoutingBgpCli import ( RouterBgpBaseMode,
                                      RouterBgpSharedModelet,
                                      deleteRouterBgpMacVrfHook,
                                      bgpNeighborConfig,
                                      configForVrf,
                                      PeerCliExpression )
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from CliPlugin.RouteMapCli import mapNameMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.IntfCli import Intf
from CliPlugin import TechSupportCli
from CliPlugin.Ssl import profileMatcher, sslMatcher
from CliSavePlugin.RoutingBgpCliSave import showBgpConfigInstanceCallbacks
from CliSavePlugin.RpkiCliSave import ( saveRpkiOriginValidationConfig,
                                        saveCacheConfig )
from TypeFuture import TacLazyType
from CliToken import ( RoutingBgp,
                       RoutingBgpShowCliTokens,
                       Clear )
from EosRpkiLib import getCacheMessage, getCacheUnusedMessage
import Tac
import LazyMount
import SmashLazyMount
import ConfigMount
import CliCommand
import CliMatcher
from CliMode.Rpki import ( RpkiCacheMode,
                           RpkiOriginValidationBaseMode,
                           RpkiTransportTcpMode,
                           RpkiTransportTlsMode )
import CliParser
import HostnameCli
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
from Arnet import Prefix, Ip6Prefix
import AgentCommandRequest
import Ark
import cStringIO
import Tracing
import ast
import Toggles.RpkiToggleLib

traceHandle = Tracing.Handle( 'RpkiCli' )
t0 = traceHandle.trace0

# pkgdeps: rpm RpkiLib-lib

rpkiConfig = None
rpkiStatus = None
rpkiStatistics = None
allVrfConfig = None
rpkiCacheCounterSmash = None
rpkiCacheCounterSnapshotSmash = None
l3Config = None
asnConfig = None
sslConfig = None

def getCacheNames( mode ):
   return [ cache.stringValue for cache in rpkiConfig.cacheConfig ]

rpkiKwMatcher = CliMatcher.KeywordMatcher(
   'rpki',
   helpdesc='Resource Public Key Infrastructure' )
rpkiCacheKwMatcher = CliMatcher.KeywordMatcher(
   'cache',
   helpdesc='Cache server instance' )
rpkiCacheNameMatcher = CliMatcher.DynamicNameMatcher(
   getCacheNames,
   'Name of cache server',
   pattern=r'.+',
   helpname='NAME' )
# Exclude 'counters' from the list of cache names, to prevent us from
# misinterpreting this token as a cache name in the show commands.
rpkiShowCacheNameMatcher = CliMatcher.DynamicNameMatcher(
   getCacheNames,
   'Name of cache server',
   pattern=r'(?!counters$).+',
   helpname='NAME' )
roaKwMatcher = CliMatcher.KeywordMatcher(
   'roa',
   helpdesc='Route Origin Authorizations (ROAs) obtained from configured cache '
   'server(s)' )

class RpkiCacheConfigMode( RpkiCacheMode, BasicCli.ConfigModeBase ):
   name = 'RPKI cache server configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, cacheName ):
      RpkiCacheMode.__init__( self, cacheName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      cacheId = Tac.Value( 'Rpki::CacheId', cacheName )
      self.rpkiCacheConfig = rpkiConfig.cacheConfig.newMember( cacheId )

#------------------------------------------------------------------------------------
# The "[ no | default ] rpki cache <name> command, in "config-router-bgp" mode.
#------------------------------------------------------------------------------------
class EnterRpkiCacheConfigMode( CliCommand.CliCommandClass ):
   syntax = 'rpki cache NAME'
   noOrDefaultSyntax = syntax
   data = {
      'rpki': rpkiKwMatcher,
      'cache': 'Configuration options related to RPKI cache server',
      'NAME': rpkiCacheNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      cacheName = args[ 'NAME' ]
      rpkiDefaults = Tac.Type( 'Rpki::RpkiDefaults' )
      invalidCacheNames = Tac.Type( 'Rpki::RpkiInvalidCacheName' )
      if len( cacheName ) > rpkiDefaults.cacheNameMaxSize:
         mode.addError( 'Cache name cannot exceed {} characters'.format(
            rpkiDefaults.cacheNameMaxSize ) )
         return
      if cacheName in invalidCacheNames.attributes:
         mode.addError(
            '"{}" is a keyword and cannot be used for a cache name'.format(
               cacheName ) )
         return

      childMode = mode.childMode( RpkiCacheConfigMode, cacheName=cacheName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cacheName = args[ 'NAME' ]
      del rpkiConfig.cacheConfig[ Tac.ValueConst( 'Rpki::CacheId', cacheName ) ]

#------------------------------------------------------------------------------------
# The "[ no | default ] host IP_OR_HOST [ vrf VRF ] [ port PORT ] command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiHostVrfPortCmd( CliCommand.CliCommandClass ):
   syntax = 'host IP_OR_HOST [ vrf VRF ] [ port PORT ]'
   noOrDefaultSyntax = 'host ...'
   data = {
      'host': 'Configure host of cache server',
      'IP_OR_HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
                    helpdesc='IPv4 address or IPv6 address or fully qualified '
                             'domain name of cache server' ),
      'vrf': 'Specify the VRF in which the cache server is configured',
      'VRF': CliMatcher.DynamicNameMatcher( lambda mode: allVrfConfig.vrf,
                                            'VRF name' ),
      'port': 'Configure transport port',
      'PORT': CliMatcher.IntegerMatcher( 0, 65535,
                                         helpdesc='Cache server port number' )
   }

   @staticmethod
   def handler( mode, args ):
      port = args.get( 'PORT', mode.rpkiCacheConfig.tcpPortDefault )
      mode.rpkiCacheConfig.port = port
      vrf = args.get( 'VRF', DEFAULT_VRF )
      mode.rpkiCacheConfig.vrf = vrf
      ipAddrOrHost = args[ 'IP_OR_HOST' ]
      mode.rpkiCacheConfig.host = ipAddrOrHost

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Set the default value
      if mode.rpkiCacheConfig.transportConfig.transportAuthenticationType == 'tcp':
         mode.rpkiCacheConfig.port = mode.rpkiCacheConfig.tcpPortDefault
      mode.rpkiCacheConfig.vrf = DEFAULT_VRF
      mode.rpkiCacheConfig.host = ''

#------------------------------------------------------------------------------------
# The "[ no | default ] refresh-interval < 1-86400 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiRefreshIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh-interval SECONDS'
   noOrDefaultSyntax = 'refresh-interval ...'
   data = {
      'refresh-interval': 'Configure refresh interval',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 86400,
                                            helpdesc='Number of seconds between '
                                                     'cache server poll' )
   }

   @staticmethod
   def handler( mode, args ):
      seconds = args.get( 'SECONDS', 0 )
      mode.rpkiCacheConfig.refreshInterval = seconds

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] retry-interval < 1-7200 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiRetryIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'retry-interval SECONDS'
   noOrDefaultSyntax = 'retry-interval ...'
   data = {
      'retry-interval': 'Configure retry interval',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 7200,
                                            helpdesc='Number of seconds between '
                                                     'cache server poll since '
                                                     'error' )
   }

   @staticmethod
   def handler( mode, args ):
      seconds = args.get( 'SECONDS', 0 )
      mode.rpkiCacheConfig.retryInterval = seconds

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] expire-interval < 600-172800 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiExpireIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'expire-interval SECONDS'
   noOrDefaultSyntax = 'expire-interval ...'
   data = {
      'expire-interval': 'Configure expire interval',
      'SECONDS': CliMatcher.IntegerMatcher( 600, 172800,
                                            helpdesc='Number of seconds to retain '
                                                     'data synced from cache server '
                                                     'after connection loss' )
      }

   @staticmethod
   def handler( mode, args ):
      seconds = args.get( 'SECONDS', 0 )
      mode.rpkiCacheConfig.expireInterval = seconds

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] preference < 0-10 > command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiPreferenceCmd( CliCommand.CliCommandClass ):
   syntax = 'preference VALUE'
   noOrDefaultSyntax = 'preference ...'
   data = {
      'preference': 'Configure cache server preference',
      'VALUE': CliMatcher.IntegerMatcher( 0, 10,
                                          helpdesc='Cache server preference. '
                                                   'Lower value means '
                                                   'higher preference.' )
   }

   @staticmethod
   def handler( mode, args ):
      preference = args[ 'VALUE' ]
      mode.rpkiCacheConfig.preference = preference

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Set the default value
      mode.rpkiCacheConfig.preference = mode.rpkiCacheConfig.preferenceDefault

#------------------------------------------------------------------------------------
# The "[ no | default ] local-interface <INTF> command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': 'Specify a local source interface to connect to cache '
                         'server',
      'INTF': Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ]
      mode.rpkiCacheConfig.sourceIntf = intf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.rpkiCacheConfig.sourceIntf = mode.rpkiCacheConfig.sourceIntfDefault

transportKwMatcher = CliMatcher.KeywordMatcher(
                        'transport',
                        helpdesc='Configure transport option' )

transportAuthenticationTypes = TacLazyType( 'Rpki::RpkiTransportAuthenticationType' )
transportConfigObject = TacLazyType( 'Rpki::RpkiTransportConfig' )

def updateTransportConfig( currentTransportConfig,
                           transportAuthenticationType=None,
                           tcpKeepaliveOptions=None,
                           sslProfileName=None ):
   '''
   Helper method that will set only the non none attribute passed as parameter and
   retain the values of other parameters in the transportConfig.
   '''
   if transportAuthenticationType is None:
      transportAuthenticationType = \
         currentTransportConfig.transportAuthenticationType
   if tcpKeepaliveOptions is None:
      tcpKeepaliveOptions = currentTransportConfig.tcpKeepaliveOptions
   if sslProfileName is None:
      sslProfileName = currentTransportConfig.sslProfileName
   transportConfig = Tac.Value( 'Rpki::RpkiTransportConfig',
                                transportAuthenticationType, tcpKeepaliveOptions,
                                sslProfileName )
   return transportConfig

class RpkiTransportTcpConfigMode( RpkiTransportTcpMode,
                                 BasicCli.ConfigModeBase ):
   name = 'RPKI cache tcp transport configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RpkiTransportTcpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.rpkiCacheConfig = parent.rpkiCacheConfig
      self.tcp = transportAuthenticationTypes.tcp

#------------------------------------------------------------------------------------
# The "[ no | default ] transport tcp command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class EnterRpkiTransportTcpConfigMode( CliCommand.CliCommandClass ):
   syntax = 'transport tcp'
   noOrDefaultSyntax = syntax
   data = {
      'transport': transportKwMatcher,
      'tcp': 'Unprotected TCP'
   }

   @staticmethod
   def handler( mode, args ):
      # This submode is mutually exclusive with 'transport tls' and 'transport ssh'
      # (TODO) so if the current auth type is not tcp, the config is overridden with
      # defaults.
      childMode = mode.childMode( RpkiTransportTcpConfigMode )
      currentTransportConfig = childMode.rpkiCacheConfig.transportConfig
      if currentTransportConfig.transportAuthenticationType != childMode.tcp:
         transportConfig = updateTransportConfig(
            currentTransportConfig,
            transportAuthenticationType=childMode.tcp,
            sslProfileName=transportConfigObject.sslProfileNameDefault )
         childMode.rpkiCacheConfig.transportConfig = transportConfig
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Restore default TCP transport Config
      mode.rpkiCacheConfig.transportConfig = \
            transportConfigObject.cacheTransportConfigDefault

#------------------------------------------------------------------------------------
# The "[ no | default ] keepalive IDLE_TIME PROBE_INTERVAL PROBE_COUNT" command
# in "config-rpki-cache-transport-tcp" mode.
#------------------------------------------------------------------------------------
class TcpKeepaliveCommand( CliCommand.CliCommandClass ):
   syntax = 'tcp keepalive IDLE_TIME PROBE_INTERVAL PROBE_COUNT'
   noOrDefaultSyntax = 'tcp keepalive ...'
   data = {
      'tcp': 'TCP connection options',
      'keepalive': 'Specify the RPKI cache\'s TCP keepalive parameters',
      'IDLE_TIME': CliMatcher.IntegerMatcher( 1, 86400,
         helpdesc='Idle time (seconds) before TCP keepalive' ),
      'PROBE_INTERVAL': CliMatcher.IntegerMatcher( 1, 3600,
         helpdesc='Interval (seconds) between TCP keepalive probes' ),
      'PROBE_COUNT': CliMatcher.IntegerMatcher( 1, 1000,
         helpdesc='Number of keepalive probes before closing connection' )
   }

   @staticmethod
   def handler( mode, args ):
      idleTime = args[ 'IDLE_TIME' ]
      probeInterval = args[ 'PROBE_INTERVAL' ]
      probeCount = args[ 'PROBE_COUNT' ]
      keepalive = Tac.Value( 'Arnet::TcpKeepaliveOptions',
                             idleTime, probeInterval, probeCount )
      transportConfig = updateTransportConfig(
         mode.rpkiCacheConfig.transportConfig,
         transportAuthenticationType=mode.tcp,
         tcpKeepaliveOptions=keepalive )
      mode.rpkiCacheConfig.transportConfig = transportConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      keepaliveDefault = transportConfigObject.tcpKeepaliveDefault
      transportConfig = updateTransportConfig(
         mode.rpkiCacheConfig.transportConfig,
         transportAuthenticationType=mode.tcp,
         tcpKeepaliveOptions=keepaliveDefault )
      mode.rpkiCacheConfig.transportConfig = transportConfig

class RpkiTransportTlsConfigMode( RpkiTransportTlsMode, BasicCli.ConfigModeBase ):
   name = 'RPKI cache server transport TLS configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RpkiTransportTlsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.rpkiCacheConfig = parent.rpkiCacheConfig
      self.tls = transportAuthenticationTypes.tls

#------------------------------------------------------------------------------------
# The "[ no | default ] transport tls command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class EnterRpkiTransportTlsConfigMode( CliCommand.CliCommandClass ):
   syntax = 'transport tls'
   noOrDefaultSyntax = syntax
   data = {
      'transport': transportKwMatcher,
      'tls': 'Transport Layer Security'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RpkiTransportTlsConfigMode )
      currentTransportConfig = childMode.rpkiCacheConfig.transportConfig
      if currentTransportConfig.transportAuthenticationType != childMode.tls:
         transportConfig = updateTransportConfig(
            currentTransportConfig, transportAuthenticationType=childMode.tls )
         childMode.rpkiCacheConfig.transportConfig = transportConfig
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.rpkiCacheConfig.transportConfig = \
         transportConfigObject.cacheTransportConfigDefault

#------------------------------------------------------------------------------------
# The "[ no | default ] ssl profile PROFILE_NAME command,
# in "config-rpki-cache-trans-tls" mode.
#------------------------------------------------------------------------------------
class RpkiTlsSslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': sslMatcher,
      'profile': profileMatcher,
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher( lambda mode:
                                                     sslConfig.profileConfig,
                                                     helpdesc='Profile name' ),
   }

   @staticmethod
   def handler( mode, args ):
      sslProfileName = args[ 'PROFILE_NAME' ]
      currentTransportConfig = mode.rpkiCacheConfig.transportConfig
      transportConfig = updateTransportConfig( currentTransportConfig,
                                               sslProfileName=sslProfileName )
      mode.rpkiCacheConfig.transportConfig = transportConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currentTransportConfig = mode.rpkiCacheConfig.transportConfig
      sslProfileNameDefault = transportConfigObject.sslProfileNameDefault
      transportConfig = updateTransportConfig( currentTransportConfig,
                                               sslProfileName=sslProfileNameDefault )
      mode.rpkiCacheConfig.transportConfig = transportConfig

rpkiRoaTable = None

ovMethodEnum = Tac.Type( "Routing::Bgp::Rpki::OriginValidationMethod" )

def ovMethod( args ):
   if 'local' in args:
      return ovMethodEnum.ovLocal
   elif 'community' in args:
      return ovMethodEnum.ovCommunity
   elif 'prefer-community' in args:
      return ovMethodEnum.ovPreferCommunity
   elif 'disabled' in args:
      return ovMethodEnum.ovDisabled
   else:
      assert False, "Unknown method %s" % args
      return None  # keep pylint happy

#------------------------------------------------------------------------------------
# The "show bgp rpki roa ( ipv4 | ipv6 ) [<prefix>]" command.
#------------------------------------------------------------------------------------
class RpkiShowRoas( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp rpki roa ( ipv4 [ PREFIX4 ] ) | ( ipv6 [ PREFIX6 ] )'''
   _prefixText = 'Only display ROAs that affect origin validation' \
                 ' of the specified prefix'
   data = {
      'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
      'rpki': rpkiKwMatcher,
      'roa': roaKwMatcher,
      'ipv4': 'IPv4 unicast address family',
      'ipv6': 'IPv6 unicast address family',
      'PREFIX4': IpAddrMatcher.IpPrefixMatcher(
         _prefixText,
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'PREFIX6': Ip6AddrMatcher.Ip6PrefixMatcher(
         _prefixText,
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
   }
   cliModel = RpkiShowRoaModel

   @staticmethod
   @ArBgpShowOutput( 'RpkiShowRoas', arBgpModeOnly=True )
   def handler( mode, args ):
      prefix = None
      asdot = asnConfig.isAsdotConfigured()
      if 'ipv6' in args:
         showRoaSm = Tac.newInstance( 'Rpki::RpkiShowRoa6Sm', rpkiRoaTable, asdot )
         prefix = args.get( 'PREFIX6' )
         if prefix:
            showRoaSm.prefix = Ip6Prefix( prefix )
      else:
         showRoaSm = Tac.newInstance( 'Rpki::RpkiShowRoa4Sm', rpkiRoaTable, asdot )
         prefix = args.get( 'PREFIX4' )
         if prefix:
            showRoaSm.prefix = Prefix( prefix )

      showRoaSm.sortRoas()
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      showRoaSm.printRoas( fd, fmt )
      return RpkiShowRoaModel

def generateBgpRpkiCacheExpr( counters=False ):
   class RpkiShowCacheExpression( CliCommand.CliExpression ):
      expression = '''bgp rpki cache [ NAME ]'''
      data = {
         'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
         'rpki': rpkiKwMatcher,
         'cache': rpkiCacheKwMatcher,
         'NAME': rpkiShowCacheNameMatcher,
      }
      if counters:
         expression += ' counters'
         data[ 'counters' ] = "Cache server PDU counters"
   return RpkiShowCacheExpression

def generateBgpRpkiClearCacheExpr( counters=False ):
   class RpkiClearCacheExpression( CliCommand.CliExpression ):
      expression = '''bgp rpki cache ( all | NAME )'''
      data = {
         'bgp': 'Bgp',
         'rpki': rpkiKwMatcher,
         'cache': rpkiCacheKwMatcher,
         'all': 'Clear all cache server instances',
         'NAME': rpkiShowCacheNameMatcher,
      }
      if counters:
         expression += ' counters'
         data[ 'counters' ] = "Cache server PDU and error PDU counters"
   return RpkiClearCacheExpression

#------------------------------------------------------------------------------------
# The "show bgp rpki roa summary" command.
#-----------------------------------------------------------------------------------
class RpkiShowRoaSummary( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp rpki roa summary '''
   data = {
      'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
      'rpki': rpkiKwMatcher,
      'roa': roaKwMatcher,
      'summary': 'Summarized ROA information'
   }
   cliModel = RpkiShowRoaSummaryModel

   @staticmethod
   @ArBgpShowOutput( 'RpkiShowRoaSummary', arBgpModeOnly=True )
   def handler( mode, args ):
      model = RpkiShowRoaSummaryModel()

      allCacheStatus = rpkiStatus.cacheStatus

      if not allCacheStatus:
         msg = getCacheMessage( None )
         mode.addMessage( msg )

      for cacheId, status in allCacheStatus.iteritems():
         summaryModel = CacheRoaSummaryModel()
         summaryModel.ipv4RoaCount = status.ip4RoaCount
         summaryModel.ipv6RoaCount = status.ip6RoaCount

         model.caches[ cacheId.stringValue ] = summaryModel

      model.total = CacheRoaSummaryModel()
      model.total.ipv4RoaCount = len( rpkiRoaTable.ip4Roa )
      model.total.ipv6RoaCount = len( rpkiRoaTable.ip6Roa )
      return model

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>]" command.
#------------------------------------------------------------------------------------

def createSubmodel( cacheId, status, mode, model, detail ):
   connectionData = rpkiStatistics.cacheConnectionData.get( cacheId )
   if not connectionData:
      return
   submodel = CacheModel()
   submodel.host = status.host
   submodel.port = status.port
   submodel.vrf = status.vrf
   submodel.refreshInterval = status.effectiveRefreshInterval
   submodel.retryInterval = status.effectiveRetryInterval
   submodel.expireInterval = status.effectiveExpireInterval
   submodel.preference = status.preference
   submodel.state = status.cacheState
   submodel.setProtocolVersion( Tac.enumValue( 'Rpki::RpkiProtocolVersion',
                                               status.protocolVersion ) )
   submodel.sessionId = status.sessionId
   submodel.serialNumber = status.serialNumber
   submodel.lastUpdateSync = Ark.switchTimeToUtc(
      connectionData.lastUpdateSyncTimestamp )
   submodel.lastFullSync = Ark.switchTimeToUtc(
      connectionData.lastFullSyncTimestamp )
   submodel.entries = status.ip4RoaCount + status.ip6RoaCount
   model.caches[ cacheId.stringValue ] = submodel
   submodel.deleted = cacheId not in rpkiConfig.cacheConfig
   submodel.activeSince = Ark.switchTimeToUtc(
      connectionData.lastConnectionTimestamp )
   # setReasonInactive should be called after deleted is set
   submodel.setReasonInactive( connectionData.errorReason,
                               connectionData.lastConnectionTimestamp )
   submodel.transportProtocol = status.transportConfig.transportAuthenticationType
   submodel.lastSerialQuery = Ark.switchTimeToUtc(
            connectionData.lastSerialQueryTimestamp )
   submodel.lastResetQuery = Ark.switchTimeToUtc(
      connectionData.lastResetQueryTimestamp )
   if detail:
      detailModel = CacheDetailModel()
      if Toggles.RpkiToggleLib.toggleRpkiCacheTcpKeepaliveEnabled():
         tcpKeepaliveOptions = status.transportConfig.tcpKeepaliveOptions
         detailModel.tcpKeepaliveIdleTime = tcpKeepaliveOptions.idleTime
         detailModel.tcpKeepaliveProbeInterval = tcpKeepaliveOptions.probeInterval
         detailModel.tcpKeepaliveProbeCount = tcpKeepaliveOptions.probeCount
      detailModel.lastConfigChange = Ark.switchTimeToUtc(
         connectionData.lastConfigChangeTimestamp )
      detailModel.lastConnectionError = Ark.switchTimeToUtc(
         connectionData.lastConnectionErrorTimestamp )
      detailModel.lastProtocolError = Ark.switchTimeToUtc(
         connectionData.lastErrorTimestamp )
      detailModel.overrideConfiguredIntervalValues = \
         status.overrideConfiguredIntervalValues
      submodel.detail = detailModel
      if submodel.deleted:
         # No need to gather TCP information for deleted cache
         return
      tcpModel = None
      buff = cStringIO.StringIO()
      cmd = 'TCP_INFORMATION {}'.format( cacheId.stringValue )
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            'Rpki', 'RpkiCli', cmd,
                                            stringBuff=buff )
      try:
         data = ast.literal_eval( buff.getvalue() )
         if 'tcpStats' in data:
            tcpModel = TcpInformationModel()
            tcpStats = data[ 'tcpStats' ]
            tcpModel.setAttrFromDict( tcpStats )
            detailModel.tcpInformation = tcpModel
      except SyntaxError:
         t0( 'Error: Unable to retrieve TCP socket information correctly for',
             cacheId.stringValue )

def addNoCacheMessage( mode, cacheName=None ):
   msg = getCacheMessage( cacheName )
   mode.addMessage( msg )

def addUnusedCacheMessage( mode, unusedCaches ):
   if not unusedCaches:
      return
   msg = getCacheUnusedMessage( unusedCaches )
   mode.addMessage( msg )

class RpkiShowCache( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_NAME [ detail ]'''
   data = {
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiCacheExpr(),
      'detail': 'Detailed view'
   }
   cliModel = RpkiShowCacheModel

   @staticmethod
   @ArBgpShowOutput( 'RpkiShowCache', arBgpModeOnly=True )
   def handler( mode, args ):
      model = RpkiShowCacheModel()
      model.caches = {}
      detail = 'detail' in args

      cacheName = args.get( 'NAME' )
      if cacheName:
         cacheId = Tac.ValueConst( 'Rpki::CacheId', cacheName )
         status = rpkiStatus.cacheStatus.get( cacheId )
         if status:
            createSubmodel( cacheId, status, mode, model, detail )
         elif cacheId in rpkiConfig.cacheConfig:
            addUnusedCacheMessage( mode, [ cacheName ] )
         else:
            addNoCacheMessage( mode, cacheName )
         return model

      # Check if there are no caches
      if not rpkiStatus.cacheStatus and not rpkiConfig.cacheConfig:
         addNoCacheMessage( mode )

      # Display all caches with a status
      for cacheId, status in rpkiStatus.cacheStatus.iteritems():
         createSubmodel( cacheId, status, mode, model, detail )

      # display unused caches (cacheConfig but no cacheStatus)
      unusedCaches = [ cid.stringValue for cid in rpkiConfig.cacheConfig
                       if cid not in rpkiStatus.cacheStatus ]
      addUnusedCacheMessage( mode, unusedCaches )
      return model

def counterAdapter( mode, args, argsList ):
   caches = []
   cacheName = args.get( 'NAME' )

   cacheId = TacLazyType( 'Rpki::CacheId' )

   if cacheName and rpkiStatus.cacheStatus.get( cacheId( cacheName ) ):
      caches.append( cacheId( cacheName ) )
   elif not cacheName:
      caches = rpkiStatus.cacheStatus.keys()

   args[ 'CACHES' ] = caches

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>] counters" command.
#-----------------------------------------------------------------------------------
class RpkiShowCachePduCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_COUNTERS'''
   data = {
      'BGP_RPKI_CACHE_COUNTERS': generateBgpRpkiCacheExpr( counters=True ),
   }
   cliModel = RpkiShowCacheCounterModel

   adapter = counterAdapter

   @staticmethod
   @ArBgpShowOutput( 'RpkiShowCachePduCounters', arBgpModeOnly=True )
   def handler( mode, args ):
      caches = args.get( 'CACHES' )
      model = RpkiShowCacheCounterModel()
      model.caches = {}

      if not caches:
         msg = getCacheMessage( args.get( 'NAME' ) )
         mode.addMessage( msg )

      for cache in caches:
         counterModel = CacheCounterModel()
         current = rpkiCacheCounterSmash.counter[ cache ].pduCounter
         snapshot = rpkiCacheCounterSnapshotSmash.counterSnapshot[ cache ].pduCounter

         # show the difference between current and snapshot counters
         pduCounter = current - snapshot

         counterModel.serialNotify = pduCounter.serialNotify
         counterModel.serialQuery = pduCounter.serialQuery
         counterModel.resetQuery = pduCounter.resetQuery
         counterModel.ipv4Prefix = pduCounter.ipv4Prefix
         counterModel.ipv6Prefix = pduCounter.ipv6Prefix
         counterModel.endOfData = pduCounter.endOfData
         counterModel.cacheReset = pduCounter.cacheReset
         counterModel.cacheResponse = pduCounter.cacheResponse

         model.caches[ cache.stringValue ] = counterModel

      return model

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>] counters errors" command.
#-----------------------------------------------------------------------------------
class RpkiShowCachePduCountersErrors( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_COUNTERS errors'''
   data = {
      'BGP_RPKI_CACHE_COUNTERS': generateBgpRpkiCacheExpr( counters=True ),
      'errors': 'Cache server PDU error statistics',
   }
   cliModel = RpkiShowCacheErrorCounterModel
   adapter = counterAdapter

   @staticmethod
   @ArBgpShowOutput( 'RpkiShowCachePduCountersErrors', arBgpModeOnly=True )
   def handler( mode, args ):
      caches = args.get( 'CACHES' )
      model = RpkiShowCacheErrorCounterModel()
      model.cachesRx = {}
      model.cachesTx = {}

      if not caches:
         msg = getCacheMessage( args.get( 'NAME' ) )
         mode.addMessage( msg )

      for cache in caches:
         current = rpkiCacheCounterSmash.counter[ cache ].pduErrorCounter
         snapshot = rpkiCacheCounterSnapshotSmash.counterSnapshot[ cache ].\
                    pduErrorCounter

         # show the difference between current and snapshot counters
         pduCounter = current - snapshot
         # tx
         txErrorPdu = CacheTxErrorCounterModel()

         txErrorPdu.corruptData = pduCounter.txCorruptData
         txErrorPdu.internalError = pduCounter.txInternalError
         txErrorPdu.unsupportedProtocol = pduCounter.txUnsupportedProtocol
         txErrorPdu.unsupportedPdu = pduCounter.txUnsupportedPdu
         txErrorPdu.unexpectedProtocol = pduCounter.txUnexpectedProtocol
         txErrorPdu.other = pduCounter.txOther
         txErrorPdu.withdrawalUnknown = pduCounter.txWithdrawalUnknown
         txErrorPdu.duplicateAnnouncement = pduCounter.txDuplicateAnnouncement

         # rx
         rxErrorPdu = CacheRxErrorCounterModel()

         rxErrorPdu.corruptData = pduCounter.rxCorruptData
         rxErrorPdu.internalError = pduCounter.rxInternalError
         rxErrorPdu.unsupportedProtocol = pduCounter.rxUnsupportedProtocol
         rxErrorPdu.unsupportedPdu = pduCounter.rxUnsupportedPdu
         rxErrorPdu.unexpectedProtocol = pduCounter.rxUnexpectedProtocol
         rxErrorPdu.other = pduCounter.rxOther
         rxErrorPdu.noData = pduCounter.rxNoData
         rxErrorPdu.invalidRequest = pduCounter.rxInvalidRequest
         rxErrorPdu.pduTimeout = pduCounter.pduTimerExpire

         model.cachesTx[ cache.stringValue ] = txErrorPdu
         model.cachesRx[ cache.stringValue ] = rxErrorPdu

      return model

def clearAdapter( mode, args, argsList ):
   cacheName = args.get( 'NAME' )
   # clearing a specific cache
   if cacheName:
      cacheId = Tac.ValueConst( 'Rpki::CacheId', cacheName )

      # check if cache exists
      if cacheId not in rpkiStatus.cacheStatus:
         mode.addMessage( getCacheMessage( cacheName ) )
         return
   # clearing all caches
   if not rpkiStatus.cacheStatus:
      # no cache is configured
      mode.addMessage( getCacheMessage( None ) )
      return

#------------------------------------------------------------------------------------
# The "clear bgp rpki cache ( all | <name> ) command.
#------------------------------------------------------------------------------------
class RpkiClearCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'clear BGP_RPKI_CACHE_NAME'

   data = {
      'clear': Clear.clearKwNode,
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiClearCacheExpr(),
   }

   adapter = clearAdapter

   @staticmethod
   def handler( mode, args ):
      cacheName = args.get( 'NAME' )
      cmd = 'CLEAR_CACHE'
      if cacheName:
         cmd = cmd + ' {}'.format( cacheName )
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            'Rpki', 'RpkiCli', cmd )

#------------------------------------------------------------------------------------
# The "clear bgp rpki cache ( all | <name> ) counters" command.
#------------------------------------------------------------------------------------
class RpkiClearCacheCounterCmd( CliCommand.CliCommandClass ):
   syntax = 'clear BGP_RPKI_CACHE_NAME'

   data = {
      'clear': Clear.clearKwNode,
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiClearCacheExpr( counters=True ),
   }

   adapter = clearAdapter

   @staticmethod
   def handler( mode, args ):
      cmd = 'CLEAR_COUNTER'
      name = args.get( 'NAME' )
      if name:
         cmd += ' {}'.format( name )
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            'Rpki', 'RpkiCli', cmd )

# Clean-up hook for rpkiConfig if bgpConfig is unconfigured.
def deleteRpkiCacheConfigHook():
   rpkiConfig.cacheConfig.clear()

def setTristateRpkiOriginValidationMethod( bgpConfig, attrName, method ):
   setattr( bgpConfig, attrName,
            Tac.Value( "Routing::Bgp::TristateRpkiOriginValidationMethod",
                       value=method,
                       isSet=True ) )

def clearTristateRpkiOriginValidationMethod( bgpConfig, attrName ):
   default = getattr( bgpConfig, attrName + 'Default' )
   setattr( bgpConfig, attrName,
            Tac.Value( "Routing::Bgp::TristateRpkiOriginValidationMethod",
                       value=default,
                       isSet=False ) )

def setTristateRpkiOriginValidationSend( bgpConfig, attrName ):
   setattr( bgpConfig, attrName, 'isTrue' )

def clearTristateRpkiOriginValidationSend( bgpConfig, attrName ):
   setattr( bgpConfig, attrName, 'isInvalid' )

class RpkiOriginValidationMode( RpkiOriginValidationBaseMode,
                                BasicCli.ConfigModeBase ):
   name = 'RPKI Origin Validation configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RpkiOriginValidationBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vrfName = DEFAULT_VRF

   def setOriginValidationMethod( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      method = ovMethod( args )
      if args.get( 'ibgp' ):
         attrName = 'rpkiIbgpOvMethod'
      elif args.get( 'ebgp' ):
         attrName = 'rpkiEbgpOvMethod'
      elif args.get( 'redistributed' ):
         attrName = 'rpkiRedistOvMethod'
      setTristateRpkiOriginValidationMethod( bgpConfig, attrName, method )

   def noOriginValidationMethod( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      if args.get( 'ibgp' ):
         attrName = 'rpkiIbgpOvMethod'
      elif args.get( 'ebgp' ):
         attrName = 'rpkiEbgpOvMethod'
      elif args.get( 'redistributed' ):
         attrName = 'rpkiRedistOvMethod'
      clearTristateRpkiOriginValidationMethod( bgpConfig, attrName )

   def setOriginValidationSend( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      if args.get( 'ibgp' ):
         attrName = 'rpkiIbgpOvSend'
      elif args.get( 'ebgp' ):
         attrName = 'rpkiEbgpOvSend'
      setTristateRpkiOriginValidationSend( bgpConfig, attrName )

   def noOriginValidationSend( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      if args.get( 'ibgp' ):
         attrName = 'rpkiIbgpOvSend'
      elif args.get( 'ebgp' ):
         attrName = 'rpkiEbgpOvSend'
      clearTristateRpkiOriginValidationSend( bgpConfig, attrName )

   def setOriginValidationRoutemap( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      if args.get( 'redistributed' ):
         bgpConfig.rpkiRedistOvRouteMap = args[ 'ROUTEMAP' ]
      else:
         bgpConfig.rpkiOvRouteMap = args[ 'ROUTEMAP' ]

   def noOriginValidationRoutemap( self, args ):
      bgpConfig = configForVrf( self.vrfName )
      if args.get( 'redistributed' ):
         bgpConfig.rpkiRedistOvRouteMap = bgpConfig.rpkiRedistOvRouteMapDefault
      else:
         bgpConfig.rpkiOvRouteMap = bgpConfig.rpkiOvRouteMapDefault

originValidationKwMatcher = CliMatcher.KeywordMatcher(
   'origin-validation',
   helpdesc='Prefix origin validation' )

#------------------------------------------------------------------------------------
#  "[ no | default ] rpki origin-validation" command, in "router-bgp" mode.
#------------------------------------------------------------------------------------
class EnterBgpBaseRpkiOriginValidationMode( CliCommand.CliCommandClass ):
   syntax = 'rpki origin-validation'
   noOrDefaultSyntax = syntax

   data = { 'rpki': rpkiKwMatcher,
            'origin-validation': originValidationKwMatcher }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RpkiOriginValidationMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bgpConfig = configForVrf( mode.vrfName )
      # Set all vars back to the defaults.
      clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiEbgpOvMethod' )
      clearTristateRpkiOriginValidationSend( bgpConfig, 'rpkiEbgpOvSend' )
      clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiIbgpOvMethod' )
      clearTristateRpkiOriginValidationSend( bgpConfig, 'rpkiIbgpOvSend' )
      bgpConfig.rpkiOvRouteMap = bgpConfig.rpkiOvRouteMapDefault
      clearTristateRpkiOriginValidationMethod( bgpConfig, 'rpkiRedistOvMethod' )
      bgpConfig.rpkiRedistOvRouteMap = bgpConfig.rpkiRedistOvRouteMapDefault

RouterBgpBaseMode.addCommandClass( EnterBgpBaseRpkiOriginValidationMode )

ibgpKwMatcher = CliMatcher.KeywordMatcher(
   'ibgp',
   helpdesc='Configure options for iBGP neighbors' )
ebgpKwMatcher = CliMatcher.KeywordMatcher(
   'ebgp',
   helpdesc='Configure options for eBGP neighbors' )
localKwMatcher = CliMatcher.KeywordMatcher(
   'local',
   helpdesc='Use the ROA database' )
communityKwMatcher = CliMatcher.KeywordMatcher(
   'community',
   helpdesc='Use the received Origin Validation State community' )
preferCommunityKwMatcher = CliMatcher.KeywordMatcher(
   'prefer-community',
   helpdesc=( 'Use the received Origin Validation State community '
              'then the ROA database' ) )
disabledKwMatcher = CliMatcher.KeywordMatcher(
   'disabled',
   helpdesc='Disabled' )

#------------------------------------------------------------------------------------
#  "[ no | default ] ( ibgp | ebgp ) ( local | community | prefer-community )"
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRecvOriginValidation( CliCommand.CliCommandClass ):
   syntax = '( ibgp | ebgp ) ( local | community | prefer-community )'
   noOrDefaultSyntax = '( ibgp | ebgp ) ...'

   data = { 'ibgp': ibgpKwMatcher,
            'ebgp': ebgpKwMatcher,
            'local': localKwMatcher,
            'community': communityKwMatcher,
            'prefer-community': preferCommunityKwMatcher }

   handler = RpkiOriginValidationMode.setOriginValidationMethod
   noOrDefaultHandler = RpkiOriginValidationMode.noOriginValidationMethod

validationKwMatcher = CliMatcher.KeywordMatcher(
   'validation',
   helpdesc='Validation configuration' )
routeMapKwMatcher = CliMatcher.KeywordMatcher(
   'route-map',
   helpdesc='Route map' )

#------------------------------------------------------------------------------------
#  "[ no | default ] validation route-map ROUTEMAP" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiValidationRoutemap( CliCommand.CliCommandClass ):
   syntax = 'validation route-map ROUTEMAP'
   noOrDefaultSyntax = 'validation route-map ...'

   data = { 'validation': validationKwMatcher,
            'route-map': routeMapKwMatcher,
            'ROUTEMAP': mapNameMatcher }

   handler = RpkiOriginValidationMode.setOriginValidationRoutemap
   noOrDefaultHandler = RpkiOriginValidationMode.noOriginValidationRoutemap

redistributedKwMatcher = CliMatcher.KeywordMatcher(
   'redistributed',
   helpdesc='Configure options for redistributed routes' )

#------------------------------------------------------------------------------------
#  "[ no | default ] redistributed local" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRedistributedOriginValidation( CliCommand.CliCommandClass ):
   syntax = 'redistributed local'
   noOrDefaultSyntax = 'redistributed ...'

   data = { 'redistributed': redistributedKwMatcher,
            'local': localKwMatcher }

   handler = RpkiOriginValidationMode.setOriginValidationMethod
   noOrDefaultHandler = RpkiOriginValidationMode.noOriginValidationMethod

#------------------------------------------------------------------------------------
#  "[ no | default ] redistributed validation route-map ROUTEMAP" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRedistributedValidationRoutemap( CliCommand.CliCommandClass ):
   syntax = 'redistributed validation route-map ROUTEMAP'
   noOrDefaultSyntax = 'redistributed validation route-map ...'

   data = { 'redistributed': redistributedKwMatcher,
            'validation': validationKwMatcher,
            'route-map': routeMapKwMatcher,
            'ROUTEMAP': mapNameMatcher }

   handler = RpkiOriginValidationMode.setOriginValidationRoutemap
   noOrDefaultHandler = RpkiOriginValidationMode.noOriginValidationRoutemap

sendKwMatcher = CliMatcher.KeywordMatcher(
   'send',
   helpdesc='Attach Origin Validation State community to advertised routes' )

#------------------------------------------------------------------------------------
#  "[ no | default ] ( ibgp | ebgp ) send" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiCommunitySend( CliCommand.CliCommandClass ):
   syntax = '( ibgp | ebgp ) send'
   noOrDefaultSyntax = syntax

   data = { 'ibgp': ibgpKwMatcher,
            'ebgp': ebgpKwMatcher,
            'send': sendKwMatcher }

   handler = RpkiOriginValidationMode.setOriginValidationSend
   noOrDefaultHandler = RpkiOriginValidationMode.noOriginValidationSend

RpkiOriginValidationMode.addCommandClass( RpkiRecvOriginValidation )
RpkiOriginValidationMode.addCommandClass( RpkiValidationRoutemap )
RpkiOriginValidationMode.addCommandClass( RpkiRedistributedOriginValidation )
RpkiOriginValidationMode.addCommandClass( RpkiRedistributedValidationRoutemap )
RpkiOriginValidationMode.addCommandClass( RpkiCommunitySend )

# Per-neighbor commands

#------------------------------------------------------------------------------------
#  "[no | default ] neighbor <addr|peer-group> rpki origin-validation ( local |
#    community | prefer-community | disabled )" command in "router-bgp" mode.
#------------------------------------------------------------------------------------
class BgpNeighborRpkiRecvOriginValidation( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor PEER rpki origin-validation ( local | community | '
              'prefer-community | disabled )' )
   noOrDefaultSyntax = ( 'neighbor PEER rpki origin-validation ...' )

   data = { 'neighbor': RoutingBgp.neighbor,
            'PEER': PeerCliExpression,
            'rpki': rpkiKwMatcher,
            'origin-validation': originValidationKwMatcher,
            'local': localKwMatcher,
            'community': communityKwMatcher,
            'prefer-community': preferCommunityKwMatcher,
            'disabled': disabledKwMatcher }

   @staticmethod
   def handler( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], vrfName=mode.vrfName )
      config.rpkiOvMethod = ovMethod( args )
      config.rpkiOvMethodPresent = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                                  vrfName=mode.vrfName )
      if config:
         config.rpkiOvMethod = config.rpkiOvMethodDefault
         config.rpkiOvMethodPresent = False

#------------------------------------------------------------------------------------
#  "[no|default] neighbor <addr|peer-group> rpki origin-validation send [disabled]"
#   command in "router-bgp" mode.
#------------------------------------------------------------------------------------
class BgpNeighborRpkiCommunitySend( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor PEER rpki origin-validation send [ disabled ]' )
   noOrDefaultSyntax = syntax

   data = { 'neighbor': RoutingBgp.neighbor,
            'PEER': PeerCliExpression,
            'rpki': rpkiKwMatcher,
            'origin-validation': originValidationKwMatcher,
            'send': sendKwMatcher,
            'disabled': disabledKwMatcher }

   @staticmethod
   def handler( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], vrfName=mode.vrfName )
      config.rpkiOvSend = 'disabled' not in args
      config.rpkiOvSendPresent = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                                  vrfName=mode.vrfName )
      if config:
         config.rpkiOvSend = config.rpkiOvSendDefault
         config.rpkiOvSendPresent = False

#---------------------------------------------------------------------------
# Add rpki related commands to 'show tech-support'.
# These commands only apply to multi-agent mode, in ribd mode we will not run them.
#--------------------------------------------------------------------------
def _showTechRpkiCmds():
   protocolAgentModel = l3Config.protocolAgentModel
   if protocolAgentModel == ProtocolAgentModelType.multiAgent:
      return [
         'show bgp rpki cache detail',
         'show bgp rpki cache counters',
         'show bgp rpki cache counters errors',
         'show bgp rpki roa summary',
      ]
   else:
      return []

TechSupportCli.registerShowTechSupportCmdCallback( '2020-01-05 12:45:00',
                                    _showTechRpkiCmds,
                                    summaryCmdCallback=_showTechRpkiCmds )

def showBgpConfigRpki( bgpConfig ):
   '''Return a dictionary of submodes and config commands to include in the
   "show bgp configuration active" output.'''
   cmdDict = {}
   for cacheId in sorted( rpkiConfig.cacheConfig ):
      cacheConfig = rpkiConfig.cacheConfig[ cacheId ]
      cmds = saveCacheConfig( cacheId, cacheConfig )
      cmdDict[ 'rpki cache %s' % str( cacheId ) ] = cmds

   cmds = saveRpkiOriginValidationConfig( bgpConfig )
   if cmds:
      cmdDict[ 'rpki origin-validation' ] = cmds

   return cmdDict

showBgpConfigInstanceCallbacks.append( showBgpConfigRpki )

RouterBgpBaseMode.addCommandClass( EnterRpkiCacheConfigMode )
deleteRouterBgpMacVrfHook.addExtension( deleteRpkiCacheConfigHook )
RpkiCacheConfigMode.addCommandClass( RpkiHostVrfPortCmd )
RpkiCacheConfigMode.addCommandClass( RpkiRefreshIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiRetryIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiExpireIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiPreferenceCmd )
RpkiCacheConfigMode.addCommandClass( RpkiLocalInterfaceCmd )
if Toggles.RpkiToggleLib.toggleRpkiCacheTcpKeepaliveEnabled():
   RpkiCacheConfigMode.addCommandClass( EnterRpkiTransportTcpConfigMode )
   RpkiTransportTcpConfigMode.addCommandClass( TcpKeepaliveCommand )
if Toggles.RpkiToggleLib.toggleRpkiSecureTransportToggleEnabled():
   RpkiCacheConfigMode.addCommandClass( EnterRpkiTransportTlsConfigMode )
   RpkiTransportTlsConfigMode.addCommandClass( RpkiTlsSslProfileCmd )
BasicCli.addShowCommandClass( RpkiShowRoas )
BasicCli.addShowCommandClass( RpkiShowRoaSummary )
BasicCli.addShowCommandClass( RpkiShowCache )
BasicCli.addShowCommandClass( RpkiShowCachePduCounters )
BasicCli.addShowCommandClass( RpkiShowCachePduCountersErrors )
BasicCli.EnableMode.addCommandClass( RpkiClearCacheCmd )
BasicCli.EnableMode.addCommandClass( RpkiClearCacheCounterCmd )
RouterBgpSharedModelet.addCommandClass( BgpNeighborRpkiRecvOriginValidation )
RouterBgpSharedModelet.addCommandClass( BgpNeighborRpkiCommunitySend )

def Plugin( entityManager ):
   global rpkiConfig
   rpkiConfig = ConfigMount.mount( entityManager, 'routing/rpki/cache/config',
                                   'Rpki::CacheConfigDir', 'w' )
   global rpkiStatus
   rpkiStatus = LazyMount.mount( entityManager, 'routing/rpki/cache/status',
                                 'Rpki::CacheStatusDir', 'r' )
   global rpkiStatistics
   rpkiStatistics = LazyMount.mount( entityManager,
                                     'routing/rpki/cache/statistics',
                                     'Rpki::CacheStatisticsDir', 'r' )
   global rpkiRoaTable
   roaMountPath = Tac.Type( 'Rpki::RpkiRoaTable' ).smashPath
   rpkiRoaTable = \
      SmashLazyMount.mount( entityManager,
                            roaMountPath,
                            'Rpki::RpkiRoaTable',
                            SmashLazyMount.mountInfo( 'reader' ) )
   global allVrfConfig
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   global rpkiCacheCounterSmash
   cacheCounterPath = Tac.Type( 'Rpki::AllRpkiCacheCounterDir' ).smashPath
   rpkiCacheCounterSmash = \
      SmashLazyMount.mount( entityManager,
                            cacheCounterPath,
                            'Rpki::AllRpkiCacheCounterDir',
                            SmashLazyMount.mountInfo( 'reader' ) )
   global rpkiCacheCounterSnapshotSmash
   counterSnapshotPath = Tac.Type(
      'Rpki::AllRpkiCacheCounterSnapshotDir' ).smashPath
   rpkiCacheCounterSnapshotSmash = \
      SmashLazyMount.mount( entityManager,
                            counterSnapshotPath,
                            'Rpki::AllRpkiCacheCounterSnapshotDir',
                            SmashLazyMount.mountInfo( 'reader' ) )
   global l3Config
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )
   global asnConfig
   asnConfig = LazyMount.mount( entityManager, 'routing/bgp/asn/config',
                                'Routing::AsnConfig', 'r' )
   if Toggles.RpkiToggleLib.toggleRpkiSecureTransportToggleEnabled():
      global sslConfig
      sslConfig = LazyMount.mount( entityManager, 'mgmt/security/ssl/config',
                                   'Mgmt::Security::Ssl::Config', 'r' )
