# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliMatcher
import CliToken
import LazyMount
import ShowCommand
import SmashLazyMount
from IpLibConsts import DEFAULT_VRF
from CliPlugin.VrfCli import generateVrfCliModel
from OspfSrCliModels import (
      OspfShowSrModel,
      OspfPrefixSegmentFlagsModel,
      OspfPrefixSegmentModel,
      OspfShowSrPrefixSegmentSummaryModel,
      OspfGlobalBlockModel,
      OspfShowSrGlobalBlocksSummaryModel,
)
from RoutingOspfCli import (
      ospfConfig,
      matcherOspfShow,
      generateOspfInstListCliModel,
      OspfVrfModelDecorator,
)
from SegmentRoutingCliShowHelper import (
   _populateSRGBValues,
   _populateSrCommonHeader,
   _getReachabilityAlgorithm,
   _getSegmentRoutingGlobalBlocksPeersCount,
   _getSelfOriginatedSegmentStatistics,
)
import Toggles.OspfToggleLib
import Tracing

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

mplsConfig = None
srSysdbStatusDir = None
srSmashStatusDir = None
l3Config = None

def getSrSysdbStatus( instanceId=None, vrfName=None ):
   return srSysdbStatusDir.get( 'default' )

def getSrSmashStatus( instanceId=None, vrfName=None ):
   return srSmashStatusDir

def ospfSrActive( mode, instanceId, vrfName ):
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   if not l3Config.protocolAgentModel == "multi-agent":
      mode.addWarning( 'OSPF Segment Routing is not supported in' +
                       ' the current protocol model (%s)' %
                       l3Config.protocolAgentModel )
      return False

   if not instConfig or not instConfig.enable:
      mode.addWarning( 'OSPF (Instance Id: %s) is not running' % instanceId )
      return False

   if not instConfig.srConfig or instConfig.srConfig.shutdown:
      mode.addWarning( 'OSPF (Instance Id: %s) Segment Routing has been ' \
            'administratively shutdown' % instanceId )
      return False

   if not mplsConfig.mplsRouting:
      mode.addWarning( 'MPLS routing is not enabled' )
      return False

   # Sanity check that smash/sysdb status is available. They should be
   # available if the above conditions are met - except a transient 
   # state where gated hasn't processed srConfig yet or if ospf instance
   # isn't up for example no interfaces etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   if not srSysdbStatus or not srSmashStatus or \
      srSysdbStatus.routerId == Arnet.IpAddr( '0.0.0.0' ):
      mode.addWarning( 'OSPF (Instance Id: %s) Segment Routing is not running' % \
                       instanceId )
      return False

   return True

def populateSrCommonModel( model, instConfig, srSysdbStatus ):
   model.instanceId = instConfig.instance
   _populateSrCommonHeader( model, instConfig.srConfig.dataPlane, srSysdbStatus ) 

def getSegmentRoutingsSummaryModel( instanceId=None, vrfName=None ):
   result = OspfShowSrModel()
  
   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   _populateSRGBValues( result, srSysdbStatus )
   result.reachabilityAlgorithm = _getReachabilityAlgorithm()
   result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus )
   result.selfOriginatedSegmentStatistics = _getSelfOriginatedSegmentStatistics(
         srSmashStatus, srSysdbStatus, str( srSysdbStatus.routerId ) )
   result.mappingServer = \
         result.selfOriginatedSegmentStatistics.proxyNodeSidCount > 0

   return result

def _populatePrefixSegmentFlags( flagsObj, pfxSeg ):
   flagsObj.np = pfxSeg.flags.noPhp
   flagsObj.m = pfxSeg.flags.proxy
   flagsObj.e = pfxSeg.flags.explicitNull
   flagsObj.v = pfxSeg.sid.isValue
   flagsObj.l = pfxSeg.flags.isLocal
   flagsObj.n = pfxSeg.flags.node

def getSegmentRoutingPrefixSegmentSummaryModel( instanceId=None, vrfName=None,
                                                selfOrig=None ):
   # ISIS does not set/reset selfOriginatedPrefix flag if 'self-originated'
   # subcommand is executed, but we are setting the value regardless of the
   # command variant. Owing to this as well as the facts that prefix segment flags
   # are different for OSPF and ISIS, separate functions are used for populating
   # OSPF and ISIS prefix segment models
   result = OspfShowSrPrefixSegmentSummaryModel()

   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   result.nodeSidCount = 0
   result.proxyNodeSidCount = 0
   result.prefixSidCount = 0
   for pfx, entry in sorted( srSmashStatus.prefixSegment.items() ):
      if selfOrig and str( entry.rtrid ) != str( srSysdbStatus.igpRtrId ):
         continue
      item = OspfPrefixSegmentModel()
      item.selfOriginatedPrefix = str( entry.rtrid ) == str( srSysdbStatus.igpRtrId )
      item.prefix = str( pfx )
      item.segmentId = entry.sid.index
      item.routerId = entry.rtrid.stringValue

      item.flags = OspfPrefixSegmentFlagsModel()
      _populatePrefixSegmentFlags( item.flags, entry )

      if entry.flags.proxy:
         item.segmentType = "Proxy-Node"
         result.proxyNodeSidCount += 1
      elif entry.flags.node:
         item.segmentType = "Node"
         result.nodeSidCount += 1
      else:
         item.segmentType = "Prefix"
         result.prefixSidCount += 1
      result.prefixSegments.append( item )

   return result

def getSegmentRoutingGlobalBlocksSummaryModel( instanceId=None, vrfName=None ):
   result = OspfShowSrGlobalBlocksSummaryModel()

   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus )

   def _populateGlobalBlock( item, routerId, base, size ):
      item.routerId = routerId.stringValue
      item.base = base
      item.size = size

   ourGlobalBlock = OspfGlobalBlockModel()
   _populateGlobalBlock( ourGlobalBlock, srSysdbStatus.igpRtrId,
                         srSysdbStatus.labelRange.base,
                         srSysdbStatus.labelRange.size )
   result.globalBlocks.append( ourGlobalBlock )
   
   for routerId, entry in sorted( srSmashStatus.globalBlock.items() ):
      item = OspfGlobalBlockModel()
      _populateGlobalBlock( item, routerId, entry.base, entry.size )
      result.globalBlocks.append( item )

   return result

# 'segment-routing' keyword for show commands
segmentRoutingKw = CliMatcher.KeywordMatcher( 'segment-routing',
                        helpdesc='Segment Routing Information' )

OspfShowSrDictModel = generateOspfInstListCliModel( OspfShowSrModel )
OspfShowSrVrfDictModel = generateVrfCliModel( OspfShowSrDictModel,
                                              'OSPF SR information for all VRFs' )


@OspfVrfModelDecorator( vrfModelListType=OspfShowSrVrfDictModel, 
                        instModelListType=OspfShowSrDictModel )
def showIpOspfSr( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None
   return getSegmentRoutingsSummaryModel( instanceId )
 

OspfShowSrPfxSegDictModel = \
      generateOspfInstListCliModel( OspfShowSrPrefixSegmentSummaryModel )
OspfShowSrPfxSegVrfDictModel = \
      generateVrfCliModel( OspfShowSrPfxSegDictModel,
                           'OSPF SR prefix segments for all VRFs' )

@OspfVrfModelDecorator( vrfModelListType=OspfShowSrPfxSegVrfDictModel, 
                        instModelListType=OspfShowSrPfxSegDictModel )
def showIpOspfSrPfxSeg( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   selfOrig = 'self-originated' in args
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None

   return getSegmentRoutingPrefixSegmentSummaryModel( instanceId=instanceId,
                                                      selfOrig=selfOrig )

OspfShowSrGlobalBlocksDictModel = \
      generateOspfInstListCliModel( OspfShowSrGlobalBlocksSummaryModel )
OspfShowSrGlobalBlocksVrfDictModel = \
      generateVrfCliModel( OspfShowSrGlobalBlocksDictModel,
                           'OSPF SR global blocks for all VRFs' )

@OspfVrfModelDecorator( vrfModelListType=OspfShowSrGlobalBlocksVrfDictModel, 
                        instModelListType=OspfShowSrGlobalBlocksDictModel )
def showIpOspfSrGlobalBlocks( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None

   return getSegmentRoutingGlobalBlocksSummaryModel( instanceId=instanceId )

class ShowSegmentRouting( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing'
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'ospf': matcherOspfShow,
         'segment-routing' : segmentRoutingKw,
   } 
   handler = showIpOspfSr
   cliModel = OspfShowSrVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRouting )

class ShowSegmentRoutingPrefixSegment( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing prefix-segments [ self-originated ]'
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'ospf': matcherOspfShow,
         'segment-routing' : segmentRoutingKw,
         'prefix-segments' : 'Prefix Segment Information',
         'self-originated' : 'Self-Originated segments',
   }
   handler = showIpOspfSrPfxSeg
   cliModel = OspfShowSrPfxSegVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingPrefixSegment )

class ShowSegmentRoutingGlobalBlocks( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing global-blocks'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'segment-routing' : segmentRoutingKw,
      'global-blocks' : 'Peer SRGBs',
   }
   handler = showIpOspfSrGlobalBlocks
   cliModel = OspfShowSrGlobalBlocksVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingGlobalBlocks )

def Plugin( entityManager ):
   global mplsConfig
   global srSysdbStatusDir
   global srSmashStatusDir
   global l3Config
   mplsConfig = LazyMount.mount( entityManager, 'routing/mpls/config',
                                 'Mpls::Config', 'r' )
   srSysdbStatusDir = LazyMount.mount( entityManager, 'segmentrouting/ospf',
                                       'Tac::Dir', 'ri' )
   srSmashStatusDir = SmashLazyMount.mount( entityManager,
                                            'segmentrouting/ospf/default',
                                            'Smash::SegmentRouting::Status',
                                            SmashLazyMount.mountInfo( 'reader' ) )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
