# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliParser, BasicCli
import IpAddrMatcher
import CliCommand
import CliMatcher
import Tac
import IraIpCli, Arnet, sys, IntfCli
import LoopbackIntfCli
import re, socket
import Intf.IntfRange as IntfRange
from ReversibleSecretCli import decodeKey, reversibleSecretCliExpression
import os
import LazyMount
import ConfigMount
import RibCliLib
from CliMode.Ospf import RoutingOspfMode, RoutingOspfTeMode, RoutingOspfSrMplsMode, \
                         RoutingOspfGeneralMode
import TechSupportCli
import IraIpIntfCli
from IraIpIntfCli import canSetVrfHook
import IraVrfCli
from RouteMapCli import mapNameMatcher
from RouteMapCli import prefixListNameMatcher
from CliModel import Model
from CliModel import GeneratorDict
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import ALL_VRF_NAME, DEFAULT_VRF, generateVrfCliModel
import CliToken.Ip
from CliToken.Router import routerMatcherForConfig
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
from RibCapiLib import EmptyResponseException
from RibCapiLib import showRibCapiCommand
from RoutingCommon import GATED_PROTO_SH_TECH_TS
import OspfConsts
from OspfCliModels import NGB_INQUIRED_STATE_MAP, \
    OspfMplsLdpSyncModel, OspfMplsLdpSyncEntry, OSPF_DATABASE_LSA_TYPE_MAP, \
    OSPF_DATABASE_QUERY_TYPE_MAP
import OspfCliModels
import AclCli, AclCliLib, AclCliModel
from DeviceNameLib import kernelIntfFilter
from MplsCli import mplsNodeForShow
import Toggles.OspfToggleLib
import Toggles.RoutingLibToggleLib
import ShowCommand

# pylint: disable-msg=protected-access

metricMin = 1
metricMax = 65535
rangeCostMin = 1
rangeCostMax = 65535
# This is is set from routingHwStatus
maxOspfInstances = 0
srDataPlaneEnum = Tac.Type( "Routing::Ospf::SrDataPlane" )

ospfConfigDir = None
ospfStatusDir = None
ospfClearReqDir = None
ospfClearRespDir = None
sysNetConfigDir = None
routingHardwareStatus = None
allVrfConfig = None
linkReadyStatusVrfColl = None
ldpProtoConfigColl = None
ldpConfigColl = None
l3IntfConfigDir = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
l3Config = None
mplsConfig = None

costKw = CliMatcher.KeywordMatcher( 'cost', helpdesc='Configure the metric' )

def routingSupportedGuard( mode, token ):
   if routingHardwareStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

def ospfNonDefaultVrfGuard( mode, token ):
   instanceConfig = ospfConfig().instanceConfig[ mode.instanceId ]
   if instanceConfig.vrfName != DEFAULT_VRF:
      return CliParser.guardNonDefaultVrf
   return None

#------------------------------------------------------------------------------------
# Ira callback handler when the VRF for an interface changes or when a vrf is deleted
# When the VRF configured for a particular interface changes the Ospf instance area
# config is updated. When multiple OSPF instances are present, the instance
# area config will not be updated.
#------------------------------------------------------------------------------------
def intfVrfChangeHandler( intfId, oldVrf, newVrf, vrfDelete ):
   warnMsg = None
   config = ospfConfig()

   if intfId in config.intfConfig:
      intfConfig = config.intfConfig[ intfId ]
      instancesInNewVrf = nInstancesInVrf( config.instanceConfig, newVrf )
      instancesInOldVrf = nInstancesInVrf( config.instanceConfig, oldVrf )

      # If new vrf and old vrf only have 1 OSPFv2 instance
      if instancesInNewVrf == 1 and instancesInOldVrf == 1:
         instanceId = vrfNameToInstance( config.instanceConfig, 
                                         vrf=newVrf )[ 0 ]

         instanceConfig = config.instanceConfig.get( instanceId, newVrf )

         # If the instanceId not configured, create warning message    
         warnMsg = 'Changing OSPFv2 instance for interface %s, from ' \
                   '%s to %s' % ( intfId, oldVrf, newVrf )

         # We should initialize the area only if the instance has been
         # created.
         if instanceConfig:
            # Also create the area config
            if intfConfig.areaId not in instanceConfig.areaConfig.keys() \
               and  intfConfig.areaIdPresent is True:
               instanceConfig.areaConfig.newMember( intfConfig.areaId )

         assert _getIntfVrf( intfConfig.name ) == oldVrf
         oldInstanceId = vrfNameToInstance( config.instanceConfig, vrf=oldVrf )[ 0 ]

         instanceConfig = config.instanceConfig.get( oldInstanceId, oldVrf )
         # Now we need to delete the areaconfig from instanceConfig for old vrf
         # if it had per intf area config. We need to set intfConfig.areaIdPresent
         # = False explicitly , then set it back, otherwise deletion wont take place,
         # as the intfconfig stands intact with vrf change.
         if intfConfig.areaIdPresent is True and \
                intfConfig.areaId in instanceConfig.areaConfig.keys():
            intfConfig.areaIdPresent = False
            _deleteIntfAreaConfigIfAllAttributeHaveDefaults( oldInstanceId,
                                                      intfConfig.areaId, oldVrf )
            intfConfig.areaIdPresent = True
      else:
         warnMsg = 'Changing OSPFv2 instance for interface %s, from  ' \
                   '%s to %s not supported when source or destination ' \
                   'vrf contains multiple OSPFv2 instances' % \
                   ( intfId, oldVrf, newVrf )

   if warnMsg:
      return True, warnMsg
   return True, None

canSetVrfHook.addExtension( intfVrfChangeHandler )

#-------------------------------------------------------------------------------
# Get the vrf configured for an interface ( looks up the L3::Intf::ConfigDir )
#-------------------------------------------------------------------------------
def _getIntfVrf( intfName ):
   intfConfig = l3IntfConfigDir.intfConfig.get( intfName, None )
   if intfConfig:
      return intfConfig.vrf
   return DEFAULT_VRF

#-------------------------------------------------------------------------------
# This module implements the OSPF configuration commands.
# The router ospf config mode are implemented in this module.
# The corresponding commands are listed below. Commands marked
# with '+' are not part of release 1 or 2.0:
#
#  * router ospf config mode commands
#     + "[no|default] area <area-id> authentication [message-digest]" 
#     - "[no|default] area <area-id> default-cost <cost>" 
#     - "[no|default] area <area-id> filter <ipaddr> <mask>" 
#     - "[no|default] area <area-id> nssa"  (ABR options are not yet supported)
#     + "[no|default] area <area-id> nssa [default-information-originate
#                                  [metric] [metric-type]]" 
#     + "[no|default] area <area-id> not-so-stubby lsa type-7 convert type-5"
#     - "[no|default] area <area-id> range <ip-address> <mask> 
#        [advertise | not-advertise]"
#     - "[no|default] area <area-id> stub"
#     + "[no|default] area <area-id> virtual-link <router-id>
#                            [authentication [message-digest | null]]
#                            [hello-interval <seconds>]
#                            [retransmit-interval <seconds>] 
#                            [transmit-delay <seconds>]
#                            [dead-interval <seconds>]
#                            [[authentication-key key] |
#                             [message-digest-key key-id auth-type key]]"
#     - "[no|default] log-adjacency-changes [detail]"
#     + "[no|default] default-metric <metric-value>"
#     - "[no|default] distance ospf [ intra-area | inter-area | external ] 
#                              <dist>"
#     - "[no|default] network <ip-address> <wildcard-mask> area <area-id>"
#     - "[no|default] router-id <ip-address>"
#     - "[no|default] shutdown"
#     - "[no|default] passive-interface <interface>"
#     + "[no|default] capability lls"
#     - "[no|default] max-lsa ..."
#     - "[no|default] redistribute static"
#     + "[no|default] maximum-paths <>
#     + "[no|default] point-to-point routes"
# 
#     + "[no|default] area <area-id> nssa nssa-only
#     + "[no|default] bfd default"
#     + "[no|default] bfd adjacency state any" 
#     + "[no|default] traffic-engineering"
#     + "[no|default] ecmp intra-area route multi-area next-hops"
#     + "[no|default] segment-routing mpls"
#
#
#  * router ospf traffic-engineering config mode commands
#     + "[no|default] shutdown"
#
#
# The config interface mode commands are listed below. These commands are
# added to the RoutingProtocolIntfConfigModelet.
#
#  * IP "config-if" mode commands
#     + "[no|default] ip ospf <instance-id> area <area-id>"
#     - "[no|default] ip ospf authentication [message-digest]"
#     - "[no|default] ip ospf authentication-key <password>"
#     - "[no|default] ip ospf cost <interface-cost>"
#     - "[no|default] ip ospf dead-interval <seconds>"
#     - "[no|default] ip ospf hello-interval <seconds>"
#     - "[no|default] ip ospf message-digest-key <key-id> <auth-type> <key>"
#     - "[no|default] ip ospf mtu-ignore"
#     - "[no|default] ip ospf priority <number-value>"
#     - "[no|default] ip ospf retransmit-interval <seconds>"
#     - "[no|default] ip ospf disabled"
#     - "[no|default] ip ospf transmit-delay <seconds>"
#     - "[no|default] ip ospf network point-to-point|point-to-multipoint|
#                             nonbroadcast"
#
#  * Global config commands
#     - "[no|default] ip ospf router-id output-format hostnames"
#     - "[no|default] router ospf <instance-id>"
#     - "[no|default] router ospf general"
#
#  * Show commands
#     - "show ip ospf [<instance-id>]"
#     - "show ip ospf [<instance-id>] border-routers"
#     - "show ip ospf [[<instance-id> <area-id>]] database"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [adv-router [<ip-address>]]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [asbr-summary] [<link-state-id>]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [asbr-summary] [<link-state-id>]
#                                 [adv-router [<ip-address>]]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [asbr-summary] [<link-state-id>]
#                                 [self-originate] [<link-state-id>]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [database-summary] [vrf vrf-name|all]"
#     - "show ip ospf [<instance-id> <area-id>]]
#                        database [external] [<link-state-id>]"
#     - "show ip ospf [<instance-id> <area-id>]]
#                        database [external] [<link-state-id>]
#                                 [adv-router [<ip-address>]] [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [external] [<link-state-id>]
#                                 [self-originate] [<link-state-id>]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [network] [<link-state-id>] [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [network] [<link-state-id>]
#                                 [adv-router [<ip-address>]] [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [network] [<link-state-id>]
#                                 [self-originate] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [nssa-external] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [nssa-external] [<link-state-id>]
#                                 [adv-router [<ip-address>]]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [nssa-external] [<link-state-id>]
#                                 [self-originate] [<link-state-id>] 
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [router] [<link-state-id>]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [router] [adv-router [<ip-address>]]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [router] [self-originate] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [self-originate] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [summary] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [summary] [<link-state-id>]
#                                 [adv-router [<ip-address>]]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [[<instance-id> <area-id>]]
#                        database [summary] [<link-state-id>]"
#                                 [self-originate] [<link-state-id>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [<instance-id>]
#                        interface [<interface-type> <interface-number>]
#                                  [brief]"
#     - "show ip ospf [<instance-id>]
#                        neighbor [<interface-type> <interface-number>]
#                        [<neighbor-id>]
#                        [detail]"
#     - "show ip ospf [<instance-id>]
#                        request queue [<neighbor>] [<interface>]
#                              [<interface-neighbor>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [<instance-id>]
#                        retransmission queue [<neighbor>] [<interface>]
#                              [<interface-neighbor>]
#                                 [vrf vrf-name|all]"
#     - "show ip ospf [<instance-id>] summary-address"
#     - "show ip ospf [<instance-id>] virtual-links"
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# route ospf config mode. A new instance of this mode is created when the
# user enters "router ospf <instance-id>.
#-------------------------------------------------------------------------------
class RouterOspfMode( RoutingOspfMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPF configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new RouterOspfMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, instanceId, vrfName ):
      self.instanceId = instanceId
      self.vrfName = vrfName
      RoutingOspfMode.__init__( self, ( instanceId, vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.constants = Tac.Type( "Routing::Ospf::Constants" )
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig

   def _getInstanceConfig( self ):
      return ospfConfig().instanceConfig.get( self.instanceId, self.vrfName )

   #----------------------------------------------------------------------------
   # AreaConfig is created when one of the attributes is configured,
   # and is deleted when all the attributes are unconfigured, i.e when the
   # the value attributes have defaults, and the collection attributes
   # are empty. Since, there is not explicit cmd to delete an area,
   # the _deleteArea function is called from the functions that
   # handle the correponding "no" cmds.
   #----------------------------------------------------------------------------
   def _getOrCreateAreaConfig( self, areaId ):
      areaId = Arnet.IpAddress( areaId )
      areaId = str( areaId )
      instanceConfig = self._getInstanceConfig()
      if areaId in instanceConfig.areaConfig:
         return instanceConfig.areaConfig[ areaId ]
      else:
         return instanceConfig.areaConfig.newMember( areaId )

   def _getAreaConfig( self, areaId ):
      areaId = Arnet.IpAddress( areaId )
      areaId = str( areaId )
      instanceConfig = self._getInstanceConfig()
      if areaId in instanceConfig.areaConfig:
         return instanceConfig.areaConfig[ areaId ]
      else:
         self.addError( "Specified area is not configured" )
         return None

   def _areaNetworkCmds( self, areaId, areaConfig=None ):
      instanceConfig = self._getInstanceConfig()
      if areaConfig is None:
         areaConfig = instanceConfig.areaConfig.get( areaId, None )
      assert areaConfig
      count = 0
      toDel = []
      for p, a in instanceConfig.networkArea.iteritems():
         if a == areaConfig:
            count += 1
         elif ( a.area == areaId or
                not a.area in instanceConfig.areaConfig ):
            toDel.append( p )  # clean up any orphans
      for p in toDel:
         del instanceConfig.networkArea[ p ]
      return count

   def _deleteAreaConfigIfAllAttributeHaveDefaults( self, areaId ):
      instanceConfig = self._getInstanceConfig()
      areaId = Arnet.IpAddress( areaId )
      areaId = str( areaId )
      if areaId not in instanceConfig.areaConfig:
         return 
      areaConfig = instanceConfig.areaConfig[ areaId ]
      # If the specified area configured with interfaces, do
      # not delete the area config
      for intfConfig in ospfConfig().intfConfig.itervalues():
         if intfConfig.areaId == areaConfig.area and \
            intfConfig.areaIdPresent is True:
            return
      if checkAreaDefault( instanceConfig, areaId, areaConfig ):
         del instanceConfig.areaConfig[ areaId ]

   def noArea( self, args ):
      areaId = args[ 'AREA_ID' ]
      instanceConfig = self._getInstanceConfig()
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      for intfConfig in ospfConfig().intfConfig.itervalues():
         if intfConfig.areaId == areaConfig.area and \
             intfConfig.areaIdPresent is True:
            self.addError( "Specified area is configured with interfaces" )
            return
      if self._areaNetworkCmds( areaId, areaConfig ):
         self.addError( "Area cannot be deleted before its "
                        "network commands are removed" )
         return
      else:
         del instanceConfig.areaConfig[ areaConfig.area ]

   def setAuthentication( self, args ):
      areaId = args[ 'AREA_ID' ]
      messageDigest = args.get( 'MESSAGE_DIGEST' )
      areaId = Arnet.IpAddress( areaId )
      self.addError( "Not yet implemented. %s" %
                     sys._getframe().f_code.co_name )
      self.addError( str( areaId ) + " -- " + str( type( areaId ) ) )
      self.addError( messageDigest )

   def noAuthentication( self, args ):
      areaId = args[ 'AREA_ID' ]
      messageDigest = args.get( 'MESSAGE_DIGEST' )
      areaId = Arnet.IpAddress( areaId )
      self.addError( "Not yet implemented. %s" %
                     sys._getframe().f_code.co_name )
      self.addError( str( areaId ) + " -- " + str( type( areaId ) ) )
      self.addError( messageDigest )
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def setDefaultCost( self, args ):
      areaId = args[ 'AREA_ID' ]
      defaultCost = args[ 'COST' ]
      areaConfig = self._getOrCreateAreaConfig( areaId )
      areaConfig.defaultMetric = defaultCost

   def noDefaultCost( self, args ):
      areaId = args[ 'AREA_ID' ]
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      areaConfig.defaultMetric = areaConfig.defaultMetricDefault
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def parseAreaFilterArgs( self, args ):
      areaId = args[ 'AREA_ID' ]
      myPrefix = None
      pfxListOption = None
      pfxListName = None
      if 'PREFIX' in args:
         myPrefix = args.get( 'PREFIX' )
         myPrefix = Arnet.Prefix( myPrefix )
      elif 'prefix-list' in args:
         pfxListOption = True
         if 'PFXLIST' in args:
            pfxListName = args.get( 'PFXLIST' )
      return ( areaId, myPrefix, pfxListOption, pfxListName  )

   def setFilter( self, args ):
      areaId, myPrefix, pfxListOption, pfxListName = \
            self.parseAreaFilterArgs( args )
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if myPrefix:
         if myPrefix not in areaConfig.summaryFilter:
            areaConfig.summaryFilter.addMember( Tac.Value(
               "Routing::Ospf::Network", network=myPrefix ) )
      elif pfxListOption and pfxListName:
         if pfxListName != areaConfig.summaryPrefixList:
            areaConfig.summaryPrefixList = pfxListName

   def noFilter( self, args ):
      areaId, myPrefix, pfxListOption, _pfxListName = \
            self.parseAreaFilterArgs( args )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if myPrefix:
         if myPrefix in areaConfig.summaryFilter:
            del areaConfig.summaryFilter[ myPrefix ]
      elif pfxListOption:
         if areaConfig.summaryPrefixList:
            areaConfig.summaryPrefixList = ""
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def _nssaAllowed( self, areaConfig ):
      if areaConfig.areaType == 'areaTypeStub' or \
         areaConfig.areaType == 'areaTypeStubNoSummary':
         self.addError( "Area has already been configured as stub area" )
         return False
      if areaConfig.virtualLink:
         self.addError( "Area contains virtual links, and "
                        "so NSSA configuration is not allowed " )
         return False
      return True
   
   def setNssa( self, args ):
      areaId = args[ 'AREA_ID' ]
      defInfoOriginRule = None
      if 'default-information-originate' in args:
         defInfoOriginRule = []
         if 'metric' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoMetric', 
                                         args[ 'METRIC' ].pop() ) )
         if 'metric-type' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoMetricType', 
                                         args[ 'METRIC_TYPE' ].pop() ) )
         if 'nssa-only' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoNssaOnly', True ) )
      translate = args.get( 'type-5' ) or \
                  args.get( 'always' ) or args.get( 'nssa-only' )
      noSummary = args.get( 'no-summary' )
      if str( Arnet.IpAddress( areaId ) ) == '0.0.0.0':
         self.addError( 'Backbone cannot be configured as NSSA area' )
         return
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return
      if noSummary:
         areaConfig.areaType = 'areaTypeNssaNoSummary'
         areaConfig.originateDefaultSummary = True
         return
      if areaConfig.areaType != 'areaTypeNssaNoSummary':
         areaConfig.areaType = 'areaTypeNssa'
         areaConfig.originateDefaultSummary = False
      if defInfoOriginRule is not None:
         areaConfig.nssaDefInfoOrigin = 'ospfDefInfoOriginEnable'
         defInfoParms = dict( defInfoOriginRule )
         for i in [ 'nssaDefInfoMetric', 'nssaDefInfoMetricType',
                    'nssaDefInfoNssaOnly' ]:
            if i in defInfoParms:
               setattr( areaConfig, i, defInfoParms[ i ] )
            else:
               setattr( areaConfig, i, getattr( areaConfig, i + 'Default' ) )
      # New command does not contain 'always' keyword. We look for 'type-5' instead
      elif translate == 'always' or translate == 'type-5':
         areaConfig.nssaTranslateAlways = True
      elif translate == 'nssa-only':
         areaConfig.nssaOnly = True
      
   def noNssa( self, args ):
      areaId = args[ 'AREA_ID' ]
      defInfoOrigin = args.get( 'default-information-originate' )
      translate = args.get( 'type-5' ) or args.get( 'always' ) or \
                  args.get( 'nssa-only' )
      noSummary = args.get( 'no-summary' )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return

      if translate == 'nssa-only':
         areaConfig.nssaOnly = False
      # New command does not contain 'always' keyword. We look for 'type-5' instead
      elif translate == 'always' or translate == 'type-5':
         areaConfig.nssaTranslateAlways = False
      elif defInfoOrigin is not None:
         areaConfig.nssaDefInfoOrigin = 'ospfDefInfoOriginDefault'
         self._setNssaDefInfoDefaults( areaConfig )
      elif noSummary:
         areaConfig.areaType = 'areaTypeNssa'
         areaConfig.originateDefaultSummary = False
      else:
         areaConfig.areaType = 'areaTypeNormal'
         areaConfig.originateDefaultSummary = False
         areaConfig.nssaTranslateAlways = False
         areaConfig.nssaOnly = False
         areaConfig.nssaDefInfoOrigin = 'ospfDefInfoOriginDefault'
         self._setNssaDefInfoDefaults( areaConfig )
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def noNssaDefInfoOrigin( self, args ):
      areaConfig = self._getAreaConfig( args[ 'AREA_ID' ] )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return
      areaConfig.nssaDefInfoOrigin = 'ospfDefInfoOriginSuppress'
      self._setNssaDefInfoDefaults( areaConfig )

   def _setNssaDefInfoDefaults( self, areaConfig ):
      for i in [ 'nssaDefInfoMetric', 'nssaDefInfoMetricType',
                 'nssaDefInfoNssaOnly' ]:
         setattr( areaConfig, i, getattr( areaConfig, i + 'Default' ) )

   def _stubAllowed( self, areaConfig ):
      if areaConfig.areaType == 'areaTypeNssa' or \
         areaConfig.areaType == 'areaTypeNssaNoSummary':
         self.addError( "Area has already been configured as NSSA" )
         return False
      if areaConfig.virtualLink:
         self.addError( "Area contains virtual links, and "
                        "so stub configuration is not allowed " )
         return False
      return True

   def noStub( self, args ):
      areaId = args.get( 'AREA_ID' )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._stubAllowed( areaConfig ):
         return
      if 'no-summary' in args:
         areaConfig.areaType = 'areaTypeStub'
      else:
         areaConfig.areaType = 'areaTypeNormal'
         areaConfig.originateDefaultSummary = False
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def setStub( self, args ):
      areaId = args.get( 'AREA_ID' )
      if str( Arnet.IpAddress( areaId ) ) == '0.0.0.0':
         self.addError( "Backbone cannot be configured as stub area" )
         return
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._stubAllowed( areaConfig ):
         return
      if 'no-summary' in args:
         areaConfig.originateDefaultSummary = True
         areaConfig.areaType = 'areaTypeStubNoSummary'
      else:
         if areaConfig.areaType == 'areaTypeStubNoSummary':
            return
         areaConfig.originateDefaultSummary = True
         if areaConfig.areaType != 'areaTypeStubNoSummary':
            areaConfig.areaType = 'areaTypeStub'

   def setRange( self, args ):
      areaId = args[ 'AREA_ID' ]
      cost = args.get( 'COST', 0 )
      prefix = args[ 'PREFIX' ]
      prefix = Arnet.Prefix( prefix )
      restricted = 'not-advertise' in args
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      areaConfig.networkList.addMember( Tac.Value(
         "Routing::Ospf::Network", network=prefix, restricted=restricted,
         cost=cost) )

   def noRange( self, args ):
      areaId = args[ 'AREA_ID' ]
      advertise = args.get( 'advertise' ) or args.get( 'not-advertise' )
      cost = args.get( 'COST' )
      prefix = args[ 'PREFIX' ]
      prefix = Arnet.Prefix( prefix )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      network = areaConfig.networkList.get( prefix )
      if not network:
         return
      if ( advertise is None or advertise=='advertise' ) and cost is None:
         del areaConfig.networkList[ prefix ]
         self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )
         return

      restricted = False
      if advertise == 'not-advertise':
         restricted = True
      if cost is None:
         cost = 0
      if network.restricted == restricted and network.cost == cost:
         cost = 0
         if restricted:
            restricted = False
         areaConfig.networkList.addMember( Tac.Value(
            "Routing::Ospf::Network", network=prefix, restricted=restricted,
            cost=cost) )
 
   def _virtualLinkAllowed( self, areaConfig ):
      if areaConfig.areaType != 'areaTypeNormal':
         self.addError( "Area has been configured as NSSA or "
                        "stub area, so virtual links are not allowed" )
         return False
      return True

   def setVirtualLink( self, args ):
      areaId = args[ 'AREA_ID' ]
      routerId = args[ 'ROUTER_ID' ]
      helloInterval = args.get( 'HELLO_SECONDS' )
      transmitDelay = args.get( 'TXSECONDS' )
      retransmitInterval = args.get( 'RXSECONDS' )
      deadInterval = args.get( 'DEADSECONDS' )
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not self._virtualLinkAllowed( areaConfig ):
         return
      if routerId not in areaConfig.virtualLink:
         virtualLink = areaConfig.virtualLink.newMember( routerId )
      else:
         virtualLink = areaConfig.virtualLink[ routerId ]
         
      if helloInterval is not None:
         virtualLink.helloInterval = helloInterval
      if transmitDelay is not None:
         virtualLink.transDelay = transmitDelay
      if retransmitInterval is not None:
         virtualLink.rxInt = retransmitInterval
      if deadInterval is not None:
         virtualLink.routerDeadInterval = deadInterval

   def noVirtualLink( self, args ):
      areaId = args[ 'AREA_ID' ]
      routerId = args[ 'ROUTER_ID' ]
      helloInterval = args.get( 'HELLO_SECONDS' )
      transmitDelay = args.get( 'TXSECONDS' )
      retransmitInterval = args.get( 'RXSECONDS' )
      deadInterval = args.get( 'DEADSECONDS' )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._virtualLinkAllowed( areaConfig ):
         return
      if routerId not in areaConfig.virtualLink:
         return
      if not ( helloInterval is not None or
               transmitDelay is not None or
               retransmitInterval is not None or
               deadInterval is not None ):
         del areaConfig.virtualLink[ routerId ]
         self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )
         return
      
      virtualLink = areaConfig.virtualLink[ routerId ]
      if helloInterval is not None:
         virtualLink.helloInterval = virtualLink.helloIntervalDefault
      if transmitDelay is not None:
         virtualLink.transDelay = virtualLink.transDelayDefault
      if retransmitInterval is not None:
         virtualLink.rxInt = virtualLink.rxIntDefault
      if deadInterval is not None:
         virtualLink.routerDeadInterval = virtualLink.routerDeadIntervalDefault
      self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def setLogAdjacencyChanges( self, args ):
      detail = args.get( 'DETAIL' )
      instanceConfig = self._getInstanceConfig()
      if detail:
         instanceConfig.adjacencyLogging = 'adjacencyLoggingTypeDetail'
      else:
         instanceConfig.adjacencyLogging = 'adjacencyLoggingTypeNormal'

   def noLogAdjacencyChanges( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.adjacencyLogging = 'adjacencyLoggingTypeNone'

   def setDefaultMetric( self, args ):
      instanceConfig = self._getInstanceConfig()
      defaultMetric  = args.get( 'METRIC_VALUE',
           instanceConfig.defaultMetricDefault )
      instanceConfig.defaultMetric = defaultMetric

   def setDistance( self, args ):
      distance = args[ 'DIST' ]
      instanceConfig = self._getInstanceConfig()
      if "inter-area" in args:
         instanceConfig.interPref = distance
      elif "intra-area" in args:
         instanceConfig.internalPref = distance
      elif "external" in args:
         instanceConfig.asePref = distance

   def noDistance( self, args ):
      instanceConfig = self._getInstanceConfig()
      if "inter-area" in args:
         instanceConfig.interPref = \
             instanceConfig.interPrefDefault
      elif "intra-area" in args:
         instanceConfig.internalPref = \
             instanceConfig.internalPrefDefault
      elif "external" in args:
         instanceConfig.asePref = \
             instanceConfig.asePrefDefault

   def setNetwork( self, args ):
      prefix = args[ 'IP_ADDRESS' ]
      instanceConfig = self._getInstanceConfig()
      if prefix is None:
         self.addError( "Invalid network wildcard mask" )
         return
      subnet = Arnet.Subnet( prefix )
      for p, a in instanceConfig.networkArea.iteritems():
         # Skip triggering network overlap warning if configured subnet is exactly
         # the same as input subnet
         if subnet.overlapsWith( Arnet.Subnet( p ) ) and str( subnet ) != str( p ) :
            self.addWarning( "Network %s overlaps with existing "
                  "network %s of area %s" % ( str( prefix ), str( p ), a.area ) )
      # Delete default old area config
      prefixObj = Arnet.Prefix( prefix )
      oldAreaConfig = instanceConfig.networkArea.get( prefixObj )
      areaConfig = self._getOrCreateAreaConfig( args[ 'AREA_ID' ] )
      if not areaConfig:
         return
      instanceConfig.networkArea[ Arnet.Prefix( prefix ) ] = areaConfig
      if oldAreaConfig:
         self._deleteAreaConfigIfAllAttributeHaveDefaults( oldAreaConfig.area )

   def noNetwork( self, args ):
      areaId = args[ 'AREA_ID' ]
      prefix = args[ 'IP_ADDRESS' ]
      instanceConfig = self._getInstanceConfig()
      if prefix is None:
         self.addError( "Invalid network wildcard mask" )
         return
      prefix = Arnet.Prefix( prefix )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if instanceConfig.networkArea.get( prefix ) == areaConfig:
         del instanceConfig.networkArea[ prefix ]
         self._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def setRouterId( self, args ):
      routerId = args.get( 'ROUTER_ID' )
      if routerId == "0.0.0.0":
         self.addError( "%s is not a valid router ID" % routerId )
         return
      instanceConfig = self._getInstanceConfig()
      instanceConfig.routerId = routerId

   def noRouterId( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.routerId = instanceConfig.routerIdDefault

   def setMaxMetric( self, args ):
      instanceConfig = self._getInstanceConfig()
      maxMetricConfig = instanceConfig.maxMetricConfig

      # set all values to their current values as default
      maxMetricFlag = True
      extLsaMetric = maxMetricConfig.maxMetricExtMetric
      includeStubFlag = maxMetricConfig.maxMetricStubFlag
      onStartTime = maxMetricConfig.maxMetricOnStartDuration
      onStartWaitBgpFlag = maxMetricConfig.maxMetricWaitBgpFlag
      sumLsaMetric = maxMetricConfig.maxMetricSumMetric

      if 'external-lsa' in args:
         if 'EXTERNAL_LSA_METRIC_VALUE' in args:
            extLsaMetric = args.get( 'EXTERNAL_LSA_METRIC_VALUE' )[ 0 ]
         else:
            extLsaMetric = instanceConfig.maxMetricLsaMetricDefault
      if 'include-stub' in args:
         includeStubFlag = True
      if 'summary-lsa' in args:
         if 'SUMMARY_LSA_METRIC_VALUE' in args:
            sumLsaMetric = args.get( 'SUMMARY_LSA_METRIC_VALUE' )[ 0 ]
         else:
            sumLsaMetric = instanceConfig.maxMetricLsaMetricDefault
      if 'on-startup' in args:
         if 'wait-for-bgp' in args:
            onStartWaitBgpFlag = True
            onStartTime = 0
         else:
            onStartTime = args.get( 'ANNOUNCE_TIME' )[ 0 ]
            onStartWaitBgpFlag = False

      instanceConfig.maxMetricConfig = \
            Tac.Value( 'Routing::Ospf::MaxMetric', maxMetricFlag,
                       extLsaMetric, includeStubFlag,
                       onStartTime, onStartWaitBgpFlag, sumLsaMetric )

   def noMaxMetric( self, args ):
      instanceConfig = self._getInstanceConfig()
      if len( args ) == 3:
         instanceConfig.maxMetricConfig = \
            Tac.Value( 'Routing::Ospf::MaxMetric',
                       instanceConfig.maxMetricFlagDefault,
                       instanceConfig.maxMetricExtMetricDefault,
                       instanceConfig.maxMetricStubFlagDefault,
                       instanceConfig.maxMetricOnStartDurationDefault,
                       instanceConfig.maxMetricWaitBgpFlagDefault,
                       instanceConfig.maxMetricSumMetricDefault )
      else:
         maxMetricConfig = instanceConfig.maxMetricConfig

         # set all values to their current values as default
         maxMetricFlag = maxMetricConfig.maxMetricFlag
         extLsaMetricValue = maxMetricConfig.maxMetricExtMetric
         includeStubFlag = maxMetricConfig.maxMetricStubFlag
         onStartTime = maxMetricConfig.maxMetricOnStartDuration
         onStartWaitBgp = maxMetricConfig.maxMetricWaitBgpFlag
         sumLsaMetricValue = maxMetricConfig.maxMetricSumMetric
    
         argsDict = {}

         if 'external-lsa' in args:
            if 'EXTERNAL_LSA_METRIC_VALUE' in args:
               if args.get( 'EXTERNAL_LSA_METRIC_VALUE' )[ 0 ] != 0:
                  extLsaMetricValue = instanceConfig.maxMetricLsaMetricDefault
            else:
               extLsaMetricValue = instanceConfig.maxMetricExtMetricDefault
         if 'include-stub' in args:
            includeStubFlag = instanceConfig.maxMetricStubFlagDefault
         if 'summary-lsa' in args:
            if 'SUMMARY_LSA_METRIC_VALUE' in args:
               if args.get( 'SUMMARY_LSA_METRIC_VALUE' )[ 0 ] != 0:
                  sumLsaMetricValue = instanceConfig.maxMetricLsaMetricDefault
            else:
               sumLsaMetricValue = instanceConfig.maxMetricSumMetricDefault
     
         if 'on-startup' in args:
            if 'wait-for-bgp' in args:
               argsDict[ 'maxMetricOnStart' ] = args.get( 'wait-for-bgp' )[ 0 ]
            elif 'ANNOUNCE_TIME' in args:
               argsDict[ 'maxMetricOnStart' ] = args.get( 'ANNOUNCE_TIME' )[ 0 ]
            else:
               argsDict[ 'maxMetricOnStart' ] = None

         if 'maxMetricOnStart' in argsDict:
            if argsDict[ 'maxMetricOnStart' ] is None:
               onStartWaitBgp = instanceConfig.maxMetricWaitBgpFlagDefault
               onStartTime = instanceConfig.maxMetricOnStartDurationDefault
            elif argsDict[ 'maxMetricOnStart' ] == 'wait-for-bgp':
               onStartWaitBgp = instanceConfig.maxMetricWaitBgpFlagDefault
            else:
               onStartTime = instanceConfig.maxMetricOnStartDurationDefault

         instanceConfig.maxMetricConfig = \
            Tac.Value( 'Routing::Ospf::MaxMetric', maxMetricFlag,
                       extLsaMetricValue, includeStubFlag,
                       onStartTime, onStartWaitBgp, sumLsaMetricValue )

   def setLsaRetransmissionThreshold( self, args ):
      instanceConfig = self._getInstanceConfig()
      retransmissionThreshold = args.get( 'THRESHOLD', 
                                instanceConfig.lsaRetransmissionThresholdDefault )
      instanceConfig.lsaRetransmissionThreshold = retransmissionThreshold

   def setGracefulRestart( self, args=None ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.gracefulRestart = True

   def setGrHelper( self, args=None ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.grHelper = True

   def setShutdown( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.enable = False

   def noShutdown( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.enable = True

   def setLls( self ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.lls = True

   def noLls( self ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.lls = False

   def setSimultaneousBringUp( self, args ):
      simulPeers = args[ 'THRESHOLD' ]
      instanceConfig = self._getInstanceConfig()
      instanceConfig.simultaneousBringUp = simulPeers

   def noSimultaneousBringUp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.simultaneousBringUp = \
          instanceConfig.simultaneousBringUpDefault

   def setAutoCost( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.refBw = args.get( 'BANDWIDTH', 0 ) 

   def setRedistribute( self, proto, mapName=None,
                        isisLevel='levelNone',
                        matchRouteType=None,
                        includeLeaked=False,
                        allowOspfInstance=False ):
      instanceConfig = self._getInstanceConfig()
      if not matchRouteType:
         matchRouteType = ''
      if proto == 'protoBgpAggregate' and mapName:
         self.addWarning( 'The route-map argument is no longer supported for'
                          'aggregates and has not effect' )
      elif proto == 'protoOspf':
         if matchRouteType == 'external':
            proto = 'protoOspfAse'
         elif matchRouteType == 'nssa-external':
            proto = 'protoOspfNssa'

      if mapName:
         redistribute = Tac.Value( 'Routing::Ospf::Redistribute',
                                   proto, routeMap=mapName,
                                   isisLevel=isisLevel,
                                   matchRouteType=matchRouteType,
                                   includeLeaked=includeLeaked,
                                   allowOspfInstance=allowOspfInstance )
      else:
         redistribute = Tac.Value( 'Routing::Ospf::Redistribute', proto,
                                   isisLevel=isisLevel,
                                   matchRouteType=matchRouteType,
                                   includeLeaked=includeLeaked,
                                   allowOspfInstance=allowOspfInstance )
      instanceConfig.redistributeConfig.addMember( redistribute )

   def noRedistribute( self, proto, matchRouteType=None ):
      instanceConfig = self._getInstanceConfig()
      if proto == 'protoOspf':
         if matchRouteType == 'external':
            proto = 'protoOspfAse'
         elif matchRouteType == 'nssa-external':
            proto = 'protoOspfNssa'
      del instanceConfig.redistributeConfig[ proto ]

   def setDistListIn( self, args ):
      distName = args[ 'MAPNAME' ] if "MAPNAME" in args else args[ 'PREFIXNAME' ]
      instanceConfig = self._getInstanceConfig()      
      distType = "distRouteMap" if "route-map" in args else "distPrefixList"
      distListIn = Tac.Value( "Routing::Ospf::DistListIn", distType, distName )
      instanceConfig.distListIn = distListIn

   def noDistListIn( self, args ):
      instanceConfig = self._getInstanceConfig()
      distType = "distRouteMap" if "route-map" in args else "distPrefixList"
      distListIn = instanceConfig.distListIn
      if distListIn.distType == distType:
         distListIn = Tac.Value( "Routing::Ospf::DistListIn", "distUnconfigured",
                                 self.constants.distListInDefaultName )
         instanceConfig.distListIn = distListIn

   def setMplsLdpSync( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.mplsLdpSync = True

   def noSetMplsLdpSync( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.mplsLdpSync = instanceConfig.mplsLdpSyncDefault
      
   def setSpfInterval ( self, args ):
      spfInterval = args[ 'SECONDS' ] 
      instanceConfig = self._getInstanceConfig()
      instanceConfig.spfThrottleTimerConfig = \
            Tac.Value( "Routing::Ospf::SpfThrottleTimer",
                       0, spfInterval * 1000, spfInterval * 1000 )
      # If setting SPF timer in secs using the "timers spf" CLI then
      # we track the usage of the old CLI command for CliSavePlugin
      instanceConfig.timerSpfCommandActive = (
         instanceConfig.spfThrottleTimerConfig !=
            Tac.Value( "Routing::Ospf::SpfThrottleTimer",
                      instanceConfig.spfStartIntDefault,
                      instanceConfig.spfHoldIntDefault,
                      instanceConfig.spfMaxWaitIntDefault ) )

   def setSpfIntervalThrottle ( self, args ):
      spfStartInt = args[ 'SPF_START' ]
      spfHoldInt = args[ 'SPF_HOLD' ]
      spfMaxWaitInt = args[ 'SPF_MAX_WAIT' ]
      instanceConfig = self._getInstanceConfig()
      change = 0
      # spfHoldInt must be >= spfStartInt
      # spfMaxWaitInt must be >= spfHoldInt
      # adjust configured values if they are invalid
      if spfHoldInt < spfStartInt:
         spfHoldInt = spfStartInt
         change = 1
      if spfMaxWaitInt < spfHoldInt:
         spfMaxWaitInt = spfHoldInt
         change = 1
      if change:
         print "OSPF: Throttle timers corrected to: %d %d %d" % ( spfStartInt,
                                                                 spfHoldInt,
                                                                 spfMaxWaitInt )
      instanceConfig.spfThrottleTimerConfig = \
            Tac.Value( "Routing::Ospf::SpfThrottleTimer",
                       spfStartInt, spfHoldInt, spfMaxWaitInt)
      instanceConfig.timerSpfCommandActive = False

   def noSpfIntervalThrottle ( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.spfThrottleTimerConfig = \
            Tac.Value( "Routing::Ospf::SpfThrottleTimer",
                    instanceConfig.spfStartIntDefault,
                    instanceConfig.spfHoldIntDefault,
                    instanceConfig.spfMaxWaitIntDefault )
      instanceConfig.timerSpfCommandActive = False

   def setLsaArrivalInterval ( self, args ):
      instanceConfig = self._getInstanceConfig()
      lsaArrivalInt = args.get( 'MILLISECONDS',
                      instanceConfig.lsaArrivalIntDefault )
      instanceConfig.lsaArrivalInt = lsaArrivalInt

   def setLsaIntervalThrottle ( self, args ):
      lsaStartInt = args[ 'LSA_START' ]
      lsaHoldInt = args[ 'LSA_HOLD' ]
      lsaMaxWaitInt = args[ 'LSA_MAX_WAIT' ]
      instanceConfig = self._getInstanceConfig()
      change = False
      # lsaHoldInt must be >= lsaStartInt
      # lsaMaxWaitInt must be >= lsaHoldInt
      # adjust configured values if they are invalid
      if lsaHoldInt < lsaStartInt:
         lsaHoldInt = lsaStartInt
         change = True
      if lsaMaxWaitInt < lsaHoldInt:
         lsaMaxWaitInt = lsaHoldInt
         change = True
      if change:
         self.addMessage ( "OSPF: LSA throttle timers corrected to: %d %d %d" % \
                         ( lsaStartInt, lsaHoldInt, lsaMaxWaitInt ) )
      instanceConfig.lsaThrottleTimerConfig = \
            Tac.Value( "Routing::Ospf::LsaThrottleTimer",
                       lsaStartInt, lsaHoldInt, lsaMaxWaitInt )

   def noLsaIntervalThrottle ( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.lsaThrottleTimerConfig = \
            Tac.Value( "Routing::Ospf::LsaThrottleTimer",
                    instanceConfig.lsaStartIntDefault,
                    instanceConfig.lsaHoldIntDefault,
                    instanceConfig.lsaMaxWaitIntDefault )

   def setFloodPacing( self, args ):
      instanceConfig = self._getInstanceConfig()
      pacing = args.get( 'MILLISECONDS', instanceConfig.floodPacingDefault )
      instanceConfig.floodPacing = pacing

   def setOutDelayTimer( self, args ):
      instanceConfig = self._getInstanceConfig()
      lsaOutDelay = args.get( 'LSA_OUT_DELAY',
                    instanceConfig.lsaOutDelayTimerDefault )
      instanceConfig.lsaOutDelayTimerConfig = lsaOutDelay

   def setMaxEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.maxEcmp = args[ 'PATHS' ]
   
   def noMaxEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.maxEcmp = 0

   def setPointToPointRoutes ( self, args ):
      no = CliCommand.isNoCmd( args )
      instanceConfig = self._getInstanceConfig()
      instanceConfig.pointToPointRoutes = not no

   def setTunnelRoutes ( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.tunnelRoutes = True

   def noTunnelRoutes ( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.tunnelRoutes = False

   def setDnBitIgnore( self, args ):
      instanceConfig = self._getInstanceConfig()
      no = CliCommand.isNoOrDefaultCmd( args )
      if not no and self.vrfName == DEFAULT_VRF:
         self.addError( 'dn-bit-ignore cannot be configured for the default vrf' )
         return

      instanceConfig.ignoreDnBit = not no
      instanceConfig.backwardCompatibilityIgnoreDnBitType5Type7 = \
         not no and 'lsa' in args

   def setServiceAcl( self, args ):
      AclCliLib.setServiceAcl( self, OspfConsts.serviceName, 
                               OspfConsts.IPPROTO_OSPF,
                               self.aclConfig_, self.aclCpConfig_, 
                               args[ 'ACL_LIST_NAME' ], "ip", self.vrfName )

   def noServiceAcl( self, args ):
      AclCliLib.noServiceAcl( self, OspfConsts.serviceName,
                              self.aclConfig_, self.aclCpConfig_,
                              None, "ip", self.vrfName )

   def gotoRouterOspfTeMode( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.ospfTeMode = True
      childMode = self.childMode( RouterOspfTeMode )
      self.session_.gotoChildMode( childMode )

   def delRouterOspfTeMode( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.ospfTeMode = instanceConfig.ospfTeModeDefault
      childMode = self.childMode( RouterOspfTeMode )
      childMode.clearConfig()

   def setIntraAreaRouteEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      no = CliCommand.isNoOrDefaultCmd( args )
      instanceConfig.intraAreaRouteEcmp = not no

#-------------------------------------------------------------------------------
# OSPF-TE config mode. A new instance of this mode is created when the
# user enters "traffic-engineering".
#-------------------------------------------------------------------------------
class RouterOspfTeMode( RoutingOspfTeMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'OSPF traffic-engineering configuration'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session ):
      # parent is an object of RouterOspfMode which inherits from
      # RoutingOspfMode and that has vrfName which we are using below.
      RoutingOspfTeMode.__init__( self, ( parent.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
   
   # clearConfig doesn't pass any variable. Hence setting args to None.
   def setShutdown( self, args=None ):
      instanceConfig = self.parent_._getInstanceConfig()
      enabled = instanceConfig.ospfTeEnabledDefault
      if args:
         if CliCommand.isNoCmd( args ):
            enabled = True
      instanceConfig.ospfTeEnabled = enabled

   def setArea( self, args ):
      no = None 
      if CliCommand.isNoCmd( args ):
         no = True
      elif CliCommand.isDefaultCmd( args ):
         no = 'default'
      
      # Area all is configured by default
      if "all" in args:
         instanceConfig = self.parent_._getInstanceConfig()
         # Value for no    Should all Areas be enabled
         # None            True
         # default         True
         # True            False
         if no is True:
            instanceConfig.ospfTeConfiguredAllAreas = False
         else:
            instanceConfig.ospfTeConfiguredAllAreas = True

            # If enabling 'all', disabled all the per area configs
            args = { 'AREA_ID': self.parent_._getInstanceConfig().areaConfig.keys(),
                     '__no__': True, 
                   }
            self.setArea( args )
      else:
         if no or no == "default":
            # When disabling, only disable in the areas specified
            for areaId in args[ 'AREA_ID' ]:
               areaConfig = self.parent_._getOrCreateAreaConfig( areaId )
               areaConfig.teConfigured = False
               self.parent_._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )
         else:
            # When enabling, make sure 'all' and all other areas are disabled
            self.setArea( { 'all': 'all', '__no__': True } )
            areasToEnable = set( args[ 'AREA_ID' ] )
            areasToIterateOver = (
               set( self.parent_._getInstanceConfig().areaConfig.keys() ) |
               areasToEnable )
            for areaId in areasToIterateOver:
               areaConfig = self.parent_._getOrCreateAreaConfig( areaId )
               areaConfig.teConfigured = areaId in areasToEnable
               self.parent_._deleteAreaConfigIfAllAttributeHaveDefaults( areaId )

   def clearConfig( self ):
      self.setShutdown()
      args = { 'all': 'all' }
      self.setArea( args )

def checkAreaNetworkCmds( instanceConfig, areaId, areaConfig ):
   areaConfig = instanceConfig.areaConfig.get( areaId, None )
   assert areaConfig
   return bool( [ x for x in instanceConfig.networkArea.values() 
                  if x.area == areaId ] )

def checkAreaDefault( instanceConfig, areaId, areaConfig ):
   if ( ( areaConfig.areaType == areaConfig.areaTypeDefault ) and
        ( areaConfig.nssaTranslateAlways == \
             areaConfig.nssaTranslateAlwaysDefault ) and
        ( areaConfig.nssaOnly ==
          areaConfig.nssaOnlyDefault ) and
        ( areaConfig.advertiseSubnet == areaConfig.advertiseSubnetDefault ) and
        ( areaConfig.defaultMetric == areaConfig.defaultMetricDefault ) and
        ( not areaConfig.stub ) and
        ( not areaConfig.summaryFilter ) and
        ( not areaConfig.virtualLink ) and
        ( not areaConfig.networkList ) and
        ( not areaConfig.nssaNetwork ) and
        ( areaConfig.teConfigured == areaConfig.teConfiguredDefault ) and
        ( checkAreaNetworkCmds( instanceConfig, areaId,
                                areaConfig=areaConfig ) is False ) ):
      return True
   return False

def _deleteIntfAreaConfigIfAllAttributeHaveDefaults( instanceId, areaId, vrfName ):
   # Scan through all the interfaces to check if the current area ID is still
   # in use, else delete the area config from the instance config
   for intfName in ospfConfig().intfConfig:
      intfConfig = ospfConfig().intfConfig[ intfName ]
      if intfConfig.areaIdPresent and intfConfig.areaId == areaId and \
             _getIntfVrf( intfConfig.name ) == vrfName:
         return 

   instanceConfig = ospfConfig().instanceConfig.get( instanceId )

   if instanceConfig:
      if areaId not in instanceConfig.areaConfig:
         return
      areaConfig = instanceConfig.areaConfig[ areaId ]
      if checkAreaDefault( instanceConfig, areaId, areaConfig ):
         del instanceConfig.areaConfig[ areaId ]

def _ospfSrConflictingPrefixFound( instanceConfig, intfName, prefix, proxy=False ):
   # TODO - BUG362053 
   # To be updated when node-segment command is supported
   # Once a prefix is configured as a particular type - prefix/node/proxy, we should
   # not allow it to be configured as another type
   if proxy:
      segType = 'proxy'
   else:
      segType = 'prefix'

   # If we find a prefix segment with the same key as 'prefix' it's a conflict
   value = instanceConfig.srConfig.prefixSegments.get( prefix )
   if value:
      nodeType = 'proxy' if value.isProxyNode else 'prefix'
      if nodeType != segType:
         return True

   return False

def _ospfSrConflictingSidFound( instanceConfig, prefix, sid ):
   # If we find a prefix segment with key other than 'prefix'
   # using this SID, it's a conflict
   for key, value in instanceConfig.srConfig.prefixSegments.items():
      if key != prefix and value.index == sid:
         return True
   return False

#-------------------------------------------------------------------------------
# OSPF router general config mode. A new instance of this mode is created when
# the user enters "router ospf general".
#-------------------------------------------------------------------------------
class RouterOspfGeneralMode( RoutingOspfGeneralMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPF configuration that applies to all OSPF instances in all VRFs'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RoutingOspfGeneralMode.__init__( self, 'ospfGeneralConfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# OSPF-SR config mode. A new instance of this mode is created when the
# user enters "segment-routing mpls".
#-------------------------------------------------------------------------------
class RouterOspfSrMplsMode( RoutingOspfSrMplsMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPF segment-routing mpls configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      # parent is an object of RouterOspfMode which inherits from RoutingOspfMode
      RoutingOspfSrMplsMode.__init__( self, parent.instance )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "[no|default] router ospf <instance-id> [vrf name]" command, in "config" mode.
#-------------------------------------------------------------------------------

def ospfConfig():
   return ospfConfigDir

def nInstancesInVrf( instanceConfig, vrf=DEFAULT_VRF ):
   return len( [ i for i in instanceConfig.values() if i.vrfName == vrf ] )

def vrfNameToInstance( instanceConfig, vrf=DEFAULT_VRF ):
   _x = [ i for i in instanceConfig.values() if i.vrfName == vrf ]
   return ( [ i.instance for i in _x ] if _x else [] )

def gotoRouterOspfMode( mode, args ):
   instanceId = args[ 'INSTANCE_ID' ]
   vrfName = args.get( 'VRF' )
   config = ospfConfig()
   instanceConfig = config.instanceConfig.get( instanceId, None )
   if not instanceConfig:
      IraIpCli.warnIfRoutingDisabled( mode,
            vrfName=vrfName if vrfName else DEFAULT_VRF )
   if not vrfName:
      vrfName = DEFAULT_VRF
   elif instanceConfig and instanceConfig.vrfName != vrfName:
      mode.addError( 'OSPF instance %d already exists in VRF %s'
                     % ( instanceConfig.instance, instanceConfig.vrfName ) )
      return
   if instanceConfig is None:
      # Get number of OSPFv2 instances currently in vrf
      instancesInVrf = nInstancesInVrf( config.instanceConfig, vrfName )

      # If moving from single to multiple instance OSPFv2
      # reject new instance if there is interface area config present.
      if instancesInVrf == 1:
         for intfConfig in ospfConfig().intfConfig.itervalues():
            if _getIntfVrf( intfConfig.name ) == vrfName and \
               intfConfig.areaId and intfConfig.areaIdPresent:
               mode.addWarning(
                  "OSPFv2 interface area configuration found in vrf% s. "
                  "Multiple OSPFv2 instance config will not be applied." % vrfName )
               return



      IraVrfCli.addAgentVrfEntry( vrfName, "Ospf" )
      instanceConfig = config.instanceConfig.newMember( instanceId, vrfName )

   # Re-initialize interfaces that are part of the Ospf instance if
   # single instance configuration. Also if any interface has areaId set,
   # create a areaConfig entity for the areaId if one does not exist already
   # The intfConfig.areaId flag is to prevent area config being generated
   # if intfconfig was due to floating config and not a intf area config
   for intfConfig in ospfConfig().intfConfig.itervalues():
      if _getIntfVrf( intfConfig.name ) == vrfName and \
         intfConfig.areaId and intfConfig.areaIdPresent:
         if intfConfig.areaId not in instanceConfig.areaConfig.keys() and \
            nInstancesInVrf( config.instanceConfig, vrfName ) == 1:
            instanceConfig.areaConfig.newMember( intfConfig.areaId )

   assert instanceConfig
   childMode = mode.childMode( RouterOspfMode, instanceId=instanceId,
                               vrfName=instanceConfig.vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterOspfMode( mode, args ):
   instanceId = args[ 'INSTANCE_ID' ]
   vrfName = args.get( 'VRF' )
   config = ospfConfig()
   if instanceId in config.instanceConfig:
      intfList = config.intfConfig
      if isinstance( intfList, IntfRange.IntfList ):
         intfNames = intfList.intfNames()
      else:
         intfNames = intfList
      for name in intfNames:
         if name in config.intfConfig:
            intfConfig = config.intfConfig[name]
            del intfConfig.passiveInstances[ instanceId ]
            _deleteIntfConfigIfAllAttributeHaveDefaults( config, name )
      config.instanceConfig[ instanceId ].summaryAddrConfig.clear()
      if not vrfName:
         vrfName = config.instanceConfig[ instanceId ].vrfName
      del config.instanceConfig[ instanceId ]
      if not nInstancesInVrf( config.instanceConfig, vrfName ):
         IraVrfCli.removeAgentVrfEntry( vrfName, "Ospf" )

#
# This hook is invoked by Ira before VRF is deleted. We can cleanup
# corresponding OSPF instance if any. We are expected to return a tuple:
# ( True/False, messages). If we return True, VRF is deleted.
#
def deleteVrfHook( vrfName ):
   config = ospfConfig()
   for inst in config.instanceConfig.values():
      if inst.vrfName != vrfName:
         continue
      instanceId = inst.instance
      return ( True, "OSPF configuration for " \
         "instance %s has been disabled" % instanceId )
   return ( True, None )

class OspfVrfModelDecorator( object ):
   '''
   Decorator class that can be used to create a top level
   Capi model that contains a generator list of component model instances

   Example usage:
   to 'capi'ize the 'show ip ospf' command with 'vrf [VRF|default|all]'
   add the 'OspfVrfModelDecorator' decorator to the showIpOspf function and pass 
   the @vrfModelListType argument to the decorator. @vrfModelListType is a model
   type generated by invoking generateVrfCliModel with relevent component
   model type. @instModelListType is the model type generated by invoking
   generateOspfInstListCliModel. This model contains the list of
   instances per VRF. 

   @OspfVrfModelDecorator( vrfModelListType, instModelListType )
   def showIpOspf( mode, args ):
   '''
   def __init__( self, vrfModelListType, instModelListType ):
      '''input: @vrfModelListType is a model type generated by invoking 
      generateOspfVrfCliModel with relevent compoenent
      model type.
      @instModelListType is the model type generated by invoking
      generateOspfInstListCliModel. This model contains the list of
      instances per VRF.'''
      self.vrfModelListType = vrfModelListType
      self.instModelListType = instModelListType

   def __call__( self, func ):
      '''return a decorated function that wraps @func
         arguments to the wrapped function are passed as is'''
      def vrfExecCmdFunc( *args, **kwargs ):
         '''
         first argument in the args list must be of type 'CliParser.Mode'
         '''
         assert isinstance( args[ 0 ], CliParser.Mode )
         mode = args[ 0 ]

         instanceId = kwargs[ 'args' ].get( 'INSTANCE_ID' )
         vrfName = kwargs[ 'args' ].get( 'VRF' )

         instanceIds = getInstanceIds( mode, instanceId, vrfName )
         if not instanceIds:
            instanceIds = []

         vrfs = {}
         status = ospfStatus()
         config = ospfConfig()
         for inst in instanceIds:
            c = config.instanceConfig.get( inst )
            if c:
               statusVrf = status.get( c.vrfName )
               if not statusVrf:
                  continue
               instanceStatus = statusVrf.instanceStatus.get( inst )
               if not instanceStatus:
                  continue
               vrfs.setdefault( c.vrfName, [] ).append( inst )
            else:
               vrfs.setdefault( DEFAULT_VRF, [] ).append ( inst )

         def instListFunc( vrf, firstVrf ):
            firstInst = True
            for inst in vrfs[ vrf ]:
               kwargs[ 'args' ][ 'INSTANCE_ID' ] = inst
               kwargs[ 'args' ][ 'VRF' ] = vrf
               instModel = func( *args, **kwargs )
               if instModel is None:
                  continue
               if '_firstVrf' in instModel:
                  instModel._firstVrf = firstVrf
               if '_firstInst' in instModel:
                  instModel._firstInst = firstInst
               yield inst, instModel
               firstInst = False

         def vrfListFunc():
            firstVrf = True
            for vrf in vrfs:
               instListModel = self.instModelListType()
               instListModel.instList = instListFunc( vrf, firstVrf )
               yield vrf, instListModel
               firstVrf = False

         model = self.vrfModelListType()
         model.vrfs = vrfListFunc()
         return model

      return vrfExecCmdFunc   

def generateOspfInstListCliModel( cliModel ):
   instList = GeneratorDict( keyType=int, valueType=cliModel,
         help='OSPF Instance list keyed by instance id' )
   def render( self ):
      for _, model in self.instList:
         model.render()

   className = 'InstList%s' % cliModel.__name__
   classBody = { 'instList' : instList,
                 'render' : render }

   return type( className, ( Model, ), classBody )

ospfKw = CliCommand.guardedKeyword( 'ospf',
                                    helpdesc='OSPF protocol',
                                    guard=routingSupportedGuard )

instanceMin = 1
instanceMax = 65535
matcherInstance = CliMatcher.IntegerMatcher( instanceMin, instanceMax,
                                             helpdesc='Instance ID' )

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True,
      inclAllVrf=True )

class RouterOspfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'router ospf INSTANCE_ID [ VRF ]'
   noOrDefaultSyntax = syntax
   data = {
      'router': routerMatcherForConfig,
      'ospf': ospfKw,
      'INSTANCE_ID': matcherInstance,
      'VRF': vrfExprFactory,
   }
   handler = gotoRouterOspfMode
   noOrDefaultHandler = deleteRouterOspfMode

BasicCli.GlobalConfigMode.addCommandClass( RouterOspfConfigCmd )

areaIdMin = 0
areaIdMax = 4294967295
areaKw = CliMatcher.KeywordMatcher( 'area', helpdesc='Configure OSPF area' )

class AreaIdExpression( CliCommand.CliExpression ):
   expression = 'AREA_INT | AREA_IP'
   data = {
         'AREA_INT': CliMatcher.IntegerMatcher(
                        areaIdMin, areaIdMax,
                        helpdesc='OSPF area-id in decimal format' ),
         'AREA_IP': IpAddrMatcher.IpAddrMatcher(
                        helpdesc='OSPF area-id in IP address format' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'AREA_INT' in args and 'AREA_IP' in args:
         args[ 'AREA_ID' ] = args.pop( 'AREA_INT' ) + args.pop( 'AREA_IP' )
      elif 'AREA_INT' in args:
         args[ 'AREA_ID' ] = args.pop( 'AREA_INT' )
      elif 'AREA_IP' in args:
         args[ 'AREA_ID' ] = args.pop( 'AREA_IP' )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf router-id output-format hostnames" command,
# in "config" mode
# Legacy:
# "[no|default] ip ospf name-lookup" command, in "config" mode.
#-------------------------------------------------------------------------------

def setOspfNameLookup( mode, args ):
   cfg = ospfConfig()
   value = True
   if CliCommand.isDefaultCmd( args ):
      value = cfg.nameLookupDefault
   elif CliCommand.isNoCmd( args ):
      value = False
   cfg.nameLookup = value

nameLookupKw = CliMatcher.KeywordMatcher( 'name-lookup',
               helpdesc='Show DNS-resolved router names' )

class OspfNameLookupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf name-lookup'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'ospf': ospfKw,
            'name-lookup': CliCommand.Node( nameLookupKw,
                     deprecatedByCmd='ip ospf router-id output-format hostnames' ),
   }
   handler = setOspfNameLookup
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( OspfNameLookupCmd )

class OspfRouterIdOutputFmtHostnamesCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf router-id output-format hostnames'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'ospf': ospfKw,
            'router-id' : 'Display format for OSPF router IDs',
            'output-format' : 'Display format for OSPF router IDs',
            'hostnames' : 'Show DNS-resolved router names',
   }
   handler = setOspfNameLookup
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( OspfRouterIdOutputFmtHostnamesCmd )

#-------------------------------------------------------------------------------
# "[no|default] router ospf general" command, in "config" mode
#-------------------------------------------------------------------------------
ospfGeneralKw = CliMatcher.KeywordMatcher( 'general',
                                           helpdesc='General OSPF configuration' )

class RouterOspfGeneralCmd( CliCommand.CliCommandClass ):
   syntax = 'router ospf general'
   noOrDefaultSyntax = syntax
   data = {
      'router'  : routerMatcherForConfig,
      'ospf'    : ospfKw,
      'general' : ospfGeneralKw,
   }

   @staticmethod
   def handler( mode, args ):
      config = ospfConfig()
      config.generalConfig = ( 'ospfGeneralConfig', )
      childMode = mode.childMode( RouterOspfGeneralMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = ospfConfig()
      config.generalConfig = None

BasicCli.GlobalConfigMode.addCommandClass( RouterOspfGeneralCmd )

#-------------------------------------------------------------------------------
# "[no|default] qos dscp <dscp-value>" command, in "router ospf general" mode
#-------------------------------------------------------------------------------
ospfQosKw = CliMatcher.KeywordMatcher( 'qos',
                                       helpdesc='Set QoS parameters for CPU'
                                                ' generated OSPF traffic' )
dscpKw = CliMatcher.KeywordMatcher( 'dscp',
                                    helpdesc='Set DSCP value for OSPF traffic' )
dscpValue = CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP value' )

class OspfQosDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos dscp <dscpValue>'
   noOrDefaultSyntax = 'qos dscp ...'
   data = {
      'qos'             : ospfQosKw,
      'dscp'            : dscpKw,
      '<dscpValue>'     : dscpValue,
   }

   @staticmethod
   def handler( mode, args ):
      dscp = args[ '<dscpValue>' ]
      cfg = ospfConfig()
      cfg.generalConfig.dscp = dscp

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = ospfConfig()
      cfg.generalConfig.dscp = cfg.generalConfig.dscpDefault

RouterOspfGeneralMode.addCommandClass( OspfQosDscpCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id>"
# command, in ospf mode.
#-------------------------------------------------------------------------------

class NoAreaCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'area AREA_ID'
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
          }
   noOrDefaultHandler = RouterOspfMode.noArea

RouterOspfMode.addCommandClass( NoAreaCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> authentication [message-digest [ auth-type] ]"
# command, in ospf mode.
#-------------------------------------------------------------------------------
authenticationKw = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Use authenticatication' )
messageDigestKw = CliMatcher.KeywordMatcher( 'message-digest',
      helpdesc='Enable message-digest authentication' )

class AreaAuthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID authentication [ MESSAGE_DIGEST ]'
   noOrDefaultSyntax = syntax
   data = {
            'area': areaKw,
            'AREA_ID': AreaIdExpression,
            'authentication': authenticationKw,
            'MESSAGE_DIGEST': messageDigestKw,
          }
   handler = RouterOspfMode.setAuthentication
   noOrDefaultHandler = RouterOspfMode.noAuthentication

if os.environ.get( 'ENABLE_CMDS_NOT_IN_FIRST_REL' ):
   RouterOspfMode.addCommandClass( AreaAuthenticationCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> default-cost <cost>" command, in ospf mode.
#-------------------------------------------------------------------------------
matcherRouteCostMetric = CliMatcher.IntegerMatcher( metricMin, metricMax,
      helpdesc='Default summary route cost metric' )
defaultCostKw = CliMatcher.KeywordMatcher( 'default-cost',
                                           helpdesc='Specify the cost for default '
                                           'summary route in stub/NSSA area' )

class AreaDefaultCostCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID default-cost COST'
   noOrDefaultSyntax = 'area AREA_ID default-cost ...'
   data = {
            'area' : areaKw,
            'AREA_ID' : AreaIdExpression,
            'default-cost': defaultCostKw,
            'COST': matcherRouteCostMetric,
          }
   handler = RouterOspfMode.setDefaultCost
   noOrDefaultHandler = RouterOspfMode.noDefaultCost

RouterOspfMode.addCommandClass( AreaDefaultCostCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> filter [ <ipaddr> <mask> | filterPrefix-list <pfx> ]"
#  command, in ospf mode.
#-------------------------------------------------------------------------------
filterPrefixMatcher = IpAddrMatcher.ipPrefixExpr(
                          'IP address', 'Subnet\'s mask value',
                          'IP address with mask length',
                          overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                          inverseMask=False, partial=True )

class areaFilterCmd( CliCommand.CliCommandClass ):
   syntax = '''area AREA_ID filter
               ( PREFIX | ( prefix-list PFXLIST ) )'''
   noOrDefaultSyntax = '''area AREA_ID filter
               ( PREFIX | prefix-list )'''
   data = {
            'area' : areaKw,
            'AREA_ID' : AreaIdExpression,
            'filter' : 'Specify the filter for incoming summary LSAs',
            'PREFIX' : filterPrefixMatcher,
            'prefix-list' : 'Specify list to filter for incoming LSAs',
            'PFXLIST' : prefixListNameMatcher,
            }
   handler = RouterOspfMode.setFilter
   noOrDefaultHandler = RouterOspfMode.noFilter

RouterOspfMode.addCommandClass( areaFilterCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> nssa"
# command, in ospf mode.
#-------------------------------------------------------------------------------

stubKw = CliMatcher.KeywordMatcher( 'stub', helpdesc='Configure a stub area' )
stubNoSummaryKw = CliMatcher.KeywordMatcher( 'no-summary',
                              helpdesc='Filter all type-3 LSAs in the stub area' )
nssaKw = CliMatcher.KeywordMatcher( 'nssa', helpdesc='Configure NSSA parameters' )
notsostubbyKw = CliMatcher.KeywordMatcher( 'not-so-stubby',
                                           helpdesc='Configure NSSA parameters' )
nssaNoSummaryKw = CliMatcher.KeywordMatcher( 'no-summary',
                              helpdesc='Filter all type-3 LSAs in the nssa area' )
nssaOnlyKw = CliMatcher.KeywordMatcher( 'nssa-only',
                                        helpdesc='Disable Type-7 LSA p-bit setting' )

class AreaNssaCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID ( nssa | not-so-stubby ) [ no-summary ]'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'no-summary': nssaNoSummaryKw,
          }
   handler = RouterOspfMode.setNssa
   noOrDefaultHandler = RouterOspfMode.noNssa

RouterOspfMode.addCommandClass( AreaNssaCmd )

class AreaNssaDefaultInfoOriginCmd( CliCommand.CliCommandClass ):
   syntax = '''area AREA_ID ( nssa | not-so-stubby ) default-information-originate
               [ { ( metric METRIC ) 
                 | ( metric-type METRIC_TYPE ) 
                 | ( nssa-only ) } ]'''
   noOrDefaultSyntax = '''area AREA_ID ( nssa | not-so-stubby ) 
                          default-information-originate ...'''
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'default-information-originate': 'Originate default Type 7 LSA',
           'metric' : CliCommand.singleKeyword( 'metric', 
              helpdesc='Metric for default route' ),
           'METRIC': CliMatcher.IntegerMatcher( metricMin, metricMax, 
              helpdesc='Value of the route metric' ),
           'metric-type': CliCommand.singleKeyword( 'metric-type', 
              helpdesc='Metric type for default route' ),
           'METRIC_TYPE': CliMatcher.IntegerMatcher( 1, 2, helpdesc='Metric type' ),
           'nssa-only': CliCommand.singleKeyword( 'nssa-only', 
              helpdesc='Limit default advertisement to this NSSA area' ), 
          }
   handler = RouterOspfMode.setNssa
   noHandler = RouterOspfMode.noNssaDefInfoOrigin
   defaultHandler = RouterOspfMode.noNssa

RouterOspfMode.addCommandClass( AreaNssaDefaultInfoOriginCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> ( nssa | not-so-stubby ) nssa-only"
# command, in ospf mode.
#-------------------------------------------------------------------------------
class AreaNssaNssaOnlyCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID ( nssa | not-so-stubby ) nssa-only'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'nssa-only': nssaOnlyKw,
          }
   handler = RouterOspfMode.setNssa
   noOrDefaultHandler = RouterOspfMode.noNssa

RouterOspfMode.addCommandClass( AreaNssaNssaOnlyCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> not-so-stubby lsa type-7 convert type-5"
# Legacy:
# "[no|default] area <area-id> nssa translate type7 always"
# command, in ospf mode.
#-------------------------------------------------------------------------------
class AreaType7toType5Cmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID not-so-stubby lsa type-7 convert type-5'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'not-so-stubby': notsostubbyKw,
           'lsa': 'Not-so-stubby LSA parameters',
           'type-7': 'Not-so-stubby Type-7 LSA parameters',
           'convert': 'Translation of not-so-stubby Type-7 LSAs',
           'type-5': 'Always translate Type-7 LSAs to Type-5',
          }
   handler = RouterOspfMode.setNssa
   noOrDefaultHandler = RouterOspfMode.noNssa

RouterOspfMode.addCommandClass( AreaType7toType5Cmd )

class AreaNssaTranslateCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID nssa translate type7 always'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'translate':  CliCommand.Node( CliMatcher.KeywordMatcher( 'translate', 
              helpdesc='Enable LSA translation' ), deprecatedByCmd='area AREA_ID'
              ' not-so-stubby lsa type-7 convert type-5' ),
           'type7': 'Enable Translation of Type-7 to Type-5 LSAs',
           'always': 'Always translate LSAs',
          }
   handler = RouterOspfMode.setNssa
   noOrDefaultHandler = RouterOspfMode.noNssa

RouterOspfMode.addCommandClass( AreaNssaTranslateCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> range <ip-address> <mask> [advertise | not-advertise]"
# command, in ospf mode.
#-------------------------------------------------------------------------------
matcherRangeCost = CliMatcher.IntegerMatcher( rangeCostMin, rangeCostMax,
                                              helpdesc='Range cost value' )

class OspfAreaRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID range PREFIX [ advertise | not-advertise ] [ cost COST ]'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'range': 'Configure route summarization',
           'PREFIX': filterPrefixMatcher,
           'advertise': 'Enable advertisement of the range',
           'not-advertise': 'Disable advertisement of the range',
           'cost': costKw,
           'COST': matcherRangeCost,
          }
   handler = RouterOspfMode.setRange
   noOrDefaultHandler = RouterOspfMode.noRange

RouterOspfMode.addCommandClass( OspfAreaRangeCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> stub [no-summary]" command, in ospf mode.
#-------------------------------------------------------------------------------
class OspfAreaStubNoSummaryCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID stub [ no-summary ]'
   noOrDefaultSyntax = syntax
   data = {
          'area': areaKw,
          'AREA_ID': AreaIdExpression,
          'stub': stubKw,
          'no-summary': stubNoSummaryKw,
          }
   handler = RouterOspfMode.setStub
   noOrDefaultHandler = RouterOspfMode.noStub

RouterOspfMode.addCommandClass( OspfAreaStubNoSummaryCmd )
#-------------------------------------------------------------------------------
# "[no|default] area <area-id> virtual-link <router-id>
#                            [authentication [message-digest | null]]
#                            [hello-interval <seconds>]
#                            [retransmit-interval <seconds>] 
#                            [transmit-delay <seconds>]
#                            [dead-interval <seconds>]
#                            [[authentication-key key] |
#                             [message-digest-key key-id auth-type key]]"
# command, in ospf mode.
#-------------------------------------------------------------------------------

helloIntervalKw = CliMatcher.KeywordMatcher( "hello-interval",
      helpdesc="Timer interval between transmission of hello packets" )

intervalMin = 1
intervalMax = 8192
matcherIntfIntervalSeconds = CliMatcher.IntegerMatcher( intervalMin, intervalMax,
                                                        helpdesc='Seconds' )

retransmitIntervalKw = CliMatcher.KeywordMatcher( 'retransmit-interval',
                                    helpdesc='LSA retransmission interval' )
transmitDelayKw = CliMatcher.KeywordMatcher( 'transmit-delay',
      helpdesc='Estimated LSA transmission delay' )
deadIntervalKw = CliMatcher.KeywordMatcher( 'dead-interval',
      helpdesc='Time interval to detect a dead router' )

authPasswdMinLen = 1
authPasswdMaxLen = 8
authPasswdPattern = r'[^\s]{%d,%d}' % ( authPasswdMinLen,
                                        authPasswdMaxLen )

messageDigestKeyKw = CliMatcher.KeywordMatcher( 'message-digest-key',
      helpdesc='Configure message digest authentication key' )

authKeyMinLen = 1
authKeyMaxLen = 16
authKeyPattern = r'[^\s]{%d,%d}' % ( authKeyMinLen,
                                     authKeyMaxLen )

class AreaVirtualLinkCmd( CliCommand.CliCommandClass ):
   syntax = '''area AREA_ID virtual-link ROUTER_ID
              [ hello-interval HELLO_SECONDS ] [ retransmit-interval RXSECONDS ]
              [ transmit-delay TXSECONDS ] [ dead-interval DEADSECONDS ]'''
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'virtual-link': 'Configure a virtual link',
           'ROUTER_ID': IpAddrMatcher.IpAddrMatcher( 'Router ID of the '
                                                     'virtual link peer' ),
           'hello-interval': helloIntervalKw,
           'HELLO_SECONDS': matcherIntfIntervalSeconds,
           'RXSECONDS': matcherIntfIntervalSeconds,
           'TXSECONDS': matcherIntfIntervalSeconds,
           'DEADSECONDS': matcherIntfIntervalSeconds,
           'retransmit-interval': retransmitIntervalKw,
           'transmit-delay': transmitDelayKw,
           'dead-interval': deadIntervalKw,
          }
   handler = RouterOspfMode.setVirtualLink
   noOrDefaultHandler = RouterOspfMode.noVirtualLink

if os.environ.get( 'ENABLE_CMDS_NOT_IN_FIRST_REL' ):
   RouterOspfMode.addCommandClass( AreaVirtualLinkCmd ) 

#-------------------------------------------------------------------------------
# "[no|default] log-adjacency-changes [detail]" command, in ospf mode.
#-------------------------------------------------------------------------------
logAdjacencyChangesKw = CliMatcher.KeywordMatcher( "log-adjacency-changes",
            helpdesc='Enable logging of adjacency events')
logAdjacencyChangesDetailKw = CliMatcher.KeywordMatcher( 'detail',
            helpdesc='Log every state change' )

class OspfLogAdjacencyChangesCmd( CliCommand.CliCommandClass ):
   syntax = 'log-adjacency-changes [ DETAIL ]'
   noOrDefaultSyntax = 'log-adjacency-changes ...'
   data = {
         'log-adjacency-changes' : logAdjacencyChangesKw,
         'DETAIL' : logAdjacencyChangesDetailKw,
         }
   handler = RouterOspfMode.setLogAdjacencyChanges
   defaultHandler = RouterOspfMode.setLogAdjacencyChanges
   noHandler = RouterOspfMode.noLogAdjacencyChanges

RouterOspfMode.addCommandClass( OspfLogAdjacencyChangesCmd )
                             
#-------------------------------------------------------------------------------
# "[no|default] default-metric <metric-value>" command, in ospf mode.
#-------------------------------------------------------------------------------
defaultMetricKw = CliMatcher.KeywordMatcher( 'default-metric',
            helpdesc='Configure the default metric for redistributed routes' )
matcherDefaultMetric = CliMatcher.IntegerMatcher( metricMin, metricMax,
                                     helpdesc='Default metric value' )
class OspfDefaultMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'default-metric METRIC_VALUE'
   noOrDefaultSyntax = 'default-metric ...'
   data = {
         'default-metric' : defaultMetricKw,
         'METRIC_VALUE' : matcherDefaultMetric,
         }
   handler = RouterOspfMode.setDefaultMetric
   noOrDefaultHandler = handler

if os.environ.get( 'ENABLE_CMDS_NOT_IN_FIRST_REL' ):
   RouterOspfMode.addCommandClass( OspfDefaultMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] distance ospf [ intra-area | inter-area | external ]
#                        <dist>" command, in ospf mode.
#-------------------------------------------------------------------------------

distanceKw = CliMatcher.KeywordMatcher( 'distance',
      helpdesc='Configure administrative distance' )
intraAreaKw = CliMatcher.KeywordMatcher( 'intra-area',
      helpdesc='Routes within an area' )
externalRtKw = CliMatcher.KeywordMatcher( 'external',
      helpdesc='Routes external to the AS' )

distanceMin = 1
distanceMax = 255
matcherDistance = CliMatcher.IntegerMatcher( distanceMin, distanceMax, 
                                             helpdesc='Distance value' )

class DistanceOspfCmd( CliCommand.CliCommandClass ):
   syntax = 'distance ospf ( intra-area | inter-area | external ) DIST'
   noOrDefaultSyntax = 'distance ospf ( intra-area | inter-area | external ) ...'
   data = {
           'distance': distanceKw,
           'ospf': ospfKw,
           'intra-area': intraAreaKw,
           'inter-area': 'Routes from other areas',
           'external': externalRtKw,
           'DIST': matcherDistance,
          }
   handler = RouterOspfMode.setDistance
   noOrDefaultHandler = RouterOspfMode.noDistance

RouterOspfMode.addCommandClass( DistanceOspfCmd )
                             
#-------------------------------------------------------------------------------
# "[no|default] network <ip-address> <wildcard-mask> area <area-id>"
# command, in ospf mode.
#-------------------------------------------------------------------------------

matcherPrefix = IpAddrMatcher.ipPrefixExpr( 'Network address',
                                            'Network wildcard mask', 'Prefix', 
                                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                                       inverseMask=True )

class OspfNetworkAreaCmd( CliCommand.CliCommandClass ):
   syntax = 'network IP_ADDRESS area AREA_ID'
   noOrDefaultSyntax = 'network IP_ADDRESS area AREA_ID'
   data = {
           'network': 'Configure routing for a network',
           'IP_ADDRESS': matcherPrefix,
           'area': areaKw,
           'AREA_ID': AreaIdExpression, 
          }
   handler = RouterOspfMode.setNetwork
   noOrDefaultHandler = RouterOspfMode.noNetwork

RouterOspfMode.addCommandClass( OspfNetworkAreaCmd )
                             
#-------------------------------------------------------------------------------
# "[no|default] router-id <ip-address>" command, in ospf mode.
#
# The no rule shouldn't have trailingGarbage, or else 'no router ospf 1'
# executed in router-ospf mode matches the 'no router-id' rule.
#-------------------------------------------------------------------------------

routerIdKw = CliMatcher.KeywordMatcher( 'router-id',
      helpdesc='Configure the router ID for the OSPF instance' )
matcherRouterId = IpAddrMatcher.IpAddrMatcher( 
      "OSPF 32-bit router ID as an IP address" ) 

class OspfRouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ROUTER_ID'
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
         'router-id' : routerIdKw,
         'ROUTER_ID' : matcherRouterId,
         }
   handler = RouterOspfMode.setRouterId
   noOrDefaultHandler = RouterOspfMode.noRouterId 

RouterOspfMode.addCommandClass( OspfRouterIdCmd )
#-------------------------------------------------------------------------------
# In ospf mode
# "[no|default] max-metric router-lsa [ external-lsa [ max-metric-value ]]
#                                  [ include-stub ]
#                                  [ inter-area-lsa [ max-metric-value ]]
#                                  [ on-startup { <announce-time> | wait-for-bgp ] }
#                                  [ prefix-lsa ]
#                                  [ stub-prefix-lsa [ max-metric-value ]]
#                                  [ summary-lsa [ max-metric-value ]]
#------------------------------------------------------------------------------
maxMetricValueMin = 1
maxMetricValueMax = 16777215

maxMetricTimerMin = 5
maxMetricTimerMax = 86400

matcherExternalLsa = CliMatcher.KeywordMatcher( 'external-lsa',
      helpdesc='Override external-lsa metric with ' )
matcherExtLsaMetric = CliMatcher.IntegerMatcher( maxMetricValueMin,
                                                 maxMetricValueMax,
      helpdesc='Overriding metric in external-LSAs (default 16711680)' )
matcherIncludeStub = CliMatcher.KeywordMatcher( 'include-stub',
      helpdesc='Set maximum metric for stub links in router-LSAs' )
matcherSummaryLsa = CliMatcher.KeywordMatcher( 'summary-lsa',
      helpdesc='Override summary-lsa metric with max-metric value' )
matcherSumLsaMetric = CliMatcher.IntegerMatcher( maxMetricValueMin,
                                                 maxMetricValueMax,
      helpdesc='Overriding metric in summary-LSAs (default 16711680)' )
matcherOnStartup = CliMatcher.KeywordMatcher( 'on-startup',
      helpdesc='Set maximum metric temporarily after reboot' )
matcherMaxMetricTimer = CliMatcher.IntegerMatcher( maxMetricTimerMin,
                                                   maxMetricTimerMax,
      helpdesc='Time, in seconds, router-LSAs are originated with max-metric' )

class OspfMaxMetricRouterLsaCmd( CliCommand.CliCommandClass ):
   syntax = '''max-metric router-lsa
   [ { ( external-lsa [ EXTERNAL_LSA_METRIC_VALUE ] )
   | ( include-stub )
   | ( on-startup ( ANNOUNCE_TIME | wait-for-bgp ) )
   | ( summary-lsa [ SUMMARY_LSA_METRIC_VALUE ] ) } ]'''
   noOrDefaultSyntax = syntax.replace( '( ANNOUNCE_TIME | wait-for-bgp )', 
                                       '[ ANNOUNCE_TIME | wait-for-bgp ]' )
   data = {
          'max-metric': 'Set maximum metric',
          'router-lsa': 'Maximum metric in self-originated router-LSAs',
          'external-lsa': CliCommand.singleNode( matcherExternalLsa ),
          'EXTERNAL_LSA_METRIC_VALUE': matcherExtLsaMetric,
          'include-stub': CliCommand.singleNode( matcherIncludeStub ),
          'on-startup': CliCommand.singleNode( matcherOnStartup ),
          'ANNOUNCE_TIME': matcherMaxMetricTimer,
          'wait-for-bgp': 'Let BGP decide when to originate router-LSA with '
                          'normal metric',
          'summary-lsa': CliCommand.singleNode( matcherSummaryLsa ),
          'SUMMARY_LSA_METRIC_VALUE': matcherSumLsaMetric, 
          }
   handler = RouterOspfMode.setMaxMetric
   noOrDefaultHandler = RouterOspfMode.noMaxMetric

RouterOspfMode.addCommandClass( OspfMaxMetricRouterLsaCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart" in ospf mode
#------------------------------------------------------------------------------
gracefulRestartKw = CliMatcher.KeywordMatcher( 'graceful-restart',
                                 helpdesc='Enable graceful restart mode' )

class OspfGracefulRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart'
   noOrDefaultSyntax = syntax
   data = { 'graceful-restart' : gracefulRestartKw }
   handler = RouterOspfMode.setGracefulRestart

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.gracefulRestart = False
      instanceConfig.grGracePeriod = instanceConfig.grGracePeriodDefault
      instanceConfig.grPlannedOnly = instanceConfig.grPlannedOnlyDefault

RouterOspfMode.addCommandClass( OspfGracefulRestartCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart grace-period <seconds>" in ospf mode
#------------------------------------------------------------------------------
gracePeriodMin = 1
gracePeriodMax = 1800

gracePeriodKw = CliMatcher.KeywordMatcher( 'grace-period',
                              helpdesc='Specify maximum time to wait for '
                                       'graceful-restart to complete' )
matcherGracePeriod = CliMatcher.IntegerMatcher( gracePeriodMin, gracePeriodMax,
                                                helpdesc='Number of seconds' )

class OspfGRGracePeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart grace-period GRACE_PERIOD'
   noOrDefaultSyntax = 'graceful-restart grace-period ...'
   data = { 'graceful-restart'  : gracefulRestartKw,
            'grace-period'      : gracePeriodKw,
            'GRACE_PERIOD'      : matcherGracePeriod
          }

   @staticmethod
   def handler( mode, args ):
      mode.setGracefulRestart()
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.grGracePeriod = args[ 'GRACE_PERIOD' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.grGracePeriod = instanceConfig.grGracePeriodDefault

RouterOspfMode.addCommandClass( OspfGRGracePeriodCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart planned-only" in ospf mode
# db XXX Not Yet Supported by gated or required
#------------------------------------------------------------------------------
#class OspfGRPlannedOnlyCmd( CliCommand.CliCommandClass ):
#   syntax = 'graceful-restart planned-only'
#   noOrDefaultSyntax = syntax
#   data = { 'graceful-restart'  : gracefulRestartKw,
#            'planned-only'      : 'Restrict graceful-restarts to planned events'
#          }
#
#   @staticmethod
#   def handler( mode, args ):
#      mode.setGracefulRestart()
#      instanceConfig = mode._getInstanceConfig()
#      instanceConfig.grPlannedOnly = True
#
#   @staticmethod
#   def noOrDefaultHandler( mode, args ):
#      instanceConfig = mode._getInstanceConfig()
#      instanceConfig.grPlannedOnly = instanceConfig.grPlannedOnlyDefault
#
#RouterOspfMode.addCommandClass( OspfGRPlannedOnlyCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper" in ospf mode
#------------------------------------------------------------------------------
gracefulRestartHelperKw = CliMatcher.KeywordMatcher( 'graceful-restart-helper',
                                       helpdesc='Enable graceful restart helper' )

class OspfGRHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart-helper'
   noOrDefaultSyntax = syntax
   data = { 'graceful-restart-helper' : gracefulRestartHelperKw }
   handler = RouterOspfMode.setGrHelper

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.grHelper = False
      instanceConfig.grHelperLooseLsaChk = instanceConfig.grHelperLooseLsaChkDefault

RouterOspfMode.addCommandClass( OspfGRHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] shutdown" command, in ospf mode.
#-------------------------------------------------------------------------------
class RouterOspfShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = { 'shutdown': 'Disable the OSPF instance' }
   handler = RouterOspfMode.setShutdown
   noOrDefaultHandler = RouterOspfMode.noShutdown

RouterOspfMode.addCommandClass( RouterOspfShutdownCmd )

#-------------------------------------------------------------------------------
# "[no|default] max-lsa <N> [<N>] [warning-only] [ignore-time <minutes>]
# [ignore-count <num>] [reset-time <minutes>]" command, in ospf mode.
#-------------------------------------------------------------------------------
maxLsaMin = 0
maxLsaMax = 100000
maxLsaThresholdMin = 25
maxLsaThresholdMax = 99
maxLsaMinutesMin = 1
maxLsaMinutesMax = 60
maxLsaIgnoreCountMin = 1
maxLsaIgnoreCountMax = 20

maxLsaMinutesMatcher = CliMatcher.IntegerMatcher( maxLsaMinutesMin, maxLsaMinutesMax,
                                                  helpdesc='minutes' )

class OspfMaxLsaCmd( CliCommand.CliCommandClass ):
   syntax = 'max-lsa COUNT [ THRESHOLD ] [ ( warning-only | ' \
                                            '{ ( ignore-time IGNORE_TIME ) | ' \
                                              '( ignore-count IGNORE_COUNT ) | ' \
                                              '( reset-time RESET_TIME ) } ) ]'
   noOrDefaultSyntax = 'max-lsa [ warning-only ] ...'
   data = { 'max-lsa' : 'Maximum number of LSAs permitted in this instance',
            'COUNT' : CliMatcher.IntegerMatcher( maxLsaMin, maxLsaMax,
                                                 helpdesc='LSAs (0 for unlimited)' ),
            'THRESHOLD' : CliMatcher.IntegerMatcher( maxLsaThresholdMin,
                                          maxLsaThresholdMax,
                                          helpdesc= 'Warning threshold percentage' ),
            'warning-only' : 'Only give warning message when limit is exceeded',
            'ignore-time' : CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'ignore-time',
                                    helpdesc='Minutes to shut down the instance' ),
                              maxMatches=1 ),
            'IGNORE_TIME' : maxLsaMinutesMatcher,
            'ignore-count' : CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'ignore-count',
                                    helpdesc='Number of episodes before permanent '
                                             'shutdown' ),
                              maxMatches=1 ),
            'IGNORE_COUNT' : CliMatcher.IntegerMatcher( maxLsaIgnoreCountMin,
                                                        maxLsaIgnoreCountMax,
                                                        helpdesc='count' ),
            'reset-time' : CliCommand.Node(
                              matcher=CliMatcher.KeywordMatcher( 'reset-time',
                                    helpdesc='Minutes of no overload before '
                                             'resetting episode count' ),
                              maxMatches=1 ),
            'RESET_TIME' : maxLsaMinutesMatcher
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      if 'THRESHOLD' in args:
         instanceConfig.maxLsaThreshold = args[ 'THRESHOLD' ]
      if 'warning-only' in args:
         instanceConfig.maxLsaWarningOnly = True
      else:
         argToAttrDict = { 'IGNORE_TIME' : 'maxLsaIgnoreTime',
                           'IGNORE_COUNT' : 'maxLsaIgnoreCount',
                           'RESET_TIME' : 'maxLsaResetTime' }
         for i in argToAttrDict:
            if i in args:
               setattr( instanceConfig, argToAttrDict[ i ], args[ i ][ 0 ] )
         instanceConfig.maxLsaWarningOnly = instanceConfig.maxLsaWarningOnlyDefault
      instanceConfig.maxLsa = args[ 'COUNT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.maxLsaWarningOnly = instanceConfig.maxLsaWarningOnlyDefault
      if 'warning-only' not in args:
         instanceConfig.maxLsa = instanceConfig.maxLsaDefault
         instanceConfig.maxLsaThreshold = instanceConfig.maxLsaThresholdDefault
         instanceConfig.maxLsaIgnoreCount = instanceConfig.maxLsaIgnoreCountDefault
         instanceConfig.maxLsaIgnoreTime = instanceConfig.maxLsaIgnoreTimeDefault
         instanceConfig.maxLsaResetTime = instanceConfig.maxLsaResetTimeDefault

RouterOspfMode.addCommandClass( OspfMaxLsaCmd )

#-------------------------------------------------------------------------------
# "[no|default] adjacency exchange-start threshold <num>" command, in ospf mode.
#-------------------------------------------------------------------------------

adjacencyKw = CliMatcher.KeywordMatcher( 'adjacency',
      helpdesc='Configure adjacency options for OSPF instance' )

exchangeStartKw = CliMatcher.KeywordMatcher( 'exchange-start',
      helpdesc='Configure exchange-start options for OSPF instance' )

thresholdKw = CliMatcher.KeywordMatcher( 'threshold',
      helpdesc='Configure the maximum threshold of EXCH-START peers to' 
               'bring up simultaneously' )

matcherSimultaneousPeers = CliMatcher.IntegerMatcher( 1, 0xffffffff, 
      helpdesc='Number of peers to bring up simultaneously' )

class OspfAdjExchStartThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency exchange-start threshold THRESHOLD'
   noOrDefaultSyntax = 'adjacency exchange-start threshold ...'
   data = {
          'adjacency': adjacencyKw,
          'exchange-start': exchangeStartKw,
          'threshold': thresholdKw,
          'THRESHOLD': matcherSimultaneousPeers,
          }
   handler = RouterOspfMode.setSimultaneousBringUp 
   noOrDefaultHandler = RouterOspfMode.noSimultaneousBringUp 

RouterOspfMode.addCommandClass( OspfAdjExchStartThresholdCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute <static|connected|bgp> [include leaked]
#              [route-map <routemap>] command, in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeProtocolExpression( CliCommand.CliExpression ):
   if Toggles.OspfToggleLib.toggleBgpLeakedRedistEnabled():
      expression = '( static | connected | bgp )'
      data = {
            'static'    : 'Static routes',
            'connected' : 'Connected interface routes',
            'bgp': 'BGP routes',
      }
   else:
      expression = '( static | connected )'
      data = {
            'static'    : 'Static routes',
            'connected' : 'Connected interface routes',
      }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'static' in args:
         args[ 'proto' ] = 'protoStatic'
      elif 'connected' in args:
         args[ 'proto' ] = 'protoDirect'
      elif 'bgp' in args:
         args[ 'proto' ] = 'protoBgp'

class RedistributeProtocolCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute PROTOCOL [ include leaked ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute PROTOCOL ...'

   data = {
         'redistribute' : 'Redistribute routes in to OSPF',
         'PROTOCOL'     : RedistributeProtocolExpression,
         'include'      : 'Include leaked routes',
         'leaked'       : 'Include leaked routes',
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( args[ 'proto' ],
                            mapName=args.get( 'MAPNAME' ),
                            includeLeaked=( 'leaked' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( args[ 'proto' ] )

RouterOspfMode.addCommandClass( RedistributeProtocolCommand )

#---------------------------------------------------------------------------------
# [no|default] redistribute isis [include leaked] [level-1|level-1-2|level-2]
#              [route-map <route-map-name>] command, in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeIsisLeakedCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis [ include leaked ] ' \
            '[ level-1|level-2|level-1-2 ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute isis ...'

   data = {
         'redistribute' : 'Redistribute routes in to OSPF',
         'isis'         : 'IS-IS routes',
         'include'      : 'Include leaked routes',
         'leaked'       : 'Include leaked routes',
         'level-1'      : 'Redistribute IS-IS level-1 routes',
         'level-2'      : 'Redistribute IS-IS level-2 routes',
         'level-1-2'    : 'Redistribute IS-IS level-1 and level-2 routes',
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }
 
   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'isisLevel' ] = 'level2'
      if 'level-1' in args:
         args[ 'isisLevel' ] = 'level1'
      elif 'level-2' in args:
         args[ 'isisLevel' ] = 'level2'
      elif 'level-1-2' in args:
         args[ 'isisLevel' ] = 'level1_2'

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( 'protoIsis', mapName=args.get( 'MAPNAME' ),
                            isisLevel=args[ 'isisLevel' ],
                            includeLeaked=( 'leaked' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( 'protoIsis' )

RouterOspfMode.addCommandClass( RedistributeIsisLeakedCommand )

# NOTE: Without InstanceAndLeakedExpression, RedistributeOspfLeakedCommand doesn't
# parse the command correctly, no need for adapter fnction
class InstanceAndLeakedExpression( CliCommand.CliExpression ):
   expression = "instance [ include leaked ]"
   data = { 
         'instance'     : 'Routes learned by other OSPF instances',
         'include'      : 'Include leaked routes',
         'leaked'       : 'Include leaked routes',
   }   

class RedistributeOspfBaseCommand( CliCommand.CliCommandClass ):
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'internal' in args:
         args[ 'matchRouteType' ] = 'internal'
      elif 'external' in args:
         args[ 'matchRouteType' ] = 'external'
      elif 'nssa-external' in args:
         args[ 'matchRouteType' ] = 'nssa-external'

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( 'protoOspf', mapName=args.get( 'MAPNAME' ),
                            includeLeaked=( 'leaked' in args ),
                            matchRouteType=args.get( 'matchRouteType' ),
                            allowOspfInstance=( 'instance' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( 'protoOspf',
                           matchRouteType=args.get( 'matchRouteType' ) )

#---------------------------------------------------------------------------------
# [no|default] redistribute ospf ( instance [ include leaked ] | leaked )
#              [ match ( external | internal | nssa-external ) ]
#              [ route-map <route-map-name> ] command, in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeOspfInstanceLeakedCommand( RedistributeOspfBaseCommand ):
   syntax = \
         'redistribute ospf ( INSTANCE_EXP | leaked )' \
         '[ match ( internal | external | nssa-external ) ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = \
         'redistribute ospf ( INSTANCE_EXP | leaked )' \
         '[ match ( internal | external | nssa-external ) ] ...'

   data = {
         'redistribute' : 'Redistribute routes in to OSPF',
         'ospf'         : 'OSPF routes',
         'INSTANCE_EXP' : InstanceAndLeakedExpression,
         'leaked'       : 'Include leaked routes',
         'match'        : 'Routes learned by the OSPF protocol',
         'internal'     : 'OSPF routes learned from internal sources',
         'external'     : 'OSPF routes learned from external sources',
         'nssa-external': 'OSPF routes learned from external NSSA sources',
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }

#---------------------------------------------------------------------------------
# [no|default] redistribute ospf leaked [match (external|internal|nssa-external)]
#              [route-map <route-map-name>] command, in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeOspfLeakedCommand( RedistributeOspfBaseCommand ):
   syntax = \
         'redistribute ospf leaked' \
         '[ match ( internal | external | nssa-external ) ] [ route-map MAPNAME ]'
   noOrDefaultSyntax = \
         'redistribute ospf leaked' \
         '[ match ( internal | external | nssa-external ) ] ...'

   data = {
         'redistribute' : 'Redistribute routes in to OSPF',
         'ospf'         : 'OSPF routes',
         'leaked'       : 'Include leaked routes',
         'match'        : 'Routes learned by the OSPF protocol',
         'internal'     : 'OSPF routes learned from internal sources',
         'external'     : 'OSPF routes learned from external sources',
         'nssa-external': 'OSPF routes learned from external NSSA sources',
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }

if Toggles.OspfToggleLib.toggleOspfInstanceRedistEnabled():
   RouterOspfMode.addCommandClass( RedistributeOspfInstanceLeakedCommand )
else:
   RouterOspfMode.addCommandClass( RedistributeOspfLeakedCommand )

#---------------------------------------------------------------------------------
# [no|default] redistribute <proto> route-map <routemap>" command, 
# in "router-ospf" mode. 
#---------------------------------------------------------------------------------
matcherAggregate = CliMatcher.KeywordMatcher( 'aggregate',
                                     helpdesc='BGP Aggregate routes' )

class RedistributeOspfProtocolExpression( CliCommand.CliExpression ):
   if Toggles.OspfToggleLib.toggleBgpLeakedRedistEnabled():
      expression = '( rip | aggregate )'
      data = {
            'rip': 'RIP routes',
            'aggregate': CliCommand.Node( matcherAggregate, hidden=True ),
      }
   else:
      expression = '( bgp | rip | aggregate )'
      data = {
            'bgp': 'BGP routes',
            'rip': 'RIP routes',
            'aggregate': CliCommand.Node( matcherAggregate, hidden=True ),
      }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'rip' in args:
         args[ 'PROTOCOL' ] = 'protoRip'
      elif 'aggregate' in args:
         args[ 'PROTOCOL' ] = 'protoBgpAggregate'
      elif not Toggles.OspfToggleLib.toggleBgpLeakedRedistEnabled() \
         and 'bgp' in args:
         args[ 'PROTOCOL' ] = 'protoBgp'

class RedistributeOspfProtocolCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute PROTOCOL [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute PROTOCOL ...'

   data = {
         'redistribute': 'Redistribute routes in to OSPF',
         'PROTOCOL': RedistributeOspfProtocolExpression,
         'route-map': RouteMapMatchers.routeMapApplication,
         'MAPNAME': mapNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( args[ 'PROTOCOL' ],
                            mapName=args.get( 'MAPNAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( args[ 'PROTOCOL' ] )

RouterOspfMode.addCommandClass( RedistributeOspfProtocolCommand )

#---------------------------------------------------------------------------------
# [no|default] distribute-list route-map <routemap>" command,
# [no|default] distribute-list prefix-list <pfx>" command,
# in "router-ospf" mode. 
#---------------------------------------------------------------------------------

class OspfDistributeListCmd( CliCommand.CliCommandClass ):
   syntax = ( 'distribute-list '
              '( ( route-map MAPNAME ) | ( prefix-list PREFIXNAME ) ) in' )
   noOrDefaultSyntax = 'distribute-list ( route-map | prefix-list ) ...'
   data = { 'distribute-list': 'Distribute list',
            'route-map': RouteMapMatchers.routeMapApplication,
            'MAPNAME': mapNameMatcher,
            'prefix-list': 'Prefix list',
            'PREFIXNAME': prefixListNameMatcher,
            'in': 'Inbound',
   }
   handler = RouterOspfMode.setDistListIn
   noOrDefaultHandler = RouterOspfMode.noDistListIn

RouterOspfMode.addCommandClass( OspfDistributeListCmd )

#---------------------------------------------------------------------------------
# "[no|default] mpls ldp sync default" in 'router ospf <>' mode
#---------------------------------------------------------------------------------
class OspfMplsLdpSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp sync default'
   noOrDefaultSyntax = syntax
   data = { 'mpls': 'MPLS sync configuration',
            'ldp': 'LDP sync configuration',
            'sync': 'Sync configuration',
            'default': 'Set global default',
          }

   handler = RouterOspfMode.setMplsLdpSync
   noOrDefaultHandler = RouterOspfMode.noSetMplsLdpSync

RouterOspfMode.addCommandClass( OspfMplsLdpSyncCmd )

#---------------------------------------------------------------------------------
# [no|default] "maximum-paths <paths>" command, in "router-ospf" mode. 
#---------------------------------------------------------------------------------
def maxEcmpRangeFn( mode ):
   # This code can be called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb. Allowing any value in this case.
   if routingHardwareStatus is None or \
      routingHardwareStatus.maxEcmp == 0:
      return( 1, 0xffffffff )
   else:
      return( 1, routingHardwareStatus.maxEcmp )

maxEcmpPathsMatcher = CliMatcher.DynamicIntegerMatcher( rangeFn=maxEcmpRangeFn,
                         helpdesc='Maximum number of next-hops in an ECMP route' )

class OspfMaxPathsCmd( CliCommand.CliCommandClass ):
   syntax = 'maximum-paths PATHS'
   noOrDefaultSyntax = 'maximum-paths ...'
   data = { 'maximum-paths': 'Maximum number of next-hops in an ECMP route',
            'PATHS': maxEcmpPathsMatcher,
          }

   handler = RouterOspfMode.setMaxEcmp
   noOrDefaultHandler = RouterOspfMode.noMaxEcmp

RouterOspfMode.addCommandClass( OspfMaxPathsCmd )

#---------------------------------------------------------------------------------
# [no|default] "point-to-point routes" command, in "router-ospf" mode. 
#---------------------------------------------------------------------------------
class OspfP2PRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'point-to-point routes'
   noOrDefaultSyntax = syntax
   data = { 'point-to-point': 'Configure Point-to-point specific features',
            'routes': 'Enable/Disable Point-to-point routes',
          }

   handler = RouterOspfMode.setPointToPointRoutes
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfP2PRoutesCmd )

#---------------------------------------------------------------------------------
# [no|default] "tunnel routes" command, in "router-ospf" mode. 
#---------------------------------------------------------------------------------
class OspfTunnelRoutesCmd( CliCommand.CliCommandClass ):
   syntax = '''tunnel routes'''
   noOrDefaultSyntax = syntax

   handler = RouterOspfMode.setTunnelRoutes
   noHandler = defaultHandler = RouterOspfMode.noTunnelRoutes

   data = {
            'tunnel' : 'GRE tunnels',
            'routes' : 'Enable/Disable OPSFv2 routes over GRE Tunnels'
          }

RouterOspfMode.addCommandClass( OspfTunnelRoutesCmd )

#---------------------------------------------------------------------------------
# "[no|default] dn-bit-ignore [lsa type-5 type-7]" command, in "router-ospf" mode.
#---------------------------------------------------------------------------------
dnBitIgnoreKw = CliMatcher.KeywordMatcher( 'dn-bit-ignore',
               helpdesc='Disable DN-bit check for Type-3, Type-5 and '
               'Type-7 LSAs in non-default VRFs' )
lsaKw = CliMatcher.KeywordMatcher( 'lsa',
      helpdesc='Disable DN-bit check only for Type-5 and Type-7 LSAs in non-default '
               'VRFs' )
type5Kw = CliMatcher.KeywordMatcher( 'type-5',
      helpdesc='Disable DN-bit check only for Type-5 and Type-7 LSAs in non-default '
               'VRFs' )
type7Kw = CliMatcher.KeywordMatcher( 'type-7',
      helpdesc='Disable DN-bit check only for Type-5 and Type-7 LSAs in non-default '
               'VRFs' )

class OspfDnBitIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'dn-bit-ignore [ lsa type-5 type-7 ]'
   noOrDefaultSyntax = 'dn-bit-ignore ...'
   data = { 'dn-bit-ignore': dnBitIgnoreKw,
            'lsa': lsaKw,
            'type-5': type5Kw,
            'type-7': type7Kw,
          }

   handler = RouterOspfMode.setDnBitIgnore
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfDnBitIgnoreCmd )

#------------------------------------------------------------------------------
# "timers spf <seconds>", hidden as its deprecated
#------------------------------------------------------------------------------
spfIntervalMin = 1
spfIntervalMax = 65535
matcherSpfTimer = CliMatcher.IntegerMatcher( spfIntervalMin, spfIntervalMax,
                                             helpdesc='seconds' )
matcherTimers = CliMatcher.KeywordMatcher( 'timers', 'Configure OSPF timers' )
matcherSpf = CliMatcher.KeywordMatcher( 'spf', helpdesc='Configure SPF timers' )
matcherLsa = CliMatcher.KeywordMatcher( 'lsa', helpdesc='Configure OSPF LSA timers' )

class OspfTimersSpfCmd( CliCommand.CliCommandClass ):
   syntax = 'timers spf SECONDS'
   data = {
          'timers': matcherTimers,
          'spf': CliCommand.Node( matcherSpf, hidden=True ),
          'SECONDS': matcherSpfTimer,
          }
   handler = RouterOspfMode.setSpfInterval

RouterOspfMode.addCommandClass( OspfTimersSpfCmd )
#------------------------------------------------------------------------------
# "[no|default] timers spf delay initial <sfp-start> <sfp-hold> <sfp-max-wait>"
# Legacy:
# "[no|default] timers throttle spf <sfp-start> <sfp-hold> <sfp-max-wait>"
#------------------------------------------------------------------------------
spfThrottleMin = 0
spfThrottleMax = 65535000
matcherSpfStart = CliMatcher.IntegerMatcher( spfThrottleMin, spfThrottleMax,
      helpdesc='Initial SPF schedule delay in msecs' )
matcherSpfHold = CliMatcher.IntegerMatcher( spfThrottleMin, spfThrottleMax,
      helpdesc='Min Hold time between two SPFs in msecs' )
matcherSpfMaxWait = CliMatcher.IntegerMatcher( spfThrottleMin, spfThrottleMax,
      helpdesc='Max wait time between two SPFs in msecs' )

class OspfTimersDelayInitialCmd( CliCommand.CliCommandClass ):
   syntax = 'timers spf delay initial SPF_START SPF_HOLD SPF_MAX_WAIT'
   noOrDefaultSyntax = 'timers spf ...'
   data = {
          'timers': matcherTimers,
          'spf': matcherSpf,
          'delay': 'Configure SPF timers',
          'initial': 'Configure SPF timers',
          'SPF_START': matcherSpfStart,
          'SPF_HOLD': matcherSpfHold,
          'SPF_MAX_WAIT': matcherSpfMaxWait,
          }
   handler = RouterOspfMode.setSpfIntervalThrottle
   noOrDefaultHandler = RouterOspfMode.noSpfIntervalThrottle

RouterOspfMode.addCommandClass( OspfTimersDelayInitialCmd )

class OspfTimersThrottleSpfCmd( CliCommand.CliCommandClass ):
   syntax = 'timers throttle spf SPF_START SPF_HOLD SPF_MAX_WAIT'
   noOrDefaultSyntax = 'timers throttle spf ...'
   data = {
          'timers': matcherTimers,
          'throttle': 'Configure OSPF throttle timers',
          'spf':  CliCommand.Node( matcherSpf,
             deprecatedByCmd='timers spf delay initial' ),
          'SPF_START': matcherSpfStart,
          'SPF_HOLD': matcherSpfHold,
          'SPF_MAX_WAIT': matcherSpfMaxWait,
          }
   handler = RouterOspfMode.setSpfIntervalThrottle
   noOrDefaultHandler = RouterOspfMode.noSpfIntervalThrottle

RouterOspfMode.addCommandClass( OspfTimersThrottleSpfCmd )

#------------------------------------------------------------------------------
# "[no|default] timers lsa rx min interval <msecs>"
# Legacy:
# "[no|default] timers lsa arrival <msecs>"
#------------------------------------------------------------------------------
lsaArrivalIntMin = 0
lsaArrivalIntMax = 600000
matcherArrival = CliMatcher.KeywordMatcher( 'arrival',
      helpdesc='Configure OSPF LSA arrival timer' )
matcherLsaArrivalInt = CliMatcher.IntegerMatcher( lsaArrivalIntMin, lsaArrivalIntMax,
      helpdesc='Min interval in msecs between accepting the same LSA' )

class OspfTimersLsaRxMinIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa rx min interval MILLISECONDS'
   noOrDefaultSyntax = 'timers lsa rx min interval ...'
   data = {
          'timers': matcherTimers,
          'lsa': matcherLsa,
          'rx': 'Configure OSPF LSA receiving timers',
          'min':'Configure OSPF LSA arrival timer',
          'interval': 'Configure OSPF LSA arrival timer',
          'MILLISECONDS': matcherLsaArrivalInt,
          }
   handler = RouterOspfMode.setLsaArrivalInterval
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfTimersLsaRxMinIntervalCmd )

class OspfTimersLsaArrivalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa arrival MILLISECONDS'
   noOrDefaultSyntax = 'timers lsa arrival ...'
   data = {
          'timers': matcherTimers,
          'lsa': matcherLsa,
          'arrival': CliCommand.Node( matcherArrival, deprecatedByCmd='timers lsa '
             'rx min interval' ),
          'MILLISECONDS': matcherLsaArrivalInt,
          }
   handler = RouterOspfMode.setLsaArrivalInterval
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfTimersLsaArrivalCmd )
#------------------------------------------------------------------------------
# "[no|default] timers lsa tx delay initial <lsa-start> <lsa-hold> <lsa-max-wait>"
# Legacy:
# "[no|default] timers throttle lsa all <lsa-start> <lsa-hold> <lsa-max-wait>"
#------------------------------------------------------------------------------
lsaStartIntMin = 0
lsaThrottleMin = 1
lsaThrottleMax = 600000
matcherLsaStart = CliMatcher.IntegerMatcher( lsaStartIntMin, lsaThrottleMax,
      helpdesc='Delay to generate first occurrence of LSA in msecs' )
matcherLsaHold = CliMatcher.IntegerMatcher( lsaThrottleMin, lsaThrottleMax,
      helpdesc='Min delay between originating the same LSA in msecs' )
matcherLsaMaxWait = CliMatcher.IntegerMatcher( lsaThrottleMin, lsaThrottleMax,
      helpdesc='Maximum delay between originating the same LSA in msecs' )

class OspfTimersLsaTxDelayInitialCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa tx delay initial LSA_START LSA_HOLD LSA_MAX_WAIT'
   noOrDefaultSyntax = 'timers lsa tx delay initial ...'
   data = {
          'timers': matcherTimers,
          'lsa': matcherLsa,
          'tx': 'Configure OSPF LSA transmission timers',
          'delay': 'Configure SPF timers',
          'initial': 'Configure SPF timers',
          'LSA_START': matcherLsaStart,
          'LSA_HOLD': matcherLsaHold,
          'LSA_MAX_WAIT': matcherLsaMaxWait,
          }
   handler = RouterOspfMode.setLsaIntervalThrottle
   noOrDefaultHandler = RouterOspfMode.noLsaIntervalThrottle

RouterOspfMode.addCommandClass( OspfTimersLsaTxDelayInitialCmd )

class OspfTimersThrottleLsaAllCmd( CliCommand.CliCommandClass ):
   syntax = 'timers throttle lsa all LSA_START LSA_HOLD LSA_MAX_WAIT'
   noOrDefaultSyntax = 'timers throttle lsa all ...'
   data = {
          'timers': matcherTimers,
          'throttle': 'Configure OSPF throttle timers',
          'lsa': CliCommand.Node( matcherLsa, deprecatedByCmd='timers lsa tx '
             'delay initial' ),
          'all': 'For all type of OSPF LSAs',
          'LSA_START': matcherLsaStart,
          'LSA_HOLD': matcherLsaHold,
          'LSA_MAX_WAIT': matcherLsaMaxWait,
          }
   handler = RouterOspfMode.setLsaIntervalThrottle
   noOrDefaultHandler = RouterOspfMode.noLsaIntervalThrottle

RouterOspfMode.addCommandClass( OspfTimersThrottleLsaAllCmd )
#------------------------------------------------------------------------------
# "[no|default] timers pacing flood <milliseconds>"
#------------------------------------------------------------------------------
floodPacingMin = 5
floodPacingMax = 100
matcherFloodPacing = CliMatcher.IntegerMatcher( floodPacingMin, floodPacingMax,
                                        helpdesc='milliseconds' )
class OspfTimersPacingFloodCmd( CliCommand.CliCommandClass ):
   syntax = 'timers pacing flood MILLISECONDS'
   noOrDefaultSyntax = 'timers pacing flood ...'
   data = {
          'timers': matcherTimers,
          'pacing': 'Configure OSPF packet pacing',
          'flood': 'Configure OSPF flood pacing',
          'MILLISECONDS': matcherFloodPacing,
          }
   handler = RouterOspfMode.setFloodPacing
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfTimersPacingFloodCmd )
#-------------------------------------------------------------------------------
# "[no|default] timers out-delay <lsa out-delay>
#-------------------------------------------------------------------------------
outDelayMin = 0
outDelayMax = 65000
matcherLsaOutDelay = CliMatcher.IntegerMatcher( outDelayMin, outDelayMax,
             helpdesc='Delay to flood router LSA in msecs' )
outDelayKw = CliMatcher.KeywordMatcher( 'out-delay',
                                        helpdesc='Configure out-delay timer' )

class OspfTimersOutDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers out-delay LSA_OUT_DELAY'
   noOrDefaultSyntax = 'timers out-delay ...'
   data = {
          'timers': matcherTimers,
          'out-delay': outDelayKw,
          'LSA_OUT_DELAY': matcherLsaOutDelay,
          }
   handler = RouterOspfMode.setOutDelayTimer
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfTimersOutDelayCmd )
#-------------------------------------------------------------------------------
# "[no|default] passive-interface <interface>"
#-------------------------------------------------------------------------------

def resetIntfsPassiveness( instanceConfig ):
   # A "[no] passive-interface default" wipes out all intf-specific
   # "passive-intface <>" statements. This is to follow industry
   # standard. We set all intf passive flag to default so that
   # the config entry is deleted in gated and inherits from
   # instance config. Currently we don't differentiate intf based
   # on OSPF instance, since we only support one.
   config = ospfConfig()
   for name in config.intfConfig:
      intfConfig = config.intfConfig[name]
      del intfConfig.passiveInstances[ instanceConfig.instance ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( config, name )

def setPassiveIntf( mode, args ):
   config = ospfConfig()
   instanceConfig = config.instanceConfig.get( mode.instanceId, mode.vrfName )
   if 'default' in args:
      instanceConfig.passiveByDefault = True
      resetIntfsPassiveness( instanceConfig )
      return
   passiveByDefault = instanceConfig.passiveByDefault
   for intfName in args[ 'INTFS' ].intfNames():
      # The passive-interface command, which is under the router-ospf
      # mode, doesn't map directly to the gated config model. In gated, the passive
      # configuration is supported in the floating interface configuration
      # and the area interface configuration. Since, we are supporting only
      # one instance, we set the passive configuration in the floating interface
      # configuration, to workaround this mismatch.

      # We set intfConfig.passive to True only if global is not passive by
      # default. Otherwise the intf already inherits global setting.
      if passiveByDefault and not intfName in config.intfConfig:
         continue
      intfConfig = _getOrCreateIntfConfig( config, intfName )
      if passiveByDefault:
         del intfConfig.passiveInstances[ instanceConfig.instance ]
         _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName )
      else:
         intfConfig.passiveInstances[ instanceConfig.instance ] = 'ospfIntfPassive'

# Sets the passive interface state on the passed-in list of
# interfaces (which are either IntfList or Intfs to the default
# value. This is called from CLI context via
# "no passive-interface <intf-names>" command and also to reset
# the state when the OSPF instance is deleted.

def noPassiveIntf( mode, args ):
   config = ospfConfig()

   instanceConfig = config.instanceConfig.get( mode.instanceId, mode.vrfName )
   if 'default' in args:
      instanceConfig.passiveByDefault = False
      resetIntfsPassiveness( instanceConfig )
      return

   # Current we treat "default" the same as "no" to conform to industry
   # standard. But it'd be more reasonable if "default" just means inherit
   # global setting.
   passiveByDefault = instanceConfig.passiveByDefault
   intfs = args[ 'INTFS' ].intfNames()
   for name in intfs:
      if not passiveByDefault and not name in config.intfConfig:
         continue
      intfConfig = _getOrCreateIntfConfig( config, name )
      if not passiveByDefault:
         del intfConfig.passiveInstances[ instanceConfig.instance ]
         _deleteIntfConfigIfAllAttributeHaveDefaults( config, name )
      else:
         intfConfig.passiveInstances[ instanceConfig.instance ] = 'ospfIntfActive'

matcherPassiveIntf = CliMatcher.KeywordMatcher( 'passive-interface',
                     helpdesc='Include interface but without actively running OSPF' )

matcherPassiveDefault = CliMatcher.KeywordMatcher( 'default', 
                        helpdesc='Set all interfaces to passive by default' )

class OspfPassiveIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'passive-interface ( INTFS | default )'
   noOrDefaultSyntax = syntax
   data = { 'passive-interface': matcherPassiveIntf,
            'INTFS': IntfRange.IntfRangeMatcher(),
            'default': matcherPassiveDefault,
          }

   handler = setPassiveIntf
   noOrDefaultHandler = noPassiveIntf

RouterOspfMode.addCommandClass( OspfPassiveIntfCmd )

#------------------------------------------------------------------------------
# "[no|default] auto-cost reference-bandwidth <bandwidth>"
#------------------------------------------------------------------------------
refBwMin = 1
refBwMax = 4294967
refBwMatcher = CliMatcher.IntegerMatcher( refBwMin, refBwMax,
                                 helpdesc='Reference bandwidth in megabits per sec' )
autoCostKw = CliMatcher.KeywordMatcher( 'auto-cost',
                                        helpdesc='Set auto-cost' )
refBwKw = CliMatcher.KeywordMatcher( 'reference-bandwidth',
                                     helpdesc='Set auto-cost' )

class OspfAutoCostRefBwCmd( CliCommand.CliCommandClass ):
   syntax = 'auto-cost reference-bandwidth BANDWIDTH'
   noOrDefaultSyntax = 'auto-cost reference-bandwidth ...'
   data = { 'auto-cost': autoCostKw,
            'reference-bandwidth': refBwKw,
            'BANDWIDTH': refBwMatcher
          }

   handler = RouterOspfMode.setAutoCost
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfAutoCostRefBwCmd )

#-----------------------------------------------------------------------------
# "[no|default] retransmission-threshold lsa <threshold>"
#-----------------------------------------------------------------------------
rangeThresholdMin = 0
rangeThresholdMax = 65535

class OspfRetransmissionThresholdLsaCmd( CliCommand.CliCommandClass ):
   syntax = 'retransmission-threshold lsa THRESHOLD'
   noOrDefaultSyntax = 'retransmission-threshold lsa ...'
   data = { 'retransmission-threshold': 'Configure threshold for retransmission',
            'lsa': 'Configure threshold for retransmission of LSA',
            'THRESHOLD': CliMatcher.IntegerMatcher( 
                         rangeThresholdMin, rangeThresholdMax, 
                         helpdesc='Range for LSA retransmission threshold' ),
   }

   handler = RouterOspfMode.setLsaRetransmissionThreshold
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfRetransmissionThresholdLsaCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd default"
# Legacy:
# "[no|default] bfd all-interfaces"
#------------------------------------------------------------------------------
bfdKw = 'Enable BFD'
bfdDefaultKw = 'Enable BFD on all interfaces'
allInterfacesKw = CliMatcher.KeywordMatcher( 'all-interfaces',
                                     helpdesc='Enable BFD on all interfaces' )

class BfdDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd ( default | all-interfaces )'
   noOrDefaultSyntax = syntax
   data = {
          'bfd': bfdKw,
          'default': bfdDefaultKw,
          'all-interfaces': CliCommand.Node( allInterfacesKw,
                                             deprecatedByCmd='bfd default' ),
          }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.bfdEnabledState = not no

   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( BfdDefaultCmd )

modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

lpbkmodelet = LoopbackIntfCli.LoopbackIntfConfigModelet

#------------------------------------------------------------------------------
# "[no|default] bfd adjacency state any"
#
#------------------------------------------------------------------------------

class BfdAnyStateCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd adjacency state any'
   noOrDefaultSyntax = syntax
   data = {
          'bfd': bfdKw,
          'adjacency' : 'OSPF adjacency BFD configuration',
          'state' : 'OSPF adjacency state',
          'any' : 'Any OSPF adjacency state'
          }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.bfdAnyState = not no

   noOrDefaultHandler = handler

if Toggles.OspfToggleLib.toggleBfdForAllStatesEnabled():
   RouterOspfMode.addCommandClass( BfdAnyStateCmd )

class UnnumberedHelloMaskTxCmd( CliCommand.CliCommandClass ):
   syntax = 'interface unnumbered hello mask tx 0.0.0.0'
   noOrDefaultSyntax = syntax
   data = {
          'interface': 'OSPF interface configuration',
          'unnumbered' : 'Unnumbered interface',
          'hello' : 'Hello packet',
          'mask' : 'Subnet mask',
          'tx' : 'Transmit',
          '0.0.0.0' : 'Subnet mask value 0.0.0.0'
          }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.unnumberedHelloMaskTx = not no

   noOrDefaultHandler = handler

if Toggles.OspfToggleLib.toggleOspfUnnumberedHelloMaskTxEnabled():
   RouterOspfMode.addCommandClass( UnnumberedHelloMaskTxCmd )


#-------------------------------------------------------------------------------
# "[no|default] compatible rfc1583" command, in ospf mode
#-------------------------------------------------------------------------------

matcherCompat = CliMatcher.KeywordMatcher( 'compatible',
                                           helpdesc='Set compatibility' )
matcherRfc1583 = CliMatcher.KeywordMatcher( 'rfc1583',
                                            helpdesc='Compatible with RFC 1583' )

class CompatibleRfc1583Cmd( CliCommand.CliCommandClass ):
   syntax = 'compatible rfc1583'
   noOrDefaultSyntax = syntax
   data = {
          'compatible': matcherCompat,
          'rfc1583': matcherRfc1583,
          }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.rfc1583Compat = not no
   
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( CompatibleRfc1583Cmd )
#-------------------------------------------------------------------------------
# IntfConfig for ospf configuration, is created, when one of its attributes
# is configured, and deleted, when all its attributes go back to defaults.
#-------------------------------------------------------------------------------

def _getIntfConfig( intfName ):
   return ospfConfig().intfConfig.get( intfName, None )
            
def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig
   
def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.cost == intfConfig.costDefault ) and
        ( intfConfig.rxInt == intfConfig.rxIntDefault ) and
        ( intfConfig.transDelay == intfConfig.transDelayDefault ) and
        ( intfConfig.priority == intfConfig.priorityDefault ) and
        ( intfConfig.helloInterval == intfConfig.helloIntervalDefault ) and
        ( intfConfig.routerDeadInterval ==
          intfConfig.routerDeadIntervalDefault ) and
        ( intfConfig.pollInterval == intfConfig.pollIntervalDefault ) and
        ( intfConfig.advertiseSubnet == intfConfig.advertiseSubnetDefault ) and
        ( not intfConfig.passiveInstances ) and
        ( intfConfig.enable == intfConfig.enableDefault ) and
        ( intfConfig.authType == intfConfig.authTypeDefault ) and
        ( intfConfig.simpleAuth == '' ) and
        ( not intfConfig.mdAuth ) and
        ( intfConfig.bfdIntfState == intfConfig.bfdIntfStateDefault ) and
        ( intfConfig.mtuIgnore == intfConfig.mtuIgnoreDefault ) and
        ( intfConfig.intfType == intfConfig.intfTypeDefault ) and 
        ( intfConfig.areaIdPresent == intfConfig.areaIdPresentDefault ) ):
      del config.intfConfig[ intfName ]

#-------------------------------------------------------------------------------
# "[no|default] ip ospf area <area-id>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

def setIntfArea( mode, args ):
   areaId = args[ 'AREA_ID' ]
   areaId = str( Arnet.IpAddress( areaId ) )
   config = ospfConfig()

   vrfName = _getIntfVrf( mode.intf.name )

   intfConfig = _getOrCreateIntfConfig( config, mode.intf.name )
   prevAreaId = str( intfConfig.areaId )
   intfConfig.areaId = areaId
   intfConfig.areaIdPresent = True

   instancesInVrf = nInstancesInVrf( config.instanceConfig, vrfName )
   if instancesInVrf > 1:
      mode.addWarning( "Multiple OSPFv2 instances found in vrf %s. "
                       "The interface area config will not be applied." % vrfName )
   elif instancesInVrf == 1:
      instanceId = vrfNameToInstance( config.instanceConfig, vrf=vrfName )[ 0 ]

      # Verify if the previous area ID is not in use elsewhere, and delete
      if prevAreaId:
         _deleteIntfAreaConfigIfAllAttributeHaveDefaults( instanceId, prevAreaId,
                                                          vrfName)
      instanceConfig = config.instanceConfig.get( instanceId, vrfName )

      # Add the new areaId to the instance config
      # We should initialize the area only if the instance has been
      # created.
      if instanceConfig:
         # Also create the area config
         if areaId not in instanceConfig.areaConfig.keys():
            instanceConfig.areaConfig.newMember( areaId )

def noIntfArea( mode, args ):
   areaId = args[ 'AREA_ID' ]
   areaId = str( Arnet.IpAddress( areaId ) )
   config = ospfConfig()
   vrfName = _getIntfVrf( mode.intf.name )
   intfConfig = _getIntfConfig( mode.intf.name )

   if intfConfig:
      if intfConfig.areaId != areaId:
         mode.addError( "Interface is not a part of OSPF area %s" % str(areaId) )
         return
      intfConfig.areaIdPresent = False
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                mode.intf.name )
   if nInstancesInVrf( config.instanceConfig, vrfName ) > 0 :
      instanceId = vrfNameToInstance( config.instanceConfig, vrf=vrfName )[ 0 ]
      _deleteIntfAreaConfigIfAllAttributeHaveDefaults( instanceId, areaId, vrfName )

class IpOspfAreaCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf area AREA_ID'
   noOrDefaultSyntax = 'ip ospf area AREA_ID ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'area': areaKw,
            'AREA_ID': AreaIdExpression,
          }
   handler = setIntfArea
   noOrDefaultHandler = noIntfArea

modelet.addCommandClass( IpOspfAreaCmd )
lpbkmodelet.addCommandClass( IpOspfAreaCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf authentication [message-digest [ md5 ]]"
# command, in config-if mode.
#-------------------------------------------------------------------------------
#Hidden md5 keyword to maintain backward compatibility
md5Kw = CliMatcher.KeywordMatcher( 'md5', helpdesc='Configure MD5 authentication' )

class IntfAuthentication( CliCommand.CliCommandClass ):
   syntax = """ip ospf authentication [ message-digest [ md5 ] ] """
   noOrDefaultSyntax = syntax
   data = {
         'ip': CliToken.Ip.ipMatcherForConfigIf,
         'ospf': ospfKw,
         'authentication': authenticationKw,
         'message-digest': messageDigestKw,
         'md5' : CliCommand.Node( md5Kw, hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      if 'message-digest' in args:
         intfConfig.authType = 'ospfAuthTypeMd'
      else:
         intfConfig.authType = 'ospfAuthTypeSimple'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.authType = intfConfig.authTypeDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfAuthentication )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf authentication-key <password>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfAuthPasswd( mode, args ):
   if 'PLAIN' not in args:
      try:
         passwd = decodeKey( args[ 'AUTHPSWD' ],
                             key=mode.intf.name + '_passwd',
                             algorithm="MD5" ) 
      except: # pylint: disable=bare-except
         mode.addError( "Invalid encrypted password" )
         return
      else:
         pass
   else:
      passwd = args[ 'PLAIN' ] 

   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   intfConfig.simpleAuth = passwd
   
def noIntfAuthPasswd( mode, args ):
   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   intfConfig.simpleAuth = ''
   _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                mode.intf.name )

cleartextMatcher = CliMatcher.PatternMatcher( authPasswdPattern, helpname='WORD',
                                              helpdesc='PASSWORD (up to 8 chars)' )
type7key = CliCommand.Node( matcher=CliMatcher.PatternMatcher( 
                                       r'.+', helpname='WORD',
                                       helpdesc='ENCRYPTED password' ),
                                       sensitive=True )

class OspfAuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf authentication-key AUTHPSWD'           
   noOrDefaultSyntax = 'ip ospf authentication-key ...'
   data = { 
         'ip': CliToken.Ip.ipMatcherForConfigIf,
         'ospf': ospfKw,
         'authentication-key': 'Configure authentication password',
         'AUTHPSWD': reversibleSecretCliExpression( 'AUTHPSWD',
                               cleartextMatcher=cleartextMatcher, 
                               obfuscatedTextMatcher=type7key,
                               returnifCleartext=True ),
          }
   handler = setIntfAuthPasswd
   noOrDefaultHandler = noIntfAuthPasswd

modelet.addCommandClass( OspfAuthenticationKeyCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf cost <interface-cost>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
matcherCost = CliMatcher.IntegerMatcher( metricMin, metricMax,
                                         helpdesc='Value of the route metric' )

class OspfCostCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf cost COST'
   noOrDefaultSyntax = 'ip ospf cost ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'cost': costKw,
            'COST': matcherCost,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.cost = args[ 'COST' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.cost = intfConfig.costDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

lpbkmodelet.addCommandClass( OspfCostCmd )

modelet.addCommandClass( OspfCostCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf neighbor bfd" command, in config-if mode.
# Legacy:
# "[no|default] ip ospf bfd" command, in config-if mode.
#-------------------------------------------------------------------------------

def setIntfBfd( mode, args ):
   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   intfConfig.bfdIntfState = 'ospfIntfBfdEnabled'

def noIntfBfd( mode, args ):
   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   if CliCommand.isDefaultCmd( args ):
      intfConfig.bfdIntfState = intfConfig.bfdIntfStateDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )
   else:
      intfConfig.bfdIntfState = 'ospfIntfBfdDisabled'

matcherBfd = CliMatcher.KeywordMatcher( 'bfd', helpdesc='Enable BFD' )

class OspfNeighborBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf neighbor bfd'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'neighbor': 'Set behavior for neighbors',
            'bfd': matcherBfd,
          }

   handler = setIntfBfd
   noOrDefaultHandler = noIntfBfd

modelet.addCommandClass( OspfNeighborBfdCmd )

class OspfBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf bfd'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'bfd': CliCommand.Node( matcherBfd,
                                    deprecatedByCmd='ip ospf neighbor bfd' ),
          }

   handler = setIntfBfd
   noOrDefaultHandler = noIntfBfd

modelet.addCommandClass( OspfBfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf dead-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
class OspfDeadInterval( CliCommand.CliCommandClass ):
   syntax = 'ip ospf dead-interval DEAD_INTERVAL'
   noOrDefaultSyntax = 'ip ospf dead-interval ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'dead-interval': deadIntervalKw,
            'DEAD_INTERVAL': matcherIntfIntervalSeconds,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.routerDeadInterval = args[ 'DEAD_INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.routerDeadInterval = intfConfig.routerDeadIntervalDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfDeadInterval )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf hello-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
class OspfHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf hello-interval HELLO_INTERVAL'
   noOrDefaultSyntax = 'ip ospf hello-interval ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'hello-interval': helloIntervalKw,
            'HELLO_INTERVAL': matcherIntfIntervalSeconds,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.helloInterval = args[ 'HELLO_INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.helloInterval = intfConfig.helloIntervalDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfHelloIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf message-digest-key <key-id> <mode> <key>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
MAX_CRYPT_KEYS_PER_INTF = 2

digestsAndHelps = {
      'md5': 'Configure MD5 authentication',
      'sha1': 'Configure SHA1 authentication',
      'sha256': 'Configure SHA256 authentication',
      'sha384':  'Configure SHA384 authentication',
      'sha512':  'Configure SHA512 authentication',
}

cleartextMatcher = CliMatcher.PatternMatcher( authKeyPattern, helpname='LINE',
                                              helpdesc='Auth key (max 16 chars)' )
type7key = CliCommand.Node( matcher=CliMatcher.PatternMatcher( r'[\S]+',
                            helpname='LINE',
                            helpdesc='Auth encrypted key' ),
                            sensitive=True )

class IntfCryptographicKey( CliCommand.CliCommandClass ):
   syntax = 'ip ospf message-digest-key KEY DIGEST MSGDIGESTKEY'
   noOrDefaultSyntax = 'ip ospf message-digest-key KEY ... '
   data = {
         'ip': CliToken.Ip.ipMatcherForConfigIf,
         'ospf': ospfKw,
         'message-digest-key': messageDigestKeyKw,
         'KEY' : CliMatcher.IntegerMatcher( 1, 255,
                    helpdesc='Message-digest authentication key identifier' ),
         'DIGEST': CliMatcher.EnumMatcher( digestsAndHelps ),
         'MSGDIGESTKEY': reversibleSecretCliExpression( 'MSGDIGESTKEY',
                               cleartextMatcher=cleartextMatcher, 
                               obfuscatedTextMatcher=type7key,
                               returnifCleartext=True ),  
}

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )

      # process the key for the validated auth mode
      keyId = args[ 'KEY' ]

      if keyId not in intfConfig.mdAuth:
         if len( intfConfig.mdAuth ) >= MAX_CRYPT_KEYS_PER_INTF:
            mode.addError( "More than %d keys per interface is not supported"
                           % ( MAX_CRYPT_KEYS_PER_INTF ) )
            return

      # validate the key if encrypted
      authAlgorithm = args[ 'DIGEST' ]
      if 'PLAIN' not in args:
         try:
            key = decodeKey( args[ 'MSGDIGESTKEY' ],
                             key=mode.intf.name +
                             '_%sKey_%d' % ( authAlgorithm, keyId ),
                             algorithm="MD5" )
         except: # pylint: disable=bare-except
            mode.addError( "Invalid encrypted key" )
            return

      # Save the auth info
      else:
         key = args[ 'PLAIN' ] 
      
      intfConfig.mdAuth[ keyId ] = Tac.Value( 'Routing::Ospf::OspfMdAuth', 
                                              key, authAlgorithm )
      intfConfig.latestKey = keyId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )

      keyId = args[ 'KEY' ]
      del intfConfig.mdAuth[ keyId ]
      if not intfConfig.mdAuth:
         intfConfig.latestKey = intfConfig.latestKeyDefault
      elif intfConfig.latestKey == keyId:
         intfConfig.latestKey = sorted( intfConfig.mdAuth.keys() )[ 0 ]

      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfCryptographicKey )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf mtu-ignore"
# command, in config-if mode.
#-------------------------------------------------------------------------------

mtuIgnoreKw = CliMatcher.KeywordMatcher( 'mtu-ignore', 
      'Disable MTU check for Database Description packets' )

class OspfMtuIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf mtu-ignore'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'mtu-ignore': mtuIgnoreKw,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.mtuIgnore = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.mtuIgnore = False
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfMtuIgnoreCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf priority <number-value>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

priorityMin = 0
priorityMax = 255
priorityKw = CliMatcher.KeywordMatcher( 'priority',
      helpdesc='Interface priority' )
matcherPriority = CliMatcher.IntegerMatcher( priorityMin, priorityMax, 
      helpdesc='Priority' )

class OspfPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf priority PRIORITY'
   noOrDefaultSyntax = 'ip ospf priority ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'priority': priorityKw,
            'PRIORITY': matcherPriority,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.priority = args[ 'PRIORITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.priority = intfConfig.priorityDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfPriorityCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf retransmit-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

class OspfRetransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf retransmit-interval RETRANSMIT_INTERVAL'
   noOrDefaultSyntax = 'ip ospf retransmit-interval ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'retransmit-interval': retransmitIntervalKw,
            'RETRANSMIT_INTERVAL': matcherIntfIntervalSeconds,
          }
   
   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.rxInt = args[ 'RETRANSMIT_INTERVAL' ]
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.rxInt = intfConfig.rxIntDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfRetransmitIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf disabled" command, in config-if mode.
# Legacy:
# "[no|default] ip ospf shutdown" command, in config-if mode.
#-------------------------------------------------------------------------------

def setIntfShutdown( mode, args ):
   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   intfConfig.enable = False

def noIntfShutdown( mode, args ):
   intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
   intfConfig.enable = True
   _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                mode.intf.name )

class OspfShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf shutdown'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'shutdown': CliCommand.Node( CliMatcher.KeywordMatcher( 'shutdown',
                                         helpdesc='Disable the OSPF instance' ),
                                         deprecatedByCmd='ip ospf disabled' ),
          }

   handler = setIntfShutdown
   noOrDefaultHandler = noIntfShutdown

modelet.addCommandClass( OspfShutdownCmd )

class OspfDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf disabled'
   noOrDefaultSyntax = syntax
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'disabled': 'Disable the OSPF instance',
          }

   handler = setIntfShutdown
   noOrDefaultHandler = noIntfShutdown

modelet.addCommandClass( OspfDisableCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf transmit-delay <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------

class OspfTransmitDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf transmit-delay TRANSMIT_DELAY'
   noOrDefaultSyntax = 'ip ospf transmit-delay ...'
   data = {
          'ip': CliToken.Ip.ipMatcherForConfigIf,
          'ospf': ospfKw,
          'transmit-delay': transmitDelayKw,
          'TRANSMIT_DELAY': matcherIntfIntervalSeconds,
          }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.transDelay = args[ 'TRANSMIT_DELAY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.transDelay = intfConfig.transDelayDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfTransmitDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip ospf network point-to-point|point-to-multipoint|nonbroadcast"
# command, in config-if mode.
#-------------------------------------------------------------------------------
networkTypeKw = CliMatcher.KeywordMatcher( 'network',
                                           helpdesc='Configure interface type' )
_networkTypes = { 'point-to-point': 'Point-to-point interface' }
if os.environ.get( 'ENABLE_CMDS_NOT_IN_FIRST_REL' ):
   _networkTypes.update( { 'point-to-multipoint': 'intfTypePointToMultipoint',
                           'nonbroadcast': 'intfTypeNonBroadcast' } )
networkTypesMatcher = CliMatcher.EnumMatcher( _networkTypes )

class OspfNetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ospf network NETWORK_TYPE'
   noOrDefaultSyntax = 'ip ospf network ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'ospf': ospfKw,
            'network': networkTypeKw,
            'NETWORK_TYPE': networkTypesMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      intfType = args[ 'NETWORK_TYPE' ]
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.intfType = { 'point-to-point': 'intfTypePointToPoint',
                              'point-to-multipoint': 'intfTypePointToMultipoint',
                              'nonbroadcast': 'intfTypeNonBroadcast' }[ intfType ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ospfConfig(), mode.intf.name )
      intfConfig.intfType = 'intfTypeBroadcast'
      _deleteIntfConfigIfAllAttributeHaveDefaults( ospfConfig(),
                                                   mode.intf.name )

modelet.addCommandClass( OspfNetworkCmd )

#-------------------------------------------------------------------------------
# "[no|default] default-information originate [ always ][ metric <1-65535> ] [
# metric-type <1-2> ] [ route-map WORD ]
#-------------------------------------------------------------------------------
class OspfDefaultInformationCmdBase( CliCommand.CliCommandClass ):
   syntax = '''default-information originate
               [ { always
                 | ( metric METRIC )
                 | ( metric-type METRIC_TYPE )
                 | ( route-map MAPNAME ) } ]'''
   noOrDefaultSyntax = 'default-information originate ...'

   @classmethod
   def handler( cls, mode, args ):
      # pylint: disable-msg=E1101
      instanceConfig = cls._getInstanceConfig( mode )
      instanceConfig.defInfoOrigin = 'ospfDefInfoOriginEnable'
      if 'always' in args:
         args[ 'always' ] = [ True ]
      argToAttrDict = { 'always' : 'defInfoOriginAlways',
                        'METRIC' : 'defInfoMetric',
                        'METRIC_TYPE' : 'defInfoMetricType',
                        'MAPNAME' : 'defInfoRouteMap' }
      for i in argToAttrDict:
         attr = argToAttrDict[ i ]
         if i in args:
            setattr( instanceConfig, attr, args[ i ][ 0 ] )
         else:
            setattr( instanceConfig, attr, getattr( instanceConfig,
                                                    attr + 'Default' ) )

   @staticmethod
   def _resetDefInfoOriginParams( instanceConfig ):
      instanceConfig.defInfoOriginAlways = \
            instanceConfig.defInfoOriginAlwaysDefault
      instanceConfig.defInfoMetric = instanceConfig.defInfoMetricDefault
      instanceConfig.defInfoMetricType = \
            instanceConfig.defInfoMetricTypeDefault
      instanceConfig.defInfoRouteMap = \
            instanceConfig.defInfoRouteMapDefault

   @classmethod
   def noHandler( cls, mode, args ):
      # pylint: disable-msg=E1101
      instanceConfig = cls._getInstanceConfig( mode )
      instanceConfig.defInfoOrigin = 'ospfDefInfoOriginSuppress'
      cls._resetDefInfoOriginParams( instanceConfig )

   @classmethod
   def defaultHandler( cls, mode, args ):
      # pylint: disable-msg=E1101
      instanceConfig = cls._getInstanceConfig( mode )
      instanceConfig.defInfoOrigin = 'ospfDefInfoOriginDefault'
      cls._resetDefInfoOriginParams( instanceConfig )

class OspfDefaultInformationCmd( OspfDefaultInformationCmdBase ):
   data = { 'default-information' : 'Default route related configuration',
            'originate' : 'Source a default route',
            'always' : CliCommand.singleKeyword( 'always',
                           helpdesc='Advertise a default route unconditionally' ),
            'metric' : CliCommand.singleKeyword( 'metric',
                           helpdesc='Metric for default route' ),
            'METRIC' : CliMatcher.IntegerMatcher( metricMin, metricMax,
                           helpdesc='Value of the route metric' ),
            'metric-type' : CliCommand.singleKeyword( 'metric-type',
                           helpdesc='Metric type for default route' ),
            'METRIC_TYPE' : CliMatcher.IntegerMatcher( 1, 2,
                           helpdesc='Metric type' ),
            'route-map' : CliCommand.singleNode(
                              RouteMapMatchers.routeMapApplication ),
            'MAPNAME' : mapNameMatcher
   }

   @staticmethod
   def _getInstanceConfig( mode ):
      return mode._getInstanceConfig()

RouterOspfMode.addCommandClass( OspfDefaultInformationCmd )

#-------------------------------------------------------------------------------
# [no|default] summary-address prefix/len [not-advertise|tag <val>|attribute-map
# NAME]
#-------------------------------------------------------------------------------
class OspfSummaryAddressCmd( CliCommand.CliCommandClass ):
   syntax = '''summary-address PREFIX [ not-advertise
                                      | ( tag TAG )
                                      | ( attribute-map MAPNAME ) ]'''
   noOrDefaultSyntax = 'summary-address PREFIX ...'
   data = { 'summary-address' : 'Summary route configuration',
            'PREFIX' : IpAddrMatcher.ipPrefixExpr( 'IP summary address',
                                             'Summary Mask', 'Prefix',
                                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                                       inverseMask=False ),
            'not-advertise' : 'Do not advertise summary route',
            'tag' : 'Set tag',
            'TAG' : CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                               helpdesc='Number' ),
            'attribute-map' : 'Set attributes of summary route',
            'MAPNAME' : mapNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      if args[ 'PREFIX' ] == '0.0.0.0/0':
         mode.addError( 'Summary address cannot be configured for 0.0.0.0/0' )
         return
      prefix = Arnet.Prefix( args[ 'PREFIX' ] )
      if prefix in instanceConfig.summaryAddrConfig:
         del instanceConfig.summaryAddrConfig[ prefix ]
      if 'not-advertise' in args:
         summAddr = Tac.Value( 'Routing::Ospf::SummaryAddress', prefix,
                               notAdvertise=True )
      elif 'tag' in args:
         summAddr = Tac.Value( 'Routing::Ospf::SummaryAddress', prefix,
                               tag=args[ 'TAG' ] )
      elif 'attribute-map' in args:
         summAddr = Tac.Value( 'Routing::Ospf::SummaryAddress', prefix,
                               attrMap=args[ 'MAPNAME' ] )
      else:
         summAddr = Tac.Value( 'Routing::Ospf::SummaryAddress', prefix )

      instanceConfig.summaryAddrConfig.addMember( summAddr )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      prefix = Arnet.Prefix( args[ 'PREFIX' ] )
      del instanceConfig.summaryAddrConfig[ prefix ]

RouterOspfMode.addCommandClass( OspfSummaryAddressCmd )

#-------------------------------------------------------------------------------
#
# ip access-group <acl-list-name> in
# [no|default] ip access-group [<acl-list-name>]
#
# command in router ospf mode.
#-------------------------------------------------------------------------------

class IpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_LIST_NAME [ IN ]'
   noOrDefaultSyntax = 'ip access-group [ ACL_LIST_NAME ]'
   data = {
          'ip': AclCli.ipKwForServiceAclMatcher,
          'access-group': AclCli.accessGroupKwMatcher,
          'ACL_LIST_NAME': AclCli.ipAclNameMatcher,
          'IN': AclCli.inKwMatcher,
          }
   handler = RouterOspfMode.setServiceAcl
   noOrDefaultHandler = RouterOspfMode.noServiceAcl

RouterOspfMode.addCommandClass( IpAccessGroupCmd )
#-------------------------------------------------------------------------------
# [no|default] ecmp intra-area route multi-area next-hops
#
# command in router ospf mode
#-------------------------------------------------------------------------------
class OspfEcmpIntraAreaConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'ecmp intra-area route multi-area next-hops ...'
   noOrDefaultSyntax = syntax
   data = {
      'ecmp': "Configure ECMP parameters",
      'intra-area': "Routes within an area",
      'route': "Configure route parameters",
      'multi-area': "Configure multi-area parameters",
      'next-hops': "Enable next-hops from multiple areas for intra-area ECMP routes",
   }
   handler = RouterOspfMode.setIntraAreaRouteEcmp
   noOrDefaultHandler = handler

RouterOspfMode.addCommandClass( OspfEcmpIntraAreaConfigCmd )

#-------------------------------------------------------------------------------
#
# "[no|default] traffic-engineering" command in router ospf mode.
#
#-------------------------------------------------------------------------------
class OspfTrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering'
   noOrDefaultSyntax = syntax
   data = {
           'traffic-engineering': CliCommand.guardedKeyword( 'traffic-engineering',
              helpdesc='Enter traffic engineering config mode',
              guard=ospfNonDefaultVrfGuard ),
          }
   handler = RouterOspfMode.gotoRouterOspfTeMode
   noOrDefaultHandler = RouterOspfMode.delRouterOspfTeMode

RouterOspfMode.addCommandClass( OspfTrafficEngineeringCmd )

# TE CLI
#----------------------------------------------------
#"[no | default] shutdown" command, in ospf - te mode
#----------------------------------------------------

class OspfTeShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = { 
           'shutdown': 'Traffic Engineering state in OSPF',
          }
   handler = RouterOspfTeMode.setShutdown
   noOrDefaultHandler = handler

RouterOspfTeMode.addCommandClass( OspfTeShutdownCmd )

#----------------------------------------------------
#"[no | default] area ( <area-ids> | all )
#----------------------------------------------------

class OspfTeAreaAllCmd( CliCommand.CliCommandClass ):
   syntax = 'area ( { AREA_ID } | all )'
   noOrDefaultSyntax = syntax
   data = {
            'area': areaKw,
            'AREA_ID': AreaIdExpression,
            'all': 'All areas'
          }
   handler = RouterOspfTeMode.setArea
   noOrDefaultHandler = handler

RouterOspfTeMode.addCommandClass( OspfTeAreaAllCmd )

#-------------------------------------------------------------------------------
# Segment-Routing specific CLI
#-------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# [no|default] segment-routing mpls
#---------------------------------------------------------------------------------
mplsKw = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'mpls',
                                    helpdesc='segment routing in MPLS dataplane' ),
                          guard=ospfNonDefaultVrfGuard )

class OspfSegmentRoutingMplsCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing mpls'
   noOrDefaultSyntax = syntax
   data = {
         'segment-routing'      : 'Enter segment routing config mode',
         'mpls'                 : mplsKw,
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      instanceConfig.srConfig = ( 'srConf', )
      instanceConfig.srConfig.dataPlane = srDataPlaneEnum.srDataPlaneMpls
      childMode = mode.childMode( RouterOspfSrMplsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = mode._getInstanceConfig()
      # reset the srConfig object to None
      instanceConfig.srConfig = None

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   RouterOspfMode.addCommandClass( OspfSegmentRoutingMplsCmd )

#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
class OspfSrShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
         'shutdown' : 'Disable segment routing',
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = mode.parent_._getInstanceConfig()
      instanceConfig.srConfig.shutdown = \
            instanceConfig.srConfig.shutdownDefault

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      instanceConfig = mode.parent_._getInstanceConfig()
      instanceConfig.srConfig.shutdown = False
      
RouterOspfSrMplsMode.addCommandClass( OspfSrShutdownCmd )

#---------------------------------------------------------------------------------
# [no|default] prefix-segment <prefix> index <index> 
# command, in 'config-router-ospf-sr-mpls' mode
#---------------------------------------------------------------------------------
def srIndexRangeFn( mode ):
   labelRange = mplsConfig.labelRange.get( 'ospf-sr' )
   if labelRange is None:
      return ( 0, 65535 )
   else:
      return ( 0, labelRange.size - 1 )

srIndexMatcher = CliMatcher.DynamicIntegerMatcher( rangeFn=srIndexRangeFn,
                                                   helpdesc='Index value' )
prefixMatcher = IpAddrMatcher.ipAddrWithFullMaskExpr(
   'IP address', 'Subnet\'s mask value',
   'IPv4 address prefix', overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

class PrefixSegmentCommand( CliCommand.CliCommandClass ):
   syntax = 'prefix-segment PREFIX index INDEX'
   noOrDefaultSyntax = syntax

   data = {
      'prefix-segment' : 'Configure a prefix segment',
      'PREFIX': prefixMatcher,
      'index': 'Set the prefix segment identifier',
      'INDEX': srIndexMatcher
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = ospfConfig().instanceConfig.get( mode.instanceName )
      prefix = args[ 'PREFIX' ]
      index = args[ 'INDEX' ]

      prefix = Arnet.IpGenPrefix( prefix.stringValue )

      if _ospfSrConflictingPrefixFound( instanceConfig, None, prefix ):
         mode.addError( 'prefix segment conflicts with a proxy/node segment'
                        ' configured for the prefix %s' % prefix )
         return
      if _ospfSrConflictingSidFound( instanceConfig, prefix, index ):
         mode.addError( 'Two prefixes cannot be configured with the same index '
                        'value' )
         return

      instanceConfig.srConfig.prefixSegments.addMember(
         Tac.Value( 'Routing::SegmentRoutingCli::PrefixSegment',
                    prefix=prefix,
                    index=index ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = ospfConfig().instanceConfig.get( mode.instanceName )
      prefix = args[ 'PREFIX' ]
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      del instanceConfig.srConfig.prefixSegments[ prefix ]

RouterOspfSrMplsMode.addCommandClass( PrefixSegmentCommand )

#---------------------------------------------------------------------------------
# [no|default] proxy-node-segment <prefix> index <index> 
# command, in 'config-router-ospf-sr-mpls' mode
#---------------------------------------------------------------------------------
class ProxyNodeSegmentCommand( CliCommand.CliCommandClass ):
   syntax = 'proxy-node-segment PREFIX index INDEX'
   noOrDefaultSyntax = syntax

   data = {
         'proxy-node-segment' : 'Configure a node segment on behalf of another node',
         'PREFIX': prefixMatcher,
         'index': 'Set the prefix segment identifier',
         'INDEX': srIndexMatcher
   }

   @staticmethod
   def handler( mode, args ):
      instanceConfig = ospfConfig().instanceConfig.get( mode.instanceName )
      prefix = Arnet.IpGenPrefix( args[ 'PREFIX' ].stringValue )
      index = args[ 'INDEX' ]

      if not prefix.isHost:
         mode.addError( 'A proxy-node segment can be configured for a'
                        ' /32 IPv4 prefix only' )
         return
      if _ospfSrConflictingPrefixFound( instanceConfig, None, prefix, proxy=True ):
         mode.addError( 'proxy node segment conflicts with a prefix/node segment'
                        ' configured for the prefix %s' % prefix )
         return
      if _ospfSrConflictingSidFound( instanceConfig, prefix, index ):
         mode.addError( 'Two prefixes cannot be configured with the same index '
                        'value' )
         return

      instanceConfig.srConfig.prefixSegments.addMember(
         Tac.Value( 'Routing::SegmentRoutingCli::PrefixSegment',
                    prefix=prefix, index=index, isProxyNode=True ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instanceConfig = ospfConfig().instanceConfig.get( mode.instanceName )
      prefix = Arnet.IpGenPrefix( args[ 'PREFIX' ].stringValue )
      del instanceConfig.srConfig.prefixSegments[ prefix ]

RouterOspfSrMplsMode.addCommandClass( ProxyNodeSegmentCommand )

#-------------------------------------------------------------------------------
# show commands
#-------------------------------------------------------------------------------

def ospfStatus():
   return ospfStatusDir

matcherOspfShow = CliMatcher.KeywordMatcher( 'ospf', helpdesc='OSPF information' )

# Performs a reverse DNS lookup for the routerId, when the
# name-lookup is enabled, and DNS server has been configured
def ospfRouterIdToStr( routerId, maxLen = 0 ):
   config = ospfConfig()
   if config.nameLookup:
      try: 
         result = socket.gethostbyaddr( '%s' % routerId )
         routerName = result[ 0 ]
      except: # pylint: disable=bare-except
         routerName = '%s' % routerId
   else:
      routerName = '%s' % routerId

   if maxLen:
      routerName = routerName[ 0 : maxLen ]

   return routerName

# We allow time to stand still for testing.
if 'CLI_NOW_IS' in os.environ:
   cur_time = float( os.environ[ 'CLI_NOW_IS' ] )
   def now():
      return cur_time
else:
   now = Tac.now

def secondsToHMS( seconds, absolute=0 ):
   if absolute:
      if absolute < 0:
         seconds = now() - seconds
      else:
         seconds = seconds - now()
   seconds = int( seconds )
   hours = seconds / 3600
   seconds -= 3600*hours
   minutes = seconds / 60
   seconds -= 60*minutes
   return "%02d:%02d:%02d" % (hours, minutes, seconds)

def secondsToDHMS( seconds, absolute=0 ):
   if absolute:
      if absolute < 0:
         seconds = now() - seconds
      else:
         seconds = seconds - now()
   seconds = int( seconds )
   days = seconds // 86400
   seconds -= 86400 * days
   hours = seconds // 3600
   seconds -= 3600*hours
   minutes = seconds // 60
   seconds -= 60*minutes
   return "%03d:%02d:%02d:%02d" % (days, hours, minutes, seconds)

def checkInstanceToVrfMapping( mode, instanceId, vrfName ):
   instanceConfig = ospfConfig().instanceConfig
   if instanceId and vrfName and \
      ( vrfName == ALL_VRF_NAME or \
         instanceId not in vrfNameToInstance( instanceConfig,
                                              vrf=vrfName ) ):
      mode.addError( 'OSPF instance %d does not exist in vrf %s' % \
               ( instanceId, vrfName ) )
      return False
   return True

def getInstanceIds( mode, instanceId, vrfName ):
   if not checkInstanceToVrfMapping( mode, instanceId, vrfName ):
      return None
   config = ospfConfig()
   instanceConfig = config.instanceConfig
   if instanceId:
      instanceConfig = config.instanceConfig.get( instanceId )
      if instanceConfig is None:
         mode.addError( 'OSPF instance %d does not exist' % instanceId )
         return None
      return [ instanceId ]
   else:
      if not vrfName:
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
         if vrfName is DEFAULT_VRF:
            vrfName = None
      if vrfName and vrfName != ALL_VRF_NAME:
         instanceIds = vrfNameToInstance( config.instanceConfig,
                                          vrf=vrfName )
         # Ospf instance is not configured for this VRF.
         if not instanceIds:
            mode.addError( 'OSPF instance does not exist in vrf %s' % vrfName )
            return None
      else:
         instanceIds = config.instanceConfig.keys()
      return sorted( instanceIds )

def insertVrArgInCmd( mode, verb, cmd='show ip ospf', instanceId=None,
                      noInstanceIdCliribd=False, config=None ):
   vrfName = None
   if not config:
      config = ospfConfig()
   # If instanceId is explicitly specified, honour that arguement
   if not instanceId and nInstancesInVrf( config.instanceConfig,
                                    DEFAULT_VRF ) == 0 and config.instanceConfig:
      cmd += " %s" % verb
      vrfName = config.instanceConfig.values()[0].vrfName
   else:
      if instanceId is None:
         _vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
         if _vrfName:
            instanceId = vrfNameToInstance( config.instanceConfig, 
                                            vrf=_vrfName )[ 0 ]
      if instanceId:
         c = config.instanceConfig.get( instanceId )
         if c is None:
            cmd += " %s" % verb
            return ( DEFAULT_VRF, cmd )
         c = config.instanceConfig[ instanceId ]
         vrfName = c.vrfName
         if noInstanceIdCliribd:
            cmd += " %s" % verb
         else:
            cmd += " %s %s" % ( verb, instanceId )
      else:
         vrfName = DEFAULT_VRF
         cmd += " %s" % verb
   return ( vrfName, cmd )

def getDefaultInstanceModel( instModel, instanceId=None ):
   # Default model(which includes MaxLsa info) is created and
   # returned when the instance is shutdown from CLI or not
   # active on gated
   instModel.instanceId = instanceId
   instConfig = ospfConfig().instanceConfig.get( instanceId )
   status = ospfStatus()

   if instConfig:
      instModel.vrf = instConfig.vrfName
      if not instConfig.enable:
         instModel.shutDown = True
         statusVrf = status.get( instConfig.vrfName )
         if statusVrf is not None:
            instanceStatus = statusVrf.instanceStatus.get( instanceId )
            if instanceStatus is not None  and instanceStatus.routerId:
               instModel.routerId = instanceStatus.routerId
      maxLsaInfo = OspfCliModels.OspfShowInstMaxLsaInfoModel()
      maxLsaInfo.processModel( { 'instanceId': instanceId } )
      instModel.maxLsaInformation = maxLsaInfo

   return instModel

def getDefaultModel( model, instanceId=None ):
   # Default model is created and
   # returned when the instance is shutdown from CLI or not
   # active on gated
   if hasattr( model, "_instanceId" ):
      model._instanceId = instanceId
   elif  hasattr( model, "instanceId" ):
      model.instanceId = instanceId
   instConfig = ospfConfig().instanceConfig.get( instanceId )
   status = ospfStatus()
   if instConfig:
      if hasattr( model, "vrf" ):
         model.vrf = instConfig.vrfName
      elif hasattr( model, "_vrf" ):
         model._vrf = instConfig.vrfName
      if not instConfig.enable and hasattr( model, "shutDown" ):
         model.shutDown = True
      statusVrf = status.get( instConfig.vrfName )
      if statusVrf is not None:
         instanceStatus = statusVrf.instanceStatus.get( instanceId )
         if instanceStatus and instanceStatus.routerId:
            if hasattr( model, "routerId" ):
               model.routerId = instanceStatus.routerId
            elif hasattr( model, "_routerId" ):
               model._routerId = instanceStatus.routerId
   return model

#-------------------------------------------------------------------------------
# "show ip ospf [instance-id] [vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
OspfShowInstDictModel = generateOspfInstListCliModel(
                        OspfCliModels.OspfShowInstModel )

OspfShowInstVrfDictModel = generateVrfCliModel( OspfShowInstDictModel,
                           'OSPF instance information for all VRFs')
@OspfVrfModelDecorator( vrfModelListType=OspfShowInstVrfDictModel, 
                        instModelListType=OspfShowInstDictModel )
def showIpOspf( mode, args ):
   command = 'MIO_DGET_OSPF_INSTANCE_SUMMARY'
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   argsCapi = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, OspfCliModels.OspfShowInstModel,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      return getDefaultInstanceModel( OspfCliModels.OspfShowInstModel(),
                                      instanceId )

class ShowIpOspfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'VRF': vrfExprFactory,
          }
   handler = showIpOspf
   cliModel = OspfShowInstVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfCmd )

#-------------------------------------------------------------------------------
# "show ip ospf [instance-id] summary [vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
OspfShowSummaryDictModel = generateOspfInstListCliModel(
                        OspfCliModels.OspfShowSummaryModel )

OspfShowSummaryVrfDictModel = generateVrfCliModel( OspfShowSummaryDictModel,
                           'OSPF instance information for all VRFs' )
@OspfVrfModelDecorator( vrfModelListType=OspfShowSummaryVrfDictModel, 
                        instModelListType=OspfShowSummaryDictModel )
def showIpOspfSummary( mode, args ):
   model = OspfCliModels.OspfShowSummaryModel()

   # ----------------------------------------------------------------------------
   # Populate summary of instance
   # ----------------------------------------------------------------------------
   command = 'MIO_DGET_OSPF_INSTANCE_SUMMARY'
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   argsCapi = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      model.inst = showRibCapiCommand( mode, model._OspfShowInstModelAcc, command, 
                                       argsCapi, clientName='OSPF', 
                                       l3Config=l3Config )
   except EmptyResponseException:

      model.inst = getDefaultInstanceModel( model._OspfShowInstModelAcc(),
                                            instanceId )
      # If we have no OSPF instance, we cannot have neighbors or areas, and so simply
      # return the empty instance.
      return model

   # ----------------------------------------------------------------------------
   # Populate neighbors
   # ----------------------------------------------------------------------------
   command = 'MIO_DGET_OSPF_NEIGHBORS'
   argsCapi = { 'instance': instanceId }   
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      model.neighbors = showRibCapiCommand( mode, model._OspfNeighborsInstanceAcc,
                                            command, argsCapi, clientName='OSPF',
                                            l3Config=l3Config )
   except EmptyResponseException:
      model.neighbors = model._OspfNeighborsInstanceAcc()

   # ----------------------------------------------------------------------------
   # Populate database summaries for each area
   # ----------------------------------------------------------------------------
   def databaseForArea( areaId ):
      command = 'MIO_DGET_OSPF_DATABASE'
      argsCapi = { 'instance': instanceId, 'area': areaId, 'databaseSum': True }
      if vrfName != DEFAULT_VRF:
         argsCapi[ 'vrfName' ] = vrfName

      try:
         result = showRibCapiCommand( mode, OspfCliModels.OspfDatabaseSummary, 
                                      command, argsCapi, clientName='OSPF',
                                      l3Config=l3Config )
      except EmptyResponseException:
         result = OspfCliModels.OspfDatabaseSummary()

      return result

   for areaId in model.inst.areaList:
      model.lsasByArea[ areaId ] = databaseForArea( areaId )

   return model

class ShowIpOspfSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] summary [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'summary': 'OSPF Summary',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfSummary
   cliModel = OspfShowSummaryVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfSummaryCmd )

#-------------------------------------------------------------------------------
# "show ip ospf [<instance-id>]
#               interface [ {<interface-type> <interface-number>} |
#                           brief ]
#               [ vrf vrf-name|default|all ]"
#-------------------------------------------------------------------------------
OspfShowInterfaceDictModel = generateOspfInstListCliModel( OspfCliModels.\
                                                           OspfInterface )

OspfShowInterfaceVrfDictModel = generateVrfCliModel( OspfShowInterfaceDictModel,
                                                     'OSPF instance information '
                                                     'for all VRFs')

@OspfVrfModelDecorator( vrfModelListType=OspfShowInterfaceVrfDictModel, 
                        instModelListType=OspfShowInterfaceDictModel )
def showIpOspfIntf( mode, args ):
   intfParam = args.get( 'INTERFACE' ) or args.get( 'brief' )
   vrfName = args.get( 'VRF' )
   cmd = 'MIO_DGET_OSPF_INTERFACE'
   argsCapi = { 'instance': args.get( 'INSTANCE_ID' ) }

   brief = False
   if intfParam is not None:
      if intfParam == 'brief':
         brief = True
      else:
         if not intfParam.lookup():
            mode.addError( "Interface {} not present".format( intfParam.name ) )
            return OspfCliModels.OspfInterface()
         argsCapi[ 'ifname' ] = intfParam.status().deviceName

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfInterface, 
                                   cmd, argsCapi, clientName='OSPF', 
                                   l3Config=l3Config )
   except EmptyResponseException:
      return OspfCliModels.OspfInterface()
   result._brief = brief
   return result

class ShowIpOspfInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] interface [ INTERFACE | brief ] [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'interface': 'Interface-specific details',
            'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'brief': 'Condensed interface status',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfIntf
   cliModel = OspfShowInterfaceVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfInterfaceCmd )

#-------------------------------------------------------------------------------
# "show ip ospf [instance-id>] neighbor [<interface-type> <interface-number>]
#                        [<neighbor-id>]
#                        [detail]
#                        [state <state>]
#                        [ vrf vrf-name|default|all ]"
#-------------------------------------------------------------------------------

neighborOspfShowKw = CliMatcher.KeywordMatcher( 'neighbor',
                                       helpdesc='Protocol neighbor details' )
matcherNeighborId = IpAddrMatcher.IpAddrMatcher( "Neighbor ID" )

# Because we do not allow space in tokenRules in our Cli, we define the
# dictionary for mapping Cli tokenRules to state strings.
allNeighborStates = { 'down' : 'DOWN',
                      'attempt' : 'ATTEMPT',
                      'init' : 'INIT',
                      '2-ways' : '2 WAYS',
                      'exch-start' : 'EXCH START',
                      'exchange' : 'EXCHANGE',
                      'loading' : 'LOADING',
                      'full' : 'FULL',
                      'graceful-restart' : 'GRACEFUL RESTART' }

def _getPatternForNbrBrief():
   regex = r'(\d+\.\d+\.\d+\.\d+) +(\S+) +(\d+) +'  #Neighbor ID, VRF, priority
   regex += '(' + '|'.join( allNeighborStates.values() )+ ')+(/(DR(OTHER)?|BDR))? +'
   regex += r'(\d+\:\d+\:\d+) +(\d+\.\d+\.\d+\.\d+) +(.+)' #Dead Time, Addr, Intf

   pattern = re.compile( regex )
   return pattern

patternForNbrBrief = _getPatternForNbrBrief()

def _neighborInfoFilter( out, stateFilter ):
   lines = []
   for line in out.split( '\n' ):
      matched = patternForNbrBrief.match( line )
      if matched:
         state1 = matched.group(4)
         if stateFilter and state1 != allNeighborStates.get( stateFilter ):
            continue
         state2 = matched.group(5) if matched.group(5) else ''
         routerId = ospfRouterIdToStr( matched.group( 1 ), 15 )
         line = '%-15s %-8s %3s   %-16s %-11s %-15s %s' % \
             ( routerId, matched.group(2), matched.group(3), (state1 + state2),
               matched.group(8), matched.group(9),
               matched.group(10) )
         lines.append( line )
         continue

      matched = re.match( r'Neighbor (\d+\.\d+\.\d+\.\d+)+,', line )
      if matched:
         routerId = ospfRouterIdToStr( matched.group( 1 ), 15 )
         line = 'Neighbor %s,%s' % ( routerId, line[ matched.end(): ] )
      lines.append( line )

   return '\n'.join( lines)

modelDesc = "OSPF neighbor instance information for all VRFs"
OspfNeighDictModel = generateOspfInstListCliModel(
                     OspfCliModels.OspfNeighborsInstance )
OspfNeighVrfDictModel = generateVrfCliModel( OspfNeighDictModel, modelDesc )

@OspfVrfModelDecorator( vrfModelListType=OspfNeighVrfDictModel,
                        instModelListType=OspfNeighDictModel )
def showIpOspfNeighbor( mode, args ):
   neighborId = args.get( 'NEIGHBOR_ID' )
   detail = 'detail' in args
   brief = not neighborId and not detail
   command = 'MIO_DGET_OSPF_NEIGHBORS'
   vrfName = args.get( 'VRF' )
   state = args.get( 'STATE' )
   instanceId = args.get( 'INSTANCE_ID' )
   intf = args.get( 'INTF' )
   argsCapi = { 'instance': instanceId }

   if intf:
      if not intf.lookup():
         mode.addError( "Interface %s not present" % intf.name )
         return OspfCliModels.OspfNeighborsInstance()
      argsCapi[ 'intf_name' ] = intf.status().deviceName
   if neighborId:
      argsCapi[ 'routerid' ] = neighborId
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if state is not None and state in NGB_INQUIRED_STATE_MAP:
      argsCapi[ 'ngb_state' ] = NGB_INQUIRED_STATE_MAP[ state ]

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfNeighborsInstance,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      if brief:
         result._detailsPresent = False
      return result
   else:
      return OspfCliModels.OspfNeighborsInstance()

class ShowIpOspfNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip ospf [ INSTANCE_ID ] neighbor [ INTF ] '
              '[ NEIGHBOR_ID ] [ detail ] [ state STATE ] [ VRF ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'INSTANCE_ID': matcherInstance,
      'neighbor': neighborOspfShowKw,
      'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
      'NEIGHBOR_ID': matcherNeighborId,
      'detail': 'Detailed neighbor information',
      'state': 'Only show neighbors of the specific state',
      'STATE': CliMatcher.EnumMatcher( allNeighborStates ),
      'VRF': vrfExprFactory,
   }
   handler = showIpOspfNeighbor
   cliModel = OspfNeighVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfNeighborCmd )

#-----------------------------------------------------------------------
# show ip ospf [<instance-id>] neighbor [<interface>] [<neighbor-id>]
#                        adjacency-changes [vrf vrf-name|default|all]
#-----------------------------------------------------------------------
OspfShowNeighborAdjChangeDictModel = generateOspfInstListCliModel(
                                     OspfCliModels.OspfNeighborAdjChanges )
OspfShowNeighborAdjChangeVrfDictModel = generateVrfCliModel(
                                        OspfShowNeighborAdjChangeDictModel,
                                        'OSPF adjacency changes info for all VRFs' )
@OspfVrfModelDecorator( vrfModelListType=OspfShowNeighborAdjChangeVrfDictModel,
                        instModelListType=OspfShowNeighborAdjChangeDictModel )
def showIpOspfNeighborAdjacencyChanges( mode, args ):
   command = 'MIO_DGET_OSPF_NEIGHBORS'
   instanceId = args.get( 'INSTANCE_ID' )
   intf = args.get( 'INTF' )
   neighborId = args.get( 'NEIGHBOR_ID' )
   vrfName = args.get( 'VRF' )
   argsCapi = { 'adj_chg' : True, 'instance': instanceId }
   if intf is not None:
      if not intf.lookup():
         mode.addError( "Interface %s not present" % intf.name )
         return getDefaultModel( OspfCliModels.OspfNeighborAdjChanges(), instanceId )
      argsCapi[ 'intf_name' ] = intf.status().deviceName
   if neighborId is not None:
      argsCapi[ 'routerid' ] = neighborId
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfNeighborAdjChanges,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = getDefaultModel( OspfCliModels.OspfNeighborAdjChanges(),
                                instanceId )

   return result

class ShowIpOspfNeighborAdjChangesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip ospf [ INSTANCE_ID ] neighbor [ INTF ] [ NEIGHBOR_ID ] '
              'adjacency-changes [ VRF ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'INSTANCE_ID': matcherInstance,
      'neighbor': neighborOspfShowKw,
      'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
      'NEIGHBOR_ID': matcherNeighborId,
      'adjacency-changes': 'Adjacency change log',
      'VRF': vrfExprFactory,
   }
   handler = showIpOspfNeighborAdjacencyChanges
   cliModel = OspfShowNeighborAdjChangeVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfNeighborAdjChangesCmd )

#------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] neighbor summary"
#------------------------------------------------------------------------------
modelDesc = 'OSPF neighbor summary for all VRFs'
OspfShowNeighborSummaryDictModel = generateOspfInstListCliModel(
                                   OspfCliModels.OspfNeighborSummaryInstance )
OspfShowNeighborSummaryVrfDictModel = generateVrfCliModel(
                                      OspfShowNeighborSummaryDictModel, modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfShowNeighborSummaryVrfDictModel,
                        instModelListType=OspfShowNeighborSummaryDictModel )
def showIpOspfNeighborSummary( mode, args ):
   command = 'MIO_DGET_OSPF_NEIGHBORS'
   vrfName = args.get( 'VRF' )
   instanceId = args.get( 'INSTANCE_ID' )
   argsCapi = { 'instance': instanceId }

   argsCapi[ 'neighborSum' ] = True

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfNeighborSummaryInstance,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      result = getDefaultModel( OspfCliModels.OspfNeighborSummaryInstance(),
                                instanceId )
      result.neighborSummary = OspfCliModels.OspfNeighborSummary()
      return result

class ShowIpOspfNeighborSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] neighbor summary [ VRF ] '
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'INSTANCE_ID': matcherInstance,
      'neighbor': neighborOspfShowKw,
      'summary': 'Summary of neighbor information',
      'VRF': vrfExprFactory,
   }
   handler = showIpOspfNeighborSummary
   cliModel = OspfShowNeighborSummaryVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfNeighborSummaryCmd )

#-------------------------------------------------------------------------------
# "show ip ospf [<instance-id> <area-id>] database [vrf default|vrf-name|all]"
#-------------------------------------------------------------------------------
matcherDatabase = CliMatcher.KeywordMatcher( 'database',
                  helpdesc='LSA database' )
matcherDatabaseSummary = CliMatcher.KeywordMatcher( 'database-summary',
                         helpdesc='Link state database summary' )
modelDesc = "OSPF database summary information"
OspfDatabaseSummaryInstModel = generateOspfInstListCliModel(
   OspfCliModels.OspfDatabaseSummary )
OspfDatabaseSummaryVrfModel = generateVrfCliModel( OspfDatabaseSummaryInstModel,
                                                   modelDesc )

def setAreaId( args, areaId ):
   if areaId is not None:
      areaId = Arnet.IpAddress( areaId )
      args[ 'area' ] = str( areaId )
      args[ 'queryType' ] = OSPF_DATABASE_QUERY_TYPE_MAP[ 'area' ]
   return

@OspfVrfModelDecorator( vrfModelListType=OspfDatabaseSummaryVrfModel,
                        instModelListType=OspfDatabaseSummaryInstModel )
def showIpOspfDatabaseSummary( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   areaId = args.get( 'AREA_ID' )
   command = 'MIO_DGET_OSPF_DATABASE'
   argsCapi = { 'instance': instanceId }

   setAreaId( argsCapi, areaId )
   
   # database and database-summary use the same MIO command. Setting this
   # param determines which one
   argsCapi[ 'databaseSum' ] = True

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfDatabaseSummary,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = OspfCliModels.OspfDatabaseSummary()

   result._vrf = vrfName
   result._instanceId = instanceId
   status = ospfStatus()
   vrfNameStatus = vrfName
   if not vrfNameStatus:
      vrfNameStatus = DEFAULT_VRF
   statusVrf = status.get( vrfNameStatus, None )
   instanceStatus = statusVrf.instanceStatus.get( instanceId, None )
   result._routerId = None
   if instanceStatus:
      result._routerId = instanceStatus.routerId
   return result

modelDesc = "OSPF database information"
OspfDatabaseInstModel = generateOspfInstListCliModel(
                        OspfCliModels.OspfDatabaseInstance )
OspfDatabaseVrfModel = generateVrfCliModel( OspfDatabaseInstModel,
                                            modelDesc )

@OspfVrfModelDecorator( vrfModelListType=OspfDatabaseVrfModel,
                        instModelListType=OspfDatabaseInstModel )
def showIpOspfDatabase( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   areaId = args.get( 'AREA_ID' )
   lsType = args.get( 'TYPE' )
   lsaTypeFilter = None
   origRouter = None
   if lsType:
      lsaTypeFilter = { 'lsaType': lsType,
                        'linkStateId': args.get( 'LNK_STATE_ID' ) }
   if 'adv-router' in args:
      origRouter = { 'ipAddr': args.get('ADV_RTR_ID'), 'advRouter': 'adv-router' }
   elif 'self-originate' in args:
      origRouter = { 'selfOriginate': 'self-originate' }
   command = 'MIO_DGET_OSPF_DATABASE'
   argsCapi = { 'instance': instanceId }
   
   printDetail = False
   trafficEngineering = None
   lsType = None
   linkstateId = None
    
   if lsaTypeFilter is not None:
      printDetail = True
      linkstateId = lsaTypeFilter[ 'linkStateId' ]
      filterName = [ value for key, value in lsaTypeFilter.iteritems() 
                    if key != 'linkStateId' ]
      lsType = filterName[ 0 ]
      if lsType == 'detail':
         lsType = None
      elif lsType == 'traffic-engineering':
         lsType = None
         trafficEngineering = True

   setAreaId( argsCapi, areaId )
   
   # External LSAs are not specific to an area
   if lsType in ( 'external', 'opaque-as' ):
      argsCapi[ 'queryType' ] = OSPF_DATABASE_QUERY_TYPE_MAP[ lsType ]
      lsType = None
   
   if lsType is not None:
      argsCapi[ 'lsType' ] = OSPF_DATABASE_LSA_TYPE_MAP[ lsType ]
  
   if linkstateId is not None:
      argsCapi[ 'lsid' ] = linkstateId

   argsCapi[ 'printType' ] = printDetail

   selfOriginate = None
   advRouterId = None
   if origRouter:
      if "advRouter" in origRouter:
         advRouterId = origRouter.get( 'ipAddr' )
         if advRouterId is None:
            selfOriginate = True
      elif "selfOriginate" in origRouter:
         selfOriginate = True

   if selfOriginate is not None:
      argsCapi[ 'self' ] = selfOriginate
   if advRouterId is not None:
      argsCapi[ 'advRouter' ] = advRouterId
   if trafficEngineering:
      argsCapi[ 'teLsa' ] = trafficEngineering
   
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
      
   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfDatabaseInstance,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = getDefaultModel( OspfCliModels.OspfDatabaseInstance(), instanceId )

   if printDetail:
      result._detailsPresent = True

   return result

# show ip ospf [ <instanceId> ]
#              [ <areaIdDecimalFmt> | <areaIdIpAddrFmt> ]
#              database [
#                            ( detail [ <linkStateId> ] ) |
#                            ( asbr-summary [ <linkStateId> ] ) |
#                            ( external [ <linkStateId> ] ) |
#                            ( network [ <linkStateId> ] ) |
#                            ( nssa-external [ <linkStateId> ] ) |
#                            ( opaque-area [ <linkStateId> ] ) |
#                            ( opaque-as [ <linkStateId> ] ) |
#                            ( opaque-link [ <linkStateId> ] ) |
#                            ( router [ <linkStateId> ] ) |
#                            ( summary [ <linkStateId> ] )
#                          ]
#                          [ ( adv-router [ <advRtrId> ] ) | self-originate ]
#                          [ vrf <vrfName> ]

# The decorator @OspfVrfModelDecorator doesn't work properly with the concat rule
# [ <instanceId> [ <areaIdDecimalFmt> | <areaIdIpAddrFmt> ] ]
# so instead, we need to pollute the shift-? completion
# to allow the user to specify the optional area id. A side effect of this is that
# an area id can be specified without specifying an instance id. The cli command 
# still has the same form, but is less restrictive

lsaType = {
            'detail': 'Detailed LSA information',
            'asbr-summary': 'ASBR summary route LSAs',
            'external': 'External LSAs',
            'network': 'OSPF network LSA entries',
            'nssa-external': 'OSPF NSSA LSA entries',
            'opaque-area': 'OSPF area-local opaque LSA entries',
            'opaque-as': 'OSPF AS-wide opaque LSA entries',
            'opaque-link': 'OSPF link-local LSA entries',
            'router': 'OSPF router LSA entries',
            'summary': 'OSPF summary LSA entries',
            'traffic-engineering': 'Traffic engineering link states',
}

class ShowIpOspfDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip ospf [ INSTANCE_ID ]
               [ AREA_ID ] database
               [ TYPE [ LNK_STATE_ID ] ]
               [ ( adv-router [ ADV_RTR_ID ] ) | self-originate ]
               [ VRF ]'''
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': CliMatcher.IntegerMatcher( instanceMin, instanceMax,
                              helpdesc='Instance ID',
                              priority=CliParser.PRIO_HIGH ),
            'AREA_ID': AreaIdExpression,
            'database': matcherDatabase,
            'TYPE': CliMatcher.EnumMatcher( lsaType ),
            'LNK_STATE_ID': IpAddrMatcher.IpAddrMatcher(
               helpdesc='IP address used as the link state ID' ),
            'adv-router': 'Router advertising the LSA',
            'ADV_RTR_ID': IpAddrMatcher.IpAddrMatcher(
               helpdesc='IP address of a router advertising the LSA' ),
            'self-originate': 'OSPF locally originated LSA entries',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfDatabase
   cliModel = OspfDatabaseVrfModel

BasicCli.addShowCommandClass( ShowIpOspfDatabaseCmd )

class showIpOspfDatabaseSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip ospf [ INSTANCE_ID ]
               [ AREA_ID ] database
               database-summary [ VRF ]'''
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': CliMatcher.IntegerMatcher( instanceMin, instanceMax,
                              helpdesc='Instance ID',
                              priority=CliParser.PRIO_HIGH ),
            'AREA_ID': AreaIdExpression,
            'database': matcherDatabase,
            'database-summary': matcherDatabaseSummary,
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfDatabaseSummary
   cliModel = OspfDatabaseSummaryVrfModel

BasicCli.addShowCommandClass( showIpOspfDatabaseSummaryCmd )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] counters [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
modelDesc = 'OSPF Counters'
OspfShowCountersInstModel = generateOspfInstListCliModel(
                              OspfCliModels.OspfCountersInstance )
OspfShowCountersVrfModel = generateVrfCliModel( OspfShowCountersInstModel,
                                                modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfShowCountersVrfModel,
                        instModelListType=OspfShowCountersInstModel )
def showIpOspfCounters( mode, args ):
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF_COUNTERS'
   argsCapi = { 'instanceId': args.get( 'INSTANCE_ID' ) }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfCountersInstance,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      return OspfCliModels.OspfCountersInstance()

class ShowIpOspfCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] counters [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'counters': 'OSPF Counters',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfCounters
   cliModel = OspfShowCountersVrfModel

BasicCli.addShowCommandClass( ShowIpOspfCountersCmd )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] counters interface
#                                   [ifName|summary] [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
modelDesc = 'OSPF Interface Counters'
OspfShowIntfCountersInstModel = generateOspfInstListCliModel(
                                    OspfCliModels.OspfCountersInterface )
OspfShowIntfCountersVrfModel = generateVrfCliModel( OspfShowIntfCountersInstModel,
                                                    modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfShowIntfCountersVrfModel,
                        instModelListType=OspfShowIntfCountersInstModel )
def showIpOspfCountersInterface( mode, args ):
   command = 'MIO_DGET_OSPF_COUNTERS_INTERFACE'
   intfParam = args.get( 'INTERFACE' ) or args.get( 'summary' )
   vrfName = args.get( 'VRF' )
   argsCapi = { 'instanceId': args.get( 'INSTANCE_ID' ) }

   summary = False
   if intfParam is not None:
      if intfParam == 'summary':
         summary = True
      else:
         if not intfParam.lookup():
            mode.addError( "Interface {} not present".format( intfParam.name ) )
            return OspfCliModels.OspfCountersInterface()
         argsCapi[ 'ifname' ] = intfParam.status().deviceName

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfCountersInterface,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      return OspfCliModels.OspfCountersInterface()
   result._summary = summary
   return result

class ShowIpOspfCountersInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip ospf [ INSTANCE_ID ] counters interface '
              '[ INTERFACE | summary ] [ VRF ]' )
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'counters': 'OSPF counters',
            'interface': 'Interface counters',
            'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'summary': 'Interface counters summary',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfCountersInterface
   cliModel = OspfShowIntfCountersVrfModel

BasicCli.addShowCommandClass( ShowIpOspfCountersInterfaceCmd )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] spf-log [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
modelDesc = 'OSPF SPF log information'
OspfShowSpfLogInstModel = generateOspfInstListCliModel(
                          OspfCliModels.OspfSpfLog )
OspfShowSpfLogVrfModel = generateVrfCliModel( OspfShowSpfLogInstModel,
                                              modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfShowSpfLogVrfModel,
                        instModelListType=OspfShowSpfLogInstModel )
def showIpOspfSpfLog( mode, args ):
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF_SPF_LOGS'
   argsCapi = { 'instanceId': args.get( 'INSTANCE_ID' ) }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfSpfLog,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      return OspfCliModels.OspfSpfLog()

class ShowIpOspfSpfLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] spf-log [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'spf-log': 'SPF Log',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfSpfLog
   cliModel = OspfShowSpfLogVrfModel

BasicCli.addShowCommandClass( ShowIpOspfSpfLogCmd )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] lsa-log [vrf vrf-name|default|all]
#------------------------------------------------------------------------------
def showIpOspfLsaLog( mode, args ):
   instanceIds = getInstanceIds( mode, args.get( 'INSTANCE_ID' ), args.get( 'VRF' ) )
   if not instanceIds:
      return
   for i, instanceId in enumerate( instanceIds ):
      vrfName, cmd = insertVrArgInCmd( mode, 'lsa-log', instanceId=instanceId )
      if i > 0:
         print ''
      RibCliLib.cliRibdShowCommand( mode, cmd, clientName='OSPF',
                                    l3Config=l3Config, vrf=vrfName )

class ShowIpOspfLsaLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] lsa-log [ VRF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'lsa-log': 'LSA throttling Log',
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfLsaLog

BasicCli.addShowCommandClass( ShowIpOspfLsaLogCmd )

#------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] border-routers [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
modelDesc = "OSPF border routers information for all VRFs"
OspfBorderRoutersInstModel = generateOspfInstListCliModel(
                             OspfCliModels.OspfBorderRoutersInstance )
OspfBorderRoutersVrfModel = generateVrfCliModel( OspfBorderRoutersInstModel,
                                                 modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfBorderRoutersVrfModel,
                        instModelListType=OspfBorderRoutersInstModel )
def showIpOspfBorderRouters( mode, args ):
   command = 'MIO_DGET_OSPF_BORDER_ROUTERS'
   vrfName = args.get( 'VRF' )
   instanceId = args.get( 'INSTANCE_ID' )
   argsCapi = { 'instanceId': instanceId }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, OspfCliModels.OspfBorderRoutersInstance,
                                   command, argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      return getDefaultModel( OspfCliModels.OspfBorderRoutersInstance(),
                              instanceId )

class ShowIpOspfBorderRoutersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] border-routers [ VRF ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'INSTANCE_ID': matcherInstance,
      'border-routers': 'Status for border routers',
      'VRF': vrfExprFactory,
   }
   handler = showIpOspfBorderRouters
   cliModel = OspfBorderRoutersVrfModel

BasicCli.addShowCommandClass( ShowIpOspfBorderRoutersCmd )

#------------------------------------------------------------------------------
# "show ip ospf [<instance-id>] retransmission queue [vrf vrf-name|default|all]"
# Legacy:
# "show ip ospf [<instance-id>] retransmission-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
modelDesc = "OSPF LSA retransmission information for all VRFs"
OspfLsaRetransmissionListInstModel = generateOspfInstListCliModel(
                                     OspfCliModels.OspfLsaRetransmissionList )
OspfLsaRetransmissionListVrfModel = generateVrfCliModel(
                                    OspfLsaRetransmissionListInstModel, modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfLsaRetransmissionListVrfModel,
                        instModelListType=OspfLsaRetransmissionListInstModel )
def showIpOspfLsaRetransmissionList( mode, args ):
   vrfName = args.get( 'VRF' )
   routerId = args.get( 'NEIGHBOR' )
   ifName = args.get( 'INTERFACE' )
   command = 'MIO_DGET_OSPF_RETRANSMISSION_LIST'
   argsCapi = { 'instanceId': args.get( 'INSTANCE_ID' ) }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( 'Interface %s not present' % ifName.name )
         return OspfCliModels.OspfLsaRetransmissionList()
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfLsaRetransmissionList,
                                   command, args=argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result is None:
      return OspfCliModels.OspfLsaRetransmissionList()

   return result

class ShowIpOspfLsaRetransmissionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip ospf [ INSTANCE_ID ]
               ( ( retransmission queue ) | retransmission-list )
               [ NEIGHBOR ] [ INTERFACE ] [ VRF ]'''
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'INSTANCE_ID': matcherInstance,
            'retransmission': 'Re-transmission list',
            'queue': 'Re-transmission list',
            'retransmission-list': CliCommand.Node( CliMatcher.KeywordMatcher(
               'retransmission-list', helpdesc='Re-transmission list' ),
               deprecatedByCmd='show ip ospf retransmission queue' ),
            'NEIGHBOR': IpAddrMatcher.IpAddrMatcher( helpdesc='Neighbor ID' ),
            'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }
   handler = showIpOspfLsaRetransmissionList
   cliModel = OspfLsaRetransmissionListVrfModel

BasicCli.addShowCommandClass( ShowIpOspfLsaRetransmissionCmd )

#------------------------------------------------------------------------------
# show ip ospf [<instance-id>] request queue [vrf vrf-name|default|all]"
# Legacy:
# show ip ospf [<instance-id>] request-list [vrf vrf-name|default|all]"
#------------------------------------------------------------------------------
modelDesc = "OSPF request list information for all VRFs"
OspfLsaRequestListInstModel = generateOspfInstListCliModel(
                                 OspfCliModels.OspfLsaRequestList )
OspfLsaRequestListVrfModel = generateVrfCliModel(
                                 OspfLsaRequestListInstModel, modelDesc )
@OspfVrfModelDecorator( vrfModelListType=OspfLsaRequestListVrfModel,
                        instModelListType=OspfLsaRequestListInstModel )
def showIpOspfLsaRequestList( mode, args ):
   command = 'MIO_DGET_OSPF_REQUEST_LIST'
   instanceId = args.get( 'INSTANCE_ID' )
   routerId = args.get( 'ROUTER_ID' )
   ifName = args.get( 'INTF' )
   vrfName = args.get( 'VRF' )
   argsCapi = { 'instanceId': instanceId }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( 'Interface %s not present' % ifName.name )
         return OspfCliModels.OspfLsaRequestList()
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      result = showRibCapiCommand( mode, OspfCliModels.OspfLsaRequestList,
                                   command, args=argsCapi, clientName='OSPF',
                                   l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result is None:
      return OspfCliModels.OspfLsaRequestList()
   return result

class ShowIpOspfLsaRequestCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip ospf [ INSTANCE_ID ] ( request-list | ( request queue ) ) '
              '[ NEIGHBOR_ID ] [ INTF ] [ VRF ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'INSTANCE_ID': matcherInstance,
      'request': 'Request list',
      'queue': 'Request list',
      'request-list': CliCommand.Node( CliMatcher.KeywordMatcher( 'request-list',
                                 helpdesc='Request list' ),
                                 deprecatedByCmd='show ip ospf request queue' ),
      'NEIGHBOR_ID': matcherNeighborId,
      'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
      'VRF': vrfExprFactory,
   }
   handler = showIpOspfLsaRequestList
   cliModel = OspfLsaRequestListVrfModel

BasicCli.addShowCommandClass( ShowIpOspfLsaRequestCmd )

#-------------------------------------------------------------------------------
# show ip ospf [instanceId] mpls ldp sync [interface]
#-------------------------------------------------------------------------------

OspfMplsLdpSyncDictModel = generateOspfInstListCliModel( OspfMplsLdpSyncModel )
OspfMplsLdpSyncVrfDictModel = generateVrfCliModel(
   OspfMplsLdpSyncDictModel,
   "OSPF mpls ldp sync information for all VRFS" )

@OspfVrfModelDecorator( vrfModelListType=OspfMplsLdpSyncVrfDictModel,
                        instModelListType=OspfMplsLdpSyncDictModel )
def showMplsLdpSync( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   intfName = args.get( 'INTERFACE' )
   vrfName = args.get( 'VRF' )
   instance = ospfConfigDir.instanceConfig[ instanceId ]
   interfaces = {}
   argsCapi = { "instance" : instanceId }
   if intfName is not None:
      argsCapi[ 'ifname' ] = kernelIntfFilter( str( intfName ) )
   ospfInterface = showRibCapiCommand(
      mode, OspfCliModels.OspfInterface, "MIO_DGET_OSPF_INTERFACE", argsCapi,
      clientName="OSPF", l3Config=l3Config )

   for intf in ospfInterface.interfaces:
      ldpConfig = ldpConfigColl.config.get( vrfName )
      ldpProtoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
      readyStatusColl = linkReadyStatusVrfColl.readyStatusVrf.get( vrfName )
      ldpEnabled = readyStatusColl is not None

      def _syncRequired( iName, ldpE=ldpEnabled, ldpC=ldpConfig ):
         if not ldpE:
            return False
         if not ldpC:
            return instance.mplsLdpSync
         ldpIntfConfig = ldpC.intfConfigColl.get( iName )
         if not ldpIntfConfig or ldpIntfConfig.igpSync == "useGlobal":
            return instance.mplsLdpSync
         return ldpIntfConfig.igpSync == "on"
      
      syncRequired = _syncRequired( intf[0] )
      if not syncRequired:
         achieved = "Not applicable"
      else:
         readyStatus = readyStatusColl.readyStatus.get( intf[0] )
         achieved = "Yes" if readyStatus and readyStatus.ready else "No"
      if ldpProtoConfig:
         interfaces[ intf[0] ] = OspfMplsLdpSyncEntry(
            instanceId=instanceId,
            interface=intf[0],
            ldpEnabled=ldpEnabled,
            syncRequired=syncRequired,
            syncAchieved=achieved,
            igpDelay=int( ldpProtoConfig.linkReadyDelay ),
            holddownTime=int( ldpProtoConfig.linkReadyTimeout ))
      else:
         interfaces[ intf[0] ] = OspfMplsLdpSyncEntry(
            instanceId=instanceId,
            interface=intf[0],
            ldpEnabled=ldpEnabled,
            syncRequired=syncRequired,
            syncAchieved=achieved )

   return OspfMplsLdpSyncModel( interfaces=interfaces )

class ShowIpOspfMplsLdpSyncCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf [ INSTANCE_ID ] mpls ldp sync [ INTERFACE ]'
   data = {
          'ip': CliToken.Ip.ipMatcherForShow,
          'ospf': matcherOspfShow,
          'INSTANCE_ID': matcherInstance,
          'mpls': mplsNodeForShow,
          'ldp': 'LDP sync configuration',
          'sync': 'Sync configuration',
          'INTERFACE': IntfCli.Intf.matcher,
          }
   handler = showMplsLdpSync
   cliModel = OspfMplsLdpSyncVrfDictModel

BasicCli.addShowCommandClass( ShowIpOspfMplsLdpSyncCmd )
#-------------------------------------------------------------------------------
# show ip ospf access-list [<name>]
#-------------------------------------------------------------------------------
class ShowIpOspfAclListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf access-list [ ACL_NAME ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'ospf': matcherOspfShow,
            'access-list': AclCli.accessListMatcher,
            'ACL_NAME': AclCli.ipAclNameExpression,
           }
   @staticmethod
   def handler( mode, args ):
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    "ip",
                                    args.get( '<aclNameExpr>' ),
                                    serviceName=OspfConsts.serviceName )

   cliModel = AclCliModel.AllAclList
BasicCli.addShowCommandClass( ShowIpOspfAclListCmd )

#-------------------------------------------------------------------------------
# OspfIntf class is registered with IntfCli.Intf class to register callbacks. 
# Destroy method of OspfIntf class is called by Intf class when an interface 
# object is deleted. Intf class will create a new instance of OspfIntf and call
# destroy method on it.
# ------------------------------------------------------------------------------

class OspfIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the Ospf IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      config = ospfConfig()
      vrfName = _getIntfVrf( self.intf_.name )
 
      areaId = None
      intfConfig = config.intfConfig.get( self.intf_.name )
      if intfConfig and intfConfig.areaIdPresent:
         areaId = intfConfig.areaId

      del config.intfConfig[ self.intf_.name ]
      # delete the associated area config if required
      if areaId and nInstancesInVrf( config.instanceConfig, vrfName ) == 1 :
         instanceId = vrfNameToInstance( config.instanceConfig, vrf=vrfName )[ 0 ]
         _deleteIntfAreaConfigIfAllAttributeHaveDefaults( instanceId, areaId, 
                                                          vrfName )

#-------------------------------------------------------------------------------
# OSPF commands in "show tech-support"
#-------------------------------------------------------------------------------
# In the future if we add commands that fork rib to produce their output, then 
# we should split the list of commands into two. One list for commands that don't 
# fork ribd using GATED_PROTO_SH_TECH_TS and the second one for commands that fork
# using GATED_PROTO_FORK_SH_TECH_TS
def _showTechSupportCmds():
   cmds = [ "show ip ospf",
            "show ip ospf interface",
            "show ip ospf neighbor detail" ]
   return cmds

def showTechSupportSummaryCmds():
   return [ "show ip ospf",
            "show ip ospf interface",
            "show ip ospf neighbor" ]

TechSupportCli.registerShowTechSupportCmdCallback(
    GATED_PROTO_SH_TECH_TS, _showTechSupportCmds,
    summaryCmdCallback=showTechSupportSummaryCmds )

def doClearIpOspf( mode, args ): 
   subCmd = args.get( 'access-list' ) or args.get( 'neighbor' ) or \
            args.get( 'counters' )
   instId = args.get( 'INST_ID' )
   vrfName = args.get( 'VRF' )
   nbr = args.get( '*' ) or args.get( 'ADDRESS' ) or args.get( 'INTERFACE' )
   countersIntfArg = args.get( 'all' ) or args.get( 'INTERFACE' )
   if subCmd == 'access-list':
      # clear counters of all access-lists
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, "ip" )
      return
   
   if not instId and not vrfName:
      vrfName = DEFAULT_VRF
   instanceIds = getInstanceIds( mode, instId, vrfName )
   if not instanceIds:
      return
   ifName = ''

   # nbr is one of the three of: '*', ipaddr, intf. If the last, 
   # we need to assign it to ifname variable to pass it on to mio api.

   if subCmd == 'neighbor':
      if isinstance( nbr, IntfCli.Intf ) :
         # does the interface exist?
         if not nbr.lookup() :
            mode.addError( 'Interface does not exist' )
            return
         ifName = nbr.status().deviceName
         desc = 'neighbors on %s interface to be cleared' % nbr.status().intfId
      elif nbr == '*':
         if vrfName == ALL_VRF_NAME:
            desc = 'all neighbors on all vrfs to be cleared'
         else:
            desc = 'all neighbors to be cleared'
      else:
         desc = 'neighbor %s to be cleared' % nbr
   elif subCmd == 'counters':
      if countersIntfArg is not None:
         if isinstance( countersIntfArg, IntfCli.Intf ) :
            if not countersIntfArg.lookup() :
               mode.addError( 'Interface does not exist' )
               return
            ifName = countersIntfArg.status().deviceName
            desc = 'interface-level counters on interface %s' \
                   ' to be cleared' % countersIntfArg.status().intfId
         elif countersIntfArg == 'all':
            ifName = 'all'
            if vrfName == ALL_VRF_NAME:
               desc = 'interface-level counters on all interfaces on all vrfs' \
                      ' to be cleared'
            else:
               desc = 'interface-level counters on all interfaces on vrf %s' \
                      ' to be cleared' % vrfName
         else:
            mode.addError( 'Invalid interface' )
            return
      else:
         if vrfName == ALL_VRF_NAME:
            desc = 'instance-level counters on all vrfs to be cleared'
         else:
            desc = 'instance-level counters on vrf %s to be cleared' % vrfName

   # except when subcmd is 'neighbor addr', set 'nbr' to zero-ip
   # so that an invalid-ip exception is not raised. 'neighbor' has 
   # an ip-addr argument when it is not '*' and was not used to 
   # give us the ifname argument, that is, ifname is empty

   if not ( subCmd == 'neighbor' and ifName == '' and nbr != '*' ):
      nbr = '0.0.0.0'

   for instId in instanceIds:
      req = Tac.Value( 'Routing::Ospf::ClearRequest', instId, subCmd, ifName, nbr )
      reqId = "%d.%f" % ( os.getpid(), Tac.now() )
      ospfClearReqDir.clearRequest[ reqId ] = req
      vrfName = ospfConfig().instanceConfig.get( instId ).vrfName
      try:
         Tac.waitFor( lambda r=reqId: vrfName in ospfClearRespDir and
                              r in ospfClearRespDir[ vrfName ].clearResponse,
                      description = desc,
                      sleep=True, maxDelay=0.1, timeout=5.0 )
         nCleared = ospfClearRespDir[vrfName].clearResponse[reqId].nCleared
         opResult = ospfClearRespDir[vrfName].clearResponse[reqId].opResult
         if opResult and 'show' in args:
            if subCmd == 'neighbor':
               print "%d neighbor%s cleared" % ( nCleared,
                                                 's' if nCleared != 1 else '' ),
               print "on vrf %s" % vrfName
            else:
               print "counters cleared on vrf %s" % vrfName

      except ( Tac.Timeout, KeyError ):
         plural = '(s)' if subCmd == 'neighbor' else ''
         mode.addWarning( "%s%s may not have been cleared yet on vrf %s" %
                          ( subCmd, plural, vrfName ) )
      del ospfClearReqDir.clearRequest[reqId]

matcherOptResetAllNeighbors = CliMatcher.KeywordMatcher( '*', 
      helpdesc="Reset ALL neighbors" )
matcherNeighbor = CliMatcher.KeywordMatcher( 'neighbor', 
      helpdesc="Reset neighbor(s)" )

matcherCounters = CliMatcher.KeywordMatcher( 'counters', 
      helpdesc="Reset counters" )

matcherInstId = CliMatcher.IntegerMatcher( 1, 65535, helpdesc='OSPF instance ID' )
matcherShow = CliCommand.Node( CliMatcher.KeywordMatcher( 'show', helpdesc='' ), 
                               hidden=True )

#-------------------------------------------------------------------------------
# clear ip ospf access-list counters
#-------------------------------------------------------------------------------
class ClearIpOspfAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip ospf access-list counters'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'ip': CliToken.Ip.ipMatcherForClear,
            'ospf': ospfKw,
            'access-list': AclCli.accessListMatcher,
            'counters': matcherCounters,
          }

   handler = doClearIpOspf

BasicCli.EnableMode.addCommandClass( ClearIpOspfAccessListCountersCmd )

#-------------------------------------------------------------------------------
# clear ip ospf [inst_id] counters [ vrf vrfName ] [ show ]
#-------------------------------------------------------------------------------
class ClearIpOspfCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip ospf [ INST_ID ] counters [ VRF ] [ show ]'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'ip': CliToken.Ip.ipMatcherForClear,
            'ospf': ospfKw,
            'INST_ID': matcherInstId,
            'counters': 'Reset OSPF counters',
            'VRF': vrfExprFactory,
            'show': matcherShow,
          }

   handler = doClearIpOspf

BasicCli.EnableMode.addCommandClass( ClearIpOspfCountersCmd )

#-------------------------------------------------------------------------------
# clear ip ospf [inst_id] counters interface < all | ifname > [ vrf vrfName ]
#-------------------------------------------------------------------------------
class ClearIpOspfCountersInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear ip ospf [ INST_ID ] counters interface ( all | INTERFACE ) '
              '[ VRF ]' )
   data = { 'clear': CliToken.Clear.clearKwNode,
            'ip': CliToken.Ip.ipMatcherForClear,
            'ospf': ospfKw,
            'INST_ID': matcherInstId,
            'counters': 'Reset OSPF counters',
            'interface': 'Reset interface counters',
            'all': 'Clear counters on all interfaces',
            'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }

   handler = doClearIpOspf

BasicCli.EnableMode.addCommandClass( ClearIpOspfCountersInterfaceCmd )

#-------------------------------------------------------------------------------
# clear ip ospf [inst_id] neighbor < * | nbrAddr | ifname > [ vrf vrfName ] [ show ]
#-------------------------------------------------------------------------------
class ClearIpOspfNbrCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear ip ospf [ INST_ID ] neighbor ( * | ADDRESS | INTERFACE ) '
              '[ VRF ] [ show ]' )
   data = { 'clear': CliToken.Clear.clearKwNode,
            'ip': CliToken.Ip.ipMatcherForClear,
            'ospf': ospfKw,
            'INST_ID': matcherInstId,
            'neighbor': matcherNeighbor,
            '*': matcherOptResetAllNeighbors,
            'ADDRESS': IpAddrMatcher.IpAddrMatcher( helpdesc='Neighbor address' ),
            'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
            'show': matcherShow,
          }

   handler = doClearIpOspf

BasicCli.EnableMode.addCommandClass( ClearIpOspfNbrCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ospfConfigDir, ospfStatusDir, sysNetConfigDir
   global routingHardwareStatus
   global allVrfConfig
   global ospfClearReqDir, ospfClearRespDir
   global ldpConfigColl, ldpProtoConfigColl, linkReadyStatusVrfColl
   global l3IntfConfigDir
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global mplsConfig
   ospfConfigDir = ConfigMount.mount( entityManager, OspfConsts.configPath,
                                      OspfConsts.configType, 'w' )
   ospfStatusDir = LazyMount.mount( entityManager, OspfConsts.statusPath,
                                    OspfConsts.statusType, 'ri' )
   global l3Config
   ospfClearReqDir = LazyMount.mount( entityManager, 'routing/ospf/clear/request',
                                      'Routing::Ospf::ClearRequestNode', 'w' )
   ospfClearRespDir = LazyMount.mount( entityManager, 'routing/ospf/clear/response',
                                       'Tac::Dir', 'ri' )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
   sysNetConfigDir = LazyMount.mount( entityManager, 'sys/net/config',
                                      'System::NetConfig', 'r' )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   linkReadyStatusVrfColl = LazyMount.mount( entityManager,
                                  "mpls/ldp/linkReadyStatus",
                                  "Ldp::LdpLinkReadyStatusVrfColl", "r" )
   ldpProtoConfigColl = LazyMount.mount( entityManager,
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpConfigColl = LazyMount.mount( entityManager,
                                    "mpls/ldp/ldpConfigColl",
                                    "Ldp::LdpConfigColl", "r" )
   # We need the interface vrfs to associate with the correct ospf3 instance
   l3IntfConfigDir = LazyMount.mount( entityManager, "l3/intf/config",
                                       "L3::Intf::ConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( OspfIntf, priority=1 )
   IraVrfCli.canDeleteVrfHook.addExtension( deleteVrfHook )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
   mplsConfig = LazyMount.mount( entityManager, "routing/mpls/config",
                                 "Mpls::Config", "r" )
