#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import sys
import socket

import AclCli
import AclCliModel
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin
from CliPlugin.VrfCli import generateVrfCliModel
import CliToken.Clear
from CliToken.Router import (
      routerMatcherForConfig,
)
import IntfCli
import Intf.IntfRange as IntfRange
import IpAddrMatcher
import IraVrfCli
import Ospf3CliModels
from CliMode.Ospf3MultiAf import (
      RoutingOspfv3Mode,
      RoutingOspfv3AfMode,
      RoutingOspfv3GeneralMode,
)
from CliModel import GeneratorDict, Model
from IpLibConsts import DEFAULT_VRF
from RibCapiLib import (
      showRibCapiCommand,
      EmptyResponseException,
      showRibDamiCommand,
)
from RoutingOspfCli import (
      areaKw,
      AreaIdExpression,
      bfdKw,
      costKw,
      deadIntervalKw,
      helloIntervalKw,
      matcherCost,
      matcherCounters,
      matcherDatabase,
      matcherDatabaseSummary,
      matcherNeighbor,
      matcherNeighborId,
      matcherOptResetAllNeighbors,
      matcherPassiveDefault,
      matcherPassiveIntf,
      matcherPriority,
      matcherShow,
      mtuIgnoreKw,
      networkTypeKw,
      networkTypesMatcher,
      priorityKw,
      retransmitIntervalKw,
      transmitDelayKw,
)
from RoutingOspf3Cli import (
      AF_INSTANCE_ID_MAP,
      RouterOspf3Common,
      RouterOspfv3AfIpv4Modelet,
      RouterOspfv3AfIpv6Modelet,
      RouterOspfv3AfSharedModelet,
      RouterOspfv3SharedModelet,
      RouterOspfv3SharedModelet2,
      _deleteIntfInstanceConfig,
      _getIntfConfigForAttr,
      afAllInstanceID,
      afIPv4InstanceID,
      afIPv6InstanceID,
      authKeyValMatcher,
      cipherValMatcher,
      cleanupOspf3InstanceConfig,
      defaultIntfBfd,
      doClearIpv6Ospf,
      espKw,
      getDefaultInstanceModel,
      getDefaultModelValues,
      getVrfToInstanceIdsDict,
      getl3Config,
      hashValMatcher,
      ifType,
      intfAuthenticationKw,
      intfEncryptionKw,
      intfModelet,
      ipsecKw,
      keyValNode,
      lsaType,
      matcherAdvRouter,
      matcherAreaFilter,
      matcherAreaIdFilter,
      matcherAsFilter,
      matcherAsExternal,
      matcherBackbone,
      matcherBorderRouters,
      matcherDetail,
      matcherIfName,
      matcherIfType,
      matcherIntfIntervalSeconds,
      matcherLinkFilter,
      matcherOspf3LsaRequestList,
      matcherOspf3LsaRetransmissionList,
      matcherOspf3SpfLog,
      matcherOspf3LsaLog,
      matcherSelfOriginate,
      noIntfAreaId,
      noIntfBfd,
      noIntfCost,
      noIntfDeadInterval,
      noIntfHelloInterval,
      noIntfMtuIgnore,
      noIntfNetworkType,
      noIntfPassive,
      noIntfPriority,
      noIntfRetransmitInterval,
      noIntfSecurity,
      noIntfTransmitDelay,
      nullKw,
      ospf3Config,
      passphraseKw,
      passphraseValMatcher,
      processShowIpv6OspfDatabaseArea,
      processShowIpv6OspfDatabaseAs,
      processShowIpv6OspfDatabaseLink,
      setForceSPF,
      setIntfAreaId,
      setIntfBfd,
      setIntfCost,
      setIntfDeadInterval,
      setIntfHelloInterval,
      setIntfMtuIgnore,
      setIntfNetworkType,
      setIntfPassive,
      setIntfPriority,
      setIntfRetransmitInterval,
      setIntfTransmitDelay,
      setSecurity,
      setupOspfv3InstanceMode,
      showIpv6OspfAcl,
      spiKw,
      spiValMatcher,
      type0DummyNode,
      type7KwMatcher,
      vrfExprFactory,
)
import ShowCommand

AF_STR_MAP = { 'ipv4' : 'afIPv4',
               'ipv6' : 'afIPv6' }

#-------------------------------------------------------------------------------
# IntfConfig for OSPF3 configuration, is created when one of its attributes
# is configured. 
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# Given address family returns the instance id. 

# Please refer to getAfFromInstanceId to get family from instance
# id defined in RoutingOspf3Cli.py
#-------------------------------------------------------------------------------

def getAfInstanceId( addrFamily='ipv6' ):
   if addrFamily == 'ipv4':
      afInstanceId = afIPv4InstanceID
   elif addrFamily == 'ipv6':
      afInstanceId = afIPv6InstanceID
   else:
      afInstanceId = afAllInstanceID
   return afInstanceId

#-------------------------------------------------------------------------------
# router ospfv3 general config mode. A new instance of this mode is created when
# the user enters "router ospfv3 general"
#-------------------------------------------------------------------------------
class RouterOspfv3GeneralMode( RoutingOspfv3GeneralMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPFv3 configuration that applies to all OSPFv3 instances in all VRFs'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new RouterOspfv3GeneralMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      RoutingOspfv3GeneralMode.__init__( self, 'ospfv3GeneralConfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# router ospfv3 config mode. A new instance of this mode is created when the
# user enters "router ospfv3"
#-------------------------------------------------------------------------------
class RouterOspfv3Mode( RoutingOspfv3Mode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPFv3 configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new RouterOspfv3Mode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      RoutingOspfv3Mode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterOspfv3AfMode( RoutingOspfv3AfMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, addrFamily ):
      # parent is an instance of RouterOspfv3Mode which inherits from
      # RoutingOspfv3Mode which contains vrfName
      instanceId = 0 if addrFamily == 'ipv6' else 64
      RoutingOspfv3AfMode.__init__( self, ( parent.vrfName, addrFamily,
                                            instanceId ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterOspfv3Ipv6AfMode( RouterOspfv3AfMode ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPFv3 address family configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RouterOspfv3AfMode.__init__(  self, parent, session, 'ipv6' )

class RouterOspfv3Ipv4AfMode( RouterOspfv3AfMode ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPFv3 address family configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RouterOspfv3AfMode.__init__(  self, parent, session, 'ipv4' )

# Any instance command that is specific to new style configuration
# will be handled by this modelet
# Example: passive-interface default
class RouterOspfv3MultiAfModelet( CliParser.Modelet, RouterOspf3Common ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

def gotoRouterOspfv3Mode( mode, args ):
   vrfName = args.get( 'VRF' )
   instance = setupOspfv3InstanceMode( mode, vrfName=vrfName, 
                                       afInstanceId=afAllInstanceID )
   if instance:
      vrfName = instance.vrfName
      childMode = mode.childMode( RouterOspfv3Mode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

def deleteRouterOspfv3Mode( mode, args ):
   vrfName = args.get( 'VRF' )
   cleanupOspf3InstanceConfig( mode, vrfName )
   IraVrfCli.removeAgentVrfEntry( vrfName, "Ospf3" )

def gotoRouterOspfv3AfMode( mode, args ):
   if setupOspfv3InstanceMode( mode, vrfName=mode.vrfName, 
                 afInstanceId=getAfInstanceId( args[ 'ADDR_FAMILY' ] ) ):
      if args[ 'ADDR_FAMILY' ] == 'ipv6':
         childMode = mode.childMode( RouterOspfv3Ipv6AfMode )
      else:
         childMode = mode.childMode( RouterOspfv3Ipv4AfMode )
      mode.session_.gotoChildMode( childMode )

def deleteRouterOspfv3AfMode( mode, args ):
   cleanupOspf3InstanceConfig( mode, mode.vrfName,   
               afInstanceId=getAfInstanceId( args[ 'ADDR_FAMILY' ] ) )

# Add command which need to be supported under 'router ospfv3' mode
RouterOspfv3Mode.addModelet( RouterOspfv3SharedModelet )
RouterOspfv3Mode.addModelet( RouterOspfv3SharedModelet2 )
RouterOspfv3Mode.addModelet( RouterOspfv3MultiAfModelet )
# Add command which need to be supported under 'address-family ipv6' sub-mode
RouterOspfv3Ipv6AfMode.addModelet( RouterOspfv3SharedModelet )
RouterOspfv3Ipv6AfMode.addModelet( RouterOspfv3AfSharedModelet )
RouterOspfv3Ipv6AfMode.addModelet( RouterOspfv3AfIpv6Modelet )
RouterOspfv3Ipv6AfMode.addModelet( RouterOspfv3MultiAfModelet )
# Add command which need to be supported under 'address-family ipv4' sub-mode
RouterOspfv3Ipv4AfMode.addModelet( RouterOspfv3SharedModelet )
RouterOspfv3Ipv4AfMode.addModelet( RouterOspfv3AfSharedModelet )
RouterOspfv3Ipv4AfMode.addModelet( RouterOspfv3AfIpv4Modelet )
RouterOspfv3Ipv4AfMode.addModelet( RouterOspfv3MultiAfModelet )

#-------------------------------------------------------------------------------
# "[no|default] router ospfv3 [vrf <vrf-name>]" command in "config" mode.
#------------------------------------------------------------------------------- 
ospfv3Kw = CliMatcher.KeywordMatcher( 'ospfv3',
                                      helpdesc='OSPF Version 3' )

class RouterOspfv3ConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'router ospfv3 [ VRF ]'
   noOrDefaultSyntax = syntax
   data = {
      'router': routerMatcherForConfig,
      'ospfv3': ospfv3Kw,
      'VRF': vrfExprFactory,
   }
   handler = gotoRouterOspfv3Mode
   noOrDefaultHandler = deleteRouterOspfv3Mode

BasicCli.GlobalConfigMode.addCommandClass( RouterOspfv3ConfigCmd )

#-------------------------------------------------------------------------------
# "[no] address-family [ipv4|ipv6]"
#-------------------------------------------------------------------------------
class AFExpression( CliCommand.CliExpression ):
   expression = '( ipv4 | ipv6 )'
   data = {
           'ipv4' : 'IPv4 related',
           'ipv6' : 'IPv6 related',
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ipv4' in args:
         args[ 'ADDR_FAMILY' ] = args.pop( 'ipv4' )
      elif 'ipv6' in args:
         args[ 'ADDR_FAMILY' ] = args.pop( 'ipv6' )

class Ospf3AddrFamilyConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family ADDR_FAMILY'
   noOrDefaultSyntax = syntax
   data = {
      'address-family': 'Enable address family and enter its config mode',
      'ADDR_FAMILY': AFExpression,
   }
   handler = gotoRouterOspfv3AfMode
   noOrDefaultHandler = deleteRouterOspfv3AfMode

RouterOspfv3Mode.addCommandClass( Ospf3AddrFamilyConfigCmd )

class AddressFamilyExpression( CliCommand.CliExpression ):
   expression = '( ipv4 | ipv6 )'
   data = {
         'ipv4' : 'Specify parameters for IPv4',
         'ipv6' : 'Specify parameters for IPv6',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ipv4' in args:
         args.pop( 'ipv4' )
         args[ 'AF_INSTANCE_ID' ] = afIPv4InstanceID
      elif 'ipv6' in args:
         args.pop( 'ipv6' )
         args[ 'AF_INSTANCE_ID' ] = afIPv6InstanceID
      else:
         args[ 'AF_INSTANCE_ID' ] = afAllInstanceID


#-------------------------------------------------------------------------------
# "[no|default] router ospfv3 general" command in "config" mode.
#------------------------------------------------------------------------------- 
class RouterOspfv3GeneralCmd( CliCommand.CliCommandClass ):
   syntax = 'router ospfv3 general'
   noOrDefaultSyntax = syntax
   data = {
      'router' : routerMatcherForConfig,
      'ospfv3' : ospfv3Kw,
      'general' : 'General OSPFv3 configuration',
   }

   @staticmethod
   def handler( mode, args ):
      config = ospf3Config()
      config.generalConfig = ( 'ospfv3GeneralConfig', )
      childMode = mode.childMode( RouterOspfv3GeneralMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = ospf3Config()
      config.generalConfig = None

BasicCli.GlobalConfigMode.addCommandClass( RouterOspfv3GeneralCmd )

#-------------------------------------------------------------------------------
# "[no|default] qos dscp VALUE" command, in "router ospfv3 general" mode
#-------------------------------------------------------------------------------
class Ospfv3QosDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos dscp VALUE'
   noOrDefaultSyntax = 'qos dscp ...'
   data = {
      'qos' : 'Set QoS parameters for CPU generated OSPFv3 traffic',
      'dscp' : 'Set DSCP value for OSPFv3 traffic',
      'VALUE' : CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP value' ),
   }

   @staticmethod
   def handler( mode, args ):
      cfg = ospf3Config()
      cfg.generalConfig.dscp = args.get( 'VALUE', cfg.generalConfig.dscpDefault )

   noOrDefaultHandler = handler

RouterOspfv3GeneralMode.addCommandClass( Ospfv3QosDscpCmd )

#-------------------------------------------------------------------------------
# "[no|default] passive-interface <interface>"
#-------------------------------------------------------------------------------
# New style cli accepts only a default, per interface config is under the
# interface configuration
class Ospf3PassiveIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'passive-interface default'
   noOrDefaultSyntax = syntax
   data = { 'passive-interface': matcherPassiveIntf,
            'default': matcherPassiveDefault,
          }
   handler = RouterOspfv3MultiAfModelet.setPassiveIntf
   noOrDefaultHandler = RouterOspfv3MultiAfModelet.noPassiveIntf

RouterOspfv3MultiAfModelet.addCommandClass( Ospf3PassiveIntfCmd )

#-------------------------------------------------------------------------------
# The interface config for ospfv3 is done directly under the intfConfigMode, 
# hence mode ( or mode attributes ) cannot be used to differentiate the cli
# styles unlike the ospfv3 instance config. 
# For interface configuration the cli handlers are the same for all 
# config styles ( new / old/ addrfamily / global ), the context is 
# determined based on the below logic:
#  - afInstanceId=None   -   Old style interface configuration
#  - afInstanceId=0      -   Intf IPv6 address family configuration
#  - afInstanceId=64     -   Intf IPv4 address family configuration
#  - afInstanceId=255    -   Global Intf configuration
#
# Decorator function is used to get the correct afInstanceId to be passed
# to the intf config cli handler based on the command entered
#-------------------------------------------------------------------------------
def intfInstanceDecorator( func ):
   def afIterator( mode, **kargs ):
      afInstanceId = afAllInstanceID
      if 'addrFamily' in kargs:
         afInstanceId = getAfInstanceId( kargs.pop( 'addrFamily' ) )

      kargs[ 'afInstanceId' ] = afInstanceId
      func( mode, **kargs )

   return afIterator

#-------------------------------------------------------------------------------
# "clear ospfv3 [ipv4|ipv6] neighbor [vrf vrfName] command, in global mode.
#-------------------------------------------------------------------------------
class ClearOspfv3NeighborCmd( CliCommand.CliCommandClass ):
   syntax = '''clear ospfv3 [ AF_INSTANCE_ID ] neighbor ( * | ADDRESS | INTERFACE ) 
               [ VRF ] [ show ] '''
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ospfv3': ospfv3Kw,
      'AF_INSTANCE_ID': AddressFamilyExpression,
      'neighbor': matcherNeighbor,
      '*': matcherOptResetAllNeighbors,
      'ADDRESS': IpAddrMatcher.IpAddrMatcher( helpdesc='Neighbor address' ),
      'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
      'VRF': vrfExprFactory,
      'show': matcherShow,
   }
   handler = doClearIpv6Ospf

BasicCli.EnableMode.addCommandClass( ClearOspfv3NeighborCmd )

#-------------------------------------------------------------------------------
# "clear ospfv3 access-list counters"
#-------------------------------------------------------------------------------
class ClearOspfv3Cmd( CliCommand.CliCommandClass ):
   syntax = 'clear ospfv3 access-list counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ospfv3': ospfv3Kw,
      'access-list': AclCli.accessListMatcher,
      'counters': matcherCounters,
   }
   handler = doClearIpv6Ospf

BasicCli.EnableMode.addCommandClass( ClearOspfv3Cmd )

#-------------------------------------------------------------------------------
# "clear ospfv3 [ipv4|ipv6] force-spf [vrf vrfName] command, in global mode.
#-------------------------------------------------------------------------------
class ClearOspfv3ForceSpfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ospfv3 [ AF_INSTANCE_ID ] force-spf [ VRF ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ospfv3': ospfv3Kw,
      'AF_INSTANCE_ID': AddressFamilyExpression,
      'force-spf': 'force a spf calculation',
      'VRF': vrfExprFactory,
   }
   handler = setForceSPF

BasicCli.EnableMode.addCommandClass( ClearOspfv3ForceSpfCmd )

#-------------------------------------------------------------------------------
#               Interface Config Commands, in config-if mode
#-------------------------------------------------------------------------------
   
#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] priority <number-value>"
#-------------------------------------------------------------------------------
class Ospfv3PriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] priority PRIORITY'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] priority ...'
   data = {
         'ospfv3': ospfv3Kw,
         'AF_INSTANCE_ID': AddressFamilyExpression,
         'priority': priorityKw,
         'PRIORITY': matcherPriority,
          }
   handler = setIntfPriority
   noOrDefaultHandler = noIntfPriority

intfModelet.addCommandClass( Ospfv3PriorityCmd )


afValNodeHidden = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
            'ipv4': 'Specify parameters for IPv4',
            'ipv6': 'Specify parameters for IPv6',
      } ),
      hidden=True )

#-------------------------------------------------------------------------------
# [ no | default ] ( ospfv3 [ AF ]
#                     ( ( authentication ipsec spi INDEX )
#                     | ( encryption ipsec spi INDEX esp null ) )
#                     HASH
#                     ( ( ( [ 0 | AUTH7 ] HEXKEY )
#                     | ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) )
#                | ( ospfv3 [ AF ] encryption ipsec spi INDEX esp CIPHER
#                     ( ( 0 | ENCR7 ) KEY HASH [ 0 | AUTH7 ] HEXKEY ) )
#                     | ( [ HASH ]  passphrase [ 0 | PASS7 ] PHRASE ) ) )
#-------------------------------------------------------------------------------
# AddressFamily command for authentication is hidden. RFC4552, section 8 states that
# SA should be per-link. But we introduced address family auth configuration while
# adding OSPFv3 AF support. Hence to maintain backward compatibility, the command is
# hidden.
class IntfOspfv3Auth( CliCommand.CliCommandClass ):
   syntax = ( '( ospfv3 [ AF ] '
                 '( ( authentication ipsec spi INDEX ) '
                 '| ( encryption ipsec spi INDEX esp null ) ) '
                 'HASH '
                 '( ( ( [ 0 | AUTH7 ] HEXKEY ) '
                 '| ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) ) '
            '| ( ospfv3 [ AF ] encryption ipsec spi INDEX esp CIPHER '
                 '( ( ( 0 | ENCR7 ) KEY HASH [ 0 | AUTH7 ] HEXKEY ) ) '
                 '| ( [ HASH ]  passphrase [ 0 | PASS7 ] PHRASE ) ) )' )
   noOrDefaultSyntax = 'ospfv3 [ AF ] ( authentication | encryption ) ...'
   data = {
         'ospfv3': ospfv3Kw,
         'AF': afValNodeHidden,
         'authentication': intfAuthenticationKw,
         'ipsec': ipsecKw,
         'spi': spiKw,
         'INDEX': spiValMatcher,
         'encryption': intfEncryptionKw,
         'esp': espKw,
         'null': nullKw,
         'HASH': hashValMatcher,
         '0': type0DummyNode,
         'AUTH7': type7KwMatcher,
         'HEXKEY': authKeyValMatcher,
         'passphrase': passphraseKw,
         'PASS7': type7KwMatcher,
         'PHRASE': passphraseValMatcher,
         'CIPHER': cipherValMatcher,
         'ENCR7': type7KwMatcher,
         'KEY': keyValNode,
   }

   @staticmethod
   def handler( mode, args ):
      af = args.get( 'AF' )
      afInstanceId = getAfInstanceId( af ) if af else afAllInstanceID
      intfInstanceConfig = _getIntfConfigForAttr( mode, 'securityEnabled', 
                                                  afInstanceId )
      if intfInstanceConfig:
         proto = socket.IPPROTO_ESP if 'esp' in args else socket.IPPROTO_AH
         setSecurity( mode, mode.intf.name, intfInstanceConfig, args, proto )
         _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      af = args.get( 'AF' )
      afInstanceId = getAfInstanceId( af ) if af else afAllInstanceID
      noIntfSecurity( mode, afInstanceId )

CliPlugin.IntfCli.IntfConfigMode.addCommandClass( IntfOspfv3Auth )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] hello-interval <seconds>"
#-------------------------------------------------------------------------------
class Ospfv3HelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] hello-interval HELLO_INTERVAL'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] hello-interval ...'
   data = {
         'ospfv3': ospfv3Kw,
         'AF_INSTANCE_ID': AddressFamilyExpression,
         'hello-interval': helloIntervalKw,
         'HELLO_INTERVAL': matcherIntfIntervalSeconds,
         }

   handler = setIntfHelloInterval
   noOrDefaultHandler = noIntfHelloInterval

intfModelet.addCommandClass( Ospfv3HelloIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] cost <interface-cost>"
#-------------------------------------------------------------------------------
class Ospfv3CostCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] cost COST'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] cost ...'
   data = {
         'ospfv3': ospfv3Kw,
         'AF_INSTANCE_ID': AddressFamilyExpression,
         'cost' : costKw,
         'COST' : matcherCost,
         }
   handler = setIntfCost
   noOrDefaultHandler = noIntfCost

intfModelet.addCommandClass( Ospfv3CostCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] bfd"
#-------------------------------------------------------------------------------
class Ospfv3BfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] bfd'
   noSyntax = defaultSyntax = syntax
   data = {
         'ospfv3': ospfv3Kw,
         'AF_INSTANCE_ID': AddressFamilyExpression,
         'bfd': bfdKw,
         }
   handler = setIntfBfd
   noHandler = noIntfBfd
   defaultHandler = defaultIntfBfd

intfModelet.addCommandClass( Ospfv3BfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] dead-interval <seconds>"
#-------------------------------------------------------------------------------
class Ospfv3DeadIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] dead-interval DEAD_INTERVAL'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] dead-interval ...'
   data = {
         'ospfv3': ospfv3Kw,
         'AF_INSTANCE_ID': AddressFamilyExpression,
         'dead-interval': deadIntervalKw,
         'DEAD_INTERVAL': matcherIntfIntervalSeconds,
         }

   handler = setIntfDeadInterval
   noOrDefaultHandler = noIntfDeadInterval

intfModelet.addCommandClass( Ospfv3DeadIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] mtu-ignore"
#-------------------------------------------------------------------------------
class Ospfv3MtuIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] mtu-ignore'
   noOrDefaultSyntax = syntax
   data = {
            'ospfv3': ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'mtu-ignore': mtuIgnoreKw,
          }
   handler = setIntfMtuIgnore
   noOrDefaultHandler = noIntfMtuIgnore

intfModelet.addCommandClass( Ospfv3MtuIgnoreCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] retransmit-interval <seconds>"
#-------------------------------------------------------------------------------
class Ospfv3RetransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] retransmit-interval RETRANSMIT_INTERVAL'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] retransmit-interval ...'
   data = {
            'ospfv3' : ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'retransmit-interval': retransmitIntervalKw,
            'RETRANSMIT_INTERVAL': matcherIntfIntervalSeconds,
           }
   handler = setIntfRetransmitInterval
   noOrDefaultHandler = noIntfRetransmitInterval

intfModelet.addCommandClass( Ospfv3RetransmitIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] transmit-delay <seconds>"
#-------------------------------------------------------------------------------
class Ospfv3TransmitDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] transmit-delay TRANSMIT_DELAY'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] transmit-delay ... '
   data = {
            'ospfv3': ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'transmit-delay': transmitDelayKw,
            'TRANSMIT_DELAY': matcherIntfIntervalSeconds,
           }
   handler = setIntfTransmitDelay
   noOrDefaultHandler = noIntfTransmitDelay

intfModelet.addCommandClass( Ospfv3TransmitDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] network point-to-point"
#-------------------------------------------------------------------------------
class Ospfv3NetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] network NETWORK_TYPE'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] network ...'
   data = {
            'ospfv3': ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'network': networkTypeKw,
            'NETWORK_TYPE': networkTypesMatcher,
           }
   handler = setIntfNetworkType
   noOrDefaultHandler = noIntfNetworkType

intfModelet.addCommandClass( Ospfv3NetworkCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 [ipv4|ipv6] passive-interface"
#-------------------------------------------------------------------------------
class Ospf3PassiveInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'ospfv3 [ AF_INSTANCE_ID ] passive-interface'
   noOrDefaultSyntax = 'ospfv3 [ AF_INSTANCE_ID ] passive-interface ...'
   data = { 
            'ospfv3': ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'passive-interface': matcherPassiveIntf,
          }
   handler = setIntfPassive
   noOrDefaultHandler = noIntfPassive
 
intfModelet.addCommandClass( Ospf3PassiveInterfaceCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 <ipv4|ipv6> area <area-id>"
#-------------------------------------------------------------------------------
class Ospfv3AreaCmd( CliCommand.CliCommandClass ):
   '''
   Here noOrDefaultSyntax would be 'ospfv3 <ipv4|ipv6> [ area ... ]' instead of 
   'ospfv3 <ipv4|ipv6> ...' otherwise it matches all other interface commands that 
   has 'ospfv3 <ipv4|ipv6>' as a part of syntax like 
   "no ospfv3 (ipv4|ipv6) passive-interface" 
   '''
   syntax = 'ospfv3 AF_INSTANCE_ID area AREA_ID'
   noOrDefaultSyntax = 'ospfv3 AF_INSTANCE_ID [ area ... ]'
   data = {
            'ospfv3': ospfv3Kw,
            'AF_INSTANCE_ID': AddressFamilyExpression,
            'area': areaKw,
            'AREA_ID': AreaIdExpression,
           }
   handler = setIntfAreaId
   noOrDefaultHandler = noIntfAreaId

intfModelet.addCommandClass( Ospfv3AreaCmd )

class Ospf3MultiAfVrfCapiGenerator( object ):
   '''
   Decorator class that can be used to create a top level
   CAPI model that contains a generator dict of component model instances

   Example usage:
   To capize the 'show ospfv3' command with 'vrf [VRF|default|all]'
   add the 'Ospf3MultiAfVrfCapiGenerator' decorator to the doShowOspfv3 function
   and pass the @vrfModelDictType and @afModelDictType arguments to the decorator.

   @vrfModelDictType is a model type generated by invoking generateVrfCliModel
   with relevent component model type. This model contains a dict of VRFs.

   @afModelDictType is the model type generated by invoking
   generateOspf3AfCliModel. This model contains a dict of AFs per VRF.

   @Ospf3MultiAfVrfCapiGenerator( vrfModelDictType, afModelDictType )
   def doShowOspfv3( mode, instanceId=None, vrfName=None ):
   '''
   def __init__( self, vrfModelDictType, afModelDictType, firstInstance=False ):
      self.vrfModelDictType = vrfModelDictType
      self.afModelDictType = afModelDictType
      self.firstInstance = firstInstance

   def __call__( self, func ):
      '''return a decorated function that wraps @func
         arguments to the wrapped function are passed as is'''
      def vrfExecCmdFunc( *args, **kwargs ):
         '''
         first argument in the args list must be of type 'CliParser.Mode'
         '''
         assert isinstance( args[ 0 ], CliParser.Mode )
         mode = args[ 0 ]

         vrfName = kwargs[ 'args' ].get( 'VRF' )
         vrfs = getVrfToInstanceIdsDict( mode, vrfName )
         if not vrfs: # error case
            vrfs = []

         def afDictFunc( vrf, firstVrf ):
            instIdFilter = ''
            firstInst = True
            afFilter = kwargs[ 'args' ].get( 'ADDR_FAMILY' )
            if afFilter is not None:
               instIdFilter = AF_STR_MAP[ afFilter ]
            instIdList = vrfs[ vrf ]
            for instId in instIdList:
               if instId == 'allAf':
                  continue
               if instIdFilter and instId != instIdFilter:
                  continue
               kwargs[ 'args' ][ 'INSTANCE_ID' ] = instId
               kwargs[ 'args' ][ 'VRF' ] = vrf

               if self.firstInstance and firstInst and firstVrf:
                  # Incase the command would like to print a legend
                  # or header just once across multiple instances
                  kwargs[ 'args' ][ 'firstInstance' ] = True
               else:
                  kwargs[ 'args' ].pop( 'firstInstance', None )

               afModel = func( *args, **kwargs )
               firstInst = False
               if afModel is None:
                  continue
               af = "ipv4" if instId == 'afIPv4' else "ipv6"
               yield af, afModel

         def vrfDictFunc():
            firstVrf = True
            for vrf in vrfs:
               afDictModel = self.afModelDictType()
               afDictModel.addressFamily = afDictFunc( vrf, firstVrf )
               firstVrf = False
               yield vrf, afDictModel

         vrfModel = self.vrfModelDictType()
         vrfModel.vrfs = vrfDictFunc()
         return vrfModel

      return vrfExecCmdFunc

def generateOspf3AfCliModel( cliModel ):
   addressFamily = GeneratorDict( keyType=str, valueType=cliModel,
                   help='OSPFv3 instance information indexed by address family' )

   def render( self ):
      for af, model in self.addressFamily:
         model.renderModel( af )

   className = 'AddressFamily%s' % cliModel.__name__
   classBody = { 'addressFamily' :  addressFamily,
                 'render' : render }

   return type( className, ( Model, ), classBody )

#-------------------------------------------------------------------------------
# "show ospfv3 [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
matcherOspfv3AfterShow = CliMatcher.KeywordMatcher( 'ospfv3',
                                                    helpdesc='OSPFv3 information' )

afdesc = "OSPFv3 per AF instance information for all VRFs"
Ospfv3ShowAfDictModel = generateOspf3AfCliModel(
                        Ospf3CliModels.Ospf3MultiAfShowInstModel )
Ospfv3ShowAfVrfDictModel = generateVrfCliModel( Ospfv3ShowAfDictModel,
                                                afdesc )

@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowAfVrfDictModel,
                               afModelDictType=Ospfv3ShowAfDictModel )
def doShowOspfv3( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF3_INSTANCE_SUMMARY'
   args = { 'instance': AF_INSTANCE_ID_MAP[ instanceId ] }
  
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfShowInstModel,
                                   command, args, clientName='OSPF3',
                                   l3Config=getl3Config() )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      return getDefaultInstanceModel( Ospf3CliModels.Ospf3MultiAfShowInstModel(), 
                                      afInstanceId=args[ 'instance' ],
                                      vrfName=vrfName ) 

class ShowOspfv3Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 [ ADDR_FAMILY ] [ VRF ]'
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3
   cliModel = Ospfv3ShowAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3Cmd )

#-------------------------------------------------------------------------------
# "show ospfv3 neighbor [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
afNeighDesc = 'OSPFv3 per AF neighbor information for all VRFs'
Ospfv3ShowNeighAfDictModel = generateOspf3AfCliModel(
                             Ospf3CliModels.Ospf3MultiAfNeighborsInstance )
Ospfv3ShowNeighAfVrfDictModel = generateVrfCliModel( Ospfv3ShowNeighAfDictModel,
                                                     afNeighDesc )

# pylint: disable-msg=W0212
@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowNeighAfVrfDictModel, 
                               afModelDictType=Ospfv3ShowNeighAfDictModel ) 
def doShowOspfv3Neighbor( mode, args ):
   vrfName = args.get( 'VRF' )
   intf = args.get( 'INTF' )
   summary = args.get( 'summary' )
   state = args.get( 'STATE' )
   neighborId = args.get( 'NEIGHBOR_ID' )
   instanceId = args.get( 'INSTANCE_ID' )
   command = 'MIO_DGET_OSPF3_NEIGHBORS'
   argsCapi = { 'instance': AF_INSTANCE_ID_MAP[ instanceId ] }

   if intf:
      if not intf.lookup():
         mode.addError( "Interface %s not present" % intf.name )
         return None
      argsCapi[ 'interfaceName' ] = intf.status().deviceName
   if neighborId:
      argsCapi[ 'routerId' ] = neighborId
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode,
                                   Ospf3CliModels.Ospf3MultiAfNeighborsInstance,
                                   command, argsCapi, clientName='OSPF3',
                                   l3Config=getl3Config() )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3MultiAfNeighborsInstance()

   if summary:   #cmd: show ospfv3 neighbor summary
      result._summary = True
   elif state:   #cmd: show ospfv3 neighbor state abc
      result._state = state
   return result

class ShowOspfv3NeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ospfv3 [ ADDR_FAMILY ] neighbor [ INTF ] [ NEIGHBOR_ID ] '
              '[ summary ] [ state STATE ] [ VRF ]' )
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'neighbor': 'Protocol neighbors',
            'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'NEIGHBOR_ID': matcherNeighborId, 
            'summary': 'Summary of neighbor information',
            'state': 'Only show neighbors of the specific state',
            'STATE': CliMatcher.EnumMatcher( Ospf3CliModels.ADJACENCY_STATE_MAP ), 
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3Neighbor
   cliModel = Ospfv3ShowNeighAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3NeighborCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 interface [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
afIntfDesc = 'OSPFv3 per AF interface information for all VRFs'
Ospfv3ShowIntfAfDictModel = generateOspf3AfCliModel(
                            Ospf3CliModels.Ospf3MultiAfInterface )
Ospfv3ShowIntfAfVrfDictModel = generateVrfCliModel( Ospfv3ShowIntfAfDictModel,
                                                    afIntfDesc )

@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowIntfAfVrfDictModel, 
                               afModelDictType=Ospfv3ShowIntfAfDictModel ) 
def doShowOspfv3Interface( mode, args ):
   vrfName = args.get( 'VRF' )
   intf = args.get( 'INTF' )
   instanceId = args.get( 'INSTANCE_ID' )
   cmd = 'MIO_DGET_OSPF3_INTERFACE'
   argsCapi = { 'instance': AF_INSTANCE_ID_MAP[ instanceId ] }

   if intf is not None:
      if not intf.lookup():
         mode.addError( "Interface %s not present" % intf.name )
         return None
      argsCapi[ 'ifname' ] = intf.status().deviceName
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfInterface, cmd,
                                 argsCapi, clientName='OSPF3', 
                                 l3Config=getl3Config() )
   
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3MultiAfInterface()

class ShowOspfv3InterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 [ ADDR_FAMILY ] interface [ INTF ] [ VRF ]'
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'interface': 'Interface-specific details',
            'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3Interface
   cliModel = Ospfv3ShowIntfAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3InterfaceCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 border-routers [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
desc = "OSPFv3 per AF border routers information for all VRFs"
Ospfv3BorderRoutersAfDictModel = generateOspf3AfCliModel(
                             Ospf3CliModels.Ospf3MultiAfBorderRoutersInstance )
Ospfv3BorderRoutersAfVrfDictModel = generateVrfCliModel(
                                          Ospfv3BorderRoutersAfDictModel,
                                          desc )
@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3BorderRoutersAfVrfDictModel, 
                               afModelDictType=Ospfv3BorderRoutersAfDictModel ) 
def doShowOspfv3BorderRouters( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF3_BORDER_ROUTERS'
   args = { 'instance': AF_INSTANCE_ID_MAP[ instanceId ] }

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, 
                                   Ospf3CliModels.Ospf3MultiAfBorderRoutersInstance,
                                   command, args, clientName='OSPF3',
                                   l3Config=getl3Config() )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      return  Ospf3CliModels.Ospf3MultiAfBorderRoutersInstance()

class ShowOspfv3BorderRoutersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 [ ADDR_FAMILY ] border-routers [ VRF ]'
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'border-routers': matcherBorderRouters,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3BorderRouters
   cliModel = Ospfv3BorderRoutersAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3BorderRoutersCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 database [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
afDatabaseDesc = 'OSPFv3 per AF database information for all VRFs'
Ospfv3ShowDatabaseAfDictModel = generateOspf3AfCliModel(
                                Ospf3CliModels.Ospf3MultiAfDatabaseInstance )
Ospfv3ShowDatabseAfVrfDictModel = generateVrfCliModel(
                                  Ospfv3ShowDatabaseAfDictModel,
                                  afDatabaseDesc )
@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowDatabseAfVrfDictModel,
                               afModelDictType=Ospfv3ShowDatabaseAfDictModel,
                               firstInstance=True )
def doShowOspfv3Database( mode, args ):
   spec = None
   if any( i in args for i in ( 'area', 'as', 'link' ) ):
      spec = {}
      spec[ 'filter' ] = args.get( 'area' ) or args.get( 'as' ) or args.get( 'link' )

   if 'area' in args:
      spec[ 'areaSpecType' ] = args.get( 'backbone' ) or args.get( 'IP_ADDR' )
      spec[ 'lsaType' ] = args.get( 'LSA_TYPE' )

   if 'as' in args:
      spec[ 'external' ] = args.get( 'external' )

   if 'link' in args:
      if 'if-name' in args:
         spec[ 'linkSpec' ] = {'ifClassifyType': 'if-name', 
                               'ifName': args[ 'INTFS' ]}
      elif 'if-type' in args:
         spec[ 'linkSpec' ] = {'ifClassifyType': 'if-type', 
                               'ifType': args[ 'IF_TYPE' ]}      
   lsId = args.get( 'LINK_STATE_ID' )
   origRtr = None
   if any( i in args for i in ( 'adv-router', 'self-originate' ) ):
      origRtr = {}
      origRtr[ 'rtrType' ] = args.get( 'adv-router' ) or args.get( 'self-originate' )
      if 'adv-router' in args:
         origRtr[ 'rtrId' ] = args[ 'ADV_RTR' ]
   detail = args.get( 'detail' )
   instanceId = args.get( 'INSTANCE_ID' ) 
   vrfName = args.get( 'VRF' )
   firstInstance = args.get( 'firstInstance', False )
   # Use PyRibDami approach, if there are no filters / queries
   noDatabaseQuery = True
   detailOutput = False
   cmd = "MIO_DGET_OSPF3_DATABASE"
   argsCapi = { 'instance': AF_INSTANCE_ID_MAP[ instanceId ] }
   if vrfName != DEFAULT_VRF:
      argsCapi [ 'vrfName' ] = vrfName

   filterProcessor = { 'area': processShowIpv6OspfDatabaseArea,
                       'as'  : processShowIpv6OspfDatabaseAs,
                       'link': processShowIpv6OspfDatabaseLink,
                     }

   filterQueryType = { 'area': 3,
                       'as'  : 2,
                       'link': 1,
                     }

   if spec and spec.get( 'filter' ):
      noDatabaseQuery = False

      # Set the query type to area / link / as
      argsCapi[ 'query-type' ] = filterQueryType[ spec[ 'filter' ] ]
      if filterProcessor.get( spec[ 'filter' ] ):
         # Call the corresponding function to process the cmd
         filterProcessor[ spec[ 'filter' ] ]( argsCapi, spec )

   if lsId:
      noDatabaseQuery = False
      argsCapi [ 'lsid' ] = lsId

   if origRtr and origRtr.get( 'rtrType' ):
      noDatabaseQuery = False
      rtrId = None
      if origRtr[ 'rtrType' ] == 'adv-router':
         if origRtr.get( 'rtrId' ):
            rtrId = "%s" % ( origRtr[ 'rtrId' ] )
         argsCapi[ 'router id' ] = rtrId
      else:
         argsCapi[ 'self' ] = True

   if detail:
      detailOutput = True
      argsCapi ['print_type'] = True

   # Print the database legend only once for the first query 
   if firstInstance and mode.session_.outputFormat_ == "text":
      Ospf3CliModels.printOspf3LegendInformation()
      sys.stdout.flush()
      
   if noDatabaseQuery and mode.session_.outputFormat_ == "text":
      argsCapi[ 'format' ] = "2"
      l3Config = getl3Config()
      showRibDamiCommand( sys.stdout.fileno(), cmd, argsCapi, clientName='OSPF3',
                          l3Config=l3Config )
   else:
      try:
         result = showRibCapiCommand( mode, 
                                      Ospf3CliModels.Ospf3MultiAfDatabaseInstance,
                                      cmd, args=argsCapi, clientName='OSPF3',
                                      l3Config=getl3Config() )
      except EmptyResponseException:
         result = Ospf3CliModels.Ospf3MultiAfDatabaseInstance()
         getDefaultModelValues( result, afInstanceId=argsCapi[ 'instance' ],
                                vrfName=vrfName )
                  
      if detailOutput:
         result._detailsPresent = True

      return result
   return None

class ShowOspfv3DatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ospfv3 [ ADDR_FAMILY ] database 
               [ ( area ( backbone | IP_ADDR ) [ LSA_TYPE ] ) | 
                 ( as [ external ] ) | 
                 ( link [ ( ( if-name INTFS ) | ( if-type IF_TYPE ) ) ] ) 
               ]  
               [ LINK_STATE_ID ] 
               [ ( ( adv-router ADV_RTR ) | self-originate ) ] 
               [ detail ] 
               [ VRF ]'''
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'database': matcherDatabase,
            'area': matcherAreaFilter,
            'backbone': matcherBackbone,
            'IP_ADDR': matcherAreaIdFilter,
            'LSA_TYPE': CliMatcher.EnumMatcher( lsaType ),
            'as': matcherAsFilter,
            'external': matcherAsExternal,
            'link': matcherLinkFilter,
            'if-name': matcherIfName,
            'INTFS': IntfRange.IntfRangeMatcher(),
            'if-type': matcherIfType,
            'IF_TYPE': CliMatcher.EnumMatcher( ifType ),
            'LINK_STATE_ID': IpAddrMatcher.IpAddrMatcher( 'Link state ID' ),
            'adv-router': matcherAdvRouter,
            'ADV_RTR': IpAddrMatcher.IpAddrMatcher( 'Advertising Router' ),
            'self-originate': matcherSelfOriginate,
            'detail': matcherDetail,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3Database
   cliModel = Ospfv3ShowDatabseAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3DatabaseCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 [ipv4 | ipv6] lsa-log [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
lsaLogDesc = 'OSPFv3 per AF Lsa Log for all VRFs'
Ospfv3ShowLsaLogAfDictModel = generateOspf3AfCliModel(
                             Ospf3CliModels.Ospf3MultiAfLsaLogInst )
Ospfv3ShowLsaLogAfVrfDictModel = generateVrfCliModel( Ospfv3ShowLsaLogAfDictModel,
                                                      lsaLogDesc )

@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowLsaLogAfVrfDictModel, 
                               afModelDictType=Ospfv3ShowLsaLogAfDictModel ) 
def doShowOspfv3LsaLog( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF3_LSA_LOGS'
   argsCapi = {'instanceId': AF_INSTANCE_ID_MAP[ instanceId ] }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfLsaLogInst, 
                                   command, argsCapi, clientName='OSPF3',
                                   l3Config=getl3Config() )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      result = Ospf3CliModels.Ospf3MultiAfLsaLogInst()
      getDefaultModelValues( result, afInstanceId=argsCapi[ 'instanceId' ],
                             vrfName=vrfName )    
      return result

class ShowOspfv3LsaLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 [ ADDR_FAMILY ] lsa-log [ VRF ]'
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'lsa-log': matcherOspf3LsaLog,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3LsaLog
   cliModel = Ospfv3ShowLsaLogAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3LsaLogCmd )

#-------------------------------------------------------------------------------
# show ospfv3 access-list
#-------------------------------------------------------------------------------
class ShowOspfv3AccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 access-list [ ACL_NAME ]'
   data = {
         'ospfv3'      : matcherOspfv3AfterShow,
         'access-list' : AclCli.accessListMatcher,
         'ACL_NAME'    : AclCli.ip6AclNameExpression,
          }
   handler = showIpv6OspfAcl
   cliModel = AclCliModel.AllAclList

BasicCli.addShowCommandClass( ShowOspfv3AccessListCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 [ipv4|ipv6] database database-summary [ vrf vrf-name ] "
#-------------------------------------------------------------------------------
afDatabaseSumDesc = 'OSPFv3 per AF database summary information for all VRFs'

Ospfv3ShowDatabaseSumAfDictModel = generateOspf3AfCliModel(
                                Ospf3CliModels.Ospf3MultiAfDatabaseSummary )

Ospfv3ShowDatabseSumAfVrfDictModel = generateVrfCliModel(
                                  Ospfv3ShowDatabaseSumAfDictModel,
                                  afDatabaseSumDesc )

@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowDatabseSumAfVrfDictModel,
                               afModelDictType=Ospfv3ShowDatabaseSumAfDictModel,
                               firstInstance=True )

def doShowOspfv3DatabaseSummary( mode, args ):
   vrfName = args.get( 'VRF' )
   instanceId = args.get( 'INSTANCE_ID' )
   cmd = "MIO_DGET_OSPF3_DATABASE_SUM"

   # parse instance parameter, exit if there is no instance
   instance = AF_INSTANCE_ID_MAP[ instanceId ]
   if instance is None:
      return None

   # put required parameters in argument list
   argsCapi = { 'instance': instance }

   # parse optional vrf parameter
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   # parse optional area filter
   # parse optional area filter
   areaId = None
   if 'area' in args:
      if 'backbone' in args:
         areaId = '0.0.0.0'
      else:
         areaId = args[ 'IP_ADDR' ]
      argsCapi[ 'area' ] = areaId

   # Call capi command
   try:
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfDatabaseSummary,
                                   cmd, args=argsCapi, clientName='OSPF3',
                                   l3Config=getl3Config() )

   except EmptyResponseException:
      result = Ospf3CliModels.Ospf3MultiAfDatabaseSummary()

   # Save values needed to print headers 
   if result:
      result.instanceIdIs( int( instance ) )
      if vrfName != None:
         result.vrfIs( vrfName )
      if areaId != None:
         result.areaIdIs( areaId ) 

   return result

class ShowOspfv3DatabaseSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ospfv3 [ ADDR_FAMILY ] database database-summary ' 
              '[ area ( backbone | IP_ADDR ) ] [ VRF ]' )
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'database': matcherDatabase,
            'database-summary': matcherDatabaseSummary,
            'area': matcherAreaFilter,
            'backbone': matcherBackbone,
            'IP_ADDR': matcherAreaIdFilter,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3DatabaseSummary
   cliModel = Ospfv3ShowDatabseSumAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3DatabaseSummaryCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 [ipv4 | ipv6] retransmission-list [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
retransmissionListDesc = "OSPFv3 per AF LSA retransmission information for all VRFs"
Ospf3LsaRetransmissionAfDictModel = generateOspf3AfCliModel(
   Ospf3CliModels.Ospf3MultiAfLsaRetransmissionList )
Ospf3LsaRetransmissionAfVrfDictModel = generateVrfCliModel(
   Ospf3LsaRetransmissionAfDictModel, retransmissionListDesc )
@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospf3LsaRetransmissionAfVrfDictModel,
                               afModelDictType=Ospf3LsaRetransmissionAfDictModel )
def doOspfv3LsaRetransmissionList( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   routerId = args.get( 'ROUTER_ID' )
   ifName = args.get( 'IFNAME' )
   command = 'MIO_DGET_OSPF3_RETRANSMISSION_LIST'
   argsCapi = { 'instanceId': AF_INSTANCE_ID_MAP[ instanceId ] }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( "Interface %s not present" % ifName.name )
         return None
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      return showRibCapiCommand( mode,
                                 Ospf3CliModels.Ospf3MultiAfLsaRetransmissionList,
                                 command, args=argsCapi, clientName='OSPF3',
                                 l3Config=getl3Config() )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3MultiAfLsaRetransmissionList()

class ShowOspfv3LsaRetransmissionListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ospfv3 [ ADDR_FAMILY ] retransmission-list 
               [ ROUTER_ID ] [ IFNAME ] [ VRF ]'''
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'retransmission-list': matcherOspf3LsaRetransmissionList,
            'ROUTER_ID': matcherNeighborId,
            'IFNAME': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }
   handler = doOspfv3LsaRetransmissionList
   cliModel = Ospf3LsaRetransmissionAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3LsaRetransmissionListCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 [ipv4 | ipv6] request-list [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
desc = "OSPFv3 AF LSA request list information for all VRFs"
Ospf3LsaRequestListInstModel = generateOspf3AfCliModel(
                            Ospf3CliModels.Ospf3MultiAfLsaRequestList )
Ospf3LsaRequestListVrfModel = generateVrfCliModel(
                            Ospf3LsaRequestListInstModel, desc )
@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospf3LsaRequestListVrfModel,
                               afModelDictType=Ospf3LsaRequestListInstModel )
def doShowOspfv3LsaRequestList( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   routerId = args.get( 'ROUTER_ID' )
   ifName = args.get( 'IFNAME' )
   command = "MIO_DGET_OSPF3_REQUEST_LIST"
   argsCapi = { 'instanceId': AF_INSTANCE_ID_MAP[ instanceId ] }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( "Interface %s not present" % ifName.name )
         return None
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfLsaRequestList,
                                 command, args=argsCapi, clientName='OSPF3',
                                 l3Config=getl3Config() )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3MultiAfLsaRequestList()

class ShowOspfv3LsaRequestListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ospfv3 [ ADDR_FAMILY ] request-list 
               [ ROUTER_ID ] [ IFNAME ] [ VRF ]'''
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'request-list': matcherOspf3LsaRequestList,
            'ROUTER_ID': matcherNeighborId,
            'IFNAME': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3LsaRequestList
   cliModel = Ospf3LsaRequestListVrfModel

BasicCli.addShowCommandClass( ShowOspfv3LsaRequestListCmd )

#-------------------------------------------------------------------------------
# "show ospfv3 [ipv4 | ipv6] spflog [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
spfLogDesc = 'OSPFv3 per AF SPF Log for all VRFs'
Ospfv3ShowSpfLogAfDictModel = generateOspf3AfCliModel(
                             Ospf3CliModels.Ospf3MultiAfSpfLogInst )
Ospfv3ShowSpfLogAfVrfDictModel = generateVrfCliModel( Ospfv3ShowSpfLogAfDictModel,
                                                      spfLogDesc )

@Ospf3MultiAfVrfCapiGenerator( vrfModelDictType=Ospfv3ShowSpfLogAfVrfDictModel, 
                               afModelDictType=Ospfv3ShowSpfLogAfDictModel ) 
def doShowOspfv3SpfLog( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF3_SPF_LOGS'
   argsCapi = {'instanceId': AF_INSTANCE_ID_MAP[ instanceId ] }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3MultiAfSpfLogInst, 
                                   command, argsCapi, clientName='OSPF3',
                                   l3Config=getl3Config() )
   except EmptyResponseException:
      result = None

   if result:
      return result
   else:
      result = Ospf3CliModels.Ospf3MultiAfSpfLogInst()
      getDefaultModelValues( result, afInstanceId=argsCapi[ 'instanceId' ],
                             vrfName=vrfName )
      return result

class ShowOspfv3SpfLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ospfv3 [ ADDR_FAMILY ] spf-log [ VRF ]'
   data = {
            'ospfv3': matcherOspfv3AfterShow,
            'ADDR_FAMILY': AFExpression,
            'spf-log': matcherOspf3SpfLog,
            'VRF': vrfExprFactory,
          }
   handler = doShowOspfv3SpfLog
   cliModel = Ospfv3ShowSpfLogAfVrfDictModel

BasicCli.addShowCommandClass( ShowOspfv3SpfLogCmd )
