#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import hashlib
import os
import socket
import sys
from collections import namedtuple

import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Ip, CliToken.Ipv6
from CliToken.Router import routerMatcherForConfig
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
import ConfigMount
import Intf.IntfRange as IntfRange
import IntfCli
import IpAddrMatcher
import Ip6AddrMatcher
import IraIp6Cli
import IraIp6IntfCli
import IraIpCli
import LazyMount
import ShowCommand
import Tac
import Tracing
import Toggles.OspfToggleLib
import Toggles.Ospf3ToggleLib
from CliMode.Ospf3 import RoutingOspf3Mode
from RouteMapCli import mapNameMatcher
from RoutingOspfCli import (
      AreaIdExpression,
      adjacencyKw,
      allInterfacesKw,
      areaKw,
      autoCostKw,
      bfdKw,
      bfdDefaultKw,
      costKw,
      deadIntervalKw,
      defaultCostKw,
      defaultMetricKw,
      distanceKw,
      exchangeStartKw,
      externalRtKw,
      filterPrefixMatcher,
      generateOspfInstListCliModel,
      gracefulRestartHelperKw,
      gracefulRestartKw,
      gracePeriodKw,
      helloIntervalKw,
      intraAreaKw,
      logAdjacencyChangesDetailKw,
      logAdjacencyChangesKw,
      matcherArrival,
      matcherCost,
      matcherCounters,
      matcherDatabase,
      matcherDatabaseSummary,
      matcherDefaultMetric,
      matcherDistance,
      matcherExtLsaMetric,
      matcherExternalLsa,
      matcherFloodPacing,
      matcherGracePeriod,
      matcherIncludeStub,
      matcherLsa,
      matcherLsaArrivalInt,
      matcherLsaHold,
      matcherLsaMaxWait,
      matcherLsaOutDelay,
      matcherLsaStart,
      matcherMaxMetricTimer,
      matcherNeighbor,
      matcherNeighborId,
      matcherOnStartup,
      matcherOptResetAllNeighbors,
      matcherPassiveDefault,
      matcherPassiveIntf,
      matcherPriority,
      matcherRangeCost,
      matcherRouteCostMetric,
      matcherRouterId,
      matcherSimultaneousPeers,
      matcherShow,
      matcherSpf,
      matcherSpfHold,
      matcherSpfMaxWait,
      matcherSpfStart,
      matcherSpfTimer,
      matcherSumLsaMetric,
      matcherSummaryLsa,
      maxEcmpPathsMatcher,
      metricMin,
      metricMax,
      mtuIgnoreKw,
      networkTypeKw,
      networkTypesMatcher,
      notsostubbyKw,
      nssaKw,
      nssaNoSummaryKw,
      nssaOnlyKw,
      OspfDefaultInformationCmdBase,
      outDelayKw,
      priorityKw,
      refBwKw,
      refBwMatcher,
      retransmitIntervalKw,
      routerIdKw,
      stubKw,
      stubNoSummaryKw,
      thresholdKw,
      transmitDelayKw,
)

import TechSupportCli
from ReversibleSecretCli import (
      decodeKey,
      type0KwMatcher,
      type7KwMatcher,
)
from RoutingCommon import GATED_PROTO_SH_TECH_TS
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
import IraVrfCli
from RibCapiLib import (
      showRibCapiCommand,
      EmptyResponseException,
      showRibDamiCommand,
)
from CliModel import GeneratorDict, Model
import CliPlugin
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import generateVrfCliModel
import Ospf3CliModels
from IraIpIntfCli import canSetVrfHook
import AclCli, AclCliLib, AclCliModel
import OspfConsts, Ospf3Consts

traceHandle = Tracing.Handle( 'RibOspf3Cli' )
t0 = traceHandle.trace2 # Errors
t2 = traceHandle.trace2 # Delete / Add
t5 = traceHandle.trace5 # Info

def ospf3GrSupported():
   return True

l3Config = None
ospf3ConfigDir = None
ospf3ClearReq = None
ospf3ClearRespDir = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None

routingHardwareStatus = None
routing6HardwareStatus = None
l3IntfConfigDir = None
l3Config = None
afIPv6InstanceID = 0
afIPv4InstanceID = 64
afAllInstanceID = 255
AF_INSTANCE_ID_MAP = { 'afIPv6' : 0,
                       'afIPv4' : 64,
                       'allAf' : 255
                     }

HmacAlgo = namedtuple( 'HmacAlgo', 'tacVal keyLen digestLen name' )
hmacSha1 = HmacAlgo( 2, 40, 160, 'SHA1' )
keywordToHmac = {
      # The tacVal values correspond to Ospf3.tac
      'md5': HmacAlgo( 1, 32, 128, 'MD5' ),
      'sha1': HmacAlgo( 2, 40, 160, 'SHA1' ),
}

#These values correspond to Ospf3.tac
encryptAlgos = { 'aes-128-cbc' : 1, 'aes-192-cbc' : 2,
               'aes-256-cbc' : 3,'3des-cbc' : 4}

aes128KeyLen = 128
aes192KeyLen = 192
aes256KeyLen = 256
tripleDesKeyLen = 192

MAX_AREAS = 64

intervalMin = 1
intervalMax = 65535
matcherIntfIntervalSeconds = CliMatcher.IntegerMatcher( intervalMin, intervalMax,
                                                        helpdesc='Seconds' )

#--------------------------------------------------------------
# getAfFromInstanceId returns the address family from instance id
#
# Please refer to getAfInstanceId to get instance id from family
# defined in RoutingOspf3MultiAfCli.py
#--------------------------------------------------------------

def getAfFromInstanceId( afInstanceId, globalInstance=False ):
   if afInstanceId == afIPv4InstanceID:
      return 'ipv4'
   elif afInstanceId == afIPv6InstanceID:
      return 'ipv6'
   else:
      if not globalInstance:
         assert 0, 'Unexpected instanceId %d' % afInstanceId
      return ''

#-------------------------------------------------------------------------------
# Getter & Setter functions for config, the present flag is set as well
#-------------------------------------------------------------------------------
def setOspf3ConfigAttr( config, attrName, attrValue ):
   assert attrName
   if not config:
      return
   attrPresent = attrName + 'Present'
   attrDefault = getattr( config, attrName + 'Default', None )
   if attrValue is not None and attrValue != attrDefault:
      setattr( config, attrName, attrValue )
      setattr( config, attrPresent, True )
   else:
      setattr( config, attrName, attrDefault )
      setattr( config, attrPresent, False )

def resetOspf3ConfigAttr( config, attrName ):
   setOspf3ConfigAttr( config, attrName, attrValue=None )

#-------------------------------------------------------------------------------
# Ira callback handler when the VRF for a interface changes.
# With old style CLI the processId identified the ospf3 instance for a interface
# with new style CLI the VRF configured on the interface is one of the keys to
# identify the ospf3 instance ( address family being the other ).
# Hence when the VRF configured for a particular interface changes it needs to
# be switched to the corresponding Ospf3 instance
#-------------------------------------------------------------------------------
def intfVrfChangeHandler( intfId, oldVrf, newVrf, vrfDelete ):
   warnMsg = None

   config = ospf3Config()
   if intfId in config.intfConfig:
      t5( 'Vrf changed for interface %s, from %s to %s' %
          ( intfId, oldVrf, newVrf ) )
      intfConfig = config.intfConfig[ intfId ]

      # If the interface was configured with old style cli
      # nothing to do for vrf change, just return
      if intfConfig.oldStyleCli:
         intfConfig.vrfName = newVrf
         return True, None

      # If the interface was configured with new style cli and has
      # a areadId associated ( auto created ), delete from the old
      # vrf and add to new vrf
      for afInstanceId in intfConfig.intfInstanceConfig:
         intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
         if intfInstanceConfig.areaIdPresent:
            areaId = intfInstanceConfig.areaId
            warnMsg = 'Changing OSPFv3 instance for interface %s, from %s to %s' % \
                      ( intfId, oldVrf, newVrf )

            # Add area to new vrf instance first, so that incase we hit the
            # max area limit we can error out and block the vrf change
            errMsg = addIntfAreaToInstance( intfInstanceConfig, areaId,
                                            intfConfig.processId,
                                            afInstanceId, newVrf )

            # Delete from old vrf - Set the interface areaIdPresent to false or
            # else _deleteAreaConfigFromInstance() will assume the current interface
            # is using the specified areaId
            if not errMsg:
               intfInstanceConfig.areaIdPresent = False
               _deleteAreaConfigFromInstance( areaId, intfConfig.processId,
                                              afInstanceId, oldVrf )
               intfInstanceConfig.areaIdPresent = True
            if errMsg:
               return False, errMsg
      intfConfig.vrfName = newVrf
   return True, warnMsg

canSetVrfHook.addExtension( intfVrfChangeHandler )

#-------------------------------------------------------------------------------
# Get the vrf configured for an interface ( looks up the L3::Intf::ConfigDir )
#-------------------------------------------------------------------------------
def _getIntfVrf( intfName ):
   intfConfig = l3IntfConfigDir.intfConfig.get( intfName, None )
   if intfConfig:
      return intfConfig.vrf
   return DEFAULT_VRF

#-------------------------------------------------------------------------------
# Check if a interface belongs to a given instance ( the instance is identified
# by processId, afInstanceId, vrfName ). For oldStyleCli only the processId is
# matched, else the other parameters are matched
#-------------------------------------------------------------------------------
def _checkIntfBelongsToInstance( intfConfig, processId,
                                 afInstanceId, vrfName ):
   if intfConfig.oldStyleCli and intfConfig.processId == processId:
      return True
   elif intfConfig.vrfName == vrfName and \
        afInstanceId in intfConfig.intfInstanceConfig and \
        not intfConfig.oldStyleCli:
      return True
   else:
      return False

#---------------------------------------------------------------------------------
# CLI guard to disallow certain commands from being configured in the default VRF
#---------------------------------------------------------------------------------
def ospf3DefaultVrfGuard( mode, token ):
   if mode.vrfName == DEFAULT_VRF:
      return 'not supported in the default VRF'
   return None

#-------------------------------------------------------------------------------
# Adds Ospfv3-specific CLI commands to the "config-if" mode for routed ports.
# This modelet is required to add loopback support in OSPFv3 since the
# generic RoutingProtocolIntfConfigModelet does not allow protocols to be configured
# on loopback interfaces
#-------------------------------------------------------------------------------
class RoutingProtocolOspfv3IntfConfigModelet(
   IraIp6IntfCli.RoutingProtocolIntfConfigModelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ): # pylint: disable-msg=W0231
      # We want to call the init the base class first
      IraIp6IntfCli.RoutingProtocolIntfConfigModelet.__init__(self, intfConfigMode)

   @staticmethod
   def shouldAddModeletRule( mode ):
      if IraIp6IntfCli.RoutingProtocolIntfConfigModelet.shouldAddModeletRule(
         mode ):
         return True
      # we want to include loopback interfaces
      return ( mode.intf.routingSupported() and
               mode.intf.name.startswith( "Loopback" ) )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolOspfv3IntfConfigModelet )

# Register some show commands with 'show tech-support'
# In the future if we add commands that fork rib to produce their output, then
# we should split the list of commands into two. One list for commands that don't
# fork ribd using GATED_PROTO_SH_TECH_TS and the second one for commands that fork
# using GATED_PROTO_FORK_SH_TECH_TS
def _Ospf3showTechCmds():
   return [ "show ospfv3", "show ospfv3 neighbor", "show ospfv3 interface" ]

def ospf3showTechSummaryCmds():
   return [ "show ospfv3", "show ospfv3 neighbor", "show ospfv3 interface" ]

TechSupportCli.registerShowTechSupportCmdCallback(
   GATED_PROTO_SH_TECH_TS, _Ospf3showTechCmds,
    summaryCmdCallback=ospf3showTechSummaryCmds )

#-------------------------------------------------------------------------------
# OSPFv3 CLI Classes
# - The Ospfv3 cli supports 2 styles 'ipv6 router ospf ...' / 'router ospfv3 ...'
# - The Ospf3 instance is implemented with the help of 4 modelets
#       - RouterOspfv3SharedModelet - Implements all config commands
#         common to both modes & both address families
#         Example : router-id a.b.c.d
#       - RouterOspfv3AfSharedModelet - Implements all config commands which
#         are valid only in address family sub mode ( which is default for old
#         style cli).
#         Example : default-metric x
#       - RouterOspfv3AfIpv6Modelet / RouterOspfv3AfIpv4Modelet - Implements
#         config commands which are specific to an address family
#         Example : area x range <ip - range>
# - RouterOspf3Mode - Implements old style instance config Cli, and derives from
#   RouterOspfv3SharedModelet, RouterOspfv3AfSharedModelet and
#   RouterOspfv3AfIpv6Modelet
# - RouterOspfv3Mode / RouterOspfv3AfMode - Implement the new style instance
#   config and derive from the above modelets
# - RouterOspf3Common is the base class all modelets derive from this and all
#   Cli handlers are implemented as part of this class ( we could ideally derive
#   4 separate classes from RouterOspf3Common but for now this class has all
#   instance config functionality )
#-------------------------------------------------------------------------------
class RouterOspf3Common( object ):
   #----------------------------------------------------------------------------
   # Four modelet classes derive from this class, define a common init
   #----------------------------------------------------------------------------
   def __init__( self, mode ):
      self.mode_ = mode
      if getattr( mode, 'processId', None ):
         # Old style cli, we will be configuring only v6 instance
         self.afInstanceId = afIPv6InstanceID
         self.processId = getattr( mode, 'processId', None )
      else:
         # New style cli, could be global / ipv4 / ipv6
         self.afInstanceId = getattr( self.mode_ , 'instId', afAllInstanceID )
         self.processId = None
      self.vrfName = self.mode_.vrfName
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig

   #----------------------------------------------------------------------------
   # Get the vrf config for a given vrf name
   #----------------------------------------------------------------------------
   def _getVrfConfig( self ):
      vrfConfig = ospf3Config().vrfConfig.get( self.vrfName, None )
      assert vrfConfig
      if bool( vrfConfig.processId ) != bool( self.processId ):
         self.addError( getCliErrorMsg() )
         return None
      if vrfConfig.processId and \
         vrfConfig.processId != self.processId:
         self.addError( "Process IDs do not match for Ospf3 instance in vrf %s" %
                        self.vrfName )
         return None

      return vrfConfig

   #----------------------------------------------------------------------------
   # Get instance config for a mode
   #----------------------------------------------------------------------------
   def _getInstanceConfig( self ):
      vrfConfig = self._getVrfConfig()
      if not vrfConfig:
         return None
      instanceConfig = vrfConfig.instanceConfig[ self.afInstanceId ]
      assert instanceConfig
      return instanceConfig

   #----------------------------------------------------------------------------
   # Get instance config for a mode
   # With 'router ospfv3' style config we should not allow a config in global
   # mode, to be overridden in AF mode or vice versa. Hence when getting the
   # instance config also check that the attribute is not set in the other
   # instance config
   #----------------------------------------------------------------------------
   def _getInstanceConfigForAttr( self, attr ):
      assert attr
      afInstanceId = self.afInstanceId
      vrfConfig = self._getVrfConfig()
      if afInstanceId == afAllInstanceID:
         v4InstConfig = vrfConfig.instanceConfig.get( afIPv4InstanceID, None )
         if v4InstConfig and getattr( v4InstConfig, attr + "Present", False ):
            self.addError( 'Attribute already configured in ipv4 address family' )
            return None
         v6InstConfig = vrfConfig.instanceConfig.get( afIPv6InstanceID, None )
         if v6InstConfig and getattr( v6InstConfig, attr + "Present", False ):
            self.addError( 'Attribute already configured in ipv6 address family' )
            return None
      else:
         globalInstConfig = vrfConfig.instanceConfig.get( afAllInstanceID, None )
         if globalInstConfig and \
               getattr( globalInstConfig, attr + "Present", False ):
            self.addError( 'Attribute already configured in global config' )
            return None
      return vrfConfig.instanceConfig[ afInstanceId ]

   #----------------------------------------------------------------------------
   # Check if an interface belongs to a given instance
   #----------------------------------------------------------------------------
   def checkIntfBelongsToInstance( self, intf ):
      # Check for old style configuration
      if self.processId:
         if intf.oldStyleCli and intf.processId == self.processId:
            return True

      # Check for new style configuration
      if intf.vrfName == self.vrfName and \
         not intf.oldStyleCli:
         return True
      return False

   #----------------------------------------------------------------------------
   # Get the address family list for the mode
   #----------------------------------------------------------------------------
   def _getOspf3AfList( self ):
      if self.afInstanceId == afAllInstanceID:
         return [ afIPv4InstanceID, afIPv6InstanceID ]
      else:
         return [ self.afInstanceId ]

   def addError( self, msg ):
      self.mode_.addError( msg )

   def _deleteAreaConfigIfNotInUse( self, area ):
      processId = self.processId
      afInstanceId = self.afInstanceId
      vrfName = self.vrfName
      area = str( Arnet.IpAddress( area ) )
      _deleteAreaConfigFromInstance( area, processId, afInstanceId, vrfName )

   def _areaConfigIsDefaults( self, instConfig, areaId ):
      areaConfig = instConfig.areaConfig[ areaId ]
      return _areaConfigIsDefaults( areaConfig )

   def _getOrCreateAreaConfig( self, areaId ):
      areaId = str( Arnet.IpAddress( areaId ) )
      vrfConfig = self._getVrfConfig()
      afInstanceId = self.afInstanceId
      instanceConfig = vrfConfig.instanceConfig[ afInstanceId ]

      # Area should be configured only under 1 instance check for this
      # There is a subtle change w.r.t to other parameters, areaId may
      # be present in the IPV4 / IPV6 context ( because of interface cmd
      # ospfv3 ipv4 area ... )
      if afInstanceId == afAllInstanceID:
         v4InstConfig = vrfConfig.instanceConfig.get( afIPv4InstanceID, None )
         if v4InstConfig and areaId in v4InstConfig.areaConfig:
            if not self._areaConfigIsDefaults( v4InstConfig, areaId ):
               self.addError( 'Area %s already configured in ipv4 address family' %
                              areaId )
               return None
         v6InstConfig = vrfConfig.instanceConfig.get( afIPv6InstanceID, None )
         if v6InstConfig and areaId in v6InstConfig.areaConfig:
            if not self._areaConfigIsDefaults( v6InstConfig, areaId ):
               self.addError( 'Area %s already configured in ipv6 address family' %
                              areaId )
               return None
      else:
         globalInstConfig = vrfConfig.instanceConfig.get( afAllInstanceID, None )
         if globalInstConfig and areaId in globalInstConfig.areaConfig:
            self.addError( 'Area %s already configured in global config' % areaId )
            return None

      area = self._getAreaConfig( areaId, printWarning=False )
      if area:
         return area
      else:
         if len( instanceConfig.areaConfig ) >= MAX_AREAS:
            self.addError( "Max number of areas (%d) exceeded" % MAX_AREAS )
            return None
         if afInstanceId == afAllInstanceID:
            # If area created in global mode then we must add the area
            # in both AF instance area config collections to ensure area config
            # SM is created for each of the instance config SMs and will push
            # the inherited area config attributes from the global area config
            # to gated
            if ( v4InstConfig and len( v4InstConfig.areaConfig ) >= MAX_AREAS ) or \
                  ( v6InstConfig and len( v6InstConfig.areaConfig ) >= MAX_AREAS ):
               self.addError( "Max number of areas (%d) exceeded" % MAX_AREAS )
               return None
            if v4InstConfig:
               assert addAreaIdsToInstanceConfig( v4InstConfig, [ areaId ] )
            if v6InstConfig:
               assert addAreaIdsToInstanceConfig( v6InstConfig, [ areaId ] )
         t2( 'Created area %s in instance config (vrf %s, instanceId %d)' %
             ( areaId, self.vrfName, afInstanceId ) )
         return instanceConfig.areaConfig.newMember( areaId )

   def _getAreaConfig( self, areaId, printWarning=True ):
      instanceConfig = self._getInstanceConfig()
      areaId = str( Arnet.IpAddress( areaId ) )
      if areaId in instanceConfig.areaConfig:
         return instanceConfig.areaConfig[ areaId ]
      else:
         if printWarning:
            self.addError( "Specified area %s is not configured" % areaId )
         return None

   #---------------------------------------------------------------------------
   # Area config specific cli handlers
   #---------------------------------------------------------------------------
   def noArea( self, args ):
      areaId = args[ 'AREA_ID' ]
      instanceConfig = self._getInstanceConfig()
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      # Check if any interface uses this area
      for afInstanceId in self._getOspf3AfList():
         for intfname in ospf3Config().intfConfig:
            intfConfig = ospf3Config().intfConfig[ intfname ]

            # Check if the interface belongs to the current ospf3 instance
            if afInstanceId in intfConfig.intfInstanceConfig and \
               self.checkIntfBelongsToInstance( intfConfig ):

               # Get the instance list for the ospf3 instance and iterate
               intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
               if intfInstanceConfig.areaIdPresent and \
                  intfInstanceConfig.areaId == areaConfig.area:
                  self.addError( "Specified area is configured with interfaces" )
                  return
      t2( 'Deleting area %s from vrf %s instance' % ( areaId, self.vrfName ) )
      del instanceConfig.areaConfig[ areaConfig.area ]

   def noAreaSecurity( self, areaId ):
      areaConfig = self._getAreaConfig( areaId )
      if areaConfig:
         resetOspf3ConfigAttr( areaConfig, 'securityEnabled' )
         resetOspf3ConfigAttr( areaConfig, 'sa' )
      self._deleteAreaConfigIfNotInUse( areaId )

   def setDefaultCost( self, args ):
      areaId = args[ 'AREA_ID' ]
      defaultCost = args[ 'COST' ]
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if areaConfig:
         setOspf3ConfigAttr( areaConfig, 'defaultMetric', defaultCost )
         if defaultCost == areaConfig.defaultMetricDefault:
            self._deleteAreaConfigIfNotInUse( areaId )

   def noDefaultCost( self, args ):
      areaId = args[ 'AREA_ID' ]
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      resetOspf3ConfigAttr( areaConfig, 'defaultMetric' )
      self._deleteAreaConfigIfNotInUse( areaId )

   def _stubAllowed( self, areaConfig ):
      if areaConfig.areaType == 'areaTypeNssa' or \
             areaConfig.areaType == 'areaTypeNssaNoSummary':
         self.addError( "Area has already been configured as NSSA" )
         return False
      return True

   def setStub( self, args ):
      areaId = args[ 'AREA_ID' ]
      if str( Arnet.IpAddress( areaId ) ) == '0.0.0.0':
         self.addError( 'Backbone cannot be configured as stub area' )
         return
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._stubAllowed( areaConfig ):
         return
      areaConfig.originateDefault = True
      if areaConfig.areaType != 'areaTypeStubNoSummary':
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeStub' )

   def noStub( self, args ):
      areaId = args[ 'AREA_ID' ]
      noSummary = 'no-summary' in args
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._stubAllowed( areaConfig ):
         return
      if noSummary:
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeStub' )
      else:
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeNormal' )
      areaConfig.originateDefault = False
      self._deleteAreaConfigIfNotInUse( areaId )

   def setStubNoSummary( self, args ):
      areaId = args[ 'AREA_ID' ]
      if str( Arnet.IpAddress( areaId ) ) == '0.0.0.0':
         self.addError( "Backbone cannot be configured as stub area" )
         return
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._stubAllowed( areaConfig ):
         return
      areaConfig.originateDefault = True
      setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeStubNoSummary' )

   def _nssaAllowed( self, areaConfig ):
      if areaConfig.areaType == 'areaTypeStub' or \
             areaConfig.areaType == 'areaTypeStubNoSummary':
         self.addError( "Area has already been configured as stub area" )
         return False
      # An area containing virtual links cannot be made an NSSA area.
      # At present, we do not support virtual links, but when we do so,
      # we would need this check. Till then, comment it out.
      #if len( areaConfig.virtualLink ):
      #   self.addError( "Area contains virtual links, and so "
      #                  "so NSSA configuration is not allowed " )
      #   return False
      return True

   def _setNssaDefInfoDefaults( self, areaConfig ):
      for i in [ 'nssaDefInfoMetric', 'nssaDefInfoMetricType',
                 'nssaDefInfoNssaOnly' ]:
         resetOspf3ConfigAttr( areaConfig, i )

   def setNssa( self, args ):
      areaId = args[ 'AREA_ID' ]
      defInfoOriginRule = None
      if 'default-information-originate' in args:
         defInfoOriginRule = []
         if 'metric' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoMetric',
                                         args[ 'METRIC' ].pop() ) )
         if 'metric-type' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoMetricType',
                                         args[ 'METRIC_TYPE' ].pop() ) )
         if 'nssa-only' in args:
            defInfoOriginRule.append( ( 'nssaDefInfoNssaOnly', True ) )
      translate = args.get( 'type-5' ) or \
                  args.get( 'always' ) or args.get( 'nssa-only' )
      noSummary = args.get( 'no-summary' )
      if str( Arnet.IpAddress( areaId ) ) == '0.0.0.0':
         self.addError( 'Backbone cannot be configured as NSSA area' )
         return
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return
      if noSummary:
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeNssaNoSummary' )
         areaConfig.originateDefault = True
         return
      if areaConfig.areaType != 'areaTypeNssaNoSummary':
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeNssa' )
         areaConfig.originateDefault = False

      if defInfoOriginRule is not None:
         setOspf3ConfigAttr( areaConfig, 'nssaDefInfoOrigin',
                             'ospfDefInfoOriginEnable' )
         defInfoParms = dict( defInfoOriginRule )
         for i in [ 'nssaDefInfoMetric', 'nssaDefInfoMetricType',
                    'nssaDefInfoNssaOnly' ]:
            if i in defInfoParms:
               setOspf3ConfigAttr( areaConfig, i, defInfoParms[ i ] )
            else:
               resetOspf3ConfigAttr( areaConfig, i )
      # New command does not contain 'always' keyword. We look for 'type-5' instead
      elif translate == 'always' or translate == 'type-5':
         setOspf3ConfigAttr( areaConfig, 'nssaTranslateAlways', True )
      elif translate == 'nssa-only':
         setOspf3ConfigAttr( areaConfig, 'disablePBit', True )

   def noNssa( self, args ):
      areaId = args[ 'AREA_ID' ]
      defInfoOrigin = args.get( 'default-information-originate' )
      translate = args.get( 'type-5' ) or args.get( 'always' ) or \
                  args.get( 'nssa-only' )
      noSummary = args.get( 'no-summary' )
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return

      if translate == 'nssa-only':
         resetOspf3ConfigAttr( areaConfig, 'disablePBit' )
      # New command does not contain 'always' keyword. We look for 'type-5' instead
      elif translate == 'always' or translate == 'type-5':
         resetOspf3ConfigAttr( areaConfig, 'nssaTranslateAlways' )
      elif defInfoOrigin is not None:
         resetOspf3ConfigAttr( areaConfig, 'nssaDefInfoOrigin' )
         self._setNssaDefInfoDefaults( areaConfig )
      elif noSummary:
         setOspf3ConfigAttr( areaConfig, 'areaType', 'areaTypeNssa' )
         areaConfig.originateDefault = False
      else:
         resetOspf3ConfigAttr( areaConfig, 'areaType' )
         areaConfig.originateDefault = False
         resetOspf3ConfigAttr( areaConfig, 'nssaTranslateAlways' )
         resetOspf3ConfigAttr( areaConfig, 'disablePBit' )
         resetOspf3ConfigAttr( areaConfig, 'nssaDefInfoOrigin' )
         self._setNssaDefInfoDefaults( areaConfig )

      # Delete the area config only when you call no area <x> nssa
      if translate is None:
         self._deleteAreaConfigIfNotInUse( areaId )

   def noNssaDefInfoOrigin( self, args ):
      areaConfig = self._getAreaConfig( args[ 'AREA_ID' ] )
      if not areaConfig:
         return
      if not self._nssaAllowed( areaConfig ):
         return
      setOspf3ConfigAttr( areaConfig, 'nssaDefInfoOrigin',
                          'ospfDefInfoOriginSuppress' )
      self._setNssaDefInfoDefaults( areaConfig )

   def _isDefaultPrefix( self, prefix ):
      if prefix.af == 'ipv6' and prefix.v6Prefix.address == Arnet.Ip6Addr( "::" ):
         return True
      elif prefix.af == 'ipv4' and prefix.v4Prefix.address == '0.0.0.0':
         return True
      return False

   def setRange( self, args ):
      areaId = args[ 'AREA_ID' ]
      cost = args.get( 'COST' )
      prefix = args[ 'PREFIX' ]
      prefix = Arnet.IpGenPrefix( str( prefix ) )
      if self._isDefaultPrefix( prefix ):
         self.addError( "Cannot add default prefix %s as range" %
                         str( prefix ) )
         return

      restricted = 'not-advertise' in args
      cost = cost or 0
      areaConfig = self._getOrCreateAreaConfig( areaId )
      if not areaConfig:
         return
      areaConfig.networkList.addMember( Tac.Value(
         "Routing6::Ospf3::Network", network=prefix, notAdvertise=restricted,
         cost=cost ) )

   def noRange( self, args ):
      areaId = args[ 'AREA_ID' ]
      advertise = args.get( 'advertise' ) or args.get( 'not-advertise' )
      cost = args.get( 'COST' )
      prefix = args[ 'PREFIX' ]
      areaConfig = self._getAreaConfig( areaId )
      if not areaConfig:
         return
      prefix = Arnet.IpGenPrefix( str( prefix ) )
      net = areaConfig.networkList.get( prefix, None )
      if not net:
         return
      if ( advertise is None or advertise=='advertise') and cost is None:
         del areaConfig.networkList[ prefix ]
         self._deleteAreaConfigIfNotInUse( areaId )
         return

      restricted = advertise == 'not-advertise'
      cost = cost or 0
      if net.notAdvertise == restricted and net.cost == cost:
         cost = 0
         if restricted:
            restricted = False
         areaConfig.networkList.addMember( Tac.Value(
               "Routing6::Ospf3::Network", network=prefix, notAdvertise=restricted,
               cost=cost) )

   #---------------------------------------------------------------------------
   # Instance config cli handlers
   #---------------------------------------------------------------------------
   def setRouterId( self, args ):
      routerId = args.get( 'ROUTER_ID' )
      if routerId == "0.0.0.0":
         self.addError( "%s is not a valid router-id" % routerId )
         return
      instanceConfig = self._getInstanceConfigForAttr( 'routerId' )
      setOspf3ConfigAttr( instanceConfig, 'routerId', routerId )

   def noRouterId( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'routerId' )

   def setAutoCost( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'refBw' )
      refBwVal = args[ 'BANDWIDTH' ]
      setOspf3ConfigAttr( instanceConfig, 'refBw', refBwVal )

   def noAutoCost( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'refBw' )

   def setShutdown( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'enable' )
      setOspf3ConfigAttr( instanceConfig, 'enable', False )

   def noShutdown( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'enable' )

   def setLogAdjacencyChanges( self, args ):
      detail = args.get( 'DETAIL' )
      if detail:
         instanceConfig = self._getInstanceConfigForAttr( 'adjacencyLogging' )
         setOspf3ConfigAttr( instanceConfig, 'adjacencyLogging',
                             'adjacencyLoggingTypeDetail' )
      else:
         instanceConfig = self._getInstanceConfig()
         resetOspf3ConfigAttr( instanceConfig, 'adjacencyLogging' )

   def noLogAdjacencyChanges( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'adjacencyLogging' )
      setOspf3ConfigAttr( instanceConfig, 'adjacencyLogging',
                          'adjacencyLoggingTypeNone' )

   def setTimerSPF( self, args ):
      spfInterval = args[ 'SECONDS' ]
      instanceConfig = self._getInstanceConfigForAttr( 'spfThrottleTimerConfig' )
      if instanceConfig:
         setOspf3ConfigAttr( instanceConfig, 'spfThrottleTimerConfig',
             Tac.Value( "Routing6::Ospf3::SpfThrottleTimer",
                        0, spfInterval * 1000, spfInterval * 1000 ) )
         # If setting SPF timer in secs using the "timers spf" CLI then
         # we track usage of the old CLI command for CliSavePlugin
         instanceConfig.timerSpfCommandActive = (
               instanceConfig.spfThrottleTimerConfig !=
               instanceConfig.spfThrottleTimerConfigDefault )

   def setSpfIntervalThrottle ( self, args ):
      spfStartInt = args[ 'SPF_START' ]
      spfHoldInt = args[  'SPF_HOLD' ]
      spfMaxWaitInt = args[ 'SPF_MAX_WAIT' ]
      instanceConfig = self._getInstanceConfigForAttr( 'spfThrottleTimerConfig' )
      if instanceConfig:
         change = 0
         # spfHoldInt must be >= spfStartInt
         # spfMaxWaitInt must be >= spfHoldInt
         # adjust configured values if they are invalid
         if spfHoldInt < spfStartInt:
            spfHoldInt = spfStartInt
            change = 1
         if spfMaxWaitInt < spfHoldInt:
            spfMaxWaitInt = spfHoldInt
            change = 1
         if change:
            print "OSPF3: Throttle timers corrected to: %d %d %d" % ( spfStartInt,
                                                      spfHoldInt, spfMaxWaitInt )
         setOspf3ConfigAttr( instanceConfig, 'spfThrottleTimerConfig',
             Tac.Value( "Routing6::Ospf3::SpfThrottleTimer",
                        spfStartInt, spfHoldInt, spfMaxWaitInt ) )
         instanceConfig.timerSpfCommandActive = False

   def noSpfIntervalThrottle ( self, args ):
      instanceConfig = self._getInstanceConfig()
      if instanceConfig:
         resetOspf3ConfigAttr( instanceConfig, 'spfThrottleTimerConfig' )
         instanceConfig.timerSpfCommandActive = False

   def setFloodPacing( self, args ):
      floodPacing = args[ 'MILLISECONDS' ]
      instanceConfig = self._getInstanceConfigForAttr( 'floodPacing' )
      setOspf3ConfigAttr( instanceConfig, 'floodPacing', floodPacing )

   def noFloodPacing( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'floodPacing' )

   def setOutDelayTimer( self, args ):
      lsaOutDelay = args[ 'LSA_OUT_DELAY' ]
      instanceConfig = self._getInstanceConfigForAttr( 'lsaOutDelayTimerConfig' )
      setOspf3ConfigAttr( instanceConfig, 'lsaOutDelayTimerConfig', lsaOutDelay )

   def noOutDelayTimer( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'lsaOutDelayTimerConfig' )

   def setBfd( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'bfdEnabled' )
      setOspf3ConfigAttr( instanceConfig, 'bfdEnabled', True )

   def noBfd ( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'bfdEnabled' )

   def setBfdAnyState( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'bfdAnyState' )
      setOspf3ConfigAttr( instanceConfig, 'bfdAnyState', True )

   def noBfdAnyState ( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'bfdAnyState' )

   def setMaxMetric( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'maxMetricConfig' )
      if not instanceConfig:
         return
      maxMetricConfig = instanceConfig.maxMetricConfig

      # set all values to their current values as default
      maxMetricOn = True
      extLsaMetricValue = maxMetricConfig.maxMetricExtVal
      includeStubFlag = maxMetricConfig.maxMetricStub
      interAreaLsaMetricValue = maxMetricConfig.maxMetricInterAreaVal
      onStartTime = maxMetricConfig.maxMetricStartTimer
      onStartWaitBgp = maxMetricConfig.maxMetricWaitForBgp
      includePrefixStub = maxMetricConfig.maxMetricPrefixLsa
      stubPrefLsaMetricValue = maxMetricConfig.maxMetricStubPrefixVal
      sumLsaMetricValue = maxMetricConfig.maxMetricSumVal

      if 'external-lsa' in args:
         if 'EXTERNAL_LSA_METRIC_VALUE' in args:
            extLsaMetricValue = args.get( 'EXTERNAL_LSA_METRIC_VALUE' )[ 0 ]
         else:
            extLsaMetricValue = instanceConfig.maxMetricLsaValDefault
      if 'include-stub' in args:
         includeStubFlag = True
      if 'summary-lsa' in args:
         if 'SUMMARY_LSA_METRIC_VALUE' in args:
            sumLsaMetricValue = args.get( 'SUMMARY_LSA_METRIC_VALUE' )[ 0 ]
         else:
            sumLsaMetricValue = instanceConfig.maxMetricLsaValDefault
      if 'on-startup' in args:
         if 'wait-for-bgp' in args:
            onStartTime = 0
            onStartWaitBgp = True
         else:
            onStartTime = args.get( 'ANNOUNCE_TIME' )[ 0 ]
            onStartWaitBgp = False

      setOspf3ConfigAttr( instanceConfig, 'maxMetricConfig',
            Tac.Value( 'Routing6::Ospf3::MaxMetric', maxMetricOn,
                       extLsaMetricValue, includeStubFlag,
                       interAreaLsaMetricValue, onStartTime,
                       onStartWaitBgp, includePrefixStub,
                       stubPrefLsaMetricValue, sumLsaMetricValue ) )

   def noMaxMetric( self, args ):
      instanceConfig = self._getInstanceConfigForAttr( 'maxMetricConfig' )
      if not instanceConfig:
         return

      if len( args ) > 3:
         maxMetricConfig = instanceConfig.maxMetricConfig

         # set all values to their current values as default
         maxMetricOn = maxMetricConfig.maxMetricOn
         extLsaMetricValue = maxMetricConfig.maxMetricExtVal
         includeStubFlag = maxMetricConfig.maxMetricStub
         interAreaLsaMetricValue = maxMetricConfig.maxMetricInterAreaVal
         onStartTime = maxMetricConfig.maxMetricStartTimer
         onStartWaitBgp = maxMetricConfig.maxMetricWaitForBgp
         includePrefixStub = maxMetricConfig.maxMetricPrefixLsa
         stubPrefLsaMetricValue = maxMetricConfig.maxMetricStubPrefixVal
         sumLsaMetricValue = maxMetricConfig.maxMetricSumVal

         argsDict = {}

         if 'external-lsa' in args:
            if 'EXTERNAL_LSA_METRIC_VALUE' in args:
               # if extLsaMetricValue == 0, external-lsa was never set
               # so nothing needs to be done
               if args.get( 'EXTERNAL_LSA_METRIC_VALUE' )[ 0 ] != 0:
                  extLsaMetricValue = instanceConfig.maxMetricLsaValDefault
            else:
               extLsaMetricValue = instanceConfig.maxMetricExtValDefault
         if 'include-stub' in args:
            includeStubFlag = instanceConfig.maxMetricStubDefault
         if 'summary-lsa' in args:
            if 'SUMMARY_LSA_METRIC_VALUE' in args:
               if args.get( 'SUMMARY_LSA_METRIC_VALUE' )[ 0 ] != 0:
                  sumLsaMetricValue = instanceConfig.maxMetricLsaValDefault
            else:
               # if sumLsaMetricValue == 0, summary-lsa was never set
               # so nothing needs to be done
               sumLsaMetricValue = instanceConfig.maxMetricSumValDefault

         if 'on-startup' in args:
            if 'wait-for-bgp' in args:
               argsDict[ 'maxMetricOnStart' ] = args.get( 'wait-for-bgp' )[ 0 ]
            elif 'ANNOUNCE_TIME' in args:
               argsDict[ 'maxMetricOnStart' ] = args.get( 'ANNOUNCE_TIME' )[ 0 ]
            else:
               argsDict[ 'maxMetricOnStart' ] = None

         if 'maxMetricOnStart' in argsDict:
            if argsDict[ 'maxMetricOnStart' ] is None:
               onStartWaitBgp = instanceConfig.maxMetricWaitForBgpDefault
               onStartTime = instanceConfig.maxMetricStartTimerDefault
            elif argsDict[ 'maxMetricOnStart' ] == 'wait-for-bgp':
               onStartWaitBgp = instanceConfig.maxMetricWaitForBgpDefault
            else:
               onStartTime = instanceConfig.maxMetricStartTimerDefault

         setOspf3ConfigAttr( instanceConfig, 'maxMetricConfig',
            Tac.Value( 'Routing6::Ospf3::MaxMetric', maxMetricOn,
                       extLsaMetricValue, includeStubFlag,
                       interAreaLsaMetricValue, onStartTime,
                       onStartWaitBgp, includePrefixStub,
                       stubPrefLsaMetricValue, sumLsaMetricValue ) )
      else:
         resetOspf3ConfigAttr( instanceConfig, 'maxMetricConfig' )

   def setExchStartThreshold( self, args ):
      simulPeers = args[ 'THRESHOLD' ]
      instanceConfig = self._getInstanceConfigForAttr( 'exchStartThreshold' )
      setOspf3ConfigAttr( instanceConfig, 'exchStartThreshold', simulPeers )

   def noExchStartThreshold( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'exchStartThreshold' )

   def setLsaArrivalInterval ( self, args ):
      lsaArrivalInt = args[ 'MILLISECONDS' ]
      instanceConfig = self._getInstanceConfigForAttr( 'lsaArrivalInt' )
      setOspf3ConfigAttr( instanceConfig, 'lsaArrivalInt', lsaArrivalInt )

   def noLsaArrivalInterval ( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'lsaArrivalInt' )

   def setLsaIntervalThrottle ( self, args ):
      lsaStartInt = args[ 'LSA_START' ]
      lsaHoldInt = args[ 'LSA_HOLD' ]
      lsaMaxWaitInt = args[ 'LSA_MAX_WAIT' ]
      instanceConfig = self._getInstanceConfigForAttr( 'lsaThrottleTimerConfig' )
      change = 0

      # lsaHoldInt must be >= lsaStartInt
      # lsaMaxWaitInt must be >= lsaHoldInt
      # adjust configured values if they are invalid
      if lsaHoldInt < lsaStartInt:
         lsaHoldInt = lsaStartInt
         change = 1
      if lsaMaxWaitInt < lsaHoldInt:
         lsaMaxWaitInt = lsaHoldInt
         change = 1
      if change:
         print "OSPF3: LSA throttle timers corrected to: %d %d %d" % \
         ( lsaStartInt, lsaHoldInt, lsaMaxWaitInt )
      setOspf3ConfigAttr( instanceConfig, 'lsaThrottleTimerConfig',
            Tac.Value( "Routing6::Ospf3::LsaThrottleTimer",
                       lsaStartInt, lsaHoldInt, lsaMaxWaitInt ) )

   def noLsaIntervalThrottle ( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'lsaThrottleTimerConfig' )

   def setPassiveIntf( self, args ):
      intfList = args.get( 'INTFS', 'default' )
      instanceConfig = self._getInstanceConfigForAttr( 'instancePassive' )

      if not instanceConfig:
         return

      # Set the passive flag for all interfaces
      if intfList == 'default':
         setOspf3ConfigAttr( instanceConfig, 'instancePassive', True )
         return

      # The same function handles old/new config styles
      # but interface list is support only with old style
      assert self.processId
      # Set the passive flag for specified interfaces
      assert isinstance( intfList, IntfRange.IntfList )
      for afInstId in self._getOspf3AfList():
         for intfName in intfList.intfNames():
            intfConfig = _getIntfConfig( intfName )
            if intfConfig and self.checkIntfBelongsToInstance( intfConfig ) and \
               afInstId in intfConfig.intfInstanceConfig:
               intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstId ]
               setOspf3ConfigAttr( intfInstanceConfig, 'passive',
                                   'ospfIntfPassiveEnabled' )
            else:
               self.addError( 'OSPF3 instance not enabled or configuration '
                              'mismatch on interface %s for %s AF' % ( intfName,
                              getAfFromInstanceId( afInstId ).upper() ) )

   # Sets the passive interface state on the passed-in list of
   # interfaces (which are either IntfList or Intfs to the default
   # value. This is called from CLI context via
   # "no passive-interface <intf-names>" command and also to reset
   # the state when the OSPF process is deleted.
   def noPassiveIntf( self, args ):
      intfList = args.get( 'INTFS', 'default' )
      instanceConfig = self._getInstanceConfig()
      if not instanceConfig:
         return

      # Reset the passive flag for all interfaces
      if intfList == 'default':
         instanceConfig.instancePassive = False
         instanceConfig.instancePassivePresent = False
         return

      # The same function handles old/new config styles
      # but interface list is support only with old style
      assert self.processId
      assert isinstance( intfList, IntfRange.IntfList )

      # Reset the passive flag for specified interfaces
      for afInstId in self._getOspf3AfList():
         for intfName in intfList.intfNames():
            intfConfig = _getIntfConfig( intfName )
            if intfConfig and self.checkIntfBelongsToInstance( intfConfig ) and \
               afInstId in intfConfig.intfInstanceConfig:
               intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstId ]
               if CliCommand.isDefaultCmd( args ):
                  resetOspf3ConfigAttr( intfInstanceConfig, 'passive' )
               else:
                  setOspf3ConfigAttr( intfInstanceConfig, 'passive',
                                      'ospfIntfPassiveDisabled' )
               _deleteIntfInstanceConfig( intfName, afInstId )
            else:
               self.addError( 'OSPF3 instance not enabled or configuration '
                              'mismatch on interface %s for %s AF' % ( intfName,
                              getAfFromInstanceId( afInstId ).upper() ) )

   def setGracefulRestart( self, args=None ):
      instanceConfig = self._getInstanceConfigForAttr( 'gracefulRestart' )
      if instanceConfig:
         setOspf3ConfigAttr( instanceConfig, 'gracefulRestart', True )
         return instanceConfig
      return None

   def setGrGracePeriod( self, args ):
      instanceConfig = self.setGracefulRestart()
      if instanceConfig:
         setOspf3ConfigAttr( instanceConfig, 'grGracePeriod',
                             args[ 'GRACE_PERIOD' ] )

   def noGrGracePeriod( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'grGracePeriod' )

   def setGrPlannedOnly( self, args ):
      instanceConfig = self.setGracefulRestart()
      if instanceConfig:
         setOspf3ConfigAttr( instanceConfig, 'grPlannedOnly', True )

   def noGrPlannedOnly( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'grPlannedOnly' )

   def noGracefulRestart( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'gracefulRestart' )
      resetOspf3ConfigAttr( instanceConfig, 'grGracePeriod' )
      resetOspf3ConfigAttr( instanceConfig, 'grPlannedOnly' )

   def setGrHelper( self, args=None ):
      # grHelperDefault is True, thus setting means
      # resetting to its default value
      instanceConfig = self._getInstanceConfig()
      if instanceConfig:
         resetOspf3ConfigAttr( instanceConfig, 'grHelper' )
         return instanceConfig
      return None

   def setGrHelperLooseLsaChecking( self, args ):
      instanceConfig = self.setGrHelper()
      setOspf3ConfigAttr( instanceConfig, 'grHelperLooseLsaChk', True )

   def noGrHelperLooseLsaChecking( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'grHelperLooseLsaChk' )

   def noGrHelper( self, args ):
      # grHelperDefault is True, thus 'no graceful-restart-helper'
      # means setting grHelper to False
      instanceConfig = self._getInstanceConfigForAttr( 'grHelper' )
      setOspf3ConfigAttr( instanceConfig, 'grHelper', False )

   def setDnBitIgnore( self, args ):
      instanceConfig = self._getInstanceConfig()
      if self.vrfName == DEFAULT_VRF:
         self.addError( 'dn-bit-ignore cannot be configured for the default vrf' )
         return
      setOspf3ConfigAttr( instanceConfig, 'ignoreDnBit', True )

   def noDnBitIgnore( self, args ):
      instanceConfig = self._getInstanceConfig()
      resetOspf3ConfigAttr( instanceConfig, 'ignoreDnBit' )

   #---------------------------------------------------------------------------
   # AF mode specific cli handlers
   # - The AF specific handlers need not check if the attribute is configured
   #   in any other mode
   #---------------------------------------------------------------------------
   def setDefaultMetric( self, args ):
      instanceConfig = self._getInstanceConfig()
      defaultMetric  = args.get( 'METRIC_VALUE',
                                 instanceConfig.defaultMetricDefault )
      instanceConfig.defaultMetric = defaultMetric

   def setDistance( self, args ):
      instanceConfig = self._getInstanceConfig()
      if "intra-area" in args:
         distance = args.get( 'DISTANCE', instanceConfig.internalPrefDefault )
         instanceConfig.internalPref = distance
      elif "external" in args:
         distance = args.get( 'DISTANCE', instanceConfig.asePrefDefault )
         instanceConfig.asePref = distance

   def setRedistribute( self, proto, routeMapName=None,
                        isisLevel='levelNone', includeLeaked=False ):
      instanceConfig = self._getInstanceConfig()
      if routeMapName:
         redistribute = Tac.Value( 'Routing6::Ospf3::Redistribute',
                                   proto, routeMap=routeMapName,
                                   isisLevel=isisLevel,
                                   includeLeaked=includeLeaked )
      else:
         redistribute = Tac.Value( 'Routing6::Ospf3::Redistribute', proto,
                                   isisLevel=isisLevel,
                                   includeLeaked=includeLeaked )
      instanceConfig.redistributeConfig.addMember( redistribute )

   def noRedistribute( self, proto ):
      instanceConfig = self._getInstanceConfig()
      if proto in instanceConfig.redistributeConfig:
         del instanceConfig.redistributeConfig[ proto ]

   def setMaxEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.maxEcmp = args[ 'PATHS' ]

   def noMaxEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.maxEcmp = 0

   def setServiceAcl( self, args ):
      AclCliLib.setServiceAcl( self.mode_, OspfConsts.serviceName,
                               OspfConsts.IPPROTO_OSPF,
                               self.aclConfig_, self.aclCpConfig_,
                               args[ 'ACL_NAME' ], "ipv6", self.vrfName )

   def noServiceAcl( self, args ):
      AclCliLib.noServiceAcl( self.mode_, OspfConsts.serviceName,
                              self.aclConfig_, self.aclCpConfig_,
                              None, "ipv6", self.vrfName )

class RouterOspfv3SharedModelet( CliParser.Modelet, RouterOspf3Common ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

class RouterOspfv3AfSharedModelet( CliParser.Modelet, RouterOspf3Common ):
   '''
   This class is used to add commands specific to address family
   modes ( commands not allowed in global mode )
   '''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

class RouterOspfv3AfIpv6Modelet( CliParser.Modelet, RouterOspf3Common ):
   '''
   This class is used to add commands specific to ipv6 address family
   modes in router ospfv3 config mode.
   '''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

class RouterOspfv3AfIpv4Modelet( CliParser.Modelet, RouterOspf3Common ):
   '''
   This class is used to add commands specific to ipv4 address-family
   mode in router ospfv3 config mode.
   '''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

#-------------------------------------------------------------------------------
# router ospf3 config mode. A new instance of this mode is created when the
# user enters "ipv6 router ospf <process-id>
#-------------------------------------------------------------------------------
class RouterOspf3Mode( RoutingOspf3Mode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'OSPFv3 configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new RouterOspfMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, processId, vrfName ):
      self.processId = processId
      self.vrfName = vrfName
      self.afInstanceId = afIPv6InstanceID
      RoutingOspf3Mode.__init__( self, ( processId, vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# For old style cli - add the instance modelets
#-------------------------------------------------------------------------------
RouterOspf3Mode.addModelet( RouterOspfv3SharedModelet )
RouterOspf3Mode.addModelet( RouterOspfv3AfSharedModelet )
RouterOspf3Mode.addModelet( RouterOspfv3AfIpv6Modelet )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 router ospf <process-id> [vrf name]" command, in config mode.
#-------------------------------------------------------------------------------
def ospf3Config():
   return ospf3ConfigDir

def vrfNameToInstance( instanceConfig, vrf=DEFAULT_VRF ):
   for inst in instanceConfig.values():
      if inst.vrfName == vrf:
         return inst
   return None

def checkProcessToVrfMapping( mode, processId, vrfName ):
   instanceConfig = _getInstanceConfigByProcessId( processId )

   if processId and vrfName and \
      ( vrfName == ALL_VRF_NAME or \
         ( instanceConfig and instanceConfig.vrfName != vrfName ) ):
      mode.addError( 'OSPF process %d does not exist in vrf %s' % \
               ( processId, vrfName ) )
      return False
   return True

def getInstanceIdForInstance( processId, vrfName=None ):
   instanceConfig = _getInstanceConfigByProcessId( processId, vrfName=vrfName )
   if instanceConfig:
      assert instanceConfig.instanceId == 'afIPv6'
      return afIPv6InstanceID
   return None

def getInstanceIds( mode, processId, vrfName ):
   if not checkProcessToVrfMapping( mode, processId, vrfName ):
      return None
   if processId:
      instanceConfig = _getInstanceConfigByProcessId( processId )
      if instanceConfig is None:
         mode.addError( 'OSPF3 process %d does not exist' % processId )
         return None
      return [ processId ]
   else:
      if not vrfName:
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
         if vrfName is DEFAULT_VRF:
            vrfName = None
      if vrfName and vrfName != ALL_VRF_NAME:
         vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
         if not vrfConfig or afIPv6InstanceID not in vrfConfig.instanceConfig:
            mode.addError( 'OSPF3 process does not exist in vrf %s' % vrfName )
            return None
         processIds = [ vrfConfig.processId ]
      else:
         processIds = []
         for vc in ospf3Config().vrfConfig.itervalues():
            if afIPv6InstanceID in vc.instanceConfig:
               processIds += [ vc.processId ]
         processIds = list( sorted( processIds ) )
      return processIds

def getVrfList( mode, vrfName ):
   if vrfName and vrfName != ALL_VRF_NAME:
      vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
      if not vrfConfig or afIPv6InstanceID not in vrfConfig.instanceConfig:
         mode.addError( 'OSPF3 process does not exist in vrf %s' % vrfName )
         return None
      vrfNames = [ vrfName ]
   else:
      vrfNames = []
      for vc in ospf3Config().vrfConfig.itervalues():
         if afIPv6InstanceID in vc.instanceConfig:
            vrfNames += [ vc.vrfName ]
      vrfNames = list( sorted( vrfNames ) )
   return vrfNames

def getVrfToInstanceIdsDict( mode, vrfName ):
   vrfs = {}

   if not vrfName:
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      if vrfName is DEFAULT_VRF:
         vrfName = None

   if vrfName and vrfName != ALL_VRF_NAME:
      vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
      if not vrfConfig:
         mode.addError( 'OSPF3 process does not exist in vrf %s' % vrfName )
         return None
      vrfs[ vrfConfig.vrfName ] = sorted( vrfConfig.instanceConfig.keys() )
   else:
      for vc in ospf3Config().vrfConfig.itervalues():
         vrfs[ vc.vrfName ] = sorted( vc.instanceConfig.keys() )

   return vrfs

#-------------------------------------------------------------------------------
# APIs to check / dis-allow mix of old style and new cli configs
#-------------------------------------------------------------------------------
def checkConfigStyle( oldStyleCmd ):
   config = ospf3Config()
   if not config.vrfConfig and not config.intfConfig:
      t0( 'No Ospf3 configuration, can accept any style cli command' )
      return True

   t0( 'VRF Configs: %s' % config.vrfConfig.keys() )
   t0( 'Intf Configs: %s' % config.intfConfig.keys() )

   # Check the config style of the first vrf
   for _, vrf in config.vrfConfig.items():
      if vrf.processId:
         return oldStyleCmd
      else:
         return not oldStyleCmd

   # Check the config style of the first interface, incase there are no vrfs
   for _, intf in config.intfConfig.items():
      # We don't need to consider this interface for config style check if
      # configStyleSet is False
      # We chose to ignore the config style for the following commands:
      # "[ ospfv3 ipv6|ipv6 ospf] retransmit-interval <>"
      if not intf.configStyleSet:
         continue
      if intf.oldStyleCli:
         return oldStyleCmd
      else:
         return not oldStyleCmd

   return True

cliMismatchErrorMsg = 'OSPFv3 configuration style mismatch'
def getCliErrorMsg( ):
   # NOTE: This function should be called only after
   # checkConfigStyle() fails, this expects some
   # config to present
   msg = ''
   config = ospf3Config()
   for vrfName, vrfConfig in config.vrfConfig.items():
      msg += 'OSPFv3 instance in vrf %s' % vrfName
      if vrfConfig.processId:
         msg += ', configured with process id %d' % vrfConfig.processId
      else:
         msg += ", configured with 'router ospfv3' style config"
      return msg

   for intfName, intfConfig in config.intfConfig.items():
      msg += 'OSPFv3 interface %s' % intfName
      if intfConfig.oldStyleCli:
         msg += ", configured with 'ipv6 ospf ...' style config"
      else:
         msg += ", configured with 'ospfv3 ...' style config"
      return msg

   assert 0, 'No OSPFv3 configuration available'


#-------------------------------------------------------------------------------
# Functions / Helper functions to get the Ospf3 instance config given the instance
# parameters ( called from interface config functions typically )
#-------------------------------------------------------------------------------

# Check we either have processId or afInstanceId, else we cannot guess
# the ospfv3 instance. Also check if the config style matches the existing
# config style.
def checkInstanceCommand( processId, afInstanceId, vrfName=None ):
   oldStyleCmd = bool( processId )
   assert bool( processId ) or  bool( afInstanceId is not None )
   return checkConfigStyle( oldStyleCmd )

def _getVrfConfig( vrfName ):
   assert vrfName
   vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
   return vrfConfig

def getInstanceConfig( processId, afInstanceId, vrfName=None ):
   errMsg = None
   instanceConfig = None

   if not checkInstanceCommand(processId, afInstanceId, vrfName):
      errMsg = getCliErrorMsg()
      return None, errMsg

   if processId:
      instanceConfig = _getInstanceConfigByProcessId( processId )
      # Check if the vrfNames match in the instance config
      # and as specified in the command
      if vrfName and instanceConfig and \
         instanceConfig.vrfName != vrfName:
         errMsg = 'OSPFv3 process %d already exists in VRF %s' \
                   % ( processId, instanceConfig.vrfName )
         return None, errMsg
   else:
      vrfName = vrfName if vrfName else DEFAULT_VRF
      vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
      if vrfConfig:
         assert not vrfConfig.processId
         instanceConfig = vrfConfig.instanceConfig.get( afInstanceId, None )

   return instanceConfig, None

def deleteOspf3InstanceConfig( mode, vrf, afInstanceId=None ):
   assert mode

   vrf = vrf if vrf else DEFAULT_VRF
   if vrf in ospf3Config().vrfConfig:
      if afInstanceId is not None:
         if afInstanceId in ospf3Config().vrfConfig[ vrf ].instanceConfig:
            t2( 'Deleting vrf %s instance %s' % ( vrf, afInstanceId ) )
            del ospf3Config().vrfConfig[ vrf ].instanceConfig[ afInstanceId ]
         else:
            mode.addWarning( 'Could not find OSPF3 instance in vrf %s' % vrf )
            return False
      else:
         t2( 'Deleting vrf %s config' % ( vrf ) )
         del ospf3Config().vrfConfig[ vrf ]
   else:
      mode.addWarning( 'Could not find any OSPF3 instance in vrf %s' % vrf )
      return False
   return True

def cleanupOspf3InstanceConfig( mode, vrf, processId=None, afInstanceId=None ):
   if deleteOspf3InstanceConfig( mode, vrf, afInstanceId ):
      # Set interfaces part of the deleted Ospf instance as uninitialized.
      # passive config for interfaces is derive from the instance hence
      # cycle through all interfaces belonging to the instance and reset
      # the passive config
      afList = [ afInstanceId ] if afInstanceId else \
               [ afIPv4InstanceID, afIPv6InstanceID, afAllInstanceID ]
      for ifInstId in afList:
         for intfConfig in ospf3Config().intfConfig.itervalues():
            if ifInstId in intfConfig.intfInstanceConfig and \
               _checkIntfBelongsToInstance( intfConfig, processId,
                                            ifInstId, vrf ):
               intfInstanceConfig = intfConfig.intfInstanceConfig.get( ifInstId )
               if intfConfig.processId:
                  resetOspf3ConfigAttr( intfInstanceConfig, 'passive' )

#-------------------------------------------------------------------------------
# Create a OSPFv3 instance config, this function expects the caller to check
# that there is no existing config before calling this function
#-------------------------------------------------------------------------------
def createOspf3InstanceConfig( processId, afInstanceId, vrfName=None ):
   errMsg = None
   instanceConfig = None

   vrfName = vrfName if vrfName else DEFAULT_VRF
   vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
   if vrfConfig:
      # Old style cli
      if processId:
         errMsg = 'More than 1 OSPFv3 instance is not supported'
      elif vrfConfig.processId:
         errMsg = 'OSPFv3 instance exists in VRF %s, will old ' \
                  'style configuration' % vrfName
      elif afInstanceId in vrfConfig.instanceConfig:
         assert 0, 'Instance already exists in VRF %s address family %s' % \
                    ( vrfName, getAfFromInstanceId( afInstanceId ) )
      else:
         t2( 'Created new af instance in vrf %s, with instanceId %s' %
             ( vrfName, afInstanceId ) )
         instanceConfig = vrfConfig.instanceConfig.newMember(
                           afInstanceId, vrfName )
   else:
      if processId:
         t2( 'Created new instance in vrf %s, with processId %d' %
             ( vrfName, processId ) )
         vrfConfig = ospf3Config().vrfConfig.newMember( vrfName )
         vrfConfig.processId = processId
         instanceConfig = vrfConfig.instanceConfig.newMember(
                          afInstanceId, vrfName )
      else:
         # New style cli
         t2( 'Created new instance in vrf %s, with instanceId %s' %
             ( vrfName, afInstanceId ) )
         vrfConfig = ospf3Config().vrfConfig.newMember( vrfName )
         instanceConfig = vrfConfig.instanceConfig.newMember(
                           afInstanceId, vrfName )
   return instanceConfig, errMsg

#-------------------------------------------------------------------------------
# Add a list of areaIds to a instance config.
# These areaIds are typically auto-created when 'ipv6 ospf area x' or
# 'ospfv3 ipv4/ipv6 area ...' are entered
#-------------------------------------------------------------------------------
def addAreaIdsToInstanceConfig( instanceConfig, areaIdList ):
   for areaId in areaIdList:
      if areaId not in instanceConfig.areaConfig:
         if len( instanceConfig.areaConfig ) >= MAX_AREAS:
            return False
         else:
            instanceConfig.areaConfig.newMember( areaId )
   return True

#-------------------------------------------------------------------------------
# Loops through all interfaces and checks if any of them belong to the instance
# (expresssed as a 3 tuple processId, afInstanceId, vrfName ) and have an areaId
# configured
#-------------------------------------------------------------------------------
def getIntfAreaList( processId, afInstanceId, vrfName ):
   areaIdList = []
   config = ospf3Config()
   for name in config.intfConfig:
      intfConfig = config.intfConfig[ name ]
      # Check if the interface belongs to the instance
      if afInstanceId in intfConfig.intfInstanceConfig and \
         _checkIntfBelongsToInstance( intfConfig, processId,
                                      afInstanceId, vrfName ):
         intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
         if intfInstanceConfig.areaIdPresent and \
            intfInstanceConfig.areaId not in areaIdList:
            areaIdList.append( intfInstanceConfig.areaId )

   return  areaIdList


#-------------------------------------------------------------------------------
# Creates an AF instance config if we have interfaces which are associated
# with the AF. This function is called when creating the global instance
# config, so that we can create the ipv4 / ipv6 instance configs for the
# vrf
#-------------------------------------------------------------------------------
def createOspf3AfInstanceIfRequired( afInstanceId, vrfName ):
   errMsg = None
   instanceConfig = None
   assert afInstanceId != afAllInstanceID

   areaIdList = getIntfAreaList( None, afInstanceId, vrfName )
   if len( areaIdList ) >= MAX_AREAS:
      errMsg =  "Max number of areas (%d) exceeded for %s address family" % \
                ( MAX_AREAS, getAfFromInstanceId( afInstanceId ) )
   elif areaIdList:
      instanceConfig, errMsg = createOspf3InstanceConfig( None,
                                                          afInstanceId, vrfName )
      if instanceConfig:
         t2( 'Adding areas to instance ( vrf %s, instanceId %s )' %
             ( vrfName, afInstanceId ) )
         t2( areaIdList )
         addAreaIdsToInstanceConfig( instanceConfig, areaIdList )

   return instanceConfig, errMsg

#-------------------------------------------------------------------------------
# Issue a warning if routing is not enabled for the instance
#-------------------------------------------------------------------------------
def _warnIfRoutingDisabled( mode, vrfName, afInstanceId ):
   IraIp6Cli.warnIfRoutingDisabled( mode, vrfName=vrfName )
   if afInstanceId == afIPv6InstanceID:
      if not routing6HardwareStatus.routingSupported:
         mode.addWarning( "IPv6 Hardware forwarding is not supported "
                          "on this platform." )
         mode.addWarning( "This means that all IPv6 traffic will be "
                          "routed in software." )
   elif afInstanceId == afIPv4InstanceID:
      IraIpCli.warnIfRoutingDisabled( mode, vrfName=vrfName )
      if not routingHardwareStatus.routingSupported:
         mode.addWarning( "IPv4 Hardware forwarding is not supported "
                          "on this platform." )
         mode.addWarning( "This means that all IPv4 traffic will be "
                          "routed in software." )

#-------------------------------------------------------------------------------
# Return a ospf3 instanceConfig given the instance keys.
# - If processId is specified it assumed to be old style cli else new style
# - If there is a existing OSPFv3 instance return the same
# - Else create a new instance config
#   - check if we need to instantiate any areas
#   - if its a global config, check if we need to create address family instance
#-------------------------------------------------------------------------------
def setupOspfv3InstanceMode( mode, processId=None, vrfName=None,
                             afInstanceId=afIPv6InstanceID ):

   # First check if we have an instance config already,
   # we could just return it. If a new config has to be
   # created we will have iterate over all interfaces to get the
   # areaIds
   instanceConfig, errMsg = getInstanceConfig( processId, afInstanceId,
                                               vrfName=vrfName )
   if errMsg:
      mode.addError( errMsg )
      return None

   # Return the existing config
   if instanceConfig:
      _warnIfRoutingDisabled( mode, instanceConfig.vrfName, afInstanceId )
      return instanceConfig

   # There is no instance config, so create it
   instanceConfig, errMsg = createOspf3InstanceConfig( processId, afInstanceId,
                                                       vrfName )
   if not instanceConfig:
      mode.addError( errMsg )
      return None

   assert instanceConfig.vrfName
   vrfName = instanceConfig.vrfName
   _warnIfRoutingDisabled( mode, vrfName, afInstanceId )

   # Re-initialize interfaces that are part of the Ospf instance
   # Also if any interface has areaId set, create a areaConfig entity
   # for the areaId if one does not exist already

   # For global config, we need to instantiate the areas in either v6 or v4
   # address family instances, hence loop through the address families and
   # check if we do have any areas configured, if so instantiate the address
   # family specific ospf3 instance and add the areas
   if afInstanceId == afAllInstanceID:
      vrfConfig = _getVrfConfig( vrfName )

      # Check if we need to create a IPV4 instance
      if afIPv4InstanceID not in vrfConfig.instanceConfig:
         ipv4Instance, errMsg = createOspf3AfInstanceIfRequired( afIPv4InstanceID,
                                                                 vrfName )
         if errMsg:
            deleteOspf3InstanceConfig( mode, vrfName, afAllInstanceID )
            mode.addError( errMsg )
            return None

      # Check if we need to create a IPV4 instance
      if afIPv6InstanceID not in vrfConfig.instanceConfig:
         _, errMsg = createOspf3AfInstanceIfRequired( afIPv6InstanceID, vrfName )
         if errMsg:
            if ipv4Instance:
               deleteOspf3InstanceConfig( mode, vrfName, afIPv4InstanceID )
            deleteOspf3InstanceConfig( mode, vrfName, afAllInstanceID )
            mode.addError( errMsg )
            return None
   else:
      areaIdList = getIntfAreaList( processId, afInstanceId, vrfName )
      if len( areaIdList ) >= MAX_AREAS:
         mode.addError( "Max number of areas (%d) exceeded for %s address family" % \
                        ( MAX_AREAS, getAfFromInstanceId( afInstanceId ) ) )
         deleteOspf3InstanceConfig( mode, vrfName, afInstanceId )
         instanceConfig = None
      else:
         t2( 'Adding areas to instance ( vrf %s, instanceId %s )' %
             ( vrfName, afInstanceId ) )
         t2( areaIdList )
         addAreaIdsToInstanceConfig( instanceConfig, areaIdList )

   IraVrfCli.addAgentVrfEntry( vrfName, "Ospf3" )
   return instanceConfig

def gotoRouterOspf3Mode( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   inst = setupOspfv3InstanceMode( mode, processId, vrfName )
   if inst:
      vrfName = inst.vrfName
      childMode = mode.childMode( RouterOspf3Mode, processId=processId,
                                  vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

def deleteRouterOspf3Mode( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   for vrf, vrfConfig in ospf3Config().vrfConfig.iteritems():
      if vrfConfig.processId == processId:
         cleanupOspf3InstanceConfig( mode, vrf, processId )
         IraVrfCli.removeAgentVrfEntry( vrf, "Ospf3" )

ospfKw = CliMatcher.KeywordMatcher( 'ospf',
      helpdesc='OSPF protocol' )
processMin = 1
processMax = 65535
matcherProcessId = CliMatcher.IntegerMatcher( processMin, processMax,
                                              helpdesc='Process identifier' )
vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name',
                                        inclDefaultVrf=True,
                                        inclAllVrf=True )
class Ipv6RouterOspfProcessId( CliCommand.CliCommandClass ):
   syntax = 'ipv6 router ospf PROCESS_ID [ VRF ]'
   noOrDefaultSyntax = syntax
   data = {
         'ipv6'       : CliToken.Ipv6.ipv6MatcherForConfig,
         'router'     : routerMatcherForConfig,
         'ospf'       : ospfKw,
         'PROCESS_ID' : matcherProcessId,
         'VRF'        : vrfExprFactory,
          }
   handler = gotoRouterOspf3Mode
   noOrDefaultHandler = deleteRouterOspf3Mode

BasicCli.GlobalConfigMode.addCommandClass( Ipv6RouterOspfProcessId )

#-------------------------------------------------------------------------------
# "no|default area <area-id>" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3NoAreaCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'area AREA_ID'

   data = {
         'area': areaKw,
         'AREA_ID': AreaIdExpression,
         }

   noOrDefaultHandler = RouterOspfv3SharedModelet.noArea

RouterOspfv3SharedModelet.addCommandClass( Ospf3NoAreaCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> default-cost <cost>" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3DefaultCostCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID default-cost COST'
   noOrDefaultSyntax = 'area AREA_ID default-cost ...'

   data = {
         'area': areaKw,
         'AREA_ID': AreaIdExpression,
         'default-cost' : defaultCostKw,
         'COST': matcherRouteCostMetric,
         }

   handler = RouterOspfv3SharedModelet.setDefaultCost
   noOrDefaultHandler = RouterOspfv3SharedModelet.noDefaultCost

RouterOspfv3SharedModelet.addCommandClass( Ospf3DefaultCostCmd )

#-------------------------------------------------------------------------------
# "area <area-id> stub" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3AreaStubCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID stub'

   data = {
         'area': areaKw,
         'AREA_ID': AreaIdExpression,
         'stub' : stubKw,
         }

   handler = RouterOspfv3SharedModelet.setStub

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaStubCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> stub no-summary" command, in ospf mode.
#-------------------------------------------------------------------------------
class RouterOspf3AreaStubNoSummary( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID stub no-summary'
   noOrDefaultSyntax = 'area AREA_ID stub [ no-summary ]'

   data = {
         'area': areaKw,
         'AREA_ID': AreaIdExpression,
         'stub' : stubKw,
         'no-summary' : stubNoSummaryKw,
         }

   handler = RouterOspfv3SharedModelet.setStubNoSummary
   noOrDefaultHandler = RouterOspfv3SharedModelet.noStub

RouterOspfv3SharedModelet.addCommandClass( RouterOspf3AreaStubNoSummary )

#------------------------------------------------------------------------
# "[no|default] area <area-id> ( nssa | not-so-stubby ) [ no-summary ]"
# command, in ospf mode.
#------------------------------------------------------------------------
class Ospf3AreaNssaCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID ( nssa | not-so-stubby ) [ no-summary ]'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'no-summary': nssaNoSummaryKw,
          }
   handler = RouterOspfv3SharedModelet.setNssa
   noOrDefaultHandler = RouterOspfv3SharedModelet.noNssa

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaNssaCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> nssa"
# command, in ospf mode.
# "[no|default] area <area-id> nssa default-information-originate [metric ...]
#               [metric-type ...] [nssa-only]"
#-------------------------------------------------------------------------------
class Ospf3AreaNssaDefaultInfoOriginCmd( CliCommand.CliCommandClass ):
   syntax = '''area AREA_ID ( nssa | not-so-stubby ) default-information-originate
               [ { ( metric METRIC )
                 | ( metric-type METRIC_TYPE )
                 | ( nssa-only ) } ]'''
   noOrDefaultSyntax = '''area AREA_ID ( nssa | not-so-stubby )
                          default-information-originate ...'''
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'default-information-originate': 'Originate default Type 7 LSA',
           'metric' : CliCommand.singleKeyword( 'metric',
              helpdesc='Metric for default route' ),
           'METRIC': CliMatcher.IntegerMatcher( metricMin, metricMax,
              helpdesc='Value of the route metric' ),
           'metric-type': CliCommand.singleKeyword( 'metric-type',
              helpdesc='Metric type for default route' ),
           'METRIC_TYPE': CliMatcher.IntegerMatcher( 1, 2, helpdesc='Metric type' ),
           'nssa-only': CliCommand.singleKeyword( 'nssa-only',
              helpdesc='Limit default advertisement to this NSSA area' ),
          }
   handler = RouterOspfv3SharedModelet.setNssa
   noHandler = RouterOspfv3SharedModelet.noNssaDefInfoOrigin
   defaultHandler = RouterOspfv3SharedModelet.noNssa

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaNssaDefaultInfoOriginCmd )

#------------------------------------------------------------------------
# "[no|default] area <area-id> ( nssa | not-so-stubby ) nssa-only"
# command, in ospf mode.
#------------------------------------------------------------------------
class Ospf3AreaNssaNssaOnlyCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID ( nssa | not-so-stubby ) nssa-only'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'not-so-stubby': notsostubbyKw,
           'nssa-only': nssaOnlyKw,
          }
   handler = RouterOspfv3SharedModelet.setNssa
   noOrDefaultHandler = RouterOspfv3SharedModelet.noNssa

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaNssaNssaOnlyCmd )

#-------------------------------------------------------------------------------
# "[no|default] area <area-id> not-so-stubby lsa type-7 convert type-5"
# Legacy:
# "[no|default] area <area-id> nssa translate type7 [always]"
# command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3AreaType7toType5Cmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID not-so-stubby lsa type-7 convert type-5'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'not-so-stubby': notsostubbyKw,
           'lsa': 'Not-so-stubby LSA parameters',
           'type-7': 'Not-so-stubby Type-7 LSA parameters',
           'convert': 'Translation of not-so-stubby Type-7 LSAs',
           'type-5': 'Always translate Type-7 LSAs to Type-5',
          }
   handler = RouterOspfv3SharedModelet.setNssa
   noOrDefaultHandler = RouterOspfv3SharedModelet.noNssa

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaType7toType5Cmd )

class Ospf3AreaNssaTranslateCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID nssa translate type7 always'
   noOrDefaultSyntax = syntax
   data = {
           'area': areaKw,
           'AREA_ID': AreaIdExpression,
           'nssa': nssaKw,
           'translate':  CliCommand.Node( CliMatcher.KeywordMatcher( 'translate',
              helpdesc='Enable LSA translation' ), deprecatedByCmd='area AREA_ID'
              ' not-so-stubby lsa type-7 convert type-5' ),
           'type7': 'Enable Translation of Type-7 to Type-5 LSAs',
           'always': 'Always translate LSAs',
          }
   handler = RouterOspfv3SharedModelet.setNssa
   noOrDefaultHandler = RouterOspfv3SharedModelet.noNssa

RouterOspfv3SharedModelet.addCommandClass( Ospf3AreaNssaTranslateCmd )

#------------------------------------------------------------------------------------
# "[no|default] area <area-id> range <ip-address> <mask> [advertise | not-advertise]"
# command, in ospf mode.
#------------------------------------------------------------------------------------
class Ospf3Ipv6AreaRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID range PREFIX [ advertise | not-advertise ] [ cost COST ]'
   noOrDefaultSyntax = syntax
   data = {
          'area': areaKw,
          'AREA_ID': AreaIdExpression,
          'range': 'Configure route summarization',
          'PREFIX': Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 address prefix',
             overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
          'advertise': 'Enable advertisement of the range',
          'not-advertise': 'Disable advertisement of the range',
          'cost': costKw,
          'COST': matcherRangeCost,
   }
   handler = RouterOspfv3AfIpv6Modelet.setRange
   noOrDefaultHandler = RouterOspfv3AfIpv6Modelet.noRange

RouterOspfv3AfIpv6Modelet.addCommandClass( Ospf3Ipv6AreaRangeCmd )

class Ospf3Ipv4AreaRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'area AREA_ID range PREFIX [ advertise | not-advertise ] [ cost COST ]'
   noOrDefaultSyntax = syntax
   data = {
          'area': areaKw,
          'AREA_ID': AreaIdExpression,
          'range': 'Configure route summarization',
          'PREFIX': filterPrefixMatcher,
          'advertise': 'Enable advertisement of the range',
          'not-advertise': 'Disable advertisement of the range',
          'cost': costKw,
          'COST': matcherRangeCost,
          }
   handler = RouterOspfv3AfIpv4Modelet.setRange
   noOrDefaultHandler = RouterOspfv3AfIpv4Modelet.noRange

RouterOspfv3AfIpv4Modelet.addCommandClass( Ospf3Ipv4AreaRangeCmd )
#-------------------------------------------------------------------------------
# "[no|default] router-id <ip-address>" command, in ospf mode.
#
# The no rule shouldn't have trailingGarbage, or else 'no router ospf 1'
# executed in router-ospf mode matches the 'no router-id' rule.
#-------------------------------------------------------------------------------
class Ospf3RouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ROUTER_ID'
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
         'router-id' : routerIdKw,
         'ROUTER_ID' : matcherRouterId,
         }
   handler = RouterOspfv3SharedModelet.setRouterId
   noOrDefaultHandler = RouterOspfv3SharedModelet.noRouterId

RouterOspfv3SharedModelet.addCommandClass( Ospf3RouterIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] log-adjacency-changes [detail]" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3LogAdjacencyChangesCmd( CliCommand.CliCommandClass ):
   syntax = 'log-adjacency-changes [ DETAIL ]'
   noOrDefaultSyntax = 'log-adjacency-changes ...'
   data = {
         'log-adjacency-changes' : logAdjacencyChangesKw,
         'DETAIL' : logAdjacencyChangesDetailKw,
         }
   handler = RouterOspfv3SharedModelet.setLogAdjacencyChanges
   defaultHandler = RouterOspfv3SharedModelet.setLogAdjacencyChanges
   noHandler = RouterOspfv3SharedModelet.noLogAdjacencyChanges

RouterOspfv3SharedModelet.addCommandClass( Ospf3LogAdjacencyChangesCmd )

#-------------------------------------------------------------------------------
# "[no|default] default-metric <metric-value>" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3DefaultMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'default-metric METRIC_VALUE'
   noOrDefaultSyntax = 'default-metric ...'
   data = {
         'default-metric' : defaultMetricKw,
         'METRIC_VALUE' : matcherDefaultMetric,
         }
   handler = RouterOspfv3AfSharedModelet.setDefaultMetric
   noOrDefaultHandler = handler

RouterOspfv3AfSharedModelet.addCommandClass( Ospf3DefaultMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] distance ospf intra-area <dist>" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3DistanceOspfCmd( CliCommand.CliCommandClass ):
   syntax = 'distance ospf ( intra-area | external ) DISTANCE'
   noOrDefaultSyntax = 'distance ospf ( intra-area | external ) ...'
   data = {
         'distance' : distanceKw,
         'ospf' : ospfKw,
         'intra-area' : intraAreaKw,
         'external': externalRtKw,
         'DISTANCE' : matcherDistance,
         }
   handler = RouterOspfv3AfSharedModelet.setDistance
   noOrDefaultHandler = handler

RouterOspfv3AfSharedModelet.addCommandClass( Ospf3DistanceOspfCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers spf <spf-time>" command, in ospf mode.
# Hidden as its deprecated, should use timers spf delay initial instead
# Available only with old-style CLI
#-------------------------------------------------------------------------------
timersKw = 'Configure OSPFv3 timers'
matcherSpfHidden = CliCommand.Node( matcher=matcherSpf, hidden=True )

class Ospf3TimersSpfCmd( CliCommand.CliCommandClass ):
   syntax = 'timers spf SECONDS'
   data = {
         'timers' : timersKw,
         'spf': matcherSpfHidden,
         'SECONDS' : matcherSpfTimer,
           }
   handler = RouterOspfv3SharedModelet.setTimerSPF

RouterOspf3Mode.addCommandClass( Ospf3TimersSpfCmd )

#------------------------------------------------------------------------------
# "[no|default] timers spf delay initial <sfp-start> <sfp-hold> <sfp-max-wait>"
# Legacy:
# "[no|default] timers throttle spf <sfp-start> <sfp-hold> <sfp-max-wait>"
#------------------------------------------------------------------------------
class Ospf3TimersDelayInitialCmd( CliCommand.CliCommandClass ):
   syntax = 'timers spf delay initial SPF_START SPF_HOLD SPF_MAX_WAIT'
   noOrDefaultSyntax = 'timers spf ...'
   data = {
          'timers' : timersKw,
          'spf': 'Configure SPF timers',
          'delay': 'Configure SPF timers',
          'initial': 'Configure SPF timers',
          'SPF_START': matcherSpfStart,
          'SPF_HOLD': matcherSpfHold,
          'SPF_MAX_WAIT': matcherSpfMaxWait,
          }
   handler = RouterOspfv3SharedModelet.setSpfIntervalThrottle
   noOrDefaultHandler = RouterOspfv3SharedModelet.noSpfIntervalThrottle

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersDelayInitialCmd )

class Ospf3TimersThrottleSpfCmd( CliCommand.CliCommandClass ):
   syntax = 'timers throttle spf SPF_START SPF_HOLD SPF_MAX_WAIT'
   noOrDefaultSyntax = 'timers throttle spf ...'
   data = {
          'timers' : timersKw,
          'throttle': 'Configure OSPFv3 throttle timers',
          'spf':  CliCommand.Node( matcherSpf,
                    deprecatedByCmd='timers spf delay initial' ),
          'SPF_START': matcherSpfStart,
          'SPF_HOLD': matcherSpfHold,
          'SPF_MAX_WAIT': matcherSpfMaxWait,
          }
   handler = RouterOspfv3SharedModelet.setSpfIntervalThrottle
   noOrDefaultHandler = RouterOspfv3SharedModelet.noSpfIntervalThrottle

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersThrottleSpfCmd )

#------------------------------------------------------------------------------
# "[no|default] timers lsa rx min interval <msecs>"
# Legacy:
# "[no|default] timers lsa arrival <msecs>"
#------------------------------------------------------------------------------
class Ospf3TimersLsaRxMinIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa rx min interval MILLISECONDS'
   noOrDefaultSyntax = 'timers lsa rx min interval ...'
   data = {
          'timers' : timersKw,
          'lsa': 'Configure OSPFv3 LSA timers',
          'rx': 'Configure OSPFv3 LSA receiving timers',
          'min':'Configure OSPFv3 LSA arrival timer',
          'interval': 'Configure OSPFv3 LSA arrival timer',
          'MILLISECONDS': matcherLsaArrivalInt,
          }
   handler = RouterOspfv3SharedModelet.setLsaArrivalInterval
   noOrDefaultHandler = RouterOspfv3SharedModelet.noLsaArrivalInterval

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersLsaRxMinIntervalCmd )

class Ospf3TimersLsaArrivalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa arrival MILLISECONDS'
   noOrDefaultSyntax = 'timers lsa arrival ...'
   data = {
          'timers' : timersKw,
          'lsa': 'Configure OSPFv3 LSA timers',
          'arrival': CliCommand.Node( matcherArrival, deprecatedByCmd='timers lsa '
                                      'rx min interval' ),
          'MILLISECONDS': matcherLsaArrivalInt,
          }
   handler = RouterOspfv3SharedModelet.setLsaArrivalInterval
   noOrDefaultHandler = RouterOspfv3SharedModelet.noLsaArrivalInterval

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersLsaArrivalCmd )

#------------------------------------------------------------------------------
# "[no|default] timers lsa tx delay initial <lsa-start> <lsa-hold> <lsa-max-wait>"
# Legacy:
# "[no|default] timers throttle lsa all <lsa-start> <lsa-hold> <lsa-max-wait>"
#------------------------------------------------------------------------------
class Ospf3TimersLsaTxDelayInitialCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsa tx delay initial LSA_START LSA_HOLD LSA_MAX_WAIT'
   noOrDefaultSyntax = 'timers lsa tx delay initial ...'
   data = {
          'timers' : timersKw,
          'lsa': 'Configure OSPFv3 LSA timers',
          'tx': 'Configure OSPFv3 LSA transmission timers',
          'delay': 'Configure SPF timers',
          'initial': 'Configure SPF timers',
          'LSA_START': matcherLsaStart,
          'LSA_HOLD': matcherLsaHold,
          'LSA_MAX_WAIT': matcherLsaMaxWait,
          }
   handler = RouterOspfv3SharedModelet.setLsaIntervalThrottle
   noOrDefaultHandler = RouterOspfv3SharedModelet.noLsaIntervalThrottle

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersLsaTxDelayInitialCmd )

class Ospf3TimersThrottleLsaAllCmd( CliCommand.CliCommandClass ):
   syntax = 'timers throttle lsa all LSA_START LSA_HOLD LSA_MAX_WAIT'
   noOrDefaultSyntax = 'timers throttle lsa all ...'
   data = {
          'timers' : timersKw,
          'throttle': 'Configure OSPFv3 throttle timers',
          'lsa': CliCommand.Node( matcherLsa, deprecatedByCmd='timers lsa tx '
             'delay initial' ),
          'all': 'For all type of OSPF LSAs',
          'LSA_START': matcherLsaStart,
          'LSA_HOLD': matcherLsaHold,
          'LSA_MAX_WAIT': matcherLsaMaxWait,
          }
   handler = RouterOspfv3SharedModelet.setLsaIntervalThrottle
   noOrDefaultHandler = RouterOspfv3SharedModelet.noLsaIntervalThrottle

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersThrottleLsaAllCmd )

#------------------------------------------------------------------------------
# "[no|default] timers pacing flood <milliseconds>"
#------------------------------------------------------------------------------
class Ospf3TimersPacingFloodCmd( CliCommand.CliCommandClass ):
   syntax = 'timers pacing flood MILLISECONDS'
   noOrDefaultSyntax = 'timers pacing flood ...'
   data = {
          'timers' : timersKw,
          'pacing': 'Configure OSPFv3 packet pacing',
          'flood': 'Configure OSPFv3 flood pacing',
          'MILLISECONDS': matcherFloodPacing,
          }
   handler = RouterOspfv3SharedModelet.setFloodPacing
   noOrDefaultHandler = RouterOspfv3SharedModelet.noFloodPacing

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersPacingFloodCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers out-delay <lsa out-delay>, in ospf mode
#-------------------------------------------------------------------------------
class Ospf3TimersOutDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers out-delay LSA_OUT_DELAY'
   noOrDefaultSyntax = 'timers out-delay ...'
   data = {
          'timers' : timersKw,
          'out-delay': outDelayKw,
          'LSA_OUT_DELAY': matcherLsaOutDelay,
          }
   handler = RouterOspfv3SharedModelet.setOutDelayTimer
   noOrDefaultHandler = RouterOspfv3SharedModelet.noOutDelayTimer

RouterOspfv3SharedModelet.addCommandClass( Ospf3TimersOutDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart" in ospf mode
#------------------------------------------------------------------------------
if not ospf3GrSupported():
   gracefulRestartKw = CliCommand.Node( gracefulRestartKw, hidden=True )
   gracePeriodKw = CliCommand.Node( gracePeriodKw, hidden=True )

class Ospf3GracefulRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart'
   noOrDefaultSyntax = syntax
   data = { 'graceful-restart' : gracefulRestartKw }
   handler = RouterOspfv3SharedModelet.setGracefulRestart
   noOrDefaultHandler = RouterOspfv3SharedModelet.noGracefulRestart

RouterOspfv3SharedModelet.addCommandClass( Ospf3GracefulRestartCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart grace-period <seconds>" in ospf mode
#------------------------------------------------------------------------------
class Ospf3GRGracePeriodCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart grace-period GRACE_PERIOD'
   noOrDefaultSyntax = 'graceful-restart grace-period ...'
   data = { 'graceful-restart'  : gracefulRestartKw,
            'grace-period'      : gracePeriodKw,
            'GRACE_PERIOD'      : matcherGracePeriod
          }
   handler = RouterOspfv3SharedModelet.setGrGracePeriod
   noOrDefaultHandler = RouterOspfv3SharedModelet.noGrGracePeriod

RouterOspfv3SharedModelet.addCommandClass( Ospf3GRGracePeriodCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper" in ospf mode
#------------------------------------------------------------------------------
if not ospf3GrSupported():
   gracefulRestartHelperKw = CliCommand.Node( gracefulRestartHelperKw, hidden=True )

class Ospf3GRHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart-helper'
   noOrDefaultSyntax = syntax
   data = { 'graceful-restart-helper' : gracefulRestartHelperKw }
   handler = RouterOspfv3SharedModelet.setGrHelper
   noOrDefaultHandler = RouterOspfv3SharedModelet.noGrHelper

RouterOspfv3SharedModelet.addCommandClass( Ospf3GRHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] shutdown" command, in ospf mode.
#-------------------------------------------------------------------------------
class Ospf3ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = { 'shutdown': 'Disable the OSPFv3 instance' }
   handler = RouterOspfv3SharedModelet.setShutdown
   noOrDefaultHandler = RouterOspfv3SharedModelet.noShutdown

RouterOspfv3SharedModelet.addCommandClass( Ospf3ShutdownCmd )

matcherRedistribute = CliMatcher.KeywordMatcher( 'redistribute',
                                                 'Redistribute routes with OSPFv3' )
matcherInclude = CliMatcher.KeywordMatcher( 'include',
                  helpdesc='Include leaked routes' )
matcherLeaked = CliMatcher.KeywordMatcher( 'leaked',
                  helpdesc='Include leaked routes' )

class RedistributeCommandBase( CliCommand.CliCommandClass ):
   """Helper base class to keep common methods required by all
   redistribute commands.
   """
   @staticmethod
   def adapter( mode, args, argsList ):
      protoString = {
         'static' : 'protoStatic',
         'connected' : 'protoDirect',
         'bgp' : 'protoBgp',
         'isis' : 'protoIsis'
      }
      args[ 'PROTOCOL' ] = protoString[ args[ 'PROTOCOL' ] ]


   @staticmethod
   def handler( mode, args ):
      modelet = mode.modeletMap[ RouterOspfv3SharedModelet ]
      modelet.setRedistribute( args[ 'PROTOCOL' ] ,
                               routeMapName=args.get( 'MAPNAME' ),
                               isisLevel=args.get( 'isisLevel', 'levelNone' ),
                               includeLeaked=( 'leaked' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      modelet = mode.modeletMap[ RouterOspfv3SharedModelet ]
      modelet.noRedistribute( args[ 'PROTOCOL' ] )

#---------------------------------------------------------------------------------
# [no|default] redistribute <static|connected> [include leaked]
#              [route-map <routemap>] command, in "router-ospf" mode.
#---------------------------------------------------------------------------------

redistProtocols = {
               'static' : 'Static routes',
               'connected' : 'Connected interface routes',
            }
if Toggles.OspfToggleLib.toggleBgpLeakedRedistEnabled():
   redistProtocols[ 'bgp' ] = 'BGP routes'

class RedistributeProtocolCommand( RedistributeCommandBase ):
   if Toggles.Ospf3ToggleLib.toggleRedistLeakedIntoOspf3Phase1Enabled():
      syntax = 'redistribute PROTOCOL [ include leaked ] [ route-map MAPNAME ]'
   else:
      syntax = 'redistribute PROTOCOL [ route-map MAPNAME ]'

   noOrDefaultSyntax = 'redistribute PROTOCOL ...'

   data = {
         'redistribute' : matcherRedistribute,
         'PROTOCOL'     : CliMatcher.EnumMatcher( redistProtocols ),
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }
   if Toggles.Ospf3ToggleLib.toggleRedistLeakedIntoOspf3Phase1Enabled():
      data.update( {
          'include'      : matcherInclude,
          'leaked'       : matcherLeaked,
      } )
RouterOspfv3AfSharedModelet.addCommandClass( RedistributeProtocolCommand )

#---------------------------------------------------------------------------------
# [no|default] redistribute bgp route-map <routemap>" command,
# in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeBgpCommand( RedistributeCommandBase ):
   syntax = 'redistribute PROTOCOL [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute PROTOCOL ...'

   data = {
         'redistribute': matcherRedistribute,
         'PROTOCOL': CliMatcher.EnumMatcher( {
                        'bgp': 'BGP routes',
                        } ),
         'route-map': RouteMapMatchers.routeMapApplication,
         'MAPNAME': mapNameMatcher,
   }

#delete RedistributeBgpCommand, once toggleLBgpLeakedRedist is enabled or removed.
if not Toggles.OspfToggleLib.toggleBgpLeakedRedistEnabled():
   RouterOspfv3AfSharedModelet.addCommandClass( RedistributeBgpCommand )

#---------------------------------------------------------------------------------
# [no|default] redistribute isis [include leaked] [level-1|level-1-2|level-2]
#              [route-map <route-map-name>] command, in "ipv6 router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeIsisCommand( RedistributeCommandBase ):
   if Toggles.Ospf3ToggleLib.toggleRedistLeakedIntoOspf3Phase1Enabled():
      syntax = ( 'redistribute isis [ include leaked ] '
                 '[ level-1|level-2|level-1-2 ] [ route-map MAPNAME ]' )
   else:
      syntax = ( 'redistribute isis [ level-1|level-2|level-1-2 ] '
                 '[ route-map MAPNAME ]' )
   noOrDefaultSyntax = 'redistribute isis ...'

   data = {
         'redistribute' : matcherRedistribute,
         'isis'         : 'IS-IS routes',
         'level-1'      : 'Redistribute IS-IS level-1 routes',
         'level-2'      : 'Redistribute IS-IS level-2 routes',
         'level-1-2'    : 'Redistribute IS-IS level-1 and level-2 routes',
         'route-map'    : RouteMapMatchers.routeMapApplication,
         'MAPNAME'      : mapNameMatcher
   }
   if Toggles.Ospf3ToggleLib.toggleRedistLeakedIntoOspf3Phase1Enabled():
      data.update( {
          'include'      : matcherInclude,
          'leaked'       : matcherLeaked,
      } )

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PROTOCOL' ] = 'protoIsis'
      args[ 'isisLevel' ] = 'level2'
      if 'level-1' in args:
         args[ 'isisLevel' ] = 'level1'
      elif 'level-2' in args:
         args[ 'isisLevel' ] = 'level2'
      elif 'level-1-2' in args:
         args[ 'isisLevel' ] = 'level1_2'

RouterOspfv3AfSharedModelet.addCommandClass( RedistributeIsisCommand )

#---------------------------------------------------------------------------------
# [no|default] redistribute dhcp route-map <routemap>" command,
# in "router-ospf" mode.
#---------------------------------------------------------------------------------
class RedistributeDhcpCommand( RedistributeCommandBase ):
   syntax = 'redistribute dhcp [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute dhcp ...'

   data = {
         'redistribute': matcherRedistribute,
         'dhcp': 'DHCPv6 routes',
         'route-map': RouteMapMatchers.routeMapApplication,
         'MAPNAME': mapNameMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PROTOCOL' ] = 'protoDhcp'

if Toggles.Ospf3ToggleLib.toggleRedistDhcpv6IntoOspf3Enabled():
   RouterOspfv3AfIpv6Modelet.addCommandClass( RedistributeDhcpCommand )
   RouterOspf3Mode.addCommandClass( RedistributeDhcpCommand )

#-------------------------------------------------------------------------------
# [no|default] "maximum-paths <paths>" command, in "router-ospf" mode.
#-------------------------------------------------------------------------------
def maxEcmpRangeFn( mode ):
   # This code can be called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb. Allow any value if routing6HardwareStatus
   # has not been initialized yet.
   if routing6HardwareStatus is None or \
      routing6HardwareStatus.maxEcmp == 0:
      return( 1, 0xffffffff )
   else:
      return( 1, routing6HardwareStatus.maxEcmp )

class Ospf3MaxPathsCmd( CliCommand.CliCommandClass ):
   syntax = 'maximum-paths PATHS'
   noOrDefaultSyntax = 'maximum-paths ...'
   data = { 'maximum-paths': 'Maximum number of next-hops in an ECMP route',
            'PATHS': maxEcmpPathsMatcher,
          }

   handler = RouterOspfv3AfSharedModelet.setMaxEcmp
   noOrDefaultHandler = RouterOspfv3AfSharedModelet.noMaxEcmp

RouterOspfv3AfSharedModelet.addCommandClass( Ospf3MaxPathsCmd )


#-------------------------------------------------------------------------------
# "[no|default] adjacency exchange-start threshold <num>" command, in
#  router-ospf3 mode.
#-------------------------------------------------------------------------------
class Ospf3AdjExchStartThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency exchange-start threshold THRESHOLD'
   noOrDefaultSyntax = 'adjacency exchange-start threshold ...'
   data = {
          'adjacency': adjacencyKw,
          'exchange-start': exchangeStartKw,
          'threshold': thresholdKw,
          'THRESHOLD': matcherSimultaneousPeers,
          }
   handler = RouterOspfv3SharedModelet.setExchStartThreshold
   noOrDefaultHandler = RouterOspfv3SharedModelet.noExchStartThreshold

RouterOspfv3SharedModelet.addCommandClass( Ospf3AdjExchStartThresholdCmd )

class Ospf3Intf( IntfCli.IntfDependentBase ):

   def setDefault( self ):
      config = ospf3Config()
      intfConfig = config.intfConfig.get( self.intf_.name )
      if not intfConfig:
         return
      # Interface got deleted, do remove it from intfConfig
      t2( 'Deleting intf %s config' % self.intf_.name )
      del config.intfConfig[ self.intf_.name ]

      # Iterate over v4 & v6 instances to clean up area, for global
      # config this is not required
      for afInstanceId in intfConfig.intfInstanceConfig:
         # Process id will be used for old style cli
         processId = intfConfig.processId
         intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
         areaId =  intfInstanceConfig.areaId
         if intfInstanceConfig.areaIdPresent:
            _deleteAreaConfigFromInstance( areaId, processId, afInstanceId,
                                           intfConfig.vrfName )

intfModelet = RoutingProtocolOspfv3IntfConfigModelet

def getDefaultAfInstanceId():
   return afIPv6InstanceID

def getl3Config():
   return l3Config

#-------------------------------------------------------------------------------
# Return the interface config
#-------------------------------------------------------------------------------
def _getIntfConfig( intfName ):
   intfConfig = ospf3Config().intfConfig.get( intfName, None )
   return intfConfig

#-------------------------------------------------------------------------------
# Create / return a interface instance config
#-------------------------------------------------------------------------------
def _getIntfConfigForAttr( mode, attr, afInstanceId=None, processId=None,
                           ignoreConfigStyle=False ):
   assert attr
   return _getOrCreateIntfInstanceConfig( mode, afInstanceId, processId, attr=attr,
                                          ignoreConfigStyle=ignoreConfigStyle )

def addrFamilyOverlapCheck( mode, attr, intfConfig, afInstanceId=None ):
   if afInstanceId == afAllInstanceID:
      v4IntfConfig = intfConfig.intfInstanceConfig.get( afIPv4InstanceID, None )
      if v4IntfConfig and getattr( v4IntfConfig, attr + "Present", False ):
         mode.addError( 'Attribute already configured in ipv4 address family' )
         return False
      v6IntfConfig = intfConfig.intfInstanceConfig.get( afIPv6InstanceID, None )
      if v6IntfConfig and getattr( v6IntfConfig, attr + "Present", False ):
         mode.addError( 'Attribute already configured in ipv6 address family' )
         return False
   else:
      globalIntfConfig = intfConfig.intfInstanceConfig.get( afAllInstanceID,
                                                               None )
      if globalIntfConfig and \
         getattr( globalIntfConfig, attr + "Present", False ):
         mode.addError( 'Attribute already configured in global config' )
         return False
   return True

def _createIntfInstanceConfig( mode, afInstanceId=None, processId=None,
                               ignoreConfigStyle=False ):
   intfName = mode.intf.name
   config = ospf3Config()
   oldStyleCmd = bool( processId or afInstanceId is None )

   # We only check for the mixing of cli's if ignoreConfigStyle is not True
   if not ignoreConfigStyle and not checkConfigStyle( oldStyleCmd ):
      mode.addError( getCliErrorMsg() )
      return None

   t2( 'Created intf %s config ( oldStyle : %s )' % ( intfName, oldStyleCmd ) )
   intfConfig = config.intfConfig.newMember( intfName )

   if not ignoreConfigStyle:
      # oldStyleCli is effective
      intfConfig.oldStyleCli = oldStyleCmd
      intfConfig.configStyleSet = True

   intfConfig.vrfName = _getIntfVrf( intfName )

   # processId is passed ( actually available ) only when setting an
   # area, if passed set it in the interface config
   if processId:
      intfConfig.processId = processId

   # Create the intfInstance Config
   # Old cli case, afInstanceId will be none, set it to v6
   afInstanceId = afInstanceId or getDefaultAfInstanceId()
   t2( 'Created intf %s, instance %s config' % ( intfName, afInstanceId ) )
   return intfConfig.intfInstanceConfig.newMember( afInstanceId, intfName )

def _getOrCreateIntfInstanceConfig( mode, afInstanceId=None, processId=None,
                                    create=True, attr=None,
                                    ignoreConfigStyle=False ):
   intfName = mode.intf.name
   oldStyleCmd = bool( processId or afInstanceId is None )

   # Only one of them should be passed in. processId in case
   # of old style cli, and afInstanceId in case of new style
   # NOTE: afInstanceId=0 is valid
   if processId:
      assert afInstanceId is None

   # Check if we already have a interface config
   intfConfig = _getIntfConfig( intfName )

   if intfConfig is None:
      if not create:
         return None
      return _createIntfInstanceConfig( mode=mode, afInstanceId=afInstanceId,
                                        processId=processId,
                                        ignoreConfigStyle=ignoreConfigStyle )
   else:
      t5( 'Checking for interface %s ( af: %s, processId: %d, oldStyle: %s)' %
          ( intfName, afInstanceId, intfConfig.processId, oldStyleCmd ) )

      # We have an interface configuration and ignoreConfigStyle not set
      # check for the mixing of cli's
      if not ignoreConfigStyle:
         if not checkConfigStyle( oldStyleCmd ) :
            mode.addError( getCliErrorMsg() )
            return None

      # Old cli case, afInstanceId will be none, set it to v6
      afInstanceId = afInstanceId or getDefaultAfInstanceId()
      # Check for the overlapping address family configurations
      if attr:
         if not addrFamilyOverlapCheck( mode=mode, attr=attr,
                           intfConfig=intfConfig, afInstanceId=afInstanceId ):
            return None

      # ignoreConfigStyle is not True, oldStyleCli becomes effective now
      if not ignoreConfigStyle:
         intfConfig.oldStyleCli = oldStyleCmd
         intfConfig.configStyleSet = True

      # processId is passed ( actually available ) only when setting an
      # area, if passed set it in the interface config
      if create and processId:
         intfConfig.processId = processId

      # Check if we have the interface instance, if not create
      if afInstanceId in intfConfig.intfInstanceConfig:
         return intfConfig.intfInstanceConfig[ afInstanceId ]
      elif create:
         t2( 'Created intf %s, instance %s config' % ( intfName, afInstanceId ) )
         return intfConfig.intfInstanceConfig.newMember( afInstanceId, intfName )

   return None

def _getIntfInstanceConfig( mode, afInstanceId=None, processId=None, attr=None,
                            ignoreConfigStyle=False ):
   return _getOrCreateIntfInstanceConfig( mode, afInstanceId, processId=processId,
                                          create=False, attr=attr,
                                          ignoreConfigStyle=ignoreConfigStyle )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf priority <number-value>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'priority', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'priority', priority )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfPriority( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'priority' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf priority PRIORITY'
   noOrDefaultSyntax = 'ipv6 ospf priority ...'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf': ospfKw,
            'priority': priorityKw,
            'PRIORITY': matcherPriority,
          }
   handler = setIntfPriority
   noOrDefaultHandler = noIntfPriority

intfModelet.addCommandClass( Ipv6OspfPriorityCmd )

#-------------------------------------------------------------------------------
# "[no|default] passive-interface <interface>"
#-------------------------------------------------------------------------------
# Old style cli accepts a interface list
class Ipv6OspfPassiveIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'passive-interface ( INTFS | default )'
   noOrDefaultSyntax = syntax
   data = { 'passive-interface': matcherPassiveIntf,
            'INTFS': IntfRange.IntfRangeMatcher(),
            'default': matcherPassiveDefault,
          }
   handler = RouterOspfv3SharedModelet.setPassiveIntf
   noOrDefaultHandler = RouterOspfv3SharedModelet.noPassiveIntf

RouterOspf3Mode.addCommandClass( Ipv6OspfPassiveIntfCmd )

#------------------------------------------------------------------------------
# "[no|default] auto-cost reference-bandwidth <bandwidth>"
#------------------------------------------------------------------------------
class Ospf3AutoCostRefBwCmd( CliCommand.CliCommandClass ):
   syntax = 'auto-cost reference-bandwidth BANDWIDTH'
   noOrDefaultSyntax = 'auto-cost reference-bandwidth ...'
   data = { 'auto-cost': autoCostKw,
            'reference-bandwidth': refBwKw,
            'BANDWIDTH': refBwMatcher
          }
   handler = RouterOspfv3SharedModelet.setAutoCost
   noOrDefaultHandler = RouterOspfv3SharedModelet.noAutoCost

RouterOspfv3SharedModelet.addCommandClass( Ospf3AutoCostRefBwCmd )

type0DummyNode = CliCommand.Node( matcher=type0KwMatcher, noResult=True )
intfAuthenticationKw = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Configure authentication for the interface' )
authenticationKw = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Configure authentication for the area' )
ipsecKw = CliMatcher.KeywordMatcher( 'ipsec',
      helpdesc='Configure IPSec Security Association parameters' )
spiKw = CliMatcher.KeywordMatcher( 'spi',
      helpdesc='Configure IPSec Security Parameter Index' )
spiMinValue = 0
spiMaxValue = 0xffffffff
spiValMatcher = CliMatcher.IntegerMatcher(
      spiMinValue, spiMaxValue,
      helpdesc='SPI value' )
hashValMatcher = CliMatcher.EnumMatcher( {
   'md5': 'Use HMAC-MD5 algorithm',
   'sha1': 'Use HMAC-SHA1 algorithm',
} )

authKeyValMatcher = CliCommand.Node(
      matcher=CliMatcher.PatternMatcher( r'.+', helpname='KEY',
                                         helpdesc='128 bit MD5 key or '
                                                  '140 bit SHA1 key' ),
      sensitive=True )
passphraseKw = CliMatcher.KeywordMatcher( 'passphrase',
   helpdesc='Use passphrase to configure key for authentication and encryption' )
passphraseValMatcher = CliCommand.Node(
      matcher=CliMatcher.StringMatcher( helpdesc='Passphrase String for deriving '
                                                 'keys for authentication and '
                                                 'encryption' ),
      sensitive=True )
intfEncryptionKw = CliMatcher.KeywordMatcher( 'encryption',
      helpdesc='Configure encryption for the interface' )
areaEncryptionKw = CliMatcher.KeywordMatcher( 'encryption',
      helpdesc='Configure encryption for the area' )
espKw = CliMatcher.KeywordMatcher( 'esp',
      helpdesc='Configure Encapsulating Security Payload' )
nullKw = CliMatcher.KeywordMatcher( 'null',
      helpdesc='ESP with NULL encryption' )
cipherValMatcher = CliMatcher.EnumMatcher(
      { k: 'ESP with %s encryption' % k.upper() for k in encryptAlgos } )
keyValNode = CliCommand.Node(
      matcher=CliMatcher.PatternMatcher( r'.+', helpname='KEY',
                 helpdesc='192 bit 3DES key or 128/192/256 bit AES key' ),
      sensitive=True )

def getEncryptKeyLen( encryptAlgo ):
   if encryptAlgo == 'null':
      return 0
   if encryptAlgo == '3des-cbc':
      return tripleDesKeyLen
   if encryptAlgo == 'aes-128-cbc':
      return aes128KeyLen
   if encryptAlgo == 'aes-192-cbc':
      return aes192KeyLen
   if encryptAlgo == 'aes-256-cbc':
      return aes256KeyLen
   return None

def getHexDigest( key, size ):
   if size == 0 or key is None or key == '':
      return ''
   if size == 128:
      # 128 bit key needed
      return hashlib.md5( key ).hexdigest()
   if size == 256:
      # 256 bit key needed
      return hashlib.sha256( key ).hexdigest()
   if size == 192:
      # 192 bit key needed
      hashedKey = hashlib.sha256( key ).hexdigest()
      return hashedKey[:48]
   if size == 160:
      return hashlib.sha1( key ).hexdigest()
   return None

# Common handler function for setting authentication or encryption
# mode - Callers mode
# name - areaId or intfName
# config - areaConfig or intfConfig
# args - Cli tokens' list
# proto - AH or ESP protocol
def setSecurity( mode, name, config, args, proto ):
   encryptKey = args.get( 'KEY', '' )
   passphraseKey = args.get( 'PHRASE', '' )
   passphrase = 'passphrase' in args
   key = args.get( 'HEXKEY', '' )
   encryptionAlgo = 'null'
   encryptKeyLen = 0

   if passphrase:
      if 'PASS7' in args :
         try:
            passphraseKey = decodeKey( passphraseKey,
                                       key=name + '_secretKey',
                                       algorithm='MD5' )

         except Exception:         # pylint: disable-msg=W0703
            mode.addError( "Invalid encrypted passphrase key" )
            return

   if 'AUTH7' in args :
      try:
         key = decodeKey( key, key=name + '_secretKey',
                          algorithm='MD5' )
      except Exception:         # pylint: disable-msg=W0703
         mode.addError( "Invalid encrypted key for authentication" )
         return

   hmacAlgo = keywordToHmac.get( args.get( 'HASH', 'sha1' ) )
   if passphrase:
      key = getHexDigest( 'authentication' + passphraseKey, hmacAlgo.digestLen )
   else:
      try:
         int( key, 16 )
      except ValueError:
         mode.addError( 'Not a valid %s key %s' % ( hmacAlgo.name, key ) )
         return
      if len( key ) != hmacAlgo.keyLen:
         mode.addError( '%s key length is not equal to %d bytes'
                         % ( hmacAlgo.name, hmacAlgo.keyLen / 2 ) )
         return

   cipher = args.get( 'CIPHER' )
   if cipher is not None:
      if 'ENCR7' in args :
         try:
            encryptKey = decodeKey( encryptKey, key=name + '_secretKey',
                                    algorithm='MD5' )
         except Exception:         # pylint: disable-msg=W0703
            mode.addError( "Invalid encrypted key for encryption" )
            return
      # encryptKeyLen is the length of the hexadecimal Key
      encryptKeyLen = getEncryptKeyLen( cipher ) / 4
      encryptionAlgo = encryptAlgos[ cipher ]
      if passphrase:
         encryptKey = "encryption" + passphraseKey
         if "des" in cipher:
            encryptKey = getHexDigest( encryptKey, 192 )
         else:
            size = int( filter( str.isdigit, cipher ) )
            encryptKey = getHexDigest( encryptKey, size )
      try:
         int( key, 16 )
      except ValueError:
         mode.addError( 'Not a valid key' )
         return
      if len( encryptKey ) != encryptKeyLen:
         mode.addError( '%s key length is not equal to %d bytes'
                         % ( cipher, encryptKeyLen / 2 ) )
         return

   # Set securityEnabled to True and set SA parameters
   config.sa = Tac.Value( "Routing6::Ospf3::Ospf3SecurityAssociation",
                          args[ 'INDEX' ], passphraseKey, key, hmacAlgo.keyLen,
                          hmacAlgo.tacVal, proto, encryptKey, encryptKeyLen,
                          encryptionAlgo )

   config.saPresent = True
   config.securityEnabled = True
   config.securityEnabledPresent = True

#-------------------------------------------------------------------------------
# [ no | default ] area AREA_ID
#                  ( ( authentication ipsec spi INDEX HASH
#                     ( ( [ 0 | AUTH7 ] HEXKEY )
#                     | ( passphrase [ 0 | PASS7 ] PHRASE ) ) )
#                  | ( encryption ipsec spi INDEX esp
#                     ( ( null HASH ( ( [ 0 | AUTH7 ] HEXKEY )
#                                   | ( passphrase [ 0 | PASS7 ] PHRASE ) ) )
#                     | ( CIPHER ( ( [ 0 | ENCR7 ] KEY HASH [ 0 | AUTH7 ] HEXKEY )
#                                | ( HASH passphrase [ 0 | PASS7 ] PHRASE ) ) ) ) ) )
#-------------------------------------------------------------------------------
class RouterOspf3Auth( CliCommand.CliCommandClass ):
   syntax = ( 'area AREA_ID '
              '( ( authentication ipsec spi INDEX HASH '
                 '( ( [ 0 | AUTH7 ] HEXKEY ) '
                 '| ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) '
              '| ( encryption ipsec spi INDEX esp '
                 '( ( null HASH ( ( [ 0 | AUTH7 ] HEXKEY ) '
                               '| ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) '
                 '| ( CIPHER ( ( [ 0 | ENCR7 ] KEY HASH [ 0 | AUTH7 ] HEXKEY ) '
                            '| ( HASH passphrase [ 0 | PASS7 ] PHRASE ) ) ) ) ) )' )
   # We need the auth/enc tokens, otherwise `no area AREA_ID stub` matches here.
   noOrDefaultSyntax = 'area AREA_ID ( authentication | encryption ) ...'
   data = {
         'area': areaKw,
         'AREA_ID': AreaIdExpression,
         'authentication': authenticationKw,
         'ipsec': ipsecKw,
         'spi': spiKw,
         'INDEX': spiValMatcher,
         'HASH': hashValMatcher,
         '0': type0DummyNode,
         'AUTH7': type7KwMatcher,
         'HEXKEY': authKeyValMatcher,
         'passphrase': passphraseKw,
         'PASS7': type7KwMatcher,
         'PHRASE': passphraseValMatcher,
         'encryption': areaEncryptionKw,
         'esp': espKw,
         'null': nullKw,
         'CIPHER': cipherValMatcher,
         'ENCR7': type7KwMatcher,
         'KEY': keyValNode,
   }

   @staticmethod
   def handler( mode, args ):
      sharedModelet = mode.modeletMap[ RouterOspfv3SharedModelet ]
      areaId = args[ 'AREA_ID' ]
      # pylint: disable-msg=protected-access
      areaConfig = sharedModelet._getOrCreateAreaConfig( areaId )
      if areaConfig:
         proto = socket.IPPROTO_ESP if 'esp' in args else socket.IPPROTO_AH
         setSecurity( mode, areaId, areaConfig, args, proto )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sharedModelet = mode.modeletMap[ RouterOspfv3SharedModelet ]
      areaId = args[ 'AREA_ID' ]
      sharedModelet.noAreaSecurity( areaId )

RouterOspfv3SharedModelet.addCommandClass( RouterOspf3Auth )

#-------------------------------------------------------------------------------
# [ no | default ] ipv6 ospf
#                  ( ( authentication ipsec spi INDEX HASH
#                     ( ( [ 0 | AUTH7 ] HEXKEY )
#                     | ( passphrase [ 0 | PASS7 ] PHRASE ) ) )
#                  | ( encryption ipsec spi INDEX esp
#                     ( ( null HASH ( ( [ 0 | AUTH7 ] HEXKEY )
#                                   | ( passphrase [ 0 | PASS7 ] PHRASE ) ) )
#                     | ( CIPHER ( ( [ 0 | ENCR7 ] KEY HASH [ 0 | AUTH7 ] HEXKEY )
#                                | ( HASH passphrase [ 0 | PASS7 ] PHRASE ) ) ) ) ) )
#-------------------------------------------------------------------------------
def noIntfSecurity( mode, afInstanceId=None ):
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'securityEnabled' )
   resetOspf3ConfigAttr( intfInstanceConfig, 'sa' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfAuth( CliCommand.CliCommandClass ):
   syntax = ( 'ipv6 ospf '
              '( ( authentication ipsec spi INDEX HASH '
                 '( ( [ 0 | AUTH7 ] HEXKEY ) '
                 '| ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) '
              '| ( encryption ipsec spi INDEX esp '
                 '( ( null HASH ( ( [ 0 | AUTH7 ] HEXKEY ) '
                               '| ( passphrase [ 0 | PASS7 ] PHRASE ) ) ) '
                 '| ( CIPHER ( ( [ 0 | ENCR7 ] KEY HASH [ 0 | AUTH7 ] HEXKEY ) '
                            '| ( HASH passphrase [ 0 | PASS7 ] PHRASE ) ) ) ) ) )' )
   noOrDefaultSyntax = 'ipv6 ospf ( authentication | encryption ) ...'
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
         'ospf': ospfKw,
         'authentication': intfAuthenticationKw,
         'ipsec': ipsecKw,
         'spi': spiKw,
         'INDEX': spiValMatcher,
         'HASH': hashValMatcher,
         '0': type0DummyNode,
         'AUTH7': type7KwMatcher,
         'HEXKEY': authKeyValMatcher,
         'passphrase': passphraseKw,
         'PASS7': type7KwMatcher,
         'PHRASE': passphraseValMatcher,
         'encryption': areaEncryptionKw,
         'esp': espKw,
         'null': nullKw,
         'CIPHER': cipherValMatcher,
         'ENCR7': type7KwMatcher,
         'KEY': keyValNode,
   }

   @staticmethod
   def handler( mode, args ):
      intfInstanceConfig = _getIntfConfigForAttr( mode, 'securityEnabled' )
      if intfInstanceConfig:
         proto = socket.IPPROTO_ESP if 'esp' in args else socket.IPPROTO_AH
         setSecurity( mode, mode.intf.name, intfInstanceConfig, args, proto )
      _deleteIntfInstanceConfig( mode.intf.name, None )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noIntfSecurity( mode )

CliPlugin.IntfCli.IntfConfigMode.addCommandClass( Ipv6OspfAuth )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf hello-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfHelloInterval( mode, args ):
   helloInterval = args.get( 'HELLO_INTERVAL' )
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'helloInterval', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'helloInterval', helloInterval )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfHelloInterval( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'helloInterval' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf hello-interval HELLO_INTERVAL'
   noOrDefaultSyntax = 'ipv6 ospf hello-interval ...'
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
         'ospf' : ospfKw,
         'hello-interval': helloIntervalKw,
         'HELLO_INTERVAL': matcherIntfIntervalSeconds,
         }

   handler = setIntfHelloInterval
   noOrDefaultHandler = noIntfHelloInterval

intfModelet.addCommandClass( Ipv6OspfHelloIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf cost <interface-cost>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfCost( mode, args ):
   cost = args[ 'COST' ]
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'cost', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'cost', cost )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfCost( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'cost' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfCostCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf cost COST'
   noOrDefaultSyntax = 'ipv6 ospf cost ...'
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
         'ospf' : ospfKw,
         'cost' : costKw,
         'COST' : matcherCost,
         }
   handler = setIntfCost
   noOrDefaultHandler = noIntfCost

intfModelet.addCommandClass( Ipv6OspfCostCmd )

#------------------------------------------------------------------------------
# "[no|default] fips restrictions"
#------------------------------------------------------------------------------
class Ospf3FipsRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
         'fips' : 'FIPS settings',
         'restrictions' : 'Use FIPS compliant algorithms',
         }

   @staticmethod
   def handler( mode, args ):
      ospf3Config().fipsEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ospf3Config().fipsEnabled = False


RouterOspfv3SharedModelet.addCommandClass( Ospf3FipsRestrictionsCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd default"
# Legacy:
# "[no|default] bfd all-interfaces"
#------------------------------------------------------------------------------
class Ospf3BfdDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd default'
   noOrDefaultSyntax = syntax
   data = {
         'bfd' : bfdKw,
         'default' : bfdDefaultKw,
         }
   handler = RouterOspfv3SharedModelet.setBfd
   noOrDefaultHandler = RouterOspfv3SharedModelet.noBfd

RouterOspfv3SharedModelet.addCommandClass( Ospf3BfdDefaultCmd )

class Ospf3BfdAllInterfacesCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd all-interfaces'
   noOrDefaultSyntax = syntax
   data = {
         'bfd' : bfdKw,
         'all-interfaces' : CliCommand.Node( matcher=allInterfacesKw,
                                             deprecatedByCmd='bfd default' ),
         }
   handler = RouterOspfv3SharedModelet.setBfd
   noOrDefaultHandler = RouterOspfv3SharedModelet.noBfd

RouterOspfv3SharedModelet.addCommandClass( Ospf3BfdAllInterfacesCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd adjacency state any"
#
#------------------------------------------------------------------------------

class Ospf3BfdAnyStateCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd adjacency state any'
   noOrDefaultSyntax = syntax
   data = {
          'bfd': bfdKw,
          'adjacency' : 'OSPFv3 adjacency BFD configuration',
          'state' : 'OSPFv3 adjacency state',
          'any' : 'Any OSPFv3 adjacency state'
          }
   handler = RouterOspfv3SharedModelet.setBfdAnyState
   noOrDefaultHandler = RouterOspfv3SharedModelet.noBfdAnyState

if Toggles.OspfToggleLib.toggleBfdForAllStatesEnabled():
   RouterOspfv3SharedModelet.addCommandClass( Ospf3BfdAnyStateCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfBfd( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'bfdIntf', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'bfdIntf', 'ospfIntfBfdEnabled' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def defaultIntfBfd( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'bfdIntf' )

def noIntfBfd( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'bfdIntf', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'bfdIntf', 'ospfIntfBfdDisabled' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf bfd'
   noSyntax = defaultSyntax = syntax
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
         'ospf' : ospfKw,
         'bfd': bfdKw,
         }
   handler = setIntfBfd
   noHandler = noIntfBfd
   defaultHandler = defaultIntfBfd

intfModelet.addCommandClass( Ipv6OspfBfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf dead-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfDeadInterval( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   deadInterval = args.get( 'DEAD_INTERVAL' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'routerDeadInterval',
                                               afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'routerDeadInterval', deadInterval )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfDeadInterval( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'routerDeadInterval' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfDeadIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf dead-interval DEAD_INTERVAL'
   noOrDefaultSyntax = 'ipv6 ospf dead-interval ...'
   data = {
         'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
         'ospf' : ospfKw,
         'dead-interval': deadIntervalKw,
         'DEAD_INTERVAL': matcherIntfIntervalSeconds,
         }

   handler = setIntfDeadInterval
   noOrDefaultHandler = noIntfDeadInterval

intfModelet.addCommandClass( Ipv6OspfDeadIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf mtu-ignore"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfMtuIgnore( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'mtuIgnore', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'mtuIgnore', True )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfMtuIgnore( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'mtuIgnore' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfMtuIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf mtu-ignore'
   noOrDefaultSyntax = syntax
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf': ospfKw,
            'mtu-ignore': mtuIgnoreKw,
          }
   handler = setIntfMtuIgnore
   noOrDefaultHandler = noIntfMtuIgnore

intfModelet.addCommandClass( Ipv6OspfMtuIgnoreCmd )

#-------------------------------------------------------------------------------
# "[no|default] ospfv3 ipv6 retransmit-interval <seconds>"
# Legacy:
# "[no|default] ipv6 ospf retransmit-interval <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfRetransmitInterval( mode, args ):
   retransmitInterval = args.get( 'RETRANSMIT_INTERVAL' )
   afInstanceId = args.get( 'AF_INSTANCE_ID' )

   # 1) We should be able to use "ospfv3 ipv6 retransmit-interval <>" and
   # "ipv6 ospf retransmit-interval <>" interchangebly for now
   # 2) set ignoreConfigStyle=True if
   # cmd = "[ ospfv3 ipv6 | ipv6 ospf ] retransmit-interval <>"
   rxIntIPv6Cmd = afInstanceId in [ afIPv6InstanceID, None ]
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'rxInt', afInstanceId,
                                               ignoreConfigStyle=rxIntIPv6Cmd )
   setOspf3ConfigAttr( intfInstanceConfig, 'rxInt', retransmitInterval )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfRetransmitInterval( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   rxIntIPv6Cmd = afInstanceId in [ afIPv6InstanceID, None ]
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId,
                                                ignoreConfigStyle=rxIntIPv6Cmd )
   resetOspf3ConfigAttr( intfInstanceConfig, 'rxInt' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfRetransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf retransmit-interval RETRANSMIT_INTERVAL'
   noOrDefaultSyntax = 'ipv6 ospf retransmit-interval ...'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf' : ospfKw,
            'retransmit-interval': CliCommand.Node( retransmitIntervalKw,
                              deprecatedByCmd='ospfv3 ipv6 retrasmit-interval' ),
            'RETRANSMIT_INTERVAL':  matcherIntfIntervalSeconds,
           }
   handler = setIntfRetransmitInterval
   noOrDefaultHandler = noIntfRetransmitInterval

intfModelet.addCommandClass( Ipv6OspfRetransmitIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf transmit-delay <seconds>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfTransmitDelay( mode, args ):
   transmitDelay = args.get( 'TRANSMIT_DELAY' )
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'transDelay', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'transDelay', transmitDelay )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfTransmitDelay( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'transDelay' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfTransmitDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf transmit-delay TRANSMIT_DELAY'
   noOrDefaultSyntax = 'ipv6 ospf transmit-delay ...'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf' : ospfKw,
            'transmit-delay': transmitDelayKw,
            'TRANSMIT_DELAY': matcherIntfIntervalSeconds,
           }
   handler = setIntfTransmitDelay
   noOrDefaultHandler = noIntfTransmitDelay

intfModelet.addCommandClass( Ipv6OspfTransmitDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf network point-to-point command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfNetworkType( mode, args ):
   intfType = args[ 'NETWORK_TYPE' ]
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'intfType', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'intfType',
                       { 'point-to-point': 'intfTypePointToPoint' }[ intfType ] )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfNetworkType( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
   resetOspf3ConfigAttr( intfInstanceConfig, 'intfType' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

class Ipv6OspfNetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf network NETWORK_TYPE'
   noOrDefaultSyntax = 'ipv6 ospf network ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf': ospfKw,
            'network': networkTypeKw,
            'NETWORK_TYPE': networkTypesMatcher,
          }
   handler = setIntfNetworkType
   noOrDefaultHandler = noIntfNetworkType

intfModelet.addCommandClass( Ipv6OspfNetworkCmd )

#-------------------------------------------------------------------------------
# NOTE: For router ospfv3 style configuration, there is a passive-interface
# knob under the interface configuration itself. The functionality is
# achieved in the old style config by "passive-interface Ethernet1" under the
# instance configuration.
#-------------------------------------------------------------------------------
def setIntfPassive( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   intfInstanceConfig = _getIntfConfigForAttr( mode, 'passive', afInstanceId )
   setOspf3ConfigAttr( intfInstanceConfig, 'passive', 'ospfIntfPassiveEnabled' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

def noIntfPassive( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   if CliCommand.isDefaultCmd( args ):
      intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId )
      resetOspf3ConfigAttr( intfInstanceConfig, 'passive' )
   else:
      intfInstanceConfig = _getIntfConfigForAttr( mode, 'passive', afInstanceId )
      setOspf3ConfigAttr( intfInstanceConfig, 'passive', 'ospfIntfPassiveDisabled' )
   _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 ospf <process-id> area <area-id>
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfAreaId( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   processId = args.get( 'PROCESS_ID' )
   areaId = args.get( 'AREA_ID' )
   assert afInstanceId != afAllInstanceID

   areaId = str( Arnet.IpAddress( areaId ) )
   intfInstanceConfig = _getOrCreateIntfInstanceConfig( mode, afInstanceId,
                                                        processId=processId )
   if not intfInstanceConfig:
      return

   afInstanceId = intfInstanceConfig.instanceId
   prevAreaId = intfInstanceConfig.areaId \
                if intfInstanceConfig.areaIdPresent else None

   # Incase of old cli, use only processId to get the instance config, else
   # we will end up with a vrf mismatch while trying to get the instance
   # config
   intfVrf = _getIntfVrf( mode.intf.name ) if not processId  else None

   # Verify if the previous area ID is not in use elsewhere, and delete
   # Skip this step if the same interface area config is being re-applied.
   # We delete the existing area config first to handle the corner case where
   # the user tries to change an area ID while already at MAX_AREAS number of areas.
   # If we add first and then delete, this will throw an error.
   prevAreaDeleted = False
   if prevAreaId != areaId:
      intfInstanceConfig.areaIdPresent = False
      prevAreaDeleted = _deleteAreaConfigFromInstance( prevAreaId, processId,
                                                       afInstanceId, intfVrf )
      intfInstanceConfig.areaIdPresent = True

   # Add the new areaId to the instance config
   errMsg = addIntfAreaToInstance( intfInstanceConfig, areaId, processId,
                                   afInstanceId, intfVrf )
   if errMsg:
      # NOTE: Should we add back the old area Id ? seems like an overkill
      # the only reason addIntfAreaToInstance() can fail is if the number
      # of areaIds has reached MAX_AREAS, assuming we did delete the
      # previous area, that cannot happen , if we did not delete the
      # previous area again we have nothing to do the areaId in the
      # intfInstanceConfig would still point to the old one
      if prevAreaDeleted:
         t0( 'Could not change interface %s area from %s to %s' %
             ( mode.intf.name, prevAreaId, areaId ) )
      mode.addError( errMsg )
   else:
      intfInstanceConfig.areaId = areaId
      intfInstanceConfig.areaIdPresent = True

def addIntfAreaToInstance( intfInstanceConfig, areaId, processId, afInstanceId,
                           vrfName ):
   instanceConfig, errMsg = getInstanceConfig( processId, afInstanceId,
                                               vrfName=vrfName )

   # For new style cli config, we will have to create the af instance if
   # a global instance exists.
   # - check if the interface is configured in new style cli ( for this
   #   we need to get the intf config)
   # - if it is a new style config we if we have a global ospfv3 instance
   # - if a global ospfv3 instance is available create the corresponding
   #   ospfv3 af instance config
   if not instanceConfig and not errMsg:
      # Fetch the interface config and make sure it was configured with
      # new style cli
      intfConfig = _getIntfConfig( intfInstanceConfig.intfName )
      assert afInstanceId in intfConfig.intfInstanceConfig and \
             intfConfig.intfInstanceConfig[ afInstanceId ] == intfInstanceConfig

      # This is new style cli config, try to check for instance id 255
      if not intfConfig.processId and \
         vrfName in ospf3Config().vrfConfig:
         vrfConfig = ospf3Config().vrfConfig[ vrfName ]
         if afAllInstanceID in vrfConfig.instanceConfig:
            instanceConfig, errMsg = createOspf3InstanceConfig( processId,
                                                                afInstanceId,
                                                                vrfName )

   # We should initialize the area only if the instance has been created.
   if instanceConfig:
      # Also create the area config
      if areaId not in instanceConfig.areaConfig:
         if len( instanceConfig.areaConfig ) >= MAX_AREAS:
            return "Max number of areas (%d) exceeded" % MAX_AREAS
         else:
            t2( 'Added area %s, to instance ( vrf %s, instanceId %s )' %
                ( areaId, vrfName, afInstanceId ) )
            instanceConfig.areaConfig.newMember( areaId )
            return None
   else:
      return errMsg

   return None

def noIntfAreaId( mode, args ):
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   processId = args.get( 'PROCESS_ID' )
   assert afInstanceId != afAllInstanceID

   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig:
      if processId and intfConfig.processId != processId:
         mode.addError( "Interface is not a part of OSPF process %d" % processId )
         return
      intfInstanceConfig = _getIntfInstanceConfig( mode, afInstanceId, processId )
      if not intfInstanceConfig:
         return

      # Reset process Id for old style cli
      if intfConfig.oldStyleCli:
         resetOspf3ConfigAttr( intfInstanceConfig, 'passive' )
         intfConfig.processId = intfConfig.processIdInvalid
      areaId = intfInstanceConfig.areaId \
               if intfInstanceConfig.areaIdPresent else None
      afInstanceId = intfInstanceConfig.instanceId
      intfInstanceConfig.areaIdPresent = False
      _deleteIntfInstanceConfig( mode.intf.name, afInstanceId )
      if areaId:
         _deleteAreaConfigFromInstance( areaId, processId, afInstanceId,
                                        intfConfig.vrfName )

def _getVrfConfigByProcessId( processId, vrfName=None ):
   if not vrfName:
      vrfName = DEFAULT_VRF

   if processId is not None:
      for vc in ospf3Config().vrfConfig.itervalues():
         if vc.processId == processId:
            return vc
   else:
      return ospf3Config().vrfConfig.get( vrfName )
   return None

def _getInstanceConfigByProcessId( processId, vrfName=None ):
   vc = _getVrfConfigByProcessId( processId, vrfName=vrfName )
   if vc:
      return vc.instanceConfig.get( afIPv6InstanceID, None )
   return None

#-------------------------------------------------------------------------------
# Get the instance config give the processId / vrfName / afInstanceId
# Used to get the instance for an interface ( for area config operations )
#-------------------------------------------------------------------------------
def _getInstanceConfigFromKeys( processId, vrfName, afInstanceId ):
   if processId:
      # This is old style cli get the instance based on preocessId
      vc = _getVrfConfigByProcessId( processId )
      if vc:
         instanceConfig = vc.instanceConfig[ afIPv6InstanceID ]
         return instanceConfig
   else:
      vrfConfig = ospf3Config().vrfConfig.get( vrfName, None )
      # This is new style config, get the instance based on address
      # family or get the global instance config
      if vrfConfig:
         instanceConfig = vrfConfig.instanceConfig.get( afInstanceId, None )
         return instanceConfig
   return None

def _areaConfigIsDefaults( areaConfig ):
   for attr in areaConfig.attributes:
      # For singleton attributes in areaConfig, check defaults
      try:
         if getattr( areaConfig, attr ) != \
                getattr( areaConfig, attr + 'Default' ):
            t5( 'Area %s has %s configured' % ( areaConfig.area, attr ) )
            return False
      except AttributeError:
         pass

      # Check that the collection attributes are empty
      if areaConfig.networkList or areaConfig.nssaNetwork:
         return False

   return True

#-------------------------------------------------------------------------------
# Delete an area from an instance if no interfaces point to it and all its
# attributes are defaulted
#-------------------------------------------------------------------------------
def _deleteAreaConfigFromInstance( areaId, processId, afInstanceId, vrfName ):
   vrfName = vrfName if vrfName else DEFAULT_VRF

   # Scan through all the interfaces to check if the current area ID is still
   # in use, else delete the area config from the instance config
   for intfname in ospf3Config().intfConfig:
      intfConfig = ospf3Config().intfConfig[ intfname ]
      if _checkIntfBelongsToInstance( intfConfig, processId,
                                      afInstanceId, vrfName ):
         intfInstanceConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
         if intfInstanceConfig.areaIdPresent and intfInstanceConfig.areaId == areaId:
            t5( 'Area %s has interface %s configured' % ( areaId, intfname ) )
            return False

   instanceConfig = _getInstanceConfigFromKeys( processId, vrfName, afInstanceId )
   if not instanceConfig or \
          ( areaId not in instanceConfig.areaConfig.keys() ):
      return False
   # Also check if all the attributes of the area are not changed from default
   areaConfig = instanceConfig.areaConfig[ areaId ]
   if not _areaConfigIsDefaults( areaConfig ):
      return False
   t2( 'Deleting area %s from instance config (vrf %s, instanceId %s)' %
       ( areaId, vrfName, afInstanceId ) )
   del instanceConfig.areaConfig[ areaId ]
   return True

#-------------------------------------------------------------------------------
# Delete an interface instance config if all its attributes are defaulted
# - Deleted the interface config as well if it does not have any instances
#-------------------------------------------------------------------------------
def _deleteIntfInstanceConfig( intfname, afInstanceId ):
   # Check if there is an interafce config
   if intfname not in ospf3Config().intfConfig:
      return

   intfConfig = ospf3Config().intfConfig[ intfname ]
   afInstanceId = afInstanceId if afInstanceId else afIPv6InstanceID
   if afInstanceId not in intfConfig.intfInstanceConfig:
      return

   intfInstConfig = intfConfig.intfInstanceConfig[ afInstanceId ]
   for attr in intfInstConfig.attributes:
      # For singleton attributes in areaConfig, check defaults
      # the attributeConfigured flag is used for attr like areaId
      # which do not have a default value
      attributeConfigured = False
      try:
         # If any attribute has present flag set then return false
         if hasattr( intfInstConfig, attr + 'Present' ):
            if not getattr( intfInstConfig, attr + 'Present' ):
               continue
            else:
               attributeConfigured = True

         if getattr( intfInstConfig, attr ) != \
                getattr( intfInstConfig, attr + 'Default' ):
            return
      except AttributeError:
         if attributeConfigured:
            return

   t2( 'Deleting interface %s instance %s config' % ( intfname, afInstanceId ) )
   del intfConfig.intfInstanceConfig[ afInstanceId ]
   if not intfConfig.intfInstanceConfig:
      t2( 'Deleting interface', intfname )
      del ospf3Config().intfConfig[ intfname ]

class Ipv6OspfAreaCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 ospf PROCESS_ID area AREA_ID'
   noOrDefaultSyntax = 'ipv6 ospf PROCESS_ID ...'
   data = { 'ipv6': CliToken.Ipv6.ipv6MatcherForConfigIf,
            'ospf': ospfKw,
            'PROCESS_ID': matcherProcessId,
            'area': areaKw,
            'AREA_ID': AreaIdExpression,
          }
   handler = setIntfAreaId
   noOrDefaultHandler = noIntfAreaId

intfModelet.addCommandClass( Ipv6OspfAreaCmd )

#-------------------------------------------------------------------------------
# "[no|default] default-information originate [ always ][ metric <1-65535> ] [
# metric-type <1-2> ] [ route-map WORD ]
#-------------------------------------------------------------------------------
class Ospf3DefaultInformationCmd( OspfDefaultInformationCmdBase ):
   data = { 'default-information' : 'Default route related configuration',
            'originate' : 'Source a default route',
            'always' : CliCommand.singleKeyword( 'always',
                           helpdesc='Advertise a default route unconditionally' ),
            'metric' : CliCommand.singleKeyword( 'metric',
                           helpdesc='Metric for default route' ),
            'METRIC' : CliMatcher.IntegerMatcher( metricMin, metricMax,
                           helpdesc='Value of the route metric' ),
            'metric-type' : CliCommand.singleKeyword( 'metric-type',
                           helpdesc='Metric type for default route' ),
            'METRIC_TYPE' : CliMatcher.IntegerMatcher( 1, 2,
                           helpdesc='Metric type' ),
            'route-map' : CliCommand.singleNode(
                              RouteMapMatchers.routeMapApplication ),
            'MAPNAME' : mapNameMatcher
   }

   @staticmethod
   def _getInstanceConfig( mode ):
      # pylint: disable-msg=protected-access, W0212
      return mode.modeletMap[ RouterOspfv3AfSharedModelet ]._getInstanceConfig()

RouterOspfv3AfSharedModelet.addCommandClass( Ospf3DefaultInformationCmd )

#-------------------------------------------------------------------------------
# In ospf3 mode
# "[no|default] max-metric router-lsa [ external-lsa [ max-metric-value ]]
#                                  [ include-stub ]
#                                  [ inter-area-lsas [ max-metric-value ]]
#                                  [ on-startup [ <announce-time> | wait-for-bgp ]]"
#                                  [ prefix-lsa ]
#                                  [ stub-prefix-lsa [ max-metric-value ]]
#                                  [ summary-lsa [ max-metric-value ]]
#------------------------------------------------------------------------------
class Ospf3MaxMetricRouterLsaCmd( CliCommand.CliCommandClass ):
   syntax = '''max-metric router-lsa
   [ { ( external-lsa [ EXTERNAL_LSA_METRIC_VALUE ] )
   | ( include-stub )
   | ( on-startup ( ANNOUNCE_TIME | wait-for-bgp ) )
   | ( summary-lsa [ SUMMARY_LSA_METRIC_VALUE ] ) } ]'''
   noOrDefaultSyntax = syntax.replace( '( ANNOUNCE_TIME | wait-for-bgp )',
                                       '[ ANNOUNCE_TIME | wait-for-bgp ]' )
   data = {
          'max-metric': 'Set maximum metric',
          'router-lsa': 'Maximum metric in self-originated router-LSAs',
          'external-lsa': CliCommand.singleNode( matcherExternalLsa ),
          'EXTERNAL_LSA_METRIC_VALUE': matcherExtLsaMetric,
          'include-stub': CliCommand.singleNode( matcherIncludeStub ),
          'on-startup': CliCommand.singleNode( matcherOnStartup ),
          'ANNOUNCE_TIME': matcherMaxMetricTimer,
          'wait-for-bgp': 'Let BGP decide when to originate router-LSA with '
                          'normal metric',
          'summary-lsa': CliCommand.singleNode( matcherSummaryLsa ),
          'SUMMARY_LSA_METRIC_VALUE': matcherSumLsaMetric,
          }
   handler = RouterOspfv3SharedModelet.setMaxMetric
   noOrDefaultHandler = RouterOspfv3SharedModelet.noMaxMetric

RouterOspfv3SharedModelet.addCommandClass( Ospf3MaxMetricRouterLsaCmd )

#-------------------------------------------------------------------------------
# "[no|default] dn-bit-ignore" command, in ospf mode.
#-------------------------------------------------------------------------------
# We guard against configuring dn-bit-ignore in default VRF.
# The [no|default] variant of the command is not guarded.
dnBitIgnoreKw = CliMatcher.KeywordMatcher( 'dn-bit-ignore',
                     helpdesc='Disable dn-bit check for Type-3/5/7 LSAs in '
                              'non-default VRFs' )
dnBitIgnoreGuardedKw = CliCommand.Node( matcher=dnBitIgnoreKw,
                                        guard=ospf3DefaultVrfGuard )

class Ospf3DnBitIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'DN_BIT_IGNORE_GUARDED'
   noOrDefaultSyntax = 'dn-bit-ignore'
   data = { 'DN_BIT_IGNORE_GUARDED': dnBitIgnoreGuardedKw,
            'dn-bit-ignore': dnBitIgnoreKw
          }
   handler = RouterOspfv3SharedModelet.setDnBitIgnore
   noOrDefaultHandler = RouterOspfv3SharedModelet.noDnBitIgnore

if Toggles.Ospf3ToggleLib.toggleOspf3DnBitIgnoreEnabled():
   RouterOspfv3SharedModelet.addCommandClass( Ospf3DnBitIgnoreCmd )

#-------------------------------------------------------------------------------
# [no|default] ipv6 access-group [<acl-list-name>] in
#
# This command is available at top-level "ipv6 router ospf" and
# "router ospfv3" config mode, but not in address family config mode.
#
# RouterOspfv3SharedModelet2 is used for this purpose.
#-------------------------------------------------------------------------------

class RouterOspfv3SharedModelet2( CliParser.Modelet, RouterOspf3Common ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      RouterOspf3Common.__init__( self, mode )

RouterOspf3Mode.addModelet( RouterOspfv3SharedModelet2 )

class Ipv6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group ...'
   data = {
          'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
          'access-group': AclCli.accessGroupKwMatcher,
          'ACL_NAME': AclCli.ip6AclNameMatcher,
          'in': AclCli.inKwMatcher,
          }
   handler = RouterOspfv3SharedModelet2.setServiceAcl
   noOrDefaultHandler = RouterOspfv3SharedModelet2.noServiceAcl

RouterOspfv3SharedModelet2.addCommandClass( Ipv6AccessGroupCmd )
#-------------------------------------------------------------------------------
# "clear ospfv3 [ipv4|ipv6] force-spf"
# Legacy:
# "clear ipv6 ospf force-spf [processId] [vrf vrfName]"
# command, in global mode.
# Though this is a clear command (operational command), it is
# handled via a config 'forceSPFTrigger'. Everytime forceSPFTrigger 'changes',
# a callback is called into ribd that will force a spf.
# forceSPFTrigger is a uint8_t with a default value of 0.
#-------------------------------------------------------------------------------

def forceSPFTrigger ( instanceConfig ):
   if instanceConfig:
      if instanceConfig.forceSPFTrigger == 255:
         instanceConfig.forceSPFTrigger = 1
      else:
         instanceConfig.forceSPFTrigger+= 1
   return

def setForceSPF( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   if afInstanceId is None:
      # old style cli
      afInstanceId = afIPv6InstanceID
      if processId is None and vrfName is None:
         vrfName = DEFAULT_VRF
   else:
      vrfName = vrfName if vrfName else DEFAULT_VRF
      processId = 0

   instanceConfig = _getInstanceConfigFromKeys ( processId,
                                                 vrfName,
                                                 afInstanceId )
   forceSPFTrigger( instanceConfig )

class ClearIpv6OspfForceSpfCmdDeprecated( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 ospf [ PROCESS_ID ] force-spf [ VRF ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'ospf': ospfKw,
      'PROCESS_ID': matcherProcessId,
      'force-spf': CliCommand.Node( CliMatcher.KeywordMatcher( 'force-spf',
                      helpdesc='force a spf calculation' ),
                      deprecatedByCmd='clear ospfv3 ipv6 force-spf' ),
      'VRF': vrfExprFactory,
   }
   handler = setForceSPF

BasicCli.EnableMode.addCommandClass( ClearIpv6OspfForceSpfCmdDeprecated )

#-------------------------------------------------------------------------------
#Registered callback for "config convert new-syntax"
#-------------------------------------------------------------------------------
def convertRouterOspfv3ConfigStyle( mode ):
   # Check if we are already in new-style, in which case just return
   if not checkConfigStyle( oldStyleCmd=True ):
      return

   # Iterate over all vrfs and set the processId to 0
   for _, vrfConfig in ospf3Config().vrfConfig.iteritems():
      vrfConfig.processId = 0
   # Iterate over interfaces and set oldStyleCli and processId to 0
   for _, intfConfig in ospf3Config().intfConfig.iteritems():
      intfConfig.oldStyleCli = False
      intfConfig.processId = 0

#-------------------------------------------------------------------------------
# Top level generator / decorator functions for old style ospf3 cli
# These provide the vrf / instance hierarchy in the json models
#-------------------------------------------------------------------------------
class Ospf3VrfModelDecorator( object ):
   '''
   Decorator class that can be used to create a top level
   Capi model that contains a generator list of component model instances

   Example usage:
   to 'capi'ize the 'show ip ospf3' command with 'vrf [VRF|default|all]'
   add the 'Ospf3VrfModelDecorator' decorator to the showIpv6Ospf function and pass
   the @vrfModelListType argument to the decorator. @vrfModelListType is a model
   type generated by invoking generateVrfCliModel with relevent component
   model type. @instModelListType is the model type generated by invoking
   generateOspfInstListCliModel. This model contains the list of
   instances per VRF. This is will be useful when future support for multiple
   instances per VRF is added

   @Ospf3VrfModelDecorator( vrfModelListType, instModelListType )
   def showIpv6Ospf( mode, processId=None, vrfName=None ):
   '''
   def __init__( self, vrfModelListType, instModelListType, firstInstance=False ):
      '''input: @vrfModelListType is a model type generated by invoking
      generateOspf3VrfCliModel with relevent compoenent
      model type.
      @instModelListType is the model type generated by invoking
      generateOspf3InstListCliModel. This model contains the list of
      instances per VRF. This is will be useful when future support for multiple
      instances per VRF is added
      @firstInstance is a flag indicating if this is the first instance being
      called from the decorators, this is flag can be used to render / print
      headers or any other initialization that the handler expects'''
      self.vrfModelListType = vrfModelListType
      self.instModelListType = instModelListType
      self.firstInstance = firstInstance

   def __call__( self, func ):
      '''return a decorated function that wraps @func
         arguments to the wrapped function are passed as is'''
      def vrfExecCmdFunc( *args, **kwargs ):
         '''
         first argument in the args list must be of type 'CliParser.Mode'
         '''
         assert isinstance( args[ 0 ], CliParser.Mode )
         mode = args[ 0 ]

         processId = kwargs[ 'args' ].get( 'PROCESS_ID' )
         vrfName = kwargs[ 'args' ].get( 'VRF' )

         processIds = getInstanceIds( mode, processId, vrfName )
         vrfs = {}
         if not processIds:
            processIds = []
         elif processIds[ 0 ] == 0:
            vrfNames = getVrfList( mode, vrfName )
            for vrf in vrfNames:
               vrfs[ vrf ] = [ 0 ]
         else:
            for inst in processIds:
               c = _getInstanceConfigByProcessId( inst )
               if c:
                  vrfs[ c.vrfName ] = [ inst ]
               else:
                  vrfs[ DEFAULT_VRF ] = [ inst ]

         def instListFunc( vrf, firstVrf ):
            firstInst = True
            for inst in vrfs[ vrf ]:
               kwargs[ 'args' ][ 'PROCESS_ID' ] = inst
               kwargs[ 'args' ][ 'VRF' ] = vrf
               if self.firstInstance and firstInst and firstVrf:
                  # Incase the command would like to print a legend
                  # or header just once across multiple instances
                  kwargs[ 'args' ][ 'firstInstance' ] = True
               else:
                  kwargs[ 'args' ].pop( 'firstInstance', None )
               instModel = func( *args, **kwargs )
               firstInst = False
               if instModel is None:
                  continue
               yield inst, instModel

         def vrfListFunc():
            firstVrf = True
            for vrf in vrfs:
               instListModel = self.instModelListType()
               instListModel.instList = instListFunc( vrf, firstVrf )
               firstVrf = False
               yield vrf, instListModel

         model = self.vrfModelListType()
         model.vrfs = vrfListFunc()
         return model

      return vrfExecCmdFunc

def generateOspf3InstListCliModel( cliModel ):
   instList = GeneratorDict( keyType=int, valueType=cliModel,
         help='OSPFv3 Instance list keyed by instance id' )
   def render( self ):
      for _, model in self.instList:
         model.render()

   className = 'InstList%s' % cliModel.__name__
   classBody = { 'instList' : instList,
                 'render' : render }

   return type( className, ( Model, ), classBody )


#-------------------------------------------------------------------------------
# "show ipv6 ospf [process-id] [vrf vrfname]
#-------------------------------------------------------------------------------
matcherOspf3Show = CliMatcher.KeywordMatcher( 'ospf', helpdesc='OSPF3 information' )

instancedesc = "OSPFv3 instance information for all VRFs"
Ospf3ShowInstDictModel = generateOspfInstListCliModel(
   Ospf3CliModels.Ospf3ShowInstModel )
Ospf3ShowInstVrfDictModel = generateVrfCliModel( Ospf3ShowInstDictModel,
                                                 instancedesc )
def getDefaultInstanceModel( model, processId=None, afInstanceId=afIPv6InstanceID,
                             vrfName=None ):
   vrfName = vrfName if vrfName else DEFAULT_VRF
   instConfig = _getInstanceConfigFromKeys( processId, vrfName, afInstanceId )

   if instConfig:
      model.vrf = instConfig.vrfName
      model.instanceId = AF_INSTANCE_ID_MAP[ instConfig.instanceId ]
      if processId:
         model.processId = processId
      if not instConfig.enable:
         model.shutDown = True
   # If this is new-style CLI then make sure to check the global config mode for the
   # same options as above.
   if not processId:
      globalInstConfig = _getInstanceConfigFromKeys( None, vrfName, afAllInstanceID )
      if globalInstConfig:
         if not globalInstConfig.enable:
            model.shutDown = True
   return model

def getDefaultModelValues( model, processId=None, afInstanceId=afIPv6InstanceID,
                           vrfName=None ):
   vrfName = vrfName if vrfName else DEFAULT_VRF
   instConfig = _getInstanceConfigFromKeys( processId, vrfName, afInstanceId )

   if instConfig:
      if hasattr( model, "_vrf" ):
         # pylint: disable-msg=protected-access
         model._vrf = instConfig.vrfName

      if hasattr( model, "instanceId" ):
         model.instanceId = AF_INSTANCE_ID_MAP[ instConfig.instanceId ]

      if hasattr( model, "shutdown" ):
         if not instConfig.enable:
            model.shutdown = True

   # If this is new-style CLI then make sure to check the global config mode for
   # the same options as above.
   if not processId:
      globalInstConfig = _getInstanceConfigFromKeys( None, vrfName,
                                                     afAllInstanceID )
      if globalInstConfig:
         if not globalInstConfig.enable:
            if hasattr( model, "shutdown" ):
               model.shutdown = True

@Ospf3VrfModelDecorator( vrfModelListType=Ospf3ShowInstVrfDictModel,
                         instModelListType=Ospf3ShowInstDictModel )
def showIpv6Ospf( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   command = 'MIO_DGET_OSPF3_INSTANCE_SUMMARY'
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   argsCapi = { 'instance': instanceId }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3ShowInstModel, command,
                                 argsCapi, clientName="OSPF3", l3Config=l3Config )
   except EmptyResponseException:
      return getDefaultInstanceModel( Ospf3CliModels.Ospf3ShowInstModel(),
                                      processId )

class ShowIpv6OspfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf [ PROCESS_ID ] [ VRF ]'
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'VRF': vrfExprFactory,
          }
   handler = showIpv6Ospf
   cliModel = Ospf3ShowInstVrfDictModel

BasicCli.addShowCommandClass( ShowIpv6OspfCmd )

#-------------------------------------------------------------------------------
# show ipv6 ospf retransmission-list [ vrf vrfName ]
#-------------------------------------------------------------------------------
retransmissionListDesc = "OSPFv3 LSA retransmission information for all VRFs"
Ospf3LsaRetransmissionListInstModel = generateOspfInstListCliModel(
                                       Ospf3CliModels.Ospf3LsaRetransmissionList )
Ospf3LsaRetransmissionListVrfModel = generateVrfCliModel(
                                       Ospf3LsaRetransmissionListInstModel,
                                       retransmissionListDesc )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3LsaRetransmissionListVrfModel,
                         instModelListType=Ospf3LsaRetransmissionListInstModel )
def showIpv6OspfLsaRetransmissionList( mode, args ):
   vrfName = args.get( 'VRF' )
   routerId = args.get( 'ROUTER_ID' )
   ifName = args.get( 'IFNAME' )
   command = 'MIO_DGET_OSPF3_RETRANSMISSION_LIST'
   argsCapi = {}

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( "Interface %s not present" % ifName.name )
         return None
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3LsaRetransmissionList,
                                 command, args=argsCapi, clientName='OSPF3',
                                 l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3LsaRetransmissionList()

matcherOspf3LsaRetransmissionList = CliMatcher.KeywordMatcher(
      'retransmission-list', helpdesc='Re-transmission list' )

class ShowIpv6OspfLsaRetransmissionListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ipv6 ospf [ PROCESS_ID ] retransmission-list
               [ ROUTER_ID ] [ IFNAME ] [ VRF ]'''
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'retransmission-list': matcherOspf3LsaRetransmissionList,
           'ROUTER_ID': matcherNeighborId,
           'IFNAME': IntfCli.Intf.matcherWithRoutingProtoSupport,
           'VRF': vrfExprFactory,
          }
   handler = showIpv6OspfLsaRetransmissionList
   cliModel = Ospf3LsaRetransmissionListVrfModel

BasicCli.addShowCommandClass( ShowIpv6OspfLsaRetransmissionListCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf request-list [vrf vrfname]
#-------------------------------------------------------------------------------
desc="OSPFv3 LSA request list information for all VRFs"
Ospf3LsaRequestListInstModel = generateOspfInstListCliModel(
                                 Ospf3CliModels.Ospf3LsaRequestList )
Ospf3LsaRequestListVrfModel = generateVrfCliModel(
                                 Ospf3LsaRequestListInstModel, desc )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3LsaRequestListVrfModel,
                         instModelListType=Ospf3LsaRequestListInstModel )
def showIpv6OspfLsaRequestList( mode, args ):
   vrfName = args.get( 'VRF' )
   routerId = args.get( 'ROUTER_ID' )
   ifName = args.get( 'IFNAME' )
   command = "MIO_DGET_OSPF3_REQUEST_LIST"
   argsCapi = {}

   if vrfName != DEFAULT_VRF:
      argsCapi[ "vrfName" ] = vrfName
   if routerId is not None:
      argsCapi[ 'routerid' ] = routerId
   if ifName is not None:
      if not ifName.lookup():
         mode.addError( "Interface %s not present" % ifName.name )
         return None
      argsCapi[ 'ifname' ] = ifName.status().deviceName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3LsaRequestList, command,
                                 argsCapi, clientName="OSPF3", l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3LsaRequestList()

matcherOspf3LsaRequestList = CliMatcher.KeywordMatcher(
      'request-list', helpdesc='Request list' )

class ShowIpv6OspfLsaRequestListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ipv6 ospf [ PROCESS_ID ] request-list
               [ ROUTER_ID ] [ IFNAME ] [ VRF ]'''
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'request-list': matcherOspf3LsaRequestList,
           'ROUTER_ID': matcherNeighborId,
           'IFNAME': IntfCli.Intf.matcherWithRoutingProtoSupport,
           'VRF': vrfExprFactory,
          }
   handler = showIpv6OspfLsaRequestList
   cliModel = Ospf3LsaRequestListVrfModel

BasicCli.addShowCommandClass( ShowIpv6OspfLsaRequestListCmd )

#------------------------------------------------------------------------------
# show ipv6 ospf spf-log [processId] [vrf vrfName]
#------------------------------------------------------------------------------
matcherOspf3SpfLog = CliMatcher.KeywordMatcher( 'spf-log', helpdesc='OSPF3 spf log' )

desc = "OSPFv3 spf-log for the given VRF"
Ospf3SpfLogInstModel = generateOspf3InstListCliModel(
                       Ospf3CliModels.Ospf3SpfLogInst )
Ospf3SpfLogVrfModel = generateVrfCliModel( Ospf3SpfLogInstModel,
                                           desc )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3SpfLogVrfModel,
                         instModelListType=Ospf3SpfLogInstModel )
def doShowIpv6OspfSpfLog( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   command = 'MIO_DGET_OSPF3_SPF_LOGS'

   argsCapi = { 'instanceId':instanceId }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3SpfLogInst,
                                 command, argsCapi,
                                 clientName='OSPF3', l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3SpfLogInst()

class ShowIpv6OspfSpfLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf [ PROCESS_ID ] spf-log [ VRF ]'
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'spf-log': matcherOspf3SpfLog,
           'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfSpfLog
   cliModel = Ospf3SpfLogVrfModel

BasicCli.addShowCommandClass( ShowIpv6OspfSpfLogCmd )

#------------------------------------------------------------------------------
# show ipv6 ospf lsa-log
#------------------------------------------------------------------------------
matcherOspf3LsaLog = CliMatcher.KeywordMatcher( 'lsa-log', helpdesc='OSPF3 lsa log' )

desc = "OSPFv3 lsa-log for the given VRF"
Ospf3LsaLogInstModel = generateOspf3InstListCliModel(
                       Ospf3CliModels.Ospf3LsaLogInst )
Ospf3LsaLogVrfModel = generateVrfCliModel( Ospf3LsaLogInstModel,
                                           desc )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3LsaLogVrfModel,
                         instModelListType = Ospf3LsaLogInstModel )
def doShowIpv6OspfLsaLog( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   command = 'MIO_DGET_OSPF3_LSA_LOGS'

   argsCapi = { 'instanceId':instanceId }

   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName


   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3LsaLogInst,
                                 command, argsCapi,
                                 clientName='OSPF3', l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3LsaLogInst()

class ShowIpv6OspfLsaLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf [ PROCESS_ID ] lsa-log [ VRF ]'
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'lsa-log': matcherOspf3LsaLog,
           'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfLsaLog
   cliModel = Ospf3LsaLogVrfModel

BasicCli.addShowCommandClass( ShowIpv6OspfLsaLogCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf database
#-------------------------------------------------------------------------------
matcherAreaFilter = CliMatcher.KeywordMatcher( 'area',
                    helpdesc='Filter by Area Scoped LSAs' )
matcherBackbone = CliMatcher.KeywordMatcher( 'backbone',
                  helpdesc='Filter for Area 0.0.0.0' )
matcherAreaIdFilter = IpAddrMatcher.IpAddrMatcher( 'Filter by Area ID' )
matcherAsFilter = CliMatcher.KeywordMatcher( 'as',
                  helpdesc='Filter by AS Scoped LSAs' )
matcherAsExternal = CliMatcher.KeywordMatcher( 'external',
                    helpdesc='filter for AS-External LSAs' )
matcherLinkFilter = CliMatcher.KeywordMatcher( 'link',
                    helpdesc='Filter by Link Scoped LSAs' )
matcherIfName = CliMatcher.KeywordMatcher( 'if-name',
                helpdesc='Name of the interface' )
matcherIfType = CliMatcher.KeywordMatcher( 'if-type',
                helpdesc='Filter by Interface Type' )
matcherAdvRouter = CliMatcher.KeywordMatcher( 'adv-router',
                   helpdesc='Router advertising the LSA' )
matcherSelfOriginate = CliMatcher.KeywordMatcher( 'self-originate',
                       helpdesc='OSPF locally originated LSA entries' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
                helpdesc='Detailed LSA information' )


# Generating CAPI args for "show ipv6 ospf database area"
# show ipv6 ospf database area <area Id / backbone> <LSA Type>

#The below map corresponds to the defintions in cli_cmd_db_area_areaid,
# ospf3_cli.c
LSA_TYPE_MAP = { "router"            : 8193,
                 "network"           : 8194,
                 "inter-area-prefix" : 8195,
                 "inter-area-router" : 8196,
                 "intra-area-prefix" : 8201,
                 "nssa"              : 8199,
               }

def processShowIpv6OspfDatabaseArea( args, spec=None ):
   if spec.get( 'areaSpecType' ):
      if spec[ 'areaSpecType' ] == 'backbone':
         areaId = '0.0.0.0'
      else:
         areaId = "%s" % ( spec[ 'areaSpecType' ] )
      args[ spec[ 'filter' ] ] = areaId

      # Apply any LSA type filters
      if spec.get( 'lsaType' ) and spec[ 'lsaType' ] in LSA_TYPE_MAP:
         args[ 'type' ] = LSA_TYPE_MAP[ spec[ 'lsaType' ] ]

def processShowIpv6OspfDatabaseAs( args, spec=None ):
   pass

# Generating CAPI args for "show ipv6 ospf database link"
# show ipv6 ospf database link <if-name interfaceName>
# show ipv6 ospf database link <if-type interfaceType>
def processShowIpv6OspfDatabaseLink( args, spec=None ):
   if spec.get( 'linkSpec' ):
      linkSpec = spec[ 'linkSpec' ]
      if linkSpec.get( 'ifClassifyType' ):
         # Filter by interface type
         if linkSpec[ 'ifClassifyType' ] == 'if-type':
            if linkSpec.get( 'ifType' ):
               args[ 'iftype' ] = "%s" % ( linkSpec[ 'ifType' ] )

         # Filter by interface name
         elif linkSpec[ 'ifClassifyType' ] == 'if-name':
            intfList = linkSpec[ 'ifName' ]
            if isinstance( intfList, IntfRange.IntfList ):
               intfNames = intfList.intfNames( shortName=True )
            else:
               intfNames = intfList

            intfs = ""
            for intfName in intfNames:
               intfs += "%s" % ( intfName.lower() )

            args[ 'ifname' ] = intfs

Ospf3ShowDatabaseDictModel = generateOspf3InstListCliModel(
                                       Ospf3CliModels.Ospf3DatabaseInstance )
Ospf3ShowDatabseVrfDictModel = generateVrfCliModel( Ospf3ShowDatabaseDictModel,
                                       'OSPFv3 database information for all VRFs')

@Ospf3VrfModelDecorator( vrfModelListType=Ospf3ShowDatabseVrfDictModel,
                         instModelListType=Ospf3ShowDatabaseDictModel,
                         firstInstance=True )
def doShowIpv6OspfDatabase( mode, args ):
   spec = None
   if any( i in args for i in ( 'area', 'as', 'link' ) ):
      spec = {}
      spec[ 'filter' ] = args.get( 'area' ) or args.get( 'as' ) or args.get( 'link' )

   if 'area' in args:
      spec[ 'areaSpecType' ] = args.get( 'backbone' ) or args.get( 'IP_ADDR' )
      spec[ 'lsaType' ] = args.get( 'LSA_TYPE' )

   if 'as' in args:
      spec[ 'external' ] = args.get( 'external' )

   if 'link' in args:
      if 'if-name' in args:
         spec[ 'linkSpec' ] = { 'ifClassifyType': 'if-name',
                                'ifName': args[ 'INTFS' ] }
      elif 'if-type' in args:
         spec[ 'linkSpec' ] = { 'ifClassifyType': 'if-type',
                                'ifType': args[ 'IF_TYPE' ] }
   lsId = args.get( 'LINK_STATE_ID' )
   origRtr = None
   if any( i in args for i in ( 'adv-router', 'self-originate' ) ):
      origRtr = {}
      origRtr[ 'rtrType' ] = args.get( 'adv-router' ) or args.get( 'self-originate' )
      if 'adv-router' in args:
         origRtr[ 'rtrId' ] = args[ 'ADV_RTR' ]
   detail = args.get( 'detail' )
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   firstInstance = args.get( 'firstInstance', False )
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   argsCapi = { 'instance': instanceId }

   # Use PyRibDami approach, if there are no filters / queries
   noDatabaseQuery = True
   detailOutput = False

   cmd = "MIO_DGET_OSPF3_DATABASE"

   if vrfName != DEFAULT_VRF:
      argsCapi [ 'vrfName' ] = vrfName
   argsCapi[ 'process_id' ] = processId

   filterProcessor = { 'area': processShowIpv6OspfDatabaseArea,
                       'as'  : processShowIpv6OspfDatabaseAs,
                       'link': processShowIpv6OspfDatabaseLink,
                     }

   filterQueryType = { 'area': 3,
                       'as'  : 2,
                       'link': 1,
                     }

   if spec and spec.get( 'filter' ):
      noDatabaseQuery = False

      # Set the query type to area / link / as
      argsCapi[ 'query-type' ] = filterQueryType[ spec[ 'filter' ] ]

      if filterProcessor.get( spec[ 'filter' ] ):
         # Call the corresponding function to process the cmd
         filterProcessor[ spec[ 'filter' ] ]( argsCapi, spec )

   if lsId:
      noDatabaseQuery = False
      argsCapi [ 'lsid' ] = lsId

   if origRtr and origRtr.get( 'rtrType' ):
      noDatabaseQuery = False
      rtrId = None
      if origRtr[ 'rtrType' ] == 'adv-router':
         if origRtr.get( 'rtrId' ):
            rtrId = "%s" % ( origRtr[ 'rtrId' ] )
         argsCapi[ 'router id' ] = rtrId
      else:
         argsCapi[ 'self' ] = True

   if detail:
      detailOutput = True
      argsCapi ['print_type'] = True

   # Print the database legend only once at the beginning
   if firstInstance and mode.session_.outputFormat_ == "text":
      Ospf3CliModels.printOspf3LegendInformation()
      sys.stdout.flush()

   if noDatabaseQuery and mode.session_.outputFormat_ == "text":
      argsCapi[ 'format' ] = "2"
      showRibDamiCommand( sys.stdout.fileno(), cmd, argsCapi, clientName='OSPF3',
                          l3Config=l3Config )
   else:
      try:
         result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3DatabaseInstance,
                                      cmd, args=argsCapi, clientName='OSPF3',
                                      l3Config=l3Config )

      except EmptyResponseException:
         result = Ospf3CliModels.Ospf3DatabaseInstance()

      if result and detailOutput:
         result._detailsPresent = True # pylint: disable-msg=protected-access

      return result
   return None

lsaType = { 'inter-area-prefix': 'Filter for Inter-Area-Prefix LSAs',
            'inter-area-router': 'Filter for Inter-Area-Router LSAs',
            'intra-area-prefix': 'Filter for Intra-Area-Prefix LSAs',
            'network': 'Filter for type-2 LSAs',
            'nssa': 'Filter for type-7 LSAs',
            'router': 'Filter for type-1 LSAs',
}


ifType = { 'broadcast': 'Filter by broadcast networks',
           'nbma': 'Filter by NBMA networks',
           'p2mp': 'Filter by P2MP networks',
           'p2p': 'Filter by P2P networks',
}

class ShowIpv6OspfDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ipv6 ospf [ PROCESS_ID ] database
               [ ( area ( backbone | IP_ADDR ) [ LSA_TYPE ] ) |
                 ( as [ external ] ) |
                 ( link [ ( ( if-name INTFS ) | ( if-type IF_TYPE ) ) ] )
               ]
               [ LINK_STATE_ID ]
               [ ( ( adv-router ADV_RTR ) | self-originate ) ]
               [ detail ]
               [ VRF ]'''
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'ospf': matcherOspf3Show,
            'PROCESS_ID': matcherProcessId,
            'database': matcherDatabase,
            'area': matcherAreaFilter,
            'backbone': matcherBackbone,
            'IP_ADDR': matcherAreaIdFilter,
            'LSA_TYPE': CliMatcher.EnumMatcher( lsaType ),
            'as': matcherAsFilter,
            'external': matcherAsExternal,
            'link': matcherLinkFilter,
            'if-name': matcherIfName,
            'INTFS': IntfRange.IntfRangeMatcher(),
            'if-type': matcherIfType,
            'IF_TYPE': CliMatcher.EnumMatcher( ifType ),
            'LINK_STATE_ID': IpAddrMatcher.IpAddrMatcher( 'Link state ID' ),
            'adv-router': matcherAdvRouter,
            'ADV_RTR': IpAddrMatcher.IpAddrMatcher( 'Advertising Router' ),
            'self-originate': matcherSelfOriginate,
            'detail': matcherDetail,
            'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfDatabase
   cliModel = Ospf3ShowDatabseVrfDictModel

BasicCli.addShowCommandClass( ShowIpv6OspfDatabaseCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf database database-summary
#-------------------------------------------------------------------------------
Ospf3ShowDbSumDictModel = generateOspf3InstListCliModel(
                                       Ospf3CliModels.Ospf3DatabaseSummary)
Ospf3ShowDbSumVrfDictModel = generateVrfCliModel( Ospf3ShowDbSumDictModel,
      'OSPFv3 database information for all VRFs')

@Ospf3VrfModelDecorator( vrfModelListType=Ospf3ShowDbSumVrfDictModel,
                         instModelListType=Ospf3ShowDbSumDictModel,
                         firstInstance=True )

def doShowIpv6OspfDatabaseSummary( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   cmd = "MIO_DGET_OSPF3_DATABASE_SUM"

   # parse instance parameter, exit if there is no instance
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   # put required parameters in argument list
   argsCapi = { 'instance': instanceId }
   argsCapi[ 'process_id' ] = processId

   # parse optional vrf parameter
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   # parse optional area filter
   areaId = None
   if 'area' in args:
      if 'backbone' in args:
         areaId = '0.0.0.0'
      else:
         areaId = args[ 'IP_ADDR' ]
      argsCapi[ 'area' ] = areaId

   try:
      # Call Capi Command
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3DatabaseSummary,
                                   cmd, args=argsCapi, clientName='OSPF3',
                                   l3Config=l3Config )

   except EmptyResponseException:
      return Ospf3CliModels.Ospf3DatabaseSummary()

   # Save values needed to print headers
   result.instanceIdIs( int( processId ) )
   if vrfName != None:
      result.vrfIs( vrfName )
   if areaId != None:
      result.areaIdIs( areaId )

   return result

class showIpv6OspfDatabaseSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ipv6 ospf [ PROCESS_ID ] database database-summary '
              '[ area ( backbone | IP_ADDR ) ] [ VRF ]' )
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'ospf': matcherOspf3Show,
            'PROCESS_ID': matcherProcessId,
            'database': matcherDatabase,
            'database-summary': matcherDatabaseSummary,
            'area': matcherAreaFilter,
            'backbone': matcherBackbone,
            'IP_ADDR': matcherAreaIdFilter,
            'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfDatabaseSummary
   cliModel = Ospf3ShowDbSumVrfDictModel

BasicCli.addShowCommandClass( showIpv6OspfDatabaseSummaryCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf border-routers
#-------------------------------------------------------------------------------
matcherBorderRouters = CliMatcher.KeywordMatcher( 'border-routers',
                       helpdesc='Status for border routers' )

desc = "OSPFv3 border routers information for all VRFs"
Ospf3BorderRoutersInstModel = generateOspf3InstListCliModel(
                             Ospf3CliModels.Ospf3BorderRoutersInstance )
Ospf3BorderRoutersVrfModel = generateVrfCliModel( Ospf3BorderRoutersInstModel,
                                                  desc )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3BorderRoutersVrfModel,
                         instModelListType=Ospf3BorderRoutersInstModel )
def doShowIpv6OspfBorderRouters( mode, args ):
   processId = args.get( 'PROCESS_ID' )
   vrfName = args.get( 'VRF' )
   processIds = getInstanceIds( mode, processId, vrfName )
   if not processIds:
      return None

   command = 'MIO_DGET_OSPF3_BORDER_ROUTERS'
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   args = { 'instance': instanceId }

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3BorderRoutersInstance,
                                 command, args, clientName='OSPF3',
                                 l3Config=l3Config )
   except EmptyResponseException:
      return  Ospf3CliModels.Ospf3BorderRoutersInstance()

class ShowIpv6OspfBorderRoutersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf [ PROCESS_ID ] border-routers [ VRF ]'
   data = {
           'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
           'ospf': matcherOspf3Show,
           'PROCESS_ID': matcherProcessId,
           'border-routers': matcherBorderRouters,
           'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfBorderRouters
   cliModel = Ospf3BorderRoutersVrfModel

BasicCli.addShowCommandClass( ShowIpv6OspfBorderRoutersCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf [<process-id>] interface [ vrf vrf-name|default|all]"
#-------------------------------------------------------------------------------
Ospf3ShowIntfDictModel = generateOspf3InstListCliModel(
                                                Ospf3CliModels.Ospf3Interface )
Ospf3ShowIntfVrfDictModel = generateVrfCliModel( Ospf3ShowIntfDictModel,
                                     'OSPFv3 instance information for all VRFs' )
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3ShowIntfVrfDictModel,
                         instModelListType=Ospf3ShowIntfDictModel )
def doShowIpv6OspfInterface( mode, args ):
   vrfName = args.get( 'VRF' )
   intf = args.get( 'INTF' )
   processId = args.get( 'PROCESS_ID' )
   cmd = 'MIO_DGET_OSPF3_INTERFACE'
   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   argsCapi = { 'instance': instanceId }

   if intf is not None:
      if not intf.lookup():
         mode.addError( "Interface %s not present" % intf.name )
         return None
      argsCapi[ 'ifname' ] = intf.status().deviceName
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      return showRibCapiCommand( mode, Ospf3CliModels.Ospf3Interface, cmd, argsCapi,
                                 clientName='OSPF3', l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3Interface()

class ShowIpv6OspfInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf [ PROCESS_ID ] interface [ INTF ] [ VRF ]'
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'ospf': matcherOspf3Show,
            'PROCESS_ID': matcherProcessId,
            'interface': 'Interface-specific details',
            'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfInterface
   cliModel = Ospf3ShowIntfVrfDictModel

BasicCli.addShowCommandClass( ShowIpv6OspfInterfaceCmd )

#-------------------------------------------------------------------------------
# "show ipv6 ospf neighbor
#-------------------------------------------------------------------------------
neighborDesc = 'OSPFv3 neighbor instance information for all VRFs'
Ospf3ShowNeighDictModel = generateOspf3InstListCliModel(
   Ospf3CliModels.Ospf3NeighborsInstance )
Ospf3ShowNeighVrfDictModel = generateVrfCliModel( Ospf3ShowNeighDictModel,
                                                  neighborDesc )
# pylint: disable-msg=C0322
@Ospf3VrfModelDecorator( vrfModelListType=Ospf3ShowNeighVrfDictModel,
                         instModelListType=Ospf3ShowNeighDictModel )
def doShowIpv6OspfNeighbor( mode, args ):
   vrfName = args.get( 'VRF' )
   interface = args.get( 'INTF' )
   summary = args.get( 'summary' )
   state = args.get( 'STATE' )
   neighborId = args.get( 'NEIGHBOR_ID' )
   processId = args.get( 'PROCESS_ID' )
   command = 'MIO_DGET_OSPF3_NEIGHBORS'

   instanceId = getInstanceIdForInstance( processId, vrfName=vrfName )
   if instanceId is None:
      return None

   argsCapi = { 'instance': instanceId }

   if interface is not None:
      if not interface.lookup():
         mode.addError( "Interface %s not present" % interface.name )
         return None
      argsCapi[ 'interfaceName' ] = interface.status().deviceName
   if neighborId is not None:
      argsCapi[ 'routerId' ] = neighborId
   if vrfName != DEFAULT_VRF:
      argsCapi[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, Ospf3CliModels.Ospf3NeighborsInstance,
                                   command, argsCapi, clientName='OSPF3',
                                   l3Config=l3Config )
   except EmptyResponseException:
      return Ospf3CliModels.Ospf3NeighborsInstance()

   if summary:   #cmd: show ipv6 ospf neighbor summary
      result._summary = True # pylint: disable-msg=protected-access
   elif state:   #cmd: show ipv6 ospf neighbor state abc
      result._state = state # pylint: disable-msg=protected-access
   return result

class ShowIpv6OspfNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ipv6 ospf [ PROCESS_ID ] neighbor [ INTF ] [ NEIGHBOR_ID ] '
              '[ summary ] [ state STATE ] [ VRF ]' )
   data = {
            'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
            'ospf': matcherOspf3Show,
            'PROCESS_ID': matcherProcessId,
            'neighbor': 'Protocol neighbors',
            'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport,
            'NEIGHBOR_ID': matcherNeighborId,
            'summary': 'Summary of neighbor information',
            'state': 'Only show neighbors of the specific state',
            'STATE': CliMatcher.EnumMatcher( Ospf3CliModels.ADJACENCY_STATE_MAP ),
            'VRF': vrfExprFactory,
          }
   handler = doShowIpv6OspfNeighbor
   cliModel = Ospf3ShowNeighVrfDictModel

BasicCli.addShowCommandClass( ShowIpv6OspfNeighborCmd )

#-------------------------------------------------------------------------------
# show ipv6 ospf access-list [<name>]
#-------------------------------------------------------------------------------
def showIpv6OspfAcl( mode, args ):
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 "ipv6",
                                 args.get( '<aclNameExpr>' ),
                                 serviceName=OspfConsts.serviceName )

class ShowIpv6OspfAclListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 ospf access-list [ ACL_NAME ]'
   data = {
         'ipv6'        : CliToken.Ipv6.ipv6MatcherForShow,
         'ospf'        : matcherOspf3Show,
         'access-list' : AclCli.accessListMatcher,
         'ACL_NAME'    : AclCli.ip6AclNameExpression,
          }
   handler = showIpv6OspfAcl
   cliModel = AclCliModel.AllAclList

BasicCli.addShowCommandClass( ShowIpv6OspfAclListCmd )

#-------------------------------------------------------------------------------
# clear ipv6 ospf [process_id] neighbor <* | nbrAddr | ifname> [ vrf vrfName ]
# clear ipv6 ospf access-list counters
#-------------------------------------------------------------------------------
def doClearIpv6Ospf( mode, args ):
   subCmd = args.get( 'access-list' ) or args.get( 'neighbor' )
   processId = args.get( 'PROCESS_ID' )
   nbr = args.get( '*' ) or args.get( 'ADDRESS' ) or args.get( 'INTERFACE' )
   vrfName = args.get( 'VRF' )
   afInstanceId = args.get( 'AF_INSTANCE_ID' )
   if subCmd == 'access-list':
      # clear counters of all access-lists
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, "ipv6" )
      return

   if afInstanceId is None:
      # Old style cli, query based on processId
      if processId or vrfName:
         processIds = getInstanceIds( mode, processId, vrfName )
      else:
         vrfName = DEFAULT_VRF
         processIds = getInstanceIds( mode, None, vrfName )
      if not processIds:
         return
      processId = processIds[0]
      afInstanceId = afIPv6InstanceID
      # Get the vrfName in case only processId has been specified
      # We need it for the clear request
      if not vrfName:
         vrfName = _getInstanceConfigByProcessId( processId ).vrfName
   else:
      vrfName = vrfName if vrfName else DEFAULT_VRF
      vrfs = getVrfToInstanceIdsDict( mode, vrfName )
      if not vrfs:
         return
      processId = 0

   ifName = ''

   # nbr is one of the three of: '*', ipaddr, intf. If the last,
   # we need to assign it to ifname variable to pass it on to mio api.
   if subCmd == 'neighbor':
      if isinstance( nbr, IntfCli.Intf ) :
         # does the interface exist?
         if not nbr.lookup() :
            mode.addError( 'Interface does not exist' )
            return
         ifName = nbr.status().deviceName
         waitDesc = 'neighbors on %s interface to be cleared' % nbr.status().intfId
      elif nbr == '*':
         waitDesc = 'all neighbors to be cleared'
      else:
         waitDesc = 'neighbor %s to be cleared' % nbr
   else:
      waitDesc = 'all neighbors to be cleared'

   # except when subcmd is 'neighbor addr', set 'nbr' to zero-ip
   # so that an invalid-ip exception is not raised. 'neighbor' has
   # an ip-addr argument when it is not '*' and was not used to
   # give us the ifname argument, that is, ifname is empty
   if not ( subCmd == 'neighbor' and ifName == '' and nbr != '*' ) :
      nbr = '0.0.0.0'

   req = Tac.Value( 'Routing6::Ospf3::ClearRequest', vrfName, afInstanceId,
                    subCmd, ifName, nbr )
   req.processId = processId
   reqId = "%d.%f" % ( os.getpid(), Tac.now() )
   ospf3ClearReq.clearRequest[ reqId ] = req

   try:
      Tac.waitFor( lambda: vrfName in ospf3ClearRespDir and \
                           reqId in ospf3ClearRespDir[ vrfName ].clearResponse,
                   description=waitDesc,
                   sleep=True, maxDelay=0.1, timeout=5.0 )
      nCleared = ospf3ClearRespDir[ vrfName ].clearResponse[ reqId ].nCleared
      opResult = ospf3ClearRespDir[ vrfName ].clearResponse[ reqId ].opResult
      if opResult is True :
         print "%d neighbor%s cleared" % ( nCleared,
                                              's' if nCleared != 1 else '' )
      else:
         print "Instance %s not active, 0 neighbors cleared" % \
               ( processId if processId \
                 else getAfFromInstanceId( afInstanceId, globalInstance=True ) )
   except ( Tac.Timeout, KeyError ):
      plural = '(s)' if subCmd == 'neighbor' else ''
      mode.addWarning( "%s%s may not have been cleared yet" % ( subCmd, plural ) )
   del ospf3ClearReq.clearRequest[ reqId ]

#-------------------------------------------------------------------------------
# clear ipv6 ospf [process_id] neighbor <* | nbrAddr | ifname> [ vrf vrfName ]
#-------------------------------------------------------------------------------
class ClearIpv6OspfNeighborCmd( CliCommand.CliCommandClass ):
   syntax = '''clear ipv6 ospf [ PROCESS_ID ] neighbor ( * | ADDRESS | INTERFACE )
               [ VRF ] [ show ]'''
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'ospf': ospfKw,
      'PROCESS_ID': matcherProcessId,
      'neighbor': matcherNeighbor,
      '*': matcherOptResetAllNeighbors,
      'ADDRESS': IpAddrMatcher.IpAddrMatcher( helpdesc='Neighbor address' ),
      'INTERFACE': IntfCli.Intf.matcherWithRoutingProtoSupport,
      'VRF': vrfExprFactory,
      'show': matcherShow,
   }
   handler = doClearIpv6Ospf

BasicCli.EnableMode.addCommandClass( ClearIpv6OspfNeighborCmd )

#-------------------------------------------------------------------------------
# # clear ipv6 ospf access-list counters
#-------------------------------------------------------------------------------
class ClearIpv6OspfAccessListCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 ospf access-list counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'ospf': ospfKw,
      'access-list': AclCli.accessListMatcher,
      'counters': matcherCounters,
   }
   handler = doClearIpv6Ospf

BasicCli.EnableMode.addCommandClass( ClearIpv6OspfAccessListCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ospf3ConfigDir, routing6HardwareStatus, routingHardwareStatus
   global ospf3ClearReq, ospf3ClearRespDir
   global l3IntfConfigDir, l3Config
   global aclConfig, aclCpConfig, aclStatus, aclCheckpoint

   ospf3ConfigDir = ConfigMount.mount( entityManager, Ospf3Consts.configPath,
                                       Ospf3Consts.configType, 'w' )
   routing6HardwareStatus = LazyMount.mount( entityManager,
                                             "routing6/hardware/status",
                                             "Routing6::Hardware::Status", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   ospf3ClearReq = LazyMount.mount( entityManager, 'routing6/ospf3/clear/request',
                                       "Routing6::Ospf3::ClearRequestNode", "w" )
   ospf3ClearRespDir = LazyMount.mount( entityManager,
                                        'routing6/ospf3/clear/response',
                                        "Tac::Dir", "ri" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
   # We need the interface vrfs to associate with the correct ospf3 instance
   l3IntfConfigDir = LazyMount.mount( entityManager, "l3/intf/config",
                                      "L3::Intf::ConfigDir", "r" )
   IntfCli.Intf.registerDependentClass( Ospf3Intf )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
