# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Str, Enum, Int, Submodel, Bool, List
from ArnetModel import IpGenericAddress
from IntfModels import Interface
import TableOutput
import IsisCliModelCommon
import IsisCliHelper
from TableOutput import terminalWidth
from CliPlugin.TunnelModels import LdpIsisSrTunnelTableCommon
from CliPlugin.VrfCli import generateVrfCliModel
from CliPlugin import TunnelCli
from SegmentRoutingCliShowHelper import (
   npHelpStr,
   reachabilityAlgorithmEnumValues,
   renderSRGB,
   renderCommonSummary,
   SrCommonHeader,
   SelfOriginatedSegmentStatisticsModel,
   SrPrefixSegmentFlagsModel,
   SrPrefixSegmentModel,
   SrGlobalBlockModel,
)

class SrCommonModel( SrCommonHeader ):
   """Base class for all SR models
   """
   _instanceName = Str( help="Instance Name" )
   systemId = Str( help="System ID" )
   hostname = Str( help="Hostname", optional=True )

   def render(self):
      print ""
      if self.hostname:
         print "System ID: %s\t\t\tInstance: %s" % ( self.hostname,
                                                     self._instanceName )
      else:
         print "System ID: %s\t\t\tInstance: %s" % ( self.systemId,
                                                     self._instanceName )

      SrCommonHeader.render( self )

class SrCommonWithAdjSidModel( SrCommonModel ):
   adjSidAllocationMode = Enum(values=["None", "SrOnly", "All"],
      help="Adj-SID allocation mode")
   adjSidPoolBase = Int(help="Base of Adj-SID pool", optional=True)
   adjSidPoolSize = Int(help="Size of Adj-SID pool", optional=True)

   def render(self):
      srAdjSidAllocationModeEnumMapping = dict([
                           ("None", "None"),
                           ("SrOnly", "SR-adjacencies"),
                           ("All", "all-adjacencies")])
      print "Adj-SID allocation mode: %s" % \
               srAdjSidAllocationModeEnumMapping[ self.adjSidAllocationMode]

      if self.adjSidAllocationMode != "None":
         print "Adj-SID allocation pool: Base: %s Size: %s" % \
             ( str(self.adjSidPoolBase).ljust(10),
            str(self.adjSidPoolSize) )

class PrefixSegmentFlagsModel( SrPrefixSegmentFlagsModel ):
   """Model for flags used in prefix segment
   """
   r = Bool(help="Set if prefix to which this Prefix-SID is attached has been " \
         "propagated by the router either from another level or from redistribution",
         default=False)
   p = Bool(help=npHelpStr, default=False)

   def toStr(self):
      flags = ""
      flags += "R:%d " % ( self.r )
      flags += "N:%d " % ( self.n )
      flags += "P:%d " % ( self.p )
      flags += "E:%d " % ( self.e )
      flags += "V:%d " % ( self.v )
      flags += "L:%d" % ( self.l )
      return flags

class AdjacencySegmentFlagsModel( Model ):
   """Model for flags used in adjacency segment
   """
   f = Bool(help="Set if Adj-SID refers to an adjacency with outgoing IPv6 " \
         "encapsulation", default=False)
   b = Bool(help="Set if Adj-SID is eligible for protection", default=False)
   v = Bool(help="Set if Adj-SID carries a value", default=False)
   l = Bool(help="Set if value/index carried by Adj-SID has local significance", \
         default=False)
   s = Bool(help="Set if Adj-SID refers to a set of adjacencies", default=False)

   def toStr(self):
      flags = ""
      flags += "F:%d " % ( self.f )
      flags += "B:%d " % ( self.b )
      flags += "V:%d " % ( self.v )
      flags += "L:%d " % ( self.l )
      flags += "S:%d" % ( self.s )
      return flags

class GlobalBlockModel( SrGlobalBlockModel ):
   """Model to store individual global block record.
   """
   systemId = Str( help="System ID" )
   hostname = Str( help="Hostname", optional=True )

class MisconfiguredAdjacencySegmentModel( Model ):
   """Model to represent misconfigured Adj-SIDs
   """
   localIntf = Interface(help="Local interface")
   sid = Int(help="Adjacency Segment assigned to this adjacency")
   af = Enum( values=[ 'ipv4', 'ipv6' ], help="Address Family" )
   reason = Enum( values=[ 'out-of-range', 'interface-type-mismatch' ],
         help="Reason why this configured adjacency segments is invalid" )

class PrefixSegmentModel( SrPrefixSegmentModel ):
   """Model to store individual ISIS prefix segment record
   """
   flags = Submodel( valueType=PrefixSegmentFlagsModel, help="Flags" )
   systemId = Str( help="System identifier" )
   hostname = Str( help="Hostname", optional=True )
   level = Int( help="Level" )
   protection = Enum( values=[ "link", "node", "unprotected" ],
                      help="Protection mode of the prefix segment" )
   srlgProtection = Enum( IsisCliHelper.SRLG_PROTECTION_MAP.values(),
                          optional=True,
                          help="TI-LFA SRLG protection mode of the segment" )

class AdjacencySegmentModel( Model ):
   """Model to store individual adjacency segments record
   """
   ipAddress = IpGenericAddress( help="IP Address of IS-IS adjacency" )
   localIntf = Interface( help="Local interface" )
   sid = Int( help="Adjacency Segment assigned to this adjacency" )
   lan = Bool( help="True if LAN-SID" )
   sidOrigin = Enum( values=[ "configured", "dynamic" ],
         help="Indicates if adj-SID is configured manually or assigned dynamically" )
   flags = Submodel( valueType=AdjacencySegmentFlagsModel, help="Flags" )
   level = Int( help="Level" )
   protection = Enum( values=[ "link", "node", "unprotected" ],
                      help="Protection mode of the local adj-SID" )
   srlgProtection = Enum( IsisCliHelper.SRLG_PROTECTION_MAP.values(),
                          optional=True,
                          help="TI-LFA SRLG protection mode of the segment" )

class ReceivedGlobalAdjacencySegmentModel( Model ):
   """Model for received global adjacency segments
   """
   sid = Int( help="Segment Id assigned to this adjacency segment" )
   systemId = Str( help="System ID of the segment originator" )
   ngbSystemId = Str( help="IS-IS neighbor of the segment originator" )
   hostname = Str( help="Hostname of the segment originator", optional=True )
   ngbHostname = Str( help="IS-IS neighbor hostname of the segment originator",
                      optional=True )
   flags = Submodel( valueType=AdjacencySegmentFlagsModel, help="Flags" )
   protection = Enum( values=[ "link", "node", "unprotected" ],
                      help="Protection mode of the global adj-SID" )
   srlgProtection = Enum( IsisCliHelper.SRLG_PROTECTION_MAP.values(),
                          optional=True,
                          help="TI-LFA SRLG protection mode of the segment" )

class SrSummaryModel( SrCommonWithAdjSidModel ):
   """Model for show isis segment-routing
   """
   srgbBase = Int(help="Base of SRGB")
   srgbSize = Int(help="Size of SRGB")
   reachabilityAlgorithm = Enum(values=reachabilityAlgorithmEnumValues,
                                help="IS-IS Reachability Algorithm")
   mappingServer = Bool(help="Set if this is a Segment Routing Mapping Server " \
         "(SRMS)")
   srPeerCount = Int(help="Number of IS-IS SR capable peers")
   selfOriginatedSegmentStatistics = Submodel(
         valueType=SelfOriginatedSegmentStatisticsModel,
         help="Self-originated segment statistics")

   def render(self):
      SrCommonModel.render(self)
      renderSRGB(self)
      SrCommonWithAdjSidModel.render(self)
      renderCommonSummary(self, "IS-IS")      
      self.selfOriginatedSegmentStatistics.render()


# Model used in registerShowCommand for show isis segment-routing
SegmentRoutingsSummaryModel = IsisCliModelCommon. \
                              generateIsisDictCliModel( SrSummaryModel )
SegmentRoutingsSummaryVRFsModel = generateVrfCliModel( SegmentRoutingsSummaryModel,
                                       "IS-IS instance information for all VRFs" )

class GlobalBlockSummaryModel( SrCommonModel ):
   """Model for show isis segment-routing global-blocks
   """
   srgbBase = Int(help="Base of SRGB")
   srgbSize = Int(help="Size of SRGB")
   srPeerCount = Int(help="Number of IS-IS SR peers capable segments")
   globalBlocks = List(valueType=GlobalBlockModel, help="Global blocks")

   def render(self):
      SrCommonModel.render(self)
      renderSRGB(self)
      
      headings = ( "SystemId", "Base", "Size" )
      table = TableOutput.createTable( headings )
      for item in self.globalBlocks:
         if item.hostname:
            table.newRow( item.hostname, item.base, item.size )
         else:
            table.newRow( item.systemId, item.base, item.size )

      print "Number of IS-IS segment routing capable peers: %d" % self.srPeerCount
      print ""
      print table.output()

# Model used in registerShowCommand for show isis segment-routing global-blocks
SegmentRoutingsGlobalBlocksSummaryModel = IsisCliModelCommon. \
                                 generateIsisDictCliModel( GlobalBlockSummaryModel )
SegmentRoutingsGlobalBlocksSummaryVRFsModel = \
                     generateVrfCliModel( SegmentRoutingsGlobalBlocksSummaryModel,
                                       "IS-IS instance information for all VRFs" )
      
      
class PrefixSegmentSummaryModel( SrCommonModel ):
   """Model for show isis segment-routing prefix-segments
   """
   nodeSidCount = Int(help="Number of SR node segments")
   proxyNodeSidCount = Int(help="Number of SR proxy node segments" )
   prefixSidCount = Int(help="Number of SR prefix segments")
   prefixSegments = List(valueType=PrefixSegmentModel,
         help="Prefix segments" )

# Model used in registerShowCommand for show isis segment-routing prefix-segments
SegmentRoutingsPrefixSegmentsSummaryModel = IsisCliModelCommon. \
                              generateIsisDictCliModel( PrefixSegmentSummaryModel )
SegmentRoutingsPrefixSegmentsSummaryVRFsModel = \
                     generateVrfCliModel( SegmentRoutingsPrefixSegmentsSummaryModel,\
                     "IS-IS instance information for all VRFs" )
     
class AdjacencySegmentSummaryModel( SrCommonWithAdjSidModel ):
   """Model for show isis segment-routing adjacency-segments
   """
   adjacencySegments = List(valueType=AdjacencySegmentModel,
         help="List of locally originated Adjacency Segments")
   receivedGlobalAdjacencySegments = List(
         valueType=ReceivedGlobalAdjacencySegmentModel,
         help="List of received global Adjacency Segments" )
   misconfiguredAdjacencySegments = List(
         valueType=MisconfiguredAdjacencySegmentModel,
         help="List of misconfigured Adjacency Segments" )

   def render(self):
      SrCommonModel.render(self)
      SrCommonWithAdjSidModel.render(self)
      adjCnt = len( self.adjacencySegments ) + \
               len( self.receivedGlobalAdjacencySegments )
      print "Adjacency Segment Count: %d" % adjCnt
      print "Flag Descriptions: F: IPv6 address family, B: Backup, V: Value\n" \
            "                   L: Local, S: Set\n"
      print "Segment Status codes: L1 - Level-1 adjacency, L2 - Level-2 " \
            "adjacency, P2P - Point-to-Point adjacency, LAN - Broadcast adjacency\n"
      headings = ( "Adj IP Address", "Local Intf", "SID", "SID Source", \
                   "Flags", "Type", "Protection" )

      print "Locally Originated Adjacency Segments"
      table = TableOutput.createTable( headings, tableWidth=terminalWidth() )
      sidOriginMap = {
               "configured" : "Configured",
               "dynamic" : "Dynamic"
            }
      for item in self.adjacencySegments:
         if item.level == 1:
            level = 'L1'
         elif item.level == 2:
            level = 'L2'
         elif item.level == 3:
            level = 'L1L2'
         else:
            assert( False )
         adjType = ( "LAN " if item.lan else "P2P " ) + level
         protectionStr = item.protection
         if item.srlgProtection:
            protectionStr += " with SRLG %s" % ( item.srlgProtection )
         table.newRow( item.ipAddress, item.localIntf.shortName, item.sid,
                       sidOriginMap[ item.sidOrigin ],
                       item.flags.toStr(), adjType, protectionStr )

      print table.output()

      if self.receivedGlobalAdjacencySegments:
         print ""
         print "Received Global Adjacency Segments"
         headings = ( "SID", "Originator", "Neighbor", "Flags", "Protection" )
         table = TableOutput.createTable( headings, tableWidth=terminalWidth() )
         for item in self.receivedGlobalAdjacencySegments:
            systemId = item.hostname if item.hostname else item.systemId
            neighbor = item.ngbHostname if item.ngbHostname else item.ngbSystemId
            protectionStr = item.protection
            if item.srlgProtection:
               protectionStr += " with SRLG %s" % ( item.srlgProtection )
            table.newRow( item.sid, systemId, neighbor, item.flags.toStr(),
                          protectionStr )
         print table.output()

      if self.misconfiguredAdjacencySegments:
         print ""
         print "Misconfigured Adj SIDs"
         invalidReasonMapping = { "out-of-range" : "Out of SRLB range",
                  "interface-type-mismatch" : "Static Adj-SID is supported only on "\
                                              "P2P interface" }

         invalidHeadings = ( "Local Intf", "AF", "SID", "Invalid reason" )
         invalidTable = TableOutput.createTable( invalidHeadings )
         for item in self.misconfiguredAdjacencySegments:
            invalidTable.newRow( item.localIntf.shortName, item.af, item.sid,
                  invalidReasonMapping[ item.reason ] )
         print invalidTable.output()
      
# Model used in registerShowCommand for show isis segment-routing adjacency-segments
SegmentRoutingsAdjacencySegmentsSummaryModel = IsisCliModelCommon. \
                           generateIsisDictCliModel( AdjacencySegmentSummaryModel )
SegmentRoutingsAdjacencySegmentsSummaryVRFsModel = \
               generateVrfCliModel( SegmentRoutingsAdjacencySegmentsSummaryModel, \
                                   "IS-IS instance information for all VRFs" )

class IsisSrTunnelTable( LdpIsisSrTunnelTableCommon ):

   def render( self ):
      headings = ( "Index", "Endpoint", "Nexthop", "Interface", "Labels",
                   "TI-LFA\ntunnel index" )
      fl = TableOutput.Format( justify='left' )
      table = TableOutput.createTable( headings )
      table.formatColumns( fl, fl, fl, fl, fl, fl )
      # pylint: disable-msg=no-member
      for tunnelId, tunnelTableEntryWithMplsVias in \
          sorted( self.entries.iteritems() ):
         tunnelTableEntryWithMplsVias.renderIsisTunnelTableEntry( table, tunnelId )

      print table.output()

   def getTunnelIdFromIndex( self, index, addrFamily=None ):
      tunnelType = 'srV6Tunnel' if addrFamily == 'ipv6' else 'srTunnel'
      return TunnelCli.getTunnelIdFromIndex( tunnelType, index )
