# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, SharedMem, Smash
import RoutingIsisShowSegmentRoutingModel
import RoutingIsisCli
from RoutingIsisCli import (
   showIsisCommand,
   isisKw,
   isisInstanceMatcher,
   vrfKw,
   vrfNameMatcher,
   isisShowCommandSyntaxWrapper
)
from TunnelCli import (
   tokenTunnelMatcher,
   tunnelIndexMatcher,
   getTunnelTableCommon,
   readMountTunnelTable,
   TunnelTableIdentifier
)
import ShowCommand, CliMatcher, CliCommand, BasicCli
import IpAddrMatcher, Ip6AddrMatcher
from SegmentRoutingCliShowHelper import (
   getAdjSidOrigin,
   _populateSRGBValues,
   _populateSrCommonHeader,
   _getReachabilityAlgorithm,
   _getSegmentRoutingGlobalBlocksPeersCount,
   _getSelfOriginatedSegmentStatistics
)
import sys
from CliModel import cliPrinted
import LazyMount

_routingIsisCli = RoutingIsisCli

_srDataPlaneEnum = Tac.Type( "Routing::SegmentRoutingCli::SrDataPlane" )
_srAdjAllocationType = Tac.Type( "Routing::Isis::SrAdjAllocationType" )
_interfaceTypeEnum = Tac.Type( 'Routing::Isis::InterfaceType' )
_srAdjSegTypeEnum = Tac.Type( "Routing::Isis::StaticAdjSegType" )
_addrFamilyEnum = Tac.Type( "Arnet::AddressFamily" )
srTunnelTable = None

ipv4SegmentKw = CliMatcher.KeywordMatcher( 'ipv4',
                                           helpdesc='IS-IS SR IPv4 segments' )
ipv6SegmentKw = CliMatcher.KeywordMatcher( 'ipv6',
                                           helpdesc='IS-IS SR IPv6 segments' )
segmentRoutingKw = CliMatcher.KeywordMatcher( 'segment-routing',
                                             helpdesc='Segment routing information' )
selfOrigKw = CliMatcher.KeywordMatcher( 'self-originated',
                                        helpdesc='Self-Originated segments' )
sharedMatchObjectPfxSegment = object()

def _getIsisIntfConfig():
   return _routingIsisCli.isisConfig.intfConfig

def _getIsisConfigAndStatus( mode, instanceId, instanceName ):
   if not instanceName:
      mode.addError( "Invalid IS-IS instance name" )
      return None, None
   
   instConfig = _routingIsisCli.isisConfig.instanceConfig[ instanceName ]
   
   if not instConfig:
      mode.addError( "IS-IS (%s) Can not find configuration" % instanceName )
      return None, None
  
   isisStatus = _routingIsisCli.isisStatusDir.get( instConfig.vrfName )
   instStatus = isisStatus.instanceStatus[ instanceName ]
   if not instStatus:
      mode.addError( "IS-IS (%s) Can not find status data" % instanceName )
      return None, None
   
   if instConfig.shutdown:
      return None, None
   
   if not _routingIsisCli.isisSrIsActive(mode, instanceName ):
      return None, None
   
   if instConfig.srDataPlane == _srDataPlaneEnum.srDataPlaneNone:
      return None, None

   return instConfig, instStatus

def _populateSrCommonModel(model, instConfig, instanceName, srSysdbStatus):
   model._instanceName = instanceName # pylint: disable=W0212
   model.systemId = str( instConfig.systemId )
   
   RouterId = Tac.Type( "Mpls::RouterId" )
   rId = RouterId()
   rId.stringValue = instConfig.systemId
   model.hostname = _routingIsisCli.getHostName( rId )
   _populateSrCommonHeader( model, instConfig.srDataPlane, srSysdbStatus ) 

#TODO this could go to others ... someone could benefit from it.
class _SrGuardMounts(object):
   def __init__(self, path = 'segmentrouting/isis/default',
         smashType = 'Smash::SegmentRouting::Status'):
      self.shmemEm = None
      self.smi = None
      self.srSmashStatus = None
      self.path = path
      self.smashType = smashType
      self.srSysdbStatus = None
   
   def __enter__(self):
      self.shmemEm = SharedMem.entityManager( sysdbEm=_routingIsisCli.entityManager )
      self.smi = Smash.mountInfo( 'reader' )
      self.srSmashStatus = self.shmemEm.doMount( self.path, self.smashType,
                                                 self.smi )
      self.srSysdbStatus = _routingIsisCli.srSysdbStatusDir.get( 'default' )
      return ( self.srSmashStatus, self.srSysdbStatus )
   
   def __exit__(self, exc_type, exc_val, exc_tb):
      self.shmemEm.doUnmount( self.path )

def _populateAdjacencyValues(result, instConfig, instStatus):
   srAdjSidAllocationModeEnumMapping = dict([
                        (_srAdjAllocationType.srAdjacencyAllocationNone, "None"),
                        (_srAdjAllocationType.srAdjacencyAllocationSrOnly, "SrOnly"),
                        (_srAdjAllocationType.srAdjacencyAllocationAll, "All"),
      (_srAdjAllocationType.srAdjacencyAllocationSrOnlyBackup, "SrOnly"),
      (_srAdjAllocationType.srAdjacencyAllocationAllBackup, "All")])

   result.adjSidAllocationMode = srAdjSidAllocationModeEnumMapping[
         instConfig.srAdjSegmentAlloc ]
   if instConfig.srAdjSegmentAlloc != _srAdjAllocationType.srAdjacencyAllocationNone:
      result.adjSidPoolBase = instStatus.srAdjBase
      result.adjSidPoolSize = instStatus.srAdjRange

#------------------------------------------------------------------------------------
# The "show isis segment-routing" command.
#------------------------------------------------------------------------------------
def getSegmentRoutingsSummaryModel(mode, cmdPrefix=None, instanceId=None,
                                   instanceName=None, vrfName=None):

   instConfig, instStatus = _getIsisConfigAndStatus(mode, instanceId, instanceName)

   if not instConfig or not instStatus:
      return
   
   result = RoutingIsisShowSegmentRoutingModel.SrSummaryModel()
   
   with _SrGuardMounts() as ( srSmashStatus, srSysdbStatus ):
      if not srSysdbStatus:
         return
      _populateSrCommonModel( result, instConfig, instanceName, srSysdbStatus )
      _populateSRGBValues(result, srSysdbStatus)
      _populateAdjacencyValues(result, instConfig, instStatus)
      result.reachabilityAlgorithm = _getReachabilityAlgorithm()
      result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus )
      result.selfOriginatedSegmentStatistics = _getSelfOriginatedSegmentStatistics(
                                           srSmashStatus, srSysdbStatus,
                                           instConfig.systemId )
      result.mappingServer = result.selfOriginatedSegmentStatistics.\
                                                               proxyNodeSidCount > 0
   
   return result

def _populateGlobalBlocks(globalBlocks, srSmashStatus, instConfig, srSysdbStatus):
   item = RoutingIsisShowSegmentRoutingModel.GlobalBlockModel()
   RouterId = Tac.Type( "Mpls::RouterId" )
   rId = RouterId()
   rId.stringValue = instConfig.systemId
   item.systemId = instConfig.systemId
   item.hostname = _routingIsisCli.getHostName( rId )
   item.base = srSysdbStatus.labelRange.base
   item.size = srSysdbStatus.labelRange.size
   globalBlocks.append(item)
   
   for index, entry in sorted( srSmashStatus.globalBlock.items() ):
      item = RoutingIsisShowSegmentRoutingModel.GlobalBlockModel()
      item.systemId = index.stringValue
      item.hostname = _routingIsisCli.getHostName( index )
      item.base = entry.base
      item.size = entry.size
      globalBlocks.append(item)

#------------------------------------------------------------------------------------
# The "show isis segment-routing global-blocks" command.
#------------------------------------------------------------------------------------
def getSegmentRoutingsGlobalBlocksSummaryModel(mode, cmdPrefix=None, instanceId=None,
                                               instanceName=None, vrfName=None):
   instConfig, instStatus = _getIsisConfigAndStatus(mode, instanceId, instanceName)

   if not instConfig or not instStatus:
      return

   result = RoutingIsisShowSegmentRoutingModel.GlobalBlockSummaryModel()

   with _SrGuardMounts() as (srSmashStatus, srSysdbStatus):
      if not srSysdbStatus:
         return
      _populateSrCommonModel(result, instConfig, instanceName, srSysdbStatus)
      _populateSRGBValues(result, srSysdbStatus)
      result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount(srSmashStatus)
      _populateGlobalBlocks( result.globalBlocks, srSmashStatus, instConfig, 
                             srSysdbStatus )

   return result

def getProtectionMode( protection ):
   protectionStr = "unprotected" if protection == "disabled" else protection
   return protectionStr

def getSrlgProtection( srlg ):
   if srlg == "srlgStrict":
      return "strict"
   elif srlg == "srlgLoose":
      return "loose"
   else:
      return "none"

#------------------------------------------------------------------------------------
# The "show isis segment-routing prefix-segments" command.
#------------------------------------------------------------------------------------
def getSegmentRoutingsPrefixSegmentsSummaryModel( mode, cmdPrefix=None,
                                                  instanceId=None, instanceName=None,
                                                  selfOrig=False, afSegments=0,
                                                  vrfName=None, fd=None ):
   instConfig, instStatus = _getIsisConfigAndStatus( mode, instanceId, instanceName )
   # Return cliPrinted( <modelclass> ) to convey the cliPrint library is being used
   modelSchema = \
         cliPrinted( RoutingIsisShowSegmentRoutingModel.PrefixSegmentSummaryModel )
   if not instConfig or not instStatus:
      return None
   isisHostnameMap = _routingIsisCli.isisSystemIdHostnameMap
   with _SrGuardMounts() as ( srSmashStatus, srSysdbStatus ):
      if not srSysdbStatus:
         return None
      if isinstance( srSysdbStatus, LazyMount._Proxy ): # pylint: disable-msg=W0212
         LazyMount.force( srSysdbStatus )
      if not fd:
         sys.stdout.flush()
         fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      helper = Tac.newInstance( "Isis::Cli::SrPrefixSegmentHelper",
                                instConfig, srSysdbStatus, isisHostnameMap,
                                srSmashStatus, afSegments, selfOrig )
      helper.render( fd, fmt )
   return modelSchema

def _populateAdjacencySegments(result, srSysdbStatus, systemId, vrfName):
   for _, entry in sorted( srSysdbStatus.adjacencySegment.items() ):
      sidOrigin = getAdjSidOrigin( entry, systemId )
      if sidOrigin == "remote":
         for ngbRtrid in entry.ngbRtrid:
            item = RoutingIsisShowSegmentRoutingModel \
                        .ReceivedGlobalAdjacencySegmentModel()
            item.sid = entry.sid.index
            item.systemId = entry.rtrid.stringValue
            item.ngbSystemId = ngbRtrid.stringValue
            item.hostname = _routingIsisCli.getHostName( entry.rtrid, vrfName )
            item.ngbHostname = _routingIsisCli.getHostName( ngbRtrid, vrfName )
            item.protection = getProtectionMode( entry.protectionMode )
            if entry.protectionSrlg != 'srlgNone':
               item.srlgProtection = getSrlgProtection( entry.protectionSrlg )
            item.flags = RoutingIsisShowSegmentRoutingModel \
                              .AdjacencySegmentFlagsModel()
            item.flags.f = entry.flags.isIpv6
            item.flags.b = entry.flags.backup
            item.flags.v = entry.flags.isValue
            item.flags.l = entry.flags.isLocal
            item.flags.s = entry.flags.isSet

            result.receivedGlobalAdjacencySegments.append(item)
      else:
         for nexthop in entry.nexthop:
            item = RoutingIsisShowSegmentRoutingModel.AdjacencySegmentModel()
            item.ipAddress = str( nexthop.hop )
            item.localIntf = str( nexthop.intfId )
            item.sid = entry.sid.index
            item.sidOrigin = sidOrigin
            item.lan = entry.isLan
            item.level = entry.level
            item.protection = getProtectionMode( entry.protectionMode )
            if entry.protectionSrlg != 'srlgNone':
               item.srlgProtection = getSrlgProtection( entry.protectionSrlg )
            item.flags = RoutingIsisShowSegmentRoutingModel \
                              .AdjacencySegmentFlagsModel()
            item.flags.f = entry.flags.isIpv6
            item.flags.b = entry.flags.backup
            item.flags.v = entry.flags.isValue
            item.flags.l = entry.flags.isLocal
            item.flags.s = entry.flags.isSet

            result.adjacencySegments.append( item )

def __addMisconfiguredAdjSidEntry( misconfiguredAdjacencySegments, intf, af, label,
                                   cfg, reason ):
   item = RoutingIsisShowSegmentRoutingModel.MisconfiguredAdjacencySegmentModel()
   item.localIntf = intf
   item.sid = label
   item.af = af
   item.reason = reason
   misconfiguredAdjacencySegments.append(item)

def _validateStaticAdjSidsConfiguration( mode, result, instStatus ):
   intfConfig = _getIsisIntfConfig()
   if not instStatus or not intfConfig:
      return
   _base = instStatus.srSrlbBase
   _range = instStatus.srSrlbRange
   outOfRangeLabelPresent = False
   interfaceTypeMismatch = False
   for intf, cfg in intfConfig.items():
      for adjSegKey in cfg.srSingleAdjacencySegment:
         af = 'ipv6' if adjSegKey.isIpV6 else 'ipv4'
         label = adjSegKey.sid.index
         if adjSegKey.sid.isValue is True:
            if label < _base or label >= _base + _range:
               #Out of range
               outOfRangeLabelPresent = True
               __addMisconfiguredAdjSidEntry( result.misconfiguredAdjacencySegments,
                                              intf, af, label, cfg, 'out-of-range' )
         if cfg.interfaceType == _interfaceTypeEnum.interfaceTypeLan:
            #For now we support static Adj SID only for p2p
            __addMisconfiguredAdjSidEntry( result.misconfiguredAdjacencySegments,
                                           intf, af, label, cfg,
                                           'interface-type-mismatch' )
            interfaceTypeMismatch = True

      for adjSegKey in cfg.srMultipleAdjacencySegment:
         if adjSegKey in cfg.srSingleAdjacencySegment:
            # if it's also present in the srSingleAdjacencySegment collection
            # then there is no need to validate it again
            continue
         af = 'ipv6' if adjSegKey.isIpV6 else 'ipv4'
         label = adjSegKey.sid.index
         if adjSegKey.sid.isValue is True:
            if label < _base or label >= _base + _range:
               #Out of range
               outOfRangeLabelPresent = True
               __addMisconfiguredAdjSidEntry( result.misconfiguredAdjacencySegments,
                                              intf, af, label, cfg, 'out-of-range' )
         if cfg.interfaceType == _interfaceTypeEnum.interfaceTypeLan:
            __addMisconfiguredAdjSidEntry( result.misconfiguredAdjacencySegments,
                                           intf, af, label, cfg,
                                          'interface-type-mismatch' )
            interfaceTypeMismatch = True

   if outOfRangeLabelPresent:
      mode.addWarning( "Some of configured Adj-SIDs are not within SRLB "\
         "Range [%s, %s]" % ( _base, _base + _range - 1 ) )

   if interfaceTypeMismatch:
      mode.addWarning( "Static Adj-SID is supported only on P2P interface" )

#------------------------------------------------------------------------------------
# The "show isis segment-routing adjacency-segments" command.
#------------------------------------------------------------------------------------
def getSegmentRoutingsAdjacencySegmentsSummaryModel(mode, cmdPrefix=None,
                                                    instanceId=None,
                                                    instanceName=None, vrfName=None):
   
   instConfig, instStatus = _getIsisConfigAndStatus(mode, instanceId, instanceName)

   if not instConfig or not instStatus:
      return
   
   result = RoutingIsisShowSegmentRoutingModel.AdjacencySegmentSummaryModel()
  
   # pylint: disable-msg=W0612
   with _SrGuardMounts() as ( srSmashStatus, srSysdbStatus ):
      if not srSysdbStatus:
         return
      _populateSrCommonModel( result, instConfig, instanceName, srSysdbStatus )
      _populateAdjacencyValues( result, instConfig, instStatus )
      _populateAdjacencySegments( result, srSysdbStatus, instConfig.systemId,
                                  vrfName )
      _validateStaticAdjSidsConfiguration( mode, result, instStatus )
   # pylint: enable-msg=W0612
   
   return result

class ShowIsisSegmentRoutingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'segment-routing' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'segment-routing' : segmentRoutingKw,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = RoutingIsisShowSegmentRoutingModel.SegmentRoutingsSummaryVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'segment-routing' ]
      args[ 'instDictModelType' ] = RoutingIsisShowSegmentRoutingModel.\
                                    SegmentRoutingsSummaryModel
      args[ 'cmdVrfModel' ] = RoutingIsisShowSegmentRoutingModel.\
                              SegmentRoutingsSummaryVRFsModel
      return showIsisCommand( mode, getSegmentRoutingsSummaryModel, **args )
BasicCli.addShowCommandClass( ShowIsisSegmentRoutingCmd )

class ShowIsisSegmentRoutingGlobalBlocksCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'segment-routing global-blocks' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'segment-routing' : segmentRoutingKw,
      'global-blocks' : 'Peer SRGBs',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = RoutingIsisShowSegmentRoutingModel.\
              SegmentRoutingsGlobalBlocksSummaryVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'segment-routing' ]
      del args[ 'global-blocks' ]
      args[ 'instDictModelType' ] = RoutingIsisShowSegmentRoutingModel.\
                                    SegmentRoutingsGlobalBlocksSummaryModel
      args[ 'cmdVrfModel' ] = RoutingIsisShowSegmentRoutingModel.\
                              SegmentRoutingsGlobalBlocksSummaryVRFsModel
      return showIsisCommand( mode, getSegmentRoutingsGlobalBlocksSummaryModel,
                              **args )

BasicCli.addShowCommandClass( ShowIsisSegmentRoutingGlobalBlocksCmd )

class ShowIsisSegmentRoutingPrefixSegmentCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'segment-routing prefix-segments '
                                          '[ { [ self-originated ] '
                                          '[ ipv4 | ipv6 ] } ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'segment-routing' : segmentRoutingKw,
      'prefix-segments' : 'Prefix Segment Information',
      'ipv4' : CliCommand.Node( matcher=ipv4SegmentKw, maxMatches=1,
                                sharedMatchObj=sharedMatchObjectPfxSegment ),
      'ipv6' : CliCommand.Node( matcher=ipv6SegmentKw, maxMatches=1,
                                sharedMatchObj=sharedMatchObjectPfxSegment ),
      'self-originated' : CliCommand.Node( matcher=selfOrigKw, maxMatches=1 ),
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = RoutingIsisShowSegmentRoutingModel.\
              SegmentRoutingsPrefixSegmentsSummaryVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'segment-routing' ]
      del args[ 'prefix-segments' ]
      args[ 'selfOrig' ] = bool( args.pop( 'self-originated', None ) )
      if 'ipv4' in args:
         args[ 'afSegments' ] = _addrFamilyEnum.ipv4
         del args[ 'ipv4' ]
      elif 'ipv6' in args:
         args[ 'afSegments' ] = _addrFamilyEnum.ipv6
         del args[ 'ipv6' ]
      args[ 'instDictModelType' ] = RoutingIsisShowSegmentRoutingModel.\
                                    SegmentRoutingsPrefixSegmentsSummaryModel
      args[ 'cmdVrfModel' ] = RoutingIsisShowSegmentRoutingModel.\
                              SegmentRoutingsPrefixSegmentsSummaryVRFsModel
      args[ 'useCliPrint' ] = True
      return showIsisCommand( mode, getSegmentRoutingsPrefixSegmentsSummaryModel,
                              **args )
BasicCli.addShowCommandClass( ShowIsisSegmentRoutingPrefixSegmentCmd )

class ShowIsisSegmentRoutingRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis [ INSTANCE ] segment-routing routes '
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'segment-routing' : segmentRoutingKw,
      'routes' : 'Routes in Smash',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      del args[ 'segment-routing' ]
      del args[ 'routes' ]
      return showIsisCommand( mode, _routingIsisCli.showIsisSegmentRoutingRoutes,
                              **args )
BasicCli.addShowCommandClass( ShowIsisSegmentRoutingRoutesCmd )

class ShowIsisSegmentRoutingAdjSegmentCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'segment-routing adjacency-segments' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'segment-routing' : segmentRoutingKw,
      'adjacency-segments' : 'Adjacency Segment Information',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = RoutingIsisShowSegmentRoutingModel.\
              SegmentRoutingsAdjacencySegmentsSummaryVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'segment-routing' ]
      del args[ 'adjacency-segments' ]

      args[ 'instDictModelType' ] = RoutingIsisShowSegmentRoutingModel.\
                                    SegmentRoutingsAdjacencySegmentsSummaryModel
      args[ 'cmdVrfModel' ] = RoutingIsisShowSegmentRoutingModel.\
                              SegmentRoutingsAdjacencySegmentsSummaryVRFsModel
      return showIsisCommand( mode, getSegmentRoutingsAdjacencySegmentsSummaryModel,
                              **args )
BasicCli.addShowCommandClass( ShowIsisSegmentRoutingAdjSegmentCmd )

#-------------------------------------------------------------------------------
# show isis segment-routing tunnel [ tunnel-index | tep ]
#-------------------------------------------------------------------------------
endpointHelpString = 'Match this endpoint prefix'

class ShowIsisSegmentRoutingTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis segment-routing tunnel ' \
            '[ IPADDR | IP6ADDR | TUNNEL-INDEX ]'
   data = {
      'isis' : isisKw,
      'segment-routing': segmentRoutingKw,
      'tunnel' : tokenTunnelMatcher,
      'IPADDR' : IpAddrMatcher.IpPrefixMatcher( helpdesc=endpointHelpString ),
      'IP6ADDR' : Ip6AddrMatcher.Ip6PrefixMatcher( helpdesc=endpointHelpString ),
      'TUNNEL-INDEX' : tunnelIndexMatcher,
   }
   cliModel = RoutingIsisShowSegmentRoutingModel.IsisSrTunnelTable

   @staticmethod
   def handler( mode, args ):
      endpoint = args.pop( 'IPADDR', None ) or args.pop( 'IP6ADDR', None )
      tunnelIndex = args.pop( 'TUNNEL-INDEX', None )
      if endpoint:
         endpoint = str( endpoint )

      return getTunnelTableCommon(
         tunnelIndex=tunnelIndex, endpoint=endpoint, tunnelTable=srTunnelTable,
         tunnelModel=RoutingIsisShowSegmentRoutingModel.IsisSrTunnelTable )
BasicCli.addShowCommandClass( ShowIsisSegmentRoutingTunnelCmd )

def Plugin( entityManager ):
   global srTunnelTable

   srTunnelTable = readMountTunnelTable( TunnelTableIdentifier.srTunnelTable,
                                         entityManager )
