#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import os
import re
import sys
from functools import partial

import Ark
from Toggles import MplsToggleLib
from Toggles import IsisToggleLib
from Toggles import gatedToggleLib
from Toggles import RoutingLibToggleLib
import Tac, LazyMount, SharedMem, Smash
import ConfigMount, IpAddrMatcher, Ip6AddrMatcher
import CliParser, BasicCli, IntfCli, IraIpCli, IraIp6Cli, Arnet, CliToken
import Intf.IntfRange as IntfRange
import TechSupportCli, IraVrfCli, IraIpIntfCli, IraIp6IntfCli
from ReversibleSecretCli import decodeKey
from RoutingCommon import GATED_PROTO_SH_TECH_TS
from CliToken.Router import routerMatcherForConfig as routerKw
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
from CliMode.Isis import RoutingIsisMode, RoutingIsisAfMode, RoutingIsisSrMplsMode, \
      RoutingIsisMetricProfileMode, RoutingIsisTeMode, RoutingIsisAreaProxyMode
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED, ALL_VRF_NAME
from RouteMapCli import mapNameMatcher
from BasicCliUtil import notAPrefixOf

from MplsCli import mplsNodeForShow, mplsForShowNode
from LdpCli import ldpForShowKw, syncForShowKw

from CliModel import cliPrinted
import MplsModel
# pylint: disable=cyclic-import
import IsisCliModels
import CliMatcher

from IsisCliModels import (
   IsisMplsLdpSyncInterfaceModel,
   IsisMplsLdpSyncModel,
   TilfaTunnelTableEntry,
   TilfaTunnelTable
)

from TunnelCli import (
   getTunnelIdFromIndex,
   getTunnelIndexFromId,
   readMountTunnelTable,
   TunnelTableIdentifier,
   tokenTunnelMatcher,
   tunnelIndexMatcher
)

from TunnelModels import (
   MplsVia as TunnelMplsVia
)

from RibCapiLib import AmiResponseException
from RibCapiLib import showRibCapiCommand
from RibCapiLib import EmptyResponseException
import CliPlugin.SegmentRoutingCli
import CliPlugin.VrfCli as VrfCli
import ShowCommand
import CliCommand
from CliPrint import CliPrint
from TypeFuture import TacLazyType
cprinter = CliPrint().lib

# Exception not raised when try fails. Suppress pylint warning
# pylint: disable-msg=bare-except
# pylint: disable-msg=protected-access
# pylint: disable-msg=broad-except
l3Config = None
tilfaTunnelTable = None
isisSrV4BindingTable = None
isisSrV6BindingTable = None
isisConfig = None
isisStatusDir = None
isisClearReqDir = None
isisClearRespDir = None
srSysdbStatusDir = None
routingHardwareStatus = None
routing6Config = None
routing6HardwareStatus = None
allIntfStatusDir = None
ipConfig  = None
ip6Config = None
allVrfConfig = None
vrfInfoDir = None
vrf6InfoDir = None
entityManager = None
linkReadyStatusVrfColl = None
ldpProtoConfigColl = None
ldpConfigColl = None
mplsRoutingConfig = None
mplsStatus = None
isisSystemIdHostnameMap = None
trapConfig = None
U32_MAX_VALUE = 0xFFFFFFFF

AdjacencyLoggingType = TacLazyType( 'Routing::Isis::AdjacencyLoggingType' )
AdjacencySegmentKey = TacLazyType( 'Mpls::SegmentRouting::AdjacencySegmentKey' )
AreaLeaderPriority = TacLazyType( 'Routing::Isis::AreaLeaderPriority' )
ArnetAddressFamily = TacLazyType( 'Arnet::AddressFamily' )
AuthAlgo = TacLazyType( 'Routing::Isis::AuthAlgo' )
AuthKey = TacLazyType( 'Routing::Isis::AuthKey' )
AuthKeyid = TacLazyType( 'Routing::Isis::AuthKeyid' )
AuthMode = TacLazyType( 'Routing::Isis::AuthMode' )
ClearAdjRequest = TacLazyType( 'Routing::Isis::ClearAdjRequest' )
ClearCountersRequest = TacLazyType( 'Routing::Isis::ClearCountersRequest' )
ClearDatabaseRequest = TacLazyType( 'Routing::Isis::ClearDatabaseRequest' )
ClearRequest = TacLazyType( 'Routing::Isis::ClearRequest' )
CommonLibConsumerSm = TacLazyType( 'CommonLibSmash::CommonLibConsumerSm' )
FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
GrHoldTime = TacLazyType( 'Routing::Isis::GrHoldTime' )
GrTimerT2 = TacLazyType( 'Routing::Isis::GrTimerT2' )
HelloInterval = TacLazyType( 'Routing::Isis::HelloInterval' )
HelloMultiplier = TacLazyType( 'Routing::Isis::HelloMultiplier' )
HelloPadding = TacLazyType( 'Routing::Isis::HelloPadding' )
InstanceStatus = TacLazyType( 'Routing::Isis::InstanceStatus' )
InterfaceType = TacLazyType( 'Routing::Isis::InterfaceType' )
IntfHelloPadding = TacLazyType( 'Routing::Isis::IntfHelloPadding' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
IsisAddressFamily = TacLazyType( 'Routing::Isis::AddressFamily' )
IsisIntfBfdConfig = TacLazyType( 'Routing::Isis::IsisIntfBfdConfig' )
IsisMplsBindingsHelper = TacLazyType( 'Isis::Cli::IsisMplsBindingsHelper' )
LabelBindingTable = TacLazyType( 'Mpls::LabelBindingTable' )
LabelBindingTableColl = TacLazyType( 'Mpls::LabelBindingTableColl' )
Level = TacLazyType( 'Routing::Isis::Level' )
LspGenIntervalMsec = TacLazyType( 'Routing::Isis::LspGenIntervalMsec' )
LspGenIntervalSec = TacLazyType( 'Routing::Isis::LspGenIntervalSec' )
LspGenThrottleTimer = TacLazyType( 'Routing::Isis::LspGenThrottleTimer' )
LspInterval = TacLazyType( 'Routing::Isis::LspInterval' )
LspRefreshInterval = TacLazyType( 'Routing::Isis::LSPRefreshInterval' )
MaxLSPLifetime = TacLazyType( 'Routing::Isis::MaxLSPLifetime' )
MaxLSPSize = TacLazyType( 'Routing::Isis::MaxLSPSize' )
Metric = TacLazyType( 'Routing::Isis::Metric' )
MetricRule = TacLazyType( 'Routing::Isis::MetricRule' )
MinLspRemainingLifetime = TacLazyType( 'Routing::Isis::MinLspRemainingLifetime' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
OverloadStartupDelay = TacLazyType( 'Routing::Isis::OverloadStartupDelay' )
PrefixSegment = TacLazyType( 'Routing::Isis::PrefixSegment' )
Priority = TacLazyType( 'Routing::Isis::Priority' )
ProtectionMode = TacLazyType( 'Routing::Isis::ProtectionMode' )
ProtectionConfig = TacLazyType( 'Routing::Isis::ProtectionConfig' )
ProtectionSrlg = TacLazyType( 'Routing::Isis::ProtectionSrlg' )
Redistribute = TacLazyType( 'Routing::Isis::Redistribute' )
RouteTag = TacLazyType( 'Routing::Isis::RouteTag' )
SegmentIdentifier = TacLazyType( 'Mpls::SegmentRouting::SegmentIdentifier' )
Speed = TacLazyType( 'Routing::Isis::Speed' )
SpeedUnitType = TacLazyType( 'Routing::Isis::SpeedUnitType' )
SpfIntervalSec = TacLazyType( 'Routing::Isis::SpfIntervalSec' )
SpfIntervalMsec = TacLazyType( 'Routing::Isis::SpfIntervalMsec' )
SpfThrottleTimer = TacLazyType( 'Routing::Isis::SpfThrottleTimer' )
SrAdjAllocationType = TacLazyType( 'Routing::Isis::SrAdjAllocationType' )
SrDataPlane = TacLazyType( 'Routing::SegmentRoutingCli::SrDataPlane' )
TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
InstanceKey = TacLazyType( 'Routing::Isis::InstanceKey' )
invalidInstanceName = Tac.Type( 'Routing::Isis::IntfConfig' ).instanceNameInvalid
nullPrefix = Tac.Type( 'Arnet::Prefix' ).nullPrefix
srSidInvalid = Tac.Type( 'Routing::SegmentRoutingCli::Constants' ).srSidInvalid
RouterId = TacLazyType( 'Mpls::RouterId' )

LVL_KW_TO_TAC_TYPE_MAP = Ark.ReversibleDict( {
      'level-1' : Level.level1,
      'level-2' : Level.level2,
      'level-1-2' : Level.level1_2,
} )
SHA_KW_TO_TAC_TYPE_MAP = {
      'sha-1' : AuthAlgo.sha1,
      'sha-224' : AuthAlgo.sha224,
      'sha-256' : AuthAlgo.sha256,
      'sha-384' : AuthAlgo.sha384,
      'sha-512' : AuthAlgo.sha512,
}

routerIdKw = CliMatcher.KeywordMatcher( 'router-id',
         helpdesc='Configure router ID for IS-IS routing processes' )
ipv6AddrFormatKw = CliMatcher.KeywordMatcher( 'ipv6',
         helpdesc='IS-IS router ID in IPv6 address format' )
routerIdV4Matcher = IpAddrMatcher.IpAddrMatcher(
         "IS-IS router ID in IP address format" )
routerIdV6Matcher = Ip6AddrMatcher.Ip6AddrMatcher(
         "IS-IS router ID in IPv6 address format" )
ipv4IPMatcher = IpAddrMatcher.ipAddrWithFullMaskExpr( 'IP address',
      'Subnet\'s mask value', 'IPv4 address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
ipv6IPMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( 'IPv6 address prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

#------------------------------------------------------------------------------
# Configuration keywords used by CLI parser
#------------------------------------------------------------------------------
adjacencySegmentKw = CliMatcher.KeywordMatcher( 'adjacency-segment',
      helpdesc='Set IS-IS SR adjacency segment interface configuration' )
afMatcher = CliMatcher.EnumMatcher( {
      'ipv4' : 'IPv4 related',
      'ipv6' : 'IPv6 related',
} )
bgpKw = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP routes' )
globalKw = CliMatcher.KeywordMatcher( 'global',
      helpdesc='Global adjacency SID' )
indexKw = CliMatcher.KeywordMatcher( 'index',
      helpdesc='Index to be assigned as Adj-SID for adjacency on this interface' )
isisKw = CliMatcher.KeywordMatcher( 'isis',
      helpdesc='IS-IS commands' )
labelKw = CliMatcher.KeywordMatcher( 'label',
      helpdesc='Label value to be assigned as Adj-SID for adjacency'
      ' on this interface' )
level12MatcherForConfig = CliMatcher.EnumMatcher( {
      'level-1' : 'Configure at level 1',
      'level-2' : 'Configure at level 2',
      'level-1-2' : 'Configure at level-1-2',
} )
levelMatcherForConfig = CliMatcher.EnumMatcher( {
      'level-1' : 'Configure at level 1',
      'level-2' : 'Configure at level 2',
} )
lspKw = CliMatcher.KeywordMatcher( 'lsp',
      helpdesc='Configure LSP specific parameters' )
multipleKw = CliMatcher.KeywordMatcher( 'multiple',
      helpdesc='configure multiple Adj-SIDs' )
p2pKw = CliMatcher.KeywordMatcher( 'p2p', helpdesc='P2P interface type' )
rfc8202Kw = CliMatcher.KeywordMatcher( 'rfc8202',
      helpdesc='IS-IS multi-instance RFC8202 support' )
redistributeKw = CliMatcher.KeywordMatcher( 'redistribute',
      helpdesc='Redistribute routes with IS-IS' )
unicastKw = CliMatcher.KeywordMatcher( 'unicast',
      helpdesc='Unicast sub-address family' )

#------------------------------------------------------------------------------
# Show keywords used by CLI parser
#------------------------------------------------------------------------------
dbKw = CliMatcher.KeywordMatcher( 'database',
      helpdesc='Database information' )
detailKw = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Display information in detail' )
intervalKw = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Configure LSP transmission interval' )
isisLspDeprecatedKw = CliMatcher.KeywordMatcher( 'lsp-interval',
      helpdesc='Set LSP transmission interval' )
levelMatcherForShow = CliMatcher.EnumMatcher( {
      'level-1' : 'Level 1 only',
      'level-2' : 'Level 2 only',
} )
lspIdRegex = "[A-Za-z0-9,_:\-\./#%+]{1,255}\.([0-9a-fA-F]{2}-[0-9a-fA-F]{2})"
lspIntervalMatcher = CliMatcher.IntegerMatcher( LspInterval.min,
      LspInterval.max,
      helpdesc='Interval value in milliseconds' )
lspKwForShow = CliMatcher.KeywordMatcher( 'lsp',
      helpdesc='IS-IS LSP information' )
singleLspidNode = CliCommand.singleNode( CliMatcher.PatternMatcher(
      pattern=lspIdRegex,
      helpdesc='LSPID in HHHH.HHHH.HHHH.HH-HH or hostname.HH-HH format',
      helpname='LSPID' ),
      sharedMatchObj=object() )
singleLevelNodeForShow = CliCommand.singleNode( levelMatcherForShow,
      sharedMatchObj=object() )
tokenTilfaForShow = CliMatcher.KeywordMatcher( 'ti-lfa',
      helpdesc='TI-LFA related path information' )
topologyKw = CliMatcher.KeywordMatcher( 'topology',
      helpdesc='Paths to IS-IS routers' )
txKw = CliMatcher.KeywordMatcher( 'tx',
      helpdesc='Configure LSP transmission parameter' )
vrfKw = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='VRF name' )

# Authentication related keywords
authPasswdMaxLen = 80
authPasswdMinLen = 1
authPasswdPattern = '.{%d,%d}' % ( authPasswdMinLen, authPasswdMaxLen )
authAlgoKw = CliMatcher.KeywordMatcher( 'algorithm',
      helpdesc="Configure authentication algorithm" )
authEncryptedPasswdKw = CliMatcher.PatternMatcher( '.+',
      helpname='WORD',
      helpdesc='Encrypted password' )
authKeyIdKw = CliMatcher.KeywordMatcher( 'key-id',
      helpdesc="Configure authentication keyid" )
authKeyKw = CliMatcher.KeywordMatcher( 'key',
      helpdesc="Configure authentication key" )
authKw = CliMatcher.KeywordMatcher( 'authentication',
      helpdesc='Configure authentication for IS-IS' )
authUnencryptedPasswdKw = CliMatcher.PatternMatcher( authPasswdPattern,
      helpname='WORD',
      helpdesc='Password (up to %d characters)' % authPasswdMaxLen )
clearIsisKw = CliMatcher.KeywordMatcher( 'isis',
      helpdesc='Clear IS-IS state' )
encryptionType0Kw = CliMatcher.KeywordMatcher( '0',
      helpdesc='Encryption type - unencrypted' )
encryptionType7Kw = CliMatcher.KeywordMatcher( '7',
      helpdesc='Encryption type - proprietary' )
keyIdKw = CliMatcher.KeywordMatcher( 'key-id',
      helpdesc='Configure sender key-id' )
keyIdMatcher = CliMatcher.IntegerMatcher( AuthKeyid.shaMin,
      AuthKeyid.shaMax,
      helpdesc='Set key-id value' )
modeKw = CliMatcher.KeywordMatcher( 'mode',
      helpdesc='Configure authentication mode' )
rxDisabledKw = CliMatcher.KeywordMatcher( 'rx-disabled',
      helpdesc='Disable authentication on receive side' )

sharedMatchObjDb = object()

#------------------------------------------------------------------------------
# Common patterns
#------------------------------------------------------------------------------

netRegex = \
           '(?P<areaid>(?:[\da-fA-F]{2}(?:\.[\da-fA-F]{4}){0,6})|' \
           '((?:[\da-fA-F]{4})(?:\.[\da-fA-F]{4}){0,5}))' \
           '\.(?P<systemid>[\da-fA-F]{4}\.[\da-fA-F]{4}\.[\da-fA-F]{4})' \
           '\.00$'

netPartialRegex = \
    '(?:[\da-fA-F]{0,2}\.)?' \
    '(?:[\da-fA-F]{4}\.){0,8}' \
    '(?:[\da-fA-F]{0,4}|' \
    '[\da-fA-F]{4}\.0{0,2})$'

netKw = CliMatcher.PatternMatcher( pattern=netRegex,
        partialPattern=netPartialRegex, helpname='NET',
        helpdesc="NET in HH.HHHH.HHHH.HHHH.HHHH.HHHH."
                 "HHHH.HHHH.HHHH.HHHH.00 format" )

# Restricting the character list allowed in hostname as we would like to support
# filtering of LSP database based on LSPID with hostname in the system Id portion.
# This requires us to take LSPID as input which will be validated in cliribd as per
# CCTYPE_STRING. Without this we could end up in situations where we configure
# hostname, but fail to filter the LSP database stating invalid input.
# Hostnames will be restricted to alphanumeric and any of [,_:-./#%+]
hostnameRegex = '[A-Za-z0-9,_:\-\./#%+]{1,255}'
hostnameKw = CliMatcher.PatternMatcher( hostnameRegex,
        helpdesc='IS hostname',
        helpname='HOSTNAME' )

def getProtoRouteType( args ):
   if 'bgp' in args:
      proto = 'protoBgp'
   elif 'dhcp' in args:
      proto = 'protoDhcp'
   else:
      proto = args.get( 'ospf' ) or args.get( 'ospfv3' )
   routeType = ( ( args.get( 'external' ) or args.get( 'internal' ) or
                   args.get( 'nssa-external' ) ) )
   mapName = args.get( 'MAPNAME' )
   return proto, routeType, mapName

# Callback hook to check if a given ip address can be assigned to an interface
def isisSrIpAddressAllowed( intfName, newAddrWithMask=None,
                              secondary=False, mode=None ):
   # If the interface:
   # - is not an isis interface 
   # - or it doesn't have a node-segment config  
   # - or the adddress isn't a /32
   # There is nothing to validate, return true
   intfConfig = _getIntfConfig( intfName )
   srIntfConfig = _getSrIntfConfig( intfName )
   if intfConfig is None or \
      srIntfConfig is None or \
      newAddrWithMask.len != 32 or \
      srIntfConfig.srNodeSegmentIndex == srSidInvalid or \
      secondary:
      return [ True, None ]

   # The interface has a node segment configuration, so verify that the
   # prefix doesn't conflict with any other known prefix segments
   prefix = Arnet.IpGenPrefix( str( newAddrWithMask.subnet ) )
   if _isisSrConflictingPrefixFound( intfConfig.instanceName, intfName, prefix ):
      msg = "Conflicts with the IP address of another IS-IS SR "
      msg += "proxy-node/prefix segment"
      return [ False, msg ]
   return [ True, None ]

IraIpIntfCli.canSetIntfIpHook.addExtension( isisSrIpAddressAllowed )

# Callback hook to check if a given ipv6 address can be assigned to an interface
def isisSrIp6AddressAllowed( intfName, ip6Addr=None, mode=None ):
   srIntfConfig = _getSrIntfConfig( intfName )
   intfConfig = _getIntfConfig( intfName )
   # If the interface:
   # - is not an isis interface 
   # - or it doesn't have a node-segment config  
   # - or the adddress isn't a /128
   # There is nothing to validate, return true
   if intfConfig is None or \
      srIntfConfig is None or \
      ip6Addr.len != 128 or \
      srIntfConfig.srV6NodeSegmentIndex == srSidInvalid :
      return [ True, None ]

   # The interface has a node segment configuration, so verify that the
   # prefix doesn't conflict with any other known prefix segments
   prefix = Arnet.IpGenPrefix( str( ip6Addr ) )
   if _isisSrConflictingPrefixFound( intfConfig.instanceName, intfName, prefix ):
      msg = "Conflicts with the IPv6 address of another IS-IS SR "
      msg += "proxy-node/prefix segment"
      return [ False, msg ]
   return [ True, None ]

IraIp6IntfCli.canSetIntfIpHook.addExtension( isisSrIp6AddressAllowed )

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------
def routingSupportedGuard( mode, token ):
   if routingHardwareStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default VRF IS-IS instance
#-------------------------------------------------------------------------------
def isisNonDefaultVrfGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.vrfName != DEFAULT_VRF:
      return CliParser.guardNonDefaultVrf

#-------------------------------------------------------------------------------
# Guard to prevent configuration in non-default IS-IS instance (instance ID != 0)
#-------------------------------------------------------------------------------
def isisNonDefaultInstanceGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId != 0:
      return "not supported in non-default IS-IS instance"

#-------------------------------------------------------------------------------
# Guard to prevent configuration in default IS-IS instance (instance ID 0)
#-------------------------------------------------------------------------------
def isisDefaultInstanceGuard( mode, token ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   if instanceConfig.instanceId == 0:
      return "not supported in default IS-IS instance"

#-------------------------------------------------------------------------------
# Added to get kernel interface name
#-------------------------------------------------------------------------------
def kernelDeviceName( intfName ):
   name = ""
   intfStatus = allIntfStatusDir.intfStatus.get( intfName )
   if intfStatus:
      name = intfStatus.deviceName
   return name

#-------------------------------------------------------------------------------
# Address family string to enum
#-------------------------------------------------------------------------------
def addrFamilyStrToEnum( addrFamilyStr ):
   if addrFamilyStr == "ipv4":
      return IsisAddressFamily.addressFamilyIPv4
   elif addrFamilyStr == "ipv6":
      return IsisAddressFamily.addressFamilyIPv6
   elif addrFamilyStr == "both":
      return IsisAddressFamily.addressFamilyBoth
   else:
      return IsisAddressFamily.addressFamilyNone

#-------------------------------------------------------------------------------
# Segment routing common infrastructure
#-------------------------------------------------------------------------------
def nodeIndexRangeFn( mode ):
   intfConfig = _getIntfConfig( mode.intf.name ) if hasattr( mode, 'intf' ) else None

   if intfConfig is not None and intfConfig.instanceName != '':
      instanceName = intfConfig.instanceName
      instanceConfig = isisConfig.instanceConfig.get( instanceName )
      instanceStatus = None

      if instanceConfig is not None:
         # obtain vrfName from instanceConfig and use this to obtain
         # instanceStatus
         vrfName = instanceConfig.vrfName
         isisStatus = isisStatusDir.get( vrfName )

         if isisStatus is not None:
            instanceStatus = isisStatus.instanceStatus.get( instanceName )
      else:
         # in the case of re-config, cannot use the isisConfig.instanceConfig
         # as instance config won't exist, so take brute force approach and
         # iterate through the Status dir vrfs looking for matching instance
         for vrfName in isisStatusDir.keys():
            isisStatus = isisStatusDir[ vrfName ]
            intfNames = [ x.intfName for x in isisStatus.intfStatus ]
            if intfConfig.name in intfNames:
               instanceStatus = isisStatus.instanceStatus.get( instanceName )
               break

      if instanceStatus is not None:
         return ( 0, instanceStatus.srSrgbRange - 1 )

   return ( 0, InstanceStatus.srSrgbRangeDefault - 1 )

def tokenLabelValueRangeFn( model ):
   return ( MplsLabel.unassignedMin, MplsLabel.max )

indexValueMatcher = CliMatcher.DynamicIntegerMatcher( nodeIndexRangeFn,
      helpdesc='Value of index' )
labelValueMatcher = CliMatcher.DynamicIntegerMatcher( tokenLabelValueRangeFn,
      helpdesc='Value of the MPLS label' )

#-------------------------------------------------------------------------------
# Isis commands in "show tech-support"
#-------------------------------------------------------------------------------
# In the future if we add commands that fork rib to produce their output, then 
# we should split the list of commands into two. One list for commands that don't 
# fork ribd using GATED_PROTO_SH_TECH_TS and the second one for commands that fork
# using GATED_PROTO_FORK_SH_TECH_TS
def _isisShowTechCmds():
   return [ 'show isis summary vrf all', 
            'show isis neighbors detail vrf all', 
            'show isis interface detail vrf all', 
            'show isis database traffic-engineering vrf all',
            'show isis hostname vrf all',
            'show isis network topology vrf all',
            'show isis spf log vrf all',
            'show isis counters details vrf all',
            'show isis segment-routing adjacency-segments vrf all',
            'show isis segment-routing global-blocks vrf all',
            'show isis segment-routing prefix-segments vrf all',
            'show isis segment-routing tunnel',
            'show isis local-convergence-delay detail vrf all',
            'show isis lsp purges vrf all',
            ]

def _isisShowTechSummaryCmds():
   return [ 'show isis summary vrf all',
            'show isis neighbors vrf all', 
            'show isis interface vrf all',
            'show isis segment-routing',
            'show isis local-convergence-delay vrf all'
          ]

TechSupportCli.registerShowTechSupportCmdCallback(
    GATED_PROTO_SH_TECH_TS, _isisShowTechCmds,
    summaryCmdCallback=_isisShowTechSummaryCmds )

def addAddrFamilyToConfig( currentAf, addrFamily ):
   if currentAf == addrFamilyStrToEnum( "none" ):
      newAf = addrFamilyStrToEnum( addrFamily )
   elif currentAf == addrFamilyStrToEnum( "ipv4" ) and addrFamily == "ipv6":
      newAf = addrFamilyStrToEnum( "both" )
   elif currentAf == addrFamilyStrToEnum( "ipv6" ) and addrFamily == "ipv4":
      newAf = addrFamilyStrToEnum( "both" )
   else:
      newAf = currentAf
   return newAf

def delAddrFamilyFromConfig( currentAf, addrFamily ):
   if currentAf == addrFamilyStrToEnum( "both" ) and addrFamily == "ipv4":
      newAf = addrFamilyStrToEnum( "ipv6" )
   elif currentAf == addrFamilyStrToEnum( "both" ) and addrFamily == "ipv6":
      newAf = addrFamilyStrToEnum( "ipv4" )
   elif currentAf == addrFamilyStrToEnum( addrFamily ):
      newAf = addrFamilyStrToEnum( "none" )
   else:
      newAf = currentAf
   return newAf

def _redistributeConfig( instanceConfig, addrFamily ):
   redistributeConfigDict = { 'both' : instanceConfig.redistributeConfig,
                              'ipv4' : instanceConfig.redistributeConfigV4,
                              'ipv6' : instanceConfig.redistributeConfigV6 }
   return redistributeConfigDict[ addrFamily ]

def haveConflictingRedistribute( mode, config, attr, proto ):
   attrConfigModeMap = { 'redistributeConfig' : 'router isis',
                     'redistributeConfigV4' : 'address-family ipv4',
                     'redistributeConfigV6' : 'address-family ipv6'
                     }
   protoStr = { 'protoBgp' : 'bgp',
                'protoOspf3' : 'ospfv3 match internal',
                'protoOspf3Ase': 'ospfv3 match external',
                'protoOspf3Nssa' : 'ospfv3 match nssa-external',
                'protoIsis' : 'isis' }
   conflict = None

   if attr != 'redistributeConfig':
      if proto in config.redistributeConfig:
         conflict = 'redistributeConfig'
   else:
      if proto in config.redistributeConfigV4:
         conflict = 'redistributeConfigV4'
      elif proto in config.redistributeConfigV6:
         conflict = 'redistributeConfigV6'
   
   if conflict:
      errorMsg = "Cannot configure 'redistribute %s' in mode '%s' " \
          "while it is configured in mode '%s'" \
          % ( protoStr[ proto ], attrConfigModeMap[ attr ],
              attrConfigModeMap[ conflict ] )
      mode.addError( errorMsg )
      return True
   return False

afRedistConfigAttrMap = { 'ipv4' : 'redistributeConfigV4',
                          'ipv6' : 'redistributeConfigV6',
                          'both' : 'redistributeConfig'
                          }

def haveConflictMicroLoopProtected( config, addrFamily ):
   msg = None
   if addrFamily != 'both':
      if config.microLoopPreventionProtectedConfig:
         msg = 'both'
   else:
      if config.microLoopPreventionProtectedConfigV4:
         msg = 'ipv4'
      elif config.microLoopPreventionProtectedConfigV6:
         msg = 'ipv6'
   return msg

def haveConflictingMicroLoopConfig( mode, config, addrFamily ):
   cmdStr = 'timers local-convergence-delay protected-prefixes'
   addrFamilyToMode = { 'ipv4' : 'address-family ipv4',
                        'ipv6' : 'address-family ipv6',
                        'both' : 'router isis' }
   conflictMsg = haveConflictMicroLoopProtected( config, addrFamily )
   if conflictMsg:
      errorMsg = "Cannot configure '%s' in mode '%s' " \
                 "while it is configured in mode '%s'" \
                 % ( cmdStr, addrFamilyToMode[ addrFamily ],
                     addrFamilyToMode[ conflictMsg ] )
      mode.addError( errorMsg )
      return True
   return False

def delIsisRedistAf( instanceConfig, addrFamily ):
   redistConfig = _redistributeConfig( instanceConfig, addrFamily )
   redistConfig.clear()

ospfRouteTypeProtoMap = {
      'ospf' : {
         'internal' : 'protoOspf',
         'external' : 'protoOspfAse',
         'nssa-external' : 'protoOspfNssa'
      },
      'ospfv3' : {
         'internal' : 'protoOspf3',
         'external' : 'protoOspf3Ase',
         'nssa-external' : 'protoOspf3Nssa'
      },
}

def setRedistributeCommon( mode, instanceName, proto, addrFamily='both',
      routeType=None, mapName=None, includeLeaked=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   attrStr = afRedistConfigAttrMap[ addrFamily ]

   if proto in ospfRouteTypeProtoMap:
      proto = ospfRouteTypeProtoMap[ proto ][ routeType ]

   if haveConflictingRedistribute( mode, instanceConfig, attrStr, proto ):
      return 
   redistributeConfig = _redistributeConfig( instanceConfig, addrFamily )

   if mapName:
      redistribute = Redistribute( proto, routeMap=mapName,
                                   includeLeaked=includeLeaked )
   else:
      redistribute = Redistribute( proto, includeLeaked=includeLeaked )

   if proto in redistributeConfig:
      oldRedistCfg = redistributeConfig[ proto ]
      if redistribute == oldRedistCfg:
         # Config hasn't changed, nothing to do
         return
      del redistributeConfig[ proto ]

   redistributeConfig.addMember( redistribute )

def noRedistributeCommon( instanceName, addrFamily, proto, routeType=None ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   redistributeConfig = _redistributeConfig( instanceConfig, addrFamily )
   if proto in ospfRouteTypeProtoMap:
      proto = ospfRouteTypeProtoMap[ proto ][ routeType ]
   del redistributeConfig[ proto ]

def setMicroLoopProtectedCommon( mode, instanceName, addrFamily='both',
                                 negate=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   enable = not negate
   if enable and haveConflictingMicroLoopConfig(
         mode, instanceConfig, addrFamily ):
      return
   if addrFamily == 'both':
      instanceConfig.microLoopPreventionProtectedConfig = enable
   elif addrFamily == 'ipv4':
      instanceConfig.microLoopPreventionProtectedConfigV4 = enable
   else:
      instanceConfig.microLoopPreventionProtectedConfigV6 = enable

def setMicroLoopConvergenceDelayCommon( mode, instanceName, delay, addrFamily='both',
                                        negate=False ):
   instanceConfig = isisConfig.instanceConfig[ instanceName ]
   if negate:
      delay = instanceConfig.microLoopConvergenceDelayDefault

   if addrFamily == 'both':
      instanceConfig.microLoopConvergenceDelay = delay
   elif addrFamily == 'ipv4':
      instanceConfig.microLoopConvergenceDelayV4 = delay
   else:
      instanceConfig.microLoopConvergenceDelayV6 = delay

def getAuthModeData( args ):
   authMode = args.get( 'sha' ) or args.get( 'md5' ) or args.get( 'text' )
   authRxDisabled = args.get( 'rx-disabled' )
   authLevel = args.get( 'LEVEL' )
   keyid = AuthKeyid().idDefault
   authModeVal = None
   if authMode == 'sha':
      authModeVal = AuthMode.sha
      keyid = args[ 'KEYID' ]
   elif authMode == 'md5':
      authModeVal = AuthMode.md5
   elif authMode == 'text':
      authModeVal = AuthMode.clearText
   return authModeVal, keyid, authRxDisabled, authLevel

def getAuthKeyData( args ):
   authKeyId = args.get( 'ID' )
   authLevel = args.get( 'LEVEL' )
   algorithm = args.get( 'SHA' )
   shaApadRfc5310 = 'rfc-5310' in args
   authKey = {}
   if 'PASS' in args:
      authKey[ 'PASS' ] = args[ 'PASS' ]
   if 'EPASS' in args:
      authKey[ 'EPASS' ] = args[ 'EPASS' ]
   return authKeyId, authLevel, algorithm, authKey, shaApadRfc5310

def setAuthModeCommon( configHandle, authLevel, authModeVal, keyid,
                       authRxDisabled ):
   if authRxDisabled is not None:
      authRxDisabled = True
   else:
      authRxDisabled = configHandle.authRxDisabledDefault

   if authLevel != 'level-2':
      configHandle.authModeL1 = authModeVal
      configHandle.authModeKeyidL1 = keyid
      configHandle.authRxDisabledL1 = authRxDisabled
   if authLevel != 'level-1':
      configHandle.authModeL2 = authModeVal
      configHandle.authModeKeyidL2 = keyid
      configHandle.authRxDisabledL2 = authRxDisabled

def noAuthModeCommon( configHandle, authLevel, authModeVal, keyid ):
   authRxDisabledDefault = configHandle.authRxDisabledDefault
   authModeNone = AuthMode.noAuth
   idDefault = AuthKeyid().idDefault
   if authModeVal == None and authLevel == None:
      configHandle.authModeL1 = authModeNone
      configHandle.authModeKeyidL1 = idDefault
      configHandle.authRxDisabledL1 = authRxDisabledDefault
      configHandle.authModeL2 = authModeNone
      configHandle.authModeKeyidL2 = idDefault
      configHandle.authRxDisabledL2 = authRxDisabledDefault
      return
   if authLevel == None:
      if ( configHandle.authModeL1 == authModeVal ) and \
         ( configHandle.authModeKeyidL1 == keyid ):
         configHandle.authModeL1 = authModeNone
         configHandle.authModeKeyidL1 = idDefault
         configHandle.authRxDisabledL1 = authRxDisabledDefault
      if ( configHandle.authModeL2 == authModeVal ) and \
         ( configHandle.authModeKeyidL2 == keyid ):
         configHandle.authModeL2 = authModeNone
         configHandle.authModeKeyidL2 = idDefault
         configHandle.authRxDisabledL2 = authRxDisabledDefault
      return
   if ( configHandle.authModeL1 == authModeVal ) and \
      ( configHandle.authModeKeyidL1 == keyid ) and \
      ( authLevel == 'level-1' ):
      configHandle.authModeL1 = authModeNone
      configHandle.authModeKeyidL1 = idDefault
      configHandle.authRxDisabledL1 = authRxDisabledDefault
   if ( configHandle.authModeL2 == authModeVal ) and \
      ( configHandle.authModeKeyidL2 == keyid ) and \
      ( authLevel == 'level-2' ):
      configHandle.authModeL2 = authModeNone
      configHandle.authModeKeyidL2 = idDefault
      configHandle.authRxDisabledL2 = authRxDisabledDefault

def setAuthKeyCommon( mode, configHandle, authLevel, authKey, keyid, algorithm,
                      shaApadRfc5310 ):
   # process all levels applicable
   testAuthLevel = [ authLevel ] if authLevel else [ 'level-1', 'level-2' ]
   passwd = {}
   for level in testAuthLevel:
      # decrypt password
      if 'PASS' in authKey:
         passwd[ level ] = authKey[ 'PASS' ]
      else:
         postfix = str( configHandle.authModeL1 ) if level == 'level-1' else \
                   str( configHandle.authModeL2 )
         key = configHandle.instanceName + '_' + postfix
         try:
            # We need a try-except clause here, as the encrypted password specified
            # needs to be base64 encoded for decryption to succeed.
            passwd[ level ] = decodeKey( authKey[ 'EPASS' ],
                                key=key,
                                algorithm='MD5' )
         except Exception as e:
            error = "%s" % e
            mode.addError( error.capitalize() )
            return

   idDefault = AuthKeyid()
   if keyid is None:
      keyid = idDefault
   if keyid in configHandle.authKeyColl:
      authKeyEntry = Tac.nonConst( configHandle.authKeyColl[ keyid ] )
   else:
      authKeyEntry = AuthKey( keyid )
   if algorithm:
      authKeyEntry.algorithm = SHA_KW_TO_TAC_TYPE_MAP[ algorithm ]
   # save password for authentication level if all decryptions worked
   for level in testAuthLevel:
      if level == 'level-1':
         authKeyEntry.authKeyL1 = passwd[ level ]
         authKeyEntry.shaApadRfc5310L1 = shaApadRfc5310
      elif level == 'level-2':
         authKeyEntry.authKeyL2 = passwd[ level ]
         authKeyEntry.shaApadRfc5310L2 = shaApadRfc5310
   configHandle.authKeyColl.addMember( authKeyEntry )

def noAuthKeyCommon( configHandle, authLevel, keyid ):
   if keyid is None:
      keyid = AuthKeyid()
   if keyid not in configHandle.authKeyColl:
      return
   authKeyEntry = Tac.nonConst( configHandle.authKeyColl[ keyid ] )
   # pylint: disable-msg=maybe-no-member
   if authLevel != 'level-2':
      authKeyEntry.authKeyL1 = authKeyEntry.authKeyInvalid
   if authLevel != 'level-1':
      authKeyEntry.authKeyL2 = authKeyEntry.authKeyInvalid

   if authKeyEntry.authKeyL1 == authKeyEntry.authKeyInvalid and \
      authKeyEntry.authKeyL2 == authKeyEntry.authKeyInvalid:
      del configHandle.authKeyColl[keyid]
   else:
      configHandle.authKeyColl.addMember( authKeyEntry )

#-------------------------------------------------------------------------------
# router isis config mode. A new instance of this mode is created when the
# user enters "router isis".
#-------------------------------------------------------------------------------
class RouterIsisMode( RoutingIsisMode, BasicCli.ConfigModeBase ):

   # Attributes required of every Mode class.
   name = 'ISIS instance configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, instanceConfig ):
      RoutingIsisMode.__init__( self, ( instanceConfig.instanceName,
                                        instanceConfig.instanceId,
                                        instanceConfig.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.instanceName = instanceConfig.instanceName
      self.instanceId = instanceConfig.instanceId
      self.vrfName = instanceConfig.vrfName

   def setNet( self, args ):
      net = re.match( netRegex, args[ 'NET' ] )
      ( areaId, systemId ) = net.group( 'areaid', 'systemid' )
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if systemId == instanceConfig.systemIdInvalid:
         self.addError( "System Id '%s' is invalid." % systemId )
         return
      instanceConfig.areaId = areaId
      instanceConfig.systemId = systemId

   def noNet( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.areaId = instanceConfig.areaIdInvalid
      instanceConfig.systemId = instanceConfig.systemIdInvalid


   def setIsType( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

   def noIsType( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.level = instanceConfig.levelDefault

   def gotoRouterIsisAfMode( self, args ):
      """Handler for 'address-family AF [unicast]"""
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      addrFamily = args[ 'AF' ]
      if addrFamily == "ipv6":
         IraIp6Cli.warnIfRoutingDisabled( self, instanceConfig.vrfName )
         if not routing6HardwareStatus.routingSupported:
            self.addWarning( "IPv6 Hardware forwarding is not "
                             "supported on this platform." )
            self.addWarning( "This means that all IPv6 traffic will be "
                             "routed in software." )
      elif addrFamily == "ipv4":
         IraIpCli.warnIfRoutingDisabled( self, instanceConfig.vrfName )

      instanceConfig.addressFamily = addAddrFamilyToConfig( 
            instanceConfig.addressFamily, addrFamily )
      childMode = self.childMode( RouterIsisAfMode, addrFamily=addrFamily )
      self.session_.gotoChildMode( childMode )

   def delRouterIsisAfMode( self, args ):
      """Handler for '[no|default] address-family AF [unicast]"""
      addrFamily = args[ 'AF' ]
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      childMode = self.childMode( RouterIsisAfMode, addrFamily=addrFamily )
      childMode.clearConfig()
      instanceConfig.addressFamily = delAddrFamilyFromConfig( 
            instanceConfig.addressFamily, addrFamily )

   def getOrCreateMetricProfileConfig( self, name ):
      profile, instanceName = findMetricProfileConfig( name )
      if instanceName and instanceName != self.instanceName:
         if self.session_.isInteractive():
            #TODO: Confirm that we should print error msg here
            self.addError( "Metric profile %s is already defined in %s" \
               % ( name, instanceName ) )
         return None
      if not profile:
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         profile = instanceConfig.metricProfiles.newMember( name )
      return profile

   def gotoIsisMetricProfileMode( self, args ):
      """Handler for 'metric profile <profileName>' command"""
      profileName = args[ 'PROFILE_NAME' ]
      metricProfileConfig = self.getOrCreateMetricProfileConfig( profileName )

      if not metricProfileConfig:
         return

      childMode = self.childMode( RouterIsisMetricProfileMode,
                                    metricProfileName=profileName,
                                    metricProfileConfig=metricProfileConfig )
      self.session_.gotoChildMode( childMode )

   def delIsisMetricProfileMode( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      del instanceConfig.metricProfiles[ args[ 'PROFILE_NAME' ] ]

   def gotoRouterIsisSrMplsMode( self, args ):
      """Handler for 'segment-routing mpls'"""
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srDataPlane = SrDataPlane.srDataPlaneMpls
      childMode = self.childMode( RouterIsisSrMplsMode )
      self.session_.gotoChildMode( childMode )

   def delRouterIsisSrMplsMode( self, args ):
      """Handler for [no|default] segment-routing mpls """
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      # Set all segment routing related attributes to default
      instanceConfig.srDataPlane = SrDataPlane.srDataPlaneNone
      instanceConfig.srEnabled = instanceConfig.srEnabledDefault
      instanceConfig.srSegmentPathVerification = \
            instanceConfig.srSegmentPathVerificationDefault
      instanceConfig.srAdjSegmentSidReuseTimeout = \
                                 instanceConfig.srAdjSegmentSidReuseTimeoutDefault
      instanceConfig.srAdjSegmentAlloc = instanceConfig.srAdjSegmentAllocDefault
      instanceConfig.srRouterId = instanceConfig.routerIdV4Default
      instanceConfig.srPrefixSegments.clear()


   def gotoRouterIsisTeMode( self, args ):
      """Handler for 'traffic-engineering'"""
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeMode = True
      childMode = self.childMode( RouterIsisTeMode )
      self.session_.gotoChildMode( childMode )

   def delRouterIsisTeMode( self, args ):
      """Handler for [no|default] traffic-engineering """
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeMode = instanceConfig.isisTeModeDefault
      childMode = self.childMode( RouterIsisTeMode )
      childMode.clearConfig()

   def gotoRouterIsisAreaProxyMode( self, args ):
      '''
      Handler for 'area proxy'
      '''
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisAreaProxyMode = True
      childMode = self.childMode( RouterIsisAreaProxyMode )
      self.session_.gotoChildMode( childMode )

   def delRouterIsisAreaProxyMode( self, args ):
      '''
      Handler for [ no | default ] area proxy
      '''
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisAreaProxyMode = instanceConfig.isisAreaProxyModeDefault
      childMode = self.childMode( RouterIsisAreaProxyMode )
      childMode.clearConfig()

   def setRedistribute( self, proto, routeType=None, mapName=None,
                        includeLeaked=False ):
      setRedistributeCommon( self, self.instanceName, proto, addrFamily='both',
                             routeType=routeType, mapName=mapName,
                             includeLeaked=includeLeaked )

   def noRedistribute( self, proto, routeType=None ):
      noRedistributeCommon( self.instanceName, 'both', proto, routeType=routeType )

   def setMicroLoopConvergenceDelay( self, args ):
      """
      Handler for [no|default] timers local-convergence-delay protected-prefixes
      """
      setMicroLoopProtectedCommon( self, self.instanceName, addrFamily='both' )
      if 'delay' in args:
         delay = args.get( 'delay' )
         setMicroLoopConvergenceDelayCommon( self, self.instanceName,
                                             delay, addrFamily='both' )

   def noMicroLoopConvergenceDelay( self, args ):
      setMicroLoopProtectedCommon( self, self.instanceName,
                                   addrFamily='both', negate=True )
      setMicroLoopConvergenceDelayCommon( self, self.instanceName, None,
                                          addrFamily='both', negate=True )

   def setMplsLdpSync( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.mplsLdpSync = True

   def noSetMplsLdpSync( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.mplsLdpSync = instanceConfig.mplsLdpSyncDefault

   def setMaxLSPSize( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.maxLSPSize = args[ 'MAX_LSP_SIZE' ]

   def noMaxLSPSize( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.maxLSPSize = instanceConfig.maxLSPSizeDefault
            
   def setMaxLSPLifetime( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.maxLSPLifetime = args[ 'MAX_LIFETIME' ]

   def noMaxLSPLifetime( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.maxLSPLifetime = instanceConfig.maxLSPLifetimeDefault

   def setLSPRefreshInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspRefreshInterval = args[ 'REFRESH_INTERVAL' ]

   def noLSPRefreshInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspRefreshInterval = \
            instanceConfig.lspRefreshIntervalDefault

   def setIsisHostname( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisHostname = args[ 'HOSTNAME' ]

   def noIsisHostname( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisHostname = instanceConfig.isisHostnameInvalid
      
   def setShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.shutdown = True

   def noShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.shutdown = False

   def setLogAdjacencyChanges( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      value = AdjacencyLoggingType.adjacencyLoggingTypeNormal
      instanceConfig.adjacencyLogging = value

   def noLogAdjacencyChanges( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      value = AdjacencyLoggingType.adjacencyLoggingTypeNone
      instanceConfig.adjacencyLogging = value

   # set-overload-bit and set-overload-bit on-startup <delay> commands are
   # mutually exclusive. The setting of one should reset the other. This is
   # similar to its implementation in GateD as well
   def setOverloadBit( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if 'wait-for-bgp' in args:
         instanceConfig.overloadWaitForBgp = True
         instanceConfig.overloadBit = False
         instanceConfig.overloadStartupDelay = args.get( 'DELAY',
             instanceConfig.overloadWaitForBgpTimeoutDefault )
      elif 'DELAY' in args:
         instanceConfig.overloadBit = False
         instanceConfig.overloadStartupDelay = args[ 'DELAY' ]
         instanceConfig.overloadWaitForBgp = False
      else:
         instanceConfig.overloadStartupDelay = \
             instanceConfig.overloadStartupDelayInvalid
         instanceConfig.overloadBit = True
         instanceConfig.overloadWaitForBgp = False

   def noSetOverloadBit( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.overloadBit = False
      instanceConfig.overloadWaitForBgp = False

      instanceConfig.overloadStartupDelay = \
          instanceConfig.overloadStartupDelayInvalid

   def setGracefulRestart( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.gracefulRestart = True

   def noGracefulRestart( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.gracefulRestart = False

   def setGrHelper( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      no = False
      if CliCommand.isNoCmd( args ):
         no = True
      instanceConfig.grHelper = not no

   def setGrTimerT2( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      t2 = args.get( 'SECONDS', instanceConfig.grTimerT2Default )
      level = args.get( 'LEVEL' )
      if level != 'level-2':
         instanceConfig.grTimerT2L1 = t2
      if level != 'level-1':
         instanceConfig.grTimerT2L2 = t2

   def setGrHoldTime( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      seconds = args.get( 'SECONDS', instanceConfig.grHoldTimeDefault )
      instanceConfig.grHoldTime = seconds

   def setPublishLsdb( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.publishLsdb = True

   def noPublishLsdb( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.publishLsdb = False

   def advertiseHighMetric( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.advertiseHighMetric = True

   def noAdvertiseHighMetric( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.advertiseHighMetric = False

   def setAdvertisePassiveOnly( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.advertisePassiveOnly = True

   def noSetAdvertisePassiveOnly( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.advertisePassiveOnly = False

   def setSpfInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      maxWaitSpfInterval = args.get( 'MAX_WAIT' )
      holdSpfInterval = args.get( 'HOLD_TIME', instanceConfig.spfHoldIntDefault )
      startSpfInterval = args.get( 'INITIAL_WAIT',
                                   instanceConfig.spfStartIntDefault )
      
      if ( startSpfInterval > maxWaitSpfInterval * 1000 ):
         self.addError( "SPF initial wait interval cannot exceed max wait interval" )
         return
      if ( holdSpfInterval > maxWaitSpfInterval * 1000 ):
         self.addError( "SPF hold interval cannot exceed max wait interval" )
         return

      instanceConfig.spfInterval = SpfThrottleTimer(
                       startSpfInterval, holdSpfInterval, maxWaitSpfInterval )

   def noSpfInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.spfInterval = instanceConfig.spfIntervalDefault

   def setSpfPartial( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.spfFlagsPrcEnabled = True

   def noSpfPartial( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.spfFlagsPrcEnabled = False

   def setAuthMode( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      authModeVal, keyid, authRxDisabled, authLevel = getAuthModeData( args )
      setAuthModeCommon( instanceConfig, authLevel, authModeVal, keyid,
                         authRxDisabled )

   def noAuthMode( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      authModeVal, keyid, _, authLevel = getAuthModeData( args )
      noAuthModeCommon( instanceConfig, authLevel, authModeVal, keyid )

   def setAuthKey( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      keyid, authLevel, algorithm, authKey, shaApadRfc5310 = getAuthKeyData( args )
      setAuthKeyCommon( self, instanceConfig, authLevel, authKey, keyid, algorithm,
                        shaApadRfc5310 )

   def noAuthKey( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      keyid, authLevel, _, _, _ = getAuthKeyData( args )
      noAuthKeyCommon( instanceConfig, authLevel, keyid )

   def setAttachedBit( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.attachBitRouteMap = args[ 'RM_NAME' ]

   def noSetAttachedBit( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.attachBitRouteMap = instanceConfig.attachBitRouteMapInvalid

   def setHelloPadding( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if args.get( 'disabled' ):
         instanceConfig.helloPadding = HelloPadding.helloPaddingAlwaysOff
      else:
         instanceConfig.helloPadding = HelloPadding.helloPaddingOn

   def noHelloPadding( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.helloPadding = HelloPadding.helloPaddingTillAdjUp

   def setLspGenInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      maxWaitLspGenInterval = args[ 'MAX_WAIT' ]
      holdLspGenInterval = args.get( 'HOLD_WAIT', 
            instanceConfig.lspGenHoldIntDefault )
      startLspGenInterval = args.get( 'INITIAL_WAIT', 
            instanceConfig.lspGenStartIntDefault )
      
      if ( startLspGenInterval > maxWaitLspGenInterval * 1000 ):
         self.addError( "LSP initial wait interval cannot exceed max wait interval" )
         return
      if ( holdLspGenInterval > maxWaitLspGenInterval * 1000 ):
         self.addError( "LSP hold interval cannot exceed max wait interval" )
         return

      instanceConfig.lspGenInterval = LspGenThrottleTimer(
            startLspGenInterval, holdLspGenInterval, maxWaitLspGenInterval )

   def noLspGenInterval( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspGenInterval = instanceConfig.lspGenIntervalDefault

   def setOutDelayTimer( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspOutDelayTimerConfig = args[ 'OUT_DELAY' ]
        
   def noOutDelayTimer( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspOutDelayTimerConfig = instanceConfig.lspOutDelayTimerDefault

   def setMinLspRemainingLifetime( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.minLspRemainingLifetime = args.get( 'MIN_REMAINING_LIFETIME',
            instanceConfig.minLspRemainingLifetimeDefault )

   def noMinLspRemainingLifetime( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.minLspRemainingLifetime = \
            instanceConfig.minLspRemainingLifetimeDisabled

   def setRouterIdV4( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      routerIdV4 = args[ 'ROUTER_IDV4' ]
      if routerIdV4 == instanceConfig.routerIdV4Default:
         self.addError( "%s is not a valid router ID" % routerIdV4 )
         return
      instanceConfig.routerIdV4 = routerIdV4

   def noRouterIdV4( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.routerIdV4 = instanceConfig.routerIdV4Default

   def setRouterIdV6( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      routerIdV6 = args[ 'ROUTER_IDV6' ]
      if routerIdV6 == instanceConfig.routerIdV6Default:
         self.addError( "%s is not a valid router ID" % routerIdV6 )
         return
      instanceConfig.routerIdV6 = routerIdV6

   def noRouterIdV6( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.routerIdV6 = instanceConfig.routerIdV6Default

   def setPurgeOrigination( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      level = args.get( 'LEVEL' )
      instanceConfig.poiEnabledL1 = level != 'level-2'
      instanceConfig.poiEnabledL2 = level != 'level-1'

   def noPurgeOrigination( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.poiEnabledL1 = False
      instanceConfig.poiEnabledL2 = False

   def setAreaLeaderPriority( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      priority = args.get( 'PRIORITY', instanceConfig.areaLeaderPriorityDefault )
      level = args.get( 'LEVEL' )
      applyToL1 = level != 'level-2'
      applyToL2 = level != 'level-1'
      if 'disabled' in args:
         if applyToL1:
            instanceConfig.areaLeaderEnabledL1 = False
         if applyToL2:
            instanceConfig.areaLeaderEnabledL2 = False
      else:
         if applyToL1:
            instanceConfig.areaLeaderEnabledL1 = True
            instanceConfig.areaLeaderPriorityL1 = priority
         if applyToL2:
            instanceConfig.areaLeaderEnabledL2 = True
            instanceConfig.areaLeaderPriorityL2 = priority

   def defaultAreaLeaderPriority( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      level = args.get( 'LEVEL' )
      default = instanceConfig.areaLeaderPriorityDefault
      if level != 'level-2':
         instanceConfig.areaLeaderEnabledL1 = True
         instanceConfig.areaLeaderPriorityL1 = default
      if level != 'level-1':
         instanceConfig.areaLeaderEnabledL2 = True
         instanceConfig.areaLeaderPriorityL2 = default

   def dynamicFlooding( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      level = args.get( 'LEVEL' )
      if level != 'level-2':
         instanceConfig.dynamicFloodingL1 = True
      if level != 'level-1':
         instanceConfig.dynamicFloodingL2 = True

   def noDynamicFlooding( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      level = args.get( 'LEVEL' )
      if level != 'level-2':
         instanceConfig.dynamicFloodingL1 = False
      if level != 'level-1':
         instanceConfig.dynamicFloodingL2 = False

   def setRoutePreference( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.routePrefRfc7775 = True

   def noRoutePreference( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.routePrefRfc7775 = False

   def setIsisLspRfc8202RxDisabled( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspRfc8202RxDisabled = True

   def noIsisLspRfc8202RxDisabled( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.lspRfc8202RxDisabled = False

   def setIsisRfc8202Disabled( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.rfc8202Disabled = True

   def noIsisRfc8202Disabled( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.rfc8202Disabled = False

#-------------------------------------------------------------------------------
# ISIS area proxy config mode. A new instance of this mode is created when the
# user enters "area proxy".
#-------------------------------------------------------------------------------
class RouterIsisAreaProxyMode( RoutingIsisAreaProxyMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'ISIS area proxy configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisAreaProxyMode.__init__( self, ( parent.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def shutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisAreaProxyEnabled = False

   def noShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisAreaProxyEnabled = True

   def setNet( self, args ):
      net = re.match( netRegex, args[ 'NET' ] )
      ( areaId, systemId ) = net.group( 'areaid', 'systemid' )
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if systemId == instanceConfig.systemIdInvalid:
         self.addError( "System Id '%s' is invalid." % systemId )
         return
      if instanceConfig.systemIdInvalid == instanceConfig.systemId:
         self.addError( "The primary system id must be configured first." )
         return
      if systemId == instanceConfig.systemId:
         self.addError( "Proxy system id cannot be the same as primary system id." )
         return
      instanceConfig.proxyAreaId = areaId
      instanceConfig.proxySystemId = systemId

   def noNet( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.proxyAreaId = instanceConfig.areaIdInvalid
      instanceConfig.proxySystemId = instanceConfig.systemIdInvalid

   def setHostname( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.proxyHostname = args[ 'HOSTNAME' ]

   def noHostname( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.proxyHostname = instanceConfig.isisHostnameInvalid

   def setAreaSegment( self, args ):
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      labelType = args.get( 'label', 'index' )
      isValue = labelType == 'label'
      value = args.get( 'LABEL' ) or args.get( 'INDEX' )
      sid = SegmentIdentifier( value, isValue )
      if not prefix.isHost:
         self.addError( "An area segment can be configured for a /32 IPv4"
                        " prefix or /128 IPv6 prefix only" )
         return
      if _isisSrConflictingPrefixFound( self.instanceName, None, prefix, mode=self ):
         self.addError( "area segment conflicts with a prefix/node segment"
                        " configured for the prefix %s" % prefix )
         return
      if  _isisSrConflictingSidFound( self.instanceName, None, prefix, value,
                                      isValue=isValue, mode=self ):
         self.addError( "Two prefixes cannot be configured with the same SID" )
         return

      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if prefix in instanceConfig.areaSegment:
         instanceConfig.areaSegment[ prefix ] = sid
      else:
         # Allow only one (prefix, area SID) per address family. The new prefix
         # overwrites the existing one in the same address family.
         for p in instanceConfig.areaSegment:
            if p.af == prefix.af:
               del instanceConfig.areaSegment[ p ]
               break
         instanceConfig.areaSegment[ prefix ] = sid

   def noAreaSegment( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      del instanceConfig.areaSegment[ prefix ]

   def setRouterId( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if 'ipv4' in args:
         routerIdV4 = args[ 'ROUTER_IDV4' ]
         if routerIdV4 == instanceConfig.routerIdV4Default:
            self.addError( "%s is not a valid router ID" % routerIdV4 )
            return
         instanceConfig.proxyRouterIdV4 = routerIdV4
      elif 'ipv6' in args:
         routerIdV6 = args[ 'ROUTER_IDV6' ]
         if routerIdV6 == instanceConfig.routerIdV6Default:
            self.addError( "%s is not a valid router ID" % routerIdV6 )
            return
         instanceConfig.proxyRouterIdV6 = routerIdV6

   def noRouterId( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if 'ipv4' in args:
         instanceConfig.proxyRouterIdV4 = instanceConfig.routerIdV4Default
      elif 'ipv6' in args:
         instanceConfig.proxyRouterIdV6 = instanceConfig.routerIdV6Default

   def clearConfig( self ):
      self.shutdown( None )
      self.noNet( None )
      self.noHostname( None )
      self.noRouterId( { 'ipv4' : 'ipv4' } )
      self.noRouterId( { 'ipv6' : 'ipv6' } )

#-------------------------------------------------------------------------------
# ISIS-TE config mode. A new instance of this mode is created when the
# user enters "traffic-engineering".
#-------------------------------------------------------------------------------
class RouterIsisTeMode( RoutingIsisTeMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'ISIS traffic-engineering configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisTeMode.__init__( self, ( parent.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def shutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeEnabled = instanceConfig.isisTeEnabledDefault

   def noShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeEnabled = True

   def setIsType( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeLevel = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

   def noIsType( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.isisTeLevel = instanceConfig.isisTeLevelDefault

   def clearConfig( self ):
      self.shutdown( None )
      self.noIsType( None )

#-------------------------------------------------------------------------------
# "[no|default] router isis <instance-name> [vrf <vrf-name>]" command,
# in "config" mode.
# -------------------------------------------------------------------------------
MAX_ISIS_INSTANCES_PER_VRF = 1

def isisInstanceByVrfName( instanceConfig ):
   instDict = {}
   for i in instanceConfig.values():
      instDict.setdefault( i.vrfName, [] ).append( i.instanceName )
   return instDict

def isisInstanceNameById( instanceConfig, vrf=DEFAULT_VRF ):
   instDict = {}
   for  i in instanceConfig.values():
      if i.vrfName == vrf:
         instDict[ i.instanceId ] = i.instanceName
   return instDict

def anyInstanceInVrf( instanceConfig, vrf=DEFAULT_VRF ):
   return any( [ i for i in instanceConfig.values() if  i.vrfName == vrf ] )

def getInstanceConfig( mode, instanceName, instanceId, vrfName ):
   """Get ISIS Instance config object for supplied instanceName and
   vrfName. Create it if it doesn't exist already"""
   instanceId = 0 if instanceId is None else instanceId
   instanceConfig = isisConfig.instanceConfig.get( instanceName )

   # Existing instance config validations.
   if instanceConfig:
      if vrfName and instanceConfig.vrfName != vrfName:
         mode.addError( 'IS-IS instance %s is already configured in %s VRF' % \
                           ( instanceName, instanceConfig.vrfName ) )
         return None
      if instanceConfig.instanceId != instanceId:
         mode.addError( 'IS-IS instance %s is already configured '
                        'with instance ID %d' % ( instanceName,
                                                  instanceConfig.instanceId ) )
         return None

      return instanceConfig

   # New instance config validations.
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( 'VRF name %s is reserved' % vrfName )
      return None

   vrfName = DEFAULT_VRF if vrfName is None else vrfName
   instanceByVrfName = isisInstanceByVrfName( isisConfig.instanceConfig )

   if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
      if vrfName == DEFAULT_VRF:
         instanceNameById = isisInstanceNameById( isisConfig.instanceConfig )
         if instanceId in instanceNameById:
            if instanceId == 0:
               mode.addError( 'More than %d IS-IS instance in default VRF '
                              'without instance ID is not supported' % \
                              MAX_ISIS_INSTANCES_PER_VRF )
               return None
            else:
               mode.addError( 'IS-IS instance %s is already configured with '
                              'instance ID %d' % ( instanceNameById[ instanceId ],
                                                   instanceId ) )
               return None
      else:
         if vrfName in instanceByVrfName:
            mode.addError( 'More than %d IS-IS instance in non-default VRF '
                           'is not supported' % MAX_ISIS_INSTANCES_PER_VRF )
            return None
   else:
      if vrfName in instanceByVrfName:
         mode.addError( 'More than %d IS-IS instance per VRF is not supported' % \
                        MAX_ISIS_INSTANCES_PER_VRF )
         return None

   if vrfName != DEFAULT_VRF:
      maxVrfs = routingHardwareStatus.vrfCapability.maxVrfs
      instanceByVrfName.pop( DEFAULT_VRF, None )
      if maxVrfs != -1 and len( instanceByVrfName ) >= maxVrfs:
         mode.addError( 'More than %d IS-IS VRF instances are not supported' % \
                           maxVrfs )
         return None

   IraVrfCli.addAgentVrfEntry( vrfName, "Isis" )
   return isisConfig.instanceConfig.newMember( instanceName, instanceId,
                                               vrfName )

def gotoRouterIsisMode( mode, instanceName, instanceId, vrfName ):
   """Handler for command -
     'router isis <instance-name> [instance-id <instance-id> | vrf <vrf-name>]'"""
   # We do not check if IP routing or IPv6 routing is enabled here.
   # We do it when we enter the address-family ipv4 | ipv6 CLI

   # Request trap resouces if multi-instance
   if instanceId:
      trapConfig.features.add( TrapFeatureName.isisMi )

   # Get the instanceConfig object, create it if not present
   instanceConfig = getInstanceConfig( mode, instanceName, instanceId, vrfName )
   if instanceConfig is None:
      return

   childMode = mode.childMode( RouterIsisMode, instanceConfig=instanceConfig )
   mode.session_.gotoChildMode( childMode )

def anyMultiInstance( instanceConfig ):
   return any( i.instanceId for i in instanceConfig.values() )

def delRouterIsisMode( mode, instanceName ):
   """Handler for 'no|default router isis <instance-name>' command"""
   if instanceName in isisConfig.instanceConfig:
      vrfName = isisConfig.instanceConfig[ instanceName ].vrfName
      del isisConfig.instanceConfig[ instanceName ]
      if not anyInstanceInVrf( isisConfig.instanceConfig, vrfName ):
         IraVrfCli.removeAgentVrfEntry( vrfName, "Isis" )
      # Trap registration is only for isis-mi support and should be removed only 
      # if all of the isis multi-instances are deleted.
      if not anyMultiInstance( isisConfig.instanceConfig ):
         trapConfig.features.remove( TrapFeatureName.isisMi )

def deleteVrfHook( vrfName ):
   instanceByVrfName = isisInstanceByVrfName( isisConfig.instanceConfig )
   if vrfName in instanceByVrfName:
      return ( True, "IS-IS configuration for instance %s has been disabled" % \
                  instanceByVrfName[ vrfName ][ 0 ] )
   else:
      return ( True, None )

def getProtectionMode( protection ):
   protectionMode = None
   if protection == 'disabled' or protection is None:
      protectionMode = ProtectionMode.protectionDisabled
   elif protection == 'link-protection':
      protectionMode = ProtectionMode.linkProtection
   else:
      protectionMode = ProtectionMode.nodeProtection
   return protectionMode

def getSrlgProtection( srlg=False, srlgStrict=False ):
   srlgVal = ProtectionSrlg.srlgDefault
   if srlg == 'disabled':
      srlgVal = ProtectionSrlg.srlgDisabled
   elif srlg == 'strict':
      srlgVal = ProtectionSrlg.srlgStrict
   elif srlg == 'loose':
      srlgVal = ProtectionSrlg.srlgLoose
   return srlgVal

class RouterIsisAfIpv6Modelet( CliParser.Modelet ):
   '''
   This class is used to add commands specific to ipv6 address family
   modes in router isis config mode. 
   '''

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, routerIsisAfMode ):
      CliParser.Modelet.__init__( self )
      self.instanceName = routerIsisAfMode.instanceName
      self.addrFamily = routerIsisAfMode.addrFamily

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.addrFamily == 'ipv6'

   def clearConfig( self ):
      pass

class RouterIsisAfIpv4Modelet( CliParser.Modelet ):
   '''
   This class is used to add commands specific to ipv4 address-family
   mode in router isis config mode. 
   '''

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, routerIsisAfMode ):
      CliParser.Modelet.__init__( self )
      self.instanceName = routerIsisAfMode.instanceName
      self.addrFamily = routerIsisAfMode.addrFamily

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.addrFamily == 'ipv4'

   def clearConfig( self ):
      pass

class RouterIsisAfMode( RoutingIsisAfMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'ISIS address family configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, modifier="unicast" ):
      # parent is an object of RouterIsisMode which inherits from
      # RoutingIsisMode and that has vrfName which we are using below.
      self.instanceName = parent.instanceName
      RoutingIsisAfMode.__init__( self, ( parent.vrfName, addrFamily, modifier,
                                          parent.instanceId ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      return self.addrFamily

   def setMetricAllIntf( self, args ):
      metricVal = args[ 'METRICVALUE' ]
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.metricAllIntfV4 = metricVal
      elif self.addrFamily == 'ipv6':
         instanceConfig.metricAllIntfV6 = metricVal
   
   def noMetricAllIntf( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.metricAllIntfV4 = instanceConfig.metricAllIntfDefault
      elif self.addrFamily == 'ipv6':
         instanceConfig.metricAllIntfV6 = instanceConfig.metricAllIntfDefault
      
   def setMaxEcmp( self, args ):
      maxEcmp = args.get( 'PATHS' )
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.maxEcmpV4 = maxEcmp
      elif self.addrFamily == 'ipv6':
         instanceConfig.maxEcmpV6 = maxEcmp

   def noMaxEcmp( self, args=None ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.maxEcmpV4 = instanceConfig.maxEcmpV4Invalid
      elif self.addrFamily == 'ipv6':
         instanceConfig.maxEcmpV6 = instanceConfig.maxEcmpV6Invalid

   def setRedistribute( self, proto, routeType=None, mapName=None ):
      if proto == 'protoDhcp' and self.addrFamily == 'ipv4':
         self.addError( 'Redistributing DHCP routes not supported in IPv4 '
                              'address family mode' )
      else:
         setRedistributeCommon( self, self.instanceName, proto,
                                addrFamily=self.addrFamily,
                                routeType=routeType, mapName=mapName )

   def noRedistribute( self, proto, routeType=None ):
      noRedistributeCommon( self.instanceName, self.addrFamily, proto,
                            routeType=routeType )

   def setMicroLoopConvergenceDelay( self, args ):
      """
      Handler for [no|default] timers local-convergence-delay protected-prefixes
      """
      setMicroLoopProtectedCommon( self, self.instanceName,
                                   addrFamily=self.addrFamily )
      if 'delay' in args:
         delay = args.get( 'delay' )
         setMicroLoopConvergenceDelayCommon( self, self.instanceName,
                                             delay, addrFamily=self.addrFamily )

   def noMicroLoopConvergenceDelay( self, args ):
      setMicroLoopProtectedCommon( self, self.instanceName,
                                   addrFamily=self.addrFamily, negate=True )
      setMicroLoopConvergenceDelayCommon( self, self.instanceName, None,
                                          addrFamily=self.addrFamily, negate=True )

   def setRouteLeak( self, isLevel=None, mapName=None ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         if isLevel == 'level-1':
            instanceConfig.leakRouteMapL1ToL2V4 = mapName
         elif isLevel == 'level-2':
            instanceConfig.leakRouteMapL2ToL1V4 = mapName
      elif self.addrFamily == 'ipv6':
         if isLevel == 'level-1':
            instanceConfig.leakRouteMapL1ToL2V6 = mapName
         elif isLevel == 'level-2':
            instanceConfig.leakRouteMapL2ToL1V6 = mapName

   def noRouteLeak( self, isLevel=None ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      self.setRouteLeak( isLevel, instanceConfig.leakRouteMapInvalid )

   def setDistance( self, args ):
      """Set the protocol addmnistrative distance.

      :param valueAndLevel: dictionary of distance value and level arguments
      """
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      distanceValue = args[ 'DISTANCE_VALUE' ]
      level = args.get( 'LEVEL' )
      applyToL1 = level != 'level-2'
      applyToL2 = level != 'level-1'
      if self.addrFamily == 'ipv4':
         if applyToL1:
            instanceConfig.distanceV4L1 = distanceValue
         if applyToL2:
            instanceConfig.distanceV4L2 = distanceValue
      elif self.addrFamily == 'ipv6':
         if applyToL1:
            instanceConfig.distanceV6L1 = distanceValue
         if applyToL2:
            instanceConfig.distanceV6L2 = distanceValue

   def noDistance( self, args ):
      """Reset the protocol addmnistrative distance to the default values.

      :param valueAndLevel: dictionary of distance value and level arguments
      """
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      args[ 'DISTANCE_VALUE' ] = instanceConfig.distanceDefault
      self.setDistance( args )

   def clearConfig( self ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      self.noMaxEcmp()
      delIsisRedistAf( instanceConfig, self.addrFamily )
      self.noBfd()
      self.multiTopologyDel()
      for modeletName in self.modeletMap:
         if modeletName != self.__class__:
            self.modeletMap[ modeletName ].clearConfig()

   def setIgnoreAttachedBit( self, args ):
      ignore = True if 'none' in args else False
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.ignoreAttachedBitV4 = ignore
      else:
         instanceConfig.ignoreAttachedBitV6 = ignore

   def noIgnoreAttachedBit( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.ignoreAttachedBitV4 = \
               instanceConfig.ignoreAttachedBitV4Default
      else:
         instanceConfig.ignoreAttachedBitV6 = \
               instanceConfig.ignoreAttachedBitV6Default

   def setProtection( self, protection, protectionLevel ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      protectionMode = getProtectionMode( protection )
      protectionConfig = ProtectionConfig( protectionMode )
      protectionConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ protectionLevel ]

      if self.addrFamily == 'ipv4':
         instanceConfig.protectionConfigAllIntfV4 = protectionConfig
      elif self.addrFamily == 'ipv6':
         instanceConfig.protectionConfigAllIntfV6 = protectionConfig

   def noProtection( self ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.protectionConfigAllIntfV4 \
            = ProtectionConfig( ProtectionMode.protectionDefault )
      elif self.addrFamily == 'ipv6':
         instanceConfig.protectionConfigAllIntfV6 \
            = ProtectionConfig( ProtectionMode.protectionDefault )

   def frrProtection( self, args ):
      protection = args.get( 'PROTECTION' )
      if protection:
         protectionLevel = args.get( 'LEVEL', 'level-1-2' )
         self.setProtection( protection, protectionLevel )
      else:
         self.noProtection()

   def frrSrlgProtection( self, args ):
      srlg = 'strict' if 'strict' in args else 'loose'
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      srlgVal = getSrlgProtection( srlg )
      if self.addrFamily == 'ipv4':
         instanceConfig.protectionSrlgV4 = srlgVal
      elif self.addrFamily == 'ipv6':
         instanceConfig.protectionSrlgV6 = srlgVal

   def noFrrSrlgProtection( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if self.addrFamily == 'ipv4':
         instanceConfig.protectionSrlgV4 = instanceConfig.protectionSrlgDefault
      elif self.addrFamily == 'ipv6':
         instanceConfig.protectionSrlgV6 = instanceConfig.protectionSrlgDefault

   def igpShortcutDisableIs( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.igpShortcutV4 = False

   def igpShortcutDisableDel( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.igpShortcutV4 = True

   def setBfd( self, args ):
      if self.addrFamily == 'ipv4':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.bfdEnabledV4 = True
      elif self.addrFamily == 'ipv6':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.bfdEnabledV6 = True

   def noBfd( self, args=None ):
      if self.addrFamily == 'ipv4':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.bfdEnabledV4 = False
      elif self.addrFamily == 'ipv6':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.bfdEnabledV6 = False

   def multiTopologyIs( self ):
      if self.addrFamily == 'ipv6':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.multiTopology = True

   def multiTopologyDel( self ):
      if self.addrFamily == 'ipv6':
         instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
         instanceConfig.multiTopology = False

class RouterIsisMetricProfileMode( RoutingIsisMetricProfileMode,
                                    BasicCli.ConfigModeBase ):
   name = "IS-IS Metric Profile configuration"
   modeParseTree = CliParser.ModeParseTree()
   unitSpeedtoSppedMapping = { 'mbps' : 1, 'gbps' : 1000 }

   def __init__( self, parent, session, metricProfileName, metricProfileConfig ):
      self.instanceName = parent.instanceName
      self.metricProfileConfig = metricProfileConfig
      RoutingIsisMetricProfileMode.__init__( self, metricProfileName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
   
   def setMetricValue( self, args ):
      self.metricProfileConfig.metric = args.get( 'METRIC_VALUE', 
            self.metricProfileConfig.metricDefault )
   
   def setMetricRatio( self, args ):
      mr = Tac.nonConst( self.metricProfileConfig.metricRatio )
      mr.speed = args[ 'SPEED' ] 
      mr.speedUnit = args[ 'UNITS' ]
      mr.active = mr.speed != 0
      self.metricProfileConfig.metricRatio = mr
  
   def delMetricRatio( self, args ):
      mr = Tac.nonConst( self.metricProfileConfig.metricRatio )
      mr.active = False
      self.metricProfileConfig.metricRatio = mr
   
   def addMetricRule( self, args ):
      metricValue = args[ 'METRIC_VALUE' ]
      speed = args[ 'SPEED' ]
      unit = args[ 'UNITS' ]
      newRule = MetricRule( metricValue, speed, unit )
      self._delMetricRule( newRule.realSpeed )
      if not CliCommand.isNoOrDefaultCmd( args ):
         self.metricProfileConfig.metricRules.addMember( newRule )
   
   def _delMetricRule( self, realSpeed ):
      r = self.metricProfileConfig.metricRules.get( realSpeed )
      if r:
         del self.metricProfileConfig.metricRules[ realSpeed ]

class RouterIsisSrMplsMode( RoutingIsisSrMplsMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'ISIS segment-routing mpls configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      # parent is an object of RouterIsisMode which inherits from RoutingIsisMode
      RoutingIsisSrMplsMode.__init__( self, parent.instanceName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def srShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srEnabled = instanceConfig.srEnabledDefault

   def noSrShutdown( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srEnabled = True

   def setSrSegmentPathVerification( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srSegmentPathVerification = True

   def noSrSegmentPathVerification( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srSegmentPathVerification = \
               instanceConfig.srSegmentPathVerificationDefault

   def setSrRouterId( self, args ):
      # Ideally we should never have the router-id command under the "router isis"/
      # "segment-routing mpls" submode. Users should configure router-id under the
      # "router general" mode or the "router isis" mode. Let them know about it.
      self.addWarning( "Router ID configured under segment-routing mode will be "
                       "overridden by the general or the ISIS router ID" )
      routerId = args[ 'ROUTER_ID' ]
      if routerId == "0.0.0.0":
         self.addError( "%s is not a valid router ID" % routerId )
         return

      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srRouterId = routerId

   def noSrRouterId( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srRouterId = instanceConfig.routerIdV4Default

   def setSrGlobalBlock( self, srBase, srRange ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srSrgbBase = srBase
      instanceConfig.srSrgbRange = srRange

   def noSrGlobalBlock( self ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srSrgbBase = instanceConfig.srSrgbBaseDefault
      instanceConfig.srSrgbRange = instanceConfig.srSrgbRangeDefault

   def adjacencySegmentAllocation( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      srAllocation = SrAdjAllocationType.srAdjacencyAllocationNone
      backup = args.get( 'backup-eligible' )
      if 'sr-peers' in args and backup:
         srAllocation = SrAdjAllocationType.srAdjacencyAllocationSrOnlyBackup
      elif 'sr-peers' in args:
         srAllocation = SrAdjAllocationType.srAdjacencyAllocationSrOnly
      elif 'all-interfaces' in args and backup:
         srAllocation = SrAdjAllocationType.srAdjacencyAllocationAllBackup
      elif 'all-interfaces' in args:
         srAllocation = SrAdjAllocationType.srAdjacencyAllocationAll
      instanceConfig.srAdjSegmentAlloc = srAllocation

   def noAdjacencySegmentAllocation( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srAdjSegmentAlloc = \
          instanceConfig.srAdjSegmentAllocDefault

   def adjacencySegmentSidReuseTimeout( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      if 'infinite' in args:
         instanceConfig.srAdjSegmentSidReuseTimeout = U32_MAX_VALUE
      else:
         instanceConfig.srAdjSegmentSidReuseTimeout = args[ 'TIMEOUT' ]

   def noAdjacencySegmentSidReuseTimeout( self, args ):
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srAdjSegmentSidReuseTimeout = \
                              instanceConfig.srAdjSegmentSidReuseTimeoutDefault

   def setPrefixSegment( self, args ):
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      index = args.get( 'INDEX' )
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      if _isisSrConflictingPrefixFound( self.instanceName, None, prefix, mode=self ):
         self.addError( "prefix segment conflicts with a proxy/node segment"
                        " configured for the prefix %s" % prefix )
         return
      if _isisSrConflictingSidFound( self.instanceName, None, prefix, index,
                                     mode=self ):
         self.addError( "Two prefixes cannot be configured with the same SID" )
         return
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      instanceConfig.srPrefixSegments.addMember( PrefixSegment( prefix, index ) )

   def noPrefixSegment( self, args ):
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      del instanceConfig.srPrefixSegments[ Arnet.IpGenPrefix( prefix.stringValue ) ]

   def setProxyNodeSegment( self, args ):
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      prefix = Arnet.IpGenPrefix( prefix.stringValue )
      index = args[ 'INDEX' ]
      if not prefix.isHost:
         self.addError( "A proxy-node segment can be configured for a /32 IPv4"
                        " prefix or /128 IPv6 prefix only" )
         return
      if _isisSrConflictingPrefixFound( self.instanceName, None, \
            prefix, proxy=True, mode=self ):
         self.addError( "proxy node segment conflicts with a prefix/node segment"
                        " configured for the prefix %s" % prefix )
         return
      if  _isisSrConflictingSidFound( self.instanceName, None, prefix, index,
                                      mode=self ):
         self.addError( "Two prefixes cannot be configured with the same SID" )
         return
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      level = args.get( 'LEVEL', 'level-1-2' )
      segment = PrefixSegment( prefix, index )
      segment.level = LVL_KW_TO_TAC_TYPE_MAP[ level ]
      segment.isProxyNode = True
      instanceConfig.srPrefixSegments.addMember( segment )

   def noProxyNodeSegment( self, args ):
      prefix = args.get( 'IPADDR' ) or args.get( 'IP6ADDR' )
      instanceConfig = isisConfig.instanceConfig[ self.instanceName ]
      del instanceConfig.srPrefixSegments[ Arnet.IpGenPrefix( prefix.stringValue ) ]

#--------------------------------------------------------------------------
# [no|default] router isis <INST> [ instance-id <INST_ID> |vrf <VRFNAME>]
#--------------------------------------------------------------------------
isisKwMatcher = CliMatcher.KeywordMatcher( 'isis',
                     helpdesc="Intermediate System - Intermediate System (IS-IS)" )
instanceNamePattern = '.{1,100}'

instanceNameRule = CliCommand.Node( matcher=CliMatcher.PatternMatcher(
   instanceNamePattern, helpname='NAME',
   helpdesc='Name of the IS-IS protocol instance' ) )
instanceIdKw = CliMatcher.KeywordMatcher( 'instance-id',
                                          helpdesc='IS-IS instance identifier' )
instanceIdMin = 1
instanceIdMax = 65535
instanceIdMatcher = CliMatcher.IntegerMatcher( instanceIdMin, instanceIdMax,
                                               helpdesc='Instance ID' )
class RouterIsisModeCmd( CliCommand.CliCommandClass ):
   if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
      syntax = 'router isis INST [ ( instance-id ID ) | VRF ]'
   else:
      syntax = 'router isis INST [ VRF ]'
   noOrDefaultSyntax = 'router isis INST ...'
   data = {
      'router' : routerKw,
      'isis'   : CliCommand.Node( matcher=isisKwMatcher,
                                  guard=routingSupportedGuard ),
      'INST'   : instanceNameRule,
      'instance-id' : instanceIdKw,
      'ID'          : instanceIdMatcher,
      'VRF' : VrfCli.VrfExprFactory( helpdesc="Configure IS-IS in a VRF instance" ),
   }
   handler = lambda mode, args: gotoRouterIsisMode( mode, args[ 'INST' ],
                                                    args.get( 'ID' ),
                                                    args.get( 'VRF' ) )
   noOrDefaultHandler = lambda mode, args: delRouterIsisMode( mode, args[ 'INST' ] )

BasicCli.GlobalConfigMode.addCommandClass( RouterIsisModeCmd )

# The "[no|default] route preference rfc7775" command
# under 'router isis <>' mode
class IsisRoutePrefRfc7775( CliCommand.CliCommandClass ):
   syntax = 'route preference rfc7775'
   noOrDefaultSyntax = 'route preference ...'
   data = {
            'route' : 'IS-IS route specific configuration',
            'preference' : 'Route preference order mechanism ',
            'rfc7775' : 'Use RFC7775 defined route preference order'
   }
   handler = RouterIsisMode.setRoutePreference
   noOrDefaultHandler = RouterIsisMode.noRoutePreference
RouterIsisMode.addCommandClass( IsisRoutePrefRfc7775 )

#----------------------------------------------------------------------------------
# The "[no|default] lsp rfc8202 rx disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
class IsisLspRfc8202RxDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp rfc8202 rx disabled'
   noOrDefaultSyntax = syntax
   data = {
      'lsp' : lspKw,
      'rfc8202' : CliCommand.Node( matcher=rfc8202Kw,
                                   guard=isisDefaultInstanceGuard ),
      'rx' : 'Configure receive side parameters',
      'disabled' : 'Disable IID-TLV check when receiving LSPs'
   }
   handler = RouterIsisMode.setIsisLspRfc8202RxDisabled
   noOrDefaultHandler = RouterIsisMode.noIsisLspRfc8202RxDisabled

if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   RouterIsisMode.addCommandClass( IsisLspRfc8202RxDisabledCmd )

#----------------------------------------------------------------------------------
# The "[no|default] rfc8202 disabled" command
# under 'router isis <> instance-id <>' mode
#----------------------------------------------------------------------------------
class IsisRfc8202DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'rfc8202 disabled'
   noOrDefaultSyntax = syntax
   data = {
      'rfc8202' : CliCommand.Node( matcher=rfc8202Kw,
                                   guard=isisDefaultInstanceGuard ),
      'disabled' : 'Disabled all rfc8202 functionality'
   }
   handler = RouterIsisMode.setIsisRfc8202Disabled
   noOrDefaultHandler = RouterIsisMode.noIsisRfc8202Disabled

if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   RouterIsisMode.addCommandClass( IsisRfc8202DisabledCmd )

#----------------------------------------------------------------------------------
# The "[no|default] lsp purge origination-identification [ LEVEL ]" command
# under 'router isis <>' mode
#----------------------------------------------------------------------------------
class IsisLspPoiTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp purge origination-identification [ LEVEL ]'
   noOrDefaultSyntax = 'lsp purge origination-identification ...'
   data = {
            'lsp' : lspKw,
            'purge' : 'Configure purge specific parameters',
            'origination-identification' : 'Enable Origination identification',
            'LEVEL' : levelMatcherForConfig,
          }
   handler = RouterIsisMode.setPurgeOrigination
   noOrDefaultHandler = RouterIsisMode.noPurgeOrigination

RouterIsisMode.addCommandClass( IsisLspPoiTypeCmd )

#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under 'router isis <>' mode
#------------------------------------------------------------------------------
class RouterIsisIsTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'is-type LEVEL'
   noOrDefaultSyntax = 'is-type ...'
   data = {
      'is-type' : 'Set level of the instance',
      'LEVEL' : level12MatcherForConfig,
   }
   handler = RouterIsisMode.setIsType
   noOrDefaultHandler = RouterIsisMode.noIsType

RouterIsisMode.addCommandClass( RouterIsisIsTypeCmd )

#-------------------------------------------------------------------------------
# [no | default] address-family AF [ unicast ]
#-------------------------------------------------------------------------------
class RouterIsisAddressFamilyCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family AF [ unicast ]'
   noOrDefaultSyntax = syntax
   data = {
      'address-family' : 'Enable address family and enter its config mode',
      'AF' : afMatcher,
      'unicast' : unicastKw
   }
   handler = RouterIsisMode.gotoRouterIsisAfMode
   noOrDefaultHandler = RouterIsisMode.delRouterIsisAfMode

RouterIsisMode.addCommandClass( RouterIsisAddressFamilyCmd )

#-------------------------------------------------------------------------------
# "[no | default] multi-topology"
# command, in config-router-isis-af mode.
#-------------------------------------------------------------------------------
mtWarningMsg = "Multi Topology is not enabled"
mtV4WarningMsg = "IPv4 address family is not configured in router IS-IS mode"
RouterIsisAfMode.addModelet( RouterIsisAfIpv6Modelet )

multiTopologyKw = CliCommand.guardedKeyword( 'multi-topology',
                                             helpdesc='Enable multi topology',
                                             guard=isisNonDefaultInstanceGuard )

class RouterIsisAfIpv6MtCmd( CliCommand.CliCommandClass ):
   syntax = 'multi-topology'
   noOrDefaultSyntax = syntax
   data = {
      'multi-topology' : multiTopologyKw
   }

   @staticmethod
   def handler( mode, args ):
      mode.multiTopologyIs()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.multiTopologyDel()
RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIpv6MtCmd )

#---------------------------------------------------------------------------------
# [no|default] traffic-engineering
#---------------------------------------------------------------------------------
trafficengineeringKw = CliMatcher.KeywordMatcher( 'traffic-engineering',
                                  helpdesc='Enter traffic engineering config mode' )

class RouterIsisTrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering'
   noOrDefaultSyntax = 'traffic-engineering'
   data = {
      'traffic-engineering' : CliCommand.Node( matcher=trafficengineeringKw,
                                               guard=isisNonDefaultVrfGuard )
   }
   handler = RouterIsisMode.gotoRouterIsisTeMode
   noOrDefaultHandler = RouterIsisMode.delRouterIsisTeMode
RouterIsisMode.addCommandClass( RouterIsisTrafficEngineeringCmd )

#---------------------------------------------------------------------------------
# [no|default] segment-routing mpls
#---------------------------------------------------------------------------------
segmentRoutingKw = CliMatcher.KeywordMatcher( 'segment-routing',
                                       helpdesc='Enter segment routing config mode' )
mplsKw = CliMatcher.KeywordMatcher( 'mpls',
                                    helpdesc='segment routing in MPLS dataplane' )
class RouterIsisSegmentRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing mpls'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing' : segmentRoutingKw,
      'mpls' : CliCommand.Node( matcher=mplsKw, guard=isisNonDefaultVrfGuard )
   }
   handler = RouterIsisMode.gotoRouterIsisSrMplsMode
   noOrDefaultHandler = RouterIsisMode.delRouterIsisSrMplsMode
RouterIsisMode.addCommandClass( RouterIsisSegmentRoutingCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area proxy
#---------------------------------------------------------------------------------
class RouterIsisAreaProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'area proxy'
   noOrDefaultSyntax = syntax
   data = {
      'area' : 'Set area parameters',
      'proxy' : 'Area proxy configuration',
   }
   handler = RouterIsisMode.gotoRouterIsisAreaProxyMode
   noOrDefaultHandler = RouterIsisMode.delRouterIsisAreaProxyMode

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisMode.addCommandClass( RouterIsisAreaProxyCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] shutdown
#---------------------------------------------------------------------------------

class RoutingIsisAreaProxyShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable area proxy'
   }
   handler = RouterIsisAreaProxyMode.shutdown
   noOrDefaultHandler = RouterIsisAreaProxyMode.noShutdown

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisAreaProxyMode.addCommandClass( RoutingIsisAreaProxyShutdownCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] net <NET>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyNetCmd( CliCommand.CliCommandClass ):
   syntax = 'net NET'
   noOrDefaultSyntax = 'net ...'
   data = {
      'net' : 'Configure Network Entity Title (NET) '
              'for the proxy area',
      'NET' : netKw
   }
   handler = RouterIsisAreaProxyMode.setNet
   noOrDefaultHandler = RouterIsisAreaProxyMode.noNet

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyNetCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] is-hostname <HOSTNAME>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'is-hostname HOSTNAME'
   noOrDefaultSyntax = 'is-hostname ...'
   data = {
      'is-hostname' : 'Configure hostname for the proxy area',
      'HOSTNAME' : hostnameKw
   }
   handler = RouterIsisAreaProxyMode.setHostname
   noOrDefaultHandler = RouterIsisAreaProxyMode.noHostname

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyHostnameCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] area segment (ipv4|ipv6) index <value>
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyAreaSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'area segment ( IPADDR | IP6ADDR ) index INDEX'
   noOrDefaultSyntax = 'area segment ( IPADDR | IP6ADDR ) ...'
   data = {
      'area' : 'Configure area parameters',
      'segment' : 'Configure area segment SID for the proxy area',
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexValueMatcher,
   }
   handler = RouterIsisAreaProxyMode.setAreaSegment
   noOrDefaultHandler = RouterIsisAreaProxyMode.noAreaSegment

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyAreaSegmentCmd )

#---------------------------------------------------------------------------------
# Area proxy commands
# [no|default] router-id ipv4 [ROUTER_IDV4]
#---------------------------------------------------------------------------------

class RouterIsisAreaProxyRouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ( ( ipv4 ROUTER_IDV4 ) | ( ipv6 ROUTER_IDV6 ) )'
   noOrDefaultSyntax = 'router-id ( ipv4 | ipv6 ) ... '
   data = {
      'router-id' : routerIdKw,
      'ipv4' : 'IS-IS router ID in IP address format',
      'ROUTER_IDV4' : routerIdV4Matcher,
      'ipv6' : ipv6AddrFormatKw,
      'ROUTER_IDV6' : routerIdV6Matcher
   }
   handler = RouterIsisAreaProxyMode.setRouterId
   noOrDefaultHandler = RouterIsisAreaProxyMode.noRouterId

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   RouterIsisAreaProxyMode.addCommandClass( RouterIsisAreaProxyRouterIdCmd )

#------------------------------------------------------------------------------
# [no|default] redistribute <static|connected|bgp> [include leaked]
#              [route-map <route-map-name>]
#------------------------------------------------------------------------------
PROTO_KW_MAP = {
   'static' : 'protoStatic',
   'connected' : 'protoDirect',
}

protocolMatcher = CliMatcher.EnumMatcher( {
   'static'    : 'Static routes',
   'connected' : 'Connected interface routes',
} )

class RedistributeProtocolCommand( CliCommand.CliCommandClass ):
   if IsisToggleLib.toggleRedistLeakedIntoIsisPhase1Enabled():
      syntax = 'redistribute PROTOCOL [ include leaked ] [ route-map MAPNAME ]'
   else:
      syntax = 'redistribute PROTOCOL [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute PROTOCOL ...'
   data = {
      'redistribute' : redistributeKw,
      'PROTOCOL'     : protocolMatcher,
      'route-map'    : RouteMapMatchers.routeMapApplication,
      'MAPNAME'      : mapNameMatcher,
      # Contingent upon the toggle, but can add anyway.
      'include'      : 'Include leaked routes',
      'leaked'       : 'Include leaked routes',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( PROTO_KW_MAP[ args[ 'PROTOCOL' ] ],
                            mapName=args.get( 'MAPNAME' ),
                            includeLeaked=( 'leaked' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( PROTO_KW_MAP[ args[ 'PROTOCOL' ] ] )

RouterIsisMode.addCommandClass( RedistributeProtocolCommand )

#-----------------------------------------------------------------------------------
# [no|default] redistribute ospf [include leaked]
#              match <internal|external|nssa-exeternal> [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
ospfKw = CliMatcher.KeywordMatcher( 'ospf',
      helpdesc='Routes learned by the OSPF protocol' )
ospfv3Kw = CliMatcher.KeywordMatcher( 'ospfv3',
      helpdesc='Redistribution of OSPFv3 routes' )
matchKw = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Routes learned by the OSPF protocol' )
externalKw = CliMatcher.KeywordMatcher( 'external',
      helpdesc='OSPF routes learned from external sources' )
internalKw = CliMatcher.KeywordMatcher( 'internal',
      helpdesc='OSPF routes learned from internal sources' )
nssaExternalKw = CliMatcher.KeywordMatcher( 'nssa-external',
      helpdesc='Redistribute OSPF nssa - external routes' )
includeKw = CliMatcher.KeywordMatcher( 'include',
      helpdesc='Include leaked routes' )
leakedKw = CliMatcher.KeywordMatcher( 'leaked',
      helpdesc='Include leaked routes' )
matchRoute = 'match ( external | internal | nssa-external )'

redistributeCmdData = {
   'redistribute' : redistributeKw,
   'route-map' : RouteMapMatchers.routeMapApplication,
   'MAPNAME' : mapNameMatcher
}

isisRedistributeCmdData = {
   'isis' : 'IS-IS routes',
   'into' : 'into IS-IS level',
}
isisRedistributeCmdData.update( redistributeCmdData )

class OspfRedistributeCmdData( CliCommand.CliCommandClass ):
   data = {
      'ospfv3' : ospfv3Kw,
      'match' : matchKw,
      'external' : externalKw,
      'internal' : internalKw,
      'nssa-external' : nssaExternalKw,
   }
   data.update( redistributeCmdData )

class RouterIsisModeRedistributionCmd( OspfRedistributeCmdData ):
   if IsisToggleLib.toggleRedistLeakedIntoIsisPhase1Enabled():
      syntax = 'redistribute ( ( ospf [ include leaked ] ) | ospfv3 ) ' + \
            matchRoute + ' [ route-map MAPNAME ]'
      noOrDefaultSyntax = 'redistribute ( ( ospf [ include leaked ] ) | ospfv3 ) ' +\
            matchRoute + '...'
   else:
      syntax = 'redistribute ( ospf | ospfv3 ) ' + matchRoute + \
            ' [ route-map MAPNAME ]'
      noOrDefaultSyntax = 'redistribute ( ospf | ospfv3 ) ' + matchRoute + '...'
   data = { 'ospf' : ospfKw }
   data.update( OspfRedistributeCmdData.data )
   if IsisToggleLib.toggleRedistLeakedIntoIsisPhase1Enabled():
      data.update( { 'include' : includeKw,
                     'leaked' : leakedKw } )

   @staticmethod
   def handler( mode, args ):
      proto, routeType, mapName = getProtoRouteType( args )
      mode.setRedistribute( proto, mapName=mapName, routeType=routeType,
                            includeLeaked='leaked' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proto, routeType, _ = getProtoRouteType( args )
      mode.noRedistribute( proto, routeType=routeType )
RouterIsisMode.addCommandClass( RouterIsisModeRedistributionCmd )

#-----------------------------------------------------------------------------------
# [no|default] redistribute bgp [include leaked] [route-map <route-map-name>]
#-----------------------------------------------------------------------------------
bgpCliCommandNode = CliCommand.Node( matcher=bgpKw,
                                     guard=isisNonDefaultInstanceGuard )

class RedistributeBgpCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute bgp [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute bgp ...'
   data = {
      'bgp'     : bgpCliCommandNode
   }
   data.update( redistributeCmdData )

   @staticmethod
   def handler( mode, args ):
      proto, routeType, mapName = getProtoRouteType( args )
      mode.setRedistribute( proto, mapName=mapName, routeType=routeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proto, routeType, _ = getProtoRouteType( args )
      mode.noRedistribute( proto, routeType=routeType )

RouterIsisMode.addCommandClass( RedistributeBgpCommand )

#-------------------------------------------------------------------------------
# "[no|default] mpls ldp sync default" in 'router isis <>' mode
#-------------------------------------------------------------------------------
defaultKw = CliCommand.guardedKeyword( 'default',
        helpdesc='Set global default',
        guard=isisNonDefaultVrfGuard )

class RouterIsisMplsLDPSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp sync default'
   noOrDefaultSyntax = 'mpls ldp sync default ...'
   data = {
      'mpls' : 'Mpls ldp sync configuration',
      'ldp' : 'LDP configuration',
      'sync' : 'Sync configuration',
      'default' : defaultKw 
   }
   handler = RouterIsisMode.setMplsLdpSync
   noOrDefaultHandler = RouterIsisMode.noSetMplsLdpSync

RouterIsisMode.addCommandClass( RouterIsisMplsLDPSyncCmd )

#-------------------------------------------------------------------------------
# "[no|default] max-lsp-lifetime <time>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
class RouterIsisMaxLSPLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'max-lsp-lifetime MAX_LIFETIME'
   noOrDefaultSyntax = 'max-lsp-lifetime ...'
   data = {
      'max-lsp-lifetime' : 'Set maximum lifetime of LSPs',
      'MAX_LIFETIME' : CliMatcher.IntegerMatcher(
            MaxLSPLifetime.min, MaxLSPLifetime.max,
            helpdesc='LSP lifetime value in seconds' ),
   }
   handler = RouterIsisMode.setMaxLSPLifetime
   noOrDefaultHandler = RouterIsisMode.noMaxLSPLifetime

RouterIsisMode.addCommandClass( RouterIsisMaxLSPLifetimeCmd )

#-------------------------------------------------------------------------------
# "[no|default] lsp size maximum <size>" under 'router isis <>' mode
#-------------------------------------------------------------------------------
class RouterIsisMaxLSPSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp size maximum MAX_LSP_SIZE'
   noOrDefaultSyntax = 'lsp size maximum ...'
   data = {
      'lsp' : lspKw,
      'size' : 'Configure LSP size attribute',
      'maximum' : 'Configure LSP size maximum',
      'MAX_LSP_SIZE' : CliMatcher.IntegerMatcher(
            MaxLSPSize.min, MaxLSPSize.max,
            helpdesc='LSP size value in bytes' ),
   }
   handler = RouterIsisMode.setMaxLSPSize
   noOrDefaultHandler = RouterIsisMode.noMaxLSPSize

if IsisToggleLib.toggleIsisMaxLSPSizeEnabled():
   RouterIsisMode.addCommandClass( RouterIsisMaxLSPSizeCmd )

#---------------------------------------------------------------------------------
# '[no|default] is-hostname <hostname>' under 'router isis <>' mode
#---------------------------------------------------------------------------------
class RouterIsisIsHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'is-hostname HOSTNAME'
   noOrDefaultSyntax = 'is-hostname ...'
   data = {
      'is-hostname' : 'Configure hostname of Intermediate System',
      'HOSTNAME' : hostnameKw
   }
   handler = RouterIsisMode.setIsisHostname
   noOrDefaultHandler = RouterIsisMode.noIsisHostname

RouterIsisMode.addCommandClass( RouterIsisIsHostnameCmd )

# TE CLI
#----------------------------------------------------
#"[no | default] shutdown" command, in isis - te mode
#----------------------------------------------------
class RouterIsisTeShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Traffic Engineering state in IS-IS'
   }
   handler = RouterIsisTeMode.shutdown
   defaultHandler = handler
   noHandler = RouterIsisTeMode.noShutdown

RouterIsisTeMode.addCommandClass( RouterIsisTeShutdownCmd )

#------------------------------------------------------------------------------
# [no|default] is-type LEVEL under isis-te mode
#------------------------------------------------------------------------------
class RouterIsisTeIsTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'is-type LEVEL'
   noOrDefaultSyntax = 'is-type ...'
   data = {
      'is-type' : 'Set level of the instance',
      'LEVEL' : level12MatcherForConfig,
   }
   handler = RouterIsisTeMode.setIsType
   noOrDefaultHandler = RouterIsisTeMode.noIsType

RouterIsisTeMode.addCommandClass( RouterIsisTeIsTypeCmd )

#---------------------------------------------------------------------------------
# [no|default] shutdown
#---------------------------------------------------------------------------------
class RouterIsisShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown'
   data = {
      'shutdown' : 'Disable this IS-IS instance'
   }
   handler = RouterIsisMode.setShutdown
   noOrDefaultHandler = RouterIsisMode.noShutdown
RouterIsisMode.addCommandClass( RouterIsisShutdownCmd )

#---------------------------------------------------------------------------------
# [no|default] log-adjacency-changes
#---------------------------------------------------------------------------------
class RouterIsisLogAdjacencyChangesCmd( CliCommand.CliCommandClass ):
   syntax = 'log-adjacency-changes'
   noOrDefaultSyntax = syntax
   data = {
      'log-adjacency-changes' : 'Enable logging of adjacency events'
   }
   handler = RouterIsisMode.setLogAdjacencyChanges
   noOrDefaultHandler = RouterIsisMode.noLogAdjacencyChanges
RouterIsisMode.addCommandClass( RouterIsisLogAdjacencyChangesCmd )

#---------------------------------------------------------------------------------
# [no|default] hello padding [disabled] command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisHelloPaddingCmd( CliCommand.CliCommandClass ):
   syntax = 'hello padding [ disabled ]'
   noOrDefaultSyntax = 'hello padding ...'
   data = {
      'hello' : 'Configure hello packets',
      'padding' : 'Configure hello padding for all interfaces',
      'disabled' : 'Disable hello padding for all adjacency states'
   }
   handler = RouterIsisMode.setHelloPadding
   defaultHandler = RouterIsisMode.setHelloPadding
   noHandler = RouterIsisMode.noHelloPadding

RouterIsisMode.addCommandClass( RouterIsisHelloPaddingCmd )

#-------------------------------------------------------------------------------
# "[no|default] net <NET>" command in "router isis <n>" config mode.
#-------------------------------------------------------------------------------

class RouterIsisNetCmd( CliCommand.CliCommandClass ):
   syntax = 'net NET'
   noOrDefaultSyntax = 'net ...'
   data = {
      'net' : 'Configure Network Entity Title (NET) '
              'for this IS-IS instance',
      'NET' : netKw
   }
   handler = RouterIsisMode.setNet
   noOrDefaultHandler = RouterIsisMode.noNet

RouterIsisMode.addCommandClass( RouterIsisNetCmd )

#---------------------------------------------------------------------------------
# [no] bfd all-interfaces in address-family v4 and v6 config mode
#---------------------------------------------------------------------------------
RouterIsisAfMode.addModelet( RouterIsisAfIpv4Modelet )

class RouterIsisAfBfdAllIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd all-interfaces'
   noOrDefaultSyntax = 'bfd ...'
   data = {
      'bfd' : 'Enable BFD',
      'all-interfaces' : 'Enable BFD on all interfaces'
   }
   handler = RouterIsisAfMode.setBfd
   noOrDefaultHandler = RouterIsisAfMode.noBfd
RouterIsisAfMode.addCommandClass( RouterIsisAfBfdAllIntfCmd )

#---------------------------------------------------------------------------------
# [no|default] spf-interval <max-wait> [ initial-wait [hold-time] ] command in
# "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisSpfInterval( CliCommand.CliCommandClass ):
   syntax = 'spf-interval MAX_WAIT [ INITIAL_WAIT [ HOLD_TIME ] ]'
   noOrDefaultSyntax = 'spf-interval ...'
   data = {
      'spf-interval' : 'Set SPF interval config',
      'MAX_WAIT' : CliMatcher.IntegerMatcher(
            SpfIntervalSec.min,
            SpfIntervalSec.max,
            helpdesc='Maximum interval between two SPFs in seconds' ),
      'INITIAL_WAIT' : CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Initial wait interval for SPF in milliseconds' ),
      'HOLD_TIME' : CliMatcher.IntegerMatcher(
            SpfIntervalMsec.min,
            SpfIntervalMsec.max,
            helpdesc='Hold interval between the first and'
                     ' second SPF runs in milliseconds' ),
   }
   handler = RouterIsisMode.setSpfInterval
   noOrDefaultHandler = RouterIsisMode.noSpfInterval

RouterIsisMode.addCommandClass( RouterIsisSpfInterval )

#---------------------------------------------------------------------------------
# [no] spf partial prefix command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
class RouterIsisSpfPartial( CliCommand.CliCommandClass ):
   syntax = 'spf partial prefix'
   noOrDefaultSyntax = syntax
   data = {
      'spf' : 'Configure SPF',
      'partial' : 'Configure partial SPF computation methods',
      'prefix' : 'Partial SPF computation on prefix only changes',
   }
   handler = RouterIsisMode.setSpfPartial
   defaultHandler = RouterIsisMode.setSpfPartial
   noHandler = RouterIsisMode.noSpfPartial
RouterIsisMode.addCommandClass( RouterIsisSpfPartial )

#---------------------------------------------------------------------------------
# [no] set-attached-bit route-map <rmName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
setAttachedBitKw = CliCommand.guardedKeyword( 'set-attached-bit',
                   helpdesc='Configure conditionally setting attach bit',
                   guard=isisNonDefaultInstanceGuard )

class RouterIsisAttachedBitCmd( CliCommand.CliCommandClass ):
   syntax = 'set-attached-bit route-map RM_NAME'
   noOrDefaultSyntax = 'set-attached-bit ...'
   data = {
      'set-attached-bit': setAttachedBitKw,
      'route-map': RouteMapMatchers.routeMapApplication,
      'RM_NAME': mapNameMatcher,
      }
   handler = RouterIsisMode.setAttachedBit
   noOrDefaultHandler = RouterIsisMode.noSetAttachedBit
RouterIsisMode.addCommandClass( RouterIsisAttachedBitCmd )

#---------------------------------------------------------------------------------
# [no|default] "maximum-paths <paths>" command, in 'address-family mode'
#---------------------------------------------------------------------------------
def afMaxEcmpRangeFn( mode ):
   '''
   This code can be called when the startup-config is being parsed,
   which happens before the ForwardingAgent has discovered all the hardware
   and populated Sysdb. Allowing any value in this case.
   '''
   if mode.addrFamily == "ipv4":
      if routingHardwareStatus is None or \
         routingHardwareStatus.maxEcmp == 0:
         return( 1, 0xffffffff )
      else:
         return( 1, routingHardwareStatus.maxEcmp )
   elif mode.addrFamily == "ipv6":
      if routing6HardwareStatus is None or \
         routing6HardwareStatus.maxEcmp == 0:
         return( 1, 0xffffffff )
      else:
         return( 1, routing6HardwareStatus.maxEcmp )

class RouterIsisAfMaxPathsCmd( CliCommand.CliCommandClass ):
   syntax = 'maximum-paths PATHS'
   noOrDefaultSyntax = 'maximum-paths ...'
   data = {
      'maximum-paths' : 'Maximum number of next hops in an ECMP route',
      'PATHS' : CliMatcher.DynamicIntegerMatcher( afMaxEcmpRangeFn,
                                                   helpdesc='Maximum number of next'
                                                   ' hops in an ECMP route' )
   }
   handler = RouterIsisAfMode.setMaxEcmp
   noOrDefaultHandler = RouterIsisAfMode.noMaxEcmp
RouterIsisAfMode.addCommandClass( RouterIsisAfMaxPathsCmd )

#---------------------------------------------------------------------------------
# "[no|default] lsp match flag attached action none | (install default-route)"
# command in 'address-family mode'.
#---------------------------------------------------------------------------------
class RouterIsisAFLspMatcheFlagCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp match flag attached action ( none | ( install default-route ) )'
   noOrDefaultSyntax = 'lsp match flag attached action ...'
   data = {
      'lsp' : lspKw,
      'match' : 'Match on some LSP parameter',
      'flag' : 'Flag to match',
      'attached' : 'Match on attached bit in LSP',
      'action' : 'Action to be taken',
      'none' : 'Skip attached bit processing',
      'install' : 'Install based on attached bit',
      'default-route' : 'Install default route'
   }
   handler = RouterIsisAfMode.setIgnoreAttachedBit
   noOrDefaultHandler = RouterIsisAfMode.noIgnoreAttachedBit

RouterIsisAfMode.addCommandClass( RouterIsisAFLspMatcheFlagCmd )

#---------------------------------------------------------------------------------
# [no|default] lsp flooding dynamic [ LEVEL ]
# command in "router isis <n>" config mode
#---------------------------------------------------------------------------------

class RouterIsisLspFloodingDynamicCmd( CliCommand.CliCommandClass ):
   syntax = 'lsp flooding dynamic [ LEVEL ]'
   noOrDefaultSyntax = syntax

   data = {
      'lsp'      : lspKw,
      'flooding' : 'Set LSP flooding related parameters',
      'dynamic'  : 'Enable Dynamic Flooding',
      'LEVEL' : levelMatcherForConfig,
   }
   handler = RouterIsisMode.dynamicFlooding
   noOrDefaultHandler = RouterIsisMode.noDynamicFlooding

RouterIsisMode.addCommandClass( RouterIsisLspFloodingDynamicCmd )

#---------------------------------------------------------------------------------
# [no|default] passive <intfName>  command in "router isis <n>" config mode
#
# deprecated:
# [no|default] passive-interface <intfName>  command in "router isis <n>" config mode
#---------------------------------------------------------------------------------
def vrfMatches( mode, intfName ):
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   ip6IntfConfig = ip6Config.intf.get( intfName )
   intfVrf = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
       else ipIntfConfig.vrf
   # There's an off-chance that only ip6IntfConfig has the vrf. Take the value from
   # there if present
   if intfVrf == DEFAULT_VRF:
      intfVrf = DEFAULT_VRF if ip6IntfConfig is None or ip6IntfConfig.vrf == "" \
            else ip6IntfConfig.vrf

   routerVrf = instanceConfig.vrfName
   if intfVrf == routerVrf:
      return True
   else:
      vrfName = instanceConfig.vrfName

      mode.addError( "Interface %s does not belong to Vrf %s in which IS-IS "
                     "instance is configured" % ( intfName, vrfName ) )
      return False

def setPassiveIntf( mode, args ):
   intfNames = args[ 'INTFS' ]
   if isinstance( intfNames, IntfRange.IntfList ):
      intfNames = intfNames.intfNames()
      
   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   for name in intfNames:
      if vrfMatches( mode, name ):
         if not checkNodePrefixConflict( mode, mode.instanceName, name ):
            intfConfig = _getOrCreateIntfConfig( isisConfiguration(), name )
            if intfConfig.instanceName == intfConfig.instanceNameInvalid or \
                   intfConfig.instanceName == instanceConfig.instanceName:
               intfConfig.passive = True
               if intfConfig.instanceName == intfConfig.instanceNameInvalid:
                  intfConfig.instanceName = instanceConfig.instanceName
            else:
               mode.addError( "Interface %s is configured in a different IS-IS"
                           " instance %s" % ( name, intfConfig.instanceName ) )

def noPassiveIntf( mode, args ):
   intfNames = args[ 'INTFS' ]
   if isinstance( intfNames, IntfRange.IntfList ):
      intfNames = intfNames.intfNames()

   instanceConfig = isisConfig.instanceConfig[ mode.instanceName ]
   for name in intfNames:
      intfConfig = _getIntfConfig( name )
      if intfConfig is None or intfConfig.passive == False:
         continue
      if vrfMatches( mode, name ):
         if intfConfig.instanceName == intfConfig.instanceNameInvalid or \
                intfConfig.instanceName == instanceConfig.instanceName:
            intfConfig.passive = False
            if intfConfig.instanceName != intfConfig.instanceNameInvalid:
               intfConfig.instanceName = intfConfig.instanceNameInvalid
            _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, name )
         else:
            mode.addError( "Interface %s is configured in a different IS-IS instance"
                           " %s" % ( name, intfConfig.instanceName ) )

passiveIntfDeprecatedKw = CliMatcher.KeywordMatcher( 'passive-interface',
                      helpdesc='Suppress routing updates on an interface' )
intfRangeKw = IntfRange.IntfRangeMatcher(
                      explicitIntfTypes=IntfRange.intfTypesWithRoutingProtoSupport )

class RouterIsisPassiveCmd( CliCommand.CliCommandClass ):
   syntax = 'passive | passive-interface INTFS'
   noOrDefaultSyntax = syntax
   data = {
      'passive' : 'Suppress routing updates on an interface',
      'passive-interface' : CliCommand.Node( matcher=passiveIntfDeprecatedKw,
                                             deprecatedByCmd='passive' ),
      'INTFS' : intfRangeKw
   }
   handler = setPassiveIntf
   noOrDefaultHandler = noPassiveIntf

RouterIsisMode.addCommandClass( RouterIsisPassiveCmd )

#--------------------------------------------------------------------------------
# [no|default] timers lsp generation < max-wait > [ initial-wait ] [ hold-wait ]
# command in "router isis <n>" config mode
#--------------------------------------------------------------------------------
class RouterIsisTimersLSPGenerationCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp generation MAX_WAIT [ INITIAL_WAIT [ HOLD_WAIT ] ]'
   noOrDefaultSyntax = 'timers lsp generation ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'generation' : 'LSP generation interval timer parameters',
      'MAX_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalSec.min,
            LspGenIntervalSec.max,
            helpdesc='Maximum interval (in seconds) between generating two LSPs' ),
      'INITIAL_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalMsec.min,
            LspGenIntervalMsec.max,
            helpdesc='Initial wait time (in milliseconds) before generating LSPs' ),
      'HOLD_WAIT' : CliMatcher.IntegerMatcher(
            LspGenIntervalMsec.min,
            LspGenIntervalMsec.max,
            helpdesc='Wait time (in milliseconds) between generating '
                     'the first and second LSPs' ),
   }
   handler = RouterIsisMode.setLspGenInterval
   noOrDefaultHandler = RouterIsisMode.noLspGenInterval

RouterIsisMode.addCommandClass( RouterIsisTimersLSPGenerationCmd )

#-------------------------------------------------------------------------------
# [no|default] timers lsp out-delay <LSP out-delay>
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPOutDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp out-delay OUT_DELAY'
   noOrDefaultSyntax = 'timers lsp out-delay ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'out-delay' : 'Configure out-delay timer',
      'OUT_DELAY' : CliMatcher.IntegerMatcher( 1, 65000,
            helpdesc='Transmit delay (in milliseconds) for link state packets' ),
   }
   handler = RouterIsisMode.setOutDelayTimer
   noOrDefaultHandler = RouterIsisMode.noOutDelayTimer

RouterIsisMode.addCommandClass( RouterIsisTimersLSPOutDelayCmd )

#------------------------------------------------------------------------------------
# timers local-convergence-delay [<delay_in_seconds>] protected-prefixes
# no|default timers local-convergence-delay protected-prefixes
# in 'router isis' global mode
#------------------------------------------------------------------------------------
convergenceDelayMin = 1
convergenceDelayMax = 65000
delayMatcher = CliMatcher.IntegerMatcher(
                  convergenceDelayMin, convergenceDelayMax,
                  helpdesc='Delay in milliseconds (default 10000 milliseconds)' )

class RouterIsisAndAfTimersLocalConvergenceDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'timers local-convergence-delay [ delay ] protected-prefixes'
   noOrDefaultSyntax = 'timers local-convergence-delay protected-prefixes'
   data = {
      'timers' : 'IS-IS timers',
      'local-convergence-delay' : 'Set local convergence delay',
      'delay' : delayMatcher,
      'protected-prefixes' : 'micro-loop prevention for routers over protected '
                             'links',
   }

   @staticmethod
   def handler( mode, args ):
      mode.setMicroLoopConvergenceDelay( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noMicroLoopConvergenceDelay( args )
RouterIsisMode.addCommandClass( RouterIsisAndAfTimersLocalConvergenceDelayCmd )

#------------------------------------------------------------------------------------
# timers local-convergence-delay [<delay_in_seconds>] protected-prefixes
# no|default timers local-convergence-delay protected-prefixes
# in 'router-isis-af' mode
#------------------------------------------------------------------------------------
RouterIsisAfMode.addCommandClass( RouterIsisAndAfTimersLocalConvergenceDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers lsp refresh <IS-IS LSP refresh-interval>"
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPRefreshIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp refresh REFRESH_INTERVAL'
   noOrDefaultSyntax = 'timers lsp refresh ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'refresh' : 'Configure LSP refresh interval timer',
      'REFRESH_INTERVAL' : CliMatcher.IntegerMatcher(
            LspRefreshInterval.min,
            LspRefreshInterval.max,
            helpdesc='Interval (in seconds) between two LSP refreshes' )
   }
   handler = RouterIsisMode.setLSPRefreshInterval
   noOrDefaultHandler = RouterIsisMode.noLSPRefreshInterval

RouterIsisMode.addCommandClass( RouterIsisTimersLSPRefreshIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers lsp min-remaining-lifetime <n seconds>"
#-------------------------------------------------------------------------------
class RouterIsisTimersLSPMinRemainingLifetimeCmd( CliCommand.CliCommandClass ):
   syntax = 'timers lsp min-remaining-lifetime MIN_REMAINING_LIFETIME'
   noOrDefaultSyntax = 'timers lsp min-remaining-lifetime ...'
   data = {
      'timers' : 'IS-IS timers',
      'lsp' : 'Link State Packet timers',
      'min-remaining-lifetime' : 'Configure minimum remaining lifetime '
                                 'for non self-originating LSPs',
      'MIN_REMAINING_LIFETIME' : CliMatcher.IntegerMatcher(
            MinLspRemainingLifetime.min + 60,
            MinLspRemainingLifetime.max,
            helpdesc='Minimum remaining lifetime for LSPs (in seconds)' ),
   }
   handler = RouterIsisMode.setMinLspRemainingLifetime
   noHandler = RouterIsisMode.noMinLspRemainingLifetime
   defaultHandler = RouterIsisMode.setMinLspRemainingLifetime

RouterIsisMode.addCommandClass( RouterIsisTimersLSPMinRemainingLifetimeCmd )

#---------------------------------------------------------------------------------
# area leader priority <priority> [ LEVEL ]
# area leader [ LEVEL ] [ disabled ]
# [no|default] area leader priority [ LEVEL ]
# [no|default] area leader [ LEVEL ] [ disabled ]
# command in "router isis" config mode
#---------------------------------------------------------------------------------
class RouterIsisAreaLeaderPriorityCmd( CliCommand.CliCommandClass ):
   syntax = '''area leader
               [ ( priority PRIORITY [ LEVEL ] )
               | ( [ LEVEL ] [ disabled ] ) ]'''
   noOrDefaultSyntax = 'area leader ...'
   data = {
      'area'     : 'Set area parameters',
      'leader'   : 'Set area leader parameters',
      'priority' : 'Set the area leader priority',
      'PRIORITY' : CliMatcher.IntegerMatcher( AreaLeaderPriority.min,
                                              AreaLeaderPriority.max,
                                              helpdesc='Area leader priority' ),
      'LEVEL' : levelMatcherForConfig,
      'disabled' : 'Disable becoming area leader',
   }
   handler = RouterIsisMode.setAreaLeaderPriority
   noOrDefaultHandler = RouterIsisMode.defaultAreaLeaderPriority

RouterIsisMode.addCommandClass( RouterIsisAreaLeaderPriorityCmd )

#---------------------------------------------------------------------------------
# [no|default] metric profile <profileName>
# command in "router isis" config mode
#---------------------------------------------------------------------------------
_metricProfileDesc = 'Define metric profile'
profileNameMatcher = CliMatcher.PatternMatcher( instanceNamePattern,
      helpname='WORD', helpdesc='IS-IS metric profile name' )

class RouterIsisMetricProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'metric profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'metric' : _metricProfileDesc,
      'profile' : _metricProfileDesc,
      'PROFILE_NAME' : profileNameMatcher
   }
   handler = RouterIsisMode.gotoIsisMetricProfileMode
   noOrDefaultHandler = RouterIsisMode.delIsisMetricProfileMode

RouterIsisMode.addCommandClass( RouterIsisMetricProfileCmd )

#---------------------------------------------------------------------------------
# [no|default] router-id ipv4 <ipv4Addr>
# [no|default] router-id ipv6 <ipv6Addr>  hidden for now since it has no effect
# commands in "router isis" config mode
#---------------------------------------------------------------------------------
class RouterIsisRouterIdV4Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 ROUTER_IDV4'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id' : routerIdKw,
      'ipv4' : 'IS-IS router ID in IP address format',
      'ROUTER_IDV4' : routerIdV4Matcher
   }
   handler = RouterIsisMode.setRouterIdV4
   noOrDefaultHandler = RouterIsisMode.noRouterIdV4

RouterIsisMode.addCommandClass( RouterIsisRouterIdV4Cmd )

class RouterIsisRouterIdV6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 ROUTER_IDV6'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id' : routerIdKw,
      'ipv6' : CliCommand.Node( matcher=ipv6AddrFormatKw, hidden=True ),
      'ROUTER_IDV6' : routerIdV6Matcher
   }
   handler = RouterIsisMode.setRouterIdV6
   noOrDefaultHandler = RouterIsisMode.noRouterIdV6

RouterIsisMode.addCommandClass( RouterIsisRouterIdV6Cmd )

#-------------------------------------------------------------------------------
# Metric Profile specific CLI (common parts)
#-------------------------------------------------------------------------------
def findMetricProfileConfig( name ):
   for iName, instanceCfg in isisConfig.instanceConfig.items():
      profile = instanceCfg.metricProfiles.get( name )
      if profile :
         return profile, iName
   return None, None

metricMatcher = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Sets metric for Metric Profile' )
metricValueMatcher = CliMatcher.IntegerMatcher(
      Metric.isisMetricMin + 1,
      Metric.isisMetricMax - 1,
      helpdesc='Value of the route metric' )
speedValueMatcher = CliMatcher.IntegerMatcher(
      Speed.min,
      Speed.max,
      helpdesc='Interface speed' )
unitsMatcher = CliMatcher.EnumMatcher( {
      SpeedUnitType.mbps : 'Megabits per second',
      SpeedUnitType.gbps : 'Gigabits per second',
} )

#---------------------------------------------------------------------------------
# [no|default] metric <metricValue>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
class RouterIsisMetricValueCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC_VALUE'
   noOrDefaultSyntax = 'metric ...'
   data = {
      'metric' : metricMatcher,
      'METRIC_VALUE' : metricValueMatcher
   }
   handler = RouterIsisMetricProfileMode.setMetricValue
   noOrDefaultHandler = handler

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricValueCmd )

#---------------------------------------------------------------------------------
# [no|default] metric ratio <speed> <mbps|gbps>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
class RouterIsisMetricRatioCmd( CliCommand.CliCommandClass ):
   syntax = 'metric ratio SPEED UNITS'
   noOrDefaultSyntax = 'metric ratio ...'
   data = {
      'metric' : metricMatcher,
      'ratio' : 'Sets metric inversely proportional to given <speed>',
      'SPEED' : speedValueMatcher,
      'UNITS' : unitsMatcher,
   }
   handler = RouterIsisMetricProfileMode.setMetricRatio
   noOrDefaultHandler = RouterIsisMetricProfileMode.delMetricRatio

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricRatioCmd )

#---------------------------------------------------------------------------------
# [no|default] metric <metricValue> if speed <= <speed> <mbps|gbps>
# command in "router isis metric profile" config mode
#---------------------------------------------------------------------------------
_metricRuleDesc = 'Given metric is used when interface speed matches condition'

class RouterIsisMetricValueSpeedCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRIC_VALUE if speed <= SPEED UNITS'
   noOrDefaultSyntax = syntax
   data = {
      'metric' : metricMatcher,
      'METRIC_VALUE' : metricValueMatcher,
      'if' : _metricRuleDesc,
      'speed' : _metricRuleDesc,
      '<=' : _metricRuleDesc,
      'SPEED' : speedValueMatcher,
      'UNITS' : unitsMatcher,
   }
   handler = RouterIsisMetricProfileMode.addMetricRule
   noOrDefaultHandler = handler

RouterIsisMetricProfileMode.addCommandClass( RouterIsisMetricValueSpeedCmd )

#-------------------------------------------------------------------------------
# Segment-Routing specific CLI
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# [no|default] shutdown
#-------------------------------------------------------------------------------
class RoutingIsisSrMplsShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Disable segment routing',
   }
   handler = RouterIsisSrMplsMode.srShutdown
   noOrDefaultHandler = RouterIsisSrMplsMode.noSrShutdown
RouterIsisSrMplsMode.addCommandClass( RoutingIsisSrMplsShutdownCmd )

#-------------------------------------------------------------------------------
# [no|default] segment path verification prefix-segment adjacency-segment
#-------------------------------------------------------------------------------
class RoutingIsisSrMplsSegmentPathVerificationCmd( CliCommand.CliCommandClass ):
   syntax = 'segment path verification prefix-segment adjacency-segment'
   noOrDefaultSyntax = syntax
   data = {
      'segment' : 'Configuration for segments',
      'path' : 'Configuration for the path to the segment',
      'verification' : 'Enable path verification for segments',
      'prefix-segment' : 'Enable path verification for prefix-segments',
      'adjacency-segment' : 'Enable path verification for adjacency-segments',
   }
   handler = RouterIsisSrMplsMode.setSrSegmentPathVerification
   noOrDefaultHandler = RouterIsisSrMplsMode.noSrSegmentPathVerification

if gatedToggleLib.toggleIsisSrTunnelVerificationEnabled():
   RouterIsisSrMplsMode.addCommandClass(
         RoutingIsisSrMplsSegmentPathVerificationCmd )

#-------------------------------------------------------------------------------
# [no|default] router-id <router-id>
#-------------------------------------------------------------------------------
# router-id configuration is not supposed to be under the RouterIsisSrMplsMode.
# Make it hidden for backward compatibility.
srRouterIdKw = CliMatcher.KeywordMatcher( 'router-id',
      helpdesc='Configure the router ID for IS-IS Segment Routing process' )
routerIdMatcher = IpAddrMatcher.IpAddrMatcher( 
      "IS-IS SR router ID in IP address format" )

class RouterIsisRouterIdCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ROUTER_ID'
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
      'router-id' : CliCommand.Node( matcher=srRouterIdKw, hidden=True ),
      'ROUTER_ID' : routerIdMatcher
   }
   handler = RouterIsisSrMplsMode.setSrRouterId
   noOrDefaultHandler = RouterIsisSrMplsMode.noSrRouterId

RouterIsisSrMplsMode.addCommandClass( RouterIsisRouterIdCmd )

#-------------------------------------------------------------------------------
# [no|default] adjacency-segment allocation [none|sr-peers [backup-eligible]|
# all-interfaces [backup-eligible] ]
#-------------------------------------------------------------------------------
srAdjacencySegmentKw = CliMatcher.KeywordMatcher( 'adjacency-segment',
                                    helpdesc='Adjacency Segment allocation options' )
allocationKw = CliMatcher.KeywordMatcher( 'allocation',
                                        helpdesc='Allocation of adjacency segments' )
noneKw = CliMatcher.KeywordMatcher( 'none',
                          helpdesc='Disable automatic adjacency segment allocation' )
srPeersKw = CliMatcher.KeywordMatcher( 'sr-peers',
          helpdesc='Allocate adjacency segments to IS-IS adjacencies with SR peers' )
allInterfacesKw = CliMatcher.KeywordMatcher( 'all-interfaces',
                    helpdesc='Allocate adjacency segments to all IS-IS adjacencies' )
backupKw = CliMatcher.KeywordMatcher( 'backup-eligible',
                                      helpdesc='Eligible for protection' )

class RouterIsisSrMplsAdjSegAllocationCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency-segment allocation '\
            '( none | ( sr-peers | all-interfaces [ backup-eligible ] ) )'
   noOrDefaultSyntax = 'adjacency-segment allocation ...'
   data = {
      'adjacency-segment' : srAdjacencySegmentKw,
      'allocation' : allocationKw,
      'none' : noneKw,
      'sr-peers' : srPeersKw,
      'all-interfaces' : allInterfacesKw,
      'backup-eligible' : backupKw
   }
   handler = RouterIsisSrMplsMode.adjacencySegmentAllocation
   noOrDefaultHandler = RouterIsisSrMplsMode.noAdjacencySegmentAllocation
RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsAdjSegAllocationCmd )

#-------------------------------------------------------------------------------
# [no|default] adjacency-segment sid reuse timeout <time>  XXX: Hidden
#-------------------------------------------------------------------------------
sidKw = CliMatcher.KeywordMatcher( 'sid', helpdesc='Adjacency segment SID' )
reUseKw = CliMatcher.KeywordMatcher( 'reuse',
                                     helpdesc='Reserve the SID for reuse' )
timeoutKw = CliMatcher.KeywordMatcher( 'timeout',
                                     helpdesc='Set the reserve timeout for the SID' )
adjSegmentSidReuseTimeout = CliMatcher.IntegerMatcher( 0, U32_MAX_VALUE - 1,
                                             helpdesc='Reserve timeout in seconds' )
adjSegmentSidReuseTimeoutInf = CliMatcher.KeywordMatcher( 'infinite',
                                                  helpdesc='Reserve SID infinitely' )
class RouterIsisSrMplsAdjSegSidReuseTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency-segment sid reuse timeout ( TIMEOUT | infinite )'
   noOrDefaultSyntax = 'adjacency-segment sid reuse timeout ...'
   data = {
      'adjacency-segment' : srAdjacencySegmentKw,
      'sid' : CliCommand.Node( matcher=sidKw, hidden=True ),
      'reuse' : reUseKw,
      'timeout' : timeoutKw,
      'TIMEOUT' : adjSegmentSidReuseTimeout,
      'infinite' : adjSegmentSidReuseTimeoutInf
   }
   handler = RouterIsisSrMplsMode.adjacencySegmentSidReuseTimeout
   noOrDefaultHandler = RouterIsisSrMplsMode.noAdjacencySegmentSidReuseTimeout
RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsAdjSegSidReuseTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] prefix-segment <prefix> index <index>
#-------------------------------------------------------------------------------
def srIndexRangeFn( mode ):
   isisStatus = isisStatusDir.get( mode.parent_.vrfName )
   if isisStatus is None or \
      isisStatus.instanceStatus.get( mode.parent_.instanceName ) is None:
      return ( 0, 65535 )
   else:
      instanceStatus = isisStatus.instanceStatus[ mode.parent_.instanceName ]
      return ( 0, instanceStatus.srSrgbRange -1 )

srPrefixSegmentKw = CliMatcher.KeywordMatcher( 'prefix-segment',
                                    helpdesc='Configure a prefix segment' )
indexKw = CliMatcher.KeywordMatcher( 'index',
                                     helpdesc='Prefix segment identifier' )
indexMatcher = CliMatcher.DynamicIntegerMatcher( srIndexRangeFn,
      helpdesc='Index value' )

class RouterIsisSrMplsPrefixSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'prefix-segment ( IPADDR | IP6ADDR ) index INDEX'
   noOrDefaultSyntax = 'prefix-segment ( IPADDR | IP6ADDR ) ...'
   data = {
      'prefix-segment' : srPrefixSegmentKw,
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexMatcher,
   }
   handler = RouterIsisSrMplsMode.setPrefixSegment
   noOrDefaultHandler = RouterIsisSrMplsMode.noPrefixSegment

RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsPrefixSegmentCmd )

#-------------------------------------------------------------------------------
# [no|default] proxy-node-segment <host prefix> index <index> [ LEVEL ]
#-------------------------------------------------------------------------------
proxyNodeSegmentKw = CliMatcher.KeywordMatcher( 'proxy-node-segment',
                  helpdesc='Configure a node segment on behalf of another node' )
class RouterIsisSrMplsProxyNodeSegmentCmd( CliCommand.CliCommandClass ):
   syntax = 'proxy-node-segment ( IPADDR | IP6ADDR ) index INDEX [ LEVEL ]'
   noOrDefaultSyntax = 'proxy-node-segment ( IPADDR | IP6ADDR ) ...'
   data = {
      'proxy-node-segment' : proxyNodeSegmentKw,
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'index' : indexKw,
      'INDEX' : indexMatcher,
      'LEVEL' : level12MatcherForConfig,
   }
   handler = RouterIsisSrMplsMode.setProxyNodeSegment
   noOrDefaultHandler = RouterIsisSrMplsMode.noProxyNodeSegment

RouterIsisSrMplsMode.addCommandClass( RouterIsisSrMplsProxyNodeSegmentCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute bgp' under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RouterIsisAfModeRedistributeCmd( OspfRedistributeCmdData ):
   _ospfV3RedistRule = 'ospfv3 ' + matchRoute
   syntax = 'redistribute ( bgp | ( ' + _ospfV3RedistRule + ' ) ) ' + \
            '[ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute ( bgp | ( ' + _ospfV3RedistRule + ') ) ...'
   data = {
      'bgp' : bgpCliCommandNode
   }
   data.update( OspfRedistributeCmdData.data )

   @staticmethod
   def handler( mode, args ):
      proto, routeType, mapName = getProtoRouteType( args )
      mode.setRedistribute( proto, mapName=mapName, routeType=routeType )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proto, routeType, _ = getProtoRouteType( args )
      mode.noRedistribute( proto, routeType=routeType )
RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute dhcp' under 'address-family ipv6' mode
#-------------------------------------------------------------------------------
class RouterIsisAfIpv6RedistributeDhcp( CliCommand.CliCommandClass ):
   syntax = 'redistribute dhcp [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute dhcp ...'
   data = {
      'redistribute' : redistributeKw,
      'dhcp' : 'DHCPv6 routes',
      'route-map' : RouteMapMatchers.routeMapApplication,
      'MAPNAME' : mapNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      proto, routeType, mapName = getProtoRouteType( args )
      mode.setRedistribute( proto, routeType, mapName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proto, routeType, _ = getProtoRouteType( args )
      mode.noRedistribute( proto, routeType )

RouterIsisAfIpv6Modelet.addCommandClass( RouterIsisAfIpv6RedistributeDhcp )

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis level-1|level-2 into level-2|level-1 route-map <>'
#    under 'address-family ipv[4|6]' mode
#-------------------------------------------------------------------------------
class RouterIsisAfModeRedistributeLeakL1ToL2Cmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis level-1 into level-2 route-map MAPNAME'
   noOrDefaultSyntax = 'redistribute isis level-1 ...'
   data = {
      'level-1' : 'level-1 routes',
      'level-2' : 'into level-2 routes'
   }
   data.update( isisRedistributeCmdData )

   @staticmethod
   def handler( mode, args ):
      mode.setRouteLeak( 'level-1', args[ 'MAPNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRouteLeak( 'level-1' )

RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeLeakL1ToL2Cmd )

class RouterIsisAfModeRedistributeLeakL2ToL1Cmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis level-2 into level-1 route-map MAPNAME'
   noOrDefaultSyntax = 'redistribute isis level-2 ...'
   data = {
      'level-2' : 'level-2 routes',
      'level-1' : 'into level-1 routes'
   }
   data.update( isisRedistributeCmdData )

   @staticmethod
   def handler( mode, args ):
      mode.setRouteLeak( 'level-2', args[ 'MAPNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRouteLeak( 'level-2' )

RouterIsisAfMode.addCommandClass( RouterIsisAfModeRedistributeLeakL2ToL1Cmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute isis instance [route-map <>]'
#    under 'router isis' and 'address-family ipv[4|6]' modes
#-------------------------------------------------------------------------------
class RedistributeIsisInstanceCmd( CliCommand.CliCommandClass ):
   syntax = 'redistribute isis instance [ route-map MAPNAME ]'
   noOrDefaultSyntax = 'redistribute isis instance ...'
   data = {
      'instance' : 'routes from another IS-IS instance',
   }
   data.update( isisRedistributeCmdData )

   @staticmethod
   def handler( mode, args ):
      mode.setRedistribute( 'protoIsis', mapName=args.get( 'MAPNAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRedistribute( 'protoIsis' )

if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   RouterIsisMode.addCommandClass( RedistributeIsisInstanceCmd )
   RouterIsisAfMode.addCommandClass( RedistributeIsisInstanceCmd )

#----------------------------------------------------------------------------------
# [no|default] "distance <1-255> [ LEVEL ]" command, in address-family mode
#----------------------------------------------------------------------------------
distanceMin = 1
distanceMax = 255

class RouterIsisAFDistanceCmd( CliCommand.CliCommandClass ):
   syntax = 'distance DISTANCE_VALUE [ LEVEL ]'
   noOrDefaultSyntax = 'distance [ DISTANCE_VALUE ] [ LEVEL ]'
   data = {
      'distance' : 'Protocol instance administrative distance',
      'DISTANCE_VALUE' : CliMatcher.IntegerMatcher( distanceMin, distanceMax,
                            helpdesc='Administrative distance value' ),
      'LEVEL' : CliMatcher.EnumMatcher( {
            'level-1' : 'Level-1 administrative distance',
            'level-2' : 'Level-2 administrative distance',
      } ),
   }
   handler = RouterIsisAfMode.setDistance
   noOrDefaultHandler = RouterIsisAfMode.noDistance

RouterIsisAfMode.addCommandClass( RouterIsisAFDistanceCmd )

#---------------------------------------------------------------------------------
# [no] authentication mode [ < text|md5|sha key-id <id> > [rx-disable] [ LEVEL ] ]
#---------------------------------------------------------------------------------
class AuthenticationModeExpression( CliCommand.CliExpression ):
   expression = '( md5 | text | (sha key-id KEYID) )'
   data = {
      'md5' : 'HMAC-MD5 authentication mode',
      'text' : 'Text authentication mode',
      'sha' : 'SHA authentication mode',
      'key-id' : keyIdKw,
      'KEYID' : keyIdMatcher
   }

class RouterIsisModeAuthenticationMode( CliCommand.CliCommandClass ):
   syntax = 'authentication mode AUTHMODE [ rx-disabled ] [ LEVEL ]'
   noOrDefaultSyntax = 'authentication mode [ AUTHMODE [ rx-disabled ] [ LEVEL ] ]'
   data = {
      'authentication' : authKw,
      'mode' : modeKw,
      'AUTHMODE' : AuthenticationModeExpression,
      'rx-disabled' : rxDisabledKw,
      'LEVEL' : levelMatcherForConfig,
   }
   handler = RouterIsisMode.setAuthMode
   noOrDefaultHandler = RouterIsisMode.noAuthMode

RouterIsisMode.addCommandClass( RouterIsisModeAuthenticationMode )

#
#---------------------------------------------------------------------------------
# [no] authentication [key-id <id> algorithm SHA]
#       key [ [0|7] < key-string > [ LEVEL ] ]
#---------------------------------------------------------------------------------
shaSyntax = ' algorithm SHA [ rfc-5310 ] '
keySyntax = ' ( ( [ 0 ] PASS ) | ( 7 EPASS )  ) [ LEVEL ] '
authKeySyntax = 'authentication [ key-id ID' + shaSyntax + '] key' + keySyntax
noOrDefaultAuthSyntax = 'authentication ( key-id ID [' + shaSyntax + 'key'\
                        '[ ' + keySyntax + ' ] ] ) | ( key [' + keySyntax + '] )'
authKeyIntfSyntax = 'isis ' + authKeySyntax
noOrDefaultAuthIntfSyntax = 'isis ' + noOrDefaultAuthSyntax
authData = {
   'authentication' : authKw,
   'key-id': authKeyIdKw,
   'ID' : keyIdMatcher,
   'algorithm' : authAlgoKw,
   'SHA' : CliMatcher.EnumMatcher( { keyword : '%s authentication' % keyword.upper()
                                     for keyword in SHA_KW_TO_TAC_TYPE_MAP } ),
   'rfc-5310' : 'SHA digest computation according to rfc5310',
   'key' : authKeyKw,
   '0' : encryptionType0Kw,
   '7' : encryptionType7Kw,
   'LEVEL' : CliMatcher.EnumMatcher( {
         'level-1' : 'Level-1 authentication',
         'level-2' : 'Level-2 authentication'
   } ),
   'PASS' : CliCommand.Node( matcher=authUnencryptedPasswdKw, sensitive=True ),
   'EPASS' : CliCommand.Node( matcher=authEncryptedPasswdKw, sensitive=True ),
}
authIntfData = dict( authData )
authIntfData[ 'isis' ] = isisKw

class RouterIsisAuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = authKeySyntax
   noOrDefaultSyntax = noOrDefaultAuthSyntax
   data = authData
   handler = RouterIsisMode.setAuthKey
   noOrDefaultHandler = RouterIsisMode.noAuthKey

RouterIsisMode.addCommandClass( RouterIsisAuthenticationKeyCmd )

#-------------------------------------------------------------------------------
# Adds isis-specific CLI commands to the "config-if" mode for routed ports.
# Enable isis commands under config-if only on routed and loopback ports
#-------------------------------------------------------------------------------
def isisSupportedOnIntfConfigMode( intfConfigMode ):
   # Don't configure isis on ineligible interfaces
   if intfConfigMode.intf.routingSupported() and \
          not intfConfigMode.intf.name.startswith( "Management" ):
      return True
   return False

class RoutingProtocolIsisIntfConfigModelet( CliParser.Modelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.intf = IsisIntf( intfConfigMode.intf, intfConfigMode.sysdbRoot )
      self.mode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isisSupportedOnIntfConfigMode( mode )

class RoutingProtocolIsisLoopbackIntfConfigModelet(
      RoutingProtocolIsisIntfConfigModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RoutingProtocolIsisIntfConfigModelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Loopback' )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolIsisIntfConfigModelet )
if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   IntfCli.IntfConfigMode.addModelet( RoutingProtocolIsisLoopbackIntfConfigModelet )

def isisConfiguration():
   return isisConfig

class IsisIntf( IntfCli.IntfDependentBase ):

   def setDefault( self ):
      config = isisConfiguration()
      del config.intfConfig[ self.intf_.name ]

modelet = RoutingProtocolIsisIntfConfigModelet

#-------------------------------------------------------------------------------
# SrConfig stores all common sr configuration for OSPF and ISIS (only node segment
# for now)
#-------------------------------------------------------------------------------
srConfig = None
def _getSrIntfConfig( intfName ):
   return srConfig.intfConfig.get( intfName, None )

#-------------------------------------------------------------------------------
# IntfConfig for ISIS configuration, is created when one of its attributes
# is configured. It is deleted when all the attributes are at their defaults
# What this means is that after the last "no isis..." command is run on the
# interface, we delete the intfConfig object
#-------------------------------------------------------------------------------
def _getIntfConfig( intfName ):
   return isisConfiguration().intfConfig.get( intfName, None )

def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig

def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   """Delete the intfConfig collection element if all the attributes of intfConfig
   have default values. This needs to be called by command handlers for "no" form
   of the isis interface config commands. Also when we add a new attribute to
   intfConfig, it needs to be added below in the comparison against default values
   """
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.instanceName == intfConfig.instanceNameInvalid ) and
        ( not intfConfig.multiInstanceName ) and
        ( intfConfig.lspInterval == intfConfig.lspIntervalDefault ) and
        ( intfConfig.priority == intfConfig.priorityDefault ) and
        ( intfConfig.metric == intfConfig.metricNotConfigured ) and
        ( not intfConfig.metricProfile ) and
        ( not intfConfig.metricProfileV6 ) and
        ( intfConfig.metricV6 == intfConfig.metricV6NotConfigured ) and
        ( intfConfig.mtAddressFamily == intfConfig.mtAddressFamilyDefault ) and
        ( intfConfig.helloInterval == intfConfig.helloIntervalDefault ) and
        ( intfConfig.helloMultiplier == intfConfig.helloMultiplierDefault ) and
        ( intfConfig.interfaceType == intfConfig.interfaceTypeDefault ) and
        ( intfConfig.passive == intfConfig.passiveDefault ) and
        ( len( intfConfig.srSingleAdjacencySegment ) == 0 ) and
        ( len( intfConfig.srMultipleAdjacencySegment ) == 0 ) and
        ( intfConfig.circuitType == intfConfig.circuitTypeDefault ) and
        ( intfConfig.bfdV4 == intfConfig.bfdV4Default ) and
        ( intfConfig.bfdV6 == intfConfig.bfdV6Default ) and
        ( intfConfig.authModeL1 == intfConfig.authModeDefault ) and
        ( intfConfig.authModeL2 == intfConfig.authModeDefault ) and
        ( not intfConfig.authKeyColl ) and
        ( intfConfig.helloPadding == intfConfig.helloPaddingDefault ) and
        ( intfConfig.routeTagV4 == intfConfig.routeTagNotConfigured ) and
        ( intfConfig.routeTagV6 == intfConfig.routeTagNotConfigured ) and
        ( not intfConfig.protectionConfigV4 ) and
        ( not intfConfig.protectionConfigV6 ) and
        ( not intfConfig.protectionConfigV4V6 ) and
        ( intfConfig.areaProxyBoundary == intfConfig.areaProxyBoundaryDefault ) and
        ( intfConfig.protectionSrlgV4 == intfConfig.protectionSrlgDefault ) and
        ( intfConfig.protectionSrlgV6 == intfConfig.protectionSrlgDefault ) and
        ( intfConfig.protectionSrlgV4V6 == intfConfig.protectionSrlgDefault ) ):

      del config.intfConfig[ intfName ]

def addIpInterfaceWarnings( mode, addrFamily ):
   '''
   Warnings are issued whenever an IS-IS instance is enabled on an interface
   with missing V4/V6 address configuration. These warnings are also applicable 
   for user-configured MT address-family commands
   @mode - The CLI mode where these warnings get printed
   @addrFamily - The configured address family(ies)
   '''
   ipConfigured = ipConfig.ipIntfConfig.get( mode.intf.name )
   ip6Configured = ip6Config.intf.get( mode.intf.name )

   # When a vrf is associated with an interface with no ip address configured
   # the default address "0.0.0.0/0" is assigned. A warning needs to be given
   # in that case
   ipV4WarningCondition = ( ipConfigured is None
         or ( ipConfigured.addrWithMask ==
              Arnet.AddrWithMask( "0.0.0.0/0" )
              and not ipConfigured.unnumberedIntfId )
         )

   ipV6WarningCondition = ( ip6Configured is None
         or ip6Configured.ipv6Configured() == False )
   v4WarningMsg = "Interface does not have IPv4 address configured"
   v6WarningMsg = "Interface does not have IPv6 address configured"
   bothIpWarningMsg = (
         "Interface does not have IPv4 and IPv6 address configured" )
   noIpWarningMsg = "Interface does not have IPv4 or IPv6 address configured"

   if addrFamily == IsisAddressFamily.addressFamilyNone \
         and ipV4WarningCondition and ipV6WarningCondition :
      mode.addWarning( noIpWarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyIPv4 \
         and ipV4WarningCondition:
      mode.addWarning( v4WarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyIPv6 \
         and ipV6WarningCondition:
      mode.addWarning( v6WarningMsg )
   elif addrFamily == IsisAddressFamily.addressFamilyBoth:
      if ipV4WarningCondition and ipV6WarningCondition:
         mode.addWarning( bothIpWarningMsg )
      elif ipV4WarningCondition:
         mode.addWarning( v4WarningMsg )
      elif ipV6WarningCondition:
         mode.addWarning( v6WarningMsg )

#-------------------------------------------------------------------------------
# "[no|default] isis enable"
# command, in config-if mode.
# "[no|default] isis instance <INSTANCE_NAME>"
# command, in loopback config-if mode.
#-------------------------------------------------------------------------------
instanceNameKw = CliMatcher.DynamicNameMatcher(
   lambda mode: isisConfig.instanceConfig,
   'Name of the IS-IS protocol instance',
   pattern=instanceNamePattern )

# When an interface is being added to a ISIS instance or being set as
# passive, check for any prefix segment conflicts that could happen
def checkNodePrefixConflict( mode, instanceName, intfName ):
   srIntfConfig = _getSrIntfConfig( intfName )
   if not srIntfConfig:
      # The interface has no node-segment configuration, so there can't
      # be any conflicts, return False
      return False

   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )
   ip6IntfConfig = ip6Config.intf.get( intfName )
   if ipIntfConfig:
      prefix = Arnet.IpGenPrefix( str ( ipIntfConfig.addrWithMask.subnet ) )
   else:
      prefix = None

   # If the interface has v4 node-segment configured, check for v4 prefix conflicts
   if srIntfConfig.srNodeSegmentIndex != srSidInvalid:
      if prefix != None:
         if _isisSrConflictingPrefixFound( instanceName, \
               intfName, prefix, mode=mode ):
            mode.addError( "Node-segment conflicts with a prefix/proxy segment"
                           " configured for the prefix %s" % prefix )
            return True

      if _isisSrConflictingSidFound( instanceName, intfName, None,
                                     srIntfConfig.srNodeSegmentIndex, mode=mode ):
         mode.addError( "IPv4 Node-segment index conflicts with an existing segment"
                        " index in the instance %s" % instanceName )
         return True
   # If the interface has v6 node-segment configured, check for v6 prefix conflicts
   if srIntfConfig.srV6NodeSegmentIndex != srSidInvalid:
      if ip6IntfConfig:
         for prefix in ip6IntfConfig.addr:
            prefix = Arnet.IpGenPrefix( str( prefix ) )
            if _isisSrConflictingPrefixFound( instanceName, intfName,
                                              prefix, mode=mode ):
               mode.addError( "Node-segment conflicts with a prefix/proxy segment"
                              " configured for the prefix %s" % prefix )
               return True
 
      if _isisSrConflictingSidFound( instanceName, intfName, None,
                                     srIntfConfig.srV6NodeSegmentIndex, mode=mode ):
         mode.addError( "IPv6 Node-segment index conflicts with an existing segment"
                        " index in the instance %s" % instanceName )
         return True
   
   return False 

def setIsisEnable( mode, args ):
   instanceName = args[ 'INSTANCE_NAME' ]
   if mode.session_.isInteractive():
      if instanceName not in isisConfig.instanceConfig:
         mode.addWarning( "IS-IS instance %s doesn't exist." % instanceName )
      else:
         instanceConfig = isisConfig.instanceConfig[ instanceName ]
         if instanceConfig.multiTopology and isisConfig.intfConfig.get( 
            mode.intf.name ):
            intfMtAddrFamily = isisConfig.intfConfig[
               mode.intf.name ].mtAddressFamily
            instanceAddrFamily = instanceConfig.addressFamily
            if intfMtAddrFamily == IsisAddressFamily.addressFamilyNone:
               intfMtAddrFamily = instanceAddrFamily

            if instanceAddrFamily == IsisAddressFamily.addressFamilyIPv6 and ( 
               intfMtAddrFamily == IsisAddressFamily.addressFamilyIPv4 
               or intfMtAddrFamily == IsisAddressFamily.addressFamilyBoth ):
               mode.addWarning( "Interface multi topology address family not"
                                " configured in router IS-IS mode" )
            else:
               addIpInterfaceWarnings( mode, intfMtAddrFamily )
         else:
            instanceAddrFamily = instanceConfig.addressFamily
            addIpInterfaceWarnings( mode, instanceAddrFamily )
   if not checkNodePrefixConflict( mode, instanceName, mode.intf.name ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      # Determine if this is multiple instance or just a single instance
      if 'instance' in args:
         intfConfig.multiInstanceName.add( instanceName )
      else:
         intfConfig.instanceName = instanceName
     
def noIsisEnable( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   if 'instance' in args:
      instanceName = args[ 'INSTANCE_NAME' ]
      del intfConfig.multiInstanceName[ instanceName ]
   else:
      intfConfig.instanceName = intfConfig.instanceNameInvalid
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisLoopbackIntfEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'isis instance INSTANCE_NAME'
   noOrDefaultSyntax = 'isis instance INSTANCE_NAME...'

   data = {
      'isis' : isisKw,
      'instance' : 'Enable IS-IS protocol on the interface',
      'INSTANCE_NAME' : instanceNameKw
   }
   handler = setIsisEnable
   noOrDefaultHandler = noIsisEnable

class IsisIntfEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'isis enable INSTANCE_NAME'
   noOrDefaultSyntax = 'isis enable ...'
   data = {
      'isis' : isisKw,
      'enable' : 'Enable IS-IS protocol on the interface',
      'INSTANCE_NAME' : instanceNameKw
   }
   handler = setIsisEnable
   noOrDefaultHandler = noIsisEnable

modelet.addCommandClass( IsisIntfEnableCmd )

if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   RoutingProtocolIsisLoopbackIntfConfigModelet.addCommandClass(
      IsisLoopbackIntfEnableCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis rfc8202 disabled 
# command, in config-if mode.
#-------------------------------------------------------------------------------

def setIsisRfc8202Disabled( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.multiInstanceRfc8202 = False

def noIsisRfc8202Disabled( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.multiInstanceRfc8202 = intfConfig.multiInstanceRfc8202Default
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfRfc8202DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'isis rfc8202 disabled'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'rfc8202' : rfc8202Kw,
      'disabled' : 'Disable RFC8202 compliance to enable peering with'
                   ' non-multi-instance capable routers'
   }
   handler = setIsisRfc8202Disabled
   noOrDefaultHandler = noIsisRfc8202Disabled

if RoutingLibToggleLib.toggleIsisMultiInstanceEnabled():
   modelet.addCommandClass( IsisIntfRfc8202DisabledCmd )
#-------------------------------------------------------------------------------
# "[no|default] isis metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
isisIntfMetricMatcher = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='Set the metric for interface' )
isisIntfMetricMaxMatcher = CliMatcher.KeywordMatcher( 'maximum',
      helpdesc='Maximum metric value' )

def setIsisIntfMetric( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metric = args.get( 'METRIC_VALUE', Metric.isisMetricMax )

def noIsisIntfMetric( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.metric = intfConfig.metricNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'isis metric ( METRIC_VALUE | maximum )'
   noOrDefaultSyntax = 'isis metric ...'
   data = {
      'isis' : isisKw,
      'metric' : isisIntfMetricMatcher,
      'METRIC_VALUE' : metricValueMatcher,
      'maximum' : isisIntfMetricMaxMatcher
   }
   handler = setIsisIntfMetric
   noOrDefaultHandler = noIsisIntfMetric

modelet.addCommandClass( IsisIntfMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric <value> | <maximum>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
isisIntfIpv6Matcher = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='IS-IS IPv6 interface config' )

def setIsisIntfV6Metric( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metricV6 = args.get( 'METRIC_VALUE', Metric.isisMetricMax )
   if mode.session_.isInteractive():
      instanceName = intfConfig.instanceName
      if instanceName in isisConfig.instanceConfig:
         instance = isisConfig.instanceConfig[ instanceName ]
         if not instance.multiTopology:
            mode.addWarning( mtWarningMsg )

def noIsisIntfV6Metric( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.metricV6 = intfConfig.metricV6NotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfIPV6MetricCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ipv6 metric ( METRIC_VALUE | maximum )'
   noOrDefaultSyntax = 'isis ipv6 metric ...'
   data = {
      'isis' : isisKw,
      'ipv6' : isisIntfIpv6Matcher,
      'metric' : isisIntfMetricMatcher,
      'METRIC_VALUE' : metricValueMatcher,
      'maximum' : isisIntfMetricMaxMatcher
   }
   handler = setIsisIntfV6Metric
   noOrDefaultHandler = noIsisIntfV6Metric

modelet.addCommandClass( IsisIntfIPV6MetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def getAllMetricProfiles( mode ):
   return sum( [ cfg.metricProfiles.keys() for cfg in \
                              isisConfig.instanceConfig.values() ], [])

isisIntfProfileMatcher = CliMatcher.KeywordMatcher( 'profile',
      helpdesc='Set the metric profile for interface' )
isisIntfProfileNameMatcher = CliMatcher.DynamicNameMatcher( getAllMetricProfiles, 
      'IS-IS metric profile name', pattern=instanceNamePattern )

def setMetricProfile( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metricProfile = args.get( 'PROFILE_NAME', '' )

class IsisIntfMetricProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'isis metric profile PROFILE_NAME'
   noOrDefaultSyntax = 'isis metric profile ...'
   data = {
      'isis' : isisKw,
      'metric' : isisIntfMetricMatcher,
      'profile' : isisIntfProfileMatcher,
      'PROFILE_NAME' : isisIntfProfileNameMatcher
   }
   handler = setMetricProfile 
   noOrDefaultHandler = handler

modelet.addCommandClass( IsisIntfMetricProfileCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 metric profile <profileName>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setMetricProfileV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.metricProfileV6 = args.get( 'PROFILE_NAME', '' )

class IsisIntfIPV6MetricProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ipv6 metric profile PROFILE_NAME'
   noOrDefaultSyntax = 'isis ipv6 metric profile ...'
   data = {
      'isis' : isisKw,
      'ipv6' : isisIntfIpv6Matcher, 
      'metric' : isisIntfMetricMatcher,
      'profile' : isisIntfProfileMatcher,
      'PROFILE_NAME' : isisIntfProfileNameMatcher
   }
   handler = setMetricProfileV6
   noOrDefaultHandler = handler

modelet.addCommandClass( IsisIntfIPV6MetricProfileCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis multi-topology address-family ( ipv4 | ipv6 ) [ unicast ]" 
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfMtAf( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   addrFamily = args[ 'AF' ]
   intfConfig.mtAddressFamily = addAddrFamilyToConfig( 
         intfConfig.mtAddressFamily, addrFamily )

   if mode.session_.isInteractive():
      instanceName = intfConfig.instanceName
      if instanceName in isisConfig.instanceConfig:
         instance = isisConfig.instanceConfig[ instanceName ]
         if not instance.multiTopology:
            mode.addWarning( mtWarningMsg )
         elif ( instance.addressFamily != IsisAddressFamily.addressFamilyBoth ):
            if addrFamily == "ipv4":
               mode.addWarning( mtV4WarningMsg )
         else:
            addIpInterfaceWarnings( mode, addrFamilyStrToEnum( addrFamily ))

def noIsisIntfMtAf( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
   intfConfig.mtAddressFamily = delAddrFamilyFromConfig( 
         intfConfig.mtAddressFamily, args[ 'AF' ] )
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfMultiTopologyCmd( CliCommand.CliCommandClass ):
   syntax = 'isis multi-topology address-family AF [ unicast ]'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'multi-topology' : 'Set multi topology IS-IS interface config',
      'address-family' : 'Enable interface address family',
      'AF' : afMatcher,
      'unicast' : unicastKw
   }
   handler = setIsisIntfMtAf
   noOrDefaultHandler = noIsisIntfMtAf

modelet.addCommandClass( IsisIntfMultiTopologyCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis circuit-type LEVEL"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfCircuitType( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.circuitType = LVL_KW_TO_TAC_TYPE_MAP[ args[ 'LEVEL' ] ]

def noIsisIntfCircuitType( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.circuitType = intfConfig.circuitTypeDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfCircuitTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'isis circuit-type LEVEL'
   noOrDefaultSyntax = 'isis circuit-type ...'
   data = {
      'isis' : isisKw,
      'circuit-type' : 'Set level of the interface',
      'LEVEL' : level12MatcherForConfig,
   }
   handler = setIsisIntfCircuitType
   noOrDefaultHandler = noIsisIntfCircuitType

modelet.addCommandClass( IsisIntfCircuitTypeCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis area proxy boundary"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfAreaProxyBoundary( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.areaProxyBoundary = True

def noIntfAreaProxyBoundary( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if not intfConfig:
      return
   intfConfig.areaProxyBoundary = intfConfig.areaProxyBoundaryDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfAreaProxyBoundaryCmd( CliCommand.CliCommandClass ):
   syntax = 'isis area proxy boundary'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'area' : 'Set area parameters',
      'proxy' : 'Set area proxy parameters',
      'boundary' : 'Set area proxy boundary'
   }
   handler = setIntfAreaProxyBoundary
   noOrDefaultHandler = noIntfAreaProxyBoundary

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   modelet.addCommandClass( IsisIntfAreaProxyBoundaryCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis priority"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfPriority( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.priority = args[ 'PRIORITY_VALUE' ]

def noIsisIntfPriority( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.priority = intfConfig.priorityDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'isis priority PRIORITY_VALUE'
   noOrDefaultSyntax = 'isis priority ...'
   data = {
      'isis' : isisKw,
      'priority' : 'DR election priority',
      'PRIORITY_VALUE' : CliMatcher.IntegerMatcher(
            Priority.isisPriorityMin,
            Priority.isisPriorityMax,
            helpdesc='Priority value' ),
   }
   handler = setIsisIntfPriority
   noOrDefaultHandler = noIsisIntfPriority

modelet.addCommandClass( IsisIntfPriorityCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis hello-interval <interval>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisHelloInterval( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.helloInterval = args[ 'INTERVAL' ]

def noIsisHelloInterval( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloInterval = intfConfig.helloIntervalDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'isis hello-interval INTERVAL'
   noOrDefaultSyntax = 'isis hello-interval ...'
   data = {
      'isis' : isisKw,
      'hello-interval' : 'Set hello interval',
      'INTERVAL' : CliMatcher.IntegerMatcher(
            HelloInterval.min,
            HelloInterval.max,
            helpdesc='Hello interval value in seconds' ),
   }
   handler = setIsisHelloInterval
   noOrDefaultHandler = noIsisHelloInterval

modelet.addCommandClass( IsisIntfHelloIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis hello-multiplier <multiplier>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisHelloMultiplier( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.helloMultiplier = args[ 'MULTIPLIER' ]

def noIsisHelloMultiplier( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloMultiplier = intfConfig.helloMultiplierDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfHelloMultiplierCmd( CliCommand.CliCommandClass ):
   syntax = 'isis hello-multiplier MULTIPLIER'
   noOrDefaultSyntax = 'isis hello-multiplier ...'
   data = {
      'isis' : isisKw,
      'hello-multiplier' : 'Set hello multiplier for hold time',
      'MULTIPLIER' : CliMatcher.IntegerMatcher(
            HelloMultiplier.min,
            HelloMultiplier.max,
            # FIXME: Is this help correct? Aren't multipliers unitless?
            helpdesc='Hello multiplier value in seconds' ),
   }
   handler = setIsisHelloMultiplier
   noOrDefaultHandler = noIsisHelloMultiplier

modelet.addCommandClass( IsisIntfHelloMultiplierCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis lsp tx interval <interval>"
# command, in config-if mode.
#
# legacy:
# "[no|default] isis lsp-interval <interval>"
# command, in config-if mode.
#-------------------------------------------------------------------------------
class RouterIsisIntfLspTxIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ( ( lsp tx interval ) | lsp-interval ) INTERVAL'
   noOrDefaultSyntax = 'isis ( ( lsp tx interval ) | ( lsp-interval ) ) ...'
   data = {
      'isis' : isisKw,
      'lsp-interval' : CliCommand.Node( matcher=isisLspDeprecatedKw,
                                        deprecatedByCmd='isis lsp tx interval' ),
      'lsp' : lspKw,
      'tx' : txKw,
      'interval' : intervalKw,
      'INTERVAL' : lspIntervalMatcher
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      intfConfig.lspInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      intfConfig.lspInterval = intfConfig.lspIntervalDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

modelet.addCommandClass( RouterIsisIntfLspTxIntervalCmd )

#-------------------------------------------------------------------------------
#"[no|default] isis passive
# command in config-if mode
#-------------------------------------------------------------------------------
def setIntfPassive( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.passive = True 

def noIntfPassive( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
   intfConfig.passive = intfConfig.passiveDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfPassiveCmd( CliCommand.CliCommandClass ):
   syntax = 'isis passive'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'passive' : 'Include interface but without actively running IS-IS'
   }
   handler = setIntfPassive
   noOrDefaultHandler = noIntfPassive

modelet.addCommandClass( IsisIntfPassiveCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis network point-to-point"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfType( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.interfaceType = InterfaceType.interfaceTypeP2P

def noIsisIntfType( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return 
  
   intfConfig.interfaceType = intfConfig.interfaceTypeDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfNetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'isis network point-to-point'
   noOrDefaultSyntax = 'isis network ...'
   data = {
      'isis' : isisKw,
      'network' : 'Set the network type',
      'point-to-point' : 'Network type point to point'
   }
   handler = setIsisIntfType
   noOrDefaultHandler = noIsisIntfType

modelet.addCommandClass( IsisIntfNetworkCmd )

#---------------------------------------------------------------------------------
# [no] isis authentication mode [ < text|md5|sha key-id <keyid> > [rx-disabled]
# [ LEVEL ] ]
# in intf config mode
#---------------------------------------------------------------------------------
class IsisIntfModeAuthenticationMode( CliCommand.CliCommandClass ):
   syntax = 'isis authentication mode AUTHMODE [ rx-disabled ] [ LEVEL ]'
   noOrDefaultSyntax = '''isis authentication mode
                          [ AUTHMODE [ rx-disabled ] [ LEVEL ] ]'''
   data = {
      'isis' : isisKw,
      'authentication' : authKw,
      'mode' : modeKw,
      'AUTHMODE' : AuthenticationModeExpression,
      'rx-disabled' : rxDisabledKw,
      'LEVEL' : levelMatcherForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      authModeVal, keyid, authRxDisabled, authLevel = getAuthModeData( args )
      setAuthModeCommon( intfConfig, authLevel, authModeVal, keyid, authRxDisabled )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      authModeVal, keyid, _, authLevel = getAuthModeData( args )
      noAuthModeCommon( intfConfig, authLevel, authModeVal, keyid )
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

modelet.addCommandClass( IsisIntfModeAuthenticationMode )

#
#---------------------------------------------------------------------------------
# [no] isis authentication [key-id <id> algorithm SHA ]
#       key [ [0|7] < key-string > [ LEVEL ] ]
#---------------------------------------------------------------------------------
class RouterIsisIntfAuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = authKeyIntfSyntax
   noOrDefaultSyntax = noOrDefaultAuthIntfSyntax
   data = authIntfData

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      keyid, authLevel, algorithm, authKey, shaApadRfc5310 = getAuthKeyData( args )
      setAuthKeyCommon( mode, intfConfig, authLevel, authKey, keyid, algorithm,
                        shaApadRfc5310 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      keyid, authLevel, _, _, _ = getAuthKeyData( args )
      noAuthKeyCommon( intfConfig, authLevel, keyid )
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

modelet.addCommandClass( RouterIsisIntfAuthenticationKeyCmd )

# Helper function to check for a conflicting prefix/proxy/node segment
def _isisSrConflictingPrefixFound( instanceName, intfName, prefix,
                                   proxy=False, mode=None ):
   if mode and mode.session_.skipConfigCheck():
      return False
   if proxy:
      segType = 'proxy'
   elif intfName:
      segType = 'node'
   else:
      segType = 'prefix'

   # If we find a prefix segment with the same key as 'prefix'
   # it's a conflict
   instanceConfig = isisConfig.instanceConfig.get( instanceName )
   if instanceConfig:
      value = instanceConfig.srPrefixSegments.get( prefix )
      if value is not None:
         if value.isProxyNode:
            nodeType = 'proxy'
         else:
            nodeType = 'prefix'
         if nodeType != segType:
            return True

   # If we find a node segment on an interface and the IP address
   # on that interface is same as 'prefix' it's a conflict
   for interface, intfConfig in isisConfig.intfConfig.items():
      if intfName and intfName == interface:
         continue
      if intfConfig.instanceName == intfConfig.instanceNameInvalid or \
             intfConfig.instanceName != instanceName:
         continue
      # Check if the interface has a corresponding node-segment configuration,
      # if yes check if the prefixes are same (this is to identify the case
      # where we are trying to configure a prefix-segment, but there is already
      # a node-segment corresponding to the same prefix)
      srIntfConfig = _getSrIntfConfig( interface )
      if srIntfConfig and \
         srIntfConfig.srNodeSegmentIndex != srSidInvalid and \
         prefix.af == ArnetAddressFamily.ipv4:
         ipIntfConfig = ipConfig.ipIntfConfig.get( interface )
         if ipIntfConfig and \
                ipIntfConfig.addrWithMask.subnet.stringValue == prefix.stringValue:
            return True
      if srIntfConfig and \
         srIntfConfig.srV6NodeSegmentIndex != srSidInvalid and \
         prefix.af == ArnetAddressFamily.ipv6:
         ip6IntfConfig = ip6Config.intf.get( interface )
         if ip6IntfConfig and \
                Arnet.Ip6AddrWithMask( str( prefix ) ) in ip6IntfConfig.addr:
            return True
   return False

# Helper function to check for a conflicting SR SID
def _isisSrConflictingSidFound( instanceName, intfName, prefix, sid, isValue=False,
                                mode=None ):
   
   if mode and mode.session_.skipConfigCheck():
      return False
   # If we find a prefix segment with key other than 'prefix'
   # using this SID, it's a conflict
   instanceConfig = isisConfig.instanceConfig.get( instanceName )
   if instanceConfig:
      for key, value in instanceConfig.srPrefixSegments.items():
         if key != prefix and value.index == sid and value.isValue == isValue:
            return True
   # If we find a node segment on an interface using this
   # index with any ISIS instance, it's a conflict regardless
   # of the IP address on that interface
   if instanceName == '':
      return False
   for interface, intfConfig in isisConfig.intfConfig.items():
      if intfName and intfName == interface:
         continue
      if intfConfig.instanceName == intfConfig.instanceNameInvalid or \
             intfConfig.instanceName != instanceName:
         continue
      # If the interface has node-segment configuration check for sid conflicts
      srIntfConfig = _getSrIntfConfig( interface )
      if srIntfConfig:
         if srIntfConfig.srNodeSegmentIndex == sid or \
            srIntfConfig.srV6NodeSegmentIndex == sid:
            return True
   return False

#-------------------------------------------------------------------------------
# "[no|default] isis bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------
isisIntfBfdMatcher = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Enable BFD' )

def setIntfBfdV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.bfdV4 = IsisIntfBfdConfig.isisIntfBfdEnabled

def noIntfBfdV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if CliCommand.isDefaultCmd( args ):
      intfConfig.bfdV4 = intfConfig.bfdV4Default
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig,
                                                   mode.intf.name )
   else:
      intfConfig.bfdV4 = IsisIntfBfdConfig.isisIntfBfdDisabled

class IsisIntfBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'isis bfd'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'bfd' : isisIntfBfdMatcher
   }
   handler = setIntfBfdV4
   noOrDefaultHandler = noIntfBfdV4

modelet.addCommandClass( IsisIntfBfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis ipv6 bfd"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIntfBfdV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfConfig.bfdV6 = IsisIntfBfdConfig.isisIntfBfdEnabled

def noIntfBfdV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if CliCommand.isDefaultCmd( args ):
      intfConfig.bfdV6 = intfConfig.bfdV6Default
      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig,
                                                   mode.intf.name )
   else:
      intfConfig.bfdV6 = IsisIntfBfdConfig.isisIntfBfdDisabled

class IsisIntfIPV6BfdCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ipv6 bfd'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'ipv6' : isisIntfIpv6Matcher,
      'bfd' : isisIntfBfdMatcher
   }
   handler = setIntfBfdV6
   noOrDefaultHandler = noIntfBfdV6

modelet.addCommandClass( IsisIntfIPV6BfdCmd )

def checkV4NodeSegmentConflict( intfName, instanceName, index ):
   ipIntfConfig = ipConfig.ipIntfConfig.get( intfName )

   if ipIntfConfig is not None:
      prefix = Arnet.IpGenPrefix( str( ipIntfConfig.addrWithMask.subnet ) )
      if _isisSrConflictingPrefixFound( instanceName, intfName,
                                        prefix, mode=None ):
         return ( False,
                  "Node-segment conflicts with a prefix/proxy segment" \
                  " configured for the prefix %s" % prefix )

   if _isisSrConflictingSidFound( instanceName, intfName, None, index ):
      return ( False,
               "Two prefixes cannot be configured with the same SID" )

   return ( True, None )

def checkV6NodeSegmentConflict( intfName, instanceName, index ):
   ip6IntfConfig = ip6Config.intf.get( intfName )
   if ip6IntfConfig:
      for prefix in ip6IntfConfig.addr:
         prefix = Arnet.IpGenPrefix( str( prefix ) )
         if _isisSrConflictingPrefixFound( instanceName, intfName,
                                           prefix, mode=None ):
            return ( False,
                     "Node-segment conflicts with a prefix/proxy segment" \
                     " configured for the prefix %s" % prefix )

   if _isisSrConflictingSidFound( instanceName, intfName, None, index ):
      return ( False,
               "Two prefixes cannot be configured with the same SID" )

   return ( True, None )

def checkNodeSegmentConflict( intfName, index, ipv6=False ):
   intfConfig = _getIntfConfig( intfName )
   if intfConfig:
      instanceName = intfConfig.instanceName
   else:
      instanceName = invalidInstanceName

   if not ipv6:
      return checkV4NodeSegmentConflict( intfName, instanceName, index )
   else:
      return checkV6NodeSegmentConflict( intfName, instanceName, index )

#-------------------------------------------------------------------------------
# "[no] advertise high-metrics" command, in isis config mode.
#-------------------------------------------------------------------------------
advertiseMatcher = CliMatcher.KeywordMatcher( 'advertise',
                              helpdesc='Control routing advertisements' )

class RouterIsisAdvertiseHighMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'advertise high-metrics'
   noOrDefaultSyntax = syntax
   data = {
      'advertise' : advertiseMatcher,
      'high-metrics' : 'Advertise high metric value on all interfaces'
   }
   handler = RouterIsisMode.advertiseHighMetric
   noOrDefaultHandler = RouterIsisMode.noAdvertiseHighMetric
RouterIsisMode.addCommandClass( RouterIsisAdvertiseHighMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart" command, in isis config mode
#-------------------------------------------------------------------------------
gracefulRestartKw = CliCommand.guardedKeyword( 'graceful-restart',
                               helpdesc='Enable graceful restart mode',
                               guard=isisNonDefaultInstanceGuard )

class RouterIsisGracefulRestartCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart' : gracefulRestartKw,
   }
   handler = RouterIsisMode.setGracefulRestart
   noOrDefaultHandler = RouterIsisMode.noGracefulRestart

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper" command, in isis config mode
#-------------------------------------------------------------------------------

class RouterIsisGracefulRestartHelperCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart-helper'
   noOrDefaultSyntax = syntax
   data = {
      'graceful-restart-helper' : 'Enable graceful restart helper',
   }
   noOrDefaultHandler = handler = RouterIsisMode.setGrHelper

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart t2" command, in isis config mode
#-------------------------------------------------------------------------------
class RouterIsisGracefulRestartT2Cmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart t2 [ LEVEL ] SECONDS'
   noOrDefaultSyntax = 'graceful-restart t2 [ LEVEL ] ...'
   data = {
      'graceful-restart' : gracefulRestartKw,
      't2' : 'Set LSP database sync wait time',
      'LEVEL' : CliMatcher.EnumMatcher( {
            'level-1' : 'Set t2 for level-1 only',
            'level-2' : 'Set t2 for level-2 only',
      } ),
      'SECONDS' : CliMatcher.IntegerMatcher(
            GrTimerT2.min,
            GrTimerT2.max,
            helpdesc='Number of seconds' ),
   }
   handler = RouterIsisMode.setGrTimerT2
   noOrDefaultHandler = handler

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartT2Cmd )

#--------------------------------------------------------------------------------
# "[no|default] graceful-restart restart-hold-time <n seconds>"
#  command in isis config mode
#--------------------------------------------------------------------------------
class RouterIsisGracefulRestartHoldTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'graceful-restart restart-hold-time SECONDS'
   noOrDefaultSyntax = 'graceful-restart restart-hold-time ...'
   data = {
      'graceful-restart' : gracefulRestartKw,
      'restart-hold-time' : 'Set hold time when restarting',
      'SECONDS' : CliMatcher.IntegerMatcher(
            GrHoldTime.min + 5,
            GrHoldTime.max,
            helpdesc='Number of seconds' ),
   }
   handler = RouterIsisMode.setGrHoldTime
   noOrDefaultHandler = handler

RouterIsisMode.addCommandClass( RouterIsisGracefulRestartHoldTimeCmd )

#--------------------------------------------------------------------------------
# "[no|default] database publish"  in 'router isis <>' mode
#--------------------------------------------------------------------------------
class RouterIsisDatabasePublishCmd( CliCommand.CliCommandClass ):
   syntax = 'database publish'
   noOrDefaultSyntax = syntax
   data = {
      'database' : 'IS-IS Link State Database',
      'publish' : 'Publish database to shared memory',
   }
   handler = RouterIsisMode.setPublishLsdb
   noOrDefaultHandler = RouterIsisMode.noPublishLsdb

RouterIsisMode.addCommandClass( RouterIsisDatabasePublishCmd )

#--------------------------------------------------------------------------------
# "[no|default] advertise passive-only"  in 'router isis <>' mode
#-------------------------------------------------------------------------------
class RouterIsisAdvertisePassiveOnlyCmd( CliCommand.CliCommandClass ):
   syntax = 'advertise passive-only'
   noOrDefaultSyntax = syntax
   data = {
      'advertise' : advertiseMatcher,
      'passive-only' : 'Only connected prefixes of passive interfaces'
   }
   handler = RouterIsisMode.setAdvertisePassiveOnly
   noOrDefaultHandler = RouterIsisMode.noSetAdvertisePassiveOnly

RouterIsisMode.addCommandClass( RouterIsisAdvertisePassiveOnlyCmd )

#-----------------------------------------------------------------------------
#   [no|default] "metric <value>" 
#   command, in 'address-family mode'
#----------------------------------------------------------------------------
# We don't support "metric maximum' because we can achieve
# the similar behavior through set overload bit
class RouterIsisAfMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'metric METRICVALUE'
   noOrDefaultSyntax = 'metric ...'
   data = {
      'metric' : 'Set IS-IS metric for all interfaces',
      'METRICVALUE' : metricValueMatcher,
   }
   handler = RouterIsisAfMode.setMetricAllIntf
   noOrDefaultHandler = RouterIsisAfMode.noMetricAllIntf

RouterIsisAfMode.addCommandClass( RouterIsisAfMetricCmd )

#-------------------------------------------------------------------------------
# "fast-reroute ti-lfa mode ( ( PROTECTION [ LEVEL ] ) | disabled )"
# in 'address-family mode'
#-------------------------------------------------------------------------------
fastRerouteData = {
      'fast-reroute' : 'Configure fast reroute',
      'ti-lfa' : 'Configure TI-LFA FRR',
      'mode' : 'Set mode',
      'PROTECTION' : CliMatcher.EnumMatcher( {
            'link-protection' : 'Protect against the failure of the link',
            'node-protection' : 'Protect against the failure of the neighbor node',
      } ),
      'LEVEL' : CliMatcher.EnumMatcher( {
            'level-1' : 'Protect prefixes in level-1 only',
            'level-2' : 'Protect prefixes in level-2 only',
      } ),
      'disabled' : 'Disable protection over the link',
}
frrData = dict( fastRerouteData )

class RouterIsisAfFastRerouteCmd( CliCommand.CliCommandClass ):
   syntax = '''fast-reroute ti-lfa mode
               ( ( PROTECTION [ LEVEL ] )
               | ( disabled ... ) )'''
   data = fastRerouteData
   handler = RouterIsisAfMode.frrProtection

RouterIsisAfMode.addCommandClass( RouterIsisAfFastRerouteCmd )

#-------------------------------------------------------------------------------
# "[no] fast-reroute ti-lfa srlg [strict]
# in 'address-family mode'
#-------------------------------------------------------------------------------
fastRerouteSrlgData = {
   'fast-reroute' : 'Configure fast reroute',
   'ti-lfa' : 'Configure TI-LFA FRR',
   'srlg' : "Exclude same SRLG links from backup path",
   'strict': "Apply strict SRLG constraint"
}

class RouterIsisAfFrrSrlgCmd( CliCommand.CliCommandClass ):
   syntax = '''fast-reroute ti-lfa srlg [ strict ]'''
   noOrDefaultSyntax = '''fast-reroute ti-lfa srlg ...'''

   data = fastRerouteSrlgData
   handler = RouterIsisAfMode.frrSrlgProtection
   noOrDefaultHandler = RouterIsisAfMode.noFrrSrlgProtection
   
if gatedToggleLib.toggleTilfaSrlgEnabled():
   RouterIsisAfMode.addCommandClass( RouterIsisAfFrrSrlgCmd )

#-------------------------------------------------------------------------------
# [no|default] igp shortcut disabled
# in 'address family mode'
#-------------------------------------------------------------------------------

disabledKw = CliCommand.guardedKeyword( 'disabled',
                                        helpdesc="Disable IGP shortcut",
                                        guard=isisNonDefaultVrfGuard )

class RouterIsisAfIgpShortcutCmd( CliCommand.CliCommandClass ):
   syntax = "igp shortcut disabled"
   noOrDefaultSyntax = "igp shortcut disabled"
   data = {
      "igp" : "IGP configuration",
      "shortcut": "IGP shortcut configuration for IS-IS instance",
      "disabled": disabledKw
   }

   handler = RouterIsisAfMode.igpShortcutDisableIs
   noOrDefaultHandler = RouterIsisAfMode.igpShortcutDisableDel

if gatedToggleLib.toggleIsisIgpShortcutEnabled():
   RouterIsisAfIpv4Modelet.addCommandClass( RouterIsisAfIgpShortcutCmd )

#-------------------------------------------------------------------------------
# "[no|default] set-overload-bit
#                  [on-startup <<delay> | wait-for-bgp [timeout <delay>]> ]"
# command, in isis config mode
#-------------------------------------------------------------------------------
class RouterIsisSetOverloadBitCmd( CliCommand.CliCommandClass ):
   syntax = '''set-overload-bit
               [ on-startup ( DELAY | ( wait-for-bgp [ timeout DELAY ] ) ) ]'''
   noOrDefaultSyntax = 'set-overload-bit ...'
   data = {
      'set-overload-bit' : 'Control the overload bit settings',
      'on-startup' : 'Overload at startup for <time> seconds or '
                     'wait for BGP convergence',
      'wait-for-bgp' : 'Wait for BGP convergence',
      'timeout' : 'Timeout in seconds for BGP convergence',
      'DELAY' : CliMatcher.IntegerMatcher(
            1,
            OverloadStartupDelay.max,
            helpdesc='Number of seconds' ),
   }

   handler = RouterIsisMode.setOverloadBit
   noOrDefaultHandler = RouterIsisMode.noSetOverloadBit

RouterIsisMode.addCommandClass( RouterIsisSetOverloadBitCmd )

#-------------------------------------------------------------------------------
# "[no|default] isis hello padding"
# command, in config-if mode.
#-------------------------------------------------------------------------------
def setIsisIntfHelloPadding( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   if CliCommand.isNoCmd( args ):
      intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingOff
   else:
      intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingOn

def defaultIsisIntfHelloPadding( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   intfConfig.helloPadding = IntfHelloPadding.intfHelloPaddingDefault
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfHelloPaddingCmd( CliCommand.CliCommandClass ):
   syntax = 'isis hello padding'
   noOrDefaultSyntax = syntax
   data = {
      'isis' : isisKw,
      'hello' : 'Configure hello packets on this interface',
      'padding' : 'Configure hello padding on this interface'
   }
   handler = setIsisIntfHelloPadding
   noHandler = handler
   defaultHandler = defaultIsisIntfHelloPadding

modelet.addCommandClass( IsisIntfHelloPaddingCmd )

#---------------------------------------------------------------------------------
# [no|default] adjacency-segment AF p2p [multiple] 
# {<label <val>| index <val> global>} [backup-eligible]
#---------------------------------------------------------------------------------
def getAdjSidKey( addrFamily, labelType, value ):
   sidVal = value
   isIpV6 = addrFamily == "ipv6"
   isValue = labelType == 'label'

   sid = SegmentIdentifier( sidVal, isValue )
   key = AdjacencySegmentKey( sid, isIpV6 )
   return key

def setStaticAdjSid( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   addrFamily = args[ 'AF' ]
   labelType = args.get( 'label', 'index' )
   value = args.get( 'LABEL_VALUE' ) or args.get( 'INDEX_VALUE' )
   backup = args.get( 'backup-eligible' )
   key = getAdjSidKey( addrFamily, labelType, value )

   if 'multiple' not in args:
      if key in  intfConfig.srSingleAdjacencySegment:
         sidEntry = intfConfig.srSingleAdjacencySegment[ key ]
      else:
         sidEntry = intfConfig.newSrSingleAdjacencySegment( key )
      sidEntry.backup = bool( backup )

      isIpV6 = addrFamily == "ipv6"
      # delete old adj-SID (if any)
      for adjSidKey in intfConfig.srSingleAdjacencySegment:
         if adjSidKey != key and adjSidKey.isIpV6 == isIpV6:
            del intfConfig.srSingleAdjacencySegment[ adjSidKey ]
   else:
      if key in  intfConfig.srMultipleAdjacencySegment:
         sidEntry = intfConfig.srMultipleAdjacencySegment[ key ]
      else:
         sidEntry = intfConfig.newSrMultipleAdjacencySegment( key )
      sidEntry.backup = bool( backup )

def noStaticAdjSid( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return

   addrFamily = args[ 'AF' ]
   labelType = args.get( 'label', 'index' )
   if 'multiple' not in args:
      isIpV6 = addrFamily == "ipv6"
      isValue = labelType == "label"
      for adjSidKey in intfConfig.srSingleAdjacencySegment:
         if adjSidKey.isIpV6 == isIpV6 and adjSidKey.sid.isValue == isValue:
            del intfConfig.srSingleAdjacencySegment[ adjSidKey ]
   else:
      value = args.get( 'LABEL_VALUE' ) or args.get( 'INDEX_VALUE' )
      key = getAdjSidKey( addrFamily, labelType, value )
      del intfConfig.srMultipleAdjacencySegment[ key ]

   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfAdjacencySegmentConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'adjacency-segment AF p2p [ multiple ] ' \
            '( label LABEL_VALUE ) | ( index INDEX_VALUE global )' \
            '[ backup-eligible ]'
   noOrDefaultSyntax = 'adjacency-segment AF p2p ( ( multiple ' \
                       '( label LABEL_VALUE ) | ( index INDEX_VALUE global ) )' \
                       '| ( label | index ) ) ...'
   data = {
      'adjacency-segment' : adjacencySegmentKw,
      'AF' : afMatcher,
      'p2p' : p2pKw,
      'multiple' : multipleKw,
      'label' : labelKw,
      'LABEL_VALUE' : labelValueMatcher,
      'index' : indexKw,
      'INDEX_VALUE' : indexValueMatcher,
      'global' : globalKw, 
      'backup-eligible' : backupKw,
   }
   handler = setStaticAdjSid
   noOrDefaultHandler = noStaticAdjSid

modelet.addCommandClass( IsisIntfAdjacencySegmentConfigCmd )

#------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] route-tag <tag-no>   XXX: Hidden
#------------------------------------------------------------------------------
isisIntfIpv4Matcher = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='IS-IS IPv4 interface config' )
isisIntfRouteTagMatcher = CliMatcher.KeywordMatcher( 'route-tag',
      helpdesc='Set a route-tag for the interface' )
isisIntfRouteTagValueMatcher = CliMatcher.IntegerMatcher(
      RouteTag.isisRouteTagMin,
      RouteTag.isisRouteTagMax,
      helpdesc='Route Tag Value' )

def setIsisIntfRouteTagV4( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfRouteTagVal = args[ 'TAG_VALUE' ]
   #Setting a route tag to 0 removes the tag
   if intfRouteTagVal == RouteTag.isisRouteTagMin:
      noIsisIntfRouteTagV4( mode, args )
      return
   intfConfig.routeTagV4 = intfRouteTagVal

def noIsisIntfRouteTagV4( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.routeTagV4 = intfConfig.routeTagNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

def setIsisIntfRouteTagV6( mode, args ):
   intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
   intfRouteTagVal = args[ 'TAG_VALUE' ]
   #Setting a route tag to 0 removes the tag
   if intfRouteTagVal == RouteTag.isisRouteTagMin:
      noIsisIntfRouteTagV6( mode, args )
      return
   intfConfig.routeTagV6 = intfRouteTagVal

def noIsisIntfRouteTagV6( mode, args ):
   intfConfig = _getIntfConfig( mode.intf.name )
   if intfConfig is None:
      return
   intfConfig.routeTagV6 = intfConfig.routeTagNotConfigured
   _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

class IsisIntfIPV4RouteTagCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ipv4 route-tag TAG_VALUE'
   noOrDefaultSyntax = 'isis ipv4 route-tag ...'
   data = {
      'isis' : isisKw,
      'ipv4' : CliCommand.Node( matcher=isisIntfIpv4Matcher, hidden=True ),   
      'route-tag' : isisIntfRouteTagMatcher,
      'TAG_VALUE' : isisIntfRouteTagValueMatcher
   }
   handler = setIsisIntfRouteTagV4
   noOrDefaultHandler = noIsisIntfRouteTagV4

modelet.addCommandClass( IsisIntfIPV4RouteTagCmd )

class IsisIntfIPV6RouteTagCmd( CliCommand.CliCommandClass ):
   syntax = 'isis ipv6 route-tag TAG_VALUE'
   noOrDefaultSyntax = 'isis ipv6 route-tag ...'
   data = {
      'isis' : isisKw,
      'ipv6' : isisIntfIpv6Matcher,
      'route-tag' : CliCommand.Node( matcher=isisIntfRouteTagMatcher, hidden=True ),
      'TAG_VALUE' : isisIntfRouteTagValueMatcher
   }
   handler = setIsisIntfRouteTagV6
   noOrDefaultHandler = noIsisIntfRouteTagV6

modelet.addCommandClass( IsisIntfIPV6RouteTagCmd )

#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa mode ( {link-protection |
# node-protection} [ LEVEL ] | disabled )
#-------------------------------------------------------------------------------
class IsisIntfFrrProtectionCmd( CliCommand.CliCommandClass ):
   syntax = '''isis [ ipv4 | ipv6 ] fast-reroute ti-lfa mode
               ( ( PROTECTION [ LEVEL ] ) | disabled )'''
   noOrDefaultSyntax = 'isis [ ipv4 | ipv6 ] fast-reroute ti-lfa mode ...'
   data = frrData
   _intfData = {
      'isis': isisKw,
      'ipv4' : isisIntfIpv4Matcher,
      'ipv6' : isisIntfIpv6Matcher,
   }
   data.update( _intfData )

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      protection = args.get( 'PROTECTION' )
      protectionLevel = args.get( 'LEVEL', 'level-1-2' )
      frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )

      protectionConfig = ProtectionConfig( getProtectionMode( protection ) )
      protectionConfig.level = LVL_KW_TO_TAC_TYPE_MAP[ protectionLevel ]
      if frrAf:
         if frrAf == 'ipv4':
            intfConfig.protectionConfigV4 = protectionConfig
         else:
            intfConfig.protectionConfigV6 = protectionConfig
      else:
         intfConfig.protectionConfigV4V6 = protectionConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
      protectionConfig = ProtectionConfig( ProtectionMode.protectionDefault )
      if frrAf:
         if frrAf == 'ipv4':
            intfConfig.protectionConfigV4 = protectionConfig
         else:
            intfConfig.protectionConfigV6 = protectionConfig
      else:
         intfConfig.protectionConfigV4V6 = protectionConfig

      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

modelet.addCommandClass( IsisIntfFrrProtectionCmd )

#-------------------------------------------------------------------------------
# [no|default] isis [ipv4|ipv6] fast-reroute ti-lfa srlg [strict] [disabled]
#-------------------------------------------------------------------------------
class IsisIntfFrrSrlgCmd( CliCommand.CliCommandClass ):
   syntax = '''isis [ ipv4 | ipv6 ] fast-reroute ti-lfa srlg [ strict ]
               [ disabled ]'''
   noOrDefaultSyntax = 'isis [ ipv4 | ipv6 ] fast-reroute ti-lfa srlg ...'
   data = dict( fastRerouteSrlgData ) 
   _intfData = {
      'isis': isisKw,
      'ipv4' : isisIntfIpv4Matcher,
      'ipv6' : isisIntfIpv6Matcher,
      'disabled' : "Disable SRLG protection for the link"
   }
   data.update( _intfData )

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( isisConfiguration(), mode.intf.name )
      srlg = 'strict' if 'strict' in args else 'loose'
      srlg = 'disabled' if 'disabled' in args else srlg
      srlgVal = getSrlgProtection( srlg )
      frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
      if frrAf:
         if frrAf == 'ipv4':
            intfConfig.protectionSrlgV4 = srlgVal
         else:
            intfConfig.protectionSrlgV6 = srlgVal
      else:
         intfConfig.protectionSrlgV4V6 = srlgVal

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      frrAf = args.get( 'ipv4' ) or args.get( 'ipv6' )
      if frrAf:
         if frrAf == 'ipv4':
            intfConfig.protectionSrlgV4 = intfConfig.protectionSrlgDefault
         else:
            intfConfig.protectionSrlgV6 = intfConfig.protectionSrlgDefault
      else:
         intfConfig.protectionSrlgV4V6 = intfConfig.protectionSrlgDefault

      _deleteIntfConfigIfAllAttributeHaveDefaults( isisConfig, mode.intf.name )

if gatedToggleLib.toggleTilfaSrlgEnabled():
   modelet.addCommandClass( IsisIntfFrrSrlgCmd )

#-------------------------------------------------------------------------------
# New Cli Parser
#-------------------------------------------------------------------------------
def isisShowCommandSyntaxWrapper( cmd ):
   '''
   IS-IS show command CLIs can be of two forms:
   show isis [ INSTANCE_NAME ] COMMAND [ ARGS ]
   or
   show isis COMMAND [ ARGS ] [ vrf VRF_NAME ]

   If instance name is specified, we show the information for that instance.
   We do not allow both instance name and vrf name to be specified.
   '''
   template = 'show isis ( %s ) | ( INSTANCE %s ) | ( %s vrf VRFNAME )'
   return template % ( cmd, cmd, cmd )

def isisClearCommandSytaxWrapper( cmd ):
   '''
   IS-IS clear command CLIs can be of two forms:
   clear isis [ INSTANCE_NAME ] COMMAND [ ARGS ]
   or
   clear isis COMMAND [ ARGS ] [ vrf VRF_NAME ]

   If instance name is specified, we clear the information for that instance.
   We do not allow both instance name and vrf name to be specified.
   '''
   template = "clear isis ( [ INSTANCE | ALL_INSTANCE ] %s ) | ( %s vrf VRFNAME )"
   return template % ( cmd, cmd )

isisInstanceMatcher = CliMatcher.DynamicKeywordMatcher(
   lambda mode : isisConfig.instanceConfig,
   priority=CliParser.PRIO_HIGH,
   emptyTokenCompletion=[ CliParser.Completion( 'NAME',
                                               'Name of the IS-IS protocol instance',
                                               literal=False ) ] )

allInstanceMatcher = CliMatcher.KeywordMatcher(
   'all', helpdesc='Clear the state in all IS-IS instances in the default VRF',
   priority=CliParser.PRIO_LOW )

vrfNameMatcher = CliMatcher.DynamicNameMatcher(
   VrfCli.getAllPlusReservedVrfNames, helpdesc='VRF name' )

def showIsisCommand( *args, **kwargs ):
   '''
   args - Contains CLI mode and callback function
   kwargs - Contains the cmdVrfModel and instDictModelType for the Capi compatible
          commands.

   instDictModelType - Instance models for CAPI compatible commands
   cmdVrfModel - cmd VRF model for CAPI compatible commands

   If vrf name is specified show the information for the Isis instance in that VRF.
   If it doesn't exist report error.
   We do not allow both instance name and vrf name to be specified
   If neither instance name nor vrf name are specified use the VRF name from the
   routing context if it is set. If routing context is not set and there is an
   Isis instance configured in the default VRF, then show information for that
   instance. If there is no Isis instance in the default VRF but there is/are Isis
   instance(s) in non-default VRF(s), show information for one of those instances.
   Sort them by instance name and show the status for first one.
   Otherwise report error.
   '''

   mode = args[ 0 ]

   # callback will be the last entry in *args
   callback = args[ -1 ]
   args = args[ : -1 ]

   # RibCapiLib will not accept these extra arguments - Need to remove them
   for keyword in ( 'show', 'isis', 'vrf' ):
      kwargs.pop( keyword, None )

   # Some callback functions take 'level' as keyword argument
   # and not as 'level-1' or 'level-2'
   level = kwargs.pop( 'LEVEL', None )
   if level is not None:
      kwargs[ 'level' ] = level

   instDictModelType = kwargs.pop( 'instDictModelType', None )
   cmdVrfModel = kwargs.pop( 'cmdVrfModel', None )
   instanceName = kwargs.pop( 'INSTANCE', None )
   vrfName = kwargs.pop( 'VRFNAME', None )

   useCliPrint = kwargs.pop( 'useCliPrint', False )
   if useCliPrint:
      showVrfModelRoutine = isisShowVrfModelUsingCliPrint
   else:
      showVrfModelRoutine = isisShowVrfModel

   sortedInstanceList = [ isisConfig.instanceConfig[ iName ] for iName in 
                         sorted( isisConfig.instanceConfig ) ]

   vrfs = { inst.vrfName for inst in  sortedInstanceList }

   instanceList = sortedInstanceList
   vrfSet = vrfs

   if instanceName:
      instance = isisConfig.instanceConfig[ instanceName ]
      instanceList = [ instance ]
      vrfSet = { instance.vrfName }
   elif vrfName:
      # vrf name is specified
      if vrfName != ALL_VRF_NAME:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
            return
         else:
            vrfSet = { vrfName }
   else:
      # If vrf name and instance name isn't specified
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName not in vrfs:
            mode.addError( "No IS-IS instance is configured in routing-context "
                           "VRF: %s" % vrfName )
            return
         vrfSet = { vrfName }
      elif DEFAULT_VRF in vrfs:
         vrfSet = { DEFAULT_VRF }
      else:
         if not isisConfig.instanceConfig:
            return cmdVrfModel() if cmdVrfModel else None
         vrfSet = { sortedInstanceList[0].vrfName }
   # Some vrf name is always set.

   if cmdVrfModel:
      return showVrfModelRoutine( vrfSet, mode, cmdVrfModel, instDictModelType,
                                  callback, instanceList, *args, **kwargs)
   else:
      for instance in instanceList:
         if instance.vrfName in vrfSet:
            invokeCallback( instance, callback, *args, **kwargs )

#-------------------------------------------------------------------------------
# Show commands
#-------------------------------------------------------------------------------
def invokeCallback( instance, callback, *args, **kwargs ):
   cmdPrefix = 'show isis'
   cmd = cmdPrefix
   mode = args[0]

   isisStatus = isisStatusDir.get( instance.vrfName )
   warnings, statusValid = checkAndShowIsisConfigMismatch( mode, instance,
                                                           isisStatus )
   if warnings:
      mode.addWarning( warnings )
   if statusValid:
      # statusValid is True only if isisStatus is not None and
      # isisStatus.instanceStatus[instance.instanceName] is not None
      instanceId = isisStatus.instanceStatus[
            instance.instanceName ].instanceId
      # The actual show command handlers don't need/accept vrfName as
      # keyword arguments.
      if 'vrfName' in kwargs:
         del kwargs[ 'vrfName' ]
      kwargs[ 'cmdPrefix' ] = cmd
      kwargs[ 'instanceId' ] = instanceId
      kwargs[ 'instanceName' ] = instance.instanceName

      return callback( *args, **kwargs )

def isisShowVrfModelUsingCliPrint( vrfSet, mode, cmdVrfModel, instDictModelType,
                                   callback, instanceList, *args, **kwargs ):
   vrfInstDict = {}
   # Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []
   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            mode.addWarning( warnings )
         vrfInstDict[ instance.vrfName ] += \
                                    [ { 'instanceName' : instance.instanceName,
                                        'statusValid' :statusValid } ]

   fd = sys.stdout.fileno()
   outputFormat = cprinter.stringToOutputFormat( mode.session_.outputFormat_ )
   p = cprinter.initPrinter( fd, outputFormat, True )
   cprinter.start( p )
   cprinter.startDict( p, "vrfs" )
   for vrf in vrfSet:
      cprinter.startDict( p, vrf )
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      cprinter.startDict( p, "isisInstances" )
      for inst in vrfInstDict[ vrf ]:
         if inst[ 'statusValid' ]:
            cprinter.startDict( p, inst[ 'instanceName' ] )
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                     instanceStatus[ inst[ 'instanceName' ] ]. \
                                     instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]
            _ = callback( *args, **kwargs )
            cprinter.endDict( p ) # this instance
      cprinter.endDict( p ) # instances
      cprinter.endDict( p ) # this vrf
   cprinter.endDict( p ) # vrfs
   cprinter.end( p )
   cprinter.deinitPrinter( p )
   return cliPrinted( cmdVrfModel )

def isisShowVrfModel( vrfSet, mode, cmdVrfModel, instDictModelType, callback,
                      instanceList, *args, **kwargs ):
   def instDictFunc( vrf ):
      isisStatus = isisStatusDir.get( vrf )
      if isisStatus is None:
         return
      for inst in vrfInstDict[ vrf ]:
         instModel = None
         if inst[ 'statusValid' ]:
            kwargs[ 'vrfName' ] = vrf
            kwargs[ 'instanceId' ] = isisStatus. \
                                   instanceStatus[inst[ 'instanceName' ]]. \
                                   instanceId
            kwargs[ 'instanceName' ] = inst[ 'instanceName' ]

            # handling EmptyResponseException raised from RibCapiLib
            try:
               instModel = callback( *args, **kwargs )
            except EmptyResponseException:
               instModel = None
            except AmiResponseException:
               instModel = None
            
         if instModel is None:
            continue
         yield inst[ 'instanceName' ], instModel 
   def vrfListFunc( vrfs ):
      for vrf in vrfs:
         instDictModel = instDictModelType()
         instDictModel.isisInstances = instDictFunc( vrf )
         yield vrf, instDictModel

   vrfInstDict = {}
   #Loop for prepopulating warnings for all instances
   for vrf in vrfSet:
      vrfInstDict[ vrf ] = []

   for instance in instanceList:
      isisStatus = isisStatusDir.get( instance.vrfName )
      if instance.vrfName in vrfSet:
         warnings, statusValid = checkAndShowIsisConfigMismatch( mode,
                                                       instance, isisStatus )
         if warnings:
            mode.addWarning( warnings )
         vrfInstDict[ instance.vrfName ] += \
                                       [ { 'instanceName' : instance.instanceName,
                                           'statusValid' :statusValid } ]

   model = cmdVrfModel()
   model.vrfs = vrfListFunc( vrfSet )
   return model

def _isisConfigVrfIsNotActiveReason( config ):
   return "Vrf %s is not active" % config.vrfName

def _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status ):
   msgs = []
   if ( not status.netValid ):
      msgs.append( "IS-IS Network Entity Title (NET) configuration is not present" )

   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   vrfMessage = ''
   if config.vrfName != DEFAULT_VRF:
      vrfMessage = ' in vrf %s' % config.vrfName

   if ( status.activeAddressFamily == IsisAddressFamily.addressFamilyNone ):
      if config.addressFamily == IsisAddressFamily.addressFamilyNone:
         msgs.append( "IS-IS address family configuration is not present" )
      if config.addressFamily == IsisAddressFamily.addressFamilyIPv4 \
             and not v4RoutingStatus:
         msgs.append( "IPv4 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyIPv6 \
             and not v6RoutingStatus:
         msgs.append( "IPv6 unicast routing is not enabled%s" % vrfMessage )
      elif config.addressFamily == IsisAddressFamily.addressFamilyBoth \
             and not v4RoutingStatus and not v6RoutingStatus:
         msgs.append( "Routing is not enabled%s" % vrfMessage )
   return msgs

def _isisBothAfConfigMismatchReason( config, status, vrfStatus, vrf6Status ):
   msg = None
   v4RoutingStatus = vrfStatus and vrfStatus.routing
   v6RoutingStatus = vrf6Status and vrf6Status.routing
   if not v4RoutingStatus:
      msg = 'IS-IS IPv4 and IPv6 address family are configured but IPv4 unicast '
      msg += 'routing is not enabled for instance %s' % status.instanceName 
   if not v6RoutingStatus:
      msg = "IS-IS IPv4 and IPv6 address family are configured but IPv6 unicast "
      msg += "routing is not enabled for instance %s" % status.instanceName
   return msg

def checkAndShowIsisConfigMismatch( mode, config, isisStatus ):
   '''
   Wrapper function that needs to be used to invoke a show command handler, to show
   reason for ISIS to be disabled / not running as configured due to a config
   mismatch.
   If the config is valid it calls the show command handler function.
   '''
   # When this function is called, an instance exists. The status, however, may
   # or may not exist. The status doesn't exist in the case of the isis instance
   # living in a non-default VRF which was either not created or was deleted.
   # If the status is None, we let users know about it printing
   # Vrf <vrfName> is not active
   instanceName = config.instanceName
   warnings = ''
   configValid = None

   if isisStatus is not None and instanceName in isisStatus.instanceStatus:
      status = isisStatus.instanceStatus[ instanceName ]
      configValid = status.configValid
      vrfStatus = vrfInfoDir.get( config.vrfName )
      vrf6Status = vrf6InfoDir.get( config.vrfName )
      if not configValid:
         reasons = _isisConfigInvalidReason( config, status, vrfStatus, vrf6Status )
         for r in reasons:
            mode.addError( "IS-IS (%s) is disabled because: %s"
                           % ( instanceName, r ) )
      else:
         if config.addressFamily == IsisAddressFamily.addressFamilyBoth:
            reason = _isisBothAfConfigMismatchReason( config, status, vrfStatus,
                                                      vrf6Status )
            if reason is not None:
               warnings = ( "IS-IS (%s) has incorrect configuration: %s" % (
                  instanceName, reason ) )

   elif isisStatus is None and config.vrfName != DEFAULT_VRF:
      configValid = False
      reason = _isisConfigVrfIsNotActiveReason( config )
      mode.addError( "IS-IS (%s) is disabled because: %s"
                     % ( instanceName, reason ) )

   return warnings, configValid

# This function allows us to reverse-lookup from the LVL_KW_TO_TAC_TYPE dict.
getLevel = partial( LVL_KW_TO_TAC_TYPE_MAP.reverse().__getitem__ )

#-------------------------------------------------------------------------------
# show isis interface [interface] [detail]
#-------------------------------------------------------------------------------
def showIsisIntfWithName( mode, intf, detail=None ):
   callback = showIsisIntf
   instance = None
   # Find the instance Id of isis running on this interface
   intfConfig = _getIntfConfig( intf.name )
   if ( intfConfig is None or not intfConfig.instanceName  ):
      mode.addError( "No IS-IS instance is enabled on this interface" )
   elif intfConfig.instanceName in isisConfig.instanceConfig:
      instance = isisConfig.instanceConfig[ intfConfig.instanceName ]
   else:
      mode.addError( "IS-IS instance %s of this interface is not configured"
                     % intfConfig.instanceName )

   if instance is not None:
      args = [ mode ]
      kwargs = { "intf" : intf, "detail" : detail }
      return isisShowVrfModel( [ instance.vrfName ], mode, 
            IsisCliModels.interfaceVRFsModel,
            IsisCliModels.interfaceModel, 
            callback, [ instance ],  *args, **kwargs )

def showIsisIntf( mode, instanceId=None, instanceName=None, intf=None,
                  vrfName=None, detail=None, cmdPrefix=None ):

   def genDisabledIntfs():
      for interface in isisConfig.intfConfig:
         if intf is not None and intf.name != interface: 
            continue
         intfConfig = isisConfig.intfConfig.get( interface )
         if instanceName != intfConfig.instanceName:
            continue
         instanceKey = InstanceKey( interface, instanceName )
         intfStatus = isisStatus.intfStatus.get( instanceKey )
         if intfStatus is None or not intfStatus.instanceValid:
            continue
         instance = isisConfig.instanceConfig[ instanceName ]
         if ( not intfStatus.configValid and
              intfStatus.activeCircuitType == Level.levelNone ):
            interfaceInst = IsisCliModels.IsisInterface()
            interfaceInst._interfaceName = interface
            interfaceInst.enabled = False
            disabledReason = IsisCliModels.DisabledReason()
            disabledReason.circuitType = getLevel( intfConfig.circuitType )
            disabledReason.routerCircType = getLevel( instance.level )
            disabledReason.message = (
                  "Disabled because circuit type %s "
                  "doesn't match with router IS type %s" %
                  ( disabledReason.circuitType, disabledReason.routerCircType ) )
            interfaceInst.disabledReason = disabledReason
            interfaceInst.intfLevels = None
            yield interface, interfaceInst

   def updateIntfDataWithSysDb( interfaces ):
      for intfName, intf in interfaces:
         instanceKey = InstanceKey( intfName, instanceName )
         intfStatus = isisStatus.intfStatus.get( instanceKey )
         if not intfStatus:
            continue
         intf.fetchExtraDataFromSysDb( intfStatus )
         yield intfName, intf
         
   command = 'MIO_DGET_ISIS_CIRCUIT_SUMMARY' 
   if detail:
      command = 'MIO_DGET_ISIS_CIRCUIT_DETAIL'
   cliModel = IsisCliModels.IsisInterfaceModel
   isisStatus = isisStatusDir.get( vrfName )
   args = { 'instance': instanceId }
   if intf is not None:
      #Verify the interface is a valid interface name
      if not intf.lookup():
         return
      name = kernelDeviceName( intf.name )
      if ( name != "" ):
         args[ 'circuit' ] = name

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if isisStatus is None:
      return
   
   instModel =  showRibCapiCommand( mode, cliModel, command, args, 
         clientName="ISIS", l3Config=l3Config)

   interfaces = updateIntfDataWithSysDb(
           itertools.chain( instModel.interfaces, genDisabledIntfs() ) )
   
   instModel.interfaces = interfaces
   
   return instModel

# show isis [ INSTANCE ] interface [ detail ]
# show isis interface intf [ detail ]
class ShowIsisInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'interface [ detail ]' ) + \
            ' | ( interface intf [ detail ] )'
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'interface' : 'IS-IS interface status',
      'intf' : IntfCli.Intf.matcher,
      'detail' : detailKw,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.interfaceVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'interface' ]
      intf = args.get( 'intf', None )
      if not intf:
         args[ 'instDictModelType' ] = IsisCliModels.interfaceModel
         args[ 'cmdVrfModel' ] = IsisCliModels.interfaceVRFsModel
         return showIsisCommand( mode, showIsisIntf, **args )
      else:
         detail = args.get( 'detail', None )
         return showIsisIntfWithName( mode, intf, detail=detail )
BasicCli.addShowCommandClass( ShowIsisInterfaceCmd )

#-------------------------------------------------------------------------------
# show isis summary
#-------------------------------------------------------------------------------
def showIsisSummary( mode, instanceId=None, instanceName=None, vrfName=None ):
   command = 'MIO_DGET_ISIS_INSTANCE_SUMMARY'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   return showRibCapiCommand( mode, IsisCliModels.IsisInstanceSummaryModel, command,
                              args, clientName="ISIS", l3Config=l3Config )

class ShowIsisSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'summary' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'summary' : 'Get summary information for IS-IS',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisSummaryVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'summary' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisSummaryModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisSummaryVRFsModel
      return showIsisCommand( mode, showIsisSummary, **args )
BasicCli.addShowCommandClass( ShowIsisSummaryCmd )

#-------------------------------------------------------------------------------
# show isis database [ detail | traffic-engineering ] [ LEVEL ] [lspid]
#-------------------------------------------------------------------------------
def showIsisDatabase( mode, instanceId=None, instanceName=None, vrfName=None,
                      lspid=None, teOrDetail=None, level=None ):
   args = { 'instance': instanceId }
   if teOrDetail:
      command = 'MIO_DGET_ISIS_DATABASE_DETAIL'
   else:
      command = 'MIO_DGET_ISIS_DATABASE_SUMMARY'

   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if lspid is not None:
      args[ 'lspid' ] = lspid
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if teOrDetail == "traffic-engineering":
      args[ 'te-detail' ] = True
   if teOrDetail:
      result = showRibCapiCommand( mode, IsisCliModels.IsisLsdbDetailModel, command,
                                args, clientName="ISIS", l3Config=l3Config )
   else:
      result = showRibCapiCommand( mode, IsisCliModels.IsisLsdbSummaryModel, command,
                                   args, clientName="ISIS", l3Config=l3Config )
   return result

# Acceptable forms:
# 'show isis database [ LEVEL ] [ lspid ]'
# 'show isis database [ lspid ] [ LEVEL ]'
class ShowIsisDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'database [ { LEVEL | LSPID } ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE': isisInstanceMatcher,
      'database': dbKw,
      'LEVEL' : singleLevelNodeForShow,
      'LSPID' : singleLspidNode,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.lspDbSummaryVRFModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'database' ]
      args[ 'level' ] = args.pop( 'LEVEL', None )
      args[ 'lspid' ] = args.pop( 'LSPID', None )
      args[ 'instDictModelType' ] = IsisCliModels.lspDbSummaryModel
      args[ 'cmdVrfModel' ] = IsisCliModels.lspDbSummaryVRFModel
      return showIsisCommand( mode, showIsisDatabase, **args )

BasicCli.addShowCommandClass( ShowIsisDatabaseCmd )

# Acceptable forms:
# 'show isis database { ( detail|traffic-engineering ) [ LEVEL ] [ lspid ] }'
class ShowIsisDatabaseTeOrDetailCmd( ShowCommand.ShowCliCommandClass ):
   # This syntax must allow at least one argument after
   # 'show isis database' which can neither be only level-1
   # nor only level-2 nor only <lspid>, i.e., either 'detail'
   # or 'traffic-engineering' must be used or else it would
   # cause an ambiguity. It must also be noted that 'detail'
   # or 'traffic-engineering' may be entered in any sequence,
   # i.e., before/after <level> or <lspid> or both.

   syntax = isisShowCommandSyntaxWrapper(
         'database [ { LEVEL | LSPID } ] '
         '( detail | traffic-engineering ) [ { LEVEL | LSPID } ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'database' : CliCommand.Node( matcher=dbKw, maxMatches=1 ),
      'traffic-engineering' : 'Traffic-engineering information',
      'LEVEL' : singleLevelNodeForShow,
      'detail' : detailKw,
      'LSPID' : singleLspidNode,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.lspDbVRFModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'database' ]
      args[ 'level' ] = args.pop( 'LEVEL', None )
      args[ 'lspid' ] = args.pop( 'LSPID', None )
      args[ 'teOrDetail' ] = ( args.pop( 'traffic-engineering', None ) or
                               args.pop( 'detail', None ) )
      args[ 'instDictModelType' ] = IsisCliModels.lspDbModel
      args[ 'cmdVrfModel' ] = IsisCliModels.lspDbVRFModel
      return showIsisCommand( mode, showIsisDatabase, **args )

BasicCli.addShowCommandClass( ShowIsisDatabaseTeOrDetailCmd )

#------------------------------------------------
#  show isis lsp log
#------------------------------------------------
def showIsisLsplog( mode, instanceId=None,
                    instanceName=None , vrfName=None ):

   command = 'MIO_DGET_ISIS_LSP_LOG'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   result = showRibCapiCommand( mode, IsisCliModels.IsisLspLogTableModel, command,
                                args, clientName="ISIS", l3Config=l3Config )
   return result 

class ShowIsisLspLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'lsp log' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'lsp' : 'IS-IS LSP information',
      'log' : 'IS-IS LSP log',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.lspLogTableVRFsModel
   @staticmethod
   def handler( mode, args ):
      del args[ 'lsp' ]
      del args[ 'log' ]
      args[ 'instDictModelType' ] = IsisCliModels.lspLogTableModel
      args[ 'cmdVrfModel' ] = IsisCliModels.lspLogTableVRFsModel
      return showIsisCommand( mode, showIsisLsplog, **args )

BasicCli.addShowCommandClass( ShowIsisLspLogCmd )

#-------------------------------------------------------------------------------
# show isis neighbors [detail] [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisNeighbors( mode, instanceId=None, instanceName=None, vrfName=None,
                       level=None, detail=None ):
   command = 'MIO_DGET_ISIS_NEIGHBORS'
   args = {}
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = lev

   result = showRibCapiCommand( mode, IsisCliModels.IsisInstanceNeighbors,
                                command, args, clientName='ISIS',
                                l3Config=l3Config )

   if result:
      #Set details attribute of IsisCliModels.IsisInstanceNeighbors
      if detail:
         result._detailsPresent = True
   return result

class ShowIsisNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'neighbors [ { [ detail ] [ LEVEL ] } ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'LEVEL' : singleLevelNodeForShow,
      'neighbors' : 'Protocol neighbor details',
      'detail' : CliCommand.Node( matcher=detailKw, maxMatches=1 ),
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.showNeighborTableVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'neighbors' ]
      args[ 'level' ] = args.pop( 'LEVEL', None )
      args[ 'instDictModelType' ] = IsisCliModels.showNeighborTableModel
      args[ 'cmdVrfModel' ] = IsisCliModels.showNeighborTableVRFsModel
      return showIsisCommand( mode, showIsisNeighbors, **args )

BasicCli.addShowCommandClass( ShowIsisNeighborCmd )

#-------------------------------------------------------------------------------
# show isis network topology
#
# legacy:
# show isis topology
#-------------------------------------------------------------------------------
def showIsisTopology( mode, instanceId=None, instanceName=None, vrfName=None,
                      level=None, ldpTunneling=None ):
   command = 'MIO_DGET_ISIS_TOPOLOGY'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )
   if ldpTunneling:
      args[ 'ldp-shortcut' ] = True
   return showRibCapiCommand( mode, IsisCliModels.IsisTopologyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisNetworkTopologyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'network topology [ LEVEL ] [ ldp-tunneling ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'network' : 'IS-IS network information',
      'topology' : topologyKw,
      'LEVEL' : levelMatcherForShow,
      'ldp-tunneling' : 'Information for LDP over RSVP SPF',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisTopologyVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'network' ]
      del args[ 'topology' ]
      if args.pop( 'ldp-tunneling', None ):
         args[ 'ldpTunneling' ] = True
      args[ 'instDictModelType' ] = IsisCliModels.isisTopologyModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisTopologyVRFsModel
      return showIsisCommand( mode, showIsisTopology, **args )

BasicCli.addShowCommandClass( ShowIsisNetworkTopologyCmd )

class ShowIsisTopologyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'topology [ LEVEL ] ' )
   _deprecatedByCmd = 'show isis network topology [ LEVEL ]'
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'topology' : CliCommand.Node( matcher=topologyKw,
                                    deprecatedByCmd=_deprecatedByCmd ),
      'LEVEL' : levelMatcherForShow,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisTopologyVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'topology' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisTopologyModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisTopologyVRFsModel
      return showIsisCommand( mode, showIsisTopology, **args )

BasicCli.addShowCommandClass( ShowIsisTopologyCmd )

#-------------------------------------------------------------------------------
# show isis spf log [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisSpflog( mode, instanceId=None, vrfName=None, instanceName=None,
                    level=None ):
   command = 'MIO_DGET_ISIS_SPF_LOG'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )
   return showRibCapiCommand( mode, IsisCliModels.IsisSpfLogTableModel, command,
                              args, clientName="ISIS", l3Config=l3Config )

class ShowIsisSpfLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'spf log [ LEVEL ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'spf' : 'IS-IS SPF Information',
      'log' : 'IS-IS SPF logs',
      'LEVEL' : levelMatcherForShow,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.spfLogTableVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'spf' ]
      del args[ 'log' ]
      args[ 'instDictModelType' ] = IsisCliModels.spfLogTableModel
      args[ 'cmdVrfModel' ] = IsisCliModels.spfLogTableVRFsModel
      return showIsisCommand( mode, showIsisSpflog, **args )

BasicCli.addShowCommandClass( ShowIsisSpfLogCmd )

#-------------------------------------------------------------------------------
# show isis lsp purges [ LEVEL ]
#-------------------------------------------------------------------------------
def showIsisLspPurges( mode, instanceId=None, vrfName=None, instanceName=None,
      level=None ):
   command = 'MIO_DGET_ISIS_LSP_PURGES'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = lev
   return showRibCapiCommand( mode, IsisCliModels.IsisLspPurgesModel, command,
                              args, clientName="ISIS", l3Config=l3Config )

class ShowIsisLspPurgesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'lsp purges [ LEVEL ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'lsp' : lspKwForShow,
      'purges' : 'IS-IS purge log',
      'LEVEL' : levelMatcherForShow,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisLspPurgesVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'lsp' ]
      del args[ 'purges' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisLspPurgesModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisLspPurgesVRFsModel
      return showIsisCommand( mode, showIsisLspPurges, **args )

BasicCli.addShowCommandClass( ShowIsisLspPurgesCmd )

#-------------------------------------------------------------------------------
# show isis graceful-restart
#-------------------------------------------------------------------------------
def showIsisGracefulRestart( mode, instanceId=None, vrfName=None, 
                             instanceName=None ):
   command = 'MIO_DGET_ISIS_GRACEFUL_RESTART'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   return showRibCapiCommand( mode, IsisCliModels.IsisGracefulRestartModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisGracefulRestartCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'graceful-restart' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'graceful-restart' : 'Graceful Restart information',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisGracefulRestartVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'graceful-restart' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisGracefulRestartModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisGracefulRestartVRFsModel
      return showIsisCommand( mode, showIsisGracefulRestart, **args )
BasicCli.addShowCommandClass( ShowIsisGracefulRestartCmd )

#-------------------------------------------------------------------------------
# show isis local-convergence-delay [detail]
#-------------------------------------------------------------------------------
def showIsisUloopDelayInfo( mode, instanceId=None, vrfName=None, instanceName=None,
                            detail=None ):
   command = 'MIO_DGET_ISIS_ULOOP'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      result = showRibCapiCommand( mode, IsisCliModels.IsisUloopModel, command,
                                   args, clientName='ISIS', l3Config=l3Config )
   except EmptyResponseException:
      result = None

   if result:
      if detail:
         result._printDetails = True
   return result

class ShowIsisLocalConvergenceDelayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'local-convergence-delay [ detail ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'local-convergence-delay' : 'Micro-loop local convergence delay information',
      'detail' : detailKw,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisUloopVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'local-convergence-delay' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisUloopModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisUloopVRFsModel
      return showIsisCommand( mode, showIsisUloopDelayInfo, **args )

BasicCli.addShowCommandClass( ShowIsisLocalConvergenceDelayCmd )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] nodes [<node id>]
#-------------------------------------------------------------------------------
trailingByte = '\.[0-9a-fA-F]{2}'
trailingRegEx = trailingByte + '$'
nodeIdRegex = hostnameRegex + '(' + trailingByte + ')?'
nodeIdMatcher = CliMatcher.PatternMatcher( nodeIdRegex, helpname='node',
                                  helpdesc='Node ID in HHHH.HHHH.HHHH.HH or '
                                  'hostname.HH format' )

def showIsisDynFloodNode( mode, instanceId=None, instanceName=None, vrfName=None,
                          level=None, node=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_NODE'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   if node is not None:
      if re.search( trailingRegEx, node ) is None:
         node = node + '.00'
      args[ 'node' ] = node
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodNodeModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisDynFloodNode( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'dynamic flooding [ LEVEL ] nodes [NODENAME]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'dynamic' : 'Dynamic flooding information',
      'flooding' : 'Dynamic flooding information',
      'LEVEL' : levelMatcherForShow,
      'nodes' : 'Nodes in the flooding topology',
      'NODENAME' : nodeIdMatcher,
      'vrf' : 'vrf name',
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisDynFloodNodeVRFsModel

   @staticmethod
   def handler( mode, args ):
      for kw in ( 'dynamic', 'flooding', 'nodes' ):
         args.pop( kw, None )
      args[ 'node' ] = args.pop( 'NODENAME', None )
      args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodNodeModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodNodeVRFsModel
      return showIsisCommand( mode, showIsisDynFloodNode, **args )

BasicCli.addShowCommandClass( ShowIsisDynFloodNode )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] paths
#-------------------------------------------------------------------------------

def showIsisDynFloodPaths( mode, instanceId=None, instanceName=None, vrfName=None,
                           level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_PATH'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      lev = 1 if level == 'level-1' else 2
      args[ 'level' ] = int( lev )
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodPathsModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisDynFloodPaths( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'dynamic flooding [ LEVEL ] paths' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'dynamic' : 'Dynamic flooding information',
      'flooding' : 'Dynamic flooding information',
      'LEVEL' : levelMatcherForShow,
      'paths' : 'Paths in the flooding topology',
      'vrf' : 'vrf name',
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisDynFloodPathsVRFsModel

   @staticmethod
   def handler( mode, args ):
      for kw in ( 'dynamic', 'flooding', 'paths' ):
         args.pop( kw, None )
      args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodPathsModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodPathsVRFsModel
      return showIsisCommand( mode, showIsisDynFloodPaths, **args )

BasicCli.addShowCommandClass( ShowIsisDynFloodPaths )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] topology
#-------------------------------------------------------------------------------

def showIsisDynFloodTopology( mode, instanceId=None, instanceName=None, vrfName=None,
                           level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_TOPOLOGY'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodTopologyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisDynFloodTopology( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'dynamic flooding [ LEVEL ] topology' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'dynamic' : 'Dynamic flooding information',
      'flooding' : 'Dynamic flooding information',
      'LEVEL' : levelMatcherForShow,
      'topology' : 'Flooding topology',
      'vrf' : 'vrf name',
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisDynFloodTopologyVRFsModel

   @staticmethod
   def handler( mode, args ):
      for kw in ( 'dynamic', 'flooding', 'topology' ):
         args.pop( kw, None )
      args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodTopologyModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodTopologyVRFsModel
      return showIsisCommand( mode, showIsisDynFloodTopology, **args )

BasicCli.addShowCommandClass( ShowIsisDynFloodTopology )

#-------------------------------------------------------------------------------
# show isis dynamic flooding [ LEVEL ] interfaces
#-------------------------------------------------------------------------------

def showIsisDynFloodInterfaces( mode, instanceId=None, instanceName=None,
                                vrfName=None, level=None ):
   command = 'MIO_DGET_ISIS_DYNFLOOD_INTERFACES'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if level is not None:
      args[ 'level' ] = 1 if level == 'level-1' else 2
   return showRibCapiCommand( mode, IsisCliModels.IsisDynFloodInterfacesModel,
                              command, args, clientName='ISIS', l3Config=l3Config )

class ShowIsisDynFloodInterfaces( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'dynamic flooding [ LEVEL ] interfaces' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'dynamic' : 'Dynamic flooding information',
      'flooding' : 'Dynamic flooding information',
      'LEVEL' : levelMatcherForShow,
      'interfaces' : 'Flooding interfaces',
      'vrf' : 'vrf name',
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisDynFloodInterfacesVRFsModel

   @staticmethod
   def handler( mode, args ):
      for kw in ( 'dynamic', 'flooding', 'interfaces' ):
         args.pop( kw, None )
      args[ 'instDictModelType' ] = IsisCliModels.isisDynFloodInterfacesModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisDynFloodInterfacesVRFsModel
      return showIsisCommand( mode, showIsisDynFloodInterfaces, **args )

BasicCli.addShowCommandClass( ShowIsisDynFloodInterfaces )

#------------------------------------------------------------------------------
# Area proxy show commands
#
# show isis area proxy
#------------------------------------------------------------------------------

def showIsisAreaProxy( mode, instanceId=None, instanceName=None, vrfName=None,
                       cmdPrefix=None ):
   command = 'MIO_DGET_ISIS_AREA_PROXY_CMD'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      mode.addError( 'Feature only supported for the default VRF' )
      return
   return showRibCapiCommand( mode, IsisCliModels.IsisShowAreaProxyModel, command,
                              args, clientName='ISIS', l3Config=l3Config )

class ShowIsisAreaProxy( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'area proxy' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'area' : 'Area information',
      'proxy' : 'Area proxy information',
      'vrf' : 'vrf name',
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisShowAreaProxyVRFsModel

   @staticmethod
   def handler( mode, args ):
      for kw in ( 'area', 'proxy' ):
         if kw in args:
            del args[ kw ]
      args[ 'instDictModelType' ] = IsisCliModels.isisShowAreaProxyModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisShowAreaProxyVRFsModel
      return showIsisCommand( mode, showIsisAreaProxy, **args )

if MplsToggleLib.toggleIsisAreaProxyEnabled():
   BasicCli.addShowCommandClass( ShowIsisAreaProxy )

#-------------------------------------------------------------------------------
# Segment Routing Show Commands
#-------------------------------------------------------------------------------
def isisSrIsActive( mode, instanceName ): 
   instConfig = isisConfig.instanceConfig[ instanceName ]
   srSysdbStatus = srSysdbStatusDir.get( 'default' )
   if not srSysdbStatus:
      return False

   if instConfig.srEnabled == instConfig.srEnabledDefault:
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing has been " \
            "administratively shutdown" % instanceName )
      return False

   if mplsRoutingConfig.mplsRouting == False:
      mode.addWarning( "Mpls routing is not enabled" )
      return False

   if str( srSysdbStatus.igpRtrId ) == str( instConfig.routerIdV4Default ):
      mode.addWarning( "IS-IS (Instance: %s) Segment Routing is disabled because " \
            "the Router ID is not configured" % instanceName )
      return False
   return True

#helper function for the below used functions which returns hostnames
#from the isisSystemIdHostname map in srSysdbStatus whenever present 
def getHostName( rId, vrfName="default" ):
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   path = 'routing/isis/hostnamemap/%s' % vrfName
   isisHostnameMap = shmemEm.doMount( path, 'IsisExportImpl::IsisHostnameMap', smi )
   hostname = None
   hostnameArray = isisHostnameMap.hostnameMap.get( rId )
   if hostnameArray:
      hostname = hostnameArray.smashString()
   shmemEm.doUnmount( path )
   return hostname

#-------------------------------------------------------------------------------
# show isis segment-routing routes
#-------------------------------------------------------------------------------
def showIsisSegmentRoutingRoutes( mode, cmdPrefix, instanceId, instanceName ):
   instConfig = isisConfig.instanceConfig[ instanceName ]

   if instConfig.shutdown:
      return

   if instConfig.srDataPlane == SrDataPlane.srDataPlaneNone:
      return

   if not isisSrIsActive( mode, instanceName ):
      return

   print ""

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   srStatus = shmemEm.doMount( 'segmentrouting/isis/default',
                               'Smash::SegmentRouting::Status', smi )

   for index in srStatus.srRoute.keys():
      print index

   shmemEm.doUnmount( 'segmentrouting/isis/default')

#-------------------------------------------------------------------------------
# show mpls segment-routing bindings
#-------------------------------------------------------------------------------
def showMplsSrBindings( mode, args ):
   if 'ipv4' in args or 'ipv6' in args:
      af = args.get( 'ipv4', 'ipv6' ) 
      prefix = None
   elif 'IPADDR' in args:
      prefix = args.get( 'IPADDR' ).stringValue
      af = None
   elif 'IP6ADDR' in args:
      prefix = args.get( 'IP6ADDR' ).stringValue
      af = None
   else:
      prefix = None
      af = None
   if not af:
      source = [ 'isisV4', 'isisV6' ]
   elif af == 'ipv4':
      source = [ 'isisV4' ]
   elif af == 'ipv6':
      source = [ 'isisV6' ]
   else:
      mode.addError( 'Invalid address-family' )
      return MplsModel.MplsBindingsModel( bindings={} )

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper = IsisMplsBindingsHelper( mplsStatus.force(), False,
                                    isisSystemIdHostnameMap )
   genPrefix = IpGenPrefix()
   if prefix:
      if isinstance( prefix, str ):
         genPrefix.handleInitialized( prefix ) #ipv4 are obtained as str
      else:
         genPrefix.handleV6Prefix( prefix ) # ipv6 prefix are obtained as Ip6Prefix
   else:
      genPrefix.handleV4Prefix( nullPrefix )
   for src in source:
      fwdEqvClass = FwdEqvClass()
      fwdEqvClass.prefix = genPrefix
      if MplsToggleLib.toggleCommonLibSmashEnabled():
         if src == 'isisV4':
            bindingTable = isisSrV4BindingTable
         elif src == 'isisV6':
            bindingTable = isisSrV6BindingTable
         peerLbtColl = LabelBindingTableColl( "plbtc" )
         localLbt = LabelBindingTable( RouterId() )
         commonLibConsumerSm = CommonLibConsumerSm( bindingTable,
                                                    localLbt,
                                                    peerLbtColl )
         helper.populateFecBindingsFromTablesFiltered( 
               commonLibConsumerSm.localLbt,
               commonLibConsumerSm.peerLbtColl,
               'detail' in args,
               fwdEqvClass )
      else:      
         helper.populateFecBindingsForSourceFiltered( src, 'detail' in args, 
                                                   fwdEqvClass )
   helper.populateLocalConflicts()

   helper.render( fd, fmt )
   return cliPrinted( MplsModel.MplsBindingsModel )

class ShowIsisMplsSrBindings( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mpls segment-routing bindings ' \
            '[ ( IPADDR | IP6ADDR ) | ( ipv4 | ipv6 ) ] ' \
            '[ detail ]'
   data = {
      'mpls' : mplsNodeForShow,
      'segment-routing' : 'Segment Routing Information',
      'bindings' : 'Label bindings',
      'IPADDR' : ipv4IPMatcher,
      'IP6ADDR' : ipv6IPMatcher,
      'ipv4' : 'IS-IS SR IPv4 bindings',
      'ipv6' : 'IS-IS SR IPv6 bindings',
      'detail' : 'Display information in detail'
   }
   handler = showMplsSrBindings
   cliModel = MplsModel.MplsBindingsModel

BasicCli.addShowCommandClass( ShowIsisMplsSrBindings )

#-------------------------------------------------------------------------------
# show isis mpls ldp sync [interface]
#-------------------------------------------------------------------------------
#
# ldpConfigColl is mounted at mpls/ldp/ldpConfigColl of Ldp::LdpConfigColl
#
# ldpProtoConfigColl is mounted at mpls/ldp/ldpProtoConfigColl of
# Ldp::LdpProtoConfigColl
#
# linkReadyStatusVrfColl is mounted at mpls/ldp/linkReadyStatus of
# Ldp::LdpLinkReadyStatusVrfColl
#
# instanceLevelConfig is instance-level mpls ldp sync config value (a boolean)
#
def showInterfaceMplsLdpSyncEntry( instanceLevelConfig, vrfName, intf ):
   ldpConfig = ldpConfigColl.config.get( vrfName )
   ldpProtoConfig = ldpProtoConfigColl.protoConfig.get( vrfName )
   readyStatusColl = linkReadyStatusVrfColl.readyStatusVrf.get( vrfName )

   def _syncRequired( intf ):
      if not readyStatusColl:
         return False # LDP not enabled
      if not ldpConfig:
         return instanceLevelConfig
      ldpIntfConfig = ldpConfig.intfConfigColl.get( intf )
      if not ldpIntfConfig:
         return instanceLevelConfig
      if ldpIntfConfig.igpSync == "useGlobal":
         return instanceLevelConfig
      return ldpIntfConfig.igpSync == "on"

   interfaceEntryModel = IsisMplsLdpSyncInterfaceModel()
   required = _syncRequired( intf )
   
   if required:
      readyStatus = readyStatusColl and  readyStatusColl.readyStatus.get( intf )
      interfaceEntryModel.syncAchieved = readyStatus and readyStatus.ready

   interfaceEntryModel.ldpEnabled = bool( readyStatusColl )
   interfaceEntryModel.syncRequired = required
   
   if ldpProtoConfig:
      interfaceEntryModel.igpNotifyDelay = int( ldpProtoConfig.linkReadyDelay )
      holdtime = ldpProtoConfig.linkReadyTimeout
      # For until-established ( holdtime value will be infinite, Tac.endOfTime ),
      # we will use 0 in CAPI model
      if holdtime == Tac.endOfTime:
         holdtime = 0
      interfaceEntryModel.holddownTime = int( holdtime )

   return interfaceEntryModel

def showInterfaceMplsLdpSync(  mode, instanceId=None, instanceName=None, 
                               vrfName=None, intf=None):
   mplsLdpSyncModel = IsisMplsLdpSyncModel() 
   for interface in isisConfig.intfConfig:
      if intf is not None and str( intf.name ) != str( interface ):
         continue
      if instanceName != isisConfig.intfConfig[ interface ].instanceName:
         continue
      instance = isisConfig.instanceConfig[ instanceName ]
      interfaceEntryModel = showInterfaceMplsLdpSyncEntry( instance.mplsLdpSync,
                                                      instance.vrfName,
                                                      interface )
      interfaceEntryModel._instanceName = instanceName
      mplsLdpSyncModel.interfaces[ interface ] = interfaceEntryModel
      mplsLdpSyncModel._instanceName = instanceName
      mplsLdpSyncModel._vrf = instance.vrfName
   return mplsLdpSyncModel

class ShowIsisMplsLdpSyncCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis [ INSTANCE ] mpls ldp sync [ intf ]'
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'mpls' : mplsForShowNode,
      'ldp' : ldpForShowKw,
      'sync' : syncForShowKw,
      'intf' : IntfCli.Intf.matcher,
   }

   cliModel = IsisCliModels.isisMplsLdpSyncVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'mpls' ]
      del args[ 'ldp' ]
      del args[ 'sync' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisMplsLdpSyncModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisMplsLdpSyncVRFsModel
      return showIsisCommand( mode, showInterfaceMplsLdpSync, **args )
BasicCli.addShowCommandClass( ShowIsisMplsLdpSyncCmd )

#-------------------------------------------------------------------------------
# show isis hostname
#-------------------------------------------------------------------------------
def showIsisHostname( mode, instanceId=None, instanceName=None, vrfName=None ):
   command = 'MIO_DGET_ISIS_HOSTNAME'
   args = { 'instance' : instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   return showRibCapiCommand( mode, IsisCliModels.IsisHostnameInstanceModel,
                              command, args, clientName='ISIS', l3Config=l3Config )
class ShowIsisHostnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper( 'hostname' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'hostname' : 'System ID to hostname mapping',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.isisHostnameVRFModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'hostname' ]
      args[ 'instDictModelType' ] = IsisCliModels.isisHostnameModel
      args[ 'cmdVrfModel' ] = IsisCliModels.isisHostnameVRFModel
      return showIsisCommand( mode, showIsisHostname, **args )
BasicCli.addShowCommandClass( ShowIsisHostnameCmd )

#-------------------------------------------------------------------------------
# show isis counters [ packet | drop ]
# show isis counters [ drop ] details
#-------------------------------------------------------------------------------
def showIsisCounters( mode, instanceId=None, instanceName=None,
                      vrfName=None, drop=None, packet=None, details=None ):
   command = 'MIO_DGET_ISIS_COUNTERS'
   args = { 'instance': instanceId }
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if details:
      args[ 'details' ] = True
      args[ 'drop' ] = True
      args[ 'packet' ] = False if drop else True
   else:
      args[ 'details' ] = False
      if drop or packet:
         args[ 'drop' ] = True if drop else False
         args[ 'packet' ] = True if packet else False
      else:
         args[ 'drop' ] = True
         args[ 'packet' ] = True

   result = showRibCapiCommand( mode, IsisCliModels.IsisInstanceCounters,
                                command, args, clientName='ISIS',
                                l3Config=l3Config )
   return result

class ShowIsisCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = isisShowCommandSyntaxWrapper(
      'counters [ packet | drop | (drop details) | (details drop) | details ]' )
   data = {
      'isis' : isisKw,
      'INSTANCE' : isisInstanceMatcher,
      'counters' : 'Show IS-IS Packet counters and drop counters',
      'packet' : 'Show only the packet counters',
      'drop' : 'Show only the drop counters',
      'details' : 'Show IS-IS Packet counters and drop counters with drop details',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }
   cliModel = IsisCliModels.showCountersVRFsModel

   @staticmethod
   def handler( mode, args ):
      del args[ 'counters' ]
      args[ 'instDictModelType' ] = IsisCliModels.showCountersInstModel
      args[ 'cmdVrfModel' ] = IsisCliModels.showCountersVRFsModel
      return showIsisCommand( mode, showIsisCounters, **args )
BasicCli.addShowCommandClass( ShowIsisCountersCmd )

#-------------------------------------------------------------------------------
# show isis ti-lfa tunnel [ TUNNEL_INDEX ]
#-------------------------------------------------------------------------------
def getTilfaTunnelTableEntryModel( tunnelId ):
   tunnelTableEntry = tilfaTunnelTable.entry.get( tunnelId )

   if not tunnelTableEntry:
      return None

   def _getViaModel( tunnelVia, backup=False ):
      labels = []
      labelOp = tunnelVia.labels

      for index in range( labelOp.stackSize ):
         labels.insert( 0, str( labelOp.labelStack( index ) ) )
      return TunnelMplsVia( nexthop=tunnelVia.nexthop,
                            interface=tunnelVia.intfId,
                            type='ip',
                            labels=labels,
                            _isBackupVia=backup )

   vias = [ _getViaModel( tunnelTableEntry.via ) ]
   backupVias = [ _getViaModel( tunnelTableEntry.backupVia, backup=True ) ]
   return TilfaTunnelTableEntry(
      tunnelType='TI-LFA',
      tunnelIndex=getTunnelIndexFromId( tunnelId ),
      vias=vias,
      backupVias=backupVias,
      tunnelId=tunnelId )

def getTilfaTunnelTableModel( tunnelIndex=None ):
   tunnelIds = tilfaTunnelTable.entry
   if tunnelIndex is not None:
      tunnelIds = [ getTunnelIdFromIndex( 'tiLfaTunnel',
                                          tunnelIndex ) ]
   tilfaEntries = {}

   for tunnelId in tunnelIds:
      tilfaEntryModel = getTilfaTunnelTableEntryModel( tunnelId )
      if tilfaEntryModel:
         tilfaEntries[ getTunnelIndexFromId( tunnelId ) ] = tilfaEntryModel
   return TilfaTunnelTable( entries=tilfaEntries )

class ShowIsisTilfaTunnelTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show isis ti-lfa tunnel [ TUNNEL_INDEX ]'
   data = {
      'isis' : isisKw,
      'ti-lfa': tokenTilfaForShow,
      'tunnel': tokenTunnelMatcher,
      'TUNNEL_INDEX': tunnelIndexMatcher
   }
   cliModel = TilfaTunnelTable

   @staticmethod
   def handler( mode, args ):
      return getTilfaTunnelTableModel( tunnelIndex=args.get( "TUNNEL_INDEX" ) )

BasicCli.addShowCommandClass( ShowIsisTilfaTunnelTableCmd )

#-------------------------------------------------------------------------------
# Clear commands
#-------------------------------------------------------------------------------
def clearIsisCommand( *args, **kwargs ):
   '''
   args - Contains CLI mode and callback function
   kwargs - Contains extra arguement for the CLI
   # The format of parser tokens for IS-IS clear commands is like this
   # clear isis [ instance-name ] <command> [ command args ]
   #    or
   # clear isis <command> [ command args ] [ vrf vrf-name ]
   # If instance name is specified we do not show the 'vrf' keyword.
   #
   # If vrf name is specified use the Isis instance in that VRF.
   # If it doesn't exist report error.
   #
   # If neither instance name nor vrf name are specified use the VRF from the
   # routing context if it is set. If routing context is not set and there is an
   # Isis instance configured in the default VRF, then use that instance.
   #
   # Otherwise report error.

   '''
   vrfName = None
   instances = None
   mode = args[ 0 ]
   instanceConfig = isisConfig.instanceConfig

   # callback will be the last entry in *args
   callback = args[ -1 ]
   args = args[ : -1 ]

   for keyword in ( 'clear', 'isis', 'vrf' ):
      kwargs.pop( keyword, None )

   level = kwargs.pop( 'LEVEL', None )
   if level is not None:
      kwargs[ 'level' ] = LVL_KW_TO_TAC_TYPE_MAP[ level ]

   if 'INSTANCE' in kwargs:
      instanceName = kwargs.pop( 'INSTANCE' )
      if instanceName in instanceConfig:
         instances = [ instanceName ]
      else:
         mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
   elif 'ALL_INSTANCE' in kwargs:
      # Clearing all instances in default VRF
      kwargs.pop( 'ALL_INSTANCE' )
      instances = sorted( i.instanceName for i in instanceConfig.values() if
                          i.vrfName == DEFAULT_VRF )
   elif 'VRFNAME' in kwargs:
      # VRF name was specified so find the instance #0 corresponding to that VRF.
      # For non-default VRF, that's the only instance.
      vrfName = kwargs.pop( 'VRFNAME' )
      instanceByVrfName = isisInstanceByVrfName( instanceConfig )
      if vrfName != ALL_VRF_NAME:
         if vrfName in instanceByVrfName:
            instId = 0
            instancesById = isisInstanceNameById( instanceConfig, vrf=vrfName )
            instances = [ instancesById[ instId ] ]
         else:
            mode.addError( "No IS-IS instance is configured in VRF: %s" % vrfName )
      else:
         # find all instance #0 of each VRF when VRFNAME is 'all'
         instances = sorted( i.instanceName for i in instanceConfig.values() if
                             i.instanceId == 0 )
   elif 'INTF' in kwargs:
      intf = kwargs[ 'INTF' ]
      kwargs[ 'intf' ] = kwargs.pop( 'INTF' )
      intfConfig = _getIntfConfig( intf.name )
      if ( intfConfig is None or not intfConfig.instanceName ):
         mode.addError( 'No IS-IS instance is enabled on this interface' )
      elif intfConfig.instanceName in instanceConfig:
         instances = [ intfConfig.instanceName ]
      else:
         mode.addError( 'IS-IS instance %s of this interface is not configured'
                        % intfConfig.instanceName )
   else:
      # Neither VRF name nor instance name was specified.
      # Find instance by looking at routing context if that exists.
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
      instanceByVrfName = isisInstanceByVrfName( instanceConfig )
      if vrfName and vrfName != DEFAULT_VRF:
         if vrfName in instanceByVrfName:
            instances = instanceByVrfName[ vrfName ]
         else:
            mode.addError( "No IS-IS instance is configured in routing-context "
                           "VRF: %s" % vrfName )
      else:
         # Use the default instance in the default VRF (i.e. instance id #0)
         if DEFAULT_VRF in instanceByVrfName:
            instId = 0
            instancesById = isisInstanceNameById( instanceConfig )
            instances = [ instancesById[ instId ] ]
   if instances:
      kwargs[ 'instances' ] = instances
      return callback( *args, **kwargs )

CLEAR_NEIGHBOR_TARGET = 'adjacency'
CLEAR_INSTANCE_TARGET = 'instance'
CLEAR_COUNTERS_TARGET = 'counters'
CLEAR_DATABASE_TARGET = 'database'

def clearCommandCallback( mode, instances, target, *args ):
   for instanceName in instances:
      request = ClearRequest( instanceName, target )
      if target == CLEAR_NEIGHBOR_TARGET:
         request.clearAdjRequest = ClearAdjRequest( *args )
      elif target == CLEAR_COUNTERS_TARGET:
         request.clearCountersRequest = ClearCountersRequest( *args )
      elif target == CLEAR_DATABASE_TARGET:
         request.clearDatabaseRequest = ClearDatabaseRequest( *args )
      reqId = "%d.%f" % ( os.getpid(), Tac.now() )
      isisClearReqDir.clearRequest[ reqId ] = request 
      vrfName = isisConfig.instanceConfig[ instanceName ].vrfName
      try:
         Tac.waitFor( lambda: vrfName in isisClearRespDir and
                              reqId in isisClearRespDir[vrfName].clearResponse,
                      description=( "Clear on instance %s" % instanceName ),
                      sleep=True, maxDelay=0.1, timeout=5.0 )
         numCleared = isisClearRespDir[ vrfName ].clearResponse[ reqId ].numCleared
         success = isisClearRespDir[ vrfName ].clearResponse[ reqId ].success
         if success:
            yield numCleared, instanceName
         else:
            print "Failed to clear on instance %s" % instanceName
      except Tac.Timeout:
         mode.addWarning( "Clearing may not have been successful on instance %s" %
                          ( instanceName ) )

      del isisClearReqDir.clearRequest[ reqId ]


#-------------------------------------------------------------------------------
# "clear isis neighbor
#-------------------------------------------------------------------------------
def doClearIsisNeighbor( mode, instances=None, allAdj=None, adjId=None, intf=None,
      level=None ):
   level = level if level else 'levelNone'
   adjId = adjId if adjId else ''
   if intf:
      if not intf.lookup():
         return
      intf = kernelDeviceName( intf.name )
   else:
      intf = ''

   if allAdj:
      adjId = intf = ''

   print( "" ) # Newline before output
   for count, inst in clearCommandCallback( mode, instances, CLEAR_NEIGHBOR_TARGET,
                                            intf, adjId, level ):
      print( "%d neighbors cleared on instance %s" % ( count, inst ) )

NbrIdRegex = "^(?!all$)" + notAPrefixOf( 'interface' ) + hostnameRegex
NbrIdMatcher = CliMatcher.PatternMatcher( pattern=NbrIdRegex,
                                        helpdesc='System-ID or hostname of neighbor',
                                        helpname='Neighbor-ID' )
class ClearIsisNeighborCmd( CliCommand.CliCommandClass ):
   syntax = (
         isisClearCommandSytaxWrapper( 'neighbor ( Neighbor-ID | all ) [ LEVEL ]' ) +
         '| ( neighbor [ Neighbor-ID ] interface INTF [ LEVEL ] )' )
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'isis' : clearIsisKw,
      'INSTANCE' : isisInstanceMatcher,
      'ALL_INSTANCE' : allInstanceMatcher,
      'neighbor' : 'Reset neighbors',
      'Neighbor-ID' : CliCommand.Node( matcher=NbrIdMatcher, maxMatches=1 ),
      'all' : 'Clear all',
      'interface' : 'Reset neighbors on an interface',
      'INTF' : IntfCli.Intf.matcher,
      'LEVEL' : CliMatcher.EnumMatcher( {
         'level-1' : 'Level 1 only',
         'level-2' : 'Level 2 only',
         'level-1-2' : 'Level 1-2 Point-to-Point only',
      } ),
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      del args[ 'neighbor' ]
      if 'interface' in args:
         del args[ 'interface' ]
      args[ 'allAdj' ] = args.pop( 'all', None )
      args[ 'adjId' ] = args.pop( 'Neighbor-ID', None )
      clearIsisCommand( mode, doClearIsisNeighbor, **args )

BasicCli.EnableMode.addCommandClass( ClearIsisNeighborCmd )

#-------------------------------------------------------------------------------
# "clear isis instance
#-------------------------------------------------------------------------------
def doClearIsisInstance( mode, instances=None ):
   print( "" )  # Newline before output
   for count, inst in clearCommandCallback( mode, instances, CLEAR_INSTANCE_TARGET ):
      if count > 0:
         print( "IS-IS instance %s cleared." % inst )

class ClearIsisInstanceCmd( CliCommand.CliCommandClass ):
   syntax = isisClearCommandSytaxWrapper( 'instance' )
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'isis' : clearIsisKw,
      'INSTANCE' : isisInstanceMatcher,
      'ALL_INSTANCE' : allInstanceMatcher,
      'instance' : 'Reset instance',
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      del args[ 'instance' ]
      clearIsisCommand( mode, doClearIsisInstance, **args )
BasicCli.EnableMode.addCommandClass( ClearIsisInstanceCmd )

#-------------------------------------------------------------------------------
# clear isis counters
#-------------------------------------------------------------------------------
def doClearIsisCounters( mode, instances=None, intf=None ):
   if intf:
      if not intf.lookup():
         return
      intf = kernelDeviceName( intf.name )
   else:
      intf = ''
   print( "" ) # Newline before output
   for count, inst in clearCommandCallback( mode, instances, CLEAR_COUNTERS_TARGET,
                                            intf ):
      print( "Counters cleared on %d interfaces of instance %s." % ( count, inst ) )

class ClearIsisCountersCmd( CliCommand.CliCommandClass ):
   syntax = isisClearCommandSytaxWrapper( 'counters' ) + \
            ' | ( counters interface INTF )'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'isis' : clearIsisKw,
      'INSTANCE' : isisInstanceMatcher,
      'ALL_INSTANCE' : allInstanceMatcher,
      'counters' : 'Reset counters',
      'interface' : 'Reset counters on an interface',
      'INTF' : IntfCli.Intf.matcher,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      del args[ 'counters' ]
      if 'interface' in args:
         del args[ 'interface' ]
      clearIsisCommand( mode, doClearIsisCounters, **args )
BasicCli.EnableMode.addCommandClass( ClearIsisCountersCmd )

#-------------------------------------------------------------------------------
# clear isis database
#-------------------------------------------------------------------------------
def doClearIsisDatabase( mode, instances=None, allDb=None, LSPID=None, level=None ):
   level = level if level else 'levelNone'
   lspid = LSPID if LSPID else ''
   if allDb:
      level = 'levelNone'
      lspid = ''
   print( "" ) # Newline before output
   for count, inst in clearCommandCallback( mode, instances, CLEAR_DATABASE_TARGET,
                                            lspid, level ):
      print( "%d LSPs cleared on instance %s." % ( count, inst ) )

class ClearIsisDatabaseCmd( CliCommand.CliCommandClass ):
   syntax = isisClearCommandSytaxWrapper( 'database ( all | { LSPID | LEVEL } )' )
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'isis' : clearIsisKw,
      'INSTANCE' : isisInstanceMatcher,
      'ALL_INSTANCE' : allInstanceMatcher,
      'database' : 'Reset database',
      'all' : 'All',
      'LEVEL' : singleLevelNodeForShow,
      'LSPID' : singleLspidNode,
      'vrf' : vrfKw,
      'VRFNAME' : vrfNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      del args[ 'database' ]
      args[ 'allDb' ] = args.pop( 'all', None )
      clearIsisCommand( mode, doClearIsisDatabase, **args )

BasicCli.EnableMode.addCommandClass( ClearIsisDatabaseCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entMan ):
   global isisConfig, isisStatusDir, routingHardwareStatus
   global routing6Config, routing6HardwareStatus, allIntfStatusDir
   global ipConfig, ip6Config, allVrfConfig
   global isisClearReqDir, isisClearRespDir
   global vrfInfoDir, vrf6InfoDir, srSysdbStatusDir
   global linkReadyStatusVrfColl, ldpProtoConfigColl, ldpConfigColl
   global entityManager
   global mplsRoutingConfig
   global l3Config
   global mplsStatus
   global isisSystemIdHostnameMap
   global tilfaTunnelTable
   global isisSrV4BindingTable
   global isisSrV6BindingTable
   global srConfig
   global trapConfig

   entityManager = entMan
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )

   l3Config = ConfigMount.mount( entityManager, "l3/config", "L3::Config", 'w' )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
   isisConfig = ConfigMount.mount( entityManager, 'routing/isis/config',
                                   'Routing::Isis::Config', 'w' )
   isisStatusDir = LazyMount.mount( entityManager, 'routing/isis/status',
                                    'Tac::Dir', 'ri' )
   isisClearReqDir = LazyMount.mount( entityManager, 'routing/isis/clear/request',
                                      'Routing::Isis::ClearRequestNode', 'w' )
   isisClearRespDir = LazyMount.mount( entityManager, 'routing/isis/clear/response',
                                       'Tac::Dir', 'ri' )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   ip6Config = LazyMount.mount( entityManager, "ip6/config", "Ip6::Config", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   routing6Config = LazyMount.mount( entityManager, "routing6/config",
                                     "Routing6::Config", "r" )
   routing6HardwareStatus = LazyMount.mount( entityManager,
                                             "routing6/hardware/status",
                                             "Routing6::Hardware::Status", "r" )
   srSysdbStatusDir = LazyMount.mount( entityManager,
                                    "segmentrouting/isis",
                                    "Tac::Dir", "ri" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   vrfInfoDir = LazyMount.mount( entityManager, "routing/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   vrf6InfoDir = LazyMount.mount( entityManager, "routing6/vrf/routingInfo/status",
                                "Tac::Dir", "ri" )
   linkReadyStatusVrfColl = LazyMount.mount( entityManager,
                                  "mpls/ldp/linkReadyStatus",
                                  "Ldp::LdpLinkReadyStatusVrfColl", "r" )
   ldpProtoConfigColl = LazyMount.mount( entityManager,
                                         "mpls/ldp/ldpProtoConfigColl",
                                         "Ldp::LdpProtoConfigColl", "r" )
   ldpConfigColl = LazyMount.mount( entityManager,
                                    "mpls/ldp/ldpConfigColl",
                                    "Ldp::LdpConfigColl", "r" )
   mplsRoutingConfig = LazyMount.mount( entityManager, 'routing/mpls/config', 
                                                      "Mpls::Config", "r" )
   mplsStatus = LazyMount.mount( entityManager, 'mpls/status',
                                 "Mpls::Api::Status", "r" )
   isisSystemIdHostnameMap = shmemEm.doMount( 'routing/isis/hostnamemap/default',
                                              "IsisExportImpl::IsisHostnameMap",
                                              smi )
   isisSrV4BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV4",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   isisSrV6BindingTable = shmemEm.doMount( "mpls/labelBindingTables/isisV6",
                                           "CommonLibSmash::LabelBindingTable",
                                           Smash.mountInfo( 'keyshadow' ) )
   srConfig = ConfigMount.mount( entityManager, "routing/sr/config",
                                 "Routing::SegmentRoutingCli::Config", 'w' )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )

   IntfCli.Intf.registerDependentClass( IsisIntf, priority=20 )
   IraVrfCli.canDeleteVrfHook.addExtension( deleteVrfHook )
   CliPlugin.SegmentRoutingCli.validateNodeSegmentHook.addExtension( 
         checkNodeSegmentConflict )
