# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#----------------------------------------------------------------------------
# This module provides common functionality
#----------------------------------------------------------------------------

import re
import os
import Tac

# pylint: disable-msg=W1401

# We want common gated info before protocol specific information and after
# general system information like 'show processes top' which are defined in
# SysMgr package.
GATED_COMMON_SH_TECH_TS = '2010-05-29 23:59:50'
# We don't really care about the relative order of the commands between each
# protocol, so use the same timestamp for all non-forking ribd show commands
GATED_PROTO_SH_TECH_TS = '2010-05-29 23:59:51'
# We want the show commands that generate their output from ribd after forking
# ribd to be at the end (relative to all commands that generate their o/p from
# ribd), so that in the event ribd isn't able to fork (and crashes), we get the
# output from commands that don't fork. See BUG162829
GATED_PROTO_FORK_SH_TECH_TS = '2010-05-29 23:59:52'

# We allow time to stand still for testing.
if 'CLI_NOW_IS' in os.environ:
   cur_time = float( os.environ[ 'CLI_NOW_IS' ] )
   def now():
      return cur_time
else:
   now = Tac.now

def secondsToHMS( seconds, absolute=0 ):
   if absolute:
      if absolute < 0:
         seconds = now() - seconds
      else:
         seconds = seconds - now()
   seconds = int( seconds )
   hours = seconds / 3600
   seconds -= 3600*hours
   minutes = seconds / 60
   seconds -= 60*minutes
   return "%02d:%02d:%02d" % (hours, minutes, seconds)

# This one is in fact code duplication for what exists in @parseIpRouteShow
# in L3TestContext, but it existed in KernelFib package creating dependency
# issues. So moved it here.
def parseKernelIpRouteShow( output, ipv6=False, protocol=None, connectedRoute=False,
                            protoUnset=False, getMetric=False ):
   """Parses output of "ip route show (-6|-4)" and returns a dictionary indexed
   by prefixes."""
   ret = {}
   prefix = None
   rtypesRe = \
       "(unicast|local|broadcast|multicast|throw|unreachable|prohibit|blackhole|nat)"
   metricRe = "\s*(?:metric (\S+))?"
   scopeRe = "\s*(?:scope (\S+))?"
   protoRe = "\s*(?:proto (\S+))?"
   if ipv6:
      viadev = "(?:via\s+([0-9a-f:]+))?\s*(?:dev\s+(\S+))?"
      rtRe = rtypesRe + "?\s*([0-9a-f:.]+([/]\d+)?|default)\s+"
      prefSrcRe = "\s*(?:src\s+([0-9a-f:]+))?"
   else:
      viadev = "(?:via\s+(\d+\.\d+\.\d+\.\d+))?\s*(?:dev\s+(\S+))?"
      rtRe = rtypesRe + "?\s*(\d+\.\d+\.\d+\.\d+([/]\d+)?|default)\s+"
      prefSrcRe = "\s*(?:src\s+(\d+\.\d+\.\d+\.\d+))?"
   routeRe = rtRe + viadev + protoRe + scopeRe + prefSrcRe + metricRe
   for line in output:
      m = re.match( routeRe, line )
      if m:
         ( rtType, prefix, l, via, dev, proto, scope, prefSrc, metric ) = m.groups()
         routeInfo = ( via, dev, proto )
         if protocol:
            if type ( protocol ) == str and protocol != proto:
               continue
            elif proto not in protocol:
               continue
         # For bash added routes the proto need not be specified. For some
         # tests where we may have multiple routes for the prefix, we actually
         # want to assert that the proto in the kernel is none. The protoUnset
         # flag is used for those cases
         if protoUnset and proto:
            continue

         if prefix == 'default':
            prefix = ( '::/0' if ipv6 else '0.0.0.0/0' )
         elif l is None:
            prefix += ( "/128" if ipv6 else  "/32" )
         if connectedRoute:
            routeInfo += ( scope, prefSrc )
         if getMetric:
            if metric is None:
               metric = 0
            routeInfo += ( metric, )
         if via is not None or dev is not None:
            ret.setdefault( prefix, [] )
            # Work around for for BUG29671 - "two routes in kernel - 256 and
            # 1024 metric for directly connected IPv6 routes"
            #
            # Work around for BUG34114 Multiple IPv6 routes for management
            # interface prefix on modular duts
            # In case of modular duts, there are multiple ipv6 routes, one for
            # ma0 and another for ma1.
            #
            # d7a:629f:52a4:1::/64 dev ma0  proto kernel  metric 256
            # fd7a:629f:52a4:1::/64 dev ma1  proto kernel  metric 1024
            # fd7a:629f:52a4:1::/64 dev ma0  proto kernel  metric 1024
            #
            if ipv6 and dev is not None and prefix in ret and \
                  len( ret[prefix] ) > 0:
               if ret[ prefix ][0][1].startswith( 'ma' ):
                  dev = ret[ prefix ][0][1]
               # Only if have a different nexthop, append it
               if ret[ prefix ][0][0] != via or ret[ prefix ][0][1] != dev or \
                  ( connectedRoute and \
                    ( ret[ prefix ][0][3] != scope or \
                      ret[ prefix ][0][4] != prefSrc ) ):
                  ret[ prefix ].append( routeInfo )
            else:
               ret[ prefix ].append( routeInfo )
         elif rtType is not None:
            ret[ prefix ] = [ routeInfo ]
      m = re.match( "\s+nexthop\s+" + viadev, line )
      if m:
         ( via, dev ) = m.groups()
         if via is not None or dev is not None:
            ret.setdefault( prefix, [] )
            ret[ prefix ].append( ( via, dev, proto ) )
   return ret
