# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from BasicCli import addShowCommandClass
from CliCommand import CliExpression, Node
from CliMatcher import IntegerMatcher
from ShowCommand import ShowCliCommandClass

from CliPlugin.RouteMapCli import CommonTokens as RouteMapCommonTokens
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
import CliPlugin.RoutingBgpShowCli
from CliPlugin.RoutingBgpShowCli import (
      allVrfExprFactory,
      routeSummaryVrfModel,
      summaryVrfModel,
   )
from CliToken.RoutingBgpShowCliTokens import (
      AddrFamilyAfterShowBgp,
      bgpAfterShow,
      color,
      communityExact,
      detail,
      distinguisher,
      endpoint,
      ipGenAddrMatcher,
      neighbors,
      policies,
      receivedPolicies,
      showCommunity,
      showCommunityList,
      srTeAfterShowBgp,
      summary,
   )
from CliToken.RouteMapCliTokens import CommunityConstExpr

# This file provides the CLI parser handling for BGP show commands that are
# specific to the SR-TE SAFI

#----------------------------------------------------------
# Locally useful expressions
#----------------------------------------------------------
class PolicyFilter( CliExpression ):
   expression = "policies | received-policies"
   data = {
         "policies": Node( policies, alias="policyFilter" ),
         "received-policies": Node( receivedPolicies, alias="policyFilter" ),
      }

#----------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] summary [vrf <vrfName>]
#----------------------------------------------------------
class ShowBgpSrTeSummary( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] summary [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "summary": summary,
         "VRF": allVrfExprFactory,
      }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeSummary(
            mode,
            addressFamily=args.get( "addrFamily" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpSrTeSummary )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] [ detail ] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTe( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] [ detail ] [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeRoutes(
            mode,
            addressFamily=args.get( "addrFamily" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpSrTe )

#---------------------------------------------------------------------------------
# "show bgp sr-te endpoint <> color <> distinguisher <> [detail] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTeEndpointColorDistinguisher( ShowCliCommandClass ):
   syntax = """show bgp sr-te
               endpoint ENDPOINT_VALUE
               color COLOR_VALUE
               distinguisher DISTINGUISHER_VALUE
               [ detail ] [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "endpoint": endpoint,
         "ENDPOINT_VALUE": ipGenAddrMatcher,
         "color": color,
         "COLOR_VALUE": IntegerMatcher( 0, 0xFFFFFFFF,
            helpdesc='Policy color value' ),
         "distinguisher": distinguisher,
         "DISTINGUISHER_VALUE": IntegerMatcher( 0, 0xFFFFFFFF,
            helpdesc='Distinguisher value' ),
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeRoutes(
            mode,
            endpoint=args[ "ENDPOINT_VALUE" ],
            color=args[ "COLOR_VALUE" ],
            distinguisher=args[ "DISTINGUISHER_VALUE" ],
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpSrTeEndpointColorDistinguisher )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community [<com>] [exact] [detail] [vrf <name>]"
#---------------------------------------------------------------------------------
class ShowBgpSrTeCommunity( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] community
               { COMM_VALUE }
               [ exact ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "community": showCommunity,
         "COMM_VALUE": CommunityConstExpr, # communityValue
         "exact": communityExact,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeRoutesCommunities(
            mode,
            addressFamily=args.get( "addrFamily" ),
            values=args[ "communityValue" ],
            exact=args.get( "exact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpSrTeCommunity )

#-------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community-list <community-list name>
#       [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpSrTeCommunityList( ShowCliCommandClass ):
   syntax = """show bgp sr-te [ AF_AFTER_BGP ] community-list
               COMM_LIST_NAME
               [ exact ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "sr-te": srTeAfterShowBgp,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "community-list": showCommunityList,
         "COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "exact": communityExact,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeRoutesCommunityList(
            mode,
            addressFamily=args.get( "addrFamily" ),
            commListName=args[ "COMM_LIST_NAME" ],
            exact=args.get( "exact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpSrTeCommunityList )

#-----------------------------------------------------------------------------
# "show bgp neighbors <neigh_addr> [ ipv4 | ipv6 ] sr-te
#                 ( policies | received-policies) [ detail ] [vrf <vrfName>]"
#-----------------------------------------------------------------------------
class ShowBgpNeighborsSrTePoliciesReceivedPolicies( ShowCliCommandClass ):
   syntax = """show bgp neighbors PEER_ADDR [ AF_AFTER_BGP ] sr-te
               POLICY_FILTER
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "sr-te": srTeAfterShowBgp,
         "POLICY_FILTER": PolicyFilter,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpSrTeNeighborRoutes(
            mode,
            peerAddr=args[ "PEER" ],
            addressFamily=args.get( "addrFamily" ),
            policyFilter=args[ "policyFilter" ],
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpNeighborsSrTePoliciesReceivedPolicies )
