# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from BasicCli import addShowCommandClass
from CliPlugin.IpAddrMatcher import (
      ipAddrMatcher,
      IpAddrMatcher,
      IpPrefixMatcher,
      ipPrefixMatcher,
      PREFIX_OVERLAP_AUTOZERO,
   )
from CliPlugin.Ip6AddrMatcher import (
      ip6AddrMatcher,
      Ip6PrefixMatcher,
      ip6PrefixMatcher,
   )
from CliPlugin.RouteMapCli import (
      CommonTokens as RouteMapCommonTokens,
      LargeCommCliMatcher,
      showLargeCommunityDesc,
   )
from CliPlugin.RoutingBgpCli import (
      PeerConfigKey,
      V4V6PeerKeyCliExpression,
      ipv4PeerMatcher,
      ipv6LlPeerMatcher,
      ipv6PeerMatcher,
   )
import CliPlugin.RoutingBgpShowCli
from CliPlugin.RoutingBgpShowCli import (
      allVrfExprFactory,
      pathsVrfModel,
      peerListVrfModel,
      queuedWithdrawalsVrfModel,
      routeSummaryVrfModel,
      summaryVrfModel,
   )
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from CliToken.RouteMapCliTokens import CommunityConstExpr
from CliToken.RoutingBgpShowCliTokens import (
      AddressFamilyAfterShowBgpRule,
      CommunityValuesAndExactRule,
      InternalWithVerboseRule,
      IpOrPrefixMatcher,
      Ipv4OrIpv6AddrMatcher,
      Ipv4OrIpv6AddrMaskLongerMatcher,
      RouteTypeMaybeFilteredMatcher,
      RouteUnicastAddressFamilyRule,
      TaskStatsWithResetRule,
      advertised,
      aspath,
      aspathRegexMatcher,
      bfd,
      bgpAfterShow,
      commRegexp,
      communityExact,
      detail,
      filtered,
      iar,
      installed,
      internal,
      labUni,
      longerPrefixes,
      luNexthop,
      luNexthopOriginated,
      match,
      neighbors,
      notInstalled,
      paths,
      peers,
      queuedWithdrawals,
      received,
      receivedAll,
      regexp,
      routerId,
      showCommunity,
      showCommunityList,
      showLargeCommunity,
      showLargeCommunityList,
      summary,
   )
from ShowCommand import ShowCliCommandClass
from CliCommand import (
      CliExpression,
      Node
   )
from Toggles import BgpCommonToggleLib

# This file provides the CLI parser handling for BGP show commands that are
# not specific to one SAFI and are not better suited in any of the other
# SAFI specific CliPlugin show command plugins.
# This file can also include "show [ip/ipv6] bgp summary" style and
# "show bgp [afi/safi] summary" style commands which show the neighbor
# status and number of prefixes received/accepted.

#-------------------------------------------------------------------------------
# "show ip bgp paths [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpBgpPaths( ShowCliCommandClass ):
   syntax = """show ip bgp paths [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "paths": paths,
         "VRF": allVrfExprFactory,
      }
   cliModel = pathsVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpPaths(
            mode,
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpPaths )

#-------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     [ community [<communities>] [exact] ] [detail]
#     [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowBgpMultiSafi( ShowCliCommandClass ):
   syntax = """show bgp AFI_SAFI
               [ COMM_VAL_EXACT ]
               [ detail ]
               [ internal ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "AFI_SAFI": AddressFamilyAfterShowBgpRule,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "detail": detail,
         "internal": internal,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNlriAf(
            mode,
            addressFamilyAndType=args.get( "addressFamilyAndType" ),
            communityValuesAndExact=args.get( "communityValuesAndExact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ),
            internal=args.get( "internal" ) )

addShowCommandClass( ShowBgpMultiSafi )

#-----------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     summary [vrf <vrfName>]
#-----------------------------------------------------------------------------------
class ShowBgpMultiSafiSummary( ShowCliCommandClass ):
   syntax = """show bgp AFI_SAFI summary [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "AFI_SAFI": AddressFamilyAfterShowBgpRule,
         "summary": summary,
         "VRF": allVrfExprFactory,
      }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNlriAfSummary(
            mode,
            addressFamilyAndType=args.get( "addressFamilyAndType" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpMultiSafiSummary )

#-------------------------------------------------------------------------------
# "show ip bgp
#     [labeled-unicast]
#     [
#       ( community <communities> [exact] ) |
#       ( community-list <name> [exact] ) |
#       ( large-community {<name>} [exact] ) |
#       ( large-community-list <name> [exact] )
#     ]
#     [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpBgp( ShowCliCommandClass ):
   syntax = """show ip bgp
               [ labeled-unicast [ next-hop originated ] ]
               [ ( community { COMM_VALUE } [ exact ] ) |
                 ( community-list COMM_LIST_NAME [ exact ] ) |
                 ( large-community { LARGE_COMM_VALUE } [ exact ] ) |
                 ( large-community-list LARGE_COMM_LIST_NAME [ exact ] ) ]
               [ detail ] [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "labeled-unicast": labUni,
         "next-hop" : Node(
            luNexthop,
            hidden=not BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled() ),
         "originated" : Node(
            luNexthopOriginated,
            hidden=not BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled() ),
         "community": showCommunity,
         "COMM_VALUE": CommunityConstExpr,
         "exact": communityExact,
         "community-list": showCommunityList,
         "COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "large-community": showLargeCommunity,
         "LARGE_COMM_VALUE": LargeCommCliMatcher( showLargeCommunityDesc ),
         "large-community-list": showLargeCommunityList,
         "LARGE_COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      extraArgs = {
            "labUni": args.get( "labeled-unicast" ),
            "epeOnly": args.get( "originated" ),
            "detail": args.get( "detail" ),
            "vrfName": args.get( "VRF" ),
         }
      if args.get( "community" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpBgpRoutesCommunities(
               mode,
               values=set( args[ "communityValue" ] ),
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "community-list" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpBgpRoutesCommunityList(
               mode,
               commListName=args[ "COMM_LIST_NAME" ],
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "large-community" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpBgpLargeCommunity(
               mode,
               values=set( args[ "LARGE_COMM_VALUE" ] ),
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "large-community-list" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpBgpRoutesLargeCommunityList(
               mode,
               largeCommListName=args[ "LARGE_COMM_LIST_NAME" ],
               exact=args.get( "exact" ),
               **extraArgs )
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpRoutesDetail( mode, **extraArgs )

addShowCommandClass( ShowIpBgp )

#----------------------------------------------------------------------
# show ip bgp large-community {<name>} detail exact [vrf <vrfName>]
#----------------------------------------------------------------------
class ShowIpBgpLargeCommunityListLegacySyntax( ShowCliCommandClass ):
   syntax = """show ip bgp large-community { LARGE_COMM_VALUE } detail
               exact [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "large-community": showLargeCommunity,
         "LARGE_COMM_VALUE": LargeCommCliMatcher( showLargeCommunityDesc ),
         "detail": detail,
         "exact": communityExact,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      extraArgs = {
            "detail": args[ "detail" ],
            "vrfName": args.get( "VRF" ),
         }
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpLargeCommunity(
            mode,
            values=set( args[ "LARGE_COMM_VALUE" ] ),
            exact=args.get( "exact" ),
            **extraArgs )

addShowCommandClass( ShowIpBgpLargeCommunityListLegacySyntax )

#-------------------------------------------------------------------------------
# "show ip bgp community regex <comm-regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpBgpCommunityRegexp( ShowCliCommandClass ):
   syntax = """show ip bgp community regexp REGEX [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "community": showCommunity,
         "regexp": commRegexp,
         "REGEX": RouteMapCommonTokens.commListRegexMatcher,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpRoutesCommunitiesRegexp(
            mode, regex=args[ "REGEX" ], vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpCommunityRegexp )

#-------------------------------------------------------------------------------
# "show ip bgp (installed | not-installed) [internal] [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowIpBgpInstalled( ShowCliCommandClass ):
   syntax = """show ip bgp ( installed | not-installed ) [ internal ] [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "installed": installed,
         "not-installed": notInstalled,
         "internal": internal,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpInstalled(
            mode,
            notInstalled="not-installed" in args,
            installed="installed" in args,
            internal=args.get( "internal" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpInstalled )

#-------------------------------------------------------------------------------
# "show ip bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpBgpSummary( ShowCliCommandClass ):
   syntax = """show ip bgp summary [ PREFIX ] [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "summary": summary,
         "PREFIX": IpPrefixMatcher( "BGP peers filtered by prefix",
                                    overlap=PREFIX_OVERLAP_AUTOZERO ),
         "VRF": allVrfExprFactory,
      }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpSummary(
            mode,
            prefix=args.get( "PREFIX" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpSummary )

#-------------------------------------------------------------------------------
# "show ip bgp [ labeled-unicast ]
#     ( ( <ip> ) | ( <prefix> [ longer-prefixes ] [ detail ] ) )
#     [ internal ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpBgpAddrOrPrefix( ShowCliCommandClass ):
   syntax = """show ip bgp [ labeled-unicast ]
               ( ( IP ) | ( PREFIX [ longer-prefixes ] [ detail ] ) )
               [ internal ] [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "labeled-unicast": labUni,
         "PREFIX": ipPrefixMatcher,
         "longer-prefixes": longerPrefixes,
         "detail": detail,
         "IP": ipAddrMatcher,
         "internal": internal,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      if args.get( "IP" ):
         addr = args.get( "IP" )
      else:
         addr = args.get( "PREFIX" )
      return CliPlugin.RoutingBgpShowCli.doShowIpBgp(
            mode,
            addr=addr,
            longerPrefixes=args.get( "longer-prefixes" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ),
            vrfName=args.get( "VRF" ),
            labUni=args.get( "labeled-unicast" ) )

addShowCommandClass( ShowIpBgpAddrOrPrefix )

#-------------------------------------------------------------------------------
# "show ipv6 bgp
#     [labeled-unicast]
#     [
#       ( match community <values> [exact] ) |
#       ( community-list <name> [exact] ) |
#       ( large-community {<name>} [exact] ) |
#       ( large-community-list <name> [exact] )
#     ]
#     [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpv6Bgp( ShowCliCommandClass ):
   syntax = """show ipv6 bgp
               [ labeled-unicast [ next-hop originated ] ]
               [ ( match community { COMM_VALUE } [ exact ] ) |
                 ( community-list COMM_LIST_NAME [ exact ] ) |
                 ( large-community { LARGE_COMM_VALUE } [ exact ] ) |
                 ( large-community-list LARGE_COMM_LIST_NAME [ exact ] ) ]
               [ detail ] [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "labeled-unicast": labUni,
         "next-hop" : Node(
            luNexthop,
            hidden=not BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled() ),
         "originated" : Node(
            luNexthopOriginated,
            hidden=not BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled() ),
         "match": match,
         "community": showCommunity,
         "COMM_VALUE": CommunityConstExpr,
         "exact": communityExact,
         "community-list": showCommunityList,
         "COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "large-community": showLargeCommunity,
         "LARGE_COMM_VALUE": LargeCommCliMatcher( showLargeCommunityDesc ),
         "large-community-list": showLargeCommunityList,
         "LARGE_COMM_LIST_NAME": RouteMapCommonTokens.communityListName,
         "detail": detail,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      extraArgs = {
            "labUni": args.get( "labeled-unicast" ),
            "epeOnly": args.get( "originated" ),
            "detail": args.get( "detail" ),
            "vrfName": args.get( "VRF" ),
         }
      if args.get( "community" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpRoutesCommunities(
               mode,
               values=set( args[ "communityValue" ] ),
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "community-list" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpRoutesCommunityList(
               mode,
               commListName=args[ "COMM_LIST_NAME" ],
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "large-community" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpLargeCommunity(
               mode,
               values=set( args[ "LARGE_COMM_VALUE" ] ),
               exact=args.get( "exact" ),
               **extraArgs )
      if args.get( "large-community-list" ):
         return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpRoutesLargeCommunityList(
               mode,
               largeCommListName=args[ "LARGE_COMM_LIST_NAME" ],
               exact=args.get( "exact" ),
               **extraArgs )
      return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpRoutesDetail(
            mode,
            **extraArgs )

addShowCommandClass( ShowIpv6Bgp )

#----------------------------------------------------------------------
# show ipv6 bgp large-community {<name>} detail exact
#----------------------------------------------------------------------
class ShowIpv6BgpLargeCommunityListLegacySyntax( ShowCliCommandClass ):
   syntax = """show ipv6 bgp large-community { LARGE_COMM_VALUE } detail
               exact [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "large-community": showLargeCommunity,
         "LARGE_COMM_VALUE": LargeCommCliMatcher( showLargeCommunityDesc ),
         "detail": detail,
         "exact": communityExact,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      extraArgs = {
            "detail": args.get( "detail" ),
            "vrfName": args.get( "VRF" ),
         }
      return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpLargeCommunity(
            mode,
            values=set( args[ "LARGE_COMM_VALUE" ] ),
            exact=args.get( "exact" ),
            **extraArgs )

addShowCommandClass( ShowIpv6BgpLargeCommunityListLegacySyntax )

#-------------------------------------------------------------------------------
# "show ipv6 bgp (installed | not-installed) [internal] [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowIpv6BgpInstalled( ShowCliCommandClass ):
   syntax = """show ipv6 bgp ( installed | not-installed ) [ internal ] [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "installed": installed,
         "not-installed": notInstalled,
         "internal": internal,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpInstalled(
            mode,
            notInstalled="not-installed" in args,
            installed="installed" in args,
            internal=args.get( "internal" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpv6BgpInstalled )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip>
#     [ ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#     ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) ) ]
#     ( routes | advertised-routes | ( received-routes [filtered] ) )
#     [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowBgpNeighborsPeerRouteType( ShowCliCommandClass ):
   syntax = """show bgp neighbors 
               PEER_ADDR
               [ AFI_SAFI ]
               ROUTE_TYPE
               [ COMM_VAL_EXACT ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "AFI_SAFI": AddressFamilyAfterShowBgpRule,
         "ROUTE_TYPE": RouteTypeMaybeFilteredMatcher,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "detail": detail,
         "VRF": allVrfExprFactory,
         }

   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNeighborsNlriAfRoutes(
            mode,
            peerAddr=args[ "PEER" ],
            routeFamilyAndType=args.get( "addressFamilyAndType" ),
            routeType=args[ "route-type" ],
            filtered=args.get( "filtered" ),
            communityValuesAndExact=args.get( "communityValuesAndExact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" )
            )

addShowCommandClass( ShowBgpNeighborsPeerRouteType )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip>
#       ( routes | advertised-routes | received-routes [filtered] )
#       ( <ip> | <prefix> [longer-prefixes] )
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowBgpNeighborsPeerIp( ShowCliCommandClass ):
   syntax = """show bgp neighbors
               PEER_ADDR
               ROUTE_TYPE
               IP 
               [ COMM_VAL_EXACT ]
               [ detail ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "ROUTE_TYPE": RouteTypeMaybeFilteredMatcher,
         "IP": IpOrPrefixMatcher,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "detail": detail,
         "VRF": allVrfExprFactory,
         }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNeighborsNlriAfPrefixAddr(
            mode,
            peerAddr=args[ "PEER" ],
            routeType=args[ "route-type" ],
            filtered=args.get( "filtered" ),
            pfxAddr=args[ "ADDR" ],
            longerPrefixes=args.get( "longer-prefixes" ),
            communityValuesAndExact=args.get( "communityValuesAndExact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" )
            )

addShowCommandClass( ShowBgpNeighborsPeerIp )

#-------------------------------------------------------------------------------
# "show bgp neighbors [<ip>] [ vrf <vrfName> ]"
#     (hidden options) [internal [verbose] [rib-out] [dump-duplicates]]
#                      [task-stats [reset]]
#
# Note: 'verbose' option prints AdjRibin paths
#-------------------------------------------------------------------------------
class ShowBgpNeighbors( ShowCliCommandClass ):
   syntax = """show bgp neighbors
               [ PEER_ADDR ]
               [ VRF ]
               [ INTERNAL ]
               [ TASK_STATS ]
            """
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "VRF": allVrfExprFactory,
         "INTERNAL": InternalWithVerboseRule,
         "TASK_STATS": TaskStatsWithResetRule,
          }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNeighbors(
            mode,
            addr=args.get( "PEER" ),
            internal=args.get( "INTERNAL" ),
            taskStats=args.get( "TASK_STATS" ),
            vrfName=args.get( "VRF" )
            )

addShowCommandClass( ShowBgpNeighbors )

#-------------------------------------------------------------------------------
# "show ip bgp [as-path] regex <asp-regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpBgpRegexp( ShowCliCommandClass ):
   syntax = """show ip bgp [ aspath ] regexp REGEX
               [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "aspath": aspath,
         "regexp": regexp,
         "REGEX": aspathRegexMatcher,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpRegexp(
         mode, regex=args[ "REGEX" ], vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpRegexp )

#-------------------------------------------------------------------------------
# "show ipv6 bgp regexp <regex> [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpv6BgpRegexp( ShowCliCommandClass ):
   syntax = """show ipv6 bgp regexp REGEX [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "regexp": regexp,
         "REGEX": aspathRegexMatcher,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpRegexp(
            mode,
            regex=args[ "REGEX" ],
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpv6BgpRegexp )

#-------------------------------------------------------------------------------
# "show bgp ( ipv4 unicast | ipv6 unicast | ipv4 multicast | ipv6 multicast )
# ( <ip> | <prefix> [longer-prefixes] ) [ community [<communities>] [exact] ]
# [detail] [internal] [vrf <vrfName>]
#-------------------------------------------------------------------------------
class ShowBgpNlriAf( ShowCliCommandClass ):
   syntax = """show bgp
               AFI_SAFI
               IP
               [ COMM_VAL_EXACT ]
               [ detail ]
               [ internal ]
               [ VRF ]
            """
   data = {
         "bgp": bgpAfterShow,
         "AFI_SAFI": AddressFamilyAfterShowBgpRule,
         "IP": IpOrPrefixMatcher,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "detail": detail,
         "internal": internal,
         "VRF": allVrfExprFactory,
         }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpNlriAf(
            mode,
            addressFamilyAndType=args.get( "addressFamilyAndType" ),
            pfxAddr=args[ "ADDR" ],
            longerPrefixes=args.get( "longer-prefixes" ),
            communityValuesAndExact=args.get( "communityValuesAndExact" ),
            detail=args.get( "detail" ),
            vrfName=args.get( "VRF" ),
            internal=args.get( "internal" ) )

addShowCommandClass( ShowBgpNlriAf )

class PeerKeyV4Expression( CliExpression ):
   expression = "PEER [ router-id ROUTER_ID ]"
   data = {
         "PEER": ipv4PeerMatcher,
         "router-id": Node( routerId,
            hidden=not BgpCommonToggleLib.toggleArBgpSameAddressPeeringEnabled() ),
         "ROUTER_ID": IpAddrMatcher( "Neighbor router ID" )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if "PEER" in args:
         args[ "PEER" ] = PeerConfigKey( args[ "PEER" ] )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors
#     [ v4PeerKey ]
#     [ bfd ]
#     (hidden options) [ internal [ verbose ] [ rib-out ] [ dump-duplicates ] ]
#                      [ task-stats [ reset ] ]
#                      [ iar ]
#     [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpBgpNeighbors( ShowCliCommandClass ):
   syntax = """show ip bgp neighbors
               [ PEER_KEY ]
               [ bfd ]
               [ INTERNAL ]
               [ TASK_STATS ]
               [ iar ]
               [ VRF ]"""
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_KEY": PeerKeyV4Expression,
         "bfd": bfd,
         "INTERNAL": InternalWithVerboseRule,
         "TASK_STATS": TaskStatsWithResetRule,
         "iar": iar,
         "VRF": allVrfExprFactory,
      }
   cliModel = peerListVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighbors(
               mode,
               addr=args.get( "PEER" ),
               routerId=args.get( "ROUTER_ID" ),
               bfd=args.get( "bfd" ),
               internal=args.get( "INTERNAL" ),
               taskStats=args.get( "TASK_STATS" ),
               iar=args.get( "iar" ),
               vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpBgpNeighbors )

#------------------------------------------------------------------------------
# "show
#     ( ip bgp neighbors <v4Addr> | ipv6 bgp peers <v6Addr|v6LlAddr> )
#     (
#      ([ipv4 unicast|ipv6 unicast] (routes|advertised-routes) [detail]) |
#      ((routes|advertised-routes) (
#            ( regex REGEX ) |
#            ( community COMM_VALUES [ exact ] [ detail ] ) |
#            ( <v4Addr|v6Addr> ) |
#            ( <v4Addr/MaskLen|v6Addr/MaskLen> [ longer-prefixes ] )
#           )
#      )
#     )
#     vrf <vrfName>"
#------------------------------------------------------------------------------
class ShowIpAnyBgpNeighborsAddrRouteType( ShowCliCommandClass ):
   syntax = """show
               ( ( ip bgp neighbors IP_PEER ) |
                 ( IPV6_WORKAROUND bgp peers ( IP6_PEER | IP6_LL_PEER ) ) )
               (
                 ( [ UNICAST_AFI_SAFI ] ( routes | advertised-routes ) [ detail ] ) |
                 ( ( routes | advertised-routes ) (
                       ( regex REGEX ) |
                       ( COMM_VAL_EXACT [ detail ] ) |
                       IPV4_OR_IPV6_ADDR |
                       IPV4_OR_IPV6_ADDR_MASK_LONGER
                    )
                 )
               )
               [ VRF ]
      """
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "IP_PEER": ipv4PeerMatcher,
         "IPV6_WORKAROUND": ipv6MatcherForShow,
         "peers": peers,
         "IP6_PEER": ipv6PeerMatcher,
         "IP6_LL_PEER": ipv6LlPeerMatcher,
         "UNICAST_AFI_SAFI": RouteUnicastAddressFamilyRule,
         "routes": received,
         "advertised-routes": advertised,
         "detail": detail,
         "regex": regexp,
         "REGEX": aspathRegexMatcher,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "IPV4_OR_IPV6_ADDR": Ipv4OrIpv6AddrMatcher,
         "IPV4_OR_IPV6_ADDR_MASK_LONGER": Ipv4OrIpv6AddrMaskLongerMatcher,
         "VRF": allVrfExprFactory,
         }

   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      """
      Note: This handler also handles the syntax defined in
      ShowIpAnyBgpNeighborsReceivedRoutes
      """
      peerAddr = ( args.get( "IP_PEER" ) or args.get( "IP6_PEER" ) or
                   args.get( "IP6_LL_PEER" ) )
      if peerAddr:
         peerAddr = PeerConfigKey( peerAddr )

      ipv6 = args.get( "peers" ) # peers keyword => ipv6
      afiSafi = args.get( "addressFamilyAndType" )

      routeOrAdvRoute = args.get( "routes" ) or args.get( "advertised-routes" )
      filteredArg = args.get( "filtered" )

      detailArg = args.get( "detail" )
      vrfName = args.get( "VRF" )

      regex = args.get( "regex" )
      commsAndExact = args.get( "communityValuesAndExact" )
      ipv4OrIpv6Addr = args.get( "IPV4_OR_IPV6_ADDR" )
      ipv4OrIpv6AddrMask = args.get( "IPV4_OR_IPV6_ADDR_MASK" )

      if ( ipv4OrIpv6Addr or ipv4OrIpv6AddrMask ) and routeOrAdvRoute:
         # (routes|advertised-routes) (ipv4Addr|ipv6Addr)
         # (routes|advertised-routes) (ipv4AddrMask|ipv6AddrMask) [longer-prefixes]
         otherArgs = {}
         if ipv4OrIpv6AddrMask:
            otherArgs[ "longerPrefixes" ] = args.get( "longer-prefixes" )

         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNeighborsPrefixAddr(
                  mode,
                  peerAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  pfxAddr=ipv4OrIpv6Addr or ipv4OrIpv6AddrMask,
                  vrfName=vrfName,
                  **otherArgs )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighborsPrefixAddr(
                  mode,
                  peerAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  pfxAddr=ipv4OrIpv6Addr or ipv4OrIpv6AddrMask,
                  vrfName=vrfName,
                  **otherArgs )

      elif ( ipv4OrIpv6Addr or ipv4OrIpv6AddrMask ) and not routeOrAdvRoute:
         # received-routes [filtered] (ipv4Addr|ipv6Addr)
         # received-routes [filtered] (ipv4AddrMask|ipv6AddrMask) [longer-prefixes]
         otherArgs = {}
         if ipv4OrIpv6AddrMask:
            otherArgs[ "longerPrefixes" ] = args.get( "longer-prefixes" )

         if ipv6:
            return CliPlugin.RoutingBgpShowCli.\
                  doShowIpv6BgpNeighborsPrefixAddrRecvdRts(
                  mode,
                  pfxAddr=ipv4OrIpv6Addr or ipv4OrIpv6AddrMask,
                  peerAddr=peerAddr,
                  filtered=filteredArg,
                  vrfName=vrfName,
                  **otherArgs )
         else:
            return CliPlugin.RoutingBgpShowCli.\
                  doShowIpBgpNeighborsPrefixAddrRecvdRts(
                  mode,
                  pfxAddr=ipv4OrIpv6Addr or ipv4OrIpv6AddrMask,
                  peerAddr=peerAddr,
                  filtered=filteredArg,
                  vrfName=vrfName,
                  **otherArgs )

      elif commsAndExact and routeOrAdvRoute:
         # (routes|advertised-routes) community {VALS} [exact] [detail]
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNeighborsCommunityRoutes(
                  mode,
                  addr=peerAddr,
                  routeType=routeOrAdvRoute,
                  values=commsAndExact[ "communityValues" ],
                  exact=commsAndExact.get( "exact" ),
                  detail=detailArg,
                  vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighborsCommunityRoutes(
                  mode,
                  addr=peerAddr,
                  routeType=routeOrAdvRoute,
                  values=commsAndExact[ "communityValues" ],
                  exact=commsAndExact.get( "exact" ),
                  detail=detailArg,
                  vrfName=vrfName )

      elif commsAndExact and not routeOrAdvRoute:
         # received-routes [filtered] community {VALS} [exact] [detail]
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.\
                  doShowIpv6BgpNeighborsCommunityRecvdRoutes(
                        mode,
                        values=commsAndExact[ "communityValues" ],
                        addr=peerAddr,
                        filtered=filteredArg,
                        exact=commsAndExact.get( "exact" ),
                        detail=detailArg,
                        vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.\
                  doShowIpBgpNeighborsCommunityRecvdRoutes(
                     mode,
                     values=commsAndExact[ "communityValues" ],
                     addr=peerAddr,
                     filtered=filteredArg,
                     exact=commsAndExact.get( "exact" ),
                     detail=detailArg,
                     vrfName=vrfName )

      elif regex and routeOrAdvRoute:
         # (routes|advertised-routes) regex REGEX
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNbrRtsRegex(
                  mode,
                  nbrAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  regex=args[ "REGEX" ],
                  vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNbrRtsRegex(
                  mode,
                  nbrAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  regex=args[ "REGEX" ],
                  vrfName=vrfName )

      elif regex and not routeOrAdvRoute:
         # received-routes [filtered] regex REGEX
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNbrRecvdRtsRegex(
                  mode,
                  regex=args[ "REGEX" ],
                  nbrAddr=peerAddr,
                  filtered=filteredArg,
                  vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNbrRecvdRtsRegex(
                  mode,
                  regex=args[ "REGEX" ],
                  nbrAddr=peerAddr,
                  filtered=filteredArg,
                  vrfName=vrfName )

      elif routeOrAdvRoute:
         # [UNICAST_AFI_SAFI] (routes|advertised-routes) [detail]
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNeighborsRoutes(
                  mode,
                  peerAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  routeFamilyAndType=afiSafi,
                  detail=detailArg,
                  vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighborsRoutes(
                  mode,
                  peerAddr=peerAddr,
                  routeType=routeOrAdvRoute,
                  routeFamilyAndType=afiSafi,
                  detail=detailArg,
                  vrfName=vrfName )

      elif not routeOrAdvRoute:
         # [UNICAST_AFI_SAFI] received-routes [filtered] [detail]
         if ipv6:
            return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNeighborsReceivedRoutes(
                  mode,
                  peerAddr=peerAddr,
                  filtered=filteredArg,
                  routeFamilyAndType=afiSafi,
                  detail=detailArg,
                  vrfName=vrfName )
         else:
            return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighborsReceivedRoutes(
                  mode,
                  peerAddr=peerAddr,
                  filtered=filteredArg,
                  routeFamilyAndType=afiSafi,
                  detail=detailArg,
                  vrfName=vrfName )

      mode.addErrors( "show command not implemented" )
      return None

addShowCommandClass( ShowIpAnyBgpNeighborsAddrRouteType )

#------------------------------------------------------------------------------
# "show ( ip bgp neighbors [<v4Addr>] | ipv6 bgp peers [<v6Addr|v6LlAddr>] )
#    (
#     ( [ ipv4 unicast | ipv6 unicast ] received-routes [ filtered ] [ detail ] ) |
#     ( received-routes [ filtered ] (
#               ( regex REGEX ) |
#               ( community COMM_VALUES [ exact ] [ detail ] ) |
#               ( <v4Addr|v6Addr> ) |
#               ( <v4Addr/MaskLen|v6Addr/MaskLen> [ longer-prefixes ] )
#              )
#     )
#    )
#    vrf <vrfName>"
#------------------------------------------------------------------------------
class ShowIpAnyBgpNeighborsReceivedRoutes( ShowCliCommandClass ):
   syntax = """show
               ( ( ip bgp neighbors [ IP_PEER ] ) |
                 ( IPV6_WORKAROUND bgp peers [ IP6_PEER | IP6_LL_PEER ] ) )
               (
                 ( [ UNICAST_AFI_SAFI ] received-routes [ filtered ] [ detail ] ) |
                 ( received-routes [ filtered ] (
                       ( regex REGEX ) |
                       ( COMM_VAL_EXACT [ detail ] ) |
                       IPV4_OR_IPV6_ADDR |
                       IPV4_OR_IPV6_ADDR_MASK_LONGER
                    )
                 )
               )
               [ VRF ]
      """
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "IP_PEER": ipv4PeerMatcher,
         "IPV6_WORKAROUND": ipv6MatcherForShow,
         "peers": peers,
         "IP6_PEER": ipv6PeerMatcher,
         "IP6_LL_PEER": ipv6LlPeerMatcher,
         "UNICAST_AFI_SAFI": RouteUnicastAddressFamilyRule,
         "received-routes": receivedAll,
         "filtered": filtered,
         "detail": detail,
         "regex": regexp,
         "REGEX": aspathRegexMatcher,
         "COMM_VAL_EXACT": CommunityValuesAndExactRule,
         "IPV4_OR_IPV6_ADDR": Ipv4OrIpv6AddrMatcher,
         "IPV4_OR_IPV6_ADDR_MASK_LONGER": Ipv4OrIpv6AddrMaskLongerMatcher,
         "VRF": allVrfExprFactory,
         }

   cliModel = routeSummaryVrfModel
   handler = ShowIpAnyBgpNeighborsAddrRouteType.handler

addShowCommandClass( ShowIpAnyBgpNeighborsReceivedRoutes )

#------------------------------------------------------------------------------
# "show ( ip bgp neighbors <v4Addr> | ipv6 bgp peers <v6Addr|v6LlAddr> )
#     [ ipv4 unicast | ipv6 unicast ]
#     advertised-routes queued-withdrawals
#     vrf <vrfName>"
#------------------------------------------------------------------------------
class ShowIpAnyBgpNeighborsAdvRoutesQueued( ShowCliCommandClass ):
   syntax = """show
               ( ( ip bgp neighbors IP_PEER ) |
                 ( IPV6_WORKAROUND bgp peers ( IP6_PEER | IP6_LL_PEER ) ) )
               [ UNICAST_AFI_SAFI ]
               advertised-routes queued-withdrawals
               [ VRF ]
      """
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "IP_PEER": ipv4PeerMatcher,
         "IPV6_WORKAROUND": ipv6MatcherForShow,
         "peers": peers,
         "IP6_PEER": ipv6PeerMatcher,
         "IP6_LL_PEER": ipv6LlPeerMatcher,
         "UNICAST_AFI_SAFI": RouteUnicastAddressFamilyRule,
         "advertised-routes": advertised,
         "queued-withdrawals": queuedWithdrawals,
         "VRF": allVrfExprFactory,
      }

   cliModel = queuedWithdrawalsVrfModel

   @staticmethod
   def handler( mode, args ):
      peerAddr = ( args.get( "IP_PEER" ) or args.get( "IP6_PEER" ) or
                   args.get( "IP6_LL_PEER" ) )
      assert peerAddr
      peerAddr = PeerConfigKey( peerAddr )

      ipv6 = args.get( "peers" ) # peers keyword => ipv6
      afiSafi = args.get( "addressFamilyAndType" )
      vrfName = args.get( "VRF" )

      if not ipv6:
         # "show ip" (not "show ipv6")
         return CliPlugin.RoutingBgpShowCli.doShowIpBgpNeighborsQueuedWithdrawals(
               mode,
               peerAddr=peerAddr,
               queuedWithdrawals=args[ 'queued-withdrawals' ],
               routeFamilyAndType=afiSafi,
               vrfName=vrfName )
      else:
         # "show ipv6" (not "show ip")
         return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpNeighborsQueuedWithdrawals(
               mode,
               peerAddr=peerAddr,
               queuedWithdrawals=args[ 'queued-withdrawals' ],
               routeFamilyAndType=afiSafi,
               vrfName=vrfName )

addShowCommandClass( ShowIpAnyBgpNeighborsAdvRoutesQueued )

class PeerKeyV6Expression( CliExpression ):
   expression = " IP6_PEER | IP6_LL_PEER [ router-id ROUTER_ID ]"
   data = {
         "IP6_PEER": ipv6PeerMatcher,
         "IP6_LL_PEER": ipv6LlPeerMatcher,
         "router-id": Node( routerId,
            hidden=not BgpCommonToggleLib.toggleArBgpSameAddressPeeringEnabled() ),
         "ROUTER_ID": IpAddrMatcher( "Neighbor router ID" )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      for key in ( "IP6_PEER", "IP6_LL_PEER" ):
         if key in args:
            args[ "PEER" ] = PeerConfigKey( args.pop( key ) )

#-------------------------------------------------------------------------------
# "show ipv6 bgp peers
#     [ v6PeerKey ]
#     (hidden options) [ internal [ verbose ] [ rib-out ] [ dump-duplicates ] ]
#                      [ task-stats [ reset ] ]
#                      [ iar ]
#     [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpv6BgpPeers( ShowCliCommandClass ):
   syntax = """show ipv6 bgp peers
               [ PEER_KEY ]
               [ INTERNAL ]
               [ TASK_STATS ]
               [ iar ]
               [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "peers": peers,
         "PEER_KEY": PeerKeyV6Expression,
         "INTERNAL": InternalWithVerboseRule,
         "TASK_STATS": TaskStatsWithResetRule,
         "iar": iar,
         "VRF": allVrfExprFactory,
      }
   cliModel = peerListVrfModel


   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpV6BgpNeighbors(
            mode,
            addr=args.get( "PEER" ),
            routerId=args.get( "ROUTER_ID" ),
            internal=args.get( "INTERNAL" ),
            taskStats=args.get( "TASK_STATS" ),
            iar=args.get( "iar" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpv6BgpPeers )

#-------------------------------------------------------------------------------
# "show ipv6 bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowIpv6BgpSummary( ShowCliCommandClass ):
   syntax = """show ipv6 bgp summary [ PREFIX ] [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "summary": summary,
         "PREFIX": Ip6PrefixMatcher( "BGP peers filtered by prefix",
                                     overlap=PREFIX_OVERLAP_AUTOZERO ),
         "VRF": allVrfExprFactory,
      }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowIpv6BgpSummary(
            mode,
            prefix=args.get( "PREFIX" ),
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowIpv6BgpSummary )

#-------------------------------------------------------------------------------
# "show ipv6 bgp [ labeled-unicast ]
#     ( ( <ip> ) | ( <prefix> [ longer-prefixes ] [ detail ] ) )
#     [ internal ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------
class ShowIpv6BgpAddrOrPrefix( ShowCliCommandClass ):
   syntax = """show ipv6 bgp [ labeled-unicast ]
               ( ( IP ) | ( PREFIX [ longer-prefixes ] [ detail ] ) )
               [ internal ] [ VRF ]"""
   data = {
         "ipv6": ipv6MatcherForShow,
         "bgp": bgpAfterShow,
         "labeled-unicast": labUni,
         "PREFIX": ip6PrefixMatcher,
         "longer-prefixes": longerPrefixes,
         "detail": detail,
         "IP": ip6AddrMatcher,
         "internal": internal,
         "VRF": allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      if args.get( "IP" ):
         addr = args.get( "IP" )
      else:
         addr = args.get( "PREFIX" )
      return CliPlugin.RoutingBgpShowCli.doShowIpv6Bgp(
            mode,
            addr=addr,
            longerPrefixes=args.get( "longer-prefixes" ),
            detail=args.get( "detail" ),
            internal=args.get( "internal" ),
            vrfName=args.get( "VRF" ),
            labUni=args.get( "labeled-unicast" ) )

addShowCommandClass( ShowIpv6BgpAddrOrPrefix )
