# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from BasicCli import addShowCommandClass
from CliCommand import (
      CliExpression,
   )
from CliMatcher import (
      IntegerMatcher,
   )
from CliPlugin.AclCli import (
      accessListKwMatcherForServiceAcl,
      ipAclNameExpression,
   )
import CliPlugin.AclCliModel
from CliPlugin.RoutingBgpCli import (
      peergroupNameMatcher,
      V4V6PeerKeyCliExpression,
   )
import CliPlugin.RoutingBgpShowCli
from CliPlugin.RoutingBgpShowCli import (
      allVrfExprFactory,
      peerGroupVrfModel,
      showBgpConvergenceVrfModel,
      statisticsVrfModel,
      updateGroupVrfModel,
   )
from CliToken.Ip import ipMatcherForShow
from CliToken.RoutingBgpShowCliTokens import (
      AddrFamilyAfterShowBgp,
      bgpAfterShow,
      bgpConvergence,
      bgpInstance,
      hiddenKeyword,
      internal,
      internalDumpDuplicates,
      peerGroupAfterShow,
      statistics,
      bgpUpdateGroup,
   )
from ShowCommand import ShowCliCommandClass

# This file provides the CLI parser handling for BGP show commands that are
# at instance level, i.e., not specific to any one AFI/SAFI
# This file can also include "show [] bgp neighbor" style command
# that describes neighbor state in detail
# "show bgp [afi/safi] summary" style commands which show the neighbor
# status and number of prefixes received/accepted.

brib = hiddenKeyword( 'brib', 'BGP RIB state' )
countAfterBrib = hiddenKeyword( 'count', 'Path Count' )
bribDetail = hiddenKeyword( 'detail', 'Include more information' )
bribStash = hiddenKeyword( 'stash', 'Path Count Got from Stash' )
events = hiddenKeyword( 'events', 'internal events tracking' )
eventsTracking = hiddenKeyword( 'tracking', 'internal events tracking' )
eventsStart = hiddenKeyword( 'start', 'start internal events tracking' )
eventsDetail = hiddenKeyword( 'detail', 'show events tracking detail' )
eventsAdvtThreshold = hiddenKeyword( 'advertise-threshold',
      'route advertisement threshold' )
eventsStop = hiddenKeyword( 'stop', 'stop internal events tracking' )
eventsSummary = hiddenKeyword( 'summary', 'show events tracking summary' )
eventsIar = hiddenKeyword( 'iar', 'In-place FEC Replace' )
eventsIarHistory = hiddenKeyword( 'history', 'show history last 100 events' )

#-------------------------------------------------------------------------------
# "show bgp [ipv4|ipv6] access-list [<aclName>]
#     [summary] [dynamic] [detail]"
#-------------------------------------------------------------------------------
class ShowBgpAccessList( ShowCliCommandClass ):
   syntax = """show bgp [ AF_AFTER_BGP ] access-list [ ACL_NAME_OPTIONS ]"""
   data = {
         "bgp": bgpAfterShow,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "access-list": accessListKwMatcherForServiceAcl,
         "ACL_NAME_OPTIONS": ipAclNameExpression,
      }
   cliModel = CliPlugin.AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.showBgpAcl( mode,
            addressFamily=args.get( "addrFamily" ),
            name=args.get( "<aclNameExpr>" ) )

addShowCommandClass( ShowBgpAccessList )

class InternalOrExpression( CliExpression ):
   expression = """(
         ( internal [dump-duplicates] ) |
         ( internal brib count [detail] ) |
         ( internal events iar history ) |
         ( internal brib count stash ) |
         ( internal events tracking start [ advertise-threshold
               RIB_ADD_THRESHOLD RIBOUT_ADV_THRESHOLD ] ) |
         ( internal events tracking stop ) |
         ( internal events tracking summary ) |
         ( internal events tracking detail )
      )"""
   data = {
         "internal": internal,
         "dump-duplicates": internalDumpDuplicates,
         "brib": brib,
         "count": countAfterBrib,
         "detail": bribDetail,
         "stash": bribStash,
         "events": events,
         "tracking": eventsTracking,
         "start": eventsStart,
         "advertise-threshold": eventsAdvtThreshold,
         "RIB_ADD_THRESHOLD": IntegerMatcher( 0, 4294967294,
            helpdesc='threshold value for RIB routes' ),
         "RIBOUT_ADV_THRESHOLD": IntegerMatcher( 0, 4294967294,
            helpdesc='threshold value for RibOut routes' ),
         "stop": eventsStop,
         "summary": eventsSummary,
         "iar": eventsIar,
         "history": eventsIarHistory,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      if "internal" not in args:
         return

      kwRename = {
            "internal": None,
            "dump-duplicates": None,
            "brib": None,
            "count": None,
            "detail": None,
            "stash": None,
            "events": None,
            "tracking": None,
            "start": None,
            "advertise-threshold": None,
            "RIB_ADD_THRESHOLD": "numRibRoutes",
            "RIBOUT_ADV_THRESHOLD": "numRibOutRoutes",
            "stop": None,
            "summary": None,
            "iar": None,
            "history": None,
         }
      result = {}
      for kw, newName in kwRename.iteritems():
         newName = newName or kw
         if kw in args:
            result[ newName ] = args.pop( kw )

      args[ "internal" ] = result

#-------------------------------------------------------------------------------
# "show bgp instance
#     (hidden options) internal [dump-duplicates]
#     (hidden options) internal [brib count [detail]]
#     (hidden options) internal [events iar history]
#     (hidden options) internal [events tracking start [advertise-threshold
#                       <rib-add-threshold> <ribout-advertise-threshold>]]
#     (hidden options) internal [events tracking stop]
#     (hidden options) internal [events tracking summary]
#     (hidden options) internal [events tracking detail]
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpInstanceInternal( ShowCliCommandClass ):
   syntax = "show bgp instance INTERNAL_EXPR [ VRF ]"
   data = {
      "bgp": bgpAfterShow,
      "instance": bgpInstance,
      "INTERNAL_EXPR": InternalOrExpression,
      "VRF": allVrfExprFactory,
   }

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpInstance(
         mode,
         internal=args[ "internal" ],
         vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpInstanceInternal )

#-------------------------------------------------------------------------------
# "show bgp instance [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpInstance( ShowCliCommandClass ):
   syntax = "show bgp instance [ VRF ]"
   data = {
         "bgp": bgpAfterShow,
         "instance": bgpInstance,
         "VRF": allVrfExprFactory,
      }
   cliModel = CliPlugin.RoutingBgpShowCli.cliShowBgpInstanceModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpInstance(
            mode,
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpInstance )

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpConvergence( ShowCliCommandClass ):
   syntax = """show bgp convergence [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "convergence": bgpConvergence,
         "VRF": allVrfExprFactory,
      }
   cliModel = showBgpConvergenceVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpConvergence(
            mode, vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpConvergence )

#-------------------------------------------------------------------------------
# "show [ip] bgp peer-group [peergroupName] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpPeerGroup( ShowCliCommandClass ):
   syntax = """show [ ip ] bgp peer-group [ PG_NAME ] [ VRF ]"""
   # "ip" is accepted for backward compatibility reason
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "peer-group": peerGroupAfterShow,
         "PG_NAME": peergroupNameMatcher,
         "VRF": allVrfExprFactory,
      }
   cliModel = peerGroupVrfModel

   @staticmethod
   def handler( mode, args ):
      pgName = args.get( "PG_NAME" )
      vrfName = args.get( "VRF" )
      return CliPlugin.RoutingBgpShowCli.doShowIpBgpPeerGroup( mode,
            pgAndVrfName=( pgName, vrfName ) )

addShowCommandClass( ShowBgpPeerGroup )

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpStatistics( ShowCliCommandClass ):
   syntax = """show bgp statistics [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "statistics": statistics,
         "VRF": allVrfExprFactory,
      }
   cliModel = statisticsVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpStatistics(
            mode, vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpStatistics )

#-------------------------------------------------------------------------------
# "show bgp update-group
#     (index | A.B.C.D | A:B:C:D:E:F:G:H | LL_ADDR)
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpUpdateGroup( ShowCliCommandClass ):
   syntax = """show bgp update-group
               [ ( INDEX | PEER_ADDR ) ]
               [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "update-group": bgpUpdateGroup,
         "INDEX": IntegerMatcher( 1, 4294967295,
            helpdesc="Index of the update group" ),
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "VRF": allVrfExprFactory,
      }
   cliModel = updateGroupVrfModel

   @staticmethod
   def handler( mode, args ):
      opt = args.get( "INDEX" ) or args.get( "PEER" )
      return CliPlugin.RoutingBgpShowCli.doShowBgpUpdateGroup( mode,
            opt=opt,
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpUpdateGroup )
