# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for BGP exec mode commands

# pylint: disable=anomalous-backslash-in-string, protected-access
import Tac, LazyMount, Tracing, os, sys, Cell
import BasicCli
import CliToken.Clear, CliToken.Ip, CliToken.Ipv6
import CliToken.RoutingBgpShowCliTokens as ShowTokens
import Arnet
from BgpLib import createKey
from BgpLib import doLogClearIpBgp
from BgpLib import PeerConfigKey
import IpAddrMatcher
import Ip6AddrMatcher
from IpGenAddrMatcher import isIpv4
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import (
      VrfExecCmdDec,
      generateVrfCliModel,
)
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import re
import AutoAggrModels
import BgpCliModels
from TunnelCli import getTunnelIdFromIndex, getTunnelIndexFromId, getViaModels
from TunnelCli import isDyTunIntfId
from TunnelCli import appendEntryLabelsToViaModels, getDyTunTidFromIntfId
from TunnelCli import (
   readMountTunnelTable,
   TunnelTableIdentifier,
)
import SmashLazyMount
import AclCli
import ConfigMount
from socket import AF_INET, AF_INET6
import CliCommand
import CliMatcher
from RoutingBgpCli import ( ipv4PeerMatcher, ipv6PeerMatcher,
                            ipv6LlPeerMatcher, peergroupNameMatcher )
import ShowCommand

traceHandle = Tracing.Handle( 'Bgp' )
# pylint: disable=no-member
# pylint: disable=multiple-statements, no-self-use
# pylint: disable=undefined-loop-variable, redefined-outer-name, unused-variable
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

bgpConfig = None
bgpClearReq = None
bgpClearResp = None
aclListConfig = None
allVrfConfig = None
bgpVrfClearReqDir = None
bgpVrfClearRespDir = None
bgpVrfConfigDir = None
bgpLuTunnelTable = None
bgpLuPushTunnelTable = None
bgpLuLfib = None 
l3Config = None
l3ProtocolAgentModelStatus = None
defaultVrfProtocolStatus = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
bgpTestControlReq = None

# Global dictionary to overloaded functions for ArBgp support
arBgpShowCmdDict = {}
ribdBgpShowCmdDict = {}

# Address family Identifier
AFI_IP = 1
AFI_IP6 = 2
# Sub Address family Identifier
PA4MP_SAFI_UNI = 1
PA4MP_SAFI_SR_TE = 73

# Following APIs enable us to avoid direct symbolic dependency on Ribd in
# similar way as arBgpShowCmdDict based mechanism works for ArBgp CLI
# implementation
def cliRibdShowCommand():
   cb = ribdBgpShowCmdDict.get( 'cliRibdShowCommand' )
   assert cb is not None
   return cb

def showRibCapiCommand():
   return ribdBgpShowCmdDict.get( 'showRibCapiCommand' )

def showRibDamiCommand():
   return ribdBgpShowCmdDict.get( 'showRibDamiCommand' )

def EmptyResponseException():
   return ribdBgpShowCmdDict.get( 'EmptyResponseException' )

def AmiResponseException():
   return ribdBgpShowCmdDict.get( 'AmiResponseException' )

def getBgpPeerListModel():
   return ribdBgpShowCmdDict.get( 'BgpPeerListModel' )

invalidPosixRegexErrMsg = 'Invalid posix regular expression'

def doShowBgpInactive( mode, *args, **kwargs ):
   mode.addError( 'BGP inactive' )
   return None

def doShowBgpOutputNotSupported( mode, *args, **kwargs ):
   mode.addError( 'Not supported' )
   return None

class ArBgpShowOutput( object ):
   '''
   Decorator class that can be used to tag that a certain cli command's handler
   function can be redirected to Arbgp implementation, based on whether ribd is
   running or Arbgp is running.
   Function calls using the new cli parser should avoid using positinal arguments
   in place of keyword arguments
   @vrfLookup: For commands which are supported in both ribd and arbgp modes,
   this decorator will lookup and set the vrfName parameter.  For commands which
   are arbgp only this is not done, unless the vrfLookup=True is also set.
   '''
   def __init__( self, arBgpShowCallback, arBgpModeOnly=False,
                 ribdModeOnly=False, vrfLookup=False ):
      self.arBgpShowCallback = arBgpShowCallback
      self.arBgpModeOnly = arBgpModeOnly
      self.ribdModeOnly = ribdModeOnly
      self.vrfLookup = vrfLookup or not arBgpModeOnly
      assert not ( arBgpModeOnly and ribdModeOnly )
   def __call__( self, func ):
      def arBgpShowOutput( *args, **kwargs ):
         protocolAgentModel = l3Config.protocolAgentModel
         if protocolAgentModel == ProtoAgentModel.multiAgent:
            if self.ribdModeOnly:
               return doShowBgpOutputNotSupported( *args, **kwargs )
            if protocolAgentModel != \
                  l3ProtocolAgentModelStatus.protocolAgentModel or \
                  not defaultVrfProtocolStatus.entityPtr.get( 'Bgp' ):
               return doShowBgpInactive( *args, **kwargs )
            elif self.arBgpModeOnly:

               callback = arBgpShowCmdDict.get( self.arBgpShowCallback,
                                                func )
            else:
               callback = arBgpShowCmdDict.get( self.arBgpShowCallback, 
                                                doShowBgpOutputNotSupported )
            if self.vrfLookup:
               # The VrfExecCmdDec sets up the vrfName correctly even when the cmd
               # is being executed under a routing-context and the vrfName is not
               # specified as an argument. In case of ArBgp though, some show cmds 
               # have C++ implementation which do not use the VrfExecCmdDec, due to
               # which these commands give default vrf output even when they are run
               # under some routing-context vrf mode.
               # The fix is to do the vrf lookup here, before invoking the 
               # corresponding show command function in ArBgpCli.py with the correct
               # vrfName.
               # There is a special case for "show ip bgp peer-group" command where
               # vrfName is not part of kwargs but rather it is part of a list 
               # represented by kwargs['pgAndVrfName']
               vrfName = kwargs['pgAndVrfName'][1] if self.arBgpShowCallback == \
                         "doShowIpBgpPeerGroup" else kwargs.get( 'vrfName', None )
               vrfName = VrfCli.vrfMap.lookupCliModeVrf( args[ 0 ], vrfName )
               if self.arBgpShowCallback == "doShowIpBgpPeerGroup":
                  kwargs['pgAndVrfName'] = ( kwargs['pgAndVrfName'][0], vrfName )
               elif 'vrfName' in kwargs:
                  kwargs['vrfName'] = vrfName
            return callback( *args, **kwargs )
         else:
            if not self.arBgpModeOnly:
               return func( *args, **kwargs )
            else:
               return doShowBgpOutputNotSupported( *args, **kwargs )
      return arBgpShowOutput

class BgpVrfRoutingContextDefaultOnly( object ):
   """Decorator class to process show commands according to the VRF in
   routing-context VRF mode.
   These show commands should display output only in case of default VRF.
   For non-default VRF, no output should be displayed.
   """
   def __init__( self, cliModel ):
      """Takes cliModel with which the cmd is registered as constructor arg"""
      self.cliModel = cliModel

   def __call__( self, func ):
      def bgpVrfRoutingContext( *args, **kwargs ):
         """Performs a vrf lookup and returns a default( empty ) instance of
         cliModel if vrfName is not default.
         For non-default VRF, the text output will be empty and CAPI output will
         be an empty model.
         """
         mode = args[ 0 ] # mode is always the first arg
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName=None )
         if vrfName != DEFAULT_VRF:
            return self.cliModel()

         return func( *args, **kwargs )

      return bgpVrfRoutingContext

BGP_ROUTE_V6            = 0x0001
BGP_ROUTE_DETAIL        = 0x0002

BGP_NOT_SPECIFIED = 0x00
BGP_RECEIVED_ROUTES = 0x01
BGP_ADVERTISED_ROUTES = 0x02
BGP_RECV_FILTERED_ROUTES = 0x03
BGP_NOT_INSTALLED_ROUTES = 0x4
BGP_INSTALLED_ROUTES = 0x5

cliribdFastPort = "9889"

BGP_SAFI_ALL = 0
BGP_SAFI_UNICAST = 1
BGP_SAFI_LABELS = 2
BGP_SAFI_SR_TE = 3

# pylint: disable=unnecessary-lambda
allVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True, inclAllVrf=True )

routeSummaryVrfModel = generateVrfCliModel( BgpCliModels.BgpRouteHeader,
                                            'Per VRF BGP route summary',
                                            uncheckedModel=True,
                                            revision=3 )

queuedWithdrawalsVrfModel = generateVrfCliModel(
   BgpCliModels.BgpQueuedWithdrawals,
   'Per VRF BGP queued withdrawals',
   uncheckedModel=True )

COMMUNITY_NO_EXPORT = 0xffffff01 #no export outside confederation
COMMUNITY_NO_ADVERTISE = 0xffffff02 #no advertise to other peers
COMMUNITY_NO_EXPORT_SUB = 0xffffff03 #no export to external peers
COMMUNITY_INTERNET = 0 #community internet
COMMUNITY_GSHUT = 0xffff0000 #community GSHUT

def getCommunityValuesScalarList( commValues ):
   commList = []
   for cv in commValues:
      if isinstance( cv, int ):
         commList.append( int( cv ) )
      elif isinstance( cv, long ):
         commList.append( long( cv ) )
      elif cv == 'no-export':
         commList.append( COMMUNITY_NO_EXPORT )
      elif cv == 'no-advertise':
         commList.append( COMMUNITY_NO_ADVERTISE )
      elif cv == 'local-as':
         commList.append( COMMUNITY_NO_EXPORT_SUB )
      elif cv == 'internet':
         commList.append( COMMUNITY_INTERNET )
      elif cv == 'GSHUT':
         commList.append( COMMUNITY_GSHUT )
      elif ':' in cv:
         ( asn, num ) = cv.split( ':' )
         commList.append( int( asn ) << 16 | int( num ) & 0xffff )

   return commList

# vrfName is defaulted to None so as to allow cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def _doShowIpBgpRoute( mode, ipv6=False, prefix=None, detail=None,
                       commValues=None, commListName=None, exact=None, peer=None,
                       showPeerColumn=False, familyFilterType=None, filterType=None,
                       aspRegex=None, commRegex=None, highPerf=False, flag=0,
                       longerPrefixes=False, internal=None, vrfName=None,
                       labUni=None, srTe=None, largeCommValues=None,
                       largeCommListName=None ):

   cliRibd = False
   cliRibdCmd = ""

   args = {}
   if ipv6:
      flag |= BGP_ROUTE_V6
      args = { 'ipv6' : int( ipv6 ) }
   cmd = 'MIO_DGET_BGP_ROUTE'
   mlen = None
   if srTe is not None:
      args[ 'safi_filter_type' ] = BGP_SAFI_SR_TE
   elif labUni:
      args[ 'safi_filter_type' ] = BGP_SAFI_LABELS
   else:
      args[ 'safi_filter_type' ] = BGP_SAFI_ALL
   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if detail is not None:
      flag |= BGP_ROUTE_DETAIL
      args[ 'show_detail' ] = 1
      if srTe is not None:
         # For SR-TE SAFI, aggregation and policy advertisement
         # are not supported. Hence only check if the tunnel encap
         # has to be displayed
         args[ 'show_sr_te_tunnel_encap' ] = 1
         if prefix is not None:
            args[ 'show_prefix_detail' ] = 1
      else:
         args[ 'show_contributors' ] = 1
         if not longerPrefixes and prefix is not None and '/' in prefix:
            args[ 'show_prefix_detail' ] = 1
   else:
      args[ 'show_detail' ] = 0
   # Use Rib fast port for show ip|ipv6 bgp prefix
   if prefix is not None:
      if '/' in prefix:
         addr, mlen = prefix.split( '/' )
         args[ 'walkdown' ] = int( longerPrefixes )
      else:
         addr = prefix
         args[ 'walkup' ] = 1
      if srTe is not None:
         if isIpv4( addr ):
            args[ 'addrSrTeV4' ] = addr
         else:
            args[ 'addrSrTeV6' ] = addr
      elif isIpv4(addr):
         args[ 'addrv4' ] = addr
         if mlen:
            args[ 'masklenv4' ] = int( mlen )
      else:
         args[ 'addrv6' ] = addr
         if mlen:
            args[ 'masklenv6' ] = int( mlen )
      if not longerPrefixes:
         args[ 'show_detail' ] = 1
   if peer is not None:
      if peer.type == 'peerIpv4':
         key = 'peerv4'
      elif peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         key = 'peerv6'
      else:
         assert False, "Cannot handle peer type: %s" % ( peer.type )

      args[ key ] = peer.gatedRepr()

      # Right now, options for the above commands (communities, detail, etc.) 
      # will still go through RibCapi
      peerRouteFilters = { BGP_NOT_SPECIFIED: 'routes',
                           BGP_RECEIVED_ROUTES: 'received-routes',
                           BGP_ADVERTISED_ROUTES: 'advertised-routes',
                           BGP_RECV_FILTERED_ROUTES: 'received-routes filtered' }

      if ( detail or prefix or commValues or aspRegex or commRegex or labUni or srTe
           or highPerf or largeCommValues ):
         cliRibd = False
      elif filterType in peerRouteFilters:
         routeType = peerRouteFilters[ filterType ]
         cliRibd = True
         cliRibdCmd = 'show ip%s bgp' % [ '', 'v6' ][ ipv6 ]
         cliRibdCmd += ' peer %s ' % ( peer.gatedRepr() )
         if familyFilterType:
            cliRibdCmd += 'ipv6 ' if familyFilterType == BGP_V6_ROUTES else 'ipv4 '
         cliRibdCmd += '%s' % routeType
   else:
      #
      # For the following show commands, fall back to cliribd for performance:
      #
      # 'show ip{v6} bgp installed'
      # 'show ip{v6} bgp not-installed'
      #
      routeFilters = { BGP_INSTALLED_ROUTES : 'installed',
                       BGP_NOT_INSTALLED_ROUTES : 'not-installed' }
      if filterType in routeFilters:
         routeType = routeFilters[ filterType ]
         cliRibd = True
         cliRibdCmd = 'show ip%s bgp' % [ '', 'v6' ][ ipv6 ]
         cliRibdCmd += ' %s' % routeType
   if showPeerColumn:
      args[ 'show_peers' ] = 1

   if familyFilterType is not None:
      args[ 'family_filter_type' ] = familyFilterType
   if filterType is not None:
      args[ 'filter_type' ] = filterType
   if commValues is not None:
      args[ 'communities' ] = getCommunityValuesScalarList( commValues )
   if largeCommValues is not None:
      args[ 'largeCommunities' ] = largeCommValues
   if exact is not None:
      args[ 'exact' ] = 1
   if aspRegex is not None:
      args[ 'aspath_regex' ] = aspRegex
   if commRegex is not None:
      args[ 'comm_regex' ] = commRegex
   if commListName is not None:
      if exact is not None:
         commListName = ".exact." + commListName
      args[ 'commlist_filter' ] = commListName
   if largeCommListName is not None:
      if exact is not None:
         largeCommListName = ".exact." + largeCommListName
      args[ 'largeCommList_filter' ] = largeCommListName

   if cliRibd and mode.session_.outputFormat_ == 'text':
      cliRibdShowCommand()( mode, cliRibdCmd, clientName="BGP", vrf=vrfName )
      return None
   elif highPerf and mode.session_.outputFormat_ == "text" and prefix is None:
      args[ 'format' ] = 2 
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args,
                                   clientName='BGP' )
   else:
      try:
         args = args or None
         return showRibCapiCommand()( mode, BgpCliModels.BgpRouteHeader,
                                      cmd, args, clientName='BGP', 
                                      highPerf=highPerf )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

#-------------------------------------------------------------------------------
# Explanation of Bgp/ArBgp command decorators
# The ordering of the decorators on these commands matters, but is implementation
# dependent

# ArBgpShowOutput takes a key to the arBgpShowCmdDict (see ArBgp CliPlugins).
# This key determines which function to call

# VrfExecCmdDec accepts a vrf function, which is uses to generate a series of
# vrfs, and a cliModel. This cli model can have the __cliPrinted__ attribute set
# to true, which means that it has already been written to console by the
# cli printer. 

# This is where the ordering matters. If the command referenced by the key in
# ArBgpShowOutput uses the c++ implementation that renders using cli print then
# it is placed first. These commands should return a cliPrinted model.
# The VrfExecCmdDec will then exit without doing any additional work,
# as the output has already been rendered

# If VrfExecCmdDec is placed first then it will iterate over each vrf on the
# test device and call the ArBgp function for each vrf name. With cli printed
# commands this will result in multiple renders and, in json output, multiple
# objects where there should be only one
#------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] [ <ip> | <prefix> | ( installed | not-installed ) ]
#  [ longer-prefixes ] [internal] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def getVrfsFunc():
   return allVrfConfig.vrf.members()

@ArBgpShowOutput( 'doShowIpBgp' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgp( mode, addr=None, longerPrefixes=None, detail=None, internal=None,
                 vrfName=None, labUni=None ):
   flag = 0
   highPerf = True
   longerPrefixes = bool( longerPrefixes )
   if detail:
      flag |= BGP_ROUTE_DETAIL
   routeType = BGP_NOT_SPECIFIED 

   result = _doShowIpBgpRoute( mode, prefix=addr, detail=detail,
                               highPerf=highPerf, flag=flag,
                               longerPrefixes=longerPrefixes,
                               filterType=routeType,
                               vrfName=vrfName, labUni=labUni )
   return result

@ArBgpShowOutput( 'doShowIpBgpInstalled' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpInstalled( mode, addr=None, notInstalled=None, installed=None,
                          vrfName=None, internal=None ):
   flag = 0
   highPerf = True
   routeType = BGP_NOT_SPECIFIED
   if notInstalled:
      highPerf = False
      routeType = BGP_NOT_INSTALLED_ROUTES
   elif installed:
      highPerf = False
      routeType = BGP_INSTALLED_ROUTES
   
   result = _doShowIpBgpRoute( mode, prefix=addr, highPerf=highPerf, flag=flag,
                               filterType=routeType, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp [labeled-unicast] [ <ip> | <prefix> | [installed] | 
# [ not-installed ] ] [ longer-prefixes ] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6Bgp' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6Bgp( mode, addr=None, longerPrefixes=None, detail=None, vrfName=None,
                   internal=None, labUni=None ):
   highPerf = True
   longerPrefixes = bool( longerPrefixes )
   routeType = BGP_NOT_SPECIFIED 
   if addr != None:
      prefix = addr.stringValue
   else:
      prefix = None

   result = _doShowIpBgpRoute( mode, prefix=prefix, detail=detail,
                               longerPrefixes=longerPrefixes,
                               filterType=routeType,
                               ipv6=True, highPerf=highPerf, vrfName=vrfName,
                               labUni=labUni )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpInstalled' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpInstalled( mode, addr=None, notInstalled=None, installed=None,
                            vrfName=None, internal=None ):
   highPerf = True
   routeType = BGP_NOT_SPECIFIED
   if notInstalled:
      highPerf = False
      routeType = BGP_NOT_INSTALLED_ROUTES
   elif installed:
      highPerf = False
      routeType = BGP_INSTALLED_ROUTES
   if addr != None:
      prefix = addr.stringValue
   else:
      prefix = None

   result = _doShowIpBgpRoute( mode, prefix=prefix,
                               filterType=routeType,
                               ipv6=True, highPerf=highPerf, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# Deprecated version of the command:
# "show ipv6 bgp community [<communities>] [exact] [detail] [vrf <vrfName>]"
#
# New Version of the command:
# "show ipv6 bgp match community [<communities>] [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesCommunities( mode, values=None, exact=None, detail=None,
                                    internal=None, vrfName=None,
                                    labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni,
                               detail=detail,
                               commValues=set( values ),
                               exact=exact, ipv6=True,
                               vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpRoutesDetail' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesDetail( mode, detail=None,
                               internal=None, vrfName=None,
                               labUni=None, epeOnly=None ):
   highPerf = True
   result = _doShowIpBgpRoute( mode, detail=detail, ipv6=True,
                               vrfName=vrfName, highPerf=highPerf,
                               labUni=labUni )
   return result

#----------------------------------------------------------------------
# show ip bgp large-community [<large-community name>] [exact] [detail]
#----------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpLargeCommunity', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpLargeCommunity( mode, values=None, detail=None,
                               vrfName=None, exact=None, labUni=None,
                               epeOnly=None ):

   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               vrfName=vrfName, exact=exact,
                               largeCommValues=set( values ) )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpLargeCommunity', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpLargeCommunity( mode, values=None, detail=None,
                               labUni=None, epeOnly=None,
                               vrfName=None, exact=None ):

   result = _doShowIpBgpRoute( mode, detail=detail, ipv6=True,
                               vrfName=vrfName, exact=exact, labUni=labUni,
                               largeCommValues=set( values ) )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp community-list [<community-list name>] [exact] [detail]
# [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesCommunityList( mode, commListName=None, exact=None,
                                      labUni=None, epeOnly=None,
                                      detail=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode, detail=detail,
                               commListName=commListName,
                               exact=exact, ipv6=True,
                               labUni=labUni,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
#"show ip(v6) bgp large-community-list [<large-community-list name>] [exact]
# [detail]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesLargeCommunityList', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesLargeCommunityList( mode, largeCommListName=None, exact=None,
        detail=None, vrfName=None, labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               largeCommListName=largeCommListName,
                               exact=exact,
                               vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpRoutesLargeCommunityList', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesLargeCommunityList( mode, largeCommListName=None, exact=None,
        detail=None, vrfName=None, labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               largeCommListName=largeCommListName,
                               exact=exact, ipv6 = True,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]"
#     (hidden options) [internal [verbose]] [task-stats [reset]]
#-------------------------------------------------------------------------------
peerListVrfModel = generateVrfCliModel( BgpCliModels.BgpPeerListBase,
                                        "Per Vrf BGP Peer Information" )

def _doShowIpBgpNeighbors( mode, addr=None, bfd=None, internal=None,
                           vrfName=None, ipv6=False, highPerf=True ):
   cmd = 'MIO_DGET_BGP_PEER'
   args = {}

   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if ipv6:
      args[ 'ipv6' ] = True

   if addr is not None:
      if addr.type == 'peerIpv4':
         key = 'addrv4'
      elif addr.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         key = 'addrv6'
      else:
         assert False, "Cannot handle peer type: %s" % ( addr.type )

      args[ key ] = addr.gatedRepr()

   peerModel = getBgpPeerListModel()

   try:
      retval = showRibCapiCommand()( mode, peerModel, 
                                     cmd, args, clientName='BGP', 
                                     highPerf=highPerf,
                                     skipNullResponses=False )
      if retval is not None and bfd is not None:
         retval._showBfd = True
      return retval
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=peerListVrfModel )
def doShowIpBgpNeighbors( mode, addr=None, bfd=None, internal=None,
                          taskStats=None, iar=None, vrfName=None, routerId=None ):
   return _doShowIpBgpNeighbors( mode, addr, bfd, internal, vrfName )

@ArBgpShowOutput( 'doShowIpv6BgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=peerListVrfModel )
def doShowIpV6BgpNeighbors( mode, addr=None, bfd=None, internal=None,
                            taskStats=None, iar=None, vrfName=None,
                            routerId=None ):
   return _doShowIpBgpNeighbors( mode, addr, bfd, internal, vrfName,
                                 ipv6=True )

#-------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     [ community [<communities>] [exact] ] [detail]
#     [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNlriAf' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpNlriAf( mode, addressFamilyAndType=None,
                     pfxAddr=None, longerPrefixes=None,
                     communityValuesAndExact=None, detail=None,
                     vrfName=None, internal=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show bgp ( ipv4 unicast | ipv6 unicast | ipv4 multicast | ipv6 multicast )
# ( <ip> | <prefix> [longer-prefixes] ) [ community [<communities>] [exact] ]
# [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
def getNlriAfForPfxAddrRule( pfxAddr ):
   if isinstance( pfxAddr, ( Tac.Type( 'Arnet::Ip6Addr' ),
                             Tac.Type( 'Arnet::Ip6AddrWithMask' ) ) ):
      nlriAf = 'ipv6'
   else:
      assert isinstance( pfxAddr, str )
      nlriAf = 'ipv4'
   return nlriAf

#-------------------------------------------------------------------------------
# "show bgp neighbors [<ip>] [vrf <vrfName>]"
#     (hidden options) [internal [verbose] [rib-out] [dump-duplicates]]
#                      [task-stats [reset]]
#
# Note: 'verbose' option prints AdjRibin paths
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=peerListVrfModel )
def doShowBgpNeighbors( mode, addr=None, internal=None, taskStats=None,
                        vrfName=None ):
   doShowBgpOutputNotSupported( mode )


#-------------------------------------------------------------------------------
# "show ip bgp
#     [labeled-unicast]
#     community <communities> [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunities( mode, values=None, exact=None, detail=None,
                                  labUni=None, epeOnly=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode,
                               labUni=labUni,
                               commValues=set( values ),
                               exact=exact,
                               detail=detail,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp community regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunitiesRegex' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunitiesRegexp( mode, regex=None, vrfName=None ):
   result = showBgpCommunityRegexpHelper( mode, regex, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesDetail' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesDetail( mode, detail=None, vrfName=None,
                             labUni=None, epeOnly=None ):
   highPerf = True
   flag = 0
   if detail:
      flag |= BGP_ROUTE_DETAIL
   result = _doShowIpBgpRoute( mode, detail=detail,
                               highPerf=highPerf, flag=flag,
                               vrfName=vrfName, labUni=labUni )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] community-list [<community-list name>] [exact]
#     [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunityList( mode, commListName=None, exact=None, detail=None,
                                    labUni=None, epeOnly=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode, detail=detail,
                               labUni=labUni,
                               commListName=commListName,
                               exact=exact,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp peer-group [peergroupName] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

def doParseNeighborIntoDict( mode, family, vrfName ):
   '''
   Sends the correct CLI w.r.t to af to cliribd and
   receives the output back in dict format
   '''
   cmd = 'show %s bgp' % ( 'ip' if family == 'v4' else 'ipv6' )
   cmd += ' neighbor pyformat'
   out = cliRibdShowCommand()( mode, cmd, clientName="BGP", output = False,
                               vrf=vrfName )
   return eval( '{' + out.replace('\n', "") + '}' )

def doParseDynamicNeighborsIntoDict( mode, vrfName, pgName ):
   '''
   Sends the correct CLI w.r.t to peer-group info to cliribd and
   receives the output back in dict format
   '''
   cmd = 'show ip bgp'
   cmd += ' peer-group %s pyformat' % pgName
   out = cliRibdShowCommand()( mode, cmd, clientName="BGP", output = False,
                               vrf=vrfName )
   return eval( '{' + out.replace('\n', "") + '}' )

def configForVrf( vrfName ):
   config = bgpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         config = bgpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

# We get pyformat dicts from cliribd for "show ip[v6] bgp neighbors".
# We fetch relevant information from these dicts and use it in this show command for
# static peer group members
# For dynamic peer groups, we get pyformat dicts for 'show ip bgp peer-group <name>'
# and fetch the information from these dicts. Peer-group information 
# from different VRFs is then put together

# example:
# peerGroups['pg1' ] = 
#       {'as':100,
#        'static':{'vrf1':{'v4':[],'v6':[]},'vrf2':{'v4':[],'v6':[]},...},
#        'listen':{'vrf1':{'networkAndAs':[],'vrf2':{'networkAndAs':[]},... },
#        'dynamic':{'vrf1':{},'vrf2':{},...}}
def compilePeerGroupInfo( mode, name='', vrfName=None ):
   '''Prepare our list of configured peer Groups and its attributes
      peerGroups members are keyed by peer-group name'''
   # top-level PeerGroup info dict
   peerGroups = {}
   # empty peer-group info dicts that can be cloned
   peerGroupMembersEmpty = { 'v4' : [], 'v6' : [] }
   peerGroupListenRangeEmpty = { 'networkAndAs' : [] }

   def setPeerGroupDefaults( peerGroups, key, vrf ):
      peerGroups.setdefault( key, { 'as' : None, 'static' : {},
                                    'listen' : {}, 'dynamic' : {}, 'intf' : {} } )
      peerGroups[ key ][ 'static' ].setdefault( vrf, { 'v4' : [], 'v6' : [] } )
      peerGroups[ key ][ 'listen' ].setdefault( vrf, { 'networkAndAs' : [] } )
      peerGroups[ key ][ 'dynamic' ].setdefault( vrf, {} )
      peerGroups[ key ][ 'intf' ].setdefault( vrf, False )

   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName == ALL_VRF_NAME:
      vrfList = [ DEFAULT_VRF ]
      vrfList.extend( getVrfsFunc() )
   else:
      vrfList = [ vrfName ]

   defaultVrfBc = configForVrf( DEFAULT_VRF )
   for vrf in vrfList:
      bc = configForVrf( vrf )
      if bc is None:
         continue
      # iterate over static peers first
      for peer in bc.neighborConfig.values():
         if peer.isPeerGroupPeer:
            key = peer.peerGroupKey.group
            if name and key != name:
               continue
            setPeerGroupDefaults( peerGroups, key, vrf ) 
            if peer.key.type == 'peerIpv4':
               peerGroups[ key ][ 'static' ][ vrf ][ 'v4' ].append( peer.key )
            elif peer.key.type == 'peerIpv6' or peer.key.type == 'peerIpv6Ll':
               peerGroups[ key ][ 'static' ][ vrf ][ 'v6' ].append( peer.key )
            peerGroups[ key ][ 'as' ] = \
                  defaultVrfBc.neighborConfig[ peer.peerGroupKey ].asNumber
         elif peer.key.group == 'peerGroup':
            if name and key != name:
               continue
            key = peer.key.group
            setPeerGroupDefaults( peerGroups, key, vrf )
            peerGroups[ key ][ 'as' ] = peer.asNumber

      # iterate over listen-range config next
      for lr in bc.listenRange.values():
         key = lr.peergroupName
         if name and key != name:
            continue
         setPeerGroupDefaults( peerGroups, key, vrf )
         for network in lr.prefixList:
            peerGroups[ key ][ 'listen' ][ vrf ][ 'networkAndAs' ].append( 
               ( str( network ), lr.prefixList[ network ] ) )
         dynPeerGroup = doParseDynamicNeighborsIntoDict( mode, vrf, key )
         if key in dynPeerGroup:
            peerGroups[ key ][ 'dynamic' ][ vrf ] = dynPeerGroup[ key ]

      # iterate over the "neighbor interface" config
      for peer in bc.neighIntfConfig.values():
         key = peer.peerGroup.group
         # if a peer group filter is specified, consider only if it matches
         # the filter
         if name and key != name:
            continue
         setPeerGroupDefaults( peerGroups, key, vrf )
         peerGroups[ key ][ 'intf' ][ vrf ] = True

   return peerGroups

desc = "Per-VRF BGP peer groups"
peerGroupVrfModel = generateVrfCliModel( BgpCliModels.BgpPeerGroup, desc )

@ArBgpShowOutput( 'doShowIpBgpPeerGroup' )
def doShowIpBgpPeerGroup( mode, pgAndVrfName=( '', None ) ):
   '''
   This show command prints information pertaining to peer-groups 
   # Peer-groups can have both static and dynamic (listen-range) members
   and they can span multiple VRFs
   '''
   vrfName = pgAndVrfName[ 1 ]
   # do a quick sanity check to see if BGP is active
   out = cliRibdShowCommand()( mode, 'show bgp statistics',
                               clientName="BGP", output=False,
                               vrf=vrfName )
   if 'BGP inactive' in out:
      print out
      return None
   pgName = pgAndVrfName[ 0 ]
   peerGroups = compilePeerGroupInfo( mode, pgName, vrfName )
   indent = ' '

   def numStaticPeers( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'static' ]:
         num += len( peerGroup[ 'static' ][ vrf ][ 'v4' ] )
         num += len( peerGroup[ 'static' ][ vrf ][ 'v6' ] )
      return num

   def numListenRange( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'listen' ]:
         num += len( peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ] )
      return num

   def numDynamicPeers( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'dynamic' ]:
         num += len( peerGroup[ 'dynamic' ][ vrf ] )
      return num

   def hasInterfacePeers( v6PeerInfo, peerGroup ):
      for vrf in v6PeersInfo:
         for peer in v6PeersInfo[ vrf ]:
            if isInterfacePeer( vrf, peer, peerGroup ):
               return True
      return False

   def isInterfacePeer( vrf, peer, peerGroup ):
      bc = configForVrf( vrf )
      peerKey = PeerConfigKey( peer )
      return peerKey.type == 'peerIpv6Ll' and \
             peerKey not in bc.neighborConfig and \
             peerKey.llIntf in bc.neighIntfConfig and \
             bc.neighIntfConfig[ peerKey.llIntf ].peerGroup.group == peerGroup

   def displayPeerDetails( peerStr, peerInfo ):
      print '%s%s, state: %s' % ( indent * 6, peerStr, peerInfo[ 'state' ] )
      print '%sNegotiated MP Capabilities:' % ( indent * 8 )
      print '%sIPv4 Unicast: %s' % \
             ( indent * 12, 'Yes' if peerInfo[ 'mpv4u' ] else 'No' )
      print '%sIPv6 Unicast: %s' % \
             ( indent * 12, 'Yes' if peerInfo[ 'mpv6u' ] else 'No' )
      print '%sIPv4 SR-TE: %s' % \
             ( indent * 12, 'Yes' if peerInfo[ 'mpv4srte' ] else 'No' )
      print '%sIPv6 SR-TE: %s' % \
             ( indent * 12, 'Yes' if peerInfo[ 'mpv6srte' ] else 'No' )

   # Iterate through our peer Group list
   # pylint: disable=too-many-nested-blocks
   for key in sorted( peerGroups.keys() ):
      peerGroup = peerGroups[ key ]
      asAndTypeStr = ""
      if peerGroup[ 'as' ]:
         asType = 'internal' if peerGroup[ 'as' ] == bgpConfig.asNumber \
               else 'external'
         asAndTypeStr = ', remote AS %d, peer-group %s' % \
               ( peerGroup[ 'as' ], asType )
      print 'BGP peer-group is %s%s' % ( key, asAndTypeStr )
      print '%sBGP version 4' % ( indent*2 )

      # Holds the information about all the v6 peers. This is used to display
      # information about interface peers (i.e) peers configured using
      # "neighbor interface" command.
      # Logic to fetch the info about interface peers:
      # interface peers = all v6 peers - static v6 peers - dynamic v6 peers
      v6PeersInfo = {}
      for vrf in peerGroup[ 'static' ].keys():
         if peerGroup[ 'static' ][ vrf ][ 'v6' ] or peerGroup[ 'intf' ][ vrf ]:
            v6PeersInfo[ vrf ] = doParseNeighborIntoDict( mode, 'v6', vrf )

      # static peer-group members info
      if numStaticPeers( peerGroup ):
         print '%sStatic peer-group members:' % ( indent * 2 )
      for vrf in sorted( peerGroup[ 'static' ].keys() ):
         if ( peerGroup[ 'static' ][ vrf ][ 'v4' ] or \
               peerGroup[ 'static' ][ vrf ][ 'v6' ] ):
            print '%sVRF %s:' % ( indent*4, vrf )
         else:
            continue
         for family in [ 'v4', 'v6' ]:
            if family == 'v6':
               if not peerGroup[ 'static' ][ vrf ][ family ]:
                  continue
               nbrsInfo = v6PeersInfo[ vrf ]
            else:
               nbrsInfo = doParseNeighborIntoDict( mode, family, vrf )
            for peer in peerGroup[ 'static' ][ vrf ][ family ]:
               peerStr = peer.stringValue
               ni = None
               peerAddr = None
               if peerStr in nbrsInfo:
                  ni = nbrsInfo[ peerStr ]
                  peerAddr = peerStr
               elif peer.gatedRepr() in nbrsInfo:
                  # Applicable only to link-local peers:
                  # if the interface is not known to gated the response
                  # from gated will have the interface name in kernal format
                  # as opposed to eos format. (i.e) fe80::1%et1 as opposed to
                  # fe80::1%Et1
                  ni = nbrsInfo[ peer.gatedRepr() ]
                  peerAddr = peer.gatedRepr()
               if ni:
                  displayPeerDetails( peerStr, ni )
               if vrf in v6PeersInfo and peerAddr in v6PeersInfo[ vrf ]:
                  # Remove static v6 peers. We will now have the dynamic peers
                  # and the interface peers for this vrf
                  del v6PeersInfo[ vrf ][ peerAddr ]
      
      # listen range subnets info
      if numListenRange( peerGroup ):
         print '%sListen-range subnets:' % ( indent*2 )
      for vrf in sorted( peerGroup[ 'listen' ].keys() ):
         if peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ]:
            print '%sVRF %s:' % ( indent*4, vrf )
         else:
            continue
         for lr in peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ]:
            peerSpec = lr[ 1 ]
            if peerSpec.hasAsn:
               spec = 'remote AS %s' % peerSpec.asn
            else:
               spec = 'peer filter %s' % peerSpec.peerFilter
            print '%s%s, %s' % ( indent*6, lr[ 0 ], spec )

      # dynamic peer-group members info
      if numDynamicPeers( peerGroup ):
         print '%sDynamic peer-group members:' % ( indent * 2 )
      for vrf in sorted( peerGroup[ 'dynamic' ].keys() ):
         if peerGroup[ 'dynamic' ][ vrf ]:
            print '%sVRF %s:' % ( indent*4, vrf )
         else:
            continue
         for peer in peerGroup[ 'dynamic' ][ vrf ][ 'peer-group-members' ]:
            if vrf in v6PeersInfo and peer[ 'addr' ] in v6PeersInfo[ vrf ]:
               # Remove dynamic v6 peers. We will now have only the
               # interface peers for this vrf
               del v6PeersInfo[ vrf ][ peer[ 'addr' ] ]
            print '%s%s, state: %s' % ( indent*6, peer[ 'addr' ], 'Established' )
            print '%sNegotiated MP Capabilities:' % ( indent*8 )
            print '%sIPv4 Unicast: %s' % \
                ( indent*12, peer[ 'capabilities' ][ 0 ][ 'ncaps-mpv4u' ] )
            print '%sIPv6 Unicast: %s' % \
                ( indent*12, peer[ 'capabilities' ][ 1 ][ 'ncaps-mpv6u' ] )
            print '%sIPv4 SR-TE: %s' % \
                ( indent*12, peer[ 'capabilities' ][ 2 ][ 'ncaps-mpv4srte' ] )
            print '%sIPv6 SR-TE: %s' % \
                ( indent*12, peer[ 'capabilities' ][ 3 ][ 'ncaps-mpv6srte' ] )

      if hasInterfacePeers( v6PeersInfo, key ):
         print '%sInterface peer-group members:' % ( indent * 2 )
         for vrf in sorted( v6PeersInfo.keys() ):
            if v6PeersInfo[ vrf ]:
               bc = configForVrf( vrf )
               print '%sVRF %s:' % ( indent * 4, vrf )
               for peer in v6PeersInfo[ vrf ]:
                  if isInterfacePeer( vrf, peer, key ):
                     ni = v6PeersInfo[ vrf ][ peer ]
                     displayPeerDetails( peer, ni )
   return peerGroupVrfModel()

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes |
#   advertised-routes [queued-withdrawals] ) [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------

BGP_V4_ROUTES = 0x01
BGP_V6_ROUTES = 0x02

def showBgpRegexHelperValidateParams( mode, regex, nbrAddr, routeType, ipv6 ):
   if len( regex ) > 256:
      mode.addError( 'Regular expression exceeds the maximum allowed'
                     'length of 256 characters' )

   if ipv6:
      ipv6 = 1
   else:
      ipv6 = 0
   if nbrAddr:
      if routeType == 'routes':
         routeType = BGP_NOT_SPECIFIED
      elif routeType == 'advertised-routes':
         routeType = BGP_ADVERTISED_ROUTES
      elif routeType == 'received-routes filtered':
         routeType = BGP_RECV_FILTERED_ROUTES
      else:
         routeType = BGP_RECEIVED_ROUTES

   return ipv6, routeType

def getRegexMode():
   return aclListConfig.regexMode

def validateAspRegex( mode, regex ):
   regexMode = aclListConfig.regexMode
   posixRegex = regexMode == 'regexModePosix'
   regexModeStr = "string" if posixRegex else "asn"

   helper = Tac.newInstance( "Routing::RouteMap::Helper" )
   if not ( ( posixRegex and helper.validatePosixRegex( regex ) ) or
            ( not posixRegex and helper.validateDfaRegex( regex ) ) ):
      mode.addError( "Invalid regular expression under regex-mode %s" %
                     regexModeStr )
      return False
   return True

def showBgpRegexpHelper( mode, regex, nbrAddr=None,
                         routeType=None, showPeerColumn=False,
                         ipv6=False, vrfName=DEFAULT_VRF ):

   if not validateAspRegex( mode, regex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, regex, nbrAddr,
                                                       routeType, ipv6 )
   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType,
                               aspRegex=regex,
                               vrfName=vrfName )
   return result

def validatePosixRegex( mode, regex ):
   try:
      re.compile( regex )
      return True
   except re.error:
      mode.addError( invalidPosixRegexErrMsg )
      return False

def showBgpCommunityRegexpHelper( mode, regex, nbrAddr=None, routeType=None,
                                 ipv6=False, vrfName=DEFAULT_VRF ):
   if not validatePosixRegex( mode, regex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, regex, nbrAddr,
                                                       routeType, ipv6 )
   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               filterType=routeType,
                               commRegex=regex,
                               vrfName=vrfName )
   return result

def showBgpCombinedRegexpHelper( mode, aspRegex, commRegex, nbrAddr=None,
                                 routeType=None, ipv6=False,
                                 vrfName=DEFAULT_VRF ):
   # Validate both regex's seperately
   if aspRegex is not None and not validateAspRegex( mode, aspRegex ):
      return False
   if commRegex is not None and not validatePosixRegex( mode, commRegex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, '', nbrAddr,
                                                       routeType, ipv6 )
   if ( aspRegex and len( aspRegex ) > 256 ) or ( commRegex and
                                                  len( commRegex ) > 256 ):
      mode.addError( 'Regular expression exceeds the maximum allowed'
                     'length of 256 characters' )

   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               filterType=routeType,
                               aspRegex=aspRegex,
                               commRegex=commRegex,
                               vrfName=vrfName )
   mode.addError( str( result ) )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsRoutes( mode, peerAddr, routeType,
                                routeFamilyAndType=None, detail=None,
                                vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr, highPerf=True,
                               familyFilterType=familyFilterType, 
                               filterType=routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsReceivedRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsReceivedRoutes( mode, peerAddr=None,
                                        filtered=None, routeFamilyAndType=None,
                                        detail=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not peerAddr and not detail

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               highPerf=True, familyFilterType=familyFilterType,
                               filterType=routeType, vrfName=vrfName )

   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsQueuedWithdrawals' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=queuedWithdrawalsVrfModel )
def doShowIpBgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                           routeFamilyAndType=None, vrfName=None ):
   # queued-withdrawals option to advertised-routes only supported in ArBgp
   return doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# Deprecated Command:
# "show ipv6 bgp neighbors <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes | advertised-routes ) [ vrf <vrfName> ]"
#
# New Version:
# "show ipv6 bgp peers <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes |
#   advertised-routes [queued-withdrawals] ) [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsRoutes( mode, peerAddr, routeType,
                                  routeFamilyAndType=None,
                                  detail=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr, highPerf=True,
                               familyFilterType=familyFilterType, ipv6=True,
                               filterType=routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsReceivedRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsReceivedRoutes( mode, peerAddr=None,
                                          filtered=None, routeFamilyAndType=None,
                                          detail=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not peerAddr and not detail
   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               familyFilterType=familyFilterType, ipv6=True,
                               filterType=routeType, vrfName=vrfName )

   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsQueuedWithdrawals' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=queuedWithdrawalsVrfModel )
def doShowIpv6BgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                             routeFamilyAndType=None, vrfName=None ):
   # queued-withdrawals option to advertised-routes only supported in ArBgp
   return doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { received-routes [ filtered ] | routes | 
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNbrRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNbrRtsRegex( mode, nbrAddr, routeType, regex, vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpBgpNbrRecvdRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNbrRecvdRtsRegex( mode, regex, nbrAddr=None,
                                 filtered=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not nbrAddr
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType,
                                 showPeerColumn=showPeerColumn,
                                 vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors <ip> { received-routes [ filtered ] | routes | 
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers <ip> { received-routes [ filtered ] | routes |
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNbrRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNbrRtsRegex( mode, nbrAddr, routeType, regex, vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType, ipv6=True, 
                                 vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNbrRecvdRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNbrRecvdRtsRegex( mode, regex, nbrAddr=None,
                                   filtered=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not nbrAddr
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType,
                                 showPeerColumn=showPeerColumn,
                                 ipv6=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
# community < C > [ exact ] [ detail ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNeighborsCommunityRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsCommunityRoutes( mode, addr, routeType, values,
                                         exact=None, detail=None, vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsCommunityRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsCommunityRecvdRoutes( mode, values, addr=None,
                                              filtered=None,
                                              exact=None, detail=None, 
                                              vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors <ip> {routes|advertised-routes|
# received-routes [ filtered ] }
# community <C> [ exact ] [ detail ] [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers <ip> {routes|advertised-routes|
# received-routes [ filtered ] }
# community <C> [ exact ] [ detail ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNeighborsCommunityRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsCommunityRoutes( mode, addr, routeType, values,
                                         exact=None, detail=None, vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsCommunityRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsCommunityRecvdRoutes( mode, values, addr=None,
                                                filtered=None,
                                                exact=None, detail=None,
                                                vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNeighborsPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsPrefixAddr ( mode, peerAddr, routeType, pfxAddr,
                                     longerPrefixes=None, vrfName=None ):
   v6 = False
   if isinstance(pfxAddr, Tac.Type("Arnet::Ip6AddrWithMask")):
      v6 = True

   longerPrefixes = bool( longerPrefixes )
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   if routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, 
                               filterType=routeType, ipv6=v6, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsPrefixAddrRecvdRts' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsPrefixAddrRecvdRts( mode, pfxAddr, peerAddr=None,
                                             filtered=None,
                                             longerPrefixes=None, vrfName=None ):
   v6 = False
   if isinstance(pfxAddr, Tac.Type("Arnet::Ip6AddrWithMask")):
      v6 = True

   longerPrefixes = bool( longerPrefixes )
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, 
                               filterType=routeType, ipv6=v6, vrfName=vrfName,
                               highPerf=True )
   return result

#-------------------------------------------------------------------------------
# Deprecated commands:
# "show ipv6 bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNeighborsPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsPrefixAddr ( mode, peerAddr, routeType, pfxAddr,
                                     longerPrefixes=None, vrfName=None ):
   
   longerPrefixes = bool( longerPrefixes )
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   if routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), 
                               peer=peerAddr,
                               longerPrefixes=longerPrefixes, ipv6=True,
                               filterType=routeType, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsPrefixAddrRecvdRts' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsPrefixAddrRecvdRts( mode, pfxAddr, peerAddr=None,
                              filtered=None, longerPrefixes=None, vrfName=None ):

   longerPrefixes = bool( longerPrefixes )
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), 
                               peer=peerAddr, showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, ipv6=True,
                               filterType=routeType, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> [ ( ipv4 unicast | ipv6 unicast | ipv4 multicast ) ]
#       ( routes | advertised-routes | received-routes [filtered] )
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighborsNlriAfRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpNeighborsNlriAfRoutes( mode, peerAddr,
                                    routeFamilyAndType=None,
                                    routeType=None, filtered=None,
                                    communityValuesAndExact=None,
                                    detail=None, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip>
#       ( routes | advertised-routes | received-routes [filtered] )
#       ( <ip> | <prefix> [longer-prefixes] )
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighborsNlriAfPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpNeighborsNlriAfPrefixAddr( mode, peerAddr,
                                        routeType=None, filtered=None,
                                        pfxAddr=None, longerPrefixes=None,
                                        communityValuesAndExact=None,
                                        detail=None, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show ip bgp paths [vrf <vrfName>]"
#-------------------------------------------------------------------------------
pathsVrfModel = generateVrfCliModel( BgpCliModels.BgpASPathList,
                                     "Per VRF BGP AS path info list" )

def _doShowIpBgpPaths( mode, vrfName ):
   args = {}
   cmd = 'MIO_DGET_BGP_ASPATH'
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   try:
      return showRibCapiCommand()( mode, BgpCliModels.BgpASPathList, cmd, args,
                                   clientName='BGP', highPerf=True )
   except ( EmptyResponseException(), AmiResponseException() ):
      return None


@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=pathsVrfModel )
@ArBgpShowOutput( 'doShowIpBgpPaths' )
def doShowIpBgpPaths( mode, vrfName=None ):
   return _doShowIpBgpPaths( mode, vrfName )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------
updateErrorsVrfModel = generateVrfCliModel( BgpCliModels.BgpNeighborUpdateError,
                                     "Per VRF BGP Neighbor Update-Error list" )
def _doShowIpBgpUpdateErrors( mode, ipv6, vrfName, addr=None ):
   args = {}
   cmd = 'MIO_DGET_BGP_UPDATE_ERROR'
   args[ 'ipv6' ] = ipv6
   if addr is not None:
      if ipv6 is 1:
         args[ 'addrv6' ] = addr.stringValue
      else:
         args[ 'addrv4' ] = addr
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   try:
      return showRibCapiCommand()( mode, BgpCliModels.BgpNeighborUpdateError, cmd,
                                   args, clientName='BGP', skipNullResponses=False )
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpUpdateErrors' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=updateErrorsVrfModel )
def doShowIpBgpUpdateErrors( mode, addr=None, vrfName=None ):
   return _doShowIpBgpUpdateErrors( mode, 0, vrfName, addr )

class ShowIpBgpNeighborsUpdateErrorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip bgp neighbors [ ADDRESS ] update errors [ VRF ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'bgp': ShowTokens.bgpAfterShow,
      'neighbors': ShowTokens.neighbors,
      'ADDRESS': IpAddrMatcher.IpAddrMatcher( 'Neighbor address' ),
      'update': 'BGP update information',
      'errors': ShowTokens.errors,
      'VRF': allVrfExprFactory
   }

   cliModel = updateErrorsVrfModel

   @staticmethod
   def handler( mode, args ):
      return doShowIpBgpUpdateErrors( mode, addr=args.get( 'ADDRESS' ),
         vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIpBgpNeighborsUpdateErrorsCmd )

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpUpdateErrors' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=updateErrorsVrfModel )
def doShowIp6BgpUpdateErrors( mode, addr=None, vrfName=None ):
   return _doShowIpBgpUpdateErrors( mode, 1, vrfName, addr )

# New Version
class ShowIp6BgpNeighborsUpdateErrorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 bgp peers [ ADDRESS ] update errors [ VRF ]'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'bgp': ShowTokens.bgpAfterShow,
      'peers': ShowTokens.peers,
      'ADDRESS': Ip6AddrMatcher.Ip6AddrMatcher( 'Neighbor address' ),
      'update': 'BGP update information',
      'errors': ShowTokens.errors,
      'VRF': allVrfExprFactory
   }

   cliModel = updateErrorsVrfModel

   @staticmethod
   def handler( mode, args ):
      return doShowIp6BgpUpdateErrors( mode, addr=args.get( 'ADDRESS' ),
         vrfName=args.get( 'VRF' ) )

BasicCli.addShowCommandClass( ShowIp6BgpNeighborsUpdateErrorsCmd )

#-------------------------------------------------------------------------------
# "show ip bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
desc = "Per VRF BGP summary"
summaryVrfModel = generateVrfCliModel( BgpCliModels.BgpSummary, desc )

def _doShowIpBgpSummary( mode, afi, safi, prefix, vrfName ):
   cmd = 'MIO_DGET_BGP_SUMMARY'
   args = {}
   if afi is not None:
      args[ 'afi' ] = int( afi )
   if safi is not None:
      args[ 'safi' ] = safi
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if prefix:
      return doShowBgpOutputNotSupported( mode )

   try:
      return showRibCapiCommand()( mode, BgpCliModels.BgpSummary, cmd, args,
                                   highPerf=True, clientName='BGP' )
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpSummary' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=summaryVrfModel )
def doShowIpBgpSummary( mode, prefix=None, vrfName=None ):
   return _doShowIpBgpSummary( mode, AFI_IP, PA4MP_SAFI_UNI, prefix, vrfName )

#-------------------------------------------------------------------------------
# "show ipv6 bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpSummary' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=summaryVrfModel )
def doShowIpv6BgpSummary( mode, prefix=None, vrfName=None ):
   return _doShowIpBgpSummary( mode, AFI_IP6, PA4MP_SAFI_UNI, prefix, vrfName )

#-----------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     summary [vrf <vrfName>]
#-----------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNlriAfSummary' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=summaryVrfModel )
def doShowBgpNlriAfSummary( mode, addressFamilyAndType, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#----------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] summary [vrf <vrfName>]
#----------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeSummary' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=summaryVrfModel )
def doShowBgpSrTeSummary( mode, addressFamily=None, vrfName=DEFAULT_VRF ):
   afi = 0
   if addressFamily == 'ipv4':
      afi = AFI_IP
   elif addressFamily == 'ipv6':
      afi = AFI_IP6
   if vrfName != DEFAULT_VRF:
      return doShowBgpOutputNotSupported( mode )
   return _doShowIpBgpSummary( mode, afi, PA4MP_SAFI_SR_TE,
                               prefix=None, vrfName=vrfName )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] [ detail ] [vrf <vrfName>]"
# "show bgp sr-te endpoint <> color <> distinguisher <> [detail] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutes( mode, addressFamily=None, endpoint=None, color=None,
                         distinguisher=None, detail=None, vrfName=DEFAULT_VRF ):
   flag = BGP_ROUTE_DETAIL if detail is not None else 0
   highPerf = True
   routeType = BGP_NOT_SPECIFIED
   familyFilterType = None
   prefix = None
   if endpoint is not None and color is not None and distinguisher is not None:
      prefix = str( distinguisher ) + '|' + str( color ) + '|' + str( endpoint )
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, prefix=prefix, detail=detail,
                               highPerf=highPerf, flag=flag, srTe=True,
                               filterType=routeType, vrfName=vrfName,
                               familyFilterType=familyFilterType )
   return result

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community [<com>] [exact] [detail] [vrf <name>]"
#---------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutesCommunities( mode, addressFamily=None, values=None,
                                    exact=None, detail=None,
                                    vrfName=DEFAULT_VRF ):
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, detail=detail, commValues=set( values ),
                               familyFilterType=familyFilterType,
                               exact=exact, srTe=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community-list [<community-list name>]
#       [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutesCommunityList( mode, addressFamily=None, commListName=None,
                                      exact=None, detail=None, vrfName=None ):
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, detail=detail, commListName=commListName,
                               familyFilterType=familyFilterType,
                               exact=exact, srTe=True, vrfName=vrfName )
   return result

#-----------------------------------------------------------------------------
# "show bgp neighbors <neigh_addr> [ ipv4 | ipv6 ] sr-te
#                 ( policies | received-policies) [ detail ] [vrf <vrfName>]"
#-----------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeNeighborRoutes' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeNeighborRoutes( mode, addressFamily=None, policyFilter=None,
                                 detail=None, peerAddr=None, vrfName=DEFAULT_VRF ):
   flag = BGP_ROUTE_DETAIL if detail is not None else 0
   routeType = BGP_NOT_SPECIFIED
   if policyFilter == 'received-policies':
      routeType = BGP_RECEIVED_ROUTES
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, peer=peerAddr, detail=detail,
                               highPerf=True, flag=flag, srTe=True,
                               filterType=routeType, vrfName=vrfName,
                               familyFilterType=familyFilterType )
   return result

#--------------------------------------------------------------------
# The "show bgp [ipv4|ipv6] access-list
#     [<aclName>]
#     [summary] [dynamic] [detail]" command
#--------------------------------------------------------------------
def showBgpAcl( mode, addressFamily=None, name=None ):
   aclFilter = None
   if addressFamily == 'ipv4':
      aclFilter = 'ip'
   elif addressFamily == 'ipv6':
      aclFilter = 'ipv6'

   return AclCli.showServiceAcl( mode, aclCpConfig, aclStatus, aclCheckpoint,
                                 aclFilter, name, serviceName='bgpsacl' )

#-------------------------------------------------------------------------------
# "show ip bgp regexp <regex> [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRegexp' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpBgpRegexp( mode, regex, vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp regexp <regex> [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRegexp' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRegexp( mode, regex, vrfName ):
   result = showBgpRegexpHelper( mode, regex, ipv6=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]"
#-------------------------------------------------------------------------------
desc = "Per VRF BGP Statistics"
statisticsVrfModel = generateVrfCliModel( BgpCliModels.BgpStatistics, desc )

@ArBgpShowOutput( 'doShowBgpStatistics' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=statisticsVrfModel )
def doShowBgpStatistics( mode, vrfName=None ):
   # vrfName is set to None so that the cliRibdShowCommand can deduce the
   # correct vrfName given the cli context in which it resides
   if mode.session_.outputFormat_ == "json":
      mode.addError( "This is an unconverted command" )
      return
   cmd = 'show bgp'
   cmd += ' statistics'
   cliRibdShowCommand()( mode, cmd, clientName="BGP", vrf=vrfName )

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]"
#-------------------------------------------------------------------------------
showBgpConvergenceVrfModel = generateVrfCliModel( 
   BgpCliModels.ShowBgpConvergenceVrfModel,
   "Per VRF BGP convergence information." )
@ArBgpShowOutput( 'doShowBgpConvergence' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=showBgpConvergenceVrfModel )
def doShowBgpConvergence( mode, vrfName=DEFAULT_VRF ):
   args = {}
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   cmd = "MIO_DGET_BGP_CONVERGENCE"

   if mode.session_.outputFormat_ == "text":
      args[ 'fastpath' ] = 1
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args, clientName='BGP' )
   else:
      try:
         args[ 'fastpath' ] = 0
         return showRibCapiCommand()( mode, BgpCliModels.ShowBgpConvergenceVrfModel,
                                      cmd, args, clientName='BGP',
                                      skipNullResponses=False )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

#-------------------------------------------------------------------------------
# "show bgp instance
#     (hidden options) [internal [dump-duplicates]]
#     (hidden options) [internal brib count [detail]]
#     (hidden options) [internal events tracking start [advertise-threshold
#                       <rib-add-threshold> <ribout-advertise-threshold>]]
#     (hidden options) [internal events tracking stop]
#     (hidden options) [internal events tracking summary]
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------

# Handle internal events tracking commands
# The 'start' and 'show' commands are handled inline by directly updating relevant
# attributes in Sysdb.
# The 'summary' command is handled by cliribd
def doShowBgpEventsTracking( mode, internal, vrfName ):
   if internal.get( 'start' ):
      if internal.get( 'advertise-threshold' ):
         numRibRoutes = internal.get( 'numRibRoutes' )
         numRibOutRoutes = internal.get( 'numRibOutRoutes' )
         bgpTestControlReq.ribAddThreshold = int( numRibRoutes )
         bgpTestControlReq.ribOutPathAdvertiseThreshold = int( numRibOutRoutes )
      print 'Starting BGP Events Tracking'
      bgpTestControlReq.eventsTrackingEnabled = True
      return

   if internal.get( 'stop' ):
      print 'Stopping BGP Events Tracking'
      bgpTestControlReq.eventsTrackingEnabled = False
      bgpTestControlReq.ribAddThreshold = 0
      bgpTestControlReq.ribRemoveThreshold = 0
      bgpTestControlReq.ribOutPathAdvertiseThreshold = 0
      bgpTestControlReq.ribOutPathWithdrawThreshold = 0
      return

   if internal.get( 'summary' ) or internal.get( 'detail' ):
      cmd = 'show bgp events-tracking-summary'
      # gated 'summary' command doesn't summarize events tracking data for all
      # the vrfs. Instead prints data for individual vrf.
      if not vrfName:
         vrfName = 'all'
      cliRibdShowCommand()( mode, cmd, clientName="BGP", vrf=vrfName )

cliShowBgpInstanceModel = generateVrfCliModel( BgpCliModels.ShowBgpInstanceModel,
   "per VRF bgp instance information models.", revision=2 )
# vrfName is defaulted to None so as to allow cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
@ArBgpShowOutput( 'doShowBgpInstance' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=cliShowBgpInstanceModel )
def doShowBgpInstance( mode, internal=None, vrfName=None ):
   if internal:
      if internal.get( 'events' ) and internal.get( 'tracking' ):
         doShowBgpEventsTracking( mode, internal, vrfName )
      return

   cmd = 'show bgp instance'
   if mode.session_.outputFormat_ == 'text':
      cliRibdShowCommand()( mode, cmd, clientName="BGP", vrf=vrfName )
   else:
      mode.addError( 'This is an unconverted command.' )

#-------------------------------------------------------------------------------
# "show bgp auto-aggregation ipv[4|6] [prefix] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
autoAggrVrfModel = generateVrfCliModel( AutoAggrModels.AutoAggregation,
                                        'Per VRF auto aggregation summary' )

def doShowAutoAggr( mode, v6=False, prefix=None, detail=None, vrfName=None ):
   args = {}
   args[ 'protocol' ] = 0
   args[ 'af' ] = AF_INET6 if v6 else AF_INET

   if prefix:
      if v6:
         # prefix type is Arnet::Ip6AddrWithMask
         args[ 'addrv6' ] = str( prefix.address )
         args[ 'maskv6' ] = prefix.len
      else:
         addr, mlen = prefix.split( '/' )
         args[ 'addrv4' ] = addr
         args[ 'maskv4' ] = int( mlen )

   if detail:
      args[ 'detail' ] = 1

   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      model = showRibCapiCommand()( mode, AutoAggrModels.AutoAggregation,
                                    'MIO_DGET_AUTO_AGGR', args=args, highPerf=True )
      model._vrf = vrfName
      return model
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowAutoAggrIpv4', ribdModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=autoAggrVrfModel )
def doShowAutoAggrIpv4( mode, prefix=None, detail=None, vrfName=None ):
   return doShowAutoAggr( mode, prefix=prefix, detail=detail,
                          vrfName=vrfName )

@ArBgpShowOutput( 'doShowAutoAggrIpv6', ribdModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=autoAggrVrfModel )
def doShowAutoAggrIpv6( mode, prefix=None, detail=None, vrfName=None ):
   return doShowAutoAggr( mode, v6=True, prefix=prefix, detail=detail,
                          vrfName=vrfName )

#-------------------------------------------------------------------------------
# "show bgp labeled-unicast tunnel [<tunnel-index>]"
#
# HIDDEN
# "show bgp labeled-unicast tunnel [<tunnel-index>] raw"
#-------------------------------------------------------------------------------
def getViaModelList( bgpTunnelTableEntry, flattenLuPush, labels ):
   vias = []
   useMplsVias = False
   if flattenLuPush:
      for via in bgpTunnelTableEntry.via.itervalues():
         if isDyTunIntfId( via.intfId ):
            tid = getDyTunTidFromIntfId( via.intfId )
            tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
            TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
            if tacTid.tunnelType() == TunnelType.bgpLuPushTunnel:
               useMplsVias = True
   for via in bgpTunnelTableEntry.via.itervalues():
      viaModels = getViaModels( via, bgpLuPushTunnelTable,
                                flattenLuPush=flattenLuPush, 
                                useMplsVias=useMplsVias )
      if viaModels:
         vias.extend( appendEntryLabelsToViaModels( viaModels, labels ) )
   return vias

def getBgpLuTunnelTableEntryModel( tunnelId, flattenLuPush=True ):
   vias = []
   useMplsVias = False

   bgpLuTunnelTableEntry = bgpLuTunnelTable.entry.get( tunnelId )
   if bgpLuTunnelTableEntry:
      labels = []
      for mplsStackIndex in \
         reversed( range( bgpLuTunnelTableEntry.labels.stackSize ) ):
         label = bgpLuTunnelTableEntry.labels.labelStack( mplsStackIndex )
         labels.append( str( label ) )
      vias = getViaModelList( bgpLuTunnelTableEntry, flattenLuPush, labels )
      return BgpCliModels.BgpLuTunnelTableEntry(
         endpoint=bgpLuTunnelTableEntry.tep, vias=vias, labels=labels,
         contribution=bgpLuTunnelTableEntry.contribution,
         bgpMetric=bgpLuTunnelTableEntry.bgpMetric,
         bgpMetric2=bgpLuTunnelTableEntry.bgpMetric2,
         bgpPref=bgpLuTunnelTableEntry.bgpPref,
         bgpPref2=bgpLuTunnelTableEntry.bgpPref2 )
   return None

def getBgpLuTunnelTable( tunnelIndex=None, flattenLuPush=True ):
   bgpLuEntries = {}

   if tunnelIndex is None:
      for tunnelId in bgpLuTunnelTable.entry:
         bgpLuEntryModel = getBgpLuTunnelTableEntryModel(
                              tunnelId=tunnelId, flattenLuPush=flattenLuPush )
         if bgpLuEntryModel:
            bgpLuEntries[ getTunnelIndexFromId( tunnelId ) ] = bgpLuEntryModel
   else:
      tunnelId = getTunnelIdFromIndex( 'bgpLuTunnel', tunnelIndex ) 
      bgpLuEntryModel = getBgpLuTunnelTableEntryModel( tunnelId=tunnelId, 
                                                       flattenLuPush=flattenLuPush )
      if bgpLuEntryModel:
         bgpLuEntries[ getTunnelIndexFromId( tunnelId ) ] = bgpLuEntryModel

   return BgpCliModels.BgpLuTunnelTable( entries=bgpLuEntries )

def showBgpLuTunnelTable( mode, tunnelIndex=None ):
   return getBgpLuTunnelTable( tunnelIndex=tunnelIndex, flattenLuPush=True )

def showBgpLuTunnelTableNoFlattening( mode, tunnelIndex=None ):
   return getBgpLuTunnelTable( tunnelIndex=tunnelIndex, flattenLuPush=False )

#-------------------------------------------------------------------------------
# HIDDEN:
# "show bgp labeled-unicast push [<tunnel-index>]"
# "show bgp labeled-unicast push [<tunnel-index>] raw"
#-------------------------------------------------------------------------------
def getBgpLuPushTunnelTableEntryModel( tunnelId, flattenLuPush=True ):
   vias = []
   useMplsVias = False

   bgpLuPushTunnelTableEntry = bgpLuPushTunnelTable.entry.get( tunnelId ) 
   if bgpLuPushTunnelTableEntry:
      labels = []
      for mplsStackIndex in \
         reversed( range( bgpLuPushTunnelTableEntry.labels.stackSize ) ):
         label = bgpLuPushTunnelTableEntry.labels.labelStack( mplsStackIndex )
         labels.append( str( label ) )
      vias = getViaModelList( bgpLuPushTunnelTableEntry, flattenLuPush, labels )
      return BgpCliModels.BgpLuPushTunnelTableEntry( vias=vias, labels=labels )
   return None

def getBgpLuPushTunnelTable( tunnelIndex=None, flattenLuPush=True ):
   bgpLuPushEntries = {}

   if tunnelIndex is None:
      for tunnelId in bgpLuPushTunnelTable.entry:
         bgpLuPushEntryModel = getBgpLuPushTunnelTableEntryModel( tunnelId=tunnelId,
            flattenLuPush=flattenLuPush )
         if bgpLuPushEntryModel:
            bgpLuPushEntries[ getTunnelIndexFromId( tunnelId ) ] = \
               bgpLuPushEntryModel
   else:
      tunnelId = getTunnelIdFromIndex( 'bgpLuPushTunnel', tunnelIndex ) 
      bgpLuPushEntryModel = getBgpLuPushTunnelTableEntryModel(
         tunnelId=tunnelId, flattenLuPush=flattenLuPush )
      if bgpLuPushEntryModel:
         bgpLuPushEntries[ getTunnelIndexFromId( tunnelId ) ] = bgpLuPushEntryModel

   return BgpCliModels.BgpLuPushTunnelTable( entries=bgpLuPushEntries )

def showBgpLuPushTunnelTable( mode, tunnelIndex=None ):
   return getBgpLuPushTunnelTable( tunnelIndex=tunnelIndex, flattenLuPush=True )

def showBgpLuPushTunnelTableNoFlattening( mode, tunnelIndex=None ):
   return getBgpLuPushTunnelTable( tunnelIndex=tunnelIndex, flattenLuPush=False )

#-------------------------------------------------------------------------------
# "show bgp update-group
#     (index | A.B.C.D | A:B:C:D:E:F:G:H | LL_ADDR)
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
updateGroupVrfModel = generateVrfCliModel( BgpCliModels.BgpUpdateGroups,
                                           "Per VRF BGP update group",
                                           uncheckedModel=True )

@ArBgpShowOutput( 'doShowBgpUpdateGroup' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=updateGroupVrfModel )
def doShowBgpUpdateGroup( mode, opt=None, vrfName=None ):
   cmd = 'MIO_DGET_BGP_UPDATE_GROUP'
   args = {}

   if opt is not None:
      if isinstance( opt, ( int, long ) ):
         args[ 'ug-index' ] = opt
      elif ':' in opt.gatedRepr():
         args[ 'addrv6' ] = opt.gatedRepr()
      else:
         args[ 'addrv4' ] = opt.gatedRepr()
   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if mode.session_.outputFormat_ == "text":
      args[ 'fastpath' ] = 1
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args, clientName='BGP' )
   else:
      try:
         args[ 'fastpath' ] = 0
         return showRibCapiCommand()( mode, BgpCliModels.BgpUpdateGroups,
                                      cmd, args, clientName='BGP',
                                      skipNullResponses=False )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

def isIpAddr( addr ):
   try:
      Arnet.IpGenAddr( addr )
   except ( ValueError, IndexError ):
      return False
   return True

@VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
@ArBgpShowOutput( 'doClearIpBgp' )
def doClearIpBgp( mode, transportAddrFamily=None, peer=None, soft=False, 
                  direction=None, vrfName=None,
                  errorsOrCounters=None, resetMsg='' ):

   if ( not peer and bgpConfig.clearAllDisabled and not errorsOrCounters ):
      mode.addError( "Cannot clear all peering sessions because "
                     "clear all commands are disabled" )
      return
   #
   # Log what we are attempting to do
   #
   doLogClearIpBgp( mode, transportAddrFamily, peer, soft, direction,
                    vrfName,
                    errorsOrCounters == 'errors',
                    errorsOrCounters == 'counters',
                    resetMsg )

   #
   # transportBasedClear is True in "clear ip(v6) bgp neighbor *" command.
   #       In this case only the Address family specific peers are hard reset.
   # transportBasedClear is False in "clear ip(v6) bgp *" command.
   #        In this case all peers are hard reset.
   # we derive the transportBasedClear from transportAddrFamily which is set only
   # in 'clear ip(v6) bgp neighbor *' command
   #
   transportBasedClear = transportAddrFamily is not None

   # if peer is specified, reset that peer.
   # if peer is not specified reset all peers if its 'clear ip(v6) bgp *'.
   # if peer is not specified reset only transportAddrFamily specific peers if
   # the command is 'clear ip(v6) bgp neighbor *'.
   if peer:
      peerKey = peer
   elif transportAddrFamily is None or transportAddrFamily == 'ip':
      peerKey = createKey( '0.0.0.0' )
   elif transportAddrFamily == 'ipv6':
      peerKey = createKey( '::' )

   k = "%d,%d" % ( os.getpid(), Tac.now() )
   if direction == 'in':
      d = 'peerClearSoftIn'
   elif direction == 'out':
      d = 'peerClearSoftOut'
   else:
      d = 'peerClearSoftBoth'
   req = Tac.Value( 'Routing::Bgp::PeerClearRequest', peerKey,
                     bool( peer is None ),
                     bool( soft or direction ), d,
                     transportBasedClear,
                     bool( errorsOrCounters == 'errors' ),
                     bool( errorsOrCounters == 'counters' ),
                     resetMsg )
   bcReq = None
   bcResp = None
   if vrfName != DEFAULT_VRF:
      bcReq = bgpVrfClearReqDir.clearRequestDir.newMember( vrfName )
      bcResp = bgpVrfClearRespDir.newEntity(
         'Routing::Bgp::ClearResponseDir', vrfName )
      description = 'BGP clear request to complete for VRF %s' % vrfName
      warningPost = ' for VRF %s' % vrfName
   else:
      bcReq = bgpClearReq
      bcResp = bgpClearResp
      description = 'BGP clear request to complete'
      warningPost = ''
   if ( bcReq is None ) or ( bcResp is None ):
      return
   bcReq.bgpPeerClearRequest[ k ] = req
   try:
      # If we are running as part of a BGP Cohab CLI clear test, we will not get a
      # response as the agent won't be running.  Timeout immediately to speed
      # up the testing.
      if 'BGP_COHAB_CLI_CLEAR_TEST' in os.environ:
         timeout = 0
      else:
         timeout = 5.0

      Tac.waitFor( lambda: k in bcResp.bgpPeerClearResponse,
                   description=description,
                   warnAfter=None, sleep=True, maxDelay=0.1, timeout=timeout )
   except Tac.Timeout:
      sessions = 'sessions%s' % warningPost \
            if peer is None else 'session%s' % warningPost
      mode.addWarning( "BGP %s may not have reset yet" % sessions )

   del bcReq.bgpPeerClearRequest[ k ]

bgpAfterClearIpMatcher = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='Reset BGP peering sessions' )
clearPeerAnyMatcher = CliMatcher.KeywordMatcher( '*',
      helpdesc=( 'Clear all IPv4 and IPv6 peering sessions '
                 '(Use neighbor * to clear only IPv4 or IPv6 sessions)' ),
      value=lambda mode, match: None )
clearReasonMatcher = CliMatcher.KeywordMatcher( 'reason',
      helpdesc='Hard reset reason to be sent to the peers' )
resetMsgMatcher = CliMatcher.StringMatcher( helpname='MESSAGE',
      helpdesc='Shut down communication reset message' )

class PeerClearCliExpression( CliCommand.CliExpression ):
   expression = '* | IP_PEER | IP6_PEER | IP6_LL_PEER | ( peer-group GROUP )'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher,
            'peer-group': 'BGP peer group information',
            '*': clearPeerAnyMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in
                     ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER', 'GROUP' )
                     if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

#-----------------------------------------------------------------------------------
# "clear ip bgp PEER (([vrf <vrfName>] [reason <message>]) |
#                    ([soft] [in|out] [errors] [vrf <vrfName>]))"
#  in privileged EXEC mode
#-----------------------------------------------------------------------------------

class ClearIpBgpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear [ ip | ipv6 ] bgp '
              '[ PEER ] '
              '[ ( [ VRF ] reason MESSAGE ) '
              '| ( [ soft ] [ DIRECTION ] [ STATISTICS ] [ VRF ] ) ]' )
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliToken.Ip.ipMatcherForClear,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'bgp': bgpAfterClearIpMatcher,
      'PEER': PeerClearCliExpression,
      'VRF': allVrfExprFactory,
      'reason': clearReasonMatcher,
      'MESSAGE': resetMsgMatcher,
      'soft': 'Do not reset the session',
      'DIRECTION': CliMatcher.EnumMatcher( {
         'in': 'Send a route refresh message',
         'out': 'Send all routes to the peer',
      } ),
      'STATISTICS': CliMatcher.EnumMatcher( {
         'errors': 'Reset the error statistics',
         'counters': 'Reset the peer counters',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      soft = "soft" in args
      vrfName = args.get( "VRF", DEFAULT_VRF )
      direction = args.get( 'DIRECTION' )
      errorsOrCounters = args.get( 'STATISTICS' )
      resetMsg = args.get( "MESSAGE", '' )

      doClearIpBgp( mode, peer=args[ 'PEER' ], soft=soft,
                    direction=direction, vrfName=vrfName,
                    errorsOrCounters=errorsOrCounters, resetMsg=resetMsg )

BasicCli.EnableMode.addCommandClass( ClearIpBgpCmd )

#-------------------------------------------------------------------------------
# "clear { ip | ipv6 } bgp neighbor [ * ] [ vrf <vrfName> ] reason [ MESSAGE ]"
#    in privileged EXEC mode      
#-------------------------------------------------------------------------------
class ClearIpBgpTransportCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear [ ip | ipv6 ] bgp neighbor '
              '[ * ] [ VRF ] [ reason MESSAGE ]' )
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliToken.Ip.ipMatcherForClear,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'bgp': bgpAfterClearIpMatcher,
      'neighbor': 'Clear based on transport address family',
      '*': ( 'Clear specified transport Address Family peering '
             'sessions (IPv4 or IPv6 accordingly)' ),
      'VRF': allVrfExprFactory,
      'reason': clearReasonMatcher,
      'MESSAGE': resetMsgMatcher
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( "VRF", DEFAULT_VRF )
      transportAddrFamily = None
      if "ip" in args:
         transportAddrFamily = "ip"
      elif "ipv6" in args:
         transportAddrFamily = "ipv6"
      resetMsg = args.get( "MESSAGE", "" )

      doClearIpBgp( mode, vrfName=vrfName,
                    transportAddrFamily=transportAddrFamily,
                    resetMsg=resetMsg )

BasicCli.EnableMode.addCommandClass( ClearIpBgpTransportCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global aclListConfig
   global bgpClearReq, bgpClearResp, allVrfConfig
   global bgpVrfClearRespDir, bgpVrfClearReqDir
   global bgpVrfConfigDir
   global bgpLuLfib
   global l3Config, l3ProtocolAgentModelStatus, defaultVrfProtocolStatus
   global bgpLuTunnelTable
   global bgpLuPushTunnelTable
   global aclCpConfig
   global aclCheckpoint
   global aclStatus
   global bgpTestControlReq

   bgpConfig = LazyMount.mount( entityManager, 'routing/bgp/config',
                                'Routing::Bgp::Config', 'w' )
   bgpClearReq = LazyMount.mount( entityManager, 'routing/bgp/clear/request',
                                  'Routing::Bgp::ClearRequestDir', 'w' )
   bgpClearResp = LazyMount.mount( entityManager, 'routing/bgp/clear/response',
                                   'Routing::Bgp::ClearResponseDir', 'r' )
   aclListConfig = LazyMount.mount( entityManager,
                                    'routing/acl/config',
                                    'Acl::AclListConfig', 'r' )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   bgpVrfClearReqDir = LazyMount.mount( entityManager,
                                        'routing/bgp/clear/vrf/request',
                                        'Routing::Bgp::VrfClearRequestDir', 'w' )
   bgpVrfClearRespDir = LazyMount.mount( entityManager,
                                         'routing/bgp/clear/vrf/response',
                                         'Tac::Dir', 'r' )
   bgpVrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                      'Routing::Bgp::VrfConfigDir', 'w' )
   bgpLuTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.bgpLuTunnelTable, entityManager )
   bgpLuPushTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.bgpLuPushTunnelTable, entityManager )
   bgpLuLfib = SmashLazyMount.mount( entityManager, 'mpls/protoLfibInputDir/bgpLu',
                             'Mpls::LfibStatus',
                             SmashLazyMount.mountInfo( 'reader' ) )
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   l3ProtocolAgentModelStatus = \
         LazyMount.mount( entityManager, 'l3/status/protocolAgentModelStatus',
                          'L3::ProtocolAgentModelStatus', 'r' )
   defaultVrfProtocolStatus = \
         LazyMount.mount( entityManager,
                          Cell.path( "routing/defaultVrfProtocolStatus" ),
                          "Tac::Dir", "ri" )
   bgpTestControlReq = LazyMount.mount( entityManager,
                                        'routing/bgp/testControlRequest',
                                        'Routing::Bgp::TestControlRequestDir', 'w' )
