#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import re

import BgpLib
from BgpLib import (
      NoOrDefault,
      PeerConfigKey,
      PeerConfigKeyType,
      vpnAfTypeCliKeywords
)
import CliCommand
import CliCommon
import CliMatcher
import CliToken.RoutingBgp as bgpTokens
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
import CliPlugin.RcfCliLib as RcfCliLib
from CliPlugin.RouteMapCli import (
      LinkBandwidthValueCliMatcher,
      mapNameMatcher,
      prefixListNameMatcher,
)
from CliPlugin.RoutingBgpCli import (
      addPathSendConfig,
      bgpNeighborConfig,
      BgpCmdBaseClass,
      BgpNEDCmdBaseClass,
      configForVrf,
      delNeighborConfigIfDefault,
      haveConflictingOutDelay,
      haveConflictingRouteMaps,
      haveConflictingWeight,
      NedModeKwMatcher,
      PeerCliExpression,
      peergroupNameMatcher,
      resetPresents,
      RouterBgpAfEvpnModelet,
      RouterBgpAfIpMulticastModelet,
      RouterBgpAfIpUniAndMcastSharedModelet,
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfIpv4Modelet,
      RouterBgpAfIpv6Modelet,
      RouterBgpAfIpv6MulticastModelet,
      RouterBgpAfLabelSharedModelet,
      RouterBgpAfLinkStateModelet,
      RouterBgpAfRtMembershipModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfSharedVrfModelet,
      RouterBgpAfSrTeSharedModelet,
      RouterBgpAfDpsModelet,
      RouterBgpAfVpnModelet,
      RouterBgpBaseAfEvpnMode,
      RouterBgpSharedModelet,
      RouterBgpVrfAfIpMulticastModelet,
      RouterBgpVrfAfIpv6MulticastModelet,
      RouterBgpVrfAfModelet,
      RouterBgpVrfSharedModelete,
      V4OrPeerGroupCliExpression,
      V6OrPeerGroupCliExpression,
      V4V6PeerKeyCliExpression,
)
from Intf.IntfRange import IntfRangeMatcher
from Arnet import IntfId
import ConfigMount
import Tac
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from ReversibleSecretCli import (
   reversibleAuthPasswordExpression,
   tryDecodeToken,
)
from RouteMapLib import getLbwCommValue
from TypeFuture import TacLazyType
# As a workaround for BUG393228, we need to use separate import statements
# for modules coming from different packages.
# pylint: disable-msg=ungrouped-imports
from Toggles import (
   RcfLibToggleLib,
   BgpCommonToggleLib,
   RoutingLibToggleLib,
)

bgpConfig = None
bgpVrfConfigDir = None

ExtendedNextHopCapabilityEnum = TacLazyType(
   "Routing::Bgp::ExtendedNextHopCapability" )
OutDelayApplyEnum = TacLazyType( "Routing::Bgp::OutDelayApplyType" )
PeerAfStateEnum = TacLazyType( "Routing::Bgp::PeerAFState" )
LinkBandwidthGenerationStateEnum = TacLazyType(
      'Routing::Bgp::LinkBandwidthGenerationState'
)
MetricOutStateEnum = TacLazyType( "Routing::Bgp::MetricOutState" )
RRClientEnum = TacLazyType( "Routing::Bgp::ReflectorClient" )
SendCommunityLinkBandwidthStateEnum = TacLazyType(
   'Routing::Bgp::SendCommunityLinkBandwidthState' )
SoftReconfigInboundStateEnum = TacLazyType(
   "Routing::Bgp::SoftReconfigInboundState" )

# Common helper methods

def validatePeer( mode, key ):
   SUPPORTED_INTFS = [ 'Ethernet', 'Vlan', 'Vxlan', 'Loopback', 'Internal', 'Fabric',
                       'Management', 'Port-Channel', 'Test', 'Tunnel' ]

   assert key is not None
   if key.type == 'peerGroup':
      if key not in bgpConfig.neighborConfig:
         return
   if key.type == 'peerIpv6' and key.v6Addr.isLinkLocal:
      mode.addErrorAndStop( "Link-local address must use <addr>%<ifname> syntax" )
   if key.type == 'peerIpv6Ll' and \
      not any( tag in key.llIntf for tag in SUPPORTED_INTFS ):
      mode.addErrorAndStop( "Link-local interface, '%s', must be one of: %s"
                            % ( key.llIntf,
                                ', '.join( SUPPORTED_INTFS ) ) )
   addr = None
   if key.type == 'peerIpv4':
      addr = Tac.Value( 'Arnet::IpAddr', stringValue=key.v4Addr )
   elif key.type == 'peerIpv6':
      addr = key.v6Addr
   if addr is not None and addr.isZero:
      mode.addErrorAndStop( "%s is an invalid peer address" % str( addr ) )

def getConflictingAttrsForAf( attrs, addrFamily ):
   if addrFamily == 'all':
      conflictingAttrs = [ ( af, c ) for af, c in attrs.iteritems() if af != 'all' ]
   else:
      conflictingAttrs = [ ( 'all', attrs[ 'all' ] ) ]
   return conflictingAttrs

def attrIsPresent( config, attr ):
   if getattr( config, attr + 'Present' ):
      return True
   return False

def configModeCmdForAf( addrFamily ):
   if addrFamily == 'all':
      modeCmd = 'router bgp'
   else:
      modeCmd = 'address-family %s' % addrFamily
   return modeCmd

def isDynamicPeerGroupByName( pgName ):
   if pgName in bgpConfig.listenRange:
      return True
   for v in bgpVrfConfigDir.vrfConfig:
      config = bgpVrfConfigDir.vrfConfig[ v ]
      if pgName in config.listenRange:
         return True
   return False

def isNeighIntfConfigPeerGroup( pgName ):
   """
   Checks whether the given peer-group is used as part of a
   'neighbor interface' config command.
   """
   for intf in bgpConfig.neighIntfConfig:
      peerConfig = bgpConfig.neighIntfConfig[ intf ]
      if peerConfig.peerGroup.stringValue == pgName:
         return True

   for v in bgpVrfConfigDir.vrfConfig:
      config = bgpVrfConfigDir.vrfConfig[ v ]
      for intf in config.neighIntfConfig:
         peerConfig = config.neighIntfConfig[ intf ]
         if peerConfig.peerGroup.stringValue == pgName:
            return True
   return False

def isPrivateAsn( asn ):
   """
   Checks whether the AS number is private per RFC 6996.
   There are two ranges for this:
      64512 - 65534 inclusive
      4200000000 - 4294967294 inclusive
   """
   return ( ( 64512 <= asn <= 65534 ) or
            ( 4200000000 <= asn <= 4294967294 ) )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-accepted-routes NUM_ACCEPTED_ROUTES
#    [warning-limit THRESHOLD]"
#-----------------------------------------------------------------------------
class SetNeighborMaxAcceptedRoutesCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-accepted-routes NUM_ACCEPTED_ROUTES '
              '[ warning-limit THRESHOLD ]' )
   noOrDefaultSyntax = 'neighbor PEER maximum-accepted-routes ...'
   data = { 'neighbor': bgpTokens.neighbor,
            'PEER': PeerCliExpression,
            'maximum-accepted-routes': bgpTokens.maxAcceptedRoutes,
            'NUM_ACCEPTED_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
            'warning-limit': bgpTokens.warningLimit,
            'THRESHOLD': bgpTokens.thresholdRangeMatcher
          }

   @staticmethod
   def _setNeighborMaxAcceptedRoutes( mode, peer, numAcceptedRoutes, threshold ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
         mode.addrFamily )
      if not threshold:
         threshold = getattr( config, attr + 'Default' ).maxAcceptedRoutesThreshold
      # As MIO commits are batched and asycnhronous, the order of these assignments
      # is critical. That is, we want threshold to be set before or with
      # numAcceptedRoutes (not after), and so the configuration must be mutated in
      # this order.
      maxAcceptedRoutesConfig = Tac.Value( 'Routing::Bgp::MaxAcceptedRoutesConfig',
                                           numAcceptedRoutes, threshold )
      setattr( config, attr, maxAcceptedRoutesConfig )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborMaxAcceptedRoutes( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         attr = BgpLib.peerConfigAttrsAfMap[ 'maxAcceptedRoutesConfig' ].get(
            mode.addrFamily )
         setattr( config, attr + 'Present', noOrDefault == NoOrDefault.NO and
                                            config.isPeerGroupPeer )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborMaxAcceptedRoutesCmd._noNeighborMaxAcceptedRoutes(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborMaxAcceptedRoutesCmd._setNeighborMaxAcceptedRoutes(
         mode,
         args[ 'PEER' ],
         args[ 'NUM_ACCEPTED_ROUTES' ],
         args.get( 'THRESHOLD' ) )


RouterBgpSharedModelet.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )
for modelet in [ RouterBgpAfIpUniSharedModelet, RouterBgpAfEvpnModelet ]:
   modelet.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-advertised-routes NUM_ADV_ROUTES
#    [warning-limit THRESHOLD [percent] ]"
#-----------------------------------------------------------------------------
class SetNeighborMaxAdvRoutesCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-advertised-routes NUM_ADV_ROUTES '
              '[ warning-limit ( ( THRESHOLD percent ) | ABS_THRESHOLD ) ]' )
   noOrDefaultSyntax = 'neighbor PEER maximum-advertised-routes ...'
   data = { 'neighbor': bgpTokens.neighbor,
            'PEER': PeerCliExpression,
            'maximum-advertised-routes': bgpTokens.maxAdvRoutes,
            'NUM_ADV_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
            'warning-limit': bgpTokens.warningLimit,
            'THRESHOLD': bgpTokens.thresholdPercentageMatcher,
            'percent': bgpTokens.percent,
            'ABS_THRESHOLD': bgpTokens.thresholdRangeMatcher
          }

   @staticmethod
   def _setNeighborMaxAdvRoutes( mode, peer, numAdvRoutes, threshold, percent ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = 'maxAdvRoutesConfig'
      if not threshold:
         threshold = getattr( config, attr + 'Default' ).maxAdvRoutesThreshold
      percentagePresent = percent is not None
      if percent:
         percentage = threshold
         threshold = int( ( numAdvRoutes * threshold ) / 100 )
      else:
         percentage = getattr(
            config, attr + 'Default' ).maxAdvRoutesThresholdPercentage
      maxAdvRoutesConfig = Tac.Value( 'Routing::Bgp::MaxAdvRoutesConfig',
                                      numAdvRoutes, threshold,
                                      percentage, percentagePresent )
      if mode.addrFamily == 'all':
         setattr( config, attr, maxAdvRoutesConfig )
         setattr( config, attr + 'Present', True )
      else:
         afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                              *BgpLib.addressFamilies.get( mode.addrFamily ) )
         config.maxAdvRoutesAfiSafiConfig[ afiSafi ] = maxAdvRoutesConfig
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborMaxAdvRoutes( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         if mode.addrFamily == 'all':
            attr = 'maxAdvRoutesConfig'
            setattr( config, attr + 'Present', False )
            setattr( config, attr, getattr( config, attr + 'Default' ) )
         else:
            afiSafi = Tac.Value( 'Routing::Bgp::AfiSafi',
                                 *BgpLib.addressFamilies.get( mode.addrFamily ) )
            del config.maxAdvRoutesAfiSafiConfig[ afiSafi ]
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborMaxAdvRoutesCmd._noNeighborMaxAdvRoutes(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ABS_THRESHOLD' in args:
         args[ 'THRESHOLD' ] = args.pop( 'ABS_THRESHOLD' )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborMaxAdvRoutesCmd._setNeighborMaxAdvRoutes(
         mode,
         args[ 'PEER' ],
         args[ 'NUM_ADV_ROUTES' ],
         args.get( 'THRESHOLD' ),
         args.get( 'percent' ) )

RouterBgpSharedModelet.addCommandClass( SetNeighborMaxAdvRoutesCmd )
for modelet in [ RouterBgpAfSharedModelet, RouterBgpAfLabelSharedModelet,
                 RouterBgpAfSrTeSharedModelet ]:
   modelet.addCommandClass( SetNeighborMaxAdvRoutesCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-as AS_NUM no-prepend \
#  replace-as [fallback]"
# command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborLocalAsCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER local-as '
              '( AS_NUM no-prepend replace-as [ fallback ] ) | disabled' )
   noOrDefaultSyntax = 'neighbor PEER local-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'local-as': bgpTokens.localAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
         'no-prepend': bgpTokens.noPrepend,
         'replace-as': bgpTokens.replaceLocalAs,
         'fallback': bgpTokens.fallbackAs,
   } )

   @staticmethod
   def _setLocalAs( mode, peer, asNumber, fallbackAs=False ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.localAs = Tac.Value( 'Routing::Bgp::LocalAs', asNumber,
                                  fallbackAs )
      config.localAsPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noLocalAs( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.localAsPresent = ( noOrDefault == NoOrDefault.NO and
                                   config.isPeerGroupPeer )
         config.localAs = config.localAsDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborLocalAsCmd._noLocalAs(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborLocalAsCmd._setLocalAs( mode,
                                         args[ 'PEER' ],
                                         args[ 'AS_NUM' ],
                                         'fallback' in args )

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalAsCmd )

#----------------------------------------------------------------------------
# "[no|default] neighbor <addr> activate" command
#----------------------------------------------------------------------------
deactivateNedMatcher = NedModeKwMatcher(
      bgpTokens.deactivate.keyword_,
      bgpTokens.deactivate.helpdesc_
)

def neighborAfActivateHelper( mode, peer, afStateEnum ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'af' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, afStateEnum )
      setattr( config, attr + 'Present', afStateEnum != PeerAfStateEnum.afDefault )
   else:
      mode.addError( "Invalid address family %s" % peer )

class RouterBgpAfActivateCommand( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER ( activate | deactivate )'
   noOrDefaultSyntax = 'neighbor PEER activate ...'

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'activate': bgpTokens.activate,
      'deactivate': deactivateNedMatcher
   }

   @staticmethod
   def _activateNeighbor( mode, peer ):
      validatePeer( mode, peer )
      neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afActive )

   @staticmethod
   def _noActivateNeighbor( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      if noOrDefault != NoOrDefault.DEFAULT:
         neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afInactive )
      else:
         neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afDefault )

   @staticmethod
   def handler( mode, args ):
      peer = args.get( 'PEER' )
      if 'deactivate' in args:
         RouterBgpAfActivateCommand._noActivateNeighbor(
               mode,
               peer,
               NoOrDefault.NO
         )
      else:
         RouterBgpAfActivateCommand._activateNeighbor( mode, peer )

   @staticmethod
   def noHandler( mode, args ):
      RouterBgpAfActivateCommand._noActivateNeighbor(
         mode,
         args[ 'PEER' ],
         ( NoOrDefault.DEFAULT if bgpConfig.noEqualsDefaultMode else NoOrDefault.NO )
      )

   @staticmethod
   def defaultHandler( mode, args ):
      RouterBgpAfActivateCommand._noActivateNeighbor(
         mode,
         args[ 'PEER' ],
         NoOrDefault.DEFAULT,
      )

modes = [
      RouterBgpAfIpv4Modelet,
      RouterBgpAfIpMulticastModelet,
      RouterBgpAfLabelSharedModelet,
      RouterBgpAfEvpnModelet,
      RouterBgpAfSrTeSharedModelet,
      RouterBgpVrfAfModelet,
      RouterBgpVrfAfIpMulticastModelet,
      RouterBgpAfLinkStateModelet,
      RouterBgpAfDpsModelet,
      RouterBgpAfIpv6MulticastModelet,
      RouterBgpVrfAfIpv6MulticastModelet,
      RouterBgpAfRtMembershipModelet,
]
for m in modes:
   m.addCommandClass( RouterBgpAfActivateCommand )

#-------------------------------------------------------------------------------
# 'neighbor ( ( PEER activate 6pe [ ipv6-unicast receive ] ) )'
# in router bgp, address-family ipv6
#-------------------------------------------------------------------------------
class RouterBgpNeighborActivate6peCommand( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor ( ( PEER ( activate | deactivate ) ) | '
              '( 6PE_PEER ( ( activate 6pe' )
   noOrDefaultSyntax = 'neighbor PEER activate ...'

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      '6PE_PEER': V4OrPeerGroupCliExpression,
      'activate': bgpTokens.activate,
      'deactivate': deactivateNedMatcher,
      '6pe': bgpTokens.sixPe,
   }

   syntax += ' [ ipv6-unicast receive ]'

   data.update( {
      'ipv6-unicast': 'Activate IPv6 Unicast for the neighbor',
      'receive': 'Activate IPv6 Unicast receive side only',
   } )

   syntax += ' ) | deactivate ) ) )'

   @staticmethod
   def _enableV6Unicast( mode, peer ):
      neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afActive )

   @staticmethod
   def _disableV6Unicast( mode, peer, noOrDefault ):
      if noOrDefault != NoOrDefault.DEFAULT:
         neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afInactive )
      else:
         neighborAfActivateHelper( mode, peer, PeerAfStateEnum.afDefault )

   @staticmethod
   def _enableSixPe( mode, peer ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.sixPe = True
      config.sixPePresent = True

   @staticmethod
   def _disableSixPe( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.sixPePresent = ( noOrDefault != NoOrDefault.DEFAULT )
      config.sixPe = config.sixPeDefault

   @staticmethod
   def _activateNeighbor( mode, peer, sixPe=False, ipv6Uni=False ):
      validatePeer( mode, peer )

      if ipv6Uni:
         # enables 6pe and v6 unicast
         RouterBgpNeighborActivate6peCommand._enableSixPe( mode, peer )
         RouterBgpNeighborActivate6peCommand._enableV6Unicast( mode, peer )
      elif sixPe:
         # enables only 6pe and disables v6 unicast
         RouterBgpNeighborActivate6peCommand._enableSixPe( mode, peer )
         RouterBgpNeighborActivate6peCommand._disableV6Unicast(
               mode,
               peer,
               NoOrDefault.NO
         )
      else:
         # enables only v6 unicast and disables 6pe
         RouterBgpNeighborActivate6peCommand._disableSixPe( mode, peer,
               NoOrDefault.NO )
         RouterBgpNeighborActivate6peCommand._enableV6Unicast( mode, peer )

   @staticmethod
   def _noActivateNeighbor( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      RouterBgpNeighborActivate6peCommand._disableSixPe( mode, peer, noOrDefault )
      RouterBgpNeighborActivate6peCommand._disableV6Unicast(
            mode,
            peer,
            noOrDefault
      )

   @staticmethod
   def handler( mode, args ):
      peer = args.get( 'PEER' )
      if 'deactivate' in args:
         RouterBgpNeighborActivate6peCommand._noActivateNeighbor(
            mode,
            peer,
            NoOrDefault.NO,
         )
      else:
         RouterBgpNeighborActivate6peCommand._activateNeighbor(
               mode,
               peer,
               sixPe=( '6pe' in args ),
               ipv6Uni=( 'ipv6-unicast' in args )
         )

   @staticmethod
   def noHandler( mode, args ):
      noOrDefault = (
            NoOrDefault.DEFAULT if bgpConfig.noEqualsDefaultMode else NoOrDefault.NO
      )
      RouterBgpNeighborActivate6peCommand._noActivateNeighbor(
         mode,
         args[ 'PEER' ],
         noOrDefault,
      )

   @staticmethod
   def defaultHandler( mode, args ):
      RouterBgpNeighborActivate6peCommand._noActivateNeighbor(
         mode,
         args[ 'PEER' ],
         NoOrDefault.DEFAULT,
      )

RouterBgpAfIpv6Modelet.addCommandClass( RouterBgpNeighborActivate6peCommand )

#-------------------------------------------------------------------------------
# '[no|default] neighbor PEER activate ( layer-2 | layer-3 )
#     under 'address-family evpn' mode'
#-------------------------------------------------------------------------------
class RouterBgpNeighborActivateEvpnCommand( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER activate ( layer-2 | layer-3 )'
   noOrDefaultSyntax = syntax

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'activate': bgpTokens.activate,
      'layer-2': bgpTokens.layer2,
      'layer-3': bgpTokens.layer3,
   }

   hidden = True

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'routeType' ] = args.get( 'layer-2', args.get( 'layer-3' ) )

   @staticmethod
   def _setActivateRouteType( mode, peer, routeType ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if routeType == 'layer-2':
         config.afEvpnActivateMacIpRouteType = True
         config.afEvpnActivateMacIpRouteTypePresent = True
      elif routeType == 'layer-3':
         config.afEvpnActivateIpPrefixRouteType = True
         config.afEvpnActivateIpPrefixRouteTypePresent = True
      else:
         mode.addError( "Invalid route type %s" % routeType )

   @staticmethod
   def _noActivateRouteType( mode, peer, noOrDefault, routeType ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if config:
         if routeType == 'layer-2':
            if noOrDefault == NoOrDefault.DEFAULT:
               config.afEvpnActivateMacIpRouteType = (
                     config.afEvpnActivateMacIpRouteTypeDefault
               )
               config.afEvpnActivateMacIpRouteTypePresent = False
            else:
               config.afEvpnActivateMacIpRouteType = False
               config.afEvpnActivateMacIpRouteTypePresent = True
         elif routeType == 'layer-3':
            if noOrDefault == NoOrDefault.DEFAULT:
               config.afEvpnActivateIpPrefixRouteType = (
                     config.afEvpnActivateIpPrefixRouteTypeDefault
               )
               config.afEvpnActivateIpPrefixRouteTypePresent = False
            else:
               config.afEvpnActivateIpPrefixRouteType = False
               config.afEvpnActivateIpPrefixRouteTypePresent = True
         else:
            mode.addError( "Invalid route type %s" % routeType )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def handler( mode, args ):
      RouterBgpNeighborActivateEvpnCommand._setActivateRouteType(
            mode,
            args.get( 'PEER' ),
            args.get( 'routeType' )
      )

   @staticmethod
   def defaultHandler( mode, args ):
      RouterBgpNeighborActivateEvpnCommand._noActivateRouteType(
            mode,
            args.get( 'PEER' ),
            NoOrDefault.DEFAULT,
            args.get( 'routeType' )
      )

   @staticmethod
   def noHandler( mode, args ):
      RouterBgpNeighborActivateEvpnCommand._noActivateRouteType(
            mode,
            args.get( 'PEER' ),
            NoOrDefault.NO,
            args.get( 'routeType' )
      )

RouterBgpBaseAfEvpnMode.addCommandClass( RouterBgpNeighborActivateEvpnCommand )

#----------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> timers <keepalive> <hold>" command,
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetNeighborPeerTimersCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER timers KEEPALIVE ( HOLD_TIME | HOLD_FOREVER )'
   noOrDefaultSyntax = 'neighbor PEER timers ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'timers': bgpTokens.timers,
      'bgp': bgpTokens.bgpTimers,
      'KEEPALIVE': bgpTokens.keepaliveTimeRangeMatcher,
      'HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      'HOLD_FOREVER': bgpTokens.holdForever,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      keepalive = args[ 'KEEPALIVE' ]
      hold = args.get( 'HOLD_TIME', 0 )
      validatePeer( mode, peer )
      if hold < keepalive:
         mode.addError( 'keepalive time cannot be greater than hold time' )
         return
      if hold != 0 and keepalive == 0:
         mode.addError(
            'keepalive time cannot be zero when hold time is not zero' )
         return
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.keepaliveTime = keepalive
      config.holdTime = hold
      config.timersPresent = True
      delNeighborConfigIfDefault ( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.timersPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                  config.isPeerGroupPeer )
         config.keepaliveTime = config.keepaliveTimeDefault
         config.holdTime = config.holdTimeDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborPeerTimersCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR idle-restart-timer IDLE_RESTART_TIMER" command,
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetIdleRestartTimerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER idle-restart-timer ( IDLE_RESTART_TIMER | disabled )'
   noOrDefaultSyntax = 'neighbor PEER idle-restart-timer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'idle-restart-timer': bgpTokens.idleRestartTimer,
      'IDLE_RESTART_TIMER': bgpTokens.idleRestartTimerRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      irtime = args[ 'IDLE_RESTART_TIMER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.idleHoldTime = irtime
      config.idleHoldTimePresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.idleHoldTimePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
         config.idleHoldTime = config.idleHoldTimeDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetIdleRestartTimerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR remote-as AS_NUM" command,
# in "config-bgp" mode
#-------------------------------------------------------------------------------
class SetNeighborAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER remote-as ( AS_NUM | disabled )'
   noOrDefaultSyntax = 'neighbor PEER remote-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'remote-as': bgpTokens.remoteAs,
      'AS_NUM': bgpTokens.AsNumCliExpr,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      asNumber = args[ 'AS_NUM' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.asNumber = asNumber
      config.asNumberPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.asNumberPresent = ( noOrDefault == NoOrDefault.NO and
                                    config.isPeerGroupPeer )
         config.asNumber = config.asNumberDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborAsCmd )

#---------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-map <name> [ in|out ]" command, in
# "config-bgp" mode
#--------------------------------------------------------------------------------
class SetNeighborRouteMapBaseCmd( BgpNEDCmdBaseClass ):

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'in' in args or 'NED_IN' in args:
         args[ 'inout' ] = 'in'
      elif 'out' in args or 'NED_OUT' in args:
         args[ 'inout' ] = 'out'

   @staticmethod
   def _setNeighborRouteMap( mode, peer, mapName, inOut ):
      if ( mode.addrFamily == 'ipv6 labeled-unicast' and
           getEffectiveProtocolModel( mode ) == ProtoAgentModel.ribd ):
         mode.addError( "IPv6 neighbor route-maps is only supported in "
                         "multi-agent labeled-unicast mode" )
      else:
         attr = 'routeMapIn' if inOut == 'in' else 'routeMapOut'
         config = bgpNeighborConfig( peer, mode.vrfName )
         if haveConflictingRouteMaps( mode, config, attr, mapName, mode.addrFamily ):
            return
         af = mode.addrFamily
         attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
         setattr( config, attr, str( mapName ) )
         setattr( config, attr + 'Present', True )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborRouteMap( mode, peer, inOut, noOrDefault ):
      attr = 'routeMapIn' if inOut == 'in' else 'routeMapOut'
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         af = mode.addrFamily
         attr = BgpLib.peerConfigAttrsAfMap[ attr ].get( af )
         if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
            setattr( config, attr + 'Present', True )
         else:
            setattr( config, attr + 'Present', False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborRouteMapBaseCmd._setNeighborRouteMap(
         mode,
         args[ 'PEER' ],
         args[ 'MAP_NAME' ],
         args[ 'inout' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborRouteMapBaseCmd._noNeighborRouteMap(
         mode,
         args[ 'PEER' ],
         args[ 'inout' ],
         noOrDefault )

class SetNeighborRouteMapCmd( SetNeighborRouteMapBaseCmd ):
   syntax = '''neighbor PEER route-map
            ( ( MAP_NAME ( in | out ) )
            | ( ( NED_IN | NED_OUT ) disabled ) )'''
   noOrDefaultSyntax = 'neighbor PEER route-map [ MAP_NAME ] ( in | out )'
   data = SetNeighborRouteMapBaseCmd._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-map': bgpTokens.routeMap,
      'MAP_NAME': mapNameMatcher,
      'in': bgpTokens.rmIn,
      'out': bgpTokens.rmOut,
      'NED_IN': bgpTokens.rmIn,
      'NED_OUT': bgpTokens.rmOut,
   } )

# SR-TE afiSafi does not support outbound route-map, so create separate command
class SetNeighborRouteMapSrTeCmd( SetNeighborRouteMapBaseCmd ):
   syntax = 'neighbor PEER route-map ( MAP_NAME in ) | ( NED_IN disabled )'
   noOrDefaultSyntax = 'neighbor PEER route-map [ MAP_NAME ] in'
   data = SetNeighborRouteMapBaseCmd._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-map': bgpTokens.routeMap,
      'MAP_NAME': mapNameMatcher,
      'in': bgpTokens.rmIn,
      'NED_IN': bgpTokens.rmIn,
   } )

RouterBgpAfSrTeSharedModelet.addCommandClass( SetNeighborRouteMapSrTeCmd )
for m in [ RouterBgpSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNeighborRouteMapCmd )

#---------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR prefix-list <name> [ in|out ]" command,
# in "config-bgp" mode and address-family block
#--------------------------------------------------------------------------------
prefixListNedIn = NedModeKwMatcher(
      bgpTokens.prefixListIn.keyword_,
      bgpTokens.prefixListIn.helpdesc_,
)
prefixListNedOut = NedModeKwMatcher(
      bgpTokens.prefixListOut.keyword_,
      bgpTokens.prefixListOut.helpdesc_,
)

class SetNeighborPrefixListCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER prefix-list ( PREFIX_LIST_NAME ( in | out ) ) | '
              '( ( NED_IN | NED_OUT ) disabled )' )
   noOrDefaultSyntax = 'neighbor PEER prefix-list [ PREFIX_LIST_NAME ] ( in | out )'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'prefix-list': bgpTokens.prefixList,
         'PREFIX_LIST_NAME': prefixListNameMatcher,
         'in': bgpTokens.prefixListIn,
         'out': bgpTokens.prefixListOut,
         'NED_IN': prefixListNedIn,
         'NED_OUT': prefixListNedOut,
   } )

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'in' in args or 'NED_IN' in args:
         args[ 'inout' ] = 'prefixListIn'
      elif 'out' in args or 'NED_OUT' in args:
         args[ 'inout' ] = 'prefixListOut'

   @staticmethod
   def _setPrefixList( mode, peer, plName, attrName ):
      addrFamily = 'ipv4' if mode.addrFamily == 'all' else mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
      setattr( config, attr, str( plName ) )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noPrefixList( mode, peer, attrName, noOrDefault ):
      addrFamily = 'ipv4' if mode.addrFamily == 'all' else mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         attr = BgpLib.peerConfigAttrsAfMap[ attrName ].get( addrFamily )
         attrVal = noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         setattr( config, attr + 'Present', attrVal )
         setattr( config, attr, getattr( config,
                                         attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborPrefixListCmd._setPrefixList(
            mode,
            args.get( 'PEER' ),
            args.get( 'PREFIX_LIST_NAME' ),
            args.get( 'inout' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborPrefixListCmd._noPrefixList(
            mode,
            args.get( 'PEER' ),
            args.get( 'inout' ),
            noOrDefault,
      )

for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniAndMcastSharedModelet ]:
   m.addCommandClass( SetNeighborPrefixListCmd )


#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR import-localpref LOCAL_PREF" command,
# in "config-bgp" mode
#-----------------------------------------------------------------------------------
class SetNeighborImportLocalPrefCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER import-localpref LOCAL_PREF'
   noOrDefaultSyntax = 'neighbor PEER import-localpref ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'import-localpref': bgpTokens.importLocalPref,
      'LOCAL_PREF': bgpTokens.localPrefMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      localpref = args[ 'LOCAL_PREF' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.importLocalPref = localpref
      config.importLocalPrefPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.importLocalPrefPresent = ( noOrDefault == NoOrDefault.NO and
                                           config.isPeerGroupPeer )
         config.importLocalPref = config.importLocalPrefDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborImportLocalPrefCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR export-localpref LOCAL_PREF" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborExportLocalPrefCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER export-localpref LOCAL_PREF'
   noOrDefaultSyntax = 'neighbor PEER export-localpref ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'export-localpref': bgpTokens.exportLocalPref,
      'LOCAL_PREF': bgpTokens.localPrefMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      localpref = args[ 'LOCAL_PREF' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.exportLocalPref = localpref
      config.exportLocalPrefPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.exportLocalPrefPresent = ( noOrDefault == NoOrDefault.NO and
                                           config.isPeerGroupPeer )
         config.exportLocalPref = config.exportLocalPrefDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborExportLocalPrefCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR graceful-restart" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborGracefulRestartCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER graceful-restart [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER graceful-restart'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'graceful-restart': bgpTokens.gracefulRestart,
   } )

   @staticmethod
   def _configureNeighborGracefulRestart( mode, config, enable, noOrDefault,
                                          addrFamily ):
      # Verify that neighbor graceful restart is not configured in both
      # router-bgp-mode and router-bgp-af mode
      allAfConfigured, anyAfConfigured = (
            BgpLib.isGracefulRestartGlobalOrAfConfigured( config ) )
      if BgpLib.haveConflictingGracefulRestart(
               noOrDefault,
               None if addrFamily == "all" else addrFamily,
               anyAfConfigured,
               allAfConfigured ):
         mode.addError( "Neighbor Graceful restart cannot be configured in"
                                 " both router-bgp and router-bgp-af mode" )
         return

      if addrFamily == 'ipv4 multicast':
         mode.addError( "Graceful restart is currently not supported "
                        "for IPv4 multicast mode" )
         return
      if addrFamily == 'ipv6 multicast':
         mode.addError( "Graceful restart is currently not supported "
                        "for IPv6 multicast mode" )
         return

      gr = False
      grPresent = False
      if enable:
         gr = grPresent = True
      else:
         assert noOrDefault
         grPresent = False if noOrDefault == NoOrDefault.DEFAULT else True

      attr = BgpLib.bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
      setattr( config, attr, gr )
      setattr( config, attr + 'Present', grPresent )

   @staticmethod
   def _setNeighborGracefulRestart( mode, peer ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      SetNeighborGracefulRestartCmd._configureNeighborGracefulRestart(
            mode,
            config,
            True,
            None,
            mode.addrFamily )

   @staticmethod
   def _noNeighborGracefulRestart( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                  vrfName=mode.vrfName )
      if config:
         SetNeighborGracefulRestartCmd._configureNeighborGracefulRestart(
               mode,
               config,
               False,
               noOrDefault,
               mode.addrFamily )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborGracefulRestartCmd._setNeighborGracefulRestart(
            mode,
            args[ 'PEER' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborGracefulRestartCmd._noNeighborGracefulRestart(
            mode,
            args[ 'PEER' ],
            noOrDefault )

for m in [ RouterBgpSharedModelet, RouterBgpAfSharedModelet ]:
   m.addCommandClass( SetNeighborGracefulRestartCmd )
if BgpCommonToggleLib.toggleArBgpLuGracefulRestartEnabled():
   RouterBgpAfLabelSharedModelet.addCommandClass( SetNeighborGracefulRestartCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR graceful-restart-helper
#  [ restart-time < seconds > ]" command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborGrHelperCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER graceful-restart-helper'
   if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
      syntax += ' [ EXT_RESTART_TIME ]'
   if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
      syntax += ' [ long-lived ]'
   syntax += ' [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER graceful-restart-helper'
   if ( BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled() or
        BgpCommonToggleLib.toggleArBgpLlgrEnabled() ):
      noOrDefaultSyntax = 'neighbor PEER graceful-restart-helper ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'graceful-restart-helper': bgpTokens.grHelper,
   } )
   if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
      data.update( {
         'EXT_RESTART_TIME': bgpTokens.GrHelperRestartTimeExp } )
   if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
      data.update( {
         'long-lived': bgpTokens.llgrHelper } )

   @staticmethod
   def _setNeighborGrHelper( mode, peer, extRestartTime, llgrHelper ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.grHelper = True
      config.grHelperPresent = True
      if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
         if extRestartTime:
            config.grHelperRestartTime = extRestartTime
            config.grHelperRestartTimePresent = True
         else:
            config.grHelperRestartTime = config.grHelperRestartTimeDefault
            config.grHelperRestartTimePresent = False
      if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
         if llgrHelper:
            config.llgrHelper = True
            config.llgrHelperPresent = True
         else:
            config.llgrHelper = config.llgrHelperDefault
            config.llgrHelperPresent = False

   @staticmethod
   def _noNeighborGrHelper( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                  vrfName=mode.vrfName )
      if config:
         if noOrDefault == NoOrDefault.DEFAULT:
            # GR-Helper is enabled by default
            config.grHelper = True
            config.grHelperPresent = False
            if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
               config.grHelperRestartTimePresent = False
            if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
               config.llgrHelperPresent = False
         else:
            config.grHelper = False
            config.grHelperPresent = True
            if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
               config.grHelperRestartTimePresent = True
            if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
               config.llgrHelperPresent = True
         if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
            config.grHelperRestartTime = config.grHelperRestartTimeDefault
         if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
            config.llgrHelper = config.llgrHelperDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborGrHelperCmd._setNeighborGrHelper( mode, args[ 'PEER' ],
                                                   args.get( 'RESTART_TIME' ),
                                                   ( 'long-lived' in args ) )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborGrHelperCmd._noNeighborGrHelper( mode, args[ 'PEER' ], noOrDefault )

RouterBgpSharedModelet.addCommandClass( SetNeighborGrHelperCmd )


#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR metric-out <metric>"
#  command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborMetricOutCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER metric-out ( METRIC | disabled )'
   noOrDefaultSyntax = 'neighbor PEER metric-out ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'metric-out': bgpTokens.metricOut,
      'METRIC': bgpTokens.metricRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      metric = args[ 'METRIC' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.metricOut = metric
      config.metricOutState = MetricOutStateEnum.metricOutSet
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
            config.metricOutState = MetricOutStateEnum.metricOutNone
         else:
            config.metricOutState = MetricOutStateEnum.metricNotConfigured
         config.metricOut = config.metricOutDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborMetricOutCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor PEER password SECRET" command,
# in "config-bgp" mode
# We do not have a 'disabled' version for the password command because the
# 'disabled' token matches autoPasswdRule (specifically rule1)
#-----------------------------------------------------------------------------
class NeighborPasswordCommand( BgpNEDCmdBaseClass ):
   syntax = "neighbor PEER password SECRET"
   noOrDefaultSyntax = "neighbor PEER password ..."
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'password': 'Password to use in computation of MD5 hash',
      'SECRET': reversibleAuthPasswordExpression( 'SECRET' ),
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      authPasswdToken = args[ 'SECRET' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      decoded = tryDecodeToken( authPasswdToken,
                                key=peer.stringValue + '_passwd',
                                algorithm='MD5',
                                mode=mode )
      if not decoded:
         return

      config.password = decoded
      config.passwordPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.passwordPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
         config.password = config.passwordDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborPasswordCommand )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-routes NUM_ROUTES
#  [ warning-limit THRESHOLD percent ] [ warning-only ]" command
#-----------------------------------------------------------------------------
class SetNeighborMaxRoutesCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-routes NUM_ROUTES '
              '[ warning-limit ( ( THRESHOLD percent ) | ABS_THRESHOLD ) ] '
              '[ warning-only ]'
   )
   noOrDefaultSyntax = 'neighbor PEER maximum-routes ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'maximum-routes': bgpTokens.maxRoutes,
         'NUM_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
         'warning-limit': bgpTokens.threshold,
         'THRESHOLD': bgpTokens.thresholdPercentageMatcher,
         'percent': bgpTokens.percent,
         'ABS_THRESHOLD': bgpTokens.thresholdRangeMatcher,
         'warning-only': bgpTokens.warning,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ABS_THRESHOLD' in args:
         args[ 'THRESHOLD' ] = args.pop( 'ABS_THRESHOLD' )
   @staticmethod
   def _setNeighborMaxRoutes(
         mode,
         peer,
         numRoutes,
         threshold=None,
         percent=None,
         action=False,
   ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily

      config = bgpNeighborConfig( peer, vrfName=vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
      maxRoutesConfig = getattr( config, attr )
      if not threshold:
         threshold = getattr( config, attr + 'Default' ).maxRoutesThreshold
      percentagePresent = percent is not None
      if percent:
         percentage = threshold
         # The Tac model expects a integer, and Python division now returns a float.
         threshold = ( numRoutes * threshold ) // 100
      else:
         percentage = (
               getattr( config, attr + 'Default' ).maxRoutesThresholdPercentage
         )
      maxRestart = maxRoutesConfig.maxRoutesRestart
      maxRoutesConfig = Tac.Value( 'Routing::Bgp::MaxRoutesConfig',
         numRoutes, threshold, maxRestart, action, percentage, percentagePresent )
      setattr( config, attr, maxRoutesConfig )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def _noNeighborMaxRoutes(
         mode,
         peer,
         noOrDefault,
         numRoutes=0,
         threshold=None,
         action=False,
         restart=None,
   ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily

      config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
      if config:
         attr = BgpLib.peerConfigAttrsAfMap[ 'maxRoutesConfig' ].get( addrFamily )
         setattr( config, attr + 'Present',
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      validatePeer( mode, args[ 'PEER' ] )
      SetNeighborMaxRoutesCmd._setNeighborMaxRoutes(
            mode,
            args[ 'PEER' ],
            args[ 'NUM_ROUTES' ],
            threshold=args.get( 'THRESHOLD' ),
            percent=args.get( 'percent' ),
            action=( 'warning-only' in args ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      if noOrDefault == NoOrDefault.DEFAULT:
         SetNeighborMaxRoutesCmd._noNeighborMaxRoutes(
               mode,
               args[ 'PEER' ],
               NoOrDefault.DEFAULT,
         )
      else:
         SetNeighborMaxRoutesCmd._noNeighborMaxRoutes(
               mode,
               args[ 'PEER' ],
               NoOrDefault.NO,
         )

for m in [
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfEvpnModelet,
      RouterBgpSharedModelet
]:
   m.addCommandClass( SetNeighborMaxRoutesCmd )


#-------------------------------------------------------------------------------
# "neighbor ADDR rcf in FUNCTION" command,
# in "config-bgp" mode
#-------------------------------------------------------------------------------
directions = {
  'in': bgpTokens.rcfIn.helpdesc_,
}
if RcfLibToggleLib.toggleRcfLimaEnabled():
   directions[ 'out' ] = bgpTokens.rcfOut.helpdesc_

class SetNeighborRcfCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER rcf DIRECTION ' \
            '( ( FUNCTION [ disabled ] ) | disabled )'
   noOrDefaultSyntax = 'neighbor PEER rcf DIRECTION ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'rcf': bgpTokens.rcf,
         'DIRECTION': CliMatcher.EnumMatcher( directions ),
         'FUNCTION': RcfCliLib.rcfFunctionMatcher,
   } )

   @staticmethod
   def _setRcf( mode, peer, inout, function ):
      validatePeer( mode, peer )
      # construct rcf field to be modified from af and inbound vs outbound
      direction = inout.title()
      config = bgpNeighborConfig( peer, mode.vrfName )
      attr = 'rcfV%sUni' % ( '6' if mode.addrFamily == 'ipv6' else '4' ) + direction
      setattr( config, attr, function )
      setattr( config, attr + 'Present', True )

   @staticmethod
   def _noRcf( mode, peer, inout, noOrDefault ):
      validatePeer( mode, peer )
      # construct rcf field to be modified from af and inbound vs outbound
      config = bgpNeighborConfig( peer, mode.vrfName, create=False )
      direction = inout.title()
      attr = 'rcfV4Uni' + direction
      if mode.addrFamily == 'ipv6':
         attr = 'rcfV6Uni' + direction
      if config:
         setattr( config, attr, getattr( config,
                                         attr + 'Default' ) )
         setTrue = noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         setattr( config, attr + 'Present', setTrue )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      function = args[ 'FUNCTION' ]
      inout = args[ 'DIRECTION' ]
      SetNeighborRcfCmd._setRcf( mode, peer, inout,
                                 re.sub( r'\(\)', '', function ) )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      inout = args[ 'DIRECTION' ]
      SetNeighborRcfCmd._noRcf( mode, peer, inout, noOrDefault )

RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborRcfCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-to-peer" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetRouteToPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER route-to-peer [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER route-to-peer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-to-peer': bgpTokens.routeToPeer,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.routeToPeerPresent = config.isPeerGroupPeer
      config.routeToPeer = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer,
                                  create=( noOrDefault != NoOrDefault.DEFAULT ),
                                  vrfName=mode.vrfName )
      if config:
         if noOrDefault == NoOrDefault.DEFAULT:
            # default route-to-peer is true
            config.routeToPeer = config.routeToPeerDefault
            config.routeToPeerPresent = False
         else:
            config.routeToPeer = False
            config.routeToPeerPresent = True
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetRouteToPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR remove-private-as [all [replace-as] ]"
# command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborRemovePrivateAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER remove-private-as [ [ all [ replace-as ] ] | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER remove-private-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'remove-private-as': bgpTokens.removePrivateAs,
      'all': bgpTokens.removePrivateAsAll,
      'replace-as': bgpTokens.replacePrivateAs,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      always = 'all' in args
      replace = 'replace-as' in args
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      asNumber = configForVrf( mode.vrfName ).asNumber
      if mode.vrfName != DEFAULT_VRF and asNumber == 0:
         asNumber = configForVrf( DEFAULT_VRF ).asNumber
      if config.asNumber == asNumber:
         mode.addWarning( "This command is a no-op - "
                          "neighbor's AS number is same as "
                          "device's AS number" )
      elif isPrivateAsn( config.asNumber ):
         mode.addWarning( 'eBGP neighbor has a private AS number' )
      config.removePrivateAs = Tac.Value( 'Routing::Bgp::RemovePrivateAs', True,
                                          always, replace )
      config.removePrivateAsPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.removePrivateAsPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
         config.removePrivateAs = config.removePrivateAsDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborRemovePrivateAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR as-path prepend-own disabled" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborPrependOwnAsDisabledCmd( BgpNEDCmdBaseClass ):
   # The 'disabled' token for NED mode is different than the disabled token
   # that is part of this command.  Use an underscore to disambiguate.
   syntax = 'neighbor PEER as-path prepend-own _disabled [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER as-path prepend-own _disabled ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'as-path': bgpTokens.asPath,
      'prepend-own': bgpTokens.prependOwn,
      '_disabled': bgpTokens.prependOwnDisabled,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.prependOwnDisabled = True
      config.prependOwnDisabledPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.prependOwnDisabledPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                              config.isPeerGroupPeer )
         config.prependOwnDisabled = config.prependOwnDisabledDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborPrependOwnAsDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR as-path remote-as replace out"
# command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborReplaceRemoveAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER as-path remote-as replace out [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER as-path remote-as replace out ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'as-path': bgpTokens.asPath,
      'remote-as': bgpTokens.remoteAs,
      'replace': bgpTokens.replaceRemoteAs,
      'out': bgpTokens.out,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      asNumber = configForVrf( mode.vrfName ).asNumber
      if mode.vrfName != DEFAULT_VRF and asNumber == 0:
         asNumber = configForVrf( DEFAULT_VRF ).asNumber
      if config.asNumber == asNumber:
         mode.addWarning( "This command is a no-op - " +
                          "neighbor's AS number is same as " +
                          "device's AS number" )
      config.replaceRemoteAs = True
      config.replaceRemoteAsPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.replaceRemoteAsPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
         config.replaceRemoteAs = config.replaceRemoteAsDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborReplaceRemoveAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-self" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborNextHopSelfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-self [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop-self ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'next-hop-self': bgpTokens.nextHopSelf,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.nextHopSelf = True
      config.nextHopSelfPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.nextHopSelfPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
         config.nextHopSelf = config.nextHopSelfDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborNextHopSelfCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-peer" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborNextHopPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-peer [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop-peer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'next-hop-peer': bgpTokens.nextHopPeer,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.nextHopPeer = True
      config.nextHopPeerPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.nextHopPeerPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
         config.nextHopPeer = config.nextHopPeerDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborNextHopPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR description <desc>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborDescriptionCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER description DESC'
   noOrDefaultSyntax = 'neighbor PEER description ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'description': bgpTokens.description,
      'DESC': bgpTokens.descriptionMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      desc = args[ 'DESC' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.description = desc
      config.descriptionPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.descriptionPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
         config.description = config.descriptionDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborDescriptionCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR send-community
# [ standard ] [ extended ] [ large ]
# [ link-bandwidth [aggregate LINK_BW | divide ( equal | ratio ) ] ]"
# command, in "router-bgp" mode.
#
# This is broken into several CliCommandClasses to simplify the syntax
# and processing.  In particular, the LBW parameter is allowed but hidden
# in some cases.
#-------------------------------------------------------------------------------
class SendLinkBandwidthCommExpr( CliCommand.CliExpression ):
   expression = '''link-bandwidth
                   ( aggregate [ LINK_BW ] ) | ( divide DIV )'''
   data = { 'link-bandwidth': bgpTokens.sendLinkBandwidthCommunity,
            'aggregate': bgpTokens.sendLbwAggregate,
            'LINK_BW': LinkBandwidthValueCliMatcher(
               'Reference link speed in bits/second' ),
            'divide': bgpTokens.sendLbwDivide,
            'DIV': CliMatcher.EnumMatcher( {
               'equal': ( 'Send link-bandwidth equal divided '
                          'attribute to this neighbor' ),
               'ratio': ( 'Send link-bandwidth ratio divided '
                          'attribute to this neighbor' ),
            } )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'link-bandwidth' in args:
         if 'aggregate' in args:
            additiveOrDivide = {
               'additive': {
                  'lbwVal': args.get( 'LINK_BW' ),
               }
            }
         elif 'divide' in args:
            additiveOrDivide = {
               'divide': {
                  'equalOrRatio': args[ 'DIV' ]
               }
         }
         args[ 'LBW' ] = { 'additiveOrDivide': additiveOrDivide }

# The class below is used to hide the link-bandwidth keyword
# when link-bandwidth token is used with standard/large community
# Note that we cannot hide the expression directly as of now
# and so we need to create a duplicate class to have link-bandwidth
# hidden (BUG 480785)
class SendLinkBandwidthCommHiddenExpr( SendLinkBandwidthCommExpr ):
   data = SendLinkBandwidthCommExpr.data.copy()
   data[ 'link-bandwidth' ] = CliCommand.Node(
      matcher=bgpTokens.sendLinkBandwidthCommunity,
      hidden=True )

def setNeighborSendCommunity( mode, peer, standard=None, extended=None,
                              large=None, lbw=None, add=None, remove=None ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )

   # For each send-community command, there are 4 command variants for each
   # community-type (standard, extended, large, link-bandwidth):
   #   (1) The community type is present with add
   #   (2) The community type is present with remove
   #   (3) The community type is present without add or remove
   #   (4) The community type is not present without add or remove
   isReplace = ( not add and not remove ) # case (3) and (4)

   preExistingSendCommunity = ( config.sendStandardCommunity or
                                config.sendExtendedCommunity or
                                config.sendLargeCommunity or
                                config.sendCommunity )

   if ( not remove and standard ):
      config.sendStandardCommunity = True
   elif ( ( remove and standard ) or
          ( isReplace and not standard ) ):
      config.sendStandardCommunity = config.sendStandardCommunityDefault

   if ( not remove and extended ):
      config.sendExtendedCommunity = True
   elif ( ( remove and extended ) or
          ( isReplace and not extended ) ):
      config.sendExtendedCommunity = config.sendExtendedCommunityDefault

   if ( not remove and large ):
      config.sendLargeCommunity = True
   elif ( ( remove and large ) or
          ( isReplace and not large ) ):
      config.sendLargeCommunity = config.sendLargeCommunityDefault

   # If neither send standard/extended/large community are
   # explicitly enabled, fallback to the default case and set the send
   # community attribute to True to enable send community for all (except
   # in the case that this is a remove command and there were no
   # pre-existing comm-type configured - in which case, ignore the cmd).
   # Otherwise, if one or two are explicitly enabled, then
   # set the send community attribute to False.
   if ( not config.sendStandardCommunity and not config.sendExtendedCommunity
        and not config.sendLargeCommunity ):
      config.sendCommunity = preExistingSendCommunity if remove else True
   else:
      config.sendCommunity = config.sendCommunityDefault

   # check if all or any individual send-communities values configured
   if ( config.sendCommunity or config.sendStandardCommunity or
        config.sendExtendedCommunity or config.sendLargeCommunity ):
      config.sendCommunityPresent = True

   if lbw:
      additive = lbw[ 'additiveOrDivide' ].get( 'additive', {} )
      lbwVal = additive.get( 'lbwVal' )
      divide = lbw[ 'additiveOrDivide' ].get( 'divide', {} )
      equal = ( divide.get( 'equalOrRatio' ) == 'equal' )
      ratio = ( divide.get( 'equalOrRatio' ) == 'ratio' )
      if not remove:
         # Add or replace existing send lbw community configuration.
         if additive:
            if not lbwVal:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthAdditive
            else:
               config.sendCommunityLinkBw = \
                  SendCommunityLinkBandwidthStateEnum.\
                  sendCommunityLinkBandwidthAdditiveRef
               config.sendCommunityLinkBwAdditiveRef = getLbwCommValue( lbwVal )
               config.sendCommunityLinkBwAdditiveRefDisplay = lbwVal
         elif divide:
            if equal:
               config.sendCommunityLinkBw = \
                     SendCommunityLinkBandwidthStateEnum.\
                     sendCommunityLinkBandwidthDivideEqual
            elif ratio:
               config.sendCommunityLinkBw = \
                     SendCommunityLinkBandwidthStateEnum.\
                     sendCommunityLinkBandwidthDivideRatio
         config.sendCommunityLinkBwPresent = True
      else:
         # Only remove send lbw community configuration if we
         # have an exact match w/ existing configuration.
         if ( additive and not lbwVal and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditive or
                config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditiveRef ) ) or \
            ( additive and lbwVal and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthAdditiveRef ) and
              ( config.sendCommunityLinkBwAdditiveRefDisplay == lbwVal ) ) or \
            ( divide and equal and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthDivideEqual ) ) or \
            ( divide and ratio and
              ( config.sendCommunityLinkBw ==
                SendCommunityLinkBandwidthStateEnum.
                sendCommunityLinkBandwidthDivideRatio ) ):
            config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
            config.sendCommunityLinkBwPresent = False
   elif isReplace:
      # Remove existing send lbw community configuration.
      config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
      config.sendCommunityLinkBwPresent = False

   # If the send lbw community configuration is no longer 'additiveRef',
   # then reset the corresponding 'additiveRef' attributes to default.
   if config.sendCommunityLinkBw != \
      SendCommunityLinkBandwidthStateEnum.\
      sendCommunityLinkBandwidthAdditiveRef:
      config.sendCommunityLinkBwAdditiveRef = \
          config.sendCommunityLinkBwAdditiveRefDefault
      config.sendCommunityLinkBwAdditiveRefDisplay = \
          config.sendCommunityLinkBwAdditiveRefDisplayDefault

class SetNeighborSendCommunityCmd( BgpNEDCmdBaseClass ):
   '''The base command, includes the no/default form.'''
   syntax = 'neighbor PEER send-community [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER send-community ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      setNeighborSendCommunity( mode, args[ 'PEER' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         isPresent = ( noOrDefault != noOrDefault.DEFAULT and
                       config.isPeerGroupPeer )
         config.sendCommunityPresent = isPresent
         config.sendCommunityLinkBwPresent = isPresent
         config.sendCommunity = config.sendCommunityDefault
         config.sendStandardCommunity = config.sendStandardCommunityDefault
         config.sendExtendedCommunity = config.sendExtendedCommunityDefault
         config.sendLargeCommunity = config.sendLargeCommunityDefault
         config.sendCommunityLinkBw = config.sendCommunityLinkBwDefault
         config.sendCommunityLinkBwAdditiveRef = \
            config.sendCommunityLinkBwAdditiveRefDefault
         config.sendCommunityLinkBwAdditiveRefDisplay = \
            config.sendCommunityLinkBwAdditiveRefDisplayDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityCmd )

class SetNeighborSendCommunityStandardLargeCmd( CliCommand.CliCommandClass ):
   '''The command which accepts combinations of standard and large communities.'''
   syntax = ( 'neighbor PEER send-community '
              '( ( standard [ large ] ) | large ) [ LBW ]' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'standard': bgpTokens.sendStandardCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'LBW': SendLinkBandwidthCommHiddenExpr
   }

   @staticmethod
   def handler( mode, args ):
      if 'LBW' in args:
         commType = 'standard' if 'standard' in args else 'large'
         mode.addWarning( 'Link bandwidth is not a %s community. '
                          'Please use "neighbor PEER send-community link-bandwidth" '
                          'or "neighbor PEER send-community [ standard ] extended '
                          '[ large ] link-bandwidth" instead' % commType )
      setNeighborSendCommunity( mode, args[ 'PEER' ],
                                standard='standard' in args,
                                large='large' in args,
                                lbw=args.get( 'LBW' ) )

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityStandardLargeCmd )

class SetNeighborSendCommunityExtendedCmd( CliCommand.CliCommandClass ):
   '''The command which accepts extended and LBW communities.'''
   syntax = ( 'neighbor PEER send-community '
              '( [ standard ] extended  [ large ] [ LBW ] ) | LBW' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'standard': bgpTokens.sendStandardCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'extended': bgpTokens.sendExtCommunity,
      'LBW': SendLinkBandwidthCommExpr
   }

   @staticmethod
   def handler( mode, args ):
      setNeighborSendCommunity( mode, args[ 'PEER' ],
                                standard='standard' in args,
                                large='large' in args,
                                extended='extended' in args,
                                lbw=args.get( 'LBW' ) )

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityExtendedCmd )

class SetNeighborSendCommunityAddOrRemoveCmd( CliCommand.CliCommandClass ):
   '''The command which accepts combinations of add/remove.'''
   syntax = ( 'neighbor PEER send-community ( add | remove ) '
              '( standard | extended | large | LBW )' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'add': bgpTokens.sendCommunityAdd,
      'remove': bgpTokens.sendCommunityRemove,
      'standard': bgpTokens.sendStandardCommunity,
      'extended': bgpTokens.sendExtCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'LBW': SendLinkBandwidthCommExpr
   }

   @staticmethod
   def handler( mode, args ):
      setNeighborSendCommunity( mode, args[ 'PEER' ],
                                standard='standard' in args,
                                extended='extended' in args,
                                large='large' in args,
                                lbw=args.get( 'LBW' ),
                                add='add' in args,
                                remove='remove' in args )

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityAddOrRemoveCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR shutdown command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborShutdownCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER shutdown [ reason MESSAGE ] [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER shutdown ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'shutdown': 'Disable the neighbor',
         'reason': bgpTokens.reason,
         'MESSAGE': bgpTokens.messageMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      msg = args.get( 'MESSAGE' )
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.neighborShutdownMsg = msg if msg else config.neighborShutdownMsgDefault
      config.neighborShutdown = True
      config.neighborShutdownPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.neighborShutdownPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                            config.isPeerGroupPeer )
         config.neighborShutdownMsg = config.neighborShutdownMsgDefault
         config.neighborShutdown = config.neighborShutdownDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborShutdownCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR dont-capability-negotiate" command, in
#   "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborDontCapabilityNegotiateCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER dont-capability-negotiate [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER dont-capability-negotiate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'dont-capability-negotiate': bgpTokens.dontCapabilityNegotiate,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=True )
      config.dontCapabilityNegotiate = True
      config.dontCapabilityNegotiatePresent = True

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      if config:
         config.dontCapabilityNegotiate = False
         config.dontCapabilityNegotiatePresent = (
            noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborDontCapabilityNegotiateCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-reflector-client [meshed]"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborReflectorClientCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER route-reflector-client [ meshed | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER route-reflector-client ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-reflector-client': bgpTokens.routeReflectorClient,
      'meshed': CliCommand.Node( bgpTokens.meshed, hidden=True ),
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      # rrClient is
      # = 0 by default, neighbor is not a route reflector client
      # = 1 when the neighbor is a route reflector client
      # = 2 for "neighbor ip-address route-reflector-client meshed",
      config.rrClient = ( RRClientEnum.reflectorMeshed if 'meshed' in args
                          else RRClientEnum.reflector )
      config.rrClientPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.rrClientPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                    config.isPeerGroupPeer )
         config.rrClient = config.rrClientDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborReflectorClientCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR stall [fail]" hidden command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborStallCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER stall [ fail ]'
   noOrDefaultSyntax = 'neighbor PEER stall ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'stall': bgpTokens.stallHidden,
      'fail': bgpTokens.stallFail
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      fail = 'fail' in args
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.stall = 'stallWriteFail' if fail else 'stallWriteBlock'
      config.stallPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.stallPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                 config.isPeerGroupPeer )
         config.stall = config.stallDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborStallCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR sockfault [greedyrecvestabfail]" hidden
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborSockFaultCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER sockfault [ greedyrecvestabfail ]'
   noOrDefaultSyntax = 'neighbor PEER sockfault ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'sockfault': bgpTokens.sockFaultHidden,
      'greedyrecvestabfail': bgpTokens.sockFaultGreedy
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      greedyRecvEstabFail = 'greedyrecvestabfail' in args
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, create=True, vrfName=mode.vrfName )
      if greedyRecvEstabFail:
         config.sockFault = 'greedyRecvEstablishedFault'
      config.sockFaultPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.sockFaultPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                     config.isPeerGroupPeer )
         config.sockFault = config.sockFaultDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborSockFaultCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-v6-addr IP6_ADDR" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLocalIp6AddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER local-v6-addr ( IP6_ADDR | disabled )'
   noOrDefaultSyntax = 'neighbor PEER local-v6-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrPeerGroupCliExpression,
      'local-v6-addr': bgpTokens.localIp6Addr,
      'IP6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher(
         "The local IPv6 address of the neighbor" )
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      v4OrPgPeerKey = args[ 'PEER' ]
      localIp6Addr = args[ 'IP6_ADDR' ]
      validatePeer( mode, v4OrPgPeerKey )
      config = bgpNeighborConfig( v4OrPgPeerKey, vrfName=mode.vrfName )
      config.localIp6Addr = localIp6Addr
      config.localIp6AddrPresent = True
      delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      v4OrPgPeerKey = args[ 'PEER' ]
      config = bgpNeighborConfig( v4OrPgPeerKey, create=False, vrfName=mode.vrfName )
      if config:
         config.localIp6AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
         config.localIp6Addr = config.localIp6AddrDefault
         delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalIp6AddrCmd )

#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-v6-addr IP6_ADDR in"
# command in "router-bgp" mode.
#-----------------------------------------------------------------------------------
class SetNeighborIp6NextHopAddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-v6-addr ( IP6_ADDR in ) | disabled'
   noOrDefaultSyntax = 'neighbor PEER next-hop-v6-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrPeerGroupCliExpression,
      'next-hop-v6-addr': bgpTokens.nextHopV6Addr,
      'IP6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher(
         "IPv6 next-hop address for the neighbor" ),
      'in': bgpTokens.nextHopV6AddrIn,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      v4OrPgPeerKey = args[ 'PEER' ]
      remoteIp6Addr = args[ 'IP6_ADDR' ]
      validatePeer( mode, v4OrPgPeerKey )
      config = bgpNeighborConfig( v4OrPgPeerKey, vrfName=mode.vrfName )
      config.remoteIp6Addr = remoteIp6Addr
      config.remoteIp6AddrPresent = True
      delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      v4OrPgPeerKey = args[ 'PEER' ]
      config = bgpNeighborConfig( v4OrPgPeerKey, create=False, vrfName=mode.vrfName )
      if config:
         config.remoteIp6AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                         config.isPeerGroupPeer )
         config.remoteIp6Addr = config.remoteIp6AddrDefault
         delNeighborConfigIfDefault( v4OrPgPeerKey, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborIp6NextHopAddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-v4-addr IP4_ADDR" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLocalIp4AddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER local-v4-addr ( IP4_ADDR | disabled )'
   noOrDefaultSyntax = 'neighbor PEER local-v4-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V6OrPeerGroupCliExpression,
      'local-v4-addr': bgpTokens.localIp4Addr,
      'IP4_ADDR': IpAddrMatcher.IpAddrMatcher(
         "The local IPv4 address of the neighbor" )
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      v6OrPgPeerKey = args[ 'PEER' ]
      localIp4Addr = args[ 'IP4_ADDR' ]
      validatePeer( mode, v6OrPgPeerKey )
      config = bgpNeighborConfig( v6OrPgPeerKey, vrfName=mode.vrfName )
      config.localIp4Addr = localIp4Addr
      config.localIp4AddrPresent = True
      delNeighborConfigIfDefault( v6OrPgPeerKey, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      v6OrPgPeerKey = args[ 'PEER' ]
      config = bgpNeighborConfig( v6OrPgPeerKey, create=False, vrfName=mode.vrfName )
      if config:
         config.localIp4AddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
         config.localIp4Addr = config.localIp4AddrDefault
         delNeighborConfigIfDefault( v6OrPgPeerKey, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalIp4AddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR auto-local-addr"
#-------------------------------------------------------------------------------
class SetNeighborAutoLocalAddrPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER auto-local-addr [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER auto-local-addr'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'auto-local-addr': bgpTokens.autoLocalAddr,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.autoLocalAddr = True
      config.autoLocalAddrPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.autoLocalAddrPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                         config.isPeerGroupPeer )
         config.autoLocalAddr = config.autoLocalAddrDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborAutoLocalAddrPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR additional-paths receive disabled"
#-------------------------------------------------------------------------------
class SetNeighborAddPathRecvCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER additional-paths receive [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER additional-paths receive'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'additional-paths': bgpTokens.addPath,
         'receive': bgpTokens.addpathReceive,
   } )

   @staticmethod
   def _setNeighborAddPathRecv( mode, peer ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
      setattr( config, attr + 'Present', True )
      setattr( config, attr, True )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborAddPathRecv( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer,
                                  vrfName=mode.vrfName,
                                  create=( noOrDefault == NoOrDefault.NO ) )
      attr = BgpLib.peerConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
      if config:
         if noOrDefault == NoOrDefault.NO:
            setattr( config, attr, False )
            setattr( config, attr + 'Present', True )
         else:
            # Add-path receive is enabled by default
            setattr( config, attr, True )
            setattr( config, attr + 'Present', False )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborAddPathRecvCmd._setNeighborAddPathRecv( mode, args[ 'PEER' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborAddPathRecvCmd._noNeighborAddPathRecv( mode,
                                                        args[ 'PEER' ],
                                                        noOrDefault )

modelets = [ RouterBgpSharedModelet,
             RouterBgpAfSharedModelet,
             RouterBgpAfLabelSharedModelet ]
for modelet in modelets:
   modelet.addCommandClass( SetNeighborAddPathRecvCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> additional-paths send any disabled"
#-------------------------------------------------------------------------------
class SetNeighborAddPathSendCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER additional-paths send any [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER additional-paths send any ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'additional-paths': bgpTokens.addPath,
         'send': bgpTokens.addpathSend,
         'any': bgpTokens.appAny,
   } )

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'any' ] = 'appAny'

   @staticmethod
   def _validateAddPathSend( mode ):
      if ( mode.vrfName != DEFAULT_VRF and
           mode.addrFamily in [ 'evpn', 'vpn-ipv4', 'vpn-ipv6' ] ):
         mode.addError( "Neighbor additional-paths send can only be configured under"
                        " %s mode for %s VRF" % ( mode.addrFamily, DEFAULT_VRF ) )
         return False
      elif ( mode.vrfName != DEFAULT_VRF and 'labeled-unicast' in mode.addrFamily ):
         mode.addError( "Neighbor additional-paths send can only be configured under"
                        " labeled-unicast mode for %s VRF" % DEFAULT_VRF )
         return False
      return True

   @staticmethod
   def _setNeighborAddPathSend( mode, peer, app ):
      validatePeer( mode, peer )
      if not SetNeighborAddPathSendCmd._validateAddPathSend( mode ):
         return
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      sendConfig = addPathSendConfig( config, mode.addrFamily )
      newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app, enable=True )
      sendConfig.addMember( newConfig )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborAddPathSend( mode, peer, app, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer,
                                  vrfName=mode.vrfName,
                                  create=( noOrDefault == NoOrDefault.NO ) )
      if config:
         sendConfig = addPathSendConfig( config, mode.addrFamily )
         if noOrDefault == NoOrDefault.NO:
            # The following setting is to support config inheritance
            newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app,
                                   enable=False )
            sendConfig.addMember( newConfig )
         elif app in sendConfig:
            del sendConfig[ app ]
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborAddPathSendCmd._setNeighborAddPathSend( mode,
                                                         args[ 'PEER' ],
                                                         args[ 'any' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborAddPathSendCmd._noNeighborAddPathSend( mode,
                                                        args[ 'PEER' ],
                                                        args[ 'any' ],
                                                        noOrDefault )

modelets = [ RouterBgpSharedModelet,
             RouterBgpAfIpUniSharedModelet,
             RouterBgpAfLabelSharedModelet,
             RouterBgpAfEvpnModelet ]
for modelet in modelets:
   modelet.addCommandClass( SetNeighborAddPathSendCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor GROUP | ADDR default-originate
#                                    [route-map MAP_NAME] [always]
#     in "router-bgp" or "router-bgp-af" mode
#-------------------------------------------------------------------------------

class SetNeighborDefaultOriginateBaseClass( object ):
   baseData = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'default-originate': bgpTokens.defOrigin,
         'default-route': bgpTokens.defaultRoute,
         'always': CliCommand.Node( bgpTokens.defOriginAlways,
                                    maxMatches=1 ),
         'route-map': CliCommand.Node( bgpTokens.routeMap,
                                       maxMatches=1 ),
         'MAP_NAME': CliCommand.Node( mapNameMatcher, maxMatches=1 ),
   }
   @staticmethod
   def _verifyNoConflictingDefOrigin( mode, config, addrFamily ):
      """ 
      Print an error if a conflicting default-originate config is already present.
      """
      # config in VPN address-families do not conflict with other modes
      if addrFamily in vpnAfTypeCliKeywords:
         return
      attrs = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ]
      conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
      for otherAf, c in conflictingAttrs:
         # config in VPN address-families do not conflict with other modes
         if otherAf in vpnAfTypeCliKeywords:
            continue
         if attrIsPresent( config, c ):
            errMsg = "Cannot configure default-originate in mode '%s' while it is "\
                     "configured in mode '%s'" % \
                     ( configModeCmdForAf( addrFamily ),
                       configModeCmdForAf( otherAf ) )
            mode.addError( errMsg )
            raise CliCommon.AlreadyHandledError

   @staticmethod
   def _noDefaultOriginateCommon( peer, noOrDefault, vrfName, addrFamily='all' ):
      # The "no" variant of the command is not same as "default" variant in case of
      # peerGroupPeer. Also, we should check for conflicting configuration in case of
      # "no" command for peerGroupPeer (as done in setDefaultOriginateCommon). But
      # the "neighbor <> route-map" command does not have this check. So, I am
      # leaving out the check here as well, till I verify if it is a bug or is
      # intentional. - pranav
      config = bgpNeighborConfig( peer, vrfName=vrfName )
      present = noOrDefault == NoOrDefault.NO and config.isPeerGroupPeer
      attr = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )
      if config:
         setattr( config, attr + 'Present', present )
         setattr( config, attr, False )
         setattr( config, attr + 'RouteMap',
                  getattr( config, attr + 'RouteMapDefault' ) )
         setattr( config, attr + 'Always',
                  getattr( config, attr + 'AlwaysDefault' ) )
         delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def _defaultOriginateCommon( mode, peer, addrFamily, mapName, always ):
      config = bgpNeighborConfig( peer, mode.vrfName )
      SetNeighborDefaultOriginateBaseClass._verifyNoConflictingDefOrigin(
            mode, config, addrFamily )
      attr = BgpLib.peerConfigAttrsAfMap[ 'defaultOriginate' ].get( addrFamily )

      setattr( config, attr, True )
      setattr( config, attr + 'Present', True )
      if mapName is None:
         mapName = getattr( config, attr + 'RouteMapDefault' )
      setattr( config, attr + 'RouteMap', mapName )
      setattr( config, attr + 'Always', always )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      addrFamily = mode.addrFamily if mode.addrFamily else 'all'
      SetNeighborDefaultOriginateBaseClass._defaultOriginateCommon(
                               mode, args[ 'PEER' ], addrFamily,
                               args.get( 'MAP_NAME' ),
                               'always' in args )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborDefaultOriginateBaseClass._noDefaultOriginateCommon(
            args[ 'PEER' ], noOrDefault, mode.vrfName, mode.addrFamily )

class SetNeighborDefaultOriginateCmd( SetNeighborDefaultOriginateBaseClass,
                                      BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER default-originate '
              '[ { always | ( route-map MAP_NAME ) } | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER default-originate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )

RouterBgpSharedModelet.addCommandClass( SetNeighborDefaultOriginateCmd )
RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborDefaultOriginateCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor GROUP | ADDR default-originate
#                                    [route-map MAP_NAME]
#     in "router-bgp-af" vpn mode ( evpn | vpn-ipv4 | vpn-ipv6 )
#
# Same as the previous but without 'always'
#-------------------------------------------------------------------------------

class SetVpnNeighborDefaultOriginateHiddenCmd( SetNeighborDefaultOriginateBaseClass,
                                               BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER default-originate '
              '[ ( route-map MAP_NAME ) | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER default-originate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )
   hidden = True

class SetVpnNeighborDefaultOriginateCmd( SetNeighborDefaultOriginateBaseClass,
                                         BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER default-route '
              '[ ( route-map MAP_NAME ) | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER default-route ...'
   data = BgpCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )

RouterBgpAfVpnModelet.addCommandClass( SetVpnNeighborDefaultOriginateHiddenCmd )
RouterBgpAfVpnModelet.addCommandClass( SetVpnNeighborDefaultOriginateCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR enforce-first-as" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborEnforceFirstAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER enforce-first-as [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER enforce-first-as'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'enforce-first-as': bgpTokens.enforceFirstAs,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.enforceFirstAs = True
      config.enforceFirstAsPresent = True

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer,
                                  create=( noOrDefault != NoOrDefault.DEFAULT ),
                                  vrfName=mode.vrfName )
      if config:
         if noOrDefault == NoOrDefault.DEFAULT:
            # Enforce First AS is enabled by default
            config.enforceFirstAs = True
            config.enforceFirstAsPresent = False
         else:
            config.enforceFirstAs = False
            config.enforceFirstAsPresent = True
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborEnforceFirstAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR update-source UPDATE_SRC_INTF"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborUpdateSrcIntfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER update-source ( UPDATE_SRC_INTF | disabled )'
   noOrDefaultSyntax = 'neighbor PEER update-source ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'update-source': bgpTokens.updateSrc,
      'UPDATE_SRC_INTF': IntfCli.Intf.matcherWithIpSupport,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      intf = args[ 'UPDATE_SRC_INTF' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.updateSrcIntf = intf.name
      config.updateSrcIntfPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.updateSrcIntfPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                         config.isPeerGroupPeer )
         config.updateSrcIntf = config.updateSrcIntfDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( NeighborUpdateSrcIntfCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR ebgp-multihop [TTL]" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborEbgpMultiHopTtlCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ebgp-multihop [ TTL | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER ebgp-multihop ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'ebgp-multihop': bgpTokens.ebgpMultiHop,
      'TTL': bgpTokens.ttlRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      ebgpMultiHopTtl = args.get( 'TTL' )
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if ebgpMultiHopTtl:
         config.ebgpMultiHop = ebgpMultiHopTtl
      else:
         # if ebgp-multihop is enabled default TTL is 255
         config.ebgpMultiHop = config.ebgpMultiHopEnabledTtlDefault
      config.ebgpMultiHopPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.ebgpMultiHopPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                        config.isPeerGroupPeer )
         config.ebgpMultiHop = config.ebgpMultiHopDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborEbgpMultiHopTtlCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR ttl maximum-hops NUM_HOPS" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborTtlMaxHopsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ttl maximum-hops NUM_HOPS'
   noOrDefaultSyntax = 'neighbor PEER ttl maximum-hops ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'ttl': bgpTokens.ttl,
      'maximum-hops': bgpTokens.maxHops,
      'NUM_HOPS': bgpTokens.numHopsRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      bgpTtlMaxHops = args[ 'NUM_HOPS' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.bgpTtlSecMaxHop = bgpTtlMaxHops
      config.bgpTtlSecMaxHopPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.bgpTtlSecMaxHopPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
         config.bgpTtlSecMaxHop = config.bgpTtlSecMaxHopDefault
         config.bgpTtlSecMaxHopLog = config.bgpTtlSecMaxHopLogDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborTtlMaxHopsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR link-bandwidth update-delay DELAY"
# command in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLinkbwDelayCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth update-delay DELAY'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth update-delay ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'link-bandwidth': bgpTokens.linkBwSend,
      'update-delay': bgpTokens.ucmpLinkbwDelay,
      'DELAY': bgpTokens.ucmpDelayRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      linkbwDelay = args[ 'DELAY' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.linkbwDelay = linkbwDelay
      config.linkbwDelayPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.linkbwDelayPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                       config.isPeerGroupPeer )
         config.linkbwDelay = config.linkbwDelayDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkbwDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR weight [WEIGHT]" command
#-------------------------------------------------------------------------------
class SetNeighborWeightCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER weight WEIGHT'
   noOrDefaultSyntax = 'neighbor PEER weight ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'weight': bgpTokens.weight,
         'WEIGHT': CliMatcher.IntegerMatcher( 0, 65535, helpdesc='Weight to assign' )
   }

   @staticmethod
   def _setWeight( mode, peer, weight ):
      validatePeer( mode, peer )
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=vrfName )
      if haveConflictingWeight( mode, config, addrFamily ):
         return
      attr = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
      if weight is None:
         weight = getattr( config, attr + 'Default' )
      setattr( config, attr, weight )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def _noWeight( mode, peer, noOrDefault ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
      attr = BgpLib.peerConfigAttrsAfMap[ 'weight' ].get( addrFamily )
      if config:
         setattr(
               config,
               attr + 'Present',
               ( noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer )
         )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborWeightCmd._setWeight( mode, args[ 'PEER' ], args[ 'WEIGHT' ] )


   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborWeightCmd._noWeight( mode, args[ 'PEER' ], noOrDefault )

for m in [ RouterBgpSharedModelet, RouterBgpAfSharedModelet ]:
   m.addCommandClass( SetNeighborWeightCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR allowas-in [NUM_AS]" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborAllowAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER allowas-in [ NUM_AS | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER allowas-in ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'allowas-in': bgpTokens.allowAsIn,
      'NUM_AS': bgpTokens.allowAsRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      allowAs = args.get( 'NUM_AS' )
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if allowAs is None:
         # if allowAs is enabled default allowAs loop count is 3
         config.allowAs = config.allowAsEnabledDefault
      else:
         config.allowAs = allowAs
      config.allowAsPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.allowAsPresent = noOrDefault != NoOrDefault.DEFAULT
      config.allowAs = config.allowAsDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborAllowAsCmd )

#-------------------------------------------------------------------------------
# Deprecated version of the Command:-
# "[no|default] neighbor GROUP | ADDR transport connection-mode passive"
# command, in "router-bgp" mode.
#
# New version of the command:-
# "[no|default] neighbor GROUP | ADDR passive"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborPassiveCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER ( passive | '
              '( transport connection-mode passive ) ) [ disabled ]' )
   noOrDefaultSyntax = ( 'neighbor PEER ( passive | '
                         '( transport connection-mode passive ) ) ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'passive': bgpTokens.passive,
      'transport': bgpTokens.transport,
      'connection-mode': bgpTokens.connectionModeDeprecated,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.passive = True
      config.passivePresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.passivePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                   config.isPeerGroupPeer )
         config.passive = config.passiveDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborPassiveCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR transport remote-port PORT_NUM" command
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetNeighborRemotePortCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER transport remote-port PORT_NUM'
   noOrDefaultSyntax = 'neighbor PEER transport remote-port ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'transport': bgpTokens.transport,
      'remote-port': bgpTokens.remotePort,
      'PORT_NUM': bgpTokens.remotePortNumRangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      remotePort = args[ 'PORT_NUM' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.remotePort = remotePort
      config.remotePortPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.remotePortPresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                      config.isPeerGroupPeer )
         config.remotePort = config.remotePortDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborRemotePortCmd )

#----------------------------------------------------------------------------------
# " [no|default] neighbor <addr|peer-group> transport pmtud" command
# "neighbor <addr|peer-group> transport pmtud [disabled]" command
# in "router-bgp" mode. With "no == default" and states to "default inherit".
#
# For this command, the 'disabled' keyword is not dependent on NED mode.
#----------------------------------------------------------------------------------
class SetNeighborTransportPmtudCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER transport pmtud [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER transport pmtud ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'transport': bgpTokens.transport,
         'pmtud': bgpTokens.pathMtuDiscovery, } )

   @staticmethod
   def _setNeighPathMtuDiscovery( mode, peer ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.pathMtuDiscovery = True
      config.pathMtuDiscoveryPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _disabledNeighPathMtuDiscovery( mode, peer ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.pathMtuDiscovery = False
      config.pathMtuDiscoveryPresent = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _defaultNeighPathMtuDiscovery( mode, peer ):
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.pathMtuDiscoveryPresent = False
      config.pathMtuDiscovery = config.pathMtuDiscoveryDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      if noOrDefault == NoOrDefault.DEFAULT:
         SetNeighborTransportPmtudCmd._defaultNeighPathMtuDiscovery(
               mode,
               args.get( 'PEER' ) )
      else:
         SetNeighborTransportPmtudCmd._disabledNeighPathMtuDiscovery(
               mode,
               args.get( 'PEER' ) )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborTransportPmtudCmd._setNeighPathMtuDiscovery(
            mode,
            args.get( 'PEER' ) )

RouterBgpSharedModelet.addCommandClass( SetNeighborTransportPmtudCmd )

#------------------------------------------------------------------------------
# Deprecated version of the command:-
# "[no|default] neighbor ADDR peer-group PEER_GROUP" command,
# in "router-bgp" mode.
#
# New version of the command:-
# "[no|default] neighbor ADDR peer group PEER_GROUP" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborPeerGroupCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ( ( peer group ) | peer-group ) PEER_GROUP'
   noOrDefaultSyntax = 'neighbor PEER ( ( peer group ) | peer-group ) ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4V6PeerKeyCliExpression,
      'peer': bgpTokens.peer,
      'group': bgpTokens.group,
      'peer-group': bgpTokens.peerGroupDeprecated,
      'PEER_GROUP': peergroupNameMatcher
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      v4v6PeerKey = args[ 'PEER' ]
      pgKey = PeerConfigKey( args[ 'PEER_GROUP' ] )
      validatePeer( mode, v4v6PeerKey )
      peerGroup = bgpNeighborConfig( pgKey, vrfName=mode.vrfName )
      peer = bgpNeighborConfig( v4v6PeerKey, vrfName=mode.vrfName )
      peer.peerGroupKey = peerGroup.key
      delNeighborConfigIfDefault( v4v6PeerKey, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      v4v6PeerKey = args[ 'PEER' ]
      config = bgpNeighborConfig( v4v6PeerKey, create=False, vrfName=mode.vrfName )
      if config:
         if config.isPeerGroupPeer:
            config.peerGroupKey = PeerConfigKeyType()
            resetPresents( config )
         delNeighborConfigIfDefault( v4v6PeerKey, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborPeerGroupCmd )

#-------------------------------------------------------------------------------
# Deprecated version of the command:
# "[no|default] neighbor PEER_GROUP peer-group" command,
#in "router-bgp" mode.
#
# New version of the command:
# "[no|default] neighbor PEER_GROUP peer group" command,
#in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ConfigureNeighborPeerGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor GROUP ( ( peer group ) | peer-group )'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'GROUP': peergroupNameMatcher,
      'peer': bgpTokens.peer,
      'group': bgpTokens.group,
      'peer-group': bgpTokens.peerGroupDeprecated,
   }

   @staticmethod
   def handler( mode, args ):
      pgKey = PeerConfigKey( args[ 'GROUP' ] )
      bgpNeighborConfig( pgKey, vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pgKey = PeerConfigKey( args[ 'GROUP' ] )
      config = bgpNeighborConfig( pgKey, create=False, vrfName=mode.vrfName )
      peergroupName = pgKey.group
      if config:
         # if this peer-group is being used by a configured listen range or neighbor
         # interface config in any VRF prevent peer-group config from being removed
         if isDynamicPeerGroupByName( peergroupName ):
            mode.addError( 'Cannot remove peer-group %s while a listen-range '
                           'referencing the peer-group is still configured' %
                           peergroupName )
            return
         if isNeighIntfConfigPeerGroup( peergroupName ):
            mode.addError( 'Cannot remove peer-group %s while a neighbor '
                           'interface config referencing the peer-group '
                           'is still configured' % peergroupName )
            return
         # cleanup all references to this peer-group for all configured static peers
         for peer in bgpConfig.neighborConfig.values():
            if peer.isPeerGroupPeer and peer.peerGroupKey == config.key:
               peer.peerGroupKey = PeerConfigKeyType()
               resetPresents( peer )
               delNeighborConfigIfDefault( peer.key, vrfName=DEFAULT_VRF )
         for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
            for peer in vrfConfig.neighborConfig.values():
               if peer.isPeerGroupPeer and peer.peerGroupKey == config.key:
                  peer.peerGroupKey = PeerConfigKeyType()
                  resetPresents( peer )
                  delNeighborConfigIfDefault( peer.key, vrfName=vrfConfig.name )
         # delete the peer-group config in the end
         del bgpConfig.neighborConfig[ config.key ]

RouterBgpSharedModelet.addCommandClass( ConfigureNeighborPeerGroupCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> bfd [c-bit]" command,
# in "router-bgp" mode.
#
# Deprecated version of the command:
# "[no|default] neighbor <addr> fall-over bfd" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborBfdCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER [ fall-over ] bfd [ c-bit ] [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER [ fall-over ] bfd [ c-bit ] [ disabled ] ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'fall-over': bgpTokens.fallOverDeprecated,
         'bfd': bgpTokens.bfd,
         'c-bit': 'Enable support for the BFD C-bit',
   } )

   @staticmethod
   def _noBfd( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.bfdEnabledStatePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
         config.bfdEnabledState = config.bfdEnabledStateDefault
         config.bfdCBitEnabled = config.bfdCBitEnabledDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _setBfd( mode, peer, cbit=False ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.bfdEnabledStatePresent = True
      config.bfdEnabledState = True
      config.bfdCBitEnabled = cbit

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborBfdCmd._noBfd(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborBfdCmd._setBfd( mode, args[ 'PEER' ], cbit=( 'c-bit' in args ) )

class SetNeighborBfdWithDampingCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER [ fall-over ] bfd [ ( [ c-bit ]'
              ' [ damping [ seconds TIME [ max-multiplier MULTIPLIER ] ] ] )'
              ' | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER [ fall-over ] bfd ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'fall-over': bgpTokens.fallOverDeprecated,
         'bfd': bgpTokens.bfd,
         'c-bit': 'Enable support for the BFD C-bit',
         'damping': 'Enable support for the BFD Damping',
         'seconds': 'Damping time in seconds',
         'TIME': CliMatcher.IntegerMatcher( 10, 3600, helpdesc='Number of seconds' ),
         'max-multiplier': 'Maximum multiplier for damping time',
         'MULTIPLIER': CliMatcher.IntegerMatcher( 1, 60, helpdesc='Max multiplier' )
   } )

   @staticmethod
   def _noBfd( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.bfdEnabledStatePresent = ( noOrDefault != NoOrDefault.DEFAULT and
                                           config.isPeerGroupPeer )
         config.bfdEnabledState = config.bfdEnabledStateDefault
         config.bfdCBitEnabled = config.bfdCBitEnabledDefault
         config.bfdDampingEnabled = config.bfdDampingEnabledDefault
         config.bfdDampingTime = config.bfdDampingTimeDefault
         config.bfdDampingMultiplier = config.bfdDampingMultiplierDefault
         config.bfdDampingTimePresent = False
         config.bfdDampingMultiplierPresent = False
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _setBfd( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.bfdEnabledStatePresent = True
      config.bfdEnabledState = True
      config.bfdCBitEnabled = 'c-bit' in args
      config.bfdDampingEnabled = 'damping' in args
      config.bfdDampingTimePresent = 'TIME' in args
      config.bfdDampingTime = args.get( 'TIME', config.bfdDampingTimeDefault )
      config.bfdDampingMultiplierPresent = 'MULTIPLIER' in args
      config.bfdDampingMultiplier = args.get( 'MULTIPLIER',
                                              config.bfdDampingMultiplierDefault )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborBfdWithDampingCmd._noBfd(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborBfdWithDampingCmd._setBfd( mode, args )

if BgpCommonToggleLib.toggleArBgpPeerFlapDampingEnabled():
   RouterBgpSharedModelet.addCommandClass( SetNeighborBfdWithDampingCmd )
else:
   RouterBgpSharedModelet.addCommandClass( SetNeighborBfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|pg> link-bandwidth
#  [ auto | default <lbw> | adjust auto [ percent <%> ] ] command, in
#  "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLinkBwAutoBase( object ):
   syntax = 'neighbor PEER link-bandwidth auto'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth auto'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'auto': bgpTokens.linkBwAuto,
         'percent': bgpTokens.linkBwPercent,
         'PERCENT': bgpTokens.linkBwPercentRangeMatcher,
   }

   @staticmethod
   def _setLinkBwAuto( mode, peer, percent=None ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if config:
         config.linkBw = LinkBandwidthGenerationStateEnum.linkBandwidthAutoGeneration
         config.linkBwPresent = True
         config.linkBwDef = config.linkBwDefDefault
         if percent:
            config.linkBwAutoPercentPresent = True
            config.linkBwAutoPercent = percent
         else:
            config.linkBwAutoPercentPresent = False
            config.linkBwAutoPercent = config.linkBwAutoPercentDefault
         config.linkBwDefDisplay = config.linkBwDefDisplayDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noLinkBwAuto( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.linkBwPresent = (
               noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         )
         config.linkBwAutoPercentPresent = False
         config.linkBw = config.linkBwDefault
         config.linkBwAutoPercent = config.linkBwAutoPercentDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

class SetNeighborLinkBwAutoPercentCmd(
      BgpNEDCmdBaseClass,
      SetNeighborLinkBwAutoBase
):
   syntax = SetNeighborLinkBwAutoBase.syntax + ' [ percent PERCENT ] [disabled]'
   noOrDefaultSyntax = SetNeighborLinkBwAutoBase.noOrDefaultSyntax
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborLinkBwAutoBase.data.copy()
   )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborLinkBwAutoBase._setLinkBwAuto(
            mode,
            args[ 'PEER' ],
            percent=args.get( 'PERCENT' )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborLinkBwAutoBase._noLinkBwAuto( mode, args[ 'PEER' ], noOrDefault )

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwAutoPercentCmd )

class SetNeighborLinkBwAdjustCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth adjust auto [ percent PERCENT ] [disabled]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth adjust auto'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'adjust': bgpTokens.linkBwAdjust,
         'auto': bgpTokens.linkBwAutoAfterAdjust,
         'percent': bgpTokens.linkBwPercent,
         'PERCENT': bgpTokens.linkBwPercentRangeMatcher,
   } )

   @staticmethod
   def _setLinkBwAdjust( mode, peer, percent ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if config:
         if percent:
            config.linkBwAdjustPercentPresent = True
            config.linkBwAdjustPercent = percent
         else:
            config.linkBwAdjustPercentPresent = False
            config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
         config.linkBwAdjustPresent = True
         config.linkBwAdjust = True
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noLinkBwAdjust( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.linkBwAdjustPresent = (
               noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         )
         config.linkBwAdjustPercentPresent = False
         config.linkBwAdjust = config.linkBwAdjustDefault
         config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborLinkBwAdjustCmd._setLinkBwAdjust(
            mode,
            args[ 'PEER' ],
            args.get( 'PERCENT' )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborLinkBwAdjustCmd._noLinkBwAdjust(
            mode,
            args[ 'PEER' ],
            noOrDefault
   )

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwAdjustCmd )

class SetNeighborLinkBwCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
   } )

   @staticmethod
   def _setLinkBw( mode, peer ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if config:
         config.linkBw = LinkBandwidthGenerationStateEnum.linkBandwidthProcessing
         config.linkBwPresent = True
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noLinkBw( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.linkBwPresent = (
               noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         )
         config.linkBw = config.linkBwDefault
         config.linkBwDef = config.linkBwDefDefault
         config.linkBwAutoPercent = config.linkBwAutoPercentDefault
         config.linkBwDefDisplay = config.linkBwDefDisplayDefault
         config.linkBwAdjustPresent = False
         config.linkBwAdjust = config.linkBwAdjustDefault
         config.linkBwAdjustPercentPresent = False
         config.linkBwAutoPercentPresent = False
         config.linkBwAdjustPercent = config.linkBwAdjustPercentDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborLinkBwCmd._setLinkBw( mode, args[ 'PEER' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborLinkBwCmd._noLinkBw( mode, args[ 'PEER' ], noOrDefault )

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwCmd )

class SetNeighborLinkBwDefaultCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth default COMM [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth default'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'default': bgpTokens.linkBwDefault,
         'COMM': LinkBandwidthValueCliMatcher( 'Link speed in bits/second' )
   } )

   @staticmethod
   def _setLinkBwDefault( mode, peer, lbwVal ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if config:
         config.linkBw = (
               LinkBandwidthGenerationStateEnum.linkBandwidthDefaultGeneration
         )
         config.linkBwPresent = True
         config.linkBwDefDisplay = lbwVal
         config.linkBwDef = getLbwCommValue( lbwVal )
         config.linkBwAutoPercent = config.linkBwAutoPercentDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noLinkBwDefault( mode, peer, noOrDefault ):
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.linkBwPresent = (
               noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer
         )
         config.linkBw = config.linkBwDefault
         config.linkBwDef = config.linkBwDefDefault
         config.linkBwDefDisplay = config.linkBwDefDisplayDefault
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborLinkBwDefaultCmd._setLinkBwDefault(
            mode,
            args[ 'PEER' ],
            args[ 'COMM' ]
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborLinkBwDefaultCmd._noLinkBwDefault(
            mode,
            args[ 'PEER' ],
            noOrDefault
      )

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwDefaultCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> route-map <name> in [disabled]" command in
# address-family sr-te block
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> route-map <name> <in|out>" command in address-family
# block
#-------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> next-hop address-family ipv6 [originate]"
#      command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
class SetNeighborNextHopAfV6OriginateCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop address-family ipv6 [ originate ] [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop address-family ipv6 ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'address-family': bgpTokens.nextHopAf,
         'ipv6': bgpTokens.ipv6,
         'originate': bgpTokens.originate
   } )

   @staticmethod
   def _setNeighborV6NextHopAf( mode, peer, originate ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if mode.addrFamily == 'ipv4':
         if originate:
            config.v6NextHopAfV4Uni = (
                  ExtendedNextHopCapabilityEnum.isEnabledWithOriginate
            )
         else:
            config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isEnabled
         config.v6NextHopAfV4UniPresent = True
      elif mode.addrFamily == 'ipv6':
         mode.addError(
               'IPv6 nexthop address family may not be specified in IPv6 mode'
         )
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _noNeighborV6NextHopAf( mode, noOrDefault, peer ):
      validatePeer( mode, peer )
      if mode.addrFamily == 'ipv4':
         config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
         if config:
            config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isDisabled
            config.v6NextHopAfV4UniPresent = ( noOrDefault != NoOrDefault.DEFAULT )
      elif mode.addrFamily == 'ipv6':
         mode.addError(
               'IPv6 nexthop address family may not be specified in IPv6 mode'
         )
      else:
         raise ValueError()
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborNextHopAfV6OriginateCmd._setNeighborV6NextHopAf(
            mode,
            args[ 'PEER' ],
            ( 'originate' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborNextHopAfV6OriginateCmd._noNeighborV6NextHopAf(
            mode,
            noOrDefault,
            args[ 'PEER' ]
      )

RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborNextHopAfV6OriginateCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR aigp-session [disabled]" command,
# in "config-bgp-af" mod
#-----------------------------------------------------------------------------
class NeighborAigpSessionCommand( BgpNEDCmdBaseClass ):
   syntax = "neighbor PEER aigp-session [ disabled ]"
   noOrDefaultSyntax = "neighbor PEER aigp-session ..."
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'aigp-session': bgpTokens.aigpSession,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         mode.addWarning(
            "AIGP-session is only supported in multi-agent mode" )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if mode.addrFamily == 'ipv4':
         config.aigpSessionIPv4UniPresent = True
         config.aigpSessionIPv4Uni = True
      else:
         config.aigpSessionIPv6UniPresent = True
         config.aigpSessionIPv6Uni = True

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      # Explicit 'no' case
      #
      # Note that, because there are per-peer-type globals for
      # aigp-session config, we need to keep the explicit 'no' at the
      # peer level (even when the peer is not in a peer group).
      #
      # There are per-peer-type global configs for aigp-session: one for
      # iBGP peers, one for confed peers, and one for eBGP peers.
      #
      # For example, when the global for iBGP peers is set to true, if
      # the aigp-session for an iBGP peer is not configured at the peer
      # level, the aigp-session is enabled. On the other hand, when 'no'
      # is configured at the peer level, it overrides the global
      # default.
      #
      # In order to override the per-peer-type global defaults, set the
      # config to false explicitly.
      explicitNo = ( noOrDefault == NoOrDefault.NO )
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, create=explicitNo,
                                  vrfName=mode.vrfName )
      if config:
         if mode.addrFamily == 'ipv4':
            config.aigpSessionIPv4UniPresent = explicitNo
            config.aigpSessionIPv4Uni = config.aigpSessionIPv4UniDefault
         else:
            config.aigpSessionIPv6UniPresent = explicitNo
            config.aigpSessionIPv6Uni = config.aigpSessionIPv6UniDefault
         if not explicitNo:
            delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
   RouterBgpAfIpUniSharedModelet.addCommandClass( NeighborAigpSessionCommand )

#-------------------------------------------------------------------------------
# "[no|default] neighbor default encapsulation ( mpls | vxlan )"
# "[no|default] neighbor default encapsulation mpls next-hop-self
#   source-interface INTF"
# in 'address-family evpn' mode
#-------------------------------------------------------------------------------
class SetNeighborDefaultEncapCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor default encapsulation '
              '( vxlan | ( mpls [ next-hop-self source-interface INTF ] ) )' )
   noOrDefaultSyntax = 'neighbor default encapsulation ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'default': bgpTokens.default,
         'encapsulation': bgpTokens.encap,
         'mpls': bgpTokens.mplsForEvpn,
         'vxlan': bgpTokens.vxlan,
         'next-hop-self': bgpTokens.nextHopSelf,
         'source-interface': bgpTokens.sourceInterface,
         'INTF': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if 'vxlan' in args:
         args[ 'vxlan' ] = 'encapVxlan'
      elif 'mpls' in args:
         args[ 'mpls' ] = 'encapMpls'
         if 'INTF' in args:
            intf = args.get( 'INTF' )
            args[ 'INTF' ] = intf.name

   @staticmethod
   def _setDefaultEncap( mode, mpls=None, vxlan=None, intf=None ):
      config = configForVrf( mode.vrfName )
      if intf:
         config.afEvpnEncap = mpls
         config.afEvpnMplsNexthopSelfSrcIntf = intf
      else:
         config.afEvpnEncap = vxlan or mpls
         config.afEvpnMplsNexthopSelfSrcIntf = (
               config.afEvpnMplsNexthopSelfSrcIntfDefault )

   @staticmethod
   def _noDefaultEncap( mode ):
      config = configForVrf( mode.vrfName )
      config.afEvpnEncap = config.afEvpnEncapDefault
      config.afEvpnMplsNexthopSelfSrcIntf = (
            config.afEvpnMplsNexthopSelfSrcIntfDefault )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborDefaultEncapCmd._noDefaultEncap( mode )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborDefaultEncapCmd._setDefaultEncap(
            mode,
            mpls=args.get( 'mpls' ),
            vxlan=args.get( 'vxlan' ),
            intf=args.get( 'INTF' ) )

RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborDefaultEncapCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> encapsulation ( mpls | vxlan )"
# "[no|default] neighbor <addr|peer-group> encapsulation mpls next-hop-self
#   source-interface INTF"
# in 'address-family evpn' mode
#-------------------------------------------------------------------------------
class SetNeighborEncapCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER encapsulation '
              '( vxlan | ( mpls [ next-hop-self source-interface INTF ] ) )' )
   noOrDefaultSyntax = 'neighbor PEER encapsulation ...'
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'encapsulation': bgpTokens.encap,
      'mpls': bgpTokens.mplsForEvpn,
      'vxlan': bgpTokens.vxlan,
      'next-hop-self': bgpTokens.nextHopSelf,
      'source-interface': bgpTokens.sourceInterface,
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      if 'vxlan' in args:
         args[ 'vxlan' ] = 'encapVxlan'
      elif 'mpls' in args:
         args[ 'mpls' ] = 'encapMpls'
         if 'INTF' in args:
            intf = args.get( 'INTF' )
            args[ 'INTF' ] = intf.name

   @staticmethod
   def _setPeerEncap( mode, peer, mpls=None, vxlan=None, intf=None ):
      peerConfig = bgpNeighborConfig( peer, mode.vrfName )
      peerConfig.afEvpnEncapPresent = True
      if intf:
         peerConfig.afEvpnEncap = mpls
         peerConfig.afEvpnMplsNexthopSelfSrcIntf = intf
         peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = True
      else:
         peerConfig.afEvpnEncap = vxlan or mpls
         peerConfig.afEvpnMplsNexthopSelfSrcIntf = (
               peerConfig.afEvpnMplsNexthopSelfSrcIntfDefault )
         peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = False

   @staticmethod
   def _noOrDefaultPeerEncap( mode, peer ):
      peerConfig = bgpNeighborConfig( peer, mode.vrfName, create=False )
      if peerConfig:
         peerConfig.afEvpnEncap = peerConfig.afEvpnEncapDefault
         peerConfig.afEvpnEncapPresent = False
         peerConfig.afEvpnMplsNexthopSelfSrcIntf = (
               peerConfig.afEvpnMplsNexthopSelfSrcIntfDefault )
         peerConfig.afEvpnMplsNexthopSelfSrcIntfPresent = False

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborEncapCmd._setPeerEncap(
            mode,
            peer=args[ 'PEER' ],
            mpls=args.get( 'mpls' ),
            vxlan=args.get( 'vxlan' ),
            intf=args.get( 'INTF' ) )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborEncapCmd._noOrDefaultPeerEncap(
            mode,
            peer=args[ 'PEER' ] )

if BgpCommonToggleLib.toggleEvpnPerPeerEncapEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborEncapCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR out-delay DELAY
#  [ changes [ withdrawals ] ]" commands
#-------------------------------------------------------------------------------
class PeerOutDelayCmdMixin( object ):
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'out-delay': bgpTokens.outDelayMatcherForConfig,
         'DELAY': CliMatcher.IntegerMatcher(
               0,
               600,
               helpdesc='Time in seconds by which route updates should be delayed.'
         ),
         'changes': bgpTokens.outDelayChangesMatcherForConfig,
         'withdrawals': bgpTokens.outDelayWithdrawalsMatcherForConfig
   }

   @staticmethod
   def setOutDelay( mode, peer, outDelay, changes=False, withdrawals=False ):
      validatePeer( mode, peer )
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=vrfName )
      if haveConflictingOutDelay( mode, config, addrFamily ):
         return
      attr = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
      if not outDelay:
         outDelay = getattr( config, attr + 'Default' )
      if changes and withdrawals:
         outDelayApply = OutDelayApplyEnum.outDelayApplyChangesWithdrawals
      elif changes:
         outDelayApply = OutDelayApplyEnum.outDelayApplyChanges
      else:
         outDelayApply = OutDelayApplyEnum.outDelayApplyInitial
      config.outDelayApply = outDelayApply
      setattr( config, attr, outDelay )
      setattr( config, attr + 'Present', True )
      delNeighborConfigIfDefault( peer, vrfName=vrfName )

   @staticmethod
   def noOutDelay( mode, peer, noOrDefault ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=vrfName, create=False )
      attr = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ].get( addrFamily )
      if config:
         if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
            setattr( config, attr + 'Present', True )
         else:
            setattr( config, attr + 'Present', False )
         setattr( config, attr, getattr( config, attr + 'Default' ) )
         config.outDelayApply = OutDelayApplyEnum.outDelayApplyInitial
         delNeighborConfigIfDefault( peer, vrfName=vrfName )

class SetNeighborOutDelayCmd( PeerOutDelayCmdMixin, BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER out-delay DELAY [ changes [ withdrawals ] ]'
   noOrDefaultSyntax = 'neighbor PEER out-delay ...'
   data = PeerOutDelayCmdMixin.data.copy()

   @staticmethod
   def _handleNormal( mode, args ):
      PeerOutDelayCmdMixin.setOutDelay(
            mode,
            args[ 'PEER' ],
            args[ 'DELAY' ],
            changes=( 'changes' in args ),
            withdrawals=( 'withdrawals' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      PeerOutDelayCmdMixin.noOutDelay( mode, args[ 'PEER' ], noOrDefault )

RouterBgpSharedModelet.addCommandClass( SetNeighborOutDelayCmd )

class SetNeighborOutDelayHiddenCmd(
      PeerOutDelayCmdMixin,
      CliCommand.CliCommandClass
):
   syntax = 'neighbor PEER out-delay DELAY'
   noOrDefaultSyntax = 'neighbor PEER out-delay ...'
   data = PeerOutDelayCmdMixin.data.copy()
   hidden = True

   @staticmethod
   def handler( mode, args ):
      PeerOutDelayCmdMixin.setOutDelay( mode, args[ 'PEER' ], args[ 'DELAY' ] )

   @staticmethod
   def noHandler( mode, args ):
      PeerOutDelayCmdMixin.noOutDelay( mode, args[ 'PEER' ], NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      PeerOutDelayCmdMixin.noOutDelay( mode, args[ 'PEER' ], NoOrDefault.DEFAULT )

RouterBgpAfSharedModelet.addCommandClass( SetNeighborOutDelayHiddenCmd )

#-------------------------------------------------------------------------------
# '[no|default] neighbor <addr|peer-group> next-hop-self \
#            source-interface <interface-name>
#     under 'address-family <addrFamily> labeled-unicast' mode'
#-------------------------------------------------------------------------------
class SetNeighborNextHopSelfAfSrcIntfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-self source-interface ( INTF | disabled )'
   noOrDefaultSyntax = 'neighbor PEER next-hop-self source-interface ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop-self': bgpTokens.nextHopSelf,
         'source-interface': bgpTokens.sourceInterface,
         'INTF': IntfCli.Intf.matcherWithIpSupport,
   } )

   @staticmethod
   def _setNexthopSelfSourceInterface( mode, peer, intf ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( mode.addrFamily )
      setattr( config, attr, intf.name )
      setattr( config, attr + 'Present', True )

   @staticmethod
   def _noNexthopSelfSourceInterface( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
      attr = BgpLib.peerConfigAttrsAfMap[ 'nhSelfSrcIntf' ].get( mode.addrFamily )
      if config:
         setattr( config, attr, config.nhSelfSrcIntfDefault )
         if noOrDefault != NoOrDefault.DEFAULT and config.isPeerGroupPeer:
            setattr( config, attr + 'Present', True )
         else:
            setattr( config, attr + 'Present', False )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborNextHopSelfAfSrcIntfCmd._setNexthopSelfSourceInterface(
            mode,
            args[ 'PEER' ],
            args[ 'INTF' ]
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborNextHopSelfAfSrcIntfCmd._noNexthopSelfSourceInterface(
            mode,
            args[ 'PEER' ],
            noOrDefault
      )

RouterBgpAfLabelSharedModelet.addCommandClass( SetNeighborNextHopSelfAfSrcIntfCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor interface <interface-range> peer-group <pg-name>
#  ( ( remote-as <ASN> ) | ( peer-filter PEER_FILTER ) )"
#  command, in "router-bgp" mode.
# The no and default versions of this command behave the same which is to remove the
# config whether in normal config mode or in no-equals-default mode. Hence there is
# no need for a 'disabled' version for no-equals-default mode.
#----------------------------------------------------------------------------------
class SetNeighborInterfaceCmd( BgpCmdBaseClass ):
   _peerFilterEnabled = BgpCommonToggleLib.toggleArBgpAsnRangePeerFilterEnabled()
   if _peerFilterEnabled:
      syntax = ( 'neighbor interface INTFS peer-group PG '
                 '( ( remote-as AS_NUM ) | ( peer-filter PEER_FILTER ) )' )
   else:
      syntax = 'neighbor interface INTFS peer-group PG remote-as AS_NUM'
   noOrDefaultSyntax = 'neighbor interface INTFS ...'

   data = {
         'neighbor': bgpTokens.neighbor,
         'interface': bgpTokens.interface,
         'INTFS': IntfRangeMatcher(),
         'peer-group': bgpTokens.peerGroup,
         'PG': peergroupNameMatcher,
         'remote-as': bgpTokens.remoteAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
   }
   if _peerFilterEnabled:
      data.update(
         { 'peer-filter': bgpTokens.peerFilter,
          'PEER_FILTER': bgpTokens.peerFilterNameMatcher,
         }
      )

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )

      for intf in args[ 'INTFS' ].intfNames():
         intfId = IntfId( intf )
         peerConfig = config.neighIntfConfig.get( intfId )
         if not peerConfig:
            peerConfig = config.neighIntfConfig.newMember( intfId )
         asnOrPeerFilter = Tac.Value( "Routing::Bgp::AsnOrPeerFilter" )
         if 'AS_NUM' in args:
            asnOrPeerFilter.asn = args[ 'AS_NUM' ]
         else:
            assert SetNeighborInterfaceCmd._peerFilterEnabled
            asnOrPeerFilter.hasAsn = False
            asnOrPeerFilter.peerFilter = args[ 'PEER_FILTER' ]
         peerConfig.asnOrPeerFilter = asnOrPeerFilter
         peerConfig.peerGroup = PeerConfigKey( args[ 'PG' ] )

      protocolModel = getEffectiveProtocolModel( mode )
      if( protocolModel != ProtoAgentModel.multiAgent and 'PEER_FILTER' in args ):
         if mode.session.isInteractive():
            mode.addError( "peer-filter is only supported in multi-agent mode" )
         return

      bgpNeighborConfig( PeerConfigKey( args[ 'PG' ] ), vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )

      for intf in args[ 'INTFS' ].intfNames():
         peerConfig = config.neighIntfConfig.get( intf )
         if peerConfig:
            del config.neighIntfConfig[ intf ]

RouterBgpSharedModelet.addCommandClass( SetNeighborInterfaceCmd )

#-------------------------------------------------------------------------------
# The "[no|default] neighbor <peer|group> route refresh demarcated
# stale-path removal [disabled|(timeout <secs> )]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class RouteRefreshConfigCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER route refresh demarcated stale-path removal '
              '[ ( timeout TIMEOUT ) | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER route ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route': 'Route Refresh configuration',
      'refresh': 'Route Refresh configuration',
      'demarcated': 'Demarcated Route Refresh configuration',
      'stale-path': 'Stale paths removal configuration',
      'removal': 'Stale paths removal configuration',
      'timeout': 'Set the time (in seconds) before timing out waiting for ' +
      'End of Route Refresh demarcation',
      'TIMEOUT': CliMatcher.IntegerMatcher(
         0, bgpTokens.U16_MAX_VALUE,
         helpdesc='Time (in seconds) before timing out waiting for ' +
         'End of Route Refresh demarcation' ),
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      timeoutSecs = args.get( 'TIMEOUT' )
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.enhRtRefreshIn = True
      config.enhRtRefreshInPresent = True
      if timeoutSecs is not None:
         # '0' is a valid value
         config.enhRtRefreshStalePathTimePresent = True
         config.enhRtRefreshStalePathTime = timeoutSecs
      else:
         config.enhRtRefreshStalePathTimePresent = False
         config.enhRtRefreshStalePathTime = config.enhRtRefreshStalePathTimeDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if not config:
         return
      # 'peer' can be either a peer, peer group, or a peer group peer
      # (a peer group member).
      #
      # For a peer group peer, "default" means fall back to using
      # the peer group's config, so 'enhRtRefreshInPresent' # is set to False.
      # "no" means explicitly disabling the feature, regardless of the peer
      # group configuration. For this reason, 'enhRtRefreshInPresent' is set to
      # True and 'enhRtRefreshIn' is set to False.
      if noOrDefault == NoOrDefault.NO and config.isPeerGroupPeer:
         # Peer group peer: explicitly disabled.
         config.enhRtRefreshInPresent = True
         config.enhRtRefreshIn = False
      else:
         # In all other cases, simply remove the configuration.
         config.enhRtRefreshInPresent = False
         config.enhRtRefreshIn = config.enhRtRefreshInDefault
      config.enhRtRefreshStalePathTimePresent = False
      config.enhRtRefreshStalePathTime = config.enhRtRefreshStalePathTimeDefault
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( RouteRefreshConfigCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor <addr|peer-group> next-hop-unchanged [disabled]
#     in "router-bgp-af" and "router-bgp"  modes
#-------------------------------------------------------------------------------
class SetNeighborNextHopUnchangedCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-unchanged [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop-unchanged': bgpTokens.nexthopUnchanged,
   } )

   @staticmethod
   def _setNeighborNexthopUnchanged( mode, peer ):
      validatePeer( mode, peer )
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      attr = BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
      setattr( config, attr, True )
      setattr( config, attr + 'Present', True )

   @staticmethod
   def _noNeighborNexthopUnchanged( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      addrFamily = mode.addrFamily
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName,
                                  create=( noOrDefault != NoOrDefault.DEFAULT ) )
      attr = BgpLib.peerConfigAttrsAfMap[ 'nexthopUnchanged' ].get( addrFamily )
      if config:
         if noOrDefault != NoOrDefault.DEFAULT:
            setattr( config, attr + 'Present', True )
            setattr( config, attr, False )
         else:
            setattr( config, attr + 'Present', False )
            setattr( config, attr, getattr( config, attr + 'Default' ) )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborNextHopUnchangedCmd._noNeighborNexthopUnchanged(
         mode,
         args[ 'PEER' ],
         noOrDefault )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborNextHopUnchangedCmd._setNeighborNexthopUnchanged(
            mode,
            args[ 'PEER' ] )

for m in [ RouterBgpSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNeighborNextHopUnchangedCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor PEER route-target export VPN_NLRI_TYPE filter [ disabled ]
#-------------------------------------------------------------------------------
class NeighborRouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER route-target export VPN_NLRI_TYPE filter [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER route-target export VPN_NLRI_TYPE filter ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export,
         'VPN_NLRI_TYPE': bgpTokens.VpnNlriTypeExpr,
         'filter': 'Set route-target export filtering behavior',
   } )

   @staticmethod
   def _setFilterDisabled( mode, peer, vpnNlriType, value ):
      collectionType = 'vrfAf' if mode.addrFamily != 'all' else 'vrf'
      validatePeer( mode, peer )
      if collectionType == 'vrfAf':
         ipVersion = int( mode.addrFamily[ -1 ] )
         if vpnNlriType.ipVersion != ipVersion:
            mode.addError( '%s export is not applicable for ipv%d routes' %
                  ( bgpTokens.vpnNlriTypeKeyword( vpnNlriType ), ipVersion ) )
            return
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      coll = { 'vrf': config.routeTargetExportFilterDisabled,
               'vrfAf': config.routeTargetExportFilterDisabledAf }[ collectionType ]
      if value == 'isInvalid':
         del coll[ vpnNlriType ]
      else:
         coll[ vpnNlriType ] = value

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
      NeighborRouteTargetExportFilterDisabledCmd._setFilterDisabled(
         mode, peer, vpnNlriType, 'isFalse' )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
      NeighborRouteTargetExportFilterDisabledCmd._setFilterDisabled(
         mode, peer, vpnNlriType,
         'isTrue' if noOrDefault == NoOrDefault.NO else 'isInvalid' )

for m in [ RouterBgpVrfSharedModelete, RouterBgpAfSharedVrfModelet ]:
   m.addCommandClass( NeighborRouteTargetExportFilterDisabledCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor <peer|peer-group> next-hop labeled-unicast originate
#    [ disabled ]
# in address-family ipv4|ipv6 mode
#-------------------------------------------------------------------------------
class NeighborNexthopLuOriginateCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER next-hop labeled-unicast originate [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop labeled-unicast originate ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'labeled-unicast': bgpTokens.lu,
         'originate' : 'Originate BGP LU route for peer next hop'
   } )

   @staticmethod
   def _setNeighborNexthopLuOriginate( mode, peer ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      addrFamily = mode.addrFamily
      afAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get( addrFamily )
      setattr( config, afAttr, True )
      setattr( config, afAttr + 'Present', True )

   @staticmethod
   def _noNeighborNexthopLuOriginate( mode, peer, noOrDefault ):
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, create=( noOrDefault == NoOrDefault.NO ),
                                  vrfName=mode.vrfName )
      if config:
         addrFamily = mode.addrFamily
         afAttr = BgpLib.peerConfigAttrsAfMap[ 'nexthopLuOriginate' ].get(
                     addrFamily )
         setattr( config, afAttr, False )
         setattr( config, afAttr + 'Present', noOrDefault != NoOrDefault.DEFAULT )
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      NeighborNexthopLuOriginateCmd._setNeighborNexthopLuOriginate(
         mode, args[ 'PEER' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      NeighborNexthopLuOriginateCmd._noNeighborNexthopLuOriginate(
         mode, args[ 'PEER' ], noOrDefault )

if BgpCommonToggleLib.toggleArBgpNexthopLuOriginateEnabled():
   RouterBgpAfIpUniSharedModelet.addCommandClass( NeighborNexthopLuOriginateCmd )

#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR" command,
# in "config-bgp" mode
#-----------------------------------------------------------------------------------
class NoNeighborCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'neighbor PEER'
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4V6PeerKeyCliExpression,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], create=False,
                                  vrfName=mode.vrfName )
      if config:
         del configForVrf( mode.vrfName ).neighborConfig[ config.key ]

RouterBgpSharedModelet.addCommandClass( NoNeighborCmd )

#-----------------------------------------------------------------------------
# Deprecated version of the command:
# "[no|default] neighbor GROUP | ADDR soft-reconfiguration inbound"
# in "config-bgp" mode
# "neighbor GROUP | ADDR soft-reconfiguration inbound all in "config-bgp"
# mode
#
# New Version of the command:
# "[no|default] neighbor GROUP | ADDR rib-in pre-policy retain"
# in "config-bgp" mode
# "neighbor <addr|peer-group> rib-in pre-policy retain all in "config-bgp"
# mode
#-----------------------------------------------------------------------------
class SetNeighborSoftReconfigInboundCmd( BgpNEDCmdBaseClass ):
   syntax = (
      'neighbor PEER '
      '( ( rib-in pre-policy retain ) | ( soft-reconfiguration inbound ) ) '
      '[ all | disabled ]'
   )
   noOrDefaultSyntax = (
      'neighbor PEER '
      '( ( rib-in pre-policy retain ) | ( soft-reconfiguration inbound ) ) ...'
   )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'rib-in': bgpTokens.ribIn,
      'pre-policy': bgpTokens.prePolicy,
      'retain': bgpTokens.retain,
      'soft-reconfiguration': bgpTokens.softReconfigDeprecated,
      'inbound': bgpTokens.inbound,
      'all': bgpTokens.softReconfigAll,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      allRoutes = 'all' in args
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      if allRoutes:
         config.softReconfigInbound = SoftReconfigInboundStateEnum.sciAll
         config.softReconfigInboundPresent = True
      else:
         config.softReconfigInbound = SoftReconfigInboundStateEnum.sciNormal
         # Setting only needs to be marked present if this is a peer-group peer
         # because it may override the peer-group setting.
         config.softReconfigInboundPresent = config.isPeerGroupPeer
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer,
                                  create=( noOrDefault != NoOrDefault.DEFAULT ),
                                  vrfName=mode.vrfName )
      if config:
         if noOrDefault == NoOrDefault.DEFAULT:
            config.softReconfigInbound = config.softReconfigInboundDefault
            config.softReconfigInboundPresent = False
         else:
            config.softReconfigInbound = SoftReconfigInboundStateEnum.sciNone
            config.softReconfigInboundPresent = True
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( SetNeighborSoftReconfigInboundCmd )

#---------------------------------------------------------------------------------
# [ no | default ] neighbor PEER default-route-target [ only | disabled ]
#---------------------------------------------------------------------------------
class RtMembershipNeighborDefaultRt( BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-route-target [only | disabled]'
   # Do not accept trailing garbage, as this command is a prefix to
   # the "encoding" command below. If we accept trailing garbage, then
   # a no/default command with a partial "encoding" keyword will
   # match.
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'PEER': PeerCliExpression,
        'default-route-target': bgpTokens.defaultRouteTarget,
        'only': 'Only advertise default route target prefix' } )

   @staticmethod
   def _handleNormal( mode, args ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, mode.vrfName )
      config.rtMembershipDefaultRouteTarget = (
               'rtMembershipDefaultRouteTargetEnabledWithOnly' if ( 'only' in args )
               else 'rtMembershipDefaultRouteTargetEnabled' )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      if noOrDefault == NoOrDefault.DEFAULT:
         config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
         if config:
            config.rtMembershipDefaultRouteTarget = \
               'rtMembershipDefaultRouteTargetUnspecified'
            delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
      else:
         config = bgpNeighborConfig( peer, mode.vrfName )
         config.rtMembershipDefaultRouteTarget = \
            'rtMembershipDefaultRouteTargetDisabled'

RouterBgpAfRtMembershipModelet.addCommandClass(
      RtMembershipNeighborDefaultRt )

#---------------------------------------------------------------------------------
# [ no | default ] neighbor PEER default-route-target encoding origin-as omit
#---------------------------------------------------------------------------------
class RtMembershipNeighborDefaultRtEncoding( BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-route-target encoding origin-as omit [disabled]'
   noOrDefaultSyntax = 'neighbor PEER default-route-target encoding ...'
   data = BgpCmdBaseClass._createSyntaxData(
      { 'neighbor': bgpTokens.neighbor,
        'PEER': PeerCliExpression,
        'default-route-target': bgpTokens.defaultRouteTarget,
        'encoding': 'advertised default route target NLRI encoding',
        'origin-as': 'advertised NLRI origin-AS encoding',
        'omit': 'advertise NLRI with origin-AS unspecified for compatibility' } )

   @staticmethod
   def _handleNormal( mode, args ):
      config = bgpNeighborConfig( args[ 'PEER' ], mode.vrfName )
      config.rtMembershipDefaultRouteTargetEncoding = \
         'rtMembershipDefaultRouteTargetEncodingOriginAsOmit'

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      peer = args[ 'PEER' ]
      if noOrDefault == NoOrDefault.DEFAULT:
         config = bgpNeighborConfig( peer, vrfName=mode.vrfName, create=False )
         if config:
            config.rtMembershipDefaultRouteTargetEncoding = \
               'rtMembershipDefaultRouteTargetEncodingUnspecified'
            delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
      else:
         config = bgpNeighborConfig( peer, mode.vrfName )
         config.rtMembershipDefaultRouteTargetEncoding = \
            'rtMembershipDefaultRouteTargetEncodingOriginAsSet'

RouterBgpAfRtMembershipModelet.addCommandClass(
   RtMembershipNeighborDefaultRtEncoding )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global bgpVrfConfigDir

   bgpConfig = ConfigMount.mount( entityManager,
                                  'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   bgpVrfConfigDir = ConfigMount.mount( entityManager,
                                        'routing/bgp/vrf/config',
                                        'Routing::Bgp::VrfConfigDir', 'w' )
