#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import re
from Toggles import (
   BgpCommonToggleLib,
   IpRibLibToggleLib,
   RcfLibToggleLib,
   RouteMapToggleLib,
   RoutingLibToggleLib,
)
from CliParser import (
      AlreadyHandledError,
      guardNotThisPlatform,
)
from CliCommand import (
      CliCommandClass,
      isDefaultCmd,
      Node,
      SetEnumMatcher,
      singleNode,
)
from CliPlugin.RoutingBgpCli import (
      addPathSendConfig,
      afModeExtensionHook,
      bgpNeighborConfig,
      BgpCmdBaseClass,
      BgpNEDCmdBaseClass,
      configForVrf,
      configModeCmdForAf,
      conflictingRouteInstallMapConfigs,
      conflictingRouteMapConfigs,
      conflictingRouteTargetConfigs,
      deleteBgpDefaultVrfConfig,
      deleteRouterBgpMacVrfHook,
      deleteRouterBgpUcmpVrfHook,
      deleteRouterBgpVrfHook,
      getBgpRouteDistinguisherInput,
      getConflictingAttrsForAf,
      getCurrentBgpAs,
      getExplicitDefaultTristate,
      getExplicitDefaultValue,
      getVpnTypeAttr,
      haveConflictingRedistInternal,
      haveConflictingRedistribute,
      peergroupNameMatcher,
      redistributeConfig,
      RouterBgpAfIpMcastSharedModelet,
      RouterBgpAfIpUniAndMcastSharedModelet,
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfIpv6Modelet,
      RouterBgpAfLabelSharedModelet,
      RouterBgpAfLinkStateModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfSharedVrfModelet,
      RouterBgpAfSrTeSharedModelet,
      RouterBgpBaseAfEvpnMode,
      RouterBgpAfEvpnModelet,
      RouterBgpBaseAfIpMulticastMode,
      RouterBgpBaseAfIpv6MulticastMode,
      RouterBgpBaseAfIpUniMode,
      RouterBgpBaseAfIpv6UniMode,
      RouterBgpBaseAfLabelMode,
      RouterBgpBaseAfLinkStateMode,
      RouterBgpBaseAfSrTeMode,
      RouterBgpBaseAfDpsMode,
      RouterBgpBaseMode,
      RouterBgpDefaultVrfAfMode,
      RouterBgpDefaultVrfMode,
      RouterBgpSharedModelet,
      RouterBgpVrfAfIpMulticastMode,
      RouterBgpVrfAfIpMode,
      RouterBgpVrfAfIp6Mode,
      RouterBgpVrfAfIpv6Modelet,
      RouterBgpVrfAfIpv6MulticastMode,
      RouterBgpVrfAfModelet,
      RouterBgpVrfMode,
      RouterBgpVrfSharedModelete,
      routingSupportedGuard,
      setServiceAclCommon,
      switchRedistInternalConfigToAfMode,
      switchRedistInternalConfigToAllAfMode,
      ucmpConfigForVrf,
      validateRouteTargetCommand,
      vpnTypeTokenAllowedInMode,
)
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RouteMapCli import mapNameMatcher, RtSooExtCommCliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
import CliPlugin.AclCli as AclCli
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
from CliPlugin.IraIpCli import warnIfRoutingDisabled
import CliPlugin.IraIpCli as IraIpCli
import CliPlugin.RcfCliLib as RcfCliLib
from CliPlugin.IpRibLib import (
   ResolutionRibsExpr,
   ResolutionRibsHiddenUnicastExpr,
   ResolutionRibsTunnelExpr,
   getResolutionRibProfileConfig,
   ipRibsEnumMatcher
)
import CliToken.RoutingBgp as bgpTokens
import CliToken.Clear as Clear
from CliToken.Router import routerMatcherForConfig as routerMatcher
from CliToken import IpRibLibCliTokens
from BgpLib import (
      addressFamilies,
      bgpConfigAttrsAfMap,
      cleanupAsnConfig,
      haveConflictingGracefulRestart,
      isGracefulRestartGlobalOrAfConfigured,
      NoOrDefault,
      peerConfigAttrsAfMap,
      PeerConfigKey,
      routeTargetToExtCommU64Value,
      vpnAfTypeMapInv,
)
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
from IpLibTypes import ProtocolAgentModelType
import ConfigMount
import LazyMount
from CliMatcher import (
      DynamicIntegerMatcher,
      DynamicKeywordMatcher,
      DynamicNameMatcher,
      EnumMatcher,
      IntegerMatcher,
)
import Tac
import Arnet
import BasicCliModes
from ArnetLib import asnStrToNum, bgpFormatAsn, formatRd
from RouteMapLib import isAsdotConfigured
import BasicCli
import BasicCliUtil
from TypeFuture import TacLazyType

AggregateAddressType = TacLazyType( 'Routing::Bgp::AggregateAddress' )
AsnNotation = TacLazyType( 'Routing::AsnNotation' )
AsNumKey = TacLazyType( "Arnet::AutonomousSystem" )
AutoAggregateDomainType = TacLazyType( 'Routing::Bgp::AutoAggregateDomain' )
ExtendedNextHopCapabilityEnum = TacLazyType(
   "Routing::Bgp::ExtendedNextHopCapability" )
PolicyActionEnum = TacLazyType( "Routing::Bgp::PolicyActionType" )
RedistInternalStateEnum = TacLazyType( "Routing::Bgp::RedistInternalState" )
ReflectedRouteAttrStateEnum = TacLazyType( "Routing::Bgp::ReflectedRouteAttrState" )

U32_MAX_VALUE = 0xFFFFFFFF

routingHardwareStatus = None
bgpConfig = None
rdAutoInputDir = None
rdConfigInputDir = None
asnConfig = None
aclStatus = None
aclCheckpoint = None
ucmpVrfConfigDir = None
bgpVrfConfigDir = None
lsImportConfig = None

def ucmpSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.ucmpSupported:
      return None
   else:
      return guardNotThisPlatform

bgpNode = Node( matcher=bgpTokens.bgp, guard=routingSupportedGuard )
ucmpNode = Node( matcher=bgpTokens.ucmp, guard=ucmpSupportedGuard )

#-----------------------------------------------------------------------------
# "[no|default] local-as AS_NUM"
#-----------------------------------------------------------------------------
class SetLocalAsCmd( BgpCmdBaseClass ):
   syntax = "local-as AS_NUM"
   noOrDefaultSyntax = "local-as [ AS_NUM ]"
   data = BgpCmdBaseClass._createSyntaxData( {
         'local-as': bgpTokens.localAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asNumber = args[ 'AS_NUM' ]

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      # AS_NUM is optional in the 'no' form of the command, but if it is provided it
      # must match the currently configured AS_NUM.
      asNumber = args.get( 'AS_NUM' )
      if not asNumber or asNumber == config.asNumber:
         config.asNumber = 0
      elif config.asNumber:
         mode.addError( "local AS number %s does not match configuration (%s)" %
                        ( asNumber, config.asNumber ) )

RouterBgpVrfMode.addCommandClass( SetLocalAsCmd )

#-------------------------------------------------------------------------------
# "bgp fec skip in-place update event { route-changes | drain-undrain | all }"
# "{ no|default } bgp fec skip in-place update"
#
# "bgp fec in-place update event drain-undrain"
# "{ no|default } bgp fec in-place update event drain-undrain"
#
# "bgp fec in-place update event peer-init"
# "{ no|default } bgp fec in-place update event peer-init"
#
# "bgp fec in-place update timeout { <iar-timeout> | disabled }"
# "{ no|default } bgp fec in-place update timeout"
#
# commands, in "router-bgp" mode.
#-------------------------------------------------------------------------------

class SetFecSkipIarCmd( CliCommandClass ):
   syntax = 'bgp fec skip in-place update event IAR_EVENT'
   noOrDefaultSyntax = 'bgp fec skip in-place update ...'
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'skip': bgpTokens.skip,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'IAR_EVENT': bgpTokens.IarEventTypeExpression,
   }

   @staticmethod
   def _setBgpFecSkipIar( mode, eventType ):
      config = configForVrf( mode.vrfName )
      if eventType == 'route-changes':
         config.fecSkipIarDivergenceDetection = 'isTrue'
         config.fecSkipIarTracking = 'isInvalid'
      elif eventType == 'drain-undrain':
         config.fecSkipIarDivergenceDetection = 'isInvalid'
         config.fecSkipIarTracking = 'isInvalid'
      elif eventType == 'all':
         mode.addWarning( 'To make this command effective, clear all the '
            'routes by issuing "clear ip bgp *"' )
         config.fecSkipIarDivergenceDetection = 'isInvalid'
         config.fecSkipIarTracking = 'isTrue'
      else:
         assert False, 'Invalid token %s' % eventType

   @staticmethod
   def _noBgpFecSkipIar( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      if config.fecSkipIarTracking == 'isTrue':
         mode.addWarning( 'To make this command effective, clear all the '
            'routes by issuing "clear ip bgp *"' )
      if noOrDefault == NoOrDefault.DEFAULT:
         config.fecSkipIarDivergenceDetection = 'isInvalid'
         config.fecSkipIarTracking = 'isInvalid'
      else:
         config.fecSkipIarDivergenceDetection = 'isFalse'
         config.fecSkipIarTracking = 'isFalse'

   @staticmethod
   def handler( mode, args ):
      SetFecSkipIarCmd._setBgpFecSkipIar( mode, eventType=args.get( 'IAR_EVENT' ) )

   @staticmethod
   def noHandler( mode, args ):
      SetFecSkipIarCmd._noBgpFecSkipIar( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetFecSkipIarCmd._noBgpFecSkipIar( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( SetFecSkipIarCmd )

class SetFecDrainUndrainIarCmd( CliCommandClass ):
   syntax = 'bgp fec in-place update event drain-undrain'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'drain-undrain': bgpTokens.drainUndrain,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarDrainUndrain = 'isFalse'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarDrainUndrain = 'isTrue'

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarDrainUndrain = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetFecDrainUndrainIarCmd )

class SetFecPeerInitIarCmd( CliCommandClass ):
   syntax = 'bgp fec in-place update event peer-init'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'peer-init': bgpTokens.peerInit,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarPeerInit = 'isFalse'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarPeerInit = 'isTrue'

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecSkipIarPeerInit = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetFecPeerInitIarCmd )

class SetIarTimeoutCmd( BgpCmdBaseClass ):
   syntax = 'bgp fec in-place update timeout ( TIMEOUT | disabled )'
   noOrDefaultSyntax = 'bgp fec in-place update timeout ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'bgp': bgpNode,
         'fec': bgpTokens.fec,
         'in-place': bgpTokens.inPlace,
         'update': bgpTokens.iarUpdate,
         'timeout': bgpTokens.iarTimeout,
         'TIMEOUT': bgpTokens.numSeconds3600RangeMatcher,
   } )

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fecIarTimeout = args.get( 'TIMEOUT' )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      if noOrDefault == NoOrDefault.NO:
         # to disable the timeout, set to 0
         config.fecIarTimeout = config.fecIarTimeoutDisabled
      else:
         config.fecIarTimeout = config.fecIarTimeoutDefault

RouterBgpSharedModelet.addCommandClass( SetIarTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] route-target export { { evpn { ipv4 | ipv6 } } |
#                                      vpn-ipv4 | vpn-ipv6 } filter disabled
#-------------------------------------------------------------------------------

class RouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
   syntax = 'route-target export VPN_NLRI_TYPE filter disabled'
   noOrDefaultSyntax = 'route-target export VPN_NLRI_TYPE filter ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'route-target': bgpTokens.routeTarget,
      'export': bgpTokens.export,
      'VPN_NLRI_TYPE': bgpTokens.VpnNlriTypeExpr,
      'filter': 'Set route-target export filtering behavior',
   } )

   @staticmethod
   def _setFilterDisabled( dummyModeForStubCommandHandler, vpnNlriType, value ):
      config = configForVrf( DEFAULT_VRF )
      if value:
         config.routeTargetExportFilterDisabled[ vpnNlriType ] = True
      else:
         del config.routeTargetExportFilterDisabled[ vpnNlriType ]

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      vpnNlriType = args[ 'VPN_NLRI_TYPE' ]
      RouteTargetExportFilterDisabledCmd._setFilterDisabled(
         None, vpnNlriType, noOrDefault == NoOrDefault.NO )

   @staticmethod
   def _handleNormal( mode, args ):
      # The BgpCmdBaseClass will always call _handleNoOrDefault with NoOrDefault.NO
      # as the 'disabled' keyword in the syntax of this command is non-optional.
      # However, to make the linter happy with this class inheritance, we need to
      # have a stub for _handleNormal.
      pass

RouterBgpBaseMode.addCommandClass( RouteTargetExportFilterDisabledCmd )

#-------------------------------------------------------------------------------
# [ (no|default) ] neighbor default send-community
# [ ( [ extended ] [ large ] [ LBW ] [ standard ] )  | disabled ]
#     under 'bgp' mode'
#-------------------------------------------------------------------------------

class SetNeighborDefaultSendCommunity( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor default send-community '
              '[ ( [ standard ] [ extended ] [ large ] ) | disabled ]' )
   noOrDefaultSyntax = ( 'neighbor default send-community '
                         '[ standard ] [ extended ] [ large ] ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( { 'neighbor': bgpTokens.neighbor,
            'default': bgpTokens.neighborDefaultGlobal,
            'send-community': bgpTokens.sendCommunity,
            'standard': bgpTokens.standardCommunity,
            'extended': bgpTokens.extendedCommunity,
            'large': bgpTokens.largeCommunity,
            } )

   @staticmethod
   def _setNeighborDefaultSendCommunity( mode, standard=False, large=False,
                                         extended=False ):
      config = configForVrf( mode.vrfName )
      config.sendCommunityPresent = True
      config.sendCommunity = not ( standard or large or extended )
      config.sendStandardCommunity = standard or config.sendStandardCommunityDefault
      config.sendLargeCommunity = large or config.sendLargeCommunityDefault
      config.sendExtendedCommunity = extended or config.sendExtendedCommunityDefault

   @staticmethod
   def _noNeighborDefaultSendCommunity( mode, isDefault, standard=False,
                                        large=False, extended=False ):
      config = configForVrf( mode.vrfName )
      sendCommunityOnly = not ( standard or large or extended )
      if isDefault:
         if sendCommunityOnly:
            config.sendCommunity = config.sendCommunityDefault
         if standard or sendCommunityOnly:
            config.sendStandardCommunity = config.sendStandardCommunityDefault
         if large or sendCommunityOnly:
            config.sendLargeCommunity = config.sendLargeCommunityDefault
         if extended or sendCommunityOnly:
            config.sendExtendedCommunity = config.sendExtendedCommunityDefault
         config.sendCommunityPresent = ( config.sendCommunity or
                                         config.sendStandardCommunity or
                                         config.sendLargeCommunity or
                                         config.sendExtendedCommunity )
      else:
         if sendCommunityOnly:
            config.sendCommunity = False
         if standard or sendCommunityOnly:
            config.sendStandardCommunity = False
         if large or sendCommunityOnly:
            config.sendLargeCommunity = False
         if extended or sendCommunityOnly:
            config.sendExtendedCommunity = False
         config.sendCommunityPresent = True

   @staticmethod
   def _handleNormal( mode, args ):
      SetNeighborDefaultSendCommunity._setNeighborDefaultSendCommunity(
         mode, standard='standard' in args,
         large='large' in args,
         extended='extended' in args )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNeighborDefaultSendCommunity._noNeighborDefaultSendCommunity(
         mode, noOrDefault == NoOrDefault.DEFAULT,
         standard='standard' in args,
         large='large' in args,
         extended='extended' in args )

RouterBgpSharedModelet.addCommandClass( SetNeighborDefaultSendCommunity )

#-------------------------------------------------------------------------------
# "[no|default] router bgp <as>" command, in "config" mode.
#-------------------------------------------------------------------------------

class RouterBgpCmd( CliCommandClass ):
   syntax = 'router bgp ( ASN | CURR_ASN )'
   noOrDefaultSyntax = 'router bgp ...'
   data = {
         'router': routerMatcher,
         'bgp': bgpNode,
         'ASN': bgpTokens.AsNumCliExpr,
         'CURR_ASN': DynamicKeywordMatcher( getCurrentBgpAs ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      currAsn = args.get( 'CURR_ASN' )
      if currAsn:
         args[ 'CURR_ASN' ] = asnStrToNum( currAsn )

      # The expression used for ASN populates 'AS_NUM'.
      args[ 'AS_NUMBER' ] = args.get( 'AS_NUM' ) or args.get( 'CURR_ASN' )

   @staticmethod
   def handler( mode, args ):
      warnIfRoutingDisabled( mode )
      asNumber = args[ 'AS_NUMBER' ]
      if bgpConfig.asNumber != 0 and bgpConfig.asNumber != asNumber:
         mode.addError( "BGP is already running with AS number %s"
                        % bgpFormatAsn( bgpConfig.asNumber ) )
         return
      if asNumber == 23456:
         mode.addError( "Configuring ASN %d for BGP is disallowed" % 23456 )
         return
      # selective rollback may leave behind stale ucmp config
      if bgpConfig.asNumber == 0:
         for v in ucmpVrfConfigDir.vrfConfig:
            del ucmpVrfConfigDir.vrfConfig[ v ]
      bgpConfig.asNumber = asNumber
      childMode = mode.childMode( RouterBgpBaseMode, asNumber=asNumber )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfNames = [ DEFAULT_VRF ] + bgpVrfConfigDir.vrfConfig.keys()

      # cleanup all bgp vrf service ACL config
      childMode = mode.childMode( RouterBgpBaseMode, asNumber=bgpConfig.asNumber )
      for v in vrfNames:
         vrfMode = RouterBgpDefaultVrfMode if v == DEFAULT_VRF else RouterBgpVrfMode
         subMode = childMode.childMode( vrfMode, vrfName=v )
         for aclType in [ 'ip', 'ipv6' ]:
            setServiceAclCommon( subMode, aclType, no=True )

      # delete all the vrf configs
      for v in vrfNames:
         for hook in deleteRouterBgpVrfHook.extensions():
            hook( v )

      # cleanup asnConfig
      cleanupAsnConfig( asnConfig )

      # cleanup mac vrf config
      for hook in deleteRouterBgpMacVrfHook.extensions():
         hook()

      # delete all the vrf ucmp configs
      for v in ucmpVrfConfigDir.vrfConfig:
         for hook in deleteRouterBgpUcmpVrfHook.extensions():
            hook( v )

      # cleanup ucmp vrf related config
      for hook in deleteRouterBgpUcmpVrfHook.extensions():
         hook( DEFAULT_VRF )

BasicCli.GlobalConfigMode.addCommandClass( RouterBgpCmd )

#-------------------------------------------------------------------------------
# "[no|default] service routing configuration bgp no-equals-default" command,
# in "config" mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] command clear all disabled" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ClearAllDisabledCmd( CliCommandClass ):
   syntax = 'command clear all disabled'
   noOrDefaultSyntax = syntax
   data = {
      'command': 'Configure command options',
      'clear': 'BGP clear command',
      'all': 'Clear all BGP sessions',
      'disabled': 'Disallow clearing all BGP sessions'
   }

   @staticmethod
   def handler( mode, args ):
      bgpConfig.clearAllDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bgpConfig.clearAllDisabled = False

RouterBgpBaseMode.addCommandClass( ClearAllDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] update wait-install" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
def fibAckSupportedGuard( mode, token ):
   if routingHardwareStatus.fibAckSupported:
      return None
   return guardNotThisPlatform

class SetFibSyncBatchSizeCmd( CliCommandClass ):
   syntax = 'update wait-install [ batch-size BATCH_SIZE ]'
   noOrDefaultSyntax = 'update wait-install ...'
   data = { 'update': bgpTokens.update,
            'wait-install': Node( bgpTokens.waitInstall,
                                  guard=fibAckSupportedGuard ),
            'batch-size': bgpTokens.batchSize,
            'BATCH_SIZE':  IntegerMatcher( 0, 10000, helpdesc='batch size' ),
   }

   @staticmethod
   def handler( mode, args ):
      batchSize = args.get( 'BATCH_SIZE' )
      config = configForVrf( mode.vrfName )
      fibSyncBatchSize = int( batchSize ) if batchSize else 0
      config.fibSync = Tac.Value( 'Routing::Bgp::FibSync', True,
                                  fibSyncBatchSize )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.fibSync = config.fibSyncDefault

RouterBgpSharedModelet.addCommandClass( SetFibSyncBatchSizeCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp labeled-unicast rib { ip | tunnel }" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpLuRibCmd( CliCommandClass ):
   _ipSyntax = '( ip [ IP_RM IP_RM_NAME ] )'
   _tunnelSyntax = '( tunnel [ TUNNEL_RM TUNNEL_RM_NAME ] )'
   syntax = 'bgp labeled-unicast rib ( {ip} [ {tunnel} ] ) | ' \
            '( {tunnel} [ {ip} ] )'.format( ip=_ipSyntax, tunnel=_tunnelSyntax )
   noOrDefaultSyntax = 'bgp labeled-unicast rib'
   data = {
         'bgp': bgpTokens.bgp,
         'labeled-unicast': bgpTokens.lu,
         'rib': bgpTokens.rib,
         'ip': bgpTokens.ip,
         'tunnel': bgpTokens.tunnel,
         'IP_RM': bgpTokens.ribRouteMap,
         'TUNNEL_RM': bgpTokens.ribRouteMap,
         'IP_RM_NAME': mapNameMatcher,
         'TUNNEL_RM_NAME': mapNameMatcher
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      enumVal = 'tunnel'
      if 'ip' in args and 'tunnel' in args:
         enumVal = 'ipAndTunnel'
      elif 'ip' in args:
         enumVal = 'ip'
      args[ 'rib' ] = getattr( Tac.Type( 'Routing::Bgp::BgpLuRibTypes' ), enumVal )

   @staticmethod
   def handler( mode, args ):
      rib = args.get( 'rib' )
      tunnelRmName = args.get( 'TUNNEL_RM_NAME', '' )
      ipRmName = args.get( 'IP_RM_NAME', '' )
      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent and \
         not ( mode.session.inConfigSession() or mode.session.startupConfig() ):
         # in ribd mode interactive session, setting route-map filter or
         # use ipAndTunnel type is not supported.
         if rib == 'ipAndTunnel':
            mode.addError( "Labeled-unicast routes installed in both Tunnel"
                           " RIB and IP RIB is only supported on multi-agent"
                           " mode" )
            return
         if tunnelRmName or ipRmName:
            mode.addError( "Filtering the labeled-unicast RIB by route-map"
                           " is only supported on multi-agent mode" )
            return
      bgpLuRib = Tac.Value( "Routing::Bgp::BgpLuRib", rib, tunnelRmName, ipRmName )
      configForVrf( mode.vrfName ).bgpLuRib = bgpLuRib

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      bgpLuRib = Tac.Value( "Routing::Bgp::BgpLuRib", config.bgpLuRib.ribDefault,
                            config.bgpLuRib.rmNameDefault,
                            config.bgpLuRib.rmNameDefault )
      config.bgpLuRib = bgpLuRib

RouterBgpSharedModelet.addCommandClass( BgpLuRibCmd )

#-------------------------------------------------------------------------------
# "[no|default] shutdown [reason MESSAGE]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ShutdownCmd( CliCommandClass ):
   syntax = 'shutdown [ reason MESSAGE ]'
   noOrDefaultSyntax = 'shutdown ...'
   data = { 'shutdown': 'Shut down BGP',
            'reason': bgpTokens.reason,
            'MESSAGE': bgpTokens.messageMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if not BasicCliUtil.confirm( mode,
                                   "You are attempting to shutdown BGP. "
                                   "Are you sure you want to shutdown? [confirm]" ):
         return
      config = configForVrf( mode.vrfName )
      config.shutdownMsg = args.get( 'MESSAGE', config.shutdownMsgDefault )
      config.shutdown = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.shutdownMsg = config.shutdownMsgDefault
      config.shutdown = False

RouterBgpSharedModelet.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp log-neighbor-changes" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class LogNeighborChangesCmd( CliCommandClass ):
   syntax = 'bgp log-neighbor-changes'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpTokens.bgp,
      'log-neighbor-changes': bgpTokens.logNeighborChanges,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.logNeighborChanges = getExplicitDefaultTristate( mode,
                                                              'logNeighborChanges' )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.logNeighborChanges = 'isFalse'

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.logNeighborChanges = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( LogNeighborChangesCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp client-to-client reflection" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ClientToClientReflectionCmd( CliCommandClass ):
   syntax = 'bgp client-to-client reflection'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'client-to-client': 'client to client configuration',
      'reflection': 'Enable client to client route reflection',
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.rrDisabled = config.rrDisabledDefault

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.rrDisabled = True

   defaultHandler = handler

RouterBgpSharedModelet.addCommandClass( ClientToClientReflectionCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp cluster-id <cluster-id> " command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetClusterIdCmd( CliCommandClass ):
   syntax = 'bgp cluster-id CLUSTER_ID'
   noOrDefaultSyntax = 'bgp cluster-id ...'
   data = {
      'bgp': bgpNode,
      'cluster-id': bgpTokens.cluster,
      'CLUSTER_ID': IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      ipAddr = args[ 'CLUSTER_ID' ]
      config = configForVrf( mode.vrfName )
      config.bgpClusterId = ipAddr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.bgpClusterId = config.bgpClusterIdDefault

RouterBgpSharedModelet.addCommandClass( SetClusterIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp aggregate-route community inheritance loose" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpAggregateCommunityInheritanceLooseCmd( CliCommandClass ):
   syntax = 'bgp aggregate-route community inheritance loose'
   noOrDefaultSyntax = 'bgp aggregate-route ...'
   data = {
      'bgp': bgpNode,
      'aggregate-route': 'Aggregate route',
      'community': 'Aggregate route community',
      'inheritance': 'Aggregate route community generation from contributors',
      'loose': 'Ignore ATOMIC_AGGREGATE in contributors for community inheritance',
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.aggregateCommInheritLoose = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.aggregateCommInheritLoose = config.aggregateCommInheritLooseDefault

RouterBgpSharedModelet.addCommandClass( BgpAggregateCommunityInheritanceLooseCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp allowas-in [count]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAllowAsCmd( CliCommandClass ):
   syntax = 'bgp allowas-in [ COUNT ]'
   noOrDefaultSyntax = 'bgp allowas-in ...'
   data = {
      'bgp': bgpNode,
      'allowas-in': bgpTokens.allowAsIn,
      'COUNT': bgpTokens.allowAsRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.allowAs = args.get( 'COUNT', config.allowAsEnabledDefault )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.allowAs = config.allowAsDefault

RouterBgpSharedModelet.addCommandClass( SetAllowAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp redistribute-internal " command,
# in "router-bgp", "router-bgp-af" mode
#--------------------------------------------------------------------------------
class BgpRedistInternalCmd( CliCommandClass ):
   syntax = 'bgp redistribute-internal'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'redistribute-internal': bgpTokens.bgpRedistInternal,
   }

   @staticmethod
   def _setBgpRedistInternalCommon( mode, no=False ):
      # Prevent configuration of bgp redistribute-internal in both
      # router-bgp and router-bgp-af levels.
      config = configForVrf( mode.vrfName )
      addrFamily = mode.addrFamily
      if haveConflictingRedistInternal( mode, config, addrFamily=addrFamily ):
         return

      attr = bgpConfigAttrsAfMap[ 'bgpRedistInternal' ].get( addrFamily )
      if addrFamily == 'all':
         switchRedistInternalConfigToAllAfMode( config )
      else:
         switchRedistInternalConfigToAfMode( config )

      if no:
         val = RedistInternalStateEnum.disableRedistInt
      else:
         val = RedistInternalStateEnum.enableRedistInt

      setattr( config, attr, val )

   @staticmethod
   def handler( mode, args ):
      BgpRedistInternalCmd._setBgpRedistInternalCommon( mode )

   @staticmethod
   def noHandler( mode, args ):
      BgpRedistInternalCmd._setBgpRedistInternalCommon( mode, no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      BgpRedistInternalCmd._setBgpRedistInternalCommon( mode )

for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniSharedModelet ]:
   m.addCommandClass( BgpRedistInternalCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers bgp <keepalive> <hold>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpSetTimerCmd( CliCommandClass ):
   syntax = 'timers bgp KEEPALIVE ( HOLD_TIME | HOLD_FOREVER )'
   noOrDefaultSyntax = 'timers bgp ...'
   data = {
      'timers': bgpTokens.timers,
      'bgp': bgpTokens.bgpTimers,
      'KEEPALIVE': bgpTokens.keepaliveTimeRangeMatcher,
      'HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      'HOLD_FOREVER': bgpTokens.holdForever,
   }

   @staticmethod
   def handler( mode, args ):
      keepalive = args[ 'KEEPALIVE' ]
      hold = args.get( 'HOLD_TIME', 0 )
      if hold < keepalive:
         mode.addError( 'keepalive time cannot be greater than hold time' )
         return
      if hold != 0 and keepalive == 0:
         mode.addError(
            'keepalive time cannot be zero when hold time is not zero' )
         return
      config = configForVrf( mode.vrfName )
      config.keepaliveTime = getExplicitDefaultValue(
         mode, 'keepaliveTime', keepalive )
      config.holdTime = getExplicitDefaultValue(
         mode, 'holdTime', hold )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      # The "no timers bgp" command operates differently from the
      # "default timers bgp..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if mode.vrfName == DEFAULT_VRF:
         config.keepaliveTime = config.keepaliveTimeInvalid
         config.holdTime = config.holdTimeInvalid
      else:
         config.keepaliveTime = config.keepaliveTimeDefault
         config.holdTime = config.holdTimeDefault

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.keepaliveTime = config.keepaliveTimeInvalid
      config.holdTime = config.holdTimeInvalid

RouterBgpSharedModelet.addCommandClass( BgpSetTimerCmd )

#---------------------------------------------------------------------------------
# "[no|default] graceful-restart [ ( restart-time | stalepath-time ) SECONDS ]
# in "router-bgp" and "router-bgp-af"  mode.  The 'restart-time' and 'stalepath-time'
# keywords are only applicable in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetGracefulRestartCmd( CliCommandClass ):
   syntax = ( 'graceful-restart' )
   noOrDefaultSyntax = syntax
   data = {
         'graceful-restart': bgpTokens.gracefulRestart,
   }

   @staticmethod
   def _configureGlobalGracefulRestart( mode,
                                        enable=True,
                                        noOrDefault=None,
                                        addrFamily='all' ):
      # Verify that global graceful restart is not configured in both router-bgp-mode
      # and router-bgp-af mode
      config = configForVrf( mode.vrfName )
      allAfConfigured, anyAfConfigured = isGracefulRestartGlobalOrAfConfigured(
            config )
      if haveConflictingGracefulRestart( noOrDefault,
                                         None if addrFamily == 'all' else addrFamily,
                                         anyAfConfigured, allAfConfigured ):
         mode.addError( 'Graceful restart cannot be configured in both'
                                 ' router-bgp and router-bgp-af mode' )
         return

      if addrFamily == 'ipv4 multicast':
         mode.addError( 'Graceful restart is currently not supported for IPv4 '
                        'multicast mode' )
         return
      if addrFamily == 'ipv6 multicast':
         mode.addError( 'Graceful restart is currently not supported for IPv6 '
                        'multicast mode' )
         return

      triStateVal = None
      if enable:
         triStateVal = 'isTrue'
      else:
         assert noOrDefault
         triStateVal = (
               'isInvalid' if noOrDefault == NoOrDefault.DEFAULT else 'isFalse'
         )

      if addrFamily == 'all':
         if triStateVal == 'isFalse':
            triStateVal = getExplicitDefaultTristate( mode, 'gracefulRestart' )

      attr = bgpConfigAttrsAfMap[ 'gracefulRestart' ].get( addrFamily )
      setattr( config, attr, triStateVal )

   @staticmethod
   def noHandler( mode, args ):
      SetGracefulRestartCmd._configureGlobalGracefulRestart(
            mode,
            enable=False,
            noOrDefault=NoOrDefault.NO,
            addrFamily=mode.addrFamily )

   @staticmethod
   def defaultHandler( mode, args ):
      SetGracefulRestartCmd._configureGlobalGracefulRestart(
            mode,
            enable=False,
            noOrDefault=NoOrDefault.DEFAULT,
            addrFamily=mode.addrFamily )

   @staticmethod
   def handler( mode, args ):
      SetGracefulRestartCmd._configureGlobalGracefulRestart(
            mode,
            enable=True,
            noOrDefault=None,
            addrFamily=mode.addrFamily )

for m in [ RouterBgpAfSharedModelet, RouterBgpSharedModelet ]:
   m.addCommandClass( SetGracefulRestartCmd )
if BgpCommonToggleLib.toggleArBgpLuGracefulRestartEnabled():
   RouterBgpAfLabelSharedModelet.addCommandClass( SetGracefulRestartCmd )

class SetGracefulRestartTimesCmd( CliCommandClass ):
   syntax = ( 'graceful-restart ( restart-time | stalepath-time ) SECONDS' )
   noOrDefaultSyntax = 'graceful-restart ( restart-time | stalepath-time ) ...'
   data = {
      'graceful-restart': bgpTokens.gracefulRestart,
      'restart-time': bgpTokens.restartTime,
      'stalepath-time': bgpTokens.stalepathTime,
      'SECONDS': bgpTokens.secondsRangeMatcher,
      }

   # The "no graceful-restart [ restart-time | stalepath-time ]" command operates
   # differently from the "default..." command for the non-default VRF.  The
   # 'default' option is the same as unconfigured, which will cause the non-default
   # vrf to use the value configured in the default vrf.  The 'no' option explicitly
   # sets parameters to the system defaults.
   @staticmethod
   def _setGrRestartTime( mode, restartTime, noOrDefault=None ):
      config = configForVrf( mode.vrfName )
      if noOrDefault:
         if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
            config.grRestartTime = config.grRestartTimeInvalid
         else:
            config.grRestartTime = config.grRestartTimeDefault
      else:
         config.grRestartTime = getExplicitDefaultValue(
               mode,
               'grRestartTime',
               restartTime )

   @staticmethod
   def _setGrStalepathTime( mode, stalepathTime, noOrDefault=None ):
      config = configForVrf( mode.vrfName )
      if noOrDefault:
         if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
            config.grStalepathTime = config.grStalepathTimeInvalid
         else:
            config.grStalepathTime = config.grStalepathTimeDefault
      else:
         config.grStalepathTime = getExplicitDefaultValue(
               mode,
               'grStalepathTime',
               stalepathTime )

   @staticmethod
   def _noOrDefaultGracefulRestartTimes( mode,
                                         noOrDefault,
                                         restartTime=None,
                                         stalepathTime=None ):
      if restartTime:
         SetGracefulRestartTimesCmd._setGrRestartTime(
               mode,
               None,
               noOrDefault=noOrDefault )
      elif stalepathTime:
         SetGracefulRestartTimesCmd._setGrStalepathTime(
               mode,
               None,
               noOrDefault=noOrDefault )

   @staticmethod
   def noHandler( mode, args ):
      SetGracefulRestartTimesCmd._noOrDefaultGracefulRestartTimes(
            mode,
            NoOrDefault.NO,
            restartTime=args.get( 'restart-time' ),
            stalepathTime=args.get( 'stalepath-time' ) )

   @staticmethod
   def defaultHandler( mode, args ):
      SetGracefulRestartTimesCmd._noOrDefaultGracefulRestartTimes(
            mode,
            NoOrDefault.DEFAULT,
            restartTime=args.get( 'restart-time' ),
            stalepathTime=args.get( 'stalepath-time' ) )

   @staticmethod
   def handler( mode, args ):
      if args.get( 'restart-time' ):
         SetGracefulRestartTimesCmd._setGrRestartTime(
               mode,
               args.get( 'SECONDS' ) )
      elif args.get( 'stalepath-time' ):
         SetGracefulRestartTimesCmd._setGrStalepathTime(
               mode,
               args.get( 'SECONDS' ) )

RouterBgpSharedModelet.addCommandClass( SetGracefulRestartTimesCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper [ restart-time < seconds > ] [ long-lived ]"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetGracefulRestartHelperCmd( CliCommandClass ):
   syntax = 'graceful-restart-helper'
   if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
      syntax = 'graceful-restart-helper [ EXT_RESTART_TIME ]'
   if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
      syntax += ' [ long-lived ]'
   noOrDefaultSyntax = 'graceful-restart-helper'
   if ( BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled() or
        BgpCommonToggleLib.toggleArBgpLlgrEnabled() ):
      noOrDefaultSyntax = 'graceful-restart-helper ...'
   data = {
      'graceful-restart-helper': bgpTokens.grHelper,
   }
   if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
      data.update( {
         'EXT_RESTART_TIME': bgpTokens.GrHelperRestartTimeExp } )
   if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
      data.update( {
         'long-lived': bgpTokens.llgrHelper } )

   @staticmethod
   def _setGrHelper( mode, extRestartTime, llgrHelper ):
      config = configForVrf( mode.vrfName )
      config.grHelper = getExplicitDefaultTristate( mode, 'grHelper' )
      if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
         if extRestartTime:
            config.grHelperRestartTime = getExplicitDefaultValue(
                  mode, 'grHelperRestartTime', extRestartTime )
         else:
            config.grHelperRestartTime = config.grHelperRestartTimeInvalid
      if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
         if llgrHelper:
            config.llgrHelper = 'isTrue'
         else:
            config.llgrHelper = 'isInvalid'

   @staticmethod
   def _noGrHelper( mode ):
      config = configForVrf( mode.vrfName )
      config.grHelper = 'isFalse'
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
         if mode.vrfName == DEFAULT_VRF:
            config.grHelperRestartTime = config.grHelperRestartTimeInvalid
         else:
            config.grHelperRestartTime = config.grHelperRestartTimeDefault
      if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
         if mode.vrfName == DEFAULT_VRF:
            config.llgrHelper = 'isInvalid'
         else:
            config.llgrHelper = config.llgrHelperDefault

   @staticmethod
   def _defaultGrHelper( mode ):
      config = configForVrf( mode.vrfName )
      config.grHelper = 'isInvalid'
      if BgpCommonToggleLib.toggleArBgpOverrideGrRestartTimeEnabled():
         config.grHelperRestartTime = config.grHelperRestartTimeInvalid
      if BgpCommonToggleLib.toggleArBgpLlgrEnabled():
         config.llgrHelper = 'isInvalid'

   @staticmethod
   def handler( mode, args ):
      SetGracefulRestartHelperCmd._setGrHelper( mode,
                                                args.get( 'RESTART_TIME' ),
                                                ( 'long-lived' in args ) )

   @staticmethod
   def noHandler( mode, args ):
      SetGracefulRestartHelperCmd._noGrHelper( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      SetGracefulRestartHelperCmd._defaultGrHelper( mode )

RouterBgpSharedModelet.addCommandClass( SetGracefulRestartHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp enforce-first-as" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpEnforceAsFirstCmd( CliCommandClass ):
   syntax = 'bgp enforce-first-as'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'enforce-first-as': bgpTokens.enforceFirstAs,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.enforceFirstAs = getExplicitDefaultTristate(
         mode,
         'enforceFirstAs' )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.enforceFirstAs = 'isFalse'

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.enforceFirstAs = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( BgpEnforceAsFirstCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath as-path multipath-relax [match <n>]" command
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAsPathMultiPathRelaxCmd( CliCommandClass ):
   syntax = ( 'bgp bestpath as-path multipath-relax '
              '[ match ASPATH_LEN ]' )
   noOrDefaultSyntax = 'bgp bestpath as-path multipath-relax ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'as-path': bgpTokens.asPath,
         'multipath-relax': bgpTokens.multiPathRelax,
         'match': bgpTokens.asMatch,
         'ASPATH_LEN': bgpTokens.asPathLenMatcher
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathMultiPathRelax = getExplicitDefaultTristate(
            mode,
            'asPathMultiPathRelax'
      )
      config.asPathMultiPathRelaxMatch = args.get(
            'ASPATH_LEN',
            config.asPathMultiPathRelaxMatchDefault
      )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathMultiPathRelax = 'isFalse'
      config.asPathMultiPathRelaxMatch = config.asPathMultiPathRelaxMatchInvalid

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathMultiPathRelax = 'isInvalid'
      config.asPathMultiPathRelaxMatch = config.asPathMultiPathRelaxMatchDefault

RouterBgpSharedModelet.addCommandClass( SetAsPathMultiPathRelaxCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath as-path ignore" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class IgnoreAsPathCmd( CliCommandClass ):
   syntax = 'bgp bestpath as-path ignore'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'as-path': bgpTokens.asPath,
         'ignore': bgpTokens.ignore,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ignoreAsPathLen = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ignoreAsPathLen = getExplicitDefaultTristate( mode, 'ignoreAsPathLen' )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ignoreAsPathLen = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( IgnoreAsPathCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath d-path" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class BgpBestPathDomainPath( CliCommandClass ):
   syntax = 'bgp bestpath d-path'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'd-path': bgpTokens.domainPath
   }

   @staticmethod
   def handler( mode, args ):
      assert mode.vrfName == DEFAULT_VRF
      config = configForVrf( mode.vrfName )
      config.includeDomainPathLen = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      assert mode.vrfName == DEFAULT_VRF
      config = configForVrf( mode.vrfName )
      config.includeDomainPathLen = getExplicitDefaultTristate(
         mode, 'includeDomainPathLen' )

   @staticmethod
   def defaultHandler( mode, args ):
      assert mode.vrfName == DEFAULT_VRF
      config = configForVrf( mode.vrfName )
      config.includeDomainPathLen = 'isInvalid'

RouterBgpBaseMode.addCommandClass( BgpBestPathDomainPath )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath skip next-hop igp-cost" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class SetSkipNhIgpCostCmd( CliCommandClass ):
   syntax = 'bgp bestpath skip next-hop igp-cost'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'skip': bgpTokens.skipMatcher,
         'next-hop': bgpTokens.skipNextHop,
         'igp-cost': bgpTokens.skipNhIgpCost,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipNextHopIgpCost = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipNextHopIgpCost = (
            getExplicitDefaultTristate( mode, 'skipNextHopIgpCost' )
      )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipNextHopIgpCost = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetSkipNhIgpCostCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath origin-as validity" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class BgpBestPathOriginAsValidityCmd( BgpCmdBaseClass ):
   syntax = 'bgp bestpath origin-as validity [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'origin-as': bgpTokens.originAsBestPath,
      'validity': bgpTokens.originAsValidityBestPath
   } )

   @staticmethod
   def _setOriginAsValidity( mode, includeOriginAsValidity ):
      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
         mode.addWarning( "RPKI origin AS validation is only supported in "
                          "multi-agent mode" )
      config = configForVrf( mode.vrfName )
      config.includeOriginAsValidity = includeOriginAsValidity

   @staticmethod
   def _handleNormal( mode, args ):
      BgpBestPathOriginAsValidityCmd._setOriginAsValidity( mode, 'isTrue' )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      includeOriginAsValidity = \
         'isInvalid' if noOrDefault == NoOrDefault.DEFAULT else 'isFalse'
      BgpBestPathOriginAsValidityCmd._setOriginAsValidity( mode,
                                                           includeOriginAsValidity )

RouterBgpSharedModelet.addCommandClass( BgpBestPathOriginAsValidityCmd )

#--------------------------------------------------------------------------
# "bgp additional-paths install" command, in "router-bgp" mode
#--------------------------------------------------------------------------
class SetAddpathInstallCmd( CliCommandClass ):
   syntax = 'bgp additional-paths install'
   noOrDefaultSyntax = syntax
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'install': bgpTokens.addpathInstall,
   }
   if IpRibLibToggleLib.toggleBackupPathWithEcmpEnabled():
      syntax = 'bgp additional-paths install [ ecmp-primary ]'
      noOrDefaultSyntax = 'bgp additional-paths install ...'
      data = { 'bgp': bgpNode,
               'additional-paths': bgpTokens.addPath,
               'install': bgpTokens.addpathInstall,
               'ecmp-primary': bgpTokens.addpathInstallEcmpPrimary,
      }

   @staticmethod
   def handler( mode, args ):
      ecmpPrimary = ( 'ecmp-primary' in args )
      config = configForVrf( mode.vrfName )
      if mode.addrFamily == 'all':
         config.picIPv4Uni = True
         config.picEcmpPrimaryIPv4Uni = ecmpPrimary
      else:
         attr = bgpConfigAttrsAfMap[ 'pic' ].get( mode.addrFamily )
         if attr:
            setattr( config, attr, True )
         attr = bgpConfigAttrsAfMap[ 'picEcmpPrimary' ].get( mode.addrFamily )
         if attr:
            setattr( config, attr, ecmpPrimary )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      if mode.addrFamily == 'all':
         config.picIPv4Uni = False
         config.picEcmpPrimaryIPv4Uni = False
      else:
         attr = bgpConfigAttrsAfMap[ 'pic' ].get( mode.addrFamily )
         if attr:
            setattr( config, attr, False )
         attr = bgpConfigAttrsAfMap[ 'picEcmpPrimary' ].get( mode.addrFamily )
         if attr:
            setattr( config, attr, False )

for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniSharedModelet ]:
   m.addCommandClass( SetAddpathInstallCmd )

#--------------------------------------------------------------------------
# "bgp additional-paths receive" command, in "router-bgp" mode
#--------------------------------------------------------------------------
class SetAddPathReceiveCmd( CliCommandClass ):
   syntax = 'bgp additional-paths receive'
   noOrDefaultSyntax = syntax
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'receive': bgpTokens.addpathReceive,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      if mode.addrFamily == 'all':
         config.apRecv = getExplicitDefaultTristate( mode, 'apRecv' )
      else:
         attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
         if attr:
            setattr( config, attr, 'isTrue' )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, 'isFalse' )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'apRecv' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, 'isInvalid' )

for m in [ RouterBgpSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetAddPathReceiveCmd )

#--------------------------------------------------------------------------
# "bgp additional-paths send any" command in "router-bgp" mode.
#--------------------------------------------------------------------------
class SetAddpathSendCmd( CliCommandClass ):
   syntax = 'bgp additional-paths send any'
   noOrDefaultSyntax = syntax
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'send': bgpTokens.addpathSend,
            'any': bgpTokens.appAny,
   }

   @staticmethod
   def _setAddpathSend( mode, noOrDefault=None ):
      app = 'appAny'
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = configForVrf( vrfName )
      sendConfig = addPathSendConfig( config, addrFamily )

      if not noOrDefault:
         newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app, enable=True )
         sendConfig.addMember( newConfig )
      if noOrDefault == NoOrDefault.NO:
         # The following setting is to support config inheritance
         newConfig = Tac.Value( 'Routing::Bgp::AddPathSendConfig', app,
                                enable=False )
         sendConfig.addMember( newConfig )
      elif noOrDefault == NoOrDefault.DEFAULT and app in sendConfig:
         del sendConfig[ app ]

   @staticmethod
   def handler( mode, args ):
      SetAddpathSendCmd._setAddpathSend( mode )

   @staticmethod
   def noHandler( mode, args ):
      SetAddpathSendCmd._setAddpathSend(
            mode,
            noOrDefault=NoOrDefault.NO,
      )

   @staticmethod
   def defaultHandler( mode, args ):
      SetAddpathSendCmd._setAddpathSend(
            mode,
            noOrDefault=NoOrDefault.DEFAULT,
      )

for m in [ RouterBgpSharedModelet,
           RouterBgpAfIpUniSharedModelet,
           RouterBgpAfLabelSharedModelet,
           RouterBgpBaseAfEvpnMode ]:
   m.addCommandClass( SetAddpathSendCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp route install-map <rmap-name>" command
# under the "router-bgp" and under the address-family modes
#-------------------------------------------------------------------------------
class BgpRouteInstallMapCmd( BgpCmdBaseClass ):
   syntax = 'bgp route install-map MAP_NAME'
   noOrDefaultSyntax = 'bgp route install-map ...'
   data = { 'bgp': bgpNode,
            'route': bgpTokens.route,
            'install-map': bgpTokens.installMap,
            'MAP_NAME': mapNameMatcher,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      # prevent configuration from both globally and at af levels
      if conflictingRouteInstallMapConfigs( mode, args[ 'MAP_NAME' ], config,
                                            mode.addrFamily ):
         return
      attr = bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( mode.addrFamily )
      setattr( config, attr, args[ 'MAP_NAME' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'routeInstallMap' ].get( mode.addrFamily )
      defaultVal = getattr( config, 'routeInstallMapDefault' )
      setattr( config, attr, defaultVal )

RouterBgpSharedModelet.addCommandClass( BgpRouteInstallMapCmd )
RouterBgpAfIpUniSharedModelet.addCommandClass( BgpRouteInstallMapCmd )

#-------------------------------------------------------------------------------
# "[no|default] network ( PREFIX | ( ADDR mask MASK ) )" command, in
# "router-bgp" mode.
#
# "network ADDR route-map MAP " command, in "router-bgp" mode
#
# The no version of this command is handled by no network command
# as it specifies trailing garbage
#
# It's possible to specify network statements both inside and outside
# the address-family block (although when specified inside, they must
# be networks of the corresponding IP version). Supporting networks
# outside the address-family block is done for backward compatibility
# reasons.
#--------------------------------------------------------------------------------
class SetNetworkCmd( BgpCmdBaseClass ):
   syntax = 'network ( PREFIX_OR_ADDRMASK | PREFIX_OR_ADDRMASK6 ) [ route-map MAP ]'
   noOrDefaultSyntax = 'network ( PREFIX_OR_ADDRMASK | PREFIX_OR_ADDRMASK6 ) ...'
   data = {
         'network': bgpTokens.network,
         'PREFIX_OR_ADDRMASK': IpAddrMatcher.ipPrefixExpr(
                                    'Network address',
                                    'Network mask',
                                    'Prefix',
                                    overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                                    maskKeyword=True
                                ),
         'PREFIX_OR_ADDRMASK6': Ip6PrefixMatcher(
                                    'IPv6 address prefix',
                                    overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                                ),
         'route-map': bgpTokens.routeMap,
         'MAP': mapNameMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'prefix' ] = args.get(
            'PREFIX_OR_ADDRMASK',
            args.get( 'PREFIX_OR_ADDRMASK6' ) )

   @staticmethod
   def _setNetworkCommon( mode, prefix, restricted=False, mapName='', no=False ):
      if ':' in str( prefix ):
         if mode.addrFamily == 'ipv4':
            mode.addErrorAndStop( "IPv6 networks may not be specified in IPv4 mode" )
         if mode.addrFamily == 'ipv4 multicast':
            mode.addErrorAndStop( "IPv6 networks may not be specified in IPv4 " +
                                  "multicast mode" )

      if '.' in str( prefix ):
         if mode.addrFamily == 'ipv6':
            mode.addErrorAndStop( "IPv4 networks may not be specified in IPv6 mode" )

      if prefix is None:
         mode.addError( "Invalid network wildcard mask" )
         return

      vrfName = mode.vrfName
      addrFamily = mode.addrFamily

      if addrFamily in [ '', 'ipv4', 'ipv6' ]:
         addrFamily = 'all'
      prefix = str( prefix )
      config = configForVrf( vrfName )
      attrName = bgpConfigAttrsAfMap[ 'networkList' ].get( addrFamily )
      networkList = getattr( config, attrName )

      if no:
         del networkList[ Arnet.IpGenPrefix( prefix ) ]
      else:
         networkList.addMember( Tac.Value( 'Routing::Bgp::Network',
                                      network=Arnet.IpGenPrefix( prefix ),
                                      restricted=( restricted and
                                                   addrFamily in [ '', 'ipv4' ] ),
                                      routeMap=mapName ) )

   @staticmethod
   def _handleNormal( mode, args ):
      SetNetworkCmd._setNetworkCommon(
            mode,
            args.get( 'prefix' ),
            mapName=args.get( 'MAP', '' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetNetworkCmd._setNetworkCommon(
            mode,
            args.get( 'prefix' ),
            no=True
      )

for m in [ RouterBgpSharedModelet,
           RouterBgpAfIpUniAndMcastSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNetworkCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp aspath-cmp-include-nexthop" command, in
# "router-bgp" mode. If enabled, next-hop would be included in as-path
# comparison. This knob allows us to go back to old behaviour
#-------------------------------------------------------------------------------
class SetInstanceAspathCmpIncNhCmd( CliCommandClass ):
   syntax = 'bgp aspath-cmp-include-nexthop'
   noOrDefaultSyntax = 'bgp aspath-cmp-include-nexthop ...'
   data = {
         'bgp': bgpNode,
         'aspath-cmp-include-nexthop': bgpTokens.aspathCmpIncludeNh,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathCmpIncNh = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathCmpIncNh = getExplicitDefaultTristate( mode, 'asPathCmpIncNh' )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.asPathCmpIncNh = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetInstanceAspathCmpIncNhCmd )

#-------------------------------------------------------------------------------
# 'bgp bestpath tie-break ( cluster-list-length | originator-id | router-id | age )'
#-------------------------------------------------------------------------------
class SetBestPathTieBreakBase( object ):
   _tbAttrMap = {
         'cluster-list-length': 'ecmpTbOnClusterListLen',
         'originator-id': 'ecmpTbOnOrigId',
         'router-id': 'ecmpTbOnRouterId',
         'age': 'ecmpTbOnAge'
   }

   @staticmethod
   def _setTieBreak( mode, tieBreak ):
      config = configForVrf( mode.vrfName )
      setattr( config, SetBestPathTieBreakBase._tbAttrMap[ tieBreak ], 'isTrue' )

   @staticmethod
   def _noTieBreak( mode, tieBreak, noOrDefault ):
      config = configForVrf( mode.vrfName )
      attr = SetBestPathTieBreakBase._tbAttrMap[ tieBreak ]
      if noOrDefault == NoOrDefault.DEFAULT:
         val = 'isInvalid'
      else:
         val = getExplicitDefaultTristate( mode, attr )
      setattr( config, attr, val )

class SetBestPathTieBreakCmd( CliCommandClass, SetBestPathTieBreakBase ):
   syntax = 'bgp bestpath tie-break ( TIE_BREAK | age )'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'tie-break': bgpTokens.tieBreak,
         'TIE_BREAK': EnumMatcher( {
               'cluster-list-length': (
                     'Tie-break BGP paths based on path cluster list length'
               ),
               'originator-id': 'Tie-break BGP paths based on path originator-id',
               'router-id': 'Tie-break BGP paths based on path router-id',
         } ),
         # The 'age' kw cannot be in an enum matcher as it is hidden.
         'age': bgpTokens.age,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'age' in args:
         args[ 'TIE_BREAK' ] = 'age'

   @staticmethod
   def handler( mode, args ):
      SetBestPathTieBreakBase._setTieBreak( mode, args[ 'TIE_BREAK' ] )

   @staticmethod
   def noHandler( mode, args ):
      SetBestPathTieBreakBase._noTieBreak(
            mode,
            args[ 'TIE_BREAK' ],
            NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetBestPathTieBreakBase._noTieBreak(
            mode,
            args[ 'TIE_BREAK' ],
            NoOrDefault.DEFAULT
      )

#-------------------------------------------------------------------------------
# "[no|default] bgp tie-break-on-age" command, in
# "router-bgp" mode. If enabled, We add new ecmp group entries to the tail of
# linked list. This is O(1) operation. So it results in better performance. But
# which path becomes ECMP head is not deterministic
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break age" which is also hidden at the
# moment for implementation detail purposes.
#
#
# "[no|default] bgp tie-break-on-router-id" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on router-id alone. This results in slightly inferior performance
# compared to tie-break-on-age but will result in deterministic ordering of paths
# in the ECMP group
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break router-id"
#
#
# "[no|default] bgp tie-break-on-originator-id" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on originator-id alone.
# Note: This command is hidden and will not be replaced. Its functionality is
# included within "[no|default] bgp bestpath tie-break originator-id". Another
# hidden command will be added that follows the new model but won't be
# advertised.
#
#
# "[no|default] bgp tie-break-on-cluster-list-length" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on cluster-list-length alone.
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break cluster-list-length"
#-------------------------------------------------------------------------------
class SetEcmpTbDeprecatedCmd( CliCommandClass, SetBestPathTieBreakBase ):
   syntax = 'bgp TIE_BREAK'
   noOrDefaultSyntax = syntax + '...'
   data = {
         'bgp': bgpNode,
         'TIE_BREAK': EnumMatcher( {
               'tie-break-on-age': (
                     'Tie-break BGP paths in a ECMP group based on the '
                     'order of arrival'
               ),
               'tie-break-on-router-id': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path router-id'
               ),
               'tie-break-on-originator-id': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path originator-id'
               ),
               'tie-break-on-cluster-list-length': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path cluster list length'
               ),
         } )
   }
   hidden = True

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'TIE_BREAK' ] = args[ 'TIE_BREAK' ].replace( 'tie-break-on-', '' )

   @staticmethod
   def handler( mode, args ):
      SetBestPathTieBreakBase._setTieBreak(
            mode,
            args[ 'TIE_BREAK' ],
      )

   @staticmethod
   def noHandler( mode, args ):
      SetBestPathTieBreakBase._noTieBreak(
            mode,
            args[ 'TIE_BREAK' ],
            NoOrDefault.NO
      )

   @staticmethod
   def defaultHandler( mode, args ):
      SetBestPathTieBreakBase._noTieBreak(
            mode,
            args[ 'TIE_BREAK' ],
            NoOrDefault.DEFAULT
      )

RouterBgpSharedModelet.addCommandClass( SetBestPathTieBreakCmd )
RouterBgpSharedModelet.addCommandClass( SetEcmpTbDeprecatedCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath ecmp-fast" command, in "router-bgp" mode. Which is
# enabled by default. We add new ecmp group entries the tail of the
# list. This is our ranking based on "age" which is an O(1) operation. Best
# performance is achieved when enabled, but ECMP head is not deterministic.
# Disabling it reverts the ECMP best-path selection to deterministic.
#-------------------------------------------------------------------------------
class SetBestPathEcmpFastCmd( CliCommandClass ):
   syntax = 'bgp bestpath ecmp-fast'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'ecmp-fast': bgpTokens.ecmpFast,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ecmpFast = getExplicitDefaultTristate( mode, 'ecmpFast' )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ecmpFast = 'isInvalid'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.ecmpFast = 'isFalse'

RouterBgpSharedModelet.addCommandClass( SetBestPathEcmpFastCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp advertise-inactive" command, in
# "router-bgp" mode. If enabled, we consider BRIB winners for advertisements in
# BGP policy code (and not active routes in RIB)
#-------------------------------------------------------------------------------
class SetInstanceAdvertiseInactive( CliCommandClass ):
   syntax = 'bgp advertise-inactive'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'advertise-inactive': bgpTokens.advertiseInactive,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.advertiseInactive = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.advertiseInactive = getExplicitDefaultTristate(
            mode,
            'advertiseInactive'
      )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.advertiseInactive = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetInstanceAdvertiseInactive )

#-------------------------------------------------------------------------------
# "[no|default] aggregate-address (<prefix> | <address> <mask>) [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [ advertise-only ]" command,
#      in "router-bgp" mode.
#
# "[no|default] aggregate-address <v6prefix> [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [ advertise-only ]" command,
#      in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAggregateAddressCmd( BgpCmdBaseClass ):
   syntax = (
         'aggregate-address ( PREFIX | PREFIX6 ) '
         '[ { OPTIONS | ( attribute-map ATTR_MAP ) | ( match-map MATCH_MAP ) } ]'
   )
   noOrDefaultSyntax = 'aggregate-address ( PREFIX | PREFIX6 ) ...'
   data = {
         'aggregate-address': bgpTokens.aggregate,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                    ),
         'OPTIONS': SetEnumMatcher( {
               bgpTokens.asSet.keyword_: bgpTokens.asSet.helpdesc_,
               bgpTokens.summaryOnly.keyword_: bgpTokens.summaryOnly.helpdesc_,
               bgpTokens.advertiseOnly.keyword_: bgpTokens.advertiseOnly.helpdesc_,
         } ),
         'attribute-map': singleNode( bgpTokens.attrMap ),
         'ATTR_MAP': Node( mapNameMatcher, maxMatches=1 ),
         'match-map': singleNode( bgpTokens.matchMap ),
         'MATCH_MAP': Node( mapNameMatcher, maxMatches=1 ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get(
            'PREFIX',
            args.get( 'PREFIX6' )
      )
      options = args.get( 'OPTIONS' )
      if options:
         args[ 'summary-only' ] = 'summary-only' in options
         args[ 'advertise-only' ] = 'advertise-only' in options
         args[ 'as-set' ] = 'as-set' in options

   @staticmethod
   def _setAggregateAddress(
         mode,
         address,
         asset=False,
         summaryonly=False,
         advertiseonly=False,
         attrmap=None,
         matchmap=None,
   ):
      if address is None or str( address ) == '':
         mode.addError( 'Invalid address wildcard mask' )
         return

      if not summaryonly:
         summaryonly = AggregateAddressType.summaryOnlyDefault
      if not asset:
         asset = AggregateAddressType.asSetDefault
      if not attrmap:
         attrmap = AggregateAddressType.attrMapDefault
      if not matchmap:
         matchmap = AggregateAddressType.matchMapDefault
      if not advertiseonly:
         advertiseonly = AggregateAddressType.advertiseOnlyDefault

      configForVrf( mode.vrfName ).aggregateList.addMember(
            Tac.Value(
                  'Routing::Bgp::AggregateAddress',
                  address=Arnet.IpGenPrefix( str( address ) ),
                  summaryOnly=summaryonly,
                  asSet=asset,
                  attrMap=attrmap,
                  matchMap=matchmap,
                  advertiseOnly=advertiseonly
            )
      )

   @staticmethod
   def _noAggregateAddress( mode, address ):
      config = configForVrf( mode.vrfName )
      del config.aggregateList[ Arnet.IpGenPrefix( str( address ) ) ]

   @staticmethod
   def _handleNormal( mode, args ):
      SetAggregateAddressCmd._setAggregateAddress(
            mode,
            args[ 'PREFIX' ],
            asset=args.get( 'as-set', False ),
            summaryonly=args.get( 'summary-only', False ),
            advertiseonly=args.get( 'advertise-only', False ),
            attrmap=args.get( 'ATTR_MAP' ),
            matchmap=args.get( 'MATCH_MAP' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetAggregateAddressCmd._noAggregateAddress( mode, args[ 'PREFIX' ] )

RouterBgpSharedModelet.addCommandClass( SetAggregateAddressCmd )

#-------------------------------------------------------------------------------
# "[no|default] auto-aggregation-domain (<prefix> | <address> <mask>) [as-set]
#      [eligible-routes <mapname>] [aggregate-attributes <mapname>]
#      [ group-by {as-path|local-preference|med|community|ext-community}+ ]" command,
#      in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAutoAggDomainCmd( BgpCmdBaseClass ):
   syntax = (
         'auto-aggregation-domain ( PREFIX | PREFIX6 ) '
         '[ { as-set | ( eligible-routes RTS ) | ( aggregate-attributes ATTRS ) } ] '
         '[ group-by { GROUP_ARGS } ]'
   )
   noOrDefaultSyntax = 'auto-aggregation-domain ( PREFIX | PREFIX6 ) ...'
   data = {
         'auto-aggregation-domain': bgpTokens.autoAggDomain,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
         ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
         ),
         'as-set': Node( matcher=bgpTokens.asSet, maxMatches=1 ),
         'eligible-routes': Node( matcher=bgpTokens.eligibleRoutes, maxMatches=1 ),
         'RTS': mapNameMatcher,
         'aggregate-attributes': Node(
               matcher=bgpTokens.aggAttributes,
               maxMatches=1
         ),
         'ATTRS': mapNameMatcher,
         'group-by': bgpTokens.groupBy,
         'GROUP_ARGS': SetEnumMatcher( {
               bgpTokens.groupByAsPath.keyword_: bgpTokens.groupByAsPath.helpdesc_,
               bgpTokens.groupByLocPref.keyword_: bgpTokens.groupByLocPref.helpdesc_,
               bgpTokens.groupByMed.keyword_: bgpTokens.groupByMed.helpdesc_,
               bgpTokens.groupByCommunity.keyword_:
                     bgpTokens.groupByCommunity.helpdesc_,
               bgpTokens.groupByExtCommunity.keyword_:
                     bgpTokens.groupByExtCommunity.helpdesc_,
         } ),
   }

   hidden = True

   _groupArgsMap = {
         'as-path': 'groupByAsPath',
         'local-preference': 'groupByLocPref',
         'med': 'groupByMed',
         'community': 'groupByCommunity',
         'ext-community': 'groupByExtCommunity',
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )

      domainArgs = {}
      if 'as-set' in args:
         domainArgs[ 'asset' ] = True
      if 'RTS' in args:
         domainArgs[ 'matchmap' ] = args[ 'RTS' ][ 0 ]
      if 'ATTRS' in args:
         domainArgs[ 'attrmap' ] = args[ 'ATTRS' ][ 0 ]
      args[ 'DOMAIN_ARGS' ] = domainArgs

      if 'GROUP_ARGS' in args:
         args[ 'GROUP_ARGS' ] = {
               SetInstanceAutoAggDomainCmd._groupArgsMap[ arg ]: True for arg in
               args.get( 'GROUP_ARGS' )
         }

   @staticmethod
   def _setAutoAggDomain(
         mode,
         prefix,
         autoAggDomainArgs=None,
         autoAggGroupByArgs=None
   ):
      prefix = Arnet.IpGenPrefix( str( prefix ) )
      if prefix is None or str( prefix ) == '':
         mode.addError( "Invalid prefix" )
         return

      argsDict = dict( autoAggDomainArgs if autoAggDomainArgs else [] )
      asset = argsDict.get( 'asset', AutoAggregateDomainType.asSetDefault )
      attrmap = argsDict.get( 'attrmap', AutoAggregateDomainType.attrMapDefault )
      matchmap = argsDict.get( 'matchmap', AutoAggregateDomainType.matchMapDefault )

      groupByDict = dict( autoAggGroupByArgs if autoAggGroupByArgs else [] )
      groupByAsPath = 'groupByAsPath' in groupByDict
      groupByLocPref = 'groupByLocPref' in groupByDict
      groupByMed = 'groupByMed' in groupByDict
      groupByCommunity = 'groupByCommunity' in groupByDict
      groupByExtCommunity = 'groupByExtCommunity' in groupByDict

      configForVrf( mode.vrfName ).autoAggDomainList.addMember(
         AutoAggregateDomainType( prefix, asSet=asset,
                                  attrMap=attrmap, matchMap=matchmap,
                                  groupByAsPath=groupByAsPath,
                                  groupByLocPref=groupByLocPref,
                                  groupByMed=groupByMed,
                                  groupByCommunity=groupByCommunity,
                                  groupByExtCommunity=groupByExtCommunity ) )

   @staticmethod
   def _noAutoAggDomain( mode, prefix ):
      config = configForVrf( mode.vrfName )
      del config.autoAggDomainList[ Arnet.IpGenPrefix( str( prefix ) ) ]

   @staticmethod
   def _handleNormal( mode, args ):
      SetInstanceAutoAggDomainCmd._setAutoAggDomain(
            mode,
            args[ 'PREFIX' ],
            autoAggDomainArgs=args.get( 'DOMAIN_ARGS', {} ),
            autoAggGroupByArgs=args.get( 'GROUP_ARGS', {} )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetInstanceAutoAggDomainCmd._noAutoAggDomain( mode, args[ 'PREFIX' ] )

RouterBgpSharedModelet.addCommandClass( SetInstanceAutoAggDomainCmd )

#-------------------------------------------------------------------------------
# "[no|default] router-id <ip-address>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceRouterIdCmd( BgpCmdBaseClass ):
   syntax = 'router-id ROUTER_ID'
   # The no rule shouldn't have trailingGarbage, or else 'no router bgp 1'
   # executed in router-bgp mode matches the 'no router-id' rule.
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
         'router-id': bgpTokens.routerId,
         'ROUTER_ID': IpAddrMatcher.IpAddrMatcher(
            helpdesc='BGP router-id in IP address format' )
   }

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      config.routerId = args[ 'ROUTER_ID' ]

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      config.routerId = config.routerIdDefault

RouterBgpSharedModelet.addCommandClass( SetInstanceRouterIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] distance bgp <distance>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceBgpDistanceCmd( CliCommandClass ):
   syntax = 'distance bgp EXTERNAL [ INTERNAL LOCAL ]'
   noOrDefaultSyntax = 'distance bgp ...'
   data = {
         'distance': bgpTokens.distance,
         'bgp': bgpTokens.distanceBgp,
         'EXTERNAL': IntegerMatcher(
               1,
               255,
               helpdesc=(
                     'Distance for external routes OR external, internal and '
                     'local routes'
               )
         ),
         'INTERNAL': IntegerMatcher(
               1,
               255,
               helpdesc='BGP internal route admin distance'
         ),
         'LOCAL': IntegerMatcher(
               1,
               255,
               helpdesc='BGP local route admin distance'
         )
   }

   @staticmethod
   def _distanceBgp(
         mode,
         bgpExternalDistance,
         bgpInternalDistance=None,
         bgpLocalDistance=None
   ):
      # The cli prevents bgpInternalDistance and bgpLocalDistance
      # being set to None independently. Either both have values, or both
      # are none
      if bgpInternalDistance is None and bgpLocalDistance is None:
         bgpInternalDistance = bgpExternalDistance
         bgpLocalDistance = bgpExternalDistance
      config = configForVrf( mode.vrfName )
      setVal = True
      if ( mode.vrfName == DEFAULT_VRF and
           bgpExternalDistance == config.externalDistanceDefault and
           bgpInternalDistance == config.internalDistanceDefault and
           bgpLocalDistance == config.localDistanceDefault ):
         setVal = False
      eD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=bgpExternalDistance, isSet=setVal )
      iD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=bgpInternalDistance, isSet=setVal )
      lD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=bgpLocalDistance, isSet=setVal )
      config.externalDistance = eD
      config.internalDistance = iD
      config.localDistance = lD

   @staticmethod
   def _noDistanceBgp( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      # The "no distance bgp" command operates differently from the
      # "default distance bgp..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         setVal = False
      else:
         setVal = True
      eD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=config.externalDistanceDefault, isSet=setVal )
      iD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=config.internalDistanceDefault, isSet=setVal )
      lD = Tac.Value( 'Routing::Bgp::RoutePreferenceOrNone',
                       pref=config.localDistanceDefault, isSet=setVal )
      config.externalDistance = eD
      config.internalDistance = iD
      config.localDistance = lD

   @staticmethod
   def handler( mode, args ):
      SetInstanceBgpDistanceCmd._distanceBgp(
            mode,
            args[ 'EXTERNAL' ],
            bgpInternalDistance=args.get( 'INTERNAL' ),
            bgpLocalDistance=args.get( 'LOCAL' )
      )

   @staticmethod
   def noHandler( mode, args ):
      SetInstanceBgpDistanceCmd._noDistanceBgp( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetInstanceBgpDistanceCmd._noDistanceBgp( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( SetInstanceBgpDistanceCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp listen [ limit <> | range network/length peer-group
# <peer-group name> ]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceListenLimitRangeBaseClass( object ):
   data = {
         'bgp': bgpNode,
         'listen': bgpTokens.listen,
         'LIMIT_KW_DEPRECATED': bgpTokens.limitDeprecated,
         'LIMIT': bgpTokens.limitMatcher,
         'dynamic': bgpTokens.dynamic,
         'peer': bgpTokens.peerAfterDynamic,
         'max': bgpTokens.maxAfterDynamicPeer,
         'range': bgpTokens.subnetRange,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                       maskKeyword=True
                   ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                    ),
         'peer-id': 'Additional specification for identifying a peer',
         'include': 'Include following fields as part of peer identifier',
         'router-id': 'Include router ID as part of peer identifier',
         'peer-group': bgpTokens.peerGroup,
         'PGKEY': peergroupNameMatcher,
         'PEER_SPEC': bgpTokens.PeerSpecExpression,
   }

   @staticmethod
   def _setBgpListenLimit( mode, bgpLimit ):
      config = configForVrf( mode.vrfName )
      config.listenLimit = getExplicitDefaultValue(
            mode,
            'listenLimit',
            bgpLimit
      )

   @staticmethod
   def _noBgpListenLimit( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      # The "no bgp listen limit..." command operates differently from
      # the "default bgp listen limit..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.listenLimit = config.listenLimitInvalid
      else:
         config.listenLimit = config.listenLimitDefault

   @staticmethod
   def _setBgpListenRange( mode, prefix, pgKey, peerSpecifier, includeRouterId ):
      config = configForVrf( mode.vrfName )
      prefix = Arnet.IpGenPrefix( str( prefix ) )
      assert pgKey.group is not None, "The peer group key is of the wrong type"
      peergroupName = pgKey.group
      for pg in config.listenRange:
         # Prevent configuration of identical or overlapping listenRanges
         for existingPrefix in config.listenRange.get( pg ).prefixList:
            if prefix == existingPrefix and \
               config.listenRange.get( pg ).peergroupName == peergroupName:
               continue
            if prefix.overlaps( existingPrefix ):
               mode.addError( 'Cannot configure range %s while %s'
                                    ' is configured against %s.'
                                    % ( prefix, existingPrefix, pg ) )
               return
      listenRangeConfig = config.listenRange.get( peergroupName )
      if not listenRangeConfig:
         listenRangeConfig = config.listenRange.newMember( peergroupName )

      peerSpec = Tac.Value( 'Routing::Bgp::PeerSpecifier', prefix )
      if includeRouterId and getEffectiveProtocolModel( mode ) != \
         ProtocolAgentModelType.multiAgent and \
         not ( mode.session.inConfigSession() or mode.session.startupConfig() ):
         mode.addError( "peer-id is only supported in multi-agent mode" )
         return
      peerSpec.includeRouterId = includeRouterId
      if isinstance( peerSpecifier, str ):                                           
         peerSpec.hasAsn = False                                                     
         peerSpec.peerFilter = peerSpecifier                                         
      else:                                                                          
         peerSpec.hasAsn = True                                                      
         peerSpec.asn = peerSpecifier
      listenRangeConfig.prefixList[ prefix ] = peerSpec
      bgpNeighborConfig( pgKey, vrfName=mode.vrfName )

   @staticmethod
   def _noBgpListenRange( mode, prefix, pgKey ):
      config = configForVrf( mode.vrfName )
      prefix = Arnet.IpGenPrefix( str( prefix ) )
      assert pgKey.group is not None, "The peer group key is of the wrong type"
      peergroupName = pgKey.group
      if config.listenRange.get( peergroupName ):
         del config.listenRange[ peergroupName ].prefixList[ prefix ]
         if not config.listenRange.get( peergroupName ).prefixList:
            del config.listenRange[ peergroupName ]

class SetInstanceListenLimitDeprecatedCmd(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):
   syntax = 'bgp listen LIMIT_KW_DEPRECATED LIMIT'
   noOrDefaultSyntax = 'bgp listen LIMIT_KW_DEPRECATED ...'
   data = SetInstanceListenLimitRangeBaseClass.data.copy()

   @staticmethod
   def handler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._setBgpListenLimit(
            mode,
            args[ 'LIMIT' ]
      )

   @staticmethod
   def noHandler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._noBgpListenLimit( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._noBgpListenLimit(
            mode,
            NoOrDefault.DEFAULT
      )

RouterBgpSharedModelet.addCommandClass( SetInstanceListenLimitDeprecatedCmd )

class SetInstanceDynamicPeerMaxLimit(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):
   syntax = 'dynamic peer max LIMIT'
   noOrDefaultSyntax = 'dynamic peer max ...'
   data = SetInstanceListenLimitRangeBaseClass.data.copy()

   @staticmethod
   def handler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._setBgpListenLimit(
            mode,
            args[ 'LIMIT' ]
      )

   @staticmethod
   def noHandler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._noBgpListenLimit( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._noBgpListenLimit(
            mode,
            NoOrDefault.DEFAULT
      )

RouterBgpSharedModelet.addCommandClass( SetInstanceDynamicPeerMaxLimit )

class SetInstanceListenRangePrefixPgCmd(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):

   if BgpCommonToggleLib.toggleArBgpSameAddressPeeringEnabled():
      syntax = """bgp listen range ( PREFIX | PREFIX6 ) 
                  [ peer-id include router-id ] peer-group PGKEY PEER_SPEC"""
      noOrDefaultSyntax = """bgp listen range ( PREFIX | PREFIX6 ) 
                             [ peer-id include router-id ] peer-group PGKEY ..."""
   else:
      syntax = 'bgp listen range ( PREFIX | PREFIX6 ) peer-group PGKEY PEER_SPEC'
      noOrDefaultSyntax = """bgp listen range ( PREFIX | PREFIX6 ) 
                             peer-group PGKEY ..."""

   data = SetInstanceListenLimitRangeBaseClass.data.copy()

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get( 'PREFIX', args.get( 'PREFIX6' ) )
      args[ 'PGKEY' ] = PeerConfigKey( args[ 'PGKEY' ] )

   @staticmethod
   def handler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._setBgpListenRange(
            mode,
            args[ 'PREFIX' ],
            args[ 'PGKEY' ],
            args[ 'PEER_SPEC' ],
            'router-id' in args,
      )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      SetInstanceListenLimitRangeBaseClass._noBgpListenRange(
            mode,
            args[ 'PREFIX' ],
            args[ 'PGKEY' ],
      )

RouterBgpSharedModelet.addCommandClass( SetInstanceListenRangePrefixPgCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp default ipv4-unicast" command, in "router-bgp" mode.
# "[no|default] bgp default ipv4-unicast transport ipv6" command, in "router-bgp"
#    mode.
# "[no|default] bgp default ipv6-unicast" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetIpv4UnicastCmd( CliCommandClass ):
   syntax = 'bgp default ipv4-unicast [ transport ipv6 ]'
   noOrDefaultSyntax = 'bgp default ipv4-unicast [ transport ipv6 ] ...'
   data = {
         'bgp': bgpNode,
         'default': bgpTokens.default,
         'ipv4-unicast': bgpTokens.iPv4Unicast,
         'transport': bgpTokens.transportForIpv4Unicast,
         'ipv6': bgpTokens.iPv6AfterTransport,
   }

   @staticmethod
   def _setIPv4Unicast( mode, overIpv6=False ):
      config = configForVrf( mode.vrfName )
      if overIpv6:
         # IPv4 unicast over IPv6 is disabled by default
         config.defaultV4UniOverV6 = 'isTrue'
      else:
         # IPv4 unicast is enabled by default
         config.defaultV4Uni = getExplicitDefaultTristate( mode, 'defaultV4Uni' )

   @staticmethod
   def _noDefaultIPv4Unicast( mode, noOrDefault, overIpv6=False ):
      config = configForVrf( mode.vrfName )
      if noOrDefault == NoOrDefault.DEFAULT:
         if overIpv6:
            config.defaultV4UniOverV6 = 'isInvalid'
         else:
            config.defaultV4Uni = 'isInvalid'
      else:
         if overIpv6:
            config.defaultV4UniOverV6 = getExplicitDefaultTristate(
                                              mode, 'defaultV4UniOverV6' )
         else:
            config.defaultV4Uni = 'isFalse'

   @staticmethod
   def handler( mode, args ):
      SetIpv4UnicastCmd._setIPv4Unicast( mode, overIpv6=( 'transport' in args ) )

   @staticmethod
   def defaultHandler( mode, args ):
      SetIpv4UnicastCmd._noDefaultIPv4Unicast(
            mode,
            NoOrDefault.DEFAULT,
            overIpv6=( 'transport' in args ) )

   @staticmethod
   def noHandler( mode, args ):
      SetIpv4UnicastCmd._noDefaultIPv4Unicast(
            mode,
            NoOrDefault.NO,
            overIpv6=( 'transport' in args ) )

RouterBgpSharedModelet.addCommandClass( SetIpv4UnicastCmd )

class SetIpv6UnicastCmd( CliCommandClass ):
   syntax = 'bgp default ipv6-unicast'
   noOrDefaultSyntax = 'bgp default ipv6-unicast ...'
   data = {
         'bgp': bgpNode,
         'default': bgpTokens.default,
         'ipv6-unicast': bgpTokens.iPv6Unicast,
   }

   @staticmethod
   def _setIPv6Unicast( mode ):
      # IPv6 unicast is disabled by default
      config = configForVrf( mode.vrfName )
      config.defaultV6Uni = 'isTrue'

   @staticmethod
   def _noDefaultIPv6Unicast( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      if noOrDefault == NoOrDefault.DEFAULT:
         config.defaultV6Uni = 'isInvalid'
      else:
         config.defaultV6Uni = getExplicitDefaultTristate( mode, 'defaultV6Uni' )

   @staticmethod
   def handler( mode, args ):
      SetIpv6UnicastCmd._setIPv6Unicast( mode )

   @staticmethod
   def noHandler( mode, args ):
      SetIpv6UnicastCmd._noDefaultIPv6Unicast( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetIpv6UnicastCmd._noDefaultIPv6Unicast( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( SetIpv6UnicastCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [address-family all] [include sub-route-map]
# direction [ in | out ] action" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
dirOptions = {
   'in': 'Missing policy inbound direction',
   'out': 'Missing policy outbound direction'
}
actionOptions = {
   'permit': 'Permit all routes for specified direction with missing policy '
             'configuration',
   'deny': 'Deny all routes for specified direction with missing policy '
           'configuration',
   'deny-in-out': 'Deny all routes for both in/out directions with missing '
                  'policy configuration in specified direction'
}
actionValues = {
   'permit': PolicyActionEnum.policyActionPermit,
   'deny': PolicyActionEnum.policyActionDeny,
   'deny-in-out': PolicyActionEnum.policyActionDenyBoth
}

class MissingPolicyCmd( CliCommandClass ):
   if BgpCommonToggleLib.toggleMissingPolicyIncludePrefixListEnabled():
      syntax = ( 'bgp missing-policy [ address-family all ] '
                 '[ include { sub-route-map | prefix-list } ] '
                 'direction DIR action ACT' )
      noOrDefaultSyntax = syntax.replace( 'ACT', '...' )
   else:
      syntax = ( 'bgp missing-policy [ address-family all ] '
                 '[ include { sub-route-map } ] '
                 'direction DIR action ACT' )
      noOrDefaultSyntax = syntax.replace( 'ACT', '...' )

   data = {
      'bgp': bgpNode,
      'missing-policy': bgpTokens.missingPolicy,
      'address-family': bgpTokens.mpAddrFamily,
      'all': bgpTokens.mpAddrFamilyAll,
      'include': bgpTokens.mpInclude,
      'sub-route-map': singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': singleNode( bgpTokens.mpPrefixList ),
      'direction': bgpTokens.mpDirection,
      'DIR': EnumMatcher( dirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': EnumMatcher( actionOptions )
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      actType = actionValues[ args[ 'ACT' ] ]
      if ( mode.vrfName == DEFAULT_VRF and
           actType == config.missingPolicyActionDefault ):
         actType = config.missingPolicyActionInvalid
      addressFamilyAll = 'address-family' in args
      includeSubRm = 'sub-route-map' in args
      includePfxList = 'prefix-list' in args
      direction = args[ 'DIR' ].capitalize()
      actionAttr = 'missingPolicyAction' + direction
      actionAddrFamilyAllAttr = 'missingPolicyAddressFamilyAll' + direction
      actionIncludeSubAttr = 'missingPolicyIncludeSubRouteMap' + direction
      actionIncludePfxListAttr = 'missingPolicyIncludePrefixList' + direction
      setattr( config, actionAttr, actType )
      if addressFamilyAll:
         setattr( config, actionAddrFamilyAllAttr, 'isTrue' )
      else:
         setattr( config, actionAddrFamilyAllAttr, 'isInvalid' )
      if includeSubRm:
         setattr( config, actionIncludeSubAttr, 'isTrue' )
      else:
         setattr( config, actionIncludeSubAttr, 'isInvalid' )
      if includePfxList:
         setattr( config, actionIncludePfxListAttr, 'isTrue' )
      else:
         setattr( config, actionIncludePfxListAttr, 'isInvalid' )

      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
         if addressFamilyAll:
            mode.addWarning( "Routing protocols model multi-agent must be "
                             "configured for bgp missing-policy address-family "
                             "all configuration" )
         if includeSubRm or includePfxList:
            mode.addWarning( "Routing protocols model multi-agent must be "
                             "configured for bgp missing-policy include options" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      direction = args[ 'DIR' ].capitalize()
      actionAttr = 'missingPolicyAction' + direction
      actionAddrFamilyAllAttr = 'missingPolicyAddressFamilyAll' + direction
      actionIncludeSubAttr = 'missingPolicyIncludeSubRouteMap' + direction
      actionIncludePfxListAttr = 'missingPolicyIncludePrefixList' + direction
      actType = config.missingPolicyActionDefault
      addressFamilyAll = getExplicitDefaultTristate(
         mode, 'missingPolicyAddressFamilyAll' + args[ 'DIR' ].capitalize() )
      inclSubRm = getExplicitDefaultTristate(
         mode, 'missingPolicyIncludeSubRouteMap' + args[ 'DIR' ].capitalize() )
      inclPfxList = getExplicitDefaultTristate(
         mode, 'missingPolicyIncludePrefixList' + args[ 'DIR' ].capitalize() )
      if mode.vrfName == DEFAULT_VRF or isDefaultCmd( args ):
         actType = config.missingPolicyActionInvalid
         addressFamilyAll = 'isInvalid'
         inclSubRm = 'isInvalid'
         inclPfxList = 'isInvalid'
      setattr( config, actionAttr, actType )
      setattr( config, actionAddrFamilyAllAttr, addressFamilyAll )
      setattr( config, actionIncludeSubAttr, inclSubRm )
      setattr( config, actionIncludePfxListAttr, inclPfxList )

RouterBgpSharedModelet.addCommandClass( MissingPolicyCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [include sub-route-map] direction [ in | out ]
# action" command, in router bgp af mode.
#-------------------------------------------------------------------------------
class MissingPolicyAfSharedCmd( CliCommandClass ):
   if BgpCommonToggleLib.toggleMissingPolicyIncludePrefixListEnabled():
      syntax = ( 'bgp missing-policy [ include { sub-route-map | prefix-list } ] '
                 'direction DIR action ACT' )
      noOrDefaultSyntax = syntax.replace( 'ACT', '...' )
   else:
      syntax = ( 'bgp missing-policy [ include { sub-route-map } ] '
                 'direction DIR action ACT' )
      noOrDefaultSyntax = syntax.replace( 'ACT', '...' )
   data = {
      'bgp': bgpNode,
      'missing-policy': bgpTokens.missingPolicy,
      'include': bgpTokens.mpInclude,
      'sub-route-map': singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': singleNode( bgpTokens.mpPrefixList ),
      'direction': bgpTokens.mpDirection,
      'DIR': EnumMatcher( dirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': EnumMatcher( actionOptions )
   }

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
         mode.addWarning( "Routing protocols model multi-agent must be "
                          "configured for bgp missing-policy configuration "
                          "under address family mode" )
      config = configForVrf( mode.vrfName )
      actType = actionValues[ args[ 'ACT' ] ]
      includeSubRm = 'sub-route-map' in args
      includePfxList = 'prefix-list' in args
      direction = args[ 'DIR' ].capitalize()
      actionAfAttr = 'missingPolicyActionAf' + direction
      actionIncludeSubAfAttr = 'missingPolicyIncludeSubRouteMapAf' + direction
      actionIncludePfxAfAttr = 'missingPolicyIncludePrefixListAf' + direction
      actionAfConfig = getattr( config, actionAfAttr )
      actionIncludeSubAfConfig = getattr( config, actionIncludeSubAfAttr )
      actionIncludePfxAfConfig = getattr( config, actionIncludePfxAfAttr )
      af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ mode.addrFamily ] )
      actionAfConfig[ af ] = actType
      actionIncludeSubAfConfig[ af ] = 'isTrue' if includeSubRm else 'isFalse'
      actionIncludePfxAfConfig[ af ] = 'isTrue' if includePfxList else 'isFalse'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      direction = args[ 'DIR' ].capitalize()
      actionAfAttr = 'missingPolicyActionAf' + direction
      actionIncludeSubAfAttr = 'missingPolicyIncludeSubRouteMapAf' + direction
      actionIncludePfxAfAttr = 'missingPolicyIncludePrefixListAf' + direction
      actionAfConfig = getattr( config, actionAfAttr )
      actionIncludeSubAfConfig = getattr( config, actionIncludeSubAfAttr )
      actionIncludePfxAfConfig = getattr( config, actionIncludePfxAfAttr )
      af = Tac.Value( "Routing::Bgp::AfiSafi", *addressFamilies[ mode.addrFamily ] )
      del actionAfConfig[ af ]
      del actionIncludeSubAfConfig[ af ]
      del actionIncludePfxAfConfig[ af ]

RouterBgpAfSharedModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpAfLabelSharedModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpAfSrTeSharedModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpAfLinkStateModelet.addCommandClass( MissingPolicyAfSharedCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp auto-local-addr" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAutoLocalAddrCmd( CliCommandClass ):
   syntax = 'bgp auto-local-addr'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'auto-local-addr': bgpTokens.autoLocalAddr,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.autoLocalAddr = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.autoLocalAddr = getExplicitDefaultTristate(
            mode,
            'autoLocalAddr'
      )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.autoLocalAddr = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetInstanceAutoLocalAddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp control-plane-filter default-allow"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceControlPlaneFilterDefaultAllowCmd( BgpCmdBaseClass ):
   syntax = 'bgp control-plane-filter default-allow'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'control-plane-filter': bgpTokens.controlPlane,
         'default-allow': bgpTokens.defaultAllow,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      config.cpFilterDefaultAllow = True

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      config.cpFilterDefaultAllow = False

RouterBgpSharedModelet.addCommandClass(
      SetInstanceControlPlaneFilterDefaultAllowCmd
)

#----------------------------------------------------------------------------------
# " bgp transport ( ipv4 | ipv6 ) mss [ MSS ]" command in "router-bgp" mode
#----------------------------------------------------------------------------------
class SetTransportAfMssCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport ( ( ipv4 mss MSS) | ( ipv6 mss MSS6) )'
   noOrDefaultSyntax = 'bgp transport ( ipv4 | ipv6 ) mss ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
         'mss': bgpTokens.maxSegSize,
         'MSS': bgpTokens.mssV4RangeMatcher,
         'MSS6': bgpTokens.mssV6RangeMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'MSS' ] = args.get( 'MSS' ) or args.get( 'MSS6' )

   @staticmethod
   def _setTcpMaxSegSize( mode, mss, ipv4 ):
      config = configForVrf( mode.vrfName )
      if ipv4:
         config.sockMaxSegSizeIpv4 = mss
      else:
         config.sockMaxSegSizeIpv6 = mss

   @staticmethod
   def _noTcpMaxSegSize( mode, ipv4 ):
      config = configForVrf( mode.vrfName )
      if ipv4:
         config.sockMaxSegSizeIpv4 = config.sockMaxSegSizeInvalid
      else:
         config.sockMaxSegSizeIpv6 = config.sockMaxSegSizeInvalid

   @staticmethod
   def _handleNormal( mode, args ):
      SetTransportAfMssCmd._setTcpMaxSegSize(
            mode,
            args[ 'MSS' ],
            ( 'ipv4' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetTransportAfMssCmd._noTcpMaxSegSize( mode, ( 'ipv4' in args ) )

RouterBgpSharedModelet.addCommandClass( SetTransportAfMssCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport listen-port <portnum>" command, in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportListenPortCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport listen-port PORT'
   noOrDefaultSyntax = 'bgp transport listen-port ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'listen-port': bgpTokens.listenPort,
         'PORT': bgpTokens.portNumRangeMatcher,
   }

   @staticmethod
   def _setListenPort( mode, listenPort ):
      config = configForVrf( mode.vrfName )
      config.listenPort = listenPort

   @staticmethod
   def _noListenPort( mode ):
      config = configForVrf( mode.vrfName )
      config.listenPort = config.listenPortInvalid

   @staticmethod
   def _handleNormal( mode, args ):
      SetTransportListenPortCmd._setListenPort( mode, args[ 'PORT' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetTransportListenPortCmd._noListenPort( mode )

RouterBgpSharedModelet.addCommandClass( SetTransportListenPortCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport qos dscp DSCP" command, in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportQosDscpCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport qos dscp DSCP'
   noOrDefaultSyntax = 'bgp transport qos dscp ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'qos': bgpTokens.qos,
         'dscp': bgpTokens.dscp,
         'DSCP': bgpTokens.dscpRangeMatcher,
   }

   @staticmethod
   def _setDscp( mode, dscp ):
      bgpConfig.dscp = dscp

   @staticmethod
   def _noDscp( mode ):
      bgpConfig.dscp = bgpConfig.dscpDefault

   @staticmethod
   def _handleNormal( mode, args ):
      SetTransportQosDscpCmd._setDscp( mode, args[ 'DSCP' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetTransportQosDscpCmd._noDscp( mode )

RouterBgpSharedModelet.addCommandClass( SetTransportQosDscpCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport pmtud [ disabled ]" command in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportPmtudCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport pmtud [ disabled ]'
   noOrDefaultSyntax = 'bgp transport pmtud ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'pmtud': bgpTokens.pathMtuDiscovery, } )

   @staticmethod
   def _setPathMtuDiscovery( mode ):
      config = configForVrf( mode.vrfName )
      config.pathMtuDiscovery = 'isTrue'
      config.pathMtuDiscoverySet = True

   @staticmethod
   def _disabledPathMtuDiscovery( mode ):
      config = configForVrf( mode.vrfName )
      config.pathMtuDiscovery = 'isFalse'
      config.pathMtuDiscoverySet = True

   @staticmethod
   def _defaultPathMtuDiscovery( mode ):
      config = configForVrf( mode.vrfName )
      config.pathMtuDiscovery = config.pathMtuDiscoveryDefault
      config.pathMtuDiscoverySet = False

   @staticmethod
   def _handleNormal( mode, args ):
      SetTransportPmtudCmd._setPathMtuDiscovery( mode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      if noOrDefault == NoOrDefault.DEFAULT:
         SetTransportPmtudCmd._defaultPathMtuDiscovery( mode )
      else:
         SetTransportPmtudCmd._disabledPathMtuDiscovery( mode )

RouterBgpSharedModelet.addCommandClass( SetTransportPmtudCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp confederation identifier <asn>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceConfedIdCmd( CliCommandClass ):
   syntax = 'bgp confederation identifier AS_NUM'
   noOrDefaultSyntax = 'bgp confederation identifier ...'
   data = {
         'bgp': bgpNode,
         'confederation': bgpTokens.confed,
         'identifier': bgpTokens.confedId,
         'AS_NUM': bgpTokens.AsNumCliExpr,
   }

   @staticmethod
   def _setConfedId( mode, asNumber ):
      config = configForVrf( mode.vrfName )
      config.confedId = asNumber

   @staticmethod
   def _noConfedId( mode ):
      config = configForVrf( mode.vrfName )
      config.confedId = 0

   @staticmethod
   def handler( mode, args ):
      SetInstanceConfedIdCmd._setConfedId( mode, args.get( 'AS_NUM' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      SetInstanceConfedIdCmd._noConfedId( mode )

RouterBgpSharedModelet.addCommandClass( SetInstanceConfedIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp confederation peers <asn>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceConfedPeersCmd( CliCommandClass ):
   syntax = 'bgp confederation peers { ASN_MULTIRANGE }'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'confederation': bgpTokens.confed,
         'peers': bgpTokens.confedPeers,
         'ASN_MULTIRANGE': bgpTokens.AsNumMultiRangeExpr,
   }

   @staticmethod
   def _setConfedPeers( mode, asIter ):
      config = configForVrf( mode.vrfName )
      for asn in asIter:
         asnKey = AsNumKey( asn )
         config.confedPeer[ asnKey ] = True

   @staticmethod
   def _noConfedPeers( mode, asIter ):
      config = configForVrf( mode.vrfName )
      for asn in asIter:
         asnKey = AsNumKey( asn )
         del config.confedPeer[ asnKey ]

   @staticmethod
   def handler( mode, args ):
      SetInstanceConfedPeersCmd._setConfedPeers( mode, args.get( 'AS_NUMS' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      SetInstanceConfedPeersCmd._noConfedPeers( mode, args.get( 'AS_NUMS' ) )

RouterBgpSharedModelet.addCommandClass( SetInstanceConfedPeersCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp asn notation {asplain|asdot}" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAsnNotationCmd( BgpCmdBaseClass ):
   syntax = 'bgp asn notation ( asplain | asdot )'
   noOrDefaultSyntax = 'bgp asn notation ...'
   data = {
         'bgp': bgpNode,
         'asn': bgpTokens.asn,
         'notation': bgpTokens.asnNotation,
         'asplain': bgpTokens.asnNotationAsplain,
         'asdot': bgpTokens.asnNotationAsdot,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if args.pop( 'asplain', None ):
         args[ 'ASN_NOTATION' ] = AsnNotation.asnNotationAsplain
      elif args.pop( 'asdot', None ):
         args[ 'ASN_NOTATION' ] = AsnNotation.asnNotationAsdot

   @staticmethod
   def _handleNormal( mode, args ):
      asnConfig.asnNotation = args[ 'ASN_NOTATION' ]

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      asnConfig.asnNotation = asnConfig.asnNotationDefault

RouterBgpSharedModelet.addCommandClass( SetInstanceAsnNotationCmd )

#-------------------------------------------------------------------------------
# "[no|default] (ip|ipv6) access-group <aclName>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpAfServiceAclMixin( object ):
   data = {
         'ip': AclCli.ipKwForServiceAclMatcher,
         'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
         'access-group': AclCli.accessGroupKwMatcher,
         'ACL': AclCli.ipAclNameMatcher,
         'ACL6': AclCli. ip6AclNameMatcher,
         'in': AclCli.inKwMatcher,
   }

   @staticmethod
   def setServiceAcl( mode, aclName=None ):
      setServiceAclCommon( mode, aclName=aclName )

   @staticmethod
   def noServiceAcl( mode, aclName=None ):
      setServiceAclCommon( mode, no=True, aclName=aclName )

   @staticmethod
   def setServiceAclV6( mode, aclName=None ):
      setServiceAclCommon( mode, aclType='ipv6', aclName=aclName )

   @staticmethod
   def noServiceAclV6( mode, aclName=None ):
      setServiceAclCommon( mode, aclType='ipv6', no=True, aclName=aclName )

class SetInstanceServiceAclCmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
   syntax = 'ip access-group ACL [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL ] [ in ]'
   data = BgpAfServiceAclMixin.data.copy()

   @staticmethod
   def _handleNormal( mode, args ):
      BgpAfServiceAclMixin.setServiceAcl( mode, aclName=args[ 'ACL' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      BgpAfServiceAclMixin.noServiceAcl( mode, aclName=args.get( 'ACL' ) )

RouterBgpSharedModelet.addCommandClass( SetInstanceServiceAclCmd )

class SetInstanceServiceAclV6Cmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
   syntax = 'ipv6 access-group ACL6 [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL6 ] [ in ]'
   data = BgpAfServiceAclMixin.data.copy()

   @staticmethod
   def _handleNormal( mode, args ):
      BgpAfServiceAclMixin.setServiceAclV6( mode, aclName=args[ 'ACL6' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      BgpAfServiceAclMixin.noServiceAclV6( mode, aclName=args.get( 'ACL6' ) )

RouterBgpSharedModelet.addCommandClass( SetInstanceServiceAclV6Cmd )

#-------------------------------------------------------------------------------
# "[no|default] maximum-paths PATHS ecmp ECMP_PATHS" command
#-------------------------------------------------------------------------------
def maxEcmpRangeFn( mode ):
   # This code can be called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb, allowing any value in this case and returning
   # the actual hardware supported (routingHardwareStatus.maxEcmp) in the other case
   if routingHardwareStatus is None or routingHardwareStatus.maxEcmp == 0:
      return( 1, 0xffffffff )
   else:
      return( 1, routingHardwareStatus.maxEcmp )

# These matchers must be defined here, to avoid a circular dependency with
# CliToken/RoutingBgp.py
maxPathsRangeMatcher = DynamicIntegerMatcher(
      maxEcmpRangeFn, helpdesc='Value for maximum number of equal cost paths' )
maxPathsEcmpRangeMatcher = DynamicIntegerMatcher(
      maxEcmpRangeFn, helpdesc='Value for maximum number of installed ECMP routes' )

class SetMaxPathsCmd( CliCommandClass ):
   syntax = "maximum-paths NUM_PATHS [ ecmp NUM_ECMP_RTS ]"
   noOrDefaultSyntax = "maximum-paths ..."
   data = {
      'maximum-paths': bgpTokens.maxNumPaths,
      'NUM_PATHS': maxPathsRangeMatcher,
      'ecmp': bgpTokens.ecmp,
      'NUM_ECMP_RTS': maxPathsEcmpRangeMatcher,
   }

   @staticmethod
   def _noMaxNumPaths( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      # The "no maximum-paths" command operates differently from the
      # "default maximum-paths" command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets maxPaths to the system default,
      # which will be 1, and for maxEcmp will be depend on the hardware.
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.maxPaths = config.maxPathsInvalid
         config.maxEcmp = config.maxEcmpInvalid
      else:
         config.maxPaths = config.maxPathsDefault
         config.maxEcmp = routingHardwareStatus.maxEcmp

   @staticmethod
   def _maxNumPaths( mode, maxPaths, maxEcmp ):
      config = configForVrf( mode.vrfName )
      currMaxEcmp = config.maxEcmp
      if ( ( maxEcmp and maxPaths > maxEcmp ) or
           ( maxEcmp is None and
             currMaxEcmp != config.maxEcmpInvalid and
             currMaxEcmp > 0 and
             maxPaths > currMaxEcmp ) ):
         mode.addError(
            "maximum-paths should be less than or equal to max ECMP" )
         return
      if config.maxPaths != maxPaths:
         config.maxPaths = getExplicitDefaultValue( mode, 'maxPaths', maxPaths )
      if maxEcmp and config.maxEcmp != maxEcmp:
         config.maxEcmp = getExplicitDefaultValue( mode, 'maxEcmp', maxEcmp )

   @staticmethod
   def handler( mode, args ):
      SetMaxPathsCmd._maxNumPaths(
            mode,
            args.get( 'NUM_PATHS' ),
            args.get( 'NUM_ECMP_RTS' ) )

   @staticmethod
   def noHandler( mode, args ):
      SetMaxPathsCmd._noMaxNumPaths( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetMaxPathsCmd._noMaxNumPaths( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( SetMaxPathsCmd )

#-------------------------------------------------------------------------------
# [no|default] default-route export VPN [route-map MAP_NAME] [always]
#-------------------------------------------------------------------------------
class SetVrfVpnDefaultExportCmd( BgpCmdBaseClass ):
   syntax = ( 'default-route export VPN '
              '[ { always | ( route-map MAP_NAME ) } | disabled ]' )
   noOrDefaultSyntax = 'default-route export VPN ...'
   data = BgpCmdBaseClass._createSyntaxData(
         { 'default-route': bgpTokens.defaultRoute,
            'export': Node( bgpTokens.export, maxMatches=1 ),
            'VPN': bgpTokens.vpnMatcher,
            'always': Node( bgpTokens.defaultRouteExportAlways, maxMatches=1 ),
            'route-map': Node( bgpTokens.routeMap, maxMatches=1 ),
            'MAP_NAME': Node( mapNameMatcher, maxMatches=1 )
         } )

   @staticmethod
   def _validateVpnDefaultExportCommand( mode, vpnType ):
      ''' This function validates the route target command to check if the vpnType
          is allowed in the mode '''
      if not vpnTypeTokenAllowedInMode( vpnType, mode ):
         mode.addError( '%s argument is not allowed under address-family %s' % (
                        vpnType, mode.addrFamily ) )
         raise AlreadyHandledError()

   @staticmethod
   def _conflictingVpnDefaultExportConfigs( mode, config, addrFamily, vpnType ):
      ''' This function detects conflicting configuration 
          for "default-route export <vpnType> <rm>" commands
      '''
      defaultExportAttrs = bgpConfigAttrsAfMap[ 'defaultExport' ]
      conflictingAttrs = getConflictingAttrsForAf( defaultExportAttrs, addrFamily )
      vpnAf = vpnAfTypeMapInv[ vpnType ]
      for otherAf, c in conflictingAttrs:
         if vpnAf in getattr( config, c ):
            errorMsg = ( "Cannot configure route-map %s in mode '%s' "
                         "while it is configured in mode '%s'"
                         % ( vpnType, configModeCmdForAf( addrFamily ),
                           configModeCmdForAf( otherAf ) ) )
            mode.addError( errorMsg )
            raise AlreadyHandledError()

   @staticmethod
   def _setVpnDefaultExport( mode, vpnType, defaultExportArgs ):
      ''' Handler for "default-route export <vpnType> <rm>" commands '''

      defaultExportDict = dict( defaultExportArgs if defaultExportArgs else [] )
      rm = defaultExportDict.get( "mapName" )
      always = defaultExportDict.get( "always" )
      addrFamily = mode.addrFamily

      SetVrfVpnDefaultExportCmd._validateVpnDefaultExportCommand( mode, vpnType )

      config = configForVrf( mode.vrfName )
      SetVrfVpnDefaultExportCmd._conflictingVpnDefaultExportConfigs(
            mode, config, addrFamily, vpnType )

      attr = bgpConfigAttrsAfMap[ 'defaultExport' ].get( addrFamily )
      defaultExportCfg = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]

      defaultExportVpnCfg = defaultExportCfg.get( vpnAf )
      if defaultExportVpnCfg is None:
         defaultExportVpnCfg = defaultExportCfg.newMember( vpnAf )
      if rm is None:
         defaultExportVpnCfg.routeMap = defaultExportVpnCfg.routeMapDefault
      else:
         defaultExportVpnCfg.routeMap = rm
      defaultExportVpnCfg.always = always

   @staticmethod
   def _noVpnDefaultExport( mode, vpnType ):
      ''' Handler for "default-route export <vpnType> <rm>" commands '''
      config = configForVrf( mode.vrfName )
      addrFamily = mode.addrFamily

      attr = bgpConfigAttrsAfMap[ 'defaultExport' ].get( addrFamily )
      defaultExportCfg = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]

      del defaultExportCfg[ vpnAf ]

   @staticmethod
   def adapter( mode, args, argsList ):
      exportDefOrigin = set()

      exportDefOrigin.add( ( 'always', 'always' in args ) )

      if 'route-map' in args:
         exportDefOrigin.add( ( 'mapName', args.get( 'MAP_NAME' ) ) )

      args[ 'EXPORT_DEF_ORIGIN' ] = exportDefOrigin

   @staticmethod
   def _handleNormal( mode, args ):
      SetVrfVpnDefaultExportCmd._setVpnDefaultExport(
            mode, args[ 'VPN' ], args.get( 'EXPORT_DEF_ORIGIN' ) )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetVrfVpnDefaultExportCmd._noVpnDefaultExport( mode, args[ 'VPN' ] )

#-------------------------------------------------------------------------------
# [no|default] default-route export VPN [route-map MAP_NAME] [always]
#     in "router-bgp-vrf-af" mode
#-------------------------------------------------------------------------------
RouterBgpVrfSharedModelete.addCommandClass( SetVrfVpnDefaultExportCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( SetVrfVpnDefaultExportCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp always-compare-med" in "router bgp" mode
#---------------------------------------------------------------------------------
class SetInstanceAlwaysCompareMedCmd( CliCommandClass ):
   syntax = 'bgp always-compare-med'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'always-compare-med': bgpTokens.alwaysCompareMed,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.alwaysCompareMed = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.alwaysCompareMed = getExplicitDefaultTristate(
            mode,
            'alwaysCompareMed'
      )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.alwaysCompareMed = 'isInvalid'

RouterBgpSharedModelet.addCommandClass( SetInstanceAlwaysCompareMedCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp bestpath med ( missing-as-worst | confed [ missing-as-worst ] )"
# in "router bgp" mode
#---------------------------------------------------------------------------------
class SetBestPathMedCmd( CliCommandClass ):
   syntax = 'bgp bestpath med ( missing-as-worst | confed [ missing-as-worst ] )'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'med': bgpTokens.med,
         'missing-as-worst': bgpTokens.missingAsWorst,
         'confed': bgpTokens.medConfed,
   }

   @staticmethod
   def _setBestPathMed( mode, missingAsWorst, confed, noOrDefault=None ):
      config = configForVrf( mode.vrfName )
      if noOrDefault:
         if missingAsWorst and confed:
            if noOrDefault == NoOrDefault.DEFAULT:
               config.medConfed = 'isInvalid'
               config.medConfedMissingAsWorst = 'isInvalid'
            else:
               config.medConfed = getExplicitDefaultTristate( mode, 'medConfed' )
               config.medConfedMissingAsWorst = getExplicitDefaultTristate(
                     mode,
                     'medConfedMissingAsWorst'
               )
         elif missingAsWorst:
            if noOrDefault == NoOrDefault.DEFAULT:
               config.medMissingAsWorst = 'isInvalid'
            else:
               config.medMissingAsWorst = getExplicitDefaultTristate(
                     mode,
                     'medMissingAsWorst'
               )
         elif confed:
            if noOrDefault == NoOrDefault.DEFAULT:
               config.medConfed = 'isInvalid'
               config.medConfedMissingAsWorst = 'isInvalid'
            else:
               config.medConfed = getExplicitDefaultTristate( mode, 'medConfed' )
               if config.medConfedMissingAsWorst == 'isTrue':
                  config.medConfedMissingAsWorst = getExplicitDefaultTristate(
                        mode,
                        'medConfedMissingAsWorst'
                  )
      else:
         if missingAsWorst and confed:
            config.medConfed = 'isTrue'
            config.medConfedMissingAsWorst = 'isTrue'
         elif missingAsWorst:
            config.medMissingAsWorst = 'isTrue'
         elif confed:
            config.medConfed = 'isTrue'
            config.medConfedMissingAsWorst = 'isInvalid'

   @staticmethod
   def handler( mode, args ):
      SetBestPathMedCmd._setBestPathMed(
            mode,
            ( 'missing-as-worst' in args ),
            ( 'confed' in args )
      )

   @staticmethod
   def noHandler( mode, args ):
      SetBestPathMedCmd._setBestPathMed(
            mode,
            ( 'missing-as-worst' in args ),
            ( 'confed' in args ),
            noOrDefault=NoOrDefault.NO,
      )

   @staticmethod
   def defaultHandler( mode, args ):
      SetBestPathMedCmd._setBestPathMed(
            mode,
            ( 'missing-as-worst' in args ),
            ( 'confed' in args ),
            noOrDefault=NoOrDefault.DEFAULT,
      )

RouterBgpSharedModelet.addCommandClass( SetBestPathMedCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp convergence [ slow-peer ] time SECONDS"
# in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetConvergenceTimeCmd( CliCommandClass ):
   syntax = 'bgp convergence [ slow-peer ] time SECONDS'
   noOrDefaultSyntax = 'bgp convergence [ slow-peer ] time ...'
   data = {
      'bgp': bgpNode,
      'convergence': bgpTokens.convergence,
      'slow-peer': bgpTokens.convergenceSlowPeer,
      'time': bgpTokens.convergenceTime,
      'SECONDS': bgpTokens.secondsRangeMatcher,
   }

   @staticmethod
   def _setConvergenceTime( mode, seconds, slowPeer=False ):
      config = configForVrf( mode.vrfName )
      if slowPeer:
         config.convergenceIdlePeerTime = getExplicitDefaultValue(
               mode,
               'convergenceIdlePeerTime',
               seconds )
      else:
         config.convergenceTime = getExplicitDefaultValue(
               mode,
               'convergenceTime',
               seconds )

   @staticmethod
   def _noConvergenceTime( mode, noOrDefault, slowPeer=False ):
      # The "no bgp convergence time" command operates differently from the
      # "default bgp..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      config = configForVrf( mode.vrfName )
      if slowPeer:
         if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
            config.convergenceIdlePeerTime = config.convergenceIdlePeerTimeInvalid
         else:
            config.convergenceIdlePeerTime = config.convergenceIdlePeerTimeDefault
      else:
         if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
            config.convergenceTime = config.convergenceTimeInvalid
         else:
            config.convergenceTime = config.convergenceTimeDefault

   @staticmethod
   def handler( mode, args ):
      SetConvergenceTimeCmd._setConvergenceTime(
            mode,
            args.get( 'SECONDS' ),
            slowPeer=( 'slow-peer' in args ) )

   @staticmethod
   def noHandler( mode, args ):
      SetConvergenceTimeCmd._noConvergenceTime(
            mode,
            NoOrDefault.NO,
            slowPeer=( 'slow-peer' in args ) )

   @staticmethod
   def defaultHandler( mode, args ):
      SetConvergenceTimeCmd._noConvergenceTime(
            mode,
            NoOrDefault.DEFAULT,
            slowPeer=( 'slow-peer' in args ) )

RouterBgpSharedModelet.addCommandClass( SetConvergenceTimeCmd )

#---------------------------------------------------------------------------------
# "[no|default] update wait-for-convergence
# in "router-bgp" and "router-bgp-af" mode.
#---------------------------------------------------------------------------------
class SetUpdateWaitForConvergenceCmd( CliCommandClass ):
   syntax = 'update wait-for-convergence'
   noOrDefaultSyntax = syntax
   data = {
      'update': bgpTokens.update,
      'wait-for-convergence': bgpTokens.waitForConvergence,
   }

   @staticmethod
   def _setWaitForConvergence( mode ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'convergenceNoSync' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, 'isFalse' )

   @staticmethod
   def _noWaitForConvergence( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'convergenceNoSync' ].get( mode.addrFamily )
      if attr:
         if noOrDefault == NoOrDefault.DEFAULT:
            setattr( config, attr, 'isInvalid' )
         else:
            if mode.addrFamily == 'all':
               setattr(
                     config,
                     attr,
                     getExplicitDefaultTristate(
                           mode,
                           'convergenceNoSync' ) )
            else:
               setattr( config, attr, 'isTrue' )

   @staticmethod
   def handler( mode, args ):
      SetUpdateWaitForConvergenceCmd._setWaitForConvergence( mode )

   @staticmethod
   def noHandler( mode, args ):
      SetUpdateWaitForConvergenceCmd._noWaitForConvergence(
            mode,
            NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetUpdateWaitForConvergenceCmd._noWaitForConvergence(
            mode,
            NoOrDefault.DEFAULT )

for m in [ RouterBgpAfLabelSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpSharedModelet, ]:
   m.addCommandClass( SetUpdateWaitForConvergenceCmd )


#---------------------------------------------------------------------------------
# "[no|default] bgp peer-mac-resolution-timeout <peer-mac-resolution-timeout>
# in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetInstancePeerMacResolutionTimeoutCmd( CliCommandClass ):
   syntax = 'bgp peer-mac-resolution-timeout TIMEOUT'
   noOrDefaultSyntax = 'bgp peer-mac-resolution-timeout ...'
   data = {
         'bgp': bgpNode,
         'peer-mac-resolution-timeout': bgpTokens.peerMacResolutionTimeout,
         'TIMEOUT': IntegerMatcher(
               0,
               600,
               helpdesc='Time to wait for proactive resolution'
         ),
   }
   hidden = True

   @staticmethod
   def _noBgpPeerMacResolutionTimeout( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      # The "no peer-mac..." command operates differently from
      # the "default peer-mac..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         config.peerMacResolutionTimeout = config.peerMacResolutionTimeoutInvalid
      else:
         config.peerMacResolutionTimeout = config.peerMacResolutionTimeoutDefault

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.peerMacResolutionTimeout = getExplicitDefaultValue(
            mode,
            'peerMacResolutionTimeout',
            args[ 'TIMEOUT' ]
      )

   @staticmethod
   def noHandler( mode, args ):
      SetInstancePeerMacResolutionTimeoutCmd._noBgpPeerMacResolutionTimeout(
            mode,
            NoOrDefault.NO
      )

   @staticmethod
   def defaultHandler( mode, args ):
      SetInstancePeerMacResolutionTimeoutCmd._noBgpPeerMacResolutionTimeout(
            mode,
            NoOrDefault.DEFAULT
      )

RouterBgpSharedModelet.addCommandClass( SetInstancePeerMacResolutionTimeoutCmd )

#---------------------------------------------------------------------------------
# [no|default] bgp host-routes fib direct-install under "config-bgp" mode
#
# This command does nothing, and only exists for update-path sustainability.
# See http://cl/2620911.
#---------------------------------------------------------------------------------
class HostRoutesFibDirectInstallCmd( CliCommandClass ):
   syntax = 'bgp host-routes fib direct-install'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'host-routes': bgpTokens.hostRoutes,
         'fib': bgpTokens.fib,
         'direct-install': bgpTokens.directInstall,
   }

   @staticmethod
   def handler( mode, args ):
      pass

   noOrDefaultHandler = handler

RouterBgpBaseMode.addCommandClass( HostRoutesFibDirectInstallCmd )

#-----------------------------------------------------------------------------
# "[no|default] bgp route-reflector preserve-attributes [always]"
# in "config-bgp" mode.
#-----------------------------------------------------------------------------
class SetInstancePreserveAttrsCmd( CliCommandClass ):
   syntax = 'bgp route-reflector preserve-attributes [ always ]'
   noOrDefaultSyntax = 'bgp route-reflector preserve-attributes ...'
   data = {
         'bgp': bgpNode,
         'route-reflector': bgpTokens.routeReflector,
         'preserve-attributes': bgpTokens.preserveAttrs,
         'always': bgpTokens.always,
   }

   @staticmethod
   def _setPreserveAttrs( mode, always=False ):
      config = configForVrf( mode.vrfName )
      if always:
         config.reflectedRouteAttr = ReflectedRouteAttrStateEnum.alwaysPreserved
      else:
         config.reflectedRouteAttr = (
               ReflectedRouteAttrStateEnum.preservedExceptRouteMap
         )

   @staticmethod
   def handler( mode, args ):
      SetInstancePreserveAttrsCmd._setPreserveAttrs(
            mode,
            always=( 'always' in args )
      )

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.reflectedRouteAttr = ReflectedRouteAttrStateEnum.notPreserved

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.reflectedRouteAttr = config.reflectedRouteAttrDefault

RouterBgpSharedModelet.addCommandClass( SetInstancePreserveAttrsCmd )

#-------------------------------------------------------------------------------
# "[no|default] default-metric <metric>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceDefaultMetricCmd( BgpCmdBaseClass ):
   syntax = 'default-metric METRIC'
   noOrDefaultSyntax = 'default-metric ...'
   data = {
         'default-metric': bgpTokens.defaultMetric,
         'METRIC': IntegerMatcher( 0, U32_MAX_VALUE, helpdesc='Default metric' ),
   }

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      config.defaultMetric = args[ 'METRIC' ]
      config.defaultMetricPresent = True

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      config.defaultMetric = config.defaultMetricDefault
      config.defaultMetricPresent = False

RouterBgpSharedModelet.addCommandClass( SetInstanceDefaultMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp mode <mode#> <maxUcmp> <deviation>" command
#-------------------------------------------------------------------------------
def maxUcmpRangeFn( mode ):
   # This code can be called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb, allowing any value in this case and returning
   # the actual hardware supported (routingHardwareStatus.maxUcmp) in the other case
   if routingHardwareStatus is None or routingHardwareStatus.maxUcmp == 0:
      return( 1, 0xffffffff )
   else:
      return( 1, routingHardwareStatus.maxUcmp )

ucmpSizeRange = DynamicIntegerMatcher(
      maxUcmpRangeFn,
      helpdesc='Value for total number UCMP nexthops'
)

class UcmpModeCmd( CliCommandClass ):
   syntax = 'ucmp mode MODE [ MAX [ DEV ] ]'
   noOrDefaultSyntax = 'ucmp mode ...'

   data = {
         'ucmp': ucmpNode,
         'mode': bgpTokens.ucmpMode,
         'MODE': bgpTokens.ucmpModeRangeMatcher,
         'MAX': ucmpSizeRange,
         'DEV': bgpTokens.ucmpDevRangeMatcher,
   }

   @staticmethod
   def _setUcmp( mode, ucmpMode, size=None, deviation=None ):
      ucmpConfig = ucmpConfigForVrf( mode.vrfName )
      # Historically, we stored a ucmp "mode" number with a mode of "0"
      # indicating that UCMP is disabled. The CLI only supports a single mode
      # ( "ucmp mode 1 ..." ) and we have no plans to introduce new modes in the
      # future. So we ignore the ucmpMode value here and simply "enable" UCMP.
      ucmpConfig.ucmpEnabled = 'isTrue'
      if size:
         ucmpConfig.maxUcmp = size
      else:
         ucmpConfig.maxUcmp = ucmpConfig.maxUcmpInvalid

      if deviation:
         ucmpConfig.ucmpDeviation = deviation
      else:
         ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationInvalid

   @staticmethod
   def _noUcmp( mode, noOrDefault ):
      ucmpConfig = ucmpConfigForVrf( mode.vrfName )
      # The "no ucmp mode ..." command operates differently from
      # the "default ucmp mode..." command for the non-default VRF.
      # The 'default' option is the same as unconfigured, which will cause
      # the non-default vrf to use the value configured in the default vrf.
      # The 'no' option explicitly sets parameters to the system defaults.
      if mode.vrfName == DEFAULT_VRF or noOrDefault == NoOrDefault.DEFAULT:
         ucmpConfig.ucmpEnabled = 'isInvalid'
         ucmpConfig.maxUcmp = ucmpConfig.maxUcmpInvalid
         ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationInvalid
      else:
         ucmpConfig.ucmpEnabled = 'isFalse'
         ucmpConfig.maxUcmp = ucmpConfig.maxUcmpDefault
         ucmpConfig.ucmpDeviation = ucmpConfig.ucmpDeviationDefault

   @staticmethod
   def handler( mode, args ):
      UcmpModeCmd._setUcmp(
            mode, args[ 'MODE' ],
            size=args.get( 'MAX' ),
            deviation=args.get( 'DEV' )
      )

   @staticmethod
   def noHandler( mode, args ):
      UcmpModeCmd._noUcmp( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      UcmpModeCmd._noUcmp( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( UcmpModeCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp link-bandwidth update-delay <num>" command
# "[no|default] ucmp link-bandwidth encoding-weighted" command
# "[no|default] ucmp link-bandwidth recursive" command
#-------------------------------------------------------------------------------
class UcmpLinkBandwidthCmd( CliCommandClass ):
   syntax = (
         'ucmp link-bandwidth ( recursive | encoding-weighted | '
         '( update-delay DELAY ) )'
   )
   noOrDefaultSyntax = (
         'ucmp link-bandwidth ( recursive | encoding-weighted | update-delay ) ...'
   )
   data = {
         'ucmp': ucmpNode,
         'link-bandwidth': bgpTokens.ucmpLinkbw,
         'recursive': bgpTokens.ucmpRecursive,
         'encoding-weighted': bgpTokens.ucmpEncodingWeighted,
         'update-delay': bgpTokens.ucmpLinkbwDelay,
         'DELAY': bgpTokens.ucmpDelayRangeMatcher,
   }

   @staticmethod
   def _setUcmpLinkBandwidth(
         mode,
         recursive=False,
         encodeWeight=False,
         delay=None
   ):
      config = ucmpConfigForVrf( mode.vrfName )
      if recursive:
         config.ucmpLinkbwRecursive = 'isTrue'
      elif encodeWeight:
         config.ucmpLinkbwWeighted = 'isTrue'
      elif delay is not None: # Check against None here as delay may be 0
         config.ucmpLinkbwDelay = delay

   @staticmethod
   def _noUcmpLinkBandwidth(
         mode,
         noOrDefualt,
         recursive=False,
         encodeWeight=False,
         delay=False
   ):
      config = ucmpConfigForVrf( mode.vrfName )
      val = 'isInvalid' if noOrDefualt == NoOrDefault.DEFAULT else 'isFalse'
      if encodeWeight:
         config.ucmpLinkbwWeighted = val
      elif recursive:
         config.ucmpLinkbwRecursive = val
      elif delay:
         # The "no ucmp link-..." command operates differently from
         # the "default ucmp link-..." command for the non-default VRF.
         # The 'default' option is the same as unconfigured, which will cause
         # the non-default vrf to use the value configured in the default vrf.
         # The 'no' option explicitly sets parameters to the system defaults.
         if mode.vrfName == DEFAULT_VRF or noOrDefualt == NoOrDefault.DEFAULT:
            config.ucmpLinkbwDelay = config.ucmpLinkbwDelayInvalid
         else:
            config.ucmpLinkbwDelay = config.ucmpLinkbwDelayDefault

   @staticmethod
   def handler( mode, args ):
      UcmpLinkBandwidthCmd._setUcmpLinkBandwidth(
            mode,
            recursive=( 'recursive' in args ),
            encodeWeight=( 'encoding-weighted' in args ),
            delay=args.get( 'DELAY' )
      )

   @staticmethod
   def noHandler( mode, args ):
      UcmpLinkBandwidthCmd._noUcmpLinkBandwidth(
            mode,
            NoOrDefault.NO,
            recursive=( 'recursive' in args ),
            encodeWeight=( 'encoding-weighted' in args ),
            delay=( 'update-delay' in args )
      )

   @staticmethod
   def defaultHandler( mode, args ):
      UcmpLinkBandwidthCmd._noUcmpLinkBandwidth(
            mode,
            NoOrDefault.DEFAULT,
            recursive=( 'recursive' in args ),
            encodeWeight=( 'encoding-weighted' in args ),
            delay=( 'update-delay' in args )
      )

RouterBgpSharedModelet.addCommandClass( UcmpLinkBandwidthCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp fec threshold trigger <%> clear <%> warning-only" command
#-------------------------------------------------------------------------------
class UcmpFecThresholdCmd( CliCommandClass ):
   syntax = 'ucmp fec threshold trigger PERCENT clear CLR_PERCENT warning-only'
   noOrDefaultSyntax = 'ucmp fec threshold ...'
   data = {
         'ucmp': ucmpNode,
         'fec': bgpTokens.ucmpFec,
         'threshold': bgpTokens.ucmpThreshold,
         'trigger': bgpTokens.ucmpTrigger,
         'PERCENT': bgpTokens.ucmpFecPercentTrigMatcher,
         'clear': bgpTokens.ucmpClear,
         'CLR_PERCENT': bgpTokens.ucmpFecPercentClearMatcher,
         'warning-only': bgpTokens.ucmpWarnOnly
   }

   @staticmethod
   def _setUcmpFecThresholds(
         mode,
         ucmpFecTrigPercentage=None,
         ucmpFecClrPercentage=None
   ):
      config = ucmpConfigForVrf( mode.vrfName )
      if (
            ucmpFecTrigPercentage != config.ucmpTriggerFecThresholdInvalid and
            ucmpFecClrPercentage != config.ucmpClearFecThresholdInvalid and
            ucmpFecTrigPercentage < ucmpFecClrPercentage
      ):
         mode.addError(
               'clear threshold percentage needs to be less than '
               'or equal to trigger threshold percentage '
         )
         return
      config.ucmpTriggerFecThreshold = ucmpFecTrigPercentage
      config.ucmpClearFecThreshold = ucmpFecClrPercentage

   @staticmethod
   def _noUcmpFecThresholds( mode, noOrDefault ):
      config = ucmpConfigForVrf( mode.vrfName )
      if config:
         if noOrDefault == NoOrDefault.DEFAULT:
            # by default, ucmp thresholds are enabled with default values
            config.ucmpTriggerFecThreshold = config.ucmpTriggerFecThresholdDefault
            config.ucmpClearFecThreshold = config.ucmpClearFecThresholdDefault
         else:
            config.ucmpTriggerFecThreshold = config.ucmpTriggerFecThresholdInvalid
            config.ucmpClearFecThreshold = config.ucmpClearFecThresholdInvalid

   @staticmethod
   def handler( mode, args ):
      UcmpFecThresholdCmd._setUcmpFecThresholds(
            mode,
            ucmpFecTrigPercentage=args[ 'PERCENT' ],
            ucmpFecClrPercentage=args[ 'CLR_PERCENT' ]
      )

   @staticmethod
   def noHandler( mode, args ):
      UcmpFecThresholdCmd._noUcmpFecThresholds( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      UcmpFecThresholdCmd._noUcmpFecThresholds( mode, NoOrDefault.DEFAULT )

RouterBgpSharedModelet.addCommandClass( UcmpFecThresholdCmd )

#--------------------------------------------------------------------------
# "bgp next-hop-unchanged" command
#--------------------------------------------------------------------------
class SetInstanceNexthopUnchangedCmd( CliCommandClass ):
   syntax = 'bgp next-hop-unchanged'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'next-hop-unchanged': bgpTokens.nexthopUnchanged,
   }

   @staticmethod
   def _noNexthopUnchanged( mode, noOrDefault ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( mode.addrFamily )
      if attr:
         if noOrDefault == NoOrDefault.DEFAULT:
            setattr( config, attr, 'isInvalid' )
         else:
            setattr( config, attr, 'isFalse' )

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ 'nexthopUnchanged' ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, 'isTrue' )

   @staticmethod
   def noHandler( mode, args ):
      SetInstanceNexthopUnchangedCmd._noNexthopUnchanged( mode, NoOrDefault.NO )

   @staticmethod
   def defaultHandler( mode, args ):
      SetInstanceNexthopUnchangedCmd._noNexthopUnchanged( mode, NoOrDefault.DEFAULT )

for m in [
      RouterBgpSharedModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfLabelSharedModelet,
]:
   m.addCommandClass( SetInstanceNexthopUnchangedCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp next-hop address-family ipv6" command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# "[no|default] bgp next-hop address-family ipv6" command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
class SetInstanceNextHopAfIpv6Cmd( BgpCmdBaseClass ):
   syntax = 'bgp next-hop address-family ipv6'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'next-hop': bgpTokens.nextHop,
         'address-family': bgpTokens.nextHopAf,
         'ipv6': bgpTokens.ipv6,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      config = configForVrf( mode.vrfName )
      if mode.addrFamily == 'ipv4':
         config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isEnabled
      elif mode.addrFamily == 'ipv6':
         mode.addError(
               'IPv6 nexthop address family may not be specified in IPv6 mode'
         )
      else:
         raise ValueError()

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      if mode.addrFamily == 'ipv4':
         config.v6NextHopAfV4Uni = ExtendedNextHopCapabilityEnum.isDisabled
      elif mode.addrFamily == 'ipv6':
         mode.addError(
               'IPv6 nexthop address family may not be specified in IPv6 mode'
         )
      else:
         raise ValueError()

RouterBgpAfIpUniSharedModelet.addCommandClass( SetInstanceNextHopAfIpv6Cmd )

#-----------------------------------------------------------------------------
# "[no|default] aigp-session ( ibgp | ebgp | confederation )" command, in
# "config-bgp-af" mode
#-----------------------------------------------------------------------------
class AigpSessionAfCommand( CliCommandClass ):
   syntax = "aigp-session PEER_TYPE"
   noOrDefaultSyntax = syntax
   data = {
      'aigp-session': bgpTokens.aigpSession,
      'PEER_TYPE': bgpTokens.aigpPeerType,
   }

   # Note that the default state is enabled on ibgp/confed, disabled on ebgp.
   # The attributes reflect this.
   _aigpAttrMap = {
      'ibgp': 'aigpSessionIbgpDisabled',
      'confederation': 'aigpSessionConfedDisabled',
      'ebgp': 'aigpSessionEbgpEnabled',
   }

   @staticmethod
   def _setAigpSessionAf( mode, peerType, val ):
      config = configForVrf( mode.vrfName )
      attrname = AigpSessionAfCommand._aigpAttrMap[ peerType ]
      attr = bgpConfigAttrsAfMap[ attrname ].get( mode.addrFamily )
      if attr:
         setattr( config, attr, val )

   @staticmethod
   def handler( mode, args ):
      # The attributes are enabling/disabling depending on the peer type.
      # Set the TristateBool value appropriately.
      value = 'isTrue' if args[ 'PEER_TYPE' ] == 'ebgp' else 'isFalse'
      AigpSessionAfCommand._setAigpSessionAf( mode, args[ 'PEER_TYPE' ], value )

   @staticmethod
   def noHandler( mode, args ):
      value = 'isFalse' if args[ 'PEER_TYPE' ] == 'ebgp' else 'isTrue'
      AigpSessionAfCommand._setAigpSessionAf( mode, args[ 'PEER_TYPE' ], value )

   @staticmethod
   def defaultHandler( mode, args ):
      AigpSessionAfCommand._setAigpSessionAf( mode, args[ 'PEER_TYPE' ],
                                              'isInvalid' )

if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
   RouterBgpAfIpUniSharedModelet.addCommandClass( AigpSessionAfCommand )

#---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "address-family vpn-ipv4" and
# "address-family vpn-ipv6" modes
#---------------------------------------------------------------------------------
class NexthopResolutionDisabledCmd( CliCommandClass ):
   syntax = 'next-hop resolution disabled'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'disabled': bgpTokens.disabledKwMatcherForNhResolution,
   }
   afiSafisSupported_ = {
      'ipv4': 'afV4NexthopResolutionDisabled',
      'ipv6': 'afV6NexthopResolutionDisabled',
      'vpn-ipv4': 'afVpnV4NexthopResolutionDisabled',
      'vpn-ipv6': 'afVpnV6NexthopResolutionDisabled',
      'evpn': 'afEvpnNexthopResolutionDisabled',
   }

   vpnAfiSafis_ = [ 'vpn-ipv4', 'vpn-ipv6', 'evpn' ]

   @staticmethod
   def _setNextHopResolutionDisabled( afiSafi, vrfName=DEFAULT_VRF ):
      if afiSafi in NexthopResolutionDisabledCmd.vpnAfiSafis_:
         assert vrfName == DEFAULT_VRF
      config = configForVrf( vrfName )
      setattr( config, NexthopResolutionDisabledCmd.afiSafisSupported_[
         afiSafi ], True )

   @staticmethod
   def _noNextHopResolutionDisabled( afiSafi, vrfName=DEFAULT_VRF ):
      if afiSafi in NexthopResolutionDisabledCmd.vpnAfiSafis_:
         assert vrfName == DEFAULT_VRF
      config = configForVrf( vrfName )
      setattr( config, NexthopResolutionDisabledCmd.afiSafisSupported_[
         afiSafi ], False )

   @staticmethod
   def handler( mode, args ):
      NexthopResolutionDisabledCmd._setNextHopResolutionDisabled( mode.addrFamily,
                                                                  mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      NexthopResolutionDisabledCmd._noNextHopResolutionDisabled( mode.addrFamily,
                                                                 mode.vrfName )

RouterBgpAfIpUniSharedModelet.addCommandClass( NexthopResolutionDisabledCmd )

#---------------------------------------------------------------------------------
# "[no|default] redistribute <routing protocol> <routemap>" command, in
# "router-bgp" mode.
#---------------------------------------------------------------------------------
class RedistributeBaseCmd( object ):
   syntax = 'redistribute PROTOCOL'

   noOrDefaultSyntax = syntax + ' ...'
   noOrDefaultSyntaxWithMatch = syntax + ' [ MATCH_EXPR ] ...'
   noOrDefaultSyntaxWithNssaMatch = syntax + ' match nssa-external ...'

   @staticmethod
   def makeDataDict( data, needRcf=False ):
      # The data dict of the base class contains all keyword and expression matchers
      # for all derived classes, except PROTOCOL and MATCH_EXPR, which must be
      # defined by the appropriate child classes.
      baseData = {
            'redistribute': bgpTokens.redistribute,
            'route-map': bgpTokens.routeMap,
            'MAP_NAME': mapNameMatcher,
            'include': bgpTokens.redistInclude,
            'leaked': bgpTokens.leaked,
            'ISIS_LEVEL': bgpTokens.IsisLevelExpr,
            'match': bgpTokens.asMatch,
      }
      baseData.update( data )
      if needRcf:
         baseData.update( {
            'rcf': bgpTokens.rcf,
            'FUNCTION': RcfCliLib.rcfFunctionMatcher,
         } )
      return baseData

   @staticmethod
   def _setRedistribute(
         mode,
         proto,
         intext=None,
         mapName=None,
         rcfName=None,
         mType=0,
         isisLevel=None,
         leaked=False
   ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      if addrFamily == 'ipv4 multicast':
         if proto not in [ 'protoDirect', 'protoStatic', 'protoIsis', 'protoOspf',
                           'protoOspfNssa', 'protoOspfAse', 'protoOspf3',
                           'protoOspf3Nssa', 'protoOspf3Ase', 'protoAttachedHost' ]:
            raise ValueError()
      config = configForVrf( vrfName )
      if proto == 'protoBgpAggregate' and mapName:
         mode.addWarning( 'The route-map argument is no longer supported for '
                          'aggregates and has no effect' )
      if not intext:
         intext = ''
      elif intext == 'external':
         if proto == 'protoOspf':
            proto = 'protoOspfAse'
         elif proto == 'protoOspf3':
            proto = 'protoOspf3Ase'
      elif intext == 'nssa-external':
         if proto == 'protoOspf':
            proto = 'protoOspfNssa'
         elif proto == 'protoOspf3':
            proto = 'protoOspf3Nssa'

      level = 0
      if proto == 'protoIsis':
         if isisLevel == 'level-1':
            level = 1
         elif isisLevel == 'level-1-2':
            level = 3
         else:
            level = 2
         if haveConflictingRedistribute( mode, config, proto, addrFamily ):
            return
      elif proto == 'protoOspf3' or proto == 'protoOspf3Nssa' or \
           proto == 'protoOspf3Ase':
         if haveConflictingRedistribute( mode, config, proto, addrFamily ):
            return

      redistConfig = redistributeConfig( addrFamily, config )
      assert not ( mapName and rcfName )  # Only one of them can be enabled/applied
      redistribute = Tac.Value( 'Routing::Bgp::Redistribute',
                                proto, routeType=intext, metricType=mType,
                                routeMap=( mapName or '' ),
                                rcf=( rcfName or '' ), isisLevel=level,
                                includeLeaked=leaked )
      redistConfig.addMember( redistribute )

   @staticmethod
   def _noRedistribute( mode, proto, intext='' ):
      vrfName = mode.vrfName
      addrFamily = mode.addrFamily
      config = configForVrf( vrfName )
      if intext == 'external':
         if proto == 'protoOspf':
            proto = 'protoOspfAse'
         elif proto == 'protoOspf3':
            proto = 'protoOspf3Ase'
      elif intext == 'nssa-external':
         if proto == 'protoOspf':
            proto = 'protoOspfNssa'
         elif proto == 'protoOspf3':
            proto = 'protoOspf3Nssa'
      redistConfig = redistributeConfig( addrFamily, config )
      del redistConfig[ proto ]

#---------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static | attached-host )
#    [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
class RedistributeMcastCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.McastRedistExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args.get( 'proto' ),
            mapName=args.get( 'MAP_NAME' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, args.get( 'proto' ) )

for m in [
      RouterBgpBaseAfIpMulticastMode,
      RouterBgpVrfAfIpMulticastMode, ]:
   m.addCommandClass( RedistributeMcastCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static ) [ route-map <name> ]
#-------------------------------------------------------------------------------
class RedistributeMcast6Cmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.McastRedist6Expr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args.get( 'proto' ),
            mapName=args.get( 'MAP_NAME' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, args.get( 'proto' ) )

for m in [
      RouterBgpBaseAfIpv6MulticastMode,
      RouterBgpVrfAfIpv6MulticastMode, ]:
   m.addCommandClass( RedistributeMcast6Cmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static ) [ include leaked ]
#    [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
redistConnStatSyntax = RedistributeBaseCmd.syntax + ' [ include leaked ]'
if RcfLibToggleLib.toggleRcfRedistConnectedStaticEnabled():
   redistConnStatSyntax += ' [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ]'
   needRcfForRedist = True
else:
   redistConnStatSyntax += ' [ route-map MAP_NAME ]'
   needRcfForRedist = False
class ConnStatRedistributeProtoCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = redistConnStatSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.ConnectedStaticExpression,
   }, needRcf=needRcfForRedist ) )

   @staticmethod
   def _handleNormal( mode, args ):
      rcfName = args.get( 'FUNCTION' )
      if rcfName:
         # Remove the trailing () characters
         rcfName = re.sub( r'\(\)', '', rcfName )
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            mapName=args.get( 'MAP_NAME' ),
            rcfName=rcfName,
            leaked=( 'leaked' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, args[ 'proto' ] )

RouterBgpSharedModelet.addCommandClass( ConnStatRedistributeProtoCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute ( rip | aggregate | attached-host | dynamic )
#    [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
class RedistributeProtoCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.ProtocolExpression,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            mapName=args.get( 'MAP_NAME' )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, args[ 'proto' ] )

RouterBgpSharedModelet.addCommandClass( RedistributeProtoCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute bgp leaked [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
class RedistributeBgpCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' leaked [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpNode,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            'protoBgp',
            mapName=args.get( 'MAP_NAME' ),
            leaked=True
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, 'protoBgp' )

RouterBgpSharedModelet.addCommandClass( RedistributeBgpCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute isis [ ISIS_LEVEL ] [ include leaked ]
#    [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeIsisCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ ISIS_LEVEL ] [ include leaked ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.isis,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            'protoIsis',
            isisLevel=args.get( 'isisLevel' ),
            mapName=args.get( 'MAP_NAME' ),
            leaked=( 'leaked' in args ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, 'protoIsis' )

for m in [
      RouterBgpSharedModelet,
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfIpMcastSharedModelet,
]:
   m.addCommandClass( RedistributeIsisCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute dhcp [ route-map MAP_NAME ]'
#-------------------------------------------------------------------------------
class RedistributeDhcpCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.dhcp,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            'protoDhcp',
            mapName=args.get( 'MAP_NAME' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute( mode, 'protoDhcp' )

for m in [ RouterBgpVrfAfIpv6Modelet, RouterBgpAfIpv6Modelet ]:
   m.addCommandClass( RedistributeDhcpCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospf3 ) match ( internal | external )
#    [ include leaked ] [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeOspfMatchIntExtLeakedCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ MATCH_EXPR ] [ include leaked ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchIntExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( args.get( 'intext' ) ),
            mapName=args.get( 'MAP_NAME' ),
            leaked=( 'leaked' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( args.get( 'intext' ) )
      )

RouterBgpSharedModelet.addCommandClass( RedistributeOspfMatchIntExtLeakedCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospf3 ) match nssa-external [ 1 | 2 ]
#    [ include leaked ]
#-------------------------------------------------------------------------------
class RedistributeOspfMatchNssaExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' MATCH_EXPR [ include leaked ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithNssaMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchNssaExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external',
            mType=args.get( 'NSSA_TYPE', 0 ),
            leaked=( 'leaked' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external'
      )

RouterBgpSharedModelet.addCommandClass( RedistributeOspfMatchNssaExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospf3 ) match ( internal | external )
#    [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeOspfMatchIntExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ MATCH_EXPR ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchIntExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( 'intext' ),
            mapName=args.get( 'MAP_NAME' ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( 'intext' )
      )

RouterBgpAfIpMcastSharedModelet.addCommandClass( RedistributeOspfMatchIntExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ospf3 match nssa-external [ 1 | 2 ]
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchNssaExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' MATCH_EXPR'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithNssaMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchNssaExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external',
            mType=args.get( 'NSSA_TYPE', 0 ),
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external'
      )

RouterBgpAfIpMcastSharedModelet.addCommandClass( RedistributeOspf3MatchNssaExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ospf3 match ( internal | external ) [ include leaked ]
#    [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchIntExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ MATCH_EXPR ] [ include leaked ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3IncludingHiddenExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchIntExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( 'intext' ),
            mapName=args.get( 'MAP_NAME' ),
            leaked=( 'leaked' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext=args.get( 'intext' )
      )

RouterBgpAfIpUniSharedModelet.addCommandClass( RedistributeOspf3MatchIntExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospf3 ) match nssa-external [ 1 | 2 ]
#    [ include leaked ]
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchNssaExtLeakedCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' MATCH_EXPR [ include leaked ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithNssaMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3IncludingHiddenExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchNssaExtExpr,
   } ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RedistributeBaseCmd._setRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external',
            mType=args.get( 'NSSA_TYPE', 0 ),
            leaked=( 'leaked' in args )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RedistributeBaseCmd._noRedistribute(
            mode,
            args[ 'proto' ],
            intext='nssa-external'
      )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      RedistributeOspf3MatchNssaExtLeakedCmd
)

#-------------------------------------------------------------------------------
# "[no|default] rd <rd>" command
#     in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-------------------------------------------------------------------------------
class SetInstanceRouteDistinguisher( BgpCmdBaseClass ):
   syntax = 'rd RD'
   noOrDefaultSyntax = 'rd ...'
   data = {
         'rd': bgpTokens.rd,
         'RD': RdDistinguisherMatcher( 'BGP route distinguisher' ),
   }

   @staticmethod
   def _handleNormal( mode, args ):
      rd = args[ 'RD' ]
      check = Tac.Value( 'Ira::RdAssignmentChecker',
                         LazyMount.force( rdAutoInputDir ),
                         LazyMount.force( rdConfigInputDir ),
                         mode.vrfName, rd )
      if not check.acceptable:
         errorMsg = check.errorMessage.replace( rd,
                        formatRd( rd, isAsdotConfigured( asnConfig ) ) )
         mode.addError( errorMsg )
         return
      bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
      bgpRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ] = rd

      # For default VRF, write to bgp config as well. IRA doesn't publish VRF status
      # for default VRF, so the config written above at
      # ip/vrf/routeDistinguisherInputDir/config for default VRF will not be read
      # by BGP. We still write there for default VRF so that RdAssignmentChecker
      # can be used to check that the RD is not already in use.
      if mode.vrfName == DEFAULT_VRF:
         bgpConfig.routeDistinguisher = rd

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
      if mode.vrfName in bgpRouteDistinguisherInput.routeDistinguisher:
         del bgpRouteDistinguisherInput.routeDistinguisher[ mode.vrfName ]
      if mode.vrfName == DEFAULT_VRF:
         bgpConfig.routeDistinguisher = 'INVALID'

RouterBgpVrfSharedModelete.addCommandClass( SetInstanceRouteDistinguisher )

#-------------------------------------------------------------------------------
# "[no|default] rd auto" command in "router-bgp" mode
#-------------------------------------------------------------------------------

class RdAutoCommand( CliCommandClass ):
   syntax = 'rd auto'
   noOrDefaultSyntax = syntax
   data = {
      'rd': 'BGP route distinguisher',
      'auto': 'Auto Generate Route Distinguisher'
   }

   @staticmethod
   def handler( mode, args ):
      assert mode.vrfName == DEFAULT_VRF
      config = configForVrf( mode.vrfName )
      config.autoRd = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName == DEFAULT_VRF
      config = configForVrf( mode.vrfName )
      config.autoRd = False

RouterBgpBaseMode.addCommandClass( RdAutoCommand )

#-----------------------------------------------------------------------------------
# "[no|default] route-target ( import|export|both ) [ evpn|vpn-ipv4|vpn-ipv6 ] <rt> "
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
# Note: Usage of [no|default] route-target ( import|export|both ) <rt> is deprecated.
#-----------------------------------------------------------------------------------
class RouteTargetImportExportBaseClass( object ):
   @staticmethod
   def _setRouteTargetCommon( mode, rt, attrName, vpnType=None, warn=True ):
      ''' Handler for "route-target import/export <rt>" commands '''
      # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
      if not validateRouteTargetCommand( mode, vpnType, warn ):
         return
      config = configForVrf( mode.vrfName )
      if conflictingRouteTargetConfigs( mode, config, mode.addrFamily, vpnType ):
         return
      vpnTypeAttr = getVpnTypeAttr( config, attrName, mode.addrFamily, vpnType )
      vpnTypeAttr[ routeTargetToExtCommU64Value( rt ) ] = True

   @staticmethod
   def _noRouteTargetCommon( mode, rt, attrName, vpnType=None, warn=True ):
      ''' Handler for "no route-target import/export <rt>" commands" '''
      # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
      if not validateRouteTargetCommand( mode, vpnType, warn ):
         return
      config = configForVrf( mode.vrfName )
      vpnTypeAttr = getVpnTypeAttr( config, attrName, mode.addrFamily, vpnType )
      if rt is not None:
         extCommValue = routeTargetToExtCommU64Value( rt )
         del vpnTypeAttr[ extCommValue ]
      else:
         vpnTypeAttr.clear()

class RouteTargetImportExportVpnRouteCmd(
      RouteTargetImportExportBaseClass,
      BgpCmdBaseClass
):
   syntax = 'route-target IMPORT_EXPORT [ VPN_TYPE ] RT'
   noOrDefaultSyntax = 'route-target IMPORT_EXPORT [ VPN_TYPE ] [ RT ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'VPN_TYPE': bgpTokens.RouteTargetVpnTypeExpr,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }

   @staticmethod
   def _handleNormal( mode, args ):
      RouteTargetImportExportBaseClass._setRouteTargetCommon(
            mode,
            args[ 'RT' ],
            'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
            vpnType=args.get( 'VPN_TYPE' )
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouteTargetImportExportBaseClass._noRouteTargetCommon(
            mode,
            args.get( 'RT' ),
            'routeTarget' + args[ 'IMPORT_EXPORT' ].title(),
            vpnType=args.get( 'VPN_TYPE' )
      )

RouterBgpVrfSharedModelete.addCommandClass( RouteTargetImportExportVpnRouteCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportExportVpnRouteCmd )

class RouteTargetBothRouteCmd( RouteTargetImportExportBaseClass, BgpCmdBaseClass ):
   syntax = 'route-target both RT'
   noOrDefaultSyntax = 'route-target both [ RT ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'both': bgpTokens.bothImportExport,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }
   hidden = True

   @staticmethod
   def _setBothRouteTarget( mode, rt ):
      RouteTargetImportExportBaseClass._setRouteTargetCommon(
            mode,
            rt,
            'routeTargetImport'
      )
      RouteTargetImportExportBaseClass._setRouteTargetCommon(
            mode,
            rt,
            'routeTargetExport'
      )

   @staticmethod
   def _noBothRouteTarget( mode, rt ):
      RouteTargetImportExportBaseClass._noRouteTargetCommon(
            mode,
            rt,
            'routeTargetImport',
            warn=( not mode.addrFamily == 'all' )
      )
      RouteTargetImportExportBaseClass._noRouteTargetCommon(
            mode,
            rt,
            'routeTargetExport',
            warn=( not mode.addrFamily == 'all' )
      )

   @staticmethod
   def _handleNormal( mode, args ):
      RouteTargetBothRouteCmd._setBothRouteTarget( mode, args[ 'RT' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouteTargetBothRouteCmd._noBothRouteTarget( mode, args.get( 'RT' ) )

RouterBgpVrfMode.addCommandClass( RouteTargetBothRouteCmd )
RouterBgpVrfAfIpMode.addCommandClass( RouteTargetBothRouteCmd )
RouterBgpVrfAfIp6Mode.addCommandClass( RouteTargetBothRouteCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target ( import|export ) ( evpn|vpn-ipv4|vpn-ipv6 ) route-map
# <rm> " command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-----------------------------------------------------------------------------------
class RouteTargetImportExportRouteMapCmd( BgpCmdBaseClass ):
   syntax = 'route-target IMPORT_EXPORT VPN_TYPE route-map MAP'
   noOrDefaultSyntax = 'route-target IMPORT_EXPORT VPN_TYPE route-map [ MAP ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'VPN_TYPE': bgpTokens.RouteTargetVpnTypeExpr,
         'route-map': bgpTokens.routeMap,
         'MAP': mapNameMatcher,
   }

   @staticmethod
   def _setRTRouteMapCommon( mode, rm, attrName, vpnType ):
      ''' Handler for "route-target import/export route-map" commands '''
      if not validateRouteTargetCommand( mode, vpnType, warn=True ):
         return
      config = configForVrf( mode.vrfName )
      if conflictingRouteMapConfigs( mode, config, mode.addrFamily, vpnType ):
         return
      attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
      routeMapConfig = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]
      routeMapConfig[ vpnAf ] = rm

   @staticmethod
   def _noRTRouteMapCommon( mode, rm, attrName, vpnType ):
      ''' Handler for "no route-target import/export route-map" commands" '''
      assert vpnType is not None
      if not validateRouteTargetCommand( mode, vpnType, warn=False ):
         return
      config = configForVrf( mode.vrfName )
      attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
      routeMapConfig = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]
      rmEntry = routeMapConfig.get( vpnAf )
      if rmEntry and ( not rm or rmEntry == rm ):
         del routeMapConfig[ vpnAf ]

   @staticmethod
   def _handleNormal( mode, args ):
      RouteTargetImportExportRouteMapCmd._setRTRouteMapCommon(
            mode,
            args[ 'MAP' ],
            'routeMap' + args[ 'IMPORT_EXPORT' ].title(),
            args[ 'VPN_TYPE' ]
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouteTargetImportExportRouteMapCmd._noRTRouteMapCommon(
            mode,
            args.get( 'MAP' ),
            'routeMap' + args[ 'IMPORT_EXPORT' ].title(),
            args[ 'VPN_TYPE' ]
      )

RouterBgpVrfSharedModelete.addCommandClass( RouteTargetImportExportRouteMapCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportExportRouteMapCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target export ( evpn|vpn-ipv4|vpn-ipv6 ) imported-route"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-----------------------------------------------------------------------------------
class RouteTargetExportBaseCmd( object ):
   data = {
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export,
         'evpn': bgpTokens.evpn,
         'imported-route': bgpTokens.importedRoute,
         'vpn-ipv4': bgpTokens.vpnIpv4,
         'vpn-ipv6': bgpTokens.vpnIpv6,
   }

   @staticmethod
   def _setRTExportImportedRoute( mode, vpnType=None ):
      ''' Handler for "route-target export imported-route" commands '''
      if not validateRouteTargetCommand( mode, vpnType ):
         return
      config = configForVrf( mode.vrfName )
      if conflictingRouteTargetConfigs( mode, config, mode.addrFamily, vpnType ):
         return
      attrName = 'allowImportedRouteToExport'
      attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
      allowImportedRouteToExport = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]
      allowImportedRouteToExport[ vpnAf ] = True

   @staticmethod
   def _noRTExportImportedRoute( mode, vpnType=None ):
      ''' Handler for "no route-target export imported-route" commands '''
      if not validateRouteTargetCommand( mode, vpnType ):
         return
      config = configForVrf( mode.vrfName )
      attrName = 'allowImportedRouteToExport'
      attr = bgpConfigAttrsAfMap[ attrName ].get( mode.addrFamily )
      allowImportedRouteToExport = getattr( config, attr )
      vpnAf = vpnAfTypeMapInv[ vpnType ]
      del allowImportedRouteToExport[ vpnAf ]

class RouteTargetExportEvpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
   syntax = 'route-target export evpn imported-route'
   noOrDefaultSyntax = syntax
   data = RouteTargetExportBaseCmd.data.copy()

   @staticmethod
   def _handleNormal( mode, args ):
      RouteTargetExportBaseCmd._setRTExportImportedRoute( mode, vpnType='evpn' )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouteTargetExportBaseCmd._noRTExportImportedRoute( mode, vpnType='evpn' )

RouterBgpVrfSharedModelete.addCommandClass( RouteTargetExportEvpnCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportEvpnCmd )

class RouteTargetExportAfVpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
   syntax = 'route-target export ( vpn-ipv4 | vpn-ipv6 ) imported-route'
   noOrDefaultSyntax = syntax
   data = RouteTargetExportBaseCmd.data.copy()

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'VPN_TYPE' ] = args.get( 'vpn-ipv4', args.get( 'vpn-ipv6' ) )

   @staticmethod
   def _handleNormal( mode, args ):
      RouteTargetExportBaseCmd._setRTExportImportedRoute(
            mode,
            vpnType=args[ 'VPN_TYPE' ]
      )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouteTargetExportBaseCmd._noRTExportImportedRoute(
            mode,
            vpnType=args[ 'VPN_TYPE' ]
      )

if BgpCommonToggleLib.toggleArBgpAllowImportedRouteToExportMplsVpnEnabled():
   RouterBgpVrfSharedModelete.addCommandClass( RouteTargetExportAfVpnCmd )
   RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportAfVpnCmd )

#---------------------------------------------------------
# The "clear bgp [ipv4|ipv6] access-list counters" command
#---------------------------------------------------------
class ClearBgpAclCounters( CliCommandClass ):
   syntax = 'clear bgp [ ipv4 | ipv6 ] access-list counters'
   data = {
         'clear': Clear.clearKwNode,
         'bgp': 'Bgp',
         'ipv4': AclCli.ipv4KwForServiceAclMatcher,
         'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
         'access-list': AclCli.accessListKwMatcher,
         'counters': bgpTokens.counters,
   }

   @staticmethod
   def handler( mode, args ):
      aclFilter = None
      if 'ipv4' in args:
         aclFilter = 'ip'
      elif 'ipv6' in args:
         aclFilter = 'ipv6'
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclFilter )

BasicCliModes.EnableMode.addCommandClass( ClearBgpAclCounters )

#--------------------------------------------------------------------------------
# "[no|default] next-hop [6pe] resolution ribs
#  ( ( tunnel-rib TUNNEL_PRIMARY [ IP_FALL ] ) |
#    ( IP_PRIMARY [ tunnel-rib TUNNEL_FALL ] ) )"
#
# Where TUNNEL_PRIMARY and TUNNEL_FALL conceptually include 'system-tunnel-rib'
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ADDR_FAMILY'
#--------------------------------------------------------------------------------
class RouterBgpAddrFamilyCommand( BgpCmdBaseClass ):
   syntax = 'address-family ADDR_FAMILY'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ADDR_FAMILY': bgpTokens.AddrFamilyExpr,
   }

   # Map address-family from args to the appropriate goto/delete methods.
   _afModeMap = {
      'ipv4': RouterBgpBaseAfIpUniMode,
      'ipv6': RouterBgpBaseAfIpv6UniMode,
      'ipv4 labeled-unicast': RouterBgpBaseAfLabelMode,
      'ipv6 labeled-unicast': RouterBgpBaseAfLabelMode,
      'ipv4 multicast': RouterBgpBaseAfIpMulticastMode,
      'ipv6 multicast': RouterBgpBaseAfIpv6MulticastMode,
      'ipv4 sr-te': RouterBgpBaseAfSrTeMode,
      'ipv6 sr-te': RouterBgpBaseAfSrTeMode,
      'evpn': RouterBgpBaseAfEvpnMode,
   }

   @staticmethod
   def _gotoAfMode( mode, af ):
      if ( af == 'evpn' and
           ( getEffectiveProtocolModel( mode.session.mode ) ==
             ProtocolAgentModelType.ribd )
      ):
         mode.addWarning( 'Routing protocols model multi-agent must be '
                          'configured for EVPN address-family' )

      childMode = mode.childMode( RouterBgpAddrFamilyCommand._afModeMap.get( af ),
                                  addrFamily=af )

      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _deleteAfMode( mode, af ):
      resetBgpAfModeConfig( bgpConfig, af, mode.vrfName )

   @staticmethod
   def _handleNormal( mode, args ):
      RouterBgpAddrFamilyCommand._gotoAfMode( mode, args[ 'af' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      RouterBgpAddrFamilyCommand._deleteAfMode( mode, args[ 'af' ] )

RouterBgpBaseMode.addCommandClass( RouterBgpAddrFamilyCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family link-state'
#
# Currently there's no requirement to support BGP Link State in
# non-default VRF. So its supported in default vrf mode only
#--------------------------------------------------------------------------------
class RouterBgpLinkStateCommand( BgpCmdBaseClass ):
   syntax = 'address-family link-state'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'link-state': bgpTokens.linkState,
   }

   @staticmethod
   def _handleNormal( mode, args ):
      if ( getEffectiveProtocolModel( mode.session.mode ) ==
           ProtocolAgentModelType.ribd ):
         mode.addWarning( 'Routing protocols model multi-agent must be '
                          'configured for BGP Link State address-family' )
      childMode = mode.childMode( RouterBgpBaseAfLinkStateMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      resetBgpAfModeConfig( bgpConfig, 'link-state', mode.vrfName )
      lsImportConfig.protoConfig.clear()

RouterBgpBaseMode.addCommandClass( RouterBgpLinkStateCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family path-selection'
#
#--------------------------------------------------------------------------------
class RouterBgpDpsCommand( BgpCmdBaseClass ):
   syntax = 'address-family path-selection'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'path-selection': bgpTokens.dps,
   }

   hidden = not BgpCommonToggleLib.toggleBgpDpsEnabled()

   @staticmethod
   def _handleNormal( mode, args ):
      if ( getEffectiveProtocolModel( mode.session.mode ) ==
           ProtocolAgentModelType.ribd ):
         mode.addWarning( 'Routing protocols model multi-agent must be '
                          'configured for BGP path-selection address-family' )
      childMode = mode.childMode( RouterBgpBaseAfDpsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      resetBgpAfModeConfig( bgpConfig, 'path-selection', mode.vrfName )

RouterBgpBaseMode.addCommandClass( RouterBgpDpsCommand )

#---------------------------------------------------------------------------------
# '[ no | default ] address-family rt-membership' config mode
#---------------------------------------------------------------------------------
class RtMembershipAfModeCmd( BgpCmdBaseClass ):
   syntax = 'address-family rt-membership'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'rt-membership': 'Route Target VPN route membership address family'
   }

   @staticmethod
   def _handleNormal( mode, args ):
      if ( getEffectiveProtocolModel( mode.session.mode ) ==
           ProtocolAgentModelType.ribd ):
         mode.addWarning( 'Routing protocols model multi-agent must be '
                          'configured for the RT membership address-family' )
      addrFamily = 'rt-membership'
      assert addrFamily in afModeExtensionHook.afModeExtensions()

      childMode = mode.childMode(
            afModeExtensionHook.afModeExtension[ addrFamily ],
            addrFamily=addrFamily
      )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      resetBgpAfModeConfig( bgpConfig, 'rt-membership', mode.vrfName )

if BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   RouterBgpBaseMode.addCommandClass( RtMembershipAfModeCmd )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ( ipv4 | ipv6 )' in default vrf and vrf modes.
#--------------------------------------------------------------------------------
def afAdapter( mode, args, argsList ):
   args[ 'af' ] = args.get( 'ipv4', args.get( 'ipv6' ) )

class RouterBgpVrfAfCommand( BgpCmdBaseClass ):
   syntax = 'address-family ( ipv4 | ipv6 )'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
   }

   adapter = afAdapter

   @staticmethod
   def _handleNormal( mode, args ):
      if mode.vrfName == DEFAULT_VRF:
         NewMode = RouterBgpDefaultVrfAfMode
      elif args.get( 'ipv4' ):
         NewMode = RouterBgpVrfAfIpMode
      else:
         NewMode = RouterBgpVrfAfIp6Mode

      childMode = mode.childMode(
            NewMode,
            addrFamily=args[ 'af' ],
            vrfName=mode.vrfName
      )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      config = configForVrf( mode.vrfName )
      resetBgpAfModeConfig( config, args[ 'af' ], mode.vrfName )

RouterBgpVrfMode.addCommandClass( RouterBgpVrfAfCommand )
RouterBgpDefaultVrfMode.addCommandClass( RouterBgpVrfAfCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ( ipv4 | ipv6 ) multicast' in default vrf and vrf
# modes.
#--------------------------------------------------------------------------------
class RouterBgpVrfAfMulitcastCommand( BgpCmdBaseClass ):
   syntax = 'address-family ( ipv4 | ipv6 ) multicast'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
         'multicast': bgpTokens.multicast,
   }

   adapter = afAdapter

   @staticmethod
   def _handleNormal( mode, args ):
      addrFamily = args[ 'af' ] + ' multicast'
      childMode = mode.childMode( RouterBgpVrfAfIpMulticastMode,
                                  addrFamily=addrFamily,
                                  vrfName=mode.vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      addrFamily = args[ 'af' ]
      config = configForVrf( mode.vrfName )
      addrFamily = addrFamily + ' multicast'
      resetBgpAfModeConfig( config, addrFamily )

RouterBgpVrfMode.addCommandClass( RouterBgpVrfAfMulitcastCommand )

#--------------------------------------------------------------------------------
# vrf VRFNAME
#--------------------------------------------------------------------------------
class BgpVrfCmd( CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf VRFNAME ...'
   data = {
         'vrf': bgpTokens.vrf,
         'VRFNAME': DynamicNameMatcher(
               IraIpCli.getAllVrfNames,
               helpdesc='VRF name'
         ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRFNAME' ]
      if ( vrfName in VRFNAMES_RESERVED and vrfName != DEFAULT_VRF ):
         mode.addError( "vrf name %s is reserved." % vrfName )
         return

      # TODO: check number of vrfs and warn
      # TOOO: check vrfCapability and warn
      # TODO: check if vrf is configured or not and warn

      IraIpCli.warnIfRoutingDisabled( mode, vrfName )
      vrfMode = ( RouterBgpDefaultVrfMode if vrfName == DEFAULT_VRF else
                     RouterBgpVrfMode )
      childMode = mode.childMode( vrfMode, vrfName=vrfName )
      mode.session.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRFNAME' ]
      # cleanup bgp vrf service ACL config
      vrfMode = ( RouterBgpDefaultVrfMode if vrfName == DEFAULT_VRF else
                  RouterBgpVrfMode )
      childMode = mode.childMode( vrfMode, vrfName=vrfName )
      for aclType in [ 'ip', 'ipv6' ]:
         setServiceAclCommon( childMode, aclType, no=True )

      if vrfName == DEFAULT_VRF:
         deleteBgpDefaultVrfConfig()
         return
      assert vrfName not in VRFNAMES_RESERVED
      if vrfName in bgpVrfConfigDir.vrfConfig:
         for hook in deleteRouterBgpVrfHook.extensions():
            hook( vrfName )
      if vrfName in ucmpVrfConfigDir.vrfConfig:
         for hook in deleteRouterBgpUcmpVrfHook.extensions():
            hook( vrfName )

RouterBgpBaseMode.addCommandClass( BgpVrfCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath skip peer type ebgp ibgp
#  command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class SetBestpathCompatibleEbgpIbgp( CliCommandClass ):
   syntax = 'bgp bestpath skip peer type ebgp ibgp'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'skip': bgpTokens.skipMatcher,
         'peer': bgpTokens.peerBestpathMatcher,
         'type': bgpTokens.typeBestpathMatcher,
         'ebgp': bgpTokens.eBgpMatcher,
         'ibgp': bgpTokens.iBgpMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.compatibleEBgpIBgp = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.compatibleEBgpIBgp = (
            getExplicitDefaultTristate( mode, 'compatibleEBgpIBgp' )
      )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.compatibleEBgpIBgp = 'isInvalid'

if IpRibLibToggleLib.toggleMixedEcmpEnabled():
   RouterBgpSharedModelet.addCommandClass( SetBestpathCompatibleEbgpIbgp )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd(
      CliCommandClass ):
   syntax = 'bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'next-hop': bgpTokens.bestPathNexthop,
      'resolution': bgpTokens.bestPathNexthopResolution,
      'tunnel': bgpTokens.bestPathNexthopTunnel,
      'nexthop-group': bgpTokens.bestPathNhg,
      'ecmp': bgpTokens.bestPathEcmp,
      'disabled': bgpTokens.bestPathEcmpDisabled,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.nexthopResolutionNhgTunnelEcmpDisabled = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.nexthopResolutionNhgTunnelEcmpDisabled = 'isFalse'

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.nexthopResolutionNhgTunnelEcmpDisabled = 'isInvalid'

RouterBgpSharedModelet.addCommandClass(
   BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd )

#---------------------------------------------------------------------------
# "[no|default] bgp bestpath skip aigp-metric" command, in "router-bgp" mode
#---------------------------------------------------------------------------
class SetSkipAigpMetricCmd( CliCommandClass ):
   syntax = 'bgp bestpath skip aigp-metric'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'skip': bgpTokens.skipMatcher,
      'aigp-metric': bgpTokens.skipAigp,
   }

   @staticmethod
   def handler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipAigpMetric = 'isTrue'

   @staticmethod
   def noHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipAigpMetric = (
            getExplicitDefaultTristate( mode, 'skipAigpMetric' ) )

   @staticmethod
   def defaultHandler( mode, args ):
      config = configForVrf( mode.vrfName )
      config.skipAigpMetric = 'isInvalid'

if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
   RouterBgpSharedModelet.addCommandClass( SetSkipAigpMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp skip rib-install [ bestpath-selection ] " command, in
# IPv4/v6 unicast AfiSafi modes
#-------------------------------------------------------------------------------
class SkipRibBestpathCommand( CliCommandClass ):
   syntax = 'bgp skip rib-install [ bestpath-selection ]'
   noOrDefaultSyntax = 'bgp skip rib-install ...'
   data = {
      'bgp': bgpNode,
      'skip': 'Skip installing of routes in RIB and (optionally) bestpath selection',
      'rib-install': 'Skip installing BGP routes in RIB',
      'bestpath-selection': 'Skip bestpath selection steps',
   }

   @staticmethod
   def _setSkipRibBestpath( afiSafi, skipRib=False, skipBestpath=False,
                            vrfName=DEFAULT_VRF ):
      config = configForVrf( vrfName )
      if afiSafi == 'ipv4':
         config.skipRibIPv4Uni = skipRib
         config.skipBestpathIPv4Uni = skipBestpath
      else:
         config.skipRibIPv6Uni = skipRib
         config.skipBestpathIPv6Uni = skipBestpath

   @staticmethod
   def handler( mode, args ):
      SkipRibBestpathCommand._setSkipRibBestpath(
         mode.addrFamily, skipRib=True, skipBestpath='bestpath-selection' in args,
         vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      SkipRibBestpathCommand._setSkipRibBestpath(
         mode.addrFamily, skipRib=False, skipBestpath=False, vrfName=mode.vrfName )

RouterBgpAfIpUniSharedModelet.addCommandClass( SkipRibBestpathCommand )

#--------------------------------------------------------------------------------
# "[no|default] vpn layer-3 export paths ( all | best )" cmd in "router-bgp" mode
#--------------------------------------------------------------------------------
class SetVpnExportPathsMethod( BgpCmdBaseClass ):
   syntax = 'vpn LAYER export paths METHOD'
   # When both layer2 and layer3 are supported change the above line to the
   # following. Also follow the instructions in the definition of LAYER token
   # syntax = 'vpn [ LAYER ] export paths METHOD'
   # noOrDefaultSyntax = 'vpn [ LAYER ] export paths ...'
   noOrDefaultSyntax = 'vpn LAYER export paths ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'vpn': bgpTokens.vpn,
      'LAYER': bgpTokens.layerMatcher,
      'export': bgpTokens.export,
      'paths': bgpTokens.paths,
      'METHOD': bgpTokens.exportMethodMatcher,
   } )

   @staticmethod
   def _setVpnExportPathsMethod( mode, layer, method ):
      isBest = method == 'best'
      config = configForVrf( mode.vrfName )
      if layer is None or layer == 'layer-3':
         config.vpnExportMethodL3Best = isBest
      if layer is None or layer == 'layer-2':
         config.vpnExportMethodL2Best = isBest

   @staticmethod
   def _noVpnExportPathsMethod( mode, layer ):
      config = configForVrf( mode.vrfName )
      if layer is None or layer == 'layer-3':
         config.vpnExportMethodL3Best = config.vpnExportMethodL3BestDefault
      if layer is None or layer == 'layer-2':
         config.vpnExportMethodL2Best = config.vpnExportMethodL2BestDefault

   @staticmethod
   def _handleNormal( mode, args ):
      SetVpnExportPathsMethod._setVpnExportPathsMethod(
            mode, args.get( 'LAYER' ), args[ 'METHOD' ] )

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      SetVpnExportPathsMethod._noVpnExportPathsMethod(
            mode, args.get( 'LAYER' ) )

if BgpCommonToggleLib.toggleBgpVpnExportMethodEnabled():
   RouterBgpBaseMode.addCommandClass( SetVpnExportPathsMethod )

#----------------------------------------------------------------------------------
#  " [ no | default  ] <protocol> instance <instance-name> identifier
#    <value> " under link-state address-family config mode
#---------------------------------------------------------------------------------
class ConfigLsImportCmd( CliCommandClass ):
   syntax = 'PROTOCOL instance INST [ identifier ID ]'
   noOrDefaultSyntax = 'PROTOCOL [ instance INST ] ...'
   data = {
      'PROTOCOL': EnumMatcher( {
         'isis': 'IS-IS protocol',
      } ),
      'instance': bgpTokens.protocolInstance,
      'INST': bgpTokens.instanceNameRule,
      'identifier': bgpTokens.linkStateIdentifier,
      'ID': bgpTokens.lsIdentifierValueRangeMatcher,
   }

   _kwProtoMap = {
      'isis': 'protoIsis',
   }

   @staticmethod
   def handler( mode, args ):
      proto = ConfigLsImportCmd._kwProtoMap[ args[ 'PROTOCOL' ] ]
      if proto not in [ 'protoIsis' ]:
         raise ValueError()

      protoImportCfg = lsImportConfig.protoConfig.get( proto )
      if ( protoImportCfg and
            not RoutingLibToggleLib.toggleBgpLsProducerIsisMultiInstanceEnabled() ):
         lsImportConfig.protoConfig[ proto ].instanceCfg.clear()

      if not protoImportCfg:
         protoImportCfg = lsImportConfig.protoConfig.newMember( proto )

      instName = args.get( 'INST' )
      instanceCfg = Tac.Value( 'Routing::Bgp::LinkStateInstanceImportCfg', instName )
      identifier = args.get( 'ID' )
      if identifier is not None:
         instanceCfg.identifierPresent = True
         instanceCfg.identifier = identifier
      protoImportCfg.instanceCfg.addMember( instanceCfg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proto = ConfigLsImportCmd._kwProtoMap[ args.get( 'PROTOCOL' ) ]
      protoImportCfg = lsImportConfig.protoConfig.get( proto )
      if not protoImportCfg:
         return
      instName = args.get( 'INST' )
      if instName is not None:
         del protoImportCfg.instanceCfg[ instName ]
      else:
         protoImportCfg.instanceCfg.clear()
      if not protoImportCfg.instanceCfg:
         del lsImportConfig.protoConfig[ proto ]

RouterBgpAfLinkStateModelet.addCommandClass( ConfigLsImportCmd )

def nhTypeFromAfiSafi( af, args ):
   pnht = None
   pathAfiSafiNhType = Tac.Type( 'Routing::Bgp::PathAfiSafiNexthopType' )
   if af == 'ipv4':
      pnht = pathAfiSafiNhType.ipv4UniNexthopsAll
   elif af == 'ipv6':
      if '6pe' in args:
         pnht = pathAfiSafiNhType.sixPeNexthopsAll
      else:
         pnht = pathAfiSafiNhType.ipv6UniNexthopsAll
   elif af == 'ipv4 labeled-unicast':
      pnht = pathAfiSafiNhType.ipv4MplsNexthopsAll
   elif af == 'ipv6 labeled-unicast':
      pnht = pathAfiSafiNhType.ipv6MplsNexthopsAll
   elif af == 'evpn':
      if 'vxlan' in args:
         pnht = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan
      elif 'mpls' in args:
         pnht = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls
   elif af == 'vpn-ipv4':
      pnht = pathAfiSafiNhType.ipv4MplsVpnNexthopsAll
   elif af == 'vpn-ipv6':
      pnht = pathAfiSafiNhType.ipv6MplsVpnNexthopsAll
   else:
      raise Exception( 'Next-hop resolution ribs not supported in af %s' % af )

   assert pnht
   return pnht

def nhResRibsConfigHelper( af, args=None, no=None,
                           vrfName=DEFAULT_VRF ):
   config = configForVrf( vrfName )
   pnht = nhTypeFromAfiSafi( af, args )
   if no:
      config.nhResolutionRibProfileConfig[ pnht ] = \
         config.nhResolutionRibProfileConfigDefault[ pnht ]
      if pnht in config.nhResolutionRibProfileRouteMapConfig:
         del config.nhResolutionRibProfileRouteMapConfig[ pnht ]
      return

   if 'ROUTE_MAP' in args:
      config.nhResolutionRibProfileRouteMapConfig[ pnht ] = args[ 'NAME' ]
      config.nhResolutionRibProfileConfig[ pnht ] = \
         config.nhResolutionRibProfileConfigDefault[ pnht ]
      return

   if pnht in config.nhResolutionRibProfileRouteMapConfig:
      del config.nhResolutionRibProfileRouteMapConfig[ pnht ]
   resRibProf = getResolutionRibProfileConfig( args )
   config.nhResolutionRibProfileConfig[ pnht ] = resRibProf

class ResolutionRibsCmd( CliCommandClass ):
   if RouteMapToggleLib.toggleSetResolutionRibProfileEnabled():
      syntax = 'next-hop resolution ribs ( RIBS | ( ROUTE_MAP NAME ) )'
   else:
      syntax = 'next-hop resolution ribs RIBS'
   noOrDefaultSyntax = 'next-hop resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'RIBS': ResolutionRibsExpr,
      'ROUTE_MAP': bgpTokens.resolutionRouteMap,
      'NAME': mapNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, no=True, vrfName=mode.vrfName )

RouterBgpAfIpUniSharedModelet.addCommandClass( ResolutionRibsCmd )
RouterBgpVrfAfModelet.addCommandClass( ResolutionRibsCmd )

if BgpCommonToggleLib.toggleHideUnicastRibEnabled():
   luRibsExpr = ResolutionRibsHiddenUnicastExpr
else:
   luRibsExpr = ResolutionRibsExpr

class ResolutionRibsLuCmd( CliCommandClass ):
   syntax = 'next-hop resolution ribs RIBS'
   noOrDefaultSyntax = 'next-hop resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'RIBS': luRibsExpr
   }

   @staticmethod
   def handler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args,
                             vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, no=True,
                             vrfName=mode.vrfName )

RouterBgpAfLabelSharedModelet.addCommandClass( ResolutionRibsLuCmd )

class Resolution6peRibsCmd( CliCommandClass ):
   syntax = 'next-hop 6pe resolution ribs RIBS'
   noOrDefaultSyntax = 'next-hop 6pe resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      '6pe': '6pe specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'RIBS': ResolutionRibsTunnelExpr,
   }

   @staticmethod
   def handler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, no=True,
                             vrfName=mode.vrfName )

RouterBgpAfIpv6Modelet.addCommandClass( Resolution6peRibsCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop ( mpls | vxlan ) resolution ribs RIBS"
# under 'address-family evpn' mode
#-------------------------------------------------------------------------------
if BgpCommonToggleLib.toggleHideUnicastRibEnabled():
   mplsRibsExpr = ResolutionRibsHiddenUnicastExpr
else:
   mplsRibsExpr = ResolutionRibsExpr

class ResolutionMplsRibsCmd( CliCommandClass ):
   syntax = 'next-hop mpls resolution ribs RIBS'
   noOrDefaultSyntax = 'next-hop mpls resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      'mpls': 'MPLS specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'RIBS': mplsRibsExpr
   }

   @staticmethod
   def handler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, no=True,
                             vrfName=mode.vrfName )

RouterBgpAfEvpnModelet.addCommandClass( ResolutionMplsRibsCmd )

class ResolutionVxlanRibsCmd( CliCommandClass ):
   syntax = 'next-hop vxlan resolution ribs IP_PRIMARY'
   noOrDefaultSyntax = 'next-hop vxlan resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKwForResolutionPolicy,
      'vxlan': 'Vxlan specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'IP_PRIMARY': ipRibsEnumMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, vrfName=mode.vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      nhResRibsConfigHelper( mode.addrFamily, args, no=True,
                             vrfName=mode.vrfName )

RouterBgpAfEvpnModelet.addCommandClass( ResolutionVxlanRibsCmd )

def removeNexthopSelfReceivedVpnRoutesConfig( config, vpnAf, vrfName=None ):
   ''' Removes 'no neighbor default encapsulation mpls next-hop-self
   received-vpnv[4|6]-routes'
   '''
   if vpnAf in [ 'vpn-ipv4', 'vpn-ipv6' ]:
      afiSafi = vpnAfTypeMapInv[ vpnAf ]
      del config.nexthopSelfLocalLabelAlloc[ afiSafi ]
   else:
      raise ValueError()

def resetBgpAfModeConfig( config, addrFamily, vrfName=None ):
   # This reset logic for af-specific attributes relies on a specific format of said
   # attributes in the config entity which causes them to be populated into
   # bgpConfigAttrsAfMap.
   # Unfortunately nhResolutionRibProfileConfig was not able to follow that
   # format and consequently does not get populated into bgpConfigAttrsAfMap
   # Therefore we'll handle this single attribute here manually
   if addrFamily == 'ipv6':
      nhResRibsConfigHelper( addrFamily, args={}, no=True, vrfName=vrfName )
      nhResRibsConfigHelper( addrFamily, args={ '6pe': '6pe' }, no=True,
         vrfName=vrfName )
   elif addrFamily in ( 'ipv4', 'ipv4 labeled-unicast', 'ipv6 labeled-unicast' ):
      nhResRibsConfigHelper( addrFamily, args=None, no=True, vrfName=vrfName )
   elif addrFamily == 'evpn':
      nhResRibsConfigHelper( addrFamily, args={ 'mpls': 'mpls' }, no=True,
         vrfName=vrfName )
      nhResRibsConfigHelper( addrFamily, args={ 'vxlan': 'vxlan' }, no=True,
         vrfName=vrfName )
   elif ( addrFamily == 'vpn-ipv4' or addrFamily == 'vpn-ipv6' ):
      nhResRibsConfigHelper( addrFamily, args=None, no=True, vrfName=vrfName )
      # nexthopSelfLocalLabelAlloc attribute doesn't follow bgpConfigAttrsAfMap
      # and hence needs to be removed explicitly.
      removeNexthopSelfReceivedVpnRoutesConfig( config, addrFamily, vrfName=vrfName )

   for key, afMap in bgpConfigAttrsAfMap.iteritems():
      attrName = afMap.get( addrFamily )
      if attrName:
         attr = getattr( config, attrName )
         if hasattr( attr, 'clear' ):
            attr.clear()
         elif attr in Tac.Type( 'Ip::TristateBool' ).attributes:
            setattr( config, attrName, 'isInvalid' )
         else:
            if key == 'bgpRedistInternal':
               # The redistribute-internal config is a special case because we
               # do not reset the attribute back to the default/initial value.
               if attr != RedistInternalStateEnum.notConfigured:
                  setattr( config, attrName,
                           RedistInternalStateEnum.enableRedistInt )
            else:
               defVal = getattr( config, attrName + 'Default' )
               setattr( config, attrName, defVal )

   if vrfName is not None and vrfName != DEFAULT_VRF:
      config = configForVrf( vrfName )
   for peerConfig in config.neighborConfig.values():
      for _, afMap in peerConfigAttrsAfMap.iteritems():
         attrName = afMap.get( addrFamily )
         if attrName:
            attr = getattr( peerConfig, attrName )
            if hasattr( attr, 'clear' ):
               attr.clear()
            else:
               if hasattr( peerConfig, attrName + 'Present' ):
                  setattr( peerConfig, attrName + 'Present', False )
               defVal = getattr( peerConfig, attrName + 'Default' )
               setattr( peerConfig, attrName, defVal )

      if addrFamily == "ipv6":
         peerConfig.sixPePresent = False
         peerConfig.sixPe = peerConfig.sixPeDefault

      if addrFamily in [ 'ipv4', 'ipv6' ]:
         l = peerConfig.routeTargetExportFilterDisabledAf.keys()
         for nlriType in l:
            if nlriType.ipVersion == int( addrFamily[ -1 ] ):
               del peerConfig.routeTargetExportFilterDisabledAf[ nlriType ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global routingHardwareStatus
   global rdAutoInputDir
   global rdConfigInputDir
   global asnConfig
   global aclStatus
   global aclCheckpoint
   global ucmpVrfConfigDir
   global bgpVrfConfigDir
   global lsImportConfig

   bgpConfig = ConfigMount.mount( entityManager,
                                  'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   routingHardwareStatus = LazyMount.mount(
         entityManager,
         'routing/hardware/status',
         'Routing::Hardware::Status',
         'r' )
   rdAutoInputDir = LazyMount.mount(
         entityManager,
         'ip/vrf/routeDistinguisherInputDir/auto',
         'Tac::Dir',
         'ri'
   )
   rdConfigInputDir = ConfigMount.mount(
         entityManager,
         'ip/vrf/routeDistinguisherInputDir/config',
         'Tac::Dir',
         'w'
   )
   asnConfig = ConfigMount.mount(
         entityManager,
         'routing/bgp/asn/config',
         'Routing::AsnConfig',
         'w'
   )
   aclStatus = LazyMount.mount(
         entityManager,
         'acl/status/all',
         'Acl::Status',
         'r'
   )
   aclCheckpoint = LazyMount.mount(
         entityManager,
         'acl/checkpoint',
         'Acl::CheckpointStatus',
         'w'
   )
   bgpVrfConfigDir = ConfigMount.mount(
         entityManager,
         'routing/bgp/vrf/config',
         'Routing::Bgp::VrfConfigDir',
         'w'
   )
   ucmpVrfConfigDir = ConfigMount.mount(
         entityManager,
         'routing/ucmp/bgp/vrf/config',
         'RoutingLib::Ucmp::VrfUcmpConfigDir',
         'w'
   )
   lsImportConfig = ConfigMount.mount( entityManager, 'routing/bgp/lsImportConfig',
         'Routing::Bgp::LinkStateImportConfig', 'w' )
