# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for BGP configuration commands

from Toggles import (
   BgpCommonToggleLib,
)
from CliPlugin.RouteMapCli import (
   getPrefixListNames,
   getIpv6PrefixListNames,
)
from CliPlugin.IraServiceCli import routingMatcherForConfig
from CliPlugin import (
   IpAddrMatcher,
   Ip6AddrMatcher,
   Ip6LlAddr,
   IraVrfCli,
)
import Tac
import LazyMount
import ConfigMount
import CliCommand
import CliMatcher
import BasicCli
from BasicCliUtil import anyCaseRegex
import CliParser
from ArnetLib import asnStrToNum, bgpFormatAsn
import CliExtensions
from CliMode.BgpCommon import RoutingBgpBaseMode, RoutingBgpVrfMode
from CliMode.BgpCommon import RoutingBgpBaseAfMode, RoutingBgpVrfAfMode
from RouteMapLib import isAsdotConfigured
from BgpLib import (
   PeerConfigKey,
   cleanupBgpConfig,
   cleanupUcmpConfig,
   vpnTypeAttrMap,
   vpnTypeAttrMapInv,
   vpnAfTypeMapInv,
   NoOrDefault,
   updateConfigAttrsAfMap,
)
import BgpLib
from IpLibConsts import VRFNAMES_RESERVED
from IpLibConsts import DEFAULT_VRF
import re
import AclCliLib
from eunuchs.in_h import IPPROTO_TCP
import CliParserCommon
import CliToken.Service
import CliToken.IpRibLibCliTokens

bgpConfig = None
bgpVrfConfigDir = None
allVrfConfig = None
rdConfigInputDir = None
routingHardwareStatus = None
bgpVrfClearReqDir = None
BGP_PORT = 179
addPathSendAcceptOneApp = True
asnConfig = None
aclConfig = None
aclCpConfig = None
ucmpConfig = None
ucmpVrfConfigDir = None

redistInternalStateEnum = Tac.Type( "Routing::Bgp::RedistInternalState" )
MetricOutStateEnum = Tac.Type( "Routing::Bgp::MetricOutState" )
SendCommunityLinkBandwidthStateEnum = \
    Tac.Type( "Routing::Bgp::SendCommunityLinkBandwidthState" )
ExtendedNextHopCapabilityEnum = Tac.Type( "Routing::Bgp::ExtendedNextHopCapability" )
policyActionEnum = Tac.Type( "Routing::Bgp::PolicyActionType" )

#--------------------------------------------------------------------------------
# AfModeExtensionHook to provide support for registering CLI config for new
# address families in their own package in such a way that unwanted package
# dependencies are not introduced
#--------------------------------------------------------------------------------
class AfModeExtensionHook( object ):
   """A hook to allow for extensions for the new address families created for
   multi-agent mode in a manner that is independent of package dependencies."""
   def __init__( self ):
      """Dictionary that contains address family mode extensions.
      @key : address family string
      @value : address family submode for the address family's configuration
      Example : { 'vpn-ipv4' : RouterBgpBaseAfMplsVpnMode }
      This will be populated by the CliPlugin of the corresponding af's package.
      Please see Plugin() in /src/MplsVpn/CliPlugin/MplsVpnCli.py.
      """
      self.afModeExtension = {}
      self.vrfAfModeExtension = {}

   def addAfModeExtension( self, afStr, afSubMode ):
      """Add extension @afSubMode for address family @afStr. These extensions will
      be used to go to the specific submode when an address family is configured.
      Please see RouterBgpBaseMode.gotoRouterBgpBaseAfMplsVpnMode() for usage.
      """
      # Each address family extension should only be registered once.  However,
      # in cohab Cli breadth tests it is possible for plugins to be reloaded,
      # therefore it's not safe to assert that the afStr has not already been
      # registered.
      self.afModeExtension[ afStr ] = afSubMode

   def afModeExtensions( self ):
      """Return list of all address families that have currently provided an
      afMode extension.
      """
      return list( self.afModeExtension.keys() )

   def addVrfAfModeExtension( self, afStr, afSubMode ):
      """Add extension @afSubMode for address family @afStr with vrf mode as parent.
      These extensions will be used to go to the specific submode when an address
      family is configured under a vrf.
      Please see RouterBgpVrfMode.addCommandClass( EnterBgpFlowspecVrfAfMode ) for
      usage.
      """
      # Each address family extension should only be registered once.  However,
      # in cohab Cli breadth tests it is possible for plugins to be reloaded,
      # therefore it's not safe to assert that the afStr has not already been
      # registered.
      self.vrfAfModeExtension[ afStr ] = afSubMode

   def vrfAfModeExtensions( self ):
      """Return list of all address families that have currently provided an
      afMode extension under a given vrf.
      """
      return list( self.vrfAfModeExtension.keys() )

afModeExtensionHook = AfModeExtensionHook()

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------
def routingSupportedGuard( mode, token ):
   if routingHardwareStatus.routingSupported: 
      return None
   return CliParser.guardNotThisPlatform

def deleteVrfHook( vrfName ):
   if vrfName and vrfName in bgpVrfClearReqDir.clearRequestDir:
      del bgpVrfClearReqDir.clearRequestDir[ vrfName ]
   return ( True, None )

# The CLI hook for BGP is used to allow external features dependant on BGP mode
# to clear their configurations. This function is called as
#
# hook( vrfName )
#
# vrfName is the name of the relevant vrf
#
# and has no expected return values
deleteRouterBgpVrfHook = CliExtensions.CliHook()
deleteRouterBgpMacVrfHook = CliExtensions.CliHook()
deleteRouterBgpUcmpVrfHook = CliExtensions.CliHook()

def deleteBgpVrfConfigHook( vrfName ):
   if not vrfName:
      return
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         cleanupBgpConfig( bgpVrfConfigDir.vrfConfig[ vrfName ] )
         del bgpVrfConfigDir.vrfConfig[ vrfName ]
      bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
      if vrfName in bgpRouteDistinguisherInput.routeDistinguisher:
         del bgpRouteDistinguisherInput.routeDistinguisher[ vrfName ]
   else:
      # Cleanup config under default-vrf mode
      deleteBgpDefaultVrfConfig()
      # Cleanup global BGP config
      cleanupBgpConfig( bgpConfig )
      bgpConfig.asNumber = 0
      cleanupUcmpConfig( ucmpConfig )

def deleteBgpDefaultVrfConfig():
   """Method to clean config under router-bgp-vrf-default mode"""

   # Currently, only VPN config is present under this mode. Cleanup VPN config.

   # Delete route-distinguisher config
   bgpRouteDistinguisherInput = getBgpRouteDistinguisherInput()
   if DEFAULT_VRF in bgpRouteDistinguisherInput.routeDistinguisher:
      del bgpRouteDistinguisherInput.routeDistinguisher[ DEFAULT_VRF ]
   bgpConfig.routeDistinguisher = 'INVALID'

   # Delete route-target config
   for attrName in [ 'routeTargetImport', 'routeTargetExport' ]:
      # Reset route-target config
      getattr( bgpConfig, attrName ).clear()
      # Also reset the per VPN route-target config
      for vpnType in vpnTypeAttrMap:
         vpnTypeAttrName = attrName + vpnType
         getattr( bgpConfig, vpnTypeAttrName ).clear()

   for attrName in [ 'routeMapImport', 'routeMapExport',
                     'allowImportedRouteToExport' ]:
      getattr( bgpConfig, attrName ).clear()
      for imoprtAf in [ 'AfV4', 'AfV6' ]:
         vpnTypeAttrName = attrName + imoprtAf
         getattr( bgpConfig, vpnTypeAttrName ).clear()

   bgpConfig.routeTargetExportFilterDisabled.clear()

   attrName = 'defaultExport'
   getattr( bgpConfig, attrName ).clear()
   for af in [ 'AfV4', 'AfV6' ]:
      afAttrName = attrName + af
      getattr( bgpConfig, afAttrName ).clear()

def deleteUcmpVrfConfigHook( vrfName ):
   if not vrfName:
      return
   if vrfName != DEFAULT_VRF:
      if vrfName in ucmpVrfConfigDir.vrfConfig:
         cleanupUcmpConfig( ucmpVrfConfigDir.vrfConfig[ vrfName ] )
         del ucmpVrfConfigDir.vrfConfig[ vrfName ]
   else:
      cleanupUcmpConfig( ucmpConfig )

def configForVrf( vrfName ):
   config = bgpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         config = bgpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

def ucmpConfigForVrf( vrfName ):
   config = ucmpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in ucmpVrfConfigDir.vrfConfig:
         config = ucmpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

def bgpNeighborConfig( neighborKey, vrfName, create=True ):
   # Create/look-for peerGroup types in default vrf config
   if neighborKey.type == 'peerGroup':
      config = bgpConfig
   else:
      config = configForVrf( vrfName ) 
   neighConfig = config.neighborConfig.get( neighborKey )
   if not neighConfig and create:
      neighConfig = config.neighborConfig.newMember( neighborKey )
   return neighConfig

def delNeighborConfigIfDefault( peerKey, vrfName ):
   '''Function to delete neighborConfig if all attributes have default
   values

   If all the attributes of a neighborConfig are default, then the
   neighbor should be deleted. This can happen as the result of a
   setAttr call, if the value it is set to is the default. In other
   words, it's possible for a set call to have the effect of deleting
   a neighbor config.

   This should be called after any set or clear operation on a
   NeighborConfig attribute.
   '''
   config = configForVrf( vrfName ) 
   peerConfig = bgpNeighborConfig( peerKey, vrfName, create=False )
   if not peerConfig or peerConfig.isPeerGroup or peerConfig.isPeerGroupPeer:
      return
   allAttrs = set( [ i.name for i in
                     peerConfig.tacType.attributeQ if i.writable ] )
   attrsWithPresent = set( [ i for i in allAttrs
                             if hasattr( peerConfig, i + 'Present' ) ] )
   attrsWithPresent.add( 'timers' )
   attrsWithPresent.add( 'metricOut' )
   for attr in attrsWithPresent:
      if attr == 'metricOut':
         if ( getattr( peerConfig, 'metricOutState' ) != 
              getattr( peerConfig, 'metricOutStateDefault' ) ):
            return
      elif getattr( peerConfig, attr + 'Present' ):
         return
   if ( peerConfig.apSend.members() or peerConfig.apSendIPv4Uni.members() or
        peerConfig.apSendIPv6Uni.members() or
        peerConfig.apSendIPv4LabeledUni.members() or
        peerConfig.apSendIPv6LabeledUni.members() or
        peerConfig.apSendEvpn.members() or
        peerConfig.apSendVpnV4.members() or peerConfig.apSendVpnV6.members() ):
      return
   if peerConfig.maxAdvRoutesAfiSafiConfig.members():
      return
   if ( peerConfig.routeTargetExportFilterDisabled.members() or
        peerConfig.routeTargetExportFilterDisabledAf.members() ):
      return
   if ( peerConfig.rtMembershipDefaultRouteTarget !=
        peerConfig.rtMembershipDefaultRouteTargetDefault or
        peerConfig.rtMembershipDefaultRouteTargetEncoding !=
        peerConfig.rtMembershipDefaultRouteTargetEncodingDefault ):
      return
   del config.neighborConfig[ peerConfig.key ]

def resetPresents( config ):
   def isDefault( attr ):
      if getattr( config, attr ) == getattr( config, attr + 'Default' ):
         return True
      return False
   allAttrs = set( [ i.name for i in
                     config.tacType.attributeQ if i.writable ] )
   attrsWithPresent = set( [ i for i in allAttrs
                             if hasattr( config, i + 'Present' ) ] )
   attrsWithPresent.add( 'timers' )
   attrsWithPresent.add( 'metricOut' )

   for attr in attrsWithPresent:
      if attr == 'metricOut':
         if isDefault( attr ):
            setattr( config, 'metricOutState', 
                     MetricOutStateEnum.metricNotConfigured )
      elif getattr( config, attr + 'Present' ):
         if attr == 'timers':
            if isDefault( 'keepaliveTime' ) and isDefault( 'holdTime' ):
               setattr( config, attr + 'Present', False )
         else:
            if isDefault( attr ):
               setattr( config, attr + 'Present', False )

class RouterBgpBaseMode( RoutingBgpBaseMode, BasicCli.ConfigModeBase ):
   name = 'BGP configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, asNumber ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseMode.__init__( self, asNumber )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   # 'show active' command extracts and displays the configuration under
   # 'router bgp <ASN>' in a way similar to how 'show running-configuration |
   # section router bgp <ASN>' works.
   # In BasicCli.py the function showRunningConfigWithMode gets the entire
   # configuration and filters it out using the filterExp provided by parent
   # mode (RoutingBgpBaseMode). The 'filterExp' by default uses parent Mode's
   # enterCmd (check ConfigModeBase in BasicCli.py).
   # when 'bgp asn notation asdot' is configured under 'router bgp <ASN>', we need
   # to filter router bgp configuration dynamically. This is because if <ASN>
   # is >65535, with 'bgp asn notation asdot' we represent <ASN> in dotted
   # notation.
   def filterExp( self ):
      return "^router bgp %s$" % bgpFormatAsn( self.param_, \
                                              isAsdotConfigured( asnConfig ) )

class RouterBgpDefaultVrfMode( RoutingBgpVrfMode, BasicCli.ConfigModeBase ):
   """BGP default VRF config mode. This mode only has VPN commands currently"""

   name = "BGP Default VRF configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = DEFAULT_VRF
      RoutingBgpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfMode( RoutingBgpVrfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      self.maybeMakeVrf()
      RoutingBgpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def maybeMakeVrf( self ):
      assert self.vrfName not in VRFNAMES_RESERVED
      config = bgpVrfConfigDir.vrfConfig.get( self.vrfName )
      if not config:
         bgpVrfConfigDir.vrfConfig.newMember( self.vrfName )
      if not ucmpVrfConfigDir.vrfConfig.get( self.vrfName ):
         ucmpVrfConfigDir.vrfConfig.newMember( self.vrfName )

   def validateRouteTargetBothCommand( self ):
      ''' This function validates if the "route target both" command is allowed ''' 
      self.addWarning( 'Usage of the keyword both is deprecated.'
                       ' Please use import and export keywords explicitly.' )

class RouterBgpBaseAfIpUniMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Unicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpv6UniMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IPV6 Unicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfLabelMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family labeled unicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-label' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpMulticastMode( RoutingBgpBaseAfMode,
                                      BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF 
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfIpv6MulticastMode( RoutingBgpBaseAfMode,
                                      BasicCli.ConfigModeBase ):
   name = 'BGP address family IPv6 Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpMulticastMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family IP Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName 
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpv6MulticastMode( RoutingBgpVrfAfMode,
      BasicCli.ConfigModeBase ):
   name = 'BGP address family IPv6 Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName=DEFAULT_VRF ):
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfSrTeMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Segment Routing Traffic Engineering configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-srte' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# XXX_FIX_BGP_CORE_CLI_EVPN_CONFIG: The Evpn CLI mode and callback functions
# should eventually be moved out of the Bgp package. But we've decided to defer
# doing so until later on. Note: Defining this directly in the
# RouterBgpBaseMode, so this command is not available in Nd-VRF submode.
class RouterBgpBaseAfEvpnMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family EVPN configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setVpnDomainId( self, vpnDomainId ):
      config = configForVrf( self.vrfName )
      config.domainIdEvpn = vpnDomainId

   def noVpnDomainId( self ):
      config = configForVrf( self.vrfName )
      config.domainIdEvpn = config.domainIdEvpnDefault

class RouterBgpBaseAfLinkStateMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Link State configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, 'link-state' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfDpsMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Dynamic-Path-Selection configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, 'path-selection' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIpMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF IPv4 configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfIp6Mode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF IPv6 configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpDefaultVrfAfMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   """BGP default VRF address family config mode. This mode only has VPN commands
   currently.
   """

   name = "BGP Default VRF address family configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName ):
      assert vrfName == DEFAULT_VRF
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfSharedModelete( CliParser.Modelet ):
   """This modelet has all the commands which are shared between default and
   non-default VRFs e.g. VPN configuration commands
   """
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName

# In the default VRF we don't CliSave attributes that match default, so return
# 'isInvalid' (meaning the default value is set). In a non-default VRF, we
# CliSave all explicitly-configured attributes, so return the actual value.
# This method should only be used in the handler for the command that's setting
# the value of the attribute that matches the default value.
def getExplicitDefaultTristate( mode, attr ):
   if mode.vrfName == DEFAULT_VRF:
      return 'isInvalid'
   config = configForVrf( mode.vrfName )
   meaningOfInvalid = getattr( config, attr + 'Default' )
   return 'isTrue' if meaningOfInvalid else 'isFalse'

def getExplicitDefaultValue( mode, attr, attrValue ):
   # In the default VRF we don't CliSave attributes that match default, so return
   # the value of attr + 'Invalid'. In a non-default VRF, we CliSave all
   # explicitly-configured attributes, so return the actual value. This method
   # should only be used in the handler for the command that's setting the value of
   # the attribute that matches the default value.
   config = configForVrf( mode.vrfName )
   if attr != 'maxEcmp':
      defValue = getattr( config, attr + 'Default' )
   else:
      # The maxEcmp attribute doesn't have a default in the bgp instance.
      # The default is set by the platform and stored in Routing:Hardware::Status
      defValue = routingHardwareStatus.maxEcmp
   if mode.vrfName == DEFAULT_VRF and attrValue == defValue:
      return getattr( config, attr + 'Invalid' )
   return attrValue

class RouterBgpSharedModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName

RouterBgpBaseMode.addModelet( RouterBgpSharedModelet )
RouterBgpVrfMode.addModelet( RouterBgpSharedModelet )

# Add RouterBgpVrfSharedModelete to both default and non-default VRF modes
RouterBgpDefaultVrfMode.addModelet( RouterBgpVrfSharedModelete )
RouterBgpVrfMode.addModelet( RouterBgpVrfSharedModelete )

def attrIsPresent( config, attr ):
   if getattr( config, attr + 'Present' ):
      return True
   return False

def configModeCmdForAf( addrFamily ):
   if addrFamily == 'all':
      modeCmd = 'router bgp'
   else:
      modeCmd = 'address-family %s' % addrFamily
   return modeCmd

def configModeKeyForVrfAf( addrFamily, vrfName=DEFAULT_VRF ):
   modeKey = 'router-bgp-vrf-%s' % vrfName
   if addrFamily != 'all':
      modeKey += '-af'
   return modeKey

def getConflictingAttrsForAf( attrs, addrFamily ):
   if addrFamily == 'all':
      conflictingAttrs = [ ( af, c ) for af, c in attrs.iteritems() if af != 'all' ]
   else:
      conflictingAttrs = [ ( 'all', attrs[ 'all' ] ) ]
   return conflictingAttrs

def haveConflictingRedistribute( mode, config, proto, addrFamily ):
   """ Print an error and returns True if redistribution is configured in
       conflicting mode"""

   attrs = BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   protocolNameMap = { 'protoIsis'      : 'isis',
                       'protoOspf3'     : 'ospfv3',
                       'protoOspf3Nssa' : 'ospfv3 nssa',
                       'protoOspf3Ase'  : 'ospfv3 external' }
   for otherAf, c in conflictingAttrs:
      if proto in getattr( config, c ):
         errorMsg = "Cannot configure redistribute %s in mode '%s' " \
                    "while it is configured in mode '%s'" \
                    % ( protocolNameMap[ proto ],
                        configModeCmdForAf( addrFamily ),
                        configModeCmdForAf( otherAf ) )
         mode.addError( errorMsg )
         return True
   return False

def redistributeConfig( addrFamily, config ):
   attr = BgpLib.bgpConfigAttrsAfMap[ 'redistributeConfig' ].get( addrFamily )
   return getattr( config, attr )

def haveConflictingRouteMaps( mode, config, attrName, mapName, addrFamily ):
   """ Print an error and returns True if a conflicting route-map is already set """
   attrs = BgpLib.peerConfigAttrsAfMap[ attrName ]
   conflictingRmAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, conflictingRm in conflictingRmAttrs:
      if attrIsPresent( config, conflictingRm ) and \
            getattr( config, conflictingRm ):
         errorMsg = ( "Cannot configure route-map '%s' in mode '%s' "
                      "while '%s' is configured in mode '%s'"
                      % ( mapName, configModeCmdForAf( addrFamily ),
                          getattr( config, conflictingRm ),
                          configModeCmdForAf( otherAf ) ) )
         mode.addError( errorMsg )
         return True
   return False

def haveConflictingWeight( mode, config, addrFamily ):
   """
   Print an error and return True if a conflicting weight config is
   already present.
   """
   attrs = BgpLib.peerConfigAttrsAfMap[ 'weight' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if attrIsPresent( config, c ):
         errMsg = "Cannot configure weight in mode '%s' while it is "\
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )
         mode.addError( errMsg )
         return True
   return False

def addPathSendConfig( config, addrFamily ):
   attr = BgpLib.bgpConfigAttrsAfMap[ 'apSend' ].get( addrFamily )
   return getattr( config, attr )

def haveConflictingRedistInternal( mode, config, addrFamily ):
   """ Print an error and returns True if conflicting
       redistribute-internal option is set """
   afMode = ( addrFamily == 'ipv4' or addrFamily == 'ipv6' )
   m = re.search( 'router-bgp.*', mode.longModeKey )
   routerBgpMode = True if m and not afMode else False
   vrfStr = ''
   if mode.vrfName != DEFAULT_VRF:
      vrfStr = '-vrf-%s' % mode.vrfName
   if routerBgpMode and config.bgpRedistInternalAfV4 == config.bgpRedistInternalAfV6:
      return False
   if afMode and \
         config.bgpRedistInternal != redistInternalStateEnum.disableRedistInt:
      return False
   otherMode = { True : 'router-bgp%s-af', False : 'router-bgp%s' }
   # Print error message as there is conflicting redistribute-internal config
   errorMsg = "Cannot configure command in mode '%s' when " \
         "'no bgp redistribute-internal' has been configured " \
         "in mode '%s'" % ( mode.longModeKey, otherMode[ routerBgpMode ] % vrfStr )
   mode.addError( errorMsg )
   return True

def switchRedistInternalConfigToAfMode( config ):
   # switch redistribute-internal config to addrFamily mode config
   allAfState = config.bgpRedistInternal
   if config.bgpRedistInternalAfV4 == redistInternalStateEnum.notConfigured and \
         config.bgpRedistInternalAfV6 == redistInternalStateEnum.notConfigured:
      assert allAfState != redistInternalStateEnum.notConfigured

      config.bgpRedistInternalAfV4 = allAfState
      config.bgpRedistInternalAfV6 = allAfState
      config.bgpRedistInternal = redistInternalStateEnum.notConfigured

def switchRedistInternalConfigToAllAfMode( config ):
   # switch redistribute-internal config to router-bgp mode config
   allAfState = config.bgpRedistInternal
   if allAfState == redistInternalStateEnum.notConfigured:
      assert config.bgpRedistInternalAfV4 == config.bgpRedistInternalAfV6

      afState = config.bgpRedistInternalAfV4
      config.bgpRedistInternalAfV4 = redistInternalStateEnum.notConfigured
      config.bgpRedistInternalAfV6 = redistInternalStateEnum.notConfigured
      config.bgpRedistInternal = afState

def conflictingRouteInstallMapConfigs( mode, routeInstallMapName, config,
                                       addrFamily ):
   # possible conflicts occur when routeInstallMap has been configured
   # at af level and global level. We need to prevent that.
   attrs = BgpLib.bgpConfigAttrsAfMap[ 'routeInstallMap' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if getattr( config, c ) != getattr( config, c + 'Default' ):
         errMsg = "Cannot configure route install-map in mode '%s' while it is " \
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )
         mode.addError( errMsg )
         return True
   return False

def conflictingRouteTargetConfigs( mode, config, addrFamily='all', vpnType=None ):

   vpnTypeAttrName = 'routeTargetImport' + vpnTypeAttrMapInv.get( vpnType, '' )
   importAttrs = BgpLib.bgpConfigAttrsAfMap[ vpnTypeAttrName ]
   vpnTypeAttrName = 'routeTargetExport' + vpnTypeAttrMapInv.get( vpnType, '' )
   exportAttrs = BgpLib.bgpConfigAttrsAfMap[ vpnTypeAttrName ]
   conflictingAttrs = getConflictingAttrsForAf( importAttrs, addrFamily )
   conflictingAttrs.extend( getConflictingAttrsForAf( exportAttrs, addrFamily ) )
   for otherAf, c in conflictingAttrs:
      if getattr( config, c ):
         errMsg = ( "Cannot configure route-target in mode '%s' while it is "
                    "configured in mode '%s'" % ( 
                       configModeKeyForVrfAf( addrFamily, mode.vrfName ),
                       configModeKeyForVrfAf( otherAf, mode.vrfName ) ) )
         mode.addError( errMsg )
         return True
   return False

def vpnTypeTokenAllowedInMode( vpnType, mode ):
   ''' This function checks if the vpnType token is allowed in the mode '''
   tokenAllowed = True
   if isinstance(
            mode,
            ( RouterBgpVrfAfIpMode,
              RouterBgpVrfAfIp6Mode,
              RouterBgpDefaultVrfAfMode )
   ):
      if ( vpnType == 'vpn-ipv4' and mode.addrFamily != 'ipv4' ) or (
         ( vpnType == 'vpn-ipv6' and mode.addrFamily != 'ipv6' ) ):
         tokenAllowed = False
   return tokenAllowed

def validateRouteTargetCommand( mode, vpnType, warn=True ):
   ''' This function validates the route target command to check if the vpnType
       is allowed in the mode '''
   # Note: Pass warn=False to suppress warnings as in the case of 'both' keyword
   valid = True
   if not vpnType and warn:
      mode.addWarning( 'This form of the route-target command is deprecated.'
                       ' Please specify one of %s.' % ( 
                       ', '.join( sorted( vpnTypeAttrMapInv.keys() ) ) ) )

   if not vpnTypeTokenAllowedInMode( vpnType, mode ):
      mode.addError( '%s argument is not allowed under address-family %s' % (
                     vpnType, mode.addrFamily ) )
      valid = False
   return valid

def getVpnTypeAttr( config, attrName, addrFamily, vpnType ):
   ''' Returns the vpnTypeAttr from the config corresponding to
       attrName, addrFamily, vpnType '''
   assert BgpLib.bgpConfigAttrsAfMap.has_key( attrName )
   attrName = BgpLib.bgpConfigAttrsAfMap[ attrName ].get( addrFamily )
   vpnTypeAttrName = attrName + vpnTypeAttrMapInv.get( vpnType, '' )
   vpnTypeAttr = getattr( config, vpnTypeAttrName )
   return vpnTypeAttr

def haveConflictingOutDelay( mode, config, addrFamily ):
   """
   Print an error and return True if a conflicting outDelay config is
   already present.
   """
   attrs = BgpLib.peerConfigAttrsAfMap[ 'outDelay' ]
   conflictingAttrs = getConflictingAttrsForAf( attrs, addrFamily )
   for otherAf, c in conflictingAttrs:
      if attrIsPresent( config, c ):
         errMsg = "Cannot configure outDelay in mode '%s' while it is "\
                  "configured in mode '%s'" % \
                  ( configModeCmdForAf( addrFamily ), configModeCmdForAf( otherAf ) )

         mode.addError( errMsg )
         return True
   return False

def conflictingRouteMapConfigs( mode, config, addrFamily, vpnType ):
   ''' Print an error and returns True if import/export route-map is configured in
       conflicting mode '''

   importAttrs = BgpLib.bgpConfigAttrsAfMap[ 'routeMapImport' ]
   exportAttrs = BgpLib.bgpConfigAttrsAfMap[ 'routeMapExport' ]
   conflictingAttrs = getConflictingAttrsForAf( importAttrs, addrFamily )
   conflictingAttrs.extend( getConflictingAttrsForAf( exportAttrs, addrFamily ) )
   vpnAf = vpnAfTypeMapInv[ vpnType ]
   for otherAf, c in conflictingAttrs:
      if vpnAf in getattr( config, c ):
         errorMsg = ( "Cannot configure route-map %s in mode '%s' "
                      "while it is configured in mode '%s'"
                      % ( vpnType, configModeCmdForAf( addrFamily ),
                        configModeCmdForAf( otherAf ) ) )
         mode.addError( errorMsg )
         return True
   return False

def setServiceAclCommon( mode, aclType='ip', no=False, aclName=None ):
   if no:
      AclCliLib.noServiceAcl( mode, 'bgpsacl', aclConfig, aclCpConfig,
                              aclName, aclType, mode.vrfName )
   else:
      assert aclName
      AclCliLib.setServiceAcl( mode, 'bgpsacl', IPPROTO_TCP,
                               aclConfig, aclCpConfig, aclName, aclType,
                               mode.vrfName, port=[ BGP_PORT ], sport=[ BGP_PORT ] )

bgpMatcherForConfig = CliCommand.Node( CliMatcher.KeywordMatcher( 'bgp',
                                          helpdesc='Border Gateway Protocol' ),
                                       guard=routingSupportedGuard )

#
# Add 'router bgp <asn>' rule to the list of global configs so that when
# the user types 'router bgp ?', the current AS number (if configured) will 
# show up in the help list.  Also, if the user types 'router bgp [tab]', and
# BGP is already configured with an AS number, that AS number will be 
# autocompleted.
#
# This is all for convenience/user-friendliness:
#
#   switch(config)# router bgp ?
#      <1-4294967295>  AS Number
#
#   switch(config)# router bgp 100       << configure BGP AS 100
#   switch(config-router-bgp)# 
#   switch(config-router-bgp)# exit
#   switch(config)#
#   switch(config)#
#   switch(config)# router bgp ?
#      100        Configure BGP AS 100   << AS 100 shows up in the help list
#      <1-4294967295>  AS Number
#
#   switch(config)#
# 
def getCurrentBgpAs( mode ):
   asn = bgpConfig.asNumber
   asdotConfigured = isAsdotConfigured( asnConfig )
   if asn == 0:
      return {}
   return { bgpFormatAsn( asn, asdotConfigured ): \
         "Configure BGP AS %s" % bgpFormatAsn( asn, asdotConfigured ) }

def getBgpAsValue( mode, match=None ):
   return asnStrToNum( match )

#-------------------------------------------------------------------------------
# "[no|default] service routing configuration bgp no-equals-default" command,
# in "config" mode.
#-------------------------------------------------------------------------------
class NedModeKwMatcher( CliMatcher.KeywordMatcher ):
   """Keyword Matcher for no-equals-default mode. Returns a Matcher instance."""

   def __init__( self, keyword, helpdesc ):
      CliMatcher.KeywordMatcher.__init__( self, keyword, helpdesc=helpdesc )

   def match( self, mode, context, token ):
      if bgpConfig.noEqualsDefaultMode:
         return CliMatcher.KeywordMatcher.match( self, mode, context, token )
      else:
         return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      if bgpConfig.noEqualsDefaultMode:
         return CliMatcher.KeywordMatcher.completions( self, mode, context, token )
      else:
         return []

# In general, commands that configure a boolean value will have a new 'disabled'
# token, while commands that set a numerical value will interpret configuring the
# default value explicitly the way the 'no' version of the command works outside
# of no-equals-default mode. See AID3714 for a full list of all affected commands.
noEqualsDefaultMatcherForConfig = CliMatcher.KeywordMatcher( 'no-equals-default',
      helpdesc='Interpret all \'no neighbor\' commands ' +
               'as \'default neighbor\' commands' )

configMode = BasicCli.GlobalConfigMode

class ConfigBgpNoEqualsDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'service routing configuration bgp no-equals-default'
   noOrDefaultSyntax = 'service routing configuration bgp no-equals-default'
   data = { 'service': CliToken.Service.serviceMatcherForConfig,
            'routing': routingMatcherForConfig,
            'configuration': CliToken.Service.configMatcherForAfterService,
            'bgp': bgpMatcherForConfig,
            'no-equals-default': noEqualsDefaultMatcherForConfig
          }

   @staticmethod
   def handler( mode, args ):
      # If ned-mode is already true, we don't want to cause churn in the system
      if bgpConfig.noEqualsDefaultMode:
         return
      bgpConfig.noEqualsDefaultMode = True
      # For all BGP peers, reset all explicit-no attributes to default values
      for peer in bgpConfig.neighborConfig.values():
         resetPresents( peer )
         delNeighborConfigIfDefault( peer.key, vrfName=DEFAULT_VRF )
      for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
         for peer in vrfConfig.neighborConfig.values():
            resetPresents( peer )
            delNeighborConfigIfDefault( peer.key, vrfName=vrfConfig.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bgpConfig.noEqualsDefaultMode = False

configMode.addCommandClass( ConfigBgpNoEqualsDefaultCmd )

def getAfPrefixListNames( mode ):
   if mode.addrFamily == 'ipv4':
      return getPrefixListNames( mode )
   else:
      return getIpv6PrefixListNames( mode )

afPrefixListNameMatcher = CliMatcher.DynamicNameMatcher( getAfPrefixListNames,
      'Name of the prefix list' )

# Peergroup names can be a combination of letters, digits, '-', '.' and '_'.
# But, if it includes a '.', it must begin with a letter
# Cannot be the (case-insensitive) keywords in blackListedPeerGroupNames.
validNameRegex = r'(([A-Za-z][-A-Za-z0-9_.]+)|([-A-Za-z0-9_]+))'
disallowRegex = anyCaseRegex( "(default)" )
peergroupNameRegex = "^(?!%s$)%s$" % ( disallowRegex, validNameRegex )

def peergroupNameFn( mode ):
   return ( x.group for x in bgpConfig.neighborConfig if x.type == 'peerGroup' )
peergroupNameMatcher = CliMatcher.DynamicNameMatcher( peergroupNameFn,
                                                      'Name of the peer-group',
                                                      helpname='WORD',
                                                      pattern=peergroupNameRegex )
ipv4PeerMatcher = IpAddrMatcher.IpAddrMatcher( "Neighbor IPv4 address" )
ipv6PeerMatcher = Ip6AddrMatcher.Ip6AddrMatcher( "Neighbor IPv6 address" )
ipv6LlPeerMatcher = Ip6LlAddr.Ip6LlAddrMatcher( "Neighbor IPv6 link-local address" )

class PeerCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP_PEER | IP6_PEER | IP6_LL_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      addr = args[ next( peer for peer in
                   ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER', 'GROUP' )
                   if peer in args ) ]
      args[ 'PEER' ] = PeerConfigKey( addr )

class V4OrPeerGroupCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if any( [ x in V4OrPeerGroupCliExpression.data.keys() for x in args ] ):
         addr = args[ next( peer for peer in
                      ( 'IP_PEER', 'GROUP' )
                      if peer in args ) ]
         args[ 'PEER' ] = PeerConfigKey( addr )

class V6OrPeerGroupCliExpression( CliCommand.CliExpression ):
   expression = 'GROUP | IP6_PEER | IP6_LL_PEER'
   data = { 'GROUP': peergroupNameMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if any( [ x in V6OrPeerGroupCliExpression.data.keys() for x in args ] ):
         addr = args[ next( peer for peer in
                      ( 'GROUP', 'IP6_PEER', 'IP6_LL_PEER' )
                      if peer in args ) ]
         args[ 'PEER' ] = PeerConfigKey( addr )

class V4V6PeerKeyCliExpression( CliCommand.CliExpression ):
   expression = 'IP_PEER | IP6_PEER | IP6_LL_PEER'
   data = { 'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in
                   ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER' )
                   if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

#------------------------------------------------------------------------------
# Base modelet for base functions for address-family config
#-------------------------------------------------------------------------------
class RouterBgpAfBaseModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

      self.mode_ = mode
      self.vrfName = mode.vrfName

class RouterBgpAfSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfSharedVrfModelet( RouterBgpAfBaseModelet ):
   """This modelet has all the address famliy commands which are shared between
   default and non-default VRFs e.g. VPN configuration commands
   """

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfVpnModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfEvpnModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfIpv4Modelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpVrfAfIpMulticastModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpVrfAfIpv6MulticastModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpVrfAfModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpVrfAfIpv6Modelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpVrfAfIpv4Modelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfIpMulticastModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfIpv6MulticastModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfIpv6Modelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family [ipv4 | ipv6] [multicast]" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpUniAndMcastSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family [ipv4 | ipv6]" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpUniSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family [ipv4 | ipv6] labeled-unicast" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfLabelSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family ipv4|ipv6 multicast" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfIpMcastSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family <ipv4|ipv6> sr-te" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfSrTeSharedModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

#-------------------------------------------------------------------------------
# "address-family path-selection" config mode
#-------------------------------------------------------------------------------
class RouterBgpAfDpsModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

class RouterBgpAfLinkStateModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfIpv4Modelet )
RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpUniMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfIpv6Modelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpv6UniMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfIpMulticastModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpMulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfIpv6MulticastModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpBaseAfIpv6MulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )
RouterBgpBaseAfLabelMode.addModelet( RouterBgpAfLabelSharedModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfEvpnModelet )
RouterBgpBaseAfEvpnMode.addModelet( RouterBgpAfVpnModelet )
RouterBgpBaseAfSrTeMode.addModelet( RouterBgpAfSrTeSharedModelet )

# Add RouterBgpAfSharedVrfModelet to both default and non-default VRF modes
RouterBgpDefaultVrfAfMode.addModelet( RouterBgpAfSharedVrfModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfSharedVrfModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfSharedVrfModelet )

RouterBgpVrfAfIpMode.addModelet( RouterBgpVrfAfModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpVrfAfIpMode.addModelet( RouterBgpVrfAfIpv4Modelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpVrfAfModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpAfIpUniSharedModelet )
RouterBgpVrfAfIp6Mode.addModelet( RouterBgpVrfAfIpv6Modelet )

RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpVrfAfIpMulticastModelet )
RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpMulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )

RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpVrfAfIpv6MulticastModelet )
RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfSharedModelet )
RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfIpUniAndMcastSharedModelet )
RouterBgpVrfAfIpv6MulticastMode.addModelet( RouterBgpAfIpMcastSharedModelet )

RouterBgpBaseAfLinkStateMode.addModelet( RouterBgpAfLinkStateModelet )

RouterBgpBaseAfDpsMode.addModelet( RouterBgpAfDpsModelet )

#-------------------------------------------------------------------------------
def getBgpRouteDistinguisherInput():
   bgpRouteDistinguisherInput = rdConfigInputDir.get( 'bgp' )
   assert bgpRouteDistinguisherInput
   return bgpRouteDistinguisherInput

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global routingHardwareStatus
   global allVrfConfig
   global rdConfigInputDir
   global bgpVrfConfigDir
   global bgpVrfClearReqDir
   global asnConfig
   global aclConfig
   global aclCpConfig
   global ucmpConfig
   global ucmpVrfConfigDir

   bgpConfig = ConfigMount.mount( entityManager, 'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   bgpVrfConfigDir = ConfigMount.mount( entityManager, 'routing/bgp/vrf/config',
                                        'Routing::Bgp::VrfConfigDir', 'w' )
   asnConfig = ConfigMount.mount( entityManager, 'routing/bgp/asn/config',
                                  'Routing::AsnConfig', 'w' )
   routingHardwareStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   rdConfigInputDir = \
      ConfigMount.mount( entityManager, 'ip/vrf/routeDistinguisherInputDir/config',
                         'Tac::Dir', 'w' )

   bgpVrfClearReqDir = LazyMount.mount( entityManager,
                                        'routing/bgp/clear/vrf/request',
                                        'Routing::Bgp::VrfClearRequestDir', 'w' )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   ucmpConfig = ConfigMount.mount( entityManager, 'routing/ucmp/bgp/config',
                                   'RoutingLib::Ucmp::UcmpConfig', 'w' )
   ucmpVrfConfigDir = ConfigMount.mount( entityManager,
                                         'routing/ucmp/bgp/vrf/config',
                                         'RoutingLib::Ucmp::VrfUcmpConfigDir',
                                         'w' )
   IraVrfCli.canDeleteVrfHook.addExtension( deleteVrfHook )

   deleteRouterBgpVrfHook.addExtension( deleteBgpVrfConfigHook )
   deleteRouterBgpUcmpVrfHook.addExtension( deleteUcmpVrfConfigHook )

# 'no-equals-default' mode, or NED mode, is described in AID3792.
# Refer to the AID for more details.
#
# The 'BgpNEDCmdBaseClass' calls an handler based on the form of the
# command executed and on the mode:
#
#  Mode          | CLI Command foom      | Handler triggered
# ------------------------------------------------------------
#  ANY mode      | <cmd> (positive form) | _handleNormal()
# ------------------------------------------------------------
#  ANY mode      | default <cmd>         | _handleNoOrDefault( noOrDefault=DEFAULT )
# ------------------------------------------------------------
#  normal mode   | no <cmd>              | _handleNoOrDefault( noOrDefault=NO )
#  NED mode      | no <cmd>              | _handleNoOrDefault( noOrDefault=DEFAULT )
# ------------------------------------------------------------
#  NED mode only | <cmd> disabled        | _handleNoOrDefault( noOrDefault=NO )
#
# Calls to _handleNoOrDefault are passed an enum noOrDefault which indicates if the
# command in question is being explicitly disabled or set to default settings.
#
# For a feature being disabled by default, the feature is implicitly disabled,
# meaning that the 'Present' attribute is False.
#
# As an optimization, explicit disables for non peer-group peers are treated the same
# as implicit disables. This is to avoid unnecessarily rendering the "no" form of the
# command when it has no practical effect. Explicit disable only makes a practical
# effect (compared to implicit disable) if there is some config to inherit from (e.g.
# a peer-group's config for a peer-group peer).
#
# Examples of features disabled by default are 'neighbor <peer> default-originate'
#
#  non-peer-group peer or peer group
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit disable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | implicit disable
#  NED mode      | no <cmd>       | implicit disable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | implicit disable
#
# peer group peer
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit disable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | explicit disable
#  NED mode      | no <cmd>       | implicit disable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | explicit disable
#
# For a feature that is enabled by default, an example is
# 'neighbor <peer> additional-path receive'. The behavior is the following:
#
#  non-peer-group peer or peer group or peer group peer
# ---------------------------------------------------
#  ANY mode      | <cmd>          | explicit enable
# ---------------------------------------------------
#  ANY mode      | default <cmd>  | implicit enable
# ---------------------------------------------------
#  normal mode   | no <cmd>       | explicit disable
#  NED mode      | no <cmd>       | implicit enable
# ---------------------------------------------------
#  NED mode only | <cmd> disabled | explicit disable
#
class BgpCmdBaseClass( CliCommand.CliCommandClass ):
   '''
   Base class for commands where NED mode is not dependent on the running
   configuration (config is always in NED mode).

   All NEW bgp cli commands (unless otherwise stated
   by cli-review) should derive from this class and are always in NED mode.
   Additionally, commands that are not setting a numeric attribute MUST provide a
   `disabled` form of the command.

   Please be sure to check in with cli-review@arista.com if implementing a new
   command in either NED mode or always-NED mode.
   '''
   _disabledKeyword = 'disabled'
   _disabledKeywordHelpDesc = "Explicitly disable this configuration"
   _disabledMatcher = CliMatcher.KeywordMatcher(
         _disabledKeyword,
         helpdesc=_disabledKeywordHelpDesc
   )
   data = {
      'disabled': _disabledMatcher
   }

   @classmethod
   def _createSyntaxData( cls, dataDict ):
      '''
      Helper method for performing the requisite parent class data dictionary copy,
      and update with the passed args.  This is done to streamline the process of
      creating new commands, and to help prevent errors e.g. where the base class
      data dictionary is not copied before updating.  The data dictionary may still
      be created/modified manually.
      '''
      data = cls.data.copy()
      data.update( dataDict )
      return data

   @staticmethod
   def _handleNormal( mode, args ):
      raise NotImplementedError

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      '''
      The parameter 'noOrDefault' must be handled if the command using this method
      uses the 'disabled' keyword.
      '''
      raise NotImplementedError

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      cls._handleNoOrDefault( mode, args, NoOrDefault.DEFAULT )

   @classmethod
   def handler( cls, mode, args ):
      if 'disabled' in args:
         cls._handleNoOrDefault( mode, args, NoOrDefault.NO )
      else:
         cls._handleNormal( mode, args )

class BgpNEDCmdBaseClass( BgpCmdBaseClass ):
   '''
   Base class for commands where NED mode is dependent on the running
   configuration (config is not always in NED mode).  If adding a brand new command,
   chances are that NED mode is always "on" for your command, so use BgpCmdBaseClass
   instead.

   This base class should only be used for Bgp commands which were added (NOT
   converted) prior to November of 2019.
   '''
   data = {
      'disabled': NedModeKwMatcher(
            BgpCmdBaseClass._disabledKeyword,
            BgpCmdBaseClass._disabledKeywordHelpDesc
      )
   }

   @staticmethod
   def _handleNormal( mode, args ):
      raise NotImplementedError

   @staticmethod
   def _handleNoOrDefault( mode, args, noOrDefault ):
      raise NotImplementedError

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      # This needs to override the parent class as NED mode in this case is
      # dependent on the running config of the device.
      if CliCommand.isNoCmd( args ) and not bgpConfig.noEqualsDefaultMode:
         noOrDefault = NoOrDefault.NO
      else:
         noOrDefault = NoOrDefault.DEFAULT
      cls._handleNoOrDefault( mode, args, noOrDefault )

#
# Rt Membership configuration
#
if BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   updateConfigAttrsAfMap( { 'rt-membership': [ 'RtMembership' ] } )

class RouterBgpBaseAfRtMembershipMode( RoutingBgpBaseAfMode,
                                       BasicCli.ConfigModeBase ):
   name = 'Route Target VPN route membership address family configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpAfRtMembershipModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

RouterBgpBaseAfRtMembershipMode.addModelet( RouterBgpAfRtMembershipModelet )

if BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   afModeExtensionHook.addAfModeExtension( 'rt-membership',
                                           RouterBgpBaseAfRtMembershipMode )
