# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import abc
from functools import partial

import CliParser
from CliMode.Mroute import RoutingMulticastMode, RoutingMulticastVrfMode
from CliMode.Mroute import RoutingMulticastAfMode
import ConfigMount, LazyMount
import BasicCli
import BasicCliUtil
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
import Tac
import Tracing


AddressFamily = Tac.Type( "Arnet::AddressFamily" )
traceHandle = Tracing.Handle( "MrouteCli" )
t0 = traceHandle.trace0
t1 = traceHandle.trace1


def getVrfNameFromMode( mode ):
   if hasattr( mode, 'vrfName'):
      return mode.vrfName
   return DEFAULT_VRF

def getAddressFamilyFromMode( mode, legacy=False ):
   ''' Returns: ipv4 or ipv6 or ipunknown.
      ipunknown is used for something common with ipv4 and ipv6'''
   if not legacy and hasattr( mode, 'af' ):
      return mode.af
   return AddressFamily.ipv4

mcastGenConfig = {}

def doConfigMounts( entityManager, configTypes, **kwargs ):
   '''
   Performs config mounts for each config type for both
   Ipv4 and Ipv6 Address family
   configType: List of ConfigType
   return : Dictionary{ key: (af,type); value: mount
   '''
   #TODO: Mount the common config
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      for typeName in configTypes:
         tacType = Tac.Type( typeName )
         assert hasattr( tacType, 'mountPath' )
         if kwargs:
            path = tacType.mountPath( af, *kwargs.values() )
         else:
            path = tacType.mountPath( af )
         mcastGenConfig[ ( af, typeName, str( kwargs.values() ) ) ] = \
                                                         ConfigMount.mount(
                                                                entityManager,
                                                                path, typeName,
                                                                'w' )

families = [ AddressFamily.ipv4, AddressFamily.ipv6 ]

def getVrfConfig( af, vrfName, configType=None, collectionName='vrfConfig',
                                                **kwargs ):
   configColl = getConfigRoot( af, configType, **kwargs )
   if configColl:
      assert hasattr( configColl, collectionName )
      collection = getattr( configColl, collectionName )

      config = collection.get( vrfName )
      if config is None:
         config = collection.newMember( vrfName )
      return config

def getConfigRoot( af, configType=None, **kwargs ):
   '''Returns config root of 'configType' based on the current mode'''
   assert af in families
   key = ( af, configType, str( kwargs.values() ) )

   return mcastGenConfig[ key ] if key in mcastGenConfig else None

def getConfigRootFromMode( mode, configType=None, legacy=False, **kwargs ):
   ''' Returns the Configuration root based on mode'''
   af = getAddressFamilyFromMode( mode, legacy=legacy )
   return getConfigRoot( af, configType, **kwargs )

def getVrfConfigFromMode( mode, configType=None, legacy=False,
                          collectionName='vrfConfig', **kwargs ):
   ''' Returns a vrfConfig based on the current mode.
   configType is defaulted to None to play nice with functools.partial'''
   af = getAddressFamilyFromMode( mode, legacy=legacy )

   vrfName = getVrfNameFromMode( mode )
   return getVrfConfig( af, vrfName, configType, collectionName, **kwargs )

def legacyCliCallback( cliCallback ):
   return partial( cliCallback, legacy=True )

def configGetters( configType, collectionName='vrfConfig' ):
   ''' Returns partial functions to access the varial config collections
   returns : ( configRoot, configRootFromMode, vrfConfig, vrfConfigFromMode) '''

   _configRoot = partial( getConfigRoot, configType=configType )
   _configRootFromMode = partial( getConfigRootFromMode, configType=configType )
   _vrfConfig = partial( getVrfConfig, configType=configType,
                         collectionName=collectionName )
   _vrfConfigFromMode = partial( getVrfConfigFromMode, configType=configType,
                                 collectionName=collectionName )

   return ( _configRoot, _configRootFromMode, _vrfConfig, _vrfConfigFromMode )

lazyMcastGenReadTypes = {}
lazyMcastGenWriteTypes = {}

def doLazyMounts( entityManager, entityTypes, useWriteMount=False, **kwargs ):
   '''
   Performs lazy mounts for each type for both
   Ipv4 and Ipv6 Address family
   entityTypes: List of Entity Types
   return : Dictionary{ key: (af,type); value: mount
   '''
   mountFlag = 'w' if useWriteMount else 'r'
   if useWriteMount:
      typeMap = lazyMcastGenWriteTypes
   else:
      typeMap = lazyMcastGenReadTypes
   #TODO: Mount the common config
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      for typeName in entityTypes:
         tacType = Tac.Type( typeName )
         assert hasattr( tacType, 'mountPath' )
         if kwargs:
            path = tacType.mountPath( af, *kwargs.values() )
         else:
            path = tacType.mountPath( af )
         typeMap[ ( af, typeName, str( kwargs.values() ) ) ] = \
               LazyMount.mount( entityManager, path, typeName, mountFlag )

def lazyGetVrfConfig( af, vrfName, entityType=None, collectionName='vrfConfig',
                      writeMount=False, **kwargs ):
   entityColl = getLazyMountRoot( af, entityType=entityType, writeMount=writeMount,
         **kwargs )
   if entityColl:
      assert hasattr( entityColl, collectionName )
      collection = getattr( entityColl, collectionName )

      entity = collection.get( vrfName )
      if entity is None and writeMount:
         entity = collection.newMember( vrfName )
      return entity

def getLazyMountRoot( af, entityType=None, writeMount=False, **kwargs ):
   '''Returns mount root of 'entityType' based on the current mode'''
   assert af in families
   key = ( af, entityType, str( kwargs.values() ) )

   if writeMount:
      typeMap = lazyMcastGenWriteTypes
   else:
      typeMap = lazyMcastGenReadTypes
   return typeMap.get( key )

def getLazyMountRootFromMode( mode, entityType=None, legacy=False, writeMount=False,
      **kwargs ):
   ''' Returns the entity root based on mode'''
   af = getAddressFamilyFromMode( mode, legacy=legacy )
   return getLazyMountRoot( af, entityType, writeMount=writeMount, **kwargs )

def lazyGetVrfConfigFromMode( mode, entityType=None, legacy=False,
                          collectionName='vrfConfig', writeMount=False, **kwargs ):
   ''' Returns a vrfConfig based on the current mode.
   entityType is defaulted to None to play nice with functools.partial'''
   af = getAddressFamilyFromMode( mode, legacy=legacy )

   vrfName = getVrfNameFromMode( mode )
   return lazyGetVrfConfig( af, vrfName, entityType, collectionName, writeMount,
         **kwargs )

def lazyGetters( entityType, collectionName='vrfConfig', writeMount=False ):
   ''' Returns partial functions to access the varial entity collections
   returns (lazyMountRoot, lazyMountRootFromMode, vrfConfig, vrfConfigFromMode) '''

   _lazyMountRoot = partial( getLazyMountRoot, entityType=entityType,
                             writeMount=writeMount )
   _lazyMountRootFromMode = partial( getLazyMountRootFromMode, entityType=entityType,
                                     writeMount=writeMount )
   _vrfConfig = partial( lazyGetVrfConfig, entityType=entityType,
                         collectionName=collectionName, writeMount=writeMount )
   _vrfConfigFromMode = partial( lazyGetVrfConfigFromMode, entityType=entityType,
                              collectionName=collectionName, writeMount=writeMount )

   return ( _lazyMountRoot, _lazyMountRootFromMode, _vrfConfig, _vrfConfigFromMode )


class RouterModeCallbackBase( object ):
   '''A base class for receiving callbacks about mode changes.
      The derived class implements
      <modeName>ModeEntered( self, **kwargs ): Call back for entering a submode
      to be notified of entering the mode and
      <modeName>ModeDeleted( self, **kwargs ): Call back for deleting a submode
      '''
   __metaclass__ = abc.ABCMeta

   def modeEntered( self, **kwargs ):
      '''Callback for entering the root mode'''
      pass

   def modeDeleted( self, **kwargs ):
      '''Callback for deleting the root mode itself'''

class RouterModeExtension( BasicCliUtil.NonOverridablePrompt ):
   '''Meta class to add a per class attribute 'extension' '''
   def __init__( cls, name, bases, dct ):
      BasicCliUtil.NonOverridablePrompt.__init__( cls, name, bases, dct )
      cls.extensions = []


class RouterModeBase( BasicCli.ConfigModeBase ):
   '''Any Multicast root routermode should derive from this class
      The CliPlugin triggering mode changes should call enterMode, deleteMode,
      enterSubmode and deleteSubmode in the appropriate context. Unfortunately there
      is no easy way of infering this from the mode'''
   # Pytlint is not able to reason abot RouterModeExtension
   # pylint: disable-msg=invalid-metaclass
   __metaclass__ = RouterModeExtension

   def __init__( self, parent, session ):
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @classmethod
   def registerCallback( cls, extension ):
      assert isinstance( extension, RouterModeCallbackBase )
      # pylint: disable-msg=E1101
      # 'extensions' is defined by the meta
      cls.extensions.append( extension )

   @classmethod
   def notify( cls, callbackName, **kwargs ):
      t0( "Router mode callback : %s( %s )" % ( callbackName, kwargs ) )
      # pylint: disable-msg=E1101
      # 'extensions' is defined by the meta
      for ext in cls.extensions:
         if hasattr( ext, callbackName ):
            t1( "Calling extension: %s" % ext )
            callback = getattr( ext, callbackName )
            callback( **kwargs )
         else:
            t1( "Skipping extension: %s" % ext )

   @classmethod
   def enterMode( cls, **kwargs ):
      cls.notify( "modeEntered", **kwargs )

   @classmethod
   def deleteMode( cls, **kwargs ):
      cls.notify( "modeDeleted", **kwargs )

   @classmethod
   def enterSubmode( cls, name, **kwargs ):
      callbackName = name + "ModeEntered"
      cls.notify( callbackName, **kwargs )

   @classmethod
   def deleteSubmode( cls, name, **kwargs ):
      callbackName = name + "ModeDeleted"
      cls.notify( callbackName, **kwargs )


class RouterMulticastMode( RoutingMulticastMode, RouterModeBase ):
   ''' config-router-multicast mode'''
   name = 'Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      self.af = AddressFamily.ipunknown
      RoutingMulticastMode.__init__( self, None )
      RouterModeBase.__init__( self, parent, session )


class RouterMulticastVrfMode( RoutingMulticastVrfMode, BasicCli.ConfigModeBase ):
   ''' config-router-multicast-vrf mode'''
   name = 'Multicast VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.af = AddressFamily.ipunknown
      self.vrfName = vrfName
      RoutingMulticastVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMulticastDefaultIpv4Mode( RoutingMulticastAfMode,
                                      BasicCli.ConfigModeBase ):
   ''' config-router-multicast-ipv4 mode
       Register Ipv4 specific default vrf only or vrf agnostic commands here'''
   name = 'IPv4 Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RoutingMulticastAfMode.__init__( self, DEFAULT_VRF, AddressFamily.ipv4 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )


class RouterMulticastVrfIpv4Mode( RoutingMulticastAfMode, BasicCli.ConfigModeBase ):
   ''' config-router-multicast-vrf-*-ipv4 mode
       Register Ipv4, non-default vrf specific commands here'''
   name = 'IPv4 Multicast vrf configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName != DEFAULT_VRF
      RoutingMulticastAfMode.__init__( self, vrfName, AddressFamily.ipv4 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )


class RouterMulticastDefaultIpv6Mode( RoutingMulticastAfMode,
                                      BasicCli.ConfigModeBase ):
   ''' config-router-multicast-ipv6 mode
       Register Ipv6 specific default vrf only or vrf agnostic commands here'''
   name = 'IPv6 Multicast configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RoutingMulticastAfMode.__init__( self, DEFAULT_VRF, AddressFamily.ipv6 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMulticastVrfIpv6Mode( RoutingMulticastAfMode, BasicCli.ConfigModeBase ):
   ''' config-router-multicast-vrf-*-ipv6 mode
       Register Ipv6, non-default vrf specific commands here'''
   name = 'IPv6 Multicast vrf configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName != DEFAULT_VRF
      RoutingMulticastAfMode.__init__( self, vrfName, AddressFamily.ipv6 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )


class RouterMulticastSharedModelet( CliParser.Modelet ):
   ''' Add commands under both config-router-multicast and
       config-router-multicast-vrf-*'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = AddressFamily.ipunknown

class RouterMulticastAfSharedModelet( CliParser.Modelet ):
   '''Modelet that has commands available only under ipv* Submodes
      Add commands that have ipAddress or ACLs as arguments here'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

class RouterMulticastAfCommonSharedModelet( CliParser.Modelet ):
   '''Modelet that has commands that is common for ipv* sub modes and
   router multicast submode
   Add commands that dont have arguments that are ipAddresses or ACL'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

class RouterMulticastIpv4Modelet( CliParser.Modelet ):
   '''Modelet that has commands available only under ipv4 Submodes
      Add commands that have ipv4 Address or ipv4 ACLs as arguments here'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

class RouterMulticastIpv6Modelet( CliParser.Modelet ):
   '''Modelet that has commands available only under ipv6 Submodes
      Add commands that have ipv6 Address or ipv6 ACLs as arguments here'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

class RouterMulticastNonVrfAfModelet( CliParser.Modelet ):
   '''Add commands to config-router-multicast-ipv4 and config-router-multicast-ipv6
      modes. Add address family specific commands that are vrf agnostic'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

# config-router-multicast
RouterMulticastMode.addModelet( RouterMulticastSharedModelet )
# config-router-multicast-vrf-*
RouterMulticastVrfMode.addModelet( RouterMulticastSharedModelet )
# config-router-multicast-ipv4
RouterMulticastDefaultIpv4Mode.addModelet( RouterMulticastIpv4Modelet )
RouterMulticastDefaultIpv4Mode.addModelet( RouterMulticastNonVrfAfModelet )
RouterMulticastDefaultIpv4Mode.addModelet( RouterMulticastAfSharedModelet )
RouterMulticastDefaultIpv4Mode.addModelet( RouterMulticastAfCommonSharedModelet )
# config-router-multicast-vrf-*-ipv4
RouterMulticastVrfIpv4Mode.addModelet( RouterMulticastIpv4Modelet )
RouterMulticastVrfIpv4Mode.addModelet( RouterMulticastAfSharedModelet )
RouterMulticastVrfIpv4Mode.addModelet( RouterMulticastAfCommonSharedModelet )
# config-router-multicast-ipv6
RouterMulticastDefaultIpv6Mode.addModelet( RouterMulticastIpv6Modelet )
RouterMulticastDefaultIpv6Mode.addModelet( RouterMulticastNonVrfAfModelet )
RouterMulticastDefaultIpv6Mode.addModelet( RouterMulticastAfSharedModelet )
RouterMulticastDefaultIpv6Mode.addModelet( RouterMulticastAfCommonSharedModelet )
# config-router-multicast-vrf-*-ipv6
RouterMulticastVrfIpv6Mode.addModelet( RouterMulticastIpv6Modelet )
RouterMulticastVrfIpv6Mode.addModelet( RouterMulticastAfSharedModelet )
RouterMulticastVrfIpv6Mode.addModelet( RouterMulticastAfCommonSharedModelet )

# TODO: Enable when ready to support the common mode CLI versions
# RouterMulticastMode.addModelet( RouterMulticastAfCommonSharedModelet )
# RouterMulticastVrfMode.addModelet( RouterMulticastAfCommonSharedModelet )
