#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from Toggles import (
   IpRibLibToggleLib,
   RoutingLibToggleLib,
   ArnetToggleLib
   )
from CliPlugin import (
   IpAddrMatcher as IpAddr,
   Ip6AddrMatcher as Ip6Addr,
   IraNexthopGroupCli,
   VrfCli,
   IraServiceCli,
   IraIpCli,
)
import BasicCli
import BasicCliModes
import Tracing
import CliCommand
import CliExtensions
import CliMatcher
from CliMode.RouterGeneral import RouterGeneralBaseMode
from CliMode.RouterGeneral import RouterGeneralVrfBaseMode
import CliParser
from CliToken.Router import routerMatcherForConfig
from IpLibTypes import ProtocolAgentModelType
import LazyMount
import ConfigMount
import Tac

traceHandle = Tracing.Handle( 'RouterGeneralCli' )
t5 = traceHandle.trace5 # Info
# NOTE:
# This is expected to be the dumpyard for routing config knobs that
# do not fit into any of the existing protocols (so a whole lot
# of not so related code is expected to come in here).
#
# Configuration knobs:
# rtr# router general
#      ### config applicable to all vrfs goes here
#      ### Example:
#      ### [no|default] next-hops fec dedicated
#
# Vrf submode - All vrf specific configuration will go into this mode
# NOTE: default is also considered as a vrf, and user is expected to
# input 'vrf default' to configure for default vrf.
#
# rtr# router general
# router-general# vrf <vrfname>
#                 ### Example:
#                 ### routes dynamic prefix-list <name>
#
# To simplify the code organization the following is the approach
# (similar ip commands which is present is a whole lot of places).
#
# - Any configuration which is a single knob can be added in 
#   RouterGeneral package (this file itself). This is a straight
#   forward case and nothing special is required.
#   The config is mounted at : routing/general/config/global
#
# - Any considerable configuration, which may need its own sm to
#   to sync configuration to gated, should prefferably be in its
#   own package (at the very least we could have separate CliPlugin
#   and tacc SM files).
#   To support config-replace and no/default for the above case the
#   new package will have to register callback handlers (using 
#   CliExtensions.CliHook() ).
#   The config is mounted at : routing/general/config/<somename>
#

defaultRouterIdV4 = Tac.Value( 'Arnet::IpAddr' ).ipAddrZero
defaultRouterIdV6 = Tac.Value( 'Arnet::Ip6Addr' )

#----------------------------------------------------------------------------------
#       Support for registering cleanup handlers
#----------------------------------------------------------------------------------
routerGeneralCleanupHook = CliExtensions.CliHook() 
routerGeneralVrfCleanupHook = CliExtensions.CliHook() 

#----------------------------------------------------------------------------------
#       Helpers
#----------------------------------------------------------------------------------
def getRouterConfig( vrfName ):
   if vrfName:
      return generalConfig.vrfConfig.get( vrfName, None )
   return generalConfig

def orderedEcmpSupportedGuard( mode, token ):
   if routingHwStatus.orderedEcmpSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def hwSupportedGuard( mode, token ):
   # Guards against enabling cli in single-agent mode and checks routingSupported
   if IraServiceCli.getEffectiveProtocolModel( mode ) == \
         ProtocolAgentModelType.multiAgent and routingHwStatus.routingSupported:
      return None
   return CliParser.guardNotThisPlatform

def getIpv4Routable240ClassE():
   return generalConfig.ipv4Routable240ClassE

IraIpCli.ipv4Routable240ClassEHook.addExtension( getIpv4Routable240ClassE )

#----------------------------------------------------------------------------------
#                                CLI MODES
# - Global mode applicable to all vrfs
# - Per vrf mode, specific to vrfs 
#   (default vrf config will be under 'vrf default' mode)
#----------------------------------------------------------------------------------
class RouterGeneralMode( RouterGeneralBaseMode, BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.
   '''
   name = 'Protocol independent routing configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      RouterGeneralBaseMode.__init__( self, param='generalconfig' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def deleteModeConfig( self ):
      for vrfName in generalConfig.vrfConfig:
         routerGeneralVrfCleanupHook.notifyExtensions( vrfName )
         del generalConfig.vrfConfig[ vrfName ]

class RouterGeneralVrfMode( RouterGeneralVrfBaseMode, BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.
   '''
   name = 'Protocol independent routing vrf configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      RouterGeneralVrfBaseMode.__init__( self, self.vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getOrCreateVrfConfig( self, vrfName ):
      vrfConfig = generalConfig.vrfConfig.get( vrfName, None )
      if not vrfConfig:
         vrfConfig = generalConfig.vrfConfig.newMember( vrfName )
      return  vrfConfig

   def deleteVrfConfig( self, vrfName ):
      # If therere any registed cleanup handlers call them
      routerGeneralVrfCleanupHook.notifyExtensions( vrfName )
      del generalConfig.vrfConfig[ vrfName ]


#----------------------------------------------------------------------------------
#                                CLI MODELET
# A modelet for global / per vrf modes each.
#----------------------------------------------------------------------------------
class RouterGeneralBaseModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrf = getattr( mode, 'vrfName', '' )


class RouterGeneralModelet( RouterGeneralBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterGeneralBaseModelet.__init__( self, mode )


class RouterGeneralVrfModelet( RouterGeneralBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterGeneralBaseModelet.__init__( self, mode )

#----------------------------------------------------------------------------------
#                                K E Y W O R D S
#----------------------------------------------------------------------------------
matcherGeneral = CliMatcher.KeywordMatcher( 'general', 
   'Protocol independent routing configuration' )
matcherNexthops  = CliMatcher.KeywordMatcher( 'next-hops', 
   'Next hop configuration' )
matcherFec       = CliMatcher.KeywordMatcher( 'fec', 
   'Forward Equivalence Class configuration' )
matcherDedicated = CliMatcher.KeywordMatcher( 'dedicated', 
   'Use dedicated FEC per next hop' )
matcherIpv4Config = CliMatcher.KeywordMatcher( 'ipv4',
    'IPv4 configuration commands' )
matcherRoutable = CliMatcher.KeywordMatcher( "routable",
    'Enable reserved-address IPv4 routing' )
matcher240_4 = CliMatcher.KeywordMatcher( "240.0.0.0/4",
    'Enable reserved-address IPv4 routing' )
matcherRouterId = CliMatcher.KeywordMatcher( 'router-id', 
   'Configure a general router ID for all routing processes' )
matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4', 
   'General router ID in IP address format' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6', 
   'General router ID in IPv6 address format' )
matcherV4Addr = IpAddr.IpAddrMatcher(
   "General router ID in IP address format" ) 
matcherV6Addr = Ip6Addr.Ip6AddrMatcher(
   "General router ID in IPv6 address format" )
matcherResolutionOverAggregates = CliMatcher.KeywordMatcher(
   'resolution', helpdesc='Next hop recursive resolution' )
matcherRoute = CliMatcher.KeywordMatcher( 'route',
   'Route commands' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
   'Static routes' )
matcherNexthopGroup = CliMatcher.KeywordMatcher( 'nexthop-group',
   'Nexthop groups' )
matcherUnresolved = CliMatcher.KeywordMatcher( 'unresolved',
   'Unreachable destinations' )
matcherInvalid = CliMatcher.KeywordMatcher( 'invalid',
   'Do not install in routing table' )

#----------------------------------------------------------------------------------
#                           T A C     I N S T A N C E S
#----------------------------------------------------------------------------------
generalConfig = None  # Routing::General::Config
routingHwStatus = None  # Routing::Hardware::Status

#----------------------------------------------------------------------------------
#                               C O M M A N D S
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# "[ no | default ] next-hops fec dedicated" 
# in "router general" mode
#--------------------------------------------------------------------------------
class NexthopsFecDedicatedCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops fec dedicated'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'fec' : matcherFec,
      'dedicated' : matcherDedicated,
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecPerNexthop = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecPerNexthop = False
   
RouterGeneralModelet.addCommandClass( NexthopsFecDedicatedCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp emulated"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibEcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp emulated'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'emulated': 'Emulate',
   }

   @staticmethod
   def handler( mode, args):
      generalConfig.fecEcmpEmulated = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpEmulated = False

RouterGeneralModelet.addCommandClass( RibFibEcmpCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp ordered"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibOrderedCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp ordered'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'ordered': CliCommand.guardedKeyword( 'ordered',
                                            'Enforce order of next hops in FEC',
                                            guard=orderedEcmpSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecEcmpOrdered = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpOrdered = False

if RoutingLibToggleLib.toggleOrderedEcmpEnabled():
   RouterGeneralModelet.addCommandClass( RibFibOrderedCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] ipv4 routable 240.0.0.0/4"
# in "router general" mode
#--------------------------------------------------------------------------------
class Ipv4ReservedClassEAddressingCmd( CliCommand.CliCommandClass ):
   syntax = '''ipv4 routable 240.0.0.0/4'''
   noOrDefaultSyntax = syntax

   data = { "ipv4": CliCommand.Node( matcher=matcherIpv4Config,
                                     guard=hwSupportedGuard ),
            "routable": matcherRoutable,
            "240.0.0.0/4": matcher240_4,
          }

   @staticmethod
   def handler( mode, args ):
      generalConfig.ipv4Routable240ClassE = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.ipv4Routable240ClassE = False

if ArnetToggleLib.toggleIpv4Routable240ClassEEnabled():
   RouterGeneralModelet.addCommandClass( Ipv4ReservedClassEAddressingCmd )


#--------------------------------------------------------------------------------
# "[ no | default ] router-id ipv4 <ip-address>" 
# in "router general/ vrf" mode
#--------------------------------------------------------------------------------
class RouterIdIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 IP_ADDR'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id': matcherRouterId,
      'ipv4': matcherIpv4,
      'IP_ADDR': matcherV4Addr,
   }

   @staticmethod
   def handler( mode, args ):
      modeletClass = mode.modeletClasses[ 0 ]
      modelet = mode.modeletMap[ modeletClass ]
      routerIdV4 = args[ 'IP_ADDR' ]
      if routerIdV4 == defaultRouterIdV4:
         mode.addError( '%s is not a valid router ID' % routerIdV4 )
         return
      routerConfig = getRouterConfig( modelet.vrf )
      routerConfig.routerIdV4 = routerIdV4

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      modeletClass = mode.modeletClasses[ 0 ]
      modelet = mode.modeletMap[ modeletClass ]
      routerConfig = getRouterConfig( modelet.vrf )
      routerConfig.routerIdV4 = defaultRouterIdV4

RouterGeneralModelet.addCommandClass( RouterIdIpv4Cmd )
RouterGeneralVrfModelet.addCommandClass( RouterIdIpv4Cmd )

#--------------------------------------------------------------------------------
# "[ no | default ] router-id ipv6 <ipv6-address>" command in both "router general"
# in "router general/ vrf" mode
#--------------------------------------------------------------------------------
class RouterIdIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 IP_ADDR'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id': matcherRouterId,
      'ipv6': matcherIpv6,
      'IP_ADDR': matcherV6Addr,
   }

   @staticmethod
   def handler( mode, args ):
      modeletClass = mode.modeletClasses[ 0 ]
      modelet = mode.modeletMap[ modeletClass ]
      routerIdV6 = args[ 'IP_ADDR' ]
      if routerIdV6 == defaultRouterIdV6:
         mode.addError( '%s is not a valid router ID' % routerIdV6 )
         return
      routerConfig = getRouterConfig( modelet.vrf )
      routerConfig.routerIdV6 = routerIdV6

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      modeletClass = mode.modeletClasses[ 0 ]
      modelet = mode.modeletMap[ modeletClass ]
      routerConfig = getRouterConfig( modelet.vrf )
      routerConfig.routerIdV6 = defaultRouterIdV6

RouterGeneralModelet.addCommandClass( RouterIdIpv6Cmd )
RouterGeneralVrfModelet.addCommandClass( RouterIdIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] router general
#--------------------------------------------------------------------------------
class RouterGeneralCmd( CliCommand.CliCommandClass ):
   syntax = 'router general'
   noOrDefaultSyntax = 'router general ...'
   data = {
      'router': routerMatcherForConfig,
      'general': 'Protocol independent routing configuration',
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go from 'config' mode to 'router general' mode
      '''
      childMode = mode.childMode( RouterGeneralMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecPerNexthop = False
      generalConfig.fecEcmpEmulated = False
      generalConfig.fecEcmpOrdered = False
      generalConfig.fecEcmpMixTunnelIp = False
      generalConfig.routerIdV4 = defaultRouterIdV4
      generalConfig.routerIdV6 = defaultRouterIdV6
      generalConfig.resolveOverAggregates = False
      generalConfig.resolveNexthopGroupVias = False
      generalConfig.ipv4Routable240ClassE = False
      routerGeneralCleanupHook.notifyExtensions( mode=mode )
      childMode = mode.childMode( RouterGeneralMode )
      childMode.deleteModeConfig()

BasicCliModes.GlobalConfigMode.addCommandClass( RouterGeneralCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfCli.VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                    inclDefaultVrf=True ),
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go from 'router general' mode to vrf submode
      '''
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( RouterGeneralVrfMode, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateVrfConfig( vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF' ]
      childMode = mode.childMode( RouterGeneralVrfMode, vrfName=vrfName )
      childMode.deleteVrfConfig( vrfName=vrfName )

RouterGeneralMode.addCommandClass( VrfCmd )

RouterGeneralMode.addModelet( RouterGeneralModelet )
RouterGeneralVrfMode.addModelet( RouterGeneralVrfModelet )

#--------------------------------------------------------------------------------
# [ no | default ] next-hops resolution bgp aggregates allowed
#--------------------------------------------------------------------------------
class NexthopResAggsAllowedCmd( CliCommand.CliCommandClass ):
   syntax = 'next-hops resolution bgp aggregates allowed'
   noOrDefaultSyntax = syntax
   data = {
      'next-hops' : matcherNexthops,
      'resolution': matcherResolutionOverAggregates,
      'bgp': 'Recursive resolution over BGP aggregates',
      'aggregates': 'Recursive resolution over aggregates',
      'allowed': 'Recursive resolution over aggregates',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.resolveOverAggregates = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.resolveOverAggregates = False

RouterGeneralMode.addCommandClass( NexthopResAggsAllowedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] route static nexthop-group unresolved invalid
#--------------------------------------------------------------------------------
if RoutingLibToggleLib.toggleCheckFecResolutionStatusEnabled():
   class RouteStaticNexthopGroupUnresolvedCmd( CliCommand.CliCommandClass ):
      syntax = 'route static nexthop-group unresolved invalid'
      noOrDefaultSyntax = syntax
      data = {
         'route' : matcherRoute,
         'static': matcherStatic,
         'nexthop-group': CliCommand.Node(
                              matcherNexthopGroup,
                              guard=IraNexthopGroupCli.nexthopGroupSupportedGuard ),
         'unresolved': matcherUnresolved,
         'invalid': matcherInvalid,
      }

      @staticmethod
      def handler( mode, args ):
         generalConfig.resolveNexthopGroupVias = True

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         generalConfig.resolveNexthopGroupVias = False

   RouterGeneralMode.addCommandClass( RouteStaticNexthopGroupUnresolvedCmd )

#--------------------------------------------------------------------------------
# "[ no | default ] rib fib fec ecmp compatible tunnel ip"
# in "router general" mode
#--------------------------------------------------------------------------------
class RibFibEcmpMixTunnelIpCmd( CliCommand.CliCommandClass ):
   syntax = 'rib fib fec ecmp compatible tunnel ip'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Routing table',
      'fib': 'FIB',
      'fec': matcherFec,
      'ecmp': 'ECMP',
      'tunnel': 'Tunnel Vias',
      'compatible' : 'Compatible',
      'ip': 'IP Vias',
   }

   @staticmethod
   def handler( mode, args ):
      generalConfig.fecEcmpMixTunnelIp = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      generalConfig.fecEcmpMixTunnelIp = False

if IpRibLibToggleLib.toggleMixedEcmpEnabled():
   RouterGeneralModelet.addCommandClass( RibFibEcmpMixTunnelIpCmd )

t5( 'Loaded RouterGeneralCli' )

def Plugin( entityManager ):
   global generalConfig
   global routingHwStatus

   generalConfig = ConfigMount.mount( entityManager, 'routing/general/config/global',
                                      'Routing::General::Config', 'w' )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
