# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import time
from ArnetModel import IpGenericPrefix, IpGenericAddress, Ip4Address, Ip6Address
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from ArnetLib import asnStrToNum
from RouteMapLib import ( bgpOriginEnum,
                          asPathEntryFlags,
                          inactiveAclMsg,
                          SetActionInfo,
                          capiMedTypes,
                          capiAigpMetricTypes,
                          duplicateOperationToCliStr,
                          renderRange,
                          POLICY_COMM_INST_MIN,
                          POLICY_ASPATH_LENGTH_MIN,
                          CommunityType,
                          matchContribAggAttrContribStr,
                          sourceProtocolEnumToRouteProtoMap,
                          originAsValidityEnumToCapi,
                          originAsValidityCapiToCli,
)
from IntfModels import Interface
import Tac
from Toggles import RouteMapToggleLib
from IpRibLibCliModels import ResolutionRibProfileConfig

# args:
# - level: the number of levels/tabs to indent by
# - spacesPerLevel: the size of each level/tab
def indent( level, spacesPerLevel=3, extraSpaces="" ):
   res = " " * ( level * spacesPerLevel )
   return res + extraSpaces

#-------------------------------------------------------------------------------
# "show ip[v6] prefix-list"
#-------------------------------------------------------------------------------

class IpPrefixEntry( Model ):
   seqno = Int( help="Sequence number of prefix list entry" )
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   prefix = IpGenericPrefix( help="Subnet upon which command filters routes" )
   ge = Int( optional=True, help="Minimum prefix length" )
   le = Int( optional=True, help="Maximum prefix length" )
   eq = Int( optional=True, help="Exact prefix length" )

class IpPrefixList( Model ):
   ipPrefixEntries = List( valueType=IpPrefixEntry, 
      help="Prefix list entries sorted by increasing sequence number" )
   ipPrefixListSource = Str( help="Source URL of the prefix list",
                             optional=True )
   duplicateHandling = Str(
      help="Enable handling of duplicate entries in URL prefix list",
      optional=True )
   
   def renderPrefixList( self, listName, isIpv6, inPrefixListMode, output=None ):
      if not output:
         output = sys.stdout

      if self.ipPrefixListSource is not None:
         ipCmd = "ipv6" if isIpv6 else "ip"
         cliStr = duplicateOperationToCliStr( self.duplicateHandling )
         if cliStr is None:
            output.write( "%s prefix-list %s source %s" % (
               ipCmd, listName, self.ipPrefixListSource ) )
         else:
            output.write( "%s prefix-list %s source %s %s" % (
               ipCmd, listName, self.ipPrefixListSource, cliStr ) )
         output.write( "\n" )
         return

      # no newline for displaying pfx lists configured with one line syntax
      if not isIpv6 and not inPrefixListMode:
         pfxListKeyword = "ip prefix-list %s" % ( listName )

      if isIpv6:
         output.write( "ipv6 prefix-list %s\n" % ( listName ) )
      elif inPrefixListMode:
         output.write( "ip prefix-list %s\n" % ( listName ) )
      
      for s in self.ipPrefixEntries:
         if isIpv6 or inPrefixListMode:
            msg = "    seq %s %s %s" % ( s.seqno, s.filterType, s.prefix )
         else:
            msg = "%s seq %s %s %s" % ( pfxListKeyword, s.seqno, s.filterType, 
                                        s.prefix )
         if s.eq is None:
            if s.ge != 0:
               msg += " ge %s" % s.ge
            if s.le != 0:
               msg += " le %s" % s.le
         else:
            msg += " eq %s" % s.eq
         output.write( msg + "\n" )

class IpPrefixLists( Model ):
   _isIpv6 = Bool( help="In IPv6 Address Family" )
   _inPrefixListMode = Bool( help="Indicates format for printing prefix lists" )
   ipPrefixLists = Dict( keyType=str, valueType=IpPrefixList,
                         help="Map prefix list name to prefix list" )
   _filename = Str( help=
         "Private variable that holds the filename for url based prefix lists" )
   _errMsg = Str( help = 
         "Private variable that holds error message for url based prefix lists" )

   def render( self ):
      if self._errMsg: 
         print self._errMsg
      elif self._filename is not None: 
         fh = open( self._filename, "r" )
         while( True ): 
            line = fh.readline()
            if not line:
               break
            print line.rstrip()
         fh.close()
         Tac.run( [ "rm", "-f", self._filename ], asRoot=True, 
                      stdout=Tac.DISCARD )
      else: 
         for listName, prefixList in sorted( self.ipPrefixLists.iteritems() ):
            prefixList.renderPrefixList( listName, self._isIpv6, 
                                         self._inPrefixListMode )
#-------------------------------------------------------------------------------
# "show ip prefix-list summary"
#-------------------------------------------------------------------------------

class PrefixListSummary( Model ):
   _invalidListName = Bool( help="No list found for given prefix list name",
                            optional=True )
   entryCount = Int( help="Prefix list entry count", optional=True )
   pfxListSource = Str( help="Prefix list source", optional=True )
   sourceTimeStamp = Float( help="Prefix list source timestamp", optional=True )

   def render( self ):
      if self._invalidListName:
         print "List not found"
      else:
         if self.pfxListSource == "":
            print "Num entries: %s" % self.entryCount
         elif self.sourceTimeStamp == 0:
            print "Failed to import from source: %s" % self.pfxListSource
         else:
            tm = time.localtime( self.sourceTimeStamp )
            tmStr = time.strftime( "%Y-%m-%d %H:%M:%S", tm )
            print "Source: %s" % self.pfxListSource
            print "Num entries: %s" % self.entryCount
            print "Update time: %s" % tmStr

#-------------------------------------------------------------------------------
# "show ip as-path access-list"
#-------------------------------------------------------------------------------

class IpAsPathEntry( Model ):
   userRegex = Str( help="Regex describing AS path being filtered" )
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   origin = Enum( values=( "bgpAny", "bgpEGP", "bgpIGP", "bgpIncomplete" ), 
                  help="Origin of path information" )
   valid = Bool( optional=True,
                 help="Entry validity given current regex mode " +
                 "for this as-path entry list" )


class IpAsPathList( Model ):
   ipAsPathEntries = List( valueType=IpAsPathEntry, help="AS path entries" )
   ipAsPathListSource = Str( help="Source URL of the as-path access list",
                             optional=True )
   
   def renderAsPathList( self, listName ):
      if self.ipAsPathListSource != "" :
         print "ip as-path access-list %s source %s" % ( listName,
               self.ipAsPathListSource )
         return
      for s in self.ipAsPathEntries:
         print "%s ip as-path access-list %s %s %s %s" % \
               ( asPathEntryFlags[ s.valid ], listName, 
                 s.filterType, s.userRegex, 
                 bgpOriginEnum[ s.origin ] )

class IpAsPathLists( Model ):
   regexMode = Enum( values=( "asn", "string" ), help="Regex mode" )
   activeIpAsPathLists = Dict( keyType=str, valueType=IpAsPathList, 
                         help="Active AS path lists under the current regex mode" )
   inactiveIpAsPathLists = Dict( keyType=str, valueType=IpAsPathList, 
                         help="Inactive AS path lists under the current regex mode" )
   _filename = Str( help=
             "Private variable that holds the filename for url based as-path " \
                   "access lists" )
   _errMsg = Str( help=
            "Private variable that holds error message for url based as-path " \
                  "access lists" )

   def render( self ):
      s = "AS-path entry codes: # - entry is invalid in " \
            "the current regex mode (%s)\n"
      print s % self.regexMode
      if self._errMsg: 
         print self._errMsg
      elif self._filename: 
         print "ip as-path regex-mode %s" % self.regexMode
         fh = open( self._filename, "r" )
         while( True ):
            line = fh.readline()
            if not line:
               break
            print line.rstrip()
         fh.close()
         Tac.run( ["rm", "-f", self._filename], asRoot = True, stdout=Tac.DISCARD )
      else: 
         print "ip as-path regex-mode %s" % self.regexMode
         for listName, asPathList in sorted( self.activeIpAsPathLists.iteritems() ):
            asPathList.renderAsPathList( listName )
      
         if self.inactiveIpAsPathLists:
            print inactiveAclMsg
            for listName, pathList in \
                        sorted( self.inactiveIpAsPathLists.iteritems() ):
               pathList.renderAsPathList( listName )

#-------------------------------------------------------------------------------
# "show ip as-path access-list <name> summary"
#-------------------------------------------------------------------------------

class IpAsPathSummary( Model ):
   _invalidListName = Bool( help="No list found for given access list name",
                            optional=True )
   entryCount = Int( help="Access list entry count", optional=True )
   accessListSource = Str( help="Access list source", optional=True )
   sourceTimeStamp = Float( help="Access list source timestamp", optional=True )

   def render( self ):
      if self._invalidListName:
         print "List not found"
      else:
         if self.accessListSource == "":
            print "Num entries: %s" % self.entryCount
         elif self.sourceTimeStamp == 0:
            print "Failed to import from source: %s" % self.accessListSource
         else:
            tm = time.localtime( self.sourceTimeStamp )
            tmStr = time.strftime( "%Y-%m-%d %H:%M:%S", tm )
            print "Source: %s" % self.accessListSource
            print "Num entries: %s" % self.entryCount
            print "Update time: %s" % tmStr

#-------------------------------------------------------------------------------
# "show ip [ext|large-]community-list"
# 
# JSON output examples:
# {'ipCommunityLists': {'com1': {'entries': [{'communityValues': ['no-advertise'],
#                                              'filterType': 'permit',
#                                              'listType': 'standard'},
#                                             {'communityValues': ['0:5',
#                                                                  '25:100',
#                                                                  'local-as'],
#                                              'filterType': 'deny',
#                                              'listType': 'standard'}]},
#                        'com2': {'entries': [{'communityRegex': '1*:20*',
#                                              'filterType': 'permit',
#                                              'listType': 'expanded'}]}}}
#-------------------------------------------------------------------------------

class IpCommunityListEntry( Model ):
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   listType = Enum( values=( "standard", "expanded" ), help="Community list type" )
   communityValues = List( valueType=str, help="Community list values",
                           optional=True )
   communityRegex = Str( help="Communities as a regular expression",
                         optional=True )

   def renderCommunityListEntry( self, listName,
                                 commType=CommunityType.communityTypeStandard ):
      prefix = ""
      if commType == CommunityType.communityTypeLarge:
         prefix = "large-"
      elif commType == CommunityType.communityTypeExtended:
         prefix = "ext"
      expr = " ".join( self.communityValues ) if self.listType == "standard" \
             else self.communityRegex
      print "ip %scommunity-list %s%s %s %s" % (
            prefix, 'regexp ' if self.listType == 'expanded' else '', listName,
            self.filterType, expr )

class IpCommunityList( Model ):
   entries = List( valueType=IpCommunityListEntry,
      help="Community list entries sorted by increasing sequence number" )
   
class IpExtCommunityList( Model ):
   entries = List( valueType=IpCommunityListEntry,
      help="Extended community list entries sorted by increasing sequence number" )

class IpLargeCommunityList( Model ):
   entries = List( valueType=IpCommunityListEntry,
      help="Large community list entries sorted by increasing sequence number" )

class IpCommunityLists( Model ):
   ipCommunityLists = Dict( keyType=str, valueType=IpCommunityList,
                            help="Table of community lists by name" )

   def render( self ):
      iteritems = sorted( self.ipCommunityLists.iteritems() )
      for listName, ipCommunityList in iteritems:
         for ipCommunityListEntry in ipCommunityList.entries:
            ipCommunityListEntry.renderCommunityListEntry( listName )

class IpExtCommunityLists( Model ):
   ipExtCommunityLists = Dict( keyType=str, valueType=IpExtCommunityList,
                               help="Table of extended community lists by name" )

   def render( self ):
      iteritems = sorted( self.ipExtCommunityLists.iteritems() )
      for listName, ipExtCommunityList in iteritems:
         for ipCommunityListEntry in ipExtCommunityList.entries:
            ipCommunityListEntry.renderCommunityListEntry( listName,
                                                CommunityType.communityTypeExtended )

class IpLargeCommunityLists( Model ):
   ipLargeCommunityLists = Dict( keyType=str, valueType=IpLargeCommunityList,
                                 help="Table of large community lists by name" )

   def render( self ):
      iteritems = sorted( self.ipLargeCommunityLists.iteritems() )
      for listName, ipLargeCommunityList in iteritems:
         for ipCommunityListEntry in ipLargeCommunityList.entries:
            ipCommunityListEntry.renderCommunityListEntry( listName,
                                                   CommunityType.communityTypeLarge )

class AggregateContributor( Model ):
   """ Model to represent aggregate contributor match clause for
       'match aggregate-role contributor aggregate-attributes <MAP>' command,
       which stores the role and optionally an aggregate attributes map
   """
   aggregateRole = Enum( values=[ matchContribAggAttrContribStr ],
                         help='Role in contributor-aggregate relation' )
   aggregateAttributes = Str(
         help='Route map against which to evalute the aggregate route',
         optional=True )

   def renderToString( self ):
      if self.aggregateAttributes:
         return "%s" % self.aggregateAttributes
      else:
         return ""

class CommInstancesRange( Model ):
   """ Model to represent a range for 'match community instances' command,
   which stores a start and end values for the range.
   """
   start = Int( help='The minimum number of communities' )
   end = Int( help='The maximum number of communities' )

   def renderToString( self ):
      return renderRange( self.start, self.end,
                          POLICY_COMM_INST_MIN )

class AsPathLengthRange( Model ):
   """ Model to represent a range for 'match as-path length' command,
   which stores a start and end values for the range.
   """
   start = Int( help='The minimum length of the as-path' )
   end = Int( help='The maximum length of the as-path' )

   def renderToString( self ):
      return renderRange( self.start, self.end,
                          POLICY_ASPATH_LENGTH_MIN )

class AttributeDebug( Model ):
   undefined = Str( help="String representation of undefined element",
                    optional=True )
   ignored = Str( help="String representation of element that may be ignored",
                  optional=True )

class DebugInfo( Model ):
   community = Submodel( valueType=AttributeDebug,
                         help="Debugging information for match community "
                         "lists", optional=True )
   communityAddOrReplace = Submodel( valueType=AttributeDebug,
                                     help="Debugging information for add/replace "
                                     "community lists", optional=True )
   communityDelete = Submodel( valueType=AttributeDebug,
                               help="Debugging information for delete community "
                               "lists", optional=True )
   communityFilter = Submodel( valueType=AttributeDebug,
                               help="Debugging information for filter community "
                               "lists", optional=True )
   largeCommunity = Submodel( valueType=AttributeDebug,
                              help="Debugging information for large community lists",
                              optional=True )
   largeCommunityFilter = Submodel( valueType=AttributeDebug,
                                    help="Debugging information for filter large "
                                    "community lists", optional=True )
   extCommunity = Submodel( valueType=AttributeDebug,
                            help="Debugging information for extended community "
                            "lists", optional=True )
   extCommunityFilter = Submodel( valueType=AttributeDebug,
                                  help="Debugging information for filter extended "
                                  "community lists", optional=True )
   matchResult = Enum( values=( "matched", "failed", "unsupported" ),
                       help="Result of the matcher", optional=True )

# additional model layer for consistency with comm-list debug info
class SubRMDebugInfo( Model ):
   name = Submodel( valueType=AttributeDebug,
                    help="Debugging information for sub-route-maps",
                    optional=True )

#-------------------------------------------------------------------------------
# "show route-map"
#
# JSON output examples:
# {'routeMaps': {'map1':
#     {'entries': {'10': {'continueClause': True,
#                         'continueSeqno': 20,
#                         'description': [],
#                         'filterType': 'permit',
#                         'matchRules': {},
#                         'setRules': {}},
#                  '20': {'continueClause': True,
#                         'description': [],
#                         'filterType': 'deny',
#                         'matchRules': {},
#                         'setRules': {}},
#                  '30': {'description': ['This is map1'],
#                         'filterType': 'permit',
#                         'matchRules': {},
#                         'setRules': {}}}}}}
#
#{'routeMaps': {'map5':
#     {'entries': {'100': {'continueClause': True,
#                          'continueSeqno': 20,
#                          'description': ['second entry'],
#                          'filterType': 'deny',
#                          'matchRules': {'accessList': 'al4',
#                                         'asNumber': 100,
#                                         'asPathList': 'apl',
#                                         'asPathLength': {
#                                              start: 20,
#                                              end: 40
#                                          }
#                                         'community': 'comm',
#                                         'communityExactMatch': true,
#                                         'communityInvertResult': true,
#                                         'interface': 'Ethernet5',
#                                         'ipv6NextHopPrefixList': 'pl6',
#                                         'ipv6ResolvedNextHopPrefixList': 'pl6',
#                                         'localPref': 1000,
#                                         'nextHop': '5.1.1.1',
#                                         'nextHopPrefixList': 'pl4',
#                                         'ospfMetricType': 'type-2',
#                                         'prefixList': 'pl4',
#                                         'protocol': 'ospf',
#                                         'resolvedNextHopPrefixList': 'pl4',
#                                         'routeMetric': 1000,
#                                         'isisLevel': 'level-2',
#                                         'isisInstance' : 1,
#                                         'routeTag': 4000,
#                                         'ospfInstance' : 1 },
#                          'setRules': {'communityDelete':
#                                          {'communityLists': [ 'com1' ],
#                                           'operationType': 'add'},
#                                       'extCommunity': {'operationType': 'none'},
#                                       'largeCommunity': {'operationType': 'none'},
#                                       'nextHop': {'nextHopType': 'peerAddress'},
#                                       'origin': 'bgpIncomplete',
#                                       'ospfMetricType': 'type-2',
#                                       'routeMetric': {'metricType': 'subtractive',
#                                                       'value': 2000},
#                                       'routeTag': 1000,
#                                       'isisLevel': 'level-1-2'
#                                       'weight': 100},
#                                       'segmentIndex': 100},
#-------------------------------------------------------------------------------

class MatchRules( Model ):
   __revision__ = 3
   _populatedFields = List( valueType=str, help="Configured attributes" )
   _matchCommands = List( valueType=str, help="Configured match commands" )
   accessList = Str( help="IPv4 address filtered by access control list",
                     optional=True )
   prefixList = Str( help="IPv4 address filtered by IP prefix list", optional=True )
   nextHop = Ip4Address( help="IPv4 next hop filtered by IP address",
                          optional=True )
   nextHopPrefixList = Str( help="IPv4 next hop filtered by IP prefix list",
                            optional=True )
   resolvedNextHopPrefixList = Str( help="IPv4 resolved nexthop filtered by "
                                    "IP prefix list", optional=True )
   asNumber = Str( help="BGP autonomous system number", optional=True )
   asPathList = Str( help="BGP autonomous system path access list", optional=True )
   # variable name must match value of matchContribAggAttrCapiName
   aggregation = Submodel( valueType=AggregateContributor,
                           help="Match contributors to BGP aggregates",
                           optional=True )
   aggregationInvertResult = \
         Bool( help="Invert match contributors to BGP aggregates",
               optional=True )

   asPathLength = Submodel( valueType=AsPathLengthRange,
                            help="Match on as-path length",
                            optional=True )
   asPathLengthInvertResult = Bool( help="Invert of as-path length match",
                                    optional=True )
   community = Str( help="BGP community", optional=True )
   extCommunity = Str( help="BGP extended community", optional=True )
   largeCommunity = Str( help="BGP large community", optional=True )
   communityExactMatch = Bool( help="Exact match of communities", optional=True )
   communityOrResults = Bool( help="Or results of community lists", optional=True )
   communityInvertResult = Bool( help="Invert match community result",
                                 optional=True )
   communityInstances = Submodel( valueType=CommInstancesRange,
                                  help="Match on the number of "
                                       "community instances",
                                  optional=True )
   communityInstancesInvertResult = Bool( help="Invert match community"
                                               " instances result",
                                          optional=True )
   extCommunityInvertResult = Bool( help="Invert match extcommunity result",
                                    optional=True )
   largeCommunityInvertResult = Bool( help="Invert match large community result",
                                      optional=True )
   interface = Interface( help="Interface name", optional=True )
   localPref = Int( help="Bgp local preference metric", optional=True )
   routeMetric = Int( help="Route metric", optional=True )
   ospfMetricType = Enum( values=( "type-1", "type-2" ),
                          help="OSPF external LSA metric type", optional=True )
   isisLevel = Enum( values=( "level-1", "level-2" ),
                     help="IS-IS route level", optional=True )
   isisInstance = Int( help="IS-IS route instance", optional=True )
   protocol = Enum( values=sourceProtocolEnumToRouteProtoMap.values(),
                    help="Routing protocol of route's source", optional=True )
   routeTag = Int( help="Route tag", optional=True )
   ipv6PrefixList = Str( help="IPv6 address filtered by IPv6 prefix list",
                         optional=True )
   ipv6NextHop = Ip6Address( help="IPv6 next hop filtered by IPv6 address",
                             optional=True )
   ipv6NextHopPrefixList = Str( help="IPv6 next hop filtered by "
                                "IPv6 prefix list", optional=True )
   ipv6ResolvedNextHopPrefixList = Str( help="IPv6 resolved nexthop filtered "
                                        "by IPv6 prefix list", optional=True )
   routerIdPrefixList = Str( help="Router ID filtered by prefix list",
                   optional=True )
   routeType = Enum( values=( "local", "internal", "external",
                              "confederation-external", "vpn" ), help="Route type",
                              optional=True )
   ipAddrDynamicPrefixList = Str( help="Ip address dynamic prefix list name",
                                  optional=True )
   ipv6AddrDynamicPrefixList = Str( help="Ipv6 address dynamic prefix list name",
                                    optional=True )
   debugInfo = Submodel( valueType=DebugInfo,
                         help="Debugging information for match rules",
                         optional=True )
   originAsValidity = Enum( values=( originAsValidityEnumToCapi.values() ),
                            help="AS Origin validation state",
                            optional=True )
   originAsValidityInvertResult = Bool(
      help="Invert match AS Origin validation state",
      optional=True )
   originAs = Str( help="Route origin AS number", optional=True )
   ospfInstance = Int( help="OSPF route instance", optional=True )

   def renderMatchRules( self, indentation='', evaluationResult="" ):
      for field, command in zip( self._populatedFields, self._matchCommands ):
         value = self.matchValue( field )
         invertResultString = self.isInvertMatch( field )
         exactMatchString = self.isExactMatch( field )
         orResultsString = self.isOrMatch( field )

         print "%s" % ( indentation ) \
               + "match " \
               + invertResultString \
               + command \
               + orResultsString + " " \
               + str( value ) \
               + exactMatchString \
               + evaluationResult

   def renderConfigSanityMatchRules( self ):
      for field, command in zip( self._populatedFields, self._matchCommands ):
         value = self.matchValue( field )
         invertResultString = self.isInvertMatch( field )
         exactMatchString = self.isExactMatch( field )
         orResultsString = self.isOrMatch( field )
         undefinedCommLists = None
         undefinedMsg = None
         text = []

         if command == 'community' and self.debugInfo.community and \
            self.debugInfo.community.undefined:
            undefinedCommLists = self.debugInfo.community.undefined
         elif command == 'extcommunity' and self.debugInfo.extCommunity and \
              self.debugInfo.extCommunity.undefined:
            undefinedCommLists = self.debugInfo.extCommunity.undefined
         elif command == 'large-community' and self.debugInfo.largeCommunity and \
              self.debugInfo.largeCommunity.undefined:
            undefinedCommLists = self.debugInfo.largeCommunity.undefined
         if undefinedCommLists:
            text.append( 'undefined' )
            if len( value.split( ' ' ) ) > 1:
               text.append( undefinedCommLists )
            undefinedMsg = "(" + " ".join( text ) + ")"
         elif command in( 'ip address prefix-list',
                          'ip address access-list',
                          'ip address dynamic prefix-list',
                          'ipv6 address prefix-list',
                          'ipv6 address dynamic prefix-list',
                          'as-path',
                          'aggregate-role contributor aggregate-attributes',
                          'router-id prefix-list' ):
            undefinedMsg = "(undefined)"
         if undefinedMsg:
            print "  match %s%s%s %s%s %s" % ( invertResultString, command,
                                               orResultsString, str( value ),
                                               exactMatchString, undefinedMsg )

   def degrade( self, dictRepr, revision ):

      if revision < 3:
         # in RouteMaps revision 1, 2 asNumber was a Int
         # containing Autonomous System Number.
         # asNumber is Str from revision 3
         if 'asNumber' in dictRepr:
            asnInStr = dictRepr[ 'asNumber' ]
            dictRepr[ 'asNumber' ] = asnStrToNum( asnInStr )
         return dictRepr
      return dictRepr

   def isInvertMatch( self, field ):
      if field == "community" and self.communityInvertResult or \
         field == "extCommunity" and self.extCommunityInvertResult or \
         field == "largeCommunity" and self.largeCommunityInvertResult or \
         field == "communityInstances" and self.communityInstancesInvertResult or \
         field == "aggregation" and self.aggregationInvertResult or \
         field == "asPathLength" and self.asPathLengthInvertResult or \
         field == "originAsValidity" and self.originAsValidityInvertResult:
         return "invert-result "
      else:
         return ""

   def isExactMatch( self, field ):
      if field in [ "community",
                    "extCommunity",
                    "largeCommunity" ] and self.communityExactMatch:
         return " exact-match"
      else:
         return ""

   def isOrMatch( self, field ):
      if field in [ "community",
                    "extCommunity",
                    "largeCommunity" ] and self.communityOrResults:
         return " or-results"
      else:
         return ""

   def matchValue( self, field ):
      value = getattr( self, field )
      if field == "interface":
         value = value.stringValue
      elif field == "communityInstances" or field == "asPathLength" or \
            field == "aggregation":
         value = value.renderToString()
      elif field == "originAsValidity":
         value = originAsValidityCapiToCli[ value ]
      return value



class CommunitySet( Model ):
   __revision__ = 2
   operationType = Enum( values=( "default", "add", "delete", "filter", "none" ),
                         help="Community set operation type" )
   communityLists = List( valueType=str, help="Community list names",
                                optional=True )
   communityValues = List( valueType=str, help="Community list values",
                           optional=True )
   communityFilterTypes = List( valueType=str, help="Community filter type values",
                               optional=True )
   def degrade( self, dictRepr, revision ):
      if revision < 2:
         if not "communityLists" in dictRepr:
            return dictRepr
         if len( dictRepr[ "communityLists" ] ) == 0:
            return dictRepr
         # first revision communityList was a string
         # containing a single community-list name
         # see BUG143367
         if self.communityLists is not None:
            dictRepr[ "communityList" ] = dictRepr[ "communityLists" ][ 0 ]
            del dictRepr[ "communityLists" ]
      return dictRepr

class NextHop( Model ):
   nextHopType = Enum( values=( "ipAddress", "peerAddress", "ipv6PeerAddress",
                                "unchanged", "ipv6Unchanged", "evpnUnchanged" ),
                       help="Next hop type" )
   ipAddress = IpGenericAddress( help="Next hop IP address", optional=True )

class OspfBit( Model ):
   ospfBitType = Enum( values=( "dn", ),
                       help="Set an OSPF protocol bit" )

class RouteMetric( Model ):
   value = Int( help="Route metric value" )
   metricType = Enum( values=capiMedTypes.keys(),
                      help="Additive or subtractive metric value",
                      optional=True )

def renderRouteMetric( metric ):
   if metric.metricType == "additive":
      return "+" + str( metric.value )
   if metric.metricType == "subtractive":
      return "-" + str( metric.value )
   if metric.metricType == "igp-nexthop-cost":
      return "igp-nexthop-cost"
   if metric.metricType == "add-igp-nexthop-cost":
      return "+igp-nexthop-cost"
   if metric.metricType == "value-add-igp-nexthop-cost":
      return "%s +igp-nexthop-cost" % metric.value
   if metric.metricType == "igp-metric":
      return "igp-metric"
   if metric.metricType == "add-igp-metric":
      return "+igp-metric"
   if metric.metricType == "value-add-igp-metric":
      return "%s +igp-metric" % metric.value
   assert metric.metricType == "metric"
   return str( metric.value )

class AigpMetric( Model ):
   value = Int( help="AIGP metric value" )
   metricType = Enum( values=capiAigpMetricTypes,
                      help="AIGP metric from IGP metric or IGP next-hop-cost",
                      optional=True )

def renderAigpMetric( metric ):
   if metric.metricType == "igp-next-hop-cost":
      return "igp next-hop-cost"
   if metric.metricType == "igp-metric":
      return "igp metric"
   assert metric.metricType == "metric"
   return str( metric.value )

class SetRules( Model ):
   _populatedFields = List( valueType=str, help="Configured attributes" )
   _setCommands = List( valueType=str, help="Configured set commands" )
   asPathPrepend = Str( help="BGP AS path to prepend", optional=True )
   asPathPrependLastAs = Int( help="Number of times to prepend the last AS number",
                              optional=True )
   asPathReplace = Str( help="BGP AS path to replace", optional=True )
   nextHop = Submodel( valueType=NextHop, help="Next hop information",
                       optional=True )
   localPref = Int( help="BGP local preference metric", optional=True )
   routeMetric = Submodel( valueType=RouteMetric, help="Route Metric",
                           optional=True )
   aigpMetric = Submodel( valueType=AigpMetric, help="AIGP Metric",
                          optional=True )
   ospfMetricType = Enum( values=( "type-1", "type-2" ),
                          help="OSPF external LSA metric type", optional=True )
   isisLevel = Enum( values=( "level-1", "level-2", "level-1-2" ),
                     help="IS-IS route level", optional=True )
   origin = Enum( values=( "bgpEGP", "bgpIGP", "bgpIncomplete" ), 
                  help="BGP origin of path information", optional=True  )
   distance = Int( help="Protocol independent administrative distance",
                   optional=True )
   communityAddOrReplace = Submodel( valueType=CommunitySet, help="Community add"
                                     "or replace modifications", optional=True )
   communityDelete = Submodel( valueType=CommunitySet, help="Community delete"
                               "modifications", optional=True )
   communityFilter = Submodel( valueType=CommunitySet, help="Community filter"
                               "modifications", optional=True )
   extCommunity = Submodel( valueType=CommunitySet, help="Extended community"
                            "modifications", optional=True )
   extCommunityFilter = Submodel( valueType=CommunitySet, help="Extended community"
                                  "filter modifications", optional=True )
   largeCommunity = Submodel( valueType=CommunitySet, help="Large community"
                              "modifications", optional=True )
   largeCommunityFilter = Submodel( valueType=CommunitySet, help="Large community"
                                    "filter modifications", optional=True )

   routeTag = Int( help="Route tag", optional=True )
   weight = Int( help="BGP weight parameter", optional=True )
   bpAsPathWeight = Int( help="BGP bestpath AS path weight parameter",
                         optional=True )
   segmentIndex = Int( help="MPLS Segment Routing Index", optional=True )
   igpMetric = Int( help="IGP metric", optional=True )
   debugInfo = Submodel( valueType=DebugInfo,
                         help="Debugging information for set rules", optional=True )
   originAsValidity = Enum( values=( originAsValidityEnumToCapi.values() ),
                            help="AS Origin validation state",
                            optional=True )
   if RouteMapToggleLib.toggleMatchRouteTypeVpnSetOspfBitDnEnabled():
      ospfBitDn = Submodel( valueType=OspfBit,
                            help="DN bit to prevent routing loops", optional=True )
   if RouteMapToggleLib.toggleSetResolutionRibProfileEnabled():
      resolutionRibProfileConfig = Submodel( valueType=ResolutionRibProfileConfig,
                                             optional=True,
                                             help="Resolution RIBs" )

   def renderConfigSanitySetRules( self ):
      for field, command in zip( self._populatedFields, self._setCommands ):
         value = getattr( self, field )
         text = ""
         undefinedMsg = ""
         ignoredMsg = ""
         fieldToValue = {
            "communityAddOrReplace": ( "community-list", "community" ),
            "communityDelete": ( "community-list", "community" ),
            "communityFilter": ( "community-list", "community" ),
            "extCommunity": ( "extcommunity-list", "extCommunity" ),
            "extCommunityFilter": ( "extcommunity-list", "extCommunity" ),
            "largeCommunity": ( "large-community-list", "largeCommunity" ),
            "largeCommunityFilter": ( "large-community-list", "largeCommunity" ),
            }
         if field in fieldToValue.keys():
            if value.communityLists:
               text = "{} ".format( fieldToValue[ field ][ 0 ] )\
                      + " ".join( value.communityLists )
            text = text.rstrip()
            if value.operationType != "default":
               space = "" if text == "" else " "
               if value.operationType == "add":
                  op = "additive"
               else:
                  op = value.operationType
               text += "%s" % space + op
               if op == "filter":
                  for commType in value.communityFilterTypes:
                     text += " %s" % commType
            communityDebugInfo = getattr( self.debugInfo, field )
            if communityDebugInfo and communityDebugInfo.undefined:
               undefinedComms = str( communityDebugInfo.undefined ).split()
               if len( value.communityLists ) == 1 and len( undefinedComms ) == 1:
                  undefinedMsg = "undefined"
               else:
                  undefinedMsg = "undefined %s" % communityDebugInfo.undefined

            if communityDebugInfo and communityDebugInfo.ignored:
               ignoredComms = str( communityDebugInfo.ignored ).split( ' ' )
               if len( value.communityLists ) == 1 and len( ignoredComms ) == 1:
                  ignoredMsg = "regexp entries ignored, not supported in set clause"
               else:
                  ignoredMsg = "regexp entries ignored in %s, not supported in set" \
                               " clause" % communityDebugInfo.ignored

            if value.operationType == "delete":
               if undefinedMsg:
                  print "  set %s %s (%s)" % ( command, text, undefinedMsg )
            else:
               if undefinedMsg and ignoredMsg:
                  print "  set %s %s (%s, %s)" % ( command, text, undefinedMsg,
                                                   ignoredMsg )
               elif undefinedMsg:
                  print "  set %s %s (%s)" % ( command, text, undefinedMsg )
               elif ignoredMsg:
                  print "  set %s %s (%s)" % ( command, text, ignoredMsg )

   def renderSetRules( self, indentation='' ):
      for field, command in zip( self._populatedFields, self._setCommands ):
         value = getattr( self, field )
         text = ""
         colorText = ""
         if field in ( "communityAddOrReplace", "communityDelete", "communityFilter",
                       "extCommunity", "extCommunityFilter", "largeCommunity",
                       "largeCommunityFilter" ):
            if value.communityValues:
               onlyLbwValues = True
               for commVal in value.communityValues:
                  if "lbw" in commVal:
                     # As in RouteMapLib.printRouteMapEntryAttributes(),
                     # the code for 'show running-config and 'show pending',
                     # here we also split the extcommunity attribute into
                     # lbw-values and non-lbw-values when rendering.
                     print ( "%sset %s %s%s" %
                        ( indentation, command, commVal,
                          "" if value.operationType == "default" else
                          " additive" if value.operationType == "add" else
                          " " + value.operationType ) )
                  elif "color" in commVal:
                     colorText += commVal + " "
                     onlyLbwValues = False
                  else:
                     text += commVal + " "
                     onlyLbwValues = False
               if onlyLbwValues:
                  continue
            elif value.communityLists:
               if field == "extCommunity" or field == "extCommunityFilter":
                  text = "extcommunity-list "
               elif field == "largeCommunity" or field == "largeCommunityFilter":
                  text = "large-community-list "
               else:
                  text = "community-list "
               text += " ".join( value.communityLists )
            text = text.rstrip()
            colorText = colorText.rstrip()
            if value.operationType != "default":
               if value.operationType == "add":
                  op = "additive"
               else:
                  op = value.operationType
               if text:
                  text += " %s" % op
               if op == "filter":
                  for commType in value.communityFilterTypes:
                     text += " %s" % commType
               if colorText:
                  colorText += " %s" % op
         elif field == "routeMetric":
            text = renderRouteMetric( value )
         elif field == "aigpMetric":
            text = renderAigpMetric( value )
         elif command == " ".join( SetActionInfo[ "nextHop" ][ 'command' ] ) or \
               command == " ".join( SetActionInfo[ "ipv6NextHop" ][ 'command' ] ):
            text = value.ipAddress
         elif field == "origin":
            text = bgpOriginEnum[ value ]
         elif field == "originAsValidity":
            text = originAsValidityCapiToCli[ value ]
         elif field == "routeTag":
            if value == 0x7fffffffffffffff:
               text = "auto"
            else:
               text = str( value )
         elif field == "resolutionRibProfileConfig":
            text = value.toTac().stringValue()
         else:
            text = str( value )
         cmds = ( " ".join( SetActionInfo[ "nextHopPeerAddr" ][ 'command' ] ),
                  " ".join( SetActionInfo[ "ipv6NextHopPeerAddr" ][ 'command' ] ),
                  " ".join( SetActionInfo[ "nextHopUnchanged" ][ 'command' ] ),
                  " ".join( SetActionInfo[ "ipv6NextHopUnchanged" ][ 'command' ] ),
                  " ".join( SetActionInfo[ "evpnNextHopUnchanged" ]\
                                                   [ 'command' ] ) )
         if RouteMapToggleLib.toggleMatchRouteTypeVpnSetOspfBitDnEnabled():
            cmds += ( " ".join( SetActionInfo[ "ospfBitDn" ][ 'command' ] ), )

         if command in cmds:
            print "%sset %s" % ( indentation, command )
         else:
            if text:
               print "%sset %s %s" % ( indentation, command, text )
            elif value.operationType == 'none' and not value.communityValues:
               print "%sset %s %s" % ( indentation, command, value.operationType )
            if colorText:
               print "%sset %s %s" % ( indentation, command, colorText )

class SubRouteMap( Model ):
   name = Str( help="Sub route map name", optional=False )
   invert = Bool( help="Invert sub route map result", optional=True )
   debugInfo = Submodel( valueType=SubRMDebugInfo,
                         help="Debugging information for sub-route-maps",
                         optional=True )

   def renderSubRouteMap( self, level, indentLevel, spacesPerLevel=2 ):
      print "%sSubRouteMap:" % indent( level, spacesPerLevel=spacesPerLevel,
                                       extraSpaces=indentLevel )
      if self.name:
         subInvert = "invert-result " if self.invert else ""
         print "%ssub-route-map %s%s" % ( indent( level + 1, spacesPerLevel=2,
                                                extraSpaces=indentLevel ),
                                              subInvert,
                                              self.name )

   def renderConfigSanitySubRouteMap( self, spacesPerLevel=2 ):
      if self.name:
         subInvert = "invert-result " if self.invert else ""
         print "%ssub-route-map %s%s (undefined)" % (
            indent( 1, spacesPerLevel=spacesPerLevel ),
            subInvert,
            self.name )

   def renderExpandedSubRouteMap( self, allRouteMaps, rmPath, level,
                                  pruned, renderedRMs ):
      rms = set( [] )
      subInvert = "invert-result " if self.invert else ""
      if self.name and self.name in allRouteMaps:
         subRouteMap = allRouteMaps[ self.name ]
         if subRouteMap.entries: # sub-route-map is defined
            if pruned and self.name in renderedRMs:
               print "%ssub-route-map %s%s (already rendered)" % (
                  indent( level ), subInvert, self.name )
            else:
               print "%ssub-route-map %s%s" % (
                  indent( level ), subInvert, self.name )

            rms = subRouteMap.renderExpandedRouteMap( self.name,
                                                      allRouteMaps,
                                                      rmPath, level,
                                                      subRM=True, pruned=pruned,
                                                      renderedRMs=renderedRMs )
      elif self.name: # sub-route-map is undefined
         print "%ssub-route-map %s%s" % ( indent( level ), subInvert,
                                            self.name )
         if self.name in rmPath:
            print '%scycle detected' % indent( level + 1 )
      return rms

class RouteMapEntry( Model ):
   __revision__ = 5
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   matchRules = Submodel( valueType=MatchRules, help="Match rules" )
   subRouteMap = Submodel( valueType=SubRouteMap, help="Sub route map",
                           optional=True )
   setRules = Submodel( valueType=SetRules, help="Set rules" )
   continueClause = Bool( help="Continue to next clause", optional=True )
   continueSeqno = Int( help="Sequence number of next clause", optional=True )
   description = List( valueType=str, help="Route map entry description" )

   def degrade( self, dictRepr, revision ):
      # in revision 1, subRouteMap was a Str; revisions 2, 3 & 4 skipped
      if revision < 5:
         dictRepr[ 'subRouteMap' ] = self.subRouteMap.name if self.subRouteMap \
                                     else ""
      return dictRepr

   def renderRouteMapEntry( self, mapName, seqno, indentLevel='' ):
      level = 0
      print "%sroute-map %s %s %s" % ( indent( level, spacesPerLevel=2,
                                               extraSpaces=indentLevel ),
                                       mapName, self.filterType, seqno )
      level = level + 1
      print "%sDescription:" % indent( level, spacesPerLevel=2,
                                       extraSpaces=indentLevel )
      for desc in self.description:
         print "%sdescription %s" % ( indent( level + 1, spacesPerLevel=2,
                                              extraSpaces=indentLevel ), desc )

      print "%sMatch clauses:" % indent( level, spacesPerLevel=2,
                                         extraSpaces=indentLevel )
      self.matchRules.renderMatchRules(
         indentation=indent( level + 1, spacesPerLevel=2,
                             extraSpaces=indentLevel ) )

      if self.subRouteMap:
         self.subRouteMap.renderSubRouteMap( level, indentLevel )

      if self.continueClause:
         if self.continueSeqno:
            print "%sContinue: sequence %d" % ( indent( level, spacesPerLevel=2,
                                                        extraSpaces=indentLevel ),
                                                self.continueSeqno )
         else:
            print "%sContinue: next sequence" % indent( level, spacesPerLevel=2,
                                                        extraSpaces=indentLevel )

      print "%sSet clauses:" % indent( level, spacesPerLevel=2,
                                       extraSpaces=indentLevel )
      self.setRules.renderSetRules( indentation=indent( level + 1, spacesPerLevel=2,
                                                        extraSpaces=indentLevel ) )

   def renderConfigSanityRouteMapEntry( self, mapName, seqno ):
      print "route-map %s %s %s" % ( mapName, self.filterType, seqno )

      self.matchRules.renderConfigSanityMatchRules()

      if self.subRouteMap:
         self.subRouteMap.renderConfigSanitySubRouteMap( )

      if self.continueClause:
         if self.continueSeqno:
            print "%scontinue %d (undefined)" % ( indent( 1, spacesPerLevel=2 ),
                                                  self.continueSeqno )
         else:
            print "%scontinue (undefined)" % indent( 1, spacesPerLevel=2 )

      self.setRules.renderConfigSanitySetRules()

class RouteMap( Model ):
   entries = Dict( keyType=int, valueType=RouteMapEntry,
                   help="Route map entries by sequence number" )

   def renderRouteMap( self, mapName, indentLevel='' ):
      for seqno, routeMapEntry in sorted( self.entries.iteritems() ):
         routeMapEntry.renderRouteMapEntry( mapName, seqno, indentLevel )

   def renderConfigSanityRouteMap( self, mapName ):
      for seqno, routeMapEntry in sorted( self.entries.iteritems() ):
         routeMapEntry.renderConfigSanityRouteMapEntry( mapName, seqno )

class RouteMaps( Model ):
   __revision__ = 5
   routeMaps = Dict( keyType=str, valueType=RouteMap, help="Route maps by name" )

   def renderRouteMaps( self, indentLevel='' ):
      for mapName, routeMap in sorted( self.routeMaps.iteritems() ):
         routeMap.renderRouteMap( mapName, indentLevel )

   def render( self ):
      self.renderRouteMaps()

class ExpandedRouteMapEntry( Model ):
   __revision__ = 5
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   matchRules = Submodel( valueType=MatchRules, help="Match rules" )
   subRouteMap = Submodel( valueType=SubRouteMap, help="Sub route map",
                           optional=True )
   setRules = Submodel( valueType=SetRules, help="Set rules" )
   continueClause = Bool( help="Continue to next clause", optional=True )
   continueSeqno = Int( help="Sequence number of next clause", optional=True )
   description = List( valueType=str, help="Route map entry description" )

   def degrade( self, dictRepr, revision ):
      # in revision 1, subRouteMap was a Str; revisions 2, 3 & 4 skipped
      if revision < 5:
         dictRepr[ 'subRouteMap' ] = self.subRouteMap.name if self.subRouteMap \
                                     else ""
      return dictRepr

   def renderExpandedRouteMapEntry( self, mapName, seqno, allRouteMaps, rmPath,
                                    pruned, renderedRMs, level=0 ):
      rms = set( [] )
      print "%sseq %s %s" % ( indent( level ), seqno, self.filterType )

      level = level + 1
      for desc in self.description:
         print "%sdescription %s" % ( indent( level ), desc )

      self.matchRules.renderMatchRules( indentation=indent( level ) )

      rms = self.subRouteMap.renderExpandedSubRouteMap(
         allRouteMaps, rmPath, level,
         pruned, renderedRMs )

      self.setRules.renderSetRules( indentation=indent( level ) )

      if self.continueClause:
         if self.continueSeqno:
            print "%scontinue %d" % ( indent( level ), self.continueSeqno )
         else:
            print "%scontinue" % indent( level )
      return rms

class ExpandedRouteMap( Model ):
   entries = Dict( keyType=int, valueType=ExpandedRouteMapEntry,
                   help="Route map entries by sequence number",
                   optional=True )

   def renderExpandedRouteMap( self, mapName, allRouteMaps, rmPath, level=0,
                               subRM=False, pruned=False, renderedRMs=None ):
      rms = set( [] )
      if not subRM:
         print "%sroute-map %s" % ( indent( level ), mapName )
      level = level + 1
      if mapName in rmPath:
         print "%scycle detected" % indent( level )
      elif pruned and mapName in renderedRMs:
         pass
      else:
         for seqno, routeMapEntry in sorted( self.entries.iteritems() ):
            rms = routeMapEntry.renderExpandedRouteMapEntry( mapName, seqno,
               allRouteMaps, rmPath.union( { mapName } ), pruned, renderedRMs,
               level )
            renderedRMs = renderedRMs.union( rms )
         renderedRMs.add( mapName )
      return renderedRMs

class ExpandedRouteMaps( Model ):
   __revision__ = 5
   routeMaps = Dict( keyType=str, valueType=ExpandedRouteMap,
                     help="Route maps by name" )
   mapName = Str( help="Name of specific route map to render", optional=True )
   pruned = Bool( help="Avoid rerendering common subtrees" )

   def renderRouteMaps( self, pruned=False ):
      if self.mapName and self.mapName in self.routeMaps.keys():
         routeMap = self.routeMaps[ self.mapName ]
         routeMap.renderExpandedRouteMap( self.mapName, self.routeMaps,
                                          set( [] ), pruned=self.pruned,
                                          renderedRMs=set( [] ) )
      elif not self.mapName:
         for mapName, routeMap in sorted( self.routeMaps.iteritems() ):
            routeMap.renderExpandedRouteMap( mapName, self.routeMaps, set( [] ),
                                             pruned=self.pruned,
                                             renderedRMs=set( [] ) )

   def render( self ):
      self.renderRouteMaps()

class ConfigSanityRouteMaps( RouteMaps ):
   __revision__ = 5

   def render( self ):
      for mapName, routeMap in sorted( self.routeMaps.iteritems() ):
         routeMap.renderConfigSanityRouteMap( mapName )

#-------------------------------------------------------------------------------
# 'show peer-filter'
#-------------------------------------------------------------------------------
class AsnRange ( Model ):
   start = Int( help='ASN range start' )
   end = Int( help='ASN range end', optional=True )

class PeerFilterMatchRules( Model ):
   _populatedFields = List( valueType=str, help='Configured attributes' )
   _matchCommands = List( valueType=str, help='Configured match commands' )
   asRange = Submodel( valueType=AsnRange, help='BGP ASN range', optional=True )

   def renderMatchRules( self, seqno, level=0, result='accept' ):
      for field, command in zip( self._populatedFields, self._matchCommands ):
         value = getattr( self, field )
         if field == 'asRange':
            if ( value.start != value.end ):
               value = str( value.start ) + '-' + str( value.end )
            else:
               value = str( value.start )
         text = ' ' + command + ' ' + str( value )
      level = level + 1
      print '%s%s match%s result %s' % ( indent( level ), seqno, text, result )

class PeerFilterEntry( Model ):
   filterType = Enum( values=( 'accept', 'reject' ), help='Filter type' )
   matchRules = Submodel( valueType=PeerFilterMatchRules, help='Match rules' )

   def renderPeerFilterEntry( self, filterName, seqno, level=0 ):
      self.matchRules.renderMatchRules( seqno, level=level, result=self.filterType )

class PeerFilter( Model ):
   description = List( valueType=str, help='Peer filter description' )
   entries = Dict( keyType=int, valueType=PeerFilterEntry,
                   help='Peer filter entries by sequence number' )

   def renderPeerFilter( self, filterName ):
      print 'peer-filter %s' % filterName
      level = 1
      for desc in self.description:
         print '%sdescription %s' % ( indent( level ), desc )

      for seqno, peerFilterEntry in sorted( self.entries.iteritems() ):
         peerFilterEntry.renderPeerFilterEntry( filterName, seqno, level=level )

class PeerFilters( Model ):
   peerFilters = Dict( keyType=str, valueType=PeerFilter,
                       help='Peer filters by name' )

   def renderPeerFilters( self ):
      for filterName, peerFilter in sorted( self.peerFilters.iteritems() ):
         peerFilter.renderPeerFilter( filterName )

   def render( self ):
      self.renderPeerFilters()

#------------------------------------------------------------------------------------
# 'show bgp debug policy'
#------------------------------------------------------------------------------------

class PolicyEvalSequenceResult( Model ):
   """
   Stores the results of an evaluated route map sequence.
   """
   result = Enum( values=( 'permit', 'deny', 'fallThrough' ),
                  help="Route map sequence result" )
   continueSeqno = Int( help="Continue to sequence number", optional=True )

   def renderResult( self, level=0, continueNext=None ):
      if self.result == "permit":
         result = "permit"
         if self.continueSeqno:
            # an expection exists here that if the continue statements did not
            # include a seqno then we was to display "continue next" however the
            # result will still contain the seqno we continued to. So a
            # continueNext flag is used to distinguish this case.
            if continueNext:
               result += ", continue next"
            else:
               result += ", continue %d" % self.continueSeqno
      elif self.result == "deny":
         result = "deny"
      else:
         result = "fall through to next sequence"
      print "%sSeq result: %s" % ( indent( level ), result )

class PolicyEvalRouteMapResult( Model ):
   """
   Stores the results of an evaluated route map sequence.
   In the case where a route map permits the matching sequences that
   caused the permit are stored in permittingSeqnos.
   """
   result = Enum( values=( 'permit', 'explicitDeny', 'noMatchingSequenceDeny' ),
                  help="Route map result" )
   permittingSeqnos = List( valueType=int, help="Sequences resulting in permit",
                           optional=True )

   # The render function is passed a bool (subRm) to indicate whether we are
   # rendering a sub route map result or not. In the case we are rendering a
   # sub route map result we do not want to display anything for missing policy.
   # It is passed to the render function as this model does not know if it is
   # for a route map or sub route map.
   def renderResult( self, routeMapName, level=0, notFound=None, subRm=False,
                     cycleFound=None, peerApplication=False ):
      result = ""
      if self.result == "permit":
         result = "permit"
         if self.permittingSeqnos:
            result += ", matching sequence"
            if len( self.permittingSeqnos ) > 1:
               result += "s"
            result += " "
            result += ", ".join( map( str, self.permittingSeqnos ) )
         elif notFound and not subRm:
            if peerApplication:
               result += " (missing policy action is permit)"
            else:
               result += " (route map %s is undefined)" % routeMapName
         elif cycleFound:
            result += " (route map evaluation loop action is permit)"
      elif self.result == "explicitDeny":
         result = "deny"
         if notFound and not subRm:
            if peerApplication:
               result += " (missing policy action is deny)"
            else:
               result += " (route map %s is undefined)" % routeMapName
      else:
         result = "deny due to no matching sequence"
      print "%sRoute map result: %s" % ( indent( level ), result )

class PolicyEvalRouteMap( Model ):
   """
   Stores the components of an evaluated route map.
   This model is part of a recursive model due to sub route maps.
   Since a sub route map requires a model in a sequence, and a sequence is part of a
   route map we can get the following:
   PolicyEvalRouteMap -> PolicyEvalSequence -> PolicyEvalRouteMap
   """
   routeMapName = Str( help="Name of the route map" )
   routeMapResult = Submodel( valueType=PolicyEvalRouteMapResult,
                              help="Result of the route map" )
   notFound = Bool( help="Route map is not configured", optional=True )
   routeMapCycleDetected = Bool( help="Route map is part of a cycle", optional=True )

   # A sequences attribute is added later after PolicyEvalSequence is declared.

   def renderMap( self, level=0, subRm=False, subRMInvertResult=None,
                  peerApplication=False ):
      if not subRm:
         print "%sroute-map %s" % ( indent( level ), self.routeMapName )
      else:
         if self.notFound:
            result = "(not found)"
         elif self.routeMapCycleDetected:
            result = "(call to route map from within itself not evaluated)"
         else:
            if self.routeMapResult.result == "permit":
               result = "(permit)" if not subRMInvertResult else\
                        "(invert permit to deny)"
            else:
               result = "(deny)" if not subRMInvertResult else\
                        "(invert deny to permit)"

         subInvert = "invert-result " if subRMInvertResult else ""
         print "%ssub-route-map %s%s %s" % ( indent( level ),
                                              subInvert, self.routeMapName,
                                              result )

      level = level + 1

      if self.sequences:
         seqs = sorted( self.sequences )
         for seq in seqs:
            self.sequences[ seq ].renderSequence( seq, level=level )

      self.routeMapResult.renderResult( self.routeMapName,
                                        level=level,
                                        notFound=self.notFound,
                                        subRm=subRm,
                                        cycleFound=self.routeMapCycleDetected,
                                        peerApplication=peerApplication )

class PolicyEvalSequence( Model ):
   """
   Stores the components of a configured sequence which are evaluated or applied.
   """
   filterType = Enum( values=( "permit", "deny" ), help="Filter type" )
   matcherResults = List( valueType=MatchRules, help="Evaluated matcher results",
                          optional=True )
   subRouteMap = Submodel( valueType=PolicyEvalRouteMap,
                           help="Sub route map evaluation",
                           optional=True )
   subRouteMapInvertResult = Bool( help="Invert result of the sub route map",
                                   optional=True )
   continueNextSequence = Bool( help="Continue to next sequence", optional=True )
   continueSeqno = Int( help="Continue to sequence number", optional=True )
   missingContinueTarget = Bool( help="Sequence contains a continue statement to a "
                                 "target which is not configured.", optional=True )
   descriptions = List( valueType=str, help="Route map sequence description",
                       optional=True )
   setRules = Submodel( valueType=SetRules, help="Applied set rules", optional=True )
   sequenceResult = Submodel( valueType=PolicyEvalSequenceResult,
                              help="Sequence result" )

   def renderSequence( self, seqno, level=0 ):
      print "%sseq %s %s" % ( indent( level ), seqno, self.filterType )

      level = level + 1

      if self.descriptions:
         for desc in self.descriptions:
            print "%sdescription %s" % ( indent( level ), desc )

      if self.matcherResults:
         for matcher in self.matcherResults:
            result = " (%s)" % matcher.debugInfo.matchResult
            matcher.renderMatchRules( indentation=indent( level ),
                                      evaluationResult=result )

      if self.subRouteMap:
         self.subRouteMap.renderMap( level, subRm=True,
                                     subRMInvertResult=self.subRouteMapInvertResult )

      if self.setRules:
         self.setRules.renderSetRules( indentation=indent( level ) )

      missing = ""
      if self.continueSeqno:
         # If the continue target is missing then we add this to the output
         if self.missingContinueTarget:
            missing = " (ignored: sequence %d is missing)" % self.continueSeqno
         print "%scontinue %d%s" % ( indent( level ), self.continueSeqno, missing )
      elif self.continueNextSequence:
         if self.missingContinueTarget:
            missing = " (ignored: no next sequence)"
         print "%scontinue%s" % ( indent( level ), missing )

      self.sequenceResult.renderResult( level=level,
                                        continueNext=self.continueNextSequence )

# See the comments in PolicyEvalRouteMap for more detail.
# Now that PolicyEvalSequence has been decalared we can add a reference to
# this type to PolicyEvalRouteMap. This is required as the models are
# recursive, ie PolicyEvalSequence also refers to PolicyEvalRouteMap
PolicyEvalRouteMap.__attributes__[ "sequences" ] \
   = Dict( keyType=int, valueType=PolicyEvalSequence,
           help="Evaluated sequences", optional=True )

class PolicyEvalPeerNlriResult( Model ):
   """
   Stores the policy evaluation results for a particular peer.
   """
   routeMapEval = Submodel( PolicyEvalRouteMap, optional=True,
                        help="Evaluated route map" )

   def render( self ):
      if self.routeMapEval:
         self.routeMapEval.renderMap( peerApplication=True )
      else:
         print "No route map applied to peer"

class PolicyEvalMultiPeersNlriResult( Model ):
   """
   Stores the policy evaluation results for a particular prefix.
   """
   # Note that IpGenPrefix is not used here since we will want to display
   # prefixs of afi safi other than ipv4 and v6 at some point.
   nlri = Str( help="Evaluated NLRI" )
   peers = Dict( keyType=IpGenericAddress, valueType=PolicyEvalPeerNlriResult,
                 help="Evaluation result for each peer" )
   application = Enum( values=( "inbound", "outbound" ),
                       help="Point of application debugged" )

   def render( self ):
      first = True
      for peer in sorted( self.peers ):
         if not first:
            # Add two lines before printing a new peer
            print "\n"
         first = False
         if self.application == "inbound":
            print "NLRI %s, received from %s\n" % ( self.nlri, peer )
         elif self.application == "outbound":
            print "NLRI %s, to %s\n" % ( self.nlri, peer )
         self.peers[ peer ].render()

class PolicyEvalSourceNlriResult( Model ):
   """
   Stores the policy evaluation results for a particular source.
   """
   routeMapEval = Submodel( PolicyEvalRouteMap, optional=True,
                            help="Evaluated route map" )

   def renderResult( self, source ):
      if self.routeMapEval:
         self.routeMapEval.renderMap()
      else:
         print "No route map applied to redistribution from %s" % source

class PolicyEvalMultiSourceNlriResult( Model ):
   """
   Stores the policy evaluation results for a particular prefix.
   """
   nlri = Str( help="Evaluated NLRI" )
   sources = Dict( keyType=str, valueType=PolicyEvalSourceNlriResult,
                   help="Evaluation result for each source" )
   application = Enum( values=( "redistribute", ),
                       help="Point of application debugged" )

   def render( self ):
      first = True
      for source in sorted( self.sources ):
         if not first:
            # Add two lines before printing a new peer
            print "\n"
         first = False
         print "NLRI %s, redistributed from %s\n" % ( self.nlri, source )
         self.sources[ source ].renderResult( source )
