# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliCommand
import CliParser
from CliPlugin.RouteCacheCliLib import routeCacheSupported
from CliPlugin.RouterGeneralCli import RouterGeneralMode, matcherRoute
from CliPlugin.RouterGeneralCli import RouterGeneralBaseModelet
from CliMode.RouteCacheMode import RouterGeneralRouteCacheBaseMode
import CliMatcher
import ConfigMount
import Tac

routeCacheCliConfig = None

thresholdHelpDesc = 'Configure route cache threshold levels'
hardwareHelpDesc = 'Configure hardware specific parameters'
utilizationHelpDesc = 'Configure routing table utilization'
timeoutHelpDesc = 'Set timeout for prefixes at this threshold level'
idletimeHelpDesc = 'Set idle time limit for prefixes at this threshold level'
percentHelpDesc = 'Threhold limit to be specified as a percentage'
secondsHelpDesc = 'idle time limit to be specified as seonds'

def setRouteCacheThreshold( mode, level, util, idleTime ):
   level = level.title()
   attr1 = 'thresholdLevel' + level + 'IdleTime'
   attr2 = 'thresholdLevel' + level + 'HwUtilPercentage'
   setattr( mode.thresholdConfig, attr1, idleTime )
   setattr( mode.thresholdConfig, attr2, util )

class RouterGeneralRouteCacheModelet( RouterGeneralBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterGeneralBaseModelet.__init__( self, mode )

class RouterGeneralRouteCacheMode( RouterGeneralRouteCacheBaseMode,
                                   BasicCli.ConfigModeBase ):
   ''' This mode is used for routing protocol independent features.
   '''
   name = 'Protocol independent route cache configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.cancelChanges = False
      RouterGeneralRouteCacheBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.thresholdConfig = None

   def getOrCreateRouteCacheConfig( self ):
      self.thresholdConfig = Tac.Value( "RouteCache::ThresholdConfig" )
      # copy the current state
      for attr in ( 'thresholdLevelLowIdleTime',
                    'thresholdLevelLowHwUtilPercentage',
                    'thresholdLevelMediumIdleTime',
                    'thresholdLevelMediumHwUtilPercentage',
                    'thresholdLevelHighIdleTime',
                    'thresholdLevelHighHwUtilPercentage',
                  ):
         value = getattr( routeCacheCliConfig.thresholdConfig, attr )
         setattr( self.thresholdConfig, attr, value )

   def deleteRouteCacheConfig( self ):
      # reset the params
      self.thresholdConfig = Tac.Value( "RouteCache::ThresholdConfig" )
      routeCacheCliConfig.thresholdConfig = self.thresholdConfig

   def commit( self ):
      if self.cancelChanges:
         return

      routeCacheCliConfig.thresholdConfig = self.thresholdConfig

   def validate( self ):
      if self.thresholdConfig.thresholdLevelLowIdleTime < \
            self.thresholdConfig.thresholdLevelMediumIdleTime:
         self.addError( "idle time limit for low threshold is less "
                        "than medium threshold. Config not applied." )
         return False

      if self.thresholdConfig.thresholdLevelMediumIdleTime < \
            self.thresholdConfig.thresholdLevelHighIdleTime:
         self.addError( "idle time limit for medium threshold is less "
                        "than high threshold. Config not applied." )
         return False

      if self.thresholdConfig.thresholdLevelLowHwUtilPercentage > \
            self.thresholdConfig.thresholdLevelMediumHwUtilPercentage:
         self.addError( "percentage of hardware utilization for low threshold "
                        "is greater than medium threshold. Config not applied." )
         return False

      if self.thresholdConfig.thresholdLevelMediumHwUtilPercentage > \
            self.thresholdConfig.thresholdLevelHighHwUtilPercentage:
         self.addError( "percentage of hardware utilization for medium threshold "
                        "is greater than high threshold. Config not applied." )
         return False

      return True

   def onExit( self ):
      if self.validate():
         self.commit()
      BasicCli.ConfigModeBase.onExit( self )

   @staticmethod
   def abort( mode, args ):
      mode.cancelChanges = True
      mode.session_.gotoParentMode()

#--------------------------------------------------------------------------------
# [ no | default ] route cache
#--------------------------------------------------------------------------------
class RouteCacheSubModeCmd( CliCommand.CliCommandClass ):
   syntax = 'route cache'
   noOrDefaultSyntax = syntax
   data = {
      'route': matcherRoute,
      'cache': CliCommand.guardedKeyword( 'cache', 'Enter route cache sub-mode',
                                          guard=routeCacheSupported ),
   }

   @staticmethod
   def handler( mode, args ):
      ''' Function to go from 'router general' mode to "route cache" submode
      '''
      childMode = mode.childMode( RouterGeneralRouteCacheMode )
      mode.session_.gotoChildMode( childMode )
      childMode.getOrCreateRouteCacheConfig()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( RouterGeneralRouteCacheMode )
      childMode.deleteRouteCacheConfig()

RouterGeneralMode.addCommandClass( RouteCacheSubModeCmd )
RouterGeneralRouteCacheMode.addModelet( RouterGeneralRouteCacheModelet )

class RouteCacheLowThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = \
         'hardware utilization threshold low UTIL percent idle-limit SECONDS seconds'
   noOrDefaultSyntax = 'hardware utilization threshold low ...'
   data = {
            'hardware': hardwareHelpDesc,
            'utilization': utilizationHelpDesc,
            'threshold': thresholdHelpDesc,
            'low': 'Configure parameters for threshold level low',
            'UTIL': CliMatcher.IntegerMatcher( 0, 100,
                  helpdesc='Hardware space utilization in percentage. default: 85' ),
            'percent': percentHelpDesc,
            'idle-limit': idletimeHelpDesc,
            'SECONDS': CliMatcher.IntegerMatcher( 100, 5100,
                        helpdesc='Seconds. default: 1100s' ),
             'seconds': secondsHelpDesc,
         }

   @staticmethod
   def handler( mode, args ):
      util = args.get( 'UTIL',
            mode.thresholdConfig.defaultThresholdLevelLowHwUtilPercentage )
      idleTime = args.get( 'SECONDS',
            mode.thresholdConfig.defaultThresholdLevelLowIdleTime )
      setRouteCacheThreshold( mode, 'low', util, idleTime )

   noOrDefaultHandler = handler

RouterGeneralRouteCacheModelet.addCommandClass( RouteCacheLowThresholdConfigCmd )

class RouteCacheMediumThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = \
      'hardware utilization threshold medium UTIL percent idle-limit SECONDS seconds'
   noOrDefaultSyntax = 'hardware utilization threshold medium ...'
   data = {
            'hardware': hardwareHelpDesc,
            'utilization': utilizationHelpDesc,
            'threshold': thresholdHelpDesc,
            'medium': 'Configure parameters for threshold level medium',
            'UTIL': CliMatcher.IntegerMatcher( 0, 100,
                  helpdesc='Hardware space utilization in percentage. default: 90' ),
            'percent': percentHelpDesc,
            'idle-limit': idletimeHelpDesc,
            'SECONDS': CliMatcher.IntegerMatcher( 100, 5100,
                        helpdesc='Seconds. default: 300s' ),
             'seconds': secondsHelpDesc,
         }

   @staticmethod
   def handler( mode, args ):
      util = args.get( 'UTIL',
            mode.thresholdConfig.defaultThresholdLevelMediumHwUtilPercentage )
      idleTime = args.get( 'SECONDS',
            mode.thresholdConfig.defaultThresholdLevelMediumIdleTime )
      setRouteCacheThreshold( mode, 'medium', util, idleTime )

   noOrDefaultHandler = handler

RouterGeneralRouteCacheModelet.addCommandClass( RouteCacheMediumThresholdConfigCmd )

class RouteCacheHighThresholdConfigCmd( CliCommand.CliCommandClass ):
   syntax = \
      'hardware utilization threshold high UTIL percent idle-limit SECONDS seconds'
   noOrDefaultSyntax = 'hardware utilization threshold high ...'
   data = {
            'hardware': hardwareHelpDesc,
            'utilization': utilizationHelpDesc,
            'threshold': thresholdHelpDesc,
            'high': 'Configure parameters for threshold level high',
            'UTIL': CliMatcher.IntegerMatcher( 0, 100,
                  helpdesc='Hardware space utilization in percentage. default: 95' ),
            'percent': percentHelpDesc,
            'idle-limit': idletimeHelpDesc,
            'SECONDS': CliMatcher.IntegerMatcher( 100, 5100,
                        helpdesc='Seconds. default: 100s' ),
             'seconds': secondsHelpDesc,
         }

   @staticmethod
   def handler( mode, args ):
      util = args.get( 'UTIL',
            mode.thresholdConfig.defaultThresholdLevelHighHwUtilPercentage )
      idleTime = args.get( 'SECONDS',
            mode.thresholdConfig.defaultThresholdLevelHighIdleTime )
      setRouteCacheThreshold( mode, 'high', util, idleTime )

   noOrDefaultHandler = handler

RouterGeneralRouteCacheModelet.addCommandClass( RouteCacheHighThresholdConfigCmd )

def Plugin( entityManager ):
   global routeCacheCliConfig
   routeCacheCliConfig = ConfigMount.mount( entityManager,
         "routecache/cliconfig", "RouteCache::CliConfig", "w" )
