# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides an interface to retrieve Rib-internal state
# for all Rib agents (across default and non-default VRFs)
#------------------------------------------------------------------------------------
import BasicCli
import CliMatcher
import TechSupportCli
import LazyMount
from FileCliUtil import registerPersistentConfiguration
from CliPlugin.VrfCli import VrfExecCmdDec, VrfExprFactory
from IpLibConsts import DEFAULT_VRF
from Arnet import NsLib
from RoutingCommon import GATED_COMMON_SH_TECH_TS
import Cell
import Tac
import sys
import RibCliLib
import os
from ShowCommand import ShowCliCommandClass

# pylint: disable-msg=W0108

RIBD_DUMP_RECV_BUF = 1024
RIBD_DUMP_CONN_RETRIES = 3

intfStatusAll = None
allVrfConfig = None
allVrfStatusLocal = None
l3Config = None

def getVrfsFunc():
   return allVrfConfig.vrf.members()

@VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
def showAgentDump( mode, **kwargs ):
   vrfName = kwargs.get( "vrfName" )
   agentName = kwargs.get( "AGENT" )
   debug = kwargs.get( "debug" )
   ribout = kwargs.get( "ribout" )
   verbose = kwargs.get( "verbose" )

   # filter out any vrfs that have a VRF status not 'active'
   if vrfName != DEFAULT_VRF:
      vsl = allVrfStatusLocal.vrf.get( vrfName )
      if not vsl or vsl.state != 'active':
         mode.addError( "VRF %s is not active" % vrfName )
         return
      doShowAgentDump( mode, agentName, vrfName=vrfName, ns=vsl.networkNamespace,
                      debug=debug, ribout=ribout, verbose=verbose )
   else:
      doShowAgentDump( mode, agentName, debug=debug, ribout=ribout,
                       verbose=verbose )

def doShowAgentDump( mode, agentName,
                     vrfName=DEFAULT_VRF, ns=NsLib.DEFAULT_NS, debug=None,
                     ribout=None, verbose=None ):
   multiagent = l3Config.protocolAgentModel == "multi-agent"
   if agentName == 'ribd':
      if multiagent:
         mode.addError( "Rib agent is not active" )
         return
      cmds = [ '/usr/bin/RibdShowTech' ]
      if debug:
         cmds.append( 'debug' )
      if ribout:
         cmds.append( 'ribout' )
      if verbose:
         cmds.append( 'verbose' )
   elif agentName == 'ospf':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/OspfShowTech' ]
   elif agentName == 'isis':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/IsisShowTech' ]
   elif agentName == 'ospfv3':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/Ospf3ShowTech' ]
   elif agentName == 'rip':
      if not multiagent:
         mode.addError( "%s agent is not active" % agentName.capitalize() )
         return
      cmds = [ '/usr/bin/RipShowTech' ]

   try:
      NsLib.runMaybeInNetNs( ns, cmds,
                             stdout=sys.stdout,
                             stderr=Tac.DISCARD,
                             asRoot=True )
   except Tac.SystemCommandError as e:
      vrfErrStr = ''
      if vrfName != DEFAULT_VRF:
         vrfErrStr += ' for VRF %s' % vrfName
      if e.error in ( 1, 2 ):
         mode.addError( 'Failed to connect to %s%s' % ( agentName, vrfErrStr ) )
      elif e.error == 3:
         mode.addError( 'Timeout encountered while processing '
                        'show tech-support %s%s' % ( agentName, vrfErrStr ) )

matcherRibd = CliMatcher.KeywordMatcher( 'ribd',
    helpdesc='Show aggregated status and configuration details for the Rib agent' )

matcherVrfName = VrfExprFactory( helpdesc="VRF name", inclAllVrf=True,
                            inclDefaultVrf=True )

class ShowTechSupportRibd( ShowCliCommandClass ):
   syntax = "show tech-support AGENT [ debug ] [ ribout ] [ verbose ] [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'AGENT': matcherRibd,
           'debug': 'Show debug information for ribd',
           'ribout': 'Show debug information about RibOuts',
           'verbose': 'Enable verbose output',
           'vrfName': matcherVrfName
           }

   @staticmethod
   def handler( mode, args ):
      args.setdefault( "vrfName", None )
      showAgentDump( mode, **args )

BasicCli.addShowCommandClass( ShowTechSupportRibd )

def _showTechCmds():
   return [ 'show tech-support ribd running-config',
            'show rib ready vrf all',
            'bash echo "show memory" | cliribd', ]

@VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
def doShowRibdRunningConfig( mode, **args ):
   vrfName = args.get( "vrfName", DEFAULT_VRF )
   RibCliLib.cliRibdShowCommand( mode, 'show running-config', cport=None,
                                 vrf=vrfName )

tokenRunningConfig = CliMatcher.KeywordMatcher( 'running-config',
                     helpdesc='Show running configuration of ribd' )

class ShowTechSupportRibdRunningConfig( ShowCliCommandClass ):
   syntax = "show tech-support ribd running-config [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'ribd': matcherRibd,
           'running-config': tokenRunningConfig,
           'vrfName': matcherVrfName
           }

   @staticmethod
   def handler( mode, args ):
      args.setdefault( "vrfName", None )
      doShowRibdRunningConfig( mode, **args )

BasicCli.addShowCommandClass( ShowTechSupportRibdRunningConfig )

TechSupportCli.registerShowTechSupportCmdCallback( GATED_COMMON_SH_TECH_TS,
                                                   _showTechCmds )

agentsNames = [ 'ospf', 'ospfv3', 'isis', 'rip' ]
agentsEnumDict = { agent: "Show agregated status and configuration "
                           "details for " + agent for agent in agentsNames }

class ShowTechSupportExtendedAgents( ShowCliCommandClass ):
   syntax = "show tech-support extended AGENT [ vrfName ] "
   data = {
           'tech-support': TechSupportCli.techSupportKwMatcher,
           'extended': TechSupportCli.extendedKwMatcher,
           'AGENT': CliMatcher.EnumMatcher( agentsEnumDict ),
           'vrfName': matcherVrfName,
           }

   @staticmethod
   def handler( mode, args ):
      args.setdefault( "vrfName", DEFAULT_VRF )
      showAgentDump( mode, **args )

BasicCli.addShowCommandClass( ShowTechSupportExtendedAgents )

def deleteRibStartup( mode, delete ):
   if delete and os.path.exists( "/mnt/flash/rib-startup" ):
      os.unlink( "/mnt/flash/rib-startup" )

def deleteRibAmi( mode, delete ):
   if delete and os.path.exists( "/persist/sys/ribd.ami" ):
      os.unlink( "/persist/sys/ribd.ami" )

def deleteRibConfig( mode, delete ):
   deleteRibAmi( mode, delete )
   deleteRibStartup( mode, delete )

startupDesc = """startup-config text file for Ribd configuration commands that
                 are not managed by EOS.  They are applied via cliribd at Rib
                 agent start time"""
amiDesc = """binary format AMI dump of both customer-added 
                 configuration, and EOS-generated configuration. 
                 (deprecated, do not use)"""
ribConfigDesc = """config file for Ribd configuration commands that
                 are not managed by EOS."""

registerPersistentConfiguration( "rib-config", deleteRibConfig,
                                ribConfigDesc, default=True )

def Plugin( entityManager ):
   global intfStatusAll
   global allVrfConfig
   global allVrfStatusLocal
   global l3Config
   # This is a little silly.  We use interface/status/all as a fake
   # CliSave anchor in our CliSavePlugin, so it has to get mounted here,
   # even though we never use it.
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
   intfStatusAll = LazyMount.mount( entityManager,
                                    "interface/status/all",
                                    "Interface::AllIntfStatusDir", "r" )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
