# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


import CliCommand
import CliMatcher
import ConfigMount
import Tac

from RouterGeneralCli import routerGeneralVrfCleanupHook
from RouterGeneralCli import RouterGeneralVrfMode
from RouteMapCli import mapNameMatcher

config = None
RouteControlConstant = Tac.Type( 'Routing::RouteControl::Constant' )

#-------------------------------------------------------------------------------
# Config handler functions 
#-------------------------------------------------------------------------------
def maybeDeleteVrfConfig( vrfConfig ):
   if vrfConfig.fibIpv4PolicyMap != RouteControlConstant.fibIpv4PolicyMapDefault:
      return
   elif vrfConfig.fibIpv6PolicyMap != RouteControlConstant.fibIpv6PolicyMapDefault:
      return
   elif vrfConfig.resIpv4PolicyMap != RouteControlConstant.resIpv4PolicyMapDefault:
      return
   elif vrfConfig.resIpv6PolicyMap != RouteControlConstant.resIpv6PolicyMapDefault:
      return
   else:
      del config.vrfConfig[ vrfConfig.vrfName ]


matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
                                         helpdesc='IPv4 configuration commands' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
                                         helpdesc='IPv6 configuration commands' )
matcherRib = CliMatcher.KeywordMatcher( 'rib',
                                        helpdesc='Routing table' )
#-------------------------------------------------------------------------------
# [ no | default ] rib ( ipv4 | ipv6 ) fib policy POLICY
# in "config-router-general-vrf" mode
#--------------------------------------------------------------------------------
class RibFibPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'rib ( ipv4 | ipv6 ) fib policy POLICY'
   noOrDefaultSyntax = 'rib ( ipv4 | ipv6 ) fib policy ...'
   data = {
      'rib': matcherRib,
      'ipv4': matcherIpv4,
      'ipv6': matcherIpv6,
      'fib': 'FIB configuration commands',
      'policy': 'FIB installation policy',
      'POLICY': mapNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      assert mode.vrfName
      mapName = args[ 'POLICY' ]
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         vrfConfig = config.vrfConfig.newMember( mode.vrfName )
   
      if ipv4:
         vrfConfig.fibIpv4PolicyMap = mapName
      elif ipv6:
         vrfConfig.fibIpv6PolicyMap = mapName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         return
      if ipv4:
         vrfConfig.fibIpv4PolicyMap = RouteControlConstant.fibIpv4PolicyMapDefault
      elif ipv6:
         vrfConfig.fibIpv6PolicyMap = RouteControlConstant.fibIpv6PolicyMapDefault
   
      maybeDeleteVrfConfig( vrfConfig )

RouterGeneralVrfMode.addCommandClass( RibFibPolicyCmd )

#-------------------------------------------------------------------------------
# [ no | default ] rib ( ipv4 | ipv6 ) resolution policy POLICY
# in "config-router-general-vrf" mode
#--------------------------------------------------------------------------------
class RibResolutionPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'rib ( ipv4 | ipv6 ) resolution policy POLICY'
   noOrDefaultSyntax = 'rib ( ipv4 | ipv6 ) resolution policy ...'
   data = {
      'rib': matcherRib,
      'ipv4': matcherIpv4,
      'ipv6': matcherIpv6,
      'resolution': 'Recursive resolution configuration commands',
      'policy': 'Recursive resolution policy',
      'POLICY': mapNameMatcher,
   }

   @staticmethod
   def handler( mode, args):
      assert mode.vrfName
      mapName = args[ 'POLICY' ]
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         vrfConfig = config.vrfConfig.newMember( mode.vrfName )
      if ipv4:
         vrfConfig.resIpv4PolicyMap = mapName
      elif ipv6:
         vrfConfig.resIpv6PolicyMap = mapName
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         return
      if ipv4:
         vrfConfig.resIpv4PolicyMap = RouteControlConstant.resIpv4PolicyMapDefault
      elif ipv6:
         vrfConfig.resIpv6PolicyMap = RouteControlConstant.resIpv6PolicyMapDefault
      maybeDeleteVrfConfig( vrfConfig )

RouterGeneralVrfMode.addCommandClass( RibResolutionPolicyCmd )


# Register callback handler for cleaning 'router general' vrf sub-modes
def cleanupRouterGeneralVrfMode( vrfName ):
   del config.vrfConfig[ vrfName ]

routerGeneralVrfCleanupHook.addExtension( cleanupRouterGeneralVrfMode )

def Plugin( entityManager ):
   global config

   config = ConfigMount.mount( entityManager,
                               'routing/general/config/routeControl',
                               'Routing::RouteControl::Config', 'w' )
