# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides "show rib route ip[v6]" functions
#------------------------------------------------------------------------------------

import simplejson
import sys

from IraCommonCli import ribRouteHook, ribRouteModel
from IpLibConsts import DEFAULT_VRF
from RibCapiLib import showRibCapiCommand, EmptyResponseException
from socket import AF_INET, AF_INET6

def showRibdRoute( af, mode, vrfName, protocol, prefix, fibExcluded=None ):
   assert 'IpRib' in ribRouteModel

   args = {}
   args[ 'af' ] = AF_INET if af == 'ipv4' else AF_INET6

   # routingProtocols means show all supported protocols
   if protocol != ribRouteModel[ 'IpRib' ].routingProtocol.routingProtocols:
      for i, p in enumerate( ribRouteModel[ 'IpRib' ].ribdSupportProtos ):
         if p == protocol:
            args[ 'protocol' ] = i
            break
      else:
         return

   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if fibExcluded:
      args[ 'show_fib_denied' ] = 1

   if prefix:
      # even if mlen was not specified, prefix was built as IpGenPrefix which
      # defaults to host mlen, so we always have '/<mlen>' in stringValue
      addr, mlen = prefix.stringValue.split( '/' )

      if prefix.af == 'ipv4':
         args[ 'addrv4' ] = addr
         args[ 'masklenv4' ] = int( mlen )
      else:
         args[ 'addrv6' ] = addr
         args[ 'masklenv6' ] = int( mlen )

   cmd = "MIO_DGET_RIB_ROUTE"

   try:
      model = showRibCapiCommand( mode,
                                  ribRouteModel[ 'IpRib' ].RibRoutesByProtocol,
                                  cmd, args=args, highPerf=True )
   except EmptyResponseException:
      model = ribRouteModel[ 'IpRib' ].RibRoutesByProtocol()

   if mode.session_.outputFormatIsJson():
      revision = mode.session_.requestedModelRevision()
      encoder = simplejson.JSONEncoder( indent=4, sort_keys=False )
      for chunk in encoder.iterencode( model.toDict( streaming=True,
                                                     revision=revision ) ):
         sys.stdout.write( chunk )
      # Output is always newline terminated
      sys.stdout.write( '\n' )
   else:
      if prefix:
         model.renderProtocols( vrfName=vrfName, prefixProto=protocol )
      else:
         model.renderProtocols( vrfName=vrfName )

def Plugin( entityManager ):
   ribRouteHook.addExtension( showRibdRoute )
