# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


import Tac
import LazyMount
import Cell

from CliPlugin.VrfCli import vrfExists
from IraCommonCli import ribReadyHook
from ShowRibReadyModels import RibReadyModel, RibReadyFlags, RouteReady

# pylint: disable-msg=protected-access
vrfReadyConfigDir = None
vrfReadyStatusDir = None
fibReadyDir = None

#--------------------------------------------------------------------
# Command: show rib ready [ vrf vrfName ]
# show rib ready # To display ready state of current vrf context.
# show rib ready vrf all # To display ready state of all vrfs
# show rib ready vrf vrfName  # To display ready state of vrfName 
#--------------------------------------------------------------------

def showRibReadyState( mode, vrfName=None ):
   """
# Sample output of command
=======================
Vrf default
=======================
Start time : 0:00:43 ago
Grestart Init : 1

   Flag                              Value    TimeTaken( in sec )
--------------------------------- ----------- -------------------
   Fib ready                         True                   2.490
   ...

  RouteType       Configured     ConfigPending     Ready    TimeTaken( in sec )
--------------- -------------- ----------------- ---------- -------------------
  v4Host          True           False             True                   1.700
  ...

   """

   if not vrfExists( vrfName ):
      mode.addError( "Vrf %s doesn't exist" % ( vrfName ) )
      return

   if ( ( vrfName not in vrfReadyConfigDir ) or
        ( vrfName not in vrfReadyStatusDir ) ):
      mode.addError( "Vrf %s exists but is not yet processed by the Rib agent" % (
         vrfName ) )
      return

   vrfConfig = vrfReadyConfigDir[ vrfName ]
   vrfStatus = vrfReadyStatusDir[ vrfName ]

   def roundedTimeInterval( readyTime ):
      # return time rounded to ms
      timeTaken = readyTime - vrfStatus.startTime
      if readyTime < 0:
         # Protocols may not yet be ready and time is -inf which is the initial
         # value.
         return None
      elif timeTaken > 0:
         return round( timeTaken, 3 )
      else:
         return 0.0

   ribReadyModel = RibReadyModel()

   ribReadyModel._protocolModel = "ribd"
   ribReadyModel._vrf = vrfName
   ribReadyModel.startTime = vrfStatus.startTime
   ribReadyModel.gracefullyRestarting = bool( vrfStatus.grestartInit )
   # The protocols dict is used by multi-agent command, but this needs to be set to
   # None so that an empty dict does not show up in the json output
   ribReadyModel.protocols = None

   flagsModel = RibReadyFlags()
   flagsModel.fibReady = vrfName in fibReadyDir.entryState and \
                         fibReadyDir.entity[ vrfName ].vrfReady
   flagsModel.timeTakenForFibReady = roundedTimeInterval( vrfStatus.fibReadyTime )

   flagsModel.ribReady = vrfStatus.ribReady
   flagsModel.timeTakenForRibReady = roundedTimeInterval( vrfStatus.ribReadyTime )

   flagsModel.recursiveResolution = vrfStatus.staticRoutesRecursivelyResolved
   flagsModel.timeTakenForRecursiveResolution = roundedTimeInterval( \
                                    vrfStatus.staticRoutesRecursivelyResolvedTime )

   flagsModel.v4RouteSync = vrfStatus.v4RoutesSynced
   flagsModel.timeTakenForV4RouteSync = roundedTimeInterval( \
                                                      vrfStatus.v4RoutesSyncedTime )

   flagsModel.v6RouteSync = vrfStatus.v6RoutesSynced
   flagsModel.timeTakenForV6RouteSync = roundedTimeInterval( \
                                                      vrfStatus.v6RoutesSyncedTime )

   ribReadyModel.flags = flagsModel

   routeModels = {}
   for rtType, rtConfig in vrfConfig.routeConfig.items():
      model = RouteReady()
      model.configured = rtConfig.configured
      model.configPending = rtConfig.configPending
      model.ready = vrfStatus.routeStatus[ rtType ].ready
      model.timeTaken = roundedTimeInterval( \
                                       vrfStatus.routeStatus[ rtType ].readyTime )
      routeModels[ rtType ] = model

   ribReadyModel.routes = routeModels

   return ribReadyModel


def Plugin( entityManager ):
   global vrfReadyConfigDir
   global vrfReadyStatusDir
   global fibReadyDir

   vrfReadyConfigDir = LazyMount.mount( entityManager,
                          Cell.path( "routing/vrf/ready/config" ),
                          "Tac::Dir", "ri" )
   vrfReadyStatusDir = LazyMount.mount( entityManager, 
                          Cell.path( "routing/vrf/ready/status" ),
                          "Tac::Dir", "ri" )
   fibReadyDir = LazyMount.mount( entityManager,
                          Cell.path( "routing/fibReady" ),
                          "Tac::Dir", "ri" )
   # This hook has been defined in Ira. Since both gated and IpRib use the same CLI
   # command, we need to call the correct method based on the protocol model in use
   # when this command is invoked. For this purpose, gated adds its callback method
   # as an extension to this 'ribReadyHook' and when the command is invoked, in IpRib
   # we look at the protocol model in use and invoke this hook if protocol model
   # 'ribd' is being used.
   ribReadyHook.addExtension( showRibReadyState )
