# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides Rib ipv4 cli functions that depend on cliribd.
#------------------------------------------------------------------------------------

import BasicCli, Tac
import RibCliLib
import CliToken.Ip
import CliMatcher
import CliCommand
import IraIpCli
import IraRouteCommon
import Arnet
from Arnet import Subnet
from CliPlugin.VrfCli import vrfExists, DEFAULT_VRF
import LazyMount
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from ShowCommand import ShowCliCommandClass

ip = IraRouteCommon.Ip4()
routing = IraRouteCommon.routing( ip )
l3Config = None

# vrfName is defaulted to None so as to allow RibCliLib.cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def gatedRoute( mode, prefix, vrfName=None ):
   assert vrfName != ''
   snet = Subnet( prefix )
   cmd = "show ip route"
   cmd = "%s %s %s pyformat" % ( cmd, snet.address_, snet.mask_ )
   out = RibCliLib.cliRibdShowCommand( mode, cmd, output=False,
                                       replaceKernelIntfs=True, vrf=vrfName )
   if not out or out == 'Rib Agent inactive':
      return {} 
   return eval( out.rstrip() )

def showRouteTag( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IP routing table %s does not exist." % vrfName )
      return

   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   prefix = routing.ip.subnet( prefix )

   routeSorter = IraRouteCommon.showRoutes(
      routing, prefix, None, protocol=None, detail=False, longerPrefixes=False,
      vrfName=vrfName, host=False, degradeNhgModel=False,
      flattenTunnelOverTunnel=False ).tacRouteSorter()
   print routeSorter.legend()
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Prefix' ).nullPrefix:
      print ""
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print ""
      return
   proto = routeSorter.protocolString( lpm )
   tag = ribRoute.get( 'tag' )
   print "%-5s   %s tag %d" % ( proto, lpm, tag )
   print ""
   IraIpCli.warnIfRoutingDisabled( mode )
   return

tagDeprecatedMatcher = CliCommand.Node(
                           CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' ),
                           deprecatedByCmd='show ip route match tag' )

class ShowIpRouteTag( ShowCliCommandClass ):
   syntax = "show ip route [ VRF ] ( PREFIX | ADDR ) tag"
   data = {
           "ip": CliToken.Ip.ipMatcherForShow,
           "route": IraIpCli.routeMatcherAfterShowIp,
           "VRF": IraIpCli.vrfExprFactoryForShow,
           "ADDR": IraIpCli.ipAddrExpr,
           "PREFIX": IraIpCli.ipPrefixExpr,
           "tag": tagDeprecatedMatcher
          }

   handler = showRouteTag

BasicCli.addShowCommandClass( ShowIpRouteTag )

matchMatcher = CliMatcher.KeywordMatcher( 'match', helpdesc='Route tag' )
tagMatcher = CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' )

class ShowIpRouteMatchTag( ShowCliCommandClass ):
   syntax = "show ip route [ VRF ] ( PREFIX | ADDR ) match tag"
   data = {
           "ip": CliToken.Ip.ipMatcherForShow,
           "route": IraIpCli.routeMatcherAfterShowIp,
           "VRF": IraIpCli.vrfExprFactoryForShow,
           "ADDR": IraIpCli.ipAddrExpr,
           "PREFIX": IraIpCli.ipPrefixExpr,
           "match": matchMatcher,
           "tag": tagMatcher
          }

   handler = showRouteTag

BasicCli.addShowCommandClass( ShowIpRouteMatchTag )

def showRouteAge( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( " IP routing table %s does not exist." % vrfName )
      return

   if isinstance( prefix, str ):
      prefix = Arnet.Prefix( prefix )
   prefix = routing.ip.subnet( prefix )

   routeSorter = IraRouteCommon.showRoutes( routing, prefix, None, protocol=None,
                                            detail=False,
                                            longerPrefixes=False,
                                            vrfName=vrfName,
                                            host=False,
                                            degradeNhgModel=False,
                                    flattenTunnelOverTunnel=False ).tacRouteSorter()
   print routeSorter.legend()
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Prefix' ).nullPrefix:
      print ""
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print ""
      return
   proto = routeSorter.protocolString( lpm )
   fibvias = routeSorter.viasForPrefix( lpm )
   ribvias = ribRoute.get( 'vias', [] )
   i = 0
   tunRoute = False
   for nh in fibvias.via:
      age = None
      hop = str( nh.hop )
      if i == 0 and 'tun' in ribvias[i]['intf']:
         tunRoute = True
      if tunRoute:
         hop = ribvias[i]['intf']
         age = ribvias[i]['time']
      else:
         for via in ribvias:
            if hop == via['nexthop']:
               age = via['time']
               break

      if routeSorter.hasV6Nexthop:
         if i == 0:
            print "%-5s %s" % ( proto, lpm )
         print "%-5s    via %s, age %s" % ( " "*len(proto), hop, age )
      else:
         if i > 0:
            print "%-5s   %s via %s, age %s" % ( " "*len(proto), 
                                                 " "*len(lpm.stringValue),
                                                 hop, age )
         else:
            print "%-5s   %s via %s, age %s" % ( proto, lpm, hop, age )
      i += 1
   print ""
   IraIpCli.warnIfRoutingDisabled( mode )
   return

class ShowIpRouteAge( ShowCliCommandClass ):
   syntax = "show ip route [ VRF ] ( PREFIX | ADDR ) age"
   data = {
           "ip": CliToken.Ip.ipMatcherForShow,
           "route": IraIpCli.routeMatcherAfterShowIp,
           "VRF": IraIpCli.vrfExprFactoryForShow,
           "ADDR": IraIpCli.ipAddrExpr,
           "PREFIX": IraIpCli.ipPrefixExpr,
           "age": "Route age"
          }

   handler = showRouteAge

BasicCli.addShowCommandClass( ShowIpRouteAge )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global l3Config

   l3Config = LazyMount.mount( entityManager, 'l3/config', 'L3::Config', 'r' )
