# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------------------------------------
# This module provides Rib ipv6 cli functions that depend on cliribd.
#------------------------------------------------------------------------------------

import BasicCli, Tac
import CliMatcher
import CliCommand
import RibCliLib
import IraIp6Cli
import IraRouteCommon
import LazyMount
from CliPlugin.VrfCli import vrfExists, DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from ShowCommand import ShowCliCommandClass

l3Config = None

ip = IraRouteCommon.Ip6()
routing = IraRouteCommon.routing( ip )

# vrfName is defaulted to None so as to allow RibCliLib.cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def gatedRoute( mode, prefix, vrfName=None ):
   assert vrfName != ''
   cmd = "show ipv6 route"
   cmd = "%s %s pyformat" % ( cmd, prefix.stringValue )
   out = RibCliLib.cliRibdShowCommand( mode, cmd, output=False,
                                       replaceKernelIntfs=True, vrf=vrfName )

   if not out:
      return {}
   return eval( out.rstrip() )

def showRouteTag( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return

   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   prefix = prefix.subnet.stringValue
   print "IPv6 Routing Table - %u entries" % \
      len( routing.rStatus( vrfName=vrfName ).route )
   routeSorter = IraRouteCommon.showRoutes( routing, prefix, None, protocol=None,
                                            detail=False,
                                            longerPrefixes=False,
                                            vrfName=vrfName,
                                            host=False,
                                            degradeNhgModel=False,
                                    flattenTunnelOverTunnel=False ).tacRoute6Sorter()
   print routeSorter.legend()
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Ip6Prefix' ).nullPrefix:
      print ""
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print ""
      return
   proto = routeSorter.protocolString( lpm )
   tag = ribRoute.get( 'tag' )
   print "%-5s %s tag %d" % ( proto, lpm, tag )
   print ""      
   if not routing.config( vrfName=vrfName ).routing:
      mode.addWarning( "IPv6 routing not enabled" )

tagDeprecatedMatcher = CliCommand.Node(
                           CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' ),
                           deprecatedByCmd='show ipv6 route match tag' )

class ShowIpv6RouteTag( ShowCliCommandClass ):
   syntax = "show ipv6 route [ VRF ] ( PREFIX | ADDR ) tag"
   data = {
           "ipv6": IraIp6Cli.ipv6MatcherForShow,
           "route": IraIp6Cli.routeAfterShowIpv6Matcher,
           "VRF": IraIp6Cli.showRouteVrfNameExprFactory,
           "PREFIX": IraIp6Cli.ipv6PrefixMatcher,
           "ADDR": IraIp6Cli.ipv6AddrMatcher,
           "tag": tagDeprecatedMatcher
          }

   handler = showRouteTag

BasicCli.addShowCommandClass( ShowIpv6RouteTag )

matchMatcher = CliMatcher.KeywordMatcher( 'match', helpdesc='Route tag' )
tagMatcher = CliMatcher.KeywordMatcher( 'tag', helpdesc='Route tag' )

class ShowIpv6RouteMatchTag( ShowCliCommandClass ):
   syntax = "show ipv6 route [ VRF ] ( PREFIX | ADDR ) match tag"
   data = {
           "ipv6": IraIp6Cli.ipv6MatcherForShow,
           "route": IraIp6Cli.routeAfterShowIpv6Matcher,
           "VRF": IraIp6Cli.showRouteVrfNameExprFactory,
           "PREFIX": IraIp6Cli.ipv6PrefixMatcher,
           "ADDR": IraIp6Cli.ipv6AddrMatcher,
           "match": matchMatcher,
           "tag": tagMatcher,
          }

   handler = showRouteTag

BasicCli.addShowCommandClass( ShowIpv6RouteMatchTag )

def showRouteAge( mode, args ):
   prefix = args.get( "ADDR" ) or args.get( "PREFIX" )
   vrfName = args.get( "VRF", DEFAULT_VRF )

   if l3Config.protocolAgentModel != ProtoAgentModel.ribd:
      mode.addError( "Not supported in multi-agent mode." )
      return

   if not vrfExists( vrfName ):
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return

   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   prefix = prefix.subnet.stringValue
   print "IPv6 Routing Table - %u entries" % \
      len( routing.rStatus( vrfName=DEFAULT_VRF ).route )
   routeSorter = IraRouteCommon.showRoutes(
      routing, prefix, None, protocol=None, detail=False, longerPrefixes=False,
      vrfName=vrfName, host=False, degradeNhgModel=False,
      flattenTunnelOverTunnel=False ).tacRoute6Sorter()
   print routeSorter.legend()
   lpm = routeSorter.longestPrefixMatch()
   if lpm == Tac.Value( 'Arnet::Ip6Prefix' ).nullPrefix:
      print ""
      return
   ribRoute = gatedRoute( mode, lpm, vrfName=vrfName )
   if ribRoute == {}:
      print ""
      return
   proto = routeSorter.protocolString( lpm )
   age = ribRoute.get( 'time' )
   print "%-5s %s age %s" % ( proto, lpm, age )
   print ""         
   if not routing.config( vrfName=DEFAULT_VRF ).routing:
      mode.addWarning( "IPv6 routing not enabled" )

class ShowIpv6RouteAge( ShowCliCommandClass ):
   syntax = "show ipv6 route [ VRF ] ( PREFIX | ADDR ) age"
   data = {
           "ipv6": IraIp6Cli.ipv6MatcherForShow,
           "route": IraIp6Cli.routeAfterShowIpv6Matcher,
           "VRF": IraIp6Cli.showRouteVrfNameExprFactory,
           "PREFIX": IraIp6Cli.ipv6PrefixMatcher,
           "ADDR": IraIp6Cli.ipv6AddrMatcher,
           "age": "Route age"
          }

   handler = showRouteAge

BasicCli.addShowCommandClass( ShowIpv6RouteAge )


#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   routing.plugin( entityManager )

   global l3Config

   l3Config = LazyMount.mount( entityManager, 'l3/config', 'L3::Config', 'r' )
