# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Reload hook for running ReloadPolicy during the CLI 'reload' command.

import Cell
import CliPlugin.ReloadCli
import LazyMount
import ReloadPolicy

_abootSbStatus = None
_rpDebugConfig = None

def Plugin( entityManager ):
   # Mount SecureBoot status
   global _abootSbStatus, _rpDebugConfig
   _abootSbStatus = LazyMount.mount( entityManager, Cell.path( "aboot/sb/status" ),
                                     "Aboot::Secureboot::Status", "r" )
   _rpDebugConfig = LazyMount.mount( entityManager, "reloadpolicy/debug/config",
                                     "ReloadPolicy::DebugConfig", "r" )

def checkReloadPolicy( mode, imagePath, syslogNextSwiVersion=False ):
   ''' Run general ReloadPolicy plugins. Return success boolean indicating if
   all checks pass plus warning/error messages.'''

   result = ReloadPolicy.doCheck( imagePath, category=[ 'General' ], mode=mode,
                                  abootSbStatus=_abootSbStatus,
                                  syslogNextSwiVersion=syslogNextSwiVersion )

   # Mlag Issu needs to install pre-2GB images in tests. The ignoreBodega2GbCheck
   # attribute provides a way to do this; if True, have the reload hook discard
   # the error raised by a failed Bodega2GB check. Thus, callers of this function
   # will not know an error occurred.
   if ( _rpDebugConfig and hasattr( _rpDebugConfig, 'ignoreBodega2GbCheck' )
        and _rpDebugConfig.ignoreBodega2GbCheck
        and result.policySuccess.get( 'Bodega2GB' ) is False ):
      bodega2GbErr = ( "This platform is only supported in 2GB swi images. "
                       "Please update your image in /mnt/flash to a 2GB swi image "
                       "such as EOS-2GB.swi" )
      result.errors.remove( bodega2GbErr )
      del result.policySuccess[ 'Bodega2GB' ]

   return not result.hasError(), result

def checkReloadPolicyResult( mode, power, now ):
   ''' Check if ReloadPolicy blocks reload due to error. Return True if the
   checks pass without error and reload can proceed, else return False.'''

   imageFile, swImage = CliPlugin.ReloadCli.getImagePath( mode )

   # Check can't be performed on remote URL.
   if not hasattr( imageFile, 'realFilename_' ):
      mode.addError( 'The boot configuration specifies a remote location for the '
                     'SWI image to boot: {}\nThe reload image compatibility check '
                     'is only supported on SWI images present in the local file '
                     'system.\nPlease update the boot configuration to specify a '
                     'SWI image present in the local file system before re-issuing '
                     'the reload command.'.format( str( imageFile ) ) )
      return False

   # pylint: disable=maybe-no-member
   if not imageFile.realFilename_ or 'not set' in swImage:
      # Boot image not set, Warning already present in ReloadCli
      return True

   # Display ReloadPolicy results in the CLI
   success, result = checkReloadPolicy( mode, imageFile.realFilename_,
                                        syslogNextSwiVersion=True )
   for warnStr in result.warnings:
      mode.addWarning( warnStr )
   for errStr in result.errors:
      mode.addError( errStr )
   return success

# Register reload hook as RUN_FIRST
CliPlugin.ReloadCli.registerReloadHook( checkReloadPolicyResult,
                                        'ReloadPolicyCheckCli',
                                        'RUN_FIRST' )
