# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ReloadCli
import Tac
import HwEpochPolicy
import EosVersion
import LazyMount

hwEpochStatus_ = None

def getMaxHwEpoch( mode ):
   maxHwEpoch = 0
   # Get max of hwEpochs for all the cards in chassis
   if hwEpochStatus_:
      maxHwEpoch = hwEpochStatus_.getMaxCardEpoch()
      maxHwEpoch, _ = HwEpochPolicy.parseEpoch( maxHwEpoch )

   return maxHwEpoch

def checkImageEpochAllowed( mode, imagePath, swImage ):
   import zipfile

   hwEpoch = getMaxHwEpoch( mode )
   try :
      swiVersion = EosVersion.swiVersion( imagePath )
   except zipfile.BadZipfile :
      return False, "Not able to read boot image at %s " % swImage

   warnStr = "Software image %s is not compatible on this hardware" % swImage

   # If image does not have swiVariant,
   # it can boot only on hardware with hwEpoch == 0
   if hwEpoch and not swiVersion.swiVariant():
      return False, warnStr

   # Reload is allowed if image is compatible with hardware
   if ( ( hwEpoch == 0 ) or swiVersion.isIntlVersion() or
        ( hwEpoch <= swiVersion.swiMaxHwEpoch() ) ):
      return True, ""
   else:
      return False, warnStr

   return False

def checkImageEpochVersion( mode, power, now ):

   imageFile, swImage = ReloadCli.getImagePath( mode )

   # Check can't be performed on remote URL.
   if not hasattr( imageFile, 'realFilename_' ):
      mode.addError( 'The boot configuration specifies a remote location for the '
                     'SWI image to boot: {}\nThe reload image compatibility check '
                     'is only supported on SWI images present in the local file '
                     'system.\nPlease update the boot configuration to specify a '
                     'SWI image present in the local file system before re-issuing '
                     'the reload command.'.format( str( imageFile ) ) )
      return False

   # pylint: disable=maybe-no-member
   if imageFile.realFilename_ == None or 'not set' in swImage:
      # Boot image not set, Warning already present in ReloadCli
      return True

   if not imageFile.exists():
      mode.addError( "Boot image doesn't exist at : %s " % swImage )
      return False

   retVal, err = checkImageEpochAllowed( mode, imageFile.realFilename_, swImage )

   if not retVal:
      mode.addError( err )

   return retVal

# Register reload hook as RUN_FIRST
ReloadCli.registerReloadHook( checkImageEpochVersion, 'ReloadImageEpochCheckCli',
                              'RUN_FIRST' )

def Plugin( entityManager ):
   global hwEpochStatus_
   hwEpochStatus_ = LazyMount.mount( entityManager,
                                     'hwEpoch/status', 'HwEpoch::Status', 'r' )


