#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ReloadCli, BasicCliUtil, Tac, LazyMount, FruCli, RedSupCli
import WaitForWarmup

fileStatus = None
em = None

FileReplicationStatus = Tac.Type( "FileReplication::FileStatus::Status" )

def Plugin( entityManager ):
   global fileStatus, em
   em = entityManager
   fileStatus = LazyMount.mount( entityManager, 'redundancy/fileReplication/status',
                                 'FileReplication::Status', 'r' )

# Get failed and unsynced files from Sysdb and return them in a tuple
def getFiles():
   failedFiles = []
   unsyncedFiles = []
   
   for r in fileStatus.requester.itervalues():
      for f in r.file:
         if r.file[ f ].status == FileReplicationStatus.unsynchronized:
            unsyncedFiles.append( f )
         elif r.file[ f ].status == FileReplicationStatus.failed:
            failedFiles.append( f )
         
   return ( failedFiles, unsyncedFiles )

def getFileList( failedFiles, unsyncedFiles ):
   message = ''
   if len( failedFiles ) > 0:
      message += 'The following files failed to synchronize:\n   '
      message += '\n   '.join( failedFiles )
   if len( unsyncedFiles ) > 0:
      if message != '':
         message += '\n'
      message += 'The following files are currently unsynchronized:\n   '
      message += '\n   '.join( unsyncedFiles )
   
   return message
   

def getFileSummary( failedFiles, unsyncedFiles ):
   failedCount = len( failedFiles )
   unsyncedCount = len( unsyncedFiles )

   if failedCount == 0 and unsyncedCount == 0:
      return None
   
   message = ''
   if failedCount > 0:
      message += '%d files failed to synchronize.' % failedCount
   if unsyncedCount > 0:
      if message != '':
         message += ' '
      message += '%d files are currently unsynchronized.' % unsyncedCount
   return message

def filesSynchronized():
   files = getFiles()
   return len( files[ 0 ] ) == 0 and len( files[ 1 ] ) == 0
   
def doFileSyncBeforeReload( mode, power, now ):
   couldReload = True

   configFile = mode.session_.configFile
   if now or configFile is not None:
      # Don't prompt the user in non-interactive mode, although we don't expect
      # reload to be in a config file. Prompts are also not shown if the user
      # specified 'now' when reloading.
      return True

   # Running interactively with the user

   electionStatus = RedSupCli.electionStatus

   # We should only worry about syncing if the second supervisor is valid
   if ( electionStatus.peerState == 'inserted' and
        electionStatus.peerRedundancyMode == 'standby' ):
      
      # Wait for FileReplicator to be ready (max 5 seconds)
      try:
         WaitForWarmup.wait( em, agentList=[ 'FileReplicator' ], sleep=True,
                             warnAfter=False, timeout=5 )
      except Tac.Timeout:
         pass
      
      waitForSync = True

      while True:
         if waitForSync:
            try:
               # Wait for files to sync before prompting the user
               Tac.waitFor( filesSynchronized, timeout=10,
                            sleep=True, warnAfter=False,
                            maxDelay=0.5 )
            except Tac.Timeout:
               # Files are unsynchronized
               pass
         else:
            waitForSync = True

         # If all files are synchronized, exit
         files = getFiles()
         if len( files[ 0 ] ) == 0 and len( files[ 1 ] ) == 0:
            break

         # If the peer becomes disabled, exit
         if electionStatus.peerRedundancyMode == 'disabled':
            print 'The other supervisor became disabled.'\
                ' Aborting file synchronization.'
            break

         promptMessage = getFileSummary( *files )
         answer = BasicCliUtil.getChoice( mode, 
               promptMessage + " Wait for synchronization to complete? ", 
               [ 'yes', 'no', 'cancel', 'view' ], 'yes' )

         if ReloadCli.answerMatches( answer, "yes" ):
            continue
         elif ReloadCli.answerMatches( answer, "no" ):
            pass
         elif ReloadCli.answerMatches( answer, "cancel" ):
            couldReload = False
         elif ReloadCli.answerMatches( answer, 'view' ):
            print getFileList( *files )
            print ''
            # Don't wait for synchronization of files before prompting the user
            # again
            waitForSync = False
            continue
         else:
            # we don't understand the user's answer so prompt again
            print "Invalid response"
            continue

         # if we got here, we don't want to show the prompt again
         break
   return couldReload

# Register doFileSyncBeforeReload to be run before a reload takes place.
# Register with low priority so it gets run after any file saving operations.
def registerReloadFileSyncCli():
   ReloadCli.registerReloadHook( doFileSyncBeforeReload, 'ReloadFileSyncCli',
                                 'RUN_LAST' )

# Only register this plugin on modular systems
FruCli.registerModularSystemCallback( registerReloadFileSyncCli )
