# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount, time, os
from CliModel import Float
from CliModel import List
from CliModel import Dict
from CliModel import Model
from CliModel import Str
from CliModel import Bool

resetCauseHistory = None
reloadCauseDebugFile = "/var/tmp/reloadCauseDebugFile"
# By default display last 2000 lines to display from kernelcrash file
numLines = 2000
numHistory = 10

# ------------------------------------------------------
# The "show reload cause history" command, in "unpriv" mode
#
# The full syntax of this command is:
#
#    show reload cause history
#-------------------------------------------------------

class ReloadCausesHistory( Model ):

   class ReloadCauseList( Model ):
      class ReloadCause( Model ):
         description = Str( help="Description of the reset cause" )
         timestamp = Float( help="Time of the reload", optional=True )
         recommendedAction = Str( help="Action that is recommended", optional=True )
         debugInfo = List( valueType=str, help="Information from debug file",
                           optional=True )
         debugInfoIsDir = Bool( help="True if debug info is a directory",
                             default=False, optional=True )
      resetCauses = List( valueType=ReloadCause, help="List of reload causes" )

   resetHistory = Dict (keyType=int,
                        valueType=ReloadCauseList,
                        help="History of reload causes" )

   def render( self ):
      if not self.resetHistory:
         print "Unknown Reload History:"
         print "----------------------"
         if os.path.exists( reloadCauseDebugFile ):
            with open( reloadCauseDebugFile ) as debugs:
               print ""
               print "Debugging Information:"
               print "----------------------"
               for line in debugs.read().splitlines()[ -numLines: ]:
                  print line
         return

      for k, resetCauseList in self.resetHistory.iteritems():
         #print "Reboot Instance %d:" % k
         #print "--------------------"
         i = 0
         for resetCause in resetCauseList.resetCauses:
            #if i != 0:
            print ""
            i += 1
            print "Reboot Instance %d: Reload Cause %d:" % (k, i)
            print "------------------------------------"
            print resetCause.description
            print ""
            print "Reload Time:"
            print "------------"
            timestampStr = ""
            if resetCause.timestamp:
               timestampStr = "Reload occurred at %s." % time.strftime(
                           "%a %b %d %H:%M:%S %Y %Z",
                           time.localtime( resetCause.timestamp ) )
            else:
               timestampStr = "Not available."
            print timestampStr
            print ""
            print "Recommended Action:"
            print "-------------------"
            print resetCause.recommendedAction or "None available."
            print ""
            print "Debugging Information:"
            print "----------------------"
            ln = "None available." if not resetCause.debugInfo else \
                                    "\n".join( resetCause.debugInfo )
            print ln

def doShowReloadCauseHistory( mode, args ):
   rebootHistory = resetCauseHistory.rebootHistory
   retModel = ReloadCausesHistory()

   if not rebootHistory:
      return retModel

   for key, reboot in rebootHistory.items():
      for _, cause in reboot.causeQ.items():
         resetCauseModel = ReloadCausesHistory.ReloadCauseList.ReloadCause()
         resetCauseModel.description = cause.description
         resetCauseModel.timestamp = cause.timestamp
         resetCauseModel.recommendedAction = cause.recommendedAction
         if cause.debugInfoFilename:
            try:
               # if the filename is a directory, just print out the directory
               if os.path.isdir( cause.debugInfoFilename ):
                  resetCauseModel.debugInfo = [ cause.debugInfoFilename ]
                  resetCauseModel.debugInfoIsDir = True
               else:
                  f = file( cause.debugInfoFilename )
                  resetCauseModel.debugInfo = f.read().splitlines() [ -numLines: ]
            except EnvironmentError:
               resetCauseModel.debugInfo = None
         else:
            resetCauseModel.debugInfo = None

         resetCauseListModel = ReloadCausesHistory.ReloadCauseList()

         if key in retModel.resetHistory:
            resetCauseListModel = retModel.resetHistory[key]
         else:
            resetCauseListModel.resetCauses = [ ]

         resetCauseListModel.resetCauses.append(resetCauseModel)
         retModel.resetHistory[key] = resetCauseListModel

   return retModel


#-------------------------------------------------------------
# Register show reload cause history command into "show tech-support".
#-------------------------------------------------------------
import CliPlugin.TechSupportCli

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
                                '2010-01-01 00:01:00',
                                lambda : [ 'show reload cause history' ] )

def Plugin( entityManager ):
   global resetCauseHistory
   import Cell
   resetCauseHistory = LazyMount.mount(
         entityManager, "cell/%d/sys/reset/history" % Cell.cellId(),
         "System::ResetCauseHistory", "ri" )
