# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Hexdump import hexdump
from ReloadConsts import wkDescription, wDescription, noRecommendedAction
from ReloadConsts import wuDescription, reloadDescription
from ReloadConsts import thermoInsufficientFansDescription
import LazyMount, Tracing, time, os, string
from CliModel import Bool
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str

t0 = Tracing.trace0
t1 = Tracing.trace1

resetCauseHistory = None
reloadCauseDebugFile = "/var/tmp/reloadCauseDebugFile"
panicBreadCrumbsFile = "/var/tmp/kernelPanicBreadCrumbs"
# By default display last 2000 lines to display from kernelcrash file
numLines = 2000

# ------------------------------------------------------
# The "show reload cause" command, in "unpriv" mode
#
# The full syntax of this command is:
#
#    show reload cause
#-------------------------------------------------------

def debugFileLines( path=reloadCauseDebugFile, base=0 ):
   if not os.path.exists( path ):
      return []

   contents = ""
   with open( path ) as f:
      contents = f.read()

   if not contents:
      return []

   if any( c not in string.printable for c in contents ):
      return hexdump( contents, condense=False, xxd=True ).splitlines()

   return contents.splitlines()[ base : ]

class ReloadCauses( Model ):
   class ReloadCause( Model ):
      description = Str( help="Description of the reset cause" )
      timestamp = Float( help="Time of the reload", optional=True )
      recommendedAction = Str( help="Action that is recommended", optional=True )
      debugInfo = List( valueType=str, help="Information from debug file",
                        optional=True )
      debugInfoIsDir = Bool( help="True if debug info is a directory",
                             default=False, optional=True )

   resetCauses = List( valueType=ReloadCause, help="List of reload causes" )
   full = Bool( help="Partial or full log", default=False, optional=True )
   kernelCrashData = List( valueType=str, help="Kernel Panic EEPROM crash data",
                           optional=True )

   def renderSystemDebugInfo( self ):
      if self.full and self.kernelCrashData:
         print ""
         print "Kernel Panic EEPROM crash data:"
         print "-------------------------------"
         for line in self.kernelCrashData:
            print line

   def render( self ):
      if not self.resetCauses:
         print "Reload Cause:"
         print "-------------"
         print "Unknown"
         print ""
         print "Debugging Information:"
         print "----------------------"
         for line in debugFileLines( base=0 if self.full else -numLines ):
            print line
         self.renderSystemDebugInfo()
         return

      i = 0
      for resetCause in self.resetCauses:
         if i != 0:
            print ""
         i += 1
         print "Reload Cause %d:" % i
         print "-------------------"
         print resetCause.description
         print ""
         print "Reload Time:"
         print "------------"
         timestampStr = ""
         if resetCause.timestamp:
            timestampStr = "Reload occurred at %s." % time.strftime(
                           "%a %b %d %H:%M:%S %Y %Z",
                           time.localtime( resetCause.timestamp ) )
         else:
            timestampStr = "Not available."
         print timestampStr
         print ""
         print "Recommended Action:"
         print "-------------------"
         print resetCause.recommendedAction or "None available."
         print ""
         print "Debugging Information:"
         print "-------------------------------"
         if not resetCause.debugInfo:
            print "None available."
         else:
            if not resetCause.debugInfoIsDir and not self.full:
               print "NOTE: Displaying last %d lines of crash log" % numLines

            print "\n".join( resetCause.debugInfo )

      self.renderSystemDebugInfo()

def mergeResetCauses( model ):
   userReload = watchdogReload = None
   resetCauseModel = None
   thermalShutdown = None
   for resetCause in model.resetCauses:
      if resetCause.description == thermoInsufficientFansDescription:
         thermalShutdown = resetCause
      if resetCause.description == reloadDescription:
         userReload = resetCause
      # 2RU systems append an origin to indicate whether the upper or lower board
      # generated the reload cause. Remove an origin from description
      # if it's present.
      if resetCause.description.split( ' on the' )[ 0 ] in \
            ( wDescription, wkDescription ):
         watchdogReload = resetCause
   if thermalShutdown:
      # when reload cause is thermalShutdown, ignore other reload causes
      del model.resetCauses[:]
      model.resetCauses.append( thermalShutdown )
   elif userReload and watchdogReload:
      resetCauseModel = ReloadCauses.ReloadCause()
      resetCauseModel.description = wuDescription
      resetCauseModel.timestamp = userReload.timestamp
      resetCauseModel.recommendedAction = noRecommendedAction
      resetCauseModel.debugInfo = None
      model.resetCauses.remove( userReload )
      model.resetCauses.remove( watchdogReload )
      model.resetCauses.append( resetCauseModel )

def doShowReloadCause( mode, args ):
   full = args.get( 'full' )
   rebootHistory = resetCauseHistory.rebootHistory
   resetCauses = rebootHistory.get( 0 )

   retModel = ReloadCauses( full=bool( full ) )
   if not resetCauses:
      return retModel

   for _, resetCause in resetCauses.causeQ.items():
      resetCauseModel = ReloadCauses.ReloadCause()
      resetCauseModel.description = resetCause.description
      resetCauseModel.timestamp = resetCause.timestamp
      resetCauseModel.recommendedAction = resetCause.recommendedAction
      if resetCause.debugInfoFilename:
         try:
            # if the filename is a directory, just print out the directory
            if os.path.isdir( resetCause.debugInfoFilename ):
               resetCauseModel.debugInfo = [ resetCause.debugInfoFilename ]
               resetCauseModel.debugInfoIsDir = True
            else:
               resetCauseModel.debugInfo = debugFileLines(
                  path=resetCause.debugInfoFilename,
                  base=0 if full else -numLines )
         except EnvironmentError:
            resetCauseModel.debugInfo = None
      else:
         resetCauseModel.debugInfo = None
      retModel.resetCauses.append( resetCauseModel )

   mergeResetCauses( retModel )

   if os.path.exists( panicBreadCrumbsFile ) and full:
      retModel.kernelCrashData = debugFileLines( path=panicBreadCrumbsFile )
   else:
      retModel.kernelCrashData = None

   return retModel

#-------------------------------------------------------------
# Register show reload cause command into "show tech-support".
#-------------------------------------------------------------
import CliPlugin.TechSupportCli

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
      '2010-01-01 00:01:00',
      lambda : [ 'show reload cause full' ],
      summaryCmdCallback=lambda: [ 'show reload cause full' ] )

def Plugin( entityManager ):
   global resetCauseHistory
   import Cell
   # BUG24704 - should look at reload causes from all cells
   resetCauseHistory = LazyMount.mount(
         entityManager, "cell/%d/sys/reset/history" % Cell.cellId(),
         "System::ResetCauseHistory", "ri" )
